%option noyywrap
%{
// This file is part of Rheolef.
//
// Copyright (C) 2000-2009 Pierre Saramito 
//
// Rheolef is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Rheolef is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Rheolef; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/* compile commands:

 	flex -+ -i doc2texi.l 
 	g++ -I/usr/local/gnu/include lex.yy.c -o doc2texi

  usage:
        doc2man < file.c > file.1

  NOTE:
	translate only a subset of texinfo files 
	into unix manual page.
*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef _RHEOLEF_HAVE_CSTDIO
#include <cstdio> // sscanf()
#else
#include <stdio.h>
#endif

#ifdef _RHEOLEF_HAVE_CCTYPE
#include <cctype> // tolower()
#else
#include <ctype.h>
#endif

#include <iostream>
#include <string>

#ifdef _RHEOLEF_HAVE_CSTRING
#include <cstring>
#endif // _RHEOLEF_HAVE_CSTRING

#include <list>

using namespace std;

static int line_no = 0;
string manual_name;
string manual_section;
string package_name = _RHEOLEF_PACKAGE;
string version = _RHEOLEF_VERSION;
string package_version;
bool in_example = false;
bool in_quotation = false;
static  char  string_buf [1000]; // be carreful to LONG string !
static  char *string_buf_ptr;
int str_previous_state = 0;
int arg_previous_state = 0;

#define error_macro(msg) {               \
	cerr << "error("<<line_no<<"): " \
	     << msg                      \
	     << endl;                    \
	exit(1);                         \
    }
void yyerror (const char* msg) { error_macro(msg); }
void begin_example()
{
    cout << ".\\\" begin_example" << endl;
    cout << ".Sp" << endl
	 << ".nf" << endl;
    in_example = true;
}
void end_example()
{
    cout << ".Sp" << endl
	 << ".fi" << endl;
    cout << ".\\\" end_example" << endl;
    in_example = false;
}
list<string> refs;
void make_see_also()
{
    unsigned sz = refs.size();
    if (sz == 0) return;
    cout << endl;
    cout << ".\\\" LENGTH = " << refs.size() << endl;
    cout << ".SH SEE ALSO" << endl;
    unsigned int i = 0;
    for (list<string>::const_iterator p = refs.begin();
	 p != refs.end();
         p++, i++) {
	   cout << *p;
	   if (i != sz - 1) cout << ", ";
    }
    cout << endl;
    refs.clear();
}
string make_reference (const string& input)
{
    string ref = input;
    return ref;
}
// get "geo", put "geo(1)"
string make_command_reference (const string& input)
{
    string ref = input + "(1)";
    refs.push_back(ref);
    return ref;
}
// get "geo", put "geo(2)"
string make_class_reference (const string& input)
{
    string ref = input + "(2)";
    refs.push_back(ref);
    return ref;
}
// get "curl", put "curl(3)"
string make_form_reference (const string& input)
{
    string ref = input + "(3)";
    refs.push_back(ref);
    return ref;
}
// get "level_set", put "level_set(4)"
string make_algorithm_reference (const string& input)
{
    string ref = input + "(4)";
    refs.push_back(ref);
    return ref;
}
// strip "class}" or "command}" for @ref{}
void strip_pending_string (char* s_ptr, const char* pending)
{
    unsigned int len_pending = strlen(pending);
    unsigned int l = strlen(s_ptr);
    s_ptr [l-len_pending] = '\0';
    for (int i = l-len_pending; i >= 0; i--) { // skip also remaining spaces
	if (isspace(s_ptr[i])) {
	  s_ptr[i] = 0;
	}
    }
}
%}
ID		[a-zA-Z_][a-zA-Z_\-0-9]*
SKIP            "AUTHOR"("S"?)|"DATE"|"METHOD"("S"?)|"SEE ALSO"
SUBJECT         "P"|"PROG"|"S"|"SYSTEM"|"F"|"Func"|"Class"|"E"|"D"|"G"|"M"|"A"

%x body section skip string_state verbatim after table arg ref iftex
%%
%{
// --------------------------------------------------------------
//  		start and finish body
//		  /*Prog:hb
//                   body....
//                End:
//                */
//
//                //<vec:
//                   verbatim
//                //>vec:
// --------------------------------------------------------------
%}
<INITIAL,after>^(("/*")?){SUBJECT}":"[^:\n]*\n {
			  cout << ".\\\" label: " << yytext_ptr;
			  line_no++;
			  BEGIN(body);
			}
<after>("//<"|"/*<"){ID}":"[ \t]*("*/")?[ \t]*\n	{
			  cout << ".SH IMPLEMENTATION" << endl;
			  begin_example();
			  line_no++;
			  BEGIN(verbatim);
			}
<INITIAL,after>[^\n]*\n	{
			  // skip
			  line_no++;
			  ;
			}
<body,section,skip>^("END")":"[^:\n]*\n {
			  if (in_example) {
				end_example();
				in_example = false;
			  }
			  cout << ".\\\" END" << endl;
			  BEGIN(after);
			  line_no++;
			}
<verbatim>"/*!"		{
			  // doxygen escapes
			  cout << "/* ";
			}
<verbatim>"//!"		{
			  // doxygen escapes
			  cout << "// ";
			}
<verbatim>("//>"|"/*>"){ID}":"[ \t]*("*/")?[ \t]*\n	{
			  end_example();
			  BEGIN(after);
			  line_no++;
			}
%{
// --------------------------------------------------------------
//  		special sections to skip
// --------------------------------------------------------------
%}
<body,section,skip>^{SKIP}":" {
			  if (in_example) {
			        end_example();
				in_example = false;
			  }
			  cout << ".\\\" skip start:" << yytext_ptr << endl;
			  BEGIN(skip);
			}
%{
// --------------------------------------------------------------
// 	skip also @index declarations, @noindent, @comments
// --------------------------------------------------------------
%}
<body,section,skip,table>^[ \t]*"@"[a-z]?[a-z]?"index"[ \t][^\n]*\n {
			  /* do nothing with indexes */
			  cout << ".\\\" skip: " << yytext_ptr;
			  line_no++;
			}
<body,section,skip,table>^[ \t]*"@noindent"[ \t]*\n {
			  /* do nothing */
			  line_no++;
			}
<body,section,skip,table>^[ \t]*"@c"(([ \t]+[^\n]*)|([ \t]*))\n {
			  // skip comments
			  line_no++;
			}
%{
// --------------------------------------------------------------
//  		new paragraph
// --------------------------------------------------------------
%}
<section>^[ \t]*\n(^[ \t]*\n)* {
			  cout << ".PP" << endl;
			  line_no++;
			}
%{
// --------------------------------------------------------------
//  		begin, end of examples
// --------------------------------------------------------------
%}
<section,table>^([ \t]*)"@example"([ \t]*)\n {
			  begin_example();
			  line_no++;
			}
<section,table>^([ \t]*)"@end"([ \t]+)"example"([ \t]*)\n {
			  end_example();
			  line_no++;
			}
%{
// --------------------------------------------------------------
//  		tables and items
// --------------------------------------------------------------
%}
<section>^([ \t]*)"@table"([^\n]*)\n {
			  // cout << ".TP" << endl;
			  BEGIN(table);
			  cout << ".\\\" begin table" << endl;
			  line_no++;
			}
<table>^([ \t]*)"@end"([ \t]+)"table"([ \t]*)\n {
			  cout << ".\\\" end table" << endl;
			  BEGIN(section);
			  line_no++;
			}
<table>^[ \t]*"@item"[x]?[ \t]+ {
			  // start an @item line
			  cout << ".\\\" start item" << endl;
			  cout << ".TP" << endl
			       << ".B ";
			}
<table>^[ \t]* {
			  // skip whites inside of @itemx
		}
%{
// --------------------------------------------------------------
// @code{...}, @strong{}, @var{}, @math{}, @cite{}, @dfn{}, 
// @url{}, @email{}, @samp{}, @file{}
// @pxref{...}
// --------------------------------------------------------------
%}
<section,table>"@"("code"|"url"|"email"|"strong")"{"	{ // in bold
			  cout << "\\fB"; 
                          arg_previous_state = YYSTATE;
			  BEGIN(arg);
			}
<section,table>"@"("var"|"cite"|"dfn"|"math"|"emph")"{"	{ // in italic
			  cout << "\\fI"; 
                          arg_previous_state = YYSTATE;
			  BEGIN(arg);
			}
<section,table>"@"("samp"|"file")"{"	{ // in bold with quotes
			  cout << "`\\fB"; 
                          arg_previous_state = YYSTATE;
			  in_quotation = true;
			  BEGIN(arg);
			}
<arg>"}" 		{ 
			  // end of arg
			  if (in_quotation) cout << "'";
			  in_quotation = false;
			  cout << "\\fP";
			  BEGIN(arg_previous_state);
			}
<arg>"@@" {
			  cout << "@";
			}
<arg>[^}\n] {
			  cout << *yytext_ptr;
			}
%{
// --------------------------------------------------------------
//  		references
//	TODO: make it as:
//		See also geo(1), csr(3).
//	and compute the all references at the end:
//	SEE ALSO:
//		geo(1), csr(3).
//
// --------------------------------------------------------------
%}
<section,table>"@ref{" { // reference
                          arg_previous_state = YYSTATE;
			  BEGIN(ref);
			}
<section,table>"@pxref{" { // reference in parenthesis, with see
			  cout << "see "; 
                          arg_previous_state = YYSTATE;
			  BEGIN(ref);
			}
<section,table>"@xref{" { // reference with See
			  cout << "See "; 
                          arg_previous_state = YYSTATE;
			  BEGIN(ref);
			}
<ref>([^}])*"command}" 	{ // end of class ref
			  strip_pending_string (yytext_ptr, "command}");
			  cout << make_command_reference(yytext_ptr);
			  in_quotation = false;
			  BEGIN(arg_previous_state);
			}
<ref>([^}])*"class}" 	{ // end of class ref
			  strip_pending_string (yytext_ptr, "class}");
			  cout << make_class_reference(yytext_ptr);
			  in_quotation = false;
			  BEGIN(arg_previous_state);
			}
<ref>([^}])*"form}" 	{ // end of form ref
			  strip_pending_string (yytext_ptr, "form}");
			  cout << make_form_reference(yytext_ptr);
			  in_quotation = false;
			  BEGIN(arg_previous_state);
			}
<ref>([^}])*"algorithm}" { // end of algorithm ref
			  strip_pending_string (yytext_ptr, "algorithm}");
			  cout << make_algorithm_reference(yytext_ptr);
			  in_quotation = false;
			  BEGIN(arg_previous_state);
			}
<ref>([^}])*"}" 	{ // end of class ref
			  cout << ".\\\" REFERENCE " << yytext_ptr << endl;
			  cout << make_reference(yytext_ptr);
			  cout << "'"; 
			  in_quotation = false;
			  BEGIN(arg_previous_state);
			}
%{
// --------------------------------------------------------------
//  		name section
// --------------------------------------------------------------
%}
<body,section,skip>^("NAME")":"[ \t]*	{
			  cout << ".SH NAME" << endl;
			  BEGIN(section);
			}
%{
// --------------------------------------------------------------
//  		package, version
// --------------------------------------------------------------
%}
<body,section,table>"@PACKAGE@" {
			  cout << package_name;
			}
<body,section,table>"@VERSION@" {
			  cout << version;
			}
<body,section>^[a-zA-Z][a-zA-Z_\- ]*":" {
			  if (in_example) {
			        end_example();
				in_example = false;
			  }
			  // make title
			  for (unsigned int i = 0; i < strlen(yytext_ptr); i++) {
			      yytext_ptr[i] = toupper(yytext_ptr[i]);
			  }
			  yytext_ptr[strlen(yytext_ptr)-1] = 0;
			  cout << ".SH " << yytext_ptr << endl;
			  BEGIN(section);
			}
<skip>[ \t]+[^\n]*\n	{
			  // skip
			  line_no++;
			  ;
			}
%{
// --------------------------------------------------------------
// 	specials
// --------------------------------------------------------------
%}
<body,section,table>"@@" { cout << "@"; }
<body,section,table>"@{" { cout << "{"; }
<body,section,table>"@}" { cout << "}"; }
%{
// --------------------------------------------------------------
// 	skip blocks: @iftex @end iftex and such...
// --------------------------------------------------------------
%}
<body,section>^([ \t]*)"@if"("info"|"nottex"|"nothtml")([ \t]*)\n {
			  // cout << ".\\\" INFO" << endl;
			  line_no++;
			}
<body,section>^([ \t]*)"@end"([ \t]+)"if"("info"|"nottex"|"nothtml")([ \t]*)\n {
			  cout << ".\\\" END IFINFO" << endl;
			  line_no++;
			}
<body,section>^([ \t]*)"@iftex"([ \t]*)\n {
                          str_previous_state = YYSTATE;
			  // cout << ".\\\" IFTEX" << endl;
			  BEGIN(iftex);
			  line_no++;
			}
<iftex>^([ \t]*)"@end"([ \t]+)"iftex"([ \t]*)\n {
			  // cout << ".\\\" END IFTEX" << endl;
			  BEGIN(str_previous_state);
			  line_no++;
			}
<iftex>\n		{ line_no++; /* skip */ }
<iftex>.		{ /* skip */ }
%{
// --------------------------------------------------------------
//  		strings
// --------------------------------------------------------------
%}
<section,table>\"		{
                            string_buf_ptr = string_buf;
                            str_previous_state = YYSTATE;
			    BEGIN(string_state);
                        }
<string_state>\"		{ // saw closing quote - all done 
			  // go back to previous state
			    BEGIN(str_previous_state);
			    *string_buf_ptr = '\0';
			    cout << '\"' << string_buf << '\"'; 
			}
<string_state>\\[0-7]{1,3} 	{ // octal escape sequence
			int result;
			sscanf (yytext_ptr + 1, "%o", &result );
			if ( result > 0xff ) {
		    	    error_macro("octal escape sequence `" 
                                 << yytext_ptr << "' in string: constant is out-of-bounds"); 
                        }  
			*string_buf_ptr++ = result;
			}
<string_state>\\[0-9]+ 		{
                           error_macro("bad escape sequence `"
                                << yytext_ptr <<"' in string");
                        }  
<string_state>\n			{ 
                           error_macro("unterminated string constant"); 
			   line_no++;
                        }
<string_state>\\n		*string_buf_ptr++ = '\n';
<string_state>\\t		*string_buf_ptr++ = '\t';
<string_state>\\r		*string_buf_ptr++ = '\r';
<string_state>\\b		*string_buf_ptr++ = '\b';
<string_state>\\f		*string_buf_ptr++ = '\f';
<string_state>\\(.|\n)		*string_buf_ptr++ = yytext[1]; 
<string_state>[^@{}\\\n\"]+	{ for (register char *p = yytext_ptr; *p; *string_buf_ptr++ = *p++); }

%{ 
// --------------------------------------------------------------
//  		others
// --------------------------------------------------------------
%}
<section>\n		{ cout << endl;  line_no++; }
<section>.		{ cout << *yytext ; }
<body,section>^[ \t]* 	{
			  // skip spaces at begining of the line
			  if (in_example) {
				cout << yytext_ptr;
			  }
			}
.			{ /* skip others */ }
%%

// --------------------------------------------------------------
//              header of unix manual
// --------------------------------------------------------------
void print_header()
{
    cout << ".\\\"" << endl
      ;
#ifdef TO_CLEAN
    cout << ".\\\" Set up \\*(lq, \\*(rq if -man hasn't already set it up." << endl
         << ".if @@\\*(lq@ \\{\\" << endl
         << ".	ds lq \"" << endl
         << ".	if t .ds lq ``" << endl
         << ".	if !@@\\(lq@ .ds lq \"\\(lq" << endl
         << ".\\}" << endl
         << ".if @@\\*(rq@ \\{\\" << endl
         << ".	ds rq \"" << endl
         << ".	if t .ds rq ''" << endl
         << ".	if !@@\\(rq@ .ds rq \"\\(rq" << endl
         << ".\\}" << endl
      ;
#endif // TO_CLEAN
    cout << ".de Id" << endl
      ;
#ifdef TO_CLEAN
    cout << ".ds Rv \\\\$3" << endl
         << ".ds Dt \\\\$4" << endl
      ;
#endif // TO_CLEAN
    cout << ".." << endl
         << ".de Sp" << endl
         << ".if n .sp" << endl
         << ".if t .sp 0.4" << endl
         << ".." << endl
         << ".TH " << manual_name << " " << manual_section 
         <<     " \"" << package_version << "\" \""
         <<   	package_version << "\" \""
	 <<   	package_version << "\"" << endl;
}
int main(int argc, char** argv) {


// --------------------------------------------------------------
//              command line options
// --------------------------------------------------------------
    for (int i = 0; i < argc; i++) {
        if (strcmp(argv[i], "-name") == 0) {
            char *p = argv[++i];
            while (*p && (*p == '.' || *p == '/')) p++;
            manual_name = string(p);
        } else if (strcmp(argv[i], "-section") == 0) {
            manual_section = string(argv[++i]);
        } else if (strcmp(argv[i], "-package") == 0) {
            package_name = string(argv[++i]);
        } else if (strcmp(argv[i], "-version") == 0) {
            version = string(argv[++i]);
        }
    }
    if (package_name.length() != 0 && version.length() != 0) {
        package_version = package_name + "-" + version;
    } else if (package_name.length() != 0) {
        package_version = package_name;
    } else {
        package_version = "-";
    }
    print_header();
// --------------------------------------------------------------
//  		run lexer
// --------------------------------------------------------------
    yyFlexLexer lexer;
    line_no = 1;
    in_example = false;
    int status = lexer.yylex();
    make_see_also();
    return status;
}
