require 'mocha/parameter_matchers/base'

module Mocha
  module ParameterMatchers
    # Matches +Hash+ containing +value+.
    #
    # @param [Object] value expected value.
    # @return [HasValue] parameter matcher.
    #
    # @see Expectation#with
    #
    # @example Actual parameter contains entry with expected value.
    #   object = mock()
    #   object.expects(:method_1).with(has_value(1))
    #   object.method_1('key_1' => 1, 'key_2' => 2)
    #   # no error raised
    #
    # @example Actual parameter does not contain entry with expected value.
    #   object = mock()
    #   object.expects(:method_1).with(has_value(1))
    #   object.method_1('key_2' => 2)
    #   # error raised, because method_1 was not called with Hash containing value: 1
    # rubocop:disable Naming/PredicateName
    def has_value(value)
      HasValue.new(value)
    end
    # rubocop:enable Naming/PredicateName

    # Parameter matcher which matches when actual parameter contains +Hash+ entry with expected value.
    class HasValue < Base
      # @private
      def initialize(value)
        @value = value
      end

      # @private
      def matches?(available_parameters)
        parameter = available_parameters.shift
        return false unless parameter.respond_to?(:values)
        parameter.values.any? { |value| @value.to_matcher.matches?([value]) }
      end

      # @private
      def mocha_inspect
        "has_value(#{@value.mocha_inspect})"
      end
    end
  end
end
