# bitmaps.tcl:
# Chess piece bitmaps used by Scid.

# The piece images used here were generated from freeware chess
# fonts available in TrueType format at the En Passant website
# "Chess Fonts" page, http://www.enpassant.dk/chess/fonteng.htm

# The authors of the original TrueType fonts are:
#   Alpha: Eric Bentzen.
#   Leipzig, Merida: Armando H. Marroquin.

set png_image_support 1
if { [catch { package require img::png } ] } {
  set png_image_support 0
  ::splash::add "Img package not found (tkimg). Some piece sets are disabled."
}

set boardStyles {}

##########
# Size 20 is only used in Material search window, not for boards.
# It has two special extra images, wm20 and bm20, which contain a
# bishop and knight, for indicating a minor piece.
#

image create photo wm20 -data {
  R0lGODlhFAAUAMIAAH9/f7+/vz8/PwAAAP///////////////yH5BAEKAAcALAAAAAAUABQA
  AANweLp8cCG02cSwNB8RSACctgBAR3iDqJDd5wlidBLCHGb1CQzzx+yPDYAWyJ1gixpSwOLM
  CClFESSRup7RImF42zxP0Vpg0EE6SGjSCqxTKYxHN4RJ6sYETHxVNa3vM2gDQyBoGkNbhIdq
  FHRBZyAaCQA7}

image create photo bm20 -data {
  R0lGODlhFAAUAMIAAH9/f7+/vwAAAD8/P////////////////yH5BAEKAAcALAAAAAAUABQA
  AANneLp8cCG02YQYgupj+5CbEgyYAAQCuJHlNYZo1wHDo7VyOjSAebQxS69R25UCvVlmMXIp
  TrmhSGgB4S5LzoVQegK+YJtWwLWEFjnzGVL7ftYQMoGQggerZ3CrLealoxomXxJIX1kNCQA7
}

image create photo p20 -data {
  R0lGODlh8AAUAMIAAH9/fz8/P7+/vwAAAP///////////////yH5BAEKAAcALAAAAADwABQA
  AAP+eLrc/jDKSau9OOvNu/8VAIBkJAhl2ohqe5xuCgTBGL/oQaMwJew30KwWhAkGA0Vv8gvk
  LALRUxJ9BTSAk40qFXWzkKZWCmQit5Uo2GdDW4ZuiFQ9koZ33mxZAjhjBidJFDNIRBgBhRQD
  Q4t9NH0NP3o1BEgEYjNTDix/SIITfQOIcROIooOFpouekV6PlQMEQ2qaK6QSsZUholGit5GA
  BJeAuMIixgDCnwrJAbKLsMPNsiY0VxeeyxGhnoZm2cTb4OMrP88C0X3NVWF+2CLaELnCUTRm
  6CfDtQuUwv7G1xb8iHUkmSV1lZy0GpErSZR9DbJVUOULCUQl3VRdPDL+rtsKRM8MxuqDjlcr
  FBIflkomK+CdLP8I2Ivg5NIOmxIe5RnygOSzhDKlLGqg01LCGjMhUHQpj1AhmfEYmHIy8JSJ
  jlZXAHIUDWRBojWcFnK1zZk/bw9oBLt09lcuMcpA7eS0CU8WVyIeMTBHD9ARdMjkjPt14BhF
  QEkddDuhSm7MqIUrrgJ0ZhSDvJIr+7o14x9dB3z9zTtCE3A+nHxiUpNXsFKgXj+mHPN3pKa/
  z5cH48LqJJwDVWoT8enYDis4W9k4cjzjliWk0p5ZBn5QcKdvOardXqqXN1nJVjFpxMTNgJw4
  4zypj3V6kRlxecYl7g0+mZtewcLQ/vYMjTb+U6lh5fXfJtmVNcpOj/xnGzL/kHaeO/AZ1xtN
  AaY3nHk9dZOHKnH0th43M1D4T2KXzebEbKKVFcoMhDEz1y8cvUjIPo3AU2MmNI0zYGEU2eiJ
  a3JUqF9PFT6nnnd5GHMdRrScQMeSC3Q23oCdxXaEapAdMI+Sisy1I0YyQslMgOi48iU34AzY
  yxlQJTfUA1hRoJMXYmJkHESOLIXIl1v+A5mAMgE2IkS9qLUGdDH9gIt0fprAaHQRxHeHeIfV
  eEc2CuV0Z6TrNVYcVrdEodp0ZY36WVVsPrPYb/HxmVFykfrYyJfLddTeCx15MZ8ovJlEVHx1
  zoNillrWICgh2zxpeluLfbZVnllK9pefNiiaSopPWLrVD0BZoqnbboOhBexxEQF7bXxuGfdg
  rlTEJxt9YDh1n0Dj7rOmjhtVmmmG6E2ArlRpapGmYsDa6+2qlwYcxAWHyrHwwxAX1h47EVds
  8cUYZ6xxBwkAADs=
}

image create photo e20 -height 20 -width 20
set x 0
foreach p {wp wn wb wr wq wk bp bn bb br bq bk} {
  image create photo ${p}20 -width 20 -height 20
  ${p}20 copy p20 -from $x 0 [expr $x + 19] 19
  incr x 20
}

####################
# Alpha:

lappend boardStyles Alpha

set pieceImageData(Alpha,25) {
  R0lGODlhLAEZAMIAAL+/vz8/PwAAAH9/f////////////////yH5BAEKAAcALAAAAAAsARkA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq4QAHSBELD0QFcuLN+rzUsC3yZAmP0WA6EiqQEM
  gq+jI9gRWKVL5YGZcUI7SW038IQNjBYnUz14haOSM3crzzyt4kvYA4gJdGg4e2xuSXARd0KJ
  dlZUQ2GBGGcABACOGH2TegMEf0kETAKgeQ1yinUafzpWkXqNeE0xAYcTd56NNrgUi0uwqWCc
  BKNjwcJFrXGzlbMWn7igjcMUAcwKfcgVqhu2vhd9r6zYENwCtIjg6N0SMXCW4hPaGp/Ccziz
  lsXHFtTGwtQV8/oJrBehHJpZ8cbJYIevnDt2AMGRaiELWiMn+9K9Q5L+Dt1EBqwW+Nm10GEQ
  k9TKASxGME45gf3+1SJCzxiZbJRgCrNkjg4cUcz6dFqgpgG/neyWKfO3cYtEV3iSvJrVlJvV
  bB1f9TS04OIBS1esETy6rFwbajn1TePUJw0Znf14ShBqTJTNng6ohdkr8wERVU4q2XjbRgFQ
  BnRjUkp8d1qjptb8ZDWI1xq4Tlor6N2bpO+UsGCpSI7SCDERP6d5nmZVpHLkxhMypR1o813A
  aPQ+IiHDGZXfM1mE/bHLhUygNjCFLo2rO7QV15ZTpksE/BxuzOoQ8eZ8E0JIp6Vf+RiJxAsu
  5ebxYIxAlukuF3BpQgMi0G7dlTpbEn0yY+n+k2AYxVBYV/GdAVdCRmlEQRCSgQNWGwiCdBk6
  EXU0kRf9ZTXYFwSiU4SB4PwVgRp6bQZdUjAtZ0kcs9knTS35GKMfA2RY0Y+LeLSCHEz/6bTe
  AyQ+xhUQWdUYxFDWRfMUSRZO0OBkO5SSXlRBkNOGblsshmUWBgkZY4REyZcidEvABUpsVJlJ
  lTkBxeViTFjasiUd04XGCZiWTUYmeNPFBmVmpUA4IXYXdUefcBfIoNIBy93o5CycXWPBm446
  OZuP2DRaF6ZYlTbpc1k0CKGTMnAX5YKTOfZnpi5c9pdWbaEJ20oKtFfpOtt1BplhZuKJRIw8
  kmKJTpR2QqZzi9L+yuh0ZTzK3a4HQMkklMICJZ5Ujxk7TVqxenOkj74G18+MDxSL6IKCSUYG
  Jb7ShS0uwVRD6lRdaNhsMtzNGW2qqLY5SlgUjSKjjJzIO6KYrXl76bjhRtvbvQs+LBVW6xrj
  hCx+VswcRBQgC2oak00Mz8MNdyUxxAUJNhVyG2VyRoidHYuwP3syujDDu/x3Ermm5fosdDsK
  xi2WL9HzbSUle+yVZiGj/MC7UdlTJckGP63uy6wQLRkntsQwzcy1ZWOm0ePwVeMZPhd0s5uk
  5tQj0kTbx1NdDT+JDmRQ1lg2JDDvVdCfAEOw2pF3EEE0doOLEq6BY1ftQGBjy2VUpCua3sOd
  d2PfB8TaSE+T+UbkdASd0hIlK1JH+97tHeDhmiXxik6e3NlMsvs2js8ka2HKXirhk+/juD/8
  ju8nm+7A7hIb+jTguoXeJAPO9/48Yqy3O7XExpcSPF8fsSH76LW/0YD3e0Dfmznkfz9i+AO2
  wL5r6T8M/vvjPxxopI+zPySQ9K8ffs1YCKAAB0jAAhrwgAhMoAIXGIEEAAA7
}

set pieceImageData(Alpha,30) {
  R0lGODlhaAEeAMIAAL+/vz8/PwAAAH9/f////////////////yH5BAEKAAcALAAAAABoAR4A
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s64aACAxD/N44aYNznf9ATmDnEdAEwRwtaRmG
  BFAk87WcSgKCwJNAkFoPgIC2ISZmZmLf14Edf6Jea9i9KHfQAfWHVg1hu35QdBcDcAqGB3AD
  FwJzDGVxGYVZeh58iyJYUIFZG5ObiVGHUZgVUXSakRiTeWYbaQNiPFlchaVCYoMWAAJcUr2A
  ib63E71hjwDBHX0fqZ0fvHBGzYIb0aCIiozVdaKWxByxXFxOHWXjBHOuFbwDtY0awEaFw/TK
  7FBE0eur4J7S8MwBzMVvwrUoBSMo+jQP0a58DPbt8aeBBjpmGcSkc4f+DkvCB7ySjQuZAdil
  Wnx8lczDIJYqCaxsTMIUBgrFCAxtCqQkLdZHBwfh/GwAsKi3bbcmrRIkUycYTTchWByH8WEA
  ApNEjvN4ISQ6LiQZfR077iXOIXEasaTAq1UhtABCYqnhcqiCnNM0XLtKwFmvcviMBrQguOhS
  UKPyGqQ0I0tIuUbQDGbLl5zdR43Ifn3GNjNYrWEreNQMljNQiHfdrRXXR2JEz1s9t/0aerG0
  y90K2zwW+JTfy34F61rg+sAnTMeJo1aejEYyj5Cu9kg3eYLGvsMlzCaNrp0prVg5gs0OIWjR
  hJm95KF+qLkbY5HWj5UvbmysC87Il9ctrW/+1AUAHQGHfqdxVx1RRzmTGGcOKbBeKs9dIt0p
  WBHIXGmXFRIXd+J1YSEv6MARIm7zXHKJWXXQ8N5WKpYFoIpEaQZiF+CF+NBthGQR3IBRVNiZ
  iGUJhUGJJh5hHTehkJITgEgegtVtMzYSlDsotjREd2tRwEdlZHFZiJYdxiIfSkMaqOVcTpYG
  mxRvEWNPly1qphhORv23gC3mCaVklXf1FFdOdjLA35l5DWocJQ3QU5R/4iw65wNzFflWBbF0
  OFYYFx2YqKVTsVgmaXwq0FYuHXGZC1cNzEjWPqB+tOMplPJ3FSm1PYCXgNpsU1hgYsiqiSsg
  nrcPXr1oaitHVcH+pCNVFh1R46MFdvcsiWaKdkSHG1JlRJYxggqMnDmSstSr+RzXxX/m9Yqj
  roI1wV+AEPSC16x/BXjPBFMlG0FckAk4Zoh2XqeZLDkaaJcxNdIWF4r/bsblV/dZwFCgxC28
  53oTe3ekbha2xN/B7zZIBpUAhddFyRF/V9oZOroh8FjQPiBlkQtL8vBX3F5hKWn6ggEbxDtj
  yC5iN3LGR4C2mDKPSD0QHatuFJNb1E2PIY0rQrXu+y2NFXmhIaiBRlPkchIHzexD3I0I080o
  DUyxz+sudadgjYXdCM3G0gl10e8mJM6U7uQUIX5hdrzfzxDHLHPCvFRkNpmEMU3zytb+VdbD
  TM5tZbiDA92h25MU7yoJf28vdBFDFGjUDrLMZkYwPjDjdiGHijuwtY2SPB4eIXnYZKJNOUOQ
  hu9jb7s53ADJLip/oBMGrs2Sbvl2feGdRE6gAkufEjmbe0Vb3hG8rNmXT9eZkaw56p60aFeO
  BJ4jXRWmvNRQPClxYW87qP7mRwvGRxPqEkR+XteZhAmNd2l7R9lsUS6wxI9tOAPfndpHHdiE
  gWI+iQ0txvO2PMWtfPIjH5hu1bPThEyCdwkgX1KRBokFkGm+oZj3ZITCFCXwSQUpkkbqB4tL
  FMOAqwrVodpXE6ahSWmm6pAm+CY//LzLWTgxUXDyYKJinNC4aVKpgSautMUeRO4gjcGaEGdI
  mqzF64bU8cfqLhGhv42tILdLmxCpR5rgxQuI3RnjFUH2xJh50HdNjNcV50g/RJmikFiglO5O
  5zw0xqxhFRpDGuyTHVXdUIhxfB4FMkmWOV4xdCFrA6SEwzmjVHKQmHRkIxPoyROC8ooyM9/c
  jCIVVKJrj1r6ZC5dSZhBxnJqHqOlrWxpm5D9hFj+W4Myl8nMZjrzmdCMpjSnSc1qWvOa2Mym
  Ni+QAAA7
}

set pieceImageData(Alpha,35) {
  R0lGODlhpAEjAMIAAD8/P7+/vwAAAH9/f////////////////yH5BAEKAAcALAAAAACkASMA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLIPDbN/4Wed8fwKmgGAY8BkDg6JRAywJ
  icseUhm1CKkiwYAwEFRz3d2Xci0Nz+NbOD0RApojLdfLVgCgjCf8cx7WIXpmfX8HdwJYB4Eh
  fXQkAXcAiCJdkYeOZ5IZfTuMCl1oGQMAXHsHcgBiHqICqEGQmTR9sBxPlh6bCp0Hn36hZ6W4
  NJCpIXcDx6yOdwEEBMYjvMQaWqKcqHeeqF0aonMNp9IcayWGv8qMWtBn4RjBuryNF/Df65Pp
  l8zNZcWszf5Psy4sI+DmVrNqYeYcQ9UsnhVMDWoFxDBuRC1ZIsoxWgb+QmOyDn20MVo4KIPE
  iBBDVJy0xR8XdhkGtnT5jFY/mramHdw0Z13DDZ+IBQV6DAukPMdgUoBXr+MvdNU6XGQ0MQK6
  q+iADhG6lWgSBkcXIDkGcuZOp8yMuWwW1uTNtRynmV1Ldxs3ZN+S+mI1JFIhLcke9e06DatT
  Ij37OCv1ECseTY4da+21oKm8wayKGAtsyJDSCMfWkpWamWBSus5yPhyFOnVVB29b41yKSM69
  XQK8IY0wUB+k3AT5Rvr3keLVz21CJkU3yi+GqVhf04t8lfb02/N2Q+hN8DcXN6X1Fb9whbj0
  B2plu3yiifVOs9hMvskn+5Fzq4RzMbNraov+EAa8oDccXHLs5B5xjCWHznkOeERdAP+1M1Ji
  oDznoGP3QWCZKSVxWCFulIE1XDVpUWPiKEgwVJAvW6z0XHjqueSiBN3IWKA/t2FwIXNLodiI
  EDgW0UWQlaVFTHmoveGPkqjtc1kwGkBHnW3ILcCUHFdW2SBwso0HQXaJUAViSh7mGGaJmCxG
  QG5KXuQjg0gGx6AdN6oHoUshTlAnIzZqldSff5r5QDf8DXkQEAw1JAY4DdTYGpZcJqmlAlLO
  +deUfZWjWwXorJUVN9QAGihyYDKF6EiVbbhLcwuu2WJu0G3RigVd0CfepHYkEWlroS3pUARA
  yjiMp5Y6ymuV5Sj+0auikY5WC2N11tXcXKL9CppjuP61xZTIDLGpgpswg8SCoHJbQbKeQAUV
  pebktSZzpDgj61XfSjCMqG1VUFSMDVVbpaE2UivoUtSKNmln/QXcJ8IPREsXYDHyN8GO7dKK
  qbeazjjocRSqqi+mB/dVJnWXJphwdK7CilW9242EjDZfWRybpP5acIi/a0UYSsEy4sqKIbau
  mXOm1uYaY18Rm9wAxTCfdyVW256hkM1QdZynzNT5fLFh20VdNSSp0StKKO4N3KMbpnn2yYE4
  asnkP0nApbS9PPsz9mWp0RX0mqOYjVvEDnuaLVPZZiMlTxpNfS5zq2y0AcUhGbc1ql/+5va0
  5bZx7PeXM41m0hzjBnrzo1rG2ZqT7dR9VuoE8du6xA3vrbBssH8M5XMUitodnxo/cPhVlo4c
  GWSTfxh72BNiDm/tFNzIfAVP1Grl6KRfNlY52ghReC6qszyBsa5v/nd9uz5ceHUbXETF9WKL
  D+BWc3W7vfCSSV78Z0NKua3VN7uPErEdqIj2joarzgCKYRsIXJ9eJLvTFW1pbPOUet4QJeAF
  8Da/81bvWmaaPzHDSxLClFuK90ARBadTXpPFYpjANgqCQA+Ay1b5cGQQfpVQQ/H7k2jmJpaZ
  vSQMy/KVpS7Ew8ZQR3EXIBlILlaYX4iqcSIjQz/0cJoWkeT+ZnEhD/XEE7w8wIh2/ptel7po
  ivBxYxRQZJy8VkMK5WiEIVnUYnQ+kEHFbHBjT0QGGbcGqkTRriZSjGB3ZBdHC7yNJkXUkyBR
  c0cHAPGJYWxYHgF1w43Bj5HRkCM1aMeL4DGNDxdD4scmGUnYmNEXB3laryKJtm4dZ15DXOQK
  xTFDXpVShFEyxiSXkyFO7Y1aOrNYXRxWygVADoRPwtS7tKZE+YRql5lpj7o2UslGcexC2ULb
  BFWTxO4FrIKhQ92LZEmXQiqSPtH4R3wuc6B0+iqbWwve5FJTuErZhIQMOkQ5OveLYOqrEj7Z
  CSZm9RBvugaVrnuPBMZCSVax6on+VVEgGOUIn9h8gkFBWc+ABIqrOpLLfpgK3UIhCTVRRZSE
  VwOW8hBzkW1h9FVO1IgVWZfQ54GGnMOEiRtMGhxTiAtf3DRlQvFkPddprx04nU0SSbiXi1VC
  p1Oin5gkgNKUVu6Yg/EZVp+Ct6G6iiIGxdMDt4iTImhTbyUMllfXJB0/8stzvhyqP2mEUlxd
  DkPrhM2DwtRM31U1qF0bKlyb18AmVTNdJLRrVTNx18fUESaNzRrVihfCySVzcopFKWP3yq7I
  QPavWkJpZflohV1OEk6mPa3vVOsAhpoUWKllbRti+1ro0RZQ0nHtbVF7Ww+uNo+Z0K1vAdFb
  3Nq2uDEgIwNyk0uI5jr3udCNrnSnS93qWve62M2udrfL3e5WIAEAOw==
}

set pieceImageData(Alpha,40) {
  R0lGODlh4AEoAMIAAL+/vwAAAH9/fz8/P////////////////yH5BAEKAAcALAAAAADgASgA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987/eCQEDwKxqPkuAQyWxu
  AAFVcDBYOnvCawYqFQqJ2l02bPmmqgRCYEBmAAQAyDt+GrcnZlTVy74f5nJwKHZ+D1VUKUFp
  Vm1QQnRuXpAiXl6FhkJ9J0p5jZIOjgGTIZWEJgB7A6Onj1CrI6kjpWClUQecphhDBEEOu70k
  AkrCKqiZryShj2eZspW0pQq4thvRDdYjwl9gJ1ADwlXIIYcAi2vdqeIZa9q03wJV0kPw1Bq/
  9Qr3dfispeofe0ppSpaO0rN81qZ1wLaAobNia9KkCWdCSTmJvJZli0j+oAo3D7vU0BKQBpii
  jB08nmugcuDDFMoq/dswrRLFjbNIHTxQS16llHwcBHRpUAUVjIs+juOIdOJKEOEwegMRclbJ
  Ulc97FnljWhRZrUYiQgY9huipWWphl3bE6goUM32qQiJEdhYUQRQDWja8SkHRxelrpkpgS7f
  w3arwSNKJXEGXKoUKKNjzItSe2xJkAUQU0hHrxg6CyQcga1pSwvjMgjYATJlmZLJXl5HEqnj
  Dsb2hgqM8RBupnxvKj5MHOVCOG8XuLpdwZEwbUO8cQ437LnGaplDVB7sVODn36evYw6/FuQn
  5T81OLcefbCxN9ugk34Ar+m3EBaLI+20bq/+/ngceKSfbX5FAI9SJ52lwFFqjHJgBN7Y5wVG
  fEg4XwM18dfaLM+RZVlHs1VAHmobeDhigRA8yABkq8G24n1yANddAHmVUyFSwm0BXI5PHDIg
  UoBpEBWFNAoGWgVT8HaYMSGy9I1fJ7kiGV4itbjYAxHyBUVtVykpEY8WiJYcbiaSl9GFC4Zl
  jkAbnqghBEOt6M+UO6WpWiT+KeHflmrQGMRFe9QGpgVTONUkodz9iCOKE7jSlGcEohnJiaTh
  lRiD3x2AqYI8kcSFUP4dNmSWoh45QU2HIummVYxKIOaac/5lGGJj4pGenUFpymZDt67mn4fl
  UIecbt5lqmOofUn+uoCAivIiGD8ToEFkJfuZSgE7z2WbLYAiBlsPXa7wmVVDxD4w66O/Fkkc
  cxSIqSyuJ3JCErcWmPjorhxgq6229NpKIk+1cBYWAw7lM28t/nEkbVnGhSZjsvki5yVfwuxX
  67VcllQdgSBlTOsFoURmMIVUqNvwdqucVJwi4hYnVnOmvftFX+FBd8+FYVXMS4bmjQgyH5CU
  VaZyAaVc5Fp7sXyPmlVeMGRTU9lDksfEfUNgqtdQ3SfF0F5wblPsGmjZLSZnNOsS8qZY9mHz
  rM3Xy9GGZy0EmWQYFnwTBoE1wWvBWrAuJz42tt1rEZE2fUf3LdJJbCXl9Ihz+/L0ymv+T6GL
  1pByfNyAYYtd99qZSzREQE2qXFzJWj+6N7ymRX6Nm3uRlZHrCDGdXb6BC/55vKSL7fczvAhI
  uOPNaTtPO9miKROQXNLjpRIYSGvx27TDpejFIqqSCdhUb9WvL6nXNbN+nTfwRra1aKv8icRS
  C/3jtqclK6XjrdpWaQf3DVifDJfvu+iro5u3ALCd0bCtazEKX6Q2dL0ALuBPIFJUfaQkAUy5
  zG32qZ6ccrIQb5nGU7Uwm8NW9S6yxUsDhKNfBdt3t60RLlgafKDJ4BaavgwLDPm5YAYmt6RW
  IaqB1UBWs/pCAQjqJ3Q9fBeqPODBuklDZxkSYfRg54EUpg/+O/brldqaaJWK2Y1YMsuY/xp1
  CL+Qiji5oIBHbGYZbcQQQxhUHU3iWJwtVUB6aKTjIt5Yu7+FJk7KAQfSpDjFb3jpDb6popsc
  KMMsvgkTM8JKxQapBj46SiIUVOSYHDEgR6jnGcY7DwN/REMRKfBjMBMiuv6DvTCtpYSNhISY
  gPc+h+VJW06BpQlHxEheOdKV6QphbRgGsRKBjo+gkEwOKYfAJNCxlBgwnct6maYh7vFnqpTI
  YsI3qHohTQSUKVP6CBnNmoHAihwUUs5Cia92RWQ773DWzoaSS1lR7U8jcI4E01jEZ1ITjoqC
  EQrVxUZQArCQaUBk6+DgFGTOspX+8yPP1Ph5quPtS2/nXCQH5DUxTB7Om3lSEtXe0FCgsA2Z
  vqDSjywntdOhdGQrReYaUvEcAJyPHjNtJpxyii3enC8ViQxQ65zBy7688WG9eSlP3nFRfsmU
  SzdtjDACM0aWMOWnNpnql3yoRpUCiauP6yitevlQF6XGTamxGvn2MEe1SuV5HtGK3NBS1Ec2
  Sl8X7abXTqlNnRYGDugErF8R9ySBmIge/9SVWPMCVnfy9WrqsU5hETtV8DS1qeLZK9So8LzB
  muue4xOfXMnz0rKO066uYk/6Kjs/pl5WkBD112gKqEWv5ewqAvEsKHiopcGY1JoYtSdg8KlI
  VQyRSZj+MW5Cl5dQtg7UMJMRHzRdJU6klbC6g0StO+/2AXR+UGrw7A3QqgpQyIj3Q9ONkXI7
  qVc8PBayjdpXKsyWP9ey9lrZ/NH3rvWIZ/n0XxfQk0g9VhmyOlJm9ssI1m7aIbntq1KOBLAp
  abY9CkeEvHLKi4cKatz0miu/Ls3XYmml00BdlLHaKAdmG9spa/a1pc3CsAxH/NVeRpiPWSxs
  3Grm3chFWML4hZ1u5YRdhHnYfF5V1KcE52LGNtOIve0DbzGZ3ks2ObaIc/GRs+ZiGZvWNHPM
  oic9dxopS/SuP8ayA6RJShu/V7TlfO1rtywnOV8WtfGRr18OcWed6s3Od3Zv4BoBLVl72BfQ
  +3VmhHvp3e/Sba7LIk9F0yxoixKaxdc4tJ0TTeYsMjrNRyozV4o86R8H8MtsWd8vy7no5964
  NK0b9WlK3WpXRljVWbyErnfN6177+tfADrawh03sYhv72MhOtrKXzexmO/vZM0gAADs=
}

set pieceImageData(Alpha,45) {
  R0lGODlhHAItAMIAAL+/vz8/PwAAAH9/f////////////////yH5BAEKAAcALAAAAAAcAi0A
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoJAzKA6PyGSoOFA6
  nylA0xUQWAPQpxSQ/Uhf1WtXuR13qq6BYEBQT81CN3yDbqmt1vf8J99fwnopagSEa34MUkYQ
  TFwoRVUBiocSgHZ4eZMKiYEMjIIDkJKZDmphlmxVnHNXAlgOkFYpl3iji5eWl6pwl64NYbEo
  s8AsAL8BjS3FrW0CyCp3hiXCU9CYB9XDRAKEag9rhbogj62iKspXzivneOmOeOEd0wrY1MIf
  2N65jqDk8CMAV/g1w8WmUKt2JYyZ6Jdq3poi0UqNu7eNmbeC0dz5IwH+UBhCE9jwHFuhUNqs
  evqu2fOAz8HJZxnVHSREQNlHER0D0CR0biOHjgCYDRyxjFk9mhG/EajTAWCkVg9SVbm5JKY5
  YeyuYk1pouMsqhrkqUwZ0guvqJfAUvQZYurOmlZJhAn6licktRlK0a3ZUIQVbsKQnjQIwqsq
  aHhZxkXxS+yJxlifjiQR8h3RrZixFrZMKitMtiCKvmVaohTPgnXnjoBUdynUEH9rMmFCc3YR
  utnilULbzQvESA0QI+J3229mdyIr65y6OvPr0M6jC4At5hUt3xKDe9ZEfEBiCkrf9k5YtCPq
  0Va+U2rWGnAvxe3j71yc9+HQBQAhgkYrNlT+38r7ReDcY7wAJsxy93kA2VbvdVCZdJwptp0m
  XNGB1RT+JeWYB6LtRFoIORHSWHymfOCWfE6Jw5587eWnYiENVgHOT3fRpMwxokk1k10HfTAg
  R7PoJNiBroGwIGYNbvAghPRhUE2MFWJw415T5rhGjTxO5mBFbzVp4Yos7pSTgmCethdPz/2U
  R5jipWdBGEnK6BoyTsFoXZoulWkjHl2m11pHFG0V4ARHShcUnhkcGYmiZjGJVWJw3rmdV9Ux
  EKmAeqL5F0CH+lkXoB44RSVzIF7BZmt3qNdJh4CdJ6KXFdx4qmtaTuDGbgyUl9GJ4ymQSh/W
  nflWnR5mumeSUmL+pmoElDpqqhrLOnShUBtusI6jtUrwZAOOLehLlAucaAyaCCJI17hwIeuk
  n+aJMNesrVWaQWxtukqYkUxCgkFRveZHE1Oi9XqAUh9SaG+9tYXXHrQSvtSWs1jZuQFmbCz5
  8MF14WrBShRyrFmuszzgr2sFGkXANm68WjLDah5jI6lNmQpvG21Op0HKO9FTr4q29TwbrL4I
  SRq0ezZBNJp6FIXoAXKySC+98gHNrLKNQtxMWifbXN80NTsM36lShxvyAgCO5fXAZ1mncpCv
  IiV0ZDNay+qr0ZL9jcIsFlOXNetmPFiXg3Y2a9gdyztIsQMLiZQejSE095/fHM1iwRb+NOuR
  hDotSWQuRt08bc4XqujofpS+9yDaGzbujU6WvwNXxewt+aqFiqe29AVrFCOsfEXYPijUoGOc
  m5Jchkk4d780ArxXwr7mVSQf4R3fMsDHJ/Cb0qm78TKFXlgN7L9TTO3HDUMYoBTJKwCxK897
  B0HkmcF4spCZwa4BID4zIe+8t83Kqd9aw13xgtee4XlucIFDnSGkx5o2mS2AFzkVtKS3MAhW
  oHsHiofVxgc7C4JHfJoTHZMSqA+rPXAC8APh1TooKItc4FrZixbWwkSsmnludyb71PEg0DTj
  JTA/sLDX/9oUigSh5VTMA5v2RIbByMgQYp1y3R0818IlJdD+Us76IS+sprr1GIRBX/ziViSm
  m4QZxzS1mQ0A0Pc4oWhgZMPyTmtgpsHBeRB7cDHWwYAyPwpIzmnVa9EO8zQ26GjuK+NL4RuR
  JJBHQcdZZ7Aag/xYEQwGhTWtE8nrvra4trwKFQzq3fTuSChjCXKJuKvd5EhJAX/pDV4uY5kE
  arhKVZLIiFs72xnWRrGghKSDX3IWKrcHSTVJ8nITcGX9TKY5X+Kyb/O5YgNk1ReHuAyQcrPl
  KP2CMVSxkgI9nBkZS4lDv3VTTLe7QNlABBldwHBNU1wkxIxTTA40EULD9FXs4gcQMXItRRxS
  JeWsBacEdYRNoFrkMRrJC+/QkUP+53TgliLKu2eKrI0IC9NDw7IVnHSROxBR4TenRg7rqYY6
  9STeMSc0y6J0rxD+LNlGk+UqWYoOShR1ozaSM5vGSJNbOQWMj7QJtpGuiqgEXOVPM2lRlb7n
  ndOKJ/EUZ5t/DRJTKZ3YSg0YATQWChWJ1KRrfhrOcS4BGXdAqg0jWc50kYCCFaSIODuJgRNZ
  D64Jy2cpJ0kZtI5Qp/ZsBZLSqdWsUjGUGAyQuKbhS9kQiRl6nZo2B7qZNUmQq9vDJk7w6re6
  GSxnIYnEeWyKvTMlgmJyPFZlMYMCplYRsxVg0j+s5tkDlC61f8LGssQlypxlDBTkwhcOAWqS
  tlbQqAL+iuj1JjorYHEgd9gQLUOhi1zreIcePTuJFAj7B+dEdl/OimJAVzmCQwrKR9LbxO74
  EVgcMkG6kPuuLbrp3IcZ97jMrehP7Sbd/PX0qiB7R1u3e4vABuS+bNzfTp2z328564v5vWuD
  VXJfdP40NgmODGpgW0o2EDgz0KPVZtQ6p/JitKLVlVR05AsBqK64tjjLWAWlGUjZlJPDlWNS
  bbUDxayl2MHZY+cAjZet+jzoyABWHyMxqCISf9I4cwXszfS3hnfkgQmVDWrCmkrJ3U1JhzS+
  ZrFkJibKqjOL9Myijx1EZe+VQ24r9WwpDhSUd1ZzYgFYEJUOZOYLnhg9LO7+jJbXSiY28Elj
  JtrRzG6009o1Cz1JRp2r0LXlH6s4XyJU83XIxFd8bZW7Xd1G6frEjpPRmIWeyqMULY3FQUNW
  rlGWqq26x7pS03o/f9Yoq5lGZquKWWWBllZ6cVvpK94TSYFaH8ksMN2VAkeAn961jCxXZ8gc
  KtgqQXVAFgS+9s6VuHiOspRbXCjRzo5fzVZwcsUtVK1WeFg4DrWrW2XgY2JbgRALaSuPHQtM
  E/PTz82zMPusrSrnG8C5viVExQ0UcqKIOaJCEahty1k2kXZj83Yh8cSNcIBr8JjtWg+EuODa
  IMUq2m4KS3/92zOCy5vlLEd4xtv93JkbBB5w/FSrqvCTh7ZeXNDs7lx9OE7CIdvxuR5HurO9
  VPK08HzklET5oA4nThdpleO7jjaHUF7wF0/z2Bsx7wg56mylr9Tsx9y61kPtdUSAPerRDl/Z
  C3t2JcH87tLEu95nefebbMK/avm73gE/5cHnjwiGP7zdE98zljBe8S3uO7Mkn8zH529ZlrcN
  4jP/5lp4/vOgD73oR0/60pv+9KhPvepXz/rWu/71sI+97GfP+gQAADs=
}

set pieceImageData(Alpha,50) {
  R0lGODlhWAIyAMIAAL+/vz8/PwAAAH9/f////////////////yH5BAEKAAcALAAAAABYAjIA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWwe
  ATNAQCAIQJ3YLOsak1Kt2rB4FBjMqGjB2Eldg8rntNq9bNM5AzssPyAQ+HdJeoEYeXN7cgJm
  hEaDjBQAaIsvfH6AjwqGARJTii0DoFSgk5gQkaIxhmmkjJqcqCyjoqClEaenXbNUXCWjvgsA
  vqBcwrQheWVVEFMDzCyJjrXAcrlyvCTFk8HCxMUiqsoPnbAq0IcvXmibNJ0Af54wp7sp0Avg
  5PUhVZYC1weRfQhMiRVKkbF448LNSEjOhbx+9BLZSzQpH4hE/h5GMyH+yyCrLZJU+Vtxyoqf
  kiNRgPs4YlazQ8hAOVNQpaMIRfw+Vnr3YmMLjWhSpgCqTijHVRGpZaK4wOIHpg3unVvh81k/
  P/ymqkgXEGu6dSumWPECtsQUrHkmBrx0AKefqhmYGYKgqyxVrS7MwS2RTq9CFQz/kgg8UU5F
  iSEY0k1jtxzeWFex5vz5RbJleUY/nOr6DmIvAWgpZl2FFR6IvilVgcn7eIVUpCD91pyHgmjQ
  o7JzN/SAGoLqzCT2oqBiuXTrEYbcFbfc7kTy4ps/hzY8WlJplngk+daOfNQDQyx/mcgNWXcA
  gZ4/y8be4bVuqMe4O1DFfoP4qLszeR/PWXL+2q27+CHWcgLSRkYVykEnFjagEehgcaaBkAcu
  D+wSYQd9yXdAQmVJtdpNspFkDQG2nRcJcHG915gH7r0HW3xoLJbfBhk2xCEDHqJ4gVvF/YdC
  OwUGSGBzIgz4IHorZtffkT1e6AFxbKlFgHAUaOLLbUYuKM+VM3JAXgoMJRhYZSce6GKS9rm4
  XksvFhbjB1Z2BFGWymwpZ30ZELcclTR+odw2wzxI5AdGMrlgfCYxuZwXeDawzWJvaaWnT8NQ
  AKRlmuyDlViiQPcFiH7VlkgZo4zZ6RQ6TmCqX2hm0KKaTkrwaIVKMYDYfAMAd6l/X+wDgDuc
  8ujVp/ElOGyjG3T+sqSiJBLLoqaYLjvTG0Iye5KzFSR0DUCTSfnOtm06UKinekoWFIFkwWgO
  shqs+t5brVrQYkHQsGsprNDE24C2jsJ3gFTgaigutNsMS9wtUwq5zZ+HXiTsXHz5ae1yyvJW
  rbkNXmagZl9k4/FL6VUQiRXI4NfVIOVGuaEiXqR0ypEIG2zsniF3EGrE+Iqmcga2TapXqhPY
  +bEwt12gkV2v2ervymm4XG5CAd0itZDgnBcp0BGAl5W9Fig7sYNvcsAMgeViKpjFOWM9YYH4
  JXjJTs2yoilAy0T2IDKSjQ3z2WnWa1bO+TZ73L16RWqOvkaXzeTGFUiFH1SvyW1Y3e7+EGU1
  3qCgF9CY1yIe9MW3iIDS1wI2WTMGindGmmV8WuDS0Da56pYjittR+1R67vwPtEei2jmzA3H8
  s+iASyLH1TbXa/qt+mSsaOtuhk2TOdM71dbkDowc5KgF/qmpqVYHj+HF156e5nkBWD2x8+Vz
  jaOw1XVqtvsSpP4g9AvcqPT8cLPNgP7zgZ+DtIeemTlId4mTDda2czB8JQwNBaLfUpRnNr+B
  ClbJYkwD9EIv5jENW+/rQ8+uQkDfjdAdCGwcgtA1rfbghHeKygPNPBfAZUXqgLHykvoUJT7Z
  FUUB3GIdBM12hQyxp3+9Aw0BY5hDFepGgo8TIaz6cLx30PD+cYV74LrYhK/s/HB3aQMi1LKG
  k1eFT4nf80sfUqgqGFKMb4VgGemmJC38eYtmOIRihWxIIB/16TZI7MySAPGQzAQyj4fsYxPb
  eCY4FQ+NpIGYBl7VoDVdUE1omwclOQhGxtVQdeaonEFIuJ53SNBdh3vWuHjIvukoCWytDI0e
  HSDAuw3uXpo4Cx73BLKSTSCRZOvExNgYAVTm61mAWyN1JOnDLFqyeV30AIeMuS4AklGZrIKX
  nlCpTPployal8sh9JskYZjFKkYu0hRuNA8sFboePyyHm56wwjj7uEGMI8mT2aumgTtwziQv8
  Zr284UV8WU1nXfqcXwwXym8kM5P+j4QGcIKYm60JbqHf0gdn/CihfRgpGBxaCQ7TWUy7WWtk
  yOGnIme5MhSu8yRDKiMcaRnLkcITQrd0nfUcib4zbc864BHbI6/4y4dKM6LHrEDuSokwSgbE
  jt95GEkvkKsC9fQLDMnckRKqUNLp85XWkuc8T2JAJlU1YToCpunKulJ1Ye8YyhlT+lYRV+UF
  9Y+AcycD8cXSEeZVXrOTzQ21aI5uORJCLN2nRwUTs8VNdTv/FNRjAXvT+VGLrcOcKa0wi9OT
  QjWqFhwBUFrVIpdONoCj8sWYEvu+nCWWmo3cUYP8KqCD+kWUOVVqKz9rC2tOr7JbOyqp1Jgu
  BnGWl6f+eenEUBXHyJKtpm9k7fWoR4IMNUZOWbyrUA1yN1cwyLXJRWrRAHs5Nf4Bm+uKYAiC
  yCu9ymq8ClilLU8rDjmQSib2PYpXWWu/r9lRb44FnmYxcEITYGZfLgruUbsCqLNaETd8haZ4
  2+WWVa3RtIFDUpFM2jmVdGhXZuWqyOSHTveW1KtEfeccZamBxt4PupIJXWJY5WEcwXap9FXs
  q5Jjghvnt6PipV/MTLXG+KmjfCaeRk15iwF/HreCiVXppnKbJxg/d71u5JIvisPcFpOPRPi1
  b6A0Zj686qUFPvZZjmn6nhM8EstitmGYm/bHunLXPz3SBJI3LOcBOyxRmV3+MwPYi8Mk08qr
  VJ6AnoLxnrPyNmYgU1HUvppB2aS4yg58YF/BObQuB6d4oCrDMMcxPg73iEkyTq5zuwcm5R6J
  mWC1pXT3599ES4APjNGqfyKtCEEDsVchYes2GENpL+vG0CXNmeE6WlmxVhl2AnXoZTo4i5n5
  GozGmnOcY1xsAn+Z22Xm6Yo74zBoF2Q8oFaXg2FGH+GJ+msgQ/avXSRvLGZael4S59CYDKla
  O3SNsBW1s9U5yinyGM5PTjXxTB1Wfm9yaWi7r7nDbGhg6rpJiZUhur79lln7VaI4A9yykakm
  j3ds4uEMN1XX41QgH+6nEPfik9vHIOD2k98bBI3+1nAOWpubjqXIMKA8AO2VFub7n9Y1YCcc
  Ct6UFm/kl6zVd5EK9CoEJlE1svq1pzdaY2WdQoTi+Mb9LNtxB+lA2FTmpV/BcM+SnYyVKTpj
  jDWOKFsH3OQDx4yLt3ZaFc8d+D6qXvpeN/GCUHZSLEqMtWVKh8Vvh1mNFLVWnUTCT8Pn6DQa
  JW27zWeW3exTJrkBtUbmrU+X6GhZVl+oFVHLz5vvD9S84WOu6NkzuRKBWQv3Bh4BnGS94PZ1
  qbjHbfQ0YT6eIn79ujY3yrVQEmhSzqwqV1x82aH+pNVfeZBd/nftvrfkD6ezbG3/rI+HcuvI
  SLM6sq9T0Ceso+73vinwxD7ldQB4gN2+vPv54d7+Msvh0fdq10Z+joRUCdMo5qdBH/QeOpKA
  atJ/+oZy56ZKElgM7KdUlCdg8Od+hxdAx0UWRSQxskZZ+9d4fXJ8RPQsHDhLBEhyQ9UwcKci
  IVhygGV7gkZopDMhKgh6Azh73DdhtaciA+MiNWh7UBR+ucGCPpg8hveDSEUBatIYPlaEs3eE
  hqeETciEQOiC21c/T+Qy4UeFV5hvWUiGW7gBLehCRhiE6wGGbeZEa4iGS2iGTygNdniHeJiH
  eriHfNiHfviHgBiIgjiIhFiIhniIiJiIiriIjNiIjviIkBiJPJAAADs=
}

set pieceImageData(Alpha,55) {
  R0lGODlhlAI3AMIAAH9/fwAAAL+/vz8/P////////////////yH5BAEKAAcALAAAAACUAjcA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CoVACoCqTYrPZDtW6/YNfgahMEzmdyWFsFrEvjmxkdUL+l7bvIPLid22d6WGiCIHx+dIGF
  UISLHHMBNgBnBJVpjhAAdhFVNY2YGZCSiZegDZoVnTSfphYDaG40c5UEpa2QdRG4myuTpLGt
  E69/NbhovI67unTIKb6JwDfROQKvY4o01gDWLG3ewF3fat95JH++EefYLc900zXvcsOsMvNn
  fb3k4Prj+iWk6RLBaAdLGrQczwQQeBXvxTx8vUhFUkCQmAKJEENMAlD+aeODjR0DNDzRBRaq
  Ue4QShzZ4iE7iQsqilyAkUTFeDdflDzXrAWudj1ZVPtDa04cGUPvBRUxaQCgiQeatmFIU2QV
  qiNAEvDoQCtXGPRmGGNWzF6iozGSBkDrDCbFlVUPZv31kW6MsDLu0SIwNCOMYWP2bkXjt5ve
  YSw1iqSFTetgYJRCJs7QlMorZHw0Xc67bpXEe9k+n40hc/IHmTHhXrTLlBRm15yh0mgquBJW
  nbBq00roc5jCSkZVOD63bXFIp1UiP4ZDp/ACe85Z4B0o2iLu6oRtqVBLKnrrX1cxPpU7AnoE
  87ER/RY8B4ZR3ewBr3hf2zeK4b92S9RvOsP+vJHPeLfCdC9g19l8BhLWF4IGLtWBTAmK1h8G
  /3FCmGeyzaBcbQemMAdHtoEIn3z3EQWfLw5u4Bg/RenDnwlpdLhaeyRQQZ4CxmxC0EkwGngd
  MxCuNZiAp0WYiwlBGnmjCAJB0OQINqbEQI6nQJPiBhsKJiOSa4EIiYi6kchcHfAVBVoJjpWp
  5ov/BLDQTF0JueUGZpF1gFHFHTnWhT1i50JFvwU5AF9rnVAndkSqqGSCE16w5Ft0kHBoKXja
  d6doiWLpJnxz7gEYcImsKaYIvq0JaqE2CWmqqbe1NiiNDaTxZqPLmAicfHrZFghg6w2mXQgJ
  CmVWYIxJSGimFC7+euZcylZHKwUVPcCaB1/ulVQfub4ZCa978WbOprp1CkJCq3L6qwcolkvo
  kXtkq65gll5gI48MaNVqVKp+NeVGDtJXGy6D7mVPrwKf60Gwwq30DXYKbfZds1dasGezx4TC
  b11L5uTAvP26a21zhDbMzIgGP6gqvHCS9Om7tT3TLpkcZnlqxBVAQnC5NoeSCC8MnRorzE1t
  vLME/uoGSbiylhnvuNg9K++kiwYsrsTVFfsZzRRMDLHOdjJw6M+jNUBlrQFvtY0mp26KhiVJ
  W1YFLUuDYM16C5a41s0s65qyB9aYCy7KyCbbpT6E5xk4A7Ie8nPBwNBWrAOXrStB32v+zkFw
  uqKiSiqmbVG833JyV832Z4e7ck/hhI/qaNiIq4avW14DdJ6q3MFs+ZuDYk4Qsdx8W/YwKCQF
  Zt7sESivzFaXWTK6njt9gHIHbqj4AZE/vjjbteKcZdKmLq8p7CXU7nlu6HCBndVXczm8qfpi
  AP5q3T33mQPvTxnZxL/NUgvbv036m/cYKJq2sAYtpRCvI2ECIAV6JpjStExzc0EO6r5xr9VZ
  LzVl09UV+ICyaUAvQ7Fbn9KMgzt1ta8DWlMgZcYHC2jw7wOTEiGESpcKEpbrhBZ4X5A2KCGw
  RQoCDIRa/xYjsoBpLWA41MDt/qXC1XGECgf8m372pindODD+ZupjWRIL6BTrvGV4h2Agm1Ij
  kgpW5V36qwXMyjW1rCWIgBJwR5Kqdha2OS8u+Xng/NDUPA7sqEqko14P6/WoM/pKNApJY+IY
  hr0irZGJVOSb2h6pPb9RC3k7sQremniBNK1qiwtknSDX9weZTU+QzYmjFFm1GMepq40SmKOU
  QoAYvSlpUKQICQgDiKm3dSRPEoFjLPvIgWHFTkLVyYgxA6JLTOmKI46DmolgCUTuwSc4inki
  3liVwb0Yr4Cr9CbyinXHTNjwk5HUwEPIgLxniFARasnUOMvEQWuuipoPkGZNFPMHfV5NjXTQ
  JfPON7pBMqtZ5VTAOuFHMRy5hAL+kZFlIuVEphROc5cX8Oc+i3Q24plBhL66oy/UtDblsas1
  IC0TKKm2FmKMs6RYJEiK5mnSNL4So6ZTFg0x5rlX5TKkJiMoQBHJR4ol9FIpYeHrKgbRTckS
  RJdRlT9BhM8FWBRRwiSjPU1l0wZ6UQM0hSnSekS8quazSyCz4jmn2E8IqjJv7sybWZFKMbYo
  anwKqYguy2kg9O3xW56jJQsNhKyIJqiZVDVQYr+lJZwOtFQN9MYvIavWdGZ0rck7kVuBFc7u
  OdZ9dRCe8lSqFFgVMKWa3eqaVpoB0ZzAFxp1zbpCBdQO6NOv8WtTYDk7WNdaQCuMGl0zRcda
  93XzFa/+rWji/Ems1Jazq+qyHFnl+tlOSpWS2guYGSEAXc92lqQnBYFv+ThRIwVmYnu95GCz
  GpDdMq23rnMjEgmrK+0q9oUo9epRN+bQu31pKse4aUIpx7Letams1V1d/g7IF5+BFrVI+65m
  d2qhz+xXbP0FT3ickj/RpFeS46PwBJTKJPjKTsG29PBjCkq646RqeMXdwNGkaz/MOle8J9uN
  ZFGW4CpSdy4Qftd2syfkHN8zvI6Mb42CGUtpfvhgCltY/Urs3vea+Ktu3NRVn0gmieK3xGNV
  2T3ORWB0WpaXg2tantj71h+/jMExI2CZ2WdjkonYSdUJnll4MR6C+uKoX9r+ZouY2qeGvuzK
  SA6lEZtmieEGs75L7uaXTXAtyKmWtGdG81kADBuS0HSEbA4hnPWz0yWiM8hTDHWNiarnZclP
  SU9Wb6/CMWtOcs1zqlYookttIos2eqiP9lWudS3DzZJ3SiszIZb92KXVDplpqH7ghfElaeI9
  m2iXlrb2Eg1iziXMDuJDlN4u/GeodTHTSTbqQeHrvCWmEIHA3tnMlvzdudqWMHBdtqJQHeN9
  E6/fFRhnOLyxTXujUtDe0EYb6OnqEFy1x+VB6OjIvahpx2l8FiedFdgTnilf1naE2fhurNVx
  kBtbktWGtBj+IGg1/Tm/dLZ4laKt32+tJzmKFVH+0EBMN2ASFm2nunM1CesQo07cVRMMj8wJ
  iXHAsjy67RA6ZAWAcHp2+OQopCmNV0DZfKP7si03k9CHSfNufR0DG/H54Ci4aYAPPUpyLNwv
  sEnvN7akeUcXLL+xbmV1R/BmA28D4K9dga7reCrfMFrDOQvh8qnA8FrUN1gl3Ni2SDDpCT+7
  dZsTZMsUUp1WgrooR7AoFsTWwnnH8U3bUvYxZnPQiAI66GCYNLUn00uLD13KAbedbJcLePys
  uJhDPOznNRuuwBdsc/8l+38ZWFJR6z3GR/cwJS39TmJ8V2XMMQZZrsS0tG+pxs2de2rNWU1z
  Q4GtGPzN1TWL0qfDvD7+4gZll9dZjUDGdJ3dnrXTN6f4rTN9AEVlygJ/8Sd/nMZt32MssrRu
  K8FWFkZvu0cyCph1lMdGBsddkVFStuYfHpM3qtMBBlQ8IxMfGch01dYdCaR5LIVrBhhi1Bc+
  DJMC/pcgYzcjdXJc3QF+zINztPUx4MGCoWRkv1d+JtN6lkRvXCYr9CdY2OVRRgha63dIU9gO
  pFeCBfOBRnGCr8ZCN8h0LBSDUGJ3JlCDEXKDhuUOXqWG+FdiQ+RCaYNvkuOEYacmdMdZowY3
  kvcA4VZHs3JdWKUpdRhdJwgJ1dYXeDMPucaGzLdNVuiEvfWFDDV9BfE0FRc1cIRoP3RJS3j+
  Fol0KCKziE41fqjXhubzhGjUgW2Why+XU7fnM7U3VZOXh5XnSIN4TXv4YKPmeOplYsNmYrVV
  eO+HUJOniRBnIbZnXvyHZ97nLFyYfQy2c4ZAhAcEW781hbiYC6ZmfxUYJxNobd14WrsohBby
  jTdEjmd1ZV/4cGd4h8OkLDwEj2hnjKpYJZeHgJm3X1KBj6mDjvRzgav3erQoCgGHijwGEclm
  h9TkSazoj61Di1zIkP/mkAF4ZdU3WAOQfCOmU12oJPNIjwn1aRgokON4VMTBj/rAhdSGknKn
  QB8yQRUChveYkuGIQSyJOoSHbVZxk08RavvIk0qHY4h2kRjpksqnMg3N2DXgZIzPdZI8WY/2
  CJQAZpKaSJTstpQ2iBn+NyFJeZTdBl9jZ4xCaZHBh2iT0ZV2lWERwpX0SJGoZGJhWZWnIZX+
  4HB0WThZg4CpIH8RE3h3KXjo8pfeMC6CWQ6BWZgX5peCmZfyt5eY15eISXCHWZiEiZjBcJmY
  mZmauZmc2Zme+ZmgGZqiOZqkWZqmeZqomZqquZqs2Zqu+ZqwGZuyOZtBkAAAOw==
}

set pieceImageData(Alpha,60) {
  R0lGODlh0AI8AMIAAL+/v39/fwAAAD8/P////////////////yH5BAEKAAcALAAAAADQAjwA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodEqtigIBq3bLhWG74HAP0BsIzoKBeM2WkXlmtLpNr58AgncugEZn7V18An+AInh6OIJ9
  hIVWgoyNHmdzOWcABASCkVqPmyCTO319nlWdpBxxeTpnmJhnpw9YAYgQALI1tgFxA1i0sBOp
  vjSiaL8NssINuZAwubu9xhWCZszDeZh4AtEKqaoR2aAziqLV2wzj5TLEr9vdyQrgaTTji2PQ
  QHGz7Dj50y/Lsr4sCBjwHMF7JczwEpQsGxZq4tYNsjcrSLd9Ny7KcwH+sODAg4xAVjShsSGx
  dCro+dmhEuUMcANcbUxU74VKjBIZSHTJgRUmPuWAtsL47+EkhDTJjZFI6QY6LzkXRFWw88S6
  oOteGk2D1IbGcDmCtVL07kW3piz4LNSFc5CstmshmvCZaWKsQUO11TC14+tMfhLRlHVxFmpW
  qYepXrV6EsJNG3zDWmpV9+88P5R/yhmcAhwAzy6EatarmHLbAGPtljiDGoBcB2ovadrLGk5g
  yxFvb6YRrw/nK1MPBK86t/FdYpBr62CYmYBDXLsuNceUiifJSZTztRBdOSBdAqzf4tVsXUNv
  tA0u/k4R2alu1W78ihqgPYZ8+Ox3ij+5X6n+ifMSqJccfjawNl1lvGF2YCsw/ROHdJT5s4Ja
  qa1j2lWtvFZCPCiNs15+BNbwHlH/jLiZQg6+h54JN5moW3nm1eSYb+6F+NJ3zZHYQjYDSGfL
  gtRN8mEHnh0IGogVEnPhSXnBqAFruOk0iI6GPMSLMPEIY2VX/5l443wtYhcbC73pNqQGYbpI
  HGNgPSBKZ1tiSSNsuix0Jpo4ZqZciWkwiB2QwajAI5BBenOCIIQmqqeNICBqSS2s7DmCa2se
  oFFTLfYI4ntOhlBmTM7dFlM2d2JQ5nulWpCmmsaxKIpJMk7qFyOXnsNUqtLkeWGnH6SS5DWA
  trnhn4k+iEI2iib+Cx6jHwilFgTU1MUrMHJ4F840hZJBaXhbRRnCqiuZRQyohTIV6oqGsCrY
  seqaOG2ufaB70bsRpLKWInNgS4Cx27rVLboh6JoXn31ic1CxwqZbcLLGMgahskZSGUK0y/qC
  bF0AbxBdc4roW2jHzW2bsSQjEhxvZqf6du4d7a7LZsuBpVCSA5+SsHFmHRO7L5QGorwLCpNN
  lw1hgkEsdLiT6szwo9ctbHR2CVfQbzJBP5veeEM/ANNvDy4YD7kZbuZ11J+U3MJ9rAZJL6Qw
  M51Q20xtMDUE66D3FdtpcB300XJQB6o7Y3vb7N5+Mquw009H6LIISleI2nQoHlp04g3+bkCM
  xd+5fcDFMmH+JrXABivAgY2HTLYH4K5tQcowt6Z52YE9nubIHYCrJu10v0qzRIik7PkooEMI
  ctijdycTubYoGKrgHuSjOPMiiJU46afjGTrOPZtuOJF+iOQ9FotnQOHKx3wXOTdOS4q+dBru
  fj2QjceRLKlNx70C2u3+DT0Gq8r+IrtH+R5IArUBlaAnUwyYlQM0ghJSYWMdEMKOA+X3wPn4
  6HWMCxo4VgCT6WFjOooYQfY4NkLskeZlLUPdeJYFmxUu7w0OTA0z6AKUACEOSDGs4MMIdT4S
  3IaDcLvNaET4otH8MAUlVNRsOPCY8vEOHrFzwGLqVTDWXfD+GvR5YQUhGCTcmSp+GIze5Dz4
  uBypj4kCkwj1VCeBAAqQIHFQ4eOyJqUydhF9YENQA6rGqBwqioLHgxgdDxcYXIEuiFMix/LY
  eIA0Ua9Sq3FhspZYwMQgxn73kWKr3Me+23RydJ9Z1hxFVUESDApyYSzbqOz4NF6AsHoXoJgZ
  lfRKL1queE+TmNTCc0bCvdCPy2uhW1LZSIHBbzyIMtoZuTciQ0ZgEbYrJD0eB0sL3OeRdUMi
  3DxQphl6knUkUsk7kgmuJo0mL7eh5vZWV7oKOdNNxZNlLne4xYBJcixs4QorK6TN6elyl7pb
  AOegRp8bVsxWAZ3RPhM1PnlOcp3+ADXROzVZsGiOq5iLKJQ9YzcLg61Km237QJl6p5uCehKh
  c1JouWKnGf35r4jSmpgv+WZLC/hEF2TE5SzfaczRLPRC/UzcP6UGPjkMRFdZQiUhusGl8v2U
  UJYYqLJqCAL8XdSHRwHn/+Y1Gp5yyohHRGG7QJCLeB0VbkuF5mBEY6K8fIYubY1pr9wowJ8R
  0Tn0VBYe8gg1iLbxfSZ8qh4Z488TzpVG3MnMLg5UQwDlSrALosb4jEbV5rWsphRAJDvG5VPY
  pROsMQPaNsWI2CBmwbEUYOuIHifZisa1snKzamA0Zc+uJc4Mee3sB5L4PMFSkgTJzKWhVLkL
  3mrxlWn+mEQcVXVPJSa3uQhbW7/yN1HhIFJbFuqOZ1laGd2gwKKFJGK8NKuRC4gmmg+0rbvU
  hjqQcEs8IsmgygSpECBJyLJ87W2wGHmcwvIXozsbY3OyuKCTLZNthe3pMQ0pknSCRL5tm8NX
  RoPZB+Dvpen8rmlHAN62wehiWp1cH0BbyM4Vx4wqEJJtf5JPriSvb/YthkhnyqBeBK66wpxe
  +3aLF3DgMLcyucZvMwtYRY34aYMUryWL0yPZzkcBLTIiWVXUrfBKDq1Y1WxYbQosq4LqESud
  bSlPvKgUUwINYd4JkJMk0/wqq4clSCzEYNsouhBYx+Q6MBXdTCgwt9KvGNj+8lzOfDsYBkbK
  VSXvploHXC1HEQP1Xd57uhpkM7E3zjfccZei2r2ALHaqMp4xdBE2XExDdkF0LlsZ15wo4Q2V
  k0/zsyCJWbsMA/Eq8V2AfL4cam5qFsfC9HAkHY2cDMTQquokcd106Kp9phrTnI4Smufca1+/
  ryMdpQz98nNqxgIaXh4s8LfxGGvjHrPC8IKkq676jf4hesrjEslXgB1sRjea2LGK5cIu7Lc0
  c/aOxaHnkDdkLx2t+KGvtmZ48LeQag6u26/8L2JYTdmEu4/PEdPp0gyr5ODcobx0cjKGR/CM
  Sdp10S2TuGLwbXFl/CnExcNCpUscYJZp3K1kIiD+A05J7ZZTQD5FzeYKzN1nn1cg0uEmaK2L
  fEyjbXtYZsp5+KDIKtZW+7AC3yePWKDZ+hEb3VDW4KQzoU5PMvs6boZzZ3BDlnIbHXTZZux9
  UzzqonO8zjcPN61/TmNxz3p/qFARR3Dj5Iwu6+027FGaZqF2saYQ2vg2pG21+hN08q6CYCdy
  bpPMwQTOt+KIj4CcUT3uW1I844yBbEDsnmim17ioyCDU1kX7nmZ4PuWVhryLVJ67IDoemg0W
  tMbEHi9ZjCVCVQbW7NG+3xlUJ6ehzx3ErTb0vCsr+lRkZdA5ij02FknbLS5p3OuJAtny/uge
  zn2Wb9eCrnvdxfMLf+n+Xa7BKzHWa7q44t091fezi8v1CKdy/Ycy2Mdlerd/gxNzhXcUMsFI
  RTJdxScLftEjkkZvAqUmFnhIuHd4Alh3z9N+iHRlrkRG9SFqAPgTC9FqU0dyBqVYgMeCJ/hH
  elZJ1kc9GbhHNWhkCGht8zJ+4EcPy8V/deODQiN/KwiDJpJ5w4dWQSZxwaWDN1hHb+Q9Bag1
  eZJ81bJPVUh112MLDDd+R9I0FLd8E0J01zeDGqAQUygeShhRcJOBxXd6NdYNplQdchgy+BKF
  C/iCeMeEHGgz7/VGDmdtBwiHxeOFt5NtWwgPD2KESagPeVN+A6htg8hjGAcxSANvLROFMOf+
  IhYoJD+GQ0F4V6JIgcekAntYiSRjWt31X534RACUg4lyhDzmiHJwH7ywiJckKp9hVTIziQTI
  h2gki5iIhqZCEEdRLf2ADBoGjDiUifYkWJ/mbTtIg0B2FjcmiTDThqmlWajRMZOCjC8SEDfY
  Yas1bGYnacKXgPLxMKzTMKYkYPNDi9GYdEtyfpszGSP2dL/Ygn8mjESWNyT0KiGTCpC3UJeC
  XPMnAa8YdbTHiuBocyd1PywHkBPAGhB4T5fyYpgWEzdBT6eCGpqmQgJJOfAYj8RYjAvZbn9y
  ZPkQhcpjj91FRAfnHOoRjFeXgNOGk6a4KNDYh9clkUEEKj9JVtH+NBK/52hE9Eli01cwxnkP
  Z3lNmZG49GzMdIkyWGrf4oHKZIyw4WUthVv+Fm8d4IxOt4iGN5A/NQ7o2JMc81T9wo07F3nr
  9mt/KD57aFKJWJYVmZM9EXM7oWzjoYvW1RpAFyYUSH3whpUbR28OlXRzh35U1iTBZYtPdksy
  OR1byHNGwmqjWFt36DWfaW10CYhaFhMnp3Dtsn3sd0t9WY0XQColhTZQSTKW6SK5CJvm5Y+3
  tZLwFJoM5ZfQYoZKZym8aUIV9oSZ2YBbyZgyKHEK1pUcVpEd6GiQqJu6dlnokz/W85qMpBZr
  +EYj+ZfhOYVoyZU9F0nLuUXrAUykxzT+36eCWjkByrmcXsl3EBec/1WfhQhh+IaOWrYt7Qkz
  hraJ4vOa90kt+Vl0+xmdxVhdDiqdWLWeTahwp4aImgMTREhCnTJZFGqVGRChybKIHmqP4xmi
  fQmg+FYeqThcDZlv3YigjCSiOjidzrk0/PWaKvp1XMYq8jJW/IOgvolRKwocLJejKbp+LOdF
  3GlhQLqbMgqU1mmkRbpRLLejxPYbiVgWGOolkCakbNSigmelPEqmxIaljqaluImUWnObxaZv
  UXpY+CaX1rWk5nCneJqnerqnfNqnfvqngBqogjqohFqohnqoiJqoirqojNqojvqokBqpkjqp
  lFqplnqpmHoDBAkAADs=
}

set pieceImageData(Alpha,65) {
  R0lGODlhDANBAMIAAL+/v39/fwAAAD8/P////////////////yH5BAEKAAcALAAAAAAMA0EA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodEqtWq/YrHbL7Xq/4PAvICgHxOi0ekM2r99wGEAQHJTvg7g6wNen5nV3ZXl+YnxnhSZ2
  AGNmfG6JX3eRJYuNgpCUXJOaIXYChD1mBKRtnZtlpyCfoTyYgqpanLEbc3eMogIBpASmtAy2
  uhVtArg2AId3h8a/FMHFPq+zzcGIE8TMNMiPjgHZzRC2rD52A7zl4AfEgxOfyjfrmNbp4Zit
  Ou6C97Hx+w35wmzEEzSvxzZvRbD54lHtloyBmQ5IS7UAYsATtna1+bb+IGOvMhxfWLzo42DI
  HhBLSjuZwuK8iXQqTixI4pnDeq9YshhJMwfEnjoAuANAip1BTDpRuGQAk8FSpbp4kQGqLmop
  kjIOKuOTdMbPIEIn+rths6uJpwqaypRGFcRXCG+1JetmFgZAfZdA8foYEUe8ti27BbDzEtSh
  T04NzwXsYapUrIl3XWXsYho5mGPhwXy346/XmUwnJmYb+BXVuDcs82BFlFRYUAwFtd7rbkDd
  P6wGG4XRZu/U0JL5Rt7bl0RvUoQjkDFXFDIN1Tvu2ou9Wd9tFK+nx0ArUfRa06XlKSedA7pP
  kHt52bqewl3w9ASyU/6j7DFI3mV8kxx1Nab+Ase8FEeCNBMQqIN5flV3Xw7SSePeMdWx1xho
  C6j1H4UoGBiBhjggmJpe8CFHEYSDzBaia3hIuEqJ6b2W2Vn52WeajPIQ59wI2d34HSgqnuCh
  DQr+2IJNCg7Qhm0zNKhdCyMFGeR8HOQI2DpIHjgidSa2uKA29Z0Y4lAxsOLlOEPG2N+MZ9YY
  IDQqHAmbBOUIaMJBEhDT1jJyOMkml3riwVePHCj5yos19WnollDh0Q5BQx5SJ2dwcYWfAF42
  B+UJn8xWTqUiIqpCMFnCB2oLZsZ3yKneqIfqMjbuxKOnwICUHG6DfgPQPrrZA+gFTYpXw127
  EGlPfLBid+ieKhz+26erskVg06Ua5FiiA7c6kKt1TJYaopwszLHIOV1WCuYKoHKKYrElaGvu
  ugRcmWFU6EDwiXAYCVrlAbmReYC0uq7QK6MzCDtbr+as162yyL6LsIIsDBrBXdjZa0y+u/GL
  bZvqpsetCu6BK5u54xqLHrujJmsVu+sulMIoGz3gkco44vHetYyI2WkewRjpW8gn/AupXStp
  HCGxCZcgbJ+7UrBwkA3n5DKHMQ8yM5g2EzBOzlPfJHKltmSVoqqohrqtkB2UizLRRQ848tle
  BuPBNhLYsqY/8zbHkqQXDHVizoOkx8ote2s9p55eO7iqoDF2LfLSSUuwNMNl9+Gsgw/+SHc3
  QnkDHiLfIEr299paoisCYWO++ULHbAe+W7qg+8aH6oR+YGfqX/6cQT5AHfdnAx5ZChdeFYgD
  sp/w1cbpa43LG2TslTw+rNWms77ZXJhh7Dx5G+D++0q8Y9gAlRYI36LfxG8Kfd97ZYk8fcVs
  LvrgLNJ+Is+jt24jpbB/ypr88bHWwSe54oj5mnOPuomoJ+I4kjPQN7z2iYqBnMrU4jaTvOBd
  7xXnquACEIe/dlWHeWXjoKHupYG7CFAs//BeRxw2AfGliTmdIkq5XHg+/sSwPfHrhb5YgDr+
  uW91IaAhfO4grvd5YiurSuKqFFU2My3nATaklwJ615/Kncz+iAZcVxbndjYgkgBxIDziBdXk
  xQ8cDRNritAK3KPENlIvjE9jYQM2EysLMcByi+rg0TQVI/MxcI9zM5kD0eaCnInNhx5L2/8y
  dj9OlbEEPlMQtKxlJsXN8WS+u1AHq+iAtW1sPWcTYqZCaUQPVKeQY3RQfzQowjRuBo4c0F3q
  NjYM71wSQ02CIvbi2JozOlB4cyDKKNEWNKJpsCPDVM+0OBYuH6bqgbTMQBTvh8EQtcx6bsxm
  NFvoRHdVZZOc7M577NbJKypSAQNE2UaUiUl2bTNaYDwmtVJ5LSISbZKO24yX1Gg9/r1zclD7
  poX4GRqnPQxEHtTnKntxz3NhZk3++OQmjzYnwUTZhqGIfF3tvNkBWQpNGvvUkQg82kWRWkBq
  baAJST14hpUWJ6W5KlA73YnQLbILZkHUkzx3xI2F9dSeCR1Br0IKk4jyynlGrRB4vledljaV
  kq+QqWT+BUMGGpCDU+Xo2yC40Z1+s1OInOmZRpDOj1azdkmtJThLmtZ9ce+OCNUPHaYponsI
  TKaH5JTu6JoyrTYRaYky09IEqjmyWQBxRFUhjB7X1rvC9ZRF6p5BN+RAJ7Vmr1bxGXNw6oGq
  FTF6ahNRXvs6zvu1NS1i/ai5OGucuLJtVmZcDsBQW9rmADU9KqOSLpLCyHWxrLfm8mv2jgXL
  C5TPp4T+5Q8ARaAnV57Seks7LTJAMdvuOG80g9it0jrYXEvxxbkK9SBZ+XoiBZoAuGwDru06
  QN5G6lW4IQAQ7X4TtRTRtrzNlOsUAVIX9FaqN+0Nrn9WpKziViCVNUNTSpnrJPDasWdILdTF
  LrhfpBiXu07KKqX4E0kMB1GErzymrGAoP9KVDrSLJPGJbrtiFDevtjd18YSQstI0VsoMRLqU
  /UgGkh2vy5JvAzFmRHxBQrClPwa2YpAcTDlmPi7J46HxBb2x1JNW1knMWY9guwvkKF1Qg8+j
  nU1xa1gK1Ni9NzYpCAL83gEz2DZntWZqbXsLxGCAzWlm8dlYiwFfGgrMFE7+LofLfI0G03l6
  Kwi02lIJuQtrOEitMR4xmxpORbSTvonOrmvTxxXXac5LxBABFVu8VjKfk8Ew7qubc9o+Se9z
  tMQoWClTqOKz9TB1sLW0Yn2kaYThYiZjZbWCmDxZhT0OfoyGyXXM52eCnMlzOj3gyi4NX1S7
  FaHTZQt/V0voPCKSgE1LNbdXfcTNslncfIF2cc887jkf758aeG6mz9BsseCiQY8e7xihfOHr
  6jrZcjxqB4Vs40O/8lynhUDGqh0CNiGvhtPTaLujVuvXzvoDeAY1w6M1MqGM9scXNeYGuIoy
  s+F64xyQd7JwYbEQb7CowRb29bxa0GMjG+DNirf+Veo9MvR1+LIo14BL1SxUwPkPmZ+OcdDN
  7OMfw5u9g6Td0jNX8W/XNZZNd6R/VZfwB/Sq60x3EDPotMLpxTy+UybV9VAg5IUleZ18keRk
  OiX3e+Kw4rl2sl4qmsKoK53cnQXdQVZlInFUptQlTdTH5ZcRqFddwIhvoN53jamJJKWewI57
  80zTaddQD3hN8/dicf700MAQxFdpDZY9ZrLa8vnf6/3m4//b7QU6Qk89pflowkr0kbubf3bG
  +u9JPd+Lp7zRPKwyMDBf91BL2DDqpF9gGWtsnM84OK0sCrH1YR+agxKauuddPuIY+TYDfqsL
  C/9aeA/23VtdaGlNJsr+TExKGYsa0i/IEfkN5Upkp9f4spN2EEZ68bc2HKR6aSIW7MRvFDBm
  4CYHlsBUGSdnU2dBjpBX2yA4/rJ1adZ+Zbd4bHNNppR1coZu0HR+9RNZEFg0HXZWsAA/tsF8
  MrM+LtCCT4URpHdqVvZLdWduQ0NIradxKNgSEkhy6lR7VmaC9jEp7DdtSrhnFegAQqRX5fCE
  thV+hPMQRags/cdrxzIpEVZ9YpFEIiRs7MR5zyQZWlErDqV77MZSN5APICiEQwh1Vvh6i9WE
  iWJNMohSY5NwU+h5ImQkh8R303eDv/I4XXheygKG1HeIUgOCyCAdgAhBhJFqmLgctAGAHPf+
  eyIoA/uTUUhoe/+nfjJhhfjlgesniCP0TB5UQTTUh8sTHDTIPjqViIpIZzioGNnEB4a4gdej
  ipqEY/yTbZwYN0YIarZ2jLezdY+UfJsmPy8ISRwIUaczfO8ljKhlJGxxKr1wKtoWhQClC8DS
  eS3yeVKjgZXXJwy4SM6ziK1Vfn94jcFoMlWHKrrgjRtlNMl4GIbjiunzi7omiaXnFskodc84
  gqgoRU1jJL3oRsHnhQRxhynBj/awkKVgQtBoKNoYZe+INu2oS7O3j4eHkaYWhFLRYRLHkPfX
  cTZIjGgTfoE4P8z4VyP5f73XZ4xTJmNEc+4xh5tjJ7uoM3q1WgL+eXN/licX9GxIeSimOGrF
  p4OBpxFtd5EfoYqcMz2SpI7UGI2f1ZF9B5S+VZB5hDAhKVG8+JBkyJVoZ4LBxG3CyGYmh191
  aJDI5QJVWR1MuXlmeTAopZZrWZNhVyQQh2gSeXB0p3LreJMbdZYSZZJ5Bpb4ohezEy/06HeM
  x0RdiYGu9kNx6ZX9oysntkbO45jIOEZ7OXlNdpmMZpqnqCQhN2n2kHcjYAZD1iJKgofoB5oN
  JJUBKI8ImZPCRxSz03g8yZtixpYd9TWmpiYBSV1Y6Wwn+DHwsQ5exXO3CF0zF3dg6TOSOUU5
  KJi/EyzAQoHWAYfwk4BGSInkhGyQKSP+ujeBJSWcGOCL6CNpuQGW+fV+zld0gmCeRugrXbmf
  z0aXsQcC2AlY0zZGwXKgbiGLgyI5Cxqe1BgcR5Y+IDUZA0gj0hl3zkmWp4mZtFMyvCaWNGV/
  hbZ6lmImQpZW8pk6uilNzCkq5XBI2eGaw+huyABr1dWSyUZzyWZbb4cwL+mgO0ihakOe9haa
  YvcROFoVPjiGggJtHniQ0iiOKfp+5DN12EmU59MdMAShFnZ8Wtpi0mOCaDqKefSe1ZmQx2d9
  A5hK5uAOnYgw1qUsPRKeaqpw+biVTRWjJdSnwfhE8Wii5mKcXVmmMqJzuzV/9zGTQTmKb/ht
  IKpLZdplqKb+qIAqcDinikUKaf3Zb3h6bXbKBnpaqeWUbIfJaM3DmCf3pPJVpkKZOSK6N/oS
  ik6HolGmqENEn48CnP9XpbVajFiqZDgHqzj3GpeSl6DHrJpZn6fqq7tapqiqS4b6bnV5Z9gI
  o8U6nrwqpNr6hDfad42qaqYKrD5UrS/HpnLmmnLDq+KZqqQXp8lmXkd6KAW0MOeqp11Hf/yp
  q534rfF6S/M6ejnIqU7pdUQqdNHadQcbj526aAU7oA+rVsfCEZ/qqxl7KA47sSOVg2Cnp/RK
  gBZQpFNyLJPkrMT1sSTrnxFbm0hKsS07mH1yshwbqKd6lhv7JBALcGAHmEC7KhhxEbREiyfB
  E7R1MXgPyR5F27QSKjtOC5g1EbVLaxxU+5BDe7VVe7RAm7RIK3RaK7XxFbZuNLVkm0SmSA9q
  u7Zs27Zu+7ZwG7dyO7d0W7d2e7d4m7d6u7d827d++7eAG7iCO7iEW7iGe7iIm7iKu7hQkAAA
  Ow==
}

set pieceImageData(Alpha,70) {
  R0lGODlhSANGAMIAAL+/v39/fwAAAD8/P////////////////yH5BAEKAAcALAAAAABIA0YA
  AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
  n9CodEqtWq/YrHbL7Xq/4LB4TC4rAuiAec1uq9Jqt3zuiw8BgrweQO/7aXZCeHp5fH+HiCkD
  AkSDhHmJbQF5gZEli42PepZrkwKVnCGeoD2jBASjoWSpqiOsQZ6apK1br7QejgNDeqenm7cR
  sxLCObbAG7m7mr/HDsQQzzfGzRiPhkC8vczUAIt5uhXeAgPXOLHL0dQM1kLL283deuAU4uQ7
  57LqFt6e8z+eAE4NShfJkTx6mu65o6Qv3CN/ddwRTGTwG8JHChdOtAH+AE65IrlQFWqnCWIM
  fIQCLWS0AKWejRwWkSMwcFghmjJ1uHwJ0mOSintIPrSx89O6hQyKwtwgLqXNkjpXLo0BdGSR
  etrGDWlq8qTGo+6Sfl1BqdekrgsWBTALKUdHNN4GpPnoo6oAukCaDhXCVdpYBSvFSmThDq0C
  ve9qvA0Qdy7flYaj7ulFQNzUFhXt1RgFh+GCb3C8CYYzzUTZU54knKbZlkfpH4gJRS62kueP
  zHhfKAW7TDA6wmEjIC3leetLyrFmK8ZKmTWh3Jtfxros6mZWlZ+Qtz7giXJNsuN6qbU5QPx2
  ydRhxN5bd7076IAGx5cPePiZvynoP9gZ0ej+nW/NmXdXf9YFKNJBbsUVEE7fwKfCagQUdwCE
  qbWUh3YDrpBcUBFUlR4Kr/FglyYOzlBUSQjiEM9CmsmAX2AWvseCS/B52N+HLWRjYIRa7bBi
  djtiaJVixzUnToknQKiRkJpg6IJe0eCjnIkSYlNbigle+QgaHBKpJZKmaSkmjCxASYGU/1RZ
  F4QBAqRDLgsG6d2RNRy5YzIxKCkRk484+aRsF1mUpn9BjLgMmCwYClllQ8rg3jJTljDmpL39
  iSUEfQ2Ko4YXysnjpiHIg9xanjIqaAxxebpipCNQSNpc3r2ahp85fiKaBIPIdV4KHW2J14/j
  /IpPAIiKcGI+OUj+dGx4uc7waGF5UirtrvlhhGtCmA37K1fC+uqVAKWqKcOqlKVaaj0whFSq
  qS0CR+q68LLFUq3Eihtjr/PyClk5dvHrXrucjgkqLrEtqCizOb1wMIvFfjDttC8gCw22Khja
  br8MAPtQwx5056m9uokqkDXroouZyPDSSW+8LB/oAkCmqsYsyCFwlUZTcairss1wAdrCsk4R
  9V65V5KacKIPP+dC0pS+XCkEvyki281B68whz4z5PGOnctK8wo/ljdxkyV3qC2DL4iwdXssp
  U3uCx985Y53XHRj0rpC6eBPngXkX2VxyHGsAtG0zaDyZd186F3gGC2upNtNiOi1xAy7+DdyA
  3QYmd4DeGPbNJmryLI7BQHK6SeXhbAdokOj7nM12XLUGm3qQP36QNcANeCMg1GvzGFk8co2O
  soEah020bHsHqLLZYrLecW1yUX1l2IM4HyjkrHoA+Zi2ywTferxD1aFMG6l7p15hD5D+xnIu
  L7XxBp6aLnOzZ37pJa6rjkbyx2ffgWX1+9v9MoA+4QDJOaCAGejylTv2UMBv7Rve8XpXusSY
  4FkDTBQGH2a0HqEAg9EDYQs2mDT/OaCA4ZvcfYLzABRWAIKLMZLIzEU/1KBBgG4jQa52pDcZ
  nIN/AcxKo3SYvzZ97njWq0BygmgqQnGgH1lz4lHudqujwK/+QnIbRxTTwbm2UbB/8erH15rn
  w+0tSnEoWBipgJbEM5kxah2oXAop9hn7UA6OEugig1ICRh6tzVwLHJ4YA/YJg8WijRcAGxNp
  l0EQoM5+BySeBU9gOKbhTgOrOZpg9hY37hQIgS3sHRY7VMRzfVJs8CNbhlKwLERaoJJvPJz8
  SCCmNUYOM+IIYOgcCan9SMU3eEwLC0nZO7vAj3PVc84xh3aKHq4AkOz6luwWGST3GeuIGLob
  JC03ATNW0WEHHGWMlLedUoprNeIMJbha5ogA6YhlszzBo0xou1jWxpardEUtA3nPiH0RbTnM
  gB1jZMd+OmCg6uyFxRSaDfX5gmv+TTxUVuiZSH4YMVi4hGAQO3InwrUKos1ZSZDiWa3oyeqk
  pPmm9sJJLd015zsKpAxJ0RnQA5CObaaQlzbhZTpKgtCV47OnPFCCT25OqDZdg4w/d8qydHIA
  oZsz6LG6AtWMrWZ6DM0OAAKS04i6Q15AXYAjgOicfIKoQTSh5g13RIgS3DR+CwmSU1GATZ7W
  9AKcKc0Rb1VKl2UxDV6LKUDX+dBUxquTRBxTWLMIGhLesypCdOuVkkom4LzxmiqsT200W1mC
  Bi0Y1gGa8VxaWAHdU2xGtekjVVe2t3UqmUGUi1xbW8+7jDSuHZVitZgaxrteADEmeSviPtFV
  WjEAuG7+JCzbbirYltENGdJaLDBNmzTVjs2PrlwWZfHzoFiOALmX0xKXrqROrYGWsIpF7QIN
  NqbrXBCtquLHGzrVV3ZGEoe6fZ5yZ3tftuY3Sf1tqm9/e6hN+pcQuQ1EVaZE2tSdrb7wei4B
  OTjfszVOqS4RouUGByQOp/YB9iRBwYC5PQX38iJCxGoz1+Y6x7pXxBA2EOzOSljeApSsD/Vg
  dfa7o+Le9sMHDXBvGSgCF3oMknUtjgsdwmMB30W4bCOpBxx7YtcO1Yx8yPACKTqxfR5IvI8z
  ozzZw2EwH8aBKM5x4g601dNcGL06BoFlUHrSRtYWJ4scxMcm+USQ9riuWQX+slWFbFeziqAk
  fHinOxUdUkhUJQNHnh0lIu3g/8Zke1yeQJnFBI6CBZIEHuanQccYy8Ui+qghfjQGjjwp402a
  a1R+MQjefKWw9iPG64LyRYmsXxvrtFQhArWfo2xpXFDCMoD2MQ6T01MLUPp148B1uIotvFKb
  YNNf4iwfvyxsL3v4Z/ZMrYeE+okCr5prlGKLRcsqJlnb7qTkS6kWT3rtktSvH8Dm81P/eVFf
  r5jXVmaihDWgu4r4e9h8qleclXjKlp3D36oauKbhDRmU1jvcK5TFp2npbe6BO5YfprI9TajA
  MpNqdaI+LSjnu9NJCPqOxy7lYtSyxijye9eW0+P+nfYX8YXTWODUrhuAfoRjG05bds4UKKFz
  /ZwAzrXb+iGkzR8GjhHhM+gYIGEIOS25N36Y1tbG5AHTreYUa0le7rpoyDRjLnwV5iGFxnoi
  pe1F6WYR6EDu4rMlDaRmMQXh8UKwLn0uqc06je1UN4Shrm45sJPx416XGrmhd2nCUnlB637z
  u6TM8ZYPuMhinczgeD5kuRPYthvl/MXxbrZ3dYOaM/n34nQNT7rz1+517OwzQy+tFilqQdPB
  n1AzDWmMp3HytZn9VQW2wKuffb0B8+/LyxuQxoAY0Pw1/QVWa9/P75j1kqdmUjPNaCdDnJGE
  hzp3R1gYgmBQXtq3ALn+YYDtX0oe+eLbN2HfnBWzJ3+iZYJwElczouJ3J2R7OGcCYzUqdDZc
  htZdTeZc6RcCtCd+zaYBOzQ7yhYvSXd/6/dM0HImJAR/RrVgpBcrU0dbEPhGxDcBjpc0rJOB
  XzZZDyU24iUguFcfvPV0rPQ04QV4+TaAziYdkzJe8cdL5+cpPBgqQAh0AJd12MdfSXhRLfhC
  jiNNbfUALwhnQshw8AUvPzJ9DTB/K4h/+tY6qOdVS4IKzndGfLM1QnaBGeVRvLd0SkiH34Vp
  LmKH67KEINBc4odY0NVwfRiFkuR9cfR48/NZg/YwKRZyCCggVah0IVZhZtiFP5iGG1RzKQf+
  KVn1hDAWgYVFAyuihYKngXiYh6Dhaz2jervFRH6IhOInQ6BYbaJYQXzIWkP0QVxXOCqlAFuI
  bqlYZGXRiiVxQ4MULZVYhvgnAu1EXXuCCsCnYmWVg/VhWMfDER8xdUUXhEc4d7mIdjNgiHdY
  iwR4i4M3iVF1c7g4hZIYZuSlGNsocsvwiMwDMeMoVAEoMIA1JuoITe5Beq5nhPb2jnBYdIJY
  J8gTRMH3cyyTkGnHkIi4AYAoEP2YEif4UpiIKTH2FvOUkeQ0gepXNDtAj1+lZkBWf8MIHsvY
  g+3XjTSRgowoZ/kzZz0mV0tkkKSGjbSoA5YBk4W4i/qEjnAlhj/+6I5/Zo6zBiEdQWVysTex
  OGFf5HbTQiy0qI4uKC0+yYIpZnd5RWeksZFCp4+klhJASTwo8V4UBHHnR0WuqIowaYAKSYh8
  N5FQSJSLhpVDGI5yApGdF5MsEhoawVV2eRHlQZVbglKDGU2W0mo4YJLRaBAq0GARJpJWZnwQ
  aJW04xF9GQtjxo5mcTOkYUqWiVk8GZLW+BRpuEhZSFd8KY7014SB941SiQ5suRufeYzuaIxo
  ZomUkprn5XX81F2nWU2F2WtgmZzjNX2UaUNFw1RimUdF9BZFqJk6WZa194AnQ5f105oXZ1LK
  2YpG6VmRR0igoYFmwotDBYuZsp0PA5z+DhCM6DCcNAae4SlfsYmX6xKdXqhQKmmd/BkBR4KY
  Vdlmb+mM3NeXKmhZZ5mdSql/kJMuyJeDUKRW+KmeuaY+QHmgCAo58MkbwjmD1bI9Eiqblamd
  Q0mgHOSX56gl5ABLVVaWDfqJH+pJoNmdHCo45bmP8xaedGYZFRaUG0qbaOiNfKhJFfNGNaqS
  SPVlekmeSWOUOOOjdDaTJRBqaviBqlg0fMIi7MedPVeaH6WfxCamTBg2gveLc7iaTAQn9Ylj
  wBJ7xHOhAMaOC4ZjacOj3gQDyBdIT2pA8UgVkNlq0gUQ/8JJhxqgBqRwnqg67oEvX2Mn9eOm
  SPOaDmopCyL+eHLpnnIaiNaEWX21E0CEM2XIW8CVlGc4a4PKMO4pVPSpp0pVope4oHfGbhoF
  jXNDpFbYhtxnTAuUpF84qZ+6erMoU6laN9ITNg/3b/YpKxx3o7F1rDoqG4fYdCG5kktJMnMa
  rA5Iq9k6oXBIbmtxDrwiPZyWnM86q9JafPvXqBNkrZ/SergKX7BkPNVzfOQYdzkYiRJ4nDDq
  rjIhE/PKquCElKkTleyabDH2cC7JrW3ib8DydbO6r8iHpusan6uqPpZkPepqLVfaKSfCq9um
  qL60Tv9KXNCTqboqnWyaZzm6bwbLgRfbiC76LgARUzK5Es6Tr/d2nCzbqXcSl3T++rHTRE0r
  AomX+KcgKlTVN7N1CINM6q1W2LE+K6CFAEI/RbJZdLKUQg5ay3vFGUAdGCYxG3dL4Qh0BkR+
  p3P+qZhCuZczOjssum9hmzpqSox1OztI2qETy4xM25C2WEJn5qGJSLVfi2rkdrctin9/unfi
  d7i8V6z4tRQJiosZgrbd96AlK7nSJ1mWGpSvyLkIe27qCmTyiVXYOkfvCYxi1meGu7LxybOF
  FrqSO7qnx7nuRrS4a4Oau0Iz+kM0O6Nze16722jc5LjgR7a4a7toOKspOau6gohR2yinexnT
  W5W6y7mQe1RlG4R1SqXg65k0Fr7gqzSnR6XIE7sYGZ7PLzsa5Bu+X7uK71ulZlptzTq/8oai
  HWC4DduxxZKxdCGfFHW90sJN8ou/stK+r3TACJy/YcW/NEa1G8E0xBC1lWe4WHm6ijiUzhsm
  Hdu/pVsNUaqa+Ehwr1u/FKDB2Ua0HVx4HxzBVCvCXUsQORuolPi6oQi9F9fC6TqrIKyuMtxq
  NGySEArBxKiuT0rAZtYQTNzETvzEUBzFUjzFVFzFVnzFWJzFWrzFXNzFXvzFYBzGYjzGZFzG
  ZnzGaJzGarzGbNzGbvzGcBzHcjzHT5AAADs=
}

set pieceImageData(Alpha,80) {
  R0lGODlhwANQAPcAAAQCBISGhExKTMzKzPwCBCwqLKyqrGxqbOzq7BwWHJyanFxaXNza3Dw6
  PLy6vHR2dPz+/ADmpQASegAAAAAMAgIAAAAAAAAAAABAFgPmAAASAAAAAIhm0OXjABLRAAB3
  AHlF2ggEA4L/AHz/AADgUwADAQG8AAABAFYMFgAAAAAAAAAAAJAA0OQAABIAAAAAADGg2pe5
  A5NPAHwAALABU+UAARIAAAAAABigFu65AJFPAHwAAHCyvQU7AJJFAHwAAP9F2v8EA///AP//
  AG3gUwUDAZK8AHwBABUMMQoAAIIAAHwAAAAAxwAAABcAAAAAAFCglAO55wBPEgAAADjwJ1zm
  xx0S0QAAd7CgMGk4pRdFegAAAAC0MQDmAAASxwAAAH6gjAA4AwBFH8AAAQCeMADGjAAARQAA
  AP+cAP/mAP8SAP8AAP8Jd/+HDP/RAf93sAAAWAAD6wABEgABAAABIQAENwAARQAAAABh0QAE
  tBcA0QAAd8A5g+U2RhIwSAAgAGJ4hgkgA4I4/XwwAbAgAGl4ABcgAAAyANs0BwUgAIJCAHxQ
  AABQAL8AAE8AAAAAALAAAGkAAAEAFwAAAGqgAAC5AABPAAAAAPy8AOQNABIAAAAAAADYAADm
  AAASAAAAAPiDAPcqABKCAAB8ABgAKO4AAJEAuXwAAHAAAAUAAJIAAHwAAP8AAP8AAP8AAP8A
  AG0pHgW3AJKTAHx8AEosHvYAAIC5AHwAAAC+UAA+7ReCEgB8AAD//wD//wD//wD//7AAAGkA
  ABcAAAAAAAC8BAHm6AASEgAAAAC+HgA+qwCCSwB8AFf45Pb354ASEnwAAOgYe+buwBKRTgB8
  ALAAGGm36BeTEgB8AEH/vjv/Pjz/gnf/fBssorMA6NG5EncAAPC+4OU+/xKC/wB8fywAhADo
  6AASEgAAAADdLAA/AACCuQB8AOMWvpQ/PtGCgnd8fEkBLGYAANMAuXcAAOAxvAMADbwAAAEA
  AAxYAADnAAASAAAAAERczmHDcHPTRyB3ACH5BAEAAAQALAAAAADAA1AABwj/AAkIHEiwoMGD
  CBMqXMiwocOHECNKnEixosWLGDNq3Mixo8ePIEOKHEmypMmTKFOqXMmypcuXMGPKnEmzps2b
  OHPq3Mmzp8+fQIMKHUq0qNGjSJMqXcq0qdOnUKNKnUq1qtWrWLNq3cq1q9evYMOKHUu2rNmz
  aNOqXcs2KYK3CNrKnUtXJNy4dfPq3QtxwIG/Bxzg5Uu4MF8GDw4EWGDAsOPHJgcEMPoAgOXL
  CwZD3swZI4PJRQNcvixAc+fTqBs6SNCAKILKoy8fSI02QIIEs2l7HFCgQFHYsQEs0G02QIEE
  w4lvZLAAQILGQhk0sHxgwIAHCQAUcKB87AHLArpr/2RuGXpQBgKCW07AXTxY2K3dX/xOfSh2
  7QwgQCAvXL7BARQxAGBQovXnX0UFGkhgduoBEN6BA7UnEQIDAmWbgxBOpACDADRgWk/pdagf
  AhDQlwCEAyhwQAMBmKdaAAIsEIBgOjGgQAABPDCddorhWGGGCm14WQEMCEVfgwCgqICOLULk
  AIwCzLiTjTgGsGMBPUoJpEIDZHebZaABdV8B+unXnILuKbDjZS5yWcBoBYR5kwEcNvjAlgox
  8OZocvp0YYPxyWdAiJcp4NAAazp3Z050ImnZokW9RYCkSyFqmYyWFfDhTgOk95x1DkzXgIQw
  ISCgdaiiyoBmFKbqapEEnf/qKqqbirSacw00sCepCOmJawMMwjrnng0m0KdQdzUlXXAJCOsT
  eg1uNxMDss66KkGtzopqQdXOWmtIXarX5kEIJHrZjzUZYG5wx/5k4wGJHaCAs5EqcBl7EJxp
  aFBCxmYsTY0i6eKZ0RLkALHqtStSc8aSeKsA6BaEAH0K6GfAm5ridLBzxSrcEwMGJBaAYhEP
  hYABSHrMaJ3rqazSxikTdKR6Jw5kKZKQmjTzaA2UTBBw/tI7U6iO/osswbF5aBRv9zqg3wCW
  NSA0TzCvt+9MFyNZwNUEIB2cbxGue6/LHr0pgAFlisaeQuUKR+J+Z04d02cP1H0lvA9oKRTQ
  PMv//SzCX/usE9N81lQ1s1zz7S9BNzdIdkeA3zsutmJbJvhLNtb9QIgFyFg3rz0ZV3TO7grA
  YQBv77cnbn7fZO9oRg+dXQHxwvvmczI7J8Bfuy8wu8G3A/bXdLGbZPYAaWfHtUEMUJeffq+3
  PhN9DxbVb7GkP+u7o+sJIH26/qJO0+H+Jl4045WDmVL6KiPg6OUzwQZxaKM9MIAD9/8JQG4+
  IWDlkAZIHQT8xzP+De4ADFrAmZADP5Qo4E1Se8uqmrO13MVpVW9Rl3YMNh1NYXBiznkcR2bn
  NDN16HLk8dCIQvS9mMAmOUTRn6MO8C2bIEBN3IsNlnryGpbproEnyVqx/7iWIJqhD0kJyN5I
  Ipcp0GHLa6MBIkwKND+T7YkxAjzZkaQIEwdsD0zIK1OZbpioBSyPJic7E+0EhMMOzWtuFERb
  mb4jrYFUpgByfJqoYpWeBohxWbhDyZkakDcE7q+GBHhd70KkNJ7IzyhMbJATbzKAL+ZQOwk8
  I8CgCCcFINIk/KGZi4r4NWwRSlwp2ZlsEIky9UgtdODhYk3OtIDnjUiM9HnAJ19ysh0hp2Ji
  DOZ+HkCwht3kAVfMIwQMsMDMxISOwCTRd5BTpf8lAJj6IVqcqnQAjN1Sm7vkyLKSNkk+MmsA
  4XzJI4miyhmmkyW9vCTsYrQn8dlEdDmsZUxIGf+bKFUpfWCq0n2K9c6MQEuHk+MWJ2nkyFgO
  pXmW248wSYQABslSJX4Zki6FydEBdnOV4ysQbsJ4y0ruKTAwgaYYT1k+MRItWmK0VKBQ8lKQ
  MoSUcfrJC10jz9GUc26G7OmQcETLgo5EMkJdAENb0k6hOnU9K3EAS/XJkNeN7SdUbKFNXrfR
  jg6wQPa0IcGUOkYBenVQsrloSG6lHWxylK0A8KRL6KhMBN7mrnfFoxjRite7iqhMGgSbSl7H
  oAQ0sG0c219LBMRYdC6EP6NaiKkGwFitjqSVQtUlTl5TLHkeoERvWoBjYRKup4owJIp76lNr
  phLMruenAzEXDFUyWcb/IpIB9BlVDSnUWMuGBIQAQJ5ZgwlR0eIEOAKI6QIe4NUxDjBRmkwJ
  W6U2wOZ6sVBzbatL81bNKs3rjwbobjXdKlOWfIdFO4ItARwQtcVsEKNtJM0bDeI/llWHXAZQ
  pd5WwlmnVvGe0bqRD9WTXF8B4LQgodCA5Ylgj6RWtUJlbUoUl1CDMWslKTJXAxRQshsCzrj0
  DVlwHrBU2tKHpF51n3B8a5IegmevDCpNc8eYqAZzBLjaGa5XF/gAFm9EpTMOcpC1ad62snfF
  bIPNjDqYktIW6wD0amojBeK/SEatwiR5sKO8Z0MZDultTt7yfuhjY47w86k9ZomWIXxJCaNE
  /3H/RQg/3UwSAyPpvj+LFr0e6KgGYNkuJw6yiqmKNWJlZo51QnFzEYCwMmeEYNQV8govM2WU
  AFnSmA4m0WZ6ko31DAJ7wrKKDZsv50T3I6Yr4F8QhkcqH6kA6CQUkQSyIQ517i+WZI1aMdJf
  oXJ5fJXznhg5mbCvPqrIbIaTjzOy5mSf72ViO/WkGiTtjliyQ4Bh9b5wDOtQwprWCMMN7ziU
  gDqiBLj50fEYL7VskCCGWMJWnXqEG+SDdsioF0EApG2JaX1TeiXTDICQ1S3prAkWJVlL7jKz
  Q2iJFWijanN0RDSYKQ6v6uJPKuxsWrm151W5PtdTKjpNJSCpUjqA5/9u6pZ37W58jtishOMe
  vtqWsXM326mcNsnNna2eg3eaiXHGllVjo0SP1PSa1jGVqRzAZ+fM5lZIH1HTN86hAxiAWhcf
  ADNPju+K3BBMBN/PkTUbE4q70ZY3VIDa1x4Afs8Yrs48iVU/nekxEqzaqK542OsezBuGiM4U
  SVFi1CskhTM6hAgJF6kh8HCGXEde6j2IEI1VQo6afQE7WoBzIQAb3DCIRepO0Z5ol5Khe7br
  HkErklhkAAfgj+lW7ieJOuWgdlPE9Dw/sEpUnnskjWdJVq/qghXGgAUXnXGJUUDkDVK1GZ2V
  UJiPmsUgLps9gb6jNvKlAUsCtRN2lKKhnpv//gRQeb5/n/McOn7q9wSxFDOWoyTCbaFQPxHV
  40aOe9+8GM367kxJ3GCpxiM/xTDMBT0M9yEEtD+AhTFcoiOUJlcPATMs4lyp8zaf0U/lNyI7
  Q359V11vo3q6dxLF51R4VxKfEXtDkiu5smCO4zCPQn9OgoLrUQDAwmAwOHEs1Xtao34RGIBY
  8lMqV2lUhnv38i2fsSZb806EEyf7hyp/ZHqjsnD7syp8w4FlcioCdF3HZmmW4Ule9Tpx5xIg
  2ACfJVEd+Db5J0z2lgA32BAVFUvNpQACECVut3+EwnIVgR6fp0zmp2mQtnwLUVpfgh8J0UeV
  hwCecixEs3hwYxkJ/4GI68Ehf2ZOCXA2t3R+JVJYfOiB8OZxKQY3wfJmvmZ7G7F1Orh6TgMt
  zWI8ntV6yCRPgKgR/naKqzeJDWFn9yI4xIZ4BRFz6lErsxgbJSgQ/GGFYrc5c8hhYgQc17Qf
  fPJcpEFSCOBFAkBI+HOFBIOHE0E4uvQWV3gyr8VLRUQkfYhp1/N/D1EgsFZd8Ec9dXhLkHhI
  KvGKXVhZ1EJZ94h1WIeP+khZbIUcpDgQ/laJNxIiNHQQ0BKFyeMgpgEbZfg0Z1IrF0JICrAA
  ECRFuZWGuHQZAsdRpaVCGnlQPjcSuFg0sfgRX0eLjiJHRxZWIUGEjpNNMkgdbRgRMlRuvf+R
  kzq5kzyZk3UCkBpBbGFYEDkIJwYBk5chN9fDM8sHH6lTSV+jTAPFXG+oHWizhLYUSpSGdmpU
  kw4RMMYSRqK3HtqYEUdXgOWIacDxbSLRfXE1Y/1VSzOGWcPYESankh1ii4fiPPrhK0JIawzH
  b5BYQWGjHYoGG0V3Jnl0ISV4Mx1Zd+7IUfKjkXsVfkE0k2WJEXeJlw1yaBBlWCQpVKgDAeTj
  KDwIElBJHWn5NA+4HFozNQHTMUfkTgehco8zTmhZks9YJvCRH+FSgAUCkhAQZsJYJoRTlxpx
  JA/JeJihTvcCZaspabN3OiIhf1MYZPRxaDNGRZlJESy4WjsSkNz/QpN9GTcHISRoWSavc3Av
  JExjhxAXmTZd2BD0sZyZZmB+FEyERZnOdWR/KRJn1k8vEaCcSWklpGSAJprQM5PgcZIbUU/R
  aWDdKZDvI3nf6Rx5xj0HaRDpc5oEID9XiIK5uSP5aTpOY1W25IvqgU3d90om+FHXlEVZM1K0
  ZXZ+Fp0TpXzflyBDaZdJ6VXIdBzk1hsKB39eopdlozu4tqRMugDC06QHEH0A4JUE0H00pIGW
  MTWG9FkCRB6BtGkoRiKtVBoHkR3tZ4Bx5XgQtInmiDHlR3sK2IfxiKQXgRgXeikTioMFuh4r
  EhyfFpfimRBuaZpYV4HUcmTcg5wcISqK/9aH9BGoFIokJbNz+9I4q7cpVkY2vrIdZQJQHGmc
  DIM2TDdBqimTl1RC8aioAbIjxhhMqWluoPR3PcafBUdWHYWoawMSVERwu6gdzXUf6FgR2aFb
  F0dyxnpxyIpBIwenkEpl6REnrXchG1oQa8pR6ykQtMRRBzU12IEbDmAAF1NuhHcptLp/IFpd
  wVmuY/Q60xoS4yRPrYYSWtkhRZlsfpZaHQlRAECnEYGUo+FWE3VJHvoRt9MA1TiHBnuwCpuw
  DIuw1WhRN1avPSoQngpVHypPJeNlmTKJjSKXy9RTaBmct7SeH8ig0AkBYEilB3EzZ4p9jMSv
  FwGngsF4MqKuHf9FTBn4R4QysBLBH/jXgQCVANZlNs0KEccToY/aEYqnKwlELmaqaPF3Jvhz
  OwDbl/SxKX+iK3tSghfymH0Ip2Gkimyaaa3UcB9hbz0Vr5GBNPaTInfaU6jTTpqnHwyyfcvB
  e4UyY5fUMyvBoE6VpwJBnJdCLgx6dQw6NUvJkQ1xIQV4eD31PAdVQl6aR73aUsMJHkUrMXII
  h3D5sidBe0LLeVFjsxxVjY06RkcysRkBuu8IWPDSp9qxXIrRXMuitihhpk5js1GbpR6RuCG0
  Zwx3uigbNanmPVk0QLABgUPIRGS3EMmbliRSGbihH/ehneUIUe3au/WaQ0kYq/fyPIL/y2bI
  U5RlqGJIthvbOxr+pIz7EV5L0mYwmxF+K1SAKxCxWT/0cr+OUgABmEPKC27qkb0IAax96VQP
  qUj6cSEyRppvexmN6zwJ1nlt16aPkrmJ5yn60U2ciqOk6SDCa2xR46ANAbpqSa6SJirxixFH
  u5qzqLK0hrAYOBiMe1Y/6bXBVLaDcR3+sgACwLM7UjGki7LDSiLEY8N1F3+YG8E8l2YkkZqm
  JiDpi3MUIjYPeSUizBBoK7Bi2sAVtxIVq1r1+8LV2E8oR2ucSWg6PE+78xAh8sARJpYYkx+e
  gpZRzC4j4lAegWPl640URVEeSCJ/orpKi8HZ9MHmlx7p2YEQ//CvH0HCcOlReMqOzfWsV3wR
  JBSh5gkukWNPz6tjvVZ7XlVccQGO5SMR+hKdwGUAMpWzfAdRbNkRn5xsZEqSU/XFPXWVYjO3
  w6s+HjGvMteRZndJtnsS89tTYRwr60J2BOpsv6a/IQgRZHbHTrVRwDUvlJxNXPwo0gyrsngk
  xoI3dQPOiRHOmqMY8ObCvYjBiDK2fXgraENw5gsAPNsXhCxoL9S6rgpBlVwRG9Oy10sxI4Go
  o1FL5lu1t6Svbxl6l8EdRTnPA8GYksx3LijPopPIX0uTebxzPfWf4lS5zkZdMLnAf2K3BlXH
  sdGRpbm/KWwR4ctzqhoRg4oZAJLNTv/VGJxE0jeVHR0Zz/KEL5dL0aFVJiZ90tJsthjR0s62
  0iMcIqn2HBzsn/jMmw7cyCFixPAnP1GtniSkEgVi0OZnpehMEA9HMHGCuYtGPYYMXJtDafch
  wIurPDgaLjX4lhGK0Ryhor2n1Fg81BAmRw+mV8xZH73M12Biqmmr1xpyxhZcEERoLHtKTOoh
  yAshJDutWvh3Xpc8nDQtz0VdtK/Bw1GqQDwspXOoQDFyAKnWOTEi2qjtpGFdpTno1Kt5Hz69
  nVP9EYqUf9bpsTOWS6+dEPfBzv3mjG5DkreDP39n1rRrGaxsruvBkTOawvfRVWmplTcanWWb
  ucR02r0zPBf/2gBOCtqi7aTHTIy2rFpXaqlwIkcjDRKudUkoPb+SHRIam3vznRGOu6c99dIE
  IK3phreR/TxDp53LHC0ldGS/thzFqnQZVG4BxOAMQDSepHQLfi2oSSi6sh5sCjLgKocP0HrZ
  xHQ87CWoatt4rLTnImhySIeCVk8rAdFZLWRgTRIFkhnWWcFBZpHp1lyDejbAdd8JMYKRtppF
  ZL3l2F/73CsYtOQZhIJtBxcLPlpnC2nnzb1vs4uAPXdJnnhOxRgZl1kr4a/JZtS9q989ReZu
  uMI8LU/AhNDlu9k7XSCIDRGVoZA6iyYoYSkj5QD0wYR9qSICQIPHETUUCSy2JtyX/7iWc848
  2Zl/k5WG8lPeC9HVHIy9i80tDCJw2VceMo5Oi6YfLYM8brnPxOPVdedaFn3Ew3k7v20QA4Uk
  1dNknsIYGeVstFNCvCfbfxJIsAzgsNMb2cy3XL3ZlwTkvFbMnKnUiVgmyF6+oydHMS3McOwc
  lw6fVrl/pBki8OQ7AUS9mIG8Hh2J4G1GEd1cCRJZH4GohizRmc4SF/LO0TnjNM5uFtMcucvB
  FbVh5UmuFlF4Mf7pCA3vaVkgbh0SXwxPCOSFuwxhPzgiBU53CTLMGxHtKunQG1HgbIbmHYHx
  Zp7gGkK0FvNUwnZQZ8rxJ/02KXsSbePPEEntLCEZxNh0mv9X6/Q6hzY/xlbjehId8YtOX9m5
  7v0W6e4O1yxM3AXfEZsK768B9Jm29NVVWkneNs24mga2wWmpQaCp8iYt6Y+1KCBjZfzbsAq7
  OySFVNwj0gwi8bKo0RAm7INF7PnU6lSG7LQY9cuewU61NRYTvANE2DmmR5aDEl1b7h18YHKf
  J2uieY7bw06EWaN5iZgW8VuuEAlUhxUIj4TfX6uoZl1I+HzXfcbObKXKwUE2pjd4JguclvfM
  wrSUEvX9ci6RxQUUeUvnWQupHZOfECh4Gzw5iEjE9c6r2CZh8pzJ0dCcKWgY7gNdJtlx4E4l
  27sc+kGpzR3YShYfsc+4npNkVan/b35rmfuJF2OeqCKaU/7l//jIyyYukbdxnSnVzi0dhOik
  D7b4HSK6LKeRvpqEBfwGofwuqRIAIQDAQIIEEzggkFDhwoUGBCQoGFHiAQgVAxAsoIDhRo4d
  Ow4oIBFAgwcBFAyomLKigQABFogEoNHjTJo1FUKEmVNnwQQDbP6kyaDBTqJFcxoAOhPBSwAC
  UjoIaRTAgpQQDViUOjBASgUDDyYF2xGiAJQqUzJgiiDsWo8ILhokiLDjAa8IzN7Fa/btSLls
  bQbA2UABYJxEIR5wSPCAWr+NEw4km1fyZLoA+jq2OXTqZM6d9RJkjFkBTgWeJz8YuICB6bxQ
  U2MOG7Uo/wPYft0WLoi0I2AAuLNSrFi4gO7aHEcTFOAAgl3PdhkcKLy4+Nqs1QEQn95RoHXu
  IkM7Pt4zpYGswy0eQIk2q9OKDjQLyF6za9Oyd9Vbjh/WwXbk3xUiqCwB4FhTibkHcBouv4WW
  gou7whrwT8GZUluNQMkYQA0/CRk6roEKLQSRLgGLY2C7BD4k0C4EMhQARQtLHAjCDY3zTSfs
  ZvTIAab6m6u7ggYsKMEZdxzQQgMQjBBHhmr0UaIbN1RAtiatO6A4BnsrC4HjisqoIgXQg+C+
  ogRgDgKmCkhSwisDKNMsBMibijYlZ2LAgB0vW6iy3toEEauBCsAzPzF7E0CAAP8cMMAABxBd
  VFEDGhBAygZ8mtOj1Orrs8DKAlXzpQaUM41P1lxLIE22BjizIlE7AxAylFblDIHtAK1UoR3H
  lLPWjd7Uk1MCMpwSSIyejO+5P5UbwIAvC2W22UOvurIASnVlaMqdiFWwVWurO3G6+9DkyqgE
  roKgpdUG4G+nFlU9E9sNcVJOVLsGwClXahliQIEFFtjIAX0N2ipTlYClFUcGohLAgAEwzWth
  B6Abadp7FaJQYDcz9FXBt6iyOFYIWIwPTgAeEHhjWDvb64F7pSxK4okTyveABfzzF9hgUwpS
  phlhBKCABxoogOWIEgha5j9dnpjJba+rdb7qgIaUuwb/4nNAsTKdhikBSMkNIMufhY5ItZT2
  4lfXqpty0SxjrzOVWgT8U6AB4UjueLm9Ct75JWnvKpO5NgcYSsaXKY6z7pQ2rfXInk8OkwEG
  BlAgcskdcDxts15KQOXstAWAzVgdRzTyAGY1KXLKGWB8JZySo3YA6xRoW9e3v3tUae4GFM7d
  6Tj3UfDBCVgaJt1DBjtISA9A/gEHBlg+AOSRB9r2garcHNgHyuQNowIWQF7hAQz0m4Hlnw8a
  N7HbQzhjBXkmd7KzPfx9pr2ITthwP/lS0r2R+kRgqAR0Htyl7LeigSBtQ+jqDahU8jh/LUAA
  BxBaAhy4gEMNIG2A65n6GqMn/8+5SXwOcMkCNKOTAgiAe4r6Hl4csDp7zUlW1pFW/DyCtaUJ
  61j50wzRErBDHvbQhz3USvwQIL0paTA7PItI0EhiAMvhBQEgFGEBavSV/FgPe0RrwAEWJZkA
  NHE5yopUAtaluhgZcEYZKsBqUseUAMiQI/Prmv1SAizp4Eh/BcgUXYQ0ODiNsW4E1FClRBSw
  5QzAJWADokTEqLwyuQY+EtIT3VQFQkTusDdesaRIHsDERiKnhXPaS3Wu50YOBc8gpVlOYcik
  JAzGRHKvhGUsE/O/+J3NlAUBII4AickFOOBkKlqOZAwpxYJMTUJ0VIkBfqmiOAJTMsp8SmA+
  aUectP+PMyvsTS5lOD8FyhEBBdEcjt7SgDxOL34ZOp/hzkaWWl0kAVuBHCIJ1ZKSQPCSIjkU
  BB41lRlFEgIOsBlPphKAkpSkJQQlqFdEghh9BiZ2CgKJj7QZPxou7VPt+dGcjoPHPqHznLeM
  CDsrJTLlxcoAD0DpAxTgxYrEM0ZnVIxnHlZQkqS0JDKVZuuYAiHWaKaObjRAVA4ixwKBc056
  RCWI9GfMwW2KqBC4VK1syaQHLOB7s/Pg21A6xT8NDzZ6qpEAUIq6t/GtQAiIGcQECpmHCupW
  1pkZKQkQUZBOajk2ox6O0JmpjX60KQ84QFVTugCb2pSwhQ3sYSsj0jlhEwD/q4qc3O45tN6Y
  Cy8C2eMxY2oWEIanKIeJnIscCz9qAYul9gmJBEn5wgQ+NSXfJEg4D+g/lSogUbbFbaKUtVtl
  Re4BqfUqTHsGtQIQ17jFRe5xjZuathYHgRFpQNfIGioELKx2SQwuZgIqRoVN10JvcxyqWNai
  iQWUO3GVIWvrCiq8stJ/C8gtb3lr291G7qTA/d1blDk7/vbXv/+d3VimKSGelWmYxJQK0Uro
  y5ToLbuNcQtGvqcjKRJRJ1jkpFpJq6sDaQVEiuOnDLdzUdeqyqg4WgBOFNzD8rG4wisOzOBc
  A9JT1gpVXhHM8laVugUiKsVdnVOHr7MwyfiNoAzD/wsDo2LViWVvSuj93VtNadddAiCvGxIy
  jH+4ZRj/aXB07abhIPJOGzPFggdowAh9VFwBlMbBSrpNjIBmmJKoOWslRI4ZhftOvuLkpxNz
  3UAk+VQV4dKOIH3w7h7AnzRDKs0CeHSkHT1pSE/a0Qhi05yeS+QCqRClSFZVj0OCtxnR1U2t
  GTHsnGiWDE1USU62FpQBbWEf8fljJ96QY4Pn6hm5ZYRdW1iwhb08YRfb2MxTcRfndJ80FyW6
  AbAzTEoIkYwsm9bIgeYA1GoU2TY2Q5mzWJSmZ8QNbScyJTaxVxLNlodxj3vIQ9AB0pVF/iDP
  3cjjtYS0jZFnMcpR/gb4v/8F7u/RKUbP3tpOfWD1pvestDNCGcm6/eI0a6owAFLKHDSFuZ2y
  STWUNZT4dNJFY6rskpYzepjM7C3kothb5QfI9/oCWsJGW9rSla45ziMdES0qidljApUB7LkT
  YmZ2Q4qLmk5IwjBoSw9SEOm4ppNdty2x7V4MiEsw0Q3bgcRcQtSOV27CpJmDN5bRp7XYQ/7k
  dcw8d2Rhfq2dfJMwWA0TyEpy2riaeCWJpNNN+oOMruhKY9jdi+s0nt5dC9LGlUmF3CgfOeKT
  yPgN3SdfkfdKm/3WHj1JZHsMMrqEyPNOABWvc60x/acwyFRdZs/WdQNx5gZ89OmBmqiH7/aG
  Znf/MK3YxZanf5NXfDk7ajk2c7Yf4MUJwvbGILHromqlRDx0anoF6fGY+bhdVSjZIBFSbW99
  JJyVL3kBzV70kidIydvrwtmZXiTj4m/r3I/+mJQ6LRXZNkbg3lKwpbEib1YSOKEbt0uiFFog
  89qsfWI93QOWBtA4deKPVaoUbUE+b8K1A0Ko3xI+CLgxgrAr3uuZA8jADCwolGousCgM70O3
  9igM82u7kfufvDAvv0uJ6rM+OOu86fGioPqTDsKLikoj8aM/T+KwIQSUW1MMn1MAmrIOASEo
  lHrClHrCE0yK+RvCsnOM+yiTHNwsldC1IPkQvWE+xwAkQgIxkUiOisAQ/9vhmOXQIyz8qvTj
  MRCJMLaqlEDzvzkUGNzDjNrhMh5SpLKIqGm7DlWJtgsjmkyrDYhwwBVMMqZwwSxMF2k5mbWB
  P2HqvNDLj+jrmS6KlaqqwC9Ss3FREgSUPN8xmCEUtOXIKMe4rjEbs964NkKstU6kQpqwQvqD
  Q7/Qwn/CvFNqj1/0ClQCwFITiARQQ1iDiU06xLApC9Q4OSgZswd8KvHRDMrDkUBbDEc0xNhy
  DJYziu3Jkso4qcN4raFLsKhzjJA4N24MEzcciFsEixl7J7TTknyKlbiJketrjP3oQdRpDj2s
  k1kZQ784Q1UcmUrBOlUkJwioN8dQRomKyO6QQP/MaEaEBIBIbAy6yhcrFJCfMYyZ0Rt+9AsE
  KpWLo7VMIooSYgB3Ksi1uBL2cMTKQMUDYi53REL68IvscaBm8UmY0y0LGpsYeZtjlCQG0q1F
  g7RmgRQTksM+hDq0u70MUcT8mA9KLLHnujLRY8RQ7Jh9S8ghwUiM0EjYODz6Uw09+ZSdvLaV
  RCsp8xFZY7e2RLyqDJkGAakdCgl1VBB6RL/CyL3siKhG5EYMIsX8GQgmwkmCIUmFSBnxcZjI
  ZJ4dGz/GA7EHsJzqOrZg25KtXIsOU0GchBPV0Cwrc8TjCD8sqyxH9L2QWMD44ESMDLm1+L0h
  7CIEmc2qsxbG+8In0y7/uqQxxtqccSKsgzLO4zwog0JO5EQpe3rN/HgLsVLO5aTO6mTOBRi1
  GTmOIjnNbIKzDMFJL6mxsGAAagvNh/sSq3HM1VE1IyE7iEzM8CyqqZDHn2AQ8VhBGKlISOqZ
  iisxfywA7QTObSGz1htLaQkQbAQKBMhFo4Ay3Swi8BhLmCDN/KBA1vSUsmwM1DAPbrxPsWyK
  3RNR8BpRtCpRsjJREYXGwCwWMyPRF03RGEXRGT3R2bHBplgL3piZFyUrFO0vHTEIz3Qy5aFR
  tHIcGH2bt/izsOANNgGwJ4XSKDXRs1nS4uBBcoLRIpXRGuVRCBgkHBGRleLSLc3SMd2nAN0Q
  /y5ESM90qwkdmah4zpqYyCaUSwiNy/pcTzcNGzytCdfAo8pBncox0kA9UkE11EIt1AjLHL2q
  rEGtLkf9HkGN1COdVNSpVNDxHz7NjD8BrOfx1E8F1VAV1eTJIZ9jincb1VRV1VAtJg21FU5d
  1edZNJ4ovI3QTZKIVZmZ1cVpjHHq1FwF1lGtqunR1JnAnAcK1mQNrCXDkWPMImUN1kWjthkR
  xiH0OTXFyCAEC7iEIbtUCDvlDjRpjIpy0wqND2hEVmjNVQ0MsU6JEXUN1h+jIgW5SPpLAFPV
  U4OQuAHtO/OrMh/R1HpFyBiKD/XCyJ6YkQa9JWrNVwDAwYb1TrBQ2P+cIK+ZMMXq2EXtgFiv
  yNik4Lux7Fj94FdrYdHamNiFNZhqVUU0TYqRDcG3sYkVUdmdcNWN2Ni4qKJ8Tc38cNkmmRGB
  RT8449aDjY1aOwDUiVloa5KQrZabZZr4+DiM3M/smLEJLVnYeJzN1Nqt5VpjW7auBduw1dqw
  ENtjQ9oFzdqyFba2U9u2Bdti7Yi0ddu5Dbadodu7tdu7xVuf09u+Zdq7PVv71My51VS/9Vu4
  5Qi5NVyx1T3FXVyuRVy5ktzJpdzKtdzLxdzM1dzN5dzO9dzPBd3QFd3RJd3SNd3TRd3UVd3V
  Zd3Wdd3Xhd3Yld3Zpd3atd3bxd3c1d3d5d0m3vXd3wXe4BXe4SXe4jXe40Xe5FXe5WXe5nXe
  54Xe6JXe6a3cgAAAOw==
}

#####
# Bauhaus based on Hartwig's 1924 design: Made by Sandra Snan with Inkscape.
lappend boardStyles Bauhaus
set pieceImageData(Bauhaus,25) {
R0lGODlhLAEZAKEBAAAAAP///////////yH+EUNyZWF0ZWQgd2l0aCBHSU1QACH5BAEKAAIALAAA
AAAsARkAAAL+lI+py+0Po5y02ouz3rz7D4biSJYmGaTqyrbIGsFHS9dvjat3jofADwwKh4hgxHgY
KpfFpRPYfDpDvNwu9WBdq9oZl7b9YkHSZ/T3EJ7L6iRbuX6jQWKXV7awG+rdPV93B2ZFJtdmQMSA
eFiItMg45whnRvc35tcXFpD5tal5qSfQ84nZ8QjlZhgHoPrGuhqZOgXb6FFpOXobCqpbmZkHahOK
m8thCjn7ikorYJzMrKjKeojsnMDTYOvpO6yQvRU4hgnGy504udCsAIesbrymPBeLeLr+K2r/h3/d
bfvdI94lIClXxxKka8cGnTt4zyTRmocEmrVB+uow4HKxX6D+YTrw8ApnZyA8iQSlmDOp0NQZamge
RoR4qqJIjhgz3uPXa+M1jzJEeTR4DuGjBihTDlXWkInLeEySYqOIM9/TcTalXqIJUhu5j+MAGa0n
lJEDWSePOmLJ1Bm9STGjUnU7M0awqnx2YPXkVdddkESDAm16gezXQlHQvmqblm21TsTAxYVwE64Y
u1xD3gK0L+9fAJw7c+7r2XOG0KJBkz4dunDS02dRGW4t8ydjDPti1/xUWQtPzMFkj0xdljQG1J1N
E0e98viq0pFYNqe7a7aF2xNz4s6dVavA7Y1fBwdeQfnnweKLI11dmvlqduuhx4UqYbJkTuCwawLY
M79v7+TWzVMoP95mAPq3HnHtzQILgu7tZ59WcvXx1l43QYgHfvqV0x+BAoInAWvslXTcOwmOyJ55
Hn4AXxa7ROiNYw5iiN0eHPlg4AQnfmhYiL+RJw0z1FARmQMRwtgiTS5eNleQG+h4hHBhBYjegK78
BmWNUULZQYpTaRYGY7U1+KUVWmrAJARODgailVeWR9Cai20wpj4q5lUkmFDVWYqaaaj3VZrIUckm
oOIBORdkXOLj5Z0aSVhonn82iWVZft444JluKndCpppuymmnnn4KaqiijkpqqRkUAAA7
}


set pieceImageData(Bauhaus,30) {
R0lGODlhaAEeAKEBAAAAAP///////////yH5BAEKAAIALAAAAABoAR4AAAL+lI+py+0Po5y02ouz
3rz7D4biSJbmiabqE7TuC8cwMk+1Ied6S+/+2/v9TICi8YhMIhHLSdOgjEqLzKn1WL1eTUJh0CWJ
fbu5MVl8OO+I2u0hKYG/29YsPWq/y0lqnTkAIfPXB5hGiIZzeDMC0Oj4CBkJmdcIIUkpmTk5p9m5
CeUZWlmiuFjWcJpYymO4WqjqJzBUIiqKCeCQeVv7uMs7CtoZrMnl+iqbmhALWzoI1prs48xK8ivM
GclAjG39KeDpi8udPe4tYnyMLKicrL466BAdy7pG2325q7Bdfn+biz+M3KhrDLrEM6ZgmbtFzBT9
gUZtXcNnQBZW1GarwT3+gPzEBeyVYCO5jx5HctwGct8Cg6gQJjylEKJDmRMpmlqHpl3IjBhFOlqA
Upc+nz/HkSw68hs+gRxfziro0ikZqK7G1Kx4swZMREB5diXqceevnkTzHE0JkuRRtS2frowq1S27
qhCv2nzW0G5ejV6HguVbC3BZbmeRplWKVmjSuDHnvmsrDXKzugaz0ovMUCw4wT7/hfI8OCBicN42
fR4dlrHOaWdYRJY8E5ZdrHjdzb6bTjPBr6HJCuUs0k7hxGGJXzusWuLbehyYwz705fZlauqk3wW9
269KDdl5d8Y23DAwxOETH5TrePUF51Qfy7a4lTruWbW1dw+XOsN93cH+waMOOp54GRXVHnvpKZcB
egdCB411773H0kXebcffYhicBtxGwv2nGIGjlScOcgsiOKKEGLBUYGwLwSeIZYXQZ2KFTckoogWB
ZdgPYYGVZtxSNcqk3m0btJYig8wIuSJNLE5XH43mODneBWP51p9oHJp2GFMd/ujgc5lR0EeRhFi1
5HwPsTgRdhTix501VGpI2JXGYbIWh+e9tlxjNsTHFZBjKlkPV3wyGSN+UUKZnxNbWiinm0Y1+pE/
/9UpZp8lNhmGPCSic6ZFQDoDYZCNcjlqohGoRCFYAZr0yUmLnlWMgRE0FhOnft4pYaAv6vkBhhSk
+huIvgZrn4fHhbic5gcKukbiVaxN5WdteE4ra6+b/UpsR4Z6de2o4XxLCp6ZWgpPtGo8i2KZhDZr
7X5qHiqpsN2qWtxmvp4j7qzkLoiuF+YSad2yHtx7KqMVbjusqgh3i2+1uOaWYr9u2RpwviAQbMmT
VC68G70cJ+uBwJKBmStcFKsLbTUMF+xhxsYq3JGj8rq7Qs0234xzzjrvzHPPPv8MdNBCb1AAADs=
}

set pieceImageData(Bauhaus,35) {
R0lGODlhpAEjAKEBAAAAAP///////////yH5BAEKAAIALAAAAACkASMAAAL+lI+py+0Po5y02ouz
3rz7D4biSJbmiabqyrZUAMfyTNcxQl/5Yfe+jPsJa8GhEbYCKJfMpvO5RDgv0wP0imVKs9zntgtW
ro7kIvC1M5CP5rWv7bYlw+GvluK10sH2PbbvBzUW9wOHJCHHQ/imuJio5viIEsgFKCYhqEd5ZbkZ
pelZlRIp2RNh2kh6Bqm6KsDGphKaacAJYVs7Kyqgu1tXp9I6Y3jjwMgqHEAcuTyMfDzZe5dLy/AH
Kt0Zqt2EjYuSXPycplCYKtx8mCCUrj5OJCv9SV2dkMW9iQ+wUOmd57/rRDh3r6CdI1cwXLMH0Nip
OYjQhLx59P7hszdxH8D+Sw2uVfynkdc9Y2VIKlxn0Fy7OMsgQnSnUpxKa3QeZAy5kePHbhgnavO3
8SK1nDgXrGE40OixmS+ZvWyqLKGzdzJXGVSwx2ZGfh5H9pTXKedOnF2neWRw1GQytI7YnnxH1Rm8
uC7pOsh6dyvXfHvBAhU50iIukPXKlWwwMKpSlm7XHoR6htwjVFJd9e2i1SfNQB31fiT6yRfh0TwR
H26MznRa1K3gxN0hOU3MiF8x59W8uWZn3IBD8+mtk7RwnYuNIH3Lmqlh5HJLTq2seMjj2793+7Vu
G3uvPmNtlQYeMnvA5cbVpjZ/lfz5yq/hPZ8aiz317LmzZe43fzu27pn+vpcGBtxx5ammHAfSQZBY
G+3B5opVh1mGFQASTkhhhUVFaKGFImSo4S0cfgjihNyBF2JFdoBWISgCHkhgehuwiJ4qCkJnznvq
MJbQfSFKqOOHG+4YwY5CdmgiiSAWORR/PibZokNNSuIBjE+SMmNaNt74oDj1AZkfhz+W6OGQYo4o
Jo8BqqgkkRSu6GRyc0U54JROPbNgc1rCN+Bzl3GpnZcgjNljmUsWKSiSZxqZ4X4xugiZlgaeVtx6
dpZ3ZVWzQVgbn1v66UGhXQqaqKKepugNiqUuSpmbtGGAY6StTRdLbHhS9uaeYPYZagegXqjProCk
KSJ4lhD1l5xQqpf+qgaLGLvsdDTKVqmDcx2b6a2b5rrBrmbaqu2apyLqrbAABTUWmwWupJiybbkq
Y10HvkkrtLViOGSg2GbQ7bbV5qtvqELu9BPA3ZnLaJ3psrpUegk+ZdezC9H5mr2DXkskvkcqyS2/
YQ1crlhIXszsqo0SZAFTyi3c8LSlJBxdweDeSy/IGFj7csz8BssxieRizCTNJsRZssIpJQUVwTDF
1Ogcmlrgc829gsqNxGTdGl7TJACNRrLpoLvas5i2KaWUEv07s8z74vyytvpkjDbZpgYj9gQuN0S0
wV3nGTcJ9ZbNKcVT30wqsFCjWPXSP+d9CrXI3lm31+va7bieYxt0PsHEuP4NeNuZh5u2iICOgjiC
807J9WkoR06I0lYHCTOuT3sqOOydDw56mxWMHmPp8aHe6ukn7E1FxawHPrvsm3N+PNoChc6mDtG+
arfpjf/uNtPEV0588v1q/3qZLnwPfvjij09++eafj3766q/PfvvmFwAAOw==
}

set pieceImageData(Bauhaus,40) {
R0lGODlh4AEoAKEBAAAAAP///////////yH+EUNyZWF0ZWQgd2l0aCBHSU1QACH5BAEKAAIALAAA
AADgASgAAAL+lI+py+0Po5y02ouz3rz7D4biSJbmiabqyrbuC8fyTNf2jed4wPf+DwwCFUFMMSFM
Ko+IpXNJfEqHMYD1is1qt1rFFvNNcMfkMKKMLnvT7G5sCu9FqRThPC694536vVLWFni15kbBRSjI
hpiItshIJuOXh2Q3kdQnyXSQyUfJeVn1mOY4ODFGKlp4kNooxgr59gnaNPvw5ylLt5mracDb2/IK
e3YaMbwqfOiarGrAbPbyq+t76wCFKx2AybtdCQf4nIVa6qC2HG41LqzuFhiZLYcNTFu9C68tf/ud
X0n9BI4u3TllDFoRCwiAnTgB7gYea/PuXjcfDP7xyzWRh7X+a/Q42vPoAqHAg8dIFjMZTmE5cygJ
IjMY7R6+ixRpxrPJKSMlmxk1UvtYK6TIBaMcQnsZkB0ElkhPjiwKAY8EmTNxVrXYM1O3jjX9TRPQ
CezNsA8AmD2LNu1ZCWrbun2LtiBcsy3Vyp2L9+24unUTGk2LtGnbqXEIS6zYCWtWSduAMunn+Ifj
yF3L5o1r7LJmzETxOvN8d7NmVILtfjbdl+7ptasHR5D6WubGT7PhNRabGFi1fvVWbmYr+neD4G59
EwfNl2He1qxTl/arvHjsfVFl1/ZjyzpXn/t0854VdLho4MeRhz5uufxcUszhtt/bvrne6VMM286O
Hf/97ZT+v1f22lVYkBl3GXnqzUfgeOkdKB1KzzWI2nMPyudaddTplw19FzZAVR8AWvTVY1jN05lw
SzGIYIIFnohihYLF5150zZU2IWcuXqcYjr/YN4mF+02GW31B+sTfh7tJlpmJC7bI2ZLLJclkk8zB
OF+EU1KJmpUcFqahNJYI2eWO8hh5TYgBcjmkgSuyGOWMTsYIZZvQvSjjiloChmWFePrYI4ZefkkW
n39y1R94/wVZ6JFEvrlenG0a8qSajw5Uo55SylgphG5uuSGnfYbwqZ9iEkqmoot6laihp4qnpIpw
irCmo1EukumeewaWp6aqCRqopzl+0KmvgwIJm5ljoYn+JJuxupoiCK0yi6IjueJ5p40K1snjr4iB
CQJsvHIz5pAgJgukuKauel6kjDbrQXCSzurQtJhVa+2aWm7LrY69dZCfqOAWWSy5RqZKJIkK3Vvi
sh2gp+ykDtb5JL31NgrxnPrui6+2GmjlL0bhmqvqTZSlOrKsFK/bYLvlNQzvwxUjKPHEut56cXgZ
97pxTjXTlk+pR3gnon9fpWsetDduwCDKSRslr19Hwwz10TfjvACaHHg89b8lTzIgyAUHbfBfrxqN
8AUtkq0eaS+bljKVErodJkjCym1EbnT7zPN2X58rMt7l3n0wzawqDAZo6ood7V+Vvp0adFmOPTfg
Vh3+eoFiGnd4kd5XdS3gsRgnTDjoRRd+8uG1MtxXcjCqtNriseRbea/aYq5531XvO+JMGqtwbQam
m7524o2zfGmkuwK/gtWx3/0r7SRHfmioobLQu9mj5xo46q4TLbjCoaugvAVU410V+RzX/jPs3MJO
/bMVXL/96cRl7/RoiN9qPwzhVyC535P3SzDG/A8fwVpB9SxwMu5tSk7FY+Cu5Ne0iEyvDja7Hee0
E0AAmk+AMDjg+9ilQIsFb2mNOxD93PW6CQJqaNBbjAa9JgsX7gEg33vXA4knQgfecIQmvF8P9VfA
FVLOXzL0VgZnOEAjCiV/pBMc8U54QB3mUIcS3F06tsonxFU5b4Mv3GIw3AcpJ8oKis+SIhmZqIM0
qnGNbGyjG98IxzjKcY50rKMd74jHPOpxj3zsIwYKAAA7
}

set pieceImageData(Bauhaus,45) {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=
}

set pieceImageData(Bauhaus,50) {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}

set pieceImageData(Bauhaus,55) {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}

set pieceImageData(Bauhaus,60) {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}

set pieceImageData(Bauhaus,65) {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}

set pieceImageData(Bauhaus,70) {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}

set pieceImageData(Bauhaus,80) {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}

####################
# Eboard:

lappend boardStyles Eboard
set pieceImageData(Eboard,25) {
  R0lGODlhLAEZAKUuAAQFAxASDB8aDyIiHC8mGjUzKUI2IlQ7JUBCQGY9KVtILWpGKlBTUHpI
  K2ZWPm9YKV5aWmJiXoRnMW1rYoh0RnJ6cp52MoSFf62JQ6iMWpKTkracXaCgmcKkY7Wphquw
  q82tbdKydtyucc67kbzAvOe5fN/GlOXNndDT0O/Vo+7hv/Xu0v742v768v//////////////
  /////////////////////////////////////////////////////////yH5BAEKAD8ALAAA
  AAAsARkAAAb+wJ9wSCwaj8ikskgwCJ3LqNQIgA4NgKl2y+1KBUOwdzwEZIsAAnnNVrN/I4Xw
  co5SQMOUqbOhCAkTbkoAKQ9FFyp1XwIEjQKKb1qCZAaJPwEoVksBEwhCERcRERBDBQABURwc
  mgAoqFtmsZBTAwYOBANbagQbkQ5iWFMZJUMsKSEgfj8DGZNIABsORQoUs0oEKikYICvWUgED
  Aq9TuwWRAWfOSpzmPxMaoqRCDKdRo2gV41MAJC3d3koMeMgWQtkUOgJS6BuT5o+WDCaGrDiW
  7A8FdUfMoFkYhZeKECoucJSCoMIEBgZGIjElQGSjNXUAGmH3Cd49IQg0blpIQCb+EhKtAGjw
  eSQDCxYTQ8jZp4LAiAUbVoTIEEDTFA4TfiiQQFQIBxVHV2QD0aFDBgwUJGA0cmDpkGZcBGw4
  MUJFtS4BPqD4cAGCqX0TAmgIgKtAhCpcAkDIAuCDypkVNJD4oCFUBAYICoSrt4SAJyIAKnRR
  wYCwpS0CTKxYnaLDg1xTIhDIYHbDBhWurXY2VmXFCTFLMoA9quJECBFmKVB4sPaQQSEmdDMB
  /oMAiBQUoip6uaTAh8l9Czw2MoAdLnEIZGsRAEHDhFMXKlBPEiDy5MqiMGt+NF4IADpEIHBB
  FwgM0EKBXRExAAgqEJSBA7BFAcAB/rAAlgkgWMACBgf+TKFAgyqU8AAFKoxAVAZ1ZWNCCHxk
  kNYDCkQ4hFtDjGDiEAGkIIERnwHQwQoWXrDLBCMqxxgFq/k2XxEFVHBBKAwUsOQzoGhgEgSj
  EBBABDIqUQEJVkJQ3gUkfMPAO/Fdpl956ECygFWYUMeBaEWIIRgJeOYjxAAQYJnTDwBEgMKg
  QyUx2wgmeODiAdIdQSGQYlmI2wYmlECUAasVZ9yHJhiySQYbtLjcAg4ccEAjcJ6wUAopAJej
  BUUEsAEYB1iYAlgnaEnFCSqIBeQGjwnAQH4F9jcEAxx8wMGTE1zmVwB/LcEPCU/2GQACHBhb
  HQIMRKlZAOC2OQtmmgxAmhv+BKAwIBEGkOCJoCSoMhkAS7aCgpUckBDBYwMoJ0EDB3RJXwst
  9NoCqymcUEIHGIRQwpQBZmrCBJUacMIGjTKRgYsSKGBqIyMZoAAkq+34gwMelJABGowG4Bta
  ElhQQgo0/lDJCS5SAFEhShTgs5TaXqLsk05WEMFJDOACYBIATLDXBYNlBgAEQH1TQGaalRcL
  RuAWUYEKFZyBydKlnHKvAAKUhy0Ddaqggc9RRiYwEY04okUBLbBAmAMTpVDCihtY0IHJSBCg
  mgomnHBBpRZEMPhBG1xAgQMec4eEAZP4swFjvq08BCB/rDCCAyNcTIHfwBFilwQbjMCHCoQf
  AQ7+2kH/gO0FAxj2ZH6YEXC1EgJosNdkyiIAzgftRDFAt5kRIA5h9CmiQUhuOE02oFkIoAIE
  AeQ5AAKFCtH9B9+rcpmQRwiQQQr+AN6xABkDepQBGEiwQuIghOpwhgoZ2uAKHJjYBEClALN8
  qldiaY2LYOS8kQjAA2IQgD8G17QWlGBWQnigG1w2ggdkg1V3kAAqZnMCnWXjBHw4Qc0yWAAN
  fOADFYBAlAagEyQIRiTDgtoFmoWA5gWgNEnAVrzAlzvNCGVdTEPAOzRgpcvsB1w1xFFghkCC
  ey1mACigVh0w4x9qESBeqjBe+EAxgAu8kANYWswRDGAACnBgY8vB3Nz+iJAkEN3KBBsAVQhO
  AIKGMedyCDyBCQ7ggRQkA1RRqIoDKMcoRzwPCQIwCyoC4A8QUMUELcAjGAJwAQy8wgAmwA6v
  ToiBDXQoADiTwAdxxiKIQauHmdGaT8z4JKjhZxQI6JOUBICAtSArXmY0jF/+g8QkiCNtaGMT
  FAHgKSMEDxX2gsfUnqaRLdUhmEy8TwEmYA5+fE8yGuAAAS7wp8KdQFEwglgZOCCpvp1ABGfB
  gAkalD8JDBKSJbCjAkLJMGZIYQApQIoJFngqaSHuAA0gWIgo4A/jPCAAJVKEywpxAoSl4Cxy
  6VRFWdWBEKSgAUuIAAeipKtBkIkvRIuhfiD+sMMo/XCO4FMF1AQQAamdZArI4kBNv4WOJUiG
  e1msDAPMWBl0TOB6/MjWO5Q1lEBYSYhMLMl7rmGhqbxmCQ7IWzZKAIIQ5BEDD3BArxB3lsfJ
  rgO3ut9cUrABT3YoCgeAFB5d5LH+NIhVWrURCErHog6U4DlCAEkHkHSri3aAACq7TcJe14H+
  uHCG/REpEycwARmuqSXweJYBPjOTCDCxpRCQWpSkQAAIKGun4eipEgbgwu8sy2i1rMDVOgGJ
  L4lRMh/ITAA4wFowXcBngRhPBKqaAbUsoUETCJEFHCCB5p5qIA3CGQZGAFgiSGBmdrygBaKY
  BL4JNI8UqOsSkLT+h7A+QAIPAFgjHuAidVisNWjVg1kA8QAPJA5Dx2jmM144Qxoq4QJDjQ8D
  ktkm8NkEAbwsJmiaRDQnouMRUghUvJzYwP4AQIl+KYDvcMGmInLkPyPdZjgD8z1Ozikzw1zt
  CCrZh4DRRwUKkC4GroBKpETXkhwAwVoEIAEMnCUtzqWXFIyCFBS6aAEH6E8AHDBPghGMVXwU
  QeAgTIVQOWCo9jXBCMD7AAMI4AAYiN+e4jVDcdCnAunBEgGkYbNrXYAD7gmt2tYCgNz1kE2x
  2MeXsKIfLXVtEBXQiwqCGp9mDdMaF46AlNJjy8qEFh1QXIK5CubV8M5RCCdAAQUCcFj+IlCA
  BRWaiCWnCxBxNMACqFYLlZfwwLAYeTkHUKf4VuNkY1AglAozy0OfEYCB2Lg4lcqfbXoihQKQ
  WcPl8VI+LqwIAakizsWa4jPqPAA8J+gSZOJzLCO92u+gYNAkiOHRLoOgaZNpUFlc1g55qNr/
  EmfL4Y0fABbgH9CAGki9MmQGHjACdQpnNfOcyOYS6QEgWWgPeZRAqXCRBAOIMAMEWwEIHD4M
  EZmhUSYe6/YYIQGyDA7JEooAmDCT2kGcQRFCSRZ+QtsT/yJBpMqagEkQTFrPfidMKFGmTAaA
  iyqCyWfc6pZ4vHFhqlVRp6fYpih6uXMGGJwFr4bQpdPnAVD+Fyy6fJBAY5OgAAX4IwAHUBix
  lfAhgoVFBWTpQ8dO9RgRKCCgVi9OCUog5ZHRJwAk0ngHusqiDXDFJ5x07U7FwV1WI4uplQ2t
  eKpNH2xNQABNWjXTJlDFQVEGl98yw2PqHAFvg8do5C68f/6Dp9KniViizyCQKmShEkwlvLEm
  B1jyVqGESZl16sxjrBlBAAlkYOc/sjpSjiEqhRcUCbdp0ECVMwHlbGy7TBNrNkawbsrq7CwO
  kIlIvw0eJ+J5CgKYQLLg3Jc+9ZLbR9hdgezcnFKcO4uX1w9PU/+fCqDkeeFKGw2JPlQrkbTa
  dtY7ABFjqyEpxjFqDxB7iXQBoCb+fGKBIcVlAgDRCAowItg3dkgwABwQUHkzEQtjGxyzcEmQ
  ARxgPAcgAcqhHArgZZKHBssSI5v1AGGlALhgAKTyGD8kPMCEeanFGevAAK1VGVCiAAjSbs50
  Cp4hNUQxACKXbi0FS09khKCBYAFAeLFQhbJkQwZgCqcAgACIDtU2gCBxFHoQAiyyQDQoBaph
  dpkCddLFOuOQEsDRBAPkUcsXE5BAABt4FFWVduBlfM0BANlhATGDahaAAYfYVjgYAKzTVhjw
  iI94FooIENfyJXxxEj0ENA8WBUAoU2kiQ7GUZ6NnBVfoOz4Th4VHAJ23F1gRWpn4RKmHf+FS
  hbSINjT+lz6nIAu5GAuQl3q282aqkAGS4y8wooCDQGtIkSm+cTH1AyxDMBAngBUI8DENxEaM
  4gAX4AF8RB0aaAJhkQKuYxvgBWuBmEcbE0p6UAIjwCoDZ0MB4GNnwVXIIALIQHfB4I5JmDWy
  aAayBiigwCyVRS7nJ4oCUEUaAFm6qHmQxwAVsCzzsTwu5GjMgzXVhn4zgTaQB0uwxABqVDhX
  yHtXgzXcIoWRYARrtjEXsDGR6C/olWR/kDPXhwHZRzc6A2TNRQHAsX8JuZO+6B8GwF6gAl4H
  0BUXV4GUBQGM1IIleQSmWCA/w3PiIS5noHhXQ3JVGAZJxy3FshA82ZUy0YU5PlORWSgt1HY1
  oJh5S5mWf7CT6tATCVk7YyAucKmWbMBGm2WXeIkAdukfUql5ssOTdBkgUtmPbBAEADs=
}

set pieceImageData(Eboard,30) {
  R0lGODlhaAEeAKUuAAQFAxASDB8aDyIiHC8mGjUzKUI2IlQ7JUBCQGY9KVtILWpGKlBTUHpI
  K2ZWPm9YKV5aWmJiXoRnMW1rYoh0RnJ6cp52MoSFf62JQ6iMWpKTkracXaCgmcKkY7Wphquw
  q82tbdKydtyucc67kbzAvOe5fN/GlOXNndDT0O/Vo+7hv/Xu0v742v768v//////////////
  /////////////////////////////////////////////////////////yH5BAEKAD8ALAAA
  AABoAR4AAAb+wJ9wSCwaj8ikcnkUDJ3MqJQJMBgNgKl2y+16jwTrz0D4mpOArDF9bruR5ffw
  4MkCTOopxST+cU4hHRsBQxQOcVEOKQ9GFyoHXgIBBJQEAnlyZohvBiQQQhoafVIBEBWQPwAa
  FxEQCEMBAwCEUQAcHJipKBO5U2m/bG0EA19QB6iZE3g/Aim0UxkpfR8rKSEhzxsPm0sZJhhF
  BhsjEl4EHioZGR0qvZHFQgfPbwQTtBGUWwARE6gBHKxcwSowi9QHDfN+FNjlrtYHFShQBPsy
  gMIIDg6gbBFQhoAJbm7yJIwSbdoKQNiGZNjmS96aAw2ZEMig4sQGFbzaDGCAoMD+yFodIYA8
  E2zilAD8FAhRFfCVkFgFSVkqIsDnF14fJOakaIJFtRAUNPoyEQBAugCSYuobohbJhhQKDCA4
  VzMECAEHDASgwNKXWCGXvhDA0GHFCg0ZfmphoIFDBZ5/a/FSJWtA2ZC0Lm9BWmESAaatEHD8
  fEkxkqiwTEfhQCKNhnZmMhhmocLEBqVcLoAgMKKDohMYEo/iQiCFnQ5tjXBIsYKFVxW1QYDY
  kAEDBb5DkxzATSRD9igC1tX0gCG5kgAXSKDQEKGn+SKuAmhAMGknhCpfAjCIw+CC6iNIXcDa
  QRcExFMBlpT1HxEEIJALABV8xwQHEtkiUTEmNLdCTdr+EKOPBzMBtw4ILXTwwHtGEKDCCsiN
  cFJkSkTDQgvOqZBCCXZRd50EfUlxAQXz3DFcEgL8NVMKHGCgQh1FTCVFARqQQAIHEzBg1WYB
  BDBBBAMUYJkACESAIhECQPDBBQQxQEIFMCYR4ICrbAnBgQNYhhoVF3CDwAVjFsEnaxD2yWAJ
  K1YDwkoeagHAAS1ouKEgGbCQQZtK3EGbCg9MAB0IKGZwggrOeZWCCYGos+MhWnjQgVgCnFDO
  EgJQAMIIGbj0wzkUYEDdMj/EQkEHIHBABhMFTNAYK1YuuAYCxkbAgLPJcpSoFAcg8AFAXCJF
  JTJMlFIBB9fGGVpPA6Sl7FL+JDCgkQAFtgUABAVGQBBbFxjWmhqWRFAgfUwMsIEJKViTAQUK
  TAseo4YZxpwJJWCQAgYnDHmeB5eqEIIDjpiQmBQGXDBOb9QNzKMDChxDwEgU6HWECSZw90M0
  FDSBm6WhrjDCAUUCUCQlWPTqwWzP5bXETvw429O5T0TQWCgXTDDBnAgMQAADSPfKGkBPlxsB
  CQZX2iUDW/I0V7lp/TIkAFeSiUIFeQBAAp9GBDBvAFmhIOWUc6UCDBsBkADR3RxQOsQDGxSe
  gQSobkFAC6BuuCJ0JZSwAQYac7uEAwl/iuMBI5zQgcRwUJBByHyRfMwBCR4BwMMpQodBZic8
  DBL+rQAQwAFtIWzQQQospJDREQcwd2MIn65QguC9IoAAAwdKskUB4HKwygUVPM2AK+VGLQUA
  E0i5SrpUF5AeilUpb2UBBUiSJTBIXPBBLwWogNAQCKDgX9woKJ/VB4/198EHEPiJitZjJQRE
  SQNtGcADKMAj1HVtCQE4QaNWxLiApeAEJeiArhiGvKWcwDAqKMEKvLEBA5wABBRAUQAUoI4L
  IO50lAgA6AwQlzUYZhBPYUEJMtCLY+jFMB5YoAQk8Ba4/MUA0PHAwK6znEX0qwDkcp4W0EMg
  6lFvS9BK357MMwAKHaQCJADTZyCQro1AEX1eKptRigAVI1RAfk4hgAb+7BcZtM3CblQTwJci
  wAGqUUUFJIhAVXjCGA08sAhSo4QeuVAA5zjDEgGrzQlIBQIMSKADEkjOAZ6jAhdxIHIWiEDu
  OkgE9OjohagLwxJ6RqZGgcABQpgAC07gHTaiggAbGljnRpCrG6FuCMVRAZUoMI4OeEMFrxpa
  ubLEBWtpAAJdQkAFmtYK5pELim0hwNZQ8IG7AdAnA3CfhI5gHwQYII3rWwIALFOE13BATLqg
  Ixt7JgAVfEA/3ruPAQvAnb59IGp8lN6e8pQEAxjAAbfIwAQeoABLFIAUKGhBCgggAc7ZKAUi
  6EAISnBCT6kAdETgAAihYyMP5O4AgrCcmxT+MILHWWMD12HoyUwDAAxswgA02mEaTqDDDIxC
  AEy6FW3UcdF0YGCHcQAACIT5gGgEbIclcGJBw1SB9qCvLKTslfs6wyV9aUBOyhvbAOajTjBO
  aQIFSFfUZBGB90mhSMa6RdPoVDZKFcCPT3mIY8rCALtdQIrJ8xAA7DaACqAAXGeS2/xSAa6e
  uO8WHBBfBBQTAAS8BTqA2IAFHpAXUlLAOScgpggDlrvCgYBUG7gkD5fgIsOcIAIrUMAFPPcA
  B2gjCivMAAg2Op2BPWABJlNfEiYQAkRMgEa2IYNXbMMdoDYXOgO7aMDU8UonKGBJEqBJwDAI
  sRKAzgEAYFaBnkb+rvUlBwG3gMDTLsA0pxGSJ5ZpkFoAkJ4pFYgEy9tPeO8ZBb1AMb/KQ6d5
  F3WaZ6phsNxEEwCy8jZZLOV6+KIQT24xpb2CzQoDSBe7BgQQ8TFALQrAS14UKUOOPcdxmLUJ
  BghjgtposAMfYS1JobMCCgRMBBQYgIm0EIYRk1hBVDjBCMIAgBHQKAUofECjPJdJvVwEE7Ix
  wQM60LuLjs47AjjtdWxUE5u8ZVKwahA676QE9B4kFGhuGtTCVKVhaOkTbnKffdPDPCutkwNV
  C+80rbTM9XWQARy4D9oigi0CdBMgskiDeuehpr9OQHokOMhk2WY7hHwr0o0pFlqRV8/+WcI0
  xP3NAI24zDIcHcqS0QDVCUSQWgyIQC0UIBRJKeCBC2pQABMYJ4NoUqND8QV1pESiNRQwoxY0
  ijkbeukCWrqBvwSAHa7aQDUumALdGSIFW97uDkFQbV0r5EzljQIBKvA/6VUPi65AUD2o5yyf
  yC0m71raviCQbp0hcIoQYM0F0t3nBSHlfxNgMIG+Jb0L0IcBVWUmMNuqAY6wt5vs0ZIBcQ1x
  VgwgAhVw8BIO0JxqG8LbAWjBCNBxo+BQZ8WcbWmVaQWxDaiFADCmMbUlZwG0qLAwRz61BBpK
  SgKIMGHVtk5tFbAA0W3gGhv4JVVa/I2bpOCixqQAB/bwdFv+EyoELksCAchocHT2650FakqA
  LVPZUFQJAuDUwEORYICLh13NyiuNObUggLa+jd/qM88+BIStuXgJfctT3pw0TgT68q9LAvIe
  0dB3wH3vBK3miVTvJvcAkAKzpQLo3A5hghZKdNE5IMzgSkqQTAC5OqqYFb0EgEwKEMyINizf
  udJlEoIMNEAB2Q0BwwTxa0qQTDEHaDEtN6DECZgzajRUwAQuMLqnh+AByoLQ26KYzSpJswIJ
  3xvaKhCKCtB7LnJzymkKoF7sGz997NPCAAJ+Jryns1s7QR/CmRaB0HgpS6VpAnsr0JNnYW8W
  65RfdbI8n5EcAzACoqJBnHVIsGD+Ah4wW9WBAdyiJQlDUiVwKBagOxIiABKwYtYxREMEE2My
  ACUQKrVBHRIAXN7WKwjFO8ZmbL1zIyCQUbYXGE1AALrTDxjjASPQgx6gI5xlCQeAAQ5wLmM1
  fV53HsZXVfT2YbD0FOu3NFUFX3KDIEQSC/+VPmXxCys4BAUARu3XEzGkGVQAJeoBHX5lfvUX
  NWtUSmECTQNAVewVJ65wCbOwhVEwACpwbMa0cwxYCDUBJAewMUSQAsamITVxKBKAKW0hCQdg
  ARhgATySOlqgALxTI7YhdKijLAHQHC9IIyzgAU2EQbmzEqZRGjYGeo7DMjlSODbVhklQAN3U
  dXWyIP3+MVnhxU5EoDThsm9UKD6q8QuxUIvpNwUNkh5hiD5k9x5dJCUo8DcfUH9FMydeMl/h
  RSERYTckMD1O0254mAgrwIcDkziVMjlWQGYBMCPNQVLWsGIZMAJt8QAjgGzJVklVsxegQhsp
  MAJAWDIcQXseAALhSCMiGEEMow0lIyH0NY8bAkRkoAAhIAI0CIkq5SZqgoRk4y7pUwZk+BSN
  QSDVRAZZkjaqU1ihgDXJMiYnozT21R7l9X5Dg2bq8TZ2qB/fZwnzZUDOaDcOkgYYB1b8Ah4I
  +By+xlkdFBW5EAAmQCMgRBsmMB2SWFzdQDEtwAEIQCgPY3mlRAH5CHohBCn+fJGQ/6EA5RE8
  oChR0qF7gSCCsCIU7FAoKRBErpZRxiRc4lZY/0OL5qJOtJCUZnILGsCErjBTJLkG/mOVhWV8
  HTk09SUlXzQnZEB2MIkE69OYkUZNTcgAiVYpZcEaM4k31yONmgmLbCQbL7hcUbdzJ9MFQOUc
  h2gjUVV7FiAIbTEBkkcrHVACItAj5yFtxQaDhmJMH6gAPLcEueOCNHKIwzMdSdcWZaEkMwYd
  G2UXkPI65gE92TglVlWLi/lWhxkn1zOYWaI9aLAnB/E0/FCAUkAA9ZWdkkYndgKAlcIeFOaY
  q1BVRkMQ1ihnd7ON7FZNdtaIJaiOoYJBUcdZBpD+VcB0Aa/XKB13Qay2WSmAPChoCRxxANlV
  kUaAAIURKqDoYmAZlpuoBNK2jrZxHU5zHeqQAQrgLoqwIvuINSkqOupAhMkhPmdoN47hft0J
  K/5zZvvWhOA3mWtQIN73d1D0GeKGo+4JEFCjjHk3X+bHAG2HFviHFnVih5XSH91nJZ5XJ1Bk
  TUqqBCxVKM8hg8L5QgqqVcVGoBsySYqIAfC4BJRwAA4wAcFBhE6ySssRKqLylJCyRLK3mknw
  jrkWABgqAdfBQHFxMjaIBhwwApU3AAZgpwzlAA5EABhahBBULAIXmE/aZ2R2Hv0BLkwjdlFT
  FvOCBqUBJsqTaOYxDHb+JyXmZjRjpz49yhZod6V7g6V2qBYyJBppgKV1UqwKUo2X4wEfNGo3
  MgIgkJqYqqAUVSiNMkEhZKCReEn/cVAL9Ac2kjtaKQQIUGt+Whsgc3IMxFAjqnUDQJbVYQHw
  CokrOjAqcx70cQA0ajghMzoYsA2PSpnaxGDYciD352frKT7Rc5+t8KQnI59ocIdxuDxzkqtN
  ok2eeS1wF2ACRpqAgX/DmiUDlgZZqgRpEaW9qkcN2xYOAALFYyO71wFpGoRHQUEfuiExKDmR
  +ADOAAsUcAGjOawcoQCiI4mHAAwEoHyESj8dwFMmeIGCgK4imrSqoysZAK/GdLXqQB0iUHP+
  EBQARHRy81qjwWFMY+omU8N97OYKYyeZWYUUacYKYSM27uawvVIlsKp9eDusCHAfKbITUYJp
  U6ixXrKML+ereqRHxYog0WSFRAKAw5oGhyuZcUixTwGydwiymFskgMUEB0Cz4Zgw1ZBZK0ZE
  tMAAVEkbGPSsHUABnLcXHBAIurcivYN1LoNeLZVspBIIgbojKvgfM5EBkQMw0ekVYEYFBpB7
  IMBRVRdJTEa5vZK30JtVAwABRVN/3xdWCLKFq7pg3GRu5EWMwyo1VKUBX/RhTYKF1EsnX0o2
  lusuwxhNzJOZg5ccs/Al0WQ+5uMsslA1mRAADrCizJcrwbEjQ/SRACdCC7iyqCtKhGjxBARA
  oyD4AIoaFoUHvRbMsWwhQw8QHKOjI5w3BaVBQ3c6AQ4AAeq6mZmgD4CHpF5CAAUrjGrwfYDH
  PPeHCeELYH3WJBdswTJRv8q4TPHZhSJ7v2pbw/ybwpoAvUNRgHh7xMJgtEgcxV9gUFRsUOZU
  xVc8d6mwhdpHWTwsxUggrMCwpm4QBAA7
}
set pieceImageData(Eboard,35) {
  R0lGODlhpAEjAKUuAAQFAxASDB8aDyIiHC8mGjUzKUI2IlQ7JUBCQGY9KVtILWpGKlBTUHpI
  K2ZWPm9YKV5aWmJiXoRnMW1rYoh0RnJ6cp52MoSFf62JQ6iMWpKTkracXaCgmcKkY7Wphquw
  q82tbdKydtyucc67kbzAvOe5fN/GlOXNndDT0O/Vo+7hv/Xu0v742v768v//////////////
  /////////////////////////////////////////////////////////yH5BAEKAD8ALAAA
  AACkASMAAAb+wJ9wSCwaj8ikcslsOp/QqHRKrVqv2Kx2y+16v0gAISAMHABUAkJBLCAMhDi6
  LBCQowrFvRigsMGAWwuBQmNDAwVVAAUMA0MACAgFA5REAQCYUQAVDHtDAQETc4SkU5mBFypz
  BCp1UxkpJUQeKrElIXsEGRMEUQErJhhGBCYrHZ5TBBcZGM0WHqOlVKdgAioIQx8VAlQBEBOJ
  QgAaFxMREAxEkgDITBUqH+0DHCqOVwAR+RETAtFYAgocIOBmRYwQBxsIARAlLoO/JxlOyBpC
  60SIESL2DMjwoBcUABsyODiigIKEh1EIbGCxoVkKaFzqUArgcZrHAuEA9fuE0on+N3CfOFzI
  BwHbEAYE2KWcAOEhAAYR2klBtKJFiwrUsAygMMIDBwcEqbj6ccFETTDRpDqBNVHIhxUWQ+Aa
  QoBCRykHDBkRYwCLLhUnMGxQwXDLAAaRCqh1cooBhLNeqGX1tS/nuKHnjApBMEDpxwIEwg6x
  kxSLqA8tUCzsOdUEixW1QlAQrYnChAAZVGQIIGAMa0VEfi/ZEEuAATgVQYQAIeCAAQR17+Kl
  LaR5FgIYOqzYrmH3FgYaOGjgNFB4EkgDAoDvPNO8pj2XqgSIUAG6HHJEB8YJMKDf4vOeWTIZ
  FRyQAMk4qlyXwXavqWDCBgrYY4oKGBgwQgcOpAAXBrv+udcEASnMAUAHAULBQQoqsPAabA6C
  4GJIGVBgknRQHHDBH0XEmIUAGXSgwggneICBh0wEcAEJKHygQQSSSCjFQgMgoAECNFHi2IBT
  qFcTAxf8p8R8FWhAAglKXoAZOon1d4mXRxy4V1RZcKDaJXISqQQFxawI2AYP9FVFBHBExOEG
  HYDQQgcWEOCnWKwYg8kIcGVAHROwpKgiC7XcshyMMj5A4xMBnEABMgA8yCYRYxGhSwocUICB
  CjAVEZovBYhJJgcTMADaqUh0hoA5iMwkAAIVxFEQBB9cUAAmDJBQAa980GdrmeacgxhOapbo
  hDKQ/YAAB91OcQEAHLRgIFb+WWRQggrb1QICR4uaEkILK6wIWwchOeiAnUVM4JoKKnTwwAQA
  mzDBAVFElMKlr6Vgwi0dZBBjpw6EuwQAxtAmwAkdTGqEGRSAMEIGeXlEgAeubhCSCaMAsFWh
  uPIaQAEThIefrp1VMQAEFywJFQNAK8YfAvw+osEHQjGZXgQFFu1NBBd8gDQ51RYlCQHprSnF
  BCQwIJoAUeekRG88D6XYogBcsN0KBrYsANQXcOZhyOzWwrFI8WpygFVrp7gCCBtYgCIGZ5gy
  Qt0qlLABAScARmLCItedQiwh4Ctxpw/k4WQReY+2wgh5K5BCx0jEG0AHC2M6wgUmO8CABDLK
  KCL+BR70fYICFvNRANwXVFDUJEWLA7UGVJ9jbWgDRJDzVB+QIBRTDEAStWagIsBAzeRUYA7Q
  kkxiByhY/qAoEhqgEEE05JIgNhFwfGLrmOPt8RQE9EclYgTlo5CkBpsnYUCPIyiBBzZglwP0
  jzF7Y1C9UlQwfGHgYRbg1//ahaITmCADBojNA3zxgAwQigOceoADHKCAA5gwDsZBAgRGIBUC
  tCAFEqCeB1IAgnAFoAQb/AEARtAwFNXrBPtSixga1656sSAFnTtCHRAAAaIgpj+5U8JTCsQB
  ofRuAtBDR3qAp4nykYAcF6Af0X5FAmgdQg1MvJb3QAE+TIQPACgg2jD+PqCCcVkCBSSg3hAM
  gCw0AABJeKxi8z7QFI/xR39frMCRnHcqRXFoYg/ISxKbkIHXXGoFKKqFw0oAgmaIbAMeQ8Iv
  2nWCFWwgBKKCgEQ2EIBJIgFrDpDY5TJXwhLmxQ5KuAArj0CBFuimZXDB0BFwg7AfKMCXwZCA
  BE6JyYq1KYMO8gAz1IUiUDZBAG6IRJNAQQWmKQmMvdMeUYqCCPqE0ggGYMAgyYECbHotAFLi
  gB6bYABEPAcnk8iaGzGxmLQtLzgoUIEGzhKBJMlRVudAAwPgoSxQMOJo0WvTANiWK0ok72gR
  eIIBGkBC50TxYlVhgQcuwIEZTq4WJrAI4Jr+EYJiOoEDLFIBCFZQEhOEoIQm6IACijaAWEps
  Ah0t2ayWYIBLdGsDvuzAogzQgpxKYBg08VPtLrjMkEgAAyk4QQYgA4DcqKBVE6CALH0EApkN
  AGtqMqMlOKAkLGJRkeKMAPcKsERopQ2PSqpAHhPDDq4F7wczS0wcZnIJbbVpQHD8akR1iKQP
  LNYSy6MHLwRQAU4oBgLk8Mddu9Yb/E0AAUy7jRPOGgfkVaEALGgBCzoSmsmhKKUmGAHgLCCB
  DpxkW8VgEaRIWoISWMBCtj1nEhASEpOM8IRDvZgBHjICeoFgJEKYAAtOAIKn8kG0QvAAC0bG
  lROMgICoK0FeVIX+ohGYhAIbuJAHVzCX0fbne1VAANKUhRNimckc6EAMdBBx0CWExovNG5PU
  IKAYACALNKawXlEMgK3vGaRI6eFDQDlwPnHoz7H+MOoQ6AEB9cCvApQIUwFwBFgkgRi0VZwS
  eOCkBAMcwAFVjNEEModWX6BAtayL5OFcK4IOyKW3EmucK/nArpgC7CUhICAEOkA4KPCHAkWO
  jcoodgD4thgDZyGGakvgEEycgAWcxDKqPEBi7Y4sN641AQU6wOWaAAAEDpJRCVzL5Q38zUsM
  Bm0FmNQ98PkiahpghACS17NyZEawlJhSEyAhJ+eBo2tPTI8ANFAYJ9gBAT2rYjnQtMb+wp5z
  EZ1oA8A48CxmBfKxgGXiKAimvAqggK1VpFKt5MdWECMLaVVkRNzOuZURZNK7SbaACA0ILQpc
  qizqwmRWk6wyNpdABBuorUM+VGS4RGAFa7gAx7Csshw2IQAKmMAGQNBbwEFyAci1chgmEAI3
  h/RBRSXAa2w6pNGQORoUiBSUXXsCUWXguQRRAKwyIIE5T04igqFQFF22j/tmZhJt9NC3hALa
  CfQse1jUb5po0t/zMKBA8z2SGoDmGzNBq6jDigT3upfWiLu0TRpoyhAioIIkKQsTHwikzMXh
  GCcV4NWIqSL8OKCY6/lpAJwFzyCFUgDtfbQQzXEOHELDn0H+aLRBiANYv5PcjA4AiYZM7oBZ
  zAOirK/gABSYHLQP4EFvg8rFHg0N1Qv7dHF4V1EAYKpqaTiqB9BLIhjMewBGwAF/WMNBD+jA
  EdPswa3+QAAgULME4FzBE6gspdAlam+alC0ACLcI3/om8UZvaDQx0Rz7DcXOk6AeMYWcBAQA
  mq4uUVm1CmFnitQV1r5X2Cd8vJDj0F/SxNA88TTFjd+IzyMKdAEBTEA8zcNPANBFAPHM7Ghk
  Cg/NLoBqJZzsQTEq4eeD8wtM+dAEBy9BxJrxQIDJFF/LbCkTBLAu96NIAbm5W4RINpUQXCoF
  IYEBEmBLdTcEUGYCEnAADWAVvsT+d18GGIrzAOD2Iy8nBGx3Lw/QOK41OeilAMaRAhyBZq4V
  MZPjHaMlFNeSHkRCfNlnJmaiPbmiXztjcbkiCUNzQMERT5rWM8PyO52RK8FDM8zng7y3gmIS
  cwEwJs5jJlDBVtInBhb3T0OAAGMSFdgTfYFWAI8hACjgBlHzRT0TCeXATdeUIhZBQB74ERmg
  WpoUCxKxHMwggLCwNiEAbVclAtb1JSCQSQAjIzN0N2AhAQXYBiVwKTJVXJE0XhoVGxKTWnzj
  Q7ARC0zmAIdjTUZwAFq3TJHoWoRCAQSQITISAvzGZXtYAoL4BAWQLCmofE0wABUgNeJxRfpg
  NaBBABb+t2mzNzNe4jJhQjxmIgliRFcL0XGoWAFDmCYOBioRAIsQoAIChh/GKB6UhggMsGdk
  uFZK0huZ5jxLEgqQoAHOB3Jmojz00X1JQAAHUC8pkGQUUDEFSACG0hWa1BIqE4cFdDjsMjkj
  gwGWl4YtlgGiCDCY1G+x0BI7tRN48GV7Z24CWEKlsS0lsDbbwXV2IUIKsAAy0iNysQGKWAQC
  gCcOIhiupUkRYwAU0ComcGRZxWYqEAIYAC0EAAEkwH1NgksfQlK8sz3a1GlSgh/0oxgDQA7n
  tFz25YJMYTV2sAitJAXw9IVhhIx+9gT4kGmGBh35VF/c4xiN0BOYpg2IgFn+AcY/SBEBOHE0
  ycIUhwEO4WMEPKIiNNQSfTJ+4oMpAgAkzyYBbNQbg8UBKrJAksgRJTAC5mEAPjZnRyYR63cG
  hsUEaZdaqrN+A2hCZkQAIRB+DiBWIfAwIIAvdrEAcWAAJNRPB5CSWuVBKPNZz0EJBpAHl+lB
  xNGSffIRekWTa7QtGoAYlaU9RLNPbtR0vngORTEGjOAlQ7MPlVWDwuhGclkET2GWYZSCdXCN
  33YY0KFnvlg1LAcK/hEGYDMe2pQfncEOLqNgVBIlSLGWRkCJDbOZGBBJBYiJI6VV7JcWoSCR
  WsdJGWAB41aBSvQAFsB+MqJMygQWnvcKloIpOWX+Veh2AGYUADCWko5oFYsXC8qBLxbQAAiJ
  BL1BKEDFALTjASMQol1RXO1oAHVwABjgAGY0aGJCm+91Kk1HLMAZAQ9ZBvxRM5oGPZwRWGzi
  efVVk4XlRtOAWWTynFczd2a0CK+IJACDR9kTVwQWnksAT+dAV1pIHxdHNejgG+zQe0+Qdn9X
  OQLYkf6VAr7GJw/AJ2xZO760Nm8YOBc0ZKNhQnRKp6VVo1BQAAtiiIpjVXlQZR/hI1XBgBJ6
  AgenmZbDoMqlAKs5Q/XCIv3GST0WErgjBoMoDtU3kynYH+5BWVjEGdZjHaiCaVUkFL7zRPBU
  AevTJrwxaEXIm90AAeL+KFdvQAnJ+ASUVSABVXNjIh7HiV82KBzsYFHPJzUBNjWyOBBBCgUn
  86jrGDGRWXeDZwIiYBwHgJdE5ojtokkjIICiKKficABpSpBvSHCs6GS1Y4jf1RJ24QBkOn8M
  w4AsMBu80QAHx3Xt+XmQ4GsSCRtnVocgAG0wIgHL+QlUqKmcJ4Ws10REU094ClhQ44TjYS07
  GiXzxKqxpw9MUmB/BbGzqkZZc65MkKmAdGr8dKVyxWAKK1HYpz/6syQ48Q1xpV/8sjF9+SMR
  4wf+yAQAoIgD4gDa2iAD+S60NTLfdjgLhH7bwXf80gfp2pfPWlwSkDl5wR/0hJe4QagwtJr+
  GBAC5fYAbgSuRpIB9MAipmSiNOEiHbC2U4tuUkAAEVCk0GmTPHsKA1IrbEUtFDsG/HEq5JK3
  hjYUNvi2cIt9RaqjnRaVTIAIWCMngURfKqeVQmMekMBEY4JI9YEJaVktkfCwSnAYrmGI3iW1
  EYKDzKoiqoUpddNvFpodbsd6BHBMLbAbuCERFWMnAhehqdsilQNJBHgq1So6RsSAmdJb6gch
  IitKvAFl9hkwhSIXk5qA6Pkx3lSktPqimFCww1Ss0jgU9PM7oLAz/LkX8CQedDUf/DC9ezEB
  zeOyj0uxa+RGyfsmyRNgSNKC+MVp6osJmKaE95sscZUPT6S+ZYD+AAopoZiUU5BEbFYgANpV
  L3wDGym1jvnpI+P7MRNQSdslMWwmAhawL6BiZ5eiWqqFScuxfsaVB8mlBMymAvKKwJRTOYhS
  OEUCAPzYN+53C7ewmQH4kivINS7rPHsGpF4qFlwCi2D0vcGpnZFQJEdMafXDC/PrfYvkvtmn
  k9jSRlMcHONBUvaLrHsGv/ubNgXiv2DYO+N0LQRMf49KL6KrfswQmSaaBhfgiKnFIHZzAtAm
  bCHxbdAaGnWAjldVb06AANqBuqmLoDLMKRLQUXqhBHYmkA5DQDKCRRkpMRawU4uWIe7nXc/j
  VjLySM0Awk2wO3LSpEvoO9mpuFEgAEf+LDW/+b2IwbegAK5B2TtRigjZNJkFUMV4dCveq0ZY
  w0b7C1dMUk+8kZfha6v7yyWFJle1aFH8tXKeSxJAElINklWcFDExkojaWwZ1LKF3rCcmYALv
  clUm8LrciY4OEFYckqJ5QSQGcCKoC7XmvLYwEiON7JBskgGEpx/XCjuxE611kKFSxAGrgzuK
  8mKeEhB5oZoL0MinMjPggUh5Bb+7Bz62px5e/E33hQ4KEAkqyB9Pp9EpJ9JC+gRYs4z/66ty
  pXENRsw8WxR20Hn7xBvtob6ttA78NGjSbKuXECzu4QAesC6GSDmyxc1TK35MmZLbQS9VwSIO
  8zBMRlsEt2j+AfEAJ5lSsYEoRIIAM1TPBXMhKhOAAerIbII1dSExGGABbu3WohwSDlBU30Yl
  BNBBElOPjSdLMQIHBg27/durQzFXiVugHxGxU5M9TnQtXEoTPMtG5qlgiGEnceBNePU8ptc9
  naa+SRmeXcpGQcpPtjp/a5KUvCm/JK2CQy0yfLhJm7m2zNCu71okCzKollQ3mBKpLSFstiUr
  FMB9ns0OvVESYkVbcx3asQtUUmHIX+aYALPNZT1NJtEA/MyzGLBSEmABEQPb3MwhszXRASAB
  stQM4z3eXRcx1fwxijI8aJzZoBGd2ulkmEZ647g9MN1GA7Ao8FRRp93f/g1PBNb+QocRHr06
  sSuHT5tNdtpZ0P3x01cpWPMnnuLpabYasjzqHgiQASRVqho+MQG6AFQLqIH61FBNLwIJGKPb
  1svkdg4cUCcAVm/gYlMXyM5xAG9wmRfgAeX8XN1yABwQuiqCUoCj1x4+bI98HsuU1xlgUwdH
  QwE0ORDiHqBwVY2XHcohF8+rHM8WeN8GCmiUJi8K2sr5EZjmgtrjO7T6Bp0bpAWg3yTgjLeZ
  xaHNm9qJE9UYHl1yBFhz52Uy2CvXPWgl00XSqv3xo9okCeTpHknhaTity26A4NZDJVIO2pRe
  6ZSuCV5lSY+KOG/aDNkdL/uWVYY66hyTAGNhHOoSqRbu1FuT2jkDcAAB4VbnQEKs+adS59d0
  K0UAgH8uUgKGuoGTc28ryA5o50GFMgKxhewhijJV5iH+/ew3/Rmyp0369aMWtU8AIQ7XmZM4
  w4rC/ZUTECaCCx0PkRSoDe3+zRj8Yegrp1++UYD+MWj4pHL5RT8MUFQELA3igO7QThesc6fH
  MXW8wZaeh0KA3Bsn5JEFvfAL/14Mz/BOxk+fiUL8VAXm/tlinr36/h59i/EZf9qfcL4aTcwo
  gdoRl9JEMNcBHwkrfxwxHvAwfxxQ0A/o/s08wU84HXGXuvFZ0LEXw/NAH/RCPwxfsPNDL/RB
  AAA7
}

set pieceImageData(Eboard,40) {
  R0lGODlh4AEoAKUuAAQFAxASDB8aDyIiHC8mGjUzKUI2IlQ7JUBCQGY9KVtILWpGKlBTUHpI
  K2ZWPm9YKV5aWmJiXoRnMW1rYoh0RnJ6cp52MoSFf62JQ6iMWpKTkracXaCgmcKkY7Wphquw
  q82tbdKydtyucc67kbzAvOe5fN/GlOXNndDT0O/Vo+7hv/Xu0v742v768v//////////////
  /////////////////////////////////////////////////////////yH5BAEKAD8ALAAA
  AADgASgAAAb+wJ9wSCwaj8ikcslsOp/QqHRKrVqv2Kx2y+16v+DwE0AADAOHwJWAUAiGgALC
  QKiXiQJBQD0FKNxIAhQKZmKGhwuHRGR7QgEDBYVVAAIMDAZwCAgFBQOekgABAKNTZRWhSAET
  dYqtXpKKBgceCnAmE5hVFCAguT8BHCcmHR0bG3xCFBMKBFQOw7CLKhsHW6F2dnmur9sFJBCF
  ARoayFQBBRMV1UIAGhcTEREQCJLnouVjCCj0RwYRKBOiTRpFsGCYAwIGcIEloFkrAhNMOBgi
  IEUGh1QylCjhK8CHFSlKjAARotwGDA4wRslwIoTKIQYMjNggYQsBDyqKneyggoL+GAINrREJ
  8FLMgAmiHEW4YyXOhAnrfnHQAE8eP0cM7k1RRSLrEQEaSATUMuqDChQozpL6IsCAg5RvspAZ
  knIbgAlxfwDIgOCKRo4UcYYMAUKEyQzMqCjYkEGgIwoSfGEhsIFFCgwYQKTwwACMnoR6ivbB
  GMkVpbW/UJtD8DQqMA0V4s0DlRWVFDITBjj+wSDCVSyPvK1oseJUAS8QT6hQkSLEBgUKsVxw
  eCFELdMqVU/RaKKj4BKEDRPJQKEuFW1IHKDXQiBDCBUt4Tvw6SVAhAoVJjBAADRLAT77kWGH
  IXMNwZQVAbCmzhnjxGYVMgmKshsTB4wiml62ZTEAB/T+kIBCKChE0IUAI6jAwgorMGcCedFZ
  UYBFBlBwwgYLbDCCBw8okBRyXglxgAETOrGBCSlw8MAEEFSQAnMiFeZATDEpk5guuwFAn02Z
  gXDiCRRgcOUWBVTwAQkfcBDbJnlVQcAFAwTAQAUICIAABAx4siMXZcCyx4VPJIgfAxDQycAF
  78hDZwF57KFHhrcJsJtuXGy4wgejQJDWBF0QkAKKJ4JkwgYUtDjJBSBgYMAJHTiwgQorhIAB
  YkFupekyPxDSwQP4RBEMqyx0ulxIIHQAgjHkUQDZlFMccEGVX2bRXgcnpFCiB156MaiHZF7g
  GwKiUgHAPAMgcAECRBEACaD+kHLhJlMADJprFH5ewAEJJExFqIOCbgLUou86AcBvRURA1BYb
  opAVURygUEEXFHCKIkgzZiDZJB7QAd+rGRgDQgsqWHDgFQJkwGoKFcqUYgZprqQcCy30ygJz
  LYUgwrAZFCtBjnwyEcAJBOQKwAPHSLHvEe2lkIIEFKzqQbNCDCiFfWHR+8E7+/UMnCjiRjCA
  uQN8NkAFHzfFwdQBhcIAvf06Ee+89bpTqDz7ceIJv2qCfQQAHIRtxQDLDtBCC/TE1sUFD6PI
  ZM1RWdHQARwsJ0IILREDn6sIZRFACi9P84ABJixXwsBTbFCCiSyfCDN4xDDmJWQSIHubCoQY
  AcD+BRukvETPBBxwAO5EtLdixhlodkERFtYRgO1L2HfBB8ybua3eUuhxlLZyaMKJHvs1sgYC
  2VZggKMEaPCB3eextjyZ9r5Np75d063YuHd/AH0SoiwajQCIcvD3PgkWocceFoqCjFj1qxB0
  gDwT6wMASvC3h/HKBMOSwJJEgIHE9WEEpmPSAwigHFRdRwoZ6MAITmA4o6GuGDUz1gMe4AAF
  HKBbRVDAA5IggBVQIIEKSEEHkCcEB/QFDhPowAp6pQITXMABfAiAAxjwAGM5cYY/qAMFNpWi
  FZwAZTkrQoIiQChCWYVbsWICAKbntqfIA25xGgC6mjKA5l0gNhDQDQL+IvABgEGBKIC6wDje
  CI98yS1RAJzQAtqSBA1wIALRiAMJShMIjAQAAlGTGpwKMQpABSoeOzLAACLgIbSgQAOIhIID
  RkAk+AyDPBvEAgAO8LdeGQ5FwijBBjrgqhOAIIz0IwABmQOfDTjgApPLQNqUEKOMkTJYjEkh
  ZFboQt3tLm0AMMDxkEAAFiDGCCLbYSAEAIgfOKpzQ3SZCkZQuVHkIQ91oINeAuABzA3xnSpI
  Q58Q0JunIClubQJZBAw5FXfE5imC6gQBGMDIrZytTF6MQJ3kNA4Y+uscBajn+jbRifYFkiAJ
  9FEc+kWAeZ2CeJwsqBY5IQRpjgkt9EIfuTD+VBCD7KGTKEhpvRz1hAFkjFjkkYADXniFBXIs
  g8vJIHgoiIFTWvAJDqCiFU9gS/AcQAHj7ACQtoIACtz0Ajll4R+cWQc9LKGdGBBN7lrQAQyU
  Y2cs6MDuiDaCDEBRABw40TBmWYIhpmADOdvLpuADnhBQsQRMUwIb6GkJfA6zCbMbG0K7aEY6
  BaoTDeFWHyYwJnu5Y061gcCYcBmI6sVNbm3ag4QMkoTZ1fFdAaAXOWCBgDGt1Ah7QAGlftBG
  tIDSEvts3jDxdpYP6IcTnCSBBn7YBAJIAGNeutmPMgoFALAsnCgyWopUBEEMWEBYn+IhMTlg
  OPgIYwNMveEJwIP+KyoMwAAYS6FWdddMoGgvCRcYAQVQS4EWhKAxixhiqugrgXVANUUdsIAE
  jutXkKgnCQYY2acy8KoOLOmu2jUCJDRBYcgeVoxj62cX3+ggQ6GpAG+McD/aUVl3cOAClWCA
  oxAQlv5NQZqcMABFK/q/lsYKAPLY2t1iqoGP/eO0RAvABcDBG7R84GCPYMAExtajIwwgRRyY
  QEXDFRZQQuG8DWihAmLCiqZcoGUq8MAFPNBOoz3YBC0JFoM3EIIKwWuXKgCBCUagAFLe9wAQ
  vKaaHFCzDEwgR+3t6oV/ACToAWADLRiGNB1hAOJ0QAKOCQCQcuGBFJnguMRCmmZO4Dr+diR4
  nBdQhlVr5uA4D/oMWzMXIK+gCua9w54VuFc8FLof/iSEXJxlx9liOpUKjIOem2jTkjWQ6yBX
  jxN2CC1BnOAoRsEBLRxoMgA8dGTk3aMQjfNtu/Aj5UdA0o7rjCmb5LTPC/RGj0hxwiq6uq88
  OBQKhGvZdOpwABMtxwQhGEEI9u2cojYnMmOYAKumu5yC6xsEv3QOBYpNhC2zpCXJXGYLEXLq
  X3STCJxrQXMoEJe9EAcEFkigINI0ghSN4AFoXk5LuoQqCZSDABQYXbE0wlRUDcnUUODaOUN7
  hSWfOLTniLU7Zj3RAvQMxLh4AjqoPSaplSmOchQTBBCkvCH+19roXQOgAI56hOP9pwjmUkG9
  JhAnbwbAyHWCrSaQcYGY1gmlZNLAfwpAlWhAgEzcGhQHMqyfNzYZwbKgwN6L5YAcaRISW0FB
  r1JQhyaWyMwyE1YISmDLECpnA7kGhgML/isP7JsCECgGBhgexQHUeeAqQqGxdPpMSYsxRinL
  uApKAKtVnuBEpZqYAC4QuyF44GUjkEDnzKwC8nSA9lZjBwiKSIEHaMTMOhRdCkAwaAMUAAL3
  0c8c5Fa/rSwPNm6qBBfLiEZNGG8Ahsw1AiqALaqA2Nz7sZMA6IiprQhgAnrc+xu/aPQam1OM
  SHFWBSB2ZsIABMEA0KZi+PBIojL+ATEVAd6AAns3NuOiCuSAEXhDAhUACcszgR+gUNpSdkqQ
  IBRQIr9yAiNQDAK2U99TcXpBASfCAlyyIiMTLTZnDMeHbyIgYMTAddRUcg9zAhFwI1V1AcIQ
  VqoSNHcEEcEDAiXACxF3MwvgQl31P2IEAhdBBBMwHCnwKRYQChlwIs2BAR/kTWMWDRSgcg+w
  fNCXAlzCC+bhB8yRARIgMmYWLcYwXmG1BA4wRvchawqlL6KlHUvwL/oHAfCAf+MAG/mhfZoQ
  N6Ehgk1wLXFHKONDAPmCNVMjBUCCPxRmCdaDdYA0WqvEBCxGZOwQAWI3NYj0Lx8AbXEECgGw
  gbBQAAn+kzfysnf1sneIomR8MEbCVQnyMiZlUkexxgBZJARbowBY1Wc5RYVeFQVb6EAP82Aq
  kkzBg2YjwWAGlEpNYDJVFGYm4AGzQCTIZwDG4IIldTwK8BZ/4EJcRgfuJVrMRkvC5AhhyDGo
  IkwBYAKm4yrl9QvxZVYxZAIgYVXK8WBGQ2r99RjSQgHxZTQwc0V+9WjFFRwUZj09M1rqKC6L
  1UWLuH/xsB+94RuQ5YnJKASDQox75FsF4GGiUACbOAX/kxBtsnOidVFkAEVLAFer5QhtVy/a
  IgqaxWOxGA69wShCJlvXt4iqBUqPcAHHoRecJDB6xJJ7tybj8m6B8FTLNY/+VFBNvgISS2Jp
  +3ZAmAFeD+YcNMESnEUApcZ5YYYAI2A0K+IGjEF6dIEBx0SH6yVo6mgAbvgcepACDTR7CEcA
  79SFJ7EHBnABw1AU3IQiJ5eGbgh9aclxPyBBHkAebSgMB2Q0FJCSFLF75qYvVigFrYU+i7hH
  hNJHn3UfFVA1bbImp/YmiuVPUTZ/6+MoGyIikxAm9HImFpaTpOkIFbB3SGGIvPZG3HI+H6AB
  EJAnREEoNNU7baSB57CILJkViPgGlhJHEDA22XJZgwIBd9IEDuAB41gzObJWt6E/mcMqwrAk
  4yUCqpOWywESyHQzJuBmTCAA4MV5RkMLnYOCiBH+ADWjl0JQHS7DmM+oO8k3BksyDAihAocZ
  EuQBZiHBGD1TIm0lEGkwMjeTkAmpQ0kTVgJqEVbFS5d5QH5VAg3ggo+wd4dSUaQ3O1JzYhtm
  T/nCLejgRZfwH26SLkvAGhPoNhqgYl+kG+jXGcE5AfQSgsH2P4oSBSxGJugppfTCoyCYYdRT
  SfiHisQTFhwgbOPANrDBCUgRPijQCdDpDsP1NWTnbEuwAUM0e/c1CLIQBQDgAC3zMhDDC4SB
  UxjwAAyQAf6YIl2IlsQAaU0QAAoAOZy3ImppS2TITfI0BQMwJA/6KYwBGTvVHytRcF3IMi2z
  AnOWbySBZhAUAhzgYID+VYZFIAHvgSpJw6hmJktqRQAlYALGAn1MVQzvEWDqmCDMo1CiyHA4
  1k/jYEa0VmvBdg7lVgGBQlCO+VrJs356xFgRNVH3IGUIIi5TGoigNYh+in4ZRm3+lB9cdC9U
  IwcRQHbpSQRfQwIVGD7iE3cVcA7kckiVQIxexAmIWAYu6KvvBB7kMarwsjEZgBNMkpY18yoW
  QAG6Y4IgwRxomQEp0HtLQAe0tCQExFQmVCo6sh5PAy2fikIYwHqkCgUGUFeF0yoMtkyAtgBN
  lDH7NhOb+hWWaQJeopDMoUPkZAAewAEt+ivTd3xGQ4bRc3cfSFAVpY58A0qMVX4U1TWhsH7+
  7qAfh7IHECCViLVJsbZh5fcfW6uO/kOu+Kp9f4Sud4Rj+Xdi2kcAFcUJcgCK9KRQutEvhmgm
  bRK25BmdXYOMFcAJYhKd79AJvQEODCcApFNEw3KoF1dcGKQMI2QCIpABaaAoBCCPChBXMcio
  aoY0wac2GEBL0WKq4yV6aUCIfaIZ4qRDaKlc8JlzBGBAFNAA7Zg0IWAC+IZMkLEAduAWOpJX
  IgMfWLgBZFYec0AHW2MACoAA84FVNmKfEqO2whm155pzE0AO+MFtAmNjoxB03Wqt69MzakRc
  IxgHcxS+sTGk9vN/VaBGYUE2oEhjAPQ04UJRc6RHDVIVNxpaeQD+u3CwT7AxDxEVD5rQJgVx
  LnTSJnmLawxXANCVZ5TLlV9hmBWxAsgEK2/AkY0TTgQ0faUiARZwRUiFARsgApQ3Xk+Ilh6z
  bOfxHuJ0Ah5wQKjEDMc5FCvUROw5cGJoNDGzAQK2O3UQK59LJMJAHu+oZW+hZfD4ue3oHi0R
  kFh6v/LwtscpALEmZI0oD91HimqUfyJZNXvAH/5yLvEAD/HnwKSlJgwQNdqSv6oGQD3MDpEl
  LzLVNhyGJIeibLFyFLH2W56At3hrt3VgUaMgB51gwEoAg8RRRP2mUz08AA7mAbL0AAdgDAJB
  CShCHJwyOf2GhRZwapTgTKosodnAoD/+UACnEk4vY3M1czNUqLYAAC2h/De8rHEmuhGFkTEW
  0LP0I2lbpgAUQCSbx1Qm8IQfPAiikMR9EAHzckj4pLVPIABJgiTcUgn0YEGUgH8T6EXrYyf5
  gcrHMzc5aWNXYH3jmS20NgfqLCHRY4uvmBZnsaPwC5v8Qc+CFQpbQ2WK1XSs+ZrK2mw0nHO/
  Zzr4xhjK1cMyAVhtAQCtcwT1NhyizLojgBkPcBm5tkosXJ8hAR40Ua/wklSyrLyqI6q6o7YC
  4DKo2krQjAYNQMQyw2CHKmKZcAFL0indxQEhJDMigJ/J1LLX+TQX0KXKOmV2mgSYCAGOUmjv
  QgD7lGGwAcj+DawH9ITKW1uSfURRCb0a1Jwt95S1g0h6a5JSnlQvtBgK8srAh9fUYDdGDNB0
  nvRJ4yIHiOiucBMgeikA/jjJcxaqnKzTiyABeIUJjHAEoivK89mFrnKo45R5LEFFuzSYxBwF
  dJCGJNyoOszSFBcFuZuCIIDRLfO6obAzlLfJW0Was+MB0TKzHoAXbTGpvEAMDGYBFcSg6Kda
  S63OH53EzRAKjpEkhlQmVOFhdKAoX4dY3wLA/ZQf8QfJiKU8lfUORbeRAES13zsOnYSv9EAQ
  54B989DNck084sI2303BozCvAlxrJh2gD0A6lrZgxvIkhr0IcCAQaNArqZoiRAT+LCdRhy7R
  J43zXNMbc8vxaMyVPCXoMoGaImnGshQwhboTjU2gAGG1SpjTSn9jQITqhCKgVmHdkxDwVDxR
  RRBTHl3yOMNyQH24v6LdLrlprlmnFWOw37C1exmmf32UvqIACVw9jGZiLuunHw5sXprVx4rr
  xkMKdKlNCWoT3ucjU2XC19eaPdS9CKLAAGwDd/haWNg3a1q+5UXQNYoK07Oseq3zPRos2gsd
  4QPHVCUgehJQVhVHFCbQQD1zAA8HF08DVaXDy0SEmLiNSryLxKDDBCIwCKX9ToTOVxtR54xB
  CJknCjIys/cmLCRRGLgNaSXOBHY71ugDDw1842beJ3P+bNVDFyjrgyhJNgA+SDyhsHdU0TUv
  WaepLjvfSy9r3TZwhE/+R9z+khsbQtAxlS2FkuVxsuvly8e+nuwnxteyoQlxLOol18tyDkGM
  YQEs9CP5LVgBMBwx+N8qMl6TK0EnkOcwQzPWFUIjLgK0Kuq7METa7kp6mjo28xaK7gQdYBn2
  ThwvQ8TNHB4zcxLz03UhU5bd5YbASxKpYwyqUyHFtn4eshzSThUKfMf+rCaqMIzIjd2ubn6i
  1SYNvghwtViy4eqRQHrbaddrjVBlHWyCLOOFiMAhqS3RWlgVhrac5RTHXSbp8+OGEijW02zM
  1jCu1EqFvhF2zsm7E+6pcAH+/t0yjj209+nQG5AC4R4AZSUB7UZvD4AZs44ENiVEMB2oA39C
  Edd8OxX1QlJFYlgMTqQME2AszmgBRy1Gz7Cf8HG0r2ZPxpJe1hXjfcJFF5/Pist/Nx7fd6R3
  zbOIRV/e8ygKJw8H6PdG4HLI1eP4IwhiJwV3yB2/V8fx14ZYg+Ku/EHAovU/nmAnzv4mAByC
  Rsc1r5+3emuwTrAYykHuGcRUhMHmoyr1sGUihN5ADiQM29iyc/bRbXEAhXf3E2uxL+sEfMMB
  m6LmKlLnOtFnq/f21a8EHBtlQEEADjBgdb9Mir4oYxAMF7A56QT9P8zvh5c7C6BT0xSpbkJZ
  KNX+5OXH8UAAAAR+ReMRmQxMLpzPU6O5XCYQCEKBQAwCwwKAkBQPuwEBQavlCgFitzEwiGhI
  ddInWqky0ohCYUBAIIBQ6O0IoMCPkKuL7VFgINCx7RDOAKJAgM1MkkASdHBoALDSskiBw+SE
  pXVFJeWkpCQkpGMDg0LCQYFA4PRNgEJlZYWlxZilWAX2xKQEBANDwsSECPgHwEBBQYIio2MV
  JATjIAy7iOBi40TluJUFdgSko4NjIyOfguJh4cD8F7AHFxAIONAAQz5pCxNmyDXBwCZTpwBM
  8bWNwsJ8CqVRcLCFwL9rwAghmPABBYoPHC5UiACBj59AgxwFRGdEAIL+CxqecOAQpeVLmDE/
  jdIGjNMZBFaYMhhg6OaPAAQiXLBD4ieVCAxiGvBTwJeoIRPfxNkk9lFaNpLKYIv0dEgkQnPp
  RvpUpgs6BRtCmHCnTIUzWrY24NLFzcC5qNkcvFrRApmyYilenYjVwaEFCRusJTFAyGYRMAcc
  5PMQIta4DA5C/zDAxg2BDBxSuGvh6sSJEPRubdzHz8E/X9iGfMunWYIF5culWcCAr9zIU6Al
  YMDsUCPDhhlwfSZ7qGQEJ1g16IGZBhDNmouzDaigwWfWlhUqWIkpU6IQA0deAxCw3whOyBiA
  gD/+gIs/iQAUY6qqPiABBfIueOk+LcCaqRD+qCzZxAy65hpEkBBBE0CR1oIRIsO0UkQxDk00
  PAUBCqaI74IMprjgt30WeOCjAyJiz7UOjIFMsmWIYSY3EHCZhi/pfijAAyQ5yGCC4AwwwEcC
  tBwkJCwRwNIBGT0Y4QQQLjgjiQFm8wueeJzhrTDf9pHgAQXMUcwSbfDJZwMQTEANNRNSACGF
  ElKoZoNN0CHkgIb27ACESHeDFJpaQhCBgtcWHUUR9EJRby42TAQmpxunmABVCCKIoA8tCMjw
  qQUFiADCnxg4kCa1HmGEFAQYqKCJCxgYVaoBGIAAvqwm2Iqr+/74hK1QiTNDEAL/8LXVTq14
  Ch0CxoqrE2s7Bcv+QF/X2NTDdNMNcRAgDwjBtshcMTIWE0RYUgLupGPAg2TiiaVSzCg44Kky
  voG0gxAMJeaVPxVw8gd1OCDTSFlqqeeWPXPR5QHh8MzzgXxEmEXQV5g5uZgVMjjguzwDeCDf
  hDsowZkUbLYZlstWzgsbSnT9OS1iLRkAgVWXFUqoCp+l5IsFAXjwjqzMu5UtMlCMRJEImOCJ
  pQkYaDmbFpeiUCYCoZVExFBfPOTVODzptFn7uDpv7TwBiJaUTtOI6ditvIMYyMCnM8A4G2vM
  ICPEN9ZFAjo7HsIIBwjwAPEcN0L8s9Y2caChORtvfB8H0gSadKCjGiKkxLkrbE/WIAfLEsWQ
  FHAA1VQh4FEBA+ASWvDex8i7bAMvLLARn3nOxoqv4L7Vxe+uJtpXPu4qZIzSrU/rJm/vFj4U
  syiB/eoCtTjWWbF8Px+p60s/ggAHGNASfgISS8xbYv0z+4wQ5bfz47DY/T9EMwHg/xbzt6lo
  iX6iANxiKAGadKEIfRG0BF6Kpy6fYQ8OW1APGXJFrPzQRVdiuNIIr/SlEpIQhV4h4ZcWk5/1
  9e5VnKCW2l4nQRveEBEfwyHYcNhDH/4QiBLU4Q2HGEQjHtEIQQAAOw==
}

set pieceImageData(Eboard,45) {
  R0lGODlhHAItAKUuAAQFAxASDB8aDyIiHC8mGjUzKUI2IlQ7JUBCQGY9KVtILWpGKlBTUHpI
  K2ZWPm9YKV5aWmJiXoRnMW1rYoh0RnJ6cp52MoSFf62JQ6iMWpKTkracXaCgmcKkY7Wphquw
  q82tbdKydtyucc67kbzAvOe5fN/GlOXNndDT0O/Vo+7hv/Xu0v742v768v//////////////
  /////////////////////////////////////////////////////////yH5BAEKAD8ALAAA
  AAAcAi0AAAb+wJ9wSCwaj8ikcslsOp/QqHRKrVqv2Kx2y+16v+CweCwFEACBoeCQzhIQCAER
  UEAYDIT8AEAMCAQBbVUAFwpySAIUGXxkjY5IC49GZgyCAQwFjFgAAgUMAwZDZnAFA6Z7cwCq
  mlMEBRVoSQEQZwSSt42stwYHIwqaFyYToVgUICbEQgEcKSYhHR0bFII/BBQOCraDDikdunMq
  Kgffm35553+HuGLkjgUkKpoBKBrUVgEFEBUIB6IRFxciCISAgFWAAQFiDeKAomAyIgYioJiQ
  sMuqixfXdVEnpF0jARdYUNA0QkUGjlUylEjRb0gADytSpChRIgQGagQ2TMh2z4P+ChPajBAw
  EFOCvSwBLnjAwJRpBg8UNHIJ2lEqBBUfNA1AUeEoFXz6CrT8ASCChgsTBjIwWCCh1yfzVETw
  c0QAhxUXUHFRhSCtwAgQBHjcAkqBAwQEBljURMAB1VsAHAgeYiADAi0qTzwU4IFFihMhRoCY
  RmTAhgc8BynYsEgWBQoGBlPJqSLFBqYgVnhg0MiPKUCPr6ga4kp2GE4ViQ9HigDCBLEuJ5z9
  S5CVGbdWyk7Qe4TBvuRbDgIgsaJFixWqCogxMAABhxEeOMBGeYWAAG0ELlAI/mi5MvBXqJTC
  Zh94BloIIpBGXAYPOCbcAXkoQcA4XxCQQQgrnIDBBs3+TOAAGQEgwAAc7AWyRS1CEFAJf7no
  4h8WAUT0nAKiSBfQQAhQcxB292RCn0ttGSfFAM8R8EELEwUwAW9hEDCBT7U5Q4ECim1xHx8g
  haDAW2SYQcQZQjaR2UMvxQQaCKPhxKCDwv04hGRc1pfBBuGEcIIKFzgQlRghVoBWBAzg4SYV
  aGA5IhrnPDTGi2SF2USIEVQAnTJm3QhYjn0gxKgUCXmJhCpxVjEAByTQgUIEAGhwahgClMTC
  CiyE000GFFTpRgohLELBCRscsAF8DBIAYIWAjBWjo0tsYEIKHEzwwAT6pFDbCSOEAMIGDtxx
  h56osQhFACMlAQAFg9a3YW7+K6hwAgUY7AnGK6SSwIEGFYxYACBa5BcBQt7lKCImAwiLrBRn
  2JOjt1FAWgGggDHAwAQAqYXJH4FUrFB2kyExAJNejIpVpxC0oMIETaYAKwuv1haCNLZmcYEK
  IAx1QgcLcJhhrhn8omh4BDTLiAISdPBAqE5wcILJKL+abpRoQrMBa+2+1i0WB1zg0bgIW2Hh
  BneOcCe77n7BwAUfoIDCB2cRhAB3WDjn7wUHJzZAAYDtsbMWl0CgCQAfXEC0E5BeoEG8855l
  qcP2JgYIj1WcUVASkWZNhcdrJUSAClyBkcgKnHdeWzQZsLdFWR7koW4HGcwZjQoiW/AAAHdb
  IUD+Bip45g3snJtUrhMZHI1yC0mHc0JNCIJO62sSTH3FBScIewQAD2wQexP2uWkhCMyyu4Fu
  YX8ZYRUBmPUBCSR8wMEFEIzoPN4IhRhQAa785tvCnmohGAdoPwfqBeTTdUXggyNf4f4kEIch
  oAClMEUgLkYoAlTgDM8738CcMAC4jaoFJFjbwsBQQc4pTVaoy8ABWnYFwTiJA+GoCa6gYQJ1
  haAEbNgdFISVgpSl4AECIEoKMyC5ZK0kaShb2vCs5bQMRE0CyUuNFVD4i+ddAFtTCMSEIGSi
  L2UgBbR6Gu1ScIEigOkc6+PUPzTwAfPRS232sRIa+hIQONirFIBgwFz+GCgcBJSPAwsTTFnI
  2BUsCAABYxyfGc+ysAK+UX6MI5QC4EYOvlmNU6oARCSL0AkA3MU8KFijERbXqQlWg3Yq8Fwz
  OEAr0WlBFQcwT+1ql7sSzIlOLMiAZEY3AVgtTWgEkNZPQACAsUzhQncCYijVZYKaXCt1r0Ge
  AyCEBZjgcGcCMEEHJMCpA1BAWSOgFRuCwhkTPGBDrOmACcJ2EAVQoAMgiA9FeliES0CMA/O6
  QAXSt7a/OQEA7YFYQCAAAUAZsBTmWJsWDoAADciLXhFYGx3+QYIJ+DKKzfET/tCmgbP4hZ4H
  TODi6FgG6VRikwHo28Duo4/pYAJ2ouBEKDv+VxCc/PEfbWQbBR0wAhPcqRnPaNcDEmO/A5jg
  PEFMl8kyNLwNdCAFGOiAZjzZTp8sTVqg2UB+iMlDex7BABR4YgdGIJposOZ4FHiAWBWggAOY
  NQ9xgt0BymWCWDZxDrTrQPcoSdZDCMYBrATiCkZAgbU2yg9/yMMdtBEAB8DkVYjlXK+sqgwR
  RWACkAWMAg6IEPuZZXADrABkIYu4UryhEkx1yTLuaNGGcaKgJKjsIA5Sh4dBa0SkkBvFOrmK
  6Y2jLVyyCwkkNYkIkOCR4kKgEAygJBKYDQXkE+A+bIGR5lIKHmZLLvkmkDEnDAUaNS2BB1gT
  VgeA4pQlME/ncrf+ynA442kYaKEFYmMF4npOXcTUkgE80IzTMLadBXjAKztASmR2dwHYKOtZ
  7RMjJUCAWlwyQApaME2UlAkE65VFCU7TkV6k7GgrzZBUZaMK5oUyZR/2jPSi0AnHqoUU+MoC
  ISYaT4BoNi0OSx9uARracdgRBS1GHz8TQqTzFbgKApibG9WGQDhWzGIYUQIAGNAQlB5hAOO7
  AAOoIiz+AfeqECgfH8piXOSab6Ll08Cm2pkqFKggzJol3FyiMIBXapFWDlgmCQdBgPOwMhzh
  IK8KaiKC24gzdQ+NQmTGC18Q0ESECjhd6O5hjdQ97QK0Sl6cBWxWCBGYCfSNWRIux2D+DCjK
  ACdgQQcw4C0CZPMBQrBLyqRp1B/GhE3iOoC0zES8We95rkx4w78QZwCA3vdTBWDx+QAiz8im
  j59r+wMc5vwEfE5AgGcxqIggECQI8G8tJUSgGzOKwAFwkrarWAIhmvyWd5BAA6iaAwKiXEUj
  BEBVWfkBlM2GNn/+Q5AQCJUZwkGPJSHQLGG+TBQM8ADkJa8BlsYbAU4g3pWaRyYQP8EJTACC
  UW/IpiXI4RQAgIAT6NkEK5hABop5ml1dK1yMVsCcMqAfCWCj0gMOxPT6YJirlWfC1HxTZ1aA
  uquVtQ0XgNUJllJwJEpgAyuxTRqfRyd1eQCsiuAAyLtxX9b+1qHIs0VKAciW4wr4yetpgbGM
  7QPIK7N5Agw5qOBI4IcDCmsAWSYBx+7RiasX+RRIDneud+SRCiCXXgaB9wWqa4S5BcAWFzCz
  BhggmCAD4qUSfEuqzozuIHsCE96p930LM+kD4OF7mwg6yvCkFA8sSybmtWkIngG1DYSAQpxC
  4dKIuQJz1jRXB/BAxVtzhYClDpkNojQYrRoby32KBS2Q5k1csmCeY6AdwjoDMWCyAhNkQALg
  RCYGsHeCbNAHduEYwfkgq4jfdyAcIPj1QfTg7cXhDe1oC8hm9anZvyDO7UHOkSfHZtyDVjSD
  IkJZIUIqGpAvQpZRekBgnWRVesT+UUKQeDjmN5pwY31DeJmSHCg0MmfgHZpVCQchOAURHGhg
  ZlIWMIB0FiMSAfNCEVAwAehQPenAbFMQdMDDRedwABkQK1ESGqsHAghyG8sSAhIwc0gwAbq0
  NLOXLh1gLdnCehBUBVtiAHYCGiGEPMFHRUToRXVlBHeAfLiyARoXG8i3AtdyB5ukHxwxApwz
  AhjwADaFZzZxTjMjAQ8FLiWgAnwlAedUAhI3M0+TLulHML+RDu6XBfAXEN4WCHQjOATUMLAF
  UFs3DIKGdsZVRtJVRhyQQGhQAfgTWpQROGjxTxq1QJwABQKACUfRM2c2L4/zAwKgKmeTF9+Q
  EDkiCIn+pwKVsBXIRT4a8Bz40IskBABZ9gEVYAoMgFnmMwzHWC/2dAfWxAFPRyseghrs0Qn3
  gAJeeAEEkDwUoIYQt0Lo9ELDAwIZoFQqsAETJHt6hmcycSEh0FdOA3sEEwC2t1I7qEXK5H0y
  5wQG0C4oMRSqVAId8AuoBDxkGDqPARIeQCNE0BkskE0S0EIQ9xNG1AGu5ADUoAAg8BMcUHBX
  9I2osz0pEIj8mA+RkhYk0m0JIUNKkBRlJCnhc4qRUlFokRYY5XYIMQCDMzAAwInJ1YtbJ3cG
  pEB7FG9A9kfSgVn08loIOFurwJJ0QBH2EFKruHid1GXnIxhT2RysMAFmti/+74BjLOY3SoJu
  VEYq+5APXAdPHxABntBGMmQflaGGY1h9NcFd3fIHUyCGa0hcGCABoVZ9n0FxT+N6dlIC5CgB
  FjAThxcFG5kuMSF+ILdZGSAaGKBy0YCRUdQYd9g5NsRqkeZyZYVW7aYEs1M6HCEACqBK0iQB
  tTABqnQtfqUGA+AB2lgEd7FXDHJT3zhOKjFNgmAhdwIVEtA7ETczc6IuG/A3gkE3goQ2C4NR
  pqB3A/du5lMJIrJ1AGE49QdbsZWI3tGYzeaTvDgBqBUwRSYwBfUBv9YRaPAwFRWfS9kw8INI
  pGgcgQAtUwkPBxUYqqCLEbgHOvIwmDIEDAAPFwD+P6QyUZgVBwggOKzgWxk0AP8wUebzAdQW
  KdTWQ+xhAIuUAf11PKhhaQdBBbX0XpApLboEmqnjgxI3AqiDAc8wNOwkBCWRO+kycStwAE8i
  E64EG3MyNFRAXDlkGAEmYIOFDpxUo/exhBnQBoGQAap0AuQ4NArwU+myMjgEpRfAhmEkBFdq
  O4rAmxGXOtEwmwFwNHlYEqj3GSeQOiHQDTlHPawFB3aqkrTVng9qRsS2nTRZSA0zbZcyiHUg
  no9yAfgTZmcxPp4AGASREFvHnv8TCEGGEDH4bfd5eKj2KHZRD4wwgsg1LwmFBhAQXWexB/Ig
  R5nQTvzzAc8BAfFJKhT+NRdEsqpkQT5zUVBkFGbwpCIxBQXZwAtmlaQU812EEph5NnvCI3Er
  Y0QYUJkSyWpJhUVP6AQjcIR4FkrWxKYmwDLlKKSMJqzEWj1+ACo1Kgq7MgLCEhsGMAKq1Axy
  FQAPwDrpIpBDCDuBUFMO8A2zY17X1wGeoaIyMU5PswC2kEP1pYeyMi3DMyc2JUvV+QbdpkAL
  1J4dgaiDJJ/xWX/pYy/9BGNyEyPo4yhJsavxh6gkoCL8NCI8xokWWwQcB6togZOYOkmCNjj5
  tgoFEF0riCoO9JOyGEnCCDfDsmSkUoJo939hlhchojfVsBUacECcKEgVhUcFADEnJQWGNXT+
  eDmaq8UBwCMrnHM0UHUCIhCjTdF0QuU0yeMMgaYESHcn2SotKrcs1GJEByFCnihvF1BDQdSt
  RhRWAvalUAAA0tKaAtAArGNnezYaDCdUNREsClAS2UQObJABQpU8cothMiFXRmQIBqAstGKc
  Kno0qBOnWNQA6keh55M409lA/ENafVqT82RAlHW1AIGSvRYIIyKDzyMd8NRiFfVSdroHRCJm
  KvYwAnQp3fZtpSgFqIU2+bmLLSZHZGM+ABEkKgIxetNIdiQvCJGUiTovkgIA0GIf9FIKKFs+
  26lBaJFITkABqxRVDKIAWSguORi2tRElqrd6aGsBxdkB6fIqZhv+DYq5ASIwp01gDHeIZzz3
  Gqb3GbmCAX/ggp7oCrSTNCuAmFAjaRDCknURrVqSv+ZxHqgHK3h2lxiwACVxjq6JBAFwAHi2
  AUdnMhH3GbchDY3xDBQgASFQGzJxNK70kSXgmueqDPAiZZTlbUw1AFM7SGBnkwVECp71oMUG
  KAckRW3RbCU1QDNrkrDVeNKxt8qgDwIELaRgnw7IBOFTRq4KAWVTPtgbEBWglGdRT3K0MJng
  FcKSqIxHAGM0vgFxihPAcR8QB8FGtRHTHvK0Fp60PbFChfXLC1TgAOUxejmKmD54LV+lUw4w
  ciC2QhWJOgrMBAqwhLoUStbHNUdDpUz+AQiG8LLy5isn8yrnhUwuZ1YpJgWgZF41ZB4pIxpc
  BQI1NXH+azQzMSVK9sOnc00CC8SuJFcE8AA/SgECG8TR8MPdwAYQRUZKrFFMZRaFQ5PGhjiP
  qDh0U1HRKWMyRxD3+wMKQ5O56wnIBgdu8TBkLCzWtrxEhnekuHE6Gbzjs7RXrE/y1MgI9LH6
  Jy4+SbSAXFFUm5YUoYLexgCKrMf5cL5r3JJ+qzKkjBrtCQCsEw8TQEwbYAFx9gBGlzyeRwA+
  oYNL8wxGJAFchXJNAABHVwIt5MDQnFMW0En/4wCQ/LcjgI8sTUWEQgExjToNktJI1CAIZ2kT
  4pG0Uq1IUBn+NxVCp+ejAslyjeEsilBTblpx2IMrDALCQhEBZfTN8jNBhLB41kZsWAwIeHcR
  ZafOjgoHgMBGJFsHXkdsXudP3faUsiwKAgBw8hKo9dk+C0QogARZBFFk8INAckOxBxEwBpDF
  HGZt8AQwDyM4hUMRpmDPfQFP8uSWc9NP2jsFAlA7sTKSTyO4aE0cBBArFDABXlNM1zcOlDoU
  eqAAfvteiEnKFFACI0Cy/SgCdpLKYx0Nf9kphQ0uIKBXKQCjuPwAZ2WxeSDTDUBWeuh6Oj3c
  qNPDCyDVk2aon4KDwkOOT/N018APmW0KQzFZDgAxjnat6sIg73wEr8C+rqspUZD+B1bpdQSe
  UOB2EZfgJ3i9sojBWh91T6kdKV+3JHaQiEm2Ccd7R0y5xJzEVAeRna5gvfHZiDKWQJSqR+J2
  2Ooc2SZGCtD9n5fXu0LGsmPGBL12MjGxe4Lru0mwBhgkACZTcVDzC3KwgO7hQUqTLhQHAn9p
  AUp1rpExJyJAExJnTFDjOtQJZBSAIRpMLaAjNaP5svLaIIpAXyulNJ+BU80qAeV9DsYhfXYr
  TbetAAxgGGQVZ5NGVi2d2eYETCGA3S8bvejD4cPCBH+UoEnxYoARJEWO4EQC2gjlTw3+4VaF
  Bu0hR4/FvO3jXJvgJAaV2BhVnx0+CPl3b8nlZdypWf3+ZC9IxgTO5ifQkkCSnZ7dpjikWGKl
  kOVwQQHIF8ms18OOsXm5BB8Tht3R8zp1cVjnMcD1unq3cQyhw1i9BHPULtX2UeNQgNV6VVTc
  lUQfbLEAYI7lUcLkzgIYVuV8FkIWwM2PEiP2i1Wml2F4JnHF1DRftR+WY9VloILyosSIIT8k
  JgAvRhDeptfkgJTBG08LM5QIgQAw2WxoUKk1y+lXQFxbJ6scMMUcvlFA9grjYzb8Rr30QuA2
  OXadQj2Wnhh9QUZunFxfZjgwptkZMTkh8a6/njymVLh3ggLx2ktVdQTWjHzL/kETR45MgYf6
  buhmcAANYAEYYAFQz9IUQ8b+P6AA9LXtoAnm3/4VJ0PuwOwZUCF1QVxMrJc64HrTaIBV0qLB
  szdxZC/khYkBjoHtgFMB49O6OJmInPJ1+vdHAnoEZYc/A/RaMo4PSyLtq3DZ/nzhWCAsD4rx
  G968FiMqlEg+xxVdxAgYAlHOmJAYdJ8Kp0Uqlx9daudikQUo8HPyVeAAbRW21R0NUsNMUnB0
  UhUKoPIjAZA0Qz+3zsAU+qUCyA4F+oVhhBbEt1Ho1UkBLRQ8PwGj3AXso3kfrRAaulcC4w48
  HbCAgMmHE+Y6eg7lSoHfkJk7w3Af/WgtIHC2rAH1GCCPUaSTvPjNsmWB1AM/XlIouF/HLB9P
  J4b+BxVzH0AQ+A2JReMPMIBcNBoOR3OpRBiIQQBw1G6LhEAhovmQoBMqAlEoeANtrJDLHUya
  TRIKRSIhBAA/4AtCsIpgoA8ujggAgUnvDo8M6w9sojIiAgINASsx8eKEJVRFZaRjg4JCQsGg
  UwvgYAHuz9WBpaVlJXdFhWX3xAQkA+NBYiRDoLUoY0RFV8UkpXkXZIMgK1krwICiOTR01yTE
  dCMD9cHhIL1tAHvIQMKLIGPl9nYFRMJAQSIjpOS/wwNJAFhhC3AhQz9pCyEYEBCAgAQQ4jpU
  HOZgwYFr7YoAIKBBT5QqVgw93Ijtjyw/rgo44fDhA4cLFy6dQeAF0JX+k50ARHgJU6aZCmYY
  FOjD8Yg1AkxekoCpYSgEBgwMqHnoRhLSAVYHRLjjKIqaAmimMpC6qQ0AZMkWoeHgCBKHCAj8
  fGFg6VKmNSuR/jDEzNsoYBswoEIngB1Hvkh2EjlgwhauXKN6PStBDYMEfhkIcMzAgRcuDmgo
  RFPRAYOBxgYpAG7hrfKJyxUTFn6gQMEBAg85KsBg7QCIFPUip+gAYmIIECXEYdC4OJkACLo7
  7BqVi8UJDw4eUNggQnnFDscAcer7oxCDpiLRFLpaV/FG6EWk/3wicwqmszgDbEXUCgAGLvip
  ApIYmYABnc4jQokLHHEkJikwqaKqAa7ACov+tZIJYJMkHHzQKfxqGmQqnVbjAhA/3tIDEj04
  qGIqCGqiIsGBkAoggxPoee21XlIwYQRyylFFt/8WpI+AFXqJ7LVdfiwhBeaEkaADFTTiqBJe
  WCClNn9MsOABa85DYAMlbYkMNhVSCE9IVCRwADfdeMMmhBQ2iAbNUG7xUbZwqDHlOY6wIACD
  Zq6zbhTmLguBIiExUCArpAqYgAM8LI1pCr0MuTDSI+kLYIILYPqgjgsQFGQ/NwSwAgGUEGAA
  KFMFuWQCCDpFyqMBWbQ0j5g0MOMsNSxMqy6IUKpAgwgGbApCKGYa6pKp0hhgvkT8YGQMEJ2K
  Qgoz9CurAGqrTYT+AAqUrIe4LX+kJgML3sxtN0+7CMADe87tJZcUfokSM35SoMDITkxQUzZy
  MMDAgg06EKGDEhQ4kVwKQDCTxz1FWVOcDchx8xx4vWingy0pxsVHfX9hDgQRkDPFATHbCUCA
  DEqQZrJRfvGzUY01FiZQQSt4axQVICF1KAXScI9YiDnyQkAxnooC2EHaw8qLggB0wldT80K1
  APP6+kIMuCx18dcJ0QjXEGLTQimMOpp41myzpNVkjVt5mhRroKCIIj8Z9ZM6jT6UPkIACkrw
  Jl1RZMvYttwMiFfeHw4KrUfJQkkBc9lEIEeCB8jRMBmFMztAgNIFIICAAx7AIIMrtcr+oDrE
  z/zmx5wT4jg3OdspczJ1TSkHlUpQoSCh4hU4qh0AHIDmOn09CKqS6FFhvXiEHQBE0AJ0RWGU
  XkUcKdyr1o48gLt+8rWJv8+6SXwsrOaJCQmntVCN9gaP4wtRH2nxKZrWTwNt4ksRShhQAQMi
  6CZpa99VEGMhk9yvIwVkQhOmAEDUWQiDY3nV3KwRsDgoYAPM0MU3RqEvlP0uM+fQnafaoCWL
  2atmvxhBMDKDARM8AIJdEIABDqCAB0yAAqw7GAXklEMjDGBAKaBYYDA3m3EUz01EWmEyMpCC
  EUzgJhBxgGaGN7zOHeBxDzGJYjhwghHcxgADIAAPEfAANyr+AE5gtNAaD6AZB8DsZV+gwweG
  JiJBaMI978HekdoQKqeRilta+58VJNEfj6DEJG0ggCamZaPz9CcMY8jDtqBGI0oKK21uGNcW
  AFAACBhlHaL8wypL58BVvswAmRBcXRqIQVs+8AvhMuIQFMABEJxAS5VRwclGwKaDdQ4dkCMk
  N7BjOXyV8ATRvMwxbQiwpenDh8TrQAhMcIKJZMZlS7vABoCZJ3UBI2McKB7wkJkOZbbiAaY6
  XR2pdzB7Hqw2FJjAKuwGvwscbwAG6I491ymMIWIxXKk7gAfxB5E94sFXUZlbAFUJuq8xAmtP
  eJuEJjSSBNaFWu+zFi1XJTcZSUX+XAuCiLJWRDaOetQAFgxlXYzYH7rAbCCr1GlKhjXI6PQU
  EIjBEIYQUwgM7fIHIATBwLTkJBMwh1HjKEznHNeZIwHABLvY0Td0ocTsnCBK40FY5zYggYCp
  pg0WRU/qHDCBDGwgBGBFZzmQRwTV+EGkXfgMnnrUi2iGZxxtwp071RoHQBzgre1ylwQs0FjH
  2tMCGNiZBBYqKJgdQLIovOdmhfjWDTzAAF7b0Bci8DOXEsWj4cOKHwpLQGTdJ5EGrACJ6GYI
  nYr0Dzs0QkpAijqxhEu0P3DILPBHAGWBBKLOqpVH6UbRpGHjEO0bKk5Nl1bEjKW1WxBc0nbK
  XVoOgA/+o+SCA3wJVkMR7B8gEI8pjulGMFr1kgGQhsWU1NXLgfUyhElhB8yaFApcgAIJ0qkk
  UwcBVEypc4/bKQEmNYE7MlQeHdCRnhTHnIrobJ2D1Q18AYTZnbkLAxemTWJRQ46UWaCyeawj
  OQxWT2G8uDasow08kCpJBpT2bVFZ5BociJXsdqINxy0V3CpRFk1UJWkhLQKHKqHLnT4ZygMe
  gFm6hr8p06FZEpJKasWCQVEGNw4dbKXpNoXBQqgRzfWrCoc70QcTdbeWw8IKeDvEEQQQb1lP
  QEgGENJFVCygYwgA4495YoBmNokeNZtMNGeoX8181kgC8AD3hqlOCjhA0GD+fC/qSqcNTYOR
  h6i4gAdG8AsQOOB0WhjAZwYGG18AKbA7A56b4oi65L1jnRtYKjQy91ThQAkaJjhejQNQx4LC
  blGMStlykOOPEJRDNYLKCYM/icFOTze3C1KLsmbyrAlE5RJHBmTSCkAQIjyEBEKDQlGcG+Vi
  bWosEtQzHrfQH7P8LGvAMnJz0cap8bEFZmRmsAYpeTayZIJaSytPsWCmxoFrcA1i0YSC8jhd
  i1+coWw5wMxgsyTriExf4djcMRkbbSIEgBuZiyZYwZo5DCRANRoiyARK0M1fRPOp/2DYKfKK
  HkErAAJtdbAZHFB03MDx6II2wNIft5s5JY8A+9D+9T+iiTmrW/0E/wQzW/yQugckpCKMGgGQ
  RjD2snugHKgWL4rc3fYBaxu8ZfHo3AB4NgzqtA8KGMIicnyJbkclcBk6d3kshAAZVSB+FRgJ
  Tkg54Jy6Hcq4aoPENUF3Su6m60gRXEkYXBUNmmUQqKLKLCNX+nZ4wQHES8hMqsc64nlRM51z
  r09bhnY/qz6foW1MH1JfPAmkIvaaKYwDVA15478SV4AINfU8K6QMoAM+R9ouD+H4w0o4IOgd
  O4Bt1WJ677fCGpSn328LMXBOPVkREMAiAD9fhfCx1gjb7U+8pYW0f+/2+PlHijVy8ltbruOB
  Humq/MAQBg4NAKeS3uCT+xZwpPLP3brAAS4AdSYQdR5n6VCn+0ipPzjtdExnnuAlKTxQBEew
  JEbQBBfkrtygAjmwn44EJ4pFutSG0BhwAecski7uAU8u8L4MpyQhu2bpqNBPC5iO6RCACI/Q
  CI9w6ZJwCXuO64xvBrGB/1Ki4VQJENiMBrNQCwVQC3erC78QDMNQDMcQPcgQC8kQDdOwCIIA
  ADs=
}

set pieceImageData(Eboard,50) {
  R0lGODlhWAIyAKUuAAQFAxASDB8aDyIiHC8mGjUzKUI2IlQ7JUBCQGY9KVtILWpGKlBTUHpI
  K2ZWPm9YKV5aWmJiXoRnMW1rYoh0RnJ6cp52MoSFf62JQ6iMWpKTkracXaCgmcKkY7Wphquw
  q82tbdKydtyucc67kbzAvOe5fN/GlOXNndDT0O/Vo+7hv/Xu0v742v768v//////////////
  /////////////////////////////////////////////////////////yH5BAEKAD8ALAAA
  AABYAjIAAAb+wJ9wSCwaj8ikcslsOp/QqHRKrVqv2Kx2y+16v+CweEwuSwmCokFAMLuRAQCA
  KQ+87/g8FW0grtt6dwRyaUoCcnOBineJiz8GByMejQAXJhN9WxQgJiCZQgEcKSchHaYbFHZE
  FBMOCoBXDiYpHY1HACoqGwdiAAFsBMHBAsSObraKBSQqEEQaKBqqWwEFEBMVvEMAERoXExER
  EBAIttQDcdJVAAgcKOSfahEoKBPIXIj4+cZfhUT2egIusFDRiMAIFRn6ZclQIkWJbKA8rFCR
  wkSJEiEwpPuxIcOEV1kkqjABywgBAytoSRAT4IKHDBhiYrCQwQOFfV5KCvmXB4L+ig8IiHxA
  UWEjFmrWsPnjdqHCN3EMyiEogO5oBRQqIvxCIoDDxAs8s8hBAK7sBAiH8KAZMKCXwgNsjBGY
  ULGRAA8pMui8wtAhvAAfWKQsMSJEiFRFNmB44GDvlAwnTIRwLMSAgREjNqwMQ8DDwAwbOnbI
  dVNtsTAA3gagfGfABAbpIlwY1AVAgQgTJkD8EWACB2/gxJEjEoABVV9HJ3wgAVshEQHPSNRD
  DWDABxXz5kWQ82aAAAUOHCAQ0PaLHFgEGu8D4GD6TgMZgnJheKIEvLsD64cYIQIxkY4PgISF
  AqFlEBZvFFAgATw5baACC4rFlMIKHjBwBzHknbOaeQD+oFfAgccIEAdxg4AoRQAIWKMbcb41
  Fdw4tgDAAAFVXdEhbgMcyEAFEQwHRgA5FrDCCi20UIEcBZgxlwcn5FIRKgqU10VAgAQUggJG
  5cEdESV2QZ8Jf4mkHwj9bZQBBYyxFsUhzhnhgIhtdkEAaCGoQMoJJ1zgQGlloCibUz0agIZ5
  VAkhAAPkrCYMg2acx+WWXPh5zW4BMMVjOOOkg+I5kFpxQBwdKoFIllsMwAEJU5GAglbPRGAG
  BSY8OCRFIUBpHqwBxJGBCQn9EgyoeHTKm7BYbDBLCQpYhoABAYhZAgghgIABG8oGkGCAakYB
  wAFqEnCAiVgQgMEGHQw5pAb+E5xpRnEVaMCBBhpUgOhU4GqLQD1xMDBBjtWxBewbdRBH7FFk
  VVCUMIcw9Q2mCCgaTHUiIqdFwEn4Um8VpqpAQofkaDDRBGUIcBALJE80UgdnSrkFAQXkZQAA
  FJywywKYvfTKiEoeCtGnL3dhbAopcPBAbhBUkEIu9RUmwgYOWOY0K9hOQ8GBMMcp57ggrCCY
  CgliwOcYBVTAwXIfvBsBogWIKCcBF2i4Y8OHQmDcADRefMUgOIPScLYnFiwvBOEwoO8FwAVu
  HIa5ilijWGklMUBUY2S8AqgAoNCCCiCTQcCEggk2kWSoqLxFJdKedEIHCzhw9AonYJBBBgoA
  wKj+F6uJAoACO0nQwQOkUsHBCauT7HkuIz0LAggdFOh1ghJEncUBB4AlKgV8hwvaaCmMcIJN
  XpvBwAXLoUDCBxqEgwACOXoBwDjnIHBBw4qyxUA4Oc7eRXEQlMTABxf0ToWfFbjAqUhAgnd1
  w0WGQ1sB0KA4iYVrHQeKwMHEkDEU+CIOHMBOBcggAAqYy3MTOV1HDCC6iXlAUASwUwdcdz0Q
  qOByFngAbcAggAzIqgPfAsAIZpUQq0kBMkdjQQuENxCKYMQwyOvImZj3AOcd5QInoBESAPCA
  DTDLCgz04ZxAEDQJUAADKqDQ14owjOotoVIaWA4By3YWRNHIf/9Dh/v+IlCAAdTRX78YAI9m
  6IVDjK0bH5LD9wi4FS0AUIAELCC8vLEwuSnwjg2E4xM6VAE+FgEAv7GkFwbwPkwWaWN6dFXk
  LjAkEOaiBCjLwAFKmAU2HMA3pxTBKErRgViRogQYgMsXaJSCraVgAw8QgAlMVoIMbKhYDXmQ
  EEvmpItE6xQwSZCCmicgLGBSBbGb4gWY5sMl0Mhb0HujTuZUEQqAZgMuTMEFTHKek9Com0yo
  1AXS+IGylW8cCECD3ZwgogHc6wI9KsC8ClBHEc0PfjlZx/h+E4E13CiNE2wlWSYAvjUasCmN
  FNxUFuidSE5MAZ2cIv80+YQ4EANUbRLAhwT+QKQiJUo+RVAc5cx4hDnlwlxOAgEHzkRC8xwg
  BEV6UBi1NhGUheZBvGrcPSZAVKTtji656IR7+BIC4CmTZGGkiEUwkrzXSVNBjOGWCVcQzNkJ
  swMSgCccBHAAdJpgBGfiVjruYoIvhkY0dRXYABRAgQ4czwOYOOaJBueudwFUbujL2+hSRNHD
  zu9sGq1jHgmQvi18CwEaUORhE3ubC0hnW1lAEeDEVk/yLXJhgTsfQSkrUwfaaAIaaI4RRMS/
  D0nhEAQIYDfOdhx/pIaU5kKVawchALIQDqCcosIAYnZTJ53Oqz2tDQBKUKQWlHIiRD2eZnqp
  AhHkcp9LuN0Ittb+zA0QgLknGEEHcIeFDPhVe7OiSEW42lWvgpUxClDAKpmQrAcsQWQroEDP
  ihAABagEnk2D6Q98MYFyCW8kI9CTNALQHgZ48atoGkIwKICXWQ3pBLBTKxwKdtymIBZ9Sp2Y
  awh3QECFA1OI6icDIFBZseixnoZ9SlT8tBwfXWE1CJjxPBdpMB0PFJIy/VcSFsCHM7orAine
  iTIqYFslrIE4AYBAZhM5vnj5WA6PE4c4epQ3EnJDVdmBRgRqDIUBOGAEJgDeKCQTTRmy0gqH
  OECRtAbCCQ0pziUgVwoycgJPgBcOnckq8ewUgsZcwE50liQTDECBC4AGziNIohKX2Lz+JuYX
  euGUohIE/KluEsAELICdPQBgww5QoJvEyK8/ZBFGIm5tBHBJQx0wBAzLAMIXHe4cnyeyC/ud
  McjgyM03GKCAqWiIC8TgBgemvUjgEE1wc6ObcRRrheKcyp6Fo3FqMEsCDdxZW0Ai6Pyufb6N
  UrajuaIcIowt5UIlAQ2novKqI0CCC1QZDrZpRG/CNw8ul21vC86HwkERAFVhR3xc5sAERHyE
  AZyzQK+TQHj2ew8CUFeZJrspeVWAkaXFxASp3E0VVGeuENoJBBhRpQLspN4Bd5sACDDnXS19
  Lca4AtShVlwTAJCC0t37AC9c4RWHEAADnIAFKxQrEgwC10T+EIADJZNM8jrQEK39Uj3hPUAG
  JkQR/VTVzySnuEnajW1EGUCykp5kAQSI498c1ylEk5twCopz9FlTOQQ0rGffJ7jjQKCihz4C
  MQrA9o0SlC1JVvjAjHABFACld9QgYPk2so7lvA/zAXjGcBAwlHnEC7KyoScHICDJDmXQ8ugy
  DuP53eVDE0ACMXnd8gJ0AMvUJgNC5HOfswq0inRihRZ4rwk2oHYkGMAr8bXTCpZfVRAIODIY
  4R29nUBZA7hu09dyhX4PMP5h5MoJOkQF5ikgxBCo2h8ehDoGHLD+DEggEbiwrgqQLwHcbyAE
  E/J18GQADpISJuABHcFCHXA0FHH+ZVJgR4zXbhvlHYU0MREwNgZ0QIRjMC7GMFOhUn/SfEbw
  MmkkPoKHAhegUjG2DrBFAiiiBcxCUMvieHekIfI2Kk4gGzTGGusjPuiiE9tgeZ93b/KUPpWD
  HR+wbbliR/ryR7CxBKYyERKXbWxBbuQTdyP4AMwDVqF2PwRwAnuGXS0lX8WHJ8enGP8HaAKw
  feH1dNFXAiswAuliEcAUM50gYIlnBAWmcxlwAQriAA0wfqBmfmw4BAYAHjZnBKxGJIEmAZri
  GSuAMg8QFtuiX5kQAETVOk00TRJgARtwNL80KFNUgCrwEkskTbsSgLwzBenGeG9XUEI3DXNX
  TxrYFAH+xIHfsGyIlTb+JBt5uGCmoiqaVQH99gtTQSMDcHjM8YuzxTKMR1CPB3nxlg80BSTn
  ICpXUW4VoGA08gz8Mx6iYkehEgBYUW4RQx6KIwDSBha9AwAaEEYkQEfkIVBo0y7kc40PeIgN
  IH69Jwy1QUpDVIqE4wEeMAvFp1VxhkQoExP/9y3dNlTEU2hxyFcmQGcHUJAg8H5akB6v41Wt
  8GmhZn7oRlLasAFChHIYAA8GkALWtUL3lwRNx0cEMCsm0H/jokRexEWj8ApxIjvEE2GsMAHm
  1JELmAuryIpsEQyQF4vT4Bv1ZG3KFkAYVRbYdj4M5E/cBgUygmaKFC/lJoH+knUv78KMZJRu
  qrVASkmB8rYmOKhN8/AbkLMTCKAq/MMAPhQH5wAIA5ALH1APh1Jk25aM80QOjuEL5dg2bEAW
  3fAasvEu+AgFEyAMCMNr50YFADlEsyEMYrdog7YfSGQYETILIXB/haiIE5BVJhNyYaRe0dI0
  F1Ar1EOWP5AsGVBVpFBfPfdzaICFGlZgp6EGqNYCg4YKmQAfwQcCG2ABZtVBv/kDMzkRI/AA
  u9IkSHMYGNABpyMBn8Il1lICKgBXCeJeJYAn9ZE8w6QCjrgHS0kMatkFTvkb56A4t9Eui1kW
  jqRalEUN3tB0UmAbE4Bm9WRw9QQBd+QLCFABHwD+AbLJcMblDSfGUWppUioXTyrlWkJAWRmk
  SH4JCs8ghAxwZyiCUKBQjh96FRDXZVRBDejCZgsGAeOzjY8zT9NmT68xTxXwmJNmGQdAARzg
  ATuVIBPgaSQ0j0eBAsuUAsHQPBSwQ2SYAiJgCtByEYX2OtipC4lYUtAXfZz5EoZBAQcAAcnz
  XVgEJApwEBAJNKUQGtKkcfr1RvyJfpQmin5gAkGFSgaCCGAoGBmZkrMVENnEJVg1AhKwK02K
  EF/EdRkAdjuhAC5kAhyAJn1RfA6RPHCYAkfpBJZRDRLEI0cmWYTwRLRYFPNzKBI0T4STURKo
  lEDiLj42SQcqjOU2AQT+1W8ahT6coo59SZZwQlHwQm3AsYsLdFLypnYA8BrnRxwFcB2KVBSI
  wABoBpdqQ2BZxmZHGI9hoyozOjaf1xub5w9XRwIwOotlk61nQzhaQXEnIZ07tEwGiBEdgSbJ
  0mRU8DKewwIjEAC4JwFuWIbLd1eGEWfHY38W4BAWwJsaljU3lT2PGoe5YU6ZhgG02RH0VwWr
  4QAM0XLkdYZn0n8/xy0yxX13oRfOMXNBhXLmdR4vdDnImUsxxUkeIKegMCvRCRlNWhFnwhBo
  tRtz0iQ28QD6SoZmmDxNcgJYMkkgSDbk4yK16i+IUAXMkkZwSRbGcVwaeCnz0m5KaVAVwHr+
  /fmqEIcu5wOudAONdYMi/GOwl2SgDAAvbHtawXGM0hhvlOME3Mq1xDGXGvMuFXAIclB5b3kB
  H7oRuYIox7oTy4CCBHCBBYRj7vI+xTWYS0FA6JO4Y1NaGAg4JhYqTYAiFLABZoo06VUrGWAB
  Ggc9a4C22sB+fHYCsNIRQ1WGohsaflUCkrE0pGsKqjQFBtFyrANQFIIAIMUkJrAY6REal3oi
  BKAA6XI8IWA8uIkmD7AA5cdA8TZJ0KIXRcBURVIRyPkyqxF8wwmxMeUSkzBbWXUCWphOkjoK
  Z3I889cPClAnedF/NiSpwBNoG9Ak9mFGDlAdJHaqb3uMctu0U7D+DtP2G9YQDrxKZHh3NueD
  KGhDgX0nggs2SGukgeCKc1UJKqTXP1bgvSolgVXpeEs5wPgwoUmAWRqgoEtRjkkbSABQeiiA
  wNTKLix6dW+JcwI0bYE3bVR2KN7wCcGoAcTwPWNjUd8YQK8BXq8ACabbj9Q7AAawAFYgO25I
  PKoJupFRKzKRAXDWJGd4nTYLXgaBxYtGITsKisvnpQOQPMeLRTraeyiEBtTLtxYKBTGTXjTy
  MjpUXVplfTTyAHtGcihzf7KTK3DGAQ6ABDY0ERngsx3AAgdZfHVVIFXCRcuXk/JVdvXREbMA
  YhT8CEAGtxHqC6GsDe6DgW27yotpZFP+ATi54Ub+YgAUtU/FUVFddlzSQQCCg1igYipFMTr+
  pGXeIHusJbcoRQUy4i6slxq20aGaFQGDQABqVG4XIG6EsD7e0KpDwAADZJeuYUBks0jtcyQX
  2g4aMBXtMkCmxQFUNnevkTZUwFb55TRrwJRY5BmmlBIMOBEVYRgLyZByRivJoxmQcQJkXJRZ
  /LsXoMYwoQAC0BF/igUOcJ1whjxngl9Bh89RYAALKDM8GQAsOcgOwTRz4jkVYXKJYwCW4GqU
  oQDDFIeLQQHyJWejkAIMuQtpINIp8BKdu75zlkpAY32l2aff48D51J5XgFlkk4GLVIvBcWS4
  YTBok5+J68H+UMAulZu0TSEdxXWffGsqWjE6AkWM2qhAwiq3NEVgAiA2Encc7FBwOYY++3PB
  11wivsA2gBtlQsBJy7GNKsq24xxbv5A/umZ5NKZlR5y03UDX12xvU+AAwqtEFPBpaz1FXhGQ
  FBFGkQE0dlKeCSgTDpKwyJM8zWMsDgkFEU2dnFmKClC/6fXQAfA6qY0Fr9lLJfNLCUhNoYa6
  O0EBR4Ny2ikAKat/FJGRMeNL7qoXBXYQcFXbBNbIKvEA1DkKZOhqoMGTK8kr5tQkoHjTqQSA
  xdQAFDsAF3jNVc0WzFgJagRuJcZITjHCdVQAjQVQbpeij8Oi4SWWB6yBsfXV7cb+LwMQWwu6
  YAL1nwV0zaoVt/F2ygtmhSvcG7AqeGdzgThmroHEyxQlblNEbhywZtzaq4EXLx9SDashAERR
  R3SXy4PpT07RMHlI0ySjH6gQIEXtfAAgRJrt2RaRkEikexhAv+VSa6TwpJphAePykllNAclk
  xpXNYUBDCjDxJhMgASR5BgxxVRORRBsbVsNQBWsYK/8MOyRTXdaVEk3yhmmaOmaqGXFxBAFw
  AKNhJ5rxif4c5aOgGK4mKKrjahIgv2XoEK/DRY142ciKoPbtbN7xixhquUSmbC9mPvhJN+6z
  gamVoquRleGoZe5Sbe9jR1BxPnwLHdzcbdVg1gg8UAz+XrgnUimWqxzYMT7gxiNufVET0DAC
  JUF0pOnOWSmn8t9XDS9qxFAiIqvrkM4qdcTWbK69CLj8sgdwSDIOAZula+i9CQItAFce4CQO
  kXtKJBMWgCavZKaCAYrqBRNAFDvNJygdEAKzcFNojifzJS0WgCXEsKABgJ3ClxJc7jXVDrNn
  YABwyLsp4X6710TitwBa+Dr/ZxiZ8SlqQgw0rVVe84mbLF8Y3Xs8Gp6bXdPIk0w4XbRgDgGn
  kug1yOpQQFnzBA4lpmNgCUlLeKCnNWNyg9+HJ8+YWh2l+t5PgU9TASoo8qYSJRuBh94LHqG8
  TrfbwGLTdqpVXYPQGGS1OmP+Z8Mv/mPAZXOjSLHiXbZmgpO4VBE24+ygdbRuCjp5m4vbksxV
  YCXyBewAudC/p4l9TAMePtt/zeMAvacA215EW3ObMNGJIxCb2vJ/tHvGTmIRpTATWELAR2Hn
  wgOdBe3vD9CFVexBUeV+aBIeDtBEnab3IQkJQ3kmV+4PTgeKqVRLknoRKHMB3NIeCw9nUX58
  OkkKMuTgzhkBnufAMJ946+Dsh3d3PUKBbMEpo2JcRAY4+NRP7hOX6AdAUrmByzaB+JD0U1Bc
  Z5bg9wmhA2wjE5UbNIiWq1X8NpiUBWAAP4/2oFAJBzw3AjVkBiSrdoSVKOIbhhVQoM5bjjLP
  V8X+vYoBr0Ag+A2JRaORQBixKILJ6GQqiTKHACAQECS5A06KxVqtxKpUCdTJSB6UkQRwlBcD
  mI4odEqp+OYTumMDQ8IKAGsOETEABCysbKRDDYOC7cCSIDFzKClEjaJBwUGCYiOkRCokbSOD
  cuGAy8DBQSEAUxMgw4TvBCQjY8PDg9UB4cAAgWBgwMBAAcFhguLid2Nk7yQkwyBAM7OggoTk
  IoIBoSB54Kr7KGlCIwChQl5+IuLKED8foICh4kJDw4UKEAiWI5BlQDlu635cKYAAQgR5AuuV
  KzBAwD18QhhqSuiOxAcOFyAwsHgRYxZ1HY8EGIDgJMwI/wBSjHCzZAH+nemyCBBgaGEiADMB
  DkRgjsHNCEcv4rs3gB+DkumgwkR2j2URAwXIhElhIs2kBwoGZB0i4ECKFltSrFC1SoEhIRqx
  IFAxJsyKu2bAghgkwcKJDLY6OsDwS0SJEif+lOi0CoOFBwTwmZ0jIMRdR3pHeAjEisLYV4Qt
  N3zwwEEbDyb24BWTIgVjEyFCrLJAiAvldQQMYGAdxZMCBQxCCQ8la5ZwY0maOaCQIU8IDA+C
  lkagIdw4COaUZQRAep2AeBcCTKBYLwKEe3P1BWBgXgOHgBVumrya5SpLAFWTRpgwoSTustAn
  jtLkIOA97EgYqaSTksloQPCyAkCAAQhA4AL+Dj4IhwQUQpJPIHoiyoknrLoZAIIJ/AGQAZR0
  evFFAs6xMCXvKHyoKUMMDIACFloQI4pOJpHAgYPMsrCDFDw4Y4MHDnhggw0KPIJCD8hoYQzN
  VPiDNkEs6MWCbfRDy5Iyzbwkt58MNKIAA07oKi8VSghkFUpEe6U6swDo4IS7VmgB0EABZSE2
  2BhTLBUqbLOCoSsCaIYZBShgzU+9djkUjbdYUeCg75KYsrShOMiOHHPQ8S4rnyCo4L/tMLIK
  gAPkoBBDDmwFUTv7lMmC1QLyvKWn7iAckEBQ1xxiKwgy/HAc+7YSltg1BfhGQxSs5cPacETS
  YB5WcbKoU2MRoQz+Cxpfiu+DdDfkcNuALkCvRRvzWXOAC3zE8qtOWGHDgLL0pOAEFE7ogIIr
  DlDj1yEAoEAtQPHKMjawMhgEgxEmE/eWnwg4oAELMIjstkp8MvJYIgRQoC1HyjDBM0EoCE2B
  SxLuKICu/hT0xzCEoeALQ6UAoTZfMqCuo5+wMICCJVXOko+IUXkrSkGKrOzYACpYt1lTu1uJ
  JffkWeoK8RBIZw4AEIgAXfncLaicXcu7gIGZEWlPmbp5IrDkIi6caVSRSLLvxZSiNXCACTQM
  x9rEPfyAW6X8Qy/AGXVkaKNaFU9cW1zPq89UjdbkzYPN8vWkkiTMenSEKZowIFYJFJj+I4AD
  VLgXSzL6MHSEj9lIAQPWJ9wYgw0W0yM2x2qTYBuMSwvAgZQ3U2EEEKKchJJZLJE7PJVpHzSE
  JnpqIMmIQ6Di4+k46sjsC6yZnYwxLB2BA18SFUEEOoUeRM1jD7qAw6xn7O4QXSMAQRDwE9ZZ
  aGa14Fu6QMSqghxFcFbB3qyAIpOboMccOplX3oYAD76FJCDpOclFVGKi0hBgWYjD3IIYMACg
  VMVxCHiWCyeIhP0wQAPruhwJNLAUnURERf/5FqyUVzQHmMBHrzEBJEDjJAKcTz8SEARvfmAI
  krWEA2FwmO1ul6/yPQB6DyiiIh6QAT34CY2GEgSjSrYNAlD+QDNhaBokPvMyIsXsABmREEMI
  EILOdAAENxtUBw6QjwBI4A+NaZJxXjGhC3gACm3Ri/tU4AEDjEwBGEgFCOonCQt4rJAlc4mC
  xFGq//mEa+hLgsaqOCDYrQpd23pXegJkgIP05Bw1NIIhUEST+NSkImMzof6wQJQNgUiETMll
  CYFSmsIBBCAqzI7RnFIAgqTnKA8KIOUwFB8OYY4DCNBHAR4HuXIcJZVmEcADRjC7OJlAYtRz
  gDGgqJ+gUK0lSEwiJd3JFzR4SQLQGUxWApABDswOUBxwBgJKsAcVdGAQBqiaG9yZxDjJKRWf
  mVhoFoDHJ+ryCJkkV1pod68UdAL+BEBLhWPqV0h8Fg0Cr1BAB/biJ0J5gAKpCR4eVKEGB/zE
  lQaiTEIO9wH/PUgl27QnEV5KB2Vp6FYj8RZBAnTL/egEpEwFAAMyxEBkKGNFLQwqvZS1Lg59
  iFv1qCVKkprUenYDHve4gFnZ5Tf00LJBwpwc+u7BgFFJMxwjKYdJIuK4tnnOMhjJBZzkuKUR
  rGJfs7ikvzhohC044kc/stQYGPOHQERGAhAlWtc4ZYJADSYLZmRMkZ64pgAowJ2Zzew70fAY
  X9gJFGiqRVZE0AQFBNJmgmKBGRRT3PrBZZiNChvAssRPFZggEirlZCTUQIgNWkZGH9TWLC2y
  NWK9lYP+GXEHVNU2S6pGDkIJgQBl1wGUANxKAxfJiEQmIM6m6skdOlyh3+gTIO4IzpB7bW89
  lKEhs2YLhDYZEeCMNqG6pBCw/L2rUrYztut2DQEZuNJsy/AVjdqJU2SrrMkIEDosldRSZ9CD
  YyZ2GxVsIJRd+09FRyC0DpjCBJIp0o4QsAE4CVd0s1HFbSnhOuV8NCshSEGUwBAozOZsBcUT
  Mp0s4NLTfQcDfdoL05rmmNp2wn4fo4WA9VOAUe1QqkvhToWSSua8ZYSrOWxXQFhEVYOoRGzs
  7UZdGMBADQDomv5Rz331g8K/dkhxIWEcd09SN2Jh4Qp7nFsFGncruvr5Ajb+SSZ33HyL9B3u
  rByqiabt3F1Ca+KNIxAkkPWSgpTahg3ztCV4PxcAK2n2ZmKAmAlWLD3QxqZgBNVF8XoxscNs
  4A4dmMLrsDsACgRSizgTnfHs97LqWS8JWWBJB4abskC1zwyxaUyiOOnrqZ1OABloaKUsxRd4
  0gbMUYuSLzBg5dPFo0N9yNa2BuKMUz56xAdJkIHLq1Y7Z7OECBxT2uhMYar6qtMM2UYBcvjN
  RCe4IudESYncG/GyoQ2a0Mx0f+0jFfscJYPhmlAB3BFLkZTXW+nByXlRXoAGs0QAFChBcAfl
  iF045rN2ssSsRxyACyB0UFuUo6H+IAJYbyAFGaD+dSICgOxBvMInI9vYAw5Tb2dmuAOMvdfz
  UBGJeYMmNNbDelaH4ONJdriO1v6PtYXmCwsoIH/6OaJDW30CD4zkXf8R/MsOc7/IOECpjSIn
  /66lAg8taD7+Rcmw0lm198xVXSHH6wORQfnKZ2I//xBIejb+IoSPEdUBKMBcVYhgBo7evykX
  3ICSC/p+2MQgNaI8hCpUNxtN6PY0GX3K0WG3h8Dk5MhIfDcUUI0+/clmTTtUSj8zpHlinYNZ
  iG3Pt0jJKCyxF39Zohh/JwADHCAUc+/6JEYzdY8YIEPO0164wULds6P9jpfICEsykAL4TeCj
  CEAUJGAUrM2OnMSWDgL+lVCvbDjgBEbgAh7Alizk/J7hNMbCevpFGTZmAQjQAbSAoFTvPT4A
  c/hNrc7pq2avmUTpbVxu1MxrOxZKmBylKezJO3oC5ZgiHS4sKw5iANCGrkBoPkQIBTNIvijv
  1MrGmsxBJTiuWCik9+SFAVvCALbD5gyp98zFbiCkXHZiCkOKA0CA15amaaSgBEYgowTB+mKm
  tUomI+DIfZRO124nCv7AL0ALnoKt0JpBAR4goDLgxhgDaK5O0rqBNy5gA/pEi1SGL9DQ7CCL
  yDjq+pCsIx7gXQqIYySg8MqHEzfxZSbgkmpvzzKNU8yvDymAE+uOEylgAojBQjaGjbqmFhD+
  YAJI0Fpk6QRrDlq+yw3h4R/Iq7xGj5aSLxkcRRm+UKuAYmSkoiQWzIU8zhAVCMICaz4yDuWO
  IUZQ6R5EURFe4ie48Amf0NGWTxN6bxt9okRKqIQqJAl2MKnMwgE4wAMW40oa6wwcI3rMbl/y
  75JK5iAkQBfiUCCZRsV+BqJACxAlgO0Uhg/9kGdWo09gY042IEy+LgOWpKJ87iuW6BEh0dpi
  zaP2ryN4w0Iy6X4+CSU/6WPo7ReGwZYWsoMKqBYO4DmEZhVusiXrDjIkUAGRkQ5mMQI2ZHGQ
  Ka8yyLt48XP2AW1sZc5GzuCKcka28fOEwr0QAvlggqoQgBt3Qxr+QU3RqrEolWkntNFRoPEI
  FlALnEIdmYlAXEK+UuUc2ywclXEDd9As56D5gEYXnqd4qI1OqMfIlqNkAOCIJmlQ2sd9UmZL
  iseTbmMDTEAh5SB5QHCXkuAAoOEXPCAPogBohuGtLgkfJGoOLsSg9mAR84IxaMP+ctIAUSMk
  FxILaFJoPokAUxIly8djgmdiCAEmG0ILDkAT1QAyUrETu84XBEECpzITsuAhIgDUGogIdZEs
  1+NYbojS4MtduuXgpJMuxIV15EI0kxELXugVYcQujQA0DSE8R1MaK24oGy75am5GyJIHMwGo
  tGAt8awntIAs0fEh3O8yqjIc5VItq+L+5lgCAZ4j06JqGjJgGi7AAF9mAU6DGIYOQNfBAMJu
  w2oH+jbLUhgjetTQMUNgtEymADygD04gflhR1oxhObhgZM7v/A5Ahi7zZR5pBHgNBC7gp+Zg
  AA7AoPaSDKMgRO0HsgwQJEcDNg1AinRSDGfjpAoFDc4ANuBpA/DOJ4kgC4Cz8CBR2YBGpe4A
  BDIF3qiAAnyHoCCtKnJwJ3ZxLfPhQsvxbDKNTllFRQItB3PQqnipAEBlvjoE8lpkLAn0CZsQ
  Kh7i9uIDbijzQFxCKq6z4IgwPmPE0djSnvgTHWVEJ6wiT48PIlwlSzcB0pQxWJIBRphCUzf1
  sEK1ISCtrfL+E1azDpVK5gAyA+lOzEfcZ5L8YDacrnwCagM+sAhayEosyg/sEMxYwUmqUlIA
  MaNwrDVsZzYyQDgOpFagAI38wDGAhrrsZ6MO0BWU9HfKKAMSQzF4zaFuhw8QswrybkIexQ+d
  tROGBzb2oF7NwA+qqxCubC77tVjiNBNeorCEiJZEKPbEMlwMoQaZCgLWBdNEpLtSYlQhjR2b
  U0X+ARdbpIgowy0fYsHA5RXb0feyTh3rMxM41iXqRkaOwiQIi20Iq23usmzKhScqxPRy8OQI
  ixz6xVFOB1Z/Fmi17ViAtAQqymbKYHb24itYSg0H4e7QtIMYJtw6Sw8Yo1BOAAP+EuCSzseA
  JuAUZGN4zFAxzE4PkeAl0A8CoKEVhQg5hGMWksMZjIEZmGGVIERPvgNYx5Re65VvY8MDyENm
  5wYoCKANfCESUiFHl2gEcnRxgyED5ulA9cNfJ9eQjoU/WhZn2fT4Nu4Z8eGJQOU6igI96FQg
  0GuY3EsZLNYfhhBc5IZc2qM7KfcJ77ZjPZVlizAHbaQQ7fOFKnYrjg9mm5Ft5GXEipdmYqEm
  fSHTbmsln4N6iowAL9ClqqPEdiZC626jkuetfsI5Co8SoNcPR2ESwKtzZdd8A3dWauH8UNEm
  o0ZoIJcc1Wlwzy8UHkDwWjFtXdMYOhdgjdd4r2pT/wuNRmYkCUiIUIcpRazwZlt2LOXiLN2r
  Qm4399JxXM7Xgln1B8ildjk3vcDxOwazdzWVZQmIO1fQf08YhQ2RmvpVlxKWhTE4hTHUhec3
  hmvYho+FGWRobnc4h3l4h3V4h3cpdgt1dzdhiMPxho33o4g4iZvYiZ9Yb6A4g2FYiqvYiq8Y
  iqm4OrGYi7v4B4IAADs=
}

set pieceImageData(Eboard,55) {
  R0lGODlhlAI3AKUuAAQFAxASDB8aDyIiHC8mGjUzKUI2IlQ7JUBCQGY9KVtILWpGKlBTUHpI
  K2ZWPm9YKV5aWmJiXoRnMW1rYoh0RnJ6cp52MoSFf62JQ6iMWpKTkracXaCgmcKkY7Wphquw
  q82tbdKydtyucc67kbzAvOe5fN/GlOXNndDT0O/Vo+7hv/Xu0v742v768v//////////////
  /////////////////////////////////////////////////////////yH5BAEKAD8ALAAA
  AACUAjcAAAb+wJ9wSCwaj8ikcslsOp/QqHRKrVqv2Kx2y+16v+CweEwum89NgsBQNAgI6Hgz
  AGAzDYCAfM/vf9V2Q25wfoU/BAAAAk2JAISGkJGSQ46IekIBAgd0XwEICAqLRAAFnwYGBKkE
  AwBFmQICAZdaAAoXmUuxGQqtk76/e44MskMBAwwFvV8CBQwMA4E/jp8IBQPX18pCeY3aV4jN
  eUsBEBOIj8Dpf8AGByMevJQXJhPeWgMgJiDRPwEcKSdMhOjQYcMGCrOEEKAwwYGCAVsAODCR
  4oG9IgAuqNhw4GIXAXneqFIFC6Q6M+giGShAQgUEZQE0oNAQgN+WAuUqIDgwKoL+hgsXIgiF
  AAGBtwDW6CS8AgABBw5GbQoCgALFBE5jumnV2ufAKohlFBU54PUXgQkpTDhQJsBDigxL75VI
  UYJfgA8sVqRIMSIEiBAIixDYgOGBA1FZAmQIiCFlGwMbRmyQENdLKg8sVHTIsCEDhg4qVFDw
  eAaW4y9iiWw67WfVhGEwI1w4B6ZAhAkTCvAsNoHDT6ERiBp1hUzplgATPmgYhtgIMxQkJrAi
  k2jAh9BVVaCI0GjPgDUKHCBQAzZMInSDWReSOMHkNgMZhlsucaJukbaZAYYYAUJEYCIDdPbA
  QxEpYNBa4yhAgQR4kJHKBiqwkAJhGICQwgoeMEDaGLH+CHBNLAGol8V5RBCQzC+K0KFNiImE
  gUA5ue2GSW+/DVWUNwAwQIBxtBAQ3HRKAFBBBAhgJYYxpJCwQgstrFBBIieiYcAACFDAwTsc
  UOCAG2Ko8cYQBJhAgYh9dEeELBtaQcBc9hEhAF56nRBCCCIAthQBGVBgWHkjkhUiEwAc0KCD
  eIagwgonYLBBCirQg6AcxiDAgClJVaYFIo6ACcGfvphJSYthGHBbbgqMMoEGNQZX1FKeDMAj
  LSw2dwQzO6Z5T26OfNCCVXm8ZisXqEzgwQmhpRXCQQSGIcBsvQiQQggKWOoHiSWCOl9Adnmw
  ggr6gdDff2DmuedH1iaxFi7+ZaTCmaEqhEDsCRc4AO4ZnkRQwQUTRDApKrGYVwBWyxoVoipS
  xUEtmOV6gcBtFeh2pk+pCscqAq4mfMUmdJCJqbRdDMABCUYVsF0eMnGHxgAUmBBhXtwKhKys
  XACQ8lV0NJpBh2+o+Kt54pxp8aUUlRDKKeMFoG1oJ/wlAggYvCFokQvpmWzMXyqhxs5ZpKJo
  ByWs4HUKGkyQJ8edMFCBBk9pMOSk1cAcEQLScRM2K3lco3OZPRfz8xYLV6DTwFdDHJSNRYpU
  63evRpQ3Emie4bEKIDdF8rb1oOHACCuvkFnLm1FQcNYFPJsBHgBxtEBkHmQwYK1k/iHAMDJu
  gkf+61OEuRcHEzyAGwQVMMrtCX31t8GWp5wib+6heMHiEgBA4DYYWm/grdcqnEABBRiMhgZS
  FXDwAQkk+LZ2NTtCT0BQrsb0twCSInMN63FYchTULtq7NgT6OjMBUIMT5Yw1sJBFJtBEtil8
  J005whoWHreCD6gIAihogQoqd4YMXCgvLGCZsZAlhoyogGkEGMEJOrAABywKUSHAQAZ2UZPt
  EeAfFOSFBDpgkT+cgFGay2BetlUsfYCgAz80iGeudz0JDIhPWSBLPIKUAdpp4Q2c6QCxRKgC
  D2BPe/RiwAW+VxXlBKUo1VBgFJo3AYrF5AJFGtg1mvGjOtArAAyAACL+KMGAD9yigFTo2wXQ
  Br4P+OYnQAEOBJwxqQIUYBUBTFzWPoHACnCKDI9DgYYyxgHtPMlxI9CcJqmnH8I4oABhUMNe
  MnCAC6QABBKQgAUwYIKkieBCYjLSGEJUySoyYBsPeNe8gMaoDDKpBSzr4V9+WJDO5KmIRpza
  pRjyPIUYoAOoQAmeNnBD61lwBSPIkxivMACffKAqKPAjvgZZjUd2QQ2TAsn+IPAdu2WiFOQj
  RhkC4JOgnAgAA0gOCRwJBj1q4Hsk8OMfLzCkoThDAdU4JOII2AV6RqBfjAvABVxlBo+RQEMD
  iJxFKViGAZzAazvkIV02ozokkosAB5hACUL+U4I6PYsgHVDZCTZQAgxsoplZM2UwK7IGE/Dw
  BHBxohMKMJeVsQCYO62PX2BqTCKmMpkmZQoHdkGajJzgMGqSBUrJYgCtCgZPacmTEEHArQvY
  wxKpQMWOcDpU2fzzA35UW3A+UQD3LKNuCAhkNSSVUAAag0iK5AIiPibOIYHEEd4kgTm1sLD9
  /bOPAyWoIJ1B168kcnEjsqN8jgAA2cyxCiq6rNuYoQgOLGlXAtusmwTYDaEiYU2h8Ron9TFV
  PUVVcQcIQZMidKgdqqAEmxEiC0yQgcNs8wkA8GkwNfOAs/jug/WQERYM4K5e6jAz2wKICUqw
  VGMes4iGOQBbpQD+AA/QcA1IEEArJTBeJWiCAtIzwQg6QAGvPFINHhDTAxRlzJhiERP4VEAG
  gAgCD+AumlZohmPTJlly/uu4TsDni071xTjmj7JJ8RABKAbhJXQEAd/zjWQ5jBTZ4C5QXXiR
  vS7gvbiiCij5UhXbyPch1uIRCgAgwXLQdR+JXiBKU3AE7/ZI0OK48VPLOpRsIydLRLDvXnql
  qBXwQaxDFUtOJNXSbUcUqBIwSbbBzIsPJ2OBzIgAAzbtcJAUECEesnQDzaVAlee7xCwMAIhU
  5ORetjunYq7wmEZ8gGEUoICyNEEBD2iCAEYwAs9dJAAK2At7myAeIwBgAh24YAZXIN/+eJXv
  EAFwwGseIAEiErGGh3AEBdxCPeoBtWpUKMWK+TckcrZNzUHq5v6ITFDcBMdGk9rRMdroBXxW
  IKAirsAEBsmKel0gcueUVDn2CEi/xZiQdDXkd0DE2l8tAL1MCAAKOPDQi5CCAw37FawxAQEN
  gO/dAVVbkZRRnTgSBX+ABRUeuulucHZRAyarQoBG0EpGGWsz2TsiGEByABN8eZPZ9VpA6kOY
  Z30maYM6DgEw42ZGyWl45wuNQJp443BT4AJRJPh8/bxCpwp60GSJeSqaMJqujjdMxK3zKDLA
  LQqMFxbJGwJIVIbB626r0eKFRSJeYZq0oqIVeXAAq3WYQ+r+yU7gkhIKbvI1SIQaUp4fEQDE
  0vbiXm99kJR9n6QezAVZMICwIsZXcJ6hiIXpWMpaMIYhs871GdP4Q5fdyueG0JF/kU0N0Ek3
  ZyNAgh/bCilA7ioXwQlvP/4NDlvJfCtkwXgUaAfe7+aAXaWAJ4Iwerse6Mz1wjt4K3T54SHd
  3OZEPhAKqawEFmh9FLqaAdleuXojV4ABPNCoYz2g5EwowANWaBCCePeYhimhQwot81RAdBwQ
  OAFlloCKvXSAvQgwgtH0YgECtN4fcEZMoDb9u72AFFEgv7EjLlBlzYVGyZveQFerwAxpT5Zt
  SeFayKVF3oNsgAQUytZ39lYp2ob+fB7WFO4WPgd4Afc2b1TyMbege0+gYfDUPtmmbdwmQDrT
  DRqYI5EzeKsAORegISmxIxeAAmiUJngQUPLRWZSHbC3WRxrwEu0lDQMgE59nR8rGYu8WAWAn
  BctnEErYchLgAA5gaMW2JrvFfvenZFXIXSVAITFFUrg2Cg5wIb53AsQCAiXAXaSkACI0X6PT
  hZiwYRTAfJ2BcuDlhIQWc2RBEkcYJG7RGEuAUqHRARiwf2diACcgITalMdmEamrAASwjEAXB
  NRekF8PTgwdgQVYGEH1RXUpWAj4HWhv2CYREWSuBOKhRACwGV3HFPwjoa/4jHAD0BoxEC/kE
  UHHnbtL+ZmSm+D0awgWwsHeg2FcgKIJoknkRtkUCowRIATkAJ0vSAGIwmEbIKG4fMBwU803h
  JFeT4lbf40eb4gQvhB3KUUZ75xPgowGqtXsSgGZ/Bl6FhmDFlgFHBVKb9jW+Y3BpsV2AuEp4
  ZgJcMl3+4HtI81Eb4C5J4zkQwGdw1kJZAw0OoELetXoDQn0HQH3WByLIdQGNRjYBQAGZEQJU
  hREUoDmJ4gAauSAdsQ1sti0mkI+ppCghAIYTogaAAiHZZQKp5xkrBIncon896BwD4IvUkFCI
  05M4FgEtFlmqaG3AoSp09R22MVFEiQR4wAD/FE5xxwEowAyUBRLI4W4c4An+wKJ3HWhIZFlj
  w7gVJddZ7JQpSdA8ymgOOBIBVYFGzCgYErWW0sAAXbSChodPCoY2yoFZnGVakIM77oMNC+Nu
  yyhwBvAATmVEDXCHqwA9J/BLrXZa3LIXmimG+EghA9lKTaOBzIMAhUg9LOVTYmMC2wVnhKgP
  B5FxiUEACvJnF7AgTSiR1QcioikEXSV8CNR7K5CF2zcEJMlxJaA6COQndiBREucBheGYtmkB
  NGVwCiCTbWkAVnYCqfNd15MBHEARKUBDDngmP7l3ZZlItbFFqUhrUKaUt3Fh5OMhC/MSWIBP
  vUECVvkTZ6NY7BNPzfNsF8WGrvAG8ESW2maWI3j+MGlgDHWJERWgjDqhDTsiE884epz1YHAg
  UdCxHLGgCAvFPj7BAZMUJAJwKDoWAXVFJRhmNqgSAXhnQCNBEh6iLCRwVBlUESHyHV+oFwa3
  XZzpFyBAGBaQjhkgAZ9FBShlAr4lciK0Aiy2XWW4ARYQAX1REOwlgOHmABcgRLW5IHRoh9a3
  m0ZwCmw5KyPQJEkzPPQ2AZoEVEaaBDsyAdJ1CEeTTfLSAQQHPAeBPZmGe+LlGI5giSOQJRNQ
  ahnAaCvHGSoTAsMJWthQEiK4cP8kYhBgDeMIZXIHbDOmUFTCYVWgBuTYRR/TRwHFjfE0ABCg
  noeEGsNmayuhUImkIlj+ug1IIi0xUZhBcaSXBk4raKE+A5sCUEkfsClwBFDKUQFrSSU/kQyl
  4grIEVA6cQ0Ls0VpQyRalKvjKQQOMBJvUBIlsWVMcVoSMhuqUIm81TJz0hf8MSciYHvPIgFH
  FmRoYWX352ZWNl9+MToXUHtjIqA/EC3w8ZJY1lQLImgO4RWxIKZnQmiwcASnoKR6cSw+xwZ4
  AI9NYgJSegqzsiydWCKZpAIjUBgZUHBIk0J8OkISsAl2SQErlU0L8oaQKIbAZRAqg0rZCiaP
  +q0gEpVO4A/i5KIDVGL38hMxBmzUoFD0hC8KSV4FcJ9VUYCVx43WYDcI0D3D6q+8KQtVC2P+
  aJdQltVtmiAFmYAMgnkIjyOBZdQsMVEVjUd3jNMURXImL3hRxlABbKuDuSELpxI3lgYB4BOt
  x7BHRxk2BqBFFfAMDlg88sIBBtZyDSFowkceoJR3CGCjKUCuKUsBH6uZzyICBPEXZVgfIJCT
  oaF/HeYPSmavVbgXqTMnHAEBzpdmWCBsCoI5Vrhnx6J67IiwAiSmdZA91lkiBuBwmQFcVJUI
  B1CZeXGcgegYsHAB8OANG5cZ2SQBI1sse2Ez2FOGxfVpKElWKpAljpkBc8G5xjudIHB8UkA0
  +KOUlNKAicCzSYAUcKU2/8IALoq/Q0u0MWZrlbUK5IA254hcVfv+MdDhG7lhihdFWRSTDWL3
  Txx1BR0CN4LLYMpma4aESN22N23ZtMMQF/RbmGoDG4nAAPgJHb0qmFrLThgxAdDhogVwbAfs
  R+JUJE27jOjgCIRFMTihnk/hPeTWDEFROE8ACMt3QcDkNXx2LHkiAZELCwo7pgBgWpozAl2V
  jhJQmmmhXUGqhB0QAq1UAt5SpH1qAYtVxGTFQ3oxqCmzAlv3hvyxAcI3YJ1BklcQIg5Avqmr
  ST1UEEyYTJIJIkKlBhmQOsErBALAZru1kpNxDhOgAkzyW4QhXvcxANA7G7LyjyCrOouRmZxL
  XBRAht+nsgqBJ7pEap28mQERXGIox8j+BxLM8FioKFddWw0V4ylU0FURIE7DsDD4yz9lp2xC
  4Xfkg0jHgKxnzAgFjJ9o+wnuNgAmQparkAcI8E+3MCKewACngircrJ9993cheJZPoLc7mIee
  wMzi4zzV8YJsu4LN5qzaHLejYMIweEhGWapx9RR0mVfmqA2dBT4T5SOnCFdPIbWyUallGm5U
  Al/P9TtV2hlD+gBcRYquRwG9xQLwkjI0pWQAoR9+rIRkyGftagEDthknWTuYI1sSFxSZBAon
  Nywr2VwmZBDqqyayKTbSA7re4seAJmgLQJEVmYeAMgF/QXKmsiQS4pq5FyIWqxcYiwE6lwnQ
  G73OYdHVozr+oYy9e1FNK+QtGLCtolALhpICHpBKeqzKdEEY1FRT7sgEayFhE4CpwozBAHyW
  atYULMYBFAg3wQEU3Wx2+cI2kwKAiPOJvsoIBBh6B/g9G4Z2wZYIeWVHN4sR7xSUoXi0nBqr
  do1iBIwq3dgTbOtFKArZ1ojCa+nPcISsQAImBYCVIrpheT2qvvEUjvQ6SmsH+PQxHKpFgFmO
  3oNGMbyCR+oEU2J+J+edf9bT7agGxlCfbKrSsSVSmSlyPM18YCyG/LEZX7wZxzerBpDS0K2d
  PpVS+bUXYjwm8EHTky0IsgAeTjh9dSgoqKBWXiKMWPoGMeWRjyALTV09P6Q6kKb+pItMGA9w
  fcvCaI0RFwagAD4VnnqyGDzqyQChQlZKyf0gZ3xRpBjpydUjhh6ZAS/5fTy7I5AXlNlW1yO4
  3s2Y1xPo139dUAb1CfhWFFNb2fQTBXB0iuATWd/DRhWYDKSwRdf8RAIEC3bjrQsljFqRzOMg
  AN2zg/SmoQf8GwIDAdYYUBRIN72QB2Yz2iD8gh+QwO2mHKOqHKjyL/nkeNvAeCTgovz8WPH2
  FBFAANfqqVMwNHZ4CjIqCwawACNCiBlUr6rb4R7eGWhG4a3UMkH0GWE13EU8Ag1dhYdyAJrb
  o/5xAANAxzWdd+bXDvJNEl7SISrC5BEmZ8ADF3hQB2f+CkyKTl8B8ACQvFsjVKSp3lUYaQIc
  8CjO0Xs2Q2qgoZkSLladsQBwsAYWgrHXA77ZCxAUp6hvgVVUEAufeKBJHhIjklcf80fdvO29
  NncY1liBPR7pYwDrtE05DudxR4R0PkgOVh3dw09dIGFngy+FhEjh3AjyS0doQxOKACUUKoFy
  5wgVAFDhk+XdAAupWkYNaoJt6yE0AudmDrRVS5+rgJXmGABnA7Vkjm6GtD+4ku+HIHUBcZNS
  U2iHXAV5YFqszi1KFhDZK7Oe6xmHrigqc4nEhAFGtAECcdJQIAA0VWWryygCpjKYCIjRkumk
  hLWW/JJUlxaGXrBAreLSQAH+BjfKAtAAsb5b3HLeyptdH6c6O4KGGc7zRrAJl/h9y3dl9ujg
  nAHVa7AooJxL073VK7lCL0kXDaDif6XPhcSpUi8N50Pw6U5ruHHBDGzLPQxj+oIAbiALx2Dn
  BHwqP5zuqAKiYBRG9rkcWLsNzSDbXySUSY4mIL8N1ew95fys0AHwQPHLg2tPsnqtp92WEUhu
  rlIA3Ay14pMMTStHAiATDXOBxlptFLO1abveAmDRgf71UtOPVlCxR7Xyy75dS7xUMj+k1ptp
  vYVCnovzq7QBIgCvYhvKRVWvmnE95e3RXw0LheroalIAekx1vxWkQ2RET2h9WNBTtLcL8Pj8
  D5f+vXxcLEAQ6mwwC8dIVdpIBITfExoNdFTV01KSWqlS3dTpSxxSCI5QqkORULjeLzqTAXWV
  BGcUn9cHCpUP54IBoaBgYEAgQE9xkWCgguMj8kND46KiYiIzAiJiU3CwoJEAYeLCcsITATEg
  gKAgAGBREWAAooKSg9L0YmKQE+KzAABAYEIDIVZWWZEPgoOEBHACAgHUUGAVNnFZljUC8m/6
  QwWFJDLXNOKWg123d4CPIeIi4hVgW4/g+4NEg0EAwDxc/PrRQ8RgAgAEk1QVmEBw0i4EAxDM
  q8BgQDJuighsWMGChYoTQjJQeKDAwICNsgA4WNGiBUgVW76UAHEzBIj+DTvjYDDJwEEGEzO3
  cDEhBAOGDiU6ZJCgcWUABR1CgKkyU4WJDBtOgPmiE4MCRAoOIFq5bMCBDSY+glyR4uiGOBTU
  OCBLwOxZPQAyXDWaggVMt1lHgBhROMQIE11zCuHQAcwGCgqgKpIQ4uqJDhg2aGmTok2JrWkI
  PCihle6Gz2+ubMDMpYMFfHqlINAQKdCgRodm034SUEM7SpUyofLEAHk13fAEFBBoaRMEYKsG
  GKDWexmrihUu6No1oQAD6Z8GDQuAUJhvWQQCIIBAENCmaoQM8WY1TH0UWhPY/eFXDh1LMImg
  lF24w4QBQtzrBAFYKsvDkWc+uEAVfYaDSIP+Chp0KIBvIjiEAQ2giQi6AQqwZRoC7snviQNK
  aOutEOSq64A7WAwABJhGyMCDKuggAoOtegrSpwcOOGACJFRg4S3QUhihqSAjo4w2AwjIgKqh
  qliBpq66MMGmDSxQABZsHrxxqhNecgsunXqqSwE7AGLxCQIMeJHLPLlMQS6f1HjgAbsUUGCB
  ByiIYwOqQkiMggNaWUYACrSEK43OPvsMhA52NIACDzigSyjQPjsh0xKcDCEsOn/AyxlzcqPv
  Guz0GuCCXDrpzhTjgFEOlEIQuYeiW4iLQDxg4AmglkoK0MsAWrbbBdppdp0PoHsKaNAAVfEI
  QICK4OOFGmsOyQb+PzphKcCU4NhJR5BQYCVEQQSSq0a8ejJakSUEav3joma6g2REDT4UgAF9
  KnilDxFJnIYQeRgul8UIAJMpBZL+jFNVB1RoQQUHAJhAJDDFtMsBCUw+2UgrFegxpCWLGilK
  CSw4jAzfAJDANVP9wurLEoQQk8xhINY2AAc6a4tJwnjySYKTamRPWwAoEPWopkxyAGtATQ60
  gSOftvMAQ+PIoMYz8wDAgAysgiuODkxwA0ym4rigjEzE9uDtL04wIVOvRsrgARvzA0AffiZs
  t5BDMqJTX39qgbYCYifCpr77hG5PoEoi/wUBvJAlJRDarH0W8siTo89BYmDRFg9iKoL+Jhp6
  diVkt2xWV/UeAjOhBl7aXQmlPkNYQdYQVwyYyMHsAICg1nULCa8UXCgBj/hqtCsFncgHGaDe
  BJOnsy+QQkpBJyJMQolOAkYIiYJiRti7BBGccvQebu2wshEFOJiYyaS5CNMphirBCCTgmwAY
  QCkiKIFVrqI3mwxhfqmTVX4CQIE5BGYwT2pKBpjmtBpNUC92SFRJGjCokm0lBKcBU6Yg+Kev
  2Qlr7BEcSw4gFB+BAFEb8IAHMjABByDgAAY4XvGEqAAEKGACFLhADpX0twcYAIQrAUAfRvQq
  4B3CbBshgDEY0J5LfBEVDRLaGMcYD+7oYnPjCQUrTCQI2sD+wkTumcAlBDQBQRjAV6kTmgBY
  px/uPcQc0pgWrGyXxbPAQl66uRYDzjgcXnSiE7NL3PB+hS9lECNzGgjXIiM5H3iYR2hxFI8g
  4NFGQbBHaHQygAeQ9pZMcfB8B6CTAFLQAhI0QQsmGEKfKJMMPbaHA0QJzBa4ZJRMYcBkFoBM
  BtTTkiCBQIEL7IrPMtUnDARujH18QqQw00qRQEkuJYmlHaJ4SAUE6m5vEyaT3gKGvS0KhxaQ
  wAK+Nictog0DedNlSSYwKAYoQFAAxVpAyfK04wH0UItBlZHKuZH2iMhVvEucmWZ4FgFw5xWl
  wISu7OFLMiILIf/SHCSRs0Y2ds7+Zsi6Fid0J59J6jGV2vwNARACu/gMEniFrKjNBDCAUcwD
  YNAohzkqIaABpaIQljNk62oxR3AlqD69693vRCG8ShLjWoXIJp0GsL5hunIIRXLAAVRCwaF4
  QAEXEElskpKUAzwIigGYgJrYWZTM+KwpFrAABzaQqZ1CSgAPkEBbOeinkx22LHuU6QAyUIIl
  iW8Le4NgSZpGzw+yLgAHoAAH1AkSmHxWMG44QQlsIoIOQNACDYDiUvEAEALgDIc+PABQlOiB
  w5hAMYfxAE/mEqgg4qWnD/CJA6DIOu5BFBCIq88qKDjHanCnE91TUUvwEFeKRK8dpzjOS7P6
  V1kAhFv+ceRVHpU6Ru+yaIq2GJEggwE89gzPkuid4i0IgoKroGCoRf3iUV36yaEtY3S/8FUc
  CcSd7gynqKighucuV15tDWBJMKHJzyh7krLmhwInIIdmKNCsAzQFaoqYWmAk/JE8cUGyHBys
  CkbwANYuQnUEOEAD9IoBvcqzaTVqwu20KYCVaUF8SZNsn+gCqCN5jmgXcAtoPyu+FHhgs3lz
  IAgoHIe8WNQ8nPIAaCBbzMyYAMzULN9OdoIBB6ROm334FuJ280kWMRI6YuzWROKLh4r86w8B
  khZydsMKh1ygobMoY/CEV97/9pE9d17vU+cDq+F9j04fCyoJ8EsO/I5oc5D+1MRxdANpKQ6j
  W7WiNH5JPdRoFNUUGyWpbraqKjt5s2ItbFqczrsMAygmfu1bAAAOIAEyLSIALvssl1rmozAk
  pWlnwMCLZ1EazixwMTXxmYwysFpmH9IBW0aakEewNLrM+siBhpSSg8xkFoRgMogQwIw74IWj
  iMCaGOBWMwFwgRGAxsR1vcqOsJQTEYigmrzloFiuzY1RXMCmVmwzePNDAM7NiRaoBLZAwJE9
  Bc/OPhTpIr0JJy9IQlKiHS34ITGHcKI+sr0vtVySnwG7UkODAxe5XBxZKjk8flLcUVDReW7j
  8pdXoh4rLTB/d4WRQ+dHAC7BIE3AKc5A4QW9OCP+gp1+MAz28JEZ4itxsY1tAiBdsy8jGLkU
  HvA+fOvJ2KTCAP20CUUCSGom4vMRbif7bQ96rtbLePsIPNABELwItCzoAP3MEwAJjDZuYjrJ
  oAjXTATY1irFPLEHJmCAJhhAARjwt2khaGN5vlWbPIddJdh8jV8NzhV4aXz9iAFsYVV8WNFp
  13tZ0Zyct44WjVSX5lDRLk+zbufO6Q7Mh0Ue2h0LvnV+ozEQLNShUqhalwvPLz7RCG2k9F+w
  GzWAxDhG58zROKnoHI9VJQByD4bFLVQDSrA+uAMsIMStPpsDSNyCPLUsaV/Yp40Fe+/2v7Ev
  hGkL1OnE+IQMxo4RtCz+7oKMJqqGt77NLrzmvVhEiN7rSl4CtFxJAjBvsDIghUgrUx5Aj7LF
  gCYgDjigmyRvCzYAAiyvFV6LyljotJApUMYKAWEsAiAqIoLB9O5DvszjN2JqLwpAXSQBHcJv
  dt4LWBbHZgICHM7hkbij91SB/KJGH/DMcCICE4DhjlDn0VYO6Winp76B0mxKWeJFXpBjPAbB
  cv5PitrDPVrO+aJBe8yjjTatk7bn6GjjEMqOAauAb4jMJIKop+QLH4QwDzLLBEisrrYgg8Ak
  rHDMA/MO2HgkwuyPA6qBAnSGC5CpWTBLfR4rJmIC/UaLhcKpSBZgUI7symiD1obhAC5oFOv+
  Dw1ugspy4t02460QUS8EIAJqZCqIiSg+4gs8wAEMhTNEIARMK1GawgEA4vS0RUUixHBIbz4a
  ARss5/Zw7xD38Ae4xRnOgR1y4aikY1eUUKW2EQqGgQH2hV8mIlh4QRB8sI+W5wIIQvsCaaS4
  EAHwyPrgqw3zo/toBR+17w+0K5LGIzn8q5kchAHkcNS0jwMEATnq5eMWEs1YBESEYttk4gSa
  TpzsIiUKUab0oAk8wLNmsSiKqStGq8zkSSnkjYIGxQSGLQNYoYZasgTk5Aa3ZSo8q/7+MCuo
  7LTC6U9KqKDwTj0ixcfmgNiYhMlQjLSmkhnlImh6kg+IIcOIYhj+/eK0cKIxTguCJGAXvXEl
  2EP48PEgJWd77CN13LAkf+Ag+EMch+PizLFdsoF70qIJgakdwAMRnAMTVIEX0YsI6+vl9FF2
  cMoty8gsZQEVDGEf8PHSTg4hjyMvHxPG7kFfIDIiq+gSwg/kPuFeNLMbEEBtPkIwSJFJ4AIk
  L6ZG4CEuo+Ak8y3wigIuFigFlICD5IkKNgArTfCxQoLflCKFTEAEnq4noaA9OkPr6o81J+wU
  38RkAkrH1PEJAoBPLCUoUzImmgTxcmIIhsACdpGCWgEDNEwYscIvfAYE4WmXhqSXlrPqHOIZ
  Sq0c+EU+2hL5MnI25/Iez8GRpqFY1pD+9rhlIhAgpfRFQMGl5lKkMJvJ4eQQP/PzDzRAwU4n
  cZCvwRqPNm4hcprHMxUz1XRlXrTqBoehMwsyH0kkNEGOC9ulNBuOAkZANSWsySiGb9yEBiPQ
  SuAS0QKAlbZOMApwb8CAKYhAnjLgC7AzO6PNFK3JAkbDtOKHSlikESxIFEcxR/NvUYrSKP8E
  a+5iAt9ofNqiSPNPb+KmMaCpmsxMRSiIL3QG7Rpob47iFsGUzFDx9/TCi+7zvoZqErbQiAgp
  +WZzVc4jeozQO6IDL6vhQJHlgHgqOC40VxISL/vUN6DIOQoSP9cLQzmNP8mF9RAwBxHMO6Aj
  FdRwXpSDdjT+lRuGgfmeMM+8I3KiA1PXcD4i1OC4YgExMMisYCp/RkxmjSzwJy7NL8Ju9Ku8
  zArAhNpszGSUwkm/kbjsAFvvpwINoEYOIE7o8+DW4kYFwy0akCnEMpzEaWvG9EexMwD2BLc2
  4AKeyEqEyF7vFX+KaFBOIk4l1Eno7gIoQEYJLXjs5JzWwAKkNa4oyD32hdRANUN6jwdjJSBl
  SlHd8UKLCujMcZCOxQ7RxmZqoXNsJ/nuoaeqUD3QMkT+w9QadAuTw4jEhWJL1TcqIrqQp/Z6
  ChtXgXI8loxMcw98sTh4J/oKD6bMowmupRpWC0i5IQA24H2Iwv7QTypzoppgqWn+xuppSjIA
  1Mo7vZPYvOxOu03FBgu3epJwwAagHiCJgqSwKMABeJKrDKBWbPQ5Myg3T1Hg6EINJCACr9M3
  aGkEOKAXPOcASsZk+FZxZw1/1I25mskAPnJeUaIRhOgAEOAYAUWgjnU3wGYBTMZRmtZpoec2
  KtNS08j4elAbLTakKjUSHGkXZO+IPEkbhYESz4ZbCq8JeGV79EgaP6fnypD4eI9YyMNVJ2oV
  7oFXleFEBiEbSPVnkXa5fNeABIBoyyi8CJbQtNFE7AHpOAAEFmM4vUz/Bgg+iSxr72JZeiwA
  huL+7K+VyLcrFkMJkgImwaxaCWdQDvZQsqQriJIs+7X+4egWardU7owCMYaAr8ZGnP7ECJ6m
  FVfC/HqhCWasAwurrTJ4bOiin1yw9ZrpAC4gTnoK85CRsBg4gynAh7bHDo6kWr9xFIzhAy7t
  dAnU+HylkD6YdRABu+rSVi+uIq/Rv5AFZEVHd3d3lKTDE4QBaJVBhjozH08NIY33eOBlZ7Wx
  iaOAIsAreaPXi+1QeJRvIxDBv0w2G78QjXX2H2l2KTnAAxYo7kjRR0YrMW6iKCkrfX9Lm9hD
  AraES0rsJU7sfxZjhYBEEjsgfw3ALgTrUDhrbXZTRizgEzUSNW1rOFnTKHCLWAXOsPIYcGnD
  TnxKASjAbe33xk45KRjYKK/+9Xb3oAAQQcbEZmzkIl3TdZZh6YlkaDlb4c7+QzEl9njJK/lE
  F5Sn6F9gD9UGBC9P1Poux0P1osHYaJHSMFOzmBF4eV+Et4bFL5iRF4sR8FeQJfrQ+Au9WKV0
  uBcv54yP1px7yqea2Zq1uK9CQEsgC8Wg9Uvj0+5W0Uq0qSXYYiZyVE/eYklEwp2YwpSbZi0m
  qFm0QQBIUD/AxgFMEIXciW9CoCSeEa6K1olRkwOQACjljjHEMz7TlW89GckMSLOEJANwTGZO
  +cbaKmE5o0/I8ka4pdc4oym+LoN7OpXdtsyeCGUP6VrUAWC2uVVBYZ0rNmqOS12yK9W+SPaM
  Fxv+YQqioaBZ9uiqgxCU6uedpaqj8sDyxmir8wGbe46Gj5Cbx+uKzpiMUopbyDmNHfd5KcdX
  ltJov7jworf24qhavtcwGAg34eeBxLLz/sS3+hmzAqAv/phcTYwouAzFagIEZLpp2k1WWkGJ
  ApYBsvqjmkDGFAACQEVasxaVyChpkciHQowjUFNTkAaT6VhvOfmkH6CEvsZJDwADyqc3E/a0
  RmODIGin4wCaxESW5LTXhAQV3ZaDmpu5MRhLfvspdtlO5OGMSlR22VpxComYSa4iUDWZO0ET
  glg5UqJkYaGs2yMToIqd9dq9B01eICA9nPhzjAEitTCSklqqptdBYPX+knJ2XCinZ4unPlLP
  EOLF7fC6jJX3l9xZcfpTmqshnqEgUjLAFMZxXeZmiRS3tgPFiI5EsaPGADqA2JrsjwWZPVuy
  MCwbZ0JAVgrAA+zLCnZEhX8oiLwGf+QElrnVcvuRWx2ALi7AtggZBC6gCRYhLTLgo4cTsvSv
  MHaJJ0y6tiOQnNTDAHCGgYkbzM6gC/ZmN+fg3uACLjZALG6E1y5YSIJkKcBSPAGOmhbFZ+SH
  AiZ8HdnIFXgXVgJcrvWou6WoW+YBWk5hC1mKPI5HObp4GDJiq32RBMjh1MDDUNubr8W5e59F
  XQKhu9nDRGyhUnNhWGLUeHulzSB8wusHG07+PWnRkHd5t1jorOFK1WS5RWd9Cl5c9XfihZTo
  /BvrRxt6fc/TOMCPnHV4rZvqr8QbsZjg4p3kp61MZiuwgwFYKaTbSbaFwGqMJCPuQyoaWTyp
  7DTObnyOgkwUoLVrxQPUk3yhlSjRdW9r27Kq3GZKQ0ii6TTAwK585CrYSSsSK9IOSLCmlCpK
  ZX67YDW4QCqbgix1verem+F/ts83wllYCg9ZChimha1fAaaEYasBwJfFsY5SYXq72tRp3eOc
  ChIsjgGWaueaQ+jE74qwlWBPvZzl7yy9umCVNg3VUCF1PtdbeS+WkJK6F+eVo1XrBRjMW4wR
  VenNWpGz3BQ2+Kf+Yemkm6ZpAMVI6ixueYjDQSXLzYxpTZJwSHmD1wBlToYunlERSrPh1z56
  I+0e7AQZVZnMNjixkh6vzcNg17Y4fEg6jtFHsQjUll7w9Y4War0twVoUfodDf7YKl2eTeifn
  rcFBgHSce0ppg7h24Gs92J7zfxZLZ26/+bvQzKseZ47WcV5XPamLX3jwh73z1z4PyqBgshVb
  txV/ykQZ1O2dgQvVscFgtzb2e1/4h7+niN/4sUFbrG14ah9bsxEIEfW0jZZkz/vhW3/wfd0+
  eH3P37u12sMt03iYcx97H+2LRZe48PV40v9efVz90V+I2P/9hUhbinbtq983ol/6+7NrjHze
  +vsfUQkHCADCIbEIIPySyuWSEDBCh08mtWq9YrNaq8EJDRAM2zG5bD6j02YDu21AuNnwOL09
  ZwqfUWOArN9DqQkOYiEQCPzxCRAyNjo+Qj4eRaIRAFBiZmpucnZ6TnpiWYaSlpouBQEAOw==
}

set pieceImageData(Eboard,60) {
  R0lGODlh0AI8AKUuAAQFAxASDB8aDyIiHC8mGjUzKUI2IlQ7JUBCQGY9KVtILWpGKlBTUHpI
  K2ZWPm9YKV5aWmJiXoRnMW1rYoh0RnJ6cp52MoSFf62JQ6iMWpKTkracXaCgmcKkY7Wphquw
  q82tbdKydtyucc67kbzAvOe5fN/GlOXNndDT0O/Vo+7hv/Xu0v742v768v//////////////
  /////////////////////////////////////////////////////////yH5BAEKAD8ALAAA
  AADQAjwAAAb+wJ9wSCwaj8ikcslsOp/QqHRKrVqv2Kx2y+16v+CweEwum8/oNJEgMBgNAoJ6
  Hg0AAO6m4R6g+/+AaWx5RHBygYhFBHcAAk4CjIeJk5SVTQAEi31DAQIHdmIBBAgKjkUABQgI
  BgaZrgMARp0CAbVeAAoXCp1MtBQZCrGWw4gLxKcEDLacAwwFwmKNDAwDhEKYqggFBQPd3dBD
  AHaMXYsFFQziTLUQE4uSx/FnmPI/BgcjHsFEABcmE3jCDKAAAoQ1IQE4pDhhIkSHhxs2UNi0
  hsIEBQQGdAHgwESKDg/AIemnQsWGAyLFQOrkqiUbAZDqzaEXz0ABEioggAugAYX+hgAHvQQo
  MKECggOnImi4cGFChKcQICBIOXQcRS0AEHBAwWFq0EIAIqBAMQGUGUZo05KTqdLInXoCLrBQ
  QQEcgREqMsAMQyBDiRJIiwTwsEJFihQm/oYIgeGqEAIbgGXsMriwCQzwjhAwYGLFRwmOxRAI
  cMFDBgyoU1vI4KEuW9FuU1YSAEHFh6lEBnxAUYFXKKIVCgQOp3RpBacRoqaThcAq5QoobEfw
  jUTAAA6FL8CalzUC8qfeIUCSreYAm25qRB7wlNkSgQmIHYAT4CFFBlEC/ZYIGuADC88plDBC
  CCCEMJERkGHgABuUZcAQY+0VwcoGI4ywAWhnZOLBXB3+ZBDRaR2U5FoiL0UYTRGf4HdMRhMo
  s1MEF7wzBioTTCCcYBNwsNR3UeEmmIvqcBHABB+QoIEyplRXQE8kTLDdWQDoVtJY0aEQwVp+
  DNCGAg44gMB588BDgAMbmEgJRxPEFI4BGfgIRl9/BUXfXAuVEMIIIIhwYBEDRIaRRlzg8qF8
  6yhAAQUS7JEhZCqwkMIGqYGQwgoeLAcILTB1Q4uKZND02DbkTdLIOIItEuoWASBQ441EDKnj
  BRWA12NKAAAZ2hWYhPekErXG2typlMFy0wortNDCChXc8YwfBgyAAAUcVOgBBxQ4AIcZbEhC
  gD8UmCnqW4KZxZd+cvrn2Qn+doYggoGh9VXtZBsdYB6nvBJwgKKLZhCCCiucAOkGKajwD6F/
  BODsNKrYNMAyY8j4GAMQ0DsMluGAOwZQqypwygQaGCerVKGl6txGQy2SZBK0FDAasF0MYCMm
  HxRLljgtsgxGKxN4cEJJhjUkkQKAkhGjMHGlEMIu8tzRnqkN+2UCf4QZhi6Bee65RgbvBs0F
  JGouIR+mgvS1wb4qnBDCzidc4IDVc6QaQQWwOsXAKmzc6kUBZgnAwAVTidLSV34ojaDFoaga
  3HAIFQfrx80xN47NUXwijqdJmBrkGdeRMNVNVorT05V0DGRCo/8VhlgIP58cBgX/1AJABgLr
  NUv+Jo8nwocshH8BWWKlsEJ3ZSWhW6AIIGAQx70GNEfAoX+CAcB63j7mCeRcZILBBh2UQCyx
  KWgwAdZ2lxEAAxVowIGOGsQ6tzZdh6EqLHYw4D0s4nhTeyC3C5Y7GKlWYJTfmYCE4ngEsji4
  wg4Ly98tFJgEO4gLcxxQgeaw4bnCAGQOAnAAXljwn/+UBDEdokCzyFAAo2VgDxRYyEkWQKF8
  ZOABGLFD9BqmtwAg7hN7mKEVdpcCDkzgATWKSgUCJjUBEUgEG7AWK5a4tosw6AsB6BbLAAAB
  CqiuDNbbQEFW4METHAoDI1LDUCrAgSKR4AM6Up826tawC0RgYeNLX3P+BIAAiDmjGyvT4Zsm
  5zg9WqF/6oNAcqbBgAkwpSlQgcA0uAGTTXViZLfQ0hSbQb0tXIdfH3gcBFDQAhVcUA19mVQH
  S9cz1FEAaJ26gAqKt5kRnKADCyBTwPrFmAwAww6AuxgBFAIQjf0gGBIAyeW8sMudcZGDpSvM
  BxNTkA6AoAMfOs2hDiWB5nlBXrqYItb8uIU4eKgDO0uBK1XQGjBWUkh7K9JYzriU5KiCG+eU
  AhUngICFIWApzQFgNwrAgOTQL5dhqAXEFsEPBnzgAgzjHwLgZj4SOBSNHTtkrKBCyHeqTEu0
  YOAWFqGKKb5NYmS4JArS4bpdqoA38cTCADT+yMWWtlRqqFMQAQpAw8NkACUXSAEIJCABC2Cg
  AyY4QZ4mZQIKSC4NooggOS0FgAegbSLhu0JfRsdBY7WAg8rsWQkK9ExoRsaW0+SpNctBgRGk
  aQmb6VArwuahEyyEWhmY1AiwllIsWEcpu1knGpuiyDWCtHp6QwAk3nMBCGhJU7UQQCrWiLeo
  CiUASnHjsn4QJSKRoDeOvUL/LqABM57xfBE9jqymoQCLZmRTGtVCLZ5CCyXUgikLS0Pm0pG5
  qWTuk2gQAOu2l0zTvTIyDxjhjCLggVb0pWwdwIAtv6lFFXSyAxZ4wCIAurXj/ucjKMHDCIhV
  Nr1cMQt92VlVkcn+wQ9ObTHEg+Zyp4koGMKLMhfw4mh49QATbMAAmX3CS+JQub5IqocSoMD1
  +NWaUwWQm00YCl4d+lAO8HVuKksoFBOYKkNGgBsz9camrLPQCDhsDJCAQBl39Aw+7M2hvfmu
  ahfKWQ4wGKJLaQqPEKaNbWgYkoHaVgUImoSsOJjHmONblDhgrAkOIFZ1tcIA4kus3n6wBB16
  4QG0BobBHiBHJShJCda1EIc8ZHT+CkEJMPAJFVePNCnwoGE28IA2dCZ4JbgPgptQgA1kGZlX
  xap5FdNV9UqTmmJ9b6AiCIxT9cNfDjCzfmthL3m1AqSZiKsJsOahyIDAMBeQDR8yYVz+6mi2
  ADDq7AdGnUZ3rrF9X1iJswzpRm3U0aKMNFg/f5XfHfr4s4uDg9IW3Js5P0JVnDUjqSMat9Gu
  bxsZwSiOt4CLg7rpCGE5KJCrMA6YuI48im0EdqyKgr49ewiOfBzlqjAQEOyMXzzrcpSrJdww
  3OEAJbAqb5XZUhMU5EIWSLMKRIAaxN1CARvMqmFKwOblnfsEI+jAPrgwgAx0YJzcNcxhErMY
  iHwVa9R8gMYdoIAp+1EBBnhAEwSw3RGIkDwBUMBhOiABRQ/BWghwy5A6IEo9m0Bt831MABzQ
  ogcEmL2HCskQ3pHC7fGrMCcABn//mIq3HZIpseqrNiQZjQH+eIcpxD4OcgSZnL7WrRn+dPeR
  iwTRYisSfgiAUZG+rdpktIOzWRftIGm8DW4ouxb3Y8ICBpHgnnAgAqgORwFcHBybGSJcENAA
  gxf/gf+B4w7NiIrkn/IraOzB6iSgEpU+oIE3JlkIK6VQUAO2kIZ0CIzupfItDmACY7U0mYbZ
  HkPQBSmaMwacIPi8awnggUZllYhmS+K2gteQDORctQb4BfZGYAI8WXy5GG/vxhXQcXnJ6/hL
  qMs9PI0EApiABZNeuFtg9xEr9kIA1D8ZJDrie/KW1+TrccR4ZtFIV7CCx+JwQH1GeUzungQo
  f3QwT1EjNTJIpWVjEtYFAlAc54P+PjEmWgSoSAhjd6PgDLqHBALlYp8VY1tHDY2QdoqnAbFF
  MttQR97hFFJnWhpmbdeWFtQlBCixDQmoCAKwFZc1WW4RAQ51ATjYQKiAg/hlWZrHeBzwP4eg
  FkgILquVeeu0eA7lQ4EHBZvhcA/HfFvlAZERdA4wZS94BfDmek3GWyXBRTwjMA4REagxOvtx
  gW8QABmwPcq0EFJTfCDXe6fDZrX2BENhSxHxEBxwcRgnAQ/gALHEcdVnfQfUhT8QABCAcBii
  BK2QAi3wGXR0BJWhUxawGUuQEATXZuGAD3q2EIfBL/+BaJrwBEoTXyWhZkeHVY+CX1YAE6kA
  MeAxd+/+pCV/tQV7U0af9SoSRYBzpxzcUAsyyIZEgBL3lHkk4ItMMQGS9yur5mIIpYhQEAeL
  VUdSV3d2F254J26MoIi1ggITBFAZ8QESdAHpsDSkIY5CtgR7AAFn5CNhwYTi2Iu8yE460Qh6
  GCU9cVKcFzcXoIHLOB15eAQD8ACVFhEXhyhdYh6qtxHxVhhYJXCkKHB2smWQ8lMm0CEosREO
  MClwWDZoAwKKcVMAVzYJd0LGyAnPwocfcgGBuHGGaH2IGEAFCQD1wUpodQDOpQLJBYuCYQAn
  4CjJZR5oVSEvJAxswAE2d4YPoT3XJXwuxw8HEFfohi4DcjazVBglYH5/NAr+qkBIFaUwmxIN
  ATlqaMSMUAeMyvGMjBQH9bSSlOUynuWLisc3doQ3qAABF1AklqKAJRiWEKaNiNWN3kgxTNAP
  KHAbwyQLNyFBnfdA14AAu0ECfDODnMATi4kb9ZRXi5k+ETA3/dRZnxUxeYgJSrWYLVJjIPhQ
  GsB2UmgAEoAaywVG7dVxrUCNVAAALHAsxzSRAIJupIcYzJRcPuVM9nUtWoBf2GF0wXMCK2Bf
  ZyNUItSIFIeHXlANDqBcgBh0g0h91neIAVSWqHgBFsI2lkgBvakCIVBobkEBXPRKCuJYUZQB
  EtCRlKUAzmUZxslTsxkCIPkoCzKVlGUAGyCc9mX+S8rlcCVARK9IoCgzAIGZDRaFi9y3EWUE
  YzH2dP4jd4kkFVOnWLCiJctZK50ljmUXkChwAYqFMCuhKorHAaniBfgloamQPDVWdzfWgmoR
  BWFRWLDgLVQUHUbiDrQiFot5mQYGWUAaC+Hoj+jYWPVTSObDeSR1CddRGMtoI97QDa3JedNx
  BdXwAGHVXg1wfYIGRQQgb4WxPcYicYcRpyfAEFsFTagxNuhiPLrJKwgwlM6pAiXQGWaVASaQ
  GGyWfAwBAhKxp04gCobiIbZ0AYgiAQ7QAIdYk3UDgFCAX1yCLz32hsdCcI/4coPxH1CWlEvw
  PB2nqaTRUifgARigcYD+ZgF2RnobgBEuhwcHinSmAXS/wAEe4RkgUZCWaKM3qqMsSKx1UADD
  9nSwAjfPKndyY0enZnWFZYyV5WIo+irlY5md8E4rMwB86VB/SRnWuFjaaHeFeZjjxgSj0Q2S
  6RbQAZlGIRKj0RNJKljAEiV6KQfrCJkMkFHWgYt0pBQOdqW8IgAawC9GcmESWkcQRj6d1Xkj
  SG4GoAAN0CXgeX+ZkEp5Rk5MwQEe4AEeEacOWqhzuhjPFBnX4xD4iQUJcXRlKFSCaijMR4cH
  QLLPBAxgMABjAn1Y80Mziak2SQV7IArtyg8bsJ4biQFAiRAGIInRmVwSsKStEBCPQRiWYZ/+
  s6mQ9ilg/5V0uNpjbMIzI+BgFkEBMAl9NFcSIPAAyioLeJQJmpKsF0Mk/+gUBFgj0Fps4CGW
  4Mphv5IFtXIBTLiB6ROCYXmL4qAqLqYBcqkIBnONdfcKp4V3aBG3PzAeqcUPF1AlB/tAWcGE
  BxWwp4JAMgR6SvUBRqo3cHMcjSWunCVkZiIOn3uO1FCws7uaMKIjTqK5ueESrpApZKBYJJBn
  KQC3dJtBkxKnWjWnzUc1kHKc9jltVGAv3/enzQedaQOsf0FwFmAAETAg0CQBvpZgDnABXiup
  7aWxNDm8jMoPrJC0QkByridUSfR4E+CqIFC9lTMkn6AIhDEXc7X+Ng8XVP7yRTQXIBYAPdAW
  SiVxthYhAR5SIQnXIRvwZiEwqgG4guEGYvf0KhDADXWXdn2LSIlUUeD6sHFpV0pxuGj0Yg+F
  RlIRYVHCl6NGU2JnglJnE8h2d9d2vtdgMI3ZKv24jG40bQAwAZuHjlGIEHgwbTVYEh9gmgb1
  YhVgWFGyUEvxDL4ULhOQeY1XT85ylqAVAXU0ux4GvBOwIMM7no1kHTNyAcXiKDHSElXZKOnW
  EHfSxwWiLpBSshuMtVWwxLOEbmNYhuiWcASSAdZyAWcYRluwC2yilWbjZzH5nddHC/Grc+n3
  XazwfcdiNBLRBkKwBxnQm9GpqOH7FTD+Ial7sQbbVRgjEKuE6lbBwxgCBk6vdJ/tEkVZpgJz
  hSi/UAJuNadQFhFvtlPAO3QenCkbFgo54mBv9EjE6HQR9R1dR2N+BWpNoalVQBTaupj3+GKk
  NsLe0LhkVMU3Uwsm3IwSqIIWem2eQAWdYIG3khFKtYzeMxWmQAv4ekbouCv6owoJ9a8koAxH
  1oTsZCPEyDFOomnweEZG0Q27mKG+awB7kz7UoKxLdABrI7IcAFYW4QAaB3KnJaGUgQAowEEp
  cMeCeABlZTomqy4P8cd/ITwOd273pXsxG5KKnG4pYBqLIREHAAEQgQFymRE2O4bcNXFnmIUZ
  x3HmsSnxiwf+YPREa8AZxjIXpxoMjHAAJ3BVK7BVp7FWRfDK+pASvPd+9kmoHzSKk7bLf+HI
  2JeflyYw1EKmfmGyAYLB2qNTcDsFrJAKguQ/cpMwOkrPlFEAfZk+eNNP1tFPDMWBHUiha/Su
  idcVurdQ2vqE3rMNhouONGY/jYBXEyAUdGRIHdOAxoGCx3ZRycoIEHoKRKEMIcOsJcHP6CBu
  DHC4B7sSspAqWrwxRPpG56CMDbhX+UQUkZkZqEnR9VQAONyAZfR3/MQUaHyhSjAICDkCokTW
  nkFxqIM1gojSMNHJEgIATMlFI4Bfs8lTfooYh5GoCok9ixFUW7VTFkBzJWABuej+BARwafTm
  GRIsqAT4C3gCKSDncJHRzK0iCn5xdPO2TCsLVjylyeaRqQjGBqxhfKqDfs51LAJTvqbyHiMO
  qIpKZiG+ZB4QI1eUENw1VwgZTn491xlAkiwXwEMXXmXTGj7nIHAKvRiMNrdKrFwDaqSJlh4T
  z1OXQIhptJDVWejoNs4AI1i3oXK3PhSqMl9HPhEjxD/g2crIz/SUjBXgs9qYEQ50TwcVKFlB
  pR0z56F12Re1rjxaB0OiAaYpGAiAE+KYRuLBCEtGJUjsgY4hUPQkYbUC6DxIABHgYmgEUefD
  N7SAAJz1bfNomfUE6WdJatc9wjCSxc/ATQbzLBRCROb+hXBnmAEWIIhbeC+4SLgU0H5pwzqR
  kWXc41bqFhkK6UyBaici8OpUeFOfdxcULntutF0eQApqqzNNK11kEhEQPnSi8D1aVCB/0UwL
  KX0LIJ7jiZmpOgEFYnxXscTFclWIwcpH2xeqfIcYIH6LGBcjK+/gVutI90IEEdf1vRCUVhAy
  lSS4sC9DzbUTLqduRXDXszNjhtZOIB9b7B19K3fZyOaYuz9TkBUB6WBSgYKsRufPCoxcLprv
  tClg+cROsIsvxoxFsmMIIIFzszIa/+ZbsAeKhdmAC2vJlqwXH9bydE98LrpicVLsJFmMQJmG
  zvH0ww/jk8UE/SlbwRUMMAr+G38+/Hw+wXHP30wIQ2YkAas3LUZ2Vco359AUUx9PzbIZhgKT
  Iw19GYebNvmQGb+/QM1dpiNxy4TJHlIgCIwnIOIQLxTgTGAAeGF03PWqneEBV0ayh2HW3cIm
  1M4F+LUlXaKx1Fd9S7QZw8vzwBsHQNWeOVcLqex6ZbOzcJtyomwZMdVmm9AJ5lnLd33KCpC9
  5Vfjd2+ySadc5bsenEABbjXMEmCeeL8QKWtLAEqJUzAak3uNXX5RPd+5P2/Gzprlc/6stSh1
  tOhOsXbzylMFuyhs6MMURWIj/eSW4+DYNL81eJcp8BrH1sGNz48JIjcFnUBGQS8Mtpv07URr
  EJD+V0WvxUAAAPx+gACjEikYiU1i4EJCfSaFAkTz0WhIJO1WGRhMLssmINKNDAKIy/bj3XI4
  EQLjcokgBkPnH/BPgUDA4ODwwNCAgFFAIEDMYCGQsvLM4ISFZUVFZYWzM1TlhPTEJGQjA2M1
  o2PE5EQlxQRkY7VjNoPAz7KSYCQlFFR0xeOAAljWRGSD4mCgdSOgl5oygMDA8DCbsZHQMeBR
  yIiguheAgnREN0DIYKQlnkUZpIOCIOBBJZ7zpKQjgwQDQiBdeMXBAS9AAjJ4UmEiwwMJHVSw
  SBHs4kWIFDakWlDuRyEQs5pJoACiU8ZSJVJtMJHiRAYHAsxRMvCIAAL+BFYIDPD5CBJBmjXP
  ueHghcMWpUu3XKgwYUIECAyo7kQAIQLUCAz4sAlggAwEhUR/4MnS5UPSPBeOTrgzderOdgAG
  VOBQgSxRulcraLgwgSvPAeAgFR43NK8TAAw0JIUQTkgBDSgok1AbdReBCnG6qIXQR4gAAAIg
  /EXQjtLioyQuMBAwhs5ZOX7XtKkglkhPDig0nC7QlwPnOXetkAG8BHHiH4QOKFCQKNsiwgOU
  VwJAwIMmT5w+XRSmLER4gKpWbRhxwrssVLYkZCB1YSxRAh3QOxTVqZibZLNAqFIgQIFU4qvO
  iQAcwMAVE2gByKQHHHDukG4II7CJQnA5YYT+Df4L4IAU4mnhE1lKqMUBAjJYYZMVYFrGlgMg
  MeACU+zBpxIATQhxBAweoKAEWeq7CL0SymvmAJoCkOAiDwLiKCOVZhmvhItAuMcACkMSAA89
  uNqpJ3GmsfIMBGRLi6k31sojKqmq2qkABLKq4CmueoKEgAj0+DIxJBr7gE+1nGpLADenCkw0
  uiaoA8zUCkBiNQ2SCKyAwYAqLCiQCAzHrrQmmAuB3VDozK+n+MCDMzIvgAAfgurMwzUhKOnp
  gjg+qMC3vuBAqykGwoHALZoAgGCKzwbAIji03vBr1DwgKAA1CglwwANTPEglAwoegJCQRIkw
  goMVWphHmBVMyWj+FH9E6CCVVdR16bsU6mHvAZdCOGDAagTYoIRY7pMlGA8UyMAEH0cACAMF
  xMggg3opHOCCEFLQRBMVTehIFQokuBYRRvCkEB2MTOigyAb2+RCUFEas9oRvOYGpBFQiwmcE
  Wdah1xIXG1q5g4vd8xGjjOzJwJZBCtkgF2t39g6mJxF22OQMGti4OjEioONUSHtiQ+G8rou1
  C6TOXOupCZ4adMudBriCDDS3MqAAoAbgqg/lEJigMTpiW2sL1wTVyWwhxtCAAW0BAWDRQzvT
  AwKdrPiJ0qCSqw4AMY/64LEAJpCiaz+1tDM4U/VgVggG0v7MVevE7Ky2AujeE1RHl1D+/TFC
  JqsgUqNK9ctRPhCowLR2oE5MAAoqmscfl61V4CZtB8qABXlC8bgEWFoOrx6EV7FAgvZw4W6T
  E0JAFwMJLMBgAxEkyNqSAE6K8j5O7FGAAg9egul7VWYSYILzBc+tgAx6hHgTKhgRuip2MQdE
  KFtgKkTAlJGBf2niQ/xQEUw8EcCUtAx8C3BAJ1giAQFYyhoHoEgnTrCB7BGNX0ACki3QRYFF
  OMBh9jBJCFKYkX8gbCQmMyEjwBSA36RFS2xiHPpq0pNM8clUcIJKmrCiJr4JJidpc0pUEte2
  whAAdMoZlq2S0hQ08eEKTtxJofDXm/0V6Ap2OdwE4iJESVH+iglWgsTU+sTGD6jgU17wXBKA
  E5umTIAPPowA78IQx1cFgI4kAJxo6jQHzqglAo8Q3RIQoAUECEB1xZJDHvYwADc5hQGgARMB
  NoAi4mGwWtcywAAOkCgAOMBb3wKXivwBAltSryPUqtgDGOAAgFUkRMogGIJuqL/EBEABHQhB
  fb4DEaCRAki1WMWGnvM7K7HSJaaM2CzWgzAKWOxBEXqEKxvyvJc0Tx4BNMEIQBCCEbDTnbAw
  RQja2QEO4IIlFFAAEYsgARqSsAMY4MjKkHaREgDNHgcgwAOiVy0m1ZAU6NrAP1PQAQu4KFFt
  yMIHgliAq4nDlRGoG1OWGAE1UWX+TYqbEybt1JQkQGAqVoxE4r5yTDfxTilnqsKiYFo20Blh
  kmfMTRtK80jEbWlxjDMMP6thqD0V61N7DFtWzsS7KTLACm6KymlKZwm6qJGjl7yGUkqlBVr5
  EJCIrMNgGHM7TkYqjWzchSGrQ4ADlMCUKmpZKsCpUBDWNQAgiMc6PPA8lrCCYuRRhQWsdYgJ
  xKwTm0DaeDDgnhRoaDSJWUQrQvCScPkImrMYkS0sYLDRPC6jyTwBimQpsXdZD5zPaQRTqYEN
  vH4Ct91xmUAt9gDfPsg5C9gRBRC2AWW6k0jXsFfwGPgkgfYsGCmsx8wMED8OfPOX0c1IPTrA
  vhSEoGD+1kwMISAwuY5GSqlWglVSTGqmv5i0jU9Mqpc8aSu/RIUBMfXKsPzStsTsZZBfWwsV
  42tFgrRJvHIMlJ2MykakopcwQaFtL9pRgJbaTW2BwSJ6z2aFNiEApXxjwFYiRZCm2u4uoQxA
  aViHqzVgiQF1op0P+yIrv/wFriNm4xK6eqkIPEw7JqtfbweRYL1ssAUqSMgESMgSCzzIARLJ
  3pQdlAgCKKCw4AImQK0nvnfeY8KKkcBEoxfd7/DLBBgkrWl7rC0DEU2b6hwYxQR6sef41chk
  6RgnTgEQazkA0FGW8m8bkLFuGGJH3lQobQFgAMsKsxXzy0iab5iBCyjUAVD+SbQHXgEkGYEA
  PfR7GWqVc50IyOq8P8GalSL3hlCWBmxb4cMjXuMTCY+jDS11qVT2gABCiGFurQnztnx4U6uC
  7SlbgRTWDtOsM47GTVw43EnZ9NHGDds6uYZKFT3Mkw171CfhplS4sWi2uZijHaXBMFYLdyw/
  VkEMtTtNG8hwtySAMb8knstflcOQ4c2DPx0RqCpJTRZGjGAeFMDfeUwhgoC4yAiQ+CA2ujEA
  BXAAyCmyoLsIeDEejcCYefkKgkQQgnyZOSWkGGC6JADxNmf0JEBurcw6QFkiZyzP8iHl9yjQ
  AOc4wCRAM7mCWsZdvl5sAX5lhAEehA9+e/UAv/T+UX864gElUWACDkAAdHztE2wYQAEIUECm
  KXCB4iZjFCGIiAFyXhPCVcBYshbMYJytHEbQTVe7g5MS9zCXcfz97z5EwrFCJZW4eBQSZ+NK
  2wHRjrNd5U284ztX2EZ3wIdGqEV4G91kxYG/DEpxEJ5U3SnkeJW2acQ4zWmyTepED9s6HOKg
  6zkEoGtHVRHErU+cuXE9jsdTJS5sQL2c/A4mtnlAm64l2PGeYSUBdKgFJPhgClawoHTtM7M/
  8DtBOrWd5gUwRO6qR/jERx9dQO5AQAOBCEqQL5UXnYDXe8BcX65ACtAwzv0YgQcI6NBryZZG
  3EwBfGvTPMv7UkSF5on+nlIBe5JOQsJsFwwAA+bHFKplApyDAX7OOcYu0IALQqwMG8TOlzrL
  e3QEo9zsdLwAcdwIHK6DQAKFd5iFDMKGin5qKLaPIDbvDeyt9TSsMDwpALUm3kCsibTi8GDv
  8urPle6AbjLnqB6sS0bPBV1pNHKCczijC/JIqpINK47w1vRsWMQGTbAq3LrNDD0KihhH9ghH
  cUBDCfNiABwA4ZqnO07gXQbugHqi9A4gYPzlAkYBIMZnFWgGENjOCCZgtVIkmELBe6rHAiyA
  AzagHhSq3wRAIgSxslbBYixmyqbsgAol+/YHGv4HgPrhY/qvt5IuQhgP3Q7AASjAsyAogj7+
  BOBUov3a6VwIyAJugvTs5TrGrHoOqJfKzureaZ3W6Z2mRZe+qcqkg5UeQKAcgO0E5zUYY9qW
  7SeA4lJUh1Z2Rw/UZCvmygEK0Xc8iQzqRi24MDBgr02Y5emoQTTCocP4RqXo7tYA7x3BhHAg
  oC/QwvMcDAoH4NcMgxWpYRxqLAtRIBQqY5OcYu/SBL7k4u+EsE2aKFJeoyLfJA+Y4saoSE5k
  D47Gwc0o4GFUZhT6zBb6Kgi1aD5SYP90KCIOoCMeIGtGA/lYizsqqB/CQ+AsAAQgYhcL8g9G
  AxGK0igNrRsI4Q0JpAAcLRFLcRT+IbGI7M5+TVsA4AHo4xNmkSv+LeJHVC7N6OlcgOaiIOGY
  jADskAd+5CcYUGQRy8UUbnH9JMqbTOsaGAHbvKpOVsPzZI1Lwg2kqsMR+HGJqoitLgkAWik1
  AqXe7MZPXkpOYG934K3UIs4RsrFxcBDwMu8H2O4K2MIfv3FNKg8zJewqMQmqKGMh8wgpHGXv
  psrwiE8kDU4oBNIc9wSJsNAfuygPWI9QklBwGKYi+AE8CIjIVonV0gGPTsAeDBEgVnJwSBKd
  QOQTFJEeCEYCMEAFRmD+8nJb4tGuGuARr+cRs+cBFMoRUiXzAMQDqA+AAkha4o8ZMQY9hTJ9
  LsAtuTKdLOLqOEDSSCHN6olaHmCcgIf+IL6iPf/NOp/HFIiOnt4llzqiRIpvf2jMqOQO3Iao
  9Abv3uYiUPrAyCLnwtKCNz8vLq4GElSnNezzDABPDMJN3O5xNjMPHz5pNYAIIEMPwkLyKscA
  CylDIYEUV16q9UqK2uZkKb3qMEATSJuUNT2nN41U1lJlRkfJAJAPYlYGJavFzvyq9N4hzURA
  4RZgICTAYNLnAICJZKpTFGAiQ1bhYhwGAwaCQK7jAKARX0wh1PxhelJBIM5NqAykPeMMPtmJ
  zr7JgKqSRSlBAC7gPbsyBPQJKJ6vAXABSE6BGcjjAcKhTlutLTdBEYdhBDgAYZRp/URgLnOp
  y/7DO32hDaL+4BqrDUbj8VIIoAsRszOvoxcLZFVwkze5cPcuMvFA7DQgR1Vyr0gj0sO2jzO1
  L9dAc5M8Ukdhb6ncjABgNSGF1DIqIJRw7fGayKQoz9Zmr4iM4G1+1EkVSQ/YBPLEBjapTZSo
  0QFu5Pu0dM78Lw8LTi/GzBawYVvwQV+1jwMgRoK0o01PwRZW4QEagjtbNR8y4DzaMkSCKZpa
  ZFetZBFgsYIgJiXWqeaoBVEdpCoLVL3ub2BAAK9mkQVAZvsgQQL8QXqc7P/oJR9P7AKiRWK3
  I0Q8YAJu4oPADgPEci7jb3xarlWt4W2kTV0x9KNakGOwSClBwghCMRBWjIv65I/+4Ctx2AYf
  CgOToFMv6IIfCc+PpkjumI0zjcDC3uCR7kuMZDUzyVU5xGB1tiBbu+ACENNFeaoLt8TyFrUI
  XnU1svVTOICrAM/CxEYr4GtL1JMaYwQqR+FeEdU5kofVDuAj8KRKB8cBSlKC3DKytLR6MMAC
  JAIYMiBgqSEAIBY/QHdcdFYWagHM9gcfrnRjoZKbINSbeiucVpE6KCQbJMDpToQrVQQEWg7s
  sjMDTK79BmhT/65KooYMMoAD8C+YQmQDIEA67lIC2om7ai6gJMABNIgQBcfUpE0LgshsJAUo
  jrYIcK1Fa1J16gaJ/AQig7VrzXXVKnNqOqdPzJZ38Gv+jC7WWdBgB/kELdJXHSlvcUbvCykE
  IwlhAKZGClgTkjxMxFCqp+TCQyGnDa5icCu4cDs0BwtncZW1J5KUKAbjAWImchWkFirmzxLh
  NViNrpaSQ0xAFkM1sgLIZO4Qe9pD7WpWdak3y/iBA8IOAXhEu8KXTq0VGSDo+7K0yejpY2HL
  WhbgA7WRQAYBA3bhAEYCnb7vQ76rA2ypnegpLM8FA+hlc/Oi9iCgOTogJ3XWKz2AAqKMAsin
  5FAVXWpOJowgMBNlF94mUzbJL8EtwkysUzcGh0mDLfrEbkKFidoof+ONWYw1SzrnLrquvv4C
  boLi2YZFN7umNc0W9LiWWn3+p4MvBQH8hmtMuWv+l/UMr6eqgkJLLSgYIIS19XC4QoNbrwcX
  byKvCUsAhlANFkNQMbZWqYab9Q8+KDt22CRbt1ykkrSyM6A29VIGQQFMgGTYgUPcQ+VKRCkF
  kCLIWB5kyYIC9HuY+WJ8DgBx4gXFVAFGojojRmV9pHmb91woBvuO1ofaQTknFrdE4RRviZ5C
  IBc7bjNHSZAieZb/MZFLM5ehmQgkiW44uYvcFqY2eIATL79YqTIhoVNiowrozsJEBRTddyjn
  l3AZMoliM/SUqmVV2KsmwCfoSDcZsiFZr2+rAhQ71Qig1ZR9Goj4TpgZVycemkB+gbVIhhYj
  hh7+Yph0E5WGMXo5AgD55KE6vfqgV6SWwAd7MgAmKABwtS9g0KOWqOV6ECoXS0AE9mmUBuAk
  KmidI4iKheydpzJkAU3pzHJuO8AiTIkWQVcl4rLoGHr97nCuSm80zFpn33JFFGSh+zpC6czl
  wGR3jkIhV9MfHYWNwm7uJgWnneUIztF/S5TAPppvuvYHa7fUBIB+7Vf3PjqmANX4BEk2RFhb
  TcUjHwwwb800S02kODLWqA34Ukq+LlprCGejc3OSbwyomwi3Qbpt3pgsgif5+IGdlbn9vidd
  bg49oRkf/lAWZelzgwxI5FoXL4ZoUPdSjEsVWu4bHGHi7EoiMrF8CcT+kyJNm75lP304LME3
  sRyKdxGIZLWmlByCwI3zm75piSKcuBDGwi2gUDjGQBjIIdzUAzzvL5ZIxCO8six8sUpEbskC
  kZj0s/EotJ1CjOaLRzlToyO5fpcCca5bpRy4HaJXaxiGuqVifX0Cg3XCcSH6M+MgXRM4HalN
  R62NIE87EBbDIQV4EewxwiTOa2GU2VoVqFSv8KyGEbicyD8MpRrXd54aATZgY73FWwAo5ZpX
  vBnQzmRLIDkzHNAbRKaTDnNLFObJqoMYQbg5ag5oEZLy0LjBEBihOXZhlNyAXWJpwMOvwA3c
  xBEVq7c3ak5kFjLkAs5T0aND1KMj0TewO4/+9jrMutM74AKwbh3JHEa/bgBNomilUbAFc8Ui
  uUlDm5KVraaVampTN2pS28Zb88Y8+qMZ2CtyUDR8HAxPlMeJu9YIOC/woXC4wKf1KB2DW+ym
  dbhZWcoHx01kzSsk7jWa1tyBHTijBkuygo3G6PJCMt4/6MMWoR2EPX0CJBa2MvkYtHhsqf/q
  rMoIgYiHHb2nU70L9qBP0lDrOzvXiSbrlOANYeweACr22Hqi0a9EA0wGAEY4ABgMG3dF63tV
  1cIjPHsOSJwAlyFSgAMA6dfsCuiyh8IpvEsPndbSvE4NgANO4NORpye+7gAQQNB8iwOf48q9
  zhAWQHxdBN/TJ7X+s+C3t52m6/GN/C6tKwESROeAkSinBoymlThSJMyH+qDgUyMc5kLicoIe
  ea+YIdqTRCoLJ9qluL3tX+/qpzbcXzqkxcC0kxDXLnOItLsmwsEAlqWlLTPWYd0eS3h/BfPi
  YEFBKVaxR4B64k/gmw6ThOoRNBa30CmfFxS0AjRhsWcCTQCtJT4tZ73ClUme6il8Fs1KL8A8
  YmGMOVYZ4OljI9HEERzpFJzlHyBvP+gAGiA7S1xdkl9dLJy3LvAmMK9ODyAPJBjshiv5e588
  NBHrtC5S7OoQst5Z50bqKyOJ7J4F/55qE+URgs1X/Yi6KznE/lLtfaLRKvNA05MI4aL+b0ts
  7y3B6T4JCEjCIYejuVQmkQiDgXgiDIgCtUAYCAQBAHcL+IHD4jE5gAUEtF4uu+12mwfeALlu
  /2kHA3ZagA0ABgoG+g0QXAGied0xigk4cJScqKywtLBgqqiknJSUhIyAgHR0bGRkUEhIPDgo
  GAgQNMrWEQRImGiu6La06O76am6WmJyYjG5gYFioZnRI0M3aARgotD5IUGRweBCnbKZ8lloY
  fEUzDiBkeIx4Y15mbpqYjISQbtybnmJQ8KuyHgAkoMUcmUMGCCiggOFUhmQWHkKEmIzhwnwU
  HNSKRbBMAQSwDjzIxjCDqXwlRzI0te+BgTQEym0cAwhBhAv+H0igEPLByAUlEJpAqaJHTSIu
  AmI2elmgpoYiH3ZqiHphapIlEK42cUIFEZ8tGmN2DTSgI9AmV38iWISUVgACNG3i1AkViU+g
  UKZUKTBUTRcua/EIQHMm7CBBb/joBXD0b+AtfQIdjqyo0KE9bP6CGaBgQwhcKjBhWrGpmKcQ
  pu2p7PdAgatDmKU5MKGLEq9LlXzp8lZJxYkTnDo0VLbswQYTz+6QcyzAABkABg84mHBqg4cQ
  vouBCHHqYmAyr9owT5qOA7vP70LzPmF6FGqSDPmpdgDwgEBoa9Fky/dQlYSI/pMJl8w9DR1g
  31+ESIDBBsCpBCCAwTm4UEPTIdP+0hav/TBTBBVwcBMJc1X1kxNB6ZWFYUZhOAYAAzBQQVRF
  FCEVVRXQGMFZdk1xhRZvhDcGOWws11wbc8RxSF5UyOGXj4Gx0WNSbb11Ewo57cRTXSPehZcV
  exV12VqNEVUYIYCYmIWJZGbRERaYMamIZGFFRuZYbb72CHm+5RIMJyZ4EgopfzaYygOsHPRV
  ihkGkAElu/AS2m2LtsPCaJyAgIxDq3SQwnF1tEXBVBQwYIBlQ7pEwAEKQABfQ8us4sBBawBJ
  QAEKTDABRoCI18wIJ1RiCWiadBICe6XgQxEqqjUQ0CEGgnVApSotYwEGpJAEHHDNmAKcgqeA
  IMIGBWL+6NgB7uXzoD7BTUjRg39K8BJMBwZgAAMbXiBjiCJmuRUWf0C22KEa1huVwFNRtUSt
  TGQFhQF6QdbFHk6CMVOtDDD8psUXO4wAAxAUgMYstaDDwAQcDPEhTxVYlTCJeenBb1/vbvRS
  H1iY6UcWeuhlSMuy6jxWRwu3xSYXo6LxchewtFyImGOlpSRjCHh6AYwwXnDKVBlcAJ/WEiww
  aCsIHPCKoSlO04EvljTqSy+z5SJaeidUCiCrnD3ALBgCFOABCsGcMAIHGdgKtgFhA1ToIQJp
  QcDgBgx+QBSDOxD1OsQYc4EDsJwzLnm4gIbenn4uWGw+x8K3igLK2k3QNKP+T5cBCPKEkEIK
  sRcDDggpjFCC7CakYMIG3b0mroTuTdhBCaIIu14HIoDwCfIhfCICBT+GK5isWeLVMs1hFtaG
  v2QL8BbBBCcxQRJXLYFlFFDUB+tY04yhRQQkqJCTERNQnDNRGIe1L1n0NoUDF2BAGmQBsgJs
  rClQOUJPbJQVHD0hL1do2SCchhQ01MxmsqLCE9R3l4508CqWeY271pAGNOlsgxyMoApXyIAk
  wSwmiZihY8RkQ0HYzERjwxAADhCCXKBtbe5gAW5EI5reFcNbEGLGBhxgtxd6oFfvOGKwsvMn
  VDzgAJaZYUK00QFhdSA7umubpHqnHQUEQAF3cIv+1NbBK7btyXn1+BOxRreP0mURda8BwAPc
  IwJPeKIbbgvGJoJxmxWYIAMHiOF90HANCzRDeZKoney+UclLdgI4BbLgWmDFv08q5lBgWBEC
  ILCECBzMKqZ0IFqwt5UucSExEAMABKT0oSodIURaSRqRukKZ/02gXh06Gf46ZkAMMm2V+MqR
  Hg43Qe3VrIJtwEwt4qC963XwgTfSJlq4sEOwDE0OcvIZCLOEI7M4sCVdCN4N2+lOHIpylOMq
  gSZA0ysiEtFtRKTEaPjkLUthQBUWcEUMA0ABb8iOE71ZqG848RsMJOA73wuMAaQzjIXyiTSA
  LIEI7DG91IHhCmBDAKr+amWrg0HAASplTTVaylKwhY1xMkWcGniIEAmQRBSeWGhCe9rTE3jg
  Amhk5F+6YKqQTIcUyRuBPEbgVKY+1QMeQIV8UMRDUGLVY6JcEVkeaE71VSEoFHyDQACgxjAA
  AAEDqwoqyTcBtJTICxPtQhpwVsq2SgVlI7LChRjhLjjAMqtv2qMiyFlOKEBQY3epj3MO1Ria
  GYIKCyunWX5i2W2+wgvfjCdnU1QLA0QOJQR7z0T20ZCtrWIVg8riOsWAEamSTmsiochF1Pk9
  RzgncsPrB3+uoQqFcOcORBOsYMmGBsWBRCEjwceAtiMfJoGUsMcdHGsc8ACT1soBV7FuK07+
  Jyom3baz4vUsV8M6hZYdaUuG2KA4PfkyFUHgrRHMSzazt6NQ1qFNcgJhwiYYJqKG4a/ELS4J
  +YAzlkETEEOBZWeHtK8NIhYrH4yrWsZr4UMNGKsFccAAnenhgzDucFsILxnAtF6B0NRMyD2d
  a2iRwRfD2EzbizGNRalOurblcIYzER8uHFJPjsmGQyKxj4u8kRoqmC/vDGxk4heA83Kvhvuz
  KiPahOOGRYbIYHCVTGW6vi4z7nFhBjOZxTxmiAltwFo+VC3eQAg/nIgPmzUyneu8EedY+WJB
  M0ebQdlXOwP6VXFSHKALbegin9nLZJbCohXd5ccJyb1v2nM0+qwo4UPTGQGMlYxLMO3pT4P6
  znPG9EtCbepTozrVnm5sqkut6lfD2hxBAAA7
}

set pieceImageData(Eboard,65) {
  R0lGODlhDANBAKUuAAQFAxASDB8aDyIiHC8mGjUzKUI2IlQ7JUBCQGY9KVtILWpGKlBTUHpI
  K2ZWPm9YKV5aWmJiXoRnMW1rYoh0RnJ6cp52MoSFf62JQ6iMWpKTkracXaCgmcKkY7Wphquw
  q82tbdKydtyucc67kbzAvOe5fN/GlOXNndDT0O/Vo+7hv/Xu0v742v768v//////////////
  /////////////////////////////////////////////////////////yH5BAEKAD8ALAAA
  AAAMA0EAAAb+wJ9wSCwaj8ikcslsOp/QqHRKrVqv2Kx2y+16v+CweEwum8/otFpMEBiOBgFh
  TccGAIC304AP1P+AgYJHbXpFcXODikoEeAACTwKOiYuVlpdYAASNfkQBAgd3ZQEECAgKkEUA
  BaYGBpuwBAMARp8CAgG5YQAKFwqfTbgUGQq0mMfIPwvJR5oMup4DDAXGZY8FDAwDhkKapggF
  4QPj49VDAHeO5lyNBRUM6E25EBONlMz4gJr5QwYHIx6KqbpgYkIeMgMogDABgpuQABxSnDAR
  ooPFDhs2UOhEhACFCQ4UyAIDwIGJFB0erEMC4IIKFRsOrCwj6VOsm7ck8RO0j5/+gQIkVEBY
  F0ADCg0BHIYJUIBeBQQHVEXQcOHChAhYI0CAgGAl0wHpOHYBgIADCg5dlR4CEAEFigmi1Kib
  S3fmzjCpVNk9JuACCxUU1hEYoSLDLTIEMpRIUSJqkQAeVqhIkcJEicshQmAQK4TAhgwTRA4A
  A1myCQz3kBAwYGIFSgmczRAIcMFDBgy4c2OwkMFD4LtnUnfbe0kABBUfuhYZ8AFFBWCjmk6o
  UMDxualUK0y4qhUCvFrgwpKugAJ5BOhJBAzgIPnCrDV4EETgnnU+BEnE1xxoQ67OoyIHgCIc
  MgRMUJkD6wjgQQoZkIKQYowpFcAHLLjG2AgjhABCCBv+GeEZBg840AZpGUyk2YCHGLABhhvA
  tsYmHvzVQQYZfYZBBy/9Zkkbt6BIxn9EhOJgPrJM8AxREVxgjxmrzDdBdY9NwAFV9HWn3GPT
  iPdFABN8QIIGz+SFhAAFGEXCBO/JBQBzL7nllgooROCIIgO4oYADeCLAn5r3EODABj5iUtIE
  OhGRRwZXjpFYCSeUoJSCf0lUQmYjgCBCh8t99oBoJCmQUQYIyqMABRRIwMeLnqnAQgob6AZC
  Cit48J0iuNxCDi5DnoFHn+HkV8kj6dTSiK9bBIAAPU9a95CU2dW3VaLnMDBbHyQRYF+aSwDA
  QAURIBBXGgHMAtQKK7Rg7gr+FeBBjSAGDIAABRxg6IEHHFDgQBxp8EgJAQRREOglc9byraIQ
  OmqEABS61mhmIYjAYWyJURDiSCQdsF+u2RJwwKmoZhCCCiuc0CoGG6SgQkGhDhIuAtl88xNY
  sbGx6xAEMAABxswErBexxRrgZAEKqDKBBs1m9WxsxoJFLUlMNSJmek0PjMYAT2ryQbktvIWO
  kTyL8coEHpzw0mSVhaCRAqOh0VcjQvSVQgi/7DSzh3jIpthEEkY22cIaWoppRxlILGLaYEhS
  KBMI1lrHJjR+rMIJmYl9wgUO/A2IsRFUcIF2VzGAwCu4oLFKXAIwcEFXpNykFk89FTEsGphP
  B6X+J9ht7ixXSCOgdN1ghIJO64w4ErMZ65HQFVBuyVlUnF2HkZAJqlYo2WQUnf30GABQUFAu
  AGRwsmG2wJJO80zGUwvviG1wUgmouPL5bHo/XgIIG4oAAgZybGyAtx5JzKkYAAjQvzoDCvJ1
  YRMk60AJyMXAFGhgAoEbHri2pQEOTIlo3PIcOA73IwSg6XcMeOAs0EGO3RkQe+Z7DPpGIZ8K
  PCV1m5BE7ap0NDmIb024WNoudKiEO0gtDetRgfG8gToNSMYggBCAAwjDggpVaGyVmREF2hWc
  ArwtA3yggERisoAVAcQ2m5rWAGVTugAo6weh4MMYseCZk6SAAxN4wHb+tlIBk+2tBBjSkAg2
  cC9X+LFycRTJ9boQAH8RCwAQoMAgg0MAktFvBdJ7HKkwoKM/MKUCHPASCT5gQQ1kEBxtkOAB
  +RUBmIXwhQJgmXemMQ4xCoITMzEWzsSAORdGwGZayQYDJlAVqxhtlQWo0y1y8QnupZAkdTqk
  NE64hSCu4APjQyQKWqACJP5BiUx04vSoJ0UqysaKIcBi9rZ4gC7KC4wiucMaEVPGM6YReGFo
  I2XgKMcJ0NGOksqjpfjoxz9+JIyLLJYhm4BIRf4BgRt4ZCRPMMlK1uGSmSTBJjv5yQKEUjak
  NKUnvZVKm2UjmANwZSBgeQRZrtMOLcwgBHL+mQ1eVoU+W/moMHFVTC3tIpkEXeYfnAlNdUBg
  mtVk5hYSAysnSk8yZTsb4azhkvutZgQn6MAC/mQyckEOAxkgxh1WdwZSRMQgQRNCMSSQkmOC
  gQAcOEFVm9hEBkLRMvQDQQfk+qnbkIpUEgjjUr1gMV8cMnAn9YIcaNQBsaUAQ2LzDSWFWizT
  ecktm/wAVbRiimAyNgqInIDujEUVb8GQHNjQygi5OooA2IxthmLABy4ADTLU8gIVlOhEidZL
  bhlNl5W1qDCNKUo2kgVaSWDLcwLLhSCiAB7E1AQH4JSunS4RktCN7t7MhgERFSA4AqBMBmRy
  gRSAQAISsMCNTGD+ghNYClYmoIDvHorWl8iqGgB4gOQ20ls2ZuAk0TNXC9r61vltSK4Y+UxW
  7wpevbKBAiMgFBNWM6NXLC4xG1DrCeo1DFiNIHCXxYJ6ptIcyH6Ak1bxDihnedbSIUASBboA
  BOo0DpqSqbIEKEBrwRWAqVwgAuvqBtW89Jz6ZuG1GtDkRC+4ORpmQwG5lQVNeeiFXGAldD2k
  jXt8DIbiwaN4QyyeNekgAO0x8Kjk2htGMvAAbzIpAh54RWIe1wGsZjUjGAGBCqjZAQs8oBGk
  LdyaK4QSmeDBACMI8wkME9AsJEat+WVrE18iKYY1jK5ZDRyBH2DgpVyAobPJ1gNMsAH+A1D5
  CTzqESMS86o3SmCSGwCZb/IDCzlsqQAclm1kORBiz1l0xmMIlw89eONgxriEuFIPAri1JLUh
  MpNUoobwTCfR5xRaC0DmgGw5SVurGFmD4QBpnXgLQH5VALXBLYuSMlyFAZxuTRzQb5a5Re5y
  X5pcYIZiCWZE5gPsdQwoPoCUSvCSEjhMIpm5CPREFoISYCAUzz4gbVLwxMlsgNJuaM1LGtUg
  4j6hABtYTPRYsF/+3nFSGrqIgCmJ1wL/jyTLJUZ+WiIyByTcCbMhhcUs9opcCIdxlQkcjT4l
  5xRcYCbDgqGa0fNjWMP2wx+mKGVByUF8k5DXNwYHy76R7W3+S6NbPgwOWaQNYm7FAT/WCjIJ
  hotv+fBS7LOutnac1TJw6Ha3Nv0CL1YLXCOwZbXgvkI6cmLMRZLpEezRb9ZQV/e2EdOYczO0
  Yl7CwG1WBgQcCFyZ7w3AAIbgXKoCmTYlM++de48FJgBV0wE4gdY0nM0pKZAdTwYCg5xxCwbw
  GKIVvWikTuQyk4L0gO9aqhDt5+VVAEBkUuIGJQigvB2QAPCX8IkDUCCh5B2BFC/WJwWl9wEk
  q1FWO5BeoqxpVBmYK/3oNQFPW7yk2Di7BYnMLREH84c3PdbZb+ydld5Sl7aGmXpKgS0xyAQB
  YjcltsMVuoMORidtrkdLx5I5F5D+SUlHNFThS7+kQW4nC9vGbbswAV8SJlBzAauVY1aAYhCg
  ObSVQTJ2EIbyCC7ReOQyRFLjNC3US1GnNFqQECAgNiADRQBHb/ZiZj8SQCWgX182PZBUIQwB
  Ai1iAQynAiKAGzJhDQrARNs0GZfxcB6BgycgfQLxBQMQfiAAVSAzPZRRGSAncgImab1HaSGh
  APZ2UgpgAA8QCYE2AlO0FwGgAJSRfMt3Lwhgd1zSAUXFVqYxApSTaZ0RAA5gJA9warx3VypB
  M5pAAQviVmH4OMTganbACgwogwOIbThlDQMwH1WhdtuBFVthiiIWSle3Yu2mBGtSAV5CbUXW
  HdqADvL+4YFDREulYDNnp3Zrx1JtV3Vwl1wrtAQLUAhPsDwcEAGjdw4FIG3U0TWI8BgBAAEa
  IGuyJlkvZA54IA1b8Y325y3owweheI1uco6SVUrkNgB/MgLlZTISQV4VYVeVRgaScAAmcC6b
  h1SNNxGN0iqAqBmF1RCtiASk4AGq4ngTBzl8xC8TRxEZYIiEZAAUcAGE5Y6VYhF1tXt4RWkh
  ogBsOHP7IZFMEBj+QHSMYAKgp3LB5T0oYVDBIAAgKSaSYBIJSXv8RYcBAgn4YQs5YUOr8QrG
  gA4OMIlGBW+RFBNJYQfuwgBYsR1QmUtIJg64VjgCgB3rB4FVsXZQaU+4BVL+u3iCtGRa0jZb
  L3Vbs5BKU/ElYLEL4RIOLOMkufQNbvd2M0WMdJFnQyAT4VCVhCAAZjF2IGh3ESBRFzCYPbQK
  IOhpXUIC54gC2MhJL5QIdVGZxuBkjgknkImNEgVHzRgFqxF+HYAhcDUvn0EqvrcNTHIAQdgC
  QwhvjLdoY0MRAYYb0FMCFsAxX+BpGdB4jGZYJ2AixGAACJlUD/BpUcAUD/BmG2ARHHCGkpZX
  ITJVayiSIxlKejkE1ZiFLsIEr5ACLfAaqWSQkeFdFrAa8sABJfBweRFAI8BfEqFWVQVdLccJ
  ULArlxabU4iUrtFpyDkmL4ZLbOeJuHB+mGU6EcX+SQJYW13JUjH1fkzxfgUZJGRhjiSwoC8F
  juLoLlJCAqyVnVSwf6xgCh6VW+LQYod3eJUJoj+gLZCZFgvGJh4KDwMyGxcAmefWBHwAAZt0
  JWzhmJA1axb0YRIlWUMBJE5wBwNgFHCySdmxHQ0oW8vol1EwAMtZIxsZOBKAJ/tBeXJHAEEo
  GW01hZoXZmMzKf42MtxHb08odw4AK74ZnI9DP5ixXVH4ONInTqRRCsOwc59hkR0ZIngCktZ5
  nbiCWQviVE2gMXOmAm3maSVlACewKm22H4uKIWRmDG3AAfxFmxhhEYsBXawiIsvXogeQAVV1
  RxkSOfOpAiUAk3pXCiT+in8t8zKH+iMF0IBI10mcyDmlGFPfCA7CJKv9twVr0pidGYHXeDqq
  lCWrAAG4OCtWCZezWpcneoEqWpmYdaPJYVboFxRfch4zQRbN4aGexQS5YBTd+gO602EooI1Y
  4TlOeXQTdTPIqVxtIllGUpfycY1OWnigaQASgBuRRnK9x4avwKJX0D0ch5RjaiE5OBlkQ17z
  02biNVfu2GmlugcQ4ZsLuQKctgEhEJwgMEUQcHtmc5wKCwWyYAAOgFXQiZofSagWE5IjOUz/
  2aIXMAIa4WOFxHEqEE5bqAoUAElRVV0+OwwS0Ka8MGemYbEWAF4DK7JwOqojQlAqkoOV4QH+
  IxdpCoRPGqthA0CtdJlbF7ixVMAWDnhBESiDLuRCNPQswkomSVIBddIFfBBC5SqLDYgCF3AL
  2OA5NeFB18gBxuI1b8kK+2Ot2oatlpmzbKFiszBA0iRED5R31+EWdAd/rlNjkksLLgonq5Ul
  3POWu1RBkoVcToBuknGhRtK449CvTnoeWbAND0BgpZJXDWAxm+ClTUYAJyCEksFA5kI2YziG
  wQlXGIEbIlteBucGK5stCDCplOiqrZFgEESxD0eRE4GEdaiL4CdgF5C7DtAANCuSrZYL0fsD
  nnYnuumKvema69mdROAAkFEhnfeITBBAbLiUP0Ab9OkBIEJpJRf+XhkHjxsgSPubtZJxAraB
  hryXARxwEq6REjlbUmMLlz9RdTh7wVLAFB7It7WlOZvzttzROSVqUcKWJEMxFlQjbZApixWw
  rLmQSiO2JtAqUdK6FHJArdl2rSiKeOpgcbMxDpyrCuRhuU+xErNhFO96Op+pCmOrTv97o5bL
  ADn0CEGslhV0AaqbLQKgASATrgWgHqzwlRSUjm2JBdugAA0wqITqCrCgKxdQLovmAVXBAfPi
  RsfLaOQVnJkBacxbEX7GBRBRiWNjXtcLkggmj8RwAB7AEDMytFXmJ5E2YHG0hjZ7nTHkwULA
  B6QAT82wAUBrAo/KGUkBniDbZhLgK7P+IZSGQACRYRoZAF7MW1enhgGlNmgKHFyxNzYjQGsf
  MQEVecnbZ0cgcJzQ1kq922IdbAZc4oBU4iRQScIlXCX4B2O2YgrBogXacgFAOluedI0a8A2e
  aIsdqgETqhpvSXXhEAsh5bjCE4LqUF8tUR5n4cVSQxZAOrpRrJ1rQsU/EETIUQ+PkA2aox2k
  OwDQ2lmibCi0UR4zWielkCQRuK9JMiVo4snLcROxYCtnQCYkwHFNlALH2btKBCt9TIbJmyH0
  0ypReyO1vAntpjEqWb0mAFUhk8eWUYW5GQEZghHKZ6BJ6gAXoH3j23t4EhLoG0Prqwqu8NBE
  IAAjcC7mxUf+6wAAE0CfIDDTSzAbExAKrhMZf3FhlWMR7hicPDtJgMgYFiBALEFULxHMH3Fq
  NIIhGGIRNCJxIUC/THkrfEdMZpBKQSaAEBBM2RaDMghTq0SBbzei5WBoV2mOMVyWm1SkSccV
  t4bDIWxRoBiXIvY5P6xkNJUORN0N4eKtj8GkF3pjmKvVbuLP+bFVeQeYL/EB9qpaQqaNrOgu
  M3yYvGCQXOKYH/AUsZvRCjol3WI6k0WSUwASHx1DPwnSTIU1q6IksXCqmfdWlBICLp0Ze2Sb
  lNHXrfzUQpOq20SmOSh9GgIqBnABFaERmMsFvxB7I7uDZzhpH8m7uGDehziTAeX+Cirpmm+j
  EcX3yQzrmgyxAbnJVbcwvofREYEmGSMAIvf1jomsGRRwI8GZfGLtIYXEbypwYaUyDAokYcE5
  bxkhcd/F0R0B2D95tqMgJSBWSjXFFAxYbXHbdqAEFka3Hf5rBavQoZobUdm4q4dNDraISbi9
  zoeQC4rtlY79dqWNB6BgBZ8wDegwPLKwXEKk0V2RF7jgxJvkxcWqQtwsFhEtRM8wAOQRpEX6
  QDLGFEPzQSzBo5tk3NIAW0OqoA9kAMz9DmssBX50AJWjx5GHhiBBaW+oZGNLGgiAAiTtc5uQ
  V84XaMY7hg1jERtScIwyP9uHgxnxvlVwyB47NqhONrb+kRkacQAQoJEHt85FPCqEEYZIVTYB
  dpodydShpL5pawCUxCOEwBrm8hedVwzqcADBi79djQEOdjAPHhBMHCNlXcsXzmhj+D0brkAl
  ACrO3aIKIGcnUy+3CyHHSxmdl3Gukcye7AqssFJwe0suU3V9Rxq5+mGeJGNOqR5OSYJtW4pz
  SZcWJQvhYo0WBLD3OWwwnKxPkqszilu6Y8SPwGFb1gW1wmtEk5WzGFNJNozqgLbdUABGIti1
  UAAf8BIX6klHog4MEM60hsWq/b/GwopCI9Gl5A4SBZlDmkm05lkirwHnIRzKlee60xQhvH5r
  i2OmE3UouahusJwjUFT7RS7+lWEZrK6lmwI6B54FUc2p8DYCnjaw4EW9lUEZyYuENdIBmeG8
  9FPLFgCIl2EBJAaackaErkHX19uVw1AprfKG4fcZ9mvIpOAAi8eCDfd4GjlgBcbUN0tMxNUG
  vRGRVzvVCjBnrnkyQj0smjABlu+qSHhw16Me5obHk6+d7EHhZFYihnXuOTcMIDBvS8sZjCM5
  vrGIJZLpwTkRMxJhj5PAF2w4sCZ2SJcd7ReMEq8zWOAKNVbY+iwf05Ako9i2syjvZTvwVrct
  FXAzp00WmOSYDG8K12i3ITXaSsNZq+XizWCLpwuB7D/9c2mXpS3EH8wlGrAVVWksQaHznnQf
  6mD+bm7S2kDAGAACP+MxkGRMEMmjEcAgqVCXAiHC4ZA+nE9Xo70gBAHERYMAPKEREmk8wF68
  326dAylELhU9gIBNcDBgAIFiYyRFhVEl5eRkZCSkYyPDQuLB4eDAYKBsMFT0CACAQoVlhYXl
  5ILCxLJEZWXlMeXxhNJyg7cDpMSkJERkAzOjAznjYG202YhgZJZ2ujViZMUDAUGB4sLjxKQD
  44HAgXfjoch5/TmAQGEiYwMEJKSkhL4yY5+iP/NhgQJOBwgUFFDGCTs2ACbU20dAHakJK1q0
  YJHCBIhiBoi4y8CiRS0TITZgUMAMiYAL3jycHCTg1AoVJzI8oECP0S3+nSceZaCQgR4GBwQE
  PAGgIIQjDxkkSMggSyeuWyVKbjihogQGA4EUGnFQylCECRVW9pkQAS0EBtquDEiSpFSprqNK
  ndHC4QIEBGchTFipAbCGPhUqTDAcYa22tYkLfEpCQBsZlHNJMaDzhgsHwSs/kKhQEIFaBmsh
  1r3w4UJEyqI4BhBQILLi0YkRFLDd9hPCtwHixj2wGgCCwBB4L4yAggoXwREK9EbwAUV0Engh
  DJn8oyMDP0MEESjAIToHBpBX3t0yPQyHCgVcW54QwIDRAVs0MDhoOUxn5XXGFCA7YbzrFPKE
  AAO4uSADDvZZ0CcKMhFoK6IKWU0Uhiaahhb+RjKUaZFbcjIhHEsWnCcEEyAZAYQOMsCgA0pU
  TIcrCn8wIJpppJnJAxOw4WSCHKcCwacC5UFHNRnZMCAJAQx0gEkHFHhSoE4MmLLAgog6aDd3
  jBSAgA5GCiEDiJAI4KOKZDohxZreUsAEkGQaqaQHQDHCtQskGQEDMQUxsM1aOqDggX1O4LDD
  nVY8ZgMJDigKCQp4GsEnCex0xENcIAEzgxBu6UACRleDqBDbYovNNgLcgqu3Io0Mzi4wyno1
  MMAGQ4tWtRSDIK3aBnDsNdiaiHG1ACzz4g0wNutsAtsYiACCZmvjDYACTkvNyGZ0O2hXtw7a
  dlfdsuyNCAIekNH+tQq00IA4lIi4IDoU0LsggiZKCQAC6Nz9QLDq4iJFWO2YI4KQC0hA4YNk
  C2hWA3zDKDYw5grxqzmj3CAhArfOAEw/he+KgADL4EWAOyMVIMqAgTqR0kosCzFggWoFAcAA
  SFaRSUOaG2EEEki+tASDno8ZwURHMqqk5w5uMSFMAeeCZhENb9bQA04oUERoYig4YABE03F5
  FHemHIjKlLnc1jVow+UaO0dPDBO+UmisyCKhU6QA1AdUMHOmElSUgCOYk7ATRA6+EkWADGZR
  AekHmuoAFZ0oPdonXjJYIEYlQcBoAwdvyqnDnO/ZxYSeHPBURgPKgKw23Lp9a17SuQ7+rrzM
  Yp09MLPOYna22kIT6yzSdk1iSr8gUFohYdHIWLOytpigIAbUspU9sMytQFW05QutAsEATL2g
  3LI023UZowBMs3QFKKUADdp99ywCSiGgAv3e1Teu8wWAYCUmihskivMusG9XEzhXwhi2HIv5
  ASWmAk8aAuCf/BRQC4SxjV8A1JzwfUoAB3iSlKa0FZUNwHpGIYAHVsECaciEFh5yGkZC0MJk
  rKhnieBJ01Kgi5I4JQOQuADxFNIlnhzuabPIxhmohhEgmUQACrCES0LIhgA4gEVAGxpTAMWk
  DRLEIAixnpKM9ogRbGBkZThACuCWoRTgYwNDIUAGVJFCcBD+AwMHSJIBLgAOZNCtekZIoo5k
  gqcH2AQqtqAUT7KCAV5cjVEBkMAtliIBRERlJxh5YQluAQK6xYdr91kJYiLTljIArInBSRjy
  aIeGV13gds0aDVtgIxbCbK8tj5lDBPLIDmFlrw6u6sPyDoIA3K2lAOcDwAAESMsm8q8A2jmP
  BirAydt473uAQFu5zNWFZEELAeBx17sKA7IBWIaUeSnNvOawEvvIZRCmOg0XKtCEBmYvMBmT
  FQNc0xeiQAECBKvOACDwQOWgQTDa+OZK9LA/l5XjGybwgIh88oAN3jOUAeAAReKWE2mAw3GQ
  KIEI9AHDnhnSBI3gUIqI9o8NjGT+GVvaQAmuEkQPtUQBGQipF1VkkkLsI6XHPMIALqCpEqYC
  cyLCgIMcOpCC1BI4FKhUOBQlgAbcrYw5wUeDTmARDp1AGJYgRwAUEI0UQEoZPPyBHA3HIU5l
  Ioc5+WFU/iQPk1zJABvASEPTWilbhGNBmjpjBhqA1LkUIgvUWWWpfidW4BBgnZjR5SkLM5YJ
  iGaVbBnAwfySP04aIJhv+SbIDEuXvaTnLpsBqH0E4Eu9SBYsE6hPZ10WrSWcBy/M6mRjHPOt
  C1KIVQvDF3ECMIGBYSZ5mxwNH/CwGYe9bwkEpZ9nNfAGDlTMLQVQLfn0oxlmNke6xDGI+tZj
  CDrI8y/+7dwVAv4jL7/20AEJXeiCAPXQ265KohQ1oUVpgVEPaZSjQv0oSEXqCJLe8I8nDUFO
  KSSAlbb0aS99kkwdEYmaKuCmYdXpTnuagp+KRHIrImqUsHjerphiqZxa1FPhFhKpAqkfVU1F
  g7NaE4h01RFgJfAgyHq4FJw1UC0VJFt/UpIwKkmuSANUXQvFE7zuQ68l4KuHbTmAwOZlsLgx
  KNoAkdhiBZexhikMZGnTGMqWpXcIwKxuNhsy4HzWPFgGDGlN20nrEHO1EzZKMgXoXHidNnWr
  s631cruF3SbBt9t8l3CX9V1dHhcQyc3LciskHOdCt4HT9ad11xOtx7qDKNz+bUyrGBZeb5IX
  leZFG0xQYUKsttAS7TVdEzlSporMl1JCA8apW5gmj2LCKUZDYSpykV8MSMACFjCkCCTAWlEE
  4CaUfJpM/sQNCvjoEZRY0egEMAEJuE/O3nlKqWeGlRRJbqiZ2AQW37saJYHoQ2CK6SpKHJJa
  PEJDK5aqLjCwAAdEAyuJ6hSwCHEAxuGMF02RK6FuMcMTlKQSFIiQAzT1J0cmBZKEVNE+LnfG
  RBXEeu/sAryivDpjs8NU1bSDsQZjmFSmhTbck8NeylIY3NXGW96Z8lyGCQF4Wle0TJjswV4O
  81JUOw0fJ9fBzGXnx3aZtiqDC5PX8ZYs2IEDj/3+QHK4oMtmRgCXA9QeyBq4LLIwJ3pIhcjT
  SVCfc86Bus5dThmW0Jzg4IsM0iXWPzcJMm3woQJCQCfXCLABVZhaGNJW9QB+w2cHyHcVK8Qq
  PeoRAo1kGIaAYoADGJwKmrGwpizqgN6YInRBcLVFP+wv0oYEiYJrpGcKKINA5qRTwp/07zNj
  4Tn44SAncUJCIQTAPhIcOpC8esUmQFEI7lQPoOnM8ZTgQAcImbknsRYAEkgKzk4gjkcS/HFK
  3sCfsPgAYDRIro+zRfV50cKcdMACcgyhGUbJcWCayjGebwYAIpCeUpoc5bNJOWFdsweA9oFZ
  VCmzkmAADEAv4INCksD+l8iidsBMWZzHVtgCWoQlWeRvacwAf6rrzpAuW7xl715nmFRLC7qg
  M+4luF7pdiprJcjCLBhAVHDlLNzpA0NhmIoONcbgdOoPY4oFX9op0twpCypGABiAgOiumbys
  WbbMfUDpoA6gBP6uFgIv1SRgEzAObSACBCoCrDygEaaiJDBgSPbBZzDAAvrhAXgE31Dh3WJs
  4sAwh1IAjACg3JphK46hRJrmRixlhoIB9SwAwuTwfOTsB0BvUEBC9oYmw8Lt9gzCAkWhQAzg
  CTEEQ2qIZ/zhj/7ISZ5kAWziJ+ShRVroi67mMdbhIF6Bc8JhqMLPcR4nRSBFSp6NA/qBwWj+
  6PSQgZI6JARsKoSIAgL8zP3yTM+ukKc0g1ZMybJ6braS7pMKgbxqp3cE8FQKoZ+sYA5ZY5gW
  8JTALAAjEOYmEDYQUBBdw5cSCy+Obra6xQMbMRTgwv/Q4C4s6/1sI1t2TlQQIHduBTEaIy48
  DHa8IIKEoIHwx5+UQwhHoyCwjj0cqDPwJX/0YLKWRSz+YAaNJAAiwMIS74xcKFIcSk9exwHu
  RgW+giFwJhiKwUkcQHGaQiU1gRO2QgG4ENbkLRfasCksQBLoRh2xQwI2wB5CZ9m6ECOEgWj8
  0Gxy0hmeaOAuDHFGoBLAbcNwryOpDMRoZiRUpIqa5I+a4hIdoAH+jMpKSsYm2IsgJrJCDOAN
  P2TivASSgoHzMuACsMgBDCMsPQBoCg4cUgQEZijaXKwaPwwLFvIX5fET3MIoSeFihKBZGIuT
  eCVbUCVVxrF2mglXTktCmNEvGEAdiaCVCIOxXA4xgKkxivKTBBEKSqv+gGsDowz+vGXmXscM
  eOexUuc2rsA7SgX+GlOzdsU7DEBXzMaWAAB/zON/vCy5yMe6rmmyxsudWO5cAFBXkglX8hFa
  +G1LDKeE5ObxhsqhVo3voMGEKKDaJmAEdGajmEKOOsI1CiJCukcBOOAiLU/ezujbmCITKKAE
  RqDYKNIAWEQE7EH0gFKj5vPXznNewvH+Ji7SIlbsq5ChDZ8S95gO5AqCJ62yAZ7EARxJHnoy
  GBzvvzZyAbBIPa0IIqyQHQDgABhMrYBkSDxgKShgAhwAAaQEAW7zERVgG+KyG/aBF6hmJnKh
  JpAklPyjWAITN0bzoAhAteiJvAiDSV1OXsAFSuclmRhQViRTlWoDVEJlLSDUiYYJNmCQM1/p
  M8WsMT4JSvuyWoZpCQho444usgRT6VrTZXjDEOLRHrEOoKr0MGoFNGnrLbBkHxVCDvkgVvyA
  Le40V3ozVeJisuzRedbCLRo1ssapMI8Aszwg9kQiHzSsqEAok8aoBUiAS25hBRCRZ05CDukE
  XBSQA6TBEFH+iEMyIkV+rSaZL0zSFIrkAQRE4B485x4cTx96xgLIoTdIEyaSIvZWLBI8ABlE
  xAwfKiozTgEwcS590lUtL5LAodYsARM+1EqIQv7cZz9BpODwykWfhAEsFEqapEmgJEoihDcV
  wAF+okRyIRfPkEubztG44M70Ak4/iTopo7TIgj0qq7FSKXpSdRCjtBCKszmvbkyvYDcMIVpp
  LlTsMToPIwL9tCiLlTSxo2NUCzV/iTE8yQMDNk3lcOWIS35+S3Y2g0n54uVC0zE9r+a0rCFd
  MFtmk2dpk7BWs2zqx1doy2PRZgDujd3cKB9KoooIwlOrJQAOIKRa4gJwRkWE7aP+ZsxSeQPQ
  ChFDnubU0iTYLIAD5qEDCGJLBEBxsBYMe0ZzHEQlVXIThElh5SxrZOGnlDUcmpUfMuFDH1Sn
  onZeOcAn2a3dXk38HuEe6oGj9MECGsB05NQZzocABC5NXvQAJq8bWFQShE/47mShGMoMN8GD
  BIDwHmCoKMABgLSJTJcI7SxiaQua5lS62gnU4AV3cCVAAMAB9oRrLdMdH5YvBgs3YYM9UJYU
  icB067GTgLZmewN5dY/omiszUAmyWMlUKnPpjikuFPINoqMRpEM5YDZmaUW2QhNcKEMzvzQt
  yrRRwfQvSgmVXk5C9pF1inaalOpVZ+JLiMb2CCJfB0H+DoxmBDzgjKiiJg7gAXjhAcQKEDFV
  FUxs185ECr/QAkAAaSyAdcUng0zGgz/4WxmxUkehAMyyEJUSq5rSWR8kSnIvlB6A+aThcNvt
  IkyvwbBqQ3dVRbhVjgJ4YYHnSaaEG3xkFizvZnIGHBZXhzOMAkaGa791hBcCC2DrF7G0AyW3
  Kw7C5kzOWcrUHiWj8Pinl/wizWD2l2YUN8nrmlalI7AFmr4lSsnymJDkYC6jetHiHjGrA3VD
  jlflNappYMCXEcQ3MzSjSRsrLaCHUsN1H00X/tBs5Fq26gZoBW8HmMwUSo+Jp1ABb1hIFzZS
  ATyByhyFCqrvT5DERFWkYo3+oh8srIxowfKuaop6RgIwQAVGgFhb63zc4QAaINiEdWy1kiAO
  ojQ+NokMOFn1llmD1QyLaph9uGsuAApnOEEvokUpgHB3IonroaNqovUE1n4N4NkW4cIypAt1
  BkSEYWnPgReEYpEnTONgVzWvuFJvyTmftLQ4C6lYhVD90boaUjTQ+Hci7X+gGQpWdR7n8X7T
  92N/ACLG8Tza9HplU89QhXuJae4CGTnWRzkIg1loBf/OOJb2BTh6o7Quo11SWtDuQLRs53ac
  aZGj+AgKBFNLKFZdaAqLikT1E2g2qh8EYAE44gAkAMK65gAYIRVeWRVupuC+6KMyQVO0ojAB
  QYH+DYmlTGQnfpUkmAKVZXod0ysFkvlMUIT2UlfcWtigQ0El8vb3SowFQqCJr6WXu+gRRoIY
  PCod0JQuYIdq2iiWbWYEFGQfWmRXRWCJIe/XVK+Pq+WhBUaeQfM2P4EJyYUAJpMMApEj5ACL
  BxEL+vkL/tnlrlRXPuiL03ofIQMi0eKl/dU2irKhB/E17Zghh7dkOfZ+j8kd7Ph7U9q58s46
  5uV9zXct9JgwE4IyCoIIvqkINXp9lkM2d2csEBnlvvi3dUoAHEBH9hcjJEEfRneYqWwnS0I9
  Dxoiyk2ifspM/i7BZm+/AuWWG3hOA2U8F8FGbGSQUO88BRFJCMAUk/X+Q5iSbxvkrHHvdHaR
  AoiPWX+BouCGBc62Y5NAAnAYq6hiWKFkGaK3RM+ALn9oQ2zEAybAdIjZQDDA8Qy7cXnmlxVl
  sSlSuTFjOeY5N4w0Tb0DXKUJO+DCvG3u5j5bVlLJVjArS5NEYlOWn6jU/vTUmQjTq9NJMwm1
  urTnjLE3Gu83rZ0IzmJlt7cJBxkWOiczMUxlGQ2aCC7mPLLcXTjgScFlDxxrtWljnKy7jpRy
  Jrg71RwkiPXaGUyUcowbf1/GARA0vdsIqWU5TYRVcRQhA/B8HTMA32D5TMj1hPwrc7BNpyBC
  nLkwmRV0JP7LWYnq9h50QrhmSq7tMdjocGv+AQQURYhrOVPuYXE7IB3ABZMo0i/2gQOQ1Zwz
  ZAMgoHQxTQLqgaQY9Nc0wd6WYcmhAAuolyHHVFdiHC6kcl4ios8XQrqME+o2o83ZIkuTmzBT
  NrBGDi8Q+T+cScYrHRA8O2MYMrQDGrPK9Ht8M5N2jiicbAsIOXnAjnn170rZA1UOAr4ioxcx
  Q6Ono5nSXEtDOlHbgqSt5xMeIBrkHEQyohKb1nSeNmWLZOH7rU0M96+RWtPxgWjIsCky5Vah
  NkFg0kw4wEa1wT5pSBz45tjdYWpKzSKqWSbVuUM1DFACooW1yGVGJk9K4QAu5/cMEW5qCBkY
  r9Y2lKPiiKGrRQD+mIUgFKADUKhmVOERPEB1UZIChm0Yvi0ZRkd5oeWY3OebAHnZ3xTM+Xiy
  xaftM/7z7ocO8CC0zCKRI3DbQ4U9UtZj8KCdbhPUtudUjh0K+MnKBF527P6MgXygl67fFZ0Q
  npSnJFmx7kDxt/F6lXyqUYUBynzgUXOVZiM6hRtSO5bhh1CmMn2+TqSj/MFJPMF0X1sQuISE
  SujVrEqkzkSjmlJYa1kctgZqR4ZNyohtkuBEPUeNAnH9QI/m29oQuw1x8IGbW5+oKrSFJWSn
  zU0EwFMBLgeWZ4aG1crVXT2/lgjeoTZ6YOIqOtycGWFvfaHxGJdBLUHFn57vGohlFav+HPFY
  st64fmQfCH7CnyDAmGg4nM9HqXleJhEIlcpAIAqCYiAwYEAGhyG5bCYDAF2EkqOZFAbbQCFS
  mSAE6fS57+8DFCB9kJCgHCIWNmlcVERMQVxlxcl1BeylXfL9cf4ASA0MRCwRFhqimDIxNkpN
  TTHAZultdv7tBSBccJiaHqZyXDTePRI/RmINYNLW/uFmnKywtExLs1ivqKSkmIx0bGRkUFBI
  PCgcEAwEMK/XChB4RLesxE+3sMyrZJuUnKSUhHzDgMGChA4qNhwAwK5PgAkTMqiwxkLFCHDg
  MHQIUcKECAsPHBBQuLAZlg0polmzV6/aNX0hQoDo4O2iOAn+EhwoyHmOAJeRZQKESLHBZMR6
  9lJSu6btxMaXMTd4s5BwmU8hXQAQwHBCBbauXPOB9fevRAkQTmUOvYhBgSaqVYUEmrALUaIm
  wqZgoSQgnaU9b6sWOXJBAxMmbqBEiVTlGAIClroIQIbgrxoEDC4YZjRhMbHNjv3+PZOGAARd
  vej6yhxFSqQrBV6H4ttWzVUClCtocKRLSSleht0EY0Wsdd5kbkfuybWEF3MSHxALawWptWvj
  oEMToDCC3rQVK1vqAwEiIMGP5gzwDL3QMTxq8+jde5/thIl+Zb0JJJghxQkK6kIHYIIK/TA1
  nloDZQCVCDKVIIICIqn3AzrigMD+FVLSGCURNv68FAJa34RT00c47eRYF38F0MFEJyFlTzQb
  LnUCfWTBFIIIINyIH0jHjXSJABmUkAJX72ETFn0mIFmjh94M1SSIGBzQVoQ/4FLBXGCp4Itz
  h90BgQKToFPJVTxOWYZjgiVR2CKrsLaYJI091sUAjhlAmQBJLAFcFMUsFkkBmpQphAFd1EEY
  L6ihsqVmrMUySWxctNUXhD4BEMETl14anCOvxAILcViA+tpnZK6ThiBPkPLbYcEMYwwErroZ
  6iykMiMABSWgdE93R0kEFlP/zITBOOUccAB6AgT6h2MXRIRhUt3NY002MTY4UwblmZQBsgBC
  lQEGEpz+s4W47hBAwAEPSICBtwmVOQAC4HRgQq7VJNXSNv8syaRF4gxL4k5zULYBPtjco89M
  /PLr0AQIW2SRBQrMEqEaDpggZD4bnjCCB8CsprDH/KrbcH47nqgeHRGYdghYiS7C6it56RUn
  rVMGNoEuaq4JxXRVSFIAT5D2VedbAAwwmMsQZBFbKK8VAOpnyZpYQGmloJZKy5t+OonPoQB9
  yZiUMVBBdBFcgZ6Yc8QJ6V5KF3Hd0EaIPRgUjrjmc7lKK/0aFrA0Cidt2CGwwVYoeeedRBPN
  9yuwaZWnk9m2JWtGAAIwKw13KhFZZD4zwhTQQDZh9MB/KDpgbLmno26A6gT+GLDTAQqEVCYB
  uVywgYCXH47Pvfmi1XBN5DhA4rFFoJjBPNuMsMEFDxiLnurPQ//86arnpEA5IU06NAH76dPN
  BRRMUB3e47NugPUSUGDB5w4MWvJfRSBQ2hJ0KbJqBcM0ChvXffkY+U9G2IwUzllTMDSzM0kY
  IA6S8oIehFYVoh2DJ3F6zC3WdgmoEeoIhtKSIq7Wplh8SWts49/XhoYAYlxBNpPbS5jEBZm1
  cUEZmDCZABgQAYUhTQsylJkM1eCOpmEBPT7a1lsCoIANjGBwhdOQkWT0D/FYS1jAKxZPIOc/
  qwSgcs7iVT2iJZ8BIWkEBhII6LjxgOxRqoqtU4D+Ax7gEHGETCAUKF259NAuAwSDAyNgUUpy
  N619xEQmTAIROPhlE3+Fi3hv+VEKRsABPEjQXA6wCfoQhjBynKeOK/SaxAzAgYwtTwGPU90B
  ENDGB6CyHMHLZGxYd4AFHDJKWzAZHTT4gUQoSjiQeNPW5ACpW1wRi2hKlSqgsycrICCESVtg
  HLBCmclpAjKzA1Wo0qEMDHrhhIYyxC/mxqi+hUp/aPPaDIcmNVlYQkyz6aEP11aJawJIAAbI
  4awyMTmlhWl8YsrEAGBzQfUIQAEcAEF9mnWNIvFnH2QZwVk81y9z8KQAwSRCAChwu2hBK1df
  TFx9yrIBgXwOA0jyj3r+sGK+nKCLAvDKSH1OYJYOfOscaHwL64KBxK1cqFfdeylaOEDIQopo
  AaX7lyIB84Ao5MFcDUhXHEHqVJBaRFjgE2XbAEBEyhwgGLDbi/keAEeoNuxAUp2AA5JWrmJF
  aUpdmN0g6gKcYViBl3oB2h6umiz42SxNTWhDpnQZV16GKZqhAIADKXWLFc4ODFagwiuaObP1
  rPWEpmGOE3TJN2oaQGtbQ9thH+uHbOoBmutkJ2m9EIpoAtS0kdqCOieYtlDc7SqSilBAB1rQ
  3CF0RgttqLekaJ5wSTSYRbDokOjxHg3hIx/8oU+BPpqfdI10dNo76flUmgGZhKClL41p7Kb+
  VNPaJbFZ9UIc8njqDZ+G1ZAPECoVfybdkQjgqHhwxwGWKhCxPvW+NJmqAapq1wdm9QJbHUBX
  v6rfsDpVHGQ1q7mK9d4iBoCthHErI+CaNXHSNQ3/LRNeB0OKNhxmFa36a6gCy89kFBY5h91C
  YnnGWD9ZB5uzQ9kuKPvWVlwWVJll2mZjaE/PMsRdofUxaYushtigNjRFOG0F99I110IGtuiQ
  bQmV7AAOeIAfFuKVcn/1EjEGEkTjAN553BFMx0hAQMXN6DzkkdxsbGRGTwEpQSRwXQk8GDnm
  w0lKM4BlJPVjWouzgAFmWhV3gcMDezQol/XBjSU5iZBSHfNvqQj+MJqybk4KoEDIemuBT4M6
  1AdWl5jZ5xgroqhpRTCXV8P6DRDBOqzgCAgFHiBEdxi6RxGWLCFYBpxvyjVmFNTwFUNSBw9/
  GFOaagXP8tfLSF0C1SqmTTqbdllPIRNQgTLRjG/GzQHemFNvghnTuPZL2gD5DHM4cj2hzMMe
  eqGZG15IaDMxTiPv8J5zmnI5QzOAI5pFzX5MKI069CFvichx5QomACg2sJUQjkhCisaACFSC
  Dnj60+SwHZ79UGivTS7FcClX6xzwkAR54CUEMoFZCukAO5pBdfUUuRlmBw49CqmPTJTRWQSZ
  FkJestKJzDM71HAAlYJI4zYJNdPVlx/+p6clpkTX9eQOkK5uee6pvc3vRWb9UVtfRa10OKGV
  SuFBcVMTNuMck1WvSLSwoYqvq2DV/RzR7LSjY93KoHmh6ypP0WAimqY9HY/9eZz+zpwZ3Jbs
  BlGxVwNmjZpaa+G5++2T0IrW3Zmfw7gsIa5+IkAOAD3saPPNdmVEWdVuW+SVQZBEi2mO4CUA
  syB7J0dyfMR00i5TFyDynu8ctysryPmKLD4ep5anAyno+BkiTIFgiIMBBrBO4Cd3VgVAgGGf
  A54Qe/jDAijAISBxXx9sft0k5qrRTIGJz50U1RANqwHmKJFjQgOAA2BgPOSRgPpkkiCM+991
  fQPGYQCI4Mj+BkjF1C2E11hdgjjg1r0fVHXa1qmLIGWABGCP2LGODYkNYlhYjoVT3rVW/0TO
  GqAMphhTIzyCQ5ANCCJAAqnQLaCY5OCCQzCA4eFbDuKbaVkGBBSAGtSCibiLBtVY/VSYMfDN
  uBUeJajTj+VaENIGa42Lk60NE+YTAcRBmOjNC/7J7sFXGljTLWRC4LkD25zWBIEeAmibkiGA
  SuURiAEDOARDBnyPJdXEAqASTiBAsfSXF3ZSBxwXxP3eRiWXjGTM8dGZTWxACIiOuhWAB6AA
  lmQMB4TDghnLJere6biQK5FSKZGSA4hDMCjaCNQHy13Ay/mhEIjBzZGiQTFRQon+kTcM0qvt
  S9D5y+koYC0QlgT8lAMSlAkERQoEI334Awg0UpBswzZsAMQ8ITswYLpYxJNcV1mYhVmcxY08
  kTX+gwiEQ9+pFW3MCRClHROa25MNW135j1VJVnAsm0N0yauQDWZRkwSdXj8RVhkUQQSQQJaQ
  gBOET7k1oQ7aEwP1k7XZQQEpwQUwwOR0ghAWABjgRp4gBrC5oGblkwrN1jOt0Od9HxCNm+SJ
  o2KFXrr5AfawnWixFjpgIY+VWOHtTeitIYqAHMhRm+ZBGRXOQSra3wGEwMU4S7T0UeZ8xTbQ
  h41k3bfY2QY4wIMxwADAg84VzHI9Ub6EA/MYB/9smp/+ZURMmMVGwF60uEQGsIUClCTtYFl4
  fRHB1UjtfYikBd0CoBUulhQBPIADigBZkEV9nASWhAVYvMgKmEAGHADMTUkmGAC6WMB1nQU/
  EKM2CIljPuZjMgXGgUtMoohAYqYMzZv9uQs83pB0VAESMgZIisrpmcoMogEE9NoAPd7YHAMT
  VpWkRCFsGeQEdGCe3NgNNqOE2FtBxg8S9kyYoM5FluOTyRB2gCPeYCGoJCG2UQe2pdBoFOaJ
  ed7SbKHkWcabeMorTF+SAYhNgmd4MmQ6HgCQ5IOGIFd8nKdXqKUIDIVTLZ0ooVFFReZyGSKB
  9AN/8AdMJUB/tR0ZNNBDxJn+jOwDgeIlXi6IN1DAoJSku+xhMkGAyYkfC0boKuUETrBR9XwJ
  H0ZPHY2TxGCFAiSleJCFIULmiTrmCXiAgFkmiKoBq1kXVGxlCJBiGI3AjXLDjXqAB4RD6czC
  Zk5bZgppLj4Q6DUndoIT01QTkvUQTwAAWQ4BACBAprTKDbFjl1SHf+GjPVXhb9qm3NxFcP5T
  SZaWaQopvoGonLCk5FXkuEmQdHIYP6XkayRQSEKnYjRbf7WFTk5Un44EnYBiWAWH79yXsCCc
  IdkEuqQSKk3Fe4HEjsKfHcqasLAPCaobVoBip43DmFESutgEp83RZlLfmZKq5dnfi7aOV01g
  pAH+nY9mwkTNSoStERu5kceQFRW0EYYaixygo5/6aoSEREFuIRMuYS+Viz/JJupRBQBAwGaQ
  m5JmZ928039eKpemYRI2RgtRECdgT6l6K0mS6YkV3vhwwf7MBp9Shgyy0HUq1mg6io8R6a/K
  ayd8q5DWnAMo5HAO5+qsTrlY6h8M2b79zM9MoStBlB8S7BQq7MKyli8x7MOKS7IMCumlDnp4
  aKTM66iQnjkm63jO68cWUU3uk3iWHjvhIy7EgTlSW+UB6RDUU00mq/f9AftEz/O84AtGD87a
  bM3mbM0GEc29z7e27LaNluex1jlmArqC7NIyrS04aWZGGGSd6Zg2LchMdh87yWrVau3W9qnq
  6KzX8mxmhe3O9qzIjSHUKi0ZaKy9cu28NkZVlZaZte3c0m3d6mLaNm132e3e8m3f+i3TOpPf
  6u3fEm7h1kIQAAA7
}

set pieceImageData(Eboard,70) {
  R0lGODlhSANGAKUuAAQFAxASDB8aDyIiHC8mGjUzKUI2IlQ7JUBCQGY9KVtILWpGKlBTUHpI
  K2ZWPm9YKV5aWmJiXoRnMW1rYoh0RnJ6cp52MoSFf62JQ6iMWpKTkracXaCgmcKkY7Wphquw
  q82tbdKydtyucc67kbzAvOe5fN/GlOXNndDT0O/Vo+7hv/Xu0v742v768v//////////////
  /////////////////////////////////////////////////////////yH5BAEKAD8ALAAA
  AABIA0YAAAb+wJ9wSCwaj8ikcslsOp/QqHRKrVqv2Kx2y+16v+CweEwum8/otHrNnhIEAsPR
  ICC0790AACB/GvYBeIKDhIVQbwZ9RIl1ho6HewACUAKRAHaPmZqbY5cEBACBQwEBAgd6aAEE
  CAgKk0YABawIiZ+2BAMAR6VwpKJhewoXCqVOAgEUGQp7nM3OSQvPSpcMvkQBAwMMBbppkgUM
  DAOJRZezCAUF2eu5R6GWzGCfsRUMoU6kEBPzmNL+m5f+GTFwYIQHD8uI7LlgYgKfMwMogDAB
  QtE1DilOnDARooPHDSAp/CJCgMIEBwpwiQHgwIGJFB0edGMC4IIKFRsOHJi5plL+qVtA38Dh
  KdBQwKJCDBQgoQICBJ4BNGhAoSGAxTIBCuirgEBnuQhSL1yYEKGsUwgIiArJOkBPqJFd9iDg
  gIJD2qtzAERAgWLCBFRt4AkWjPTR0cIEDSLkubDhQzMRJ1bchVEjR48dQG4QeaTkyZQDVrZ8
  GVMtkpo3c+6847NOUFtwKhV2dLioUqZOoUqlajVN1q1dD3wNO7ZshLNpd6lzSwoYgLl17zL5
  s7fvX9NnBmuPNzvNK1jYpQm4wOImBZ6fRqjIAOcMgQwlUpQQbiSAhxU3U6QwUaJEiP8YwCUE
  ARtkMAFoYth3nwomYNCPEgQYYMIKMEkggYBoEBDABR7+ZJABBiCGaIEFGXhwXndqPKhQeM4I
  AIEKH7DC0wAffIBCBcWkotUEFRTglUJghVUBWcc5Zc8u6DCHoRakVIACjBHkqIQAA3CAn1jt
  BPZcWUQa5xcElbAoyAFvZDOIJEYcYIqK/xAwwX4mOMATHB6kkIEqEMEnH14/BPABCytQKN8I
  IYDwH2dGEIjBAw68kaCHG4XgYBOJbDDCCBtYuKQZn3hQngodZADShx6ZJ+YgQrGpBppFnIIn
  UrhMUA1ga5V1ASiqrlRABH75SN8oE3DAgQbFmeVUcvVto2QYpEzwAQkaVPNdEgIUMBUJfmXJ
  xh40fnATX0/yFUEkmQwAhwH+CrSEgFChbVvbDwQ4sEGu0rDkl2xD7GFABshyCt8JJfApgKf5
  ARzCCCCIIAKiRQxQ4AMIAqOAAqPKiY8CFFAgQSKnivHJBiqwwEIKG4QIgn4reHBkJr3Akc0x
  pNDbybsEpNOxJpK4RVTM3J0RAAL6TOBrEQEEO+wFFRiHnGkAMKAhIJtisQcBvEKg7RJNJ80K
  rWtgE8tSgbbQwgoVVLAHN44YMAACGXNwqQccUNASHWykShJDFMjsjCW7cF3Ge/EFfIQAfwaa
  kX8hiHAohu9RwKhKKwGgk4Z6K0TAAX/cHMYnGYRw0wonlLxBCjc1ZPEj2KwdDitKLRf15lOT
  xAD+BK8ixXc55KphFa9CT1zOBFIRm7SxaGH4c1tQd5IVKNNOuTwpmoMxgNBTf7BCC30BImv0
  ZBjgpgca3aTCfiGEpEC7aowHyhDjpRACMbNFoioo3GMB+EZ82off+AaDkPDCjMuA4xqFPjFU
  Al9NkNMxmle392zgPzc5QQhOoAKNXMABDCvEz8pSAaSRhQGse8PrwBALWgmAARdIiyqAwifD
  eCJR9OsaAnjVox+NIkjCU9qxjIcA5OUuDKd4R+UGFInm4KFKJJDRUlAwLlJMZVyGiIgJQiay
  QI2PI+WjgCvWAAAKNMQXAMgAg9hDCjh8Qmf1Q0MRd9Gzv23gJSVwRSL+DEALDS0ogiUwlAj8
  h4E6YI6OMcsYxCBHBsk1oglvWA2qCICBDXikBIEynFQmIMAR+oYBFZCKsKSStNWhA4HZQcAE
  cvEOBmhglEVcBxoB8pb6uEWGESgbAmIGG70Er1hFKp5rbKEHcy2rkIDAxzsIUSUVkCCJ5giF
  VPDjkEIIwAHqERmgAJUfE5ggVBRQW4oK4D4P/YECGcnJAhZgKYNkAGIp0cMQ0/CGcATgVz/Q
  iVXeNQYCvSQFHJjAAx7wJQhUgHQFKwGh/rcBB8zxoBjUZ0oYCAZkrK8JAIAABRi6hk800n8g
  WAE1T5AxCmDgRBosQAU4UKNjfmBYwwNhAUT+yE4CXCAC2SCFKbnCCwTMbhvreNo63fMOtZBi
  ljvdwgbLFoFwQOA44WDABMQiFiKdJRzqiI0vSgG9VhbSXDcDgDbS+IViBuoDOoMAX1qggmYS
  4j0p0GigqogfOGXxfFy0CR8jNIITdGABLRmdCkAnKQx4iBiPGYQqMJIChyhACMsAgARKY0kt
  EIADJyCdRqUJKCuKbz95BEEH/JeZAn2ooxmTwCALCMQDDCOrAgyqAR3oEQqmoK4qOIhHQUqI
  AKDwWSTgCwk+QCyzsEIdXLVCRCfAirYggFizpCUuBgAOBhyHlC3UnW0hMA+FhOMDFzCiz2bY
  QQ0I65i7HRaxmDr+vCIl9bcrxUUvfumx5/QLa7FcoSCKyRd7QO+xHFDBjYLrBbSqdbJWdOtm
  4LoquYLAQQao613zKlkJguiveoiu7h6rH8Midg+LlUljs/DYyO6VspXdXzUzu1nNjuqzoBVt
  xMagk9M6IYx5I0QdRNXa8cFWth/lrxdsewHc6pa3Lz3Ob69mhuEW92fI9YUtstHc5+qrttOt
  br6um90NM4m7F/AuB8B7UuKIpbxG8mQ6PmEumCWvnu7Nanw/Md/81vctlxCWfs1GTGiqla3U
  zIgES9aoAqRIAPqx004ukAIQWEgCFsBAB0xwgv+l1QQUCKJgHys+lXUDAPukIEe1W8/+DLyE
  imITWxXFd8U8FgozJxZgRy002r9RYAQToGgRIhQq7y1SVBrJSNwykNYVjECAOuYClcBiI3CF
  dywTgIBKKcfOE7KiEm66gNVc5stSpOO3NeM0G0gBliCj7Qda9cuzcGRlLAw1yyU16SaF1KUw
  KwC9ZIYZe5kVgLIcgwmkwJK214DEcGh1y2nJxpbNKggBeDGSbA2wRjJzTm2uKgIH8d57KtiB
  B3vIkRvwnwrI2gELPIB+Eh7DjENmuA7shA+XsuIJ2CNrLrzHwyITNWXzc7j/JI6znlX1qve5
  4gRdgKMaoskDTLABA5T7CUIx4xI4dzL9cEACHgXZXk10Ktj+qDYKWSE2eNWNbGWjQ0NHN3db
  9PCzpUZAHQOo2TrkTSUEJI1+V9dCJSBA0vFSDxDhuMAxcdTyPGBZy1wWL1P9okMxp4MdZrbq
  SlxagYdOYy63aiO/U8gtDogNmVpNWrC1MICfrzWSlpVPCUJ1zgOQ1gzQvlywSnCTEigsBBnp
  CGamGLoQlAAD8uy7PDaUAmrmZwMPiIM19wewO7FZDAXYQHyoGHNp7k8/APOPZlHtoY+umtU9
  B0Z+lXGqmoTOAbqPAtgvpxPMjb8zaLWmAEVVII2n4ALYod8KI3TG8Esh6+iuUY1Q6tuvgxL1
  b7E2S/VS6CALIFSAh3cM2hAByXX+D+z0HFt2Ul8WAXQwNVSjAeNWOyI3Q0uFgbi1f142JIV3
  gGO2dom3b14AAAqAXe91GhGAXTEkVHuwXm4ha9UiCQJgJaGGAjLyM+Ehb1UVO10AOJ8DetXk
  PxwgQA/gcN4gOQcQAmKDH+XxedQ0eqEyKiFjAh4CfpsXBQAwARPiexEUE27yJuJDEQ4hOWKw
  LxNEOiAmMns1PhnBH9L3ER6iczvnAGRif1cAAPcRE3GwBALAaB0gAXzYBKVwABSQcSZwKdhE
  JhoILwMDaY7TSKm2aLQ1ClqFMRngEf4DNxNgdMfHJODQgVp2NMPjdcvRhVCgVaygD1n2Uso2
  O2WRVKv+4zpU8gk9xIpMoBPPgYEkcDRII2S5QAq7cgED54Rk8DNH1UHCon/iNV649FSzUIIv
  A0aKBwwTAC3SYoMbgl3c0IXQ5k+xOIzKElj5IglisVegd0wqpBbMg2VMxYDI0wWRQUHseFkS
  5BEeIjdMmB2S0x+hhnCWpVYU4T+ZYgG9pwIKAyKKVGQKEE0ilh8lAHyfQAH4eAIj0AETw4tG
  MACdCAKwFUk0h1mI8xE5p3OitU+MMjGmp1oTYwAPQAkj4GvZxCIBoAD6UYiHaFCsAB5F0wFp
  RVn40YgXBHZDoCEOICsM8ABQB1odJRNJiSsUUCdGqHLKoHSkKI9MNYxGMmT+/3dVvCMWxDEk
  hHccxANCIqQNzmU1HpkE3FIBz9JlX0YWykZKG9RjmLeMqzA7A3hLSGOWZnFe6IV2UjVVt9ME
  44QIUBAVdREBYQKXBbBlPSJ5SUAHhzQK+aABW7d1vEVTPcMtRnUWR1UWs8QdHDMAYJFb4AIu
  vAVTlkkFAyAvI8Bo+sE/l1F9jpN9ZFAJB2ACAzlNRthroGNNAFMyQtlXHdBof4AVquABJEdq
  ALVnLeFSEcQRHhJ0DWUAFHABNHYpjTh9qdaPobWSLakA5Vd+ZwQF55EIp8CHBGACLKCFCQGX
  YgQTE/UEcDAxW8Q+e+ASJPeG1DQCkaYmkxAmvBD+G7sUId5zaaTgAFY5TVXEVjghTyH3BNhg
  U17iF3YZDu92eCgYBnCAQ8ISjeRFFhwaZiqVDbp4jssoU1sWXtLoVOKQCwIwQ5ypATGlPMxl
  gM5FeF53Dtb4MjS4HeTQBCeXDiGaKDmIAiRQmeGhF8d0AeEoTOnwbT9gdEXzLK3ZmeHFFUAl
  BNuhHWvBbbmlX07qpcEYa2FJBRHSiR1wKbWZRx5QIFGph+PgDQdQAgP5eWv1KRpFagzSESAR
  IlNUAhbQnGtIChlghHLoWpHyVwYAneQDEg8QdhhaAA9wcSDhERyQkqomWg6AVy3hkulJJp9w
  DBd6DRCgESNwIZTiPSn+0AIVcqMIgAT6U2gWwKCIyAEVCXzf4YQjMGoZEWjsCCihU1CgUG5T
  Ixb4qFak9qcUAhJGJ3fVYlNOYRxIJWa+FHdV0DTIWFInJYwnCqSzaCSukxWrKAY7wQqc6aTk
  2lTIcZoCuGVi0RsGlHayMAs3VZiHV21ThY2DsapTxgBOKh1MgAveQgIXYA/0BC8b4qSU1zF/
  AAEmhSx6EQGsmabj+owfKBVPwSoYqlW7oV+7JTxDgozgxQFlsaRRMACbqhmacYcaIwEtQSan
  1wkEwKf7Q6GCCq2k5h+uVzIgsmik95BfwBIOQJwqFz6N1h//kQEHEJEVtJG19pbXsArJwKn+
  BeKdO8coKGGq6XkLMCMFfqgfBzZEn3AAGwcqGKClrmQAJzAyFYeqiHQpHiKVkigso3YZndoB
  8aFW+qGshwhuB8BrksU/AmVzHmZF/ZGfTLIK/GqLB6gU1aZGBRCu0BivdWmuyoYchlkHxYW1
  Yjo9zrKy0siZKXRT56gVekkCK2NA1bKvrHCLV/qvQCiwg2G2YoECMeKAS5AVTAEtUeI3YooA
  NsKwyTVCTsQXv/sDPVQjrqkBnQRCzoVuu3VStMOslPYtvCUrCIijJiUVLVgFMMt+MkuzFnKz
  hFRkOzs2N+Gz0SqGrWd7IkC0inZNHoK0KdgSTIsfGqFpIAC1ISD+tVSrkfyoqA2ltXc4s14b
  WvtUqmKrnrDhsqdRJ4U2KQl7OW1bcXBLNFYxtzCBe6MIIXh7Tt3wBnxLTX7rSIA7lIJCuF54
  uLfZVgBDKP/RuPjxuIWrBKpwDpRbuVFlwXGRuSS1udJYrnZpJKBLpJJLZFwQbqd7TOSqutgK
  VXrgurgVuyI6u0CsUrdrmAEbhLvrhb37u5YkvMakAcXLNMgrscuLb47pu8kRvcXmu9RbVNYb
  Adi7f9vrhd2rX98LxrKwmierAeXrpgYgARanmxrDc+ZHsFsQRiwwNn+acMbaVnIIJ3lUcSOi
  aCJJdJj5BUYnLI5aQRqxAkQ3QRoBAtn+1Kr88R/Ah6/yMA4O8GApGZXnmZ7oOTlJR8TucAGX
  shlpjAyVrAIFnFjgQQEaZVcY4ADAjAwZYCGKpII3Ab/XhAEWcGiM/EDESTKNwod8ADIBZgJ1
  Wn1+1YnLJ4dF18NIQCXXdg7VmIBlpEYRcMTrlsTkVTZdQjzoZS7HWAFl5gV/YEoY6KR0KRZ0
  cQHVcl4+IUqcGYw/0z3YcKW04K//CrBoJBhhl7HSlgtDFFFPAi374HhA4rwpBD0JW29i4Zbg
  ZrDhgl3KAj0VrVRaxlv29QT/tj9rilPssDaryVtsjKlHMA4PkGKP3AAUnLMJQgAn0Kf4EUli
  U5KBBn0b0cn++PtAjHZ755KCCKARISY+/TEhsOZp1lSRS4iRB7kZCswsBDAxyeBZF1CzNtsA
  vXyqFUwKklwERjcxBtUxe9CoY1ORFwIX0GwfgDJ6KAzYU2uha3EBn3cCHrAoR12eiKZ8t0ky
  CwVRBkDOEdQheNhRGcABLyEoMUHU9dGj19Y6oWsNPpO5IKjPZbPPIsgleqyKK9V2fByyKTg9
  34XQqJhJDFsK1fg0AwABendMXJwgZhTPYYx2RKq7luCtYBdTWeUkawymRHFGU+G7KTQUENWj
  6tQnF8AXa8wAx4CD10jcZeFdDQu80yAAGrBXx8TGBQDPNnWLmISBUgFTqN0w6NL+AA0gwX8k
  cd4qXJDdAnAYWwp9EPdU1TRnnBJ0agwHIg9kcvybBQGQX/koPo02Eb6GMbVpTcl8AB5wkKFS
  n2OQdvFyh/3oF7s8tmQLzEqQOarwsLizAceczYkwElZhALOqyhUnAbFJElYBckl5H1akhRZi
  iScGdRjgP9CnDI4yDfsiPpfCAXPtF93p4h4ilOIDApfKBanD4suloK7tM8FSI3bHOxw623Wp
  rXrsSf7Xdg0YFyh0TD8mXmWTo7VbjcUoSt+lAaSLfhU9z2NGZmcu3f89FGtEE+WtX1ruTozx
  HKwp0+AdvFp1X0JQTDfxAfsgCZjUQUMy09mA3MiF4/n+ErHGJBbiILp8HBbgy8fiNUr/3TCv
  kaoukz5LUckiox+XGm9Lm1bGysmWMVAIqc2MHCoWYtL285sJd4aXcgKgg4x02B8bkKgRgMOZ
  YYgH3pgB4AAX4MB5iBJ4napHKjUcozdwMAJR2GgFdTpi6hdq1WjTXMKdUTSnkChKXh6/hkFx
  WpuuuhkepWj6gahqgu/5glY3MQLCYhJQJypyarWiMiE3EQKwSuY/fZhpjnrHRZdOoQ4WHUsd
  NHjaqqJCuly0C8VV8AaGvOeeaVLaixYr1UvI3WMfsFJqpDpINQusfeZAKD9aEArFiG9TYUxa
  XhYmvQcT4JoNm+lJEGExKAT+OSg+H7C9DIBb41sBbumKmUSlKnhYfbONvgum2TBD4VqibGxT
  5RgB2nkFJwEUrqGgscHUwJDgCv5+t3ILhxu/l4VFBxP4hvIf9wsi9/QfRL7XWGOGcShiE/k5
  KuCIhZIBBnUB/5EZeVPoiPXjnRNZ+5hqEBzBvSxCtGwFGjIx7YEEcySfY+M+BB+IWRpGGVDJ
  qgwC2T5H1DIeE8VABFCTVjQCi+JpuebhkpIxirZwEuAqMIQMrBf5oZoMgOv5GjF6IGHxKmBo
  t04SiKeg6+3OVVA0JJX0MSVvuzLbsi7nYZbybXGMY1H6wlUAwaLnvktSXgqNIb8OoeB2I3X1
  mj/+BFp6bkBwmUAgDAYCUSgQCIOBIBANAKiCww+b1W652YCAUQAEuljmgKNSkTiayQQBEGCh
  AQ0KRfpcGE5AGWsKSUor4AJvjSFgoAIvjwTyw00pqmDCbYCqDAAisgLBaYDhQoOD4wO1bcJg
  VKOioo8McPbHwNbgwIGCw8M0I4OCYsLhoVjBoElgQIm2eTYKAYWlhSXlgolA4uGAYmRFLSU8
  PEREpKMDJCSkpOTkpATkt+NETWXD4M/ZOSBtxd+/XsB64Tx4yKBuA4UDEM512IDhQD59E38Q
  WKSg2wg1/1SIMxHC4QZgwShoc6DgAJM6AQxQ5ALAAAYKBOZ0IWDLRAv+aipKdPilgAqVA+6o
  rYCXAYMtAmUECLjgAahNDyyoqhiRQUIGEwPFqTDxiwKGDuwyOGAiKwsVBSDUmOBAoRiFDCXE
  1e2ZYUOJFeFAPEDrMoutAgggRHhVYUKEIwgMKFEyIEqVmoD3BShwAZWrAlEYRFDWucIFUhpE
  v4lwuggS1UnOLIKgoVQcyl0AIKjAAdIjVY4vkOBjRHWoKQJOa/gwQeJsWlAEIHgz2lQb0Ygj
  pEaiJNnKKUEnUwZgSVGULpU+qGHjCtZ2KkZI4GHDp+mYMtAgZHo5ARGJCMsqtM9j6oNTOBAN
  gUouiSCKpbSgggDcPvhkmQIgwCzA6ASMoID+US44rcDuJqLJlgcyGCGFFaahyp8UTChBnRBE
  KumBY1QSoCXlusAHAA5Y8IeFEQyIQgIMJJDghB3DMSGcE0wAYYMmm+xAHRPcgSceCSzoIJwS
  LAhAQRvNYAsgFVIc4S0KTFhhhDcmCGZEJjE4Rh6RHPjLSy+4dGAuMf/h8RtwlnToF2CGfOCk
  AwxVSYouXaIpA4No8vAHARRQQAWdvjEhJAkIYHACNXTiyU1DIW1qgKeugfQHfji6SkR6Oqrr
  yK/kAqEnCQz9C5sM6DnBA7iG1PXVJN3BFK9dT9hAATonEqCKAiIwTg9UJLmgguqMWKwAJ9TT
  pE4tfgzg2QD5CAD+CcWcFW000qZL7LTFVmMtmUWMqAACLrut7bbcIFFFNQ1IqGAAAhzDDrIx
  kDDuAmW7/YEKchm4BLaII14XNWybcEI7KYLi1qUoLqlXvC2gIUGF/9CDgFkqSj3EvQH7yERh
  zuAIeUEGICn5giUiwE2PUwIsxZQLEIACAVKQSI5hACKAROiAT7vA51QA5ACCc+mtGgBFJ1oE
  CQo2IFGgFNwZAaQmM7DAJENteUJhGwEIRoUdqTrhgmBMEKkEPZNMsmyRnESnBBNWJGcDtDNo
  KIOIFiZAoz3/ofs0bzxAACPRPFCyAwweIMABB5z0a2EtLCJAgQnwAgH1dagMKdBgtCn+ZgGU
  DsWmKUKUA2CCdDI4awvcJ1hBp2qWbNICfLjMgKoW9vrooUmR/sKpgqLqQgAKNlLhhAweoIAt
  cOo6IUlgMkAdg85pSmvSEDoy6Nc8v3cnhRIe2oAedpLSWh8HghoAgQgmCA1d1LGWapYAGSls
  K3QMQwLUggYB5yQGAhMQjcRI8wo1JcZdRwBOEtjGJQIggVn3GgXPIiEddH3gXx8swrUQYBEq
  LHAPbauT8QQwmHddy11JIBjGMhaZjQHAULOpDWyIMIXeKQ0FJdODujAUFASgwhHvqU8mkNYw
  BtDLPlwQmCnwwAEGfJCB0dkX0DhQgc2AQQgTYElanIAbDTD+oCkaKgUKlwggoRUgNBMwwqZm
  wzUEeA1sARHbCcjGOrQRSm0GYFsCGQY3ubGAbnbDm97ExjfW/Y1WgmORCAqXFcQprluM05Pj
  IBcByVGOApbDnOY457kmgY6Royvd6VLHDtSxDiwwgp3sUkK7lbgtdyHYnUV69wbgUUNFTCqc
  8QiAvGksz0Vvmp4XmvIUqCAtC9W7Xva2171g7U1s4iOf+SYDgPStDytZcZ843AE++WGAfjwp
  wf0ooz+V9e9/6KIWu6zDGgNqLCgJrE3RonMBB5omghOUGLUQoybFqEaD2OogE0CITe+MEBIl
  VJdoUFgBFeKwhduBYcIYGZgx1PD+XQgAKQF3+IQebksoV/AOAohYryoiUYnT2pAYXgjFlhnU
  DzcNwBXrY9EtcqCLXySohcYYHTN+YRRvWCMWANBGErwxjhM8RSSAtock5HGPFtWHIgNmgMpl
  gAOBal1JtIESZNAkCgMoKe4cFyaOfCMFwfIToNS6gXRIaWwg8IlY1OETv+CPMgZo3F2/wSsP
  nMkDB5jAY5EUv3jMxBZ42QAsS/otGk2qc52b1GgTiQylzKiHiHWJAAjQgY/8gpiBCMAvpqE8
  7AnWsFFQgAmSd6loFgMK1BTNCEaAgdhywayCM1IH4KKrE+A1r7DKHgYOt4EhHWAyAQgG+K5S
  kgsQV6/+g3SHMA8SjnNIAFUfssgilJDS6yyhCQDdmAyFWDSo/Uyf+qRgBat1moohgQGFQQ0H
  2dYUG9rLS0O9Lx1/tlEUTiBD1SGCAzdDBSVghqQl1YJ2SKWtpnxYGS/14XwJ8IAEC+A2RDQi
  VaPAMpOJJgIFGgMEfLpEoFJRImMYarXE0DZD9AYFx1HCa4wDG55JImIYWoQEc5bj00BiP+RK
  F4ONbIoIEEBDMQaFWF2iABAZCsy4eOuM4mqABWiYYQYoEorqESaBYK+dSnIRUjBQ58ONQEp+
  EuxDxIKlr/CxTozLa5vfLKbIdiMcHTEBJxXihOpyFs2pYoItwjxmlTyqdlD+CEqCBEqBsV3F
  IjhS7Ah0spNkCnYmAXhApSyFvbsMSdQtJq5b9DcLAfxCTF7RngTOEbeuwEpWZsvAAhREI9Sp
  KCES4B5XOhJn+eEFSdkrS3oBYwAofPC9F3MpQL+A5towsGf7Ffe6IAjSwRTGoUe4mBQMIEEI
  cHlrrUCFRvWJmwlgeYWpqXBVGVFG+gq0qoSpAGn0eASCffiAI5aDQBkANA3YNGVKuEPL1JWY
  TRGgAvMe4403xiwBEEEIMgYEAIzAM/gM4A2mgA2DkbwhyNiGXvloAhc1UCA8dpWrZDSjJYQQ
  BjFQGzA0OQBpb4EMhC8i0gwjwFSqwtiuXE9FHyn+7C/qXOevgW/QHembkLLijgvA+0PzSHSf
  3ryCyX030clEigKaogCRTDPSAeicWPD8p18om1CTArMvbRc6GmFpbMiC6wHCUWqAxA8eGzBL
  Bky0o70oiZN1PkAUaHQBJWEq1bae1Jm+UdztUSBvewuWO63eJIXMIQASEAf7vAYrdkbdJ3Ph
  SzBuksCswthi8ZKPt2lKx3DvlxT5FULF3NVe/x1Gj/5MVAQ29O99XLGrqVDXK+zdHAkDpwAp
  c8IEOICgpC/oO1fkGXoe+pijJ3wKqu0jiqcGYZEiAKm6mT4cREEhekPgLJsmAPNFA0eOadEJ
  MEMPPsEybi5ifK/lFOH+4yLofDgBDz5gil7DZ7hqNAik/gxqM1ZscRzgcgTHA0QCGGJE787H
  21SFGlhgIPRkWLpiSkrAHESi6qpuA7ZiI/AKtx5iUGYwBEDJRgQgL9yh0LoCKrQCHAhpsJJl
  EX6BByNtZUIgBagCRY7EbGRiUHjpLJwPMN5m0FSkA2xFABrAUwzPe47C004Qr95hzrTHIhRA
  I1KAuBIH7CZvtnItBbqwGICl2bZQHJhLs94ERDbgSEIQWH6tkjAl9pwwfjKgARoACyliEXam
  gQxuhzSQkbJGADMKv4Rvn/7nDVYoh7JFQpisNB6qMVZCFLbMS2rjEixE5fQLq/qHCAiIirb+
  743ALnQs7GF4ZkA2xIF0KBReauEEiqZKYYlA5g3aI6NMCPeYD0AyEUP2DctEsajA7oX8hQ2i
  zDIgZo6SETbM6DssoV5oYuLMiH/uCxPVxRWQgH/ySMZ2b2G0CQpdbc6CIUasDc3wAXmCBwW9
  R9FWREpYRB1uieowAG1+BUu+ofFW4ATIwSG4zgLgSQQkwBabQbtohS4Eok+YCyMei51AAikc
  4MMmQFMm0kYE5hfyBgp3RA3gAVBkAkYcYHaaoqRohAYVTZgUwJlqy/BSRGzExPEGgkVCAgMW
  wAE0gieaBL3Uz04OoAPq4ViGRAIAEbpeT2we4hwSAhkcABGZS9n+1Mf13Cn2xidLrAsbGMky
  Mk5cyu8x2IYkF6XfpEX6+OuCUAM1+gk7muCDRNGCBij7DkhgKNE7BgACBs7IJgZd6E9Cru86
  UqYpLkE2vk+2JCRf3kMIrMMxeCjh3HFhouA0ooYDhgAVlCha2mD6DIOMSJPg1NEyQGNDlEwz
  bcIidgaFsIpZ9g86NE46PENmxOCJYGNoLGGrKBDGQIF/mA8WcCyWNqDxUBANfYIej2EAZKoS
  HeCY4pEO3wF1UicElEnYkAIujABPtuIn/cQjq24sfEIi6yQAFABKQgB8AqJPvqJYsE5s3ORN
  agclgivporNJziQl90JFXAQESUICCiX+JWpnrjLgzVTkCXWSzQRnBNKBuNIhBPBsWLazsDhA
  7OQnGOCOFgBAAtRHDU7gHGRCKr8pWEpAs5grJR4gcGSl9b6pkkjUSUS0Di1ADkuKXIwDFTbE
  4M6ALTVMaRyOgtKIXaxlpQgovr7gXGCDoSSsCPqyzBxoqvqIXAwjNAzzMJUgwFaIgLZjN9uy
  JOnjEndxgHQoWzAmM8VUH/htFQWEjvIgE5EvMURxOtbFCNoL3bRsO9q03/RAaIaGS55l5Vgu
  HQ2kQHbmNJ6g4Qx1OjAkQgZzCDZFxxKIAA5AL6BwL1jkRUoCJpmgpCwCBHTiKgqCK97JzoSt
  6pDCAuACzCb+wCjjBkCjawRiz87AB1mCUTmQ4XCgBEna7PDaySMSjyCRUA5C6PvW8xye60SM
  JJn4qs48FSVmhE2d4SbYoa72JAXmzCV1Ke9GawE+7xeepEVGICFuRSm3oCnMxE8wxUSj6+nE
  Abeu4gAMgAJ6oUyIMOvYCbfYIbpC4E0acVE+7sh8FE1/kWYWp1Ta4GmCbxQVczUGRk11bB0n
  xjS6NAz+STBJI/tsBB+K0zA0cRSJ4BM1TccGo0qTDnr6p0xLwzLXsnYO6P8208L47zRH8UcF
  Jk0jxDEAbIMArDAylooE9tsE5BViQUJIAUDOcT8Y9YsMgxJuDoVarjJB0Qj8B2v+ZjaUMHU5
  N3UePbWXQjUARrUFStUDTpXPqEtV7YwgXdVQYLUenJVWbZW6cBUogM4ZePUcQuBXa/AM4edI
  iNUCjDU+IFNZ58FEzPBI9mwgpbWXCjeWDABbszVFuJVAi+FbJyVc2WRFy/VcJy9dN6x6aBD2
  3hVFg2VeE8de8ZVNtmJfk6RfL3JbA1bDaAICClYtd7YOYmlhNaBhN4qf+um9YHYK4so2LBaD
  SDZbpGBjcwZvaeFj+ydkRRaChLcvG8YyCqRGkrUpWJaOdnEIsAVmM0Zr1bNmtaqghi9n15Jn
  20ulflalgjZNG2YiijZAjhYykhbnuKppweBpd+4soYX+aofAajuDgAPKLCOgQasifjzSdWLk
  uG7RASpFBfTnDUZ0RQpHtIgBKjsYkZBhUs52H2W1TxaSbq2EuGaiWl8iKjdgHfoWPr1Hk8qm
  WBXOcF2pRP6zLWp1CqkwRmYngm8xGEqkLTwyBEOLGIphUAilAcCslybNULandXqJTWFCV+CV
  WA6nsupiRe7iFy4gJRxATaTYA/CMnZREsECAPhdSDZ/XO/aPwQz2Mp+gYCqxaN5IMIWvWhaj
  wIRD4XSMZSfmFaqjOpDgUbhGghhghRcke/vnFfSYOjboMWBK0yCTqrr3WZIRxrzUMS6mhxb5
  Qw3GfzpxeDuZCTo5FPw4rkL+QWAMgIN8yBE5IYxcxmqZrMpgA8I2Yxn4p0CgQYJakVpizGqJ
  wFy2I3QB49biJh6TqUmitR7dWL0YBwUpQABgdQSGxQWx4lZ07AuwYcwGQAFMYYH5pE8si6+0
  AfRGID0TLCbEQgTW4T3DpgUZV0jkcNMMl3tAoEHNEBxqlW5hxAqDeHGYwIWdswFGywGUDS9e
  eIZv8EW0YQGeeNJEa72OuXcOgAiLMB40qyAMQhgcAAHq1RZayAluwhYUgHLEOJUuYFy/JuvY
  +AF+xNvwCBPleIcqeaAJ4BIUAeYOYy97+YeCetO4NEuddJAnLGfjKkMKJMGqSk+PLzSQjxTN
  D6b+uMOSGcYJHkaAwddLD3bbZFbDIoN/frS9WLNIBci/9I19paB27pki5ID5JIZeruN90/o6
  tGXTNmYZ3tdLIWOvWciFytdLGsMDltNI7M6ZUUKu/M4KUqAFSIC1SmQF/mQKnScYq5pc0kBP
  TgQhD29JBKvOhuRKsseiP3TuTkcE2MHZWOQG6cwCNueHLLl61MEnNXVER8AD+GokRPCJBbZj
  1rMYiEEuCqJveQRFABR8LnQ7RaIgIxobaEKwaWFTYmKLMQcYJmC0GAC0RksBkPgkuFttJs2V
  u1su+Ha8NGfy4o6marqQ57itS3s5miM0NkOCLMg0qiMDqyAQhFpeHkb+aaVjkNslqY1XoLNw
  Ef5awBxK3/xYqKN7rphAqzWZFyX54MgXvt9aDgKGZc0xox4h3KZDqguDk/9pYwIzgh65MoXW
  CQaGxbHjLgNGTeuAO2zoMUq8pAagKHsrRU5gz5z5Uxe7W6LgALYCKi4AznziIavOUMTqR8Zg
  Ap7LRPbkzdBQsH7BAiyAA/yqA8K2B5viAaxkVauuJNiqg4cEJlNGvyFzAOYCJXUYewwRBAXl
  ASJ6dnzbEQMgF3bhV6Gw1Po8eGT0HeCBIa/ybBrA2vqUMmqzhat8GA6AATonlQpi1iaduD6w
  O+kRJuv1UaLzAaLVAWQazT7DGt8jd39xd4P+PBs/wTZa01qCllIBwAFupHj55znISC6RlMAM
  aDA2A1SVg1koL0JSqoC0o8Gz5qoZZsgGroRclsILCK5k1s7r18IGzvcQQQ2iyMYYSqr9q70L
  xsYN3IYErGroeK/RDYDSRUvLrYXaGnvlC5RpQbuesCga6yMakgo/NdoBoaSxBLfjx2y24QFe
  ycEZRgAKm0dsCyHZDA0F9CEsAAS+ongSVojkwIkr3uKB2JcGfmEKwACcy7BV0tVwSc717nHz
  vRmo4AHmQU/8nOWDp5JeZUoGBwTKwSeYWw5NPlXG4KSPwazuVXDyKsrJbkSx2ZZmXuQpwMuK
  97nf/eT3j4QG5KH+dEjbcLpOPkxSSZmDmuMIUkY6aaM5mIwVN6pavFTboAExMjAVsbfDzs/d
  G/zYf2TIoGbZW3ODBMPUZZbpR66GbgOFHCGJrt3Dw82n7bsu9S2wNb4MKBXYL8Y5jAwufQ8T
  Vc5J90nAk8Bkg5oJjby2PLtswKKtFKkSPa1kSDQYmpwpd0fiX4IC5H0ngz5Y7Q4pgsQqYDvv
  L3kMLrUBrjzJdR8qt+FR1qv2nUFSPCAcPr6ETSC3YdAljSEmcf4ZDCEhW14fqSIFPJoDtniQ
  VoRCdfsBTn020DwKVjevUtLxhL7ZBKeh98xJnKR8Dj/4AeEsfQ/qJbGAJjaBdiyYgVr+DtSx
  UmehNuIaCDScD4ej0VySEwiEwUAQBtJAoDC5MKi/Lbfr/XIB4jEgID2fqVQyGex+w7+EAAIR
  uXBIJKKy+UQUBBZICaitAcQlvgEMTAzp6aFIqkiiQH5oVERARHRGTIB2MjkBRgWMKS6SCSDg
  kVTCSkLuGSXZVoBOeEY8FcyxpQZ3ERh4rLAgr6yopJiEdGxkZFBIPCgcEBAIJwYYjJiUiFAI
  CCwYABxIKARwH6iwrLTItygfq9zfp6ScjGxg/FcLkQLDOUTb3AAgcOABhg0lTpwwcULfiRIW
  Q4SIJsHAKTEHP/4I4MCDvmPIkt3jB2JDtH8UqD1wcO0ANnb+IOGQu3Dy5LyeLEK8VFAogIAD
  DRp00LfPmQgRLf89ICrgZhgxrUakUGHyWL1l+EZwkNYBI4imZaGxjCYNgzpyBqnGmdNqFocL
  vAAFinJGQBmbcEMS4AQBAV9E5wb84kbgjpAhRI5cwKVrVJ1BA4YOQPBEC1wyAQho3vVJVOVA
  HVH9DUalzp08e5BcmPxHkGVDh1KD+exqViw9RSowGDCGyoACCDjtemKgthi/Hwk0H8BAwwfe
  vSFHwHuc04QKknc10YxAuEfcYAQ4MHGsBbxlzUZAk/YyJrap5n8AkMASAzEDW8TMYR8cAXDA
  kzzKIIMPPvo4488/GDyQgQojPFD+3l8BRDjCRFrVw+GC+6yEwQFr3OcGRwRQYMI9JiXIzHsd
  xCcfTDLRVN9nJWZDQQgjeNABCCUo0xN7HdB0Wl8BSFCRRBaxZIE1ClyTkDadAdCKBxpm1WFX
  KnjgwQQGkEMMlBiEUNZZMLZkgQVs0fRWiVyYMd0sSNwliF7kFGLhXwn5wlc2/60h4BsBQFDB
  EUV88BgSuogS3nLZGCKAL5xRNcYAhV5wxKFGIPFdLwOc9qYb0FXRSabVkVBLbJv8UZoptnXk
  HG5mTKCpppHIQkIShLHRXAEMMLFJeFBcFiuVdGTqGiSwpFqHkWMU8Ml3ngwLBYmiboGeeiyw
  pwwzJsD+p9Z8DtRXYn779fdfQkRxUyAy8yDIgoL5NJPRgxBKSKGeVGGYAZYceustvSeEOOIp
  2HZxYoor7kQvuGhKM82MM9k4pXk57tjjj0H6ROSInlGRZAlLltDkk1H+SaWV/2rZYZdfhmnA
  mGWKcGZ8GKjJ5gFuihqnBnPa1UtehJDTkXl8EuDnlH0BIKgbhBraWKKcrtqocr7MoYakWe8p
  xqUVZKppEZB5CgioqCHMBanRMoaqqrLV4SqosDb3Jq22HoGrJbpewGuvVQAr2LBR5FnGsa0I
  McsrlTSLwLNiRNtdKNS2mjXPJQpwwQk7CcxPjC+powCYb6KzwBxdtBEHAA7+pMAtvAh6hdJS
  IHSwlpMSYJUBObP6u+JWKkSUZYcprERBNpffN4cBFHig1U4nuVcv7SxFDHpMFNdkhnkGcC9B
  1gRkwHFPK+gDggQHyCzBPxmEYJFFtHcQFRvcXxjAFdJwEALAXnnLEgQGJO0z2ZAACMoEowN2
  gC0xWQC5dpY2/CzmZ3vARNBKQ7Sh7ItKdatKBr8AACtILVGPUQJ4ZgOpU5jhbMjbxhgigCgR
  8gEXEQDbqnpRuAf+ADqLSZZj9JAo2MjQD3VYziCGQrcO3kQAxenTADqRh1ylqhYREETcNOOE
  wIWnAGoYg9MO0py4QUBZekuVd7ITKyn8KheiGUz+KcizQrhcRgAPGME7oOcVE5hgJWsZF/rI
  MYD7bJCDcaDCAUzwLva0R2DQ+1YJpucPC0hAAuzLQMoulAEONM919OCAAupAAYtkhRkJ3Mg5
  sEUFFNHxXapcZEpKYEC07JECD1gAlGYSQFndBEoE4I8YDgACfbiOW5psQQqeAYJjFrBM4AiB
  CGAkImCkhhybwIYCOsA/3x2DIh6ggAMeQIF/bEAEzJzeATPgAL4Q5VpvOp50KvCIH1YQL4Qz
  4nCOZjjUIRFORIEAHqZWhLFFhlHCGpzlzGAaXHpRDAxIgmN+Q6zMeCc2T5hbPnHDCH6iSnE+
  pFoQr4iAR03BEEfi4qz+HOe1JGRUo1MLqECZMAqPqrCiqWhORxggxsXtDRJ1ecIVgSWaTpAi
  pjI9yGUYII1tPa9FnqOexCQgE+5d5o84xEk2PGAgVXoFYMCDSAnQ8kj1JTAqCP1IABRAAAWY
  YHyUJKQ0IFIRB2TDLT2DUgfqyB6fIBIe+ABHAQ8oLpgcBUo0iespoykOASjgl/WAnpDYw4yK
  uM99zUxLBtaBNrhQQYtNo8AJtLolBZkARsgkCzMRqJGdQTN5VYhWPy/BB120yjJF26IYujhV
  LgyFAbVq6NggMxmXUgYvRpQOBKRwAA2u5p9HmEARSzXDCfjtsub6YK1QFYtcEaEWmnipBYv+
  RdvhGG1PupCCC3u4LChSMAky3ASrPFqAwgw1EZ5pxU1xNYsYTmuNw3KjdDGLgAycAEHzEOYi
  X/Q5mAjWFLc9DwG6tJ71+ERgDHpIClypFpxJoK4bQO2sQCGhho0gYhiAUftMIAInwRU6dkMA
  S1LAorsOOK/ekp5fYxlJp9YSG0krRGoCIJAW23XAh5RHMpSiJLKgBRoWQO0bt7HFXXb2msvI
  6oJc6b5kwo+yGfiHZfvbGSsUAaeMo0VkNhE32c7tuwv2QiGccAXq+FNTfXACcP9wQiqwYjyZ
  OdxCp+Zbl+5iCb/wcmqkxE9lXVcWIqyFbEghCDR8d6QWuwkADKX+CTz8M6WuVZV6W+pRs6Wu
  UmOgr6Y1+hogeqelL3U0f5u8DVRCGF5Cll0ej5mmajiARgC0LQ7nYIx4dCvWiYxeRCbSSLRg
  OAP7oAClMKuiiVTE1mvBmTQ20IFmdiAcUHL1QaJAgWM6L5iNtSMzXPkMNP2VGrimUTZOeKEO
  yMvFBuqWXl20j4i4r0zjrNn0MADX+CoiUBkogfA+OzA8OiOZ504Lwy/8MYCngg7ujISC0GsE
  XDChk0OLo5rXnMMA6PbNjskuZFYF3OBCIWtqQAwV/CNqAUiN06qus2YhLgyOBCIC1LFObybI
  6PbOxjKE+C5t96RzvMlZvZTjKZ1JUUX+Vxmpa9Q9FAyzS7VONwo5J0f5e0MNFwF8kkXwOFBe
  T9LKi9wMJtagya49PocL1FGTQ15svI1ckclGA5IP2EAKdMfrgwTA2ls+X1zxJIB2K+QB6tty
  m0qUmQzACKmrJDIrm+FK06pFYtajkY4DuKcNBKyO7gltNF5iegrkYgKnjxjrLSAUm8OhOekJ
  pYf2waO6KCH1qTc9Blg/bQv8+549Zq1rJnGPnI5wu8MKBNFsA/uvg/wK/Vw03uyy9eUT1vkA
  cDmlGZEppTPBbGegTdwGLao5BAKjehuzz7UL9DNv3Lu0DW+lGAC2VMP2o4TTmhG1pjU0DJ3X
  gURz2F8S2Er+mQkN4gHg+BmHFX2atQQSVYDGBkQZY8XD8+QDZKEd9ejdk6BPFEzaAxEF3LmO
  sB2IwaXEMunRlkESWInVrJALACEe4nEP9xCDjh2AWalY8rQCS6hIrFFeMkTPMiFQjMjIjcWE
  rhGWJSnDe7DEBTwA+gBQDVJhFcpgf9SSAlgDdHDbTCWEsn0LfHTABaDepyzgAmKh4kkABawJ
  JDkARzQbVRTCcUwfs9ACR8GW0zHfXmzRKewODlGBbmFaotBC0i3KQAXXcoRUdBQG9w3gRT3B
  LcGKkRDFFDyfIqBfGu0cFPnQCAVUe9VBJ+GF0A3dGhRdZ9gBUNVBmkmFFNyJEfn+kXf1Smrd
  BJ4xwCeAAht13d9EGhsQRTY0oP51xN8JQ1ltgIYADD2wUgZWRJkcU4xggLrRR32EIMJQAQkO
  EyKND1cIDPDg0QisIFuoD7hUSBcqQkIcngHkYK49ACh8k4hFI7kUnjkGwwAYAKZhxYs9zxDy
  lV9pmYyADo7VyBJ+nbKBBXQRlkI4wI2t4emdXjVcwxUWQtEIH5UYAAfwAxSKDgAhBvccAAKw
  4wOIpALkmhKeQTao4wJEErng2awATnV9wJgpyidyV2W8Ij3RFCBGHw/B0AHawm+FhyiK3yFU
  AXlUIwuVQeHg2eE9XWUI1SUmwhxIgR3s3CvcF0d1VKv+VNGjNZ8phsqXDcZ7GYL8zeLfxGIA
  0iJI4JkBhCV8gZcSoYFenCEGRQfz0R9mKcAxRlmQsAiV3ZsrIRM0SmMDJc1R9kwAYKOQDBk9
  cGNWRQS4hGMkYQA5QqUHFaY6kmQ3uWPvTRs36Zhc3Yc94qO8rVKBNUM/ohtTAaS6cd5njpUw
  CIBBcgBCBpBCMqRDPuTaSaRUoJBFbR9GjoBG2mBHqiNIdpNIWkNJRqRcouQBqKRTGQwxFuNL
  wplM4qGwOF0bxRFO9oVOCmII/VDSkZBLPYFQ1kasDAKAHItSSgVoNOUgPM75pRBVWtdVkk1W
  NuUo7gU9HQI9LkIBtOVYcmf+WTbHWRZLr9SPALAlILglTcHlSZ5hKQ7HEqkQbiAWB4CARMSd
  EB6cRYwARhjQUwzmNSRNAUxVISxMV5Sg2HXIwBRbIzkIhk2mCTDba7KQmECJGlKAWISAREBE
  X7HJ8eDIPbIEMh4SBoYhOC4cB6jmalYDAw0WQcrhA8QGYSiEUagPZ97Llj6IiL0EKIhO4Uhn
  MPRSEpiVEnGPFr7Elvrelt3LlzqAU2ZDjcQhv3wGAsAk43gifsLfBYlUbY0p9OFpsixab5Xc
  epFndmKNCqWQGDjiR/hie4JG4GARq6QnocHFHGSDHeyG4gDUtGTnEI3ios6WZ1QmnGQGOpUq
  gbL+qmekQU5+XSFYooMqUf9NYqS4ol545Vf+BXpg0kNwCIHJi4tA1o4E5ucgoa4d3gNBigSo
  yP6cIMe0aLmNDL712z+0IORJQJ0OoMzIhOK9xCV5gESEkj5c3gZYAPf4pyI8XpdgRdyVXT58
  44empmpGY0DiGsVI6U2gJGIowDdxppuqycAS7Jqsz1pkQOZxEwDdSI9VAWF8xkKEK+tFA1NV
  bJtWm5vKEsMe3ql6wSlxKiHujSc22lbuoa2exh+eH3SwFnhyCmxgHSJ+Wl6YAnipp6jR1MoV
  hxU5IHmalMeOigBx6iMgX/KFghA9HW0I3Wydxl2CBEWagVuA163+aVn+GpRwqGwSoVOB9l+r
  ziKenWTNCuBNDEBeFtCzbqh7QAQ4mNu5wdK9RpIHyuADiUF6BEyEDVsTOs9WQVtXuSnO6N0G
  mMC22ug5oBA5PGoYNKcDTEDEbIAHhAC0RUQBycc58YWJuGXijsp/FcG7liZKQASIwg8sJWz1
  5GZr7iulBcAB7GjmqcmNFWzsPkjONAT1sImNOpnWpENDwEjpcun6/K6WOq4/PAAcAq0+GccM
  IcpGdQqf9inTflfWTlc7bYpywax3YO9APaB2Fs78mMgq0I8H+WLO5qrS2qWXcU/mCoOmgkbb
  4NQdHiIo5qed7Gcvrut59AXK3upQ4IlUQC/+nuxsHPUqfNmv1fYnL1aicbilhToAhv7LiizI
  Y3ro6NaYS8QtfcytNVKBhNSDT7RHveltglDEPsDovehdUhDuoKBIElCAExhAq73lnEIJBJge
  C7ZgTDBsWR4eARSAAoACXKlBVP4XjGiIPpod8ADm6DaciGkeYNlSu+HuTB0ABqzEhUHSmvRu
  wtZO7UBefNROQ0RMWaDriJiHKaZD6TJVlzIx8PrevYhF73oPpgLeZxjALUbUoTJKTZos4ZTi
  FklvGa8GYyCdLZRZJ4RCT2mlIlatGThqbtABKDAAc3jtJFNyX2QGsBRARcbFfOpW4thnIb+f
  ydIGGnCn0z5H/sL+peHVaixahlwixiDIZSAMkRZVktZ6zbPEii8e3n6GFKzs7M/erxuwwo4y
  1D/90wVIgy1kABniJjUsgEjmWiex3bIizPYZgDUJ2IB1sMF1o1uBY4ximH6EwAueRwF4AArg
  A0SAxTRMQJyyHfqw3QzumACpo0d+ZA0egAO8RBK4q0Tk0QVYrmFywQAcwCVxwDdsaIFFBDiS
  k5YprEPma5Sezp4YgH60aTRkqDMIRDEthbn+0ggQ3Gk2wwa8HiCXwe5SLMJm22hhGYzUTBKD
  qCuJwDQURDA/jSXzcDDCHykaHtUecNMEKqSSA6cS8i2AAsYFy11UkQFUkSSSQXE0chf+FE0E
  kAAlpMpyRfLJDigl66wSybIdg01jYIFUYGIK/UqhCAHJwUaj9dR70i+kHVGhLaXhXUZOy7J4
  5CcgVFHgjMfx5hAX9ieeLSViuCL5nZldU5F4xJRLjlQf+vRj+1FPC3Sh0YT+OM82YlVXTJmL
  +DMzPQVASNIGOECdOsEAGENpqm0zHpPbTkMUHgB5eOW/XtJYwE8BjUwoLda3hEBlrYMCwAFo
  YNqV7GVWLQVgYkQRMpyX5uYCDCSkWBQBRIhaiEBkkauLzUs+4ENjmkAGHMDlTlc3KJ4F8KgB
  PcTaKoW96YOLPFZF1M758OqFVHJ8E+gfW9RUIgcugofWNUH+tSRtXvTipT6qGEBAdViCP2lX
  EGURH8YKJVbiK9fBLXYHoRptJMcXqSSwcdw30vpC2GbDXPa09o3tKRMHGuo0ItdZT/msUQJS
  dLAiXCZ2Uz4gnYkCA7zwgrvkY+N4jmtN2qBDQZfAPdgR9CQSkG82RSyTU4BzJLleQXwBFVBA
  VtidW0muke1DAiUAmPCFoLiFATRutboVOKxtZFkEtm3AS3DEb2cGTSCAAkAA4/6wm2+CctaS
  TMhEFiIA21lh4c2WPZ1VJCXsMVmElCvFoBP6CXiAmZqyb5ZB4nlTtZHYan8DuIzApH8DpXdJ
  5XZ3bU2XfHM6yGALI+wsIsO4Vo7+4lZC2iwmjRj4dhjUgZyt1ycQMlJ/Svdq+VuiUR1wR1gD
  kVJPysHEBYFWbadXsj2tnGE/Xag++NMF0M3eR2FErR/lNBHtNaUCCxb5AY1372R73LZvcjfo
  c5sqs4wA747eq+kh4QWLJGrFYYpdejNPbBsvrB/+nZ/oc8BOwxoyZDXc2De9xDnBAQwLe8DL
  MaT2xQ16k70n7BJjOgHjEAHTcQ6SZDvqngO4VDfVOfoQAklx+8Z3m9cY9h4ukdLqBTDaZQGX
  gXQBABMsQWkYtnsxn5jihK3/cls/lK1yGxcKfM4P/Ku5ashzZf2mEF2OgbZLHSO+comjnAXh
  ZBRzfNP+D0PLfbvvhbvEjPuWQXTconsUwioXsLsHNLFDYmw8cgRQvwG9v2P14DtDquEa3qu/
  vwHA63ynf/fDH7zvNZzCkgvDP5DD3yCUaObEV/z1RGTGa7rTG/5ve3xig7z5bjhi2DUrWi3K
  q7wuKu2DO9rLD2PMO+itO6Cd9bE6vQHOx73AJ0/Pmy8AzpaEkgHRa5DR5/ReJz1ekKUmH37t
  49PoxzcYEECuYYE80+AU2iCkNE0qqGqu7tiOqXLfk6hAI78qO//z0zX0S//ziwr3NO0p0SDi
  UWQELpjlgIxgUy3IBLXtN/2uRq1j6/iwsxkdmI3+HknXFn4iNOiRBLsOx8GbG1oh93wUU1vh
  ENXg/gOBQTgkGhBF4dHwYzadT2jUKQBUrVds9iqQdr3fJoEQyAbMAYFgcDZjzQRwXD6n1+13
  vB0gpmr91gC4urG/wiqyvETFxSgDwr/AJcZJykrLy8UkJKVNpM4ipahDMkOtQLvH0j9M1tYf
  BAIB0kI0Qddb3FzdXbo9W148AgBg4mLjY+Tk4j1lOuFm6GjpriAAOw==
}

set pieceImageData(Eboard,80) {
  R0lGODlhwANQAPcAAAQCBKSCPKSmpFRCHMyubFRSVISCdOzWpIRmLPwCBDQmDNza1LyaVKya
  dERCPOTChCQmLHRuXJx2NKSKXMy+pHRWJPzmtMSyhEw2DHxqRKSejHRydPTu1IySjBQSDLSO
  RNzWxMTKxMSmZGRWPJR+VFxGLJSOfCQmHOzOlLyypNS2bJRuLLymdGxuZKySXOzu7LyiZNzG
  lDQ2LHxWNNy6hBQWFKSGTGRiXISGhPTitIx2RNzKpDwyJJxyRKSilGxGLOzatHxmPDQqFOzm
  1HReLPTqzIRuRGxaPHx2ZDw6PKyupJSKdOTWrKyihFRKPGxWLHx+dPz23LSSVOzevNTS1FxO
  NNS2fKyWbOzGlGRiZAwODLSKPMyydISGfIRmNDwmDMyiVLyidKR+PJyObLy+vLyylJSWlCQa
  DLyWRMzSzExGNDwuHLy6tLSqhKySZPz+/LSebMy6lBweFEQ6JKSinKyurOTezOS2fFxKHPTa
  pOTi5LSeXOTKhIR2VJR2PJSGZPzyvIx+bFQ+LFRSTGRaTExOTJyanPzy5OTm5Jx+ROTSrHx+
  hCwWBDwqHEw6FPT29MzGrBwGBHRmTMzKzMyqbFxaVCwuLHReNPzyzGxOLHR2bBweHCwiFMy2
  jEQyFCwqJLS2rIxuNLymfGxybPTy9Dw6NJyGVJR6THReRFxOROzixOTKnNze3KyefERGRJyW
  jLSSTNTOxNSqZKSOfOzSnEw2JEw+PPz65FxKNMSibFQ+JKSKZPTmvJyelNTGnNTKrHxuTKyK
  REQmFLSifPTChGRWRAwOFNyydLyWTLy6vCweFMSeXIRmTKyGPIyGdIxqLHxyXPzy1JSWjJyS
  fLSWXGRmXIyKhKymlDwuFGxaLPTivNy6fOTKlMTCvMS2lCweDLSWZNS+lOy6fOzq5JRyNMSq
  fHRaJHR2dFxKLNy+hKSKTPTuzDw+PGRmZIRqNNSubPTWpMSiZKyilIRmPIR2ZLSupOzWrHRW
  LIR+dIyGfLSurGRKHJx2PHRybIRuTGxmXCH5BAEAAAkALAAAAADAA1AABwj/ABMIHEiwoMGD
  CBMqXMiwocOHECNKnEixosWLGDNq3Mixo8ePIEOKHEmypMmTKFOqXMmypcuXGQed4CGkw8E0
  PD6VegSz58l2xjbV6CMxQo1Nxlr4XMq0qVOTqTg5mSPNIJ1aTk6oecp1Yz45QmvIkIjv6FE5
  n7qqXcu2balIALQAiKR0YAcZMiIxAmBJT0tXDrJU+lelHUIZqSrdKFApVSpXkF0VgpzEFUJX
  pRxodlDK8skkljipUaNFSyWIgwAIySbohCXPbWPLxghqzOyIliKd8ADAwwaCUGpoqfHJgyWY
  WWQMqgZlVKEbqQxivpGl+g1CBQpp315o7MFKMlzh/01SKol3ktVS1QDwydUmvhA1AajhYI1c
  AGtu69/PUF+lV/wxFAEjfYzxRwQAnEBQJQDIoQkcqW3FkhaehBKICwOcVxAAN0yzi4Gn6ODF
  iF5kc4kXCgCAEIWC8CAMbzWcVIAwNZCwSic6pAgRAElcAAQNRzDCSAFNKdZOC/1oQo+STNLT
  wpItRElkgPxZ44E1VCZ0gxYGQHKDK1ocN5ADm0ABBSijABCdSyew50oBBZwgDCMGzXdDOzfc
  UIkt7JQigyWWlGLJewi1yRsAiCJqElxyFCAAG11YAkAVDuUDwCbWhGDIIMUQylQliuVJXZ5Z
  iCpqAXpmud85ADihKkKM6P/SSRzeTOCBhu3wJQAt/azXUiOovINCHokMgpAHAliQh7K0bPOO
  Cu9QIsI7Ihyh4EE8hGJANZfgooyGIwEwwgzl5PCAFBgYA5EW7WgTzSpiCKKLikwliuh99uab
  6Ku3GQJAKfwWpEUp3UTRQRaWJEHQZALUwYoZbb7kwBmVtKAJFK78gIFBNRRiCB2G4NDBKC20
  s0E7KLfjCr0GOSBHKbuVdgbLI70HgSbdUPEKmJE4NAcAMgjwAhkRtCAHzT3pq7S+ActmBgCn
  NT2QEKlQ4AsF03wjpkD9/FuGL0jU4EFLJwQRTzoo9LCmQVq8woQivqwSgxUsyEJJPOCEscc9
  chT/mgEJN4wwhxxbk6RFBBFcMEUMMOyzyUMCeBAICKo8cEoJumjRVCQesIMLOiV8ngk66DxR
  BeiClOBACZpLzZYAAMDWtDKuQLKAJnEqPNAgrtQhABXWyDG2S6V44MoN/YxSiiBfcJwEDmZY
  g8M5o1STJ2OMrYxQErfKIZcWvJn0ngw4hBCLNGAOz5AasQswRDLt9BMxU3KdYP/9J2ySv/72
  i42062Sri0HkA66ARcIaUShCER4ghMIZAACEAII2pLCG/53kE6c4xrASUYwVtSEaRVAFEOBB
  CyzQQBzHoIQV3oGAxx3kE+qIRx9QIQjjnMQDolBcEeyxjRUUMCEAqAUX/4owhQe4gBwIUJ9P
  tLCJDuzgArmAwQeQQUVYoGELYgjGBXwxAQsCUCWluAFCupAgAHIIEqAowBy0wA6CpKIQbKhD
  N6rhK5fIQAuDuMEoRvEJ5nHMFh3ogBk60AVNtKNUgqlEFgrhRZgVYhP3CV9J2jQIUAwhFgLI
  wtEc4r0N2CEKFOhCF2TgxZYgyhKaWIQBltAFKIhSlB2wxg1mVsquzIIRZxiOFtjwlJVBwSD4
  iAQPapEW1zViEDF4QAyu4AEIEKQFABBEHFBgqxix5ARGuAM3iOWqg2iBDlGIBge0sQpa3CEO
  VqAEF1JYAhca5ASmYAACTOeBRpwEAKYwRSe4Qf8DSiCgmA1pjQ244A1RJAIBRKjlShjRiGFE
  Yx1FWAU1TLEHcJjCBgwQQTo4AAQSKFQtHgDfR1fCoH4YZBaREM4ZXHeCGqCiAMLj0YJOQJ0W
  zAwmlsBjCyJwAx74sSAncEIv6DDILuDpVKBihxf/NIhcxkWSJHmPGnAAClDgoBBic4gt5NAC
  UFBBAHmSVFMQFQkosIENmCgCFRawgBAugAyMlItsiieHE8wBHSMYgSs+IQdO4HEptkAUPQpC
  Dw+YIhH2NKMaIqAJRmooC72pggtWloxf6eAYWIBHItZWEA84lIj2gMcq7nCMOFCCEilEhzsL
  sgkbwCAIPxDEGQoXrhL/lCARInCBKeagrod4oBZBOIUfssGDCjZFC2ughj2iEIV1pIMaWwhG
  M2AQgwPYowjDYNBsijGcGpzhDJvYhBzO4AHh5KcnKwOAJAqSDw/ArHVS6w1v5GKLgkABAJyY
  30tK0SjkVUMGuhCCQeRQiA4IoKjnKFU7GJOFAigVIRA4QSFyKReojmQ9Z3DFTivxjZEORAab
  OIcm9uSKTTLFA96z3w2ogAkO3OINt1BCP6qBKFK2RRB/GoE5oNGLC1ygFWYABjrUUIszOMAn
  K8ulABPQAhmUgQYjANgXL6Umg+iBDgDo8CTYdNltJqIQCDmDAKLAAQ5MgRYxEAcXVniMclDi
  /x6rJUgjTmGKSzzBCbNFCaJqMQNdNCISfYMIBDygC3Q0QlHHXcMEuAAEVSgwHS7YAizSAQRe
  qCIcLSiGh9GrBfB2xhXsYEdOTsCJkAb6JQxCVNQEgg85fEkLPADg96BmkA3EBdEvYYcHBjEK
  PPGgBLX4ozTMEL3pVaNUi2mw9g4igxOwg7yIsjBJCgA0mMUFIuxrRLPlcDRQNEXCpfheMUKw
  jkeMQwDhLg2qNt0SNdDHGReggS+mMAUg0MIXXJCCbp+ADdm5JBUOYBBsWmCJFMBjD9oG4HsQ
  teSBdGNfLmnEZWNgD1NwliAeaANERUgLbqg5Wsdg57UMsgYdmMJEeP+mbUlIebSRO+QVvGlT
  fZ+ihUZQ1Ao02AEHmJCOePgICHlYxTlGoAN2u+SRrhhEJc7RnDwNItT1OZrKV0JtCMRuIF3w
  jSYkVYpBoIIQlSAEKorRwQCt5wRawIdB8qGFNsGXeDUowD8MKYMSeMIgm2BHIAc5PVJVogCC
  YcfbCUImXUfS6Bfpw714Q6eHaLo0ifL2t7sACvespxoLeIEhSBmXLnRDPmshAwCOcAFFFGEd
  mIBoEXIgNy7UbQIzUICxkMMJUUCDphU7wRriYI90nCJ2esrTkfSkmErgQD9uB0DDBfJwXF/z
  FGiDhxRqEQEDGCAQogxEAbTQAAXao94xuIP/Ok8rrST2IxDOoIc1oIAnThjBBtl4gi5utZIk
  tP3UDjn0n9vIFYbaYA/xsAfvcF0ipA0WQAECIA0tcAqnMHhccQObMAhdsAg4IA3SQIHVkx2F
  UAqlwAMy9RKsAh5QUwmjYAkeMAodUAnroS9y4QB5snS3cTTes3xQoAVH44ArAWJ3ImL8tWv9
  YDEzpgWuIEiDRD3IlkgFACaVoAlMGAjnEAjtUANysAZO1RuIdxF0kCvfAxErUz+GIABPcQI4
  UAQHliYeUAmjxBd6ZAdDwCprUQlqwAJMgAnMFQ3iNAT1dgDpQANcQA0kkAGRYBgvQW1wYAFL
  8AXYIBdzwAXaEAPg/1ALjKAFClACmVAFguAJcJEoOIAPWNIWGqAFxvAJYdINBDEJTSAfcqF2
  LfEJAfAAqwAP5TAIHvANjKAAMrAGwuMAFxBC9GYPWHAMVrBO0SILNtAmuIRitFQKQYAEdiZb
  U0cS/aAFMzAXEKE/86AFv8QVyBUADBAPd5MO0RAFt8BRV9ACSAAMUpAIV5gSaVINHQAyDSMA
  AtAL1jAKepKE4uGBM9cSuXIDXeAA9rIJo2AGymMMIbUJMuAAk1FB9oIDONCJa9EB0QYAJlUQ
  ODAfvAGRLZEEjdICzfFILdVS4+UBEbgI1iA9XWA91PF32cMJiXIoiVIDYnMo0lYSIdAbcv/1
  EKXAF0LYFZtgAiCgDeMQAjcwkZiSBmlwCDkQCOtoEucQCW5wAKrAXGUWjgqkCvbADb7Qcy6g
  AyPQlBtBB5agBjvwAFxAAKegDJwwAQewCnvwAzMwA6eABfYABEwQBy7AD6GACrhASzigD2tB
  B1ogA4EwDVviCaSYAGmAC3qhC0Hgkl3ACizRCH8QA7SQB4rABtcAB1dwDdfgAx2gAUogQVOg
  CkyAZufEBSKAWpSwB0YQBONyBKaTV0FgBBEgmzUEUCiRCrpAAmdgTQ4BAINwBUKwfEuBXP93
  WiIQAxywQ4oABGFwCiRwClvwe2wBAIVwDnRQVWwQAiHgOyHTDtb/UwAOIAOf4IGrthL6YJ5G
  5SfFsQntYAbnUGLfdQKSMRkywAMc2Cf3cQ51kAZd8TSEA0lgVhAQyA4lCAA2YUebcAMtwHQb
  QD1dYA0dsAjncKEbMIGvdFSLoUiFQJ6CsgmfwAnctgnfBT7fU5MkkSvWBhGWkBT81RUnYA2U
  MwQgcA6lIReNYA2xEAKHYAcPpBZmIARcQAvh2Jzh+FDRoA3wEAMx0E+uWQVK1BIXqQ/acFpg
  gA5ngAo7wA0iUImpYASMVgTRoIcMEADB4AbUgAtHA5YksQD41QtAYAHeoFSUkgD+IBdyEACy
  YA68QQfXhAOKsArXJY63cAviyFzhpAr0/6YNOYBmKUQA8SACBHA3eyAC4OAGaGADRqADp+AP
  OvAMRPAENfRDI8EMjBAB+IANvRWcpyAFCnB8XAEAjTAB4LAHe8BorBcHMVAEvtAGDZABH2Cd
  a6EFBdAFSkAG3ckKadANdRA9owB4DiYenwAAv+ES/tIC/wAZJtigZtAFrhBSncYJ5Mptn1AI
  ihEloogoJtAV81EA7XCL8PFhWqBHZBA8booRpbAJldAOUGANZnANj+Kd33lg0YMPr5RgflcJ
  gzAI+CgoliAEjdBS3+UBNJmvEjEfoMZulvAJ1jAIpTGlPgEB1jAFdsAKdXBokFev8sgLwIOx
  IEEKXfAFLAAEHP9gh+uQpOKEldzgpK3plR4AqCAIAPUABFxACQyADh4gCb5AC/HwA+NiBCpw
  AFNQBIqwDS7QDFLgBlLApqUxDV3xcJ/QAb4ABBcAkCMgEPQgFwqAW0fAG70wmV2wA7QABKen
  qFR5euNUmli5A+EXLatJACIAA3tADRNgChOQCDqgA52qA0FgDqRKfylBD2cQAfkgBMDZEFoQ
  CsQ5WFwRCYoGgDDABfagCrQQB+GQAxTQBq3gB8FArGoBNV0gAMlABiFABWlABnVAj/a4GJCR
  nwCQBS9hDQAgnqBmgkPyra4wk8TBCZ8QXmvADg1bDQUgiqVhAFwBCvMxCAgjivwnEO7/Aa+g
  0AXvAYYtsa8keA5dYAaG0At1QAbKqgR0IEg40AX12wXnYD14wpIFwDuYwQOhEWEVa7FPBbMP
  MQ3zUQgACRGf4LEgO7FhiANTEAtfJSncVho3QAe9UARpcJFdMQpn8Ac7EI7rUIdkSmaq0JZo
  hrTBgAo84AFx6xJPowY5AA/wQAkz4AEncA3wQAChkA1eEAr8AAu5EAOrkE7vcAdAsA4xcDlt
  kp5Lob08YAZTYGm+sAQQYE0yoADmoANAcGaJ4IGymhInAH2EWsLi+AYwhqgJxKj0Zm9ppk6C
  Gw90TA17IAUT0JV9sLg64A6iSqp5lhKV4AmiUAZq4HILMYVS/wADR0AXn/sJFJUOtLADbWkI
  OmADD0CmU9AJpgC7syoDWWAGApAGuIu7uguamlAN/dC/nMFX6uUSsPMJ+Ih2A9MPXZAFn/Bd
  dWWiFusBn4AXCmwJM/HEToFl3moNhySL10JKg2AxbEAG5AsAMbwS6OuvIlMHddCd8KsE7TtI
  CPYPiMQYDFsA6OAA/8sDJyAEJyA8vTyRKXGRa4An+rUQh1YJvVAAISWyPSGGaQAFT8cevdAN
  N8A5J1AKbNAN99UVN8AIMKAI4lQEdbgOZTZOogUP6VC4LkAEunBAL0EKT+MMceALqNAIyCUK
  jZgLwRAANmAKwWAKMNCa0rJRj3ClYv9QAY1AkpvQTT2hBDVnCtoQTuMUA8GwP3LwB7MiTtqw
  A+BgH6qIEo0wDzEAD6XLXDCGt8ylDVhJQuHHBWuWQqdFuC4wARPgqZ0KqiVSAVWgC1qjEuIR
  B3FwBHKQWAwBAPtAA1bgBcJJElBwAniRn1K2EAw1ASLADQcAD4ogDe4wDwFADavQnNrQCp6s
  ELhQCmrQGk1NEv/SDh2Qzd3QDWTg2b5jDRtQDe3wd+U5otgJE/jhXXKxCa7QDziQHPlzAp+w
  zuFFXhYrhQVdMsUTXn/9EoJpGu7bC6IEBYUAFgsXcDdgBp0NBZCkkSgBYpUQoetrCD4ACvDL
  BtxMbPVrDfn/27vX83eDoBmCkp/8s84We3gpcV8OgAOj4IEOIZM4QAez5HxLcQLNsXCNYAiP
  8AYhUAn3UQrWAHpckRxtYA83G07RMAQSHUKqkAcokA53wAAukNElcCUwwQaXoga4oG03MAbb
  wKjbAAbLsAyU8KTxwACXmqtTsHMiwACy8AElICQD0xO9sAke4AwogMKWBgTpIAbYAACM0AB5
  YLPaMAV6SA3FUQzGGRIyYAPpwA12i7ePEE51SG8HQAsHEH53cJYiMKlfvgeHCwshogNmPSKQ
  W6oqkQRJAALcoAPG1RDK5wtW4A9nGBKPsbyMQItnICSRYQtVcRC+nAhYQAto5gZ9/5AIE7AH
  6sAC28AN6+ALJlBK+hBqSaBrT2WDACchIgEADlANAFsHoCDqVSUAhmBUd5KEtmCebXJxK3Ev
  pQEBDrABUIAD7eAA6PwJ7LCBSZAT+oPj5HUCBZBg1XpKMNEL61EIdKC7AmANi9AFzrEXiCIH
  +nkOhvDM5OsBC5oSScCvyAM9SvAoyjrqvTBI6ydKvTYqxNe/v1tcz5s/I0nAVpgSrCIJAgCu
  HiYHMtALAlAJ0g4SN8DmSeAEDhBqBl8K03wQcrBViHICG/wGj/AIaXADYJEg1LYQpaDA7FDw
  7CADUOfqHNEOgtAGKKAKqrDgRVBmQ0BEQLADK1QOTdDJ2f/AOtC9EqIn5DVABN6wAEPABC1+
  AEzABJV2ekwwWoO7B2FgBVbgAnFQBPBADcxgBKTkbyrxBrnSD74A0dEgQm+cDlKgpRHwChr3
  xg9AAyRgdQBgGyZRConwDg8g9CVsh4gajlGgDd53AAfwAOkgDl1OCTBw9BWeuIsLqmaOAAhw
  BOhQZM8oEpbgAHbABOqgtBGguW1AC1xgA18A8hnBSJHAxXTmBn+wC6eADo6QIpyQBdhbEJEg
  BA0ABCzQlV7wfzAAA3ksBeAQA3mwBBZkDZVQrVrwBY5gBH7gqacQAa6AdoMABf8AEk8TcKPQ
  Bb1gCF8oj9GD6ouBnyegDFf3Er3/8QkJ08z40AH4cP0KiSrZARnaIR5Z7AFnwAP5OV4QUOMt
  MQbVWgg48L7zaA34a48buB6d9g0AUUhThzpszmmRYyLBQoYNHT6EGPHhnE03KrXoYk2AAFBs
  6tQR0MuMGWuLrHUZZfHGjQIsCxRy5cqSDB6cTmw6IcfDTi0AAJwBIFHo0KEbABhg06GAh0FE
  W2gZtLGaHJ9ErTpsdwOAlhMyCrErVUrGphpaKt5Q+FBLTydoOUQ5dOvWm26GcGwCcCLoQ0OV
  bjgwaymsjFKfjPkEUOjq4obtPMW7A0SbKg4ciqgqog0IvHRcRISZ5kLHJV1arDFGvbgSAGU9
  +MV48+ZW/5EiUSpHi1Y5SpRoqyjFgyHFhhg/q25FCyeLko5skdilhi6xzic1cYDYLjJFu6op
  B97ZGCBDBqoYku3soCFKkhyg0aHzIEHASoxV2nbvxsQBUxEgReyhwAKLGNJ55xgCKIFhD3Co
  mcAUU07JwJ0g3KEwiHuqwKUWD85wzz128MgMhmz0SqaoNVDIg4tgviigw4is0UoNI+CI4YAc
  eOFFG1piKMeUCEqJxDQcGuJqiQuQkASVZ2ChRoomTWmQARb82GuhaXrBwadS+gjDl1VooQWF
  Aw4A4Zc2UGFESxcjMgOAUgqp5hxrrOlAGjs7QKmalV5ywBII9HJlTRez0MKBAv8GaWEDKDro
  AIdRsmpppSxaqqSASgtgx5Iz1vLgBAccqCQVAJIQFLpvtrKki4JAMQPPLlC6YVJ2jOmJqwJG
  6QIUMqrRIhIPSl3MFg8K8WuUc+w0hA4BDGF0kQ4MgAKKlFbas4CXwnJAhk8+UUAOTs5gZKed
  tuoJ2Kug0EKAEMy4gaoqI9rqHDZ6uUGvd9e0RhOfBulAACXqSAYUUOowRJpzCqnBp3OGbMin
  M1rAoQMKQKkHlBRSEJgOV8algyEN6MDBAwA2KaTfjshgI2U6oMAUgBbNZaiFRibgwhcggMBM
  mymAsAeeGBBcUB1TjBhBEC0YhpmxQlg7xZ48HnkkmkP/irAsGqpVYQIIDnDbYRsurNgmHj9W
  oISXImJYJp4PRJjAA1KTJmrgGs5oQ2urt9NMGybSocSGEpRZAxVfVFGFFl/ucCGII7QwA26I
  tLiEGjjCuCCdVaYAQRsQaOHmgRgesIIFSnIRIZc9XJCiwVMSOeWUIIyYJ4gKdPkBnX2cKEEX
  QRrZRBPoUmHkC0awSW2QS/a7QIcSAOhmKAD8oQUIK9TBppLUFGDkEzmsdygJLcdIJ4do5CL/
  OHuAgESDPubwKZ86FnJYBkss+QQDR3LHo4QS8kdHEEeqhAJiSiGFcNgDE7e4TxGGwBteAOEC
  p1jDCSIhg6QZAgAycMWtzrFB/01A4RxQ0EQ1LFUJJziAHdoClOOusppCcDAjHWgVDqzhqH48
  6h/U0lMldPgSGUDgDDXgAUxitTQKqjAi+hBZRXrBBjKAhCQ46IImWkAtV9BKCx6QQQEG0o06
  sGMrhjDiQyBwwSS4glijUBQOoLjBFkBqEJYaRCoc4IoTiucEJ1iDTs4grrVwqieIQc0/IgGU
  qqCmEDJgAxWUMIpPiGwUQrmgNLohgEpwopCL+QQgHTIILeFAXWkIQQi6QQZRhoAMoBCANKph
  CZ9soA5UgJ9ParAJOcxyE2TByS0T5hMBLCRLPWFHF5QQAiqwYgFUQGYaqJCGJnaBkwD4RNKq
  4QEdiP8gHDSIQZjOh4IYxIEL8XCBKdTRDAuhgwcA6EAYh/KPvPSAF494wyM4sI7c2OY2WGNC
  ZdbBhFXsYBUHSEc8YAELGtAAGWKwASUUQYxERMIS6nRICHzCiTEwoQh3U4VkDsAEVWhjFTQg
  wCmYgYsTQMEeRdhROnJBDRecE6IJYIcHnoCEQMBCCuC4wAUKWg4YMOB0UghngxxkhAwEwQsI
  mEEFSnAPXJRAEIKohSB0UYta6KIRVJlDaowAgEagogoj+BVjqOAKYGACE9twQRUA0DFIhqGB
  XJCCJwLFGGgeIQjz+EJWGeKKrXBiAv3Bjn4wEYXB7gYz4TDFCI5gyX4k4IL/MymFIMKShLC4
  Qg2umGMxUCEThgCAfn3oxEl3Q0/evKW0VttGEH6gCwCoAWZt+oQtCkGtG7SjGu3AraUKoAYn
  2MISPMgJAOb6UocYxQEb6IUhzLAROohkJFCEoiZsG4EWtGBSlSiEtQrhgE/c5AxysEQhCpMX
  4iaADQDwgDFcgQM2hMAjI1lEF3BgANtmwS/sWANePFCKApxDAMlYBA9KAQBQEHc1WmjEJ2TA
  DpgMohoPttYgwJIEdoCFfsqoJS3Zs0c+IuaPiAEAKnDBGHYaYxOVGMQYSQyAQbCBmThwQsIU
  E5FuuKy9AqiGvRjTEwcMohKf0AJDIjAyGVijvWlA//IyU5YyMqBsYNaoRBb0UggqdEN+88Ny
  lucHASzLr3EBpN85QqAHO4BgEiirw8A8wkQyKMEEDjjnJmCGDw9UgRp7iEcn4nCBODzgGJRo
  w+l0YAQdZOAexXDEJ87A1vI2ZDU9yEM8H4GJdeiGarWJQnaoFoWOckdnnaEEnmERgGZ8AAva
  iMEHvNAp4i7gDHs0gj0yXZnJMIEWO6BF4VYRAxrswQ2mmGordsaNdIhiDwyYgRZKBNE51KAE
  5NiFC/awhzuLoBXycMEEUjcB0RSaCESoQiZwUYwRYGi1VdXFVHXBCZtgWA7/SE0jZJCCPBzg
  AggIq1XWgI0z7CIaj8DCHv9WoABLnCCdEBGAB7wRjRzc4QMVOMEXarEYD9ABCyI4HTbWwBB2
  nEEXqLhOFNaBNW5wYxUooIUiaDNYlIpABDpwgMiKiBq+aqEYjKLFdraWHcngjBem5UAOKCEC
  EmARZgJAyIKz6xccVuqNhUhCtnLigUi0oNGN0cIoNPKvlIVEJPhYBBRfdY42/gNSk7JWOwrg
  ik9sQlwK7i4AakBcRmghJhsARQjSMC8zvCq+UGzHDmFSip7IoRSVUBUo5rSJIBP3KZ+Qty3m
  yI4M3mAQdKQJTUrBA5zoRFwe4BRPfCIyEF/RCalJRUWgoIduFCBhi2mBy2IxBTZYoxp4AUAq
  4OX/AShMIg0CaIcMtrIYYQi3DmmoCysX4ooTbDAEC0Cykw3RC5Ek919NrgP1cSB88qYmC1sp
  BB3okAZVgKAbqIRiDasBMQE0uYkbUDsA5AAzHJzBCNJW0DIUxABu7+IU7kCASyCEKrAqLXif
  q2MIWwAAI8iBR1iHyTCtaBgCbZiMKZgMVYgG29AZnckocTgGDxQoG9iCYLAmLnABFzgFRpAz
  iJoEABiBEbiAShuCSluH7lgFbrCHKeCGAhEHLvizU6gFdBAFbVCEnwEOF0AHAAAjiFqaUigG
  HeA2U1AQatuFLkBBQjMCIniCChiBYkCHEmiqdPMEqdKFCBKGE7iqhIGC/+jYhH84ACBAAREQ
  A05YjDXgBGFoBbk4gHgIhhKgijV0iHOQvyrIg2gAAi5AhnlYgzOYOaI4gQvwBXCYgET4AghY
  CFwAgBYQAArYDQ4onG3IhQTZAxgQASvggynIwGgYEy7wB6MpBejgq3lgAXiAhymwjDzwmW3o
  hG0oB0rIJnugDA7ghXQQB0oIj6RJhrygPFeQLac7lJeIujkouO8CgMZCwATgq0owg174CCVA
  pV6gg76TL7+Lohb4h0ephMCzFh2qBDrqrhOYG06wBFdIgnxTp7xogWqQBvdjA+USu0UQADbo
  gphwiUE4AZH5BFcYBTPQFTLQikbrgrwQjzkyof/sGoRs4YHMo4lb8rzPE72tADGQtIXokIkO
  sIYF0AdXKJer0IpRID8yMIMNOMhRgRcHEABQEoAWsATQazyicCRimoReUD4c0IIWSJllooJ+
  hJU3MqMb0DpD0JWPEAAoEL5oSo3VSAU24A5eSINkoINFEKE+kZ83aYFeGCU2MANDOAdaei1G
  mAcXAAdqm7Y7axDRCIJn8IJUGIECLLBrTADvYcDZ0JlUpA3t4EALzEAOoEDtKAImOIZyuINj
  EKhgoMx4+JkTTMGeDKMFAIAqqIILyI9Kq4wiWIUbVIQp+Jk78MBeJIFaKIEL0AxfuIB4iIcJ
  wERGCyMvYodisIFf07b/aduDP8ABUyi0QUOALTyCYqiC2jm3dIuqWrAJYWiEM9CLDYiOGgCG
  VbCHGBCBAKDDq/gEdmuAW3gERYiHLfBDAADEhog9D8AFeIgCIKAEZPCCNVAGHiC+OIhEF7CB
  SlwIBWwBqDwE7PgoBBlFUjyGU8tAVTgAWiCAVtQC10qNpbkEFqAFeNAGy7CHHeADGvAmK1CB
  bJKM0UyHO6AEXXhFmKkxHoiJCjOjNxoEH4uJzauFnKAKfPBLvroBa6ADUPgXfQgJZlEjKIqv
  LjiHUfiHfLQW3aoUliiMW3q1MygFVyC8e1GhvLiBaugF9yMDkRC7LjAEMhjI2TqUQlgDkVmD
  /0JYyPZKhocsL3yAps1bRsp7oz7JSB74LeA6AQ5Lr8/7MJD0iZ5oRMaQiZFgBVAohMNYDK1o
  ARCYAjIwBEYSmbd5COG6yUkAvp0UmasQmd6DPkOwhEhIAEFsB49Isn50FBEqgB67gYFgyFOq
  AyXAAe95qNRoBxZLgQq0AzIQgC6ohsujiTV4k1EI01MyBENYBLKAmXwAgEsQg4E6QaByAXU4
  BR3wAwQwByLABV2YAx6IhHrwS5hSgHSIgke4jIvijWhYB9rwj3rrD6s5TSA4gJ5JB157h/mU
  gHyFBUqgBMrhgiA4gcsLozRQgxqQoV/AhCGwjOzQtVVQuRiIBy74nP8BoYYjmIMs8IUDoAEu
  4Ng9eIIaWIO+VCdROYERyABY+AC43AMo4bZq1QHZyYYRwAVckCr/eSqpKoE1aAR28xa8sDo2
  vIFfoAUagAEdsMeh2JQSCIc34AAmoIQJqAKqsM5ABAAFIIF1iAJ74AJY4AfCOwGKM4UwSIR5
  MAJGsMQEsAQtaAUQAIFbsIxVsAKBIg4/oFtYEIFOQIFLK4IDgIEg4IHtQY0s8AB0CAOLIhx7
  OAAKuABZAAc4QAagmpw7wIIcyA5ce4AMEALuARZQiFMZsIWYAF3QZYc5EIupw4t28MsC0AIo
  4EYl2AjliiEoeJXZjSJNeBTa2pOV2K5SOAH/4KJOV2gBTRiEGkiF4XKcASsEGTKEbjCl7JOG
  IrWGOpiELvgES/CxQykFkXGbGwDTbkiGDqgBSn0pOL2gsOABW2CHy8JIAbvTTYAATtAJLQgX
  nvQwDxOZkYyOQjiBLoACOtgAGVhJqygAALAGVnhUMziHv6VJiACASugGKoDJUZCBcbkKn7iB
  VxAAM2gBOQgy1ZUGKggBO2CmXjiHAuDdO/KATVgDUNEEjdgIJYhUvWgKxtiA/RIAO8gBbYgF
  NpAGTaiEPnnHP8miczCEEKgDGNq+r4WZdtCCbFiBREgEEmgdfkgEL/ACIvgBQcidNXCpcO0C
  LQgCTIinQ5gn3sCE/1QsV1WwB8Q9HyCghTEBE396gDt4h3jYg+EQg2ZggHeIAUXgBhEgAp9A
  BSMSgAuiAApYIKoZApxRhVVIhwdggnXohD8Ah2yiABR4h0QIDxIIB26wAi54hz3wA0EAAN8h
  rn5oLSs+BVNokgmYgECAhlCgkGyYgRnQHeesqtxpBJ2F33EBACTokEaoAWjYBRLQAdZajJ44
  glWIpxw4BinwAktCGoZYnkfkjQegBNFgraOFpBKYhxkgguXBzwToXUiwjdlggjjYA3LIVqMK
  AnNAAHIQAynYBgydjAOgBFQQBi2IgMUYBQAYAC74uRnMgRgIAziYAB1oHbo9BX9IBDhogv8d
  gAdVyIEdsQFPsNI1yZJGELCwKAUHEAvz/ZNexgvjbTQc8BRu3IiRaBQ1mt1zeBUc8KAPmpbb
  uq11XDt2yEgIYAROcIB2mBO+ymhzkb92oD51mQQ2EIBwVCOYrgMx3QkHgAmZ4ISeOAM1NUtR
  4qSrMwRrkL8EEw8WPaGZwFP5uQl4nJsrAr1ALT2fmLHoEBWwcAUZoApCrQEBoALaQ+BP0YLn
  eIgb0IJzKKZ5aQdNHWqIqIETcIWL1DF2OAFQgD5VCAEBGAUH+IZaIRdOOYFSaIdoWS4z+Ae9
  0FyiCCAI8AF6ioVYqAMcGAQZOGsFk4EkoJ/DA9N+bJRC2IRHgpn/YtACQbiEeTiCeQDuIMBi
  qZqDOcijSLBGv9SCGlg4UpAL2gABX4CEWHBUPYiGKciB/3jjVbgZeMgDf9o1LPCMUIMFBoCF
  CdgDSuACGrgDZOBiAlOhZDyHA8iDiyKc7NCObZAFK1C5C0iFDCiHXXsAGni5dIODoRUBBDGF
  SwCARWi0AeYBHhiBUzhBKBmDl5WEIIjZH8idqpoDqdLZnT0BThCZGjiHfFgTZQjUWrjRq+gJ
  LziAWziEKaCBeAiAOfAAkwZsDygBe3BbiGUAMRgAubsKQVwDHY8ErrqGhZCBGwABdY0GVeCG
  YQgAbPUDUyABG7BWI3gGd1CHeHgASAYC/26AgUuoBQA4hwougXIAgiFo5B2IAw0gARKABmBg
  Bmio1h5YnV3QAC6AB/4oc1gYgBWEmX/eBD0VsATbFg3jsMN+KbczAKjUh1Z5lmiRXXwYBWMx
  APmCAmc4h6ywiCzYkzdqx8Gw0U0IL7UDvSwwosQ4B9clAxB+LzPAASk6Bzpgg7uogQWjMG2h
  Ci0ohR0dmGRok4NrNFRGr97FILCY8DwNz5y4IznoyD/9oxsQFBBba6XZhDqwgykIASxBsZuo
  1As6PmYyg1EoBfaAdIdYi024bAAQ1QToMTIoM22oA2tIBdDzAAhgB6l2glKwBMbjimzJAkax
  hizAi+UWig7Ii/9esAPJ3ohzMMgfsoS22IAjvQE6UsguCAmSGIS6TposSHJBmIMfUIMMgSoe
  WIM1ULQTMOVwlYN2sAPy3A174AUWsLx20IQ+6AM3yCbvvlBaoMADuEHL+ZI4+OShC7UJaAZy
  kABqyGZB+AEP+Nmk0YNzEAZIYNd11ZkpoHIuYAEuAIJouABcAIY24LUYiIFjAAcvwIZ/8Iaf
  +SZqyAAtSIUFaLQNPgNB6IMx0DZX7gPqUpxw6x+oGoCpaoRGKPH31QklXhN5KN4k0IIzBYAu
  sAqR0QFeeINDAIKfQQaZBQAHaAhXkAMH6ANMkCdaMMFESML5Iwo4lYE54IRlBFcnB4b/DKUa
  1ASHIHBWBuBXfmWBMDCFUAgCG2CAZbiD6HkAF/iBNDeyhu/MPcCZKdiBMiiDDjgHf8iAFvAH
  U3gFOICDBgAH1LGBeMi5KbCDVdgDXYD9pBkFWsKJO8qJTXi1DakBCvZLNpCBQegFfwEIAR2g
  nGtXaFAlQpVu3Bi14VwXKIugtKjG8EaWGwUGDSpQyBXIUp84nTmzqRS7TScADErg8iXMmDJn
  xtTSwhqZEGkWhCCjpJchOtIWdRBAhk6LLKM8frR0wgOATa5GSTPDhs2JSjS3cu0q05omS1pO
  lAJZqJQlGaVkCPn06QQPTidOnPHgwZgHLVoA8NXi9a9LV5Jc//GF2vLvoE1k0qiaJMCExkIA
  /L60FvVGCCpU2PTa4MDDGQCAAUT6JGdsgZcOKoHgpW2KknYn9DrpYsaMoV7WoNwotcmuFks3
  bOOotAmAia50AMjBMclON0MtWrjaVOOTgxvnenGX1mXUIFcFbEvrgGPQiRqA17+Uc0YOJzny
  43Mi6YGRFhns968H8EnJLQFGwcEBtBiRF0k1aCEIDTkUUcQBB9gzxQExxJAODbRYGAcXBFDy
  4R7BSOCFGMtQIoUffqADgFb8eWXZPLREEQ0HRdQ4hSrarLJDOL4wUcQqvdSxQzpWHHPHHTAE
  gE4SBqRDCRfx7JHIHCy6eKVLC0AliP8mDcQDwx4TmHCKH6fogMAlR6DyhCC6uCmIII00Igx8
  jdSC5VYAlDKHHKJ5JYQWbkTzyCFAUgJDEJJwkgRMlvCgyTS3DHiAFVNWIAcPXkEBADtICOJn
  TKVAowoHHEQxRRzgHBHEBFZQEMcqEvoSDjg6+GFDALBYAc8Bx4hRwiehQUGTGQCo4cIB2qiy
  ShO9dODHP/MYYUo8LMQQThy+cCECNS5QEwMHONISzz6b4NkVD2PJsYl8dZ1Rg10eRELZuXhq
  8ckidQhgRgfnFJCKDB7I8RZZlYwCBRQ4dGHNdwxldMNCBUhcCDtJgCSDWyrJ4MpBhNVLEwCW
  tNALGzlREQL/KEoI0E0Ik6RBRslkwMwGFJVUkgQPQtQwmQxZdNGFAGy0AEALH19JWCmFFDBK
  Ox6B5MAcafEwErvvxZvXZACcYTRMk80FgCt/FZIEGXZMQYYhHbTTTiVamPtSO8x1QUUaL5sx
  igMlgcoVAB6U0sheWcBUyiCxqIJjHVmcJsMiIbDcTTdk7NuFbHp5cIIrmnTQAS5acMII35to
  wsbL0hSQRCnrsqOdNALoa4gPhuBQgAPjmYEDDucMIocHXLukRglnfLLGGm9x0ohp7ZbEiV2+
  u6jFKHbc8sgtmKxCUCnKFC9nCaYook0R0UyhLDyrWMEFJVZ0wkUcVlDyzvlGxgOL/xiUHAAE
  PHeIY4MQACDhfAJwAIBX5GEKUyiCKlSxjvGR6nDKGl8R1gGkOHRiD+ijBjDWMIhXqGBbe7AB
  LlgCwK2wQQFamMMcSECAB9jDHrSwgghgAKYJuMEFOjCCH4iQjR0WAx26qIUnNnEGvQjBeXUp
  wQh4AIB+cOUbjvgEHNYxvSLQIkqnOEUVtFAFl4TwEtQox/Q4sIoowQIBPFhDEbcytEK8IghD
  nEkpSLCOaKxjgeFARjaMwAJarIIWBzCgKuzBDS4EQwf+sAEsuACPGMTDHT9YiSZkgg/mcOIU
  2xjfDnJhpgx4wQ+m2MMx0rGD8tFCQ++IBzX2EIMHqYIWDP/YhxxGmAB2QeAEEPhGDeTjHjmc
  4Dce2JksAbOJQfRCCXXoQBcqcQI5GEM+djEGDypxDk0kzBC3a0c1HpaFShSAm+Fhhysc4ABL
  YCM9ZyiFz3BgAHYA4B8ABEAhRmEImWWmGwKgQx2uAgpQxExlAhCAvwbBA5U8BQCcqAQU+iEk
  HGihRcGcCTYmExyDnQMH1hgFQyRWgJBYwi1zecrVoFIYen1sE5EoRCE4AQBGbaURoLkBJHhx
  Nh+YYQP/qIYM+uaSQfTNEgLYSTdA0YtRVOc4XgFADT5SA5PGRAa2YMMCGNONG8CrEHSIKhWS
  cRWY7WsUpSjFbPymOU0EbG8xSQb/ALJAhm7UwQw3kIFKPuGKc0jDELCrA14FoBtC8GAQ5zDD
  Oc5B1N/USxAoFAQPBDECYLyiCa8wxBhsYIogGOEfPXxCBdBRAh4g7wSckMEaCvFQmLiNHd2I
  wvQAsQMonAB5eHiGm3RwAVbaAx62jdAdKHEMSsRDBFDigjjEQYMdrEIcXHCBKahhBXvwojVc
  uISVnNeFTbBBgaVaR42UBT7w5eg1RchDHuLQBji44HwiSAQ6bBEIEcRDSsjwAgBSM1pJuG0N
  JfDHBTqhiAReVxV56OMqaMACMO3BhkYwgg508Ix7jAAXJWhEa0HjTHx8DABrSEQwPuVOmpyh
  Fo24QjTe/3AL8W0jHi5wQQ8iqhcAfKMBtLDHG2aUjnfUMBTEiyVNbgAAB/jgCrow60tKAYx1
  zOhU4XCDO/xQjlU4kFfwoAUQtJGOCUzACGKQwjtocAcXoOITgHOoSzaA1CNwwR5UTIEJ3FGB
  e+jAFLLgRoH8GKEDYOEdBV7GHXgBhANwAxyOeFu9oIGNr7oCYqPQxCgwegOKsWMOPHCUDAaq
  BXhN+gwnGO1LtICpUQgAFGkbhQz0UooW2GIshbhBPxaxCBykjWGjqIZFHrZRV4ATnA4oBQ/m
  Aq9PZOE2AAXbO29gADrILA09YcNtNOGASFuVDF3opkdqzYlNQAA/ADhBFrLQgf9jauEGmHaJ
  ZTbBiexUYxQ42BwdaGqGn3VBExapxDcdwA4ZWEIIdJEDvOLVl72U4mPwwkioPQDomKxEDqMw
  3NkmJ1gH7EULULFEIaqRGSrYs195GyKQuyaVSlgCAAKQiQxOAAoqgEAbZNDEMgdhBp2cTK9K
  UFkdDPGzAmxCC+ccRQc2MBuSuqQLDudBL7TagUpgjAc8KMA5Oq1XOvSCDoZ4uTUGAfGFKTpg
  W8PSGkohBCHoAgG76MQqphANUswIgapAQTqOUY4PiQC5GcjGPnSxBmGcgRNfncNDVQoAa5iq
  CFFYxxU+sRdHBEAMFTDCKqIQBfytIga02AEfrSClPSz/Yw97iAcXykGDPtLiDu+gRK1WYAMu
  5AAIWEAGBmziPGt4AgvaKNXhbASkcChhF6cIAxCKEIeEaoIEZnJDJwgADlh44ROXKLCUXMAM
  DxxGlhhr7SVsQAOxDyga1o+GpFTxoBwc4AEEiKEbJoBcHYTCHea4xAhK4KZaxEdeBj2BIM6l
  NRuI4golqMEa1DATD+hiDp04xBscAiZoQwxQguXZADqIlAe4Qgrwgh2IWBGIgwrsAQOQQy3U
  AujIhBrwgAfYwjRcQz4oQ8YlQCNkgR0MSLikihfMAxdEWSAVyJPBQ22lQwy4QZnEwzGIQzyE
  wgAIQiQ0XwJAwXvUAjUwGRPE/8EYZIA5mAMCJII35EBtRciTrQIV0gADTMAehMMLPoApeIJ6
  4AnB8E4p3AAOBE1egYJAdME5VIPEcEQquEISyAC+idS1WcLAAdAa8MUgWIMAKIEZQEEh5EcS
  FEA74MxwqJo18Is1qFpEvNoNWMRGWUyt0RoE1Idp2MUaMMQ5QEGoeZvvsIgB+ECxHVvaZMFa
  QMAgGMUomMYnGB0EyEAlpsdeYI4rbJsAbILgjJYrpIccWEwL4EZQ2JUwps3CnEMLrI1GhIct
  TM1HlYTl5AXW8MVc3OF+GANLdMAidJx/zMRKnMAigMAUvAxNtdsoFAIdssTCsEIanIxQjIIt
  rEskjP9gAkRFAVBHn8yEJdQAHUxCCPBCGuBAKYxbF4AC5Hia6wiANDRd7AzdU9SAK+BALxQA
  O+RHTAiQFrRDTvTCWw2MxPyVMRnTPXGH61hDIcjAcFCOxKxE72AJc7CDLaCCDVzAKjhI+Gif
  NgCBPawCBRwDAVRLgYlBKHjBJeCCYZ2Rw8ljvWxKEAQBCAwIJtDRBUTAERxBA1hBIrjAMWgD
  jcRZrJRPbu0BNXTLHoDDHvhWKGFBDIjDMRxDPNjAM7gDNUAJJSwDKkSCJazeHNCCKtAIAkUQ
  BxDgK1TDIPyBmbWCK6hBENCDJhgA2KWDCMDCCqwBD0gCKozlB50BmAFQaMj/QQlIwh+EgRXQ
  AIakQzqg5QPQgBWwABewADjAwQTYAJkEgRd4QTZkgmYJQgmsQSPopmfxTp9sQjLUyw+KQqWM
  QCR4gGjFBAAQwg38wvQ8wjrkpJ3tASyYA+CMRRAowoyMmD3EwLYggxGUQAkAWeewxCuUwRWU
  wkrOhAyUgC8k0AKtAjggwBG4QAwczp6VDwzGCi1wQwP0QB9YECV8wAhUgS7IARO5hC14QAkc
  QTpMARN0QgPogDlkgCmYwjFA4X/xCnH1UZ1JgQ3sQdg9gDi8gx8Ig37gCV/0VCEsZiJujnmQ
  YztAjFm4QqQ1Ar7VhftJYzBxCjtsxz1JwzmcxSNu/1MltEM+EMXm3EYHGABBuFtFzOjTsMNH
  uMIgFMKtOYUQ2YVb1E4pRIXzpBXQxIzM/IQZGMA5bIC58dMqykDHFIITUMwaYE0NyFsX6Esj
  eOJDqQFpnAB2KA03aZRF3ADbbATHlII4yUAjQMBbuEcNPGpI6YWk8oUyZCCW9BIU0MEGoBQA
  pChMHIcMmMECgAAVkEFb4Y7urIRebMI51IESaIaxCcQ57GLNAVkluEIuoZoyAYAZzMQgeIAm
  KEE90I0A3MAnCEfrhCQ+sUFbmQHTOcsgMM8wXVQ1fEYgvAQOQMAmyMA8CUALWIKljUcXdEAv
  dBpXAYUP/JNXlYImtFs3qf/BpIWAirppIfQBA1jBTGoX+NjDAfRR+4iA5VGDKawAApwfLtSC
  +sXJXgSTACWCDmgDCkYBJsQBPhgBAtAAF1wlFpAKB9hDHvQnr4yRlMSDFGCh5ZXlO2yDL8SA
  2r1DLmzBM3gBFjIAmPRBJHiq0eBALTABf5UKBwxBqaxCF2RBAQQCEKhCKzQaAhjAql0AcYnA
  B7hDLfDACBzByQaAB8iXLG3mD9xAIFCgCrzDOxCAFYjDNsiCCogAAzAANYifGxgSOZhDBcjt
  D+iC+v2QXMgJfJxGyHyMFhRAGLzDMQQBuwAZAGQTJEyPpKAAy47lByCASgmMFyhCgDwCFcUA
  DcD/AIqgw6fIhERlQRN0wjR8wRey5wCEQxFMwRBowwFcQTZcgjo8QI5QiB89GXE9GRCIQoK5
  wed9QBAUgxMAQCOohgdUwSU8gDbkwQWMgR8gADmI3zFMARCgwMf2Kx/RAhZQAiw0gyoVARYc
  gyzwA0nI32Rsgi0QajVAwcLkg6tZBCSODRyyhXXEi3wcJVLKH8XYRi/4QGfQTjW0AMQgncJ0
  gCLeRrsZ4//+w4wOAjglAUrVTni4gkjwjl141sbk1F1+4ih0gSFEjsywgQCkDe5YQyLCzCic
  gAzYjMRwkyt8AtYsoCtAwT9hG6YVw4oKQVnImziB00awsAOABDtMTa7J/4VKWI3DXU1oNNyK
  NsI3nEsSaMEGyA43GVVMHEcpCEDqvgwo3E5EVMInrOgmdMFV1E3LCekuzobh8oXIQEEBrARN
  fEo//BOxtkMrFgAB8681KcEkcAYB04FAFAAn2FwhfEc/EEb8BQZzlEIdJIMh0PFY+Iz+Bo3M
  tFUvmIG+tIMMyID6jkIxeITNnUvbbMImYIMXkEAYdAIkPEAsrOwDxAEN0EA5SAksqIM6+IEX
  EAE6HCwP4IdetMAGyBIZOEAhlEEb7GWp/J0vXEMbDIMihMMFxAE3mFkRwMMOHAATMF6UgMOJ
  uYAUhKVvcYGFxMADmJcLhIIYgIOHaO48mIQ++P8ODpTADryez2pfqdhDGwTCKFzBj7SCGqQC
  ErRCEzRBOnRfLriAGMzBJ1RBFbjAHsCADjBCZjpPPpzDGnhOLaDDKVwoDbnBFYCDKUyAKWS0
  IRlBEBzBJWRWJtStmxDPGtzbe6xoKWyANVSYDOzBNpTDKaADOtTAsLzEOQBAH0AD5T6C5XZn
  OVieFLhDHnpAI5AAEMwIdAKBL1gBDFRgJuiCFvS0S0CBB8hbILTBBZjCPdLEJ2BAJ7xGNCyL
  CLjDJSTC9OWeIjxZC5USNdPClOmAOoQBJVCDF4zAD3TgS7jCGRzBPHADEKyCKEwAOaACCcAB
  HMQATsbKYfORh7bdBIj/AJTRAFuawxrUy9D0RV8VwA0cYzuQtgq/4VqsxTLxjv2uqCv88ggV
  widohDWkqzVtANFuQD9swHScgzXggBkkoiKi6ShkQTW0Q7ZpBEhYqUcot42utl0ogBzIADuU
  Al7QwSfexLKWDAhbk8KoYQdcxSqyQwEQrUa5AgfuRQ2wgwOcg8qcgJ4+lC8Txi/RmyU4iiWs
  RVp01KPVW0fFRy65BzQCh+WsKF8ogNGsxMhcFKgZQztQwUt0AFSwQx2M6svsC8P4S7rwxSfg
  AF6xwslIjjUMKS/tRXwnAFTIwVRQB1RsxSdUwj2to7fmWgFYg13plRkEDQjTwW3cBqjVwM1t
  /yIh6MlLZIGeFAJBRp0YyvS5PV3JeJoZiGQvVAIPtPdFXSm8UiN/SILNEN03fMERVIMR4IM/
  zIO0hII/GMERVAFm+VAttAVoMEIjvOGJ+04aPIVzjliNIJAdPAj2RYOfT8GEcJ49qAIQ2DW3
  uEAATIA6gDQDsIAViBItKMIBcIgILIMLBEM8wE9q2sB4Xp3RdEAJKAIQ7DkCgQ8C2YMvUEA4
  AMEQNMAcSEITrAIK3A8t0MA7eDQuCINkMjQM2EDWYlovuMIbugInEE8pDMCe1AJLs7QweBYn
  JA+E4ZsyyEteONxepMJG+Q59BQHjriYchFAXQM6m8AA4gMMUvAH1FP8B625LDE3AE+TFHHCB
  4k2RjtwBt/BDNhBBIwBAT28KO7xCL2RLGFRADdT5S+TDGvzBHcTAgACBFZhCNjwBODzAfUZD
  DqAATkZhgQBBOriADYRoPJwCOczDN+ifS8gAOsCCG0zvBUzAmfjDNewIhNACPAh6v1LhKtyB
  R0sBDRyAIpyIC3gCzmIJAN+qDERC3clAElgCO9x3artFfcxhNJLGR2y782nBOWwOSN6ONWwO
  7pwb7lgTAW/OiB/aRYh2MrIDO3QE+lrEWRBMXezaavwqz52LTUAkXrFBN7CBEhhCIn43DrAB
  KLQDD5xaN1UC0bqCJfCtFhSPDNMBfH9bYN//qhYcp6VBPQTw0rqUxLpQMIFfzV4YuAx0k++c
  Qw2Ugo0HxTnk1CDIzHLYRTWwAd2UKk0tQkTcgCtAhRY4gDTsk074BL9sQFFlTfMxxwkUgmAV
  Qp+EDU0oEw5AQTKkARt0QQu/uHn8/eYIQDKEQMw5y+1kQcAgJxSABb69BL39TBqAQhbUnAzg
  3G0ISU7UAR0QsCHIAx20/idUg8JkAa19KUBoSTCQYEGDBxHKwyXDw5o12GpFXNOoVqMTyjbV
  kOOBoxYZqXDJQziSZMkErDZpGQYoWstoqrRNUcWBAxCbU3Cqkklr1aodD1Y9KLdHio0t6gIE
  kwIjTgxuimAeWMVF/8Qeas2CEcWiiAaJGjVMhjVorRatA0BwyiySVlVbbUWKtBIUpNyOA/am
  2HtwLJ6NSzzkbAoGA0YiRpXEJhaLCECjRnNKCHpSBU8VXTwcfuLRiFMjZcrknJGjZbQWAAC0
  cHSgmHWCTRhEwfMVLtyeKgAYmdaii0K4IW8eHYqizR63O5RU7DFywgOecLceRV83BV46LuAS
  rXA3xwMAjgBKaXjQtEwgDydMutJl6koRDrxWsfDzxM+ePTmi2NthNu8BswcOSMcUHRjYwwVw
  TvnCFoK0cGGbdPLwZg8dEAhCigfWmoKnHIDoaZUDdhAnHhcmEOFDWq6YgARhZGhNrFJK0f+C
  EwcKKcSJUmT4ZJMTTpAjNI66A8CSE85x0UVLjDmnFwEEIIMNOszoIEozqIxSyivNWKQLA0Zp
  p50bbihgkEIKKIQdHtgp4IZq2omghWrKlIGTEzYBUoFBurDGFYGMHEmLFnAwQ4k62HBSgF7M
  6AIHKBQltAVlXAmzgAIqGcQVGXJDLbBReulFjiz6DHWgGgCQo5QCRung0C7aqaQAV3AUAoLQ
  fgTSAy1MO800UUsqAIANevGhF0OgKOU0XWXgYRQyqEgjjSfNsKaLc778hLRqeqkDFDK4FUCa
  Ds4pZE7TgjzthCzaOaeLQSw5raQ9BxlEADZCMGMUhtYAUxoBOuj/wJA6yOgGFDYM6beLGyA4
  r5pzzmEHrAQcAOAcUEBJA4c9T7jBmn0N2ZaMOgo2g45OcXDgEwc0YbUAGRpJKRVeEUpGxhKq
  QMeJTASZY45a5lgjNNGAhFkxUv+Y4pYoosjBvXWKUCUaDuBZhRYgYpqJOm6woIESLiiJZ4IJ
  1AkmmC2CASccbsxiAiZV7NmmKqsCQAABHbjIg4IbHhY6AWswsIIWeKbmYC1tCE8LCFXicmWD
  C/7G64AYuHAhGD948EAOXQ70B4BC9E6MCi2qICQRMC6QGgUuksPODyOyQQUddEqoJWfHGpGD
  kRq6Q69zg2zxQIcHpFolHEqC4OGEMzz4/4GWPKIB7pZ1pgMinTuOocSFKk44xeiWaIo6hjtE
  oEZsXTiqQZhBeqElhlUIUKcKLRYsKRVGSgiCFpqKWAWGFS7RwYhjpgAEs3xIFfA4QNT+doyk
  UGMPy3BBBr4wkCRggAv2yMMq2uACL2RgAlYQIDwMGDUD/u0BkHOBKfaABRAeIxGnqIIHJLG7
  gnCkEFCwBg4O9ipYfWJOx/NAJABgCBmOJCNQEIDHPuYtKklpStawRge6EEUpnmNNYBrEq9jh
  ijNZwgGVaMEooACFc4ziVaX4BJ2AVAMHjAIHhcib0LSQLmkESwB1qIMS6NCBRSxqFF3QVjXk
  IAMwUUpMrrCEtf9QcwJO3MAMOPBAO4YoFlKdwAE3MMC8ukGHMFKxEK6AAA8+QSvkAWmU3+HT
  ELMAHmsYgg5KEMC6PhEkB7DjHN1oFrd8kKUobqAartiEKzTxL4CxARRQAlcheFQuLZyBB+3I
  Uxfu1Z03HuQcHtiEDHAAimSAogMI8wAPeNACJlrDDADjFrQSVYBb2QJMN0AexOQgAD3EwhCV
  6JErlCQAJXiMYNZYBJUUdYMTWKIa0rpBKTYCgBYNERSMKMYwYsCHGFygASccEBFgF5k5cOKU
  kZyhxEBwi6NFASbaAII9nJY/9f3tLKrgCQ2o1zUDjS1swbBBVtKBlinwgia8sIcVRBD/Dxjs
  oRleQMATEiGOHexBGB3lFQ6EwAWlTk0nMpEJcUCIllbgIhFxOJwAVxEHSkwAFkZYA65OQII9
  zAMABfDoSBYgB0WxIB2AQBrSOCCVA3CBAdSYQDB0YFQEVEAXuhCELtagSADozqNJUIAVcoCT
  x3FBDD/4BCNmAAReHCI60FkHL7TBBFpggQCyMEUQaHCIka7FP9OjhBRO+IRGeIATfdgBdeIQ
  DjjMIxKuCEsEcrMGFrhHFUCwgg2yUYFsTCAGOMnBfuAxBeL4xx4o2EMPTAGOPSCDHKWYkxBO
  8YDnpsMFOsgG3aj2oQMEkBbthUcM4hEPcGSXBnmwxx32MIMZ/3zjZR4lzSBU1alzFLIUPJDB
  GiDAiU3IAQBvLUhKqkGHQiUjBGQAhQAaKcUoWiMfeRpFC9qhiQ+36opYdIUWjfeJAlRjA13Q
  EqtuUAlYgRN35XNAARwwms4BoBKjMIMhfGCIXtDBEIZIFA7zZORCBFJNYWKHAy61hu5ogUeM
  hAIAQOXgkZCKE66ohCYE0I0QCACHVJSxA2SQo03MqZQcGeWuPLoBLFOpF2wg2Dki5oE19cJZ
  hSIDhpEcxRsMohLnkIYP6ggKIWsJCoUA5xlO4wEZFKAFisLBKJKQu7BcuQaFMEPAlNAFB9SY
  Bxpb4hG7wa06MPEcMqCyDJygiT3F8v8GZGAFGe51zXbsC4nJMIS0yAmFE5+xAOoaRSoYbBp2
  RJIOWjDFFJAGiHUw4QF34MIeJqCDIBjhCbqYQyS0PBA1AOASQIDO0V4CE3tQlQnr28b6ULCN
  bViBBlx7h9f+GgwxrOAZz/CDC/bADbhoIxpRiEYRaGEFSlACBgwoqlHFoA6uVaDBesMBI6hB
  gHSIgxZp0QZMmqYNX+yhDyPIQAPSQQt7AGgVQHWBOvwgCA+Ipho6QAePw02QbnxCBrvYBSUO
  wIG7RgETiDtA9UQADsklQgcZOOoICuuYE1DZwa4AwC7QEpPJwuAvEyh4FM4dnUfABRMa2kEM
  FFGEo9FEJyv/FwfXwEENagTgBzwgwQ5gsg0YhCEYgsB5WCrhnUtYASZT2GsoZoAAL+wBBXBZ
  NzxWHrVVGDAGkttDfDXYiDOoARxYKM6EKuSCVWjjgKX3zw7iLQIXnNAFd4iaiMSws0gM4q1U
  bgcd6rCqMBnSwIBZMIPDLQgtDILXFLMjhqM0LSh0qUvtqEY1wPSlGJsJyjjCzBlPUIobtBgH
  +GDYKMJUiUKUAgKlXCY44Sw0ANjCmVUyQy+utCgc4MAALUDmJwrxJVdZih2W4ChqGEEOsKUF
  AOAGcu4gTmCxnCBdmqQbBMAEcICKCokdksUSLCEwzqDNUsNdPOrKZGADOEVb7mhd/wrBEMLM
  DlLtY3LvW6QFBzShGvpoWOhgBpsICiqBHdihTuQgTdoBCnCgHexv6gBgmhDCGrzjBEYB9/5s
  A0qhIzihAPIkSoSJDbqhDuDvYOrE9qqBBwBgE6wBZMyAHeTgEwbBGo6IWyyMDqRFS9phEHjE
  EgphFKzhHOwJV9rKo5KBEWAAE+5q2lDg7URgF07hFNyhAmJHDopBy1hBoXzhEN7gDURqOGRC
  FQCEFh7ACraBC6wAE7mAAIQKBuIB2ybABbYgERDgCCqhCi7BFGgACKJAcBAHLtSnHCjBgdRB
  AsQAFmRhGahBCsyh4oQGBwDACCjhGLhg8qwqJjjg41ZBOf/mQR3CwSf+JmoIAAbU4QN6YA4y
  0AOcYARqAQB86wBBAQB0gSi4AAi+DmnWAQiYYBW2gRIayBRc4BQyoOkQoBgOqxYA4wz4y8HY
  oRFooWmkix1X4QJyIQbADjoe8RbeAGlsot0egBvYEQXQRmo4BAgOoN625h1gwAT+wIPsIR36
  IhGMQAvCMTFwQAuaobl0Agi4ABbUQQz8gBoOAHFWbvJUbt3ygBaOwQYAzj6CYA5OoASWAQUs
  0RT6IAhsYBtMyij9A4HsAQtCEbaoIR1wghYooRnuoTtOMpK0wBKMyI7ogFoshWWwr/yAT8vA
  0RrogEmKrMiWCAr6oUuyYE3qclL/JkX8XKGTOsmMfIYj5IAHBmEDyGn+NIFaYiwLymQNSqkG
  OIEHbAECnEpUAEDFskCK3G/+oigC3RD/1AQvXQHKZOAMdIMjiu0GCvAAC+IGbgUOs6ADhonM
  ugAKBimLCgyUIIBONuJWUuNWgDGSNsAjZCCf6kgJpIEOuMUOFsBJQKYXvgXJNEETQoycpKE6
  pQEH+iUCq+GKeOCaIgUGN6AQBqERTMMSEiMZzEChcMAQCqUXqsEC0QpPjExQCMXCBKBg8KwG
  IqHTWsAWtIAd4E8TBMoDkiAsuWVgDOHFoKAaSoFHIMAB2qEf+uEGXO1YEGOIyCAVfiAdhE4d
  pwALjHFE/3TAHzIgG6qgFk4A3BysEjzABRzRDuIgDn7hF3SCtVbhDuJBCmChF8FhqJQOFmzg
  A9TBFIKhhbwAHdTAFgohAs6RF+ygJYrgAJjAJgLkGKjBFILUBqTACqbCFDLh72CmFxiB8yhB
  BLggBqZGut4CLlyqE2iABqRmgFYhHd5hD4IhALxAyjhCAaTMFZJBNUFBC9BBqGhgJsAOaaLh
  AC5Ia+zjhGwgEZgOAUbAMmphDTghobQsEkxBfRAnFnmBF5qGJjhAEcKKBsqBBdzABXYhEfxg
  EEnAH7ZNB/qABLLLDeJuD+AxHi5AEx+ABvbAhWbgG3GgNQBgDhJhFXBi3WggHf/AAbCkIAaK
  q/TSdOUOAAVyYQ8YIAxYoAcqNRv2oAxSoAF0QAdMgQWm1B5swoJOD3L2AAbeQQVakfSsABaI
  4AyKxMGqwQM8jWLIQJN2Lwk+4RNaZhNAA34cTBJqgFMMoQsO04k6wInmcE2kr5BOrJOSoBQs
  Ic0skBNqjCM2YY0CpZHkckLzUvx0JI0s4QZwoAUSUG/2UwYKAV36wdJwyAdnc2U7qRJ0tpO0
  aIdIM9I8oBQGQQYAAB9UkyC8wwMsAVWkwWNeaVpiDMcuhQcKLDBAw1Y64jcjyTu0oBQ0AQee
  iJVcCY8KZlpaQPwKQQ1qxAEcoBSgLF7gltCgL8ZO7GL/XaHJcOwTLKEGTKO/UBJ+cIAOLswa
  2MQB6uQECu2JmEQJksHO2IBfsuATNKLUXEEOXIFwYURhVKVQQMYQapaMGvQTQrPHzjZGTqMd
  zMCjjPASiMGukAYTigAFrM1ATsEI/CADLIM8HaweSkUFHFERYMANRKENcsIeosZKyWYLtHRE
  5HFstmAUt2AFTiEbCgsDBAEKwgEngGAdomAKUMAoTcoXhsIU/qoZWNECsEAd5oADYeYVAEAQ
  pCC+KIEGUAClcIImomEIjDd47MKAXMtOxcALeGADaQsAhlU19QEAqgAGRCAdeAETItHg2jEj
  ZSEUJ2AX1EF55wEBUEEXSkCR/zgBebR2iABgHohBHGhiHUT1EKLhFgquCPhA9YKhGYwgA4zA
  C4gAD55gH3BBEAZAEA5rAPahHk0hVUFRBN7BCgigEzpBOXShFtKSNQCAEzKABmQCCFDgJ+Jh
  ck7BgzSkEoHADjDSgIAKBrggDk5hDnjACESByDrACIxgAtKBbQ7ADj7uggCEBoJKBI5BHFBg
  CnKAGyghAG4j3EqlEjrAzwTgHODEFpIlWXbkeBgsy7b2E7ogyAbMmbogWtTlHP7hBtqBUmTs
  xKDMFXrPjOTgeBgBNTiCkvohWqJkQatBMXW2TM7IAxjBA2pgOOmgC4xFb8ylktBFwzazmLuk
  lrUoi//QLJdL0wNENwGN9mgTwM0mDVCctmZbpQBs08CGhFY8wBg6IkjU8jTIUF0CBcjaEocM
  APo6yf/k4DF7RA5q4ARAaQ3OSIuuqFKy6MQoBUwcjUdyZaEUQ6GgYF/qQBrOAQoKQMp2MAus
  YViWhF5SzQpzLZUnhQydYRAEVtJShQ6SgQxCRlFe0A03QTBdpRpaQKOPxcGsQQtQgQ/SMQp4
  wTi4AAbKyx/8QFIFQXffKhkqkwse4Q0UoQ+Y4RVa4eOmQGpi4B1goYXEgAQmABkYYAISwQbE
  IABswAYkwAiC4Al2phZwYQl8oQhiwj2ilB3zIhy44ErLNxHiYRW4wQVKAFf/9IYNzMgFCIML
  aCAG8EItOvWkaGE//IMnqg0eYYEfEOCsuiMAf+Vo6wAARuDy0oHgDpXo2JEW0qEcLHgCAkB5
  M8ALRgAdBKEirnaEZQgCqiATi+DgBIesbYIdwwFVTYEETiEIvMC2syG5nmAEetiwdOEH0OES
  jiAIBhFLYSEM3oESyqEcbCAoBUEOIMBFkiASguAc2SsGIAcWTsEUuGBqpFWzPognq4ILOoEE
  EAsVpkEDemEJiNsFWtGCtCEHsCpq7oAwCIAGtoEbmKDdKEEMbi7caKsA6sxJ/PVVHllHTqAG
  QAMANKH2LOEcjOwG4hAHFuFhFQUK9G9ncQRGZoll/05gInzElCJBaQthA5iIj7yklmXM/2yl
  FNoBiiJGbyyByu7Wi6SoBW/WbKFvUni2FCiwNLi2IXLwBJRAmhMgNc7AEgYhVQplzEwAaivl
  xCyh96C7BtoMedLPo4g2Ek5gEBK6CzpAGqIFXLokXnboeDTjjDZhE/S2zXnOEnCw+tgBRpzg
  im7AkEK8C+91oE9gA/bFW7bkoIyBE7TPhn5sXuqgcevADM6hQUFjEFLhBArhHFzhlkshVY7o
  cQGKVSrBZLIvL1vgzlM0y4coJZcAP+6qCPRjE0WAAWqXQp6gBLjwrVghPUmgCA6hCEiAypBA
  E4AA4WJAHN5BHUIBF6oAiP/RAQ/2oQKIoNmb3RzQARcqVRnUoAqAwRe4twh4YQq4l+1cym0M
  RAqCQR3cwRxMIRw64RTWwLT7hA0AIBQKhBLuYK+jKydwgri0AYSw27VCUd/y9DtQQ2KOdhyZ
  wT4egHuHjgPawh7ioBOGyhabwQYoxBxw4YlRtBG+YTJJOAOCAAvWggl8Ya1hYVyDoAIEYQRm
  YARQ4RIUzx38wB0ULwhm4AdKgOZ1gTM4QRk6wzHQ4QjIFUuDwFITkA6MZEeMgAamRhumZir8
  ygVEIAakIg/2gxvQfeH2wAZomwQ2wFLVoBgI4QZQ4R7mwRSmQRQuYH18ooTUGgbm9yyAQOof
  YBn/TmEfzmATDDljmsRQGNnAe29HzoBOGtujAABPzOAGrqmPiJlLJBxGmCM3UiPEcSVXjiU1
  NmEQsgA7lygCMRxvQ1OSdVl0q+Fg2N1IhnANbIEd8m8UWqyYoyg6Q30Cb5M0JT/SVjl1jdw7
  zmDSRiHv2UAJ5s/MCumRjYdHdtNWSMWQUWPSnK8axA9j7VkzZABGeg/Opf+MrP8EJDc1KL8G
  1iyVP2ENSoExs5aSFYNoh1NKlqRg2sHT4bAawmj+wryRrGEUzmTqUlkOLKEUNiESUoOdNOGh
  AaIDDihdWtxwJWPTiVIFbjQswO6EFgAATiS4iDGjxo0cN+JTUK5IlJFR/zgAOfCAErg9sILZ
  CGIOnaBIHWva1Ngvkhw1O6IUcaNAxqsY4VatikHjDiVTCGoJUkbRQ40aHjjJkdNozQkhJ07U
  MiGgXo5oI6MV4RBlHQdtU4BgOUYpHgMXLvg9uWeKWBwEAG76zagPgCAbWxhQ4nIs3Y48U7Sx
  dTwFHi1aSeHC2OMiGL8KjbR4oDgRx9/Rf+toSUTgXYx1I2/dihKNV1taF0TEc2HKhst5XmIK
  0qWr1gllWkgb18ioKzUuIiacCvJDUBV0xYL4IeFizzZxMWhxi7HNyjsRlC7vmaAjFIIfP2px
  agRfWSNdxaoIOuHBg5Z6xzFGUtZDOYpoo0pktP9YsYcp2e0xTBNtNKADMDdUwU4ptazBQyns
  cHLCJydguMYcJbiiCy43SOLPQBNMwMAeYfhygCq8qMLENjCcgs4aHvTHIwCljMIGGWSwQcc5
  NxTiiiU8yMADBJxschUA7fDoVySc1NKCGe2ccUIW52iCzzlfftnODezoFwlFAGiRpppuTqTF
  CTIwBAUO1phhhjVQQHFONQVU4lAh7FjyiVWeyTmIK31RudGaJ/BgCUM33OClJl1A0Q8Uo5SZ
  6BxL8rDJGfqBtmapAPTCaKoAyHECOzecQwcoZNQhQBeXtlNArq64UgqkoJ4gxxlU5SfVoqn6
  tSoANciAZCmffLKkK4X/3DBKnb0YYoYhvZiBw5fVVNMOuNUEWkqkv24yrAdniEpqPzyeAIAD
  G9zZAZ5dtIOkA5Y4MEghDd1QCUTsyHDCJptoMREAo6rZYRLSDlJJIYM4gJ8Ha5TycAGFWPJN
  wsYe21EXZxRDAVkkRaMNLTEQcJsNzbz0RAm6FAcyRy0AoMwlOTxChQMA9AEIB0WsQgAle+wh
  xQRi/IDOZwBUQgorbprqgQxd/MJWEWQ9covQqnDAQTSqwLOKFSIQAEMwwaj3jBhc0OLCGTVj
  1A0AR6CSiBQuxEPJMeKIswotO9CySjriWMEFJbZhprYYCJQgR6kTGTI3aXV4cAoXXCiyDiYk
  /5WkijZAxOBNG/HsQY2CpuigQzYl/FBCIyc0wgnNlXcEwBnoIGDOERU8oYMpw7TSSQz25DBF
  ESKVJTQtBwRu1FHHlGO0c88QgUBwssPXyDcUpQrvCe5wscoB5gvORYJGRFCFE3gUksoATowg
  iIi6CIJLFSM4MIPM/gtSiwCegBMhKoUDcDECIyQiHulYxeDSEQ9y3KMRO7pdTQDgig6wIQRD
  qtWkENIrUHGiBuuaSAssmICKuAIK+bCFFkphgG5poh0R2NSkKsGD/CQMI05TU8K0cAZ2FGAQ
  LcCHGXpBBzN0gE/nyMINcJUrB/CAE584g2c8cAIpfgxkyurKJ0pRCP9AlakFZDSIQ3jVlROI
  qodTo0gdUDiaVWWxGrAiAwcFgINFQAFQGttVr5j0iVCtEWH5WRMcN1KD3A3wE67ikzUEUAc2
  BIkNkwhBkMighG0tYhEG+BIUNGGQgDmkX67gwQmGhTAApGkRjLIEAArRgkydowvWGEWgfuWh
  JcmASUriAbSe9Qk5PGuAV7kKsNQlBx2CJijS2qWO3HRIjLTAAyRQRGuEZpZVcENxe1DHFoKR
  gQrQTyt0QOHN5DADexQhDoJwRQqKwIQDUMINYggGLGzwgWCUABcAgAIVMrKANFABH2rywCDK
  cBa0FGEd0RDaFB4qG1Uo4i3Ui8eKVrACU2j/gAHboEQJ1CCDctasIieoQCjIEQxkqBR1yACD
  CxJxikQkQgL8IAc5/HDSFcyjCsLw2GcEEM2OkOEMbrDCMQ6AligAYiTrmJE9CieCy6xoAqYw
  gjuesI/f6AI+cqBJUC+isFqogQiB0MA2uMGEaGACEK7h2htu8Vau3SIaUzhfHiTzFnFw4R3H
  EAEDlrEM3LijPRX7Xqoix4l7wIISBNjGeJzRB/oVQA0lIEIEDHCFMYgiDBcQxTAu0AlRtKEc
  rZiAOlzgHBIkIgNBiJkgGnEhYCnjYhXwAxoYAAtJHIERm/gqRhRSgA4kIxlkAIUZoJCFAiRp
  SZaQAwQEmchCoLAi/w7IVCE+UQA9taMF7ciCKLNQiINRhBGBwB1o5GCJQfQjC13ogLbo0IuB
  YGpc3u3XwBQSqvzIgB2f8EApStGBkV6RS5aIFJIO3Kw1eIhY+oETqRzs2wt+qBDtOIchulFc
  AVijC0aKWCVcUSFIyaBQA2SwMiO8iag4YBTWMIQkg0SFNDxUFQGlAgcxaQZbdUETe/qkLcvk
  xOQmKZg9nMixXGEMGfCrEFnYLq4IJhUtCJMdEqtENUahiVGcYxFd4HIXuoXlftygX4LqrzJB
  cwY51IARhdzhV29whj0AIQpcA1sRgHAU5uzhAwGwwTwQMAJdrAEA50BhJRSGizscoBO7aP8D
  ZUSQiz344R5PEANuEaBma5CiI4hgQwcUZgB1nuUWh+DAFEx9ajurghbbOMZeV0KNYJCDAXFg
  QUuaBgB8zI0OFDmDLjJxj2D/4B6Z+ME+fgAcZANHFzILoC5ksIZGJNNpookwRsjggDlIIQbc
  mILJrlkge8TgDpcJ7ARwYwQEVKAEggCgcCjoCt+uCQ+JYAEFVKGKdbgmro94xBv6/YhD9Ps1
  QMgDLYBgD8Id5Q7puIOr+zYeFzAgGEaoACcKCTJ4yWEOlxjBPIIwAjWUQhdHWAIL4FCOcMQi
  ecoTXVtyoAjC+SIG4uDrMWTBgmXEAxZUZZ0/RlALK67pGyUogVj/jRxhHABABjfoxQaL24s+
  JcoWMlASJzgEpcg5YLpa+MSrAHblLEgrFc6SQRStiHSbKCy9mthAF3qBxGvF91K3CtggJlYo
  hchBCzVwxSgK0F8AnHCka8oPIzihgBNYIo0FS9eoUkmqqZnB2htBhRaUVIgNgEVIdfABDi41
  qSE6gFc88BSroLTGQW7xkIk8QymgYIg6kEGgaZiEHfA9BRCk4cZDqoO9xjSmUYyLUv8qgAMI
  JvRScVFhV2nEnJxVA0s4oR2jGEUXpHFEM0CyDnUAhSQjaQg6YMteHC6Txthx3wb7kFQ1iHAL
  NtEGXkTBNWCzw1MfYDbMqMMUGfDCEe53/yoolAxaIgd7QAv20A1MQAznoQ6X0AhnIAw6cAym
  IAQUsQA3QQoAMAdcEA1dcwi38AhCs1BnURKmdgBIYQVGQw16Qw4rsA1TcAXmsAaRw0pzwy0Q
  UBUeYDDwYWz7kAlEYA77sA8VADs/gAe0QxVcIgxs1k+UhxHWAACXEAbPozVR4DlMNQU0EgPp
  wAKnQ1UfYANGYATm8AToMADBMRyRI28eEAHcwAuYgAkcQAoAN3BcI1dv8AZpwQRWAAcUcDw5
  kAcJ10DpQAOUcBjlcBsu4AYuIAYlgHEgIwNRoQVWwQmecAS70Am0sA6biAkmEw1i81CheACD
  swrhkA6Gkzh9df8Z8eAGywAODFBVJSARqqQsFUF5vPYjAnBJxgUFR8Iru2RKm8AhUSJdcOQB
  lbAn52AQEQEvCtMIytUZFDF4ancCrtACOIADHUAHAgB3eCIQi4ADmpBcAVMApfAkXVEDnuEE
  BAF4ADAKleNDCENIJiYVUeZgU+Mx1eaE/uFfGHMOHSAAlgQKAtABHOYQuhJic2AJsoN6+kEs
  aRdU+vF6hiAAxBUC3RBjMpZ7VGBjTzcrvWANG6AJIzkKLQBK4RJ6ARMxztIIxiCPq8cjh+Zg
  6iIDlVBHHBQClRQCF1lJGLaT3QAK2iIQHWANn1eSEdAOQKYxzrJ3HpMmveVbbKAGTsD/BqxR
  Z+twZyrDHOCQNKbgBwjwBJmgC2dQCYdUBwDgD6uQA7FANDiiC18wEcowDwwwA5uwAVlwE1Sw
  AR5QDjmACf4Ggo+gFiJBV8pTBKqQcCpTPTYACxJgDi4wOgjwTFBgQbawKh6gAPtwCT2ADPFA
  DWAgCyoQD/GADGoTDOqQCB03B1axMPt4Eb0AAP9QBs7DC982EmtBI9wQDsNwNJmxBYngDzoA
  hLhgC8GxBmvACTBpQQCwCTjwgYcwf3EFgtPJNdFgATvgC+EwAYSwbjfwB3CQC+XADQcgGQvX
  N1xwOlKgDlIQDPvwiCDTDopCETWQBCSQibyAFq9BFhyQb0OQ/4XaoDxAAAnpwAXbgAUxEAOu
  9g5x4ZnU0CKoswfL8BK6cDAQGWE3oAWu0AWyJySgIA1dMGakZ3XPBSzJaQkitZyuIEvtkArw
  EglZkEgLIQMT0Q9TchNQkKH9gAN40gtKUAfip0R6siftUAkBgyQRAQFdQSwMYSTwsgGVk0ge
  owXGICelMCfP91+WQKLT5jEU4VWueRGe4TCDwGIC0KF5MhDj0kcI0UsykKRqdkUn5lt7dwPS
  oARk0A0YlgyTkJELYGOTwAag8KPWUA0IgaVVdgPVEGSTkigDY0r4oXwCtiZSpqG6GAK1R3sC
  xZFpkAyRVAeGEI4lKUvn4GNIOXyJ6v8QDvBfm+A0l2dtM0oCB/Btg8kEWpkLy7AiphAE7pAN
  MlELAHADcEQGSbAGYdAJ7yALplACVhIIfRAqRXcGn/AXvNYHrIEJHzhwYNNQ8NQYWQgPByA4
  F0ANzdAMwSABROAFXGAPe4AOiUSZt+NC+4AKNiACBmoP2sAE9gAEQKANCYcFq4AFp2gF8ZAI
  XqAjbuaEZDBNfdAJ8VSFVjhXRSA6TBADFyALR0NVwZAIOmBVFdBxv+EUnWEGZBBUVqIJQCCx
  U5ADQKAKgUmHebgNXlAB6NBVoMEIayAMAxAP4Do4WEADCyoL1HAKp0AOYmAEA6AFCmA7AlYD
  hYADZQAjyQP/G2eBCWehFviqCIrgDWMABUGQAadAF1IQDw93c6dzNNRAVbrRDKHwBPChnIfU
  DwCAXdZgR0Hie02kXErCJAVTMOjSQyiUBHLgCv1iC/CiBb2lAekILAoTR6XQDtloCD6wfT6A
  JwMRLor6JxHDDlLUFd9ALHIgA65QuADgLnMTpcryCQ7wRBswChvgSWJSfZXQLw6gqv0FeUu7
  j2awKtJyAxsgDRzqe1zWJ6EHYgOjtxxCj6MSYRJRCdJAB/oACtBbB92QBtUbY0JCBx9aCZbA
  RoS0EP8yKcTbK4GUTJEKMqUQFdWIA4bgfUOyQdULv2lABh0gJhABLFxSCrYwuDeg/wnnIHxl
  MinjKC3KJREWGlSFIAxXAASek4ccgJVAoDJxIAKYIQWnEJwVQAglwAOCd0hUAAAIcDSw8ARy
  IAxqcBHssCrJaRF+cTmgQBIg+FZFsFQ+kYUPRSD2AA87UIh7EAxbgE/k8ARhoArh4AVn0IS3
  4woKcAovsgrwAKBgwwFDoA1gUyCqkAPkSQvlsAcMEAoS8Rn7KAlrMgc2sAoPVQRXOBIPJRko
  EA6dMJrmhhsZoAO9MQK4EBwApEZpkg/R1HV+QABxUAZhIAIqIA4L5VbTGQVc8AONIAyR5xla
  8A2nQAsMRwsNdwzUsAwrYA7+kwmDdgKMcDsUsUJtcAHh4P88pxaHBdJQDlwEZSAK+ZAKteMB
  wjQHgvAEsLBXC2oZi5MZYrACEhAMXsBsbwtH5wAAlnADIyskIVAHGxaiTDJiWLcuJaQscLQq
  vsQICHMGjXARirJ3uTMaWlBhHVAvcMcGApBE1kC/ihq+f1IIpbAGMkBFJnYGphQ50wgyrZoE
  LWAAi4AnPCoA46cno4CUZzRiC4O7lNdryExH7iUrZEC5GwYFihoxxQtigHRKkaBDV5Qw7BBU
  EjEI61svStQBshcCMcYKQlKQozCLpuJDnxBGgDKOyoUQT8IuBswoUSEDo4AnZmpHQ3LSaRBQ
  mFQABLN+blIK1JJlBe1ExLu5FDP/y8RsQa7wDQpMFq9RtXd2ghKMGVTVAzrwBFVQAjIQCdVw
  SJMAADPQItRQAR+CCxeRBHIEADwwrTXgwq9BnWqBhVPAC2yxapJBoBMXDM3wAaFABEIcB+5g
  xO9aOUmcCA2wA4EzBYNpFl/jUKIjGbSAnrDgDhSUJvv4D1I2BxNAC9rAC8tDEjlAI5IRA53A
  y6rTA/5AxyMwB8smbYlkAtHECI0QBCzwAG0QBntACemgFtSJh1GwDYG2Bl06y5GgAImwCjQQ
  A4UjDsdwNOe6D0UnaABAQaH8QufQBJ1gyjECxYcpNIN5AW0wDYSwBuvyIYJQAk+ADOIBFws6
  mhPsAuog/wESUE9eMAf85ITGzHTKzHn0e0Z6G4wQ8A1pdtPWrAwyQDsI8yEXkQqr8tnh3LvY
  h0ST640bQF8OkVyCsiRbYWJywAM8EDnvaLo6nQV1gn1KJHe/J3wGgSSLl3wJHWFQOKVM9yru
  VQeWJADSkEdQAMDKxQ67wg4nrqT0+JLxFk2bEAmD0AUk7Y0cmpF2lGMbIAPfLDlRwQNhVABO
  xEejJ8/Bwgjmu3xawAOjQM4+jacbFGMxFgJKYCal0KWSk9TtMJJXJtOhJyggdnFfbG1qwA5l
  MGckoWpPxdWohR7p5j4loAX4bEELAAC1cAklAAAFsGkXYZmJJNUX0QFa4A95MP9/dAjFoMME
  y7MWivA86RBVKxIMAQCWiQAj4DBo1mBB7DAALOALbIFNUzsF3uHElu018JA+EzACavTplVMK
  crAG6GAKZSC1n4PoqsCzBCoCIrAMmKE6RjDH4jQD9xMcW1UDe1eWKKQAFvIMghAsclAL4BA2
  dOhWHEA46aAOmcAD8/EE5BAAieACd/A3NCAOd2Ab8UAEQveSQgDK8CgHN9AFYyAFTXAN20Ds
  oJhQRRDcf+AAXCEqWLEJjeAH75A4BX/f8fAB7oAAl3AJP4AKwxzgAMAO7dALHfqpBZkFicIr
  SlIwycSlhgRHcDIRTt7NtQiVyOK4HQB326cE+oAtnzf/0UHmXeeX5FREFRNxBgoxKni54sfc
  BT5gCD4K9tyoJzVqfd3CKWj0ePuIof7VDlBgDdsiAEoAChjGfaBqKzYUMTT9R/acvBs9FYd0
  MP/FlCeQqDgACiEQUBhGBtwYX+dQAKb0CfulIewwCLli+QCjXM1OLNA0N6ABRkgyCncqJMtM
  e4sPBaw7CDMKGocLugWAlN0lRqjqCg5gC6ZUOzhtQYLvCX+gCKidFvzZFjHwACyANKZgCgHg
  7RWAC4LgCRlSCm+NQmhNEcGaEa60dH7RAotXArTQGm81teU97XOFmJWsFJ4JDrAgBs+AAF4A
  BzQQBpg+CxaUBHgQByngCyIR/5i3YAfSEASTYp9YABAo7BVRZW8bmGBEeKwBkMDhQ4gRJU6k
  mOBEEg+OJvhShSnKrSgfb0ULGYUDrylTmKS7w0VEvHjLqDGYYCqYHx2X0P0ooQuXLl2NPGgB
  UKjiUaRfMDQipwvAU12UDhxY9+jNoyhYb229VWQbNSl7lokQ8Y4LF0oxxB0rR+mlmFpP5WpB
  WnciADnVRhUbUCqIFQ5TtBHUpqoIBw7RDjyIce3PJVwDqoRaIYbaOyvH3O7ZA1OWC2A8PHj4
  xukMp4Z2VVcchKtUjU+DoPSqQwaUEgEdzt0owM5VKR6fTpw4c2b00KcyLK1mbuzpoIhJnp5Q
  /cnDjf8uhgTUESCPjoDuZnB0GVXtRpZqlbJUKlDIVRJOxJF7OCF8EwAozJHWeGqpSy9ptgMF
  PDM0cYUHGU5woJ0uoICimvZkMM4DAOjS70KI2NHCkgK6qIMNMtgAEcRkQgQFwA6sgWKDFm7g
  rZBBCgEuwfiUOc4DY7TQAjkA1lAOQ6Rk2KQRueQY5RxpQOyGDCZFdBI3ac4ZpR3zCnCxgEqy
  ZK+AAgZxxZKhiCIKSIqe8uAM/ja5QQAyQmgyBCrSoILJOkCpow5rRvkyiULcK6SALM5zMdAt
  CxHyDB0rTI1MpFwp5IR9BJmAliIAKSkkQDgoQiUU4oCBGlBNsUGHDLwYAZf/Nb4ZqhLoGHWV
  Ik2IqqWTKEh6BCuSDkFMlZSKiOZXVYDYYQca3omHsz1gkcALdwJwARk/GMLh1YgcwAaVUvyB
  x6pbD1EEmBqIEqYHKXKxZ4cDsCAhkznW4OSbRamlyJIzvjklnCIw2SqrN/b9yCNMCEpJERru
  YMGlzvZwYQIXdNDBnQp+kPgHoGpphDgAtJT3ITmkOIURLdDZAwVeDCOpVg4eueXWq6LhJd0Y
  aDjmmLXeSWctSo7lLBE1KHyqho0fwssBGShUQAdupiCIA1VUWScaVTTVZooctGGiHBfU2UOW
  srgohwBKjpFlDxjIWsaFeXQhStExg5ZokEGGqsGB/1GsMQQUEO8UQJoHbyjEAXbYKWU4OeSY
  cLS13d6YqBoGaaGDXpTgrpdeDOnADDN0qyaL9dYDtJK/v+RB1aE82KSQSiypoZZp5OUPAA8g
  sKSSXgQwBHNrBjkhzBoKaYHBFlw5Qce141UcIve0cOAc2tycJI00YpkkhDTcZKMOQ3oxwxop
  ecuyS8Fl4AECTmoo3QNGniK+eMWVa2f43qWpHe8QmQzh+RBEJKOOXnAwwAAolOc87OEN3AZR
  gN/sTn3Gc9UCK8SOLtCBSRMkQzeoN4kldcOCdfBBF7IwihaYx0UjvEE7RojAL/lsgcdziCVC
  1gZa0EIb0eDKR0JCEqhpY/8V4biACDizMBIYwQ9BOAI6SlCCWhDFFSwEEg4AUIUINGAdW6Gh
  yrbSr2hsShUlAwIQaJGOByzGCpQAiwuCIYZnkIMADzAFHviTn6C5QhmuWIMOFMEVlQHBDSOo
  QhVGYAMXwCAdMXjAHfghjEY0Qg4UYmILT2CDRMRgiiXB461AApKQFGEwQFhFOo7BhXjAIJR7
  MMUEEuEPP3ghG/cgQhWM2JNEKkMubpODC7Bwiie4gBs5wOEhbuFLGq5MZVYBiSoOwA0UYIEl
  WPjkMXwIKrKJYQ4+s1DQYIcoACjgXkojyDoKM4TDFAEIhRGMKlahgnfcYWZnOQZmcgYDsexh
  Aqb/JMcPXqeoRiZACycYxSg6UBsNTmJ6tgGFIayhiRZ5KXCl+MQmDCcH83nAfGtbQz71g5ck
  0O0cZhCAEgZUuV5czhoHvcGW2PM9hZ6AB4s8jhz+JCQP/MN1FfKAHE7QDvD4AHM4aEchwseO
  SrioHREogBzWJiaLJqAQTymEP2vDBjjJSapxchN3CGSNLlyJSzE6EA8sMZxNKIOlKnyKceTA
  QFedoBQd2IAtXHEOOnznOxUMQTKelwYMMkkJdMDqSMlTDRFmgUtccg9wXtc2eS2wBq6Agg8E
  sCQK2g+DyehGMrBn0C6QRxOjcFElTGglLrGqsEadC1rl9YlW0AITVrzV/yN8qbIbAisHKFhF
  OeIhC3BIwQWJ8EMozNHHKpTAE6UAgC2SahdpAEAQFGCCHS55mI+0FiSIIYg9VkGLmMUgXZqR
  pwuU9Qw/WKEIXDgCf/DhtlI4QQlNoEER1sELC1gACNzooQhkgQYwUEK/d4jBMWARDBcwYw0M
  aWSFdiECLgChUr+CLTEtiZXYqiIH8HhAOd4hAlHugRoLg4XDHuYOBPARHVXQRS1OcDHrZCFo
  HjAFL5jwjhhoYx0h4Va/RhLdQ5AEEOtYx2xRAA8sbCMGmXnHO8myh2C44xOMUxwAsMkDGzwA
  CFNgmjbASRDDlCwH9sgBEOxBiyCr4A7iuEORQf/JGZmAQ2FmlIINnoAcxCrOFVqohjQ+1A07
  2CENdgCBnPJnm73xtKeOUkOCPvENZezuOIiiENCOixQA1KAAgD2HNSKnNx8YtAMdWEQAPSdY
  wbLHFb7hhBDOp4UayGBw35DDBmaqvHNkx06GkMamTbDZFoCwHYE6TxYcoED1WbQSyuXBDXDQ
  C1DUj3rQg16c5nS922COPFc6YAEcUApLyKChnNjE8BBXvBsBQAYbs4QlQooDKFzOELf7zp1u
  QwftWK4DODhHvesdwM/ypkswCt8aqGlaIM2lFMwzRB2SwQa8iYh+dbLTiUbqICltAKEt6qy+
  t/oeGXDiqABnFA+eEIP/KYQEJMPU1RtEcqmm2YOZZ6EEOFxgAzGEwgtPOEIfgSIIAJTi0RUx
  BAAscYHERONWtSKJMEeiqcRwAAhTocUqULDdnHUGFmJAgBdMgQIm7IEHAOiC25KQAUDkIQbM
  XIaoEjEPI8zjGe4wxzz8oQMbTIABneECMagBFI67yhUAQAU1REAAewyhJLDdiiX9FQWCFMEe
  3PikS2CiMylMIIhvd4c7iDACVPykFsLgxMVQM4rFJSJTKQnJIT7iy1tgIgebWoUvYnYBGnTi
  AmVgAQso4RJw6Fe/mqEEDMABiyAI4qzV3Jii5EACccRgnL6KBgd4LE57gAAEvvAFBTrRiTJ4
  /yMM24dDGODww2XsQc1oVpgppi4ICsXZbZHYQBceS4ZJgEDCU1AF9CDLBttJY9pZIIQTfsNQ
  CBiO4giTHcm748KLAmCRfpAG7aiTy7od8eiCXTOpAmgH9vCpUfsEThir0TgN5PAHeUm/G8Ae
  H1CCkFqEcwigLoEbGGkPVskSV1gypGokawCAE5ABGagEa/gQ+yGDqWK2NCiRARGAzJFAamsP
  3xCfTwjATfiEAQwT2NER5HAAcjsBHNg0LBwpa0C3FgihG3AFBwDD32CHTygFMywFMCSabJOB
  JCiFNhSfEyCtRLGmpyg2E9i0BrQqOkgRFYGCfviHLvQbuHEUR+kS0P85qRMaNTCUEKKIBAPE
  kCfgAnvAoZBYh0N4gxzDpEuJhilbBTITh22IupdLhCBAgAp4AkEAijkQhEgYt52LiF4AAAjQ
  gJAbPKsQuZXpCsT4FXGCB1qwB3jYgVV4gHY6ltxKhBXIhmwYGXCQFq/bBxEwhUQgB3OohRow
  HCnckcLZRk9Ahwrwg0QIACKoBQJjoQLYhAhwgz0QgRgogkvJxZO7IZOYmvliCVDKMBhYBnCY
  pwk4BQkwAh0wAi8whxEYgZ6oBQ0UBgjAj8XxA224hcJAiXXABF7xBRoIgzYjlUsIAo4MAi+o
  uGLwgiDIgEvQAVM4BVNIyZTkByLwN9hpsh3/qYJOyINx0obV2xTsCoc2mIYOGIV+OId2IAQs
  +Z6SCqogMAJUkAAbSMl5KiV1aIaYcwIiccTjKQUtIIQWsBsQkZNYaBpVoIJuyBvb6YAuAMrv
  KYCMyjZFMpzDOY5HLDAIKIAWwIcu6AABqQN9uJ1Ny5ztGQX24BzP4Y005AFO+AQOPI61QYKZ
  apx5g4JzaAHQQaAEqQHKvMZPWINNWAMzJBpVqZB8mgW8wMEC0ARDABEyYIU4YbY/ux4l0Kn+
  IQ/POSBFZAfxkYFGEALYcCjEHA0zoZAlkhdL2IQjIY/ySA/3sARsKB/j2JEzmZD1uZHR2MZt
  3ARVKZ63VA0zcYUj/4GCRdg0v2oR90iQODyTDdzGGkAT9KyPbGMHrvoTRRyc9LtO1eAEGTCF
  B1AE6KrER2CwTdyUKfDFbbiDAP0knXEDMVgBBHiCCkAHzVsD4qiFV3yIGpSBDlCFj3iDaJgi
  lTE5YcoU59OUlKC/KaAFiywzSpCnCTgjmVsBKYABL0CNrgsaBxCCESDHE7vRixGCRviEWpiD
  HrWYWrCYE+uJE0ANJkIFOciCaZg75UsMrugXk7sUTGCaA/DFGGgJDHsJGIAnU1gYKSCBPjAC
  BEAAtquCYgiuEuu8E2BI15kHIIgCbUCBGOgEDTMBScCDWuABHmgEPeWBPJ0DH20XH+WBOf/g
  geGAAD6dA104ohI4xVo4Kw9oMgDghFN4gGChhXAIh84wASQIgixwhQMBqxPotm4Dq02AgG7j
  hDzVhX24By9AACMghwA4UC/ABdHgD8UhLktwhRbAAR/gjqiKkxKpAzrQnszJh8eEkAPyElcQ
  n1Ml1cNZnzOQz+I7Abmcje/wKMvKDdzBge7MkyshShRiBwQZVcQxk03QuKeAhpnyAAcAnVJQ
  AOccjYg6V0XpwDgkikhtpA4AAGWwBB4ohHbAgdLshjiJrOvhHzNYhMwCoSvpE3ZwAHQA1do0
  TFHlQNhxS3Ygtxogod5IIej8hjM4gdIg2SccDUYY2U2oVwBwRLn/UBR8Kr4dcYUbqIYuTI9r
  KwVTe1kxkcJvg1kxoakzEIITWIOIdQVbcJegdRst0IhO8AVKSbr9PBmT8Ept8KJV8EQuIAAC
  KDJ4YphgOLuqe4IqSEWldcVXPAdxawJeEDnpilKjewRfqVoQbRrsioFw6ISYqAlYIAcEeAZq
  eIA4cIdpPa+gYQcAaIQzuMw5KIEqSAWao5IMWMpSCsdRKYaguBjjoFbmKAAtkAQz6AUfiINV
  yBeRAAmPAInm04apWIUYYAm3gAnOAAeZaEoXOIUgcBiB9IKa+4FaKLHhQA04ohY5MIKmK4MO
  qAY6uhjDaQRBGIFsqLq/NQIjOErqNYIM/yCVmUMHRyBHZdhTTjiTwiGKTYDJRugAX+CGa0CC
  bKgCRTXUT2iEQGUHJ6iCTKiACrjfEsCFggyuID3UIdHRWiiBHyCCZwiCEZgmXHUbGQCAT3CF
  AhgF2qiDqEoDveJWstSEXTsPwvINW9A2sDqDwunA4lG/pIqEuJzLy6mdO8FLvizCLhgPzgpM
  BEIHNcgoB1gpRRMTDzDUtamEVzMcnxmK4rApdkiCI05iW3BCHmEfGsSLAHSADqED06we07we
  Y+1DTdicLkHCUQvDJEQQS0g0h4pWtySKjZUXGdCCAyJXDvyGTbiRnq2QRGEbMyniTcjjTTAO
  l12h4qupNWhjHv/AJrLaERHW4zzmNkReZJY6KtP5BPExKhN+FQ+oBXWwgnCIWsNwrRkLCQ7Q
  hqnhlRjagbVwJrfwoXmChVMQU3fIhiNIxVrQOLTduVEQN0OY21u4iqvApNZ6A1JYBw6oFWDW
  JFXQhh0gpLxF5Qno2xlYAQZYhTjwAgoZXnmRjt1phN8tgREghCe4hA1oBx1oBliYJxvYgpso
  2yIdYc5djUrQgmIwg7iKA0q5FAiLAo9AvCmoUm5gDGd6CRFggLnjDFOgBoYJojDVAWa5BCco
  MaBohPhgU+KdB1rghhQwg2pgh3RNH2MQBl2YgVd91XlY5VX2sLfTAVNEoj2Vgw1ctM3/LV+3
  eQoM6IAdQIF6gIYFXQPClAOHSqRaSIJUKIYRwF/83d8ROFNBsJgNFKtugwAhAIpLeAYvqIJa
  qKl1DhIbtIVK6AczqAMlMFi8qiBi1UsoQCjvYY9BCJxm/QQyLuJv4+GqZhQbvIEWqMvs2Q6u
  5h9rEKlFYNiDqhILRCBdUANQTYKvggB79QBtW5sbeDVGIK2Mbc4HBlVFBJyGss5G/Ey8SLQk
  qAQc2A4mYYX6IYNkUALM2Z5z0IQtdhH3TIL3GOzfEGNoxdifTWNqcaFUqARyZbTSOQFGQMyX
  5dnS0oKU/YbhCN4mjtnEOhMFGIRKKAWX/O2MHdninm7qnu5G/56LHsbB3ZlkV5EDbJiGcniA
  VWACdzyZXwFmXgHGL8OCB0iHOLCCdygHtICnNgtbIyCCBJ0BdAgKOWAE43rFDihSYKAAbdjF
  TLwV51NdGjoMDrADw9AGe4hw1+UGLECLPVAHdTgjMmWApjsF1FgEt7EFAFgIXUhFdLiES/CC
  DCABEoAFcCiHcrCCC+ACBthHP6gApGYkFqoERriBDqAGOCBdIAjmrMAkTIqGwmi6TB0jUDoW
  DWgAgI68mrhdD3uYbMiEKtC8RhiSNaVmV+GEJ2gCDQiEAthRntaFCggFI+gDSJIAMUiEsI3z
  RBADG1DKUziFHkglPvKSWEaUtWkEmP+shS5QgjHAFiH4hD6dg31I8VBQcz8gBx04hUhf5T7o
  AXLgB3Jo9EuogsLKtvhdgwEeAYpZYAa2QRkYhFHYaiWAkzhBOCL0K6D8rEqQbIrVNodiyzOx
  Vyd764DbBAfIgg04NvDoqO/IjcuB4S7QBMA8qfaAWDA01FEFGZoqWp+JgBCk40ggHvPZwBNg
  hz+JkWrrkgTCpmB7Yps6AVeoBCiQBlDIoAnqBlC4htuxBgPopw0WdzAGw1r4DYAFWE4IQDn4
  hoiai0T5b2ohLkdhqZNNFOuEbuheHym8xvGUC6osPuO4QUbs2ePY4w5EH49vzhpghPM8T5tK
  1/KBqB3RxuX/5G5GOYMSMAN5KAfaippaoaLmoz9fPGZ1OoY4QAsuuLB/ngAbsIFUcgcvqID+
  O+pG0AI12LkOmIVwyQApi5oiqPop+qXDUIWViQJgBlGVWAwsiAFuiIGYubA9aIZmCAYJeIZn
  sIFh4IJTIJIY3Rjp8IkRQIcnMIfK84NdaIUGoAHxtoerhQe1cCZkEAN04AFs4HWkaOd/uAIY
  CPJVGPKaxySA4ZX5ioE4iIML0K9jYQC/r915UgdT0AHKK8VsGLGfcIQ9LVKIfpVIEIZiKATV
  EYYS8AIvCIVEkDtkkIKnhEoxCIAAaAbhF4Ng+IBgCAYpcIMrQAZAcoFA+ANmeGXi/9BXmCyF
  GxgEfzuD762AEpiHSJ8nNGAAakAGBtAtdXSBeNAt9pcCgraBXTCF6s2AGi1PORgwHS91OXjg
  x7mTbrA/gCBTx5AZHFDOjbpxo8AgV67YOWAng4eMExbPeMioUQsAAFrOaEkgciTJkiZPoiR5
  AkCpG1A6GOqlBFSdOr3M3OxgbWcXKC2qKSxQqdLCQQUKsWNXSgYnISfkcPTIyEPUfymvnqzR
  saOWqVMt8nA1SGGhAgtvtGuXxZWDTR49Yo0btwMAOU6TFDjXQQAbMt3IAGZTh44haV3O/QR6
  w6irJA6SNF46kSKPE8pOYPSAMSqAqQDYyQ2tBsAJt5FqyP/ZvFUL69asI2mB7Xor7a0eamz6
  tImjHAChQ3/UkpGrRs1nMrtOznE5gEhSz8g5UfkEJzlUmXP8/VuOoDFtWGxbpaqIqmi3oqCP
  pg3IDlq0YtwRx6WTFRHx7O/ZM8GFjiBeECDwRBX76KLLGlrIoJ2CJ3W0iRfpaKPKFNFEc0g0
  jzwSTRHRcMDBhYcUwQEv2kxRBC+r0JBODDusIs478VADSzDBBCAGOUQg4AIl5HACAA5YgWJI
  TL3QQYcAoKTkCgAllFBFCTNkQIIpplxxzQUxHGAPEFMAQQsKNFASxh4uZLPGN76hlIaROBlC
  Rz3aFVADMHAIIEAnBwABCHrnoYf/Hiaq2HPAKunQ0AkNBOwRT34MMODCfhOYAosOGfRnhDvu
  VHBEFbrUUosul/UIxYIp4bMJdI3gYoQLsEghxQQ2JCIBOc88I8EKt+K6ggS7ihHMq7HaAEuj
  DFBjig4zCKLMJviMilVdJ6zhQSSNlHCJGBPsB46r2LqgjhTIuGDDBNRMAAukwUxpgymutroq
  LOqccokgQmR0RkfNYiUDaTIUUk0XAihRRwghdMOGAIZ00EUXmgB1VAGNPeSADJZUJod1whXH
  mb34KrjcCYNs0EEHhNVZpxmGWJMwDucojFA7ChFVgMyDIPUQD598YhEjnLXW0Q0ck6TVcqxd
  B10Na5Ti/4ADSReCSyGFpFJIQ5xEFRLQWFnzLCfsuASTwFQQHEIdJhekiVowP+wQO2w5tBRl
  pZUmxzfFvcXaZ1ePVAgA18lhKlXNEV2c4BnVkJlsssVmdyQefDKxB3JEgvdIH721twdTteZB
  bZt35NzmmcuBGic4f9JbR5pLTtImukzwDhc07KCN7OtQGAUmHExxAC07cJOOODRw4fp9IsCw
  BzUuBGODDkYg4EU2eDhRQi08fDNH6nGBwhEuF4RYhDYXYvjIeRxEsQ4HRZDH4SpAkB8NE7vT
  onsM74DjQgBbqCOGrtkQQQktifSoC1cBxZmq1hFGgIIMyQBMMv7SDUuQRg5rqP8FLm5AglOQ
  YFwNoMQDercNGrCAATAwVhCc0Ii/1UGBKuxGN0IgCM7RoRt6kAsqNhEBNzShCXE4ABOisac+
  pacIU1gFN7hAieCJgHiK0o8LXLAHUzgqETpY3vJCYY4RoEMQBaqFMHIGAFGlDoKOUFV+IDUB
  /T0DQH4QgwT4IQYxuPGN/JDAG2cUDP01IxHqaOIepEANMSCAEaW43kj2BpszzOESAWDAHpLY
  RCnYoBkBCEYiYCVJdDWDRsmzYyVf1QxTQGoZMGCAGC5xQrsRUiT6OkEpXNGOc0jDDNdI4dh6
  kTBNjGIUimFIIRzikEJQpHTQuU3GlgOSVKJEK5soRDv/umCGk92kFznZySJ6kstRaMJslRDK
  No2SFLaUomKh+5sBO9KHlHDGNt9AiebSqTmpsIYRNfjGGTixBurgEzN/g8tJQMK5Bc2iLjlz
  QCU0YQ1DgKKFaaBCN5IhAGmaoQsvu0EWGNIQBzzkIUmRSDBPAIFvnKBwdFsNAGyRuhca4zYg
  qYEWWKoZwg3OOsZJzePqSVOWouYyJ/hE0k6AutRFYmceyYjdPve5jxBzaEX1iHCEEx1bsMN0
  zknlGdTQgQtYIQ6rgIeESqQKEHCJFquIQThocAcalIF4+SkeDIqHvAAkwghe8MITnjCDEsxh
  DWdIEDIZlJF57KBDG3rDLTDE/6cQ3UI9UyhPEQ4woSgUAQgHyMNWdwA8EVBjRgH4gBhCkQ0v
  EOAAfujRj1ACCgDwgAQa6MU0SJCFaqBjJVvxHHFuQ51GFMgWqAiCEYJAAlsBiAjmQEcValEd
  5myOEcrgRClwAYwu0AMVMoAKANrxgrhUQg432AMMRBCHBxQBiG8Aoiq0YY9VbMMKlCCAfeKh
  KHDsYRnawtYEfDXFPujAH+7IRiZ+oEVdWARaX7zeJjwgAVlQwgWNmoA6AhCKUIgBFpSg3wS6
  m2ACuGEXMKAEJcZECXCoo4xScAMX7iACdRDBA5ZAZt14UIFTuAAcpqDGmEA5yWA0Mcc2cMEW
  cAyLHf83cT8fmFGwmphEEYDDBk8QwlaQWQoAcOIEcxhEJbB5DhxYwyBQYBjajjKIiIB5aRS7
  SHT8hrGioamvhRSODAqgCRyI7Jlm6IAZrKGwlmVBIWhpRzW87IqjYBSj4bTExYhqmxM2JwIo
  qUsBDNCFUVRiKZZIJ+d4RhtKc65ymN5K3yDgikq0YBS20IqCsta3E7DDKF3ohQAaSoa+gOKh
  ZjgIRdG2TbZg9CGTmchENrEJj8phKlSBjeUA4IrUVaM5VDnOr1MDneNYJjW/5mnOcLYJCEzb
  EhbRjUeXwo5UhLMRWihFI663t46kxtLDWVxz6uaB0ZUGAk+RQ84Gp5FNyGD/baVgTirlIANg
  JGIJUMABHaCJE3mYAR8TkAI+urCIP0wgH8vrwRSNQHEdIOAHmTBHFUYwA3S4QgYFErcg1GwS
  1PIgETnoEPkeQVjxcch8GeKA+VRR3hJxgEQSWsUqaOELI1KjxzPSHwJCIQIruKNH+TAtABzQ
  hjSUtwj2cE8cBHCFU/ThH1VwxRZrMQdPcaoRUT5BI6ijGS1EpxFz4FSndCEIW0zPU87VAQlw
  gIM2tMEbMeh5DCiBA0kcV9FYqQQj5tGANsUhBj30IRDRE1l4jDUcRvxwovJTYydCygWnMELF
  p7jfS4ygQLoQBid4AIBzXO8EWrhHIpgIRVD60QVc/3jADqzwjgcAAQVWcE8M4ncA18EjHHfA
  QpewgKVVsCAUawDAilMJAKGtoQIRhqILTFE/bOF4GdvgAgEIwAUjWmHy3T3GO8ShXu9/3wrb
  P0Y8TBGETzQ5lXMYag04YU9LyCAirkiaRE4gg0ZYDGbUwOVkhr1lTG1AjskVkma02SjgwJ01
  XBc8oC61wAYkhFDo2cMoTUbxQCtBRhJwQgFyxuPIgFYAgOCNBD7sjRxkQTKogh1Mgk08UwtE
  mgz0GmYUGm1VTqbVhnBEwhk0207JQCtVQhacgyZ0wUHRQS90QXb1xm+YWmnwQFk0kxnQASj0
  AkHoxAO2gEIMglF84RcWgv/S2AJGKQUP4N90WMSvuRRVuKHmgEbqVEJd1NSznQFunIF02GAH
  /hLE+NIfDqEv2UIhxMx03QshnYG4DVVUwMZL1QsQfgIn/Jp0pOHaQIbEQMAa8NqgsYXUyECB
  vUUqDaBm9E3oPEVI1UDcWMwmlKLFuOIrumJNycHY2d8sQgVfKaBIAACnSAEQ8EJ5RMEbYMgb
  RAPt7MktmI/5FME6rEOJRJb8cMMqWMExYJYpBMMW2EAwSAARzIMLxAMqrERplUQluMIJtME6
  JJb49Mk6aAM8YIEv9B0luBclLIMU1JgO3APbmY5tvNsR9IApdAAcLEOHEUs8cBgNYIEiTMEU
  CFb/FHAAEMDDA1iBC1SBIHyBBwQCVhSDHLSDBtTJBYTDKuQA7fjQeEid7ngQF1iBFajAkSUR
  WxWPGzQRLCSCP5yCDvTBKayAOWTCf+lCzpgeGKWOvqCDDsAXtuTCMdzBNhzDNuxdewBBNNjD
  FMgOiZgXPCxkiVAI7TQWJVyC6SWBAj7fGqBD0UnAfuzBEbEXDNxBNBJRDMCDe+SOe+wOVuoO
  ENgDCvCci7hBIhBBLbxTml3Pk8UGUyXOa7QG4ijmcjgHRzQm4HjEYzZmAuZiXeBbAbwWn41C
  P1RDFvyDWmQBUeTZNpnFw6yNLUSMvu0hJ2SEUAVmZTgmEpQEFDxfFtRB/zdQQRqwASjogwAs
  YUTh0ii0A2n2kgyUTtzcBqYth0ZI4ujIgC0kASFWwyhsgDPJRKzNmTUYQD9Emua0QGjQBcbk
  YQeWQgG0Qwu0QD9MFM0UQimsgdhZRCN8QiOUXYDljEWE4HEcx+UIDnaU1PW0Q0cMYGlchhxg
  22V8AkTYwoLKDEPIzFAMhUU56FDcwDYZIgCcQV9xRSSU2Zk9TnRMhCW0EjtcomOwQ1kcRVJE
  J4qijRqUXTsdornx4IzSaI3SRobmoi4WiCncARCoAjoewi3cQpBGwXkI6SEUqYZwgJCSJC/k
  QaDQAhZswza8wx6Aw5B9AGchgDs0wyXgQm8I5f9InAMACMEpjCQvWMCIAEEO2MN6aEkewAM8
  5MGcxqk7vsMHIIsQKOIJTAIVTMIkKElHVEEgNMAdoEB7tEdcEgMt5ACcogAtcMOjcoOUvgMB
  wIARnMIIIMjgeYADFMAMFEMloEIg0IMzvIIJWAMJQIMzmIAzQAE9+EOsBsGsXsI8+EMQZEAQ
  XIKuesElDMIIVAA6lAAeDCuTlAAPeEIjfAMnfMM3gGkqKYkH6EIQ+IEfSMERLSUxpEP8wIMi
  AMGafmu4fiub5oCapgAdKME8LB8AFEJlng4n6EImPIM7nAIDwIF7EcDtpdc2PMBWxSWL8Nwq
  KAKi+gIFdAILsAB/zMD/HNhLjIqijUJsxG5Ojj6LDETnNmFsL2msQ5hFQ2zsEAaiL5XCffJA
  gW3aGlBNR/gDSWzAJkRnmxCGYWjCOfTDObyETmSZT2RBnhWAxMwbZ+BiAtjCamREdOxUK1Fh
  CyAhTBDEnPkEObYnD8iBMdRAFshFQLXbYTbivPnUdSCOazSm1iYO4BSmAVEaR8Th9Ywp2pli
  4xinxaKmL/WSUDzNxpZFQ7iCxv6fT3WEMphcI3DaUzDCN7Bi4/AAZazN2uRtg07og+7sQryM
  A6wBdXVOX9WEPtCEPtQBb9IE59aBPmgu5n5u5obu535u6Wqu6dbE6tYBG+SoSEAHdwTBHzSA
  /yhcQDm0gSiIQhsIwDUIwO62QQNcwb3CwQWwwBUMQzmAg0yCA6Qkwimcgh+EAjmEwlwRgXCV
  QNH6iEkIghY4Qd6FgRvQFyjR1wTAwfDCQRjkQhik7x78QRCggytqQSW0WknQgT7ggBxYgi4A
  gym4AQzkQvpegQskAracghgkAgJXayLwA/R+wE36AyeE5VXMYQ3sjEbUwAD2DWs+jq9tAmq8
  m+BYBwcXxyi6oYdeh2toBeoh0zlowQlggxDowhOMgBf4gSkgA/vCARyAQy4QAIcBMSXscPiS
  gA6MQClYQun4CB28bgLMYUq9mwKQZcfhKq6SQCCQQKkGgjP0ATT0Q/8/0AN+9QEwAAMqdNwc
  CMI9ZfB/NnEbuzFUaMEmkGcptNL+2eBONQ4d2+Aez6e8fUIHTkwjmPA+ndtyCoexpQFJyIAW
  VBQUDOcglCEdS7JRmEWFjqEeP4Ubnt5V4EBitlR0WEIjlMLcKoRafKEtTEzOsKIGM59c4EDd
  ZNqmxTIPppN/csVbOEfaXg8OsNRreEB8zjH+GWcj4F8jTKKvEfN9fvA39MwgNPEGtNR1/HKU
  4czh1vHaUNlQ9FLeukIqLA0Sl90a7AxrHJsbm/PrWk4jVIER2AA4MAA4UEOjjMFOWAONMYAN
  zIMNoAEaUAMaiAEOPzD06oAN+MEzUC8RkAP/OXiBOSAAEczAANDGBpiEMHhAFcTBMPCHH1DR
  FGl0BuzYwnGLKXBWEFTBHJggACxdSghoFQDDZuWY5nmBO4RCcDWPOZiDOwCIDgTD5ulAIzgA
  VtyAxAr1UF/aJmsoV5zBGqwdApzCB0jBS3sjd+WHojySDaCCGghDbZgzbVAF2RUILuACKphD
  BCDEF+dSqO5sJRyBORwBLhDr6LFiw3aEM59zXauZ54AoD9RC/tnxfXKCDEwPr3Xg2J2AMPjf
  Xn+CCTPCLdfGPlktSRTAJvTDDURA2rytYKMmij4Mr53QHcbfVayGeHpUI/zSzKQNALIiUWlv
  u86FpjF2pe0gLZPU/2v3YG1PsPNdGr35H0WEE7l5cHTQlHD8WgFuTj+0cQ9ahBw0giaWghp0
  c97u7M4+TCr8kttwcEttxc/Y9Xan0hx6xBl8gi6UADqIN5M4N1uQN5NM0FvjQS0IAjo4ArLO
  gSfwAAZ8ATYogAJgwxfkd39HxafVgUmsAQ8QQj1Mgz/sFiooOK4uuBEUMTOQQA9Aw/LMwxHw
  gHMUAA1ixT9kweMIApSQsazewwgcwQjUVV2hAzrU1SWgAj9MCjDUwhpgRaASdY1L7Hf21VB4
  7eVgg8dlgK36g62egilAr0AHwTygwgA0ws5sQmiGpjkXIY0PFSMcRyNgA2/rsXt+giUk8f9+
  69UZjLPdCMWTc3eZS04LcAXZBdhS1GfhpCLZ1WfZ3WHsEjZIzMYsE5RJaFs7mEV72iDpkA7+
  JY3SGGcfs1RHFIBcVCjRokZu2CD/EboMQEApjqJwaMUJXO1saTpjE1vnhHa7xTZtifotjzpT
  sXEqSUJo3sA+vnmAJXcppnYNLI7FnPRWuEJFMXETF6F3HxVufHBu4F/csCIrLpWth6YhmHmy
  X00SWKwNcvmzQ3u0W0ISSPuzU/u0czm1azu2a7vF3vZE+2olyBYt241G3DKClHNo/IMl1LrE
  aoYyCIMCsCaOogQ+OEAqUDe+5zu+dzO/63u/73u+V0LAA3zB//v/weP7IMhArqtZBAw8sycB
  BPCgDtIGBDS7OG43dTdGs0N8UswBRDA7VHn8RpEox7uCLaiBsqv81aQCOwjNcpKTUpm6cnyt
  xDZfSQBubuyGjXtE0MpFJZQgzyNgXLCBJESAJNyAJPyD0it9BGwA0qMn0y/9P1RgCyC90kf9
  1Ev91lO91Wv90qdnMZhAjj4nBNW4HEB8EqT8drcDvqc9x0dEGaqBGlB3EpQhRNhg2jvE2q98
  3/s9viQBAMwnt7l64Ru+q2sbji/IK/hfgMlb4T++RxX+r/na8/293zuBGki647v64+dfFVx+
  6Iv+OZvAHCzEDKRCFeD7CKTCDaQC6qc+bL6rvuujvurH/usXgxOMAJUVQ+r3fhWMxQzgAkro
  C75F/vFzvuQr/wloG77kA/MvP/Irv/SXBgBAwOgrOzhNTPQnP/OTG/aDf/iLP+B3heKPP0lY
  /vmr//qzf/uPvr4IgPvrIqbLf/3bP74EBAA7
}

####################
# Fantasy: based by Maurizio Monge <maurizio.monge@gmail.com>
#http://linuz.sns.it/~monge/wiki/index.php/Chess_pieces

if {$png_image_support} {
  
  lappend boardStyles Fantasy
  set pieceImageData(Fantasy,25) {
    iVBORw0KGgoAAAANSUhEUgAAASwAAAAZCAYAAACLvRPfAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAAEsAAAAGQBYy9roAAA4yklEQVR42u19aXRc
    1ZXud869t+ZRJZXmWfIo2ZZnPGCDDQaDzWgCCYlJQgLpMKRDCJk6eWTo9Et3XifpTtJJZw6B
    kBEykAAhGIPBlkdhW5MllYaSSlWqebp1x/N+1C1aGEm2k17r5a3VZ61apSqX69x79j7fnr59
    iuB/xv+MixyEkDe9Zoz9zV/n7Gs03idz/Je33MiF7u28tSDnf8/f6tr8rcim9Kzr+iXpFfc3
    pmCEEEJJ8Q1GCHnLzfzPmHv9SHHgv2PNZssDQDmAWgAmQggDoDPG2P9Pcnnf+95nsdvtdGxs
    7E3A8tfcw3z/9/9noJpLf8h/jf+WvdjU1IRkMol7773Xs3btmjq3220LBAI6Ky6cfqF5uL+B
    xbEAqLJYLN62trYPyrKckSRJBqCcfwN/6YIRQmC1WsDznN1qsbSoqqYxxlBRXtZmNgn+QkEq
    ABAopa7iXiTkYhbv/EEpBc/zJaWlAHgA3PkCdzgckGX5khToPCWCr8wLs8lUUVdbdbsoimWK
    qqYJIRoA7a+Vyaxhv/Md7/j6/v3v/ExgZKQsFo+PAsgTQqR5Pv83Mfbu3YvOzk66c+fOlY2N
    jTZK6d+ZTKaOvr6+kLE+ymzPqHQPfwnYvOc970FnZyfdsuXy5oqKCsu5c4PaO9+5v3zjxk1L
    4/FYQVEUvP3tb3f5fD4hEAi8gWuXqle6roOxAbhcLo4Qjc9kclTTGPlL9sV8egUAH/3oI+js
    7HCsW7fpps7OjkUnT56IA9AIIepfIxNKKQghuPHGG/H73/+ePPXU0//Q29v/Fa+3bPH4+OiI
    qqo5QkjhQgbh/ylg2Ww22Gy2RXv27PmepmmLVq1a9b5CoVA/MzPDVVVVrqurrV2aTKY0Xdc1
    QojGGNP+kk1CCEFjfR2cDvv2Je2t35wKR4Y1TZNXdSz9rM1q2RyORIc7Ojr+rqOj4xMcx/nj
    8fg0IURmjMlzzUUIgclkgqZpAGAG4AdQxxiDruvaova23Tuu2PqhlZ3Lb6goL9sRCoVVVVWz
    HEftPM+1FQoFs7FZmOG1LOhNEkLKGGPOWYrOtm/dCMbYLcuXtH95eHR8RFHUGCEkBUCeTzEv
    tG6EkNKmFQD4ADivumrn2zqXL1v9xJM/ixQKkg4gSQjJMcZ049r/5saaNWuwZs0aob+///9k
    s9m3M8Y2MsZ21NbWclu2bNna2bnymsbGxvKZmbBYKBRUACoh5C/yGnfu3AmHw+7N5fQfappe
    3tt7OrBq1drdjAmPRiKTZ7q6ujaZzeb/bbVat4qiOJ1IJERjY845H2MMHo+bLF++zNHV1VXT
    2tpW0dfXpxw92u1QFN8HFcV6V3l5095lyzpXxmKRWC6XU+65596arq5VNZSqfDgcAWNv6NW8
    cj916hQopc7ly5eWTUxMMFEUCQB9167dUFVtXSJR+H4kEkoPDQ1OAcgQQsS/Rq8++clPwuPx
    0Nra2rI//vEPjkQiuSsYnN5BKUmqqir5/f5cLBZLEkJ0Q7f+WwDLYmzQErL/xa51aXPk83n7
    /v37/66zs/PK2tpam9vtbtV1/bJsNru9vaV5d5nHuXkyFBYBzBiWXWOMXbJ1l2UZhYJUns3l
    rkymMoMACrKstMcSCY0XTI1ut3t3NBodq6qquioYDI4wxhIGAMy5eJqmkY6Ojrfdtm/fp99+
    x+0PLF+29L2KJK5wuRwdzU1NN33+0x9/255rdnb5K3xrAyNDS9oaazop5a+7967bP7Vp3cq9
    VrN5/XQkqimKmizd11zXnQr24qnfP3vv6hXLPhUYD44xxkQAufBMonH3tbs+LxZEb8/p3n4A
    KULIJKWkwHG8zWIxL9E0TWWMEcNCsov1Tsxm87Uf/OAHv3vllVfcYjbx7RS6o6rcW7lr+6Yr
    pqbDXDyRGieE5ABIfw2wMAa8+GIX1q6tRSKRo9lsyXH7y0NbQggURcGuXbv0np6eK4LB4L5E
    IlGmKIqTELLa5XKtlCS5w2y27W5tbamamgpOFgqFLACFEHLJXnVbWxsKBZHabLZrgsGx7MjI
    cLStrdVlsZhbJicn4q2trfckEolJQsg6XddZMBicJIQkZxuW0nU/8sgjeP755729vb2PDA4O
    PTw6Grzf4XDdvGhRx2KHw75yYiJyT2/v6BUzM+kuj8ezurV1+eKOjtVdHGe6JxyW7/f56q9a
    tmz54nw+HU8mkyVPeE65l5X5oaradYUC9292uzUxMjKcAiB2da1wjI1NPVooyMuHhnrPxGLR
    FCEkyBjL3n333fxll21aCoBMTk6CEKLP9/3nj3Xr1sFqta44ffr0j8bHJ94FcB2ECB6Hw+cp
    L2/eWlNTXTE01Duk6yw/29M6f/AXmqiivBypdLpy2+Vb39mxbMlWUczjtcPdx3pOn/0DY2yE
    EJJmjKmzrPNFx/8VFRWIx+MuSqlw5513CnabDZqqWDKZdPVzzz2b/853v58ZODfS6Ha53uty
    OesngpNPEULGCCGZ0pwXO282l+evv/66a67cvr3z8SeeuHV0bGz44Y98eO/A4Lncd7//w6ND
    Q0MvLVq0aJGiKJQxVgXAY4R1846VK1dc+dGPPnxzXU0Vnnv2WfhclqapqWkpmcln/RXlsFlN
    qK6soLu2b1pX4bWvffbgUVyxZT0WNddXHz35+srewWGaFwsJACfPV+A3XPRP/xMYQ9Jmt3d5
    Pa5rY/GkDiDlcjkat1y2fuXU5ITwq98806DrrJsxpldWVq0oL/O+o7626tZXXuv+Ziqd/S0h
    ZAJA/mJk09DQgFgsVtPV1dUFprNsKo58Ns3amhscqixSRVFaALQwxsKEkPRfAlSrVi3H7t03
    4FOfgr29feaqZDK9edOmNqqq0snDh185HIlEwwYg6peiVyUP5cYbb8TAwIBHkqTG8fFxpFIp
    yvMCCKEOr3fQ2tLSnPH7K0SXy7l1+/YdVZOTwcdPnDj+iqqqYQBiydu60LwPPvggAoExVFX5
    1584cXwtgCWrV68xxWKpDcHg9BKXy53u6+s7ZbPZdJPJZEsmk27DGx+a6/t0XYcgCI7h4ZG3
    9/ScbidEgM3mhtnsWWGx8ElRlIRCQQbPExQKustq9VzL8yamaXkEgxnC8+ba6mpLJyGIAhAB
    ZOfTK4fDhkgkFNc0tFFKryMEOcaQ0zTdHY0mN0gSsZtM5mYARxlj2r33fqCZMXJrJqO+v6mp
    7enu7u4fMcbGDKN+waHrOhRF8c3MzKxKpzN2WWaM42yQJM1mMlEul8s06DpbBCAMIH7JOSxC
    CMxmM9KZjOd9d7/33z728EMf2rP76iVbL1u3uHNp+9bg5FT96HgwSAhJGe7iRbvUJWXIZDLm
    /fv3P3r//fdfWVtbA4HnYBZ4OO12srKzw7Rp4zpbYHSMSySTZTu3bdoaDkeqMrl8kBCSLrnV
    FzPfZz7zacTj8b0f/chDX3zH22/3NDbUNZl4buWDH7zHt3zZUt/h7qPM5XI32e32ytOnT7+g
    quoAgDFCSBSAPt/G+M53vtu188orr9A1FbIsQdUYamtr+TWrV9qkQgHT4TBURYFgtkBllCxd
    1EbqayqJxSxgbHyS/eJ3f0rIihoghIwAyM1eH0opGGM4frIH27Zd/q6/v+/e7cHgZPnAueEx
    ABPpdGZmJDB6xYsvvxpLp7NnAZwGkL7/A/d856Y917zj6d/9YTo4FR4DECKEzFyMN0QIwejo
    KILBYPallw4kv/f9H4xGZyKVr7x6WP3eYz/r+/MrR84kkumTAAKEkMmLBcHz59i//92w2azW
    M2f6/tfx4z3/NDQ0sS0azVxGqe3qpqb6smBwdEpVtSyAwqV4PIQQ7Ny5E+fOnQOl9K7u7u4P
    RqNRjlIOhHCglIcsKzQWS1oIITa73UYBbnFtbf0ajqOJqampaQMotYuJHLZs2Yrq6sr6EydO
    fePEiZ7OdDpf0d6+eMX4eKR5YiJkKyvzes1mHuXl5RuDweCxs2fPHtd1fZQQMnW+PAghaGlp
    gSDwXDgcuTESidbbbHZD1xTE4ylLPi+brFYr7HYbNI1B1wuQpDxJJPJEVQFCeJjNijo4eGpa
    kuRhQkgQgHp+GMfzPGprG6AoyrU9Pf37LBZ7dSYTm8rnxbFEIj61evXqtbouCsePd59SVbW3
    ra1lzGp1fWFwcOphs5nke3qODcfj8ZABLuLFyGbp0qWYmJjIO53OtKIoozU1tU6Hw24ZGDje
    l0qF+sbGho6LojgCYMJwSC4dsC7fuhV2h+Oqjz780D+sXb3KBE0BhY7aqnJKmF7/3IuvZDVN
    mzA2tnopStvU1ASOcqse+shDn125otOeSaXAUYDpGjRVha4p8JV5yOaN64SzvX36gYOHJFXT
    ymRFjei6HiaEpEpe1kLD5XLhueeeq/nQgw9+7W1v29dOCUNjXS3HmCbEYzNY2bGMiAWpYmR0
    3H7y5MnnJUk6AaCXEBJgjOXms7TDQ+dw+PARUzqd3pTNZssUWYbZLMBiMYPpOrLZLHLZLPL5
    PDhKYDYJoJRgcmoar3afxM9+82x2KDDRB+A4IWToPMELjLFKY02JiefdU1OhzUePn4wmU+kh
    AENlZV7z+rWr331uaOSMKIp9BmBpvjLve1RFth14+dV+SZL7AAwAmJ6dNF0o5Mrlcrwsy3Iy
    mRqxWS3Vi9tbr+CgWSYmQ8Oqqv0ZwGEAowCSpeT17KLIxYBLV9dqJJOpy48ePf6l8fGQE+DB
    mIBCgZmcTk8rpXIyHo9NEkLil1pE2LdvH1avXu3u7u7+/MjISKvT6QIhHAgh4DgePC+AUg6Z
    TJ4CRLBaTSmTSTAXCgUyMxOZUhQlVvKyFprn/vvvx7vedSd96qmnHzl27MQdoigRTQPJ5yVT
    ZWUlSaVEEEIdbW2NtaOjoz1Hjx59Sdf1fkLIoOFB6HNFHcHghFZdXV3j8XjXWSwWE6UcNI2B
    MYDnBZjNZphMZqO4Q6Drxffdbjs8Ho5NTw9GxscDpxhjPYZR0QDguuuuw4oVnXTnzh1VyWRS
    /9OfnmOLFy82eTyeTbouK/39vf26rg9fdtnGnK5z7w2Hp6KBwMgpAL1dXSujfn/tnYWCUh8K
    jQ339p7pBdAPIFgC3vlyWrfccgOWLm2HJBX448dPsJ6entGamhpzbW3T1Tab2yOKucn+/t6X
    CoXCK2YzP0QIErrO5PkKQfzCqLgEqqpW+cq8Fk1VoGsadE0F01T4y8vMJpOwRFaUBmNTFC7F
    uwoEAti7d8+VHo/b/41vfANlXg9uumEPBJ4D01TougboGvy+MvKJjzxo2//++2Sz2exvbuJu
    7D5+chhA1LBUc7rvpfdSqRR279799r1796y3WsxQFQmAjss2rMNTT/8G1X4fcrk8Fw6HFVmW
    dcMjOccYS87nwRFC8NhPHgeAw0/89KdfKvd5/1loaXZbLGZoqgpNlaFpKpimgbHifRAwFMQC
    RFHEMy+8nDty4vRpAEcAnJvtqfgryiHwfEdtdeWjp3v7vywWpJGe02deVnX9NZ0RvwFsrLW5
    afmqFR21Bw6+8pqxAZJrV3XcOjDQzz3122d+CiAKIMBxnLuhtvqGvCjGwjOx1wkh8VJSkzGm
    u91upFKpsqVLl25dsnjxBo/HXSPLEj1z5uykx+3afs3VV7kUWUQknmpOJFOrVq/oWGmxmFg8
    kYyeGw70jk1MHWWMjRpWUbmYcCoWi4NS0pZOZ1wcJ4DnTeA4AYTwkCTmMJnMXQCOAhi51DyZ
    LMsYHR3riMXiK5cuXQaXy42zZ/uQz4vgOP6NB6UcpqejvNVqdRLCcuXllTuqqqpPj4wMTxg5
    JulCIc6PfvTjrv7+wXclk2nC82ZQyiGVysHnU1BTUwNBANF1HcFgUAWQNsBqylint3jsPT09
    AKBOTk5+b+3a9c2yrO6XZRWMARxHwXGc8UxBKQWlBJQSCAIHjmNsbOxkuKfnxFHG2HEDTNT/
    2svLoKpqZS6nfKOxsfXHfX39J37+858N3HHHHc+Ionq1pmm5YpqmqnFiIrpI13HUMEozixat
    3CrLSms4PPRkf39/3G63D27YsMHc2Ni6Q5YV/OpXTx4RRXFmVpSlffWrX0UoFLJHo9GNsVh8
    sygWGlesWG3u6qIhj8ezZGoqVqvrHBobW5utVssyr7eyVRAshONoElAHw+Hxo0eOHDmnKGqK
    ECKVDMiCgPXiiwcwE40OnDxxItrSUFNBoIMShoIs4dCRY2o2l9cML+2SsqMmkwmEkLLPf+6z
    e9pbW5BKxNBQVwuTwENTFTBdhSSKGBoeRltLE1qb68l977/L8ZWv/6ckCFwtgDbG2AAhJDKr
    RP2W4XQ6QQip+7evfe3O9rZWqmkqAoEAnHYrvC4HOpctweM/+zXSuQI2rllpDQQCJgAZQ7m0
    hUrdhmDEeDz+7KOf+4L/ut273/+Fzz3aQMBgNnEQOFoELlmCIkvQVBk/+fmv2Zf//bspgJ0B
    8ByAlwxPTpqtwJIso7LS31Hm8/3j4e5jjydT6T8vam/PcxxXMTY+rgMwNTU2rGlqqLMBcAGg
    bpdz+ca1q+47cKh7EMAggOGbb9x745aN66599rlnzUPDo8fCwD8yxl4jhMiGm65PT08v+vCH
    P/zPe/fuvaqluclqs5pBmI7u7m49FJrCbTffAFWRoOmssa2poXH5kjbKdA2ZdAqjYxPKcwde
    6X/y13/4fiyR/CMhZMLwShcErbq6WkQi4fGyMl8mn59xUcqDUh48b4LFwuljYzEOgPVCOcS5
    vJ777vsAfeihj94cDke8giAglcqgUJDA88IbQOX3V0AUZciygmg0ZfJ4nIrJZCJ+v3/xyMhw
    NYDRkkzOvw9CCO6//378679+md500y13TkxM1BJCjYq3E/m8gkgkgWXL2mG1Eogi+LKycvv0
    9LQEIAFAnm9tjPnY9HQ4Eg5Pf2vRokV0ZGRs38jImEUQzNB1Co4zgefNEAQLeN4Mq9WOhgao
    gUDf9MDAwGHG2LOEkKMAIrMME0QxD13XUSiotTab89GNGzc9efjwq0+n09mMomjlfr8f09PT
    tFBQOlKpnI8Q5gAgrFq1qiqRKHw0lYpm+vv7BjRND2zZsq0rn9e+pCjETAg/Xl1d/ejIyMiz
    hhdP77zzTnF8fNw/MDDw+b6+vrfFYgmXquogRIDP52P19fWst3cEhHBoa2sq5zjPjefORSml
    PEwmC+x2u1ZW1jZ+5ZXexw8ceO6XkiQHjD2p04WSl719fZiZmTn5L//61e9+81v/mTvV8zqO
    nTiFL331W+w7j/0iAiAAYPpirWBp8Z761S+xccOGPVds37auutKPa6++CksXL4KmyNA1Bbqm
    QlUVZNJp6KoCTZFx1fbNQlVlBTcyGrRZrZbFACoBWOfbFIwxPPDAA9i8efNVmzdvWq6qCqYm
    J/Gf3/0BjnQfRTqdRk1VJaKJNBa3NuDWvbssbpfTB8BplPUvJsGrGV7e480tzYOZbAavnz4N
    TVVRVemHv7wclf4K1FT50VBXjfIyr84YG2MMBwG8QAg5yxhLzQ4/IjNRRGPxsbqGxolPPPyh
    TS1NDbcBWARAXr9ubZ3JZGoCUFNbU9VV5a8gAs+XAXDt3LbpnuaG2vrA2EQUQArANM9zSyor
    fBUjo8Gx4bHgWQA2w7gQxhj6+vqEffv2PfLhD39477Ztl1tra6qhqSo7cfIkSyQSNJ/LU1VV
    UCgUkMvlaTKdpS+8dIiNBEZZhc+L9V0dwoN3v71z/217PkIIuZYxVkcIueDaDQ4OoLf3bHdT
    U/3PmpsbZafTCZfLiepqF4vHA9FQKDQOYGa+hPHcxvV58DyPr33tG12BwOi+bDaPdDqLcHgG
    lPLgOOFN3lXRWxEgyxpJpwsWQjR4vd7FNputgTHmIYTQucISxhgsFgs++clPNY+NjV+fzxdg
    s9nR0tKEpUvbYbVaoeuALEsghMPUVI54vX4fpcTFGLNeTNGAUiodPXr01MTExBPZbDbn9XpQ
    VlYGWVaRz0sQRRn5vIR8XkY+L2N6OiK+/nrPGVEUXwDwsgG4hdlG8Mknn8TBgy9EM5l4b3//
    WIfdXnYHz3MrCwWRZTI5b2VlVTsAfzYrrs3nJY4Q6gHgqKtruz0ez62bmZkKa5ouer2eQEWF
    v1WStMrJyfHEs88+fWxiYsJcclwYY/jxj3+M8fHx9x45cuS9w8PDrnQ6A8YI8/vLmdPpgCAI
    tGhABFitFsJxoJWVbuZ225mqAum0xE1OFpop9d+/fPmK2wC0EEIsjLG3WrDZiTkA9s7Ozl1L
    lixZFAyF1dd7B3G6fxipbIGt6Fhqrazw+QHYDEHwJbrBQmS1qspK7L5+T8stN9/4wIqO5RZF
    loogpatvhJtMU1EQcxifmICuKdBUBW6HDdddvd3U1tLoWL1i+XqjUlVBCJl33i984QvC+nXr
    drW1tfInjp/AF7/0ZThdHrzy2jF85evfRvfxU8hkc/C5rGiuq6LVlRXVBhA6LqRUpTkZYyqA
    iMVsFm1WG9xuFwKjY5iengbAoOt68aHpAGO6kajsI4QMMsZSpbDzPODNHjly9Bf/+//828R4
    MGQDUBeJRJSVKzrsHcuXrQXQUVlR3uor88But7ktZtPq66/evjORTIm5vKgBkNqa6m576aWX
    yD0PPvyzocDYa0bOYbJEyDVyG741a9ZsqKmuhqYqiEWjuP+BD6lf/+a3dUopO9nTwzLpNIvH
    YuzUmX7mLPOzIydO6++5/5Hc8VOndV1TYbWYsHPr+upqf/lVABYDsC+0dp/4xMexYcMGfvny
    jjZKidTUVJNpbq5CY6MPbjdlHEfR2tpKrVaLyhjjAdCF9Ko0nn/+RZSX+5znzg09ND4erC+C
    U3FT8Dw/KxwUUF7ug9lseeO9TKbAiaLCO51lixctWrwKQD1jzDaXMaSUYnJyCpOToU3hcKTJ
    YrFiw4Y1KCtzQtcLWLt2KRoaqmC38yDEAkXhIQiOMpvNVgOgrJQ3no+DNUu/xJaWlggATZJk
    MMZQUVEGQeCNvUTfeHAcJxNCxgCcJYSMzQar0nemUimcOTOgKor826VLm3utVk7Qdb2+UCgg
    mxU5q9WxzuNxL8nnCx2SpIEQzlVfX9cuSfyNkqSwaDQkAShcffXeHU6nu5XS3C96eroPzszM
    9CqKMm4YFxWA+thjj1nC4fCWeDxOCSl6Te3trVpbW6NKCCFOp505HA4mCCY4HCaWSoUYYyKz
    WLI5n8+qUiqAEIpMhrhcLv9OAB2MMTchZO6QkBACVVXLrr766i+++67971zZudzqtFmgqTJ0
    Vcaea3bQVDLhO3mq54YfPP5L/6HuE//GGDtkhGjzUhwopZgOh+3v3v+uT9x0w97VAk+hqQpy
    2SxURYLTYQdjOigYDncfY6Nj44SCFfNBTMdla1bQHz7xS1nM56o5jq7UNP2UkXNSz79+Sik0
    TfMtWtS+3Go2Y/myJXjH7bfi2T+9iLqaSqxZsQwvHHwVbU11SCSSoNDgcTldAHyMMfvFJI9n
    3aNaZLlzGB4ewZL2VgCAoiggKIIbip9lhmDzRgXsLWBlzKucONXzpN/vd67fsOHBQ4deWRQK
    hRSv26XfetPezpOnemJlXk+F1+1Eha+svLaq4vK1qzpczx84NANAb6qv3tzVsfiG5146/PNs
    NjcMYMIoJAQByIwxCIIAg8JBGNNBip4Pe+aPz6Yf+MD7HAP9vfqx4ydlWSq4GdNx6lRP3uWw
    0o1rVvDf/eFPJg8fO1nXtazNzjQNlIIArBJAjeH1JuZav4cffhi6zmxnz579UF/fwP2RSKxS
    knRCqRmCYIXFYqdOp6+qqal537Jlqx09PUe+PTY29johJGZwyebMVb7//e9HY2MNOXDg1ff1
    9vbdXCz9C+B5AU6nE6IogRAKSnn4fGWsqsqPcDhBSmGiqjKSz8tEEGS1pqZ2SW/v2QpZlu1G
    xfBNE27duhU/+cljuP76Pauz2Tyvqgxnz55DY2MdHA4bRkdn4HDYQCkHWWZwOFygVLaYTKZy
    IOc2AEtbSKcIIfjIRz4Cm82mFvNXHDweN0KhGIoVzxJYveH56UYOOT9fyGl0VrDf/e43L1xx
    xXa7y+X+pxUrVnXkcrlzjEHKZqXW9vZFWyVJaVBVBp43u1pallyWTCo1DgfNpVJJbdWqNYss
    lvLb+vt7Dx4+/Npxw/ieIYSMA5CM9hqEQiG+KJpirs3hcIIQPRuLpXmetyGXS2UFQfCoKkUi
    kZDMZoumqhZbOj0RLitzegXBVkYID0I4qKpaBqDOcCIIv8BG9IVCIfmxx37yp29m0n6B5+pv
    umFPTSqZZL/+zW8LTNdTJoFPhSMzNgDLAfQZMbo633e+bd+t6Dl9+uprdl11e3WVH7qmAroG
    MZ/DsRMnUFPlh0ng2dFjJ/SfPPkr8tB9dxMwDbqmgOkaqv0+UlddqT774qEeAGOlKtqcDFeL
    BblczuXxeDwAg9ViQdeKTrx08BD85WVgjCEWT+IdN+/C2bO9iMeiMAmcyQib3mCiX3zhk5BU
    KgXGdIyMjuLc0DmsWLYYHUvbja9hFwWAhrIxAIlIJPKbpUsW39bZsfzmb3/7P09Go1Hljn03
    uw8f6d7msFsddpsVq1Ysq2xtqCVOhw3pTAY8zzXfccPVXX1DozOpTC5ulImPAAjP5q4pigIA
    qTNnTg+kU6llTrsVvrIyUl1VpYWnJtSZaCxXVelnFrPJzVOgtqZSOny4u9Dgd3t5nkt6XM56
    xnQwpuF07zktEk0UjI3DFspdaprWyBhbazabpqqrK7MOh9OnKMwrCGbGmKbIsiIqSk7kOEdX
    fX39trGxsZCRVxTnWzOv14uBgZGWcDhyTzabM/9X6MfD5/MxSglTFA0Wi0VraKgm8XiKU1Vm
    eGBFyoMs61xv76nRUCh0XFXV1Hx0FpfLhT/+8Xf8v/zL16p1nYEQimQyA59PhNPpQC4noabG
    i2RShqrqcLv90PUcpZRaDeL1RZO1GWOUUkrMZgsopaiuroCuU4TDKcjybPL0m3o/yULcR8ZY
    dnR07M9tbe0nBMG82+VyHEgk8kowOGNfsWLxVkUhZk0DBMHhttlszlgsS5xOgrIyX0Vj4+pN
    6XROOHPmeMggch8BMGF0hDBCCBobG/HTnz4u1dTUnXU4nNdmswVoGoPJZGZ2e5nC83atUIhk
    VJV5AQJNI5qiSLHa2ipzODyT4Hmzp1jV5WGxUDY5GSrplT5v0t0ArNHTp09/6fTp04sBNKxf
    t/a23dfsqpkOTZJ//fdvJqLR2GEAp4wcVqhkuRfyTB57/Alcs+uqDZlM2p5OJZFJp+H1uPCH
    Z5/Xv/Lv/xFPp9MzPM8zq9Vc/cA9d7nXd3VAUeRi1VDTYOIplrQ1cc++eCgGYBxA5AJ0Csp0
    jTBdh66rABjcLieCUyEMB8bQ2liLlsY6TAaDCE5OIZvL/RV9eKBVlZXYccV2DA8PIR6LwuN2
    QZYVcCUvC4xcJPrB7/cr4XD43KlTPc98/rOf+bimyt7p6Wl+2+YN5IuPfsLLE0DXVNx1x82C
    1cSDMB1VFT7n3XfcsGF152Lbj37xTK8R/gWNfJBSsryzZJR/7rnnf7v3+ut2XbFtq62xvhYf
    +fv7vc//4beQC3myZ+8NnNVihsYT7Nl1pe2xn/6CBAIBbFnf1XT5xtVWXdcQiyfx9LMv5VRN
    mzQsbn4+HXj66achiuJINpv9mCjml8qy3LJ27frrx8dnrrTbnYSxbPzcuYGTPM/3AgjrOhud
    Dzje7D0oyOfFOkJojdlsgcvlQaEgo6LCp1OqJaPRyJjb7RHLysobYrF0VSgUJzzPG7mtIj9L
    1yl1udzqxMREyKAD5OZq8s7lchgbGwMAQikHxig4joei6KCUoqamDBxHkUiI4HnA7zdDkgpM
    lmV2qcUpAETXGYnHk8jlJNTX1yGfLwAoVgeLaTZ6Scx8s9msBgKBqVWrun49Ojq5kxBul8vl
    4nR9BkNDIbfFYgPHcYhGRV4QGAAKQbCZly/fvFYULU5ZHjkzMzOjAJgCECoVJ0oe3ejoKAKB
    gH7rrbf+ob6+4Z2Dg4FKSVKRSORdTU3LNLPZwyYmcoIkZUCpAFU1mRsaWhwOh5uZTK21MzPU
    AxCYTAIISUoTE4FJI4rKAmD8AmisAYgBOMUYG9t02YZ3edwucETD0kVt7OVobIQQ8pKReC+F
    OQuC1ti5XkxPh/uymUz0hz/+iZuA8RMTwcIPf/JkMJPNHucozV6368ptD7x/v3v5knZaSsAz
    vUinoIShvrpCIIQQxliGEJI4v8JWGqIoAkA6EonENVWpZboOSoC6Gj+e+PkhVFRUYOdN14Cj
    BC6nHUe7j2BicjprsIOlS/CuDLwCCU2HcOiVQ6ip9sNus+H5Fw/ixmt3wu20gel6SVvJQpaw
    tH6RSAQApEOvvvrcyPDwez/x8If96VQSiiyhurICmiJDUyTUVfmL1UhFwvvvvMlitwh4+o8H
    5KnwTB5AghAyPVdPZMmTGxwc/NPjjz9xYNmSRbvLPC7cfutNQlNdNcbGAsI1V+2AbhiLW/bs
    MtksgimbTuOed6/2V1WUEV1T8Ps/vaIdOXk2CKDX0IX8fDp15swZGCA6DSBfWekfNptt1+bz
    BRBiht/vUXRd75ck+TnDIBVmd1LMN0RRRCIRH+V57mBTU+NGn8/nikZjSjwemTp7dvCU11sW
    bG1dvDUWE6tSqRxfTLqXku/FZ8YosdtdZuP6svMB79DQEE6dOqhZrdaQyWSBLGuglIcoyrDZ
    7LDZNMzMFCBJOiglMJt55HIZMZfLZYy1uUijSKDrjHAcRXm5D+XlFchmC7DZrHA6XRgfjwG4
    eFJtSa8kSQJjTN+z5/pXysvLh/r7R5bb7U6DVKtB1xUIggWSpEPTNAiCBek0FTjOItjtuh4K
    BfJGb2GoROydS9Y1NdUnVq1a8+tYLH1POp0nkUiS6+k5w9XU1GJiIgagaChSKZlzOHgvpTlE
    o1xVLqcTjhPgdFI2Pn4mnMlk+wAMA0gv6GEZTYj5vXuuy/f29a/dccX2FWazCZzLiT27ry57
    rfu4X1VVRgiJ+v1+0dhgCy5g69IV0DTtD4988tOghG71+byLQtORqKZp52xWi3zPu99xx93v
    vK213OuGqsjQDaAqARYIg8dlpyaBd0iyYkfxJATMRT/QdR0A4t3dR0/ctPe6zvIyD3Rdw5qV
    nXjltaOorSpHc0MtdE0DYQzPHeyWovFUyEDz9F/iYnk9HthsNjjsdphNPBRFwYme01jVsQQe
    p22h85jmVC4AKBQKZ450H+3ZtXP7VeU+LzRFAtP1IpDMWh8ChkqfB5JUwOu95xTGkDWAIXl+
    rmx2LyZjbPpXTz31n1s3b1x7+76b/Rwl2LxpAzau64IiS1AVCbqqwCzw5MZrdxhhvApVkTEy
    GsSPf/lMUlHVMwB6CCGTC4FLKd/CGEt/+tP/K51KJeoOHjzSoaoMkqTBZHJ5amtrbcHgZJoQ
    Mlri9i2kU4QQfOtb/wFd16cbGxs/zfP81kIhu35qaoqFQqHxpqam5NKlK/dNTcW78vkCKRFH
    Z1cMKS3mS0wmsxWAgzFmmU+vxsfHcd99f2Kf+9xnjpSXl4uhUNRKKWfQGZLweKyIxyWDrEph
    MilscnIgqmn6lMGNu0jAKrbpahojkiRD03RIkgJKBTgcDvh8bmSzyuyjhciFdKsk9y1btmB0
    dGi8q+uyQxaLdXk+L0MQLCjRS2avDyEcFIUYXp2kRaPTWcOTjpUOI5iLCTA1FcosWZL7QXNz
    47be3uGlhHCIRpNIJPIQBDN43lz6fjI9nUE0WkAxl2mC1WqCJE3mhocH+gCcIoQESkwEfqFK
    xXXX7cYLfz6w/BMf++hnNm/a6CVMByHALXuusb5++swVj//86SFd12cikcgoGCtggWM6CCGl
    w7qSkiQ/D2AgOBmqBqAJguC97337H3n//ttb7TYLFEUC0zTouvrmZzCYeI5ylFqMfAB/gWR4
    4ffP/OHxyzas3fm2W26szeezONvXD7EgoaGmEolEAjwFQpEZDAaCMVbkL42U0PyS8Mp4rFuz
    CpOTk4hEZrBt80aosgSeoyAE4Cid/fGLHYme119/ZWh4+Cozz2EkEEA4PI10Kg1FkQGmw2rm
    4XE6UF3hgcNqwsjElGhsjpkLtU4QQtR0OvPyr5/+7e82rF39HgKGkZERTE5NIRaLQcznoesq
    OErgsFnh8zhRVVGG6govfv67PynnAhMjAI4bHlZqrkLC7NcPPngfCOHBmFrd3z/08YmJqWpC
    ODAGxOMF56JFK7dpmnYmFJoOAZgihCgL8eFmGSdlbGzsnEF2PArA0tzcJHR2rv3I6Oj0mmIi
    3mTQGWZ7V6WwkANjnGDolbBQDnN4uB8TE2MHm5vrD+Ry8rWMUTgcLtjtAiSJgFIOVqsAi8WC
    cDiQHxkZHAYwaBSltAvdT/HUDwZCikRMRVExPT2DqqpKZLMypqfjb1xisbedXJJOHTp0CADE
    LVusB32+sv25nGwuK/PC6XTCYrEa3QACAA6aRqAoHDTNBF1PydlsNmakYnLzgboBWtqf//zi
    mb17b3qioqL8M4WCwnm9XrhcLthsVphMRcACKHSdQFEAWSaQZQq7XdPPnDkVlGXlJIDXDXDU
    56wSzo5HBwfP7fj4xx75p9Vdq9a++uphaEqRAAlNwV2331hfXe79uyd+9XtPMBT+CSFkyAjT
    FiQNGuFJ2Ag3exlj9rfddN1X33nbDUvtVjM0VS6Ck6aeB1gqCBhURWGMMXohwmqxFYNDKp3u
    /trX/+PrgUDgY4vbWlzHTp3B0EgA0elJ7Nx2GZiu4Td/fEERC9KYUUkbvRR2tclkgizLrpFA
    wNk/OAgKhmQyhXw+i7HxIDRVQWB0DAQazgXGKMdxNk3TLIwxoXTO0UJEwg8/8Hfa93/0+Ctf
    +vJXQ26Xo9pmtcDrdkLM53H81Gl0dSyBzWrCSGAc6UwGmWwOgyMTScO7ii9UBJmd4D/02uFf
    fP6f/vlyr9vVQimhPq8bTocdJqcdmq6CpxTZbBavT03htVweqUxGP/Dq8WkDrE4SQuZkcM++
    l+uvvx4VFdWIRMLLjh498Y9jYxPXu91umEw2gwhpRaFgWbxq1bYPNjSMW48dO/IbTdMnZrfK
    zLdOJc+NEJJljAV5ntAtW674+/7+sW2FgvIm4uibgYp/g5OlqjoMsipdSK/27t0LAJFt27Z/
    ZfXq5Q3JZH65y2WH0ynA5aqAICShKIDTacHAwNmoJMn9RkdD6mIoGp/85McRDEbgdNoqKirK
    BSABQSiCrc1mAWP0je4AQTBDEPKC1Wq1ZTJZM2OMo5Rqs+Q75xyTkyF84AP3nOjoaB/QNKwo
    goYOQnj4/W7kciokicHpNMFkMoFSgqGhmYwkyWHDGEoXig4A5PP5zO+WLWu7njGymuMEXpY1
    KAqDouhv0IMEgYfbXZxHkgosGOyPT01NnQRwjBAyPjvtw80lELvdDkVR6j/1yU/+6AP33tNl
    NQvgCIPdZoHdaobVYkKt34fN61Y4rGZh+UuHjyc1TQ8ZLQ3aRfq72mXruiSTyXTZB9/7jo8v
    W9Ri0TUFU1PTiCcScNktb/CydE0D01UQpqOn95z+x5e6RxljJwghA+d7ELNJfoblleLxRLC6
    qvKyttbmxpHxSUwGx6FrKpa0t0DgKQ4cOpoJRaLdAA4Y1U55doJ6LgWz2WxQFMXX2tpy7a23
    3vKZ22695fL62hrO7XKiyl+BupoquJw2uJx2uB02OB02NNZWUZvF5EmkMlZJkmWj7UCe75wv
    QghePdwNsVBIVPorOj70gbs79l6zAxvXrsS5oQBS6TSa6muwf991WN3ZjiUt9Th45KR8emCk
    F8ArhJDXS97iAsxqUEqZKIrR8Ylg9uYbdl9x897dpkQyhSu2bEAml4coSti1/TJMR6LYsXkt
    avw+fP/J3yZD4Vg3gOcJIccAxC4EKjfccAN0XfN3dx/9wbFjJ3dRKhCHwwWTyfLGQxQZEUWh
    vKqqdgUgRqPRaOlUCO3iInPCHnnkY9qWLZv9o6PT/zg9Ha/hOB5mswU+nweKor+Jk1V6tljM
    SKeno1NTwVMASlSKNzfectxsENDHxkbDnZ0dZfm8skUQQLxeL7FabchkJBQKGihV1IGBU335
    fP5FAN2G9zdv7+Vdd92FlStXEoC05/PZ/f39Qx8bH5+skySFyLKGbLYASdKgaQyqCqgqQbHp
    2WZqaWn21tfX8eXlvtzkZDCLWQdgziWLZ575HQ4dOpTt6FhRHQhEto6MhBGJZGGzWeFw2KAo
    DGNjWSSTOvJ5CqCgnTr14nA2mztk9L/GF2pdK807OjqaqawsnwRM28+dm3J6vR5Eo1kQwqGp
    qRYDA2H4fC4kEgpSKQ35fDD/2msvnFJV7XkAhwkhodldJwuFVLymaUJoagpgKpwOOzSFhypT
    aDJBPJmCIuWRSmfMjLF2ANWX2vv1p1/9EPveff/G9uYGN4zewdfP9mFxSwMoGHSmA0wHgQaw
    YqVvYGRM0XQ9Y5xuoMx17f6KCuTFfHNtbe1GMEjJVJKc7e3Pz8QSckNdjcks8ExRNRw+dpJk
    sjlksjk4HTa33WpZ47BZG2biyclUJnecEJI8nzE8u1q0d+/eB973vrs/nkmnhWQiiXgsOgtk
    jWeDEFt63bGk1XP5hpU3HTlxesPXf/jLzzLGXjBIf8oCS5XWNPUsx1HEEwkk4nG4nHYkU2nY
    bRYMjozCYTXBZjFBVpSCQRANAkgvZNFL/2Z4eul0JnOI6fo0AWtNJpMIjI4DulYMM0fHEU8k
    oaoKmK6xZCozYzRcv260gWgX8hyKrSEFdzabbdJ1hlxOhCwnYDLZYDKpMJkYzGa7wY0iZbqu
    rwHQY9yLfCkhuigq9bmc1Fg6pcHlcqB4xAw/y6sq/c3DZKIskYiW2rLE8zcjIQQPPfQQFEWx
    i2LhykwmY8tmszmeF6xtbbWZTCbj1jTCJEmC222B3W4mspxjVVV15sbG1ra2tiU32u3WzOHD
    B4+cPXu2dCrEmyqgfr8fNput/KWXXvrexERws9PpJh6P+w1vilLBAFnTG6RY4zXleWcLpVUP
    eDypFT09J74oy2qPcUrHnKDS19cPxljhjjvuPM0Ykc1mk8lmc4CxYm9iLqfB4bCAsWJoWChk
    lGw2GwIQPK8fdUEvS9d1MZWKH/P7fUMAqTGZBDidNug6QSSShNNphdksgOcZZBlIp+MpRVF7
    58uJzglY+XweAKb+5ctffvQHP/jBXk1TGxjTLWCMMMaKSV9dY4qqslg8EVdVbdAAEP1SlOor
    //F9TIbCowdf7U7VVlU4VFUh0Vic2CwCQtPT0HUNTNOKxFFdY7FEEn9+9WTE4GBF5lNis9kM
    SZJafGVl99rtNn9F3sdV+ct96ZxIT585GwuFpmc4jqar/b6K1qZ6j9UkkMDE5DoKdCqqUogl
    068AmDFCnDkrhoQQ7NixQx8cPBeTCqJFlgoCYzqYAbzMeOi6xkpN45qqQNNUBKcoORcIMgD1
    KJ7sObFQCHLdrh360eOnev7h81+aFvNipc502CxmousafvvsAfzitzIURYHHaWdn+kfixveF
    56vinD+Mk1N1AGO/++Pzx0+9fro1mUyj+/hJ0OIBp9BUFaqqord/EKlMVsnm8pMARgghobmY
    1XONY8eOIZvNTlZVVX7VbrffIMtaJSHERAhHislvBkJERkiBTU9H46OjgclLOQWkNFKpJDKZ
    zJTP5zzO89x6xiBYLGbKGFBe7jCO66VGwr1IbszlIsnJyeCgsXbZBeThVBRlnyyrq61Wu6Cq
    ui2Xyzs0Tc2ePn0ynsulI9XVLXaLxVNhMplMHR1rmwRBqCSEkxRFmtA0LW9EBYXz90uhUICq
    qkQQBNFqtUYBmCklHFDku5VOZwAYGNMYYyp0XYGuy9A0HpTyJB6ftsuy2mhU1mYWWqfvfOd7
    IIQNdHQ0DptM5sU8zxNNK9I1KioEVFYWm60VhWF0NJmWJCloUJjyF9KrEmjpuq67XKbIsmVr
    XvN6PZstFhPn9TqMCicHv98FQigcDhPyeVELBmPTAEpH4+TO99rJfAQzoyXCB6AVQKPRckHm
    CO1EwwKOGLkp9UKKe96octhtN3Ict44xVkUp4Y1TMgEGAjBWjHXBFFVVC5I8BuBVEHKIFEvf
    b0pizkJ9G4BFlJClhNIKSukqs9ncmM1mA8aijxkhsR+AswhMLG1U16JGiT5Qcn3Pvx+O46Bp
    WiOAywB0oPhjDfOWe+b4Ow/gLICjRhe/vEACE0Zbxx4A26wWS6PdZrUJAsczVqwipbN5Xde1
    AmMYAPAigJcNobOL+QUY4zOc0+HYoGnauymly3w+r8dutQgEIAVJQiKZ0rN5UWOMRRRFPQTg
    OULI8dIGX2ie2XMQQio5jlvGGGsrMpiL8p5V8NEZQx7FY2z6jftQLkavSm1liqKYlixZvJox
    XC6KhTbGdFuxklaqppX2QXEkk4lENpt73QhDBkuRwmy9Mr6Xq62trec4rkPTtHqfz1fT3t6+
    7eWXXx6PRCIzAIZNJiHvdrt9lHI+Y/EzqqpmcrlcSpKkCQBDBi3gTXulsrISuq4LVVVVKzOZ
    zDpRFBcxpluLn3mLSrH/Wi+w0nv5vDgjiuIJI2ybmk/+hBCjCX3Gvn79+qs4jrtaUdQ2xoiD
    EMoTQqmRGtQlSVSGhoZG4/H4S0bD/tDFFA9myZysXLlqSVNT8926rq0uFKQKXScCIZRSyhGO
    gw5o2tRUKNnbe6ZbUZTnCCGvMcbib2k+X2giQghnsL7NhjdG5th8qiFcqRQWXAxYzbLIZgAV
    hrdRft485/98kmY09gYNy/6WSsUswOUBuBhjZUZDsweA27jeKUOYMM5KdxqfZ8YcosFheiMk
    PL99xlgfiwHq5cbpAhcCrLnuJWrkaPQLAJZgzNNQaoEx3qOzKlpKCWxLFupiN3npT2M9GgG0
    G3IxzSpwlNYnbYDJ0KUYqVmyEgyDYp9VkXvTGhnVwTyA/Gwe2SWALzXk6jfuw7wA/61keKOE
    kHCpeDSPISTGdXsBeDiOc5hMJp8oitTQmXHjGGuroW/mUkRsGKWUwR/MGUcMn39PxLjuCmMO
    /iJ0avZ7kpFPjDHGxIUKOiUDYszTgOKvIzkMmcyWuWrk30aNNpz0xRjC8+ayGXu8lD6ynDeH
    bujrBIBzBnn3LUb8/9lPnsxWQEIIbwiGx8JHipQ2jGLEtvP14s2Oo0tVH8HYfMwAV2kWL4qe
    B46sZH3n2ijnzUEXIoJe4F4WnGeOuTjjHkoKRc+bVzdAS2KMKZjnrLALbXRCiMlQqNlgRWZd
    s2KENIXZOYZLOcr4UgiPf4VuEUM+HC58VI2O4u8FvOnHNRaQOzGikJJMqAGyhSKv+Q2+AT1P
    5m/5mbTzQXEWp4r+BVurpFM6uQDN6DwdLukVP4desfP0Sr9UvTLuqaRX5nn0Sp2lV8pc1/9/
    ARDQaw8YszwHAAAAAElFTkSuQmCC
  }
  set pieceImageData(Fantasy,30) {
    iVBORw0KGgoAAAANSUhEUgAAAWgAAAAeCAYAAAAB6YS9AAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAAFoAAAAHgCDQwRVAABJ0ElEQVR42u29Z3hc
    13Uu/O5T5kwDZjDovREg2HunSJFiEUmJsiSrWnIkR7JsSe6WFNuJncTJzb1Jbm6K832x48hW
    5KJIluUiWqasQom9gwUdRMdg+mD6nLb392P2UBDFAlDWd5PnyXme84BEmX3OPmu/e613vWsd
    gv8+/vv47+MjPQgh1/yVq/yMXfabjM10bHK5z5zu5/z38X/nEP6zG/aKFStACIEgCBeNjRBy
    8fzv46Od/8udH9U4/BAByIQQmRAikdzxX/55E0LAGLvcKVwCnJeev6+DTTn+26Y/YpvOH2vW
    rBZqaqrkvE0DEGYyrviffUK/+93v4ic/+YkVwDzGmElyd0UAmNP0Tj7cDiYIfBgi8LEEPskC
    IYSIosimXseHfeAulwtWq/Xiqarq/++eHv9qA1AOQCSEEMYYCCH0oxy/qqpq5+pVq77kKXJv
    9vsDGqU0SwgxARj/1UHh3nvvRSaTIcuWLZMGBgbwxBNPrFm4cGHz6dOnYwBkfo/sowamS52z
    G25YSMrLXaLXGxb499iVAO2/cuRy5523yGvXri23253KyMgIOI58JPMtiiJ+8IMfoK6ubqmi
    KE/V1tbfWl/fYO3t7U1we9anO7b4n3Vi161bh2g0uuFXv/rVcl3X3Zs2bfo3SZIKw+HwBDck
    kwO1QAgRALDflzERQlBWWgyLxdJaV135SDqT0XTdyAqCULZg7uynBYFUJJKpACGksLKycrWi
    KA2pVEonhAh8E6EzvY7876uqCk3TkM1mLaqqlgAoBVAMwMk3ChBCGCGEiaL4oRfQ1L/nIAzG
    mPuGNSv+V5Gr8LP+YGgAgEoI0QBoH4Unkz+WLVv2xP/6q//xpGKRFrzx5lvjAEIAkoSQTN6g
    /yuCBSEEd9xxB+rr6+tcLtc3mpqaBFVVH0+n03/ocDgGvV4vbW1tlVeuXKlkMmkxkUjkvYLf
    m01/6UtfwsqVy60rVqz95IIFC+f29HQGAUg33/zxO8vLm+/R9ex4IBAgDz74YNuKFStWulwu
    YWhoKP/nxoe5htbWVnL//fcWLl++tLS1tbWkoaGuSBAESygUzv8K/X3Y0SUOBggh+MIXvoAN
    G1ZKouj+MiHWbzNmxLq7u6P8ntSPYjOQZRm1tbXQNHXHwYOH/zISiS9TFDk5ODgwDiAFIJ13
    dq41r9J/VqP+l3/5Fzz00EObd+7c+cUf//jH7zQ1NTUB+PTg4KCDMSbPnt3qVizyZEdH115N
    17sJIVEASb47XQSc6z0KnE5omraopbnh675ASEcmmwHgriwruT+VSv8OwEBDQ8N9ra2tdzPG
    jLNnz/6bz+d7hRAyxhgL853yig9TEASYpikAcAGotFqttZRSU1XVLsZYsq625q6d27c8XuR2
    KYauk9NnzwXfeHv//8MYOwsg7nYVrDIMU0+m0mMcyGLc4PT8w7/S/T/y8IP4wb//pB5AqWma
    44SQNIAsIUQTBYEZptlQVlp8u5rN9gIo4p70ZP4z89dPKb34/5lymlN+rwRALQB9fHwsFQ4F
    cODgoTiAFlmSFJPSNyml7YSQFGMs+1F78R/VYZomRFF0dnV13S+K4m0ej0fOZrP19fX1X2pu
    bh5yu901iURSWb9+45GOjnOvnj9/vp/bUZpv+h8yEhRhsYiOcFh7XNNYRBDIIGMspOt0k2Gw
    G9Pp9MENGzYsB/C1ZDJZUV5efqKqqupvvV5vPyFkAkDmWmN89rOfVQRBKDFNs9o0acXg4IWR
    N954c6yxscEWj6t/5fdHFgNEsFhcdNGi5QfHxkZ+mExmxj/xiQfcHk/JgkgkON7Vdc7X2dkV
    yWb1FLdnM785X862GGP46U9/ikOHDpcArKmzs9P/5ptvxgCoANSHHvoD+sILP/YYhnpHPB4v
    On78mAygmmPFB2x2qqMyU3v2er345je/6XQ6nU2RSIQJggCbzZktLCygkUioeNmy5bdaLPKh
    w4cPH2CMxQFkroYTH8aDlgBY+d8L+XDh97Hb5/9W0zT09fU13XPPPXfU1ta2trW1WSilztbW
    lvler3eeadK6ttbmNVUVpTcNDo85GGMTHGSMvEFf77UQQpBMJpHOZJ2BYHh1LJ44xxg0xpgW
    mYzNCUWiA4Ig1lVWVj7Q09PziiRJdrfbPcfr9XbxSY9dKyxnjFna5rR9+eMfv/NP773nnifu
    vfvjD6WSyW0et3OB1aqsLCgoWPWX3/r6xl3bbypdv2Z5aSwer4qEAm2tjTVrdcNYt/XGdV94
    7JP33N9UX7NTsUhrA6GIqBtGlBBi5D3dy907IQTuwgJkMtlP7Ny64XuTk3ESTyS93ItJU8Y8
    d95+29+0tbYuP3zsxEgsnohw8J8AkJEkCYyxegANjDGVECIxxuhUj2AmU33DDTd886tf/erf
    bdly012lxcVL6uuqC7LppLJz68bZN29et35waNQajsYGAWh8I/koAZrkQ//fd2i/dOlSpNNp
    MR6P756YmJgXCARchmGQaDRapSjWWaIoOtLpjF0Q5Bubm5sWMcbCPp8vyu/bmLq+rudwu10Y
    Hx9jtbU1K+PxaObUqVMjlDK6dOmSYkJMR1dX5+jSpUs/o2laMhwO77NarVtUVZ3w+XwhQkiC
    e36XPf70T/8Uy5cvnxMIBP9mcHDwywMDw4/F48n76usbV8yaNXely1U4f2IitOPs2f6WQCBW
    GotlympqqhqamxcuWbhwxVqXq2D38HDii9msfFtFRf3WuXPnzUmlYuFoNJrlkZt59bldBsPQ
    V4XD2RecTldjX19nn2EYAJB2OKyi3x/5+sRE6HZBoMGOjjNeXTfShBAvYyzxxS9+EStXrihe
    vnzF/PHxcTWRSEh8jmdkz8XFxTAMA9ls9r729vYfer3e+1VVX5/NmkU2m1O220sb7faK1eXl
    5XWjo/09mqZphJDUtXBiph50QV1d7c4F8+dtddhsnqHh4Ymz5zveyWbVU4SQMGMswY3pw2Yz
    8P3vfx+NjY22dDqDRx55RCwocOLWW3bBZrMWnj93jv70hReiL//8laimqoIkSZsdDrslEU+8
    bua8rUnG2MXdaSbetMVigaZp0HTDsnrVql233rqrbc+e1z5+6PDhvQvmz2t95FMPrdvz2m/L
    Xv/dm6f7+/v3UkojRUVFFX6/PwqgAoB3mhufuHTJko3f+uY315cWe5BKJtDT1eFort1Su//Q
    seyp891hxWJhnNNAgcMu7956w6Km2vJFr755EGUlxbh1+43YtnFV08kz55s+9/X/qWeyqp8x
    ZlztwTPG0NXTh3gi0RVPpGRJllcB6AXQCSAGoGLF8qWr79y1VTh4+FDj8Oh4SZ5SYowVUkpX
    rl21/IvuQufs3/xu318yxo5w8I5fB6cnlpWV1W3durVUzWYwOjwAT6GTtbU0S26nIscmo8Qw
    jQoAzdyDD/y+kPiZZ54GwCRNMxfG44n1mqaXUGr6LlzoPXXkyNFhxlicUyv0w0Rj+b9tbm5G
    b29vtaqqRaOjo4jH40SSZDAGsbOz21FRUak0NTXGHA5bMJk0C2fPbvtURUVl25Ejh1+ORqPD
    3KazedpjJiqO3bt3o6GhEYzRpefPn1ttGEbBkiVLYpIkRQcHR+7Rdc1TV1cXHBgYGIrFYj01
    NTXLTdMUYrGYm1Nsw3zzYldZs6VdXV27z5/vKgQENDY2wu2uWp1OG6tdrgI/YxFb/pIpZTAM
    qVjXnesZkyHLQDQaRDxuFFut1uLaWnujIJBh7gVn+NcrHk6nA3193cOSVJDQdbqksLBweSaT
    PQ8grigKmZxM3NDdPWyZM6epyuFwVqfTmW7GGL3//vsVQshCXcfDyaRx2/r1G77z4osvvMoY
    myCERK61MVy6rnRdh6qqlSMjI5XpdAaplAZKZWaxGEySqNViEYiqqkW6rrcCiALwA8h+aA9a
    EAQwxgpuumnzX379maf//JGHH1y5Y9vmOTeuW7XCVViwsbO7V8xksiFCiMq9KvphOFGPx4NM
    JtP2xBNP/PVjj326wuPxQBYF2G1WSAJBTVUV2XjDOvvsllnOEydPYzIWK7z7jt3rZUlsmfD5
    TcZYjF+LTgiZ9spyOBzIZrPYu/c1vPvu/o8/88xT337s0UcL57a1VnV2dbd95pGHl//hww/a
    Kyoqyva9sz+jqmp27dq1d8TjcbWjo+MdxtgIgHFCSCBPtVztmdbW1m24ZdeuZQUFTlBqQtd1
    ZDUDDY0N4m27dhTU1lQR09ChayrcrkLoJpDIaPB4SrB2xRKUlxZBJMDQqJf97NU3opqm+wB4
    CSGhq3maiUQSL7344p2ff/LxW1ub6qvfPXBEM03TRwgJAoj6/f4Fb+8/2HD85JnTmq4PAugB
    4F23ds2XnvnS5/5R07Kzj59q90Yn434OmtGpPPFM9uLJyUnlnXfesX3ve/869MYbb7BkMuH6
    i7/+PyO/2PM73+/2HRwOhCLnGcMYgAlCSHgmi+ZKx7e//eeglMrBYOSzZ892fqejo/fu0dHg
    xmRS315VVbOmuNiljo6OxnkkouZB8cPQdRMTE87BwcH/ceTIkQ2xWAyECCBEgCCIAAiSyZQQ
    DkftimKzuVyFkmHQKkmyrpw3b251Op2KRCKROA/5jZlGKrfffjvsdlt1Z2f3dw4fPrrc6w0U
    NDU1tRQXly87fbqjJhAI26qrq4pHRob6W1tbb3A4HAvGx8ePdHZ2nudg5Z1KCVy6Xrds2YJ4
    PO7wen13T0z4CkVRAiEiJIlA09IIBMIOny+iUAoiSTIEQYSqaiBEBaUZRKNJRCIZMCZAECTY
    bIbe03Pan82qvquNnR9/zZq1yGa11R0d3Z8KhZLV5eUlyvDwoBdA4NChg5F169ZV1NSUr/b5
    Rs739/dcYIz1Ariwdu26XefP9z+vaVgP6KHBwX6/1+v1AYgQQpIzidZUVcW+ffvw8ssvk+Li
    YndBgXOsqMidLC+vKjGMuC8UGhknJOUdHu47HwwGh7g9B66V15mWB/2pTz2Mt9/ed9vnnnj8
    kV03b7PpahqGlkWBrR6P/+EDZaFg8JP/+u8vxDm1kOHe1HV7G5FIhGzcuPGT991333yn0wlG
    DVDTAKUmGDVBqQlJINh18zaxoqzE9YWv/NFk+5lzqaJCx5ICpyMdiydHuQedwRS1x7W8jkwm
    A6fTie3bd8z6o2ee+dptu3e7GTOxYvlS8Ztff6ZqeHgQmWQCa1cuE++/5+NL33znwMJoNOo7
    ffr0qwBGAIwCCE4DnAHAHBgY6H9n3z5TUSzi5GQUhq7DMHRQ0yCTkzF0dXeDUQpGczQcAQNj
    FBaLBf2DwxgZG0dtVRn2HzllJlMZAsCCSzS1UxMm/P4JIYQtmD/fyGYzZjqdFhhjhQA8jDEb
    gKTD4SBeX2A8mUp3AYjk+We32zVvzcplzpHhwfTg8NgQNy6ZAzO7lI+ezvOemJh4fWJiYgjA
    /FlNjY9NRqMiAdLpTKZ3imc/kPdoGGN5h+G6wNLj8YAxgng8eVN7+9lvdXb2F0uSFRaLA+k0
    k9JpLJg1q/bJiooK0+fzHeAhdvLDbAjpdBrJZHJ1d3f3rsnJSQA5YBZFCYIgXjx13UBv7wXZ
    NKmzpqYiQYgkSZK8glLzFN8IU3kHaLpjf+5zn0Nzc6O4f//BJ9rbz2xMpTKQJAUdHX3u+fPn
    wW4vRDQahdcbrlyxYtVO09Tlnp6e186dO3eWOxxeTm9cccJDoRCi0Wi0rKx0tKmpsaagwAVZ
    tkAURQiCBFEUSU1N+cX7JSR/EjBmAhBRWVkIwyBgTISu+/VUKmW5HD5dyg0zxsif/Mkfs8cf
    f5xKkmQIgkx0Pe3IJ9YpBUyTMsNQY11d586YJg1z71UzDLMlHE6UUWo14/Hh0fb20zG+hmYs
    RaQ0ty4BnBFF8S8opW2bN2+5v6CgYGEqFdH7+noHKaX93J77AfgZY9fEiWl50KdOnRZu3Ljx
    c4986uGVNqsMQ9NADQ2GrsEiEmiaanvtjXdUwzDHCSF+AInrBWi++CoeeOCBb968fVtlZ+d5
    jI+NwVPkhkAAahpgeaA2DVRWlJK21mbrz37x69SpMx3p6qoqJ0+2TRBC4hyopzXbjDFomiZu
    2rTpj5766lfuqK6qhJkDTNTUVKKzowuJeAyN9bWITsblnv5B8eTJk+dN0+wH0E8I6SGE+Hgo
    es3xQqFQoP3MGdfcOW3zNt+4Qayvq0FjfS0a6+vQ1FCH5sZ6NDfUoqmhDk31NWiorUZ9dSWq
    K8tQVuyBYRj4++89b/xy7z6vaZqnAZwkhPRxQJ2aWCGyLG0jBCWU0iQAKRAIjB4+cqzs5Jnz
    FbFY/DSACQBDAGy37971FVXVtIGh4RMAxgB0i6KgpJKJB15/ax/ePXi0L53JjADoBtAFIDwl
    FyHkaYH8yaWKU3neAp54rOBJUrWwsOCG9WtX3VZZWiQx03COev0XALQDGOdhYP6zCWOMXK8m
    fs2aNfjbv/02eemlX3zh5Mn2G00TkGUrZFmBLFtBqQibzeESRTXj9/u8PKpIXq89f+xjH8Ou
    Xbtw8ODBz5w7d26zx1OM+voGGIYJXTc4UOfPHIAlEmnRZrOJjGkxq9VqKyx0VyaTifFEIhHm
    np0+HS+aEIINGzYiGAytPXHi5F9duDBYIIoyJMkC0wRMk6K+vhbxeBaMgVRUeOyRSGjixImT
    5yilQwB6CCFDnO6hVxpjaGgIhw8f0ltbW4NVVVXzR0bGy8LhSSQSKSQSKcTjGSQSGf41i0Qi
    i1RKQyqlI5MxkM2a0HXA7bYyTRtNnzp1uDeZTJ0AcIYQMjaV4rDZbHj22R+iubl52eLFSxYd
    P348/Gd/9mey1+sNLF68gFRWeuafPHm8PR6PewEM79y5LUGI8mQ8nqyemBg7nMlkxgB0r1+/
    NtnU1HK3211QGY16ezo6zo3out4PoEMQyMSCBbNZIpEium4IU1RZZOqcf+MbX0c0GsXOnTvt
    y5cvL503b15FZWWlJxKJmBs3bmwuLa16SNNgr6ioc1osxO/1jp2sq6sdmjNnTrKxsYG4XE4h
    FosR06Qkr9q5VNkxbQ7aNA1CqQFmvufF5oHSNAyBMVYMoIwxOAi5KIObcXJw3rx58Pt8yxcv
    Wjh7z549OHr0KFavWoE5s2eBQvjA2Mw0sWLpYvKlJx9zP/3Hfx5atWxxTWd3z+oz5+OnGGN+
    QsjkdK6FEIKSkhJEo9HFd9xx+71z2mbDNHUQAjAwCABu2rQBL738CirLSnDmfAfC4RBTVVXh
    yolxDiap/ERfaRee4gV4BwYGvv/LX/16UW9v7+Ky0lJSVOTmmxDN3SejF++TUhNgFJlMGqNj
    40gkU/R4e8eEpumnAZzkHmf0MqGZc92q5c9kMpnEkRPtf8NVHxGLYj0zd+6cnVxzrANgRW53
    67w5bXUDg0MDHOSTANQlC+Z+1u0qnPfmu4d/yhhTAYxIkph0Op2rXQXOWwudjvqOnv6fUEoH
    +CbNOKAaeU/HYrG0LVy48I4FCxasr6qsqLFYLLZEIq6dOHFyUCBovOeuO5XSYg9mtcy2TQQn
    F9ZUV9YsXjC3wGG3yslkKjs65p3o6O5tHxgefZtS1kUIiTDGUvm8x3SiJMMw8NJLrwi6rrso
    BffwJAiCdPHflBKRMdTwbH8BAN910DdgjGHNmjU4dOhgxcSEb7OiWNHW1gZN0zE25v2ABy1J
    EnKKUYLx8YClqanGGQ5HjIIC9yyn0zkLwAXuSaencz1PPfUU6uvrlLff3vdoX9+FCkAAITlK
    RRBERKMJlJVpqKysgCwTKIoMTTME0zRlACFCyDBjLHItpYHP5wMAfc+ePQduueWW5+bNa/um
    aVLn5GT84ljvndKUrzkqxOGwQ5YlJJOh7NGj+/sSieQJAGd4RPo+9ciuXbvwwAM6DhxQH8pk
    jPWLFy/8o/b2s2NerzcYDodOOhxOeDzFGBsby+RoxIbKgQH/PMPQCXcUMgASixev2hGNZj7W
    29v1m8OHD/oAhOvq6sZWr16z0O0uWsGYvKyysv8Xe/e+1s4Y83L7khlj2fci7mz1ypUrbwuF
    Qlvi8WSjYZgFkiSZa9asHy0qKrL39w8XT04m4XK5perq2tbt20sUSbIXMCYooijq5eUtwXnz
    VnaEQuPvHD586HQqlQ4SQpJ5apYQMj2AZozR2traQ4cOHrr/tltutgIMoBQEDIlEEnvf2q+p
    qpbhwHDdFVCMMZw7d86ye/etty9busQhSSIUi4QlixZCFkUYhp4L9xlFLBpFV08Pli9eCDCK
    3Tu2iK+/8XbB2XPnNU3XKgA0ABjkHmD2ags4//1gMChs3br1/q1bttQIAoFpMgxcGACYidrq
    SpSWeNDS3IjvP/cTZHSK23bcJI8MDzvDkSgFkOHhJ5uOPIfTAVnGWM9bb+/79oGDhx78/r9+
    d/fOm7eL2UwasihAEgmoqYMaOgxdg6GroIaBcx2d+Mfv/Xs2OhmfAHAawH4Axwghg4yx5GU8
    KyOeTA3X19Xes8FqKzp85Ph/6IaxjwCZZUuXOtrb290+n58CkOvrapc0N9XbRVGwASgEQGxW
    ZeWN61c9dvpsp48xFuIAP3z3x+985tabt6x78Wcv02AgkBZFIU0p/SmneRgAiTFmMMZQXV29
    86GHHvrrO++8c05jYwOxKRYQMICaOHLkyJyOzg7sunkbTENHU30tCQSDzffccWtzeYkHjBrQ
    VBWJZGLewMDQ5l/vfeuu//jlb38cnYz/mnt4YULItOSVHR0deOyxJ81PfvIPzpSWlt3v80VE
    QqSLwCFJFtjtxOjr8xvcgRFw9VLsq8nOYJoUgUDwprGxsXmBQBC6bsJiUaBp+iUALaC2tgaT
    k3Fksxo0zcDkZNpRU1OStFoVoaamdnZvb+9pHnGEMY3iHavVisHBoWX9/Rd2JJNJiKICRVFQ
    XFyCWCwNQkRMTIQwf/5syLKJWEyD3e4usNvtYjqd1qbKwK40p1NoLcoYi585c+bX8+fPh9Pp
    fPTYse4WXacQRQsYEyCKMkRRgSRZIEkKZFmBJCmYNauUjo72x/r7LwwkEsnDAA4SQs4C8HGw
    ujhePB4HIQ/h0Uc/O2KaRmtj45z/qSiOV44ePbyHMWZEozHB4ykuI4RQxphomqwtFktW2GyW
    LCFwAxDXr19XFw6nv5LJZMyBgb4xAEGr1dqzbNmKewcG/Pc1NdlhtRK9oKDQQQiijCHAIwjF
    YrFkT5w4gWeffXZRd3f3/zl//vwNwWBYMgwKQkQIgoyioqLWOXNmo7d3CICASCSJwkJnRSCQ
    qZicDEMQclGMxaK0OZ329aWldXdt2VL0yr59e1+IxeK9PGrLAtMo9c6HqGNjY3v/9u/+/oXn
    f/zT7PDwMILBIM51dOEv/vc/01d+84afc7A+QpCcKUDnF1VLyyzY7fZVu3bs2FVTXYXyslLs
    3rUDVZXlMAwNjBoXqQ3D0KFms6DUgGkacNqtePCe262jXp8xMu5TREFo4BrbAsbYNakcRVEA
    oHHTphtvaWxsgK5pSKdTePa557Hntb2gpglqmmid1YRAJIaaihLcvnMLWpobivgYdkKINMOI
    weT67f02m+0tm81mvPnW23jp5Z+jr78fVsUCWZIgyxIUxQKb1Qqnwwanww5RFBOcYjgK4Agh
    pJcxNnkFbyfj9QVPPfTJB61/+rWvbCgrK7kVQEsimUBjfR2ZP29eG+fsymurK5dUV5RDsVjs
    HKBdN65f/Zn5bS1VPf0DeU9yDEDM6bDXNjfUSZlsNnrg2Onf6boRAmBjjOVXFLFarQBQ/8AD
    D3z7qaeemrtk8WJS4HBAFAjC4RDaz5xBOBJBNDqJbCYD0zQQDIWQyWTRPzCEfQcOsa6efsYY
    RaHDjsXzZwtfeuzB+k/d+7HPSpK4mzHWTAhxXVIyfcUjHM4VR4yODu9pa5v1VnV1JbVYFEiS
    DJvNhqqqQub394d9volxvtGkZirtywPK/PnzoarZsqGh4UcmJvxWQkTE4wlEItEPeM+CIEHX
    DQACRDHnycdiSUFVTYVSzbTZbDWFhYU1PFK1XjrWpePff//92LDhBuL1eu8YGRktIyRHoVRV
    VWD58gWw2+0QBBHZrAFNy0IUCXy+NAxDdpSVlRUBcABQZsLFEkLU0dHR4YMHD/wqFosP2O12
    tLW1orm5AZIkwTQZKM2dpklhGLkznc7S3t4+XygUagdwCMApbmPpSx2e119/HZOTk9D1zOmh
    oQHj3LnBxR5P+R2CQOZHo1Elk8nqlLKWkpKScgCeVCqzJJnM2hmDRRBENwClqqrxE4FAckU2
    G/eFQiEKIFhdXTVus9lr0mnNEo8ntKNH33rnt7/d088Y7FNv8VOf+hR+9rOfOYeGhv744MGD
    m0ZGRiRVVcEYoCg2lJWVssLCQsiyDEVRIAgSFEXJrV+bBZWVxaykpJDl5gNIJnUyOpoq1nX3
    g0uXrnwAwBzGWDFjTJoWxcEnRpYkqWrMOxH+1x88Hzlx8nRVkbsQvX39aD/XYfJyZwogy3IE
    ocTD/mmJvfMA3dfXX3jP3Xd9fvu2LeWMmjDzfPMltAoYhaZmkYjHcz9jJgxKsWzRXGHZ4vnW
    yXjSJoniyuOnzpxgjJXy7H/qSh4WYwzbtm1DV1fX+jWr1zTJoogf/8d/4PTpdgiShGB4En/9
    99/B3JYmWK0KTMrQUFWCsmIX5rQ0u44cP13OAc46HUF/fnx+LSaAiCzLIavVinlts5FoaQZA
    EY5EUei0AYxdPHP9Gyi4JngYQCchpH8qOE/9/Pz9JhKJjld/81qPqWstkeikAqA2Hk8Qq1XR
    Nm28oeWNN99qBRCprKyYXeR2wVVYYAfgdNhtq2/ftWVTOp3RJvyhvFIjW1lW/PGfv/JL8403
    3z445p2I8WSSnyeI85GU2dLSgmg0unTz5s1zXS4XTEMDQHHs2HH2tW/8sdnU3CzesHY16evr
    RyAQQFVlGTq7e1g4ngGxFpKfvfxrvLP/gPZHn39UvvOWrQI1DdgUC27bvqFozxvvbuvuH8qH
    4XFMo9Lxpps2QxAEwhjLiCJ7a9WqBXPicbVG00y43YWIRieM/v5A0mazR9PpdJQxpk2pEJ1W
    cpIxhn/6p39CNpsi4+PeT/b09K7TdeN9oJxLoL3/9HiKYBgRmCbjVAuQSGRlp9PKHA5XW0tL
    y5yTJ092cvB6XwL8ksQZ6urqsH//gYrxce+WeDyB4uJSLFq0AIQA0WgEc+Y0ABAxPh6FLAOM
    KVBVAZomykVFxRVDQ0MlAAq50uCqUsNLksPZpqYmn6Io6VQqjZGRMUiSwh0gc0rOQJiaP2CC
    IIQB9BFCugCM8dzRZfEjm80iFAqNlJQUn/R4ym/IZicJpawmEokMlZVVp6PReG1tbe2CYDA4
    mk6ri7JZHaYJi8VicTU1NbakUuR2VdVJKhVKmCYVCEF848YtNxUWFtY1NcUPDQ31Zvr6esco
    ZT6ez8nbs+7xeKBpWv3IyMjanBonl/B0u91s3rw51O12kbGxCeJ0OlBU5EYgEENBQQGzWoFE
    wkuqqmohikwrLHSKgYAhASIIERCJGEpZWcUmh8Pem0qlo5yaNaSrgabb7UY0Gi3fvGnTF3ff
    essnVixfWlVbXSnaFDkXdmYziEbC8sDAYN3b+w/u/vVv37RcGBxRGWNdU9z0q+o2CSGglAKA
    MHdO28MPPnD/LVUV5RfpDEYpGDNB6XvJQYEQdHX3oLu3D1tvXAfKAdxps2LrxnXisz9+SV88
    b3Zt+1lptW4YQzxUyl6NS/v1r38t7N69e+2sWc2SSU3U1lSjv78PFsWORCIOQ8+t/aMnz8Jp
    U+D3B2FqKqorS63cUy9Crn9FdCaUDiEEs2Y100QiqWmahmPHT+B0+2lUVZRj0YK5WLpwfs46
    ODgTxsCnUudgOIlcObRxOWPOj5FMJk9979+ee7qmpuZbdXX1tT09PSWGYUQikYi6fetNxT94
    7vml/RcG4hVlJZU2qwWV5aUKgIq1K5euXL9qmeP7z7+YNE2TAdDrqst3zGlp/OSh42dfHhga
    HuVA0Q+glxAyMqX6i9bX10PTNLfL5ZIBBkYp1GwG//id7+gHDx1Of+HJz7oS8Zh58MhRPRQO
    K9WVpSQYDOK3e/dqN2/ZKN/9sZ3kF7/aE3nuhVeUG9cs83hcTpimAVeBAx53YV4j3cUjuKsC
    9Le+9S2EQiGbqqp3jY6OPzowMLogmex1miYBITKsVhtsNofU2Li0fv58aVsiEYqfPHkslE6n
    KV8w2rWcDkIInnnmGTQ2NuHNN9+4sbOz6wsTE3455xWLF/nuXGT6Hjg7HA5UVJQhGJx8H5An
    k6qg6yYURSFz5ixY5/f7zo2NjQ9zVUvqStcgiiJisVhbIBBsopRB0wwkk2nY7VZkMikUFZUi
    mdRRXFwAWRYgihY4HE5QyojN5nBzVU8hd7a06dryjh072Lx5bdmengHqdDpQV1cHRbEhHk9j
    fDwEXafvA+cce0QYV1zFuD2nrzTPufUyC8lkcrStre0bK1aseDqVsq+pqqoqj0QiE7W1jZlQ
    aFJqa2taUlpaMqiqWnPOU2eS01ngqaio2RgMpisdDsUIBIIGAGPDhs0LRNH1xODg0IlXX/3l
    Ua5MGgLQTQgZBGDkKUlRFJHJZAqy2awtdz256KeuroZOTobjyWSmIB5XiWGoWZvNaiUkQSwW
    GYlETLdYbOLYWEx0Oo0kwLTCwqaKRMIghIhgTABjUpEkSbMAnOf5hpRwtQmPRqNKRUXFN9au
    Xful5ubmqkgkipOnTtODh4+yQ0eO4dCR4+x8ZzdLZzJYPH+O58a1K+92OuxfADAXgOdaHnoe
    uNeuXQO3233Dpx/5w6duvGG9dSrX3NPbg+HhYUhCLvMoABgdHWXff+4nrLK8FGAUzDTAzJwU
    b8XiuUIqndH2HzkZJQR2HqYpvF/HFcNRALaamupGV2EBqGlg9coV+PzjjyGdjEORCL7yuUdh
    s1kRDEdx282bQAQBwYAfTpsi8HDQwT3oGfOVgiAAjAk2qxXr1q1BRXk5li9bgiKXC/0Dg8hm
    sxepfca/XjIIu9qi4V8zAE4yxn552+5baxYuWLAegHN83JtpndWET9x71xxBENYWe4oKBELQ
    3FgvOx322bft3NJQ6LTDFwhSAMyqWGbv3rbhIUZZMJFKj3IZ3nFCyDFCyAAvYc2HpWxoaAjh
    cDjs9/tVMArGKHRNQyQc0SVRTGQTUbz40kuT8XgiIokCoZTC43bD7/fH/vX7/5Yy1RSxWi3h
    aCweSaZSuQXLGCZjcfiDEQCw83kXriU3SyaTkiRJX+7r6/9/Oju71geDIZemGaIgCJBlCQCB
    ppkkHlflWAzNlZWtT+7adevjdrt9DmPMg2km1aurq3HgwP6K3t7+b3V19dQQIlwEZkWxora2
    GrJsuSi1s9vtbO7cWYwQgem6eZHiEEUJpkmRzRpid/eZid7ervFMJmvy6xCvrsvVkEqlGpPJ
    lEMQJGSzKs6c6YLfH0VpaTn6+30YGQmgsNCCdJphYiIKj6cAgmCFJFlsyPV+sWMG1cZTPWlC
    CInFEpiYCCCVysDnC8JqVeBw2POgDOB9CpwP5K6utAkmk0kA0Lu7u/tGR0dfSiZT8pw5c290
    Op3lgiBkdJ0iGk3XLViw6AZdp2WMEWSzJqmvb6yXJNfcVEonskxoMplgtbW1pZWVcx9JJLKW
    M2eOdfFNr50QkqcNo9zJZPmIRdf1uMPhSObAORcNSZJEVTWbsNuLWHV1bdrnG/dTCiYIEkyT
    kUAgmLJYhMmSkipYrY5kJpMKWCwSy3+GJEkwjAzjwG/Lz/s1DW5ycvK3/++//MvQ3//DPzSY
    plllkeVZTz/15QVLFy0g3/235+jv3ng7KkpiUCAkJAhCLJtVxwFYrwaKlx6HDh2Wd9y87aHZ
    s1uqDUOHIkswTQMCAexWBW+8tY+dLHDC43ZhaHjE/PGLP6euQqd047qVxDS0i941KEVVeQlp
    rKvGoePtXVymNTlNXbLidDhcoiDkFBSmAVmSeNg7iXA4jCMnz6CqvBjLFs1FNBrGyOgYhFz0
    JzMGC66zt4mU866Iqqrk4MFDIAQ4fuIUgoEAWmc1wr1mBWSnfQrV8cF9ZhoboQ4gPD4+/hvT
    NO/72h89vfGrTz8jDwwOQNd1PPTAvY7+/v5lblehhZoG2mY1CltvXF+1dsViwTQN6LohAKi4
    65ab1i6Y3ez+6S9eP8bHnSCE9ACY4DTA+66ur68Pqqr2HDlyeHDrTZvmigKBzabgps03Wt7d
    f4AMD/TSgN+faZ3VLJWVFMM0dDTUVZPSYo8xPj6eHBkedKiqlmxprPO4C52g1ABAcby9g42M
    +xJc0ZC9llKHEALTNC2GYURcrsJ/b2lprlBVtdTlclUZBhri8YxQUVFMfb7xJKXZOGPpWCAQ
    S1ksgl1RlJJ0Oj3CF821SvgRDIYQj8c3pFKp1Xa7A7qeQK54I+di2Gw21tbWbOi6CUEQTJfL
    SR0Om6W/f0RijCDvbeeKWASYpiDGYvH40NCZ81wpNK0iCk3T3YZhClM9dVXN5WxkWYbLVQBJ
    EuDzqdB1huLicmSzIhgTRa4HlmfqcOSUEoRQygSXy4XS0hIIgoDW1kZQShAOJxAOJy/SHFM+
    ftrjcHs2AEyOjY3tnz277fDIyPiu1tY2zWKxKFarFV5vSCotnb2EEFkgREAmYxCXq65sZCQu
    5NQshDidDtfs2StuTialJtMcPTE8PGzyoqtuTh9ql9rz0NAQgsHgRHl5+Rmns6A2m9UBCIjH
    M2J5eYXkdBabAFUFIc18vqwgiiI0jcLtdrNkMhYvLy/xTE5GEqWlVYqmMSGvC3c4JITDo0lV
    1VJT7Vm6RqJDy2azB7LZ7BCAegAlxR7PLZtv3LBo+ZKF2H/gAHtt7xsBwzRPAejgEqA4/6rP
    4Lk6i9zuuRMTEzANHV0X+lFfWw2/34+//86/6D998eehrKqGLBY5axhmyfbNN1R94ytPCEWu
    ApiG8T5+2m61oKWxVjp0vD3LpUI+AAneL+Kqa0vXdZNSEwIIGKMQBKCwoAATvgD2HzqGcDSG
    3VvXw2GzorKsFCNDfQiFw5SxiyH9dalXJEkEAMFms2LVimXIpJLwesdx+MhRLJg3B7qu56ge
    5CiOmQyT92q4oWmMsb533nn3jQfuv2fu//7rv1r7yiu/pIlEHMVFbvIX33zGykwTmq6hpqqC
    fOPLnxVLitwgzERNRaly841rFn3mwTvsL/zy9WQ4Gsvrcf3cizYuDUvz3fkAjO15dc+rt+zc
    0bZy+VLBoBSfuO9uqbOzs3xgYIAokuC5Y/dO2VXohKHraGqoxa07thQdO35cHx0do1UVpQ0P
    3XtbSYHDBmroCEWiePk3b2d0w5jgErjItYDzH//xH/MSq5/yRNQsABVr167bHAzGGwOBGESR
    0J6eLl80Gj3PqyeD/LPDM0kUJpMpaJpeQylTTJOitLQEhAjQdQMNDTVmIhEN9faOjDqdzmhN
    TU0xY86W/v5xcXIyBUmSPsBNmyYEj6eYDA0NxriEMci58SteQ2lpGYLBIM11PBSQ99QMg4JS
    hqKiAjgcErJZIJFQIYq5eiOHw45IJHPd9pyjLE0QAhKPJ5HJDEEULWhoqIOuM6TTWUiSNMWD
    FvjX6YH0FHsGIcTo7+/3L1u2fA8gbO3rG16yfPlizWazsWw2Rbq7Ry1Wqz0/h+juDoiGQSAI
    MhiTxMWL17dFIjabzWYaIyM9UZYLTwNcpaRdUksAQghefPFFqKqauOeee16pr2/Y1Nc34BAE
    ET5fSCgtLSkrLS0lhEiOiYm0PZHI0VWqaoIQd0FzczmxWAqY01lTGgqZrljMgCjKPKGY0i9c
    6PZze76o4pCmMRkqL2KYZIw5Fi9aeFdVRQUMTcOi+XNFu92mp9OZIULICS5r0/iHZ3hfiGlV
    Svr8/o5Cp3P5nt/8lvT29WHRgrn4h+98N3XwyLFRxth5ABMVZSWLP/OpT1TcfdtOpcBph8mp
    EDpFEy0QhprKsrw8Ks15rfg0qq9Uv98fyKRTKHDawSgFYQw1VeXY89peRCIRVFWUYn5bCwwj
    x4EGQxG0n+/ReFiUwXW245RlGYIokEwmQ956ex9Onz6NupoqOOx2vP7mPrgLnbhz9825JCGn
    N6b0xZ62UXNDS59ub3/96NFjD37yE/cWLV04Dw6bjfO6TpiGDtPQIQpAVUUpTF0DpSbuu/1m
    WQCVBcJwrL0zzQExSggJTC3MuTQsVRQFqqqqZ8+de/n5H/1oR+uspgUOmxUet4v8jz/7E3nP
    b/agdd4i+x27d158jhZJwhc/8wfWl8o91nQqhb/782fKF89tIaZpAIzi1d/tp8fPdAYADAAY
    JISEZmBrGl8EKatV6SspKds0NBQggiBC06hQWlqajkajfYSQw7ySLt/UJj5dbf/27dvw4x//
    6KTNZuuqra1uqqurlXy+oCDLQnpo6ML40NBQp8Ui982bN7/VZnOtHB4OFqjqe7K7qQlEQnLg
    4nQWKLySMksISfIKx6vQLFXo7e322e0OjZC4JV9Wns3qkCQF1dU2pFIJTEyooBQQRQGMAU6n
    xPr6AskpkcmMFSyMMcIYiMtViPr6OiiKFem0CkkSUV5egqGhIJJJ/bobUuXtmeeu9Hh88kBV
    VeWFEyfOzTl06LQ1J+nLqSSyWeMinZTNmpCk3L9jMSaoquIwDMBuT6fHxkY0AAmeFE3hCh30
    uMNBz507+9asWbPfDgajtySTGVDK0NU1KKkqUFZWLg4M+JGLhnIbpN+fkRwOl1uS0giHURYK
    6YSQnA7c5ZLg9Z6OhsORQV41689jybU4YsareNTy8rKo3x9YdtPmG5cUe4pgGBqWLJpPli5a
    UHLg8DELNxz/1JubwcSrb+9795/One8Iqll1QWNj/bxnn/tRwjvhG+fFF+Mrly2++anPPbp2
    /aplEkBh6NrFMmg2pQRcIAwlRYWiKAgOk1LblFLka2Xgsx0dHe0Dg4O3LV4wD4zlaI6li+Zj
    ycK5GBz1YfnCOXA6bDB1DRaLjLNdF3DyXE+c77jh6RYQfGCeeemy3WbDymVLEAmHsHrlMrgK
    HOjr68fA4CACgRBcBXZIwnRh+crSPtM0zx4+cvT87bfuvKGqsuKivpqxnLZdIDxtzSgACkYp
    Sj0ugJo429mDgeFxlT/nAIDJqyVfNU0DIcRkjPX8/Oe/eG7D+rXf/titu2zUNFFY6MS9d90J
    0zRAGAU1dX6NFKUeN3nkwbthaBokkcDQNRAw9A6M4Ec/fy1pGOYFnhwcRK7bHp3GvbM88DDG
    /E888UTr6dMd8zOZLARBQSyWEaqq6lwDAwO6YRghQsgFvvFO+w0nhBBs374NFovlzIIFC54G
    sD4Wiy4HdHd7+/mJycnJEYfDcWHz5i1rEgl92/Cwz8YYQb5HxQfPHC1isVhthMDJZV/i1Xja
    XDk8gWEYvWVlpX6vN1ibB3tdNzE6GkZTUylUlSAez0IQJC7vEwBE1PHx0RCA8HS6Ml4FQEks
    Fsf4uA8lJSUYHw9BkixobLTAarVA14GcGpNcynDMlOpgR48eGdq8efshl6twTiKR5frq9+Yz
    r5TIKy7yHnU2S+FwWJFOj2mJRCLDPdfw1aL/vOa7s7PL39DQ+GxLS/Pyc+d6KgiRwBjB4OAY
    RkcDvOjJwscWQSnI4GAYspwAIbmfCYIIq9UCIJjt7j4/zO25j1/D1SmOqQ9eEAT4fH73x267
    9cs7tm+tISTX37a02INPPXhPRVdP78pwZLKTMYwTkmvLOV1JUn7hMMaGgsHQjwG0nDl7vpkD
    awxA4sb1qz/xja88ccPCubMF09DfV0k41Xtm1AQjgN1qEURRsHKAVoArVzZOuUZ65uy5va+8
    8ssHWpsbmxVZBBNITiECEbVV5Vgyvw0kb/y6jpPnerXwZMLLFQQTuEpLxqvOAQCBCETNZvHu
    /gNQVRXvvHsA0WgUZSUe6JqG/YePYdP6lXA57RffUjATY857HTabDel0OtB+5uzB4ZGRG+bP
    aYVpGEgm4kjEY0gkEtDULKhpQCSATZFgU2QosgBZJBganWDReCLF1Sr5TQnXKGZghJBUIBj8
    9X+8+NL2NSuWbS0vLYamGkgmk4jHY4jHYshk0jm1DKOQRAKbYoHdaoFNkSGLBLpu4Ce/2Gtc
    GB4f58bcxdtGqtdSV0z9/lNPfZWWl5dazp/v+cPBwdGWPO+bSmlgzF41f/6CBe3tp88yxoYJ
    IZk8PTaTDnKapmVOnjx5GkCQENIOoIgxpttstuj27Tt3BIOJW32+kHKpx3w5+R1PMsqCIFhN
    kyrXStwRQvDaa6/h0KGDF5YuXbF3fDzwSCSSuMiBi6IAVVURiegwDAZZzvcG0Vlf36lIKpXK
    95W52NFtJn2R833CXa5ClJeXQhQlzJnTBEoJGCPweAogyxrC4cxFJQe5TnsmhCASmUwpivxu
    cbHnE8mkzyoIImTZAqvVBqvVdhGwc2+cEpGTGEsgRILFIiAcDmUoZZPcnq/I718Siar79r19
    eMeOW34xMVHyWDgcI4IgwGJR+NuQbMjr63MboAjGRFAqgLH3+pEUFjLa3X06kEqlu7lkdoRX
    x7KrArQoijBz4kFomla2ceOGP3ny8c9+vLa2BoaaKygwdR3bNq0Xg5/79Op/+O4P015fIMsY
    jhFC/FN7YEzj4VIeXgxxo+hmjFkA0KWL5j/51c89unnh3NlCrljlvbJndhmgBkGeApD4/UnT
    SaLxkKnzp//x4g9mNTV8Y8G8ObZgMICe3j4Mj3lhFRnePXwMFkmE3WrB6JgX4Wgsxj24C7zl
    pno93kYqlYZpmlSURGzbehNMXYPPN4FDh49i59YbwUwDpqkD1EQmk4VhGFMNekaVbul0GgC0
    /gsDR955d39ieGio4HxnJ/w+P5KpJEzDQFZVIRICURQgCQR2m4KSokLMaqjGmc4+0zDMSeQ6
    fsWu5m1MXUTcExt7Z//BF1/b+/q65sYG+9lz5zAyOoZYLIZsNn9fADVNEAJYJAl2mxUlnkI0
    1FQCjOK3+w5Pci+jgxAyMB3veWo097WvfQ0As/b1DX769OmzjwWDYUGWbRencWIirjQ1tW5c
    vdoaaW8/NZnNqpSXlGszbOKucalpgjE2yhUR5u2337HL74/f5fOFFd5I6Aqe8/vBmj9jaUp1
    41WP559/HgBSDQ2Nz7W1zVrd3z8232q1wel0orjYBofDiaIiA3a7g3t5MpLJiNbV1TXGqaNh
    XF8LWdhsViaKIkskUujq6ocgyGhoqIWmMUxOpiHLVoiiBZIkQxQF8LfHkffJO6Y5Ln8WNJNJ
    tVdWlnkBqam01IOCAicURbmYcM1J2XKcN6UEug5kswSyTNjQUDTBHY4oprT/vEarBppOZyKp
    VOJnjY01tyiKtaakpBhudyFsNiu/N/Fi18IcsBOYJqDrDNksg2GIUNWh9NDQ4AC3514euVyM
    Wq4I0KZpor6+HhMTE23333/f3/zhww/fHA4FpGd/8NzFaj5q6hDAsGn9Kmtdddn2f/33F6v2
    HTrxXUrpO4SQcd4f+oo66EtkOQY3iCRPPDFPkWvHQ/fdeeeyhXPFHKVxBWA2p3yPMKhqFpRS
    dh07cnpkdOzFP/7Tb9d98v57PrV08Xyps/cCAoEAxkaGUFJUgBWL52MyFsPLr75uxhJJPwfo
    IQDhmRQzvK/8uLOzoKmpcdG7+w+I/X39ME0dhq4jnUrhhZd/+f4IgVHE43FYZNnBtddOxpiN
    t1el13q7ST4sTKVSZ/7thz86NG9O6/ba6irMaqpHaXER4vE4Xnl1L+qqK7BxzTJksxkEAiGM
    T/hxpqMHx890p3koGOARjjkD/js7ORl792//zz/9esmiBTeXl5e6aqsqsGzRfBQXFUIggM8f
    QHlJEahpIBqdhC8YwrjXh32HTrBDJ86m/cFIP1fndCHXEWxawMkYwz//8z9jcHCw1OfzP3Pu
    XOenNc0oaG2dBUmycMCwgBAZoZDmLilpuWf79try06ePPDsyMnJqqtMxnaKNfFKWA3WCMYbP
    f/7JWq83+gcTE6HCPPi+1+FNuCJYS5IMSk3GbXomxmW++eabHVu2bPnrZcvm/cXgoK/O7XbC
    YgFKSkpgsxkYGAhBEEQUF9vh9Q7EM5nMMHc4vIyxzEyoys985jGoahqaZrY4HI6aOXNake/c
    Z7EoIEREWVnx+3qfCIIEh0MWKyoqi4LBgMs0qZMrSN6noLhWHUVTU+PIxo037S0pKf50KqWK
    6bSGSCQDQZBQW1sCURQxPp7g7UwV2O0WlJVZEI2OasFgIAggwLXu+nSlhAD0M2faz23atOXF
    oiL3fZSSynRaxeRkFJpGwRiB0+mArjMwJvBqQiscDgUul8j8/kGtt7d92DCMs1xkMX5pgc5V
    Oejh4WFh+/btj//xN75xS+usJvi848imkzCNXDc7U1dBDR0FDgU1FR6xrKhgsS8QvL+zdzBf
    gZWZzg1fMuGU5chP64a1K+/ftH5lUQ58DaTTaZw604FlC9sgCeSyVAcAhCKTVDdMnXtt5pWM
    mrcHvMiV8msd8weCr+q6fmckGvNQRkh8MgLTNBEIhTHhD0KWCDJZVeWSp2EAXsZYymazcc0y
    rvkaKB6hlJeXl29Yt27t/ffe/fFtK5YtFUSCnGzQyG2ApqHlOuoZOkzel4OaOua2NJT86Gev
    bm/v6FHCkZjbpPQIT5alrvTShKkeLWNswh8M7nnqi4/ftGvbJkkSADAT3/vhT1BZXoJkMoWK
    Ug8WzG5CNptBNpPGC7/6HfvtO0fDU6oGE/lwcDrVosj1axgfGBr+u7bZLY2PP/IHKxOJBF5/
    6x3MbW1EKp3G6+8exaP37YYkWnG8vQPbblgOz+Y1+LO/+545NhEYBXAOwNkpfUemFaXddddd
    ePTRR/HII488sX//gS/5fEGhrKwSlIoQRQpZZpAkQBQJNA3wetP28vKiHfPnL6MjIyMa33y9
    0+FkL70Wu93OvvWtv8To6NBNPl94kWky3hhJRHl5MbJZA+m0dlnvmRARsiwhFktrjCFv03Sa
    4zMA8RMnju3btu3Wc6Io1BFiMLvdTQABsizCZrNA1xl0XWd+vy/M6bpRxlgkX0F4aRRy6Sui
    vvCFLyAej4s2m32epuk7z53rfKC3d2BuJqNd7Dshiu9tgpKkvO//Fosi1NYunVtb23JHNht3
    BIO+37W3t/cwxmLIvSWJXakCeEqElly0KPyrdFq4p6dnzJOjMGTU15dzqSJFTm2RgSxTyDJF
    SQnB+Hh/PBwOe3niODqdRPDUTXhiwhf7xS9+9t2dOz9W1d/vuzed1rFs2RyMjUUQjaZx003L
    cf78KMbHY2huroLXm0AgkERlpcI6O08GRkZGOrjD0Xe5aFC6dmRIFF3XEY/HYLUqsEiEA7MK
    U5Ng6Cp0LYtMSkUylYam6WUAmnglzPgM5XZTj4rF89qWFxcVgpoGCBj6LgzC5w9AEue832um
    JsBP3TTROzBiTFFWqFcCaE3T4PEUIZlMVRR7PK2arpvpdDoLwPHCz14Zr69vKJjV3GhRLLm6
    +nGvH2NeH8LhKAaGR01CiCAQUqwo8mLFYmlNpjN+zlte8/15pmmKmzdv/sbjjz/+WDIRt4RC
    Efzmtb2877XxARrnfZECMwFGhY1rllV/8s4d9+9588DGn/7yd3/BGDtICBnG9Nq9arHJWLem
    qUmBwB2djCGVTGDh3Nk4dPQkGmsrYbcp8AWCUGQRVouETCZLKWURbsyBqfzzdL1o3iDqQiwW
    O8OouVLXNVDTRCKRhEUWsX75AkiigGQyBdPMvcRAIEAgFFXBW5/yUDA0k7f3iKKIH/7wB1I6
    nW6anIwJhAiIRuNIJnXIso2fdsgyhcVih8UiQddBRBEthKCNvzQgcD1Js0WLFuHJJxP4/OfV
    JamUKucb2iuKgsJCJ5LJyBV56NzvSSwaDaV4dJm+0jUQQvD0008jEokoFotlfiqVdkaj0Uwk
    EpZsNkt44cLmWDAYKBQEC0wz57eUlztzrqCeplarlVZVVdmrqqqbZ82aXSAIJH3o0P6+oaGh
    Sd5a4LLjOhwOKIoy+8SJEy8MD4/M9niKhYqKct4c6f2dAnP3Jb+vo13uZ7KVUsdyu71qSUtL
    2aKzZ8/8A6Wsl1OH08EQQ9ezA6Lo9BMieCwWCxTFhkzGgGHkqNp02oDdrnD+WQSllMXjkzHu
    bPjwXpuCaZf0A9DT6ey4IJDjgiDdJQhUzPH8ImRZxtmzQ0gkNCiKBYyBb8wMhpE1E4m4D7mW
    rvk6gg/kUq4F0Oa77777/Gcff7zM7XK1mIbuYoxK4P0gcv0yTFDTZLquM58/mBkc9XZxI/qw
    b77QfYFgcsIXgCKLIGAIhyNoaapFMpm62BeamrkycGrmKg8HR7w4crorPiUje9Vm43V1dbhw
    YWDn/PnzvlXsKVISiUTGIktSQ31d2bgvJIdDQT0WTyRHx73xirJi2+b1q9yuArvc0z/oaD/X
    vc4w9GWKRWLJZJqe7x18XdeNf2CMjXIZ1NU8HcnlctW43S6L3WZl2XQKjJnvNejnQHyRc+eV
    kqaZk8EZug5mGiACIYyxAq7rHZji2V5D2icxXTfGfrVnr//U6TPuQDAEVVVBwBCLJ5FMJtE7
    MAQCwG61oKqsGKc7enU+pwHOy6ozbdDP7SI6PDp25rmfvKRLoiAHQ2E8+6MLULVcgnDfgSOQ
    JRFWi4w9b+yHw27FhaGxFPfcRzi1MZMcB/bu3YsXXnjBuPfee19ZtWrFgmQyUwUIVkEQBUGQ
    IYoSEQSJvQcoIiMkrnZ2nhnjHy9cryF3dXXBbj8EQfjshdLSIj2Vyso54LVA1ylsNutFxcal
    dEfuHZDxzOjoyDgAP2+udUXAUhQFHo+nMhZL/G0mo861251aQYFbj8XSxclkpkBRbCwQCGai
    0UDU4SikVmuxxzAkm6JYhEWLNs5SFKFKkpS7c2MbE2VlZX85NDR0jNvVFSMzXdddmUymXFVV
    Eo8nGJ9TPpfyVCCGIOgcvGWIIoUo5qIXQIQsW8REIl5BKWueQqXp17Kr559/Hs8++/1QXV3d
    iNNZMMdut0GWc8VBOW25gLa2nB6dMQJNIzCMDM1m01E+r2Fc4/VTV4kKU4oinW1qqppUVbOY
    UhO1tcVoaMg9y7KyAu7Fm/B4FBQUWJHJeNV0Ou3j9uxFrl0DnWkFGmGMuQC0ApjPi1XsV/g7
    OkVj2sN3hosh4Uwy4Px3lbISzydmNdU/KAqkglLqZIwJoiAQIF9NxwhjjOV6J1NGGUUwPJkd
    HPX1MOAggHd528LYla6hob4eo2NjFYWFheutijKPMVYJMLsoSmtESbKOjIwM8NBnFEDWIsvF
    dpu1VBQFhRd+xBljk1lVm8yq2gXGWDeAYS451K9y74Ldbt/qcDi2UkobKaVlYExi+ch0KvRM
    aYJz8YcMYGCMMQZV1WK6YR5GrqtdO5f9XZW34z9zKorlAVEUH6SmWeV2u5xFrkLZblUEQkA0
    TUMsnmTBSJQSEEPV9VHDMN8F8CYh5ChftJhhx7MctyZJs+trq/8sncncKMuyo6qiXHYVOokk
    CsQ0DMQTCfgDIZZIpqkoChl/MHKeMvYGH/sM9yZnyvcTi8VSXF5evoRSukTX9Qb+FhmSZwRy
    PU/AKDVZNptVM5nsGHK9Ec4RQkav8UyveixdurSppqb2oUxGXaqqajljzJpTWQp8LZJLxS+M
    MZMODQ34R0dHTwE4RAg5yRVD9HJ0w5133omRkWGlvLximapqKwzDaKSUFhYVFdV6PEUrzp49
    29vd3R1NJBJeWZaCdXV1do+nrFwQhEJCBCaKYhJATNe1WDQanhgeHurQNG2A9ztPXG7Mhx9+
    GF6vt8Rqtd4XiUQXptOpOtM0C4D3du8pMsep8DGFY85Lc8EmJrwXJia8B5BrodubB86r2XNN
    TQ1GR0fl2267bUdJSdkThKBJ102XaUJmjAiAQARBIJJEmCSBGYZh9vb2BE+cOHZIVdU3CSEH
    OV1JZ4pXlFIsWrSwfN68xU+5XK57AFaYyWgWXad83FxHSotFZJIEmkgk9WPHDl7o7u56C8Ab
    hJBj3PH5IOV7rYVEctoUF+9F4Oaytcv9HeNgnOCNXKL5l1zO9IbzHiZjrBxAGz8r8f7SU3KZ
    8RlfQIEpoYM3L/q+XOMV/n0rci/HrAJQxjehRTzCiHNlySA3UAtPzhXya9D55xtTEpxePgfG
    VcYljLEivuk18XGvJJ9i1/hevphoiDcrSk4jCw0AMmOsArneKXORa1Cf146LU7LplN9niG++
    57lGOHk9AM3/xgFgDoDlAOYKglBCCLEQAhEMhIGBUkZ5SW+Kz/8ZDpRjV3qm0xhXAW+tilyT
    K+Uy3jGbYs8x7mEFZ8J5X2FsJ4AaHu3UI9e/hVzBptlUhRN3EPr5JpG6HN89ZeMV+Vqt4DZd
    UF9fX11WVtZ66tSpkGma+U6I+faxhdymFX7PeXtW+TOf4A5H+ipjWvh4TfzeCq7GDlzle4yv
    uRHkipB813LyplyDAKBEkqQ2SRLniaLUIAiCQxAEmeTeVCAwRplpUmaahqGqWoxS2sftOf+y
    CzbTZytJEnRdV4qKipocDvsKxjCfEFIpCIKVEIGPy0CpSQ3DMLPZTDYWi48xxs4h99aYAU7H
    zgygpxgByd/gVWRdbMpiNqeG99dpzIQQYuWG5uEPXLxGmUa+oUkC771L7+LLTK8C0AIfy85P
    C6cNHDxRkn8Db4aDdr6hiTQFxExu0Gnkqij1Ky3mKXNrwXuNlpQZqE4unVCTX1uaMZa+VpP1
    S+bYxhdnMY+WFHywUT2bsgGFeDIyfqUOetd+vgyMQSKEFPJNuBy5RvT5jUGYMi7lHlSYL9YA
    T4TS6wToqRJM+RqaYpPft8ErFWcMzpc4OyLvG16AXDMiyzQBWuM2nZj6tvqr2BXh95a3Z4VH
    CgX8vtMccGN8nLztW/jcT92UM3jvZRTm5QA6HxHyz8nbs3Sdtpx3stJ47x2MbJoOR34DdvNN
    2D1lnYqX2DPlY4T55juZr868noiQt6R1ckernGOWcglmTZ3XKAAfVwfFr/hM8Z/wuMTQyJRF
    S6b5wOmUk11LdnY5+oExJvFNCXyB6pf8LpmOZzDNMX8fz2FaY19m/LyWWrzMPJNLKCzK58Kc
    ypddzwY8ZVFLlwAzucLzNADoU4HyeqmG/4v2fOmcT+fZTwVqdh22ld+IhakOBbdneg07ZNNU
    Xn3k9jxNldBUu5pqz8IVNkBzyib8oe2Zz3Penq/2Np5L7fmKY/9/ZXGlHkF7daQAAAAASUVO
    RK5CYII=
  }
  set pieceImageData(Fantasy,35) {
    iVBORw0KGgoAAAANSUhEUgAAAaQAAAAiCAYAAAAQ0q1EAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAAGkAAAAIgCWmqNlAABaf0lEQVR42u29d5xc
    1ZUn/r0vVe6q6u7q6qwOanVLamW1EkISBgQimmQTHcE2OIIz9ng83rG9k+zZ2Z2xZxxnscHG
    GONAsEEkJZRT55wr5/CqXrq/P+pWqxACdUvyzM7+9unzPtVJ9W7de/L5nnMI/gtfhJB3/D2l
    FP/v+q9zZv9R50UIedOz3m5N/7fTz/n4B8A7/QG9lHtWshZy9jP+Hx//n8lLxeefj5cWshbu
    v+IhFO9S4VK8/09Y18Xc/5Hr+s+4Ss/qP+u8KKXgeR51dXX4yAP3F9fBUUo5/P/sKj2DknMh
    51AMZ9//qYL3P5KP/qN5dr6Xw+HA4sWLYTab/1N5KZ1Ozz2f8RDHaGjhe/1fUSEJggBVVXHj
    jTfi97//PQFgBaARQorMogMw/iOth7ezugkhMJkkmCUR8WT6Lc8+m6AvZG0LtZTmy0QLWct8
    vFVCCCoryhEKRwCAJ4Rw7LwMAMafk6lK18fW4jGZTOt4nncrihJWVXWIEJICkAGQLxW6/zdb
    6MXPRggh999/fzkA2w9/+MMw+5kGQKXsj+ZDN5fAQ0JXVxfuu+8+LFu2FFdeedUcb/3yl09i
    5cqVWLq0Y+5nf/u3f4upqSl8//vfh6qqbznni+WjIt0KggBKKQzDmFPcf479mG/Up+TcuKJj
    weSfUfg1/bPzEQDcd999VrfbvV5RlNpMJpPdv3//wPDwcJQQkgWQY7J43nvB/1djIFEUoaqq
    AKB2cHAQAGpaW1v/jud5RzabDRJCTAA0i8WiFQn0bOL5M1o1FgAtTNgSZi24KKVr5FyeZ1TC
    MwKysr8T2PcUAJ3v2s620M512JIkoWvdGkzPzBa9YcJkD0cpZUt8e3f7QqzAt/tbjuNAKYWi
    qB1VleXv03W9S9U0H/v8BgDtP9LKdrvdtz3yyCM/vf9DH7jd7/NtnJicGgKQJ4Tkz1ZI/zdf
    8Xi85YUXXlgaCASyLS0tDymK8oXy8vLR8fFxjdGMws6HAODOxU+XWvhxHE88nsqao0ePVA0O
    DmmKoohVVVW801nWvmfPnkXj42M5WZZFACgvLzf39PRIk5OTfJHv3skYPR99V1VVoaamBolE
    AoZh4Pe//x0eeeTz+N73/gV/+Zd/WeTp4l5wxc//7W9/G5qmYXx8/C0G6cV6UueSXcFg0Pun
    P7303k2bttyg62p4dna2+Dv1z0m7Zxve119/fefAwMAvjh07+T673XGtJAn+iYmJOACFECJT
    So2FfHbhUscu/9yeiMvlQigUKl+xYsW/chw3ffLkyaObN2++NRgMXv7SSy9xhmGMEULkbDar
    AshSSsNMyCiUUoUQYrxd/PNir8qK8hs3rF357WMnux/3B8OPAUhXeSpu27Jh7V/uP3j0h4FQ
    5N8B2JYtW3a3x+NZPjo6emBqaur3hJAQgBizzo2F7K3VasU1O6/Gb575LQEgMoVjANA1TWvo
    HxjaWuWpdFNKRU3T9GxW9ucV5RghpPR5PACdUgoQQgnzMOdrBZ5jXRxTepQQYlBKKcdxAMCt
    6lz26W2b13308ad++ys5lz/OzkcBIM8nt3Mx53XbbbfhqaeeAgAoipJe1tEurF29UorF4zkA
    iwBkKKUZZt2pRc/tPzoc9efmodJnG4Zxw8zMzJc3b97875lMZm0ikdjq9Xo/V1NT87PW1la6
    aNEiIoqiMT09NfrSS7snKaUyIUQuUVSXnJfuu+9uVzQqf5/jHNVr1qz+4t69+yZWr15pzmbp
    vxmGVNbWtvgzhw8fmb7zzjvXi6L43vb29mxZWdmTL7744mlKaRxAgtHUea/vfve7qK6uxo4d
    V8DrrQIhRMjlcryu6xSA8Y//+D/4VatWbfnJT368+IEHPmrVdY3TdV0eHR05vWfPnlEAabPZ
    nDtx4qDQ3X3IwJm8lw7AYF7LhQp8gjNplbn3/vrXv45YLHnD9HT8e06n+bTf7zsKgKeUzjLa
    BQB4PB5ks1lkMplLEokpDesW1xoOh+RkMk2np31SfX290tvbU8mM8hylNM2eq8+Xj4QLZaC3
    +0Bna/NLzVCJRAIAaFdXl/eGG2649gtf+MLl1dXVJlVVW5qbmz8xMjISsNlsizZu6CqTBD5x
    /OTJ3waC4d2U0glCSBRAutSKuJRrNHRdy+dy5YqimgBUA4jmcvmkz+fXMlk5B6ChsbHxfdXV
    1Teoqhpcvnz5lkQiYUomk39g1pbK3Nz5hlsrs9ls88uvvNre0d7exnGktrev/wCl9ASAhKey
    4j2PfOqhr69Y1sHruoacLNMf/PRnMy++sue7lNJhANNul/Oqxc2NXcFwZHh6xj+kG8YIBaYJ
    IfGzBc95vVdB2FxRUX5VOp3uTmeyxwkhGUaUOcMwKIAlqqqsPnayJx4Mx/oBNDBiDRTfw2az
    4ZYbrsHPfvk0WCwaJaHYiwlNCX/60592NjQ0LMtms+lIJFLZ09Odm5meNA0MDpk9leU3O2y2
    a3zB0HOynHuWKezkAs7jQjx9EELQ3NwMk8mEUCiEaDQKXdffZGH/uRQTpRS6rid6e3s95eXl
    n6irq9MymQycTueObdu2VVVXV7symawzkUiaPZ7a8Vtvve2Z48ePvTI2NjYKIMyEn3ap+T0S
    CWqElJkMQ7dPTU25AOQGBgbTLS3LU4qiCMPDw9YNGzZsE0Xxq7lczjAMw1FeXt7q9Xq/HQgE
    hplBFjsf3VJKceTIEXNlZWXDnj17FiuK0v6+932gRZJE5fTpk3+IxSIRjoPJ7498Z2BgYoVh
    EMJxAiorXUZtbf1TbW2tzwB0dvHiJZzH0/LAu9/9oYyqZgcCgZnhwcHB8fHxiaBhGGkAORb+
    nNf1uc99rknT6N2ZTMb3+uuv7B8YGEgQQtLMwDYikXAFx4mXlZVZMDbWf3R6eqaSeSThUloJ
    BoNF+ikqtYvmI6/Xi09/+tPrFUW5/MMf/rAmy7JRXl6Rb2pqhtPp0Nvbl+3YsqXqMkHgjv7x
    j88+lk5ngoSQBIDsfJ69YIVUzN8wQWEDYGfCMccEqkYpVUs9kUvFTCXKLjM6OjoCYN0jjzyy
    tHXxYrzy8su49pprVj///PO5ffv2xU+ePBVbu3pFzdaN6x/53QsvbVMU9XuU0mPMpU/9OZRS
    NJ4IHDvdNxFPJDkAdQCEVDojjE76fJmszBNCtpvN5mu6u7t/FwwG+6655povVFRUrE8mk6fY
    nsUXIAArurq6/vW22269rK211d3YUC/29fXhhz/8wQ31NVWh/sGR8OSML9u1bo2wqWst0dQ8
    lHweu199rfb26674ssUsyRPTvkgonvZ+44ufqFNVFf2DI9lDx08HXnhl/6/TmeyvCCGzAMLz
    XZMginU377ryK0Mjo8Mv7zn4JUrpNHuPPKXUsfOqK7/5kQ+9b8PevXty+w8dW6pqmgwgUmQY
    T2UFgqGw5enfPX+L2WTy5fL5PkKIyog5d5HMVPXud7/72w888MDKbDarj40Mq/W1VZLIc+Rf
    vvOt9kW1VZycTePRv/6Oubt/eJxSOsKe/WcJ36mqin/7t3/DmjVr0dW1Hh0dLWIkEuIMY05o
    GKVW8Z/j+sxnPoN0Oj1VW1ub8vl8rvHxcdjtdgSDQZOuG6tbWlqyTmeZX9OULKVcW3l55UfX
    rl1bHwgEHs9mszylNEQIyTCldEl46QMfeD9ee22ves011w/rumZLpVIeAERV1SghalzTssQw
    jLq2tra7AWjd3d3/s7m5+Rqz2bzZ6XQuDQQCGcbfifkYUl6v99aBgYFvTE/PelOptC2f18iG
    DV1qU9PyG5uaIFsspD8YjFdMTwd5gAfH8QAIV1e37Pp16264XBC4tNvNZ2dnEytnZlJ8WZlV
    dzha4tu2tfRI0sv/q79/4DQhJDDf9QAAzxNrIJD+iK4TZ01NzRcGBgaOAwhQSpVHH/28Fomk
    Pn7yZM/dVmsZx/N8GyHERymVAQiUUgwODmPJksXkkUceufpjH3vQ8etfP/VGKBTKMwPzTXmd
    hV4f/ehHxbGxsc8cOHDgblVVqdPpVgTBIsiyjkBArlNVV0MwyKGx0dnsdDr70unMafa55/Xc
    BSskwzAAoLympvq2667deV21t6oumUzmT3X39A4OjbxIKe0mhEQopSkWKrukDKUoCgA4zWZz
    ZVVVFa6++mpYLGZs2bQJZrNEbrv13ZbTp06afvbzJyxPPvVUOpvJxnRdd1eUl9+dV/Lt6XTm
    BQCTTGvLlzB3UX7Zli0f+ehH7l/24ku7Kx9/4hdlAKL33n3nrbfe8u7Wv//OP0p79u7bOzw8
    /FvDMEZbWlquMJvN1lgslgdQwbyEheT0aEdHR+1DDz5YZbdZQUARCgWxuWuNd+3K5V5CXtQi
    8WRQFARomgZN06HrOtzOMqGl2lVVX12B/uHxRb9/5RC8VZXwVrixdkW7dWlbU/O+Qye60pls
    N6VUZ4n+8yoki8UCWZZ7T5zumwDg4Thug2EYZqb4kwAsVVWepq61K7loyGcRBL5a1TQTI1KN
    UmqKxuJdDod952037vrU6Ph4+PX9h79BKT1OCPEXjZ2LSU1YLBappqYG4XCIt9msxGGzktGR
    EZpMJuks0WksFkUimRKZ5xYD4P9zKYO/+Zv/jn379ooDAwNdH/3ox66U5fyiDRt2KKqa6z1+
    /Ojh/v6BGQCposD/c0QbUqmUwPP8Fr/fbx0ZGWG8RcBxHAjhyNTUjM3r9Ta3tbWmzWY9FA7L
    stvt3nT99Tc2x+PR51988cU/UkoDJbykX8x6lixpw5EjB3HttdddfepU903JZLJq06Ytenf3
    qdfWretacfjwiWtNJslYsWJl6uTJk8Oqqu5zu90NJpNplaqq6UgkYgPgppSayTxjzaqqmgcG
    BpuHhkY4npdgMpmRz6tiLJZaIooiyspctYX340AIB47jQSmgKMSRywkOQkRUVIgwjATSaQWy
    TPloVK1oarKuyefza5hyzLHIzHkVEqUUR48e9bW1dZ5UFP0aSTKt53k+p+s6ByA5M+PLEmJq
    DocTJodDQnW1qZoQYqOFS6utrSHf//6/rPjsZz+7LRRKPUwpV7ljxxX/8KtfPfk8O6vQxZxT
    Mpkkuq5LmUyG6LpBbDZD0nWV2O12yvMcNZtNhiRZia7naSqVqmM8FMA8AXQLBjVQSr2Xb936
    3a98+Uufe/AjH+687pqr6q664vLGzevXrNM1bdPA4Iiq63qCEKJRSvMLcVXn6SEJS5Ys+fRX
    vvKVe6+66ipeFAVQw4AgEFDdAM9zqK+rIdsvv8zcvqTNfrqnVwgGQ9Ytm7rar9x+2aa+gSGH
    qmk6yxWolFKNEGJcJNBBXLJkyef+4qtfefA9t98ubVi/tiwUCrW6nGVr/vZb36jvWr+GVHk8
    Fa/t2WukUqng2rVrb2xqalp+/Pjxk6FQ6DA7ND8jlnmHiKq93p27rr12qcNhBzV0mCQRmWwe
    Bni0ty8hd73nNkfHksWEgMLQNIBS1NVUQ9GBeCYPQTLjso3r0NmxGBwBQCn6hkbp08/tjquq
    FmJhmRDzUN4270EIgaqqqKmpvukzn/rEXXfd8e6y3r6+2plZfwBAjIVKUz6f37XvwMHLnvvT
    y6PBcKQPwDSAEQCzHo/n+k9/4sGf3nDNlbv2HTyk9g+MDGZlOQggynIC2YUw0jmgufrs7Kz1
    ueeeU3/4wx+O/eKXT6ZsFpPnsSd+lfz5k09PPPfia6GX9x6ciMYS3QBmAQSLn/1SQ3wffvjT
    kOWcFeAeOX267x9OnOi7aXR0em0gEO/Sdf7KtrbFqwSBJAOBQNFgUgkh+qVURo899hgSicS2
    kydP/vfjx4+7NE0DIRx4ngfPC+B5AQBBJpMlwWDEZDKZ7R5PpUlR8lImoyytrKxaunz5sqpw
    OBRJp9M5lrPRsABgztnXo48+ivr6xhX9/QP/vH//wbbZ2YAgCFLDxo0b26emgmv6+0dMsVhC
    amysr56ZmRqvrq5urKuru8UwjPzg4OCrk5OTowCCjJfm5ZF0dXVVBQKBWwKBkMRxPAjhQCnA
    cTpkOYWZmYB1ZiZk0jQdHCeA43hoGkUuV3A2KJWRSKQRCKShKIVyAkEQYbOp+d7eY8F8XokQ
    Qop0rJ+PjwDgpptuXjU4OPzwqVOD7urq2iZZTsQSiUQUQPT06e70ypUrtJaWpqttNiFx8uSR
    U7FYbAbAMICJ++9/oGVsbPpng4NT73O5nGWZTGxgbGwk4PP5QoyPiikLXAgv9fX10aVLl/Je
    r9ficrnGRZEL1NU1VjgcFn1qamAsmw2HDSM5MzY22OP3+ycZHwUBJAkh9Hy8tFAPietob7//
    85/77F27rrmKV/MydDUPkaNYt7oT3oqHGiORyAdf2P26TCnlGJGqF2s5lVoPHMd13n777ffv
    3LlTotQA1XVQQyvAMQ0d1DCgGzpEgcdN1+/i6murHZ//8l8kTnf3phbVeZ3LljTffvx0v65q
    eoiBHHKlIYcLuWw221Uf/MD7P37drmslTVNQVeXB1x79kusHP/ox0qkElJyM7Vs3c/fc+Z4V
    j//yqfq6urryV1999U+pVOoksx6KobH8Ah6rTU5OTp48dQrDw0OYnZ2BLMtQcjkoigJNU8js
    rA9HjhyDYeig1ACoAQICnifgCIHA8zAMAy/veQMuhw2L6qtx7FQfzcr5IgpQON++MMudcBxH
    CeFESRIFsyQSQogFQDkAFwAzgITZbFIUVdNGxiaOsc+dZBak7iwrW77j8i3epjoPnnjy1+lQ
    JDrKBIqpNPb9TkWsZ+dbzgJIKD6f72c+n+8AgGae56/PZDKrNE01AAQVVQ1BxThTkJNMsL0l
    THgp0GW5XA6iaHrPkSPHvtLTM2jjOAmSZIUsU6TTuimbNTYvXrzcMjU1+b1oNHaMKaMYLiHA
    ore31+z3++8/ffp0ja7rzDPi2c3NvRLCQ9cNDA6O8Iqi2ZuaamNms5gAiFuSTFtyudxRFo4q
    hlUvaI0PPPAAwuFw2czM7JePHDm2QpZlCIIJY2NTvMVir7VYbDCZrFBVBaOjM+716ze8CzAc
    uVzOd+TIkednZ2dnAfiYMkrNV+bkcnKostIT93qr7OXllTCbzRAECYLAgedF8DxP3G4XeF5A
    wUsqKK0zeAMDqkrhdFrgcBDkchQmkwRVDevZrGxh9EveiY9UVYUoisW8MC0AgiAaBoFhQDIM
    Wg7ADcBqGAbRNE3mOJ0EAlPd4wVYX4LxUp5SuigYjCyZmkrwFRWVuaNH943NzMymUUAAM7Kl
    5wUNnc1LNpsNkiQhGo3iscce2y2K4oSqqk3t7Uu21Na2rqbUQCwWjfv9gQDjnxEAEwB8lNLs
    ufj3XLy0UIXk3rhx4w2Xb93C65oKQ9dg6DoMXYeuqaiuqsCuq7bXv/Tavss0TZ+zbi+VQgKA
    5ubma6+99tp6URQQDPhBDQNulxMAnVNIpa+rVywnf/etrzsf+vTn0z/62a98ZpPJajZbFmmZ
    zGoWd82w/I16gUuquuaanZ+75567PYLAQ1cVGLqGurpq3HTDLvz298/hIx+sgNvtxpXbt1oG
    R8YtPT0946lUapYJYx8hZApAaCEKiRCiD48MP/noV76ydNe1175rx7atgr2+DpIoQOA58Bwp
    YLxRUETU0GHoGjRNhaaqUPJ55PM5ZLNZpFIp9A2N4p9/8gv98InuMKV0kjF3lIEaznW5ADSz
    XItOKdVmZ2f/9Fd//e1/t1qtdw6PjA2z/ysyT9y0tGPJxrbWFmH/G4eSJd5XjFJK/X6f+Fd/
    /W3Z5XQkTvcOjDMjIUkIiVJKswy1hJJQDH0nsI3FYsGhQ4fQ2dnJs9i6xNYzLQjCki2bNmyw
    Wkzcus4lrng81hCKxAcADAHoZ2eRZO9lYmgpg0FY6cUm8auqqst6enrvGRwctgEEgiAyr6Tw
    Go/LyGTcyxoaFm2PRmNJJmzSCzRY3vbatetaAFg8NTV1eSKRRHl5Oex2BxKJFHK5fIlCOqOc
    COEwPe3jzGaz2+EwhSjVZLvdblu7dt01x44dzcTj8QzLUSQvRCn5/T5YrdZbTp06dXMoFIYg
    SOy5PEZGJrB6dScaGuoxNeVHMpkjqmpUiyKUvr6+4dnZ2STLRRZzlvNewxtvHBzr7Oz8l8su
    2/yp0dGJ6mQyjQLArqhwCmso7IUAjjtzTgXFJUIURUiSBJNJQkODhcbjM7kTJw5NaZo2wwyb
    tw3X3XbbbdKXvvTF5Vu2bJ7ev/9AFoDx4ot/Gli5cs13W1ubHpmenvLPzMzEmawWeJ6D1epY
    Oz0ddFNKMsyQDRV5lVJqXrVqea6qKpyKxXz+YDCURwFNGwWQppRqJaAonEtREkKwZcsW7N27
    F+l0Gna7HYQQzuOpFDyeSjEUClNVVQPLli0rX7u2a3MuRywmUxnZseO6lt27/zARCoXGAPTy
    POcjBAlNMzQAEgpIXsrwBcY5DNsFKySzy+V0SaJQsLoNA5TqhdvQAUNHudPB8Txfp2l6PaV0
    mBAiXixSSZKkYu7ItXbt2quqPJXc97/3PUxMTKC1pRnvveM2mExSwVuiBgxDBwwDlBpQNQ0r
    lnWQv/jiI/aHHv6Cks+r5K47brn8t8++YPj8gRlKaZy5sRfkJTU0NNx+7z33bq2rrYWmKgUB
    RQ3ouoaVncvQ29uL3a+8iht27cSp7h7kcjk6NjamMq8hxZTRTNFTWIjDqGl674kTJ/+u2uut
    vfyyzZ2hcBg5OYd8PgdFyUNTFeiaVtgPagCUgiMAxxEIHIEg8BA4AofNAl8gTF/dfzhsGLQf
    QC8TzoG3E4Leqsrra7yeL/f0D31eVbVBJoziE5PTv928edNNTmeUhsPhdIl3413W0b6y3O0q
    Fi5nACQopUmb1bppzcpl7+/u7T8USyRPMQU2AmCYUpoD4DGbTe/K55VRBpTIUUqTxRAuy0GB
    EILy8nJEo1FTNptdsmPHjvWdncs7HHaHB6BiNBZLTUxMDtpstu2dy5fVtbW1oramhp/whWpC
    kWO1VquFNNTVbimz2yRVVfOxRDLkD4ZG8nmljynpJMuVKMWzuhClpOtaWSqVrlcUDTxvmhO8
    RaFHCI9czhDNZnMrgEWU0vEiOORSKKTnn38Bra2Lt83O+mo7OjrQ2NgIRVHR3z8ARdHO4SWd
    UVCTk7Pc0qVtFfF4LGgYpLy83LOtqqqqNx6P+5nQyyxUIX3pS1+EIIh1p06demhoaMRazNcU
    nsvBMICRkSl0dnYgHs/CbBYhCASGoStTU1NFL95PCJlgQIt571N3d3emt7f3iZtvvrnMarV+
    LpvNCzabGYIgMkOhoHyKn7/oJQEcC++deQU4xGLTuVdf/eO4LMs9zLiZfCcFuWhR01JZNv53
    U9Pif92//8DzANTu7p50JpN99vLLt91BqWHhOC5tGIYOgLzrXdst2WxuQyKR5iSJNxg9JgEk
    PvGJh2ryee4zuZwcPHLk9T/6fD5J141ZQkj/kiVtyerqamt7+9KdqVQy+8QTT/Qxg7zIR2Kp
    8bl371643W7u/e9//yJN09Z/8IMf6szn87WGYZgJITlVVUatVmuVohgrQ6EwJwhx1NRUVSxf
    vrw+m5WTTU0tnZJkMXEcpwM0pqq58WBwtu/QocMjqVQ6zkLhecZH9EJDdunBwcHxUCjU4a0s
    Z56IPueRGLqG7r5BQ1FUygSuhJJiuotE1kEUxbWXb71sLc9zEHgOhqFj86YNMJsk6Lo25xWB
    GpBlGXI2gzKHHRo1sH3rJvLBe99b9o///MNoPpchJkmoBFDD8jdBAJmzC0XncdXt3Hn1B3fs
    2C4ZTBlSakDXC2ug1MBVV2zDj376GF7c/Qr6h8dQ6S6D11Np9wWCHAtzREuV0XyfzywNmVLa
    v/vlVx7bv//AnS2tLR1f/cqjlqrqagg8V6i2ZcYCZV6sqipQlTwUJY90KoV//N5P6MDwqKqq
    WswwaA+AYwCOEUKGmdWpnsutt5gtDZvWr+3w+UMfD4TCz1JKewkh45TSzPJlyw2e4z17w2EZ
    rJ7JJEntS9vb6tLpNGHITIl95srLN637Yn1tdeWBwyeeYdZeAsD4kiVtN69dtWJDX3+/uHRx
    05LDx08dGBmf/jtK6RAzIoqMNJf0j0ajzWvXrv3Url273r1xw4b6hoZ6wWoxA9RAMpnASy/t
    zo2NjZG/+PLnYbNaQA0dS9razD/4yWNXvfe2m8S2lkW8SRKhqQpisThGxsbzbxw+Pv3KvkO7
    Z/3B37Fnhxlo54KMmGg0KluttojZbIGuExAizOUnikrJbBZpIJAQAThwpoj6oviIUopdu65F
    Z+cK1/j4xE3BYEiwWq1wOMqQzyuIxxPg+bM9Ix7l5S7Ich6aVqiPnZ4OCIsX15dxHFGtVrtU
    Xl7eCKCKUlpOCImwnNe816qqKtLpzK19ff1rcrk8BEGaU9DF10wmj3A4jubmevC8wbon2Ezl
    5eWW2dlZlRCSZPnK3AL3RTMMIzoyMvLcunXrlqxb59jU1zdQMzo6TgoeEjfnGZ3xjsQ570gQ
    RHi9bni9Ft3v98vHj5+YkGX5JIAjALoJIdMsX03PnYIg3kgk2WS32++/6qqrLIcOHepOJpMj
    JpOkpNPpvK4bDdXV1fr09LQGgC5a1Fzt96c6ZTkPSeJsTM4aHk8FbxjSg5FIelMiMfPM9PRM
    AIDK8/zQjTfe1Mnzpi+7XC5eUYQ2h8Mb3Lhx4387ePDgIRbeBKMxtRjS+9SnPuW66667PjQw
    MPCBqamptlgsYVZVHYRwkCQTqqqqtLa2VvWNN04IqZQMjhMwOTnLdXUtW+fzxbqGhiKiplHw
    vABJMsFut2guV0voxhubDgwNnX768OEjJymlQXZu+WLukV+gYtBDoZBotZi3r+xcZpZEAdC1
    gvDVVby+/xD97vd+moonksXQxzCDPCoXkxAuxLhh3rhhw5c++cmPb17Stpgs7ViCjiVLUF9b
    85ZwHQEwPDyMA28cwrL2NlBdAyFAY30tv/fAYXLydE8uk8lac3llmiXd5pTCQtZZW1t718MP
    f+ZDq1eu4A29ABp44Y9/xOt79mLF8mUANWA1m5BKpfDTx38Nm82OO2/ZRULhKPoGh8cAjBJC
    JgHEFyrY2DoNQoii6/pkXlGG1q5du+qhBz9WJfAcDF1DlacC7W2tqPZ6CndVJepqvKiv9WJR
    XS3qaqrw3EuvakMj4z7dMPoAHAZwkBDSywAH5ww18DyPZColm8yWqz/1iY911VRVbDzV05cy
    DCMIgL/iiu031dbWVLzxxsEelvsI1dfVXvnB++7cGY8n8MKfdk/TguKZrfFW7frwvbd/4GR3
    v69/aPQwU0YDALhPffzBR//yy59dHItGqn//wksT0XgillfUKQAhligFABOlVPF6vchkMnXX
    X3/9v3zzm9+875577nEvXbaUq/JUwu1yoqLcjdqaaricZUIikRA2b1w/RyuhcJhUuF3Ce2+9
    gatwO+F2OuAqc6C6qgLLlrQIW7pWlS9paVwzOe1b7AuGQygU8SpYQI1W6TU9Pa0vX77co2nG
    5bFYmi8IOBMEQQLPSygvd8JmU+XTp4+OKoo6BGCEIQ2VC+WhYiJ5cHAIzz///A3Hj5/85PT0
    jJTL5RCJRBGLxUu8NH5OQfI8j+bmRcjlFGiaAZ4XoGkGLBarIIrISZJICYE6NDQ0DiDOFFJu
    vuCGzs6lWLVqjXdgYOCbJ06cXERpQUHb7XasXbsSiqIhl1PB8wJyOQ1tbQ2wWIBUSkc4nOXK
    yqTE9PT0IIAJtkf5Beb3KCFECwQCyZ6enm6n00lmZ30bfL4gJ4oSKCWIxVLIZHKQZRWyrMy9
    ZrMqMhkVoijQkZGTqddee3UsFoudBHAAwHFm1IXfKR1gt9sy1dWedSMj41t43rqloaFGGh0d
    meF5Tq6ra7gyEol1uFzOkampqVkAkTVr1jbPzkbel83mTVYrHx4fH5sFMHvDDTcuTiTol1RV
    I6Ojp/eGw5EYgLGlSztmKyu9n3njjdMbDEOqLSsTouFwIDAzM+WLRCIzzLMkAGzM28JnP/tZ
    ayqV+tq+ffsePXLkSK3f7xfSaRn5vIpcrvDZ02mZc7mc4uxsGJQS8LwAUZRgt9uFwcFZXpZV
    qCqFqlIoioF0WuXiccWhqsLSpqbG1SYT0j6fL8V0Q9FTmp+HVNIuxEgkkn/6++/+07/09/c/
    sOvqKyprvR6SSiaw98BB+oun/6DM+AKzTJj5WV2NeqEx91JEhiRJ22695d03LVncSjRVgcDz
    aKivga5pDNjAPBT26nI60NbaXMhxMWVVV12F99xynfmrf/0PWbfLJdpt1sXpTLaPUupmilMt
    MtI81lm2ceOGWzdt3CjqmoZIJAI5m8Yrr76ORCKBXTvfBZHn4bBb4HY74XC60FTnwZrOdqxf
    s8LyzB9e8BiUljFvQUSxU8ICcR4otLsJUkoHrBZLKhQM4sAbbwDUQOeyDtRUeaBrekm+T5sD
    gmiaDtaVIQRgEMBJAD3Ma5xTRudal67rsZaWltxtN9/AOcyC5/Ff/WaNpmn9AMI5OSfv2L6t
    9sfOny5PJJJ+ANV1tdWramu8ZGp6mucFwWWoqgSgbddV2z7Y2two9fQPhlioLsue33bw0GGx
    raneOHGqOzE14+9jiko9F50EAgG0tLTc8dBDD127detWYug6NFUBqIFoJIJ9+/YhFAxSWZZJ
    T28fYrE4nA4bDGqgu7cPQ8Nj+NFjTyIUCtIyuxVXbN1IFtVVQ9c1mCQRV16+kSOgG7743/4x
    6Q9FVAaJz19IqNfn8+sHDux7auXKNR1Wq/3WYDBh1TSOmExWVFS4qM1maMePH/Cl05kplmOM
    FT/3xXhJDz/8ML70pS9VT01NPzQ2NmYvekNAgcfOeCTcm7ykeDzFlNEZ7ykcTnAezyKrKPK6
    zeZoam/vaBsY6B8H4Cwp+TgvCOTLX/4qjhw5cvnY2PiqXE6B2WyF1WqD1+tBfb0XiUQWuZwO
    QgQYBpDNpmE284jFNMiyQurqPBWiKLpUVXWwxH1yIWEOtp86gJSqquN2u30CILSmxgu3uwKq
    qkOWVWgaZfvx1psQAk3T04ZhTADoBnCKKaPo+YyIwcHBjMfjTY6NTRGbrcLe1OReSQg5kkgk
    ZY7j5Gg0zre2Nq4URXFYVdWKfF5dk0rJNl0HOI4rA2AtL3eXS1LZvVNTiSqvV5ycnp7WGV0G
    Wltb9aqqKntTU8pwOi3548cPDfX09HSzMB93liwp7snG7u7uD/X09JgAzHmHkmRCXV0tysvL
    KUBIebkLTqcD0WgaAAeXqwxlZRa0tnphsViorgPxeJ4kEgo4rtAhLZFQAAgtLS2d9w0PD2UT
    iaTBjAgNgD6v7saUUvT29IJS6nU4HFctWtS4OBiK6K/uPYhX9h7EK/sOo294AuXl5aSyotzB
    Et48pZQHYGaveCeU1LmscKBQ96TruveqK9/12VtvubmSIygIVF0rKCOjiLJjQtfQAUoxO+tD
    d28fQPUzCsvQcdW2LVxTY524asVS2+YN69ahUMDqYWuedwiT5/lll225bJ3HU4lkKolv/fe/
    wV9+41uwOcrQuKgZ//BP38ejf/UtnDzVjUNHTyGblVHptEHkgI7WRbzDYfcAqKCUuljifMGI
    wxJCygII84KgVnoqccX2bdi0oQuCIGBoZAS6oZf8KS38oyy1cwbBNUkIKeYqUu9k+bNatPCr
    r73+5Kc++8Xp7/7zD3LZrKwD8AKwhiPh7KqVy7n1a9euRKGmp6W2urrNVVYGl7OMmE2SE4B9
    UUPdTXfcdG1zIBg2fIFQhinmNMeRxtZFdZ9/7fXXRz/04Gd++4c/vvwqy7MVaxoSJbkpjdGK
    1N7evnnNmjU8pQWPmVCKYCCATz/8Wf3+jz6kprM5tLa2IBaPY3hkmBIC5HIyjh4/Tavqm/Cu
    nbuQ14G/+OZ3co989Vv5yZlZSpj3rWsaulYtI2tWtK8GsJp9LgcW0DH/ySefxEsvvYSbbrpJ
    XL16tccwtNnFi+tDXV3LsGJFE1aubMTixeXI5eJ6WVkZrays1AHILBHNFxPRFxJtMAwDiURC
    SKVS9/f09G7NZmWWGxHe4hGdUTwCBEFEY2MdLBbLm7wnVdWRTMpCMpmihsF7m5ubl3Mc56WU
    elnB/Lz2ZWBgQIhEItfOzvoshBA0NTXiyisvR2trA4aHh1BT48amTcuxfHkLKiudkCQKi6UM
    isJB1wXwvKXMbrdXMUSnHRcwwaCkB1zc7XYnKKUIBEKYmppBNBpDWZkDZrOZbf+5b47jZAYE
    Gme5rHCpMno7HTkzM5PTNO2ZdetW9HR1tavpdEw2DKNSURQ3x5GsLOchy2pbQ0PDYgB1+by6
    QpbznK4b4HnRAcC6cuXqrcFgbpNhALKcTKXTGQFArrm5SWpuXvo5k8littmMZ0ZHT73U398/
    wnjIX4LcpAwQUcjLpNPrZmZm3IW9KRgpoiihs3Op0dHRqrhcdsRiMWiaAa/XQ4s5xspKF81k
    ohgaOoVoNAhCkorNls16PA5amofMZg1omqXV6/VuQKHNkLtYgnReAWy325FOp/ntO7bvuufu
    uz9/ww3Xda1asdzi9VTCLAmMWRUoOZnE4zFpcnKy/vW9b5Q/89yfvN19Q6Ku6wcJIX4GHshf
    gNA1LVnS9skHPvyhKxrq66Br6pzHQygt5K7ORtdxwKEjx5DP5wBa/H0hr1NX7SGXb1ovvLbv
    kHbV5RvcZXbrtmQ6O8UOKHE+YVy8amqq161ataqcI4DFZMK1O6/GsePHoRUyQbBazOhY3IRI
    JIqRyVl4K52YnvUjnUqg0l1Gyhz2skQy5QJQdiEK6SyoMwWgCDxP/f4AXn/9NYyMjKK1eREa
    6mpQX+OFwJGS4QHsC0rBfppjSi3D3HbjnZiIPVeenJr+35NT0/5ly5f/ZVtbW9vQ0NBJAKZQ
    KJyxWiy4/dab6/buP7Ain89rNdVV1SZJhKfCTZxlZWWpdKb1+qt3bFqxtI3sO3hEzeUVnbnt
    9o1rlj/osFnbJmb8/6ppetGSm2ahqykAIZZDAwCFKUjBbDbbJUliYBsDHAH++KcX6ZO/eiq1
    pG2xeNN1V4vP/PZ32suvvq7c8e4bLACFpqqYmZ0xXt27X121tNV0+4078cQvnsy+cfh46pU9
    bzTcd8cNfNHzFgUebqfDBqARhdZQDpZ/nBcY5Y477sAjjzyy1O0u/8Tk5NQNwWCkJpnMCppG
    QYgAQTDBbLbAbneYamuXN3d0mB2qmpKOHz+S9vl8Ogv/pHEBhbJLl3bguuuuv+7o0eOfmJyc
    EksT9WcS9m+GfhesXicqKtwYGZliORR+DmyQTOaI02kmgsBr9fXNq7Zs2TK0d+/eaUqpj61z
    PiHGylAovC6VSoPjBITDMUxO+uB02lGEo6fTecRiWXi9ZQAIdJ3A5XKggJkRJZvN6orFYk4W
    duIXAg4qhTlTSnVJknRCCGpra9DQUA+AQyqVhd8fRSqVe5vxE6TYGV1mPJQtDdO90zmpqqb/
    8pe/2N3Z2Zlav3791yoqKloWLVpUOTExYdc0LSuKEp2ZCTpaWlrX+v0+v6JorapayKHxvGTz
    eCqr3e6azrGxlNnlstN4fFoBQKxWi3755Vd/IJslN05N9f3ixRdf7GMeZIDx0QRToLTEqKUA
    oGmaQ1VVckYhcfB4KsHzNDs9Pa3KMlyiaKO6nstKkmDheZ4UCoZ1mk6n83V1jVIwKHN2u03r
    6zvmX7fuqrpk0jAX8nGF9zMMTuB5oR5ALaXURQiZoZQq57Um0uk0CCG71q5d8/1bbrllW0N9
    vSUUCuPkqdM48MYhuv+Ngzhw8DCOHDuBoeFR5PMK1q3utL733ddtXdRQ+2UAV1NKawkhZZTS
    BXeG8Hg8tz740Y98/MortosFqHkxSa/h4OHDCAYD4AlhNTYUHKE4euwkfvnr39LWpsY5ZVSA
    PevgeYLLN63lg+FI/oVX9kU0XTcDsFBKTcxCn5dRVV1d01ZXW8MZug6OI7jqiu345IMfgaEr
    mJ2dwf3vey/uuv1m9AyMgBoG3nPTNTBbzJiYnIJJ5GAxSyaWSLSW1CpcDPCDUEq5RY0NeO8d
    t6OxoQ7XXH0lutatRTQaQyKZLDRGLHpJlOmlc7zdfMMcLJS0D4bx+t133dXa2tKyBUB5MBjM
    JhJx3LDrGuGqK7avAbC2sqLcQQhQVVGO2poqV3VV5aqbd73LyXEEgVC42E1abG9d9J5rd2ze
    PD7tO61perH3YA8h5AghZIAxlFziJRhMIWnBYDAYi0bZ5ysopWg0Sg3DSJskMR+YHseTT/06
    ns3K4TMF1TwqK8qNmZnZyI9+9JOcKqeJ1WLOUsAfCke0Ar0VRg7Isozp2WCxNspKKRUXcm5f
    +cpX1odCocfeeOPgQ6dOdTcGAiFRUVTCcRwRRZGNOADJZhUSCqX5mZmsVxA8d+7cef2Xly7t
    uBxAPTNgFlTQ/k//9B3cdNPNbcPDo391+nS3l1Iy5xERwsHhcKCpqbEERVZAkjmdDqxc2UEV
    RaWqqs95TsXi2XxeJYpi8H19JwMDA91j4XA4w/ZDmK+noiiKN5lMegsWP490OovTp/sxPu5D
    XV09YrEsTp4chaJosFg4yDIwPh6E222DKFpAiMiJomhjfGTBBU4wKFEanGFQMjo6gelpH6LR
    OEZHp6CqGmw2y1yIs/Q+u2/p27zv2/ESBZDu7u4+HYlE/hCPJ5zLli3fXldXV5dOp/I2m5WG
    QjGi68KSVatWbVBVrUrXKXSdglLBunLlqqWpFK3P5w2YTDxNp5OUEGDHjqt2KIrjtmw2NXP0
    6MExFhIbIoQcJYT0oFD7mCmNws9tAMcF7Xa7XlAg3ByQIZ/P5ziOkx0OD6mvr81OTIz6NM0w
    ikpLVQ2Mj48nLBYp6fHUwmSyqfm84iNElwXhDL0VDBudJpNJgZ3ZnPwT5nlY6RMnTvzoMw8/
    7FGUfLWhG+W6oVddd+01bVdfeYUwMTGBf/3hT9RMJpMihKQIQZLn+ExWltPMarFSSsViu/aF
    lGxcsWPbQ1desd1FDR0UpIAa03UQQmExSTh24iRqqjxwucqQSadx4NAR4/s/+t90cXMTt2Ht
    Suiq+iY0oGFQdLQ2Ea+nwhifmh0AcAJvUwT5ThG7MoejwmIxs/koBZi3wHNwljkQCIaRy+Vx
    uqcP49M+NDdUY+WydszM+jA0MoaVy5aAL1C2xO4LHgNS0n2X8DzPjU9MYPdLuxGNxfDU088g
    FovBWebAyuUd2HbZRoBS5iC9fZ/F+VjerJYgA2BqYHDwN5IkXvMXX310y6Nf/QvF5/MLwWCQ
    tjQ1ks8//InKyampjWUOu4lSA2UOO1Z1LitrXdRQ1tHWQlRVRS6XBwCpwu3c/ND7bls3Gwjl
    h8emJ9jeTBNCBlmRnYy37++m9vf3Hzp2/NjdrS1NEjUMUAKsW7OaeCorhVgspoyPDiIYDCVr
    qqv4xvo6YugaRJ5DW0szByA7MTUlTU6MWXK5XEIQeNrUUCsUlRshFAMj4+gZHC0WgOYXirLT
    db1FFMXJtrbWWHNzk0vXdYfJZHaIotkbjyd5m80OSeL0WCyW43k+x/N6TtNieUWxOSorPe1A
    /wClNMq8j3l7AX6/H+l0+ha/379SkkxQFJ3lPwRwHAdFUSEIIjo62qgs5wkAarNZ9YqKMkMQ
    OKGvb4K8uXCWZ3BsAsPg+Xg8nj516lQPg+onFgK+UBSlLJ9XrEUlWAwJKooOTdMgCDxEUUR1
    tRMAQTicQzarob6ewGKxguMIIYQTGK2IuMiROqygH21traipqQYhPCorK8BxAnI5BRMTIeTz
    xls8pHMZdPP0YCkrYwiGQqHny8srbh8eHlu+evWa98hyZsRqtaqUEtPo6Ixl3bplGw2DSIzd
    oSiQmpraGk+c8AvFM1FVVVi5cnW7zda0QZYNKZ0eHQmHIxyL/AyzO3U23Pos1OOp2tq60MTE
    dHXhTAQkkxk0NtaKspxJl5VVU1HMZ202ez6X07ji2eXzBnE6XWoqlZSrqlpdhCQzLpczR4hZ
    Moz8XPjPYhGhKOFcOBzKMlqZ6ysqzNMafiMQCI4CaEKhTX8lIWTdju2Xt91793sxODCAnz72
    uDLr808xohxnBVspFgpb8IwOXddhtVpaXU7n8udf+CM+9P57QQAYhg6rxYxwOIIXXtxt/OSx
    J9JKPp8tL3drqXRaD4Uitu2XbXR+9XMf5+w2CwyWZzLoGUh4ZbkTzY114vjUbI4h3MIsnjpf
    RiIUlCvWHFGDAoYBQgCnww7CcZiemcGxk73gOILLulZDEHgsqq/Fvn370NJYA1nOGQvxSubJ
    TaSxoR4feN894DkCOZvBz594ErfdfB2cDtubQnWXaPCnRgjJ6Lp+8KWXdh/+t+//885v/be/
    2vH1b/x1cnx8gm5Yt5qsWbmc/K9/+HaZxSTC0DQQQnH3bTfymqYUoPu6DmeZnbdZLYs//oE7
    KjauWS4+/PXvBgxKi8ibELuzZ4dSzx5VEYlE9jzzm2d6r9h2+Wqn0wFN07BuzSryyYc+WvG9
    f/03JeSbpjyhZdu3X25Z1FAHnSEjr9i6kfv3x39VBkPnwqEAjSVSpisu21B92YbVvKEXcpL5
    fB6/ef5VPRJLRBi9RDHP/mTF64c//OGzqqoeV1W1Wdf1VsMwvJ2dnZ1mc9lNfX2jvNfrhddr
    z+/fv2eyUL9CpghBhFIkKKVTF3pgPC+K6XR2tdPp4my2OHI5FU6nE+l0BpJkQnNzIxVFkvb5
    piI2mz0jSaImikYZpageHJwS4/EUKYb4zuQCOGYVU87jqeImJyfTAKKs2Dk1333RdZ0rrTsq
    hgM1zYCm6RBFHl5vOcxmglyOIJnMgRABqqrBbreB5/OU1SiSCxmXci7eNgyKyclJzM4GQSmH
    xsZ6mExmTE2FwHEiOE6cA4Kc8ZIu6rk6APn48WMDt9/+nj/q+siyvr6RVV1dqxtMpiwnCAKy
    WQWnT4/b7Xbb3BmEwxlO1wUunzfAcSJ0nZD29qU1oljVGgoZNo/HSA0MdCeYko4y4Fax5vJt
    DdCDBw8OrFy58jWPp+o9kUiMFGD3MkKhhKOzcwlHqQc8nzeLol5z+vQ0V/SaE4kcaWpqKtP1
    fMpqLaMcJ2P58nWLEwlqLXpbHMfD6eToyEhfPJ9XImxdyWIYer4oO8raPxQT3ubamppli1ta
    eFVR4HaVYcniFjI2PhFnUO8TlNJppoiyzGqSF8K8BW2vGRazWb/x+mthGDp+/vgvsXXLBiiK
    ir/5+39Un3/x5aCmaWMApvzBkFFd5Vn98Mc/XHPfe94tlruc0DX1LZ0bKDVglgQ01deIbD0J
    BvuOsDXOh+mNRCIRy2TSzEMpeEkEFFWeSqSSSew9cAixVA5NdVVoX9wETVNR7iqDnMthaGSM
    hqPxXIl1cCn6/RGO54jfH8DBgweRiMdhGBomp6bwy6eeAaUGrrlyG1oW1bOQFmWYhoUrxLMY
    XwcQPHb8+AuHDx/eced7bpMqy90VJkmgulZA9q1c3gFdVwvnoevoaGuGoWvQNQWEUrzrsi6h
    pb7ae/2VW8jRU720d2g8yd47eS4B9w6W5+QLf3zhid/9fkfHfffcaaZGobfhgx/5sOhyOsQT
    h/fTlSuWVzzwwXshicJcPnLFsiXk0Uce9Dzz+2fp6b5h3HjNjuaHPnAnX+l2Qtc0EFC8cew0
    ff6V/UkGrpg+G4k4H8MuEono7LPMMi9rtK6uvnZsbJanFMhmFZhMVioIYiSfzw8wxJaP0UmK
    0eiCGxaPjAwbFotjJhaL00xGJu3tbbBYLJiZ8aOpqU4LhwORI0d6xhRFGbFYzMkNGzatEEX7
    4tFRvyWbzc95UmfXJxV41CBOp0tk60oxwy45Xw9O07S02WzOcZzgLApaQnioagHH4XCYGKDA
    QDgssz5zHBRFg91uBqUpLZlM5Bh8WMNFtleiFITjONTW1sDj8YDnBdhsNgiCCJfLBVlWMTkZ
    Ls0dXRAPnUvG5vNKVpazLzY2Ntx76NAJ7+HDpyvc7vK58GoBeq7PdbHI53UEgyn2PYd02oDd
    3lAZCmlEFDnkcsGU3+/XmQwu1vdp78RDhBD09vYm29ranli2bOm2gweP1hhGwTuemgpwgmBy
    LF3qojabzT4ykoQs58HzIivm1hCNUsfSpS1WjpNAqdsbDoNPJBRS/Bu7XYKq+nPDw4M+xkez
    zCmYP+ybHXa8CBWklFZ0da1b0ra4leiaBpvVgm2XbZJefPk13jCMOIApQkhfSaLMwAKGNJ1R
    SOrw75997kfpdOrOaCzmyOfyznQ6hccefzI+NDI6C2AUQB8A8bJN66/7zMc+2H7ZxnUcBwpd
    V9/aSogyMAQoqqvKeRa/LO7BQhSDPj09MzQ+Pq63L27hC0WwBTTWsvY2mEUOp3oH0djQiC1d
    qyGKAnRVgdkkQVFVvLLvqJHOysU+bmlcmhEHBBRw2G3oaG/DCy/8CTdefw1uvv5a6JqKX//m
    dxA4DjzHMUlR8JAu1LIrKiX2qiQSiT2vvb5nctfOKxdfsX0rdFUhuq4VOmcwyDkMgxVQF/J5
    xY4Wa5YvIauXtoIaGk50DxrJVLo4tTVyNnT2HZQRJYTkYrH4H370459cuX7t6p3tSxbDMHSY
    TRLef++d2Lqpi3Ac0FhXy+iDec2guOX6q7mOxYuQSqawrL1FsEgiNK3g2Eeicfz4F79TY4nU
    LIAxAGOs5iW7wHNTWN4tTSmdWrZsqd3hcH4yHu/neF6AomigVJBqamq08fHxAAtVDrK90HFm
    vIu2kHN7/PFf0iuuuOJnzc2NldXVnrVut9uTSKQrBYHEBwZ6/ZOTk5MA+urr65MbN265NhpN
    d42OzoiGgbm8USlKqrSdkGEAVqvVwnGc2TAMieWO9PnuSywWC7pcTp/JZPLqOt7kIcViWSxd
    Wo1oNARZ5pBI5Fgni0K+wuuVMDHhy6ZS6WJrpQwuvk0ZoRQkkUjB7XaDUoKenmEQIqCiwg23
    2wXDoOC4s0N25IKU0lnGndbb29O9cuXao2VlzuvC4Tji8eycwind/2L4q9jZo6CgKJmelsFx
    IlwugQYCkzlN0zXmhSzE4NZ27959+IYbbvxVU9Oih8bGpoWiRzwx4UMsliZWqw3JpFzy/ML6
    olGZnDo1KzidCSSTqqBpYAqLgyAIcDp149ixI6FcLl+sw5wuNTjnq5AoYwTcd9+9ePYPz15x
    43W71pWXu6EpOYBSXHPVDuEXv/pN0+ne/ipKaRkbTa1cKIGwIWWZ0bHxn4yOjZ8GsMxmtXa9
    +PKrvKqqxf5NIzzPu2676dr3Pvzgh5pbmxqgaRr0N3lE+luQeBwB3GV2juOI1TCohVJqYeud
    NyQ9GAwefuWVV33bLttcz3Nk7hleTwVuvPYq/P7FPaivrkB7azODpxdQXxPTAbyy72iW5a2K
    PajkSxKyIyA8z0PTCgnoiYkp+EQBqqogkUzh4NHj8AeDWNGxGAI/NwDubAzrhTCTAWDs8JGj
    B8cnJha3tTQVoOHUYLOeAcIRGIUh7XPRQsMAy2kV9i6fy6F7YFSjBe8hwRTSXL7knWLybC0a
    gOmDhw7/+Gc/f2LVV770Oa/Ac3Oh2sWtzYChQ9dV8ITA4AgIJTAIAeEIOjsKRdSqqkDX1Lke
    gL954VV64OjpKFNGg+z1nF0s5pF3y5tMpvz99z8Aq9Wy/uDBYxszGRmCUIAVRyIZYfHi9rrp
    6WmbpmkCK8CNnVUnsmDSeOWVVyYFgf+fkmRaWl5evsLtdjePjIxkstlsAsB0e3u7smbN+nvH
    x/3rw+H4XBjmXM1WS7+nlEAQREkQeKuiGG9qJjqfHMr+/fsi27bt2O/xeFYFAhFSmkuamYnC
    7XZAECgikYJ3VBrWM5uzxvBwT4RSWgzrJi8+2kAZuqzwnSiKcLudEAQRdrsNVqsJ9fUepFJ5
    5PO0FIlPSsLvF8pHtL9/ILJhw6ZXq6o8OycmZoWix1jarqj4+uabn6uL4nkeoqjRYHC26LUW
    e4qq8wRa6Ol0Oj40NPhEW1vL1nA4sVaWVdYNvhA+zOUMpmjIW9aUzarI5dJzrZeK63O5JBoK
    nU5NTk5MoNCarAhBz807h1RqCa9YsQKPPfazro888OEvXbfrGlsxBGboGpob6/HJj36g5tFv
    /O1l4Uh0irWFUFjPMX2Bib45AUMICVFKjwLwZbLZPlZroAGI8jxfd88dN7/vc598oMlbWQGN
    ARiK+aJzheuooYMSwGySCEc4yYBuKgIL5rs2hurq/80zv3368ss2f/zaq9/FE46AgCCTzuJE
    dz9MkoitG9fAbJKgaQpACTRNxcjkrB6Jp4LMXZ1hIan8JdJHSKVSmJqaRvuSxcgreRw+chQr
    O5eia+0qqKoCJZ8v5JAKfAee494OtbegancAycGh4T2nT3ff3rG4xaToGhKJOCKRCGKxKBKJ
    BHKyDIN1zTBLAhxWC8rsFpTZLLCYBKTSGYxP+4rhqXjpwMJ5MBFY81PZMIz9v/nt736/a+eV
    H9qyqYtTNAPJZAqRSATRSASxWAzpTBqKojADhcAsibDbzHDabXCW2eCwmiAJPAbGpvHEM3+U
    VU2fYkw0xGq1FtyzjVKKjo4O3HnnnVCUfP3Y2OQXh4bGKovMXLAw06SqqnHR+vXr1x88eHCc
    Uupn5RLZCw1HMUWY1TR9RtOy+Ww2Ozs9PV3B+D/d0dEurlu38eHBwamuWOxMvuhcCqi0p9uZ
    0B3lOY6XAFVk70nmSzvj4xPKypXRX3V0tF2TTGZbVbUAuKSUoKKiDBynIZcTEY/LKNiMBeEn
    ihwdGjqZnp6emgEwxUKbSVx8yA6EEJSVlcEwKNLpDCoqCt2+E4ksFMVg9VsFR/BM6I57i4e0
    EGg+pYXRFbquK7KceaOqqjI4NRWs5TgekiTBarXCYrHCYrFAFCXWfbywH4bBQdcJdJ0HpTwb
    HZLV4vF4ioXDYguhV2ZJ5I8ePTrY0tLys+bmxo6+vlErIRxEUYTFYmXrMUOSTBBFsRTODcMg
    0DRA0wgMgwOlPEwmEZKU1E6fPuEzDDoMYJDVayWKI4rm3VyVbSrX29u7/Z677/qbT3/y48ud
    ZQ6oSg6GVihQNTQN1129g5ezmTX/4/s/xejEVFGpTLKHKgsJb5TMYk+y/xthYAkzKxSsvX7n
    FV/9zIMfbPJWlkPTlHOG5+g5vCVKUGg2Wkj2CeyVW6BFkxkbn/jZP/2v762TBH4LYJBwKIyR
    0VH4IklMT4zj9b1WnO7uhUkUYLWYkE6n4Q+GsiwPMVFIXCN6IVbdOZSGoagKrSgvx0037IKu
    qcikUwgEAti4fg1sVjMMTYOha1BUBfm8DlXJg1JK5ri8cJOFKKaSOg4tnU4ff/X1PUGTJDT0
    9PRhenYW6VSajeqmUBQVuqbBbJIYcg2QRAFlNisa66ogCTyC4ZjMmCiOM5N9F7InGoDIyOjY
    008/87vrdU2tOdXdQ8cnJkg0GkMuJ0PXdOiGDp4rCAxD10FIIUwsiQIcNis85U60NNbiwNFT
    xtjkbIBZc4OEkHF2ZupCDSzDMPD444+ju7u7eXbW/+1Dh469Kx5PQZIsc/kISoGJiajU1tZ6
    2bve5cgfPXpIjscTOgqdT9LnywGcJ1xYbJHlQ6HUgWtoqBe2bNn21f7+icvOgBe4t/GM3vqz
    QlGyBoByJXy0EA9Bf/bZ507feuttP+zsbH/U54s4HA4HLBYr3G4LRBFwOJwgxARKCSgtCD2T
    idIjR06HDMOYQqFtkI8J3QW34KKUorKyEu973/sgy7LBcRxmZ30IheLgOBENDYDJZIbPF2Xt
    nUTwvARRFOcUNiFznhFXCrBYyLgS1iKNjo2NjrS1dQ60tjbWOp1lcDodMJlMEARhzgsRBAG6
    Tue+p7RQn6UogKpy0PWwwrzfREm/uPOuocRb0yml6b6+3heXLl35nnQ6v8npdMDtdsFqtUAU
    RfA8NzeK44ynRphBAeg6oCgUskwhihxGR4/E4vF4McowgkILsDcZ4/MNU5kWNTZ+6J577v7S
    Hbff2rhv/wH84Q9RwNBhGIVWNIahwSyJ2Ll9s9DSULP+Rz9/qvbF1w7UZ7LybwkhY6wL70JD
    U5RRezE+HGUHLnUuXfKJB95/5+q66qqCZ0QLaKgzSkh/k1IqnZcEDsjl8qDUuJD+cSjJOY28
    8trrfzc5NfW1j334/WsXNdRhfMoHRTcwNjYGOZvC/ffeAbNJRCqZwq//8KLhD0aizDuaZKOH
    U6Xj3udLvMXfm81m5HI5DkDn7Kyv6gc//gkK3QUKYbBMJoPHn3y6AGRgOZzSMGYimRJQ6FLh
    oJTaCCFmRozz7utXUpw79ofn/vjc+PjkB2trqqS6mgLcvdLthKap+P0LuxGIx3HjzuvhsFkQ
    T8QRCEXg84dw+EQv+ocn9Gg8GWMotrlw3XyEb4knTwkheUrpqZ//8le/7u3rv66xoa7ZWebA
    2lXLUeOtgsvpgMUkIRyJQOQ5lNltyGYziMcT8AWCmPEFEAhFsP/ISe3Q8Z4YLXhGPcxD8i8g
    Fv+m62Mf+xjncDi2j49P/NXp071bzWYzWbasDRwnzTXt5PmCkzE7m7ZVVdVeu3PnzbXDw90/
    P3bs2OsMKLTgAvMSXlLZXVT69Pbb77hhcHDq1nA4zpV6RsVczVvBDG/OZfC8AMPIU03TcYG8
    RHVdz+ze/eJvrrzyak9VledjIyOzVovFBEBFeXkNTCYrFIUiEEhDkiS43Vaoql+ORqN+ZtRN
    MZpRFspH3/zmN7F8+XJ0dXXhO9/5BzfP8yubmhpIRUUF80J45gEIcLtdZxUSF26r1QRR9Drs
    dpsznc7YKaU2FnVRSwq456sc6fHjJ2LLlnX+dsWKJatzOdWdzeYRDqeQz2uglIPX60ZNjRUT
    E3FkszokSYLFYoLFIsLhEEFpjp48OZxkqY0w8xwXFF5m+6Z2d/dMrVq1+snOzlarrmO5plE+
    Hs8gk8lDVQ3oemEagyiKUBQKQRAhSSKsVhOsVhPMZgm6njQCgeH08PDgGAqTBPoJITM4R3uy
    eSkkk8nU9cADD3zjM5/5VCUPCrvFDCUvw9BU6LoKQ1NgaCpAdTgcNqxduZQ01j5YxxPc/evn
    Xg6zUIqykJzSOSCctCgoeZ5fed3VO25d09lBNLXgGQEGevqGYLOa0VBTVVIQ+1ZvyaAU4WjM
    0A2qloAtzitgCvmZOWfGIISkKaWHUqnUXm9V5VpFVanV7iCh8VGUOeww9ELnb4vJjrIaCwSe
    11i8e5bdYY7j8oZhsHj8/GUcc+9tuVxuaefy5TfdcMN197z3jtubvZ5KEJQoZUNjnlFxfpU2
    d266pmHntk3iz5763co/vfYGnZ4N8KxAeIgQkqCUyvOdE8XWngyHI083NtS++0uPfMLrKrOz
    DhE6fvDvT0DTNDQvqseJ7l58/qH3Q+Q5aEoeWTmLY6f68Plv/YtsUFrsvh5eqMVbat0BiMXj
    iX8+1d3D3XHLDQ/efP01GB4ZxdHjp9DW3Ai304GfP/UHOB1WfOrDd+Jkdx/84Ri2bVqLcqcd
    L772Bp5+/pVMXlGLwJleVt2exAI7sxev8vLy5SdPnvzegQMH2xXFQF1dAwhRwfMEgkDA8xzL
    2xjQNIqZmbQoy851S5asK/P5ZhWfz3+Ihb/ViwxNUQD0s5/9tCWZzN3j84VcpYrGbDajqsqN
    QCCOM/VHbw3XcRwHSRKQz8uqVkCAaPMFNFx99dXYunUrAOBrX/taPhqNzY6MDP2+qanjdlnO
    NxCig+d5QikPQs4omXy+0Gw1GPSlFEXxsWiDn1Ka+sEPfkCffvppPP/883N/f76w2Z49e7hY
    LFaze/fuy/z+wL39/UNXzcwEOIBnobkCzLvY4bvoHZW+iqKEigpv1VVXvXtHMhngksmoMD09
    c9TvD/jZCPo85jm+nBACXTfUw4cPvbBu3ea7T54c2ZBO50FIIURXV1cJp9OKUCiNyko7enuj
    UJQcBEGHKJrgcAA2W0zt7j4VpHQuT50E67q/0JwWpVR+5plnfr1lyxbdZHJ/q6dn3GYymbF0
    aTMCgQTC4SS2bm1BY6MXzz57DJTyaGx0w+9PIxBIweWyQRQjyt69r07qutGHQpH7SGmUYUEe
    EsdxEEXR29zc7LTZbNCVHJoWNcLQFOiaAl3NF75WC99rSh66lkeZ3Qq3q8wJYBkLtfmZ1bvg
    th5nWz31tdU7NnetruE5UlAQRmHcxMt73sD1V259y1iMN+WPDAOarmF0clZnlmIR0XXesBlz
    qSUUquXnZlUEQ+GZL3z1G1MNDY3VrYtbxcnJSWiqhjTP4e//+ceFSa6qSoPhqIoz7XkIpbSc
    WVNZNvdHma+w0XVd6uzs/Nodd9x+r7OsrEaSRHLixElobHCirmtzITpdL/VkS6fHUnAEZHFT
    g622qmILR8jqP772xs0Hj/f8PaX0BEOSxc93ZiVhO9WgdDSXy884HXavJArQFAWaqmD7li4M
    j4zDHwziQ3e9GwR0Lp/ksFogCjxkOT83cKy0a/QFIpfylFJfIpk6pKrq+x12q21waATd/YNY
    uWwxqipcWNHRCrvVVIAUR2I42TuIJYtqUO+tQFaWaSYjJ5gFPkQIGWXMfUEdtwtj3pWl4+MT
    rfF4ApJkw8yMn3X5NkMULezVDEEwQ5LMEAQO6bQGVTU1WK3WdSwBXExQGxfCR282NC01sVhk
    vSyrKK0zqqx0slDcW4thz84tmc0iQqGIzHJc8nwRo4cPHyY1NdUOTdPMGzduMEZGhgmlNOvx
    OIdraiqdPt9MmdtdAZNJBMdx8HodqKqyQ9c1qih5OjmZkXmey/A8r1ZWlls6OtrrfvCDf1WO
    HTuRYWekzIenOzs7u4aHh//n1NT0clGUrIIgoLl50dwY9zNjJ870/Ct6TqVeEscJnCTZG93u
    yrtqarhrFy2aferXv/7lz9j8ruACgUv69PRUcN26zQO6TjfoOiCKhTBdMplHJJJBVVUZZmbS
    KKDYCmsphu1SqUQ+l8tFSgy7NC4g31kMxWcymagsZ46azeV+VTVay8staGqqRSajIBTKwO+P
    I583YBgEkiShpsYFWTaQTKYBEMTjkYyuG8Ux68MsZDwXZSilz3kVxqbT6ZM/+vGPXh4fH9tk
    lkQTNXSeGjopztoxmPDTdQ26plJVUeAPBPVnd+/xMQIVKaX8xRSPlS5aEgWHSRI5DhQcKAhH
    MD3rQ7nLgca6KoAa4EBByZnbYMWgHAdMTgdxone4iORKlsbmz7cGm832wc7lyz7ocjr1dCad
    SKfTmZpqb3NVZUVlOqcJNpOAbVs2GnsOHMpVV1Vyu6683NxYV41kMkmOHO+2HDp+enkul6+y
    WkzXmU0il0pnuf6Ryd25vPLvDOAQnyfxOtauXXvlQw99vDYaCWNqcmIOFQamcMAKd0H1ud8V
    z0vX1JL5SAqcdisaaz3WWX+w7eDxnlUsEZrFPGtKStB2kfHJqcCLr7yOTDqNYDCEWDwBRclj
    cnoWkWgMz+/eg92vH4DDZkGlqwx11RXoGxpHLq+kmdESxgIr/s+F/COEpDVN6z7V3Rs9dOS4
    TZIELGluxP6DR/Hiy69DVQuw7sHhUVjMEpa1NiCRTKF/eAw9g6OGbhgxZoFPMis8XQToXEjH
    7XQ6M9Ta2jJoGGinlOM4TiQ8L82NnSj9ujBvR6IWi4nm86FkIBAQWVhVJJeg+hMAZFnWRVFQ
    TCYJhoE5xWMymRCNpvHWfndvDt8JggBRNPTp6ckYo9vEfI2Ij3zkI43ZrPydaDTW0Nranl68
    uCNOKSS/P7bUapXsHCeQUCgDjksqHAfVMEymVEoXKOWJxSKStrZ11W1tK3aYzbZVVqtdF0WT
    KElGVJLM/2Pv3r2ncNacsbdNrCnKit7e3q6JiQl4vXWQJNOb5h8VodVnz0QqnY1UmB4L5PME
    sgzeZjNVCoK+EkA7o+HEAhUSTaUyabNZmqyr80BVDdjtNpjNhTyS1VoIz1VUWOFyWWEYBKpK
    oCgF0EE0KucMg0YBRAghMRbpoBdiRDGelpPJxEhT09Lx6urKVkkS0dMzBkkS0dnZAI7joesq
    2to8MAyC0dEgFAVwOs2w23n4/bFi3nKS9aybAzIs2ENiiDLfyy+/8vWXX37lMgBLUGgsKZ0D
    4ljaAkBlGnoAQIAQksUlaA3AcRwmZ3x7/v2Jp4f7B1Y2EkJ5ApB0JgtD1/C7P71GKIMc0zPN
    V6lh6ND1wn3gWI8yOuUbZ8ACH0oKxs4v8IwxWc6N2W22VkEQWm1WqwRKxWA4yidSaeWVl/si
    6XQmqWqav7t3IH3iVK+7o625utxVZtd1nbOYTQZH4CCESFk5n4kn0xFdN9IAylg+ab7gitzB
    gwcPfPKTn7QQwK4bupXqukCLn5sabGruXOEupQb7mrU7oqzLBKUGNQwDmqZhYGQiwujChkIr
    lnkLv0IcWcme7u574evf+odViqJ6KTU4V5mDlDnssFpMqCx3YXhsErKcQzyRQjKdgc1ihpzL
    ZzRd95V40hdbU0JNJpOey+WmX92z/+WJyam7EsmkBBA4yxwwm6Q5UIOuqchkZaTSGVDDgMNu
    xcDoZJGJfMwzuihlBAA/+clPhrdu3frlpqaGnbIsN6uqVlFQMDoBZFIEFOo6oGmUUkppKJRV
    xsfHZ9PpzAgK9SSXomYNAPDss8+GNm++/LElSxoeUBTNSSk1EUI4ngeprLS9CUF2VkPRAsqJ
    gzE21uP3+wP9jJeCmGdtVjKZzBgG7dZ13anrhpfj+EWUUt5kEkwcB04QuOj09GhyeHgonM1m
    Q62trXxDQ2u1KNoqAEPieZGaTA6JEM6dz+dkWU4n8/lUYGpqygTAznjpvLSbz+dHOjs7DzY0
    NFRommGnFBIhhHsrrLq4Bxwt1uQQYqAQ0dZBiEoJITCbOQhC1hgaGk6g0HzXjAW0MyqhLU2W
    0/va2+u7TSapned5QVUNommFvI0s67BaJdZaiYMoFmrCYrG4MjAwGyih21QpyvkCL31sbCKx
    bt2ml1asaF5nNptcHEegaTo0jTJkYgHgIggCBIEDzxdqxaLRQC4cDhXXE2QOgHYu7wjzFTZs
    DHk5pbQWZ8Y1mN4Gc198gs5cRT+DN/sBpCilFwXLJIW2EeUAtnMct5EQ0gSggiNEBJmDmxJS
    yhVnPjqllFLDoBkWRuwBcJIVIEbOtzYWj7YCqEWhhVIdI7oKs9n8XgCBXC43zazEGUYQOoOq
    l6NQiKsxYZtizJtnCjEAYJa595l5wJwFSmkTC4kuYeuxnSdv8HY/K/1dGoU6m37WzHSGoRrn
    ZSxQSnlKaR2AnQDWAWixWMzlJkmSRIHnOUI4ChBd15HPKzQj53SAapQixM7jOIATLM6cuVAF
    UIIQNNms1pUgeL8s55bbrFZvudtls1ktgiDwHAElqqohK8tGPJE0UpmsLgpCPq+o45TSEwAO
    E0JOMC8pfzHrYfk5DwrdwpuYYfdOfFQsKi92iJhghYSxS8BHoJSKFouloba2dr2qKktVVW2k
    lNoBwhVQY2cXe9ICAzFG0jRNj8fjEcMwRliObYCtU3mnPWLPFgC43W73Ip7nGgkhHgCmzZu3
    bHW5XKuef/7546FQKMeMkxkAaZ7nRKvV6uY4zolCgX6OUprSdT2jqmpOVbUs8659ODNHSnun
    rgRlZWUut9u9TJblDkVRWg1Dr6K0VIHQUkg4LX5feEtagpI+g5imlOr5vOI3DGOQ8dEIgMQC
    6YYAKN+0adN2s9l8GaW0jVJUEcKZCeF5juN4MpfoMSigG4qian6/LzkyMtLPaPc4O5MY5lkX
    9nYoRACCx1O1aMuWLfdZLOaNqqrVKYrmMAyIhHAcIRwhhCMcRyjHwQA0PZfLqd3d3f6pqclT
    lNLDODOJOv12fETmydiEhQqsTOhZzmM9FxmpWD+RoZTmCCH6hVqXZ63JSin1oND9uAFAxVnr
    IecRygoj3OniOAMU5s7MZy/MAFyU0kqmZIr7UV+0bJgAmSKEhFky0cLyTsVhfCgBUxTzShEG
    y40DyM9jLRwKw9A8KMwhKi/xWuejjN7u5woLdwSZckwuRPgxg8HB1tRYcj4mnOkCXYQGF4Vu
    0XiZZYbCBMshKRdLL0xxu9k6FjNF4MaZpralMGWD3cU9GMOZVv0xXNgQxVLGJoQQCwu9OUvo
    4XwKSWajppNsHPal2BcwoV4GoIopxyoUumZz5+GlUqMzwyxfH6OX9PkS6OzZPKOTcgCVAJwA
    TJWVlR4ArnA4rDL5EWCIrAxTYg7GSxa2tuKZ6ezv4yW89I5rYWdiZvTpZcbCO824mi8fGezZ
    xcnGEQC5hQ4nZXLDw2h3ETufYncZ/hx8ZLDQYIB5rOOsf518MfTCDE2OnVcdgFYALRzHVXEc
    Z2It6Obg7oZBKWvBrzJZOMn4aKyYh6UXM2rgHP/nfJX99BwhvHnBMOe5Hp4pBhtj6uIhkXf4
    bKVr0piiLMLJc/NBoRSVMxMixU7dxa8djLgNhqyJlCQTS/+eL1lbURireHNfOzrPQlCBEGJi
    wl7E29eAnG/T6VnMpDKlqBBCtIUyEjsLC2NuR0n4jy9RSDhrD3KMeBNs33IXatWd47yKQBQX
    U05nr+dsJaCyEF2UeSNJpgToRa5ljsfZzeP8lf1GyR4ZbxfquIj1CIx+zOcwNMl5aIYyus2V
    ABrOG9YseTbPnlcsTheYEC7ONVIIIQmcaXzMncV33FkGsH4WoOEd11JUyiXvaTqHHJkvH539
    O43tR3E99AL4qCjn7IyP7AAkNvD0TbTDwqhFYypVzI9fKkeArUcseNBwMaPOwfaMO4fsMUoM
    ljjjpcT5ZC25AGa6JMCES8BIZJ7K8Z2YiWJ+TTvPtw+EKYc5S5vV8dA/115dqjO51Od2jvMp
    JdZzGQz0LO/EWMiZXKAS4N9B6NKzBNxcD8ZLqAT+03npbQxNYOH92N7SNv5iaJeNf+BLvjcu
    JgcyT8X4fzIf4Swe4t7mvOhZBp5RejaXkF5ICQ9x7yB/S71ovYSv33E9/x9SnTv4PAIrOwAA
    AABJRU5ErkJggg==
  }
  set pieceImageData(Fantasy,40) {
    iVBORw0KGgoAAAANSUhEUgAAAeAAAAAnCAYAAADXTpstAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAAHgAAAAJwB/JB9aAABtMklEQVR42u39d3yc
    V5k2jl/nKdO7Rr3LKpYs17gkjktsx04lvUNCS2ihhASWssBSloVt77Lv0hbYBbKwlIQQ0puT
    OHbiKnfZ6r2MpvfytPP7Y84jjRUnkWQFvr99dz6f+ahrzpxz7nbd133fBP9DH4SQt/05pRT/
    +/jfx1/6Hv7vnVz4fs33sRj7W7C+woXS/z3D/72PCzl74X/6ZumbMvtnhJC/uLDMV9n8r3Bf
    2D7+JfaPUnrOXbPb7Ugmk+f87H8fb7t/ZIHy8uc8bKK/3v86/v9z5Xe2LVkM2f0fJ/2iKEKW
    5ekN8ng8CIVChBBC2GbSdxLOv8RB64c5+7Vnf/+tDv3dWPNCLthfSsEUGjlBEKAoCmHrp3/p
    c6WUcgCsABxM5nKEEBmADEACoOSX9/+2cjYajcjlcrO/zZlMJi6bzYIdKAWgsedfMgIGpRSi
    KMJoNCKZTE7fQY7jcN1112F8fByHDx8+r8P/dsHC/2sOfqGOK9BzpGD99C8sv+Tqq682lZWV
    OWOxqHjs2DFpYGBQAiATQnT51Rayvv+R7jcztBwAM4DLAIwBGGYHKgHIAVD/0pexvKwEkz4/
    AHCEEI5SSvN+AqU6OsFuoQZANRgM1OPxwOfzkYLo6YIMzFyEdPb/NZlMkCRpWmDeLQ92rmvT
    hYQZuDJm4CLsjCVm6P4sBm72mtlrNrW3t3+rpaWlMRoJq4cOHX45kUw+SggJAYgCSDIh/n86
    UmIGmFxyySXi/v37KQB6//33X2owGHadPHnyt7t37/YTQlRKaRpA9qGHHtJ++MMfIpPJ/FkM
    2tvdPwA8c/IpAM1iIVw6TbkCJa4BUAt+/4IdxLeTD0op3G43fD4fTpw4gXXr1r3pb95teHWe
    /59s377VVF/fUJZMpsUnnngikMlklAJHVf1LyC8AfPKTn7QB+IrPN7UNILzDYe9+5ZXdPxsY
    GBxjeibO9My89+t/nAHmOG4ppXQDpfQNAMXbtm3771QqNXjo0KGvAwgQQhKU0hCALNvwacPH
    hOHPFjEJgtDotNuuS2eyPZls9gghRKGUym6381qHzbZxYnLqj7KinGEXUABQy4zKOPO8cux9
    yPNd71wgeI7joGka2GubAIgAOLZHCiFEBaBQShWmGOckeHO+nIRAFEVIkjQXoV6+sr3t66Dq
    ipNnuv9T0+gfAaQJIbqB094NJTPHvV3/4IMPPvO1r3y56JWXd+O+j37ihWAo9DPmFI4CCFJK
    5dnR1f+LBviGG24wVldXP5TL5YRDhw79obW19ePBYPDjbrf710888cRPYrFYAkAIQKzQaWF3
    UWXn/GeR4fe9771iSUnZLlXVms+cOf3iiy++6Acg3XXXXZ7i4rJ7FUVJ7979/O+7uroThBDp
    Ax/4QJHVai0dHh4OPPnkkyEmQ7nCgGAu6y28X4QQrFy5Elu3bkVLSwssFgs+8pGPQFEUXXa5
    LVs2GhVFNSSTSSESiWJsbFyldFp+ZUqpKssyFQRhTvDqfPd0aGgIhw8fxqc//WlMTk6eF0n4
    1Kc+6QHEr0gSvcJoxLGf//xn/5hIJBMAooSQONMzi7qud9rfH//4x3j44Yexbds2z9DQ0BPP
    PPP8pSaTFZs3X9K/Z88r/zA1NdUNYASAjxCSXci9Ey50of9fgzfMZvPF11133Y8GBgZePXjw
    4IGKigovx3FV/f39HwuFQk9SSnmTycQbDAY+kUiEKKVnmQeTBZBhnoz2drDwYj0cdtuNN167
    6+/37j/8anff4JcppTEAUl111QfWrlx2+a8ffSIpK0oQgK2xsfHj7e3tV2Sz2dzRo0cf9fv9
    vyeEBJgyiutGeL6ee3V1NUZGRggAIwALAAMzXIqmaTIhZFltTdW9Hre7mAAGVVVJJpvNjo1P
    dKbSmSdASJgQEmcevgdAjlKa1aN25sGqlFKNOTnzWqMkSQK7p4XSV6hswXEcDAbDmuuu3H7T
    qTNnp46f7ooDqAAQZMb3vPfw3eAFFPxPvkC+4pMT4z45ly167bW9mWAoFAHQDECklEqEkBwh
    JMMMioI/Y/5yIQjIu/koKSkRg8Hgtt7e3m3Nzc2rVFUtUhSFU1X1xnXr1sUjkcjoihUrSnme
    t2oa1VKp1NHnn3/uUDQajQJIEkJSesT0bstwbW2VOxjM/k0qJa0pKSn9EoBnAaRsNuuqUCj9
    gMEgBBwOxwkAIzfffPMqVVUf9Pv9tW63u2/Hjh0/2L1791FKaYQQEgaQniusTinF0NAQamtr
    QQjB8ePHBavVYu7r67WeOXNWkvIeq7Jr105je/uKDyWT6TWKoloopTwAVRC40IkTx/546NCh
    HqY7Up/61P3OXC4t1NdXpQYHxxQmz4ruaLMgZd53q6qqirv77vcaTp48hlgsTVnkrxXIML75
    zW8gmUxUjI5G78hmNS+lkecSiWQp00WZwkDRYrGgvb0dPT09iEQib4roF0N+2f8g//Iv/yL2
    9PRwJ06c0G655ZYhk8lyaUlJiRYM+seDwUC13W7TeF5Qo9FomlLKMV2nYB7pEX4xcPu3+9li
    JqzfaT0GgwGKotR99KMfvenmm29u2bt3b/OuXbuciURCEAShdmJiolrTtOsqKipu3Lp50w1t
    S5uvTqdStfFEUo/ipuHewk1c7LVP7w9oMSi9YmzCdzCTzfWwCDOtKkqlPxiqnQqEngUge73e
    u5qamu4YGxvrJIQYq6urLxkaGhqhlEbZZc7O1wCz12qIxWIbPR73DbW1tXe3tbbeC+C6WCyW
    ZXCuw2Qy7XroM/d/6YsPfrr9+muuWHr9tVcs3XTJuvYjx07W+gPBBAAvM9wXr1vV/n8b62ve
    o2rq2lQ6W0spdTBojivYW22ee7WpsqLsm1azeXUylUoA0AghvC7AHMdBVdXaIo/7vmQqtfyN
    Q0cPSJLcU+BIRJgQ01nID8f+D0EBiWYREBhQSotLS0sfqK6uvsNkMm2WJKlVFIT2FcuXeb//
    wx/ncrmcuaaqYlVNZfnmZCptlyR5hK1Hm68AL1ak7vF4IIridA52tmL7c5DFVFWF1WolHo9n
    88GDB9eoqtrkcrnKBgYGxNLSUgPP841NTU3tdrt9fTKZXB2Ppy7mOHHrsmXLljidDs3n82UV
    RaGEEIXdj3fVc7DZLFpxcfkmQeBKTp8+9tzExIQEgNjt9nRtbd3FqpobfemlFw40NTXVNTQ0
    fEGW5VIAp3O53CqbzVY2MjLSLUmS7qRm8Q5pMf1RVlYGv99f9Morr6xet27drlWrVt9ptzs/
    bLc7PlhbW1sGaGpJSUmR2WwuJ8TwjQMHTm0fHPS1jo0Fl05MRFrNZssqh8Mom0xGY3l5maeq
    qtJVW9v0TUUxfbyxcemm5cuXL62sLC/hec4QjcY4TdM45jxr89lTQgg++tGPVFZUVP9dW9uq
    XXV1taSr60yaUmrQ7zqllDY2NtlF0fA+juN3Go107IUXnt2byWREABJDsHS5hyRJGB8fRzab
    5XTYH/Mgwc1lzbfeeqth165dd1dWVn5k9erV21taWlaZTOaWcDjeUF9fqxqNBm3Jkpal7e2r
    L21rW9bs9/sGk8mkxl5bmes5LjgCZiQXAMDqVStx7PgJAwAbU+oKg9NUJghKoTf6bnnThBCY
    TCbE4/HR48ePhysrK6u+/OUvV7a1teGJJ/6ED37wA/adl++4+Iknnkx1njnjf+6FFxOXbli7
    ZOdlm279zR+e2JBMpX9OKX1ez81QStMMoqGLufbi4mKEQiFQSpHN5sJdfYNjyVSaACjXodJE
    Kg2VIkwptQFY73A4rh8eHj41MDDwU4fDceUVV1xxs8FgaMtms4Nsr6MAUvNUOA3r169/+JZb
    bm5rbmqyVldVEpfTgW//3Xfo+OjwCqfdmuvs6g329A+NlZeVqnV1NYIi5QCqwWgQUVVeUrOy
    ufYLAJWD4Whs/9FO7aZrdzVdc/lmDAyPbjnb0y8fPHoy9OKeA0+l0pn/IoRMAggwYVLnc922
    b77kao/TZv+/P/1VjlL6AnM8VACSqqpiS3Pz57/4+c/e4XFYyTf+7h+ajp44vRTAAAAfAE4/
    N57noapqsyiKV8my/BqldJLd0cx8FOAckIU1H/7wh790xx13WBKJBEaGh7RsOoGK0mJ86aFP
    WZ02c2uZ14V4LIrPfuU78skzPR1svRpbi/LnMLwMotS/xzGHnAAgLOI5B9L9c1QOvPLKK0pT
    U9NAY2MjjcViwu7duwVRFHH2bBfGxsY8NTU1zqqqqoTBIA6pqiTxvNgqCIYdLS1LK8bHx37c
    3z9wgp0BKThTbTEjYY7jcPfd78Mf/vCY1tzcNiKKCKVSKRfy/ANBkmQBkGPZbCqrabR85cqV
    1wqCUBcIBH7e0dHRuWnTpjpVVRvsdnt9IpHIUUpTzFGU5vL6V155JW+z2T535szZeycmJp3x
    eEJMpTKoqKjAunXr25ua1rzPZDLIPK8ci8VSpng8BVUFOE4Ax0nI5RSusrLxdoul4QZB4DI2
    G4nkclLtiROTdqPRsNrhMFOHoz6xeXNDF8dx/3rmzNnTAPwAIgwun/ND0xQtHpcu8vkya4uL
    3SUA/o1F3ZRSmgOg2e2O60+ePPPXExNRa2trY0lJScnKUCh0Fnl+hKCfG6UUt9xyi6uhoeGO
    UCjU/1//9fBpWVZUBv9mFgs9amxsLO7r6/vcvn37lgGA3e6kLleRFo+nMTISFIxGa6MscyAk
    haYmb4nL5XxjcnISlNIxhmTl3lUDXHCJLceOn7ispbnpmuqqyqUEEMfGJ3x9/QMHZUXZz7Dx
    GDMOOQD03RBi/X8mEgkAKCKEiGVlZdi5cydMJhNWrVwJg0EERyDcc/d7nbt3v2z9xS8fDu7e
    szdiNBhC6XQmZTQaL9E0zS7L8n5KaSchJEwpTbIcjbqY62WKz7usre0j93/i421HOo5U/urX
    vzFLktRht9uv+sTH7rt5WVub8+vf/PZNA4OD+4eGhnZrmnaKEFLf1ta2NpPJyCwvWsSEYiHn
    qC5btqzok/ffbzMZjdBUBZl0Eg67jWy//cYKr8eF//zVI5U9/UMejuOIpqr5dVMNmqbC63YK
    2zYsd5YWORGLJ7x9Q2PgeQ7FXg/cTjvWrlgqtrc0lB3oOLk8lc60MKWYYXdBnce5nn39wJGO
    mqqKyziOa1BVtQXAOBO4JACDt8izZP1Fq3mP3YjqyrKioydOF7HcjFbwWuWqqrY3NtR97D1X
    7rj+6Rd2H+rpG/p7SukgISRYCIktwl0U3G43V1dXh3AohFwmxQmkCMlkgkpSDumUhhBRMTU1
    hXgiyQOoBBCnlIaYIXzXHswJwcGDB0AIIQ8++Nk6gGz6xCfuX6qqqlGWZd/IyNDpV199dUBR
    1DiDdN91p0DXKx/84Ac9siyvGR8fx+Tk5HREPj4+AUI4hMMRvqenz1VbW+Oor6+L5XJJfzQa
    SFZUVFjWrdvwodWrL+o5evTI4wMDg10AIoSQBOZAupyv/P7ylw/joYce2tLXN3Dr2NhEXWtr
    2z2ZTOZZAOG6uoZr9+07eLHX60mvW7cu0dl5Omi3O/57cHCwd+3atVtVVS2llI7EYjE7ABfy
    ZNE5n/vk5KRmsVgMp0+f9o6NTYDnRRDCo6ysHIFAwjkxEXF6PE7U1ztN8XjaDhAQAnAcQf56
    EZJOE2s0ylkBAUajoVIQNMiyBkmSkEppJBiUHI2NtqW5XK4dQIIhhKn5GBeO4/DrX/936JZb
    7thdXu5YI8vJUpfLtSwcjowzZyMJQFZVtS4QiNpCoRTSacVms9lKAfQzY6pQSrW7777b8cUv
    fqGtrq7+uqGhwAMOhy14xRVXfvepp546yOQmuBioh9lshqIovCzLQiwWA8dxEAQT4fkUD/Aw
    GIxUEETK8yIRRQsI0bRYLFbCnK/YfHUxfwEG2F1bU/OVT3zso3/7uYce2PLeO26tv/bqXbU7
    tm5aVl1RtmV4bLwxEo2FkadqawX5rXcTSivfuHHjd7/2ta+tXLt2LREEHlTTIAo8CNVAqQab
    zYoV7cu47ZdttZnNJuuxEydpOpNx1tXWNN11242XBEOh+kg0FmPOwjnEjsWA4dLpNAAYKysr
    v/DlL3/pox/64PvFSzast2TS6frOM2eqPvLhD1z2hQcfcKxetZLwPF+yd98bkiRJXVartWXL
    li23ORwOz+uvv96RyWROAQgDCLLLN98ImC8rK3vPVVddWW02m0CpBkKAKb8ffQPD8IdiKC4t
    4W696XrXZVs2cjarZRrrMYoizBYLhsam0Dvsw7AvhKKiIly+5RKUeD1QVQWgFGd7B/D4s69E
    JFkJAoizXFdstjLUIc7ZTwAQRfGS6697z0c/+P73OUSe1J/q7MpRSmOMmJEAkInFYuaBwcEt
    e/a+jpdefb0nm80NApgAMMSi4LJLLt7wo7vvvO3zHOiKsYmJxODQaCKdyU4wLzvFlIuySHeR
    8/l87bt376Y/+vGPfT/96c8Cg4OD1pGRUf7rf/ePvieeeyn85HMvJ1549Y2gLxDspBQTBWcZ
    ezeNHcdx+Pu//y4ee+yP4pVXXnnbxITve6dOdX24u3tw29hYYGMyKe8oLS3f2tra4p6YGI9m
    Mlm9dE9lOf13LQ/d09PDe73ejx06dOijXV1dgqIoIISA43hwnACe58HzPDSNIhKJklAobHY4
    nNaysjIzxxGrLGv12ayyqrm50V1UVGTw+SZTiqLoSJxWGAlfyOP73/8+VqxY0To2Nv79PXv2
    tQ8NjXKSpJSsXr2qqra2fsXp070r+/uHuUgkZmxubiwfHOzv9/l8sc2bN19tMBi2EUKiZ8+e
    3TcxMTHGzj3AZGNOaaT+/n5s2LChdXx8YlcoFCE8L4DjeAiCALfbAYMBoDSHQCBiHx72mZLJ
    DCGEg/6UJAWCoMJkUmEyKex3k4hGsyCEA8fxEEURdruaPX26YzKbzcWZMxNhqMJ5z3D2UxAE
    3HXXXZWxWOKzJ06cqVFVvqKqqswyMNAXopQmmS7ItLYuVcrLyzaVlXldPC+PHDvW0Z3JZHxM
    fsdvu+1W6nYXfaW3d/gfsll1h8lkpLKcDYdC/vDQ0NAUSzUlmVG/IAMsyzJEUaTV1dUNZWVl
    7rq6uoDb7Zysra3mbTaHxenkw7GYL2AyqQmzWY1MTg719vT0DDBib/Dt9mjRImAAfFlp6Uc+
    97kHP/PB999tFHkCRcpBlXMoctrR3FBjqqks2/7Xf/uPhvEJ3y8opWKBEZbeLd1SUlLyvo99
    7GM7Vq1aRTRNBdW0/JOyz6k2/b3yshJ84aEHxKYl9d6vf+s70eGRkYDDZqnavmn9WqrKtQMj
    Ez+hlL7EhHbaeViMCN5oNF5z993v+/jtt90qUk2Dx+3Gl/7qIWsum125cvkyWMxGqIqEO269
    SXhj/4E1jz72uGnjxo3tiqJkX3jhhackSeoBMDULFprvonJTPp8/HA7DbDIiGgkjk05j5Yrl
    aFrSgFwuC0WRiaYq5NTpM5BlCZRFwASAIHBY1rYUBBQ8x8FqNsJmMSEYjsBsFGE2iBgd9yGd
    yRpZpG5j9428Q/4UmqZNK2WTydSwedPGsisvvwwB37jjt4/+qSEnSd2UUhchxEQIScUTiTM9
    vf2p0fHxkWQydYR51klGbJFFUVx32y03XfHxD71X/OMf/4gPfuJzg9lc7nW2Z9bC6GM+LMu3
    KX2ZPHXq1F+fOnWqiaUWWowGQ6WqKkZZVoKyrIQymWyCneEIgEkAYYYQaOf734uJGj3//PNY
    vXrN9Z2dZ793+PCxkkxGhiCYIAgmBIMp3ueL17S21t27efNlRc8889QjiqL0Fxjhd8U5uPfe
    e+H1elvPnDnzkc7OTmMeMeGYQcg/CeHZx/zXqVQGJ050ipIk251Oq99stmocJ7hNJufOhgZj
    2aFDB32YKTvMLcY6b7jhevT39zvS6fQXDx48dFE4HIEgGDA2NglRNNa2t7chGk1BFE3IZGT0
    9Y0UrVixeqOmKTlK6VJZljuPHz/++tDQ0DhLyQRZlD6vCF3TNL/d7pB5njdaLBYYDCZQCvT2
    9kMUDRAEERwncAaDiJqaMvC8AEJ4PQJGPJ5CIpEFwIFSAlUlMBgEUMpB0/Kfy3IcqVTaw6J0
    E96hamZ2FYQsy3A6nd7h4cnmoaEJIsui2NhYVM/zfLmmKaOEEAulVDh6tGP40ku3TPA8cT/1
    1PP7wuFwnAUUKQBSTU1NRX//yG2HDp0qLi6uxsqV1VPPPvv4a9FoLMS4Knzh2ubTK+F8crZn
    z57c4ODg93me3x0IBKqy2UzVVVe95yaeF72RiC9x+PChYYYK+ZEvcR1DngybKrQTc3n9hRrg
    2u3bt73/rjvuMIoCD0XKQVMVaKoKVZEBqmLX9k1k/6HDa//9F7/tYx5Bmi1QXkxyREGpTPnG
    jRtvv+qqK0VCCAgIKAEU3fhq6jkGWKMaeI7glhuvI2aj0fW5L32V+8FPfxlwO+1CPJHSADgB
    VLGNTjMBXgzlU7tt27bP33fvvR6T0QhFkUBVBW63C5/99P344Y//Ha3NjVi6tAUOqxkffv/7
    bN09fWtsNht5+umnX5ckqZtFbX4G8YcopZn5MowB5PoHBvZ9/wc/2FpZUeFOxOMQRQECz0Pg
    OQgCD57nwBECnstDWKAUVNOgqgoUWYaiyJBlGZIkQZZl5HI5qKoCh80Ku81Cf/P4c1lNo1Fm
    DHX4SnsbEg6PmfpIANCSyeSR//rVfx8Z6O1Z/dIrr8YlSfIzroERAE8ppS6Xs23r5kuLn3rm
    ucFkMqULQoSVnCmyLMf3HziQWdpQJe7bf0hWVFVihpcwBa17zvSdSh3OVypksVh0ZANXXXUV
    nnvuOQnAOKVUAZB22O0XZzIZ85TPx3k9Lm8wHJ1iEXofg9omWU3wtCKevY7FLFFav3596cjI
    2GcOHeooSaWyMBjM4Hlh+inLGvr6/Mbly6uuqKurnejr69cwU/KWfDeg50OHDmH79u1X9PT0
    NORhZ1JgbPlzPi80ypRSdHf3c62tzSVmsyUajwfG3G6LNRgMZGw2a2MqlUowPkdmMdIMPT29
    pLGx6aaTJ0/fPDIySnheZGvhMTo6CavVjtbWRpw82QNFkeHzRUhdXU29x+NAOp0Ovv766wcC
    gcAkc5x9hJApzCqpmssjFAp1L1nScFIUxbVms4UQwoFSQNMoKAXyZfEzz7wzw4Pj8iiC/jF/
    5vnomeM4ABwyGQ1Go6oODQ3FZFlOIM8uz7zd3rE7yZlMRi6bzemBCu3t7R0rKyt+4eKLV91s
    tzu4wcEzY7KsEORJkiIAumHDxRWhUKxFUeRsJpMNFshvjFKa8/v92aIid6yhoQ5Op4eGw4F0
    PJ6wMvslA5AopSoAWK1WpFKpOcmwLmOqqs7+uToyMhIE0MPzfGLjxktrvd4StyQReL0O1+Tk
    xER/f/8wk98+ZoCDACKUUpnpMDpbjs8nwws1wA0rV66odjjsUKRsQaSZf2qqBpHnsWZ5m0kQ
    hFZFUfoopX5CiJ8JMF1M4WXEsLVbtmxe6na70XX2DHp6euCw27Bu7UUQeK4gAj7XEINquGrX
    dhIJh+xf+Oq3cgPDY1JZaam9va3iPV3dPbyiqmlmgLPsYlwIQ5UvLy//wIc/9MF1dbU1UGQp
    b9QoBdUU1NZU4spdl+PRPz6Bz3ziPjgcDnCEkuUrlosdHUcjkiRl2ZlFCSETBZHTvL17Qoga
    j8cf/fGPfszfeOMND/7zP3631FvkAdVUaKoKTVWgKjJUVZl2rqimAmz/UPCkmgZNU6DIEjLp
    DP794Ufo937yq3Q6kx1l5KIRFu29Zf6XUmrmee4+TaMDlNL9etMUSunAy6++9vXjJ0//MwAT
    zV90lUWtBIBYW121fuXyZcJTzz6vMiMWKyiQF4o8rq179+4jL+1+5XAimUwqijLF1jPK9jDO
    GigYCxyZQmX9lixpJmAi88Qdzz77rKlArrKEkIbVq1ZctWrFMoMmZ0GVleXPvrI/ns1JJxn8
    mAYgUErdTClZMFMOV0hi1M6Tcpl/4j+vbJrHxyfaE4kkRNE8rZjzxi2vlBWFIh6XnSUlpav7
    +vrHKKVh5iSk8S6wtLdt2+YMhUKXT076+KIiL+rq6mCxWDA8PIqpKf909FsYCetGGSDo6xvi
    TCaTgxBIiiJzdrtzzdat20sOHz6YGxwc1BERie3tgh6f/vSnYbFY6gYGBj/V2dlppZSgENol
    hMPg4BiKijyoqanC8PAEAA2yrPDZbJaOjo6GAoGAbjT8hJBxdg8T8zXAv/vd7/o2b9781SVL
    Gr7Q0XFi6+DgCJeH6QVwnDjtTOmf5wlYPPsosD2c+T7P53/P67XB45Hlw4dfDw4NDfUBGGac
    i7cliS1b1oprrrnuPaqqVP3nf/7Ho5FIVCWEyH/4w2OZ1tbWf966dauDUnrZ2NjoBHv/HOsQ
    R2022/K+volip9MyxvO8bnzDelmmy+VZJwim8rY2tXNwcCB8/PixoKZpfia/+toUjoOYSqV0
    wdAK9pQrRJYKZUdVVRBC+J07d1rdbreTUmq55pprxPHxMa6z84yyatUqY1vbsht9vqiH40xw
    ODyuLVsubwoE/F3xeCJSUlKcqKqqJGaz2S7LkuDzTZlGRsbSlFK9AZByPsdPX8e8DTAjcRBV
    ZcZMU/MQL1XP+RpUhaqpAKiNQZDOucAY880d8TwPRVHEurq6nevXrbM+9oc/YPfu3bDbbViz
    ejVUVQFPhALDqxuQPKte01SAarjp+mu4zrNdRT/4yc8nPW4Xbr7u6iX/8oPRFfFE8jRj3UaZ
    8F6I8ll29dVX3b1z5+W8pqqM2aexyJJC01Rs3LAOR48dx6t79mLHti14ec8+pNMZDA4OgpE1
    NKa4JxiElVvgmjRCSEyj9JlX9+xZ+W8/+NFdNosFsqJA4DnIsgyVGV2NOS8oWC9AWS0PwHMc
    ZFkCRwCDKOKpF/fkmPHtAdDNojzfO+Sp6y/bdMnno9HYYMeJ01+klMYJIUlCSJJSOtxQXx9x
    uV1LX3ppt85a1pumFDUuaVhVW1OlCxple5KmlGaNRsOOS9ev+UhP/9BoV+/A8wwKT7J1hRlE
    7PQWubcosiJG44nH9NIHdu5gnrpU6NEyIS7meX7zkiVLttbU1LR5izwlRqPRLEmS6vP5Ime7
    ujoJIY2XbFhv2bBuDXieQ3XtEtIz7Cs92927uqa6cm1jfa3L5XSYNU2jsXg86ZsKTI6OT56N
    J5JHKKVdzMFKsvcsv5Uimc9DUVSbJMmG2Ya38HNCeOR1E+dlBBMPu3+LngR+8MEHIYrisvHx
    iTWCIGDt2rUAgEQiiVQqPR2x6ZHvTCRMpqNjSoGBgVFh2bJGZ09Pj7+srNpuMhlLBYEvB1DC
    iDox5hQtqCtaNpvlCCF3njp1ekUymQLPG84xvhyXz0/39Y1g+fKlCIeTMJsNsFoNsNttNB6P
    88zJyrDI18eiX3m+eo9SKu3du7eT5/n/WrZs6criYq/HaDRCUbQ8cjV9ptw5UL7+OTDjvBCS
    j4BVlcBsJtqBA3uDAwMD/UxGepihi77dOq+99iprPJ79SDarXHTRRWt9L730UjdjOcfOnj0b
    WL16zWgulzNXVVUZJiYmkrpBamxsMGSz8rpwOC44nVa9XFHWkccHHvh0XTpNvihJCcP+/Xtf
    6O7u0YlWAzabbeTSSy91ud2uJU6np1UUDe0nThz94969e0copXEmO2B2JztbT950003Whx56
    6KJPfOITOxKJxOp0OlMhy4pNEAykqWlpctmyFb1ms1ENhaKVY2NjkGUNVqsFdXWVzvXrN7RZ
    LLZam83topTYAEJ4nsu0tXFBQOkLhwMdR48eOTk2NjHFyLwppkPUC4Kg2Rsa6ejoGA2HQq0O
    u/XcCFjLK+hcNocDR45riqLKDKPnF1N4dfiJlUPVb9iwfkdLSzN4jmDVqpXo7enF5ks3wigK
    eVJQYQ6YaohGo8hlsygu8kDTVBhEHve9/y5+/8EjroGhYck/NQmDKLgAlDBDN8UuobJA5SfU
    1dXddeutt9Y77A7Icr6kJ/8eZBDkjRvPc7j2yp34xX/9GooiY9wXgNdtx8XrVlv27DtgZESY
    zGxId77rYY0xMgD8oVD4kX/+P98TAaz78pe+WHvHnXcRgSPgeS6/LuakaJoCTclHxoosQ5Zz
    kHI5KLKEh3/7GP3pw79TWYQ0DuAMgNMAThFCBtgeZs9ngNl+Wtwul+aw29Z2dvd9MpvNHqSU
    niKEjACQzWZzdv26dfZXX91jUxQli3ypDAAsaWlurHO7HOA5zgLAzjxiDUDVxnWrP79iWYvn
    lX0HX2VrSzOFEl+5YsUDmy/dcFFHx1G1trKsNJNJq0+/+JpHUdVHCCEjzBCDKU25AFbiAGzZ
    tGnTX91www2bN27caKuproLFbALHEWiKjEQigeeee/6Sg4cP088/+GmYTXmmObZcimQq6Roc
    Gr7s7jtu5irKiiEKAjRVQS6XRSgURndv/zV7Xj8UeOm1/fsmfP4/UErPEEICjHyWuRAYled5
    pNNpn8fjCRsMRssMxHuuEeZ5AVargY6NBSlTYAYs8vAWXY76+vqEmpqa64aHR7ypVBpHjx5H
    S0szYrE40unMeQ0wz/MoKytFLJaAJCngOA7ZrASfL2KqqKh0GgwGyWZzGGtr65b09fV3U0o9
    LG2T0Mlk85GZe+/9MIqLixs6O8/cNTQ0JMyGxWfWxyMeTyMcjqG+vhKCkAdTkkmVVFZW23t7
    ewmLjlIFKbl5pxYIITKlNN7V1XWkrKzsJ8uWtWzPZrPLX3zxFXM2m2MGlgchQkEEnI9yBUEE
    z+fzxIIgwum0o62tRPP7J3Pd3SORkZGRngLZ7WbO/jvkqTmDouT4ZDJb0tjYfD+AV/bv3388
    lUoNAogKAp8eHw8LJSUlXo7jspqmUQBYvXq1J5FIr0wmM6AUBkKIA0CUUkqrqipNlIqfmJqK
    rDGZlGPDw8NBZkf8Nptt6IYbbnxPIpG9keet0DTRHY1qlra2Fa2nT5/6XiQS7WXENj1wkVHQ
    r/nBBx9ckkqlPn/ixIkbBwcHS8LhKCRJYakPAQaDEcXFxWvWrFmhHT58kovFkuA4ERzHI5uV
    xfb2xvVdXaNcPB6BpmEaRTCZjLDbrTuKikoSO3Zce2J4uOcPr72254Cm0SmGyul5YjpvFnQB
    di5N+aZcVqtlw4r2VkEU+LxyUfOKWpEl/OmZF+lPfvnbeCabG2AR0ACDTROEkAtmFOtNzyml
    nNfr/fADn/n0LWvXrOEqKsrQ1LgEay9aDY/H+WYilpbP/b5x4CBOnu7E8mVL8xCrpsLpsEEU
    eOHpF17OjU9Mqul0xpaT5HHGUowxb3WhJLKm66+//pv33fthr87OJqB49dU9+K9f/wYr2ttg
    MhqgqSpcTjsGBofx+8efgcPlwpXbLkFFeRn30qv7IpTSAeRbUU7pcP4F9GnVCWYRAGd4njd8
    4J571q5ft46LRiOIx+MwiCLKS0tgtZhhMZtgtZhgtVhgt1ngtNvgdtlRUuTBma4e7HnjUJzB
    zd0AjgM4TgjpZQIcfweYLTEyPlF5ycWXbPzUx+9tz6RTK/sGhkI6yaysrHTbDddf13zo8OFE
    NBrrZR5ywGa1brrnrtuur6oo5x574qlEOBIdY47SlLfIfdv9H37vHQPDo+n9h4+9zpSIH0A3
    z/P1n3/ogU9+8cFPOR1Wk/0/f/XbqaGR8f50OjNF8+uNcBwXZk6jqfDceZ7fdOedd/7ku9/9
    7sXXX3+9obq6ClaLGQZRgCjwMBoNcDocEEUB4+MT5NJL1kPT8rA+1TSc7DxDLl67mly6YS0M
    BhEGkYfI8zAaBHhcDjTV15BL1620NtVXtw6PTiz1+YNhFrkpBbDWgtM4mUxGam5ubspmcytj
    sRTJK2MDeF5/iigpccFkyqSPH+8YlmV5AMAQg0wTC+AbvKUu+c53vgObzdY+MDDw9RMnThYp
    igpJkjA15UcsFp9l3GaMnSAIaGpqQCKRhqIo0znNbFYiXm+RkEhEEhaLCZRqSk9PzzBzXnQ+
    wry5HC0tS4nRaPzAoUOH7vD5prg8qUmA0WjCRRetgs1mQySSgM5IzmRktLXVgedlqCqHvr4A
    KStzw+cbH85ms6NMFwYLkZV5Oi6UEKIkk8ns6dOnB/1+/6jd7lh39my3m+cFmExmCIIBmUwW
    sqxCVen0U1FmnrmcBo7jkcuFpGeffWpydHSsT1XVUwCOAjhDCBlm63xboufAwKDc3LyEAOrO
    U6d6Wtzu0vWlpUXa4OCAD0B6xYqVrWNjE1vsdgeJRiOnU6lUGkBo3bp1pdFo+r6xsYCtvLxI
    Hh8fHk6n0zEAk9ddd/2ySET9q0xGMqfTU4d6enommWz3bd26RQKEz73xxvHaeFxxVFR4srHY
    VFcw6A8MDg6OyLIcYJ0CCUO9pkvRPve5z1WFQqHv79mz585jx47ZwuEwcjmZ5c4BTQMURUM2
    K6G42EsmJ4NQVTp9tg6HjVDKkYEBH1SVsnx7HlVQFIp0WkE0KhkzGVJTU1O1xu22ykNDQ1HM
    dAacRrKEBRhfEEKkcCTyu3/6P/9aMTg4eOt1V+9yVFeUEmgqJiYn8MzzL9P/fvSJbCgS0/OU
    Pgb3ZRYj/6uvhZGvll6+Y8c927dtFVRVhqaqsJiMMBsNeYVXkO/VjbCqUdRWV8LjckBVlBl4
    mlLs2LqRW7tqhWXfgcM5i8VCBZ6vVFS1iLFurTqzdr6F/W63+8orr7yiIZ83z0HKZaEqCl7b
    9waOnTyNcDgMu9UMTVPBc0BlRRk83mK4bSasXdmGYo+LKyku8kz6/A4W5ZmZA3UhkDgF6zbD
    HJpJXuBx4OBB7N27F6IoYPmyVpSVbJ1GEDSNspwvnYangekpJil23r1MeHsKo453WItkNpnT
    115zFbly+6WYHBspe+6lV1dSSocASMlEMltZWYGL1qxpGhoarmCv5S0uLlpZV1PFG40GmEwm
    CxM2HsDynZddeveaFW3Cw7/9Y5gpXcIEOKCqKjl+4gQ5dmwZjp88rUwFQkOyrHSy6JjHW9Ss
    U0pta9as+dTnP//5lvb29jzxkKVeCKUIBoI43XkaiXgc0WgUPb29CAaDKPEWgVKKZCqF051d
    iMXiGB2bQCaTpg67laxoa0ZdVTk0TQVVVYiigB2bN0DguWVf/Nvv3Tk26ZeY7OkCvJC8Pwgh
    OHXqVNrpdP5He3tLndlsu3RyMiyqKoe84jaitNRDnU6qHDr0hi+dTk8y9CdcCOMtRjUApRS9
    vb0ms9l875kzXQ2yrDC2biFkei7zWTfGhHCYmPCj8G90KNrvj/B1daUWo9FATCZzSVtbW+Pp
    06dHmQwHmNHT5sM0r62t9Q4PD98wOjomzDgBIhwOByorSxAIxDA05Jten6JoSCQS4DgZsRiP
    ZFKGLPPmsrKyomg0aqOU2gghRgCp+XrQBWkQlRCSopROORyOIUJIzmKxoKWlhfVQV9HXN4xs
    VikoD+LOKRXiOKLvm6yqWhD50p+zAM6yPuWRApLiW+7R5OSkpihKcmxsnOvtHYGqGq3V1Y5V
    hJCTlNK4JEmyplE1EomW1tfXN01NTcUBuCglrfF4yp0nj1GDKAoOAIbS0hKvwWC/r68v4K2o
    sKdOnx6Ms1RQEoDf6/VaLRaHUFlZAZerCFNT477nn3/6lKpqIzrSOovISAHgrrvuIoqi3NbR
    0XFlX18fm4ox4+QZDEaUlJTAZrOBEB52uw0ejwsTE8Hp3/N63XA6rWhuroIoitA00GRSJvkG
    gnlHkBAO8biEoaF0SW1ty501Nf3JkZFRDjOVQCoAKsxHcDVNQ2lJKab8U0YApdFYLPToY3/y
    dRw9bm9sqCM8RzA4PILe/kEaTya1guR3jtXU6t12Lqikp+BvzEsaGj5+z93vXepxu6DK8jkw
    8+zIV88BE0IwOelDJBrBspbGvNKjGqBp8LgcuP7qHYY3DnUo2zZvtCVTyUv27DvQzcgI7oJI
    bj4Ldyxf3n7lhvXrBVCKwcEhfP+HP4TJaEIsHkdTUxN+++gfAUpRXOTGjddegY7jnaAUqK8s
    hs1sREWJFw011c5Jn7+IlQXY2YWUF6r8CqaxZAGECCFxjuNx2dYt2LDuojwJS5GRzWRhEAWd
    JwwKyj7OfE3zP00zZT3CPOdJ9j11DkpOjcZiB3/5X78a7uo8Vf3IH59Ks360pQD88Xg8xQG4
    YucO75NPPb1MkqQQgKrSkpKlJSVemI0GOOw23QDbqivKr7zjxmsqo7E4BkfHY2yvVHZ+jaXF
    no88+oc/Trz00stT4WhUlWVljBl1HenIMgePznIeqjds2LChra1t+k7ppVkHDx+mf/P1b2p9
    /QPk3g9/iCv2uJDLSTh1+gwuv2wTAGB0dAxjvgA2bLoM5aVePPrII/jtI4/JLUvq6Le+/IC4
    fnU70dg91TQNG1Yvw+YNq5f95vHn17K1x5hDM++axxdeeAGXX345CCGqz+cLWK2WP61c2Vy8
    fDnXFo0mCcDB5bJDllP0yJGDmWQymWRM1Dg7RwWz2v4t1Ajrf+f1enceO3b8zrGxCS4fYbyZ
    ZHUu3Jv/KAgi6uqq0Nc3glQqc87vxGJpIkmqGA6HJUoN5c3NLSs6O0+foBRFzCFMz8eBuf76
    60ApXT06OroylUqD40S0tragqqoSmqYhFArCaDRj1aoWmEwmjIwEGNFJhtnswPBwDIJgRCZD
    ea+32At0uRknxsIcQvUCAhEFQLympibAcbySTKbQ0XGcIRoieF6EIPAs10vO+TjzBDiOKISQ
    COvmNMyYveG5GF/94fP5eioryzsIMV5cVlbC9fd3piilxQC8mUxaMhqNyvj4lHHFipaVoiiO
    yrJcoqra8mQyY6CUQFGoaDKZHQAMa9as3er3pzdkswo4Tk34fD7K7Edy6dIWU3l57f08T8wt
    LZUHI5Ewd+hQx4SqamkmH9ECsh0tfA8VFRUWv9+/bXh4mE2b46bLsxwOB1aubNfKy0tpLBbj
    fL4gkWUF5eWl8PkizEAb4PU64PMNY3LSj8rKGhQXu1SeT6ouV5FhdDRBCjkB6bSKZFIor62t
    v3RkZDRIKU3oXAQA6pwjYKaM+Cn/1Ib2Zcs+snnzpu0XrVlV1rSkQSgu8sBoEEE1BZl0CpFw
    mBsaHracPnO2cd/+w46jJztFVsydZp1+9O402nyFuCDyJEaj8fo777z9vVs2XcqpbPoHLWDm
    0rcoQaIc0HHsBESRf9PvElBs2nARqamq4P1+v7J2ZZv9wKGjbTlJGmZEDt0jnE85Vf2a1WuW
    V5SXQ1NV8DyBlJMQCISwYuVKNDU14bE//AHZbAYCtwRHjp5AMJpAOhnH2PgksukUzCYB1ZWl
    JmZ8HZRSOyHEwJQJLtAIawByHMfJhBC8umcP9u7dh2QyAYMgwmaz4M7bbkKp15N/y5Td68LB
    BvlvSuxso5gZszenKS+EEE2W5TcefexPf/WnJ595f2Vl5XaDweCRJMkKwJFIJlLhSJju2LaV
    X7F82fIjHccGAKjlpSWVdpsVPAFKir0mVl7UetXOLRdftKKN/Om53Vo0GtfZ4xkARSvbGj9p
    t1paDxzt/I8J31SIKcMEc7KGWN45WrA/uQKo311SUmI3GGYQFoAiHA7hm9/6trr75VeSV191
    pe2TH7uP27f3NXqqs1Pt7unld2y9lBBQBEIheuzYca26upp862tf4oqdZrz86h7pdFdv9KcP
    /87b1lxvNBvEaYKeKPBYUltlAlAPoIpSOk4I8S2kmuDyyy/HAw88UP6ZzzxwQygUujEYjLSf
    OnXWk83KkGUKjuNhMplhMplIVVWrrbGxfYmiZOnY2HDo9OnTI4qiRFg9f1p3/BYiv5RSRCIR
    fPvb326amJj80smTp715BOZcw3uu8Z35GcDBarXA43GDkLFpWFr/O02jSCRygtMpSIJg0AwG
    a+2GDRevOXz4cEhV1aA+XWeuhi8UCpFUKn3pxMSknVL9PQAmkxGqqqCzsxsVFRUoLi6FKBph
    NBrh9dpBqQaLxQabTUYmo0KSQGw2u4PdNwe7q/xCDPAsB1quqqrK+P0Barfb0NzcDKvVxtZJ
    MDkZxPh48E1GeKbhDdGnSRUaMH1AzVyRPvrss8+OrFu37m8aGxtv83jM741E3B6TyWTNZrPO
    WCwmm81mKRKJGSVJa2hoqG/s7u4JKoranMlIJN8gRBMsFovD5XKWOp2ll/X0RIwmkxG5XDKT
    TKZ4ALLDYaebN1/+0UyG3BIIDD39+OOPHlYUpYjpnjEmv8PMeZjOuuh7zHGcOZPJeGd6nufv
    FM8LaG1t1kIhf5IQaujuHjISIqK2tka1281EEASO0rwBBlTNaDTRbFblu7rGYDZb6PHjR4Lr
    1l3qcDottkRCJoUoTi5HObPZWg2ghvFjdOb2vEgVgsvluuvGG2/85kfuu7d2eXsbTAYRmipD
    lSUoigxNkQHNheqKMixvayZXXb5FuPu2Gyqeem73rb/4zR8auvsGH6aUdrD8R3ihrGJ2GdZd
    e83VX/3wB9/vFgUeagHcrKoK0skkzGYTIxHNGGGOAKNjE3h+96v44HtvzUe/sxjSFaVeXLJ2
    pfCnZ3dnq8uKzMUe5yVjvkA/gCEGHUXnE3nyPN+0tHWp12AQIUs5VFVW4lvf+Comxsbwy1//
    Bn29vbjs0nVYu3oFHDYr/v0Xv4amUdx09XZ0nunC6Pg4aqur4HLYeBbhWRkEbcAiDhLIO8LA
    iuXL4XI48MSTT+Ku22+B3WqB02ED4ci0U0kppqFnMm2Up5nJ2nzXxHJaaUrpEUVRTEtbWtas
    XLlyzRNPPDFEKR3KZDJZn8+n7tpxmXDX7beWnTrVuTInSYbSEq/baBDBgaKupsoAwFtVUbbk
    xmt22kWRx/iET5NkWV+XoaGm8kM7N2/Y+MfnXj2gqOoIk4FBBj2PMrJThBCSLoCvCnOGcjqd
    llWFgSBavoPY+Pg4Tpw4mQOQqKkqt0UCE/inf/nXdF//YCKXzZZQqhGqaTAZDERRZOnnv/hl
    ttzrcr7/9uu4itJibXzCN3W2d8AQCIaKaypKZ5xFaEhnMmDRUiHysRC28bpQKPytzs7ObSMj
    44Z0WmKQmU7Kmc4DE1E0EbPZYisqcq5qarqopqGhqXrv3lcfDQQCfaycMIaFNYABAHz1q19x
    K4r61SNHjm6IxeIMtpsxolarFYIgIJ3OnlN+pHdYamlZAkEQIMtqgeHmpw10IpEhdruDz2TC
    qdLSSkNjY1PrmTNn9sfjcZ1Qxs3V8K1evdqYSCSWR6Mxohv7vr5BTE4GUVlZgZaWFnCcEV1d
    E0inJTiddhiNFJQa0N8/Ba+3CNGoDEoFiKLRJAi8TVFUC+MW8IvgQIPneQKAy2Ry6OnpR2Nj
    A1KpHHy+ICjlCn3lN0XAs2DaN41lnQ8z+/Dhwz39/f2P7Nq1axXHcWs3bdo09uqrr/aHQiFa
    XV2fo5TYh4YmLE1NS1f7/YGQomiVkqSAEA7ZrEycTpfTbrcvj8WU6nRagdNpQzIZVDRN43ie
    U3btuuqKbNZ8B89n/EeO7D+uKIrKOCYTyI/31NMlhfXq02hHJBLROI6TeD5fDaZHvwaDERaL
    WT179lS8vLzKLQh20tJSJ4+ODga83govzwuGfAxKEI3G1Z6erkR7+2pbb2/EwHEWCIIY8fsn
    sm53my2ZVM9xHHmeQyaTM+jBEzt3bs4GmBm8VoPB8IDJZHI+9/zz8SeffJLXVIXXVIXjOU60
    Wi0EVEMykaCZTAY0H1ZSQvKtjYo8rpUcx71P07QQaz6evoASGs/6deu+9PmHPttWWV4ORZEL
    yos05LIZPPLY47ji8stQWV4GlVIQjkDTgJHRMXz7H/+VBkMhrGhrIXlijHoOZC0KPDasWc4/
    8sRz2ceefWWM48gYg640dmu5+eR/HQ5HVVlpqahn+SnVYLNa0NzUiJJiL4LhKKoqK9Da0oTn
    X3wZwWgSLpsJO7Zeing8gc6zPaitKIXJIHLIN6DQn+KFWtxZ4/k4kn9AkiRkszn09Q9AFHhI
    uRw4AqxYthR5pHcmEqZvI6tz2SOmSCgjGkUopUd7e3v3/c3XvnqLIsu7nn7mmaAkyXRy0qdo
    qircfMO1/J69+5b/6clnTB6P28RzHKCpWLa0iRcFofbyrRsdbc1LiCTJiETjlJ0Z73TYLv/w
    He+5bNIfzA6MTJxh+xcHMEgI6deZ2mzW8lvdy0B3d/dkNBotcTkd003izWYT7HY7nfL706lE
    TOnr6hS6unuCHMflvEWeUr2HttvlgMft0qKxeHDP3n2Gqy9bZ9VUJQsgwHNcmc6G1yPrdDqD
    0139eh0hdw5uOI/Hl7/85SWZTOZ7o6MjG9PpDC0pKaF67SerByX5OlJxVr2ogHhc8lRXV9x6
    zTXXeB5//PF/j0ajasHotQUQmpphNJpuPHr04C3DwyMF0PNMeZHFYkFJSTGGhkaRJ8zmS46s
    Vgvq66toVVUp+vvHiKpq0zm3wkg5l1MACPyxY0cmJGn/mCzLp+LxeKIARsc7NV3RHwaDwZpO
    Z8okSZ7OFQIc0uksgsEIampKoSgKIpEEAA5lZU4AKmIxBYFAEqWlXlitFnAcwHGSQAhnBNRC
    R2AR5JdyNF8ZOL2PNpsVHo8KURQhCAakUjnW/erNLV8XKrvnY2aHw+F+RVGeiUZjawwG46Vb
    t14WPHu280xDQ3PGYDDA7w+TiorS2g0bNlysqrRIVfPoRzKZI42NlcWECKUDA0mBEA6iKNB4
    PAMAZP36DctMpoprUinVpKqjncPDI3o+dZQQ0gdgkpX8SG/lXHV0dGRWrVrV73Q6t2QyuWkI
    mk0ghNFozBDCW73eSqvZbEr294f8RUWV3jzywUNVAZ43IBKJxjKZpFpSUl4siiYVoCFRFD2F
    DWT0nLHZTLTBQZ/8Jtx/rgaYeTfjgUDg8z/60Y/Kke8QVYn8KLqyK3bt3PClv3rIzBOKH//0
    P/Hr3z6SZCF2hMEZMeaRTDDFJrD5iQuiQhuNxksu37FtB88RpFJJGA3idJcmqqkwm4zYsHYN
    nn1+N6wWM6oryyHLEk6f6dL+8Kenle7efv7rX3yAq6oozZOwZuWJNVAsbawnXo+L9/lDAU2j
    /Sy3GV9I7s1kMjnMZhOnK1bKiEwEFDarFSaTCX5/AJFoFHsPHAEBwcUXtcNpt6GlqR779r0O
    KTcNFuhzZoUChbxYo/QIIQTHjh3DocOHUVLixcDgEPr7B1Ba4kVpsRdLmxtYMFaQ+S24KrM+
    zvehMCh4dGBw8Ime3t6d3/zG1yoymfTtL7/y6vHhkRE5l8uZitxufPlzn7EHA8HlLqdDBM3X
    UC9vbcaK9qVFV+/cSgyikG+fSTUCwGAQheXvv+WalRtWLxM/+43vjVFKU2wvw4xsMonzNFop
    VEKs61roaEfHG8ePH12x7bLLiI6cVFaU48ordpq+/8N+ccrnU6Ymx5DJZLIlxV5TW2sz0VQ2
    xMLjxpL6Wn5gaCQXjoSzE+Oj1kQymQKA9qWNjiK3a3roBQHFmZ4BHD3dpXeR0xvCzCvvz0qP
    FFVVH62oqHjN6XQWZzKZYkmSXEajyV1cXNI8MDBqJISgtraUTk2NZ+PxQEoQhBTP88lMhk8N
    DEymOQ4hHXmhlPILkV9KKa677jp7MBi+SVFUc2VlJXw+/znlPABBOp0Fz/N09ep2TZJkKIpG
    DAZRNZlE2WIxcT5f0DgyMvmWNcKaBmgaxzudLq2rqyvEnKsICppezHX5qqoaFUWxahp9U91v
    nlWsghAuz353OiCKGgAjAoEIZJkil5Nht1sZGSrNEQKhQI4XZfgGyRc3E7PZjKamBlbXS1FV
    VQqj0YR4PI08EJQriIAxKyd8wQ+VoVjq+Pj4CxUVlbe9/vrB9hUr2q9dtqzdxXFEMZnMSKWy
    6OkZFdeuXbbMYDByehlPOi3BYnE5EwkJmUwMHCeCEI5omio0NNRXLlmyun1yUnWVlpLEwYPH
    fZRSEwuMxhn8rE9Je0s7dvToUXnNmjV7a2vrbvf7QxYdXVFVDcFgVGhqarEQAtnjKQMhGam0
    tIxPp2WB0nxZpqZRKArhS0qK1Ww2k3K73cWCoEo8z6VLS6uX+P0K0aNqQjiYTCKAuDw2NprA
    zNS1aRsyHxZ0nFJ6nDGa/UxhOQC0rVm1csMlG9YCmoaOo8fw3797NEMpHWeMuhFGY88W5AYX
    3I7SarXC6y1alclkbM88+xzu/9h9yGoKEokE3C4HQCmOHT+Jf/rXHygv7H4lrqpaym6zKqqq
    yrF4wlDsLfJ+8bMft956/dVkmsWrw88F3bzKit2oLC8Vff6QzEovogzamO7mNNf8l6apmqrm
    Z9JTSln9b56443LYwfMCItEo9h88gmgiA6fViHWrl0NRZFSVlyKRSiMWiyESjRXCu4s+F45F
    wNh5+Q5ctmVTvvZXVfCDH/8EmzduwLKWJshSjiEGmG7IAdDFqvCmbOpKmlJ68IUXXzxxz/vu
    2vLP//Cd6s/91Zfc/f0DhlQqCZvVgqXNjeQH/+c7Bo5Q1q1LRUV5Cfnmlx4gTfU1+dIUAMVF
    bk4Uherbrttp+/Cd1xn37O/QegdG/ezuS+xMz5sOOQ8DGgBywyMjT//+94/csO6iNeUmo5Hl
    agXc/9F7ha6urorJiREul4pSgee9O7dtsbQ0NuRTJJoGq8WEG67Zadh34LDHIPBCIhahgVBE
    qKuuXPbem6+xmk0GqIoMSilyuRwee+YVLRiORZDvNRtm8jNv6Pd73/veOICHAVQAqGVOtKex
    sbFRVcmS06e7wXH5utDJyZFUb2/vCIPlJzDT5zaE8wzTmO9DEAR7JpOpczodCAbD4DgOxcXF
    yGSyyGYleDxuNDXVa7FYOHbkyOmgzWZLms3mrMFgMFZWVlYlEqRoaGiSyLJ23hyxDlXncgop
    KvIKem6T1YVGcJ6mDG/3yOVyIIRoeUOvnWOEVVVjJSoqbDYrSksdoDQf/eZyKjhOQCqVg9Xq
    gKbxkGVJVRSVFjJzF0lwCCGEJBJJHDlyYrrbVWVlOcrLy3D27OA0KUufjnRuDnjR1qECyB49
    erSvpWXpS263u+3w4eOOdetWby8tLVatVgsyGRmSpOD48T7RZrNBjxo1jaCry0dUlUw7VLJM
    aV1dvaeurs3r8xE7xxHkcqHgyMiI3pAjyjq0nbfL3iyEDwBoZ2fn/vr6+qMDA0ObotHEtDM1
    NDRGVq9uL6msrFQiESdstoxVFGn1iRPDXGEDk0AgRVpa2j2UapIo2mAwJLVVqy5aoii2snQ6
    Nb33hHDweAQ6MtKdiMcTuvxGUNBNTpingpaZAGoAwpRS0Ww21y9d2mygGoWqyGhpaoDT4UA0
    FgszTL6TeSd6z93kQlmcAJBMJmG1WHLVVRW47eYbYTIa8btH/gACivdcfQX+8PgT9J//7w+z
    vX0DeqvBiXAkmuI4rmHb5ktWffK+e6yXrFuVnw5/noENOinLajaipqJU6DhxhrI16wY4Nh8W
    tMvlQiKRjMSiUQ2U8udEwYSivKwEyUQCo+OTGB6dACEE61cvg8NugyJLsFst4DgeE5M+9A2O
    KgUR0KIPHdcjmgMHD6Gj4wjS6TRURcHg4BCmfD48bTbB63Hj5uuugskg6M2ToXfFWqQ1gL0v
    X2fnmRf27t278Z733Sn82/f+ybZnz2v5M9I0qKqKmuoKUFVhKIYGgeOwZnkbVEXO16VDw8a1
    K8gXP/mBohuu2AKzUcT+jlNyVpJ04lqCKeXkHJnaIIRolNLjTzzx5JOX79h+7w3XXcvpyEtt
    TRW+90/fMf3kp/+BV/Yfo2vXrPB87N57iMlozBtV5uBds/MybnhkrOyN/Qdw4OhpLG9tKfvo
    Pbfy61a2kXzLTwqOAPs7TtLn9hxIMfTFh3zv4HkP32Dt9sBY5cmCaN/Y1NRUFImkDHpXqXQ6
    h+LiYrW3tzfIcuPd7Pf1gQY6C3vBTrTf78+azeax8XHfslgsgYqKCixd2owTJzpRXl6O8vLi
    XE/P2WBvb+8IpXR4amoq0N7eXl5f37gxFEp6JydHOU2j54WeCyNhSdKIw+E0EEIUSmmaIVjR
    +XJPwuGwZDAYY6JogKJI55RJ6XW1FosRlZXF4PkcCDHB749M/14mI8PjEUCpQAOBUE7TtMI2
    hdpiGmGbzYqamhpYLBbwvAiLxQyz2YyLLmoFx/EYGQkiFsucBw0luBD0Sofz9XuWy+VS8Xhs
    d11dzV2jo76Sjo7TpkQiA0XRpqFZSVIQjSYhCIbp/YzHs6xhSD4VkkopKCoqdoTDCkmlVFRU
    GOj4+EBclhUO+Slmuk6ea0MTun//fl9tbe1vli1rW3nwYIddb1qiKBpOneoVAV6sqbHBbLZY
    e3sjJJnMTBtVnucRi2WI3W5xt7TUarJshNFI7bFYsWN8PMUBXAGPwQBCwvLZs51+Jr+TBZPr
    5m2A1QL6fphBJ7alLc2Nq1Ys5zXWQ7i5cQlpW9psfuPgYYV5zGNMiPUX1We0LujyEUIw5ffv
    /u/f/v7laCR6cSwWEzqOHRc/+uH3c1/82jfl3z/6eDSRTE6y6LsPQNLjdl16zx03bfjQ+241
    lxZ78wp6mhWtnrdkSeA5lBUXCTpRgnmY6nwdh+LiYvT29g719vamFUWx672fQTWomorGhnqI
    HMWBQ0fQ1r4cNrOA9WtWsDaQeRas0WDA8dPd6B4Y0Wfg6h2dFnWyFEHeMbbbbSgrK8Nrr+3F
    Tddfiysv3wZCgJ7ePnT39MBoNEx38ZplOMlCYejCSSrMUGRTqdTeV/a8Nn79e66uraupQvWd
    t0Fjtd756HymTeZ0+kAn47HOXS1LaklLQxWgaZic8uPk2T69ID9b4JFm51pqwe5s3Dc19duf
    /PRnl65euXxZVWU5NAaD19VU46+/+Dl0dBwjHrcTzUvqz1kT1VSYjAbc/+H3chetaEUkGsXH
    7m0XvG7ndJRMQBEIhvGfv31SjcYS+sSVERaNRuZ77gVNV/TOS1FKqeByuYzl5RXv6+09xOtR
    QCyWIi0t5QaTyZTOZrM+lhvvY/eNsr+X2RoWJL+/+MUvk9dff/33V61qNyaTqSa322VKp3Oe
    pqZ6VdPk8P79+6ZCodAYgD6j0Tixffv25Vare8fAwJg7Hk9NN904X7lSoQFWVQqj0WwSBMEk
    y7JYoO/mte6Ojo7U1q3bhux2+9ZsNnwOS5tSIBxOora2ErIsIxzOIBaTkcspbCJRvjmDwcCD
    4yQ6Pj6SwEwXrAvqajZ7a/VhDIlECm63G+l0DqOjU6zVpIimptrpfPpMDnhaVM+Z/bEIRlg+
    derk6YsvvvSY2+2+IhpNoKtrADxvYMbs3A5ievvTc9uhclAUkKGhJCunMsBgkJXR0WG9T/qb
    UMm5VFtQSrOvvfba7h07Ln9hctJ/08jIJNHvkSyrOHGiGxMTQRiNJhIMxt50zwjhMDkZJ9ns
    GO/x2BGNZvl0WmVdx2Y6tXm9HD11qiOSSCTGmfyOY1ZTk/kYYKqTLgo2+uKdl2+/uL6uFqqq
    QtNUFLldeM/VO62HO45VyIpiZUQDinkMYn9nO4GhIx3Hvnmk49gWAHVOh2PtX3/9286R0TEf
    e4NTyA8ByCypr73xgY9/6JIbrr5cEEUBqiK9qS74TbXDmgpCgCK3g2M5LzNm2vHNK+c6PDwM
    AL0HDhwcCAQCK4s8rmkImmoaXE4HPnHvPfi3n/4SHOGwYXU7HHYrFFlmFzofwTy5+6AWisSi
    Bbn1+IWwUN/idoIQoKW5CTaLGd3d3Sj2Fk23UbRazNBUDT7fFExGEQ6rZaYv9LkYFrnwpRCV
    UtrX0XH0SH//QG1729IZxKLgvFDQm7qQ5AY6M2xDUzUQaBgem8TYpD/HjEeczMoJzpEwphFC
    cpTSrtf27vuv//7Nb7/24Gc+ZSHQjb4Ko8GASy9Zz4ZYKAwh0MlqM+VFmy++CFRToSrKdJcs
    /U4+9uwr9MCx01FmfIcKGJ4LHQiiFaBQoJTii1/8woqensGLw+Eo0aHcVCoHVRVtS5YscXV2
    dnIsJZBkim5RjIWqqvSxxx47Wlxc/C2j0bjMYDC0NDc3r5mYGMt0d3eFZVmJAhi12+2+q6++
    5rJ0Wr3+zJl+iySp5zTdOH+N8MzXlBIIgkEURcEsy7JpNut4riSjvr5+adu27QcrKyvuDAaj
    htl54MnJMByOSbhcHHjeDL8/cg4Rh1ICq5XHxMR4dnx8IlyQikstogEGIYRkszlMTEzB6XRA
    USgymRxE0QBK83C50ShCVYF8B9/Z9cAXDkQX7KnW29sX3LRpy6vl5aU7YrG0UNiucyZPyr9p
    oMXspiv6R6NRgCSF5FAoJLNAJFLAy5nPQ56YmPCPj4893NbWclEoFK/L5eRzGnKEQglwXJrB
    9jz03tmFz3g8i0RCKRh8kX8/AAePx4hYrD/d29s7gZnyxvHZ0bqw0A2mlFZcvGH9Q7ffekux
    KAiQ5dy0l3/DNbu43a+81vLSq/v6KaWjTHFkZkN9F3DRcsiXBEkAqmPxeHcsHndiphmEH4DW
    3tr8oS8/dP/G7Zsv4aimQVOU8xvc8zTrAEdgs5gIR4hRo7SQdUzmww6UZRkAfG8cOPjEE08+
    1fbBe94r0mlDkYcah0ZGYbE5YBII1q9ZAU2dMS6aqmLSH6JHTvWkKZ2eAewvqEde5CAY2L37
    Fex57TXkcjn86je/Rzweh9lkAs8TyJKEX/z6ETQ11OI9V24Hz0oROW7RksCFMHR0cGh439Gj
    x65rb20WCfJsdgICRaVQZBm5bBaSlGNjMPPRo8ATCATgCAElhJHFNPQPjdFEKq1PtYoiXw86
    r3yg7t0TQuKSJD37u0f+sP3y7ZftWrViGUApe01AZS0Vs5l0vl82myw1vUaOgyhwEDgCQvL3
    gCLvcJ3pH8ZvHn8hpyiqjwnvAKtNDl+I06Xf2w996IP45CfvtwD8x7q7++plOc+U1dmg4+MR
    sbl52VKfb7I7FAoPschbj9joIjlXyUAgMMT2f2JgYOAMc3QVAHG73Ra9/vob3hMOp27s7x81
    6ZDzmwlXbx5RqBthllvkBUEwYGb8Hb8Ahi/t7+/bW19fc3x01Lc+H4XPKGJBEGA2C1AUCclk
    noHN8+J01ylB4MFxSa2z80iYjdPUA4XkYhlgSimhlMJut6G1dSnMZjMoBSory9l0KxWCIKC6
    ugSaBgwPh5DJKOcblXehRMrpXCulNBePRw+Wlhb7BwbGKvIt1LnpCJzj+PwIVGF6fjE4Ls8v
    PXdgRP5MTSYeiURIyuVyKZbTj80n+i0IHFVKafq11147dvvtdzza0FD7ma6uQVE/Lx1G5nkB
    oihCFEXMVAzwzMjy0PtKFa417yiIsFozyp49RwKKooywYHCY6e5z0h/CfJQipRRGoxG5XK60
    uanpGw8+8OnLl7W1QpVzICwPqFENpcVefOmzn3DE4vFth4+eTLA+rDnMTMRR5gH5nS8Sl5Af
    WaUwj2KCUmpjO5IBwDcvqf/slx+8f+OOLZdwGovOp6OmtzC65xhlAALPAYQIoFRnHc97oATb
    u1w0Gn3sp//x8y0r2lu3XrR6JbKSglw2g0g4gjcOHYOiKGhuqgdAkUgmwREKnhDkcjn4/EFN
    1WiI5eLGMTNFRVkEZVionClAsOnSjVizajl4Pl/e85+//BU2XbwOLU31UOR8dyyCPExPVQUE
    BHzeAr+pxc58W3bOWpuUzWY7jhw9NnXLjddWxeNxDA0NY3h4BJM+HyLRKFKpFBR5BuLlCYHR
    KMJuNaPI5UBFSREqy7zwuuwYGJ3QNI2mWNSbYAowN1/ngJVM5QCMne3q/t3jf3pyfXtriyuV
    TGBoeASDQ0MYG59AKBRCPJFAOp2GLEnQHS9B4GEyGmA1m+B02FDscaGyzIvq8hLYbWY88tRu
    bXjcF2bR7yCLgKcwj8Ymb0VEefjhh/HCC88b3W7PfUeOHH/f2NgkN2MspmFoRCKuks2bt219
    443Xgn5/IIx8N7sAmz2tzTeKnL0kNlQiwBzmAKXUwZxcVRTF3E033XxjOJy+LW98wZTz3CLf
    GSIWD4AjhJBC+V0I65i+/PIrY7fddtt/LF/e2nT48Cm3pgGCkC/bKi/3gudlGAw2RKMxGI0G
    ECJMK2qjUUR395lMf3+/Pgd6grXEvKDxprNlixCCVCqDEyc6p1+7rKwExcVe9PSMTpPsOC7v
    h+SNHZkBv8i5xnehM6gLflft7e3tW7Vq7VmbzVYhSQqcTiecTgfsdhvMZhMMBsM56QSAsGgd
    rFc1kM0SSBIHo5HQQCCUphQJFvkm5+tAF8ixrChKtKen+4mGhuYrR0f97Yqiwm63w+12wm63
    w2o1w2jUh1YUIiv5ftH6GiUp31M7k6GQJAKPx0DHxk4mp6amJpjsDrLoN45ZFQxzjoDzhAOV
    5HK55atWrvzqA5/51A2XXLxB8Pun8jlVRc7n5xQFmiqjurIM3/nq50r/7Se/uOWFV153Z7K5
    PxFCTjHjocOnC718Cru8OrPax7pCEUopKfK4P/XxD71317bNF3M6Qee8HbGoOmuK04xR1gCo
    ioJzOm0vQHgL2sWNHj9x8sff/PbfV1y+fWtjMpkk2UwGvf0DKC6rxOlTp3Hm1HHsP9QBt9MO
    jhCIAo9sLoeuvsEc854mMDOI4YLgq9nCm29qIJGpKR8mfZPIptP5qE3JjyUcn5wEoOX3ZLrm
    WgUohchziERjwMzUq0IXUZ2vMM/q0DX4xoFDXd/9p3+t9Pv9JBqL5Qdq8BxEQYDJaEA6ncbo
    hA91VeUQRQGRqISR6XGKGkwGEV6PE4dPdOmtKGPsY3q+Bq0wQmds7TeefeHFY06H/bJgKIhg
    MEyy2QxUVYXAc3kPWuBhs1rA8xwUWYEkSUil0ojGYhgZmwDPczCKAkwmAyUAnntlf5oZ3wEW
    /eoDJuQLcbYopXjggc+4vd7iTxw/fuqhEyc67ZTinA5UuvEaH49wPF/avG3bVbf393cajx8/
    AUVRhAISyYUMhNBzyTqvJEIIESmlPADtnnvu3pDJqB/s7x+zUooCstX5I923ygHnoUNKNU0r
    lGGyAFmhlNLsiy++8PJVV13z3xs2rPpAJiNZTSYTjEYDbDYDotEQqqryfYmTSYnVlXIMfhbo
    wYPDMU3TJtm56o2IpMWQ3a9+9auYnJygRqMRDoedvXfDdG9vnufhcjmmI7jCGm9CeFgsRvC8
    ylE63SqYR35e75si5LnKiT4worPzTHjDhksOr17ddhkhHG9lpFIArFQMrA2kGbmcCp3Znkc7
    ZiJSWQZ4XsWZM7FkgfwmMc9hFgXBpIY8W7unrW3ZC6tWtbblSWw2ko9481eFDfcrKDfLIzE8
    z0MUuek1ahqBpoEqigYgqezbd1bnIfXj3EqgcwLPORtgVVVhMpl2XnXVVf/44AOfWVFaUoyf
    /efPkUmnC4xZXuFpqgK304Hbb7gSf/fXn3WvX91+wy9/+6eGnoHhnwHoYN6AXvqx0IdO6NJn
    R+q52Q3X7tr2vuuu2iFCU2dFtufP96LAKOslSRoFUumMflC60qDzFRK22bqyPvDi7lf+MRKJ
    /M0//O3XKq0WM3728O9ACEEsGoLDZkVlWQnuuf16pFIppFIp/N+f/opOBUIxFgFNEEImC6BI
    LNRT1Xt7i6IIWZZ5RVGWN9TX7zzdeYb4fD7mBrP/RTV0nulG55mzMw6Kpk1P+NE0DVP+ILxF
    bk8wFCkB4KKUOtnwihR7/wvNW0Z7evt+ZRCF1hXtbZVrVrSjtqYS5SVeuJx2RKMxPPncS+gf
    GkFTQy0u37weHAEi0Sh8UwGMjPswNjGFIyfP0rN9Q3HMlPNELiQCYZ27JAC+051nf/XD8H8k
    Nl968c7SYq+5vrYdS+pqUOz1wGGzwGgQ4ZvyY3h0DMtbGmE0CEin04jF4wgEQxgYHsXw6AR8
    gRD27D+aCkViY4z41MOEOFgIXS20//Jf/dVfNafTma8fO3bipvFxn3H58jYYDMZzBrPrnysK
    xfh4nMvlbI2Njevuq6mpr3/11d2/DYcj3awV5oImg81qfqEVEDxzeXj8A1az2XZPZ+eZakVR
    pyNfHerNDw4gczLAgiBA02RNlmVdVyxYhgkhUiQSDT333LM/v/baa62hUPrurq5hvry8GIoi
    wOUqAqUEoihgbGwK6bQCg8EIt9sGQuLSyMhIkAUfE6yT2PR0sPkMhdB/7vf7UVxcDAD49Kc/
    VWS12m5IpXKlS5c2MYj33HxlTU3peUcn6pOlDAazqa2tray7u8sjSbIL+RLTHACZ8R7oQhwX
    VVWloaHBp+vqlmzNZKRLotEkksks0mkZkqSB4wSUlbnhclkhSTJGRmJQVQKDwQCz2QiLxQCr
    VYTZTNDffzo9MTEeLuDBJBaKAuqBkaqqifHxsadLS0uLcjm6S9NoeSgUQyyWQTotQVEoNI1A
    FEV4vU4kk/l1C4IIgyE/ftBmM8FmM8Bo5DE8fDLn8w0HwuHIACMf9xe0jn1TwDSfHLBzw4YN
    D337b/92RWtLEzLpFO7/2EegyhJUtTACZi0pqQqjyAPUgHtuuVawGMXVf/3dH1ydzmQTTCHn
    LjAKnq0PKQChrqbq9huu2VlltZimJx0BFP5AECdOn8VlG9dOz959q0g4z0TV4A+GNXpuyc+8
    oY6CR44QEqKUHqaUTmaz2crTnWepYDCRocF+VJaVYmzCh1AkiuOnzsIg8jAZRUiSpLHL5mdG
    OAggWVVVpY6Ojp5XWcxjbQ5Zllurqqqu2Xn5jtvff8/7mi5avSrfcVJ3qFTdyCqgjFSkPwu/
    lqQc2bn1Yu/Dv39iy5ETZ+yxeNLBYMUBlquZd+czffKWqqpvKIra/9EP3V3Z2rwEBIztrKn4
    2cO/w8nTXbh251a88OrraKqvwuWb1kGpLAFtXQJVluEPhvDQt/5NVVUtxPYvNJv9vEAjojHD
    +MLEpC+Ty+Uu/swn7jM77Vbs3rMPTz63G9s3b0B7SyOeeuFVHDnZhQ/eehU2b1iN51/Zh6lg
    GFdsWYf1q1rhD4Twqa/8oxqKxAIs8u0D0FcAXV1QuuELX/iCJRwO//Urr7xy5+DgKMxmG0ZH
    JyAIxulWlIUtKQEBmkYQi0nI5ZLO+vqy61atWpN6+eXdqj4KDwusZHi7DlTFxSUrBgYmd0Wj
    yTf1eK6qKkUqlUMslnoTQWd2VJxvLyggl4spkiQXdu2a83qXLFmC97///SCE4Ctf+YoGIBkO
    h0disdghjjPeabGYObfbCiBLRNEMTdMgCDycTgsMBgWalm/eEIvFsul0WieHBiilYQDZH/zg
    B7j//vunZ5vP53HfffeJjY2NVZqmbYpEYnf29PRt7ekZsOQjyJnZv/lyHnG6Blh/zv6ezWYx
    trRsXNHc3G6ORv3OaDRiHx4ePhUMhgKMPT/vBjC6k/Xiiy92fehDNQeGhnwXDw1NEX1GMSEi
    yso8qK31YnIyjqIiG8rKHOjriyGXU5BKEfA8hShSVFXxdGioPxaPJwJMhsMLdaALomBKCJFe
    euml0263+6fvec+NLceO9ZeHQgmUlHiwalUzxsaCGB4OoKqqGDfffBlef70Tx46Nwuu1YMmS
    UgwPRzE5mYYgyKirM8PnG4719fUNMfntLZgoJZ8v7TofA2zxer2llZUVAACDQYTL6WRGV2Iw
    tDRtgFVVgipL+ZIRjqC4yE0Enq9Gvql8kBkU7gKN2jkQCaW0dNXy1m1tLUtIfjiDOt2rd//h
    Y0gmU+AI8iSn6RGEb4anQTVkczkMjE6qTMHq7NE51z6y9XCYKWPSIdrMyVOdfR/46KerKquq
    vCtWrBROd3ZCFHioioqX9ryBp194BXKeBU3jyZSMmbKFHPufztHRUSvyYxGVAuWizmMfbUuW
    LPnO9ddfd2NtTW0ZxxEyPDyC/r5+1thCyX9UlIKvVTalZ8ZhIYypzRGA5wjZvH510eb1q3aE
    I9FLXnztYM/ZvuH/Qyk9xbx+3ejNR0krAEKxeKwrnU5vEXgOiqxOdxO7ZudWRCIR7Dt4FNdc
    vhnrVy2DqqnTndEIASjVEI0nZcyULYSYU1A4ZGGhhDEFQJxSejYcjgwpslQiyyI6jp3EyPgk
    Nq5bCZ7ncNGKVhBQLKmtAqUaQuEoOrsH0LakGrUVJchksgiEojkmG6MsbzTCnIXcBXAmAABG
    o9EVCoVWDw+PAiBQFBWBQBg8b4QoGsHzRgjC+Z75MppYTBXNZls78oPa4wURyEIRhLdA2uj6
    UChWrLee1I2pxWKCy2VHOJx6ExnrfJ2w9C5EU1PhrKZpGczUMc/Zkenr68MzzzwtOhx2U2Vl
    BcbHJwBAkyQpWF9f1VNfX14zNjbqcDqLYbdbWA4TaG4uAaUaZFmmkpSjZ86EswUkNsXlspuq
    qyu83/jG1xQAsqZpha093/GAy8vL0dzcfPXw8PDfDg+PNHIcb1IUBUuW1J8DM5/7+bklPjNM
    5BkSkaoKRoOhtN3jKW2preVurK4e+tXjjz/2GKV0iuWs5xyEFOSlNUppSlHkTqPRkKOUmGYc
    JR6xWBb9/UHU1RUhnVYwNpaYJjQVEt1UNafFYtFEgfye09BioVEwaxySjMdjI4B20mAQNwCE
    FBU5UVtbjlAoAUJ4RKNZHDrUhcnJ6LRzV17uRiIhIRJRwPM8VDWrRqORKOPpDDHi1dTbpbrm
    Y4Cjhw8fful73/vXppUr2s35Bvwq0aMkquUNnsbyhpqqQFVkyJKEaCxG//Tcy5lEKhVnRIsL
    7oH6FoJsdzsdLjOrUyWsPCWRSKK7bxC3XLODDQ5gaNR0GUv+6/zv55X26KQfZ/uGcww6mDaA
    cz1wTdM4k8l0z5KG+veYTCY1nU7HkslkTFVVsblxSYvJZHTaXUW8gaP4xIfvwZ7X96vxeJJu
    2bhOWLmsGZqiYHzSRw4fOyUeOdFZk0pnOYMoNFvMRslkNIiJZBqjE/7diqo+zYxJfJ4XsnjL
    li1XfPvb3y4PBYPoOHKYsXTpdGkPYeUz+b2Z2TMdztejX1VRIMsyZFmCw2ZGW1MdKStyWcOR
    WPPZvuEVzEDJrJwlO18POi8gieHevn7YrWZMTEwiEAwiGo0hl8vCHwhheHQC45NT+P0TL8Bo
    EOB22OBx2lDicSIcjSEaT0oMNo0gL7yJC8mpzoqC0wBGxycmu/oHhtZXV5Zj04aL4A8E4PP5
    8etH/gRZluC0mbH3YAeMooDyEg+qyjbA63YiEo1hbHIKwXAkywzuBBNi/3xYnm/3GBoaSrpc
    rrONjY3Lg8GIPnRh9hAGxkg1MIOc/9xoNMHt5tSurmEV+YbyViyggmKucpNviXouhOpw2JFM
    ZiFJyjkzgWeXseg/EwQBJhPRxsZG4wWku3nxJj73uYcqFUX7QiwWrykvr05qmhbVNDVFKanx
    +6Nem81kVVWVhEJZhEI+arcbVU0jJBKR+GxWg8EgErNZQFnZEscVVxS1iaLocjhcl9hsTioI
    opHn1VBHx/6H33hjfzeT38RcSIHvf//7kUgkNr/xxhvt4+MTqKurh8ViY3uhnkNCK6yvzX+c
    baCF6TIaQeCQTkuIxahYXGwu5TisZg6XXl0yL4NXMK0pR6k24nDYsi6X3WSzWWGxmGEyGSGK
    IoxGERaLAaoqo7zcxkhYBIrCQZY5dtVSajab1RupRFAwDGShI211QiWlNKeqWojnuZMul0NN
    JHJCNJrCa68dA88LaGgohSAICASiMJtF1NV5oGkEhw8PQpIAk0mAIIhQ1YicSqXCLNUwXpCq
    eUs9M59WlMrIyMh/fP0b3wiaTKYVhJBykh+rJYDMFJHlm9PnN4VSSjWqUVVVFVXVwiwsn7oQ
    7P4dvJnAsZOdp/a8frC2rrqClYUAqVQKLQ01IIRidHySlQAVEK6olodaWTMRSZbx2ydfVsen
    gn6mBIMFMxznvCSe50pcLteqmurqYko1QzqdJgQgSxrquFA4wk8FI/T40Y7M0+lUamJyKgQg
    fqarx3zRqvbSlctaHG6njW9qqCVGg+gNR6JFACjPEU2SpJwsKzEQDAAoYrWa882nS319fcEf
    /ejHlYLAc4RSTqMq0TtNaYXM8AJ0QCv43rmfa9OTpQaGRpGTJBw51ZViytqNfIkJv0AHS0mn
    M70/+tnDEzabpSKRSAGgMBoMMBjyJKe66gp09Q7i1Jle5CQJuVyedexyWMERQiOxhF7LGiZs
    3u9ipT/YXU74pgKnvv+TnydtVostFo9DykkgHAfoE7kona5ZVtV8OZLRaIDHaUcoEqOpdCbO
    IuAA6/CTulDiVQH7OXPllVd+f9WqlVI2m22RZcWtaZoZIDwrCyEFo+lo/qNKKU1TTYur3d2j
    oc7Ozh4s7t696REMBvZXVZWcFUVxqaZRtigCo9EIACgr8xQMluemP9fJMHrUZDTyWiDQHxob
    GxthCjEwH6crT+AiRlVVl6TTmbWKotoFQeB5XoAkKVw2K3GEKFxRkVeJxZKZkZHB+MGDw0GP
    xyM3NbUVWa0OryxzJkoJMRhcBre7pJbn+RqAUE2Doig0o2nySC4nVSA/REBiUO87Pnp7e1Fe
    Xh5uaWlJl5WV8ZRyHCEcVxhZzm5qcW7Lztk/0/eOwmjkYLcLsFqp1tU1oQIowgIncBU6qel0
    ZqKiorK7pMSz3mgUSb71JIWaB6uQzaowGASUlRnA8xx4ngPHEagqkMupCAYT2VxO0tvGRjEz
    n3oxbIdKKc0qitxVX182XlpaVGsyGVhTjXNnKOt3boaxna+pVlVKJyeH09lsLszuWhCzZgec
    z1Eg81gkD8BNKdV7ydYgP4zB+A7/R+9fnGXGbAT5ge2TAGIFJKfFMMAmAFvKS4vvtdus9ZRq
    TlBYCCEcxxG9flfn880MlJ92FiioRqmkKNpUIBKRFaWHeYAnWEvNUQDpuXhcrDm6C0ArgGae
    5ys5jngIIQZFUWsrKip25XK50UAgMMyUgp4jyjBj5UXewTFwHJEJSJqCJjSN6uy/KPJlSVOM
    BR5AvsPRXPfTDOAyAKvYWVay1+Pe4Szn+jVlnuppAGcBdBFCBtiZz5csQSilZQZRvJ3n+Rsl
    WS4XBcHh8biMDpuVt5iMnCDwBABRFAWZTBbhaFwLx+KU5ziVUhrLSfIZAMcBHGVs/GGdBHgh
    kWUBO1QAsGRJQ+1nclnphkQyaXO7XIaK8lLe43ISk8lAOJJn1mcyGYQjUTrpD9FYPEGtFrMW
    T6SCiVT6NIAORlQ8ye5bbr4Eu7dxtj2EkDqj0dhMKa0HUAxQJr95DiObqEPZ55TmHWiNUqSZ
    IRtkef1R5NvRLqYhJjzPu7ds2bLNZrNtz2Sy1ZIkFecdBd1HIIUlMqQwAqIzD2Qyaamnp2cy
    mUz2AugEcIZNzAkD0ObQMQmUUnH9+nVVdrtjmaZp9aqqlgGwUUoNra2tWyil9ceOHTvZ1XU2
    kUymkkwWw0ajgZaXlzudTlcRx3E2tvYcISRJKY0pipJIpVLJYDAQjMXik2xfp1hP47noF7Jz
    584Wi8WyIxqN1qXT6SpJkooppca3+Vs6M0YU+lnTcz+f+bmqqtL4+HhfKpXqBHCWENIDwEcp
    VRagl+FyuRzr16/fXl1dcy/Pc3UA3LKsmTUNAqUcl9+i/DAYjiMQBEJ5nmqqqmqBQDBz4MDr
    /T6f7zil9CjTx31MB16w/Op2ob6+vmT58lV3lpdXfNJsNhYpimpKp7O8JKkcI9KTGdKaQA0G
    jooioYqi0uHh4eSBA3u7/X7/UQBHCCHHkWdAJ95OfudjgAnLZzqYIXYxz8gwRwOcAxBnQw10
    CFC6QKUyWxEaKKVeAHUAGtlHL/Pe5tImkRbAnhkmUEOMyaY3QlDmuGZCCLFSSt15RYdiBt8J
    AOoIIZdQSgeYY5ItMKRplue1MINoYetVCvLQeivPhJ5PZ8KbmKtCZGUfXmZ4q5Fv0u+cw52g
    8/h+hjldY6yUxjdXB+Y8d89KKa0EsARAE3MC3ez+FdZok1n3TmHe8jgTiF5CyBDzUuXFuH/s
    7nEAHJTSJgArAbQAqOEIcYIQkRDwAGFlMBSUQsv7fFRvbxpld60bQDchZJDB0coiyQhhd8nD
    7uK0gzcP+Y0yNChAKY0SQrJ0sQR4RhlaAZSwO1nD7qUd5x/FSN7GEdRLziaZwz/C7mJmHmsW
    mb4rYmsqYoiOYdeuXZslSbK/9tprw5qmaQwZ0Ks7FLav9gIdSQvkV5dhienCQAHqkZ2Dc0AY
    wbGc7VMVO9O5RKlzlV+VvZcJJr8TACLsvi40OCrheb6B5/kmURQbeJ4v5nnexHGcwHGEJ/lk
    NNE0Ck3TqKqqVFVVVZKkLGtgMgigl+njSXaWi3r3bDZbdUlJ8SqALOU4roHjuCKO4408z/Es
    CGUtPzWqqoomy7IqSZISj8cSiURyDPnKhW7mIPgYb+dthXJekQhTdiLO7SzzTgKsw3RywXPO
    7f/mqQStADyU0mImMI4FGuBChRNkgpKZa86BrcfIlJ6DCaINgEgpNRcIpoZ8U3Ed7s6y92Fm
    RscCwFAw+o0WGOD0rPxWjs7dO9D3ykkpdbL1mOawN/MRbH34RpxB+MmFGBTduWL76GVn6y3Y
    Q73meHadp75Xet9nP8vL6NNTtEU0wISdt5tSWsaUY1nBGs9XS17oJKTY+eulZkE957bIToKR
    nbOJfS7MUX5lJhO6wygv1tpmrVNgcuJiTr5zVvqCvI3+orPOXZ+CFCmELechv7oc2gpk2EQp
    NbDvWdnvSCyw8LHXUZnOsTKHwgRAmFV+pRbKB2Z6vM91fSYAdia7zoK1LJb86twGXXYTzDmg
    C7x7PGuWVFQQkDgL7iD/FvKrt1CNMfmdYg5LfLGcUz0KZkGJkzlb5ezpwkwLU+48Tr5+1zJM
    r0wyZ0VPtaqLaYAX+n/oO+T4FkvB6FCboUDJGM4THWEOCkeZpXCmm8/PQ4D1BgB6cbswy7iK
    yDfJzxUIoT4w/HxzfwvXpwux/tTmQ0jQheI8r3Oh+dDzRSKFJSB0gQaYMKWmn6ven1uYZdzO
    t08yZkgkejnUu+IAUkqNrP7Zxp4mtm7ubRSM7iQkC5ypzDsJ7wLlA+dxRskcZeJNY/QW0wAX
    nDM/y9Gf3cGKzOEOaudx+ud8/2a1ZiyUE54ZE71HPAdA52AkMJMf5wp+X3gLx0tfo1rw9VwG
    CqBArwgF6yOLKLso0C26/NIFdrQr3Ecjzu2vL57HuJ0PxcoUyK/uANJFlg9dx1gK5NdyHhty
    Ph2TY3KbKHSm3mmN5AKE+IIf74Lwnu+9kQW8X/oWH+e87ndyWPT5nW+h4N71PV3ss3w3z/0t
    zpa8DST5VsaDLuQsF2jgzqdQyDsYOK3g47u5vv/Pye881zgXA3xB657DWkihUWU1pdqfY1//
    /0l2F0l+8VYy/C47qNw85fe889rfbo3/P8s9kQmH8aUUAAAAAElFTkSuQmCC
  }
  set pieceImageData(Fantasy,45) {
    iVBORw0KGgoAAAANSUhEUgAAAhwAAAAsCAYAAAAuG2H4AAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAAIcAAAALADPaAVuAACAAElEQVR42u39d3hc
    13UuDr/7tOkDDAa9gygEAQLsvYlForpkyaqWbEmWLUuW5V7ie32v80vuTW6qEzuOnWI7jmNL
    sSTL6p0SJfYGkgAJEr1jgOn91P39MfsAA4iUCBB0fL/reZ55BgTAwT579lnrXWu9610Ef3j8
    4fGHx/9fPAghc/p9SukfNm2B93S+j4X+LD5q3X/47P/wmO85v5yzI/xhc/9wQ/7hcfk34u/D
    eaGU/s4c5P/LR2KhPq7f0VrJrL/5B8P2B7tx0bVmr4fjOFBKP2BXCCHzXrfw/+KHb26gxWKB
    oigwDIMQQujs//P7BDqulCP5XV7j/21RV/YZIIRAEARomvZ769jZmgjNLJpj9zchhHDM0RgA
    dPb6B8czj3uEkMz2sjPAzToHxu/b0s3PedY1TJ3f2ffc75uj/L8xOLyY7zC///toO7LWS0y7
    YRgGx+yHaTf0y7Ub/8+EQ4QQ8DwPwzBM1MYByAWgEULULIOhA9B+Hw+56fBM5JkNniilmQ8z
    YxHNQ00vJQK+ktfyYTdXtjP/fQZI7G+aNyKXtbfmeaG/L5EKpdTC8/w2i8WykuM4h6IoQUVR
    jhBCRgAkACQBpGef8T9k9D763JHMB0/uvfdeT3FxceP+/ft7Dh48mGTnQQOgANDm41SuZEnF
    BEizwUb2z0ywmnXm6Uc5yyttNz7s/X9fbcZF1i4A4E3zzOyGAcD4fbEbHo+HfPKTn1zKcdx2
    TdPyNU2Th4eH219++eUzmqYlCCEJACkAarZfmev6+f9XwAallBiG4aaUSsx5lOXn5/+txWJZ
    mUqlugkhIgAL28yPRHKzb9zfxcMwDAFAPqXUQgjhKKUSIYRn4KmcPTVCiARAzIp2eQA8IUSk
    lApkeuH0cvZ09nO2Icj+nnkwBUEwnTdH2IOtc4aDv1g0cKG/ebnrvtiTrYUH0CQKwsd4jrvR
    MIwoIURjRuSSzsqVPNcX2Iui1tbWH//xH//xp+69+86toMb29o4zMUrpKHMoKgCZrfsPj0t4
    3Hvvvc79+/e3FhQUcOfOndM3b958ZzKZ/GFRUZGtp6enN51O29h5MIMVyvaafFhQ97twNua5
    aGs7bnvllZfyJyf9JBaLicwWcNu2bbW2tZ2or6qqdJ05c9Zg9tE82ybIFgAIhBB+vk7/Uu/f
    2YDmu9/9Lv74j/94ymZgukxELpS5vtD7XK7N+Kj3u8j7k8985uHS7duvum7Tps13NjQ0uNra
    2kIALGzh+n9VRmz2eimluOGGG3iPx/PNY8eO/c+TJ9uvUhRte1lZaeHZsx3nVFUD8yvy7EBl
    rvv5/xKHw+JwOL5ts9mK/X7/TwHkbd++/VqXy2X5xS9+kVQU5VVCSJpSGmAOWs1ylgYhRMd/
    cTraarXcuLhu0dcm/YGu0fGJ7xFCwgx1VrQubfye024rPnik7c8NSt9hB8Phdrt3OJ3O8snJ
    yU5VVQ8zpJpk0W56Pof+YulY83sskzQFagzDEAF4AXg0TbMxMATm+CKEkDghRKaUyswhGtkR
    ZRY4onONhC62fsMwsmuUHyipUUpx/fXX4+WXXy5qXrL4B1dv27D5+VfeHOsdGD5JKVUBRNmZ
    SF/qzbfQDsYEZJIkAQA0TYOu64rDYU/u3HEVvLk5+PWvf52ilBIAlWz/EgBilFIu65r/kN74
    kEdpaWlDKpX6RXFxce911133g1QqtXZgYKCksLDwMxs2bAi/+uqrRwkhiaqqqmhdXZ3C85x+
    +PCRZCgUUpmhVpg9+S8Bp1/4wuMOUbT993Ra237TTbc89dOf/uQ5RVF0APGVK1ffHo9r3yoo
    qOhbt27d/zh48KCPEKLcf//9NTk5OVfpus75fL6Dzz33XJeu63KW/VDmei2z79Xu7m7U1tZm
    3ztTN5DH4yEPP/xpt8Ui5T/88EPOVCotpVJJLh6Pq2NjvnhnZ2dEVVWZOUGFUqpl24ysYMVY
    CJtxoSwxYfW1CwGfe++9k3e7PU8MDQW/kJvrFhVF+RMAAwCSlNIAIeS/zG5c6D2PHj1Ky8vL
    Iz7fBOnq6oPHk2+MjY3Fk8lUOdvDJKU0xvbbmK/dWBDA8X8JO16oqalZ/ulPf3r3r371q6WH
    Dx8+zHGcBYBtyZIlD5w8eVKjlKY4jsv3er1Om9XKT/r9w6lU6h0A48yxZ6eVjI9yvgv9kERp
    5a5tGze9d+Bw8ej4xG8opQKAKICcitLiWk+O23vgSFsRgHwAttLS0sc3bdp0i81mEwKBQOzd
    d9/9t3g8/jQhJAAgwA5Mej4H5/bbb8dvnnsOuqYBgATAgUyJyqPregqAnxCiUkplACuXNDb8
    WWNDXRHP8aJBDU5TNWPS75dPnGp/M52WfwEgTAgJs99fDECilE4CiLP9Vk3DTSnVWF2RzudG
    o5TaAOQYhqEBULIMk8KAmkEIwcsvvwwAntrqikV5uTnEHwwdMLMI7PcjppE0jZjNZoOmadA0
    bU5A7TIyd5wsyw4AVrYmqa+vf6ivp4f2GBrefHvPBAA3gBYAXhbB6oSQOLteBZdQQvxdROG/
    hzbDXB8/MDDg6erqum7dunUFgiCA4zgYhpFXUFDw4OrVqwuam5src3JySuLxuEvXqXbbbRVH
    jh079nxbW9sQM9QxZj9M20F/F7YjcyYt9nhc3e7zxdd5vWIfIeQAA/tE1/XiaDS9yOWyaIlE
    vACAes011zQIgvDfhoaGlhBC4HK5ujdt2vTXe/fuPU4pDRFCgszuqPO496b29NZbb7HV19fn
    OJ3OvI9//OPCq6++Oh6Px2UAypo1axw5OZ7/5feHNyqKbgFE3mLxUIcjX120aHF/KpX6++7u
    7j5KaRRA9OGHHy76+te/Wn7vvXcHjh8/Hu7sPC9TSs0yl0II0VjgaMwXdKxdu1Z46KEHPR/7
    2M3c66+/mU4kkpSBdpXdQ7pZigoGg5LXW9YQjcp2uz158r339g4DKGF2MTbbbmSDmQtldRfy
    nJgl4p07d1o9Ho9jcHAQx44dM8Lh8FhBQZE8ORm1lpYWxY4ePSTn5LibvF5vnihKjp6eHl3T
    NMEEebPLK78TwJG9GTzPQ9f1qdS5+fUFkOF/hRHRotHo8Pr167F48eKVn/nMZxatW7fOMj4+
    joqKivJoNPqpvr4+DUBhWVmZfeXyVgmGrh44eOiGrt6+FzRNPwxgLOtmS2enpa8kydSMyHme
    O/ryG3v8Y77JM+xv57FfmThy/FQbz3E1lNIxAAUOh+PahoaGmzo6Os6MjIyc27Rp07XNzc23
    Hjp06Cyl9Cy78WT2nPPCn376aQeAZkLIUq/Xu7SwoKC+rKy0ghDieX/fvuPJZOopSmkSQBDA
    qltvvH79Fx/7jEgpBTV0gBo4eOQYHvniN/V0Wo4CCFBKRwCkFlWV/8+1K1srhkbGx/sHh3t8
    k4Gzmq6fpZR2ARhnzjJ5GaWBVWUlxX/udNhjfQNDLyuq+iaLOGIA4izTQgEUezy5D49PBPJ/
    8stnotFYYghAGYAQOwN89jkmhCCVSk2Vi5hRMTMzVyQzxnHcuurq6q/m5ubmxGKxeDAYDETC
    4UXDw4M4d+48EolEbkV56e6y4kLJapHIiVNn3o3E4j+glPoIIREGmn7nkXe2DaCUIicnB3a7
    HRMTE1N240Llud81h4dSikAg4Hc6neMjIyOFZ86cWdXU1KT39fWhuLgYoVCobsmSJZ/Iyckh
    uq4jnU5L6bTqNAyyfOXKNVsXL1782v79+98aGhrqZ84mOvvcXinbYb7vk08+Fbn99ruPlJbm
    Lu7t7eyUZdnNnEV6cnLidElJRW80Gjpz5sxZvaysrK6kpORziqLUOZ3Olw3DcMbj8S0lJSW3
    iqI4oaqqWTpKzxVwbN26lX/iiSdqJElqfeKJJ1pkWWlMpVLV6bRcWFBQGNu5c8c/T05OjOq6
    HlNVhfj94c3vv39icTqtguMEcJwAm82KtWsXFzU01N9QXFw0QAgmYrHYkMVivSMY1O6wWAoC
    V1113cCmTds6A4GJs729fZ2dnZ1DiqJEWHYmnVX2whzX7+E4/vs2W27pAw88fPitt15/srOz
    M8TeN8aCUXrbbbdZq6ur75ZlbeWiRfn64GB319DQcC7ztwrL8BJ2/5rn3SR3E5ZxNYOgK2I3
    HnvssUJRFP9bLBZb0tLSIre0tARFUbTn5OTSkpISeL0efuXK1estFockCDbB4RCHeJ778zNn
    zp5lNsO0G3Oyv8LlHmjzabFYkEwmeQA2AFYW4ZnRos7q3hp+x7XjLKOlTUxMnH/vvff0srIy
    /k/+5E88zc3NeOGF57Fp00bymYcfXvSrJ38l79u3P3Ty5Mn+s2fPKtu3bFhy4+4d63/2y6eX
    TAaCrwL42az0ezr7QFwJo2im/TNlChodHB4fTssyBVCclc7Xk2lZZQffA6DE7XbfFggEwmfO
    nHkSQGhsbGx1eXl5EYAaAH5KqcocTnSetcTVGzZs+Nkdd3y8rKG+XqysKEdhQT7Onj2Lb34z
    UFReWrSJUmq0nznn7+rt77BYJN1ut4u6roIaBjhCYbVa0FBTsfimneu/rKqaFonG0kdOdYbL
    S4vL/8fXHrPJslwwMDTccuZc980Hj52KvLn30FvJVPqfKaWjhBA/gPCcD3yGeJtoXtJQefsN
    Oyv+5K9+6Bwe852nlEZYjVohhCiUUuTm5n7qK1/8wuO3XH+N8PSzz0p/+b1/XJdKp83P2gqA
    M1OqzHEKAG4mhJRRSl8FEGPALnU5hu7DIhWPx3P9t7/97dt37tyJWCyGkeEhDA8O0GVLm7Go
    sgzLmxsKSgu98LjtCAUDeOwbf7yirf3cEgAiK6uk55vlWoiUbhZYI5FIhMd0rR5ZEemMe+x3
    ne04evRobM2aNcPNzc2tqVSKvPTSSwIAnD/fhYGBAU6SLEWLFi1SS0qKw5qmdimKbDidnlZN
    M2qLiorubGxspENDQ69kLomSLNBhXMlMqfleg4PDRNPkMVGU/LKcdjHbkQKgqKrioVSJpdMJ
    nRBStGrVqi1Wq3VtMpk8+sILLzzV2Ni4rKSkZD2ltNJms5WpqqpRSpOEkBAD/Ze84LVr17pV
    Vf2btraTO8fHffZIJIZkMg1KgWuu2Wnk5pb+fzZbgepwWFRdT+5PpxVJVXXougFKDVCqQ1U1
    ECI6i4qaH/B6oQsCp9hsCNtsxNHePpQbiSi5TqetNifHtiMnpzK5aVNVl2Ho32tv7zgNwM8C
    oMRczzulFPF4RJYkp62ra2JLRUVBY25uTjuALkppmPk3GYDR0FC/pqdn4H8dP95ZXFBQQqur
    81e43e6JaDQ6zNYgsLMwBa6/8IUvtNrt9ut7errffvrpZwbZ2ZdnZ9QX6pGTk9Pc1tb24IED
    B5w8z8PhcMHrLaDptIZ4XMHJk/0uTSMuTUuD46xYsqTI7na7VwJIUErHWOk7+TsDHOYNYhgG
    AIjJZLLV5XJeW1Zautxms7rjsXhqeHTsXCqVOkgp7WI3WTyr/mdcaeMxKzUlUErduq6jsrIS
    a9asgSRJaG5qAscRWCwSdmzfZuno6Ch+5plnPP/562fGXnvzncn9h44G44lkmOO4AsMwdgM4
    Qyk9RQjxMZSXMq/nCkSvUxEgpbSosrLi4c8/+ujSnt7eun//xX8IqVRqH4CC8rKy6770xce3
    a5rO/dn/+ct4JBI56vP5Do+NjfUCCFksli2LFi0qHh0dTbHyRx4DGhLm36mktLa2Oh///OdF
    QeBhaBoIKE6rGtasbLV86p7bClRFwd/+8F9zu3r7KaWUz1wHi2pBYegGKksLhY9ft11wOa2g
    hu768x/+R0EorkDgebjycpHvcWN16xKupbHWc/h4e30yla5jf1/NynJc8iFiN3jf4WNtb/Ic
    HowlElYAtQyEkSw2NlwuZ/26NauExXU1WLeyVbDbrcWpdNrBHKLBShOUUmpBhrC76urtW/+s
    tqai/D9/8+KWYCjyY0rpJDPOwSuQSSAcx4mFhYWorKxEPBaDKBB43A7iyXVj0jcKSeChyGnE
    Yhp8ExOIxRMcS+2aYFP8XTpvSZKgKMrUvx966EFPfn7+si9/+SvNmqZ5VFVJ+f3+/kOHDnUP
    DQ2FKKUpQkiKfc6aWYq40qAj+2+sX7++JpFI1PT39yMcDkOWZQBAIBAEx3EghIPPNyHm5uYW
    1NfX55WVlQSTyYhvdDQcLS4upvn5+evvvPPO6mAweHDPnj3v6bruQ4YDlGC244oGYV/5ylfW
    +XwT954/372opqbqznXr1uUcPny4Y+vWLc0Wi+Ped97ZV19dXVG9bt3axLlz55KKorx09uzZ
    w3l5eQWVlZWbU6mUqOt6OpVK5QDIAWDHPBoOZFnW0um00NbWZg8EQuD5TNbCbndA1ynX2TmS
    l05rKC31orLSuVyWk26T2jGT5Ei4UAjWZJKCUuIoKBA9OTkElAKaZiAWkxGP65iclO2Nja4a
    WZYbAEQY2ksxuzHn8/CjH/1z8vOf//zzixYVXS0IILquVzE75GOgPZk5o6Tc7w95g8E4OC5J
    amqEfJvNlheNRsfZ/a9TSg0A3Oc//1ix1WqpSaeNb58/P35tcXH5ya1bt/7l3r17z7EgKIAr
    QDDVdV2QZZlEIhEWzBLwvIXIMoXD4QZAIAgieF6CKNogCBwNh8N5zHaYmWtuzryGy0XPABx2
    u/3Tu3bueOLWW26qaW5s4OxWC8LhEE6dOm0889sXR97bf/BZVdVeIYSMMu6A6aiN32HEsumG
    G2647/777+fLysoyiNnQYbVaQA0DuqbCIolYtXI5WpqXWK679prq7/3d99Mvv/r6hGEYpXl5
    noZtmzduCgQCwb37Dv4rpXQvAB87EJGsEsuCGkT2Pjav1/ulLz7xxB2ff+xzQiwScXpy3bu+
    /4N/9Dgcdsu3v/nV5Z+8715O1XQMDQ2t/9GP/zmlG8YpACguLv7Exo0b1ySTSXr8+PE25mQu
    K8XP1uQfGRkJxGKxfLfLBcPQwYEiz5uHUDSBf/2PZ8ARDla7S3rssw817ti2JdMWZr6HQVFT
    XYH6xY147u3DUOQ0QA3EUhqWL22EIPCghpF5XwIEw1EkUikRQBVLXwYBTM6TF1BWW1u7dMfO
    q9HY2Nj6s188dU8oHHmFfX5hMz3qG/ft+ccf/8stnR2nva+/tScVCkeHmYFJmZEHpVRyuVwP
    b1i35lFD10ttNiknFAoldU1vAdAE4DSlNE0IEbDAbeiEEBoOh0/96Ec/mnz11Vetvb09cndX
    d5oQuL/8+CPuv//hPyXHxn0xm9XCu5wO3jB0bWB4rJf9d5E5jd9pq5UJNioqyrm7775nYyKR
    /PK5c71b/P5QnqZRzmazGwUFeeFrrrn2xPnzZ59+77332xjJLsQCFuVKg47ss/PYY4/lEUK+
    euTIkUafz5f1O9wU2DBfo9EY2tpO8mNj4wVNTY3ukpISV25urkQprENDY2JFRfW6LVu2OIaH
    h0/39PR0U0r9jPi94LaDEIL9+/fjN7/5TX0wGPrT998/0Dw8PAqfz1+xZs2KjxUU5Fc6HLnN
    hw61lY2OjiEUiuSsX79y9+Dg0CuvvvrqmbVr11bX1dXtTKVStTabbfL48ePtaqZdgc7X+b3+
    +uvyrl27hpxOF4LBCDiOByEcdN1AIhFHZWUuNE0Hz4P6/dFFo6OTnKJoyFQaMtZD0yiGh30o
    LPTC6bSA5ykkyUAwqEOW9SlQwnEEgsBD12UjEAiUsIAgwmwGudgef5jduPHGGx2UYu3QUK8o
    yzRv8eKl94+Njf9meHi4g1IaZRljeWJi4vSiRVUned62yul067FYYDgYDCYYaE5SSmWe5+mX
    vvSlLdFo/I/D4dASu93hFQReMwy91G63r2VlW53Z6wW/RycmJoaqq6vP7dq1q8FisSiCIKbt
    drvo98e9TqfNkOV4yDAAq1XgJYlHMDg0MjAwmLhcu3FZGQ5KqeBwOB789EMP/snXvvIld3Fh
    PnRNga4qoHoJWpfUc1s3rK74i+/98OFfPfNbt6bpLxJCLCbImi9/YC6OkZG7ilpaWr721a9+
    taqsrIxpcRgAe6WzXjkCbFq/DnU11da/+8EPy3/8Lz+JhMMRn80ild5/560lY2OjD4yOTWxI
    pNI/B9Bh/jmzhrdQ2Q1zqwVB+Nhdd935yEMPPiBxHIecHDe+/pUvS8lkct3Q4BC57eYbCKgO
    iyjg85972HLs+PG1Bw8dUYqLi2t3797d1N7e7mtra9uv6/p5BvgC7CBfzv4nJicnA9FoFJ7c
    HBBw0DUN9XWL8Gf/60+gyLK5l0QUeAGUwh/ww9B1UMMAqAFJFHH/3bcjnUohLaehqSp4jiDH
    5YAkigDNZJwJKMZ9fiSTaStLB4+w0h3/UfyACxkSQkjrx265efkXHv0MxkcGpGPH25bu3X/4
    PIAgpTSXEDJJCJEVVd3/zt73T+8/eHi9b2LiPQCjWZFMinWr1Fy1dcuX//LP/r9FAlXwlW9+
    R3nqNy+/D+AM+3MuXDlyNqdp2vsvv/zyV5DpQskBkF9eVrrD0DV3OBJNRKKx/kg0po1P+BPM
    iA2z11iWo8OHtRheice9935i+8jI6A8OHTrWOD6eaQwTBAsEwcJbrePesrKSXYsXL2lQFPUn
    hw4d2s9awX3s3Cq/i5LPypUridPp/NihQ4du6u7uJqbTmw00ZoOPyckAjh49YVm2rEVMJJIT
    qqoEBcFeGQ4nSpqbV3y5pqZm38DAwI9UVSUsa7Pgpeb7778fP/vZz1xWq/Ubhw4d3jQ8PAKe
    FzEx4ceRI6e8O3Zs2dnXN8L7/VFIkg2hUBw9PUOFy5evWG+z2c7X1NRsSiaTuaIodhw7duxQ
    d3f3IAP5IUyTX+f0OHfunHH11VePO51OiqlOEkDXDRw6dBwWi4WBEJ5QSgSAgySJLBPCg5DM
    c2QkAJ8vAkEQIQgCCMlE6KoKEMKD0ky2QxR5pNNxkkgkCpEh0jsv5V6cbTfMfy9Z0pg/PDy+
    48iRUwLHOWG3u6rLysqWDg8P+wGMMZsU+8lPfjL4yCOfe62lpXZFf3/fyXfeefuMqqpJZjeS
    ANTrrrvWmk7Lj+3ff2xbKKRg+fKlyMlB+3PPvXggFAr72L0cwAfVYi/bblBK8eyzz46tXbv2
    O5qmNfX39+eOjY3ltLa2rnK5SryUUvXgwf3DwWAoJYqiLAhCWFWVMVXVJtj9N0OPYy6243IN
    4eKrrtr2xNe++mV3SVEhVDkNQ9dg6Bp0TYOhaSgtKcKXHn3Qcb675/pDx05OMPKoqQWgXsl0
    okm2lCTpmjvuuGPbunXrYBj6DICBbMCR9bVBDXi9HvzR17/MeXLcOX/xN3+vvr33/aSqpFFb
    VV4SjcX1RCpdxDopstO+C5b6YuWqxRs3bvz6Fz7/eY/L5YKuKTB0HU6HHd/4ype4//NXf4M9
    77yLG67bDYPoqCovw+c+85DnzNlzm1avXu0+d+5c8NixY68AGGSGwo8MeS1oZpnmubzk0NDg
    4IEDBzYODvTj+PHjSCQSSKWSkNNpqKoGXdeg63oGZFCWUMnUVFiZyGCfE8BzGUMuCQKsFglW
    q4QclwP1NRXIcTnw/uE2QzcMJQvcGRdbu3ngGdg029fM6IcahjF+ur09ePLU6aKBvh46Mjqe
    YDe1DRktFlOHwLNieWuVqqqyb2JiiEXYMfY065eiKAqCzSKCaBosFslU+bSxJ5ldSrmQeNJH
    gaSLXSchJEopPQRgiKU7m6lhiN09fTAMXWLOLAhggoGNIZYCnmBZgwuCzisJQL72ta+WBgLB
    /753775Gn8/PUrYCeD7zNAxgdDQIQsTK+vqmu7u7u5OBQIBn+2h2BFxxDYOtW7d6A4HAvR0d
    HfbMvWim9T+Y3ZgNPmRZRVtbO9fS0lyQn58b1nUaEkWeV5Q0398/oKqqWsyi3QSzHQtqC/fv
    309uv/32244dO3HXuXNdHMdl1sVxPAKBEA4ePM43NzciLy8PkUgUHMdjYGCMlJQULdqwYUNB
    OBzO4ziu/fXXX389EolEWObPD2CSRfJzbovNmDRjoLy8TA4EQtby8nLY7XZIkgRBEMFxPHie
    Z3tpvhKW4SBsf80uDo6dy4w/NjLxIzQNSKd1xOM6CgqsiMWGFUVR50zczmqxNUEHDYfDCafT
    MVRWVlqnaSLsdinNWp/Ne10CQJqbm0TDQMPY2CQ3MeEbicfjAWYvYoSQBKVU0TSNGIYhGQbA
    cSIMg8DnG6c+34SVvZeQtV46F7vxUbaDEIJoNKq8+eab7QzUlFgslqqKiorVlNo4h8MilJaW
    csFgKMqEA027McbsiP/D+BsfJg53WSUVi8Wy6dprrqkpKy2FIqemHXaWAzc0FRWlRdi1bWPB
    oWMnVwKYYG1iYWbA9SsFNsyST3V19Y27d++28TwPqmtQ1Qw4IwQfzHLMAiIWScQjD3+KS6aS
    +X/1vX/w//q3r/gB5LDWpiXM+ZjdHgpz4vrlpkYZ2LAWFhZ+7pHPfqa1oaEeuqZm3pNS6IYO
    r9eDB+6/F9//4Y9Qt6gKixvqoVMDm9atIVs2bfA6nS761ltvDTGgYZYLJgkhEyxSuRzAkR4d
    Hfvpf/vv3ynctHHDll07tov1dbVwOR2wWiyQRAGCwEHgeXCEgCNMCZWVsnRDh65p0FQVqipD
    kRXIchqpdArRaAzhSBShUBi/fv4NnOw4p53u7AqwDIN54KMXM3pZ/CI3MkJpPkb0pZRSHcDp
    555/8S+PHz/x+VQqVTY8OnaG7YUpB85RSinP80tWr1xe1nnufJztYRhAyDy7rNUOe955N/WF
    L39DFQVOfmPPe8PsdxVkNEYiWTcnnWXM6FxKQmY7JuMQcGydOnNYE5TS3Oqqyh3bt24sskk8
    rt++0bVn/5GGgeHxQwxkDlJKe1laOcbIfwZLkxpsf4wLqdQuZAlDEMRN3d0968bHfRAEKziO
    n+WwM44nEIijtDSvtqamZl0gEDDXG8viTV2x7MYDD3wKNput5ezZs8sDgSAI4eB0OiFJFqTT
    MjRNu2B2I/tV1w2cOXOeX7myJTcWC40LgqDn5DS48vLy6jdt2rTt2LFjejqdNkm7KuapizP7
    8Z3v/Hek03Ld+Pj4EydOnHDpug6eF2fs79jYJDweD5qa6nD8eCc0TYGmafD5glJursMrCEKy
    ra1tIBKJGGxdQVZCnmSlCXU+azt27Nh7TU1N/+L1eu8bHBzO9fuD0DQdmmbAMOgUgGDHm2U1
    OHZGhCnehwlQzSyHJImwWCRIkoj8fDtKSqgxOtqVOH78yDCl1ATY4Y/KQlNKsW7dWvELX3i8
    Yu3aNcHDh4+YbfP6T3/6s+htt93+N2vXtog2m319X19PT09Pz2SWL+UB0C1bthREo+nWcDgG
    URRlZjcS7O9HAcihUMjS0uJMbdmyTg8EojLPG+G9ezsmmJ2IM7sRp5QqF8skfJTtMAPu2T/O
    4l7IAEJOp5O/8cabbiFEWppKGeB5QVi9+qo6UZSSx48f72EZ5V4GOCLIEEdN8UOOUSMo48h8
    qN24rAyH0+msKC8vE6bKE7MzBSybQABUlBZzgsCXaJpexhY+yv6+fKWMBnssWr5s2drFixtw
    rvMsXnvtNQwPD6G6qgqfuOduxuHQMxG3kWFCm1+bzlESRTz66U9xg4NDef/x1DMGIUTcsX3b
    EqskVr/6xlsWTdMjzMksNAlsww3XX3/XDddfz5mS7DAzA5TCMHQ0NtRh+9YtePLp3+BrTzwG
    h8OBtpMn4c0vQNvJU0oqlZKR0chQAAQYj2aCASXlMgCdTik92dfX9+eqqnD333fvVVs3b0Q8
    FkMymYQip6HIMtLpNFRFgaap0DUts7/sTHAE4DhA4DhYJBF2q4QCby6kqnJYJBF9A0P4t6ee
    1zq7+/wsQ9PPXsfxIToA7EYjNqv1ExaLeG84Evs2pbSPOdc0IUSTZfn5kbHxliVLlnxizDeh
    Mu0QPetmFPLzvWtWtC61dHf3xFlUncjKcCQopdaiAu/nSooKFj3/8mvvsHPNsZuyD0A/a1OO
    sbNexqIckwSmZF2DOe+EzlZZZWdZUFXVA6CUPT0sG6NTSsPsM23JzclZa7FYhXAsCUGycl5P
    bv7A8HgBAxMOAA3I6JwYrCwUYAAkCCCZ1WOvzCarLRToUBSlye/32zIOxXzyzKlMPykliMdV
    we12L2IlIz+7ztCVLqucONHG5+V5tw8NDXkcDgdaW1vg8eSBEILu7h709w9+ZGklAzp0nD3b
    za9Y0Vzo841ORCIRLpVSG2tq6koikchYe3t7kAVgpu2YN+AwM3o9Pb1CcXHxJ0+ePLUsFApn
    gQ2Std8cenuHUFDgRVVVGfr6RmC1SvB6PUgkYkil0om+vj6BRe1xVs4aYxExI0fOnXN06NCh
    ifb29n++8cYbVVEUH+3vH7QKgghJskIURQiCBEEwAYU4BTTMrEcm25F5Ugqoqg5VpUildFCq
    gOcF2O0c7ew8HDt58tSIruv9yAhujbIz/pGgbtOmzctE0fL9FStW/dvhw0deZPeirGma8p//
    +dTxm2668cni4tKVGV9HE1l7QQAYbnfO4v7+rnJZVgye53UGcuIAopTSuCDw6qZNW25PpfTr
    AP1cd/fJw/39A1IkEkky+9ZLKR1m69VvuOG6osHBIeX06fY0c+hK1vnns3k1s9vNAXC7du1y
    VlVVFUuSVE4pzdd13a6qCkKhcOLcuc6JnJycvKKiol2jo34nxwG6LhOLxeNobm4pb28/3eZy
    ucQlS5aUl5aWVVitVqrrupJKJUI+n2/y/PnzgcnJQIzpJylgImzZOkkzlKYv56ZMp1OxYChk
    UMPgpoEGS59TYzptTnUEw2Gq64bIatpOZISKuCthLLI3XZKkrVu2bikbHxvDX//1XyMnJwe5
    OTlYuXw5RIGfAkWzSyoEdMa/3S4HvvjYw0L7mU5326n2MZ7AtrSx3vHK628Vsig6wsoUccxD
    EMV8eDwehEIhAHDU1i566L777ytyu13QVGVqjQCd7l4xDOzetR3Hjh/HvgMHsXb1CuzZuw+6
    QTE6OmKm9jnmaEPMaEdw+S2aOlNm7R0eHnnlf/zP/291SUmRM5VMobKyEpIkTpVLCCGsCDkN
    mExCqKHr0E1eByiSySRGx3yw262IRGL0fG9/iBmMbgBdAPoIIePMiX+Y0bNXV5XvvvqqzZv/
    /cln7wtFoj9jrcBRdv1pSZIS69au4QcGBvL8fn8cmZZYs1STU1VRvqK+rhaEIxzLAnDsbyqU
    Uk0UhY9t37L+3tGxiTCA48xBCwwQdbO93wkgUlSQvy7H7Vw3MDT6S1lR9jAHYxKOwQz7B6IZ
    1mrb5HK5bm5oaLiqvr6urqy0NNftdls4jpBEIqH19/XHjh473pVMJnM/df+9wvZtW2C3WZFO
    JbHvwCHyv//ib2ujsZi9tXlJbmNDndubl2shBEjEE5pv0p/sGxjy9Q4MnfUHQvsopSdYJBtj
    ZzmdDTwWAnQYBgXH8ZQQjmQDDLNGPxOAcFBVzc4A1rw7JObINcAPfvCDwmAwuGtsbJzU1NTA
    6XRBURSEQmEMD49eICtz8RJLKiWjr29ErKgocPb390eKiirchkFJMpl0Mm5BmDmXODLKpPPe
    46997WuQJGlpV1fXPV1d3Xw2wMguSRDCQdMMdHUNoqWlEX5/FA6HFQ6HAIfDBoCIHMdZjUyI
    HM/ifiUuI6CihBAlkUgE2traXt+6dds1RUVFzRaLhEQiiVQqjWn5GjKjhJW9zybZdHqveTid
    NqZpQWkwOJw8fbp9WNf1PgDnWXRucpc+lLd2993XgeeFjb29vvVut42vq6vt7e7uCZmtwJRS
    XRSlWCAQMCRJys/Ly1P8fn8SGXVkarVaeE3TVwcCEbvdbjU4DlJWeUQBoDzxxBcWJxL61+Jx
    2TU52Xv85MlTfcwvhgF019RUJzZt2rzWbrenBUEsIkS6uby85lQoFP758PBwiGWZTMBhZf5G
    mT376bOf/WyJy+W6Nh6PXxcKhVqi0Vh+Oi3bDMPgeV4wHA5HavXqdUNutyvS1zdcePZsD9Jp
    BRzHw+PJxZIlNfk33HDTVpvNeY2qEk86rVtTKYPneUG3WPLT9fUlwaam5T2xWPjwqVNtB8+c
    OTvAyLMmP0yZbTeEy0HTiUSy7d139wZvven6fJtFnJXdyIAPAopwJIp39x3RKaXpWamdKwY2
    mKCKd3FDw01bt2wWvXkePPboI/jt8y8gkYhj8eI6cByBYeiAQVlmwwA1MmBxcGgYBEBpSRGo
    YUCjBmqrq/C5h+6zfeXbf5w7MDio1VYWSw6btTAaT+SzmzHHjBRx+Yzz9ddcs/vadWvXEl3X
    s7IbFHJahsCTqb122G248bpr8NzzL8Lv9yOaSEPiOdx7x63CD//535xME8Uk6c7Qg7iM9RmM
    uxKilL5z6PBhB4DNixc3rPmzP/8zV1VFBThCwRECUAOGoQEMZFBjmuejayo0TYWmqjB0FUdP
    nMJnvvhtI5FMpdkN2M+MxlkA5wkhAywi/yjDx8XiiSjP83penudmRdPciUTyKGtpHgYQV1U1
    WVdXS+rr6or9fr/ZrsYzAZ7yutqa2qLCfIiZATBuZrB4piZYt2ZF65c3rlmR83/+/p8PMeDA
    s8xLLwDXhvXr/ui6a3YuPnHihBaNRGyrlzeRJ599qWxgeCzN1pHMAhzibG4CpdRutVrv3759
    +5fuvPPOho0bNnAlJSWwWqQpQGzomiWdSjlee/314hdfegX333sXbFYLdF0DqIGiwnzsP3Ao
    f9Xylvzrd+9AjsuJzEeiw9A1KLLsDARDhZ3nu1vefHf/9a++/f6R0fGJpymlbcywBbMydwvC
    m9B1rbOoqDjV0zNkN8GFWbPPdiiCIMDpFGlvr0/LSlnzV9J2UEqxePFifPOb39zW3t7RGovF
    0d3dg7ExH5qaliAYDEPXDfA8f0GAIQgCqqsrMT4+CVlWp67J7w+hqCjfVVhYqNntNsXhcDla
    W1ubBgcHezRNczOHE8RlduH4fONCfn7hXWfOnK1JJlMQBHFWqUqAmVniOB7BYAzhcBSLFpVD
    kgg0LYVIRIAo2u1FRUWWkZERys6lWTbWL5PXowCIDgwMdPf19f59aWnpTo/Hvb6jo6Py7Nlz
    IETIApwCOE6cKqNkSioi4/qIrG1ThMViwapVVTQcnlCGh0fivb29o6qqdgHoBNBJCOlh2ZmP
    VEf1+QIoLzdkjkMcIC3bt+/8Sk3NomPvv//+0VQq1Q9gkuf5ZCKR1HXd8NTU1Dj8fr9ipl1W
    rVrlSqXkFeFwnNhsVsKCawcybbmkrq7WAUifHx72tXg8lsmurvODbE8nAAyVl5cHd+zY9ejo
    6OTuWEwj5eUlwvh4QiwszFm7fv369NNPP/08E1QMZQEOsCAIAOB2u/lHHnlkczgc/m+HDh3a
    0tPTYw2HI1BVnYE4ATzPQxBEm8eTl7d58zp0dHTB7w8xLo2AZHISXq/HlpeX29zW1o9USgWl
    3FRwwPOi3WKR8lwue11hoXvHhg07ztfULHrurbfefCedlsdY9+YMYvrlcjgIgPZXXnn1xZXL
    W+5/4L57eIEnMCid6i7gCEEyLePf//M5uvfA0Sim2fHxy+QPfBTZB7qug+O47bt3795QX1cH
    UeCR425CZUU5YBiwWi1TBFLQ2V0qBHvefQ8WScKdt900VQagoNi9cyvZsXWj++XX3o4k4glo
    ul4JoIKl7FzsAPCXeW3WioqKu2695Rav3W6DpigAY3QfPnwE//4fv8TnHn4QSxbXQ6MZJ76i
    tRmvvP4mfvHr51FXvxjLmxZh5bIW8pvnX3ENjYxKWcaaX0AioMrSwX0AnqGUnsvNzS3P93pd
    mqYhGPQjnUrBYbeipKgQlCPgdIASCgMATwgoz8EQeVCLCA6AJ8cNURQUdgOOsKxGOzMcAyx7
    8KEZGnZdyeGRsX976Y13t3/1y18st0nC3X/3Dz9e13a6458ZYBhWVTXlcjrpmjWrCg8cPFjI
    sj821hHRvLi+Lt9us8JqtQhZny0HwObJcd93z203rAiFI/q4b3KIfT/JzsGA3Wa74+GHHmj5
    xB234uSJo9KDj3459uN/O92TSstBZoRcLHuBrJIKyboGwW63f+qhhx7631//+jdyKysrAGpA
    1zUYup4BypSCI4DTYUftohrkeXIzrH9NYz83oMoKnE4Htm5ej6KC/EymzDCgMwArCjyKCrwo
    LsjD+pUtzm0bVm3/2x//vKat/dxPKaUH2SDAiQ/jzMzVr7e3tx+vra07PDw8ftXIyOSMskp2
    lqO4OBeKEkoODg6YzPg0rpDmTXZ9/NFHH80Ph8Of6urqcgAEum4gFovjyJFjIIRjDvCD2Y3M
    z0QUFxciFIpCVfUp8AQQDA/7uMbGavvwcF+kurrGKghCPsdxeSxQcbISWWq+GYRt27ahvLyi
    tre379b+/gFuel2ZSdmtrU1wOl04fvwMDANT6xoYGMfmzcsRj4eg6zZ0dwdQUOAWq6qqXCMj
    I0IW2OMWwHbohJBkOp2eePPNN98G0HP77beTZDJVqesGXC4bLBYLCOGRSKRBKYVhUMa5y3xt
    DlylVIeuZxxoJBI13n77rXAwGBxjgP8MgA5mm0YYmPtIUv+ePYeRSODVDRs2fqyrq2/3+Hho
    d11dzca1a9f+8t13330VgJZMJlVCOG1iYsJdVVVWffTo0XOUUisAa3FxcXkslqxPpWRk+Oqc
    iwEOAQDZsmXLrrGxyF3xuMy53cbI8PCIkgU4+rds2VI0ORnccujQSbvTWQCHw6VJUvLs2Jg/
    OjQ0QJEZHyFeyG6YIPVzn/vcmtHR0R+88847S4eHh9n5NgEczzogCXSdIhpNIB5PMiVrYUaG
    kRAO4+NBRCIJVt4iM0i8qmogFEohGlXFnBxHc01NTcWuXdeUvvrqKy9qmjbAMtGh7DKWMN8o
    gH0dm/T7//XP/+KvvUODQ7tuu+UGa0VpEeEIhZxKorevH08+8zz9z+deTqXS6XHGjvdlRU3a
    hzFt52s0eJ6HpmmFzU1Nn77zjtvdkihA01TAoLDbrJlIO0u9jhp0RreKAYqlTYvBcxx0XZtB
    hHU57Ljn9puEd98/aBkaHdcFQUgjI6TloZTmEELM1JiOeeiMsHJKw/r1669es2Y10XUdZpaR
    GsDe9/eh83wPBoeG0NLUCEPPMLQlScSi6koMjgUAXcHW9avgyfOgeUmDc2hkNBeAg1JqZwqZ
    PBZmfgZlCntxdsMMS6Ikp9Ip/OrJJzHQ3w+e57C0aQnuvP1W8BwBRUb8C5mvQJHVtQKaAawg
    GtvDEVaaOE8I6WWAIHYphDW27+mVK5YL9971cQhUxbHjxyraTne0sGgnoWmamk6nja2bN9t+
    +rOfN8disRF2LR6X09lat6hGEngOLofDzHCYgGPV9i3r79q6YTX/3f/z/ahuGFHmLEKEkCCl
    NKioqnr27Fk6NLiCdJ7rQiAY8ocjsTMMMMkfVhZga2/Yvn37F7/1rW/lZlq59alzCJoh4PqD
    AQwPDUGW0wj4AxgaHsH4uA8V5aVTvzM8MoKh4RGcOHka7R1noWsanE47qstLUFpcOMUFooYO
    geewY9MaSCJf/a0//bt7WCaGY5yTqXkRl5tB2LNnz4TT6fjHNWuWFdvtPYvHxoLEMKadtiRJ
    KC/Ppzk5hrpv3/6JVCrty+KZxLPtxkIrcxYVFRK3233L4cNHtk5O+j+Qyfio8gmlFL29gyy7
    MbPskkikEY/LluLiEt1utwmyLLsbGhbXdHS0j1JK3YQQJ7OLM7opLtU+joyMQNf1a3p7e2qT
    ySR4PqPrRymB1WpDeXkxOI6HKIqQZQ1mi286rSIWi0NRUtB1J2IxFZIkk5KS4jxBEFyapjkw
    3YWRope/6TohJEUpDQLgHA5HmFKKqqpK1NfXgVJAUTScOdONZFLO6kwhM0ouMye0EsrOho/x
    p84z2zHGbMklq+mGQkE5kUiQjo7ziMV02Gy5roKCvBUsYxIPh8OkoKBYnpwMkPr62sa8vLwT
    gUBAAeCyWKxl0WiiSFV1aJpBeJ53mmXAioryIklyPtzZOZ6Xl+cwJidHJlVVtWQR0v0Oh90t
    ilaan18Auz0XlGrJPXve6hofHx9hNlb4sPP76KOPOhOJxOP79u2bBTYyTyAz7DEnJxdWq23q
    fvN68xCJJKfOqiiK8HpzEI2mUF1dAkEQYRiALGuIxRRQOp0lI4RDLKagpyfmrq+v+lhzc1P0
    5MlT77NzbNoNZc6AI5v96nA4aDweNwDExn0Te/76735Q+uxvn1/e3LiYz811w+8PoOPseTo0
    MmqynGUWASaYnHI20eayZYuzIxRN03ibzXrnXXfesa1laXOmu2N2C+yFulLMrwHEYnFYRJGR
    YfWpn+mUYu3KVrJ25TLbW3v3K7t3XVU0MTG55cjxkyOsTuhmB2zeLbJWq237jh3by3Nzc2Bo
    Gk6dPo1XX3sNDrsDPT19KC8vx7HjbYhFY4jFYti0bhU8uW50nOuB3W5HbWU+igvyQDgOy5oW
    W1598518FkWZaxMuF3BkkRopu9YwAD/Hc5rT4cAjD38alOpT5FCBIyzNTzHdHosM2DDpHQTs
    59BYFmOclT/MVs4P9P9/yHmhACbaO86c/+3zLxY7rAIOHjmeYI6+EMCEYRhyMBhUb77hOuvS
    5iWLDxw83M7+RpHXm7eksryUEELg8eRwbO8kADklxYU333P7jYWKquJcT1+M/S0BQJpSGgPQ
    ZLdKt/7Tv/wk9Nbbe+Lj4z7NNxnoziprTfXjZxvh7DKXxWLZuHv37top3RjDJDVTaJqK5194
    gf7jj/6JhkJhsn37VcQiCYjF4zh2vA2V5aUZHhIBjrWdgko5TERk8JyCd97ZQ999b59RVV5K
    v/jIp/jrd20lyCIjG4aBtcubsfuqDbX/9Itn1zOiW4rNr5lX9J0l/W5+LsoLL7x4fNu2rT9c
    saLxs0uXoikQiHKqSmG325CX56LhsM84cGBffGxsLMSipDCrDcsLTWTNfo/77//k4vFx3+c7
    Os7aKSVTgONSWmEJIRBFCZWV5ejuHmCchGzQQTAxEebc7hIhGAxSRUHpsmXL1vb19Z5NJBKm
    iqfZATLnfb7mmmtyQqHQdQMDgyLAged5NDTUIy8vDwCXAZbUwKJFFTBbSYeH/fB6c6FpSVit
    TgwOyuB5CamUDknKdbndbk8wGHRTSl2EEJtJep7PnmcTodk9HgcAm82WpJSiv38Qo6MTbK8E
    BojEGQAjmxeWLVFBCCjjRU0AGCGEDLFsY3g2b+Oj1u73B1I8z51uaKjbPjkZF8vKvNrAwJk0
    gAIA3nA4rJaVVaSSyTTC4WhJbW1tQyAQSADI4ziuORaLOTMZAB2iaHMSQuyUUnH16jU7JyaS
    q6NRGRUVuXJPz1CCBSoqgPiyZa0Ojyf/EzzP2zZs4AZkWebOnz81NDHhM9umUxewG1o2T8Lj
    8VSfPXt2swk2ptuJM1mJ4uIiLF26hJaWliAYDJB4PAVB4FFWVoTBwfGp33W7nbDbRUxODgFQ
    kZtbgIICr+H3B3RKPUJfX5joOmac/3Rah99v5NbU1G85fbrdZxhGKmu6sAqACnM9MKIoQlVV
    Lh6P19nt9tsaGxdf19TYuHhxQ31uRXkp58nNgSjwSKeSCAQCGBoe4c539eSc7+6p6+rpI7F4
    IkwpHSaE5DDUE1vIzo66ujp0dXWtvGrbtsfvvecuG8+RaVIivbDQ12wgAkKxd98BVJaVYPWK
    pR/4fbfTgWt3beXfef+gwYHqy5c2uo8cP1kB4DylND9rzoAxD9DhqKmpvmr9unUCZc6548wZ
    vPTKaygpKUVrayuuuuoq/PznP0fXG+9goL8fdpsEiyRC1ih6e7pR7BahqTIEUURtTQWxWqTc
    tKyYdWIHFkiIatYI6DSABM/zhqqqOHa2EyMjw0gmEpBlGYRQbN6wDrU1VWZeg2U5MBOEZEyI
    zoxRiHFjzIF52hxSuhTAeHvHmb/40te/PS5K4s50WjbbXp3MuCt+v18tKMi3Xn/t7oLDR441
    6boeBFBZWOCtys/PA6UGCvO9nCAIbk3TLACWX7tjy9p1K1vJ0ROnMD7hj2eRwpIASuprKr7p
    ctibT57p+uWJk6fHWQZMYxmawazSUDLrOqbmJVBKidPprF60aJEAc3+mypTA22/voV/68tfU
    cZ9P/dwjn7X/yXe/g76eLhw/fpweP9GGW264hlBKoRsGznaep8Mjo2hpaSHX7NiC1cuacKLt
    lHbmXHf0L/7+n5y1VeW2xvpqGFkkb4HnsKK5QbRapdp0Wqlin8FEFuluzufEfL3pphsdixYt
    Wqso6vXxeHJDJBIr1HWKjMQKhaYBsRgFpQJZtWq9Y+nSZInfP+Hv6ek9Hw6HLQBszPGZ3ILL
    mrqaHajcf//9bk3Tvnjy5KllpuTzhwOMma8Agc1mRUGBF319wzN4Keb/SybTUBRDNIx0iued
    FBDzampqajs6OibYTI5IFtdqThfkcrnqBgeHlodCkSm+RnV1FXJynEgmk2hvPw1BELB48WII
    ggC73Y1wOIWysjyoahKlpRUYG5uAIFiYOKJgNQHHrJKivkCBigIgLkmSSing8eSiuLgYkmSZ
    6laJRBIYG/MjW+J8GmTMzHAQgjQDbCbBNYysNthLPR+hUCh99OiRf1+ypJmvr6+6XpKE6t5e
    HSzLkxMKBUOiKCZ4XsDAwKilubmu5eTJk8OyLBdRShrj8RRPCAdF0SAIkoPneZvX6y11uwt2
    d3RMWjJiZkZ6YmLCMAn9eXkesnnz9s+n0/w9sZhv/6uvvvBuIpHIj2dGEURY0DXAugzjWdcy
    FbwzUF8ciUQ8GekHzAAbOTk5WL58qT4yMpSyWETr/v2HBV0XsHbtClpcXACr1UpkOcPzcLkc
    VBAIEokEOjq6icuVj/Xrl9Fjxw5HGxoWi+Xl5a7BwQiZ3WWWSOgoLnaXW62WmmQy5WejHczO
    prkBDp7noaqqJEnSbRs3bvz6PXfftWzb1s18aUkxrJIIUB2GliEEMmIg0TUVcjrF+/1+19Hj
    J5f/+rmXqva8f6gikUw+QwjpYkbYdNDzYmib/8fhcKCrq6u0rrb2vz3+2KMN5WWl0yWRqewF
    qwXq+gVBBwBMTvpx9PhJNDXUzgIbGfKmQSnWr2wl5aXFwomT7dpD99xsra8u39jVP+xnB93D
    Dsl86s3FDQ31SyorKzK1el3HtbuvQY7bhe7uHpzpPI8DBw5ASSXQsqQON+zagtbmRvz4p78E
    x/HYuLoV/slJjI/7UFZajHyPm1gskj0tKw4GNmyYnla40HKSBiGEaJqGwaEhjI6OoqenB+vX
    roYo8BB41sFFgWwRsCmnOi3/QJlR0y6Uyr+U88F6wlVK6clQOPx9u92u79ix467jx4+3jI6O
    9rFMjz7p98uKIrtuufF6/qlfP9Pc3nFmAIBUVFCQ73LYQXUd5aXFxG63uaLRmKe8tHjpx2/a
    7bBaJAyNjNNYLKEywKECsFWUFD60c/OaDa+9c+CEbhgdzEin2Rk3o65JTLcXZpPpaFbWTzDV
    ZqcycZRCVVU88+xv6LjPFxVFkVu7arldTkTwF3/1N9qbb7+bKCoocGmaRggyGRHd0NHR3p56
    /PHH+R/+3V9KrY2LSHNjHRkb94V6B4ZTB44cL2+sq+IyxGljqrQlSSI4wjkBeFm50Mnk2eed
    SPjqV7/aVFNT+8TAwODN/f2DRYFAmCSTCnQdU6TAjKORiMViIw6Hw5Kb6yqtqmr1NjUtLx8c
    7HHu379fV1WVZ6A+hgXQrqCUYufOnWKmlfT0J7q7e7jpzhmOSWVzs/6dnfmYjiIrKkphsUjQ
    dZoFNqZFrCgFYrE053YLxGoVNFG0ua+6asc1uq6Pnz17djSrW2FOKsDl5eXQdb3F5/Pla5oO
    QRCh6waOHDmBsrIyeL15KCsrA88LkGUgHE7C7/fB4bBBEDQQYsPISAgFBW6Ew0pGsM/gOIfD
    bpIe7ZhW91UXKFBhwQohQMZ+O51O5Od7EQ7HkErJ0HUjK8OBWdkOMvtnBqaVWzXMY3aRqblx
    7Njx0WPHjv9sx44dE2VlZX+0aNGilng8PtDZ2WkNhyMiIUjYbDb4fH7U1y+qbGxsXHz69KkI
    pbQqlcp0esiyBkGQ7Ha7zbl0acvKQECujMcVeDwuKEpSjscTPABDEATt+utvvCWVstzHcXLk
    xIkj7/t8E2ZpLcFsxjBrTTY7hczHjHOiaZpAqUFMesc04OBQXl5GA4HJlM83Hi8qKrPpuoSm
    pkZKiBqLRiNWSbJIipJiXCQe/f39is1m1desWWM7f36CyDLPeTx5qfb2U/6rrqqyiKJoMQwy
    o+U6Q+2ExbQbWZnhuZNGdV3nOY67b/v27X/xjW983dtQVwdq6AgEAhmQoWeEnAxDy3SosDbZ
    TCeIgZWtTagsK/KUFhfc9stnXqyIJ5I/AXAqC/Em5ms8Ml0zCZvX6/3S5x/73PXbr9pKMroP
    M0FFMBhEd3cPlrU0Q+C5GT8joFAUBf/+q1+jr38ANVUVDDzpM7tvYKCspAArWpeIz738VvRn
    Tz2vyLICtskCpVQEIJDpIQBzeZRUVlZ6nQ5HFm/Egd27dmLXVVvxF3/zPbS3t2PV8qV48L67
    QUDx7G9fgmIAVEvjntvvwX/+5kWcOtOJ0iIvLCIPi0USmbGwUkqthJAFG9g1y/lzADin04kH
    PnkfBgcG8A8/+jFuu+UmOOzWKT0OysopWTnO6UzHB80DmQ9RjZ0pjaX0xpPJ5BuCIOx85JHP
    Lvrbv/nb9eFIZAyANuGbSCbiCdQuqsan7rsn/zvf/dNlaVnuzc/Pc1gkCbqho6ykGIX5ec5o
    NFZ79VWbqpYuqYemqZgMBKiqaeZ5JS6H/db7b79u+0QgJPcPjx/LikBCLMU7woDHhcoT2RdH
    k8nkxOjoqAFKuSkgYFDomoZgKESR0cwQOaqh41Qb3tqzNwggKkmik0wZdaDAmwcA6f6BwfSv
    fvVUYeM3vyDYrZIBIEwplSYDwTJD12ZK/VOK4VEflWWFMuNhwXRb8JwfO3fuxKpVq9b5/f7v
    HTp0eF1f3wA0jbJOA4kR0rgplcnMzAwd8XgKyaSGycmkJT8/t3nRouYyr7eg8q233nguEon0
    MQMcXgjQsWrVqi39/QPfPH78hEvXjQ+0XhJCUF5eBkEQMDQ0+oGfEcKjtLQQTU31dGIiQDRN
    n8HhyM5yxGIpkpOTK5w8ecxfVlau8Tw3HovFZGTkvjk2tXhOnTjBYICoqrooEomI2a2k0WgC
    sVg3cnJysXbtUnAch7NnRxAMxiBJFqxYUQdVTQBwY2jIh9bWWjgcdiSTMijlOUmyWBhotjLH
    wS+k3ch83pSjFBgYGMLwsA8bN67F6KgPY2MBiKLEOm1m8jZMkDENQi7JPl3q2gzGMQl1dHQc
    LCkpOT04OLyxtbV1i6qqIz09PQlN0xJ2ux3BYAw9PYPW5ubG1mAwENY0o1BRdNbObUDXiaW6
    urokP7+0rKcnyhPCwWIRIctRXVVVDoC2bdvWFaJYeG8splsFYeJAR8eZMLuoCUKIWaqfxPTw
    0+xy+Ixzn0qlwg6HMyEIgisD5KfPncViQSqVSBPCJXneQj2eEhQVeVNHjrw3umzZqgoAksnL
    UBSdOBxEP3bsWODqq6/35udzdkIssNttiUQiMUmIUSqKvEVRpv8GwMFqFZBOB3RFUaVZZ4bM
    hzTqoJSu6O/v7/nud787pCiK1dANm2HokmEYktVqcS9tahIlUUAikUD7mbNUTqczSJNAI4DK
    EaJpmqbpul6MzHCrbuYEp1TL5nOALRYL0un05ttuveXBBz55n8gRfDCLgYzBPnTkKMpKi1BU
    kA+ey/g5XTMw5pvAL556hv7gxz/FDbt3YFFVOdFnAA46pTViEUWsam3inn91jzo44utFRrDl
    NKZ15uf7yM/3em0cx8FgnAeT1EdA4Xa5IIoiorE4KDUwOTmJ/UdOgON4NC+pRVV5KVqWNODE
    iRPYsXEVeAKIPM8zhyFmMc6vxIPwHEdSqRR++8KL6O7uxvi4Dz/4x38CpUZGWVRTsbRpMa7Z
    sRXEJIxOqfcy4EEu35iZkUpWFuHI4cOHDzz04Kdu+eIXv7DmL//qb+LJZPLgxORkKhwJw5Pj
    wl0fv5U7dvx445O//k2OJzdH4jgCQ9dRWOBFfW2NIxAMLbr5up2SJIrQNAXxeNIEWTzP82vv
    vHnXVetWNgtf/5PvDzBCnJUZiiHGnJ/ELF2Li4HnVCp19ujRo+F777k7TxT4TEmFGhBFAUub
    lnC/ff4FaJqW8k+MgSoJGgpHggD0mqpKwnEEmpbhcFSWl0KSJENRlFDn+a4c3/iIkEgkNABB
    QkhBvieXEJIlFEQpkukUDp1o13XDMB25jssY9tfS0uKUJOlRXdcXFxcXR0pLSzlCOB7gOI7j
    OFGUBF2nHECYeqRANU2nTKuDchxPAY4aBrVVVVXfuH79Ov9rr70epJTGWWvxvDtXMqz+R5yJ
    RPIzbW2nypPJdJbQlJm94EEIQSKRhNfrhceTC0VRYQITh8OO0tIiWlpaYMiyzPX1jTD+wUx5
    7gyIISzytQihUDDZ0dHRzwiOY9m17rleR3V1Da8oSlE6LX9At4IQAbpuIMM31mCxWADEUVTk
    BcepEEUHBgaiSCZVpNMycnIckGWTbMiJWbZj3qDz4nw1C6E0854lJcUoLS1BTo4LdrsdVVXl
    rBWWh6Lo6O0dg6oaFyCPLoDRmBmsmLy0kM/n66GUviKK4qr29rNVa9euu1mW5d/E4zHZ4XBQ
    juPJxEQIhYXR4g0bNq3nOMFjngvDABIJjV+1ak2136/ZE4kMkVgQeKiqQiilXGPj4oqqqqVX
    jYyonqIiPtbW1taraZqNZcjHGAF2NKvJ4kOzNn19faP5+fm9bnducTgcmXEWotE4KiqK+IkJ
    X8pisWleb4kAGHGAhnleqsruqpJlFXa7B4qixBKJqOT1FtktFhtVVTXscrlThIi8rstZ2b3M
    WXe5BDoyMpLWMloM2XaDzgdwJCil//vcuXNF586dK0OmHbSMRfaFq1et3PmlLz6em5frxtDg
    EO751MN6T29fKKumFmQRiTlIamBGXuwySF+pVEosLy+7eeWKZfl79ryDLZs3wmHPlHqzyyne
    vFysXbUCf/v3/0gFUaANtTWE4zj0DwwZe/cdUI+1nUZrc6P02EP3EYskwoz+PkA2BbCkvobk
    uJxCKBJNseuKsGeCdW/M6x60WCwCsro3zFdCALfbCavVinAkCkWWsXffQeiUg6GlsX1TZlZM
    3aIqvPH2O4hEIjB0FUYmz8VhurWNu0IlFUIIRyilSKdTyM1x48brdsNikdB28hTsLieWNi1G
    gdcDjpAMgDNTHXRWOmOBEjBZ/KCR0dHRl97es2fnH33jG85kInHVD/7hH10+3wSZ8E3Q6opy
    4slx49tf+5ItEU9UF+Z7Sebs6LBbLdi6Ya1glUShpbF+ivwqCLyZPqzdvW1902c/cavttT0H
    jN7B0SFmnCky8uY+VjqM4yPmv5gcDgCd7723t+3MmTM7Vixrhcb4FQTArTffSJ75zXOeznPn
    A6Mjw4ZEijld14x8r9e9ZtUyDlPCe0BjfS0pKSqQBoZG1EQiofonfAiFI2kAemVZccnq5c3c
    FJBmRNP2zh4cbjsjs5JFgmVT0vOt3Q8PDysej+eH6XT6BZ7nCqPRaEkqlSpWVdVjsVjymptb
    Vp06ddYtyxpyc3NRU1NinDp1NKrreozjuAjPc1Ge52OEcAlBECLhcHgEF2jVnO8jLy+vpqen
    f0NJSTEcDie6u3uz+AH8jJRxfn4eragoUlmyk3AcMTgOiq6r1DB0W3f3EGdOQr0Y9yPThWFw
    hYVF/OjoWIrtcTSLczBn8GSzWTnDMGyZttGZqqJMEAuZZJwCh8MGSZJQWuqBqsahaVZkRC55
    RCIpeDxOBIMaOE4wzyI3y24s2EOSJACUy5TsM0TRoaFRaBqFzWZDQYEXIyOT0DRz/QSzR9Zn
    EUcX0m5ohJAkpdTo7+97q6am9u6zZ883nzlzfsnGjZv4dDoVcDodBsdxPEDQ1TXEORxLqnJy
    BEx/BhxCoQS3aFGJc2JiiJgETjPYLy4uyl+zZsuasTFaDHDQtJDv/PnzMrMdCaaDY7akq5eS
    wdmz5+3Q/fd/8p2ampp1bW2n+GwNlvHxSVJbW+2srKyyC4Kgu90FICSuV1ZWOhIJ2aKqxtS5
    TaVUCIJNLCwshKIoaafTBYuF11OpZGLp0qWFiQSsTMAvK4MiQpJSem9vd9S0G7NJrsI8kF+c
    UiqxVAlhN8gogEWLFtVwpSXFsIgCUF6KyvIy9PT2JViHwSBLDfkZekxhWiRLzZrhMN/0nL2y
    vLypu7sHgUAA27ZshKYqGBsfR4HXC1HkkUwk8OxvX8Tf/uBHyvmunhAjOaqEEDUzGwH527ds
    9H77K4+RhtpqouvqB1pms1tny4ryUVSQJ4YiUQ0Z+d9oFvCYb5eKriqKMaOLZoo0COTl5oJS
    A7JiYGBwCAeOngQvWNBcW4bSkiLouob8vFwIooiJyUmEIzEkEikNM9z6FZvQSwgBcdjtuOO2
    26YG+RFQ+MZ9EAUOWzauh6rImexNNp9jxnNhQYc5b4RSeujtt/d0PPzgA+u+9fWvWiRRXPvj
    f/lXdWBwEGtXL4eu66isKMPf/PkfcwbTuzDVXG++bie5auPqjH6LroEjQElhPuF5vnDzmmUV
    X/vcJ6x2qwV7D7UlNV2PMKNhipddcto/q7wY6Ow899xTTz21rqmxwcFzhGkS6Gha0ojvfPsb
    zm/80XfEgYFB0lBRAEHgi6/esdXR2ryEkaQpDKqjpqoct1x/teMH//xv+Q6bJKaTcUz6g3DY
    7UseuPuWosa66imRMIAilUrj6ZfeNvzBcJgFBSG2/vh8M3dPP/20hsxU5XFkBsyVAigC4C4v
    LyuPx+MrhobGoOtAPC6juDiP+ny+WDweH2a2ZYzZClMi2lRa1BlX5/KIRwb1yrKcK4oSgsGQ
    yfaH1WrF5GQQgsCjoqIcVVVlWm9vV6irq2vSZrPFHA57mhCOlpeXF5eWVlQODwfE0dFJApgC
    SdnR33SWAyBIpzXi9eabox1ijCwawjz1TsLhMABo2Q4gG3RQCpb10mG1WlFUlAeOUyAImewG
    pRmeSmb/eSY1ztFkMqFfSbuRSb5mPPDIyBh8Pv8Unyc/Px/5+V6MjfmR6brJiFJNY54PjVXJ
    QtgNAPLRo8d6m5qa91RUVCw5e7aLA7iGtWtXpAQhTMxJtrpu4PTpbmK3OxjgyDjuaFTGsWMD
    JJnUptR0ZdlAdXWJc/v2a5vDYZsrHNZQWWkzhoZ6JtNp2RQANHWrErjEjkJCCJLJlDowMPBa
    Q0P9x/v7BxtiseQUl0hVNbS3d4lr17aW5efnEVm2ICfH6rJYOMfx433c9HnhoWkUk5NJYeXK
    NYU8zxuU2mGx6Hp9fX2Z3V5S2dub4GYr2ObnS3Rs7HRiYmLSnCocYj5RnnOGI+um1lkac5IZ
    gMmMOiNKmpY0Wq2SBZqmwGa1omnJYm7P3vfNaZVDjCQ6nJVWNtG92VtPLwNwGNFYLLVp43ps
    2bgeDrsNe9/bh1deewPf+toTGBsL4O9/+E/0l089E4/F4z5myCZ0XQ8D8BQV5q++/67bCj55
    98f4onxvhmz6oRNlKdwuB0qLCsTO7n4g058eZYbDJH3NKSKUJAmKokQCwWBa13XbFKmSgQ6D
    UpSWFCEei8FqkfDKG2+DCBLUdAJXbV43Y+Cc2+nExGQAPYOjNJFMmTr3KqZb7uiVARyEJJMJ
    PP3OHgwNDSGVSkFVVAwND4MaOs6eOw9d07Bl41psWrdqahnZQJ0JoZPLMR7ZxDQGlFUAo+fP
    d739zjvvrv7sww/wX/3SF4S6RdVCUWH+dPlN1+H15MJgaqimOmperhsetwO6pjHlVAOrWhvx
    xKfvzrnp6s2oKS/B6bPn0XG+18xiGMyRfMBZf1Rd2ZyorOv6O79++ul9u3Zuv2b7tq3T0vbU
    wE3XX0vsNqv1l//xS+zZf4yuXbXc89hnHiB2mxXaFHcpk4X57Kfu4v2BYGlvbw/eP9wGT26u
    55EH7vbecdPVhGMDnijT7dh35CTe2Hsohem5JROEEFP/ZD7zM7INuIJpZdUoAHHx4sWOVEqx
    ZNQ7JTbEixCv1yuzKZuDAEylSBnT7ZQRZJRuL7u7LZGIB51Oh//kyfacSCQGt9uNlSuXYXh4
    DIlEGo2NddRmE1MHD+6bHB0dHQEwkkgkJggpEDds2LhOFO2LurtHpGg0cUmS5wCBomhwudwW
    nucNNscne06PPtdgZWhomPI8H8xoK5APgA5KgXRagcdjhWFY4XRaoShRcJwNkUhyat2KYoDj
    ALvdCp5P6MFgMD2LvG0sMOAgmaNHUFZWgrKy0qnpsVarFW63E2vWNDMdExm9vePIEBWzO1Wy
    wcflZctn2Q1QSnVFUeLBYPCtqqqKu3p6Bgt6egaIouh2j8eTFeHz0HWKRCI9Nbsmk80iiMXS
    TDGVY9kDHem0255IGAgGZUiSBElS5J6e7hRbe5LZjQjm0NLLfqa//fbbXffee+8zzc1NXz1y
    5ISUneUIhSI4cqRDWrmSR0FBDWw2h6Orq5tEInFwnDDj3I6PR4jDUeStqys1YjErXC5NUpTy
    ur6+KJ85J9O/73BIsFji+unTbQE2QX2CYYTIvABHFsI1b3pz1DkPwFJUWFSzdvVqESTjIHme
    w4a1q8jP/v1X1kQyaWYzRpjxiGYZZT2rNmXM08uBUiqfP9/19L/+9N9aAwF/eSwawzPPPY9P
    3nsneXfvPnzvBz9SDhw+GqSUjrOauimRXbdu9YpNX/zcA0XbNq4lU2JfWSTRD7TTmo5d5FFU
    kCewtLqZKzOyHPt8gNN4X19fKB6LeVxO+1S7ImgmYq0sL4NV5HGirQ0TE6UoLa9EQ0MNylh2
    g1IDPEfgcjowNj6Bg8fbDd0wEgzkmWqNKq7QwxwFz/MCLJIFHR1ncNMN12LLpnXgSUaL4/kX
    XwHPatvT122qCRJMjV65zChl1ghnA0AynU6/u+eddz915+23ljoddtxx+63QNRW6rmfNeTFF
    tuhUdskwdFBdn54RZOioLCsmjz9459TPT3f20MlAOMX+vClbPuc0OQNIGoCx/v6BX/zox/+0
    bGnTkqI8T+5U6yoBsGv7NjQtrsPe9/aTxfWL0LykgV2HMQVUDUNHfp4H3/3G49yrb+1FMBTC
    9/7XTfyiqrLM9Rra1OygCX8QP3vqRT0cjQcxPWBxFNMdNfN1NhSZWQ9Rdv4ClFKR53m+trbu
    hqNHT1spzbTwGQZFNJrmKioqhIGBgRgyc136GQcmgel2WCXreVlO8Nlnnx3YtevqH2zcuO7R
    SCRampPjhtVqdYyNTZA1a5Ypfr8vdODAcV8ikRhFRmp/aNmyZZ5ly1bsHB8PNZw718OZdftL
    aaHNzDGhkCSrJEmSJZVKmfM2uPnaQUVRdAB9eXm5Wl8fJ3xwhgoHvz+C2tpmiKIFPt8IrFYn
    BgYiMPc+M2wuc8/k5loRjw/IoVA4yc5vaj5B1Ec9WCfWlFKmoigoLCzA5GRoKrNBCI/CQi8k
    ycKyBzN1ORa6pJJtN9gUVPXUqVPtW7de1Zafn3f1xEQIo6MT8PlCLBtDZrSGzo78Z4NQSoH+
    /vjUYDq7XUQy6UtPTvrNeSuRLOrBJXfnmbw1RVESR48efX716rXbRkd9G6e1TTKgIxKJY9++
    Y6ioGAUhPBkd9c+aVTNd+uvrC5BkUucLCnIwOJjiQiF5SuI8+z2LikTa1XUo6vcHzGB+lE0l
    nxGozAdw6OypZG3Cyg0b1m1avqyFTKehdaxdtZwsb23O23fwiBvTcqym9sZCOz2qqOpbL73y
    Gn3jzbe364aRb7VYGgWBL+k40xn0B4LjDPSMM6MxIUnihltv2H3jlx99MKe2ujLTCmaCjdna
    HBf4HiFAQV6uOSDNZHJb5kvK1HUdAHwdZ86c7e7uXrRyxbJpFU72N+12Gx57+JP4wY//FURy
    IJ2IYseWG4AZXA8Ki0XEsVNncPTUOZNfEgUQZV0b8x4ud0nMYocDt916MwYHBjAyOoKN69fC
    YbPCYF1M7+x9HxaLNF0umtLgoKbluVBq9LI4J1ltsudOtLWd6Dx/vnT1imWMmHuBz5lmf501
    dG6qhXq6rdcwDOiqitOdPYam62YmaV7ZjazfM8tAB19/463nfv30Mw9/9tMP8DRLL4MaOooL
    C3HX7bdA17VpoDzjGjJlGIfdhttvvHpqMnJGDG9aZVfXdTz78tv08MkzMXaPjLCspClPLF8m
    4DAj5KT5WT7yyGerUill/fj4JDEJlYRwmJwMk6amsly32y1Go1Hz/4QZ8NFnpfcv6xwTQjAy
    Mqr+/Oc/f6GpqWnUarW29PYmy5Yubd1YVJQnnD59wtfT0xuklIbYngzs2LGjsqKi5s6ursHK
    iYnglHT43BRJAUEQRUkSbalUysLk9E1SpvFhdfqL7XE0GukoLi72Wyzni3WdznJ6HEKhGNrb
    B9DYWAFZTkMQ7Cy7IWStK5MxyMkx6OHD52KqqkaY7Yixz2GBAQcBk/zDyMg4fL4AnE4X/P4Q
    fL4gBEECx/Gw2WzweERQiimJ7Q+WVMiCz9hhe691d3f7t23btresrGS73x8RPtjFdLGv+QuQ
    eHlkly8cDp5OTo4oqqqagby551P33By6bSgAub29faCurvbnLS1NjcFgLE9V9RkAQdcpBgbG
    GSlXuABQyqwN4DAxkUAgkBGxM2X9s8nQeXkWpFIjckdH+wQDG2Yp1D/7zAgL8GE4i4uLHrn7
    zo9XeXJzoakyGyqloyA/D/fddZv9ZPuZxng80UMpLWQGLJ6V3VhIZnGCUnpYUdUxABWJZHLx
    u+/tL8b04CHTmAadDvs1n7rn47c9/pn7bXmeXGiadtFMxsWyHBwBXE47wXTroMSAFT/HQ5K9
    n7H+/oHXfvPb53c0NzXaeIKsybsZB+dyOiBZbSCChJqqIpSVFE0BJRN0pNMK3nj/uBEMx8IM
    aAUwPcvmSo32JoQA6VQK//nrZ9DV3Y1JfwB//w8/njoThqFjdHQcwWAIb+7Zi1XLmrF1w5qs
    N0Bm4NsCktNmda0Eh4ZH9h0+fGT3qmUtQva+mhkWjuOYBLsBAgKDyUMTjsCgBAajWWQDkGg8
    jq6+oWwlUdNIT7W/XqJ+iBlZGZRShRASTCQST//7f/xy47Ytm1oW19dCm8XtMTMePCGgbO2Z
    yT9k+mmQqa5jcw7LdLaE4Mz5Xvzqt28omqZPMmNhcq4mskqeC8GnwR133EEjkTDvdLo+fvr0
    saXxeAI8L00Z5Gg0iWQS7mXLli967729XZTSPDYyIIrLn3J8of3WKKWxjo6OM+weKevvH+hT
    VdWpqirFtI6K74Ybbmjweos+1d7eXRoOx8Hz3JyAhvl1xrHznChKpt0wbQeZr3Lq4cOHu7Zu
    vWp/RUXZbQMDIx9wIAY7A+FwADabC0NDkSnuRvbv2WwcotFhpaurKzDLdiw44Mi+1LKyYlRU
    VMDlcsJut6O6umIqO2CxWCCKIpYvr0U6rWFwMMjmqEzHI1nlO7IQ9iNb2JBSmg4GA4eKiwt9
    nZ39ZabOSjaomwmAsjm2M7+X3bWUuTYDExO+NDtnUZYJTMwzKDcDq/iePe+8d9ttt7+yaFHV
    PefP93Ozy2zZc34+7Gm2vH4QlPCQJAG5ubqxb9+RUCqVHmPVg0FCyBimtajonAHHRW4CyWKx
    PHjfvffcfc2unZw5MCo78rth9w5y8nR7zb/+/Kl1qqb5TfIXiziTuPzpg1PcL1bTDbL3SmJ6
    gqtJxIkBSNhttlse/uTdt3/hs5+yOe22TLT3IZmMmRoFWd8jBFZJACEQKZ3RdspdjvHTNO2t
    J5/69bub1q+99ppd22HoGXlwyhEQSvD23n0Ab4GSiuPqq24Ez3GZIIE5Q0PX4Q8EaSAcM7k2
    5gybyfnW4udqQJxOBxrqatG6tAkCz+F420nYLBJam5fAMDKTYlVFQVGBF8B0ayxztpfECpvn
    GZFVVT165OjxifvvvbPUYbPCoARpRUE4EkYgEEAoFEY4HEYymYTKhp3xHIHVIsHttCPHaUOO
    ywGn3QqRJwDHwR8IYXwioGZxkswhhfJ8HSTrdEoCOHfy1Omnn3n2ufpvfOUJK0cAcBxkVUE0
    GkXAH0AgGEQgEEQ8HkMqnYauZQTvOI5AEgXYbRbkuJzIdTvgcTvhtNsgChwoR5BOpfCr5143
    hkZ9EQbIh1h2YxSzhi/NE0TPkA//9a9/jW9961vb+vuHHjt3rtuSLb9sOuP+/gm+tXVRY2tr
    eOTUqVM+Sqmf2QvjQmWUyxyJoDHpdoLM1M1IMpkcQ0bwipjByg03XN9YUFDy0KlT50uj0cRF
    lUgvFXwAhOMzrRnZA9LmfdY7O89FlixZ8h8tLUvWBoPR8mRSmbp9KCWwWESUlnogyxFQakco
    lMC05EfmKYoCVDVpHDmyP5xIJEzbMcHsqjmG/UrEKtB1A/F4ApFIHJQCdrsdBQX5GB6eYKTR
    TDROKblgaYVkt6ws/EPr7Ozs3rRp69mcHHdZMGiCNR4WiwU2mw02mw1WqxWSJLKpq5kMQWa9
    BJoGaBqBrk83/QgCD45T9FAomJxlN+YUqMy6FwwA6UgkMtHf3/d0XV315tHRyapUSpkCEJIk
    wW63Tq3bYsnwZgSBn8rAGEZG/j6zbgpVBXSdgFITOHHIz7fQsbH25ODgoM/MALJxFP4LAdR5
    zVJhF+WUJOnB+z5xz3e+/MUvuJ0OBzRVBkznxxHAyEyy/PJjnxY0VV31y6efRzKV1gGorNYf
    yE7xX+ZchOz6l8zS2eOYlvLWKKWqIPA33PmxG+557NP32Rx260wl0mxgcSGAMft7nBmJE461
    dmW3ns73oQPwDQ4N/eR73/+HJWWlxVXFRQXwjfsQDocwOenHvsPH4XDlIhoO4cy5bnR190KS
    BFglEQ6bBRZJQN/QCGVpaNNojGdJzC4I8Wt2dwBlPXR2ux3X7b6aZTQ0cABGRkchiQI2b1wL
    TVGmxOHMEfVmLJ65AbkLAo1Z52++a9YopT3tZ86eP3v2XClAcfLkafT29yMYDCKdlqfOhJlx
    MTMBMM0ERyAJAnLcDlSWFqKpvhrDo+MIR+NpZiziWUZjXlyeWfLPEU3T33jp1ddu+djNN6wU
    RR7Hjrfh3PkuTEz6kUwmoMgKNE2DqqpmaY5lYSh4noMo8OB5HqLAwyqJcLscqCgtQmtjLQLB
    MN58/7CM6Qm9ZnZjcnYd+XLOCbsu8o1vfGOzzzf5l4cOHa1JJJIQBEvWFEqOifjJ6OkJOFpa
    Vm91OBzy0aNHE6qqyuz9wpg19nohgCh7vwS7T2wsa0kopdquXTsrysurHm5rO181mxw61+xG
    dkqa2Y3Z7erzyvBSStVXXnn12J133vnLZcuanmhr67SKogSHwwGLxYqCAg8oTSM3Nw+plICS
    krwsh8hB1wGn0wqfb1Tp6ekJmHaDvYYvBzxfgHM3VUY2k4w+nx9+f4RNLRVQUOBFQYEXPl8Q
    lE53qZjj6TMOfXoiAvlgmoFcwEbNCzADMDo7zwW3bt12tLi4YCelhBQV5SMvLxd2uw2iKE4B
    0OzMwHTphDAVTgJdB9JpimSSwjAEaFpMi8cTZqBitpMqZkZwrnbDBNGU0uTBgwdP3nPPvW+W
    l5c+NDLiI0VFBcjPz4PT6YDFIkIQhKmOqmkBLzKlikuZMBKbGcnWbiAe16GqBDZbSt+372TA
    MIyxLLvhw6xuK/M6hPkYQwA1BQUFX/rk/fc9cN8n7nH39fXj/PlzoMzBZFRHdVAWybocVnz1
    8w9JiyrL1v3sV88Wnu8dKKGUvslGB5tRd3oBHKGeBTzMWRxCVv2/ZePaVZ9+7KFP5OS4Mh0H
    0wBiui6eTQykU/X66cmyU8RCEKiqStmpWBA9kSyVu2Pvvrfvh49+4StfqK+rKcvNcROb1YoT
    J0+jtmEJTp8+heHBAXR192DD2hWQRBGKLENWFIyNT+DAkZMKyyb5AIwxAo9pqOlCRCnme0iS
    BKbdL42OjQmvvvYaBJ6HIsvQ2Kj0wcEhgFL87BdPTXWEmK2YhGmMCDyH8YlJpGWZw7TQUHaZ
    ypgv+MiKtCmAQF9//+Hv/umfr3c6HRZd0wgv8LBIIqwWCxx2F5x2G5KpFN58Zz8WVZejpbEe
    6XQaiWQSyVQKyXQa0VgM/YMj2HvwBEZ8kzQlK2YpJTKfcsrFnCEhJE0p7e88d/7tb33nu0sK
    8r28KAiSomRAhigIKMjPg8vpgMNug81qgSgKoIYBRZaRTKUQjycQiycQTySQlpVMa7UsY9/h
    Nnq2q88IBCN+ltnoYynR8dkp0cs5M5RS3HvvvfZvfOMbt4yOjn3n4MGjS0KhCGw2+5TaKM9L
    7Jn5OhZLo7s7nFtb23p9aWm5t63t2PM9Pb3HKKWjTNp8IflI2dw083MklFKyYsVyy9Kly77S
    0dGzMhg0HeJMpdHZNf2PIo1mnCWlmqYtSLrAzI6m0+no22+//fT27TvKdu3a9LF0WrGbgk75
    +S4MDPSgtrYOAI/CQhmhUCZTkxFc4+FySfTEic6Epul+BjbGmR7EZWdGZw3xA6UUHk8uL0mi
    pby8FDabHZIkTXEKLBYrOI5DQ0PVjH2bzhDxWTwIKxFFwcwUZWebLwp+5wiYDUppOhaLHW1u
    rvNVV5cXCYJAKAUMA9B1A7KsQdMy07CLi3MhCDx8vjgI4SAIAkSRB88TCEKmbJWXxwHQ6fh4
    RJZlOWry7C6jnPKBoZqpVCro90++3tracGNVVWkOwFk4jiOZduRMt5SiyEynhU5piPA8z9Y8
    /cyUDwG7nYfNxlNB0OjQUF9sctI/zHiR/UwZ1Wxh/4A/F+ZxYBa1tLT8w5e++MTVd95xO3/+
    3Hm88tpr0NTsssT0KG1d11BWXIjbbtyF+++8mVuxtKH2n3/x9EMvvr7Xo6jqbxhRahzTA8Au
    +95jTwMzh31ZCwu8937q7tsaqspLMVv2XNc0RlqkF5woe6HMh65TROMJM0KaN4kt23mySEUh
    hIQNw3j5yLHj7uXLln71f/zR16yRcBh/9f1/gsvlRDwaRllJEXiOw90fuwENi6ogy2kosoy/
    +uFPaCQWj2F66NYEpoWn9ItkJ+ZlPDIqgIoIYHFRUeETrS0tlUPDIxAFATxHwLG2taLCAoBS
    qCrjyhg6DNZyajDCo6ZqUBQFq1qbHO8eOFqlafogpdRLCMljN2KSUqqyVsj57rMBECWZTD3T
    1dO7YuO6NVcva2lCQ201ykuL4clxw26TMDA4jOdeeh0lRfmwWixYtWwJVixdDEPXEYvFMOkP
    YHB4DD0Dwzjd2Y2e/pGUYRjmwKgwW28al9HqnWU4NEJILJVKP//GW++ktm3eeEd9XU3j8pYm
    NDU2oLS4ADarBQ6bFTxHEAgG8cY7+1BVVozVy5pAqYF0KpUBTIkkhkbGcLarFx3nutE7OKJ0
    9w8Hkena6gPQy2TYA5jndNgLXcvjjz+eI4ri19vbOx49cuR4ns1mx6ZN65jz5rPIdPzU3JJk
    UkNfXxBnzwYcFRV5V23durumoODEfxw8ePBdNgBy8nI5SbPmesy2H/if//O/I5mUV4+PB28f
    GZngZgMLnudhsUhQVSOrNv7RWQ5RFGAYuiHLio7LEKGZBbx1QkhyeHh45KWXXvinj33s9vyu
    Lt/u/v5xVFaWQNNicDpzYRgE6bSMVErBmTNjIESAIIhwOm2oqhL1zs6zIWYvJligMqOsdjm2
    Iwv4k89+9jOFn/70w3f4fJNXE0KQm+uewXeg1EA4HJtSaDWzQub5MAfUZWwQx69atb6c47iy
    oaGhfl038gDkssy5wvbmsgPa/fv3H928ecsvRNH+xORkWAqHE4jHZSiKDk3LBF/l5fmsvRtI
    JjVMTCRhGBxEUYLVKsFul+B0SnC7eUQio/r+/e8FNE0z9W4iC8WVYTww+eTJk20rV3L/R9e5
    q+1297XhcBzBYAzxeBqqajBuTwYUlZfnQ5J4DAyEYBgZ5V9RzJwPu90Ct9uK3FwrksmwcfDg
    m+FoNDJMKe1HpvvU7PyMX4wqMSfAYRgGnE7nLZ/73OeueeDBBzlDU9Dc1IjF9V9lQ9vUzHPq
    aw2GpmZa7wwdhq6ipbEOX33kPmd33+COU2e6eiilGiFEYcZtXi1us9ofL/bzJRvXrLxpy4bV
    RDey5MrZuO/fvvIWljcvRk1l6axMxsVBh67r8E0GzYzKZbX3Zn8oLO1vCqZ1+v0B9XT7Geu+
    g4epZHOQ9tOnUVleiq7uXthtVuw7dAxdPX2wSALsFgvGJwKUGeIAMxzmtFXZZrPRVCr1AUM7
    13IWq57k6rq+JDc397otmzffdt8n7mm8/tprOLvNxjogtKl2UjbMj/1bmyq3UPPf7PcMQ8Od
    N10j/Pr5V+uee/UdV1fvQLGsqIUAjjEEHWERgDxfAiEh0ChFv29i8tSWjet2fereO0gG9GTO
    LmDgyPGTeH3Pe7j71utx8NhJ/NuTv0Xzd74Et8sOm8SjIM+NproqaJqC7/3LU/SN947G2T5f
    UPBmAUhrCiHkPKU03X6mc+2XPv+Zxl1XbUY4HMFTzz6P7t4B7N6xCVvWrcL7B47i+Tf2Ic9l
    Q31NBXRNxb899TwUVcMNOzdgzfImrG5djH/7zxfoU8+/mWKAf5BFKYPs3MUwt+m8F33k5ubC
    4XDccOzYsS+///4+u6YB6bSKY8dOQxDEGVmNzKvISKSZer2mUQwOxrh02l3T0NDy8c7Os4Fw
    OGJmJdTLzXLMuvdmfL+trU2or19yy8jIRImmGTN4GwCB2+1EaWkBenpGpwiYH11iIZAkEbIc
    1WRZNrt35jVsjFKKZ599Fqqq4s477wQhRAUQCQZDg4ZhDOTkOFFaWkArK/MRjwdIUVEZEok0
    3G4HVNVARUUeDINA1wmsVgs0LammUukwsxt+1p0TR0acEaIoZpXs5mY7GGHYWllZWf21r31t
    SzAYuuPIkWObOjrO2ePxJGsxNcsmH/UqMG2LzKsoSqS42JO/adP1V8lyqDAUmiwKhQL5vb39
    neFw2M9K92aJ05jn+TD6+/tDK1YsPyyKJHHyZLdkOmVzXTabBXV1RQgEktB1itraPCSTBsJh
    FZpGkUzqSKc1hMNAQQEHWQ7LExM+U2AvmAU4jAWwG5QQog4MDEwODAy8dMcdd3hjscSuU6e6
    RUJ41NSUYtmyxejvH8f586Ow2y245prVcDodePLJvYhEZNTUeFFTU4SeHj/GxpKIx1NIpwG7
    PaGOj48GNE03sxvmJFuz9HbB9c+1pEIkSSqqrKjgOI4HZfVsQRBggMIgGQXO6a8NGOBh6AYM
    qgOsHu5y2OGwWV0AagCMMy6HHwuku3Chh81q3bx5/aoyt9M+JYwEwwAhFIPDozhyoh2b1yxj
    zvEjOBxMJj2RTGJgeNzsTDA1AdTLQKciAD5LTlgHEHrtjbfP7X1/f2VZeXnepk2bhWPHjkMU
    eFBQBELAX/7gX6EomSCP4ziaTssm2dDsoVdphunjTCaTBkvTG1lp5PlkDPLKysr/evfua3a3
    trYUiQJPdF3Hb59/EZqmZjIXmjbVrqlr05mMTFZDnzE4z2SIcgQQeIL8PA9/+w07SjnQkq6+
    oW3vHjxxfmjU9z1K6Rl2VsyUOp3HjagDSGiadmZ4ZDSt65rNnFtDWPHtxt3boSgK3nhnH/Jy
    3fjCQ3ch1+3MgFEA1DBYm6+GsYkgZUY5jGllzhjmoRj5EQTSBIDxeCJ+Ymh45GrDMLhQKIRD
    R9ugajpURQEhQMuSenSc60ZdVRlcDhtCYRmBcBTD434011eiobocmqqib2iUUkpjLCoZZkRR
    c8Dc5bTBzngUFhYSVVWX9ff329NpBZJkZcPZMt0pgqAxsGFAEAzwPIUgADwPCIJlylnHYhqK
    iiwldrt9STgcCTGhvfBC8EwuBj5qahblJhLpDaFQjFyIJFpUlAdZVj8ANj4su5HpBJEQDIZk
    xktJZ9mQOe/5bbfdRvLzveJdd93FYbodQlZVdbS+vrynqammYHh4yOX15iM314WMJDWB1+uA
    12uHrmuQZYUqioLBwUlZzUxzS2bxWiSn0+JmNiPbdsxJCOzjH/84qampeWBkZOTLAwMDVaqq
    WxRFRXl52VQZxXw1Z6iYX5ucGVOt08yImeRFRoEh6bTg5rji9UVFJSvq6/m7Sko6f/rSSy++
    wkjHAUxrQM0nk0QppXIgEOitrs73SZLkURR9ah0Ah1hMxunTo6irK4Qg8Dh3LohIRMla63RJ
    SBRBx8cDKUphqsyGWUCVxsIJrE0NoovFom0uV2FYEMQCQjgUF3tRUVGMsbEgAA6ybOD48W5Y
    LBYkkxoADpIkorg4F+FwEhMTMswBbaHQZErT9ADjfA1hWkX8Q7VD5ixtHo1GD//qySf9brcz
    P8ftmpoISw1tZiRrPjWVORwVmqoglUzi3QNHaMe53giry9tYnX5BhwLNXrrNZikvKykUTEGk
    7MmY+w4dR2NdFfI9OWxQ2keXVDgCDIyMo2dwVGabbIrjzJnMRikVRFG4IT+/4CqOAKlUOpqW
    0xFZVtKSJFYtbqhzUEpdTUtb+GQsgu984wl0nu9G5/kuuqi6kqxd0QKrJGIyEED/4DBOnO7k
    Os5156dScj0hxC6KwlKLJOpWi2hJpWQlFI29Qyk9xpj5F5pAeCmPqquv3nXt97///aJ4PIaX
    XnwBwWAoc58w1cpM1wnAEQ6cSEBEAYSYTCTWwmtkzo+uazA0HZqmQtVUqIqKHJcDuzavIZLA
    uWLxxOKhUV8rMtL6lM32SM4z8jZYmnVwbNyXjESitng8hmAwhGAwiEg0CllOQ1UU9PYPwd5Y
    i+6+QfQPjcBhsyDXZYfLYYPbkcnk+INhHVnzMLLa2i6XvzEjWqGUpgGEZFk53dPbn0yn006n
    045HHrgbkUgEVouEt/fuh6IoaF1SB54j2HvgKGwWCTfs2ADD0FFdXgJVVaAoCvoGR3UGjPys
    Vj+eVXpbkOwGAAQCAUop7S8rK1fHxyfFDAFQmnpmOBwS4xGIUxGj+bVZS/Z6bUinQ2o4HHEh
    M/rajis3iDCjV6LrgmFgqpMmGzRYrVY4HDb09o5/KNiYDToyuhIcHR4eyiYJJucDUB9//PF8
    i8VyfyqVXmQYRlpVlXA6LcdUVaEcx9dHo0mH3W6xJRIJGIYVp071o6Agh4qigEhEJtFomkXm
    IqxWHm53gXXTpi1luq5pbndOQV6ed5PLlUN4XpQMQ544eHD/iydOnBhiGYP4XJxjfX29LRgM
    XvvWW281+HwTaGhYDI8nb4ooPHOcOmCwLLNhaCDE1BWh4DhTOjwDSjmOgOcJeJ5j501GMglL
    ebm9jOeFFcgMx+Mw3bI+Xw0oCkALh0MTkiRO2GzWRp7XWaeHdap1VxQFSFKm1CZJPIqKHKCU
    g65nOlR0nQelPAQBSCTiKZP3xcBGfCEClVklWRVAPBqNnC8urhq1Wi0Fqmrg/PkRjIxkCLlF
    RbkQBAF+fwwcl0Benh2UEgQCSbz11lmYU2Ep5WGxgLKuGpMjOM5Kbx/J+ZprhoPTNG3/k08+
    +advvfXW9RaLpYgALkJgzWZYE9ApchCTTaaGYUDXdappmh6JxeOapvdmpW4VLLBk7oxFcxyN
    J5I9J06dUdYsa5Z4nkwpMWqaBqtFxIqlS6cj7wsNa5sFRBKyjBffOkAnAuHIrLr9fOpvkiRZ
    bl/a3HR37aIakRoGYvGYkUwkNU9ujrBuzUru8LE2xFMK+gd66UB/n3K07VQsHk9Ejhw/xXX3
    9Ht3bl3vbKyr5hprq8m6FUuFtvbO4sGRsWJD1yGKPASOQ1qW0dk9mA7HEhZKjYGM3yZqVnli
    LvwNEo/HceLEcQi8gObm5mkJ+CmlTp2VrnRWMjG5G/r0q571b/a1MZUF0XC2qx+JZAIDI+Mq
    ADeAAnbQxflmxCil4DhOo5ROHjxyLPCt7/7vvFAoTBKJJFRNzYi6mdF5vge+yQB+8qvnpiTA
    eZ6DzWJBfp4b3lwX+jNZLpMsaupvLLQqI2XclRSltHf/oSPBv/y7HzqCwTCJRqNIJJNQzS4V
    bbpcSAiBIPDgOQKLlGnr9XrcyHE50Ts4orE1Z2stzCBiLgS5OBAIkLa2ttfr6mr/vbCwcKcs
    K05KIVFKBEIImSmMNKPXnzIVR8pxhKbTk8nDhw92JZPJ8BXY3ws+Dhw4GN2wYfPhioqiVRMT
    YS5jxDNlEYtFQigUB8dxcDhsHxiYNlu90fza6bQgkZhIDAwMmEP9glkqrHOyg06nsyydlh+Z
    mPA3KIpKMuBM0gTBZgSDcSmRSCMvz4bCwiLoumDoejp17lxbRJbTiZqaOrvXm+eNxXRLLKaR
    RILC6cy11tQU14kiV2dmDpjeBbXZpMm8vLxhADEG+k19o0t6xGIxQgiB1+uFKIpIJlNIp8ez
    MhjClLhUBlAIszIcZtYj+/sCOM4Ax1HwfObVYhFgs/FwOjnj3DmfAKCQne1JXJ5kAQDogUAw
    ZrdbRleurNc5juctFrNDZWaTjKYZqKzMyRoEON1mmk4DkqTTdDptCn2ZRPP5BH+XQoiWA4HA
    uNNpHWxpqW0lhBCrVYIoimZn4Ix1TrdTm6RYQFUpZJnC4TBoJBKJsT0NZJfs8RHt83My2KyV
    1QWgGEAtK4mUIzMtVsSHd2lkk7EUtlhTYtys/0Sp2X+4kOmNzLobSooK/nTtypbtTrvNCkAg
    hHCEgFDDgMDzyMiymy0t08qdhjmwTTdgGAbVDQOhSMw4eOJsMJ5MdQI4A+AUIaQdQDf7EPQ5
    GGuOENJgsVi2u5zOJkkSSwjhcggHC0c4QZbl4qqq6ppJvz/c29s7xjIpZp01BkAihBS6nI58
    URDsHEd4jiMGKBRKacqgNKHrejwtKxFZVsYMSjtYrX6SlSfCAORLXS8DHIV2u/1LOTk5LZTS
    XMMwciilToDypmDo1CRYdsvOPojTX1P2q/QC/6QwDEoTyZRPN4yDAM4COEcI6WHluHnP96CU
    5gs8f5cnN+dLhmGUphVFdNhsfEF+Hsl1u4jDboUkCiBMkj2dlhGNJeDzBxGJxqkkCQAgB4KR
    IQq0gT0JIWdZqlFZKKedLWpEKS3Jz/N8tnFx3edjsXh+IpmC2+VCQX4ecnNcsEgSeI6AGgZU
    VUEimUQoHEEwFEEyJcPltCGeSCk9AyOjlNITAE4AOEEIOcPWLS9UdoOtnaOUunNy3NVFRcXL
    KDXqVFUrNwzdw/RrpqQU2CmZaqMyDEoNQ6eqqurxeDwty8oksxk9AHpYz3/oCtkNUEqlrVu3
    Lq+rq39C02irpmkuw9DtlJrdb5luTJIlkZs9P+aD1RpKU6l4+vDhg4Ojo6Nd7Dx3EELOsbR0
    Yi734bp16+yLFy9eRwi3UdPURbpuFFBKXWxfJa83ry4ej+UmEsmRc+c6Q/39/Uk2oyYoSZJc
    W7vIVVFRVWS3Oz2EEEvGr3Iax3FyRveEJnRdiyWTidDExHh/Z2fnmVQqNcZsR4DZn0uydV6v
    V7jtttvu0jTtY8lk0qMoSq6qai7DMCzIkjef5Z4oIdPtmbOb08x/m40v2fut62r66NEjR/1+
    /ykAnYSQbuZzEvM51yZXRRAE23XXXbe1oqLqm1arbbUg8BYAvCxrnKIY0HVKKCWszJbJvAgC
    D0niIEkcKKU0mUzrHR1tk0ePHmmPRCInmN1oZ3yI2ELbDUoprFaL4+abb73Zbs/5pt1uX2ax
    iNA0Hem0AkXRoesGKDXJ0ALL1GSyNYKQUcdNJJJGb++ZwIED+9pjsZhpN04jM3og/lGBCpnH
    BVgB5FBK81m0mc8izwtFnLPlqGlWisckNWY7vfSVEJVhBi8HQAOA1QwolbG0rIALaOReACzN
    fjWjw1G22V2EkF727zjm0HpKMqGEg1Kay9B4IYA8Vm4SAVzjcDg2JpPJg5TSIPv7YUb2M9UX
    RWQ0R9zI6AeYConJWaUela0vkGU0InMFHABslNISANUAqgBUsHVLl5ianMvPTI6EqYA5YHZS
    zAdwZF2HnVJaBmAtgCYANYSQYkHgnTzHiRzHCWS6QMzm2VCqahrVNN2sYSeYEethqdvzhJAB
    Bga1hTrPWYaDALCLolBDKf24pumtAEoddnue02G3WqwWQRR4jiMclwFrBlFVlcqyYsSTSSOR
    TOsE0Diei+i60QvAdHqdbGbJgq47C/PbKKV5yEyJLWGvuZeYqcq2GxFkuq7MElCQldbolQAc
    AERKqVeSpCqbzVav63qVYRil7D7jMnN/poXqpp3j9NIzegZTgJuqqqqyoZEjAPrYvg+xa1Hm
    CPxFAC4W9BUSQgp4ns8lhIhutzvnjjvueDCRSAgvvPDCkXA4bM6gmcS0xgqYvXBxHOdk9lAD
    aNowaPZ4cZOrFmFnZJKJgV0y4GDvnS+KYiWltJpSWsVsiOOjz8DFPuGL+jdTw2bCDGwJIYPM
    Zqbme1w4joNhGCKA/KamphV2u30ZgFqO48o5jsvleUHieUHg2IMp+RBKKdV13TAMzVAURU8k
    4umenl5fOp3uzbIbvcjMMEovtN1gm2QBULBr19XXer15V1FKK3Rdz9d16gSIQAjHE8IRQjgG
    pM37zjAo1XRVVbRgMJQ6duzoUCqV6gLQmQXkxpiP+VCgROaxeI4dUDs7KHZkZohcisImzXLW
    qazapUlQMq4Q4CBsnV5KaTEy47GLmcGTMPfJpGamxpyqOY7MsJpsgRzMEXDYGJDzMLDhztrX
    Mva9OPsvaQYYxlgqVmeG0QHAya7VLJdlD8kzsgCH2U1hpvLUOQIOCYCHUuplwNPL1sx/COCc
    DwihWdccAmASjAPsOozLiFYkQkgOgEJKaSnb50K2h6b2B4+ZGsXZ51jPytaZMwRGWD0zfiXO
    M9t7CzsnJQAqGeAryQL+s4Wk6AXOQort4VAWgPNdiXWzNQvsjLsAuFmm1D5HwKEhM0kzhpnS
    8doVshtAhsTtYOe7gJ0PL7N9/AWyuuQjzrF5HXFMt55OImtA3hw1IoRMZhG5LIDysD0WBUGw
    NjQ01Pr9fkxkGH8G+7vmFOBkliNysqeVXRfNshnZGiURZjdCyOqouJQ1m5kutn/57Olh/mQ+
    NuPDAhRgesx7AIBJGg3PxdZd5Dp4tucF7L4rm+VPeFxYzG22ZIM5amMYwHCW/1AX+v5jJSGe
    7X8RCxCrCCHlHMflEUIshBAzwGIgaTrDbBiGwbKIJugcYZnGwazulI+8D+cDOMz/l62QNxd1
    TXoBJ2gsZM34YpEKM3BuZOTO3cxoiB+S2SAXOdjmNZjzWbJrcHOWgc5an4UZZfNpTpG0swyI
    yPYszW74bFawkPX/rez9sh2OaUA0tu5skqs6F0OXBTytbG3m0zLHGin9EAMz+3ta1pqnum8w
    TxGzWdfgYmcil2XCTMA222iQi5QHY8iQRYPsxouzSPVKRd0m2TqXZQ1Mw+3ANAGbu4AzzD4H
    aTNaZQDOjFYVLJAw3AXsBo9pGW8Rly7nPdtQz24lvZJ2g7C12tj+OrOCLO4SAcds22ECvgSm
    pfDNMfB0HvehyNZjyw4AKaUSW7ON/Z4p4W7ydUzdIzHLblhmBY80y1arWVlTs7PmktfMAj9p
    ls2wYm5cwrlkR00bnd2xd1mBbdaZMPc22244LwA4LnQPmoFfktkNsy12isdzhQIVDoDEQIdp
    N/KysuIXW3N2sGLau2AWUI5c6rrnCziuyOPKaPTPOCQ8ZipYXjqPPjwAAAFNSURBVMjgkUs8
    8DTL+JntsDN6vOdoOLKBXLbDMJ2LjVLKg/VWs5vH7CagF/i/3EWcjTEL9NHLWC93geeCHIWP
    cDqXDVIv4ASlWU/+Qxw3ZjlABTNbG7UrmK3LPieWLEeTDVAvJpNNZxm8bAAnZ5/fKwQ4LtcG
    0d+l7bhIgMVn7TGZg+2gs4y3Pus55/NMPjhVefa9ONUFSCnlWDurmuV89YvcxxdzkrPtxpxs
    x6yzy+Ey5dwv8WwYC2UzLmD7+AvYDeEC9vBioF9h910asyQVrhDguBBAtWWBvpmDdT64ZtPn
    pbPshrl241KAJ7kCBuT3Cmx8xHqv2EGfh3TuR+0PuYAi4n/JZ3Clgefv4uxcwpmYjyOkv0Mn
    eKH1znfN/xXr/r/ddvyXX88c1kbmkCW4Ymv+fbIbCwA4PmyPyeXYjSsMOGaveb72bl427/8H
    o8eFSEj8u+4AAAAASUVORK5CYII=
  }
  set pieceImageData(Fantasy,50) {
    iVBORw0KGgoAAAANSUhEUgAAAlgAAAAxCAYAAAAGAtdnAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAAJYAAAAMQCkzSEIAACAAElEQVR42uz9d3hd
    x3U9gK457faCi4tOEEQlQRDsTSRFsUikumR1yUWSLctW3GRbLnES5xfHcZw4dlySuMY97rJs
    Wb1RpCiKvYEEQPReb++nzvvjzgEOQFKWSFD2e0/n+86HQhA4ZWbPmrXXXpvg7ePt4+3j7eMC
    D0LIn/0ZSunbD+otfuZzfVyKd/j22Hn7+P/1mCW8/bgvLHi9PfHfPv7/eZExr5tS+mfvwfrv
    b8+bS/M63uT3ZwyvN/n9S3Ht5Dx//+3B8na8uiTXTQiBYRjnvSdrfLsUE/PtAWMZLOf7mb/m
    xeJS7nLfXiTn9t1Yn+e53ttf+zjjOA6UUkIpJQA4AIRSSgkhFIBhXSjfHjuXZl5bnis5x8/Q
    v8Y5/1YycW+Pu0uzoTrf+vjXCrKs88R6snugs863AdZcT3Q2YAh7sIQQwrPvmQ/dmKuH/1Yc
    PM/DMIypiWD96Pf7EYvFzvr+GwmAb/W9X2gw/mvfTVk+JwykwDLZ2S381QYrAsANoAJAAbvm
    BIAEIUQBILNTBaC/vdDN7UJn/bbT6eS2bNni3LVrl5bNZs2fM9ipXSzIeisAFqUUgiBA07Sz
    FkSO43DttdfiT3/6Ewgh4Hkeuq6fFbPeIAh9O1ZdHDgxD866dswlMLlU8WrNmjXiypUrSwVB
    KM7lstLw8EjmwIEDsWg0mrPELMWcM28DrLl5+AxTUXP37QBwNYAsgIOEEDNQqewFvKGA9Zce
    ZBzHmTQoRwjhLNdlsPsls/+N3afOcRw1DIOwnycMYOJSsRFvBMxdih3vXNzDm00rW8AJAeAB
    UAxABDBBCNHZ+FJNYPLXEKzOQ7FLDofjUytWrLi3rKzMnUmnjCNHj/ZMTEx+A0AnISTBAFeK
    3Qt9m1WYm4XO4/EI6XSaMwyDAiDvf//7mwsKCh5Jp9Mv/eQnP3k6lUrpbCzlWBzT6Eya668q
    Zp1jAefMBZwtdObizbF5Y/03w7Lxnc1IzHnMeqNZj/P9Leu/vZmYNlfv5iL+Jqmvr7OtXLmi
    SJJs/mPHjodPnTqVZeujYtlI/cU3hefLEnz2s5+9Mh6PfykUCpUDHBcIFMRjsejPfv3rXz8L
    IEkIiQNIsnljXOzzF94OWASU0gJK6SoAgwBGATQ2Nzd/2eVySfv37/97SulBAAYhJAMgzgKW
    bgGpZ03k10H9b9lhGEYxgI0AMpTSk4QQlQ0amVJaapOkmymlmqKqzxJCYpZ/o7que9mvyTAw
    ZjCEPzWJLub+Xo9Wnh2AeJ4HpRSGYYDjOGIYBm8JwMTy7Kll126YE/319EJvpT7oHH/7ipKi
    4INul2Np3+Bwp6bpXwCQoJRmCSEpBrT+Kg8G3HmP17vo05/6VP1VV21Hd2cH7nvvA8LExGQV
    gDSllGdjLsvGzdvHHB0f/OAHr+B5fsfk5ORzP/rRj07zPL+jvb39Lo/Hs2nbtm36448/foxS
    qrF4BfYOpgAIZrLxU8zDXypmEUIQDAa5e+65u8Fmsy0fHR3t+dWvft2rqioFoLndbvW9733v
    MrvdflUqler83e9+98rExITK5oi8Y8cOsby83N3Z2am++uqrOTbXZ2xULsWCfb54bxiG+X94
    dprAcAp9MGCiY1Ya3dwYz45XcxWr3ijjN5sx/dCHPlTpdLo+Isv6Zrtdcsbj8X88depUC6VU
    YfEqdbHP+VIDeJ7nS7u6upbt2fOqJEkObN68qUQQaDWAeQAmKaUaISR7vrXizT73/78HWOyB
    NdXV1f1IEITh9vb2bwJwNDQ0FK1YscI7MjLy0YGBga8BGKeUphjjILMJYVKKBltITGrR+EsH
    LACc0+H40M7tmz85ODQcO3z81KcppYfZrjYpSdJ77njHdX8XCoXVp1/cw1NKn2HX7Q0EAu+q
    ra3dmM1msx0dHc8qivIsYyNSDN1nLmbBNJ+Jz+dDKpWaseMzDIMwxoxjLA81DMOglDoBLDcM
    o5K9A4mxjSpjSboJIaPmjp1SaqamQCnlCCFWMGYFYRcNiK3Xz3GcqUUis1LLU9Q50yxJgQL/
    uz7xoQfuGhwaxHd+/KvDAHyUUglAlN0HmYNd50UfdXV16OrqmgK7jNkklFI9lUz2TU5OUodN
    IidPnqCdnV2DAFwAygAobM6k2LgzZi3qbx8XuDgSQjZ0dXV92ul03nrbbbd9PZfLLdQ0DbFY
    bF5JSckH582b953JyclobW2tUV9fj8LCQg6A3traOrx///5JNm9kQkiOxS39LxmzKKW49953
    FxgG/9WRkeT2QKDkcHPzks8cPXpsEoDc1LQYgiD9Q09P+OrSUn9vU9Pij0xMTPQC0O666676
    wsLC+zKZTENzc/NweXn5o4899thhTdMyhJAki1vZi1n8XS4XUqnUWaHMZNMsoJUuXbqUe+SR
    Rxo//elP1auq5tc0za5pKqeqqiHLcnp0dGxw9+7dvaqqZgFkCSE5SqnC5gav67r5O8G0jCYr
    RM04cDEibAuDbsYkDmfrkGDdoF533XXw+bzre3snPxwO5+z19cFT7e3tYOx7ksXazF9DvDof
    +waApFKpYZfLHSeEL3K5PPD5PKG9e/ekGcAy2MYwRQiRLXPCuNB4NecA682mcf7SVKIoilBV
    ld+wYYPjrrvuWvv5z3/+Hw8fPnzM6XTaurq6sGTJkuWhUOhdmUzmIIB6l8tVNG/ePI/dJpHB
    waGeSDT6BwAdANKWM8cGnHGxk+GiBhgoKstLnOFwOAnADiDAJrKm6/qEnMtpqXTaBE4BAJLf
    7//g1q1bb3U4HEQQBFJeXr78pZde8mqa9jQhJMImvnXndUGHIAiIx+PmGHQgr+EpBlACwE0p
    bSeEhNhzzAFY0dzU+MOrtm8ptdtsPCEguqZTRVXo8ROn1N2vvvZ/um78Enn9T4xN+mYAywEM
    U0pHAERYoJXZBFIopSrb6RoXw8pRSr0AagzD0EyAxN67Na2sW+aHg+f5QCaTxsuvHuxTVe0E
    gCL2LnJs1zsD+BFCpj5/I9V7c3V0d3ebukSHpmklbCxpALRMJhNua2vNhSYnHL/7/R+UZCqV
    ALAQeV1WA4ATAI4yEJzDtMbhr2an+/+NlcOU0sTg4KA+ODhYe+WVVz7icrky4XAYFRUVyOVy
    K9avX3+3oiiJhQsXNsuyXBAKhe2GQfVly5afLioq+s3u3buPJRKJCKU0yUBIxhKz6Fsdsyil
    yGQyVBRdQiaj2dxulQuHwx42TjJ9fX3J9es3jkmSBE1TJnt6egmAgtWrV5f6fL6/Hx4eXs5x
    nKGq6tLCwsLFq1ev/vf9+/efNDdi7PfkGBv/pq8vmUyCEMKVlJTYly5t9paVlRfb7bZinheC
    AwMDw08++WQvYwzl4uJij2HgK319wxszGZnPYxcOPM9Tj8ep19XVH9q/f/9XVFWdABCnlMZv
    u+22wrq6uitzuVxifHx04NSpU6HTp9syhmHk2GZRYRkEzQqGL/S4/PLL+RUrVtRqmuro7OyI
    v/DCS0kLmFPMOWpKSZ588kmyZElTEaUcHwy6s/39XXs6OjolFrOohSk9b1birQbs8+fPF666
    6qpijuO8oVCIdnR0yIcPH0JtbX3Y4/EVVVdX0VQqEUomEyU1NTXekpLiWkmy9R0/fnxvPB4X
    2ebDjFkX9Mz5uQxSsxcCjuNmI8jz/r+/VGAVRRGaprmCweCNDz/8cGFVVVVgz549NXfccYe9
    oqKCtLa2crW1tfO6u7vnU0qXSJJUVV9XN3/VimXzVy1f0qxp6ppwJBrQdF1mA5I7B/3+lg0y
    85mLokhlWfFNTIa29A+O9OZk5WUGZnhCiGwYBg1HoiuGR8fHZUXZC4AXRfGqlStX3jMxMRF5
    4YUXftrV1ZVYvXr1wmg06onH420AVEu+Xb5QgMX0O0EAWwSevzVYVPTemurqD6xcsfy9my/f
    9E5BEG4YGRm1AXACWACgEEDDdVfvuPmfP/+3rs0bLxM2rFvNb1y/Wrhi43ohm83ann9pDwxK
    fQBqAZQDKHDY7fffcdPVH1+zYsm1Drvtat3QN2Vz8kLDoEFzc0EI4WelTC70WFZSXPTTJY0N
    92Qz2RXZnDzInpXIfreOaaF3wOVyfby6av47unr6Xadaz7QalPYy4GIASDMtQAaAYQFThFJq
    vkPBfJ+Ymaa+VPS6YLfbP7BkyZJ/a2houMfv998kSdKVsiwvrllQNb96wXzhe9//oSGJgr+2
    ekHj2hXNK5oW1V02MRmuyObkdnZfhAVw9SKf9ZzFKwBTKej169cjk8kgk8mcc76a/+cvFa+s
    17R69eqSXC53w6lTp2yUUn9BQUFhS0sLKS8vh2EYXDabmzdvXkWJYRhCNpuVVVU1NM1whUKx
    6mCweP2KFctq7HabMjIykjYMg1p0pvpbHbPMv3Ho0GFt48YN9V6va1k8Hn5lz549bQBsAJBO
    p9PLly8v8Pu9i1Kp2Mk9e3af9vv9gcsvv/zduVxus9frPZ5Op3/t9/sLEolEbUFBQba1tbWX
    /X4d0wUXb3qe3HXXXY7XXntt3aZNm25sbGx8j8fj+4Cqqg/EYon3aJpxS3l5WZXX69FraqoX
    zJtXWUqpUeR0em577bUT5X19o8L4eEQYG8ufmYwqVVQUCn6/R6ytra2pq6utLC0t8ZeXl12h
    KMIXQqHc9S6X/9qamtqty5cvW1pTs6Dc5XI6IpEI1TSNt+hlL4oFfsc7bna5XO7/zGb1T5WW
    VlxZW7sg19LSEgVgY+9DR17/SZubm4X3vOc9N2Wz8sd5npR7vVJo795XDsdiMYFtuLOWDIdm
    YfQJi09mnBJZvLXewyUbYw888MBaQsj/pNPpBxwOxzuqqqquKS4uXmWz2WqGhsbdCxfW0kCg
    QKyqqq2dP79uqd9fvq6ysnKVYagDg4ODcXatZpHIBck1hLmcIAAgSRIURSGUUhtjJniWQjM1
    MRpjDP6iqTTzmhVFAYDowMDAwPPPP1+fTCbJV77yFWdTUxOeffYZ1NXV4v777nOsX7e27tln
    n8ucOHFics8rr4y+um8ff+WWTc03X3vVglAofH/f4PAqAP9NKW1lO0DrBDBmD6ZLda+mt4em
    aRyA5NhEeETVVBVAKWN10mxHVJiRFVVRVY2xVxUul+uWXC7Hnz59+lnDMPYkk0leluUrCgsL
    g/39/eXIa7lUNpn4C00Tsne98/LLN33r9ttv9zfU15Oq+ZUIFPjhdDrw+X/8J/DEuG9eWek9
    iqrSE6fawv2Dw/tN5sYwdFDDADUMcBygGzrWrWha0dRQvUxWFCORTCnt3QPxwdFJx41Xb+M2
    rlnuDoUj7v7BoQWnz3Rtfu3wycxLew+8msnK/00pHSKEhAHEWFB50xOJVT5FSoqD2qc/8sDi
    7//4F6XP7Nr3CtvRptnYNxkzyvP8NXfcdssnP/Xwh53hyVH87ef/pWnv/kODALoZuyZZgtDU
    ZkXX9WIA7wMwQCl9he1qDbbLyl1i4OKqqqq66etf//qyxsZGpFIpjI2Nor+3B8GAH/MryvCf
    X/6C5HbagkUBH1x2EaHJSfQPDtVGYvFFbPzzllSC+pcGKeYc1PU8mfbaa69NafswbTVhMP2h
    MRt4/AUrabnx8fGxgoKCyIoVKzy5XA6PP/44p6oqenp6kEwmMTY2Ls2fPz+4YMECtbCwMJRI
    JDo1TeOCwdIVmqa77Xb3urq6OuHkyZPhZDJppoLMlJdiSede8phlt9uRzWYBgMvlshNOpyuk
    qrIkCEKppmkKAMHpdGqqqrgcDlXOZNIghBQtW7ZsicfjuTKTyYS6urr+sHv37u7bbrutWdO0
    RZIkzRNFsURVVYNSmmOsNnch86O6uro6mUx+7+TJloaJiUkhHk8ik8lCUTQ0NNRh48aN20pK
    ajYAxPB47HI2G9+laao0Dcg55HERASEAzzvLi4ub7wWIIQicJklI+XxE6+oKS+3tE5zNJpW5
    XPYyr9expqCgQl63rrJHluWvnTzZ0gIgzBhy/UJZYEIIhoYGlZqa+om+vtB8h8MzPxBwdrH4
    k2OSCpN5x86dOxfE48nP7917aFkuR7BsWWNRY2Pjhv7+/oMMRE3OJmsopZg3b554zz333ArQ
    qkOHDj2xa9fLYTanFJZJuKRMtiAIm/bv37/l2LFj4HkeTqcLBQVBOBwupNMyWlv7OYfDXaAo
    gGEI4DgbFi8uL3a7Xc0ARiil4wycZ9lzeesBlkV7AsMwOEVRqgBcUVBQsNbjdpXquo5oLDaW
    yWRPAGgBEGJptIzlIf/FdrNsgNhyuRyvKArq6uqwefNm8DyHhoYGGLoGh8OOlSuWc++9/173
    7t17XL/45a+Cu15+eejZF3ePHTh8zJZMpc17WoW8PqgNwAibCClMl6obb9XCQSktDQYL3/fg
    +9+/LBqN6j/7+c+5VCr9MoAxAAublzS966Mf+dCio0ePqz/44Y9Sqqoej8fjpw4fPvwqgP0A
    5geDwbXBYFDYv3+/jLyuxs/uR8LFV6BmVqxYafubhx4iHEdg6BoMXYeqKNA1DQ+9792Oy9as
    cKTTafy/L3/d0T843KDrumBNfwIUlBIYuoGljXX8vbddzdtEARyhtkefftnznz/8PahhgOc5
    FAZ8CBZ4sXppI1naWOc6dPxUVSYr1zCGyGB0cPZCboQt0CNtZzr/+MOf/25x7+CICVrnsdSY
    xsa7DEAXRbFu9coVztrq+SgNuLB4Ya177/5DxQCG2XOdSsEyfYRb1/Uih8P+vltuvPYTyUQ8
    88wLe36kqOqj7H3ELUHXuARzBAA4QRCE4uJiFBcXIxAIwOt2IuB1w+d2IJtNwetxQeQJDF2F
    nNMQjUaRSmcISyOE2fONW8HjX2jOT32+atUq28aNG6olydao63qJqipcPJ4It7ae7j9y5Ogk
    W5yzFgbkotMzFzu3nU4nV1xc3DQ4OOjr7OxEOp2GLMsAgOPHj08t6B0dnaS3t18qKSkpb2xc
    VFxQUBCKx8Nj0Wg0GYt5ZafT5di5c+ddqqr2nD59em9XV1cXgCgTLF/yBdC8Lwau8NGPfnRF
    Npt7z759r9XMm1cR2LFjh/2FF144Qil13XDDDTsyGfmuo0f3Fi1aVB9cu3Zttq+vLxkIBF4d
    GxtrO3z48NBll1221GazNcqyTNPptKppmo/Nuxhb6y4oZhmGIScSCe7o0aNCKpUBzwvguDx5
    I0kSRkYiUnv7iMTzAmpqSlFcLDRqGvXkARWZwX5yHIGigBsd1ThNIwAEm9cruAoKJHBcHt+q
    qoF4XEYyqSMUkm1NTd5yWZZrAcSYPtUKUC7oePTRx7T3ve+9f6qpKb1F01CWTo/bBUGo1DQt
    zZ5Tlj07XZKk4ng8MW9iIgpB8CIczgolJZ4SjuPcuq7zJkPPqjmxcuUK+9q1awMOh3NDJJL6
    10QiV9HUtHxLKBT+VktLywiLWVHM1GReCrJBymazNJPJkDyZooNSATyfhVl3oOsUHCeC5yWI
    ogOSxNN4POFhhIRCKU2wbMEFZQmEuQhWzJPEBuDGxYsbP37Nzh3L165e6SguKoQiy+jr68PL
    e/YmX9i158hkKPwbSmkLYw2imC6J1N/KXaGFxpRsNtvdt95669pbb70VLpcrz5LoBmySCFAB
    hq4DlKK4qAh33HYr2XrF5Z7fP/aHxv/+9neTbe1nRgEUOhz26sZFC1d63a7s0eMn9yeSqR8D
    GCSERDFdqq5cygBNKYXD4UA2m3W73e6HP/iBD9z2mU8/IqiKLJQUF234+je+5Y3F46eaGhdt
    /vIX/6nyyu3bcOXWLVJ/f//mp555NkcpPc2A4tL6+vorN23aVNfS0pIdHBw8zQDinKSiBEGA
    oihjQ0ODiUwm43TY7XlGihrgOAKH04nf/uEp7HplP3RdRyyRFrdu3tS8edNlNoHnAWoAFMgP
    FQOLGmpx5NhxfO1/fweBAySBR//wOIoKC+DzevKA06AwqA4OQDSWQDYrc8jrhEzhfvQiQaPP
    6/HW+gqLyFU1tV5Zfe6dff2DfwJwho3vBAtYOVmWjz7+pyfGy0sCJeHJCezZdzDOFgHNkvM3
    bQ2WlpaWPFJRVro8mUxW1VTNs3V2yjzPc9dBRRultJPtNlO4RLYrbJ7IIyMjZ77//e9vKisr
    Q1dXl9rW1pobHRnh33ffu1yhUIj+369+l7FJouF2OfkCn4fL5nK0b3BkiN2HZFnk/hrsYcjH
    PvaxOkmSHgiHo9dPTg7Pz2ZlG8+L8Pk86VWr1vUtXNj4wq5dL704Ojo6CiDGtEppc7P0VrNY
    5t966KGHlsTj8YcPHjzoj0QilvfETTEmHJcHWZRSjI2NIxKJCgsWVJUuXNiQDQSIvaKiwu1w
    OBxDQ6O8zSZpGzduWphIxH+eTmdG0un0BItbSVhsaS7V/VJK8clPfrJaVdUv7Nu3f9WZM53o
    7u4v2LBh3bVXXnmlT9O0FCBs37v3kCccDiOTkQNr1y6/emho6OnHHnvskMvlyl177bU7PB7P
    tng8XuTxeMZeffXVbrZ5vuijr68vUVBQMOp0uhal07mp5wwQKIqCwkI3li2bD4DA4RAppcai
    iYmwIMvKrJQyQSajIJmMoaLCDTNbJoqArqvIZjWYoMw8RZGHpilGOBwuZvEqzWIFdzHzuaqq
    SrDbHXWZTNw5ORknhYX+KzdvvoLu2vXSQQaUkuxUh4aGhoLBwhNNTYu2ZLOUq6wMKP39rRO6
    rpss1xQYv++++wLBYNGHksn09YZBKlMpuVRRDEMQhMsqKytPtLS0vMBAYvpSb7RkWe5qaGhI
    CYLgdrs9mtfrk51Oh6aqcA8NTQj19fNy6XRaFgQbz/MSLwgSdD0R6unpTrLUtAhL9edbniJ0
    Op3IZrPQdZ3nOO6Oq3fu+Pe//cynS1euWAqBI9A1BYau4bI1K3D9zm2eJ595fstXvv4/8zt7
    +n5IKT1CCLGxG6A4h+/EW0C1A8DGrVu3PvTwww87XS5XPsVGDbYo5xd9UIN9TkGpgUCBH++7
    7z1YsWyp54v/+m+OZ59/MSXLStrQDf/HHnpf0Tf+5/vb9u4/RHTdaKWUvkgIGWH3ecnApDkh
    s9ksx3HcbTfffNP7P/LhD0t2mx02UcTDH/kQD0qb//vb361/+MMfdG7ZvAGqkkNpSRE++fGP
    uNra2y/r7evPAXBs2LBhS2Njo/+1114ba2trO0ApbWfMQ4JNcOVigJYoilAUJTYxPhGPx+Ol
    LqcDFPnFgCPA33zw/RgbHQPHEQg8B4fdxnk9brvL6QABhaFTcBxAQUANig3rVmFp0yKEQiHE
    4nFkMhnomg63y466qkoGphkgI8DI+CQy2ZyIfKVbCHlrDvH1JtIb0Aks2bnzqmu/+uUv8nae
    oqSooO7//evX1um6kaSUxgkhE4wdUCmlB1/Ytfux061tDyaSyZFEMnUCeQG+YmF2NUqpu7S0
    9FP/8LlP37N103o8++xz+Jf/+EY0FI7uA9DDrtlDKY1a0juX6tCi0ej/ffOb30xRSv2GYXAs
    CDXGY7Hm/v5BY2x8YpiNcbBxngIwgGnh/iXdYLyZ45FHHlmRy+W++tprhzZ3dvZyuZwKQgQI
    gg02m8MfCASWL1pU1bhz5zVLn3326V+Mjo52s1SIOY+Vv8R93HvvvR5K6UcOHz7cPDIyMgtc
    TQOr2UDLMCh6evoRiyUdq1cvlyYnJ6MTExPj5eXz5vX2jtqXLm3cdNttd1aPjY28+Pvf//6P
    zGKD4hy6rLmWNNx///2eYLDoE0eOHNjW0dEJjuMRiyWwb98hx+bNl20uLCw0XnzxNXsymYUk
    OTA6Oom+vuGy1avXrI9EIge3bNlSJ4riukwmw3s8no4jR47s7+/vH7DErAwuwmz18OHDuauv
    vmbC7XZjcjJs2aADfX1DiMdTMKWchgGiqrpNVQ0AHNP3AYTkY0g6LePw4TNwOOyw2WyQJAmC
    IIBSglRKmwJiZryRJB65XIqkUulixgSPs3nHvYk17qxnftVVV3oNg975yisH/dGojKqq6kBT
    0/yNHo9nMpFIpCmlk6zIKPPTn/409K53veuHa9cuWS7LinTq1Mljhw8f7mNzIIt81bYKgJaW
    ltzY3t792RMnOhx+fzHWrVushcOjh1988Ujb0NBQxJIF4XBpN4Tc/v37DzU2Nn55wYIF1ZOT
    k8KpUycFjuNKly9fc5nb7RZGR4cju3fvGpUkCQ6HU7XZpHQ6nRmPRqOT5kbXUgD11gMsy65q
    ydq1az/3xX/+QunSJYuhKjJUVQXVNRiaCl1XYbdJuOX6naCGVvO5L3zlznAkKlNKBaZtMP1M
    lLciSFlSmv7y8vIPPPTQQ/Oqqqry3iNTAIudBj3ra/Pzlcub8Y2v/pvwz1/6N/cvf/Noou1M
    R/zQkWOONcubXBJPr9978PiCTFY+wzRPU5VrJu0+VyDLnExMf7V01apVn/rkxz/uDwYD0FQV
    1NBht9vwoYce5BLxuHNoZAS5TAaiJIEaBtasXIH73vPO4n/+0r+vLikp8SxdutT/xz/+sWN0
    dPRFBkBSDACEmZbhokqeWTojPTY+Hh4fH4PX48bg4CDSqSRSySSSqSSymQyymSxychZyToai
    yNA0Dbqm5TVYNE9hEQJwJA/ERFGATRJhkyQ4HRJADZzp7oXTboPb5YDX7QSoge6+IWj5tJ6D
    Baup9MH53sfrVe0xwJg35WIgnCOEAsSOvNO5i1LqIISI5q8rChZ6GxsXkV27X+kG0Mcmcdqa
    SgTgKSstWbTtistRVV6EzZetwn97vWooHDXTjk4Gsl43uM6u5Hk9A8Q/E7TO6Lr+U7aTLgdQ
    KopCtdvlIDZJJOyazGrBMIAJlpKeYGfM3GC83t+71KzQRz/60TJFUb7wyiv7tpw50w2OEyGK
    NnAcD0J4GAYQiSRx7FivbenSmu2bNl1u/PGPj/1WUVSJgQ5T//KW+5SVl5ev6+jouLGtrY2Y
    izHHTet8Xg9kEcIhHk9g//4j/KpVywp8Pp/KcSRbXV0pyrLmstnc9dlsroUt5DKlNGNJj16S
    o7i4mDzwwAM3t7W1vfv48ZM8pflCHY7jkEymsX//MWnjxrXw+byIRBKg1AAhGjo7+0hJyeq6
    LVu2UJfLVS3LMsdx3P5nnnlmfyQSSTJwFQEQYQUjF1yUMzg4qPM8N+b3+yEIQ/D5/LDbHbDZ
    bLDZ7JAkCaIoQRBECIIAQRBYGpGf8ezzc8iM1QSUAoaRP3Wdwu22sc8JdD3/M263hExm3JBl
    2YZ8EcybSnWer4qPFUOR/PgQTLZTAKjbjFmMdeYAUEmSHKFQzKHrysTBgwdbZVnWWawyN4Qq
    AKLrxqLh4TFHPJ4BpVlEoxna0dGRaGlpybLfZ7eQKueMPee65guJWQcPHowdPHjwUUu8Klmy
    ZMkqjhN4gIPT6SSKoqYURZVTqXT8HPEqzGLyFIP7ZuPVXIjcOYfDceOdd9zesLR5CTRVmQYp
    hhWY5M8dWzfhmRdern/siedWIy+azjG6MItzuD1fKoTLwMiyjRs3br3iiitADQME+QXbYItk
    /j7ozHug09/TdAMVZSX4p7//rGAYRsWvfvNo9D//+wejDrvE8xwnZrLyEICghbmylnzOGVvH
    nJ2RTCY9gUDgIx/84Acam5uboWvaFOtGDQNulxMfeuj9+H///CU89exzuPG6q0FBwPEC7rrt
    HdwTTz5TUz6vkrS0tKRHR0dPYNqPKWECLLZIZi4GYLFnnxgcHNz1b//274uXL1/u7+7uBs9z
    sNvtsEsSJAaUJFGAIAqw2WxwOhzgubxQFMgbjxqaBlXToKkKZEVBMplELicjJ+eQy8nIZLJQ
    NRUcIVi6uB6j45P0D8+8nGX3ZrZ1Sf+5AGxW8YG1tLE2fGNmiK3PPv/CkyLPvcPrdtr/+OQz
    YV3XBy3zTLQElsKa6gXLljU3kV2798hsfOjsWhKWSZ3t6e3t/b9f/mrllVdswFNPP0tHxydM
    yw2J/R+zfPuCWYZzATJraoMFEINtEBKsklEHUFPg91fsP3iEJJMJzud1l8QTqTjyjvQTAPop
    pUMMpMfOUxn5utcz12DL4/HA5XJdc/To8e1nznSCEAE8z7MFcHpB5HkBmmagq2ucX7Fi/mXV
    1dW9Z8506MxU8aJKty9wR47NmzeLqqpe09nZWWhqrvLP6GwgNf01OQt45XIKjh07xa9du6I4
    nU6Fh4eHe6qrq51AzhgZGYbb7a5OpVJm+sdMV895huF73/sezpxprw+Hww8fPHjYl8vJ4HnR
    AgzzTNaJE21YsWIJksksIpEYOI6Homjo6hq0XX756sb+/l7O6XQOP/PMM6fi8bi56MfYhjBs
    Af0X9K4URdGSyeRrzc1LbnQ6XdV+vw88L8AwKHRdh64b0HWDgSXKTg2U6lOpRHNs8TwPjuPB
    8/kxJgg8A2U8BIGfApeZjAZVpXC7FfXIka4opdTaFUH+c/HX9DI112gWs6b++emnn07ecMON
    v12/fmVDNJouLioKKJ2dp7uTyZSMmZV/qK2t4V0uz+rOzgF7IODWOI4zsxhm/Ewx+x+qaWpP
    Q0OdoqqiVFBQCEmi6bGxUcI2s0C+Ej2H6SK3N205c754dY5YobJYGgKg19fXe1atWl1tt3sk
    u90Nu53zr1+/PnD06NEeRVFChJAhSukAYwmjyFtoJNk1n3ej/Xrxai4AlqOkpGTVmtWr8sZo
    xkwwQqkBMNbHMHS4HDasWb7E9ocnn6+n+dL0GFu8zYWbXupgxV4Gb7c7tm7ZsiXo8/mgqQrG
    x8cQiUTgcbtRWlKcX8hnsVbUmkJkIKsoGMDfffrjfCgU8j37wq6RbC4nCIJgLykpXhKLxe2y
    LD9nCVbm5JiTFIN5P8lkEgB2XHftte+45eabydSzp5QV9FLoho6y0mLc9+578J3v/S8WL6xH
    XW01DI1C5DnU1dXydqebvvDCCyE2+CU2OCNswQyxneFFBVvmcizLsvzT3/7u0Uz7mTMPfeXf
    vjxv/bo1EHgOAs8xOskcPwYMagCGzlhGnb0Ls5pQn/rcMHQYug7D0KAqCnK5HHK5HFrPdOHf
    /+tHxonTZ9K6YYwCGGKpwbOYlfNNaEqpG8A7AHRTSk+ziGUysMnJydBXf/SzX6QKCwvvS6VS
    5vMyYDEjZCY8Dc1LFs8vLS6CYeQrDRlQMts0mACrjudIw39/5wfp//3xz0Yi0VhalpUhFtgm
    AYwzGj9hGU+zqffzikjP17SVeV6ZlhAiixMCY/soISRJKV1QV1t75+233DAv4HVBy6XJ4tp5
    wUef2kUmQtFBSmmc3f8kC1ZZdu829tcNCxM024T0LHAxV8fOnTvt2Wx2a09Pr13XDYhinrWy
    gqs8k5VfDHM5DfG46ikvL28+c6ZjFNNFBRe1aF+I9mr58uWlsVhs89DQEAEIgsEgysrKYLc7
    MDY2homJybOA1vlAliwrOHGijV+9emmBKIqqKIoOn8/nWrx4yTWNjYtrd+9++Tejo6Om0Nlc
    EJW5fB/PPfecrbq6+r0nT7YsGxsbn2JSrIwPx/EYG5vE0NAoFi+uxaFDbVBVBXmBMjA5OcmL
    oqh3d3dPxuNxs/gmxcDVBFtbplK6F3jtxi9/+ctXtm7d8g9NTY0Pt7ScXtXS0kby+8T81M6P
    H36KBc1/Lrzu96Y/z4N8kwFzOGyoq/PTXG5Afv75/ZNjY+MDLFaNsftJ/zn2lBCCkpJS8u53
    v+sKSo2yn/3s58+Nj4/rhBBKCNEGBgaNX/3qV49v3ryZLlpU+9lUKu3q7OwYNFN9mK6mpVu2
    bPHLsroiHI6RYNBrdjTJAIizmJUCkLv66p1ejhPXVFSUcW63oz8ajaQPHtw7Nj4+EWbXPY68
    W3ps1hrCnydeWbt0vC6YsVhCiBxHBEKIoOuGycBlCwoKHNu2bb86HE6uHRkZ4Ww2L1wun2Pl
    ys0LVVWNHjlypINSGjGZK0KQohQai39OZgBtnCNeva5Z9UUBLOYXI9hsNndeqKxPA5HZKTUG
    uAgoXE474Tji1XUaBFBAKXWz0nRyqcFVPq8tIZfLFS6omr9lw2XrycjwEH7285+j9fRpGIaB
    xY2L8MEH3w+n02EBjNMgywq6QPO7l4qyUvztJz8q9g8MlrR3dE1KkkTvf/c984aHh8t+/qvf
    xSmlY8whNjmXbJ1pXQCguLa29v33v/f+AhMwzmDfGNjSdR2rVy7DqpXL8avfPoZHPvYQbDYb
    XnjpZfCiDadOnabpdFplFDGHvM1AiA28yMWmB62aHkJIjFL6VEvLqcDjf3riI/W11VIikUAs
    FkM6lUI2l4Wcy0GWc1AVFZqmQtd1y1jKaxt4Lg/KBIGHKHAQBQE2SYDDboPLYYfLacfeg8fo
    8dPtacOgYwxcDbKPYxaARf/MIreyrrrqK/FE8vBkOPJ37H3q7BmplNIJAKeWLV2qDw4N+To7
    O802MbrFJZl32O2rljcvcRuGbpbGa5gWi2aQLwn2lRQVfnzlsqYle/cf2T86Ft/N9AuEvYse
    AIMsKMiM1l/Ngu8AS72b4E21BLIZfl8Wtsp0oPcg7z3WyPN8rdPpLBUFwW1Qilwul87lckOU
    0g4AKwlB5fjEJBRFhWHoiGcNiKLkRd63LABghcDzGx1Oh4vnOE7TNCWbkyO6rg8DtJ9SDDNd
    WhZnm5BekvYtHo9HkmW5LJPJsgXOCjxMsMVZSus5yLLOSZK9CPnKonFKqYfpRy+5WN967w6H
    Y2V3d8/CdDqDFStWoKamBul0GqqqIRqNTQFD09ZjJlA5G3Sl0xl0dPSKjY01gWPHjo00Ny/l
    x8Yivurqysbi4uIFo6Ojo8yINDHXGYa9e/fiiSeeWDkwMHj36dOt/DQItIq8p99BV9cgiouL
    UFVVgd7eIQiCiOrqechkUvD7/Wq+ABJONrYThJBJBu5jb4Tx+TPvwMjlctmnn37m0KZNm75f
    XV09b2BguNQwAIfDAZvNflaKMM9SWccTscimOFCaTxHmPxKWGsynB30+O+LxEeX555+dzGSy
    wyxWmW3cIm80g3DLLTf5DYN8Vte5pk2bNmUeffTRFjbfs4SQbDgczp06daqloCAQy2azxeXl
    5dL4+HiaxWZz3NHCwsKa/v6xmlQqm7dM5aZkPab+KlteXmosXbrstlhMuYOQXHj37pce6+zs
    4nRdd7DNSA+Afgay0qIocrfffttip9NR+Pvf/6E9EokoLJamLFIh0+NPtc4Fi8cWysvLbTt2
    7Cj3+XyLCCH1uq5XGobhNwwqUGoosixPjIyMdGYyGcPlci1PJrOc3U5AaQq6roPjiqTKysqi
    jo4z3vr6+rqGhoWNXq/fIwiCCFBd1/WELOfGIpHwYHd3V39ra/skpTSDvGltjjHa57WcmgsG
    S41GI2MjoyNY2rx4RmoQlJ7FZmmGjoGhUarrBliq4ywdzKU+crkcACxftWpV84IFVfjRj36M
    3p4eLG5sBM9zuPvOO+Cw26YZknNoskyGRWeMlmEYWLqkER968D7bZ//xS450OhONhCY9oDql
    lPLIO5UnmSDZDFjGxQZflu4kHMddff31121av24ddF2zAC92WsAWAcEtN12Hf/znL+PI0WOo
    q63GK68dgN3pRTIeJYLAuzRN5zFtLxBn54yJfZF9CHXkDeoilNJ9v/r1b65/6aWXatPptFBf
    X4+lS5dCFAWIJn3O8+A4As4cvIwR1XUdmpoHX6qqTgGxTDaLA4ePI5VKwWG304lQRGHgqp9N
    9m4A/YSQcXZvb8SEcPktN15TNDQ8svkXv3v8KgAHmS9YDNNVgJny8nKjoMDv7+zsFBjAsQZD
    d3Fx0crFjQtJW/sZcBxnYxW4JrDQKaWE57nbd2zddJOiqEoqlT7Ffg9hHwcZMFzsdDhuy8ly
    3Otx1yxdvPDKgaHh/r7Bka9RSs+wnaZuAVgiG3OKJViZ2jMPgC0lJSV3Njc3r1+yZElZTU21
    PVgYIHabDYauIxaPob39jLp7z57R0dFx4R//7rP8lduugN2W1/Kl0yn85tE/CF/41/9YWVwU
    bF6/dpVvSWODLRgoIALPIZfLIRKN0f6BIbW9szvS3tnTMTo+uZdSug/AEJsXpqv4JRHEh0Ih
    WlVVlbbZ7LM0SvzU5/lzmtUSRZ7G41mBgVg3i1ki3sJqyG3bttlVVbu6r6/PLQgi7HY7xsbG
    EQwGEAqFMTQ0PEvzQ85KGQqCKbieBlqTkxEUFxfZ58+f74nHY7mCgoCPEE5JpVI2FrPiALwW
    xk6/2B6klFJ873vfsxcVFb371KlTlalUGjwvTL2PaRJ22rBaUTR0dg6gqakeExNxeDwOOBwc
    KLXD7faJhYWFzlgsRlg6x6y0M/vGXuwYomwxTZ4+ffpkU1PT0Ztuuna7KIo2WZYxMDAIVdWY
    popi+tEYIIRa0s+wpAj5GZ8HAm54PE6oqkZlOWfs2dMeymSyA0yf2QWgjxVLRd/o2uH3+yvH
    xuKNY2PJisrK4rv8fl8qFovLbGMYBiA7HPaMYRjq5GRIqKys9B8/fnwq3WeiQUK4ZZFIwq+q
    Ophu2m4+V+b1p991191LUinto+FwyuNwqMe7urondF33sHEzDqD/sssucy9btuxdoigRnue4
    XI5ex3Gc9+qrr/n2o4/+7jlZllMWJ3kwXGC2Q5sBrgoLC4V3v/vdS3ievzMWi13V09NTG4lE
    3alUmtc0HQCBKErw+bxGSUlJdOHCRdEzZ3rKjh1rhSyrICSfmi0qKsSqVYuq3vGOW29QVc6b
    SGSdY2MTnGHk3REkSYTTadf8/nmpTZsWDK5de9nBtrbTuw8ePHTGMIyYRdIhW9Oe5jifE4AV
    CoVffeaZZ2/dvHGDXeQJK6M/WzDOARgcGcOuvQc0yyB5S3qTzcqb2j0ez407dlxV4HG7cecd
    t2NycgLf/vZ3sWjRQvj9XqbxydszzEh3MqBysuU0fD4PKivKWWPO/G3ccPVVZNeeV/2/f/xp
    vaOzU2+oqRKcDnt1Ji8ejbJg5WQLiXYxwcrCXhXV1ta+6/bbbnfZ7bY8e2U+2nwLCoh8foLn
    qyN1lBQVYcf2LXj8qedQV7MAlJOQSsTw4QfvI1/5xv+4+weHBQtlOycNU2dfvkXf1RqJRL4U
    iUQ2Ath4//33L/7bz36GkHydIAjy12zo+lQ60DA/6hpLC2owdCaC1zXE4wnc96FP0baOboXt
    iiZZsOoG0AmghxAyyL5vNlY+54tgBRGEUjrR0nom5PO6gwUF/gdlWVmfyWT2sXQhx4J6VhJF
    vXnJGvcf/vh4sa7rrbN2Y+VV8+c1zJ9Xjq6uLvA851RVuNkE5cyikRXNiz+0Y+sm11f+6weD
    NL9zpWz8TADo4ziuefu2rX93283Xl7S3txu7X3mVu2LDahxvcZYOjYy/V9P1/6KUTrBAamWs
    tXOkPst9Pt+nrr322nfdc889hatXr0ZhIABB4KbSs9TQAcOArmviiy/tmv+Tn/0cV267Am6X
    M+8DRgCvx4MrNl1GNqxfE/j4hx5Ec2MDRFGYnc4lqqpK8Xi8tL2zu/SZF19Z99QLu68eHp34
    LaX00Ky0ThZzbBGwb98+uaam5nhFRfn1Y2Nhzlz8zsdiSZIEj0c02ttHFMvKz+EtZK8opXjk
    kUcaJyYmdwwNDRNZlnHkyDEUFgag6waGh0fPIaiemRqUJAmLFtWhr28I2aw8A4j19Q2RlSub
    CtLpWMznc2t2u9O7fv36lSMjI/3ZbNYElXbMUTPfTCaDL37xi8t7enqu7+7uITOvOf8eJElC
    vuXl9PcmJmKoqkqjtrYCDoeATCYGu92H3t5xrra21t3T02Om4M1NxYw5fRHjx+w7m4zFYiP7
    9u379oIFC04VFxevz2Zz63fv3iMpij6VBiSEn/LKsp7W7/H89Nc8L2Dp0iqaSIxr3d3dmdHR
    sejo6OgAi1VdyPdZNdmfNyTYp5QilUpnHA5p0OmUKux251W33np7cGBg4MiePbsPyrIsAphQ
    VU3hOE6ZnAyRJUsay71eL+LxuMFIAW7BgiqHqmorIpG4wOqKbIRwHky38yJNTYu9PC99uK9v
    rNHjsSnDwwOdmqaZRTtRAAMrV66kzc1LP9nR0b+OUp5btKiWJhIyoVREaWnFBxYubJg8ebKl
    lckKYNkQcrNiFbn22mvtzc3Nd01MTDzS0tLS2NPTQ+LxJJOezE7J8pzH4y3ctm1jYU/PINLp
    LAQhvzcyDCAUiiOdVpzxeMbZ2TkMXacwhf/T75IXRFHyu90Of0mJd0lz87odlZXzn3z++eef
    jcViplF13LIxnHo33MUGAMag7Pn9Y3/c86tf/waaploCwzTw4AlBLJ7At3/0S3ry9JkEW4zM
    RpyZt0LgTilFMBgEgKXr1q69btvWLYRSA0VFhWhctBBf/tIX8cD994LjSB5cGbO1ZNN6n8ef
    egbHT7SAEApKmf5H1+HxuHDv3bcJJUVB194DR2JPv7gnp+tGJfKmk9ZgZS64F9wqwNKKaNPW
    rVvXr1ixPJ9CM3vVgeDkyRZ8+GOP4NnnX8j/HTZLDEPHlssvQyQax6N/eg4OhwNVFcW4budW
    LF2y2M5ShDzO3Q1+Lrq5UzOFRQiZIIQcAPArAPucTiflOA7RaBQDAwM409GJwcEhqGo+Rajp
    OhOX5jVZZud50HxFIc/zEEUBHMcZDMCNMuaqG0A7gDOEkD5LwMq93sJhTIstXnzupT1PaZTH
    z374vbp//ofP3DKvvPS9AJoopeWEEA8AJSfn9JUrV3BlpaU1piCdUmpn3Q0a6mqqS31eD+w2
    GwRecDLAbeoF3F6P+/1333rDYlVV0dM3OGRhcmJsFzvh83lXffD97yu571134+N/8z6uuKhQ
    /cp//W/rC3v2H9QNfRSAD4CdGZXCommYUS1JKfUHAoF/+MQnPvGhb33rW4XXX389SoqLwXEE
    uqbl07KaynRtOjiOoDBQAJfTmS8IYZsoc+ORzmRQVBhAY0MtJEmcYrANXYeuadB1DQQUfq8H
    l61ahr/96AO2L33uYyuXNy38GIBrKKXVlNIylg61Yw7beQFAOBw2BgcHX2hoqO0sKyu1LOBn
    pwc5jse8eQGkUhPZ/v4Bk3XOWXarl7Q9kXns2LHDxnHc3R0dHVXpdJpdK0EkEsPx4ycQjyem
    0oPTDBw3BbryLJyIoqJCiKJ4FlOXy6kYGQkJomizdXe3hzOZWELXdTfP8wUAvEx7+KYr2M53
    3H333aKqqrecOdNRkU5nzgKGDQ212L79chQU+DDtC5W3centHcWCBcWw2ylsNjvGx9MYGYmS
    QKDY7fV67AAEBg6sLVnmApzrhJAspTTa0tJy6k9/+tNjR44ceSKXy2V13YAkSfB6PSgo8KOg
    wA9BEGe9CzKr1dK0q0o+Pajj5MmW7P79Byb6+/sHFEXpQd5Dr50Q0oNp4+os3mD16re//Z3B
    XC71v0VFjvRrrx0KnDjRs72srOrBlStXbmHrUSAWixKO4+RUKoNsVi5bsGCB2UbMDsC+YMGC
    okxGXpxIZMylQ+I4zs3YJR4At2HDhh3j48lbwuE053Ry0c7OzggDRykWYwdWr15dOjw8tuzo
    0dNcW9sg+vsniceDoWx2+HBnZ9up8fEJO/LV1vyseDXjHTY3N3NLly69rbe398tPPvnk4sOH
    D5NIJALDMCzaNqsejoOiqJBllbkHzJzjPM8zZjsGRdGmChKsc8S0OUkmZfT0REhra6TS4Si7
    b+fOq+9zuZyLKKXzmCzCOXuOCHMQAAwA46NjY//zhS992Tc0NLjq1puvF8pLipAnswxk0hm0
    nTmD//3Zb+ifnnkpo+n6ONuJTzIxomlod0majFrddCcnJ51+n+/eu+68Y35paQljRfIAym7P
    pzqobpxXb2UuGGtWLENZabHl/+fTVrpmYNWyJly17XLnz3/9mDI4Mq4wIOlii56X6VwclkB9
    QToTnudhGIazpKTkHddff53H4XBAUxVwhIBwHCgoXt7zCvoHh9Da1o7rr9kByhEYNF9F6fN6
    UFuzAAZvRzwaxp3X3QKvy4l1q5YJTz33YkDXDXPxd1gAwFyzWGalhwk4Y5Ikobu7C1//xreQ
    zWYACixaWI/3v/c9cNikqZQnTKA3xX9Si8P7FIjLIF/9OAygl+0GB9j3Em+k2MAiANc8Ho/w
    zrvvIju2XYHVSxfi5d17aoZGxpoAhFluniYSCaV6QRWWL1s6f2h4eD67Ry8AL8/zSxrqalyS
    KMDlckIUBRcbGwIDQxsvv2z1O67eton7zo9/qaXSmUkWzMxxFAYQlWUl19PTjVh0OYaGhzEZ
    CmdzstLNAvHIn1sMLR0Yrr711lvv+fjHPy54PJ4ZbYiYMTMAIB6LIRqLwtB1yLKMcDiCwcFh
    LGyonaq6JSA43doOVVXR2t4BRVFADR1Ohx3BwgIEC/x5NpKxqNQwIPActm1cA7tNLP3bf/nG
    7T39w2l27SYAn5Pmtta48sQTT3Tedttt392wYc1njh9vKxkfj1oWvHxMlyQJCxYUUa9XVV9+
    +bUJWZZD5rNnTufKpY5X5u9csWLF6pGR0Ts6Ojr5PydeP5c1A8cRaJqO9vZuyLICnj9bjzUx
    ESZlZUFnWVkp3G6PGA6HbFVVVZVtbW2jhmF4WMyKmcyQNV69mWfg9/vR0NCwIBQKXdPb28dN
    G6Lm/78oiliwYB48Hju8Xg9isTSzNsjfVzKZQyqVRioVh9MZRCg0DE3TQKngLC0t88fjCQeL
    WXa2yM9lIYLZMDpOKSWBQGACgCFJElatWomCggJQCiiKihMn2pBMZi1TkMzSl2HG99lHmT3j
    Uca29xBCetk6GZ+dGvxzz1tVVYMQQvv7B8UzZ3pgs/lRUBD0FxUVL2ebzUwikYwSgiylwNjY
    pKeqqmrRiRMnxtlzcxcUFJSn09nKbFaBKIqgFJIgCCbA4mpra8okyfW+trbRAodDhCynJ8fH
    x822WFlCSJRSGhFFMeXxeAyv1wdRdMHptBunT58a2rt3bxu7P+2NbKZ27txZHQqFHt69e3fx
    xMTEFE80UzeZ7zHsdLogitIUQCoqKsTERGwGyHW5nHC5nLDb7SgrE6d8y1SVIpvVpip0rXMq
    lVLR0RGzL1pUsmPNmrWJl19+2TRP1S3xSr9ggGUVm7Ego1BKO4eHR378z//6Ffqr3z66evnS
    JUJFaQkURUZ3bx+OnThFJ0IR03F66mQ6kbNEiJfQJXnL1m1bb73u2mu4qeozBo6m2SpmUDJL
    sA+LYN/jdsHBtCmzqwttkoibr72KPPHMi85UKo0rLt/gGRwa3tTR1RNmk8fDFtXMxSweqqoC
    QP2yZcsuX7tmzZTj/OnW0zjd2gqfx42+/n4UFRUhFI7g+ImTiEaj8HrcWFRfg8nJEAaGx1BQ
    4IdH1LGwtgqapmDp4gbi93r94WjMh7yRpZsQYvaVnBOANavc1RwXFECa4wjmz5+PTz/ycUiC
    AJ7nIAo8XE5HPlVFTXRGLTiNaR+mHN4pCAFlQSkCYJQQMsTAxyTOYZj6BrxW1Gw223/g4AF9
    yaIavqerE739gzkWzItMejuRSGQddhu2b9/qfvb5Fxarqhpi4KjQ43EvrllQxVFK4fN6IEmS
    wwRYAAIlRcE733nrDYWiKOB0e6cperezwJuhlKYBNOmauv4//vObuT2vvKKOjIwax1tau9n9
    WEXzs1udzBZiOktLS6+/9dZbPXlwZUyxk2YRQTyRwO8e/T1+/9hjNJ3Oora2hoAaGB0fx4GD
    h7CwvibPYIEim8vh4JFjiCQyePSpXSCgOHXqFM6c6TAqykvwzttuJLfdsJOIAj8955h+cc2y
    xbj56q3zvv79X2wyDCPOxKNZzEHLltl2E4qi5H7xi188d/XVVzvXrFlybyKRrZ2YiHGyrEMQ
    8kxEUZGbRqOj+osv7otPToYibAxF2UJ31tiZ63hl/q7777+/QFW1vzl5smV+vk0L/wasGMhZ
    oEsUJZSVlSKbVZDLyRbGK/+zqqojEknyBQV2IZlMcopC561Zs06KRqN9IyMjoxZpQ/ZiYlY0
    GoUg8JsGBgbqE4nEVDpt3rwKuN0eGEbeVy6bzcHn86K0VIfNZkc6nUM6LSMY9CGXS0GSHAiH
    Veg6ASAgndaFkpKSwJkzZ8wNrAkCUhcLsGZV2Jq6VKOwMBDTdWqoqooTJ06BEJ6BQQ66bjAm
    BOdkrWYCejMjQRQ2z8cJIcNsUzjONoLymwFX5o9ls9mxQKAgVlxcUkqIHQUFLm1oaMhA3j6o
    UFGUnGHQtM1mw9DQCFddPb+xoKCgLRqNKgB8kmRrSCYzBbpugOMoDIOKdrvdA0AihEirV6/d
    MTGRWRWJpFFfX2pMTg5GVVW1YbpYJ7VmzWqvy+W+xuv1izabPamqBsnlEhOnTrWELWuKcg4c
    YMyWw3zuc5/b0NPTs3gaXJGzWOf58yuxaFEDLS4OIpFIEE3T4fF4YLfb0dk5CMOgUyxVUVEB
    RJEiGBRAqQC324NAoICGw1FD00SupydCsln9rDmnKAZGR3OOysqajR7PkaFkMqlSSrOz8Yxw
    oQPO1KWwF3W52+3eWV5Wtmr+/HlVZaWlpLS4CG63C5qmghNEeDxejIyOCYPDI75QOFKRzeYS
    yPvk2Nji7WQXNXVxcxG0rOyVYRjzqqsXPPz+9723JBgM5H2iZgvyZ1sxnEN/pWsaHv3jE9i4
    fg1qFsybErqbTJYOA8uaFpI1K5YKz7+8N1NVWS7VVM0r6ujqqQTQTin1E0L8mIP2ORzHbdi4
    cWN5MBgENTSoqoLvfv8HaDnVioqKCgQCATzwwAN4/PHH8c3v/BCpVBrJRBT/9R//gpf3vArJ
    4cKpU6dwxZol4GBAVxWUlwRRVlrsNgGWRdg7p+kaS+AyGxYbAGSO4yDwPGySDalkEplMGooi
    Q9c1BAMFKC0pmgZXNM9azRD0g5pP06rzCjMAFMJ0Y2fjzQQrAIosy4995/s/rN21a/fmcCRa
    0ts/EGe/x8sYSjkWi2WymQyu3HoFV19Xu7C1rb2TveuKAr+vpqKiDIZhwOtxw+NxO8YnJs0U
    4bodWzeu3LB2BQaHR9E/OJLGjCbJJAPQBZXlJZ93OexNZ7r7n/jTU8+NsnSaCVLHAIww8X4E
    +X5a5j3MdiX2lJeX19fX1089SzDgAwDZbBZf/drX6Te+9S0lk8mSv//7vxMfevABxKMhfPU/
    v4n9hw7jzttuBs8RtnjG0HK6nW7YtJl86pFPotDnxvPPPYv3PfRh43hLW6pvYMjmdtrtN+7c
    SjAF5PJziucI1q9cwgV8nrpQNF6NvN9WlBUQpHERjW1nvV/uxhtvLPJ4PGsAUk0I1LKyQr2g
    wMPJsgqO4+F02kGpjliMoKFhoVRcXOwaGxuXwuEwZeOFZ6ax4rmErXOVGiwvLxdKSkruOH26
    9cbu7l5iMk9v1O/KFIjnW7jYUVFRgsHBUSbyPfv/hUIxUlhYKUSjEZnjnJKuwx0IFJSOjIwM
    UEoD7F1kLMzimz5uv/12Rzab29bfP2DLC+4J3G4XNm5cB1VVoCgKRkeHEIlE0Ni4GMXFXrjd
    LqTTGg4f7kBlZRDpdBR1dQ0Ih8PgeRsMQ0M6rRG/v8DPcZzHMAw3pdTFxNg85sC/y0oosJit
    FRQEMqFQGJQCmqbDbpcgSTaIogSeF6DrFKlUFmd3hyIzGC3L5yZLHbHEqigsBThvcnzRXbte
    Orlz5zXfvvzyVffYbLYFoshpBw4Mp8yMiqqqUV3Xk06nA4ODo0gk0qULFy5s2L9/fwpAoSAI
    i0KhlM0MDapqCC6X0wtArKysnOfxBK5paRmXCOHhdApad/ewaSiqA8iWlBQLGzZs/rCqCvek
    Uon2lpajLyeTycKhoSGazWZz7B6HCCFjLEbLlntU81OCIBAIIBAIcA888EDD5OSk3cqeWpmr
    mpoFWLy4QYtGI1oyKdn27NkLwxBQX1+P5uaF8HhciMfTU3OjqKiAynKWHD16BKFQHIWFZdi4
    cQ1aWg5nCguDqK9vdLW1hTjDIJitFUwmNZSVuUoLCwvrk8mk6RFptphSAFDhAhd1GIYhAdhW
    WVn5N9u3bbt827Yt/iWLG1FWUgSX0wGOeUjlhccKFFkmyUSCjE9M2NvPdFbseuXVopf2vFY5
    MjbhpZRqbPBKbDGcKgmei4oVdr1Oh93+0Lvf+c4tmy/fNANcTYEn49ypwSn9FVt4IrEoOrq6
    sWXT+hm+X9OMF4Xb5cD2zev5l/bs446faNFvvf5KoSQYaBwPRXqRryaMswqE3J/TAL3OYSss
    LFy3Zs1qMZ8GoBAEHjfdcD0M3UAqnQLPc/D7/RB4HtQw4HTYsH7VZuRyOew/cgLegiCCfhe6
    +wYRCoVRWFgAj8uBYMAvMnBlTROKuMCGyG8ggJnpQsrzPNra2/E/3/4O0uk0MukMAgV+UGpg
    02XrcMtN1015e1nA1EwohBmgQsZMG4AL0c9QRgH3x2Lxrx44fPRgSUnJw8uWLSs9fvz4PMMw
    BlnQUpPJZDoSjWLJ4kW4/tqdwfYzHY2GYYQBcEXBwtKA3wdD1+D1uFAcLLR3dfe6ABRWlJVs
    uv2maxwOhx3Do2MIR2M5C6jNAdRXWhR4cMtlK9fv2X+sleabcguWVOckY+jG2M7XtHCwPgtq
    0ZXZbTab3eFwzLL0yLcsOn78OH74ox+rmUw2UVJS7Llmx3bic9nw4x/+gf7md48py5c1i7lc
    jnM57QClyOWyCIcj9Lvf/a6aSSXFz//tJ8ma5YuxbtUK8uSzL6Zj8WT8t48/U37FhtWi2+mw
    6DQpDEpR4PPA6XS4EI0XM02Dj427C9aK8jwPTctn4R988MEKv99/Rzwev210dGLx5GTYE40m
    uHQ6B13PW//wvABRtMFudxCv1yN4vRXe8vJ616pVfGE8HvKcOHE819fXn2Qxi8d0BdFFbwpn
    /V9yzz33rJ+YmPzE0aPH3JqmWdiraYbKqlE6n/8VQFBcHITNZoNh0HM6jHMch1xOhapSgeM4
    2et1G7kcdaxfv3EzpUicPn06RiktYL5HuQvdFFZWVpZEItHlk5OhKcYhm5XR1taJiooyCEL+
    fcmyzHQ1ArJZDUNDkwgG/eB5DXa7A6mUhqIiDyKRHHSdh6JQ2GxOhyiKblmWXSxm2XAJ+t2Z
    Qnq73a5TSmG327F69Sr4fF5IksT0PkAqlcWpU13I671xjhQhmQWqicHmsdWUWr1QgEgI0UdG
    RpM///nPflNXV9e+YcOGBxUFW1etWlUXiUT6k8mkHYAjl8umnU4nNQxKensHpaamuqaWlpbh
    dDodJYSrTadzXD6NSyDLGu/xeD0AHEuXLl0XCuUqE4kc7HY7eN5QwuGwwWKWIkmSft11N9ya
    zYr3cpwqt7WdeOno0aPDmO6bGmPxapTFqxC7b/OQzYcUDodRVVXF67ruVhRlFntF2CbCgYUL
    64yjR48kA4EA0TRiS6d1LFgwH/Pnl6iKkoXT6RDj8axFdwmEQiF17dq13KlT3Xw0qiAalVFU
    VKIdOLA/UlY2rzgQcHvC4cxZmxrDIDAMThRFoYhlMAos2YgL12AZhmG32+33b9u27e8+9tGP
    Vly+aQMcDjuoruUFsZoKw9DyPeMIAXgedrsNNimAYMCHxQ01uHr7JunQ0RMLf/DTX5e/uOe1
    qmxO/j0T802yhSGFOTDjZMwVTwi57bprr/nAA++7XxQFHoY+0+Uc1Min3CgFz5Oz/K9MRouA
    4tnnd6GvfwjFwcCUZmW2cz0hwJrlS0hZaZHY2tEtL2qrFGqryudPhKNXUEp7MG1RIeLCW1H4
    SktL62uqq0Et4vYtmy/H2tUrEI1E8bVv/Bd+85vfIJ2I4eG/eR/8Pg+KAgX4+a8fhd3pQTg0
    gQ/cezcee+IZtLSewZaNa8BzgNfj4tj1Odi1Wq006BwGq9lpZ45SoL6uDn//uc+ir7cXf3z8
    CXz8o38Du00CRwCe55jhqJWosmiyXsfOavY/vsnF0EC+5VGIUnowlUrt27Rp0x1ul2vFnlde
    GWKUvi2VSqcnJychCE2449Z38H964unGtjMdgwD8wUDA63Hne166XU5UVpSJjP1auGPrpqql
    TQth6DrGJ8JIp7Mapiv/BKfDfs+dN161JRJLyAMj4wcsAWiSBSeTuYphum+kNktDMgU8dF1X
    c7lcLp1On81gEQ5nOjowMTkpA0h7PW6XxyGi5cQx/OBHP00nU6mopmmluq5xZkrdJklwOR2G
    LMvJn/7859Li+ir3u++4kVSWFRMGzGO9A8P+UDjq8zjtU+L4vHcARSqdQU6WCVscXbO0fxd0
    mODqE5/4xBpFUf7p8OEj29razkixWBKGAXCcCJ63ngYUJa+/iMdz4PkYkSSb4PN5iufPD+68
    8spra7u62n77yiuv7NV13UEIiVj0MepczYuPfvQjNbIs/9PBg4cawuHojNTgNHji4fW6oao6
    VFU9J7tFSN69vrGxDrmcDE0zZnl/Teu0ACAezxCPR+AjkfF0aWml3eVyFjY0NNS0tbUdMQzD
    ZO34C40DkiRVxuOxsmw2N3Ufum7g9Okz6OrqR0FBACtWLEJhYSHSaQPd3T1QVQOUEqxc2YBU
    KgyvtwSnTw9i8eIquFwOpNMyKOXB87woSaJDlmW7JWbxlypWAeAoBavqPA5JsmH16uVobe1G
    PJ5ibu8EHCecIyVIptKHryOVvJhYZW4KVVVVE21tbWfKy8ufKywsXJPLyTXbtm2bePbZZwdz
    uZwzkUjknE6nxvOCODo6idraBeXLli1rPHToUAogZdmsMgXm02mFeL0+d3FxcVlhYdnS1tYw
    DxDYbCIMQ1WTyZQ5V9WrrrpyvSAE7ovHdafTGX/t8OHDg2z8mLY/owDGLFV4qVlzaAbpkLdl
    oxmbTaJgbX6sGw2XywWOI/ro6Ghq/vz5Dk3jiNdbjEWL6rTW1hPj8+ZV+nheEK2Ml6bp6Ovr
    y1JKtaVLV/qOHRvgFYUjfn+BrihyOByeED2eGnckkiOz/fHyAnkDbLy5LNo//mJF7tcWFhZ+
    YeOGDcF4PI4/Pv44E4abgm99quqIWquMrA7czHR0/aplnq7e/hvaO3tdlNL/Rd6jR7YIwC9K
    88NxHHRdX75u7Zq//cynHyksLS6CpmtTjJSpBSEAjh0/gaPHjuPed90FSRBmpA/z4lwDu199
    DV//n+/TyooyUlVZzpgw/ayUog6KyvISNC2sE57d9WrsF489GxEEPsUaJytswhBW3n+hR0Fx
    UVHQ7/flnc6ZLYNhGLDbbCgvK8WCqkq0dfbC7XGhumo+HHYJg0NDOHTsFLwFQVS5JaxavgSd
    Pb04evI0NqxuBijgsEkcW9ys55ymCM8RPAghhCMEsNttKAoGMTkxAV3XkMlkoKnKlOcVqAG/
    zwNR4GdlBU1yi76hYHkB6QGZgRc+nU6/0NPdve0TH3+4KBaLbTvZ0hIC0J5IJtPjExMwdAML
    6+vwwP3v8f/DF/5lRSaTHSwsDNhEMT+2REHA4oX1PICikqKg/aZrtos2UYSha4jEYlTP0yoE
    AOE57oqbr75i5/qVTcLn/u07/ZTSQTaZswBCzHJiFNOmqW/EXDEzOjra39fXu6autmYGowRq
    wGF3QBRFQ1GUjCIrWiYRxfHjJ+nQyOgkgHSB31cmCsLUXHLYbSguChK0nckpipp86eU9jpuu
    2ihoqkzZM4uBUs3sJUlneLMBXb0DNBpLquyeectCflHHZz7zmRWKovz3kSNHVnd0dIFSAo/H
    zUqxReZFJFpK6vkpwGV+zzAMDA5GeUUpWLhkycq/cbvdweeff/4pWZYJM3W1Ojxf1HH99dfb
    bDb7gwcOHLqiu7vXwjidnQasqpoPQji0tnacw6KBwO/3obm5gXq9Lpw82UHMaqvzGZImk1lS
    VFQk7N37yoQkHYsLgjgSjUZbDMOw6uDIRczz8mQy7chrevgZC5aq6ojFklCU/J+RZQXxeBoc
    J2D+/FJwnAKHw4WJiTSi0TRyOQUFBS7kctT0LON4nrfGK/FSxCxL7OAAEMOgyGZzU+ahoihA
    kkQIgjRlxaBpFLqOKebKqsl6vRagc1BAobFWNnxra+uBa6655mRra/umBQsWrL7yyivDzzzz
    TCoSCetlZfMVnhfEfDFEr7h6dWNTPB5P6ToNqKo2NeZSqRyqqwu8S5curU0k9IJUSgYhPCRJ
    gKYpVFEUDoCxdOnS6srKhTcPDiqB4mIh09Fx+lQ6nRHZGIpaNGYhTPdffV0nAUVRDMPQ+wsL
    gyohnGQdO6aQnRBCRVHMGYZBHA4vDQY5IopcamRkeLy2tt6jqsmpMQ8QKIoOj8erHz16JLxk
    yTKxqKjEI0lOEKJkKUWUEOI7n/WJ3S5A19NaLBY3MO18P8PC5UIBVnZycvIX//6Vrzh1XfdS
    Sn2UUg8odVLAYbfbK2675WZ3eVm+Su/goSPY/cqrZqpGBsl/JECOEE7RdS3HHrIDgMg0DnMy
    MXRdt/l9vvc88N77Fi5d0gRd02Y1bzamdDxVlRXo7+/H+Ng4vF433E4nAApNUzE2PoGnn3uJ
    fud/f0rHJ0LkkQ+/n/o8HmJMgStjRjWhmY5b1rSQe+7lfaCUjmqaPgigw5IGvdhSb7fX53VL
    osiYNjojvQlC4fW48y0/5BySqSTsUgGef2kPHG4PouEJ3HrPzRAEHksXL8T+AwcRi8Xg9bgh
    5IU1Zk8qYdaCdynL0zme50l7+xn88Ec/RiwWQzwew5f+/avQmE0Dz3FwOR1433vuRn3tAmhm
    9eBMlfslwYGm1okQYlBKD+4/cODQBx58/7Vf+pcvln38k5+8ubOz63FFUbTh4WFD1zUOBLjz
    tptJW3t79Q9/+osCn9cj5s1S8yzn8uZG4vN6ijesXck3L14InY0nWVZAMVVuvmjTuuWbPviu
    dzj+9Pwrev/QaLflfZi6qyHGZGVxHm85a6BmKcLcxMTE4Vf3vnrTliuuEKdbQFEYuoGlS5pQ
    vaBKOtPRaSSSSTk0PorR0WGqqlocAL+ooY6zSVJ+Y0UNOBx2VFfNM9sHJYaGR5TQxLgQi8V1
    BvyM8tJie8DnnfaOY+8snclhz4HjVNU000vG6r12wS9UkiQIgrBC13W+vr6+paamVjIMw2YY
    VKKUigAEt9vtymRkSdPyCwnTjhrZbFbnOM4ghNM5jtc4jtMJIbqqynpZWdmKQCBwYHR0dHIW
    q3PRR1NTU8PY2Pgt3d29vMPhgKKoM5gma7Dv6elHMBhETU0VstkcslkZhOTTJSUlQQSDfs1m
    EzA6OskPDY3PsHM4V9Vh/m+JgsfjoT09PREG2sOYNoC9qJhlGIafeTCdBQZNKwbDyMdch8PF
    zB4lzJsXRDodgtdbivHxEVAKxGJpBAJFCIVU9lzAEUJmxysOl+4glFJit9uxbFkz/H4/XC4n
    li1bDEoxVZEGEPT1jWJoKHwWuJqpybok12h6TtLR0dEeVVWfLSsrW3PgwCH75s2btm/fvl3t
    7u4eraqqVWw2ySXLOkKhGHp7x/1r165bAwhOTTOm3lUqlYPTWekuLAw6jh8f5cw0XZ7IUImu
    63xpaWlg9eoNq0dGjHn5MJMcOXmyJcHWd1MLO8zSg0nLxoT+uc1wKBQ+XVZWPup2e6rS6ewM
    /7pMJodcThWqq6uFTCaTKyz0UkWxE11XMzabLWuzOcVMRp6xEUmnFRIIFBJd19OZTDrt85V4
    bDY7VHUiKYpiurCw2BmNamQ2uMpvXiRMTnZnk8mkSQaps3WuFwqwdimKckxRlBLku1RXIN9G
    wgegsKS46OoPPviAu6mxAYau4Uc/+Rmee3GXQik1S53N3kSmaDLJFofkuRD8hR6CIEBV1YrF
    ixu3cBxHvvXf/4N33n0nfB43rO7mJoPldjmhaRoe+tindEEQ0FBXQ5wOOyZDYaOltU1tbe/U
    OEJsH3zvO8Vrr9pCTCYOxtlO72bZemN9NXE67GI6kzVbC5j3m8TFN1AVBV7gp1rimdYFoKDI
    35Pf5wPH5cugI5EolFwOh461IFBUioBLRGNDHXRNQ0VZMQjHY3h0DG7nAiiKYu0Hxc1ucHwp
    ARYBUFQUxLXX7IRNEuFyOGCzSThxsgUdHR2445abwBHA73WbQGEa/1j0V+Ts4UPmYBE0dRIG
    gJFQKPTM4088seVrX/k3579+8QtVn/27f7ilq6t7pK+vX9dUlSME8Hrc+PTHPyzqmlZUVTmP
    pa3zbOei+hpsvXy9dN1VV8Bpt+V95EDhsNtAQCQKOm9pY13dJx98p9dhk7D7tWMpw6AhFqyS
    AJIsTRXDm/DIYc/a0DTt1Weefabvnnvuqq9ZsGCqelanBmpqFuDed7/T/oUv/mtROp2hY6ND
    VJOzACAECvyByzes43im/aNGvp/lutUryC9/90d7JpvTVEUx4rEIQuGICkCVJHH+ji0bnD6v
    C4ZuTKV0CYCW9i7sP9KiWFKbpj/eRc0RVVVx9OjRP+RyuQORSKQomUyW5XK5ck1TS3XdKJAk
    yb9167aNLS1tJaFQDDwvYcmShVCUuHz48OEYx5E4ISRGCIkTwiUIIWlmKzMWi8UybG5wczE3
    zIWE47jlkUi0cuXK5QiHozh1qtWiteKnAj3P8ygtLUFNzXw9l0vnRFGigiBxhMDQdV1NpZJq
    JpO0E+JznTnTj7yI//wC+byuBJBljZSUlAg9PT0Kewcp9l6SFyvbyINbg5sNrMyv80LqvF+a
    x5NvO1NSEgAgw27Ps1eKooPjeMTjWZSXc5AkGwQhL9kxTHdSqx38pTs4Sik0Tcfg4AhGRiag
    afmmz3a7HQsX1uDMmQGoqg5NOzsdeHaK8JJcKmUsVoZSiu7u7pcXLWq86+TJ1qZXXz3g3rTp
    smuamppOiaKg2+0OKEoahHDo6Rkmdru9sLi4kFVETlfPRSIZ3u128cmkPMWu5lOhvFBYGPBt
    2bK9LpGwzwuFZFRXu/TBwdNDyWTS7HyQIYRELTIg9Y0AK9PB/dVXX+299tprX12wYMH81tZ2
    YmWwdN1AZ2cft3z54qLx8dGI0+miuZwLhERRV1fvT6VkWy6nWHzJOMRiaVRWltuDwUJO13XN
    bnfB4RDpyEgqs3hxY6HdXlAUj8dnzJt8SlSA260ZR460J1lltxmvZsRg4QIWQKvQT7GAI8VM
    WZSWlgrBwsBUEK2sqIDb7aLJZMp01DbRa8giPE5atAw65sDdXdd1ACiqKC8vO3z4CBoXLYTd
    JsGgBgYGBuB1u+HzegAAff0D+Na3v09/9dvHsolkMgogtWvPXusu2ltcVFj+4L13i/fdcxux
    26R8CvRcAnmzuTUBKsuKUVjgE9OZrG6CKyZuj1s0Mhd6n7qiyJquaZaUi9UjCggU+KEoMiSb
    DaFwGPv2d8PlLUB4chw33XUTRIGHpqlwOx0oKgxgcHgUCypKMBmOGpb38Ja47bMQQwCgKBjE
    ls2bpxzaAYqenh6kMxkUFwfZM9dm3O9ZIIvMiFrkYhc/azd6TBsP7t+z55UzHR0dK264/lqI
    Al/xuX/4f8Ge3j4+lUrB43bB0HUUBwvxhX/4NHRNnUop5ysJXfj8pz4Cj8vBTGLzwHheWREk
    SSyomldW+NkP3etoXlSDXfsO40xPf4S9Dw55T57ErHnzZlKhFEDPsWPHn/rdb3/34U98/GF+
    2qeKguMI7nv3O8nw8HDwhz/5mT7QPwCfUyIcR+ZduXWza/WKZZa2TAYMSrH5stVk6+WX+Z98
    bpfsdTsFXcliIhSmHCG1N1x1xfwbd24hM0yIKUUml8Vv//QinYzETAPiGPKmqglceAHI1P0/
    /fTTaRZ3qEXbkQHgDAaDJZRiYzQaZw7nBmKxJHw+yUilUgnGDlqLBkwBchxzUAV8nrgVlGVF
    0jQN/f0DAPI97yorKzA4OAJN0+BwONDYWE99Ppdy9OihSF9fX0gUxYTT6cwSQtTCwoB/yZLm
    Wo6zu1pbe/lUKnOWNcP5WLFcTiWBQKHI7i3DAGV8LmKWYRiqIAgGIRx/rtQLQCDLGiQprx9z
    Oh2oqChEOh2Cz1eG8fGRKQPIbDZ/GQ6HDTwPqGpCk2VZnxWv6KUOV7quMzf9aXd2r5cywbuC
    XE6b+v65zEbPA83JHIMsA4B85MiR3ubm5t3V1QsWHzt2kuzbd8i1fv3qVcFgIXW7XUgms1O+
    ZGfODKC/fwKqSmcwnz09IYhiHJQSC/Opw+322XbsuKZeUQrcIyM5IkkibDYl297enmQsr2pZ
    5885jv5cvOrr60tOTk4+vnhx45WDgyPFmcy0ls9sDN7b63UtX75YEgQPSaU4FBSIPofD7j1y
    pIunFDPAUjqtIBbTbBs2bCy12118LmeH3c4ZJSXBgurqhXVDQ1m7qhpnNSIPBu0IhbqyQ0ND
    UUvMOuu+LpTBMphOKsGCSwrTrtEFCxvqXX6fl7Wa0bGgqhLlpaX8mWSXjHwZ4zAzUBtmgc5k
    BbLsAt/UTvzP7AjVRDIhf+JjH8bChlo4HXZ0dffgX//9q/jwBx+Ax12PXXv24itf+5a+78Ch
    BKV0HNMVDTEAAs9zSzasXTXvQw+8x3n5ZavBc9y0IeN5LR4MBnC8KCkKiAPDY4QF5vSUHmUm
    PfqmgS6lNB2JRJO5XBYet3OWb1f+mspKi5HLZsC7PGhtO4MTrR0IFJXCZydYvLAOOtPD8ByH
    4mAhxifDGJuYRP/QqGbJi2vM/0W/5AGLEMJxHPoHBvD88y8gFosim8lClnOYDIUQCoXxz1/+
    KqhhwOVy4u5bb0RZSZFZw5zHWudnsDCHIMtsITTU29u7e9eul5c1NS7krrpyGwoL/LZ9+/fP
    6mFpwOmwgxoi806jU7q9suLgVMsfUAqDGmhaWIv33X2za+uGVVjRVA9N03DoeKsRT6YTmLJt
    QGqWOPRNlXJTSg1CSFqW5cd//n+/uHLrliuaVq1cbml1pcPn9eDvPvNJvry0hH/1lZfh4A26
    dtVy/4c+cD/cLsdUyt3sDhDw+/CZjz0oZjLZ8lwmQfYcOAZVp877735HzQfuvZ0L+D1Tfm2g
    FIRQvHLgOF7cezBnYbdNpvssc8UL3BBSABrbaYZZrIkCEOrr6wVN0+3TO1sOqVQWlZXziMPh
    yGaz2TBLv/YzoGUCWZnN37M8ii46p6NpQ16vJ7Vv3wFPKpWBJNmwZEkjPB4PBgdHUVxcjMWL
    G4xkMpp+4YXnJ2Kx2BiAMV3XJ3VdV9atW9dQU1O/aHQ07B8Y6CKm3un1GkBbgVYup6Gw0C1J
    kgRFUWTm75PGdPsi40LjgK7rEYfDqXAcJ55L15Ivf8+gsrIAhACVlSVT7NXkZBqKooHnRcZY
    UGhavqrQZgMSiYgiy4pqSdfoc/lezg2CKMmb0lbB6/VCFCWIoghJssHptGPlykYYRn4cDgxM
    YnIyiZk2DdNYiljdR+c+XlEAuizLqcnJyV1VVZV3nDnTHYzFUnj11SNCJJKApukzHMxZ1SDr
    Ezn9fvIsnT4DdCgKRSLBSYBfGh3NglIOHo+IVGosOT4+Zr6DDKtCjc9mpt9IzMr3o6faSy+9
    eOTWW297ZvHixncdPXqCm82CdnUNEFWl0vLlS1BWVgaHg7g6OroRiUx3PJg2JyXo7Z0kjY0V
    RWVlJTQctsHrpZwkVS7o6orxkUjuLHDldIrweGT94MGjUU3Tpux/GDOXvigGy5LXNbUSCfZ7
    OACiw2G/Z/26tQ5JkqCpCqhhoKQ4iBXLm8UznV1mWXOEMVh9pjaDnVYnVG2OJsfg/gMHn/zJ
    z//v3huvu8aeTqfx0//7Jaqr5sPjduFr3/wf+oMf/zw3PDIaYcDKFN9NAPAW+H3b77n9psb3
    vfsOoYI5vxuGdg53dzrL4iG/6DhsEooLAzymS4bNXZV+MXQ7A1jR0bHR8bGxsUXFwcJZ7FU+
    dVlcFESwwIdXDxxGf28nGhqbMTE2gvfddSNEkYemqqxqDPD7vejuGMaJ0x0YGQ/lWDC1Whxo
    uMQHYYnIbDaLZDKJVDqNrq5u7LxyG1YuXwqB50AA9A8MoLXtDJxOhyU1ihmPk5C5595nBS0N
    QEpRlN27Xt59zz133V7s93mwYvlSNDc1AnTa5X9Ko2f1WLO0LTJmgDEdxYUF5OEH7wFHAKrr
    iCdSONHaqWG6EWqWzb3E+dirN5IiZL+vq7Wt7Vff/d73P/sf//6vLpdpoQDKTHXd+NAH3ocN
    69dgzyuvkr+5bC2aFzfOaCo+bbdgoKG2Gl/9589yj/7pGUQiUXzxcw9zK5sXwSYKUyydqUEb
    mwjjJ799Uo8n0zHGMo0jb7QYYoH4oiuJTcbKYhIZppQKAPimpqaNPT2DblXVIAjS1ALP8zax
    uLjY6O/vTwCYZEUEfQxkWPvdKRYWfi42H9yePXsOrFq1+mdXXLHpznA44vN6PUZZWZlw8OBR
    buHCOpSXF+fOnGmLnDp1alJV1VEGAIeDwaC+ffv2zYLg2NDa2uuMxZJTqcRzWTmcC2SZVX2i
    6JTsdrtNURTRomkCLrLoKJVKjfh83qTNZnOpqoHZRpGEcIhGk2hsXACXywG3W8XwcB8CgQqM
    jQ3PAGSGQaBpBhwOCU6nQs+cGcxQSq0WBzLmtvPEOTRYeUbEbrczX6VCjIxMIBSKYXh4AoRw
    EEUJCxaUMwCDWayVlc2a+xzhTL8oYlBK5ZMnT57avv3KE8XFRduHhsaQy8k4caJ9FsvGneck
    OB/zODCQZsUG+UIRj4ej/f39WVXVyKwNyQyW501sCCkANRyOxNra2n6zaFHDuuHhsYUTE+Gz
    bEkGBkYRiyVRXT0fqqqjr2/snGMtDxgp2tvHSSqlkeJiH7q7s2RiIs0rCj2nsW9ZmY329h5J
    Dw+PhBhOGGMfo7Nj8QUzWCygTFXDcRxHKaX1jYsWbdq08TJiBRoOmw07t1/BPfHUc4WpdNrM
    xersYuKWReNSsCOZeDzxne9+/4fR3/zu95s0VXNmspnid9x4XcWHP/EZee++/SFV1UxriDEA
    gwz8zaurqbrxYx+8v/qma64iNpsIfVb14exm1lb9lZkqFHiCYMDHY9ruwI58hctFWR6wQZka
    Hh45dfzEic1LlyyeYd5ogiybJOHdd90KXVXQ3t0Hm82G8iI/Fi+qz6dQLX0FnXYbwtE4Wjt6
    DVlRzT6RqVms4iVPE1KDYmF9PRbW1+FkSwu+/4Mf4vprd8LlcMAwNHAA9h88jFOt7fC6XeA4
    AmoQWO0ayDQQumQaDJoXpqmU0rYTJ1taWlvbtm+8bB10XQcBZS1kpq0PZqZxrUyjJc1psZrg
    CAE18r9reGwCPQMjigWYp1gKbYadyZs1IgSgsiqjZx//0xMbtlxx+dV33n4Lmdb0GayJKrBq
    eTNWLF3MegtqLC1rWOwWjKmfLy0O4oP33gld1yHwXL4JN0uNmvepGzoefeolevhkW4btAscx
    7YsTZuNuLnqUahbWyfTgwS23vCNotzvWDA2N8lYAoqoaYrGMUFdX7+3v759tVhu2zANrCmrO
    0lGHDx9O9Pb2fm/p0qVn7HZ78+Bgv99ul1YtXFhbFImEwrt37xofHR2NWTaqA7W1tdyWLVtv
    iURS61ta2gVFUczKujfZWodA1wGeF0VJkkyrAxtep4r4TYw50tvbO1xf39Dr9/tLJyejZwnd
    OY5DOp1De/sQGhurkc3G4XS6MTmZRt5pX5ihhdF1Co9HhCyPqwMD/YlZMSt3iQEWgHy144kT
    p2Gz2bF1qx/Dw2MIhxNTVahOpwNVVWVQVd1Sf3O2k/tbEa8AaF1dXZNbt259pbKyfPPo6KQ4
    sxqPnEOj98ZOq+BdEHhIkmYMDw+plpS8qeObei9vJl6x56NTSnN79+5tnT9//q+WLVvy6T17
    9jvMJs3W606nZbS29pgNm2cwUbM94wwDGByMYXQ0BYCb0cvQmtItKLCB0pB6/PixMKV0gsWq
    UUKI2c5ItsYBYQ5eGgRBoJqmiTZJuuumG65vrF5QBUM3LDt0A5dvWEc2Xram+NkXXp4PYICZ
    1nnZQ9cuxURgbIMBYJJS+vtIJHoU+SaXi3/7+8cb2CQ0q2PiDGCNAli4esXSd/7txx+q2Lhu
    VX4x0LRztNQ5l0HpzK85Avg8boKzLQ+E/A7oomKyks1m9zz99LP3XH/t1QVet3vGIm32fGuo
    q0FtTTWIzY3R4SHcd+f1EAUBmqZOVd+ZV3HsdBcGRiZkS07ZmltWL3Wwyo9lgmQqhe6uLvT2
    9cHhcODFl3aDkHxKwNA1jI1PQNd1PP38LkiigIbaBagoLZoKYBQzmmFfEuqdEGL2ywuNjo3t
    27d//xXr160WZoNcq/WBdR1masazfhZW1ovZmXT3DyGStzBQZwWrzEUyiwYDzyORaPSXP/zR
    T5Zftm51WWVFOWPVpq/NmOpRqJ8NJYh1u0Cnig84AjZ3ZgJKQoCWti78+vEXNE3TwxZwZfXy
    utgikNlgcgYIXbhw0RXt7V1rJycjMxo9AxyGh8Nk+fIFJRUV5UXDwyPWbgbcpdpoWNKZajgc
    ju7ates1lpqs6OvrG7TbbYHh4RFV0zSzf2cUwMiSJUtsGzduun9gYGJNd/cAlzcTFd6A3urc
    rXXyDuscZ7PZTGA1w/+KXkTQeu2118KNjY17amqq14bDCf5cQvd8b8Qo5s8vRjweQTBYgc7O
    EebVNROMcRyB36/T1147nYrHE1NaGLb5yF5agJVvkshxHAoK/HC73ZBlBcXFQfj9fvB8XpMl
    CCIoBUpLA1AUHdmshkQi93opwkuhw5oBUCYnJw6UlBRPOBzOCkXRzslOmc/bfOZnM0DknGMI
    4GC381DVpBaJREzj1CTT8qUvhO21dv0ghMiU0sTevXuf2blz56bq6vnburoGyGzAZ23WPA0A
    yYwU7fTn035a5wOVkiSgqIgYhw4dTiQSiUmGFYYtG8KzqmyFC1lcrIOBEGKa+W3bvHnTA3ff
    dbvEc1zea8os+WbajA++911SW3vHkoGhkRFGqYXYbkM7125jjtpOaCw9EKb5RlEZNgnbGJtk
    7lDj7CEt2bhu9b3/+JmPlS9fsmhGSmP2wjeDzTpPix2OA5wOCZYgZZ4X3ZmeUb5HXnr55Rcf
    +8Pjt977rrsJQKfL39mgHBgYxNGWVpSWz0dp0IemRQ0zdDCUGqAAUqk0OnqHDEXVYlYtzLly
    y5c4TYjBwUH88Mc/haZrEAUB+/YfQDqdQS6XRUlxEQgoCvw+HDt5Goauw+Ww5wEWW0sJgWme
    eK6a6LlMFeqEkKyu64cOHjoSjsdiJR63a+oaeI6DAQO6kQeHuWwWiiJDU9UpRocnBIJAIAp8
    vvSJ4wDDgMHAl2bo6Oobooqq5iw6RXOXPqX/uUBvL8PCYh0+eOjwk7//w+P3f+ShB3krGCIk
    3zzcMChUTZvSxamKAk1Tp/RjHLtnUeRhE3nwbBGklOQLFBmgz2Sz+L/HnqFDoxMJlho0U10j
    bOxdkK7s9eKVec+UUnzyk5+oSySSHzl1qt2naRoEwTZjkUgkMpiczLjXrbus6Zlnnh7MZDJj
    LG2ZZPPgrEViDnsRqmwx4imlOoD0+Ph4GHlDWtHCCMSampocl19+xXu7u4fW9vYOkfwCwb8u
    a3VukGVNGRIAhIiiYFoezE4TXui8prlcTuns7Hy6rq76+qGh0abx8chZVg0AQVlZEIaRg8vl
    xuRkxqIFIrMWPR6RyIB66lRLxMIwzol+78+PrfxHSZLQ2FgPj8cDXTdQVBQAx3FwOBzIZGRQ
    CuRyCrxeJ3heQDyeRb6yfzpFyNoGzW5UONdkwxSAb2tr79q6dXtrYWFBxehoaMazn26uzUMU
    BQiCBEGY9ojLOyhxoJSfwVxZWTCHQ0A6HVXT6Yw1XqXmCPTqALL9/f1DY2Ojv164sG7Z6Ggo
    KMvaDGd38z7yjJoIURSn7iOf+pvWY+U9zGYXn05rtQAORUV2hEJduc7OzhAjY4aR15NPzNaV
    mbFAuJgXZulHuKG5ecmXPvXJT8yfX1kJXVUsC/e0AHbDmpX41Ec/UPgv//FfV4xNTGYopUlC
    iNnt3Jq/tDakxVy8DPb7zCqGEAAvpVTCtHlkBsDiVcub3/f5T3+UgSttBliy9huc+bVxzobR
    1DBAQayeUtaTw9xYBkSj0djPvvu9Hyxf3txUt3zZEhDKwQCFQfOVlM88vwsFhcUYHR7EPe/Y
    CYHnQFnaByTP9HCgGJ2YpKqmZyzgapKdsUu5GzSDq5FfgWEYBurravFP//j34DnmOskBu17e
    g0OHj+LhD70/P6aYMNxqYEvYfePcDNZs8D0X48tksTrb2s909vUPlKxY2oRkJo2xsXH09/ej
    f3AQY+MTiMfjSKczjIXL664IAI4QiGK+mbXf40ZpcQDzSotQXlIIn9sJTdXQNzhqWMBVGtMl
    wRecQrOCRBYcwjlZfvyxP/xp243XXl1TvWA+ZE1FKBRCX18/evsGMDI6gkgkimQqBTknQ1EU
    5t2VB08cIZBEAZIowum0w+9xo6QogHllxZhXVoRCvxc2kcehE614Ye8hmY21CRas5py9mv2e
    DcPAQw89VOp2u//++PFj60dHx8FxomXhnl5o+vomyLJlNVXbt1952csvvxRLJlNxtlkzMN2e
    yLD+rTkcTxkG3jMsNk4AcDLtmAEgV11dbd+yZesn+/pGN/T2DhPrTvz84OncX88GY+x3nStm
    XaweSH/hhRc63/nOd/50+fIlf//KK4c8sqyxBS4fkGw2CRUVhYjFxlBUVImuruEZ5pDm4pn3
    mVLpa6/tSyQSCTNmmRv3OOa4+OBc75kQAkVRcOjQCUtKiYfH48HKlU1oaemCLOtT+iarme10
    6mnq954TWM1VrLL8f72zszOybdu2wyUlwW2jo2E+34PTAa/XA5/PA7fbCbvdDkkSp3RIJsNj
    noYBqCqFolDkchSyDKhqHqjY7QQTE1FZ1/UMG8umzES+kPTgOVgshVKa2rdv3/5bbrntlfnz
    593c0dFH8uPHBo/HA7/fA4/HDafTAZtNZIwij3xPT2IZS3l3mrz/GoUsG8jldGSzBmQZMIy8
    qSgTtsdVVTVbkg1beihmzrU+Chc6uFirDRuAncuWLv3CF/7p88u2bsmX1XMcl2dPKZdf7DkC
    UAIicLjtxquJJHAV//ntH93U3tnjZh4yHNslRTEtnp+riWHqxXQLUxUCYGcNW4F8hVHFgvnz
    Hvjkhx+oXt7ceJbb++xeg+fyv5opejemqtpYumS2P8tcGOCZdOmpEydbfvKlf//qIx/70Ad8
    qVQKfX39CEfCCIXC6BsaxbLlK/HCc88CWg5PPPMibJIAu02C02mH1+VCYYEXx062gVKaYtqO
    SQATbNeemJ1bnkPKGgBgs9mQzWaZyUIevNtttimbBl2nEAQBvMBDUVUQBtrpVLeAaaBlNivm
    OZ6ctR2ZFcAuJnjNAiihkdHR408/+/ymU6dO4ejxk5icnISu6/nFwZKlMvR8WyZKKUSRB0cI
    eI5DIpnC2NgETrV3QtfyzF1FWRA1lWUYGB43LMAqbdGYaHMQfE0tVppS2tpyunX3k888W72k
    cRHZs3cfBgaH8v5cJD+WdU2Hpmv5ezAoCJevJqCGAYMayBICURSQSqcRicTQNziEfZoOg1IE
    C3xYtrgOf3r+FRpPpGIWcGUGqwguUAT7Ro6PfOQj1S6X+x+OHj1598mTp3lzUTcX+Onde97l
    ubV1hG9qqmy+7rob+IMH96Onp1dl793sAXnBffneQMwyTSIT7LmI+VZSlLpcTu76669/YGIi
    dk139yA/M7Vx/grB10sNmj9n9mgzDJ3MYoBnJIIvIkWVef7555+74YYblq9c2Xx7T8+Q4PN5
    4fG4YLPZ4fO5kcvF4fV6EQwGsHSpCF3Px1Jdz5+qmndv1/WsNjQ0HLeAq0nWeiU115IGM1bw
    PI9EIoFPf/rTrGkEYZZAHHh+mpHKV29y4Hk6la4y05rT6auzGKzz+njNBdAyAYphGNlEInGs
    vLwklU7LvqKiAHw+DwRBmAIdZpMRc4zkW5Rhar4QQmCzzRx3qgqkUhQeD0+7umLWDWH6YjeE
    s4kTQkg2FAqPh8Ohp6qr521LJjO+0tIiBIMFEEVxah6b12c2S7emZM17zCdzCCQJcDqJpZCC
    Ipk0IEk8xsdbMyMjoxOW1OAIpv07lXNlEi642bOu60FJkj5w5fbtf/OZTz9S7rDb8J3v/mBm
    OxzDbIuT/1zXdVRVlOKaKzejal5p0fd/+pt3PL/7tepEMvUYgMOEEBumXYNzcwyyDEsqMg2L
    8zKl1OZ02D/67jvfseaKDWvzAl7jHO7sr8NY4VwNoqkBQwdkWbEKYOeE/mVBniLfWihBKX3y
    yaef9R09fuLeKzZeVtS8pBHzystwurUdVQtqcPToEeSyGRw53oJlSxZh57ZNyGSySKVSGJ8M
    4cnnX8bu1w4rjD2YClaYNoTT5nqxs2iZkM1mAcClKErwyaeeQf/AAHK5XF4creswDAO5XBa5
    XA5f+NevTgvDrQ7u+amfTw1Sit6BIYLpwgJrfzJx9kS/SKBlAMhks7nXfvyzX1y7eFHDAkEQ
    OI4jsEkifF4PCnw+BPxe+LxuaJqGH//iUWRzOTzwzlvhsEuIJ5KIxuKIxOKIJ5L5nZSioLtv
    GLtfO0bbewZMoXUCQIIQkprjHbrpaxfNZrMvfu0b/721oa7W1dzUGJQkkSiKDEkSUVJchLLi
    IhQG/PB73XA6HFO9OwkoZFlGMpVCNJZAKBzB2MQkItE4KDVQ6PdBEnn6te/9Qu8fGsthuqik
    n6UHJ2YzQ3Mx3iilWLFihXjVVVdtzGazn9u//+C2zs4e3u/3QxBE1o9QOOdHXSdobx8XFiwo
    bt669erChoae8v37Dzwfi8W6LKmBDOagivBcZfWYrlaULXGDvve9712pqsZ9HR0DDl2n5+gv
    +PoNoGeDKiv4yjMWoIqiGucYIxfNYAHIjY+PT7z88ss/3LTpcqm6uvK64eFJeyKRRiYjo6ws
    gLGxKJqaloDneZSU+DE4GEEikYMk5W0QvF4RJSUu2tZ2IpvJZKJmzGJi45g1PTgXUhPz3RiG
    AU3T4HQ6hYcffri4tLRE2LBhDWNG+KnCAo7jYbPZsHx5wxRmOhdLauIov9+BVMojEULslFIb
    pdROCLGzWKXNVayysqS9vb3tq1evOWi311+uaYY9j7wATdOhKBpkWYeq6tB1oLy8AMXFPvT1
    RRCP52OBzSbCZhMgSRwEAQxwcbDZdCrLk1okEklaYlbKolG6qJS/1XoFQObUqVNHduzYeeyy
    y5YtikZTAV2nknk9sqwim1WRy6lQFB2aRlkPUg4AD0HgIQgCbDYRdrsEh0OCzSaA5/MGyopC
    QWlUJ0TXz5xpmzQMY5jFqwG2IYy+XnbnTQMs5o7u9Pl8n77//vs/8vDHPmqvrCjDgQMHoCgy
    qKGzVM9skJUHWKqmghoGmhvr8cXPfti2flXzum/+4BeBweExg1J6grFZZnCRMbeH1WhQBwBR
    FKEoysp1q1fcftuNV/Ncfuc2Q6yuaxoUVYFNFM6RHqSvq8cyDAOxRAqYadp5wT4y5xBCapTS
    DCFkwjCMRycnQzXbt17+jnfeeSvp7e3DK/sPIxAIoKPtFILBQng9LhT4fbh+x9Z8Ok3XQQ0N
    //ndn9IX9uxPs+AUYdqriGWXbpzn71/QZCGEQBDyPd50XZcANLhczvt2XHXVHdfs3EHcbhdE
    IU/ncmCux4wBopbxZbDxpet6nunSNGiaCkVREPB7uV8+9vS8rt6BGIBxSukIISRopoQppTIh
    RL+Y92FqS9jO/NV4PPHrbC73yHU7r+SaGhtQNa8cPq8bNlEAzxGcam3Hr3//BIqCBSCEoLtv
    ELffcCUaauZD01TIuRySqRRGxyfR3TuIwyfbcOhEm86E4HHLmZoLVnF2wCKE5Cilh8fGJ/7B
    ZrNd7fG47rli02VYtXwpqiormNWCBqfDDo4A6XQav3v8GYyNT+C2G3ZgfkXeyiSdToMyfVks
    Hkdv/xD2H23BS68epmMT4YSqaRMMVA2wYDV6KVLRlFLs2LGDX7FixW2jo2NfPHDgYE1v7wBq
    a2uxaFEDAP4cbM90SiQWy6GvL4re3giJx70VCxYsvPvaa4vmPf30E7+IRmMSY7KIBaBiDkHW
    7E0iKKXYvn275HS67jx5srMmnc6cs1eh02lHvtea9rpg6lysV35e6kYul9Mt8eqCzYbPcT8q
    pTTV1dXVo6rKT668cmdje3vv4nA4gQUL5iEcnoTD4YSmcfm+oyBIJnPo749OaWhsNhsEQaen
    T59KGIYRs2ivopjlOH+xMcuaXgbA33fffaXFxcU3h0KRj3R0dDnT6SysSQqrHsnULJlVaTNP
    M2UlIJPRSHl5ffDKK1F/8uTJyYmJiVFKaZDN8wTLVGgsXs2FhMY4ePDgiM1m/+7ChYsXhULh
    yomJGBKJLGRZZ/eTt5+oqSmFzSYiFErB7ZYQi8kYGkqy6jwRoijAbpfgdkvweiW4XBpeffWV
    xMDAYJStH3FchJ3Mn2N6u7u7R48dO/JVSsnKurpFH41GU4VDQ8OIxdJQVYMVbky/j7KyAGpr
    yzAwEMb4eIr1HhVY82YeoijC6ZRQUGBHcbGDHjzYmmtrawnJsmzGq37GYIUt5MM52fY3DbBU
    VQWA+ssuu+yuz37mM/bi4iA0RcaqlSuwavlSltJRp1I71PK5+W+aqkDXNbicDtx23XZ09/bX
    fOenj24AEGODyBTzXtSucPbNnmuiKYoiupyOm6/fuXVeSVFhXndlEasTAAePnURv/xDuuHHH
    jPY65+o/OIPdogYUVcX4ZNg0pbQ2hH3TaQUrRc0KC0AIoRbT1zHDMCZOt50hL738Cp5/aTct
    CBSRI0eOwO20I5lMon9wGAG/Dz//7R9hEwU4bDY4nTZ09Q6A7TDMNgamQ20agCaK4lRaay46
    5lBKOVVVCwAsttvtO1etWnnTHbff1njn7bfyJcXFDJznQVMeSOmgUx5kOgOGuuVzbeprE9Rr
    qkIuW9Xs/r9Hn1y1+7WjJcNjEwsopQcBnADQzyqNzFTbxaaldUJIIplKnQoWBtL33nOH32GX
    8gJwTZsypm0704UX9+zDTddshyQK+NVjT6G2qhx1VRWsATSPAq8HBV4XltRXweOy4/fP7DH9
    5mLsPMueYe7wCFUJISFK6aH+gcGqrZs3yO999912UeDR29eP//nloxgdn8A12y/HNds3o39g
    CM++/BrAiSgNHsI9t16H1jNd+PGvH4fNZsONV27A8qYGFAV8KAr48PundumpTDaBfNXgEGOw
    zKavScyBsH32sXz58vnhcPjTzz//fM3Y2ARE0YbBwRGMj0dYU2dpqrnzdJNnaarMHsgDrnhc
    xpkzCVtjY3BTY+PigX379snMrNVkmOYk/TG7RcjsY+nS5qpEInPN2FiYm2nFYLbQEVBbOw/h
    cAKTk/HzlKafP2UoSQIUJavnclnTtNM8jQu9n1n3ZHZBiMmyMiyKYqKkJAiv14OamjI6OTlI
    6uoWIZ3OIRDwsL5vLtTVCTAMAsMg4HkBokgMVVWSLF5FAEQppTO6GjCd8EXHrLvuustdXl5e
    IwjCFfF44h3Hjp1Y39LS5giHo1Oaqmnmc/q0fn22Dktgwv38x1DIIVZWLmy45pq6QDodropG
    Q5WhUPhYV1d3ZyqVijCz1xk6povIGlBN0+TBwf7OhQsbh3p6hitDoSQb/8KUuajNJmH+/ELk
    chpGR5Ooqwti3jwfIhGVgRYCXSfIZilyOQ3pNIeKipwRiYSSuq6b8SrOijbmJCtl3RQSQjRN
    09J7977asmjRQtLYuCTc0TFQGI9nIEkSmppqsGRJPbq7h3HqVD8o5bB4cSWuueZyvPjiAYTD
    HeB5AU1NZaisLEZHxxgGBlJIpXTkcjm4XASZTDyTzWbN/olDAB7teKYAABblSURBVIaYVvTP
    6vwuVIPlCwaDLr/flwcjACvj1qdYhXxaR2eeOfl/M9i/mTkyg1LwHIHP4+YAlCFvoRC3TJg5
    bSw8O2Az0FBUWz1/y7qVSzlqdWdnwElRFTz/8j4sW9yQLzlnzucz9VfncnPPb/YSyRQGRyd1
    yyJ+0U1sGbjikK8ysubrFU3Txn/wo593/fAn/+evmFdZcO211/IHDuyHrmkQeB4UwLMvvYI/
    PfsSDN0Ax3MQBYGqqmrqPcxTZqkvgaXuqMWawAoUL3TClBcXF3/18ssv37x+/bpir8fDFQYK
    8Oqrr+VF4EY+NagzVir/kbFUTNxuHWMmqCWM7cpXs+Ur89atXCI21i2oTqXTVS3t3VceOHa6
    IxSJfYNSeobpNWIW3csFCy/ZZOsZGhqZiEQi/tKSIJsf+eo6UIqd2zaBEIrf/+k5aJqGv7n/
    TlyzdcP0UJiyRMhrlgZGxqmsKDlMm/TFmRvyRTfePRdrwnbIWUJIjFJ6srdvIBSJRucVFQYw
    ODSMoydPw+FwMM0JRXlZMdavaMLo+CRWL28CQKHrOmLJNBKjIXT1DmJpYy0IpRgYHkMoEpPZ
    HDe1DJfKlsE6z+dPTk7On5iYnGKnDMNgPe8IBIFjCzfHTh48r4NSHjxPwfPTlVKqSpFOU5vb
    7W4A0Ilpywxzhz6nOsVzgUy73dE8OjqywHSfn81Eeb0uOBw2JJPZs9itPyd0BwicTgmJxJic
    y8mmgaoZty54vLHxNdUwnlLKAzB0XcsCtL+5ubrMZrMVjIwMe7xeHwIBPzPmzEsIamoKQWn+
    ncmyQhVFQTg8oWTz2oIsGzum5koyNa4MXBmzYtabuoe7777bVllZ+cjY2Ni7+/sH5qXTaUmW
    VRQWFqCoqIh5LE0bbJqeS2czVtwMIbyV6TLPTEbnDEMqJqRi27x58zcsWoTxYLDoey+88MJu
    SmmEVXSble8XM9YMAMrAwGCIENrp9bovi0TSlmKC/MdMRsGxY4NYuLAUVVUFiEZz6O1NTKXZ
    ZoN3m41HLpdUU6lUysJexVn8uhRFB1Mt+0ZGRns5Dme8Xk9DIpGDKIqoqChCeXkRBgfHYVYL
    dndP4IUXDqC7ewKGkZ//kiSipMSPcDiJ4eG8E70kCQAUPRwOm4VxYzjbp+91taIXWkU4eOTI
    kY6nnnpq/fp1a8ERCsxmHHQNVGcLpTaT1TJ0FbqmQpFldPb04dmXXzMnsQuAg1IqElN9dumP
    wrLiopKiwoIpVsRc5AgBOrp6EY7EsHZF09RiPtuKgZ5Lf2UY4DiC3oERDIxMKJYgkLMEA3oB
    AVcghKy3221rdN0QFEUxG0dnAfCEEF8wGICqaq5Vq1ZxQwP9+OSH3w9N03D85GkIAo/lSxpR
    4PMgHk9gZHwC/QNDaOvsJf1DIx5N00vY9XkALBEEHpIoSKqmp1RVOwCglxBiNq3OXUQat3Hn
    zp1Xf/Ob3/QCFD/76U9x+MjRGWLwKVUtMVOElq8tPzedztVh6PlUtK5r0Fi6UFM1NNRU4s4b
    t3PJZMr7qS9+a9GLrx5pBpBmTazNRtwXS1erACYi0ehELB5vKA4GkEylkE6lEI/HkUqlkMvl
    4HY585qreAIelxNnuvsgCTycDhvcDjvskgBJ5MFxwGQ4BsOgVoBl1TNcUDXOGwCKCvsbPUMj
    o/2Tk+F5xcEAli9twuc/9SGoiooCvwftnT1QFBnrVy+FrqqQFQVtHd3wuJz48L23QlEU1Mwv
    yydOOKBvaJRmsjlrock40zFdUmF7Op2O+ny+uN/vD8TjSVj12tZqIqsmxqqRsabeHA4RTicx
    OjrGBQCFyFcj20m+dv0tOSilHlXVxWlLhpngqaQkgHjcbIr8+vqrc4Esl0ugZ84MZSilZqWq
    OdcvyKblvvvu8wSDwat0Xa/TNE3L5XKJZDKZSqVSMqXUoygK4ThR4nneHg6HEQyW4/DhTni9
    Lvj9buRyGqLRDCglTCsjQhQ5iKKDX7Kk2Z9IxCvsdrseCBQGi4uL13q9PoHjeCGbTY/t3bt3
    X2dnZ4hVY5pg7E3dR3V1deHExMQNTz/9dE0oFMaiRY2oqKjATHsCcyxxsHornc893JqKzgMy
    bgqEaRowOpolhsE7FixwVtnt9hUsLSVYdMRzUXSkK4qSAtDvdDooITwRRQGSZIPNZpvSu/E8
    D0XRUVbmQyQShd0uwG7nkRf35+0azH2+KHKQ5bSqqlrKjFmMvcpgDosOZrNYlNJMIpGYIIS2
    er2u6wgJc6pqYP/+Npw40QtVNeBy2SEIAjIZBSdO9OP/0961Ncd1ZeVvn30ufVWrJctqSY5j
    Ywd7AhW7qIQkBKqA4EmFYooqXlLDG7xRBfPIIw/8gCl4HHhimCqKhzwMYQwJkCmoMkmGZJKM
    49iWLNmyJFtqSX2/nevi4awtbR237Va3WglTuEpu9dHtrLPX/tZlr/UtQKBQSINIYHFxB6ur
    dXge9kYbOY4Jz6v6rVZbZUnLHAzuYsDGr2EdrPLNmze/+2ff+c6fnlo4ddY0ZVoIYQtAAqSS
    sUIZvyiuRSKKIpXdoiAM4Hl+VN6peNu7VTWepqWlP8fOGM71ZM2t7d3a9vbOs4WzpwESIGXc
    BbC6/gAv/8ovYfZEcY84lFRqTSSonAXtvzcA13Xxb9c+oVqjpZyglmYchz3eKWQy6b/49dde
    u3Lm2WcRBn7YaDbder3eDYOAfvG5c4Xf+LVXrB/+6F3Yto3yw3Xs7Obpvff/y125t9oyDCO8
    t7qe+9Ybv5V59aVL4uTUJIBIrG9smv/5wcel28v3Tniu96Jtm0basaUppeF5Hj6/tRws3V3/
    ewB/Q0Q1jWTz0HJw2j6Ty+XMMAwghMAffvvbBxoiDjZLhDEvFH9tz5nfOzbcf6+OC6NQy3aF
    8ft6o4VKrY5qvRkCKAKY5fWoYUSOH622r1Gt1Xd/8I9vw7FtPHi4iWazhU6nAz8I9sg6gyCu
    Y/rbH7zNET5gSblHbVCamcLp+Rks3V0HG4WWpkOKrG9cJIoq41quVmv3r773H6/95ONPsP7g
    YUzR0GzB9Vy4rgc/CPaydOCxIY4d0zRkMylMTU5g9kQRZ5+Zwxe3lxFGkcr4VLVj6AP1ZEfp
    XAHA1atXV19//fXvf/ObV/6kXN6ejCIYgDBUFsEwpNA+1z9If2+aJjkOws3Nxa3btxe32WBE
    ifrOsf9rNOrLU1MT2/l8bqHX8w4UTsfGMECt1uauKfnErsKkc1UopOF5De/u3buVRPZhqPmw
    2WwWc3Nz3/A876/X1x+ecl2PTNMMbdvuFYszPSLI9fXdyW7XE6VSHpOTRQhhYWIiFQGd7p07
    99qTk0VRLJ6YqFQ8p1xuIwiEME0Tk5M55/z5V87atnjGMGQkhJSAYYShYcRjayYbFy9e+Mul
    paX3iUhqo6EGti+WZYGIHN/301EUwXEcrK7ex9raA41PydQyWOaBbNaj1/Z5mPodJcav8ddM
    U8KyEO3u7qQBlDggrHHd38jOiRAijEcL0dbCwokolXJkTM9gw7LMRH2eQLvtY3Y2i9nZHFSI
    GxfFA54H9HpAKmWiXPb8MAwVXinMGuckEEXb0IiiaKlUmvKDgJx8PoN02oFtx3ViypHdJ02N
    3RRF0xCG2Cts73TiEUzdbstz3V4zUTrzyBD6x2HWKKNyPtrYeFDd2HhwEcBpPuKbwD6BpnhC
    Sk+9qrEfDU69PQBQ4Whj7KSWvV4PAHZuLi3/01997+/O/+ZrL2VNGbfNx8dMAj3XRS6Txo+v
    /Q9nqvjWtSxWxMeCIR+Nhtz1dufeOv3z+x82iGida062eYGGYkXnRWx6nv+961/c2CmXy+cz
    mcy0IUSKKLKEEMad5bvNn12/kSkUp1O3lpb9L7+8UX/333/cIaIaK0jzk8+/MD774suZ0smZ
    2dLJ6QnHtk3DEAIE8n0/CsIgJKIgiiLPdb1Oo9muVRutdQA/RdyJZzFgjVLYcPedd975bHFx
    8TQAmyiyKSIzPvKkeOwPtNEx++/jp5AYM0OJ79Ouk37ddT1aXt0os34pRv2j4CRT+6JXrdX/
    9e0f/ui5iXz+edf1EEURMpkUCszJkrItWKa5R3vg+z56rodWu40H5R24rgfHtmBKgzY2d1oc
    MVUA1LhubCwjQBIF7yGATrVWv/bO1fd+9dTC3LlOtwvPdSFl3CHl2DYcx97bEyqA8jwPjWYb
    D7diFbfj5hC6sbjiakClDHgHR9e63fdoanFx0d/d3f2Hy5cvrdu2c8nzvHnP86ajKMpB6ybW
    dYCVRqkQoigk3/fDer3eW1/f2PY8b42POetcMxodk38lrl79l8U33njj+y+8cO6Pul23QKQ4
    qpghVAjkcqm+rNVxNviRYcN8/A8icr0PP/zJWr1ev8dBb1kIUcNjeH4GyB5iY2NjpVgsfrdY
    nPxt13VPh2E0EeNIzOk1NZWv5fPZXK22axWLxc729k5zaWmxvbKyUu10OlXLstrnzp1LXbhw
    sXT+/PSMEGYGEJJICMPwSQgjBKKQyPeFgBdFYavTcSubm5WlTz/97CFiBv7usPv8+vXrtXPn
    zn305ptv5lzXs4jIJiKLCIyB+vM88DkdfPY6PUByXfZeuYCdIEQAonb7/v37bewz6uvd70fi
    nOzs7HxeKs29OzmZvyKltKSMB2n7foggiPhzPlXQGPRN00A6LWGa8b0HQYRqteyWy5sVzRl5
    pJzhqAKoBF4RgKBSqdwqleY/tKzSq1JK2zTjuDkMI+4AjZBkb5dSwrZV0CGgRjD5fitYW9tq
    RBFV+gSDAw2qFkOClg2gAOAEEc1yJmAm4WD1+xvUxyApVmI1LmOLW22rAHp01OHso/VkaQC/
    AOAP0innRSHENMuWFkJInj3Muan9wQYUj0lgy6KgmAefsB/g+b4bRaRa0VcALAkh7iDuRNjl
    OqfD3rMkogyAkwAWOLI5yUd6FoCsYRi/f+bMmfNra2uf+b7/gG+5zccyVe1YL8Oy5vhnI+zT
    WOjMu55Wa1Lle9/lWqD2YddI1fEB+GV+9qdYllm+p0GOh2mAa/3eK1Lb+4iH994TQtznc/XW
    qBwzROQAmJXS+D2K6HcionkA06mUk8vnsnYmlZKObRmmaQrDEAIgEYYRgiCgbs9FvdGKmu1O
    JIQILVP2fD9YJWCZ631uafpTRdy5eOR7QgMOB8DcqYW5t3o994/rjUZRCJGePXnSKp2cMYrF
    gpFNp2CZUggBRGHINA1tVKo12trepd1qgxzHitIpx92tNlajKFoCcBvATSHEIq/BLuJu2HHt
    c4eIVMZyXts3hQReiafolMKrNvbZ5xV/1w7iztSxZrKEEIKI8oVC4Znnn//GK0IYF1y3N+f7
    wQxRlAWEjIecC2iv4H6dA+5jPNIoDkDiOCSiSqXa2dra2kFczHsPwIoQ4h7L2jyszvGesADk
    Fxbm51Kp9LyUsmQYxrRhiCwAc2pqqvTiiy+91W63rQ8++ODjmzdvtsIwVGSuijTUNwxD5vP5
    XCaTKUhpZAEYPMC7S0TtMAybYRi2Pc/rdTpd1/O8Nv9sFcAO1y/VGXcHEoL5HlOXL19+zrKs
    i61W65Tr9k75flAKw7DwaOabZ0kkOP61/9UVZTUSfgcd+G7fD9qNRmOdiFY1vNpgYx+Nom58
    kiABTL788ssvXbhw4S3DMM4KIWaIUIgiOIAwAcMQwjBi7TOEctKFABkGkRBRFAR+2O26waef
    /nRraWnxThiGd3ifL7L+bCm7My7MIiKZSqWKL7xw+dVLly7/eT6fPWcYRsZ1PbvbdU3PC0UY
    kiASQnUUSik5UyjJsgyybVAURdTpuOFHH/335s2bN5Zd170N4Bbj7zLXYnUHkUUMKYzBmYw8
    1yAU+PMM+o+AEU8weOpMucnReUNPSY8Tr1gWB8AUEZUAPMPGfp6PkGwcfsxKMkPXxj6Z4jpv
    jvIwYKUcLMS1agU2GlMAJvma5DX4XY7abmhGoclKXuGuQzCTfZY/UryuqpZIL8ZX15q8sVWx
    ZWNYoyKEyAKYJqKTnP2c5ZqW9ChBzYDrohz6MoBNPlevMFCPqlMW74dZ1qVnWa+U86iiUHmg
    6Ofg/UVadrfGmd01xJ2Pii+qxU0I49wfFuvZMwAusDN8hmXJabIYCaeYEnKoBopd3gd32VCs
    sk42RjUWT9njpgomiGiK97baM+YhHHq1p9WQ+ipTmSjaDG/M/pUio0yxHGrvzHOAm39KgDvI
    3gh4PbYRD7F9yGtUZZweZrvbrC+T/Oyn2LlNAZDz8/OzV65c+dbKysrmtWvX1pgOwcM+g32d
    h6oru6MwS+Gz2iuqEF/pnasFhQqzmrxONKj+ME5O8TOe0xIKOYxGGE0DXPcYm8qcfFAnIa0h
    1yIpn0FEWZZnAcBp0zRPm6Y5L6UsSGlYhmGYQhiSvSuBuElBEEUURRGFYURhGIZBEAS+77ew
    3yGs80XVENNzjHNvGESUtSyrNDMz85yU8ryU8qxpmgtSyknTNB0ppTQMQ6pEoiICD8MwCoKQ
    fN8LPc8Ler2eV6/X677vP2Q57jL+bvB6+ON0sMDAarGSq6GgdiLlPoiDpYj0PN4Q+kbBMThY
    Jm/WIhFNs5GfYrCyH+NciQE2BmnZuRpnfHZ4cfYK5IZwsHSQyRPRBN9rmmWxOZOY4rVQ3W11
    LiiuYr9bzkbcVJDln7c0wEgaSR/xmBD9XL09ZBZO/W3dQdczaWJIYBo0y+Vin19GcbS0RwUA
    rVMqzfJMM5fNCTYsao10p8Tos0dIc2pbHMVv8/rtMeuPO7vL95bhe5/RstUntLXS5RB9nrXq
    NPVYb1TmRxW410Yw3IPKIfi5O7wGGX51NLwaROf0soYDHbd8jY7BwQI/d5V9nuRAa4Kv9ZNH
    DLhHCPvDxGt8NFhTgdQwmKxhbJp1RmFWTmGUECJl23bBdV2lT6qTtaIFoz5/zQGQ5Sx+KuFQ
    6muk1qmVwKzuYfc51zxlE1iV53sZNts+6Peowd46VjWPIhuk7Q2HiPJs9xReTSeCKH28m+jz
    zEkLpKoaXqnTqO44Mu59ZLF5jU4wVs1yIKLbxH5yJLE30ALxzUQg3h5UllEcLP13PO6GBzWC
    umAHlO0YHCw1pkePjFRGx3pC9koMAFYqitLHm3QwwjwmzYibj3FuLex3Y5oKaJj5u6KBlZqP
    qP+OZD2SbiADzRE+EC0OKYOh/e2kDIcxCsM4WKEmhwuNw+iIAMvU9CmX0Cn1jPUM1pMcrC72
    i0XVNHr3mLK7Qsv+5NmwFNhApodwsJShqKFP8fQYHay9UxHt2csn3PeTdCnSgg6dz47GjVeJ
    vaMcRn1KQTLAfRpm0WMcSBcHO55dXiMacq+LxwTkFmKqGYf3h6MyUlzXprJPqv7LSOCe1ScD
    mcycegnMCg8jh3b/lnbfCrPMMWMVcJDJX09CjMxSn0iWOLzPFWblEnt8EAdLNcf0K3AfGV8P
    IUuK7185wxOcRHAO6WB1NcdWDzQGDqbEEYDW2P4dE1jpwGtqzosc0NgPEvHqZH0jAXLi2feb
    E2ZpYGskHIpkQfHjZo31OzpIfgy9TpoMT/v7I6tQH31/okxHBFhC0ydTMwQmHj0aFE/RHT/x
    oY5A6BgcrKRxdBLGTT4luEpGt7pD62mGYmzOyXFh1XHg1WN0TPRxcsUI+0V3ih9hcB8Rr5J7
    3khkF5WzpB+RJ2c9Hitm/bxi1WNsoNSwykpk2/sFg/0CVz0Q9xM2D8fgYAktaZLELJnI8D4N
    e70+mHWoYn3xVYPZcYDS1wF8j0r2r/qej0KOr6MMR6GPxy3XMTlYR4kd9DUIpv7POlf/j1lf
    jRxfV7w6Ygfr5x2vhsEsGlWe/wVGRVPV+Oe5vgAAAABJRU5ErkJggg==
  }
  set pieceImageData(Fantasy,55) {
    iVBORw0KGgoAAAANSUhEUgAAApQAAAA2CAYAAAB5hAnwAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAAKUAAAANgCNpmkAAACAAElEQVR42uy9Z3gd
    13U1vM6U2zt6bwQBAuy9q5CiRFG9d9lqLons2I79JnHcnThxeuIkfuPETmzHlizLsnqxCiWx
    9wKAAEH0fnF7n3q+H/cMMABJSRZByX4/DZ95Lohy78ycc/ZZe+291yb46Pjo+Oj46Pg9OQgh
    7/o7lNKPHtTv2JhczONij/d7vb+P5t1Hx+/bmpzrOSt89Og/MiwfHR8d/y8BFuP3P1qHH96Q
    vcv/3xduPM/XH+Q9kVn3Y1yH/tGQf3R8ZCs/ApQXbVA/2sw+On4fjMrvwzyllJrviZwLqLDf
    oWaw8dE6/PCG7OwhpB86kzkH9/TRZPr/Mfj6fbOVlFIyfatktp28KI7ZR4Dyt5hsbDAIIYSy
    wSKzDOWU0fl928w+KGP/u/BM/l9lmt9pzv0+b+jsvoyFxwOwARDNRpEQogFQTedHAODDBf7m
    DY1jp3kT0/F7zOyZ15lxz+/E9ryftfdh2p//P0XjZt/D77njY6w7rqioyBoIBKwTE+MkFotT
    Bix1Zh8VABo752wQPwKU79E4GiifUspRSosYO5Jg0J8y42hsavq7Tczf1YXIcRx0PX/5s8Ey
    IQQWiwWyLBNKqfnn73nz/jCB9vsNB/y+GE/T9RFCCHRd59ga52bNU2Ou/t5s6GzetIiieLvN
    ZqsDYFEUJSFJ0kFK6T4AKQAZdmaZwfzo+IDnnrGZAcD27ds9ixYtuiwWi43/4Ac/OMWcAsps
    pMzGSP8t5/bvGuAg59grzH9gDpWf11aey+H7MGzl/+s28hzP1hgfntlKYzxn20n6+3JPH//4
    x33FxcW3apq2WtM0D6VUzmZzvbt2vf1qe3v7OCEkCyDN7CRl9/gRoPwgWBHGhPAAdLYpFzsc
    jr/mOM6ZSqW+C6Cf/YnKBijzXgDW71qeF7sOTtM0B9sQVAOEUEp1ABqlVMzlciVsM0iawKQG
    QCGEqGxB8uzV2EBmLMqLYSjfzRBSSs8XOiXnMOznDAm8l8/4MAy78bmmOWWhlM4DsIIQMp9S
    epxSupcZSYmdud9FQHmu++Z5HqqqghCy46abbvqzO++4g+M5YPfu3fjev31/WSqVmqSUjgGI
    snuU2XqksxizjwzbRbIdlZWV/LXXXluTy+W4xx57LJjNZml1dfWmVCr174IgjF1//fVf+fWv
    f32K5o2CzByANGaxyedj0n9XIj6m6+C3bLnM2dnZpY+MjOqGraOUagC066+/1llUVFzY1taW
    3Ldvf86wK5RSMztkMLecKV1Dv5gA5rcsajtneonJ7n+oNvL92knzfnf//feVFBQULOd5YbGi
    KOrzzz//dHd3d5qNkWEn5d9VjDL7/nieR1lZWV00Gv3Tffv21abTWbjdPqxcuSxVXl6eaW9v
    30MpjbO9XTMc77lyDD4ClO86ZuQmQshGXddfpJQeBbB09erVW1evXl34/e9/X08kEv+IPFMp
    A4izBZZjr4b3o5uM5gxj8TsUGieEkI0et+shTdelVCr9E0ppLyGEEkIkSilsNutDleWlt8Vi
    8eFQJPZ3AAbY/UiUUhuldAEAP4AxAP2EEIVtGDm2OJW58IbeyYhwHGdmS2ZsUMziEwBWABbm
    LAjMqFMGis1Ms3IOD5Wcz9ifL03igzIslFIIggBVVVfUVFd+b92KJc0n2jutHad7/hZAN9vM
    kibQ9Xtx6LpurJNYbW2tdu21OziqKjh1qoNKUi4MoBDT4ZssgPTvCuPz/5dj8+bNTq/X+02b
    zdby4IMP/u8TTzzxgq7rKwYGBgoymUxxa2vrZyoqKv5pZGQkTinNsLXHsXk4ZStNITlzOI7+
    rjjhhBBce+21lubm5psAcvOSJcv79u7d89O9e/dGmJ3IXX/99e7585u+JEn6xnXrNu7PZrP/
    dvz4iRgDm7mbbrrJX19f36KqqnVgYGDgueeeG1UURWN7iGSylfr7nbPnAwjvFo430hQ8Ho+l
    vLzMSikssiwJ2WyWSySSNJPJqOw+dEYgKIxsMIdOySzQedFs5PkckPOtf/PPH3nkYY/fH/j2
    5GTqRp6nPq9X3K8o8n4AMUpphhCSZPdHftcYynPdM8dx0DQNuVwuLctypre3H8lkBpWVNZBl
    Kdrd3S0AKAfAUUolQkjq3fa0jwDlHB2MFeHsdvulO3bs+MTAwMC1Bw4c+GcAvN1ud9jtdqxa
    teqKN954Y0TX9dcopTIzjobnqQOIAggzYCWbTuViM3bv47A47PaHHrrv9nuHR0bx+FMvJAD8
    D6VUJ4SkAVgL/P6bH33kvmVPP/9Ky+tv73+DUhpn9+Hz+XyfWr169ZU+n8/Z09Mzfvz48SdU
    VX2aEJIAkACQZIyENNfMGFtchFJKNE0zwrzGBkWZ8dMopV5CyG0Wi9jMcbyH44gVFJyma7qu
    05yqqiFd198ghPTCFD6llCqMpSYma8mZ2NmzNr+5YH0IIUQURaooCoeZeWiYxWTQWRtG85Vb
    Lll4zdZN4pETbQMAuhnoyrHrzLxfL/9ib+imPOWpDY5Satz7aOepU8lYJBJIxqN4+pnn0oqi
    BgGUsT+XKaVZAFlTaFWf67H56Dj7sFqtYjqdrt63b9+yurq6hu3bt1coilJit9s5m82GZDK5
    ae3ataPPP//8W/Pnz7fPnz/fFQgEXBaL1UoI5LGxsbZnn322TZKkDACJheVyJpv6O8OmNzXN
    L9A0fL6ra2xVZWVBura2pn/v3r1vsUhOorCwoD6dlm/q7p4sbW4uKwwEAq8AOANAu/baa+vK
    y8v/aGhoaBWlVCguLu698cYbf/SrX/1ql6qqKWYvk2yNyu93zp5njRLTaTjShP2uDkDbtGmT
    uGHD+h2UkrWapvk0TbPpus5TSimluqRpWnxwcHDfc889d4ytNSN0Kq9YsZyEwyGuv39wRuoD
    AI2Bz3Oyrxey/5kiiYZTYtgKcj47adgWl8sZSKWkNe3tw/7586vkVGri4ODgkJ3Zep7NPzIX
    jOhFspPme+eYnST79u1LL1++YrSwsKhFkibQ2FivR6OR4f7+fiezlRqAHAPNRvrJnIT3Lxqg
    /H1LQp59FBcXY3R0FLqup7Zs2UIXLVpU/eijj37uyJEjnVVVVWI0GoXP53M0NDTc1N3dTQCU
    AJhXXl7ua2iot1tFgZ7q7OofGR37BaX0EPLhuDRbgAawOivf8sN4BmxiqpIsd46NjWsjo+MJ
    ACMAvMz7JACSsUTiwGs7d6080zvQA2AYgA+AYLPZ7t+wYcOtLpdLD4fD2RUrVlSJovjQvn37
    kpTS3YQQkS3MGTmmF3rNLEfQSin1AigCUAWgAkApY4vfQD60JjPj0Lqgaf6XH3nwvoqCgB9c
    /j2gqgrS6TSe+NWz6pu791kopS8wox4nhMQAVIqi8HEAkqKoPZTSAQAThuEnhBgMrHweVvP9
    Gg2fLMub2PuMAxgzGWYzk6EZoFrTNBuA0onxCfLD/31C7u4d3MWeeTF7JmlmLM9iN2czFedj
    GT6AuchRSssB1DEmOcc2LUtvX28sNBkM7Nu3jx45ejTOGPGVmA5PHWHjHmF/k2Pn72xO5f8L
    VfnDw8NKa2vrRCQSQW9vr+eqq666v7S0NHX8+HHi9/thtVqtbrf7mi1bthTU1dU1eTye0kgk
    4hwZGSWapqOkpHjsrrvueu7gwQOvtbW1D1JKEwxcpTCdE6vDlMbwYTwLXdcxMDCQqq6uO+12
    O1dxHB3t7u6OsXmoANCPHTs2sWnT5W1VVYWlqpo7efz48RQAX21trbuqqurzY2Njl3AclxUE
    QRsaGmopLy//dGtra+r48eOdJmdYMzG1FzS9AIgbNmzwlJeXF/r9/gqr1VLF83yZoqh4++23
    Xz9x4kSQfVZOEHi/JKmf7+joWRuPpwAQEMKD43gIgoDKymIUFhZVM5IgbpwrV660bt269SFd
    px5JyvVOTgb7Ojs7x7q6TsfT6XSGUppjjoJBqqiYo9D+2rVr+Y0bN64iBBWxWDR08ODh4WPH
    jqVYeoViWv+qqaCWUxS1VBBsjpqaEmq3a6eeffaNbl3Xi5mNlDFd+Ifz7dEs6mVOk/pAIiIG
    mLzhhhvctbW1TZRSbyKRUEdHR9O9vb2CosihQKCQptMyqagolU6cOKrU19cvKikpUYqLiyRR
    FEcPHjz0wsDAQB9z3mY7cL8bgPJ8GxTOrUVG323APiwDH4lEAIDmcrm+M2fOKA8//LDlS1/6
    UvnnPve54gULFgh1dXX4y7/8S2zZsqUkk8ncPDIyIgOwyrIsioJgn9dQb1m7annxnr375u8/
    dGxfOpN5iVJ6khASAhBjC9JsKC/6JHyn8eI4TldVtfPtvQdHUplskm3IXtP1pbPZXO+hoye7
    IrHEEBtPPyFkVWtr6zXpdFp58cUXn9B1vbeuru6+LVu21B0/fnxDNpvtZuBEM236yhwY9noA
    l1gslqUFgUBzSWlJbWVFRaCmptot8Lz488d/MRoMThZTSsPsMycBeCsrym233HAdfF4PdF0D
    1TWA6lAUBcdOtHNv7t63FkCAbWRBSmk/gLq1K5Z+fPuWTdbu3n65s7s3OjA0OhIMR0+rqtpG
    KW0D0Mc2wDRjFwyH4UIMRkUg4P+b5YtbSwYGh4Jn+gZ/SCl9gRnJFAO0hmcJSqlTFIVPNzc1
    fSorq/yJk+05ZsjLMJ2AHTGzrcY6ZYaRY0ZUJ4SYq3JnsHwXc46ysE3A5/N9d/ny5ZfabDYS
    i8WkcDicnJiYSGUz2UAsGsax48dht9kDtdVVmyvLSsTqyjKBIyBPv/ja0mAoMk4p7UE+FcVg
    0dXfJYbyXLbO+JqlLUy9mu3o71pOqDEX3n77bXnRokVniouL6cjICDl58mSA5/lAJBJBQUEB
    vF4vTpw4EZg/f/5luq6nwuHwZC6XG3U6HXZKScHAwGiZz+f7+OWXX3FJa+vCV3bu3PnKxMTE
    CCGEN33cDMbuw7CVlFL84hdP5D73uc+3l5R4QpKUGjp9ultitlICoJw+fTqxYcOm0y6XWBuJ
    RMaj0ZjNZrO51q9ff00ymVzn9XrDiUTix/F4nAQCgfuj0Wh1bW3tquPHj0+abGXOFFF4X8cn
    PvGJIq/XewmAFbIst2Qy2fp0Ol00Nhb0SJJkaWioyyxfvryyqqqqhxBIsizHJEnKZjJZT2/v
    MKLRJDhOAMfxIIQHIRxEUcT8+cXN11133V08z6cAPZzJZAYdDiefTiv3Dw7GS2w2UfV4ShLr
    15eNrVmz7kw8Hm0fHBxsO3r02Ol0Om0QKxl2fxe8LltaFvBWq/XBcDh5B8e545deevmuXC73
    L52dnVFm95LMXmqUUupyucgnP/nJTYqifisej1cWFTkRi4Xi4XDEh7yCRJT9vpGWYWYEAYCr
    qakWBwYGzYU9ZjZU+yBsTU1NDWlqarojGo3+WTwet4uiqFZWVqZramqiDoezwGZzIBAogMfj
    EZYsWblU14moqkRUVcIVFjqzra2ZzMDAgMRyKuPsbS/IiREuhpE0NgYWrhIwS+aDeQnUnIOB
    37EQsCRJxnX0nTp1KvH0008XhsNh8k//9E9iS0sLdu7cierqatx2261k+1VXFj/xxC/VPXv3
    JPr6+odff2PnxOtv7HRvu3zzkuu2b/VNhkJXnew4vQTADymlr7L7102J2DMS0z9IYG1aKByA
    +MRkZIIxf0UmI84wnO5KZSVeUTWNgS6H1Wq9SRAE7/Hjx/fruv46ACEajYYdDked3W4vyWaz
    RQBUSqnCcjbiuMCcDfZc7rj88su/euutt1jnNzaiuqoSAb8fDocNZ7rP4O1du8pXLl34BbvN
    SnM5iR472REZGZt4TdM0oqoqVFWFrqugug5Qnf1fI1dsWrVqXm3lKkmW9WQyrQ+OTKQPt52W
    qipKxftuvx6KLFti8UTJwNBwSVvnmeV7Dx2TXt918Eg2J/0jpXSAEBI1GZf3bVjYmIRtVsvI
    x+68aX5PT4/3m3/77ys0Sg9RSiUAggmkE47jqKZpq9avW/f57/7lN0vKiwP4xS+fdH7jO3+/
    KZFMKQCCAFKEEIvp+RspAUYYbAOAGwC8wkCyzj5DNjk/2sWejwCqli5deul//ud/lhYUFCCb
    zWJyMoihgX5kUklUV1bg3jtuIdu3bLYXF/jgcdlg4QmCwSCOnewoDYYijWzDEo3wDrv+3zlm
    0sjzNcwmAE5VVePVYD70c4ULfxfspPH5uVyOZrPZ/uLiYmn16tW2WCyGX//615AkCTabDRMT
    EzhzpgdnzvQ4ampq7XV1NR6n0zk+Pj52GoClqqp2pSSpFkJQXlRUtL2iomJsYmIiZ+RyMAeH
    YGa60AfKVlosFkiSBAB8Op0Ker2+cDota6IolDBHMgdADQQKbJqmilYrz2cyaQAomD9/fnVR
    UdGOZDJJFEV586mnntpfWFhYeeWVV6ZyuZzfbrdXAChgYVmJOafchVxvYWHh+nA4/H+PHz8R
    mJwMIZFIIpvNQVV1iKIVJSUlLqvV9aDfb9E5jqMejz0Xi02+rKqKheM4TJ8EhHAghAPPc+B5
    b5PP55xPCEcFgdPKyojkciGZSuX8AwMRaBoEq9UScDqtAa/X3ur3l+5YsaK8P5VK/92xY8dO
    MqeWmyvwdfToUbW4uOR0KJSwjo9L5UuXlq+2WMRqADylNM3WkMGM4u677/Zrmv75ffuObBwY
    CKKkpBwrVsxf1tDQMHHq1Kk+xmTamONtLhgjAHDnnXeW1NXVfjKXy0289NLLr3R0dGRmsaHS
    xXZgKaVYuHChVVXVLbt27arp7u4Gz/Ow2ezwev3w+QoQjaagKBRvv31c1HVOVBQOhIjgeRsW
    LCiz22z2ZgAdLMoHxh5nLuS6hLk2jkboTdf1AgAreZ5f43I6awlHrJIkpbLZXC+l9CSAQUzn
    X2RNzJX2YRt506BxAMRkMqmrqorm5mZcdtllIISgoaEB995zD+x2G3iew+WXXSp0dnYGnnnm
    Gd+Tv3qqrONU5+CrO3eNHjxy3JlMpSYZK1kOYAWltI/dv9XEVMpzFQ7+bceMGcpyt9v9wP33
    3bsMAH7y0//l4/H4qyz0bSOEXHPpJZvu/Nj99xU8/otflr/w4ktZAB2SJJ05cOBAH6X0LbaA
    ltTV1dXquo5MJiMD8LBJmmbhS26OLj+xcuVK/hOPPAKAQtc06JoKjgCJZBINtdXcV7/4qM3t
    ciCVSuGLX/0ONzI2Ua9qmkgNG0FnbYyUkkvXLSfXb9sIDuAIoTje3m35g6/9EzRNg67r4HkO
    BX4PCvwLsGLxAixeMM968FhHaTYn1QBQTAszdyHAmW2Q0bHxiSf/6yePr3LarU4tL+dUxt5b
    ZHMmA0DWNE0DULuwtSXQ2jwfVMli3cqlJODzFiaSKR+bfwSzJIMopSIAJ4B5VZXl39p2+ebN
    h44cu+FEe9f3KKVvs89KzvLAL/buLVitVs7v98Pj8eRPtxMFPg+gKbBaBVitIgoLfLCIPKiu
    QdV1JJNJpNJZsI25hI1HkhAyJQfyu3QweS7+7rvvLiwqKmwASLmqKvZ0Op0dHBwMHjx4MJhI
    JDMAZJZWYc7T/tBvxpznumzZMofNZmtJpVLCiRMnkMlkIMt5X7SrqwsAAcfxkGUFsdhxcvp0
    t626urp2wYKmclEUJiYnx8eSyURmbIxPut1utbq6em1VVVVtKDTZduTI0WPZbHaSsSiZD2uv
    YGASn/rUJ1t5Xvz4oUOH5wcCgeorr7xKfOWVV94KhUKhkpIS95VXXrltfHxyR3//oGPRogWu
    VatWyv39A5Hh4eETiqJE9+7d+6bdbnevXbt2VTabLbRYLHo8HteQTyGS2J4gXqitpJSmwuEI
    PXz4CGRZAc9Ps40OhwBdBzl6tNeaSGQhCCJaWqpsDgeppxTO/NgSsO0chMCIZCCR0MjAgEwA
    HoTwvN0uWFpaLG6eV6bmhKLoiMclJBIqwmFZWLjQUyRJuRoAEUopx8ikHPK5fBc0LkePHsP8
    +fN/U1FRfafNJi3lOF2JxWIBtkYSzGYZGEPzeNyeRCLdMDo6AVUF4vEcMhnN4ff7SpAvKDWD
    Xd1gJbds2eIoKAh4KyoqPj08HP4CIbx66aWXr04kEv8zPDwcZXMz/kGxlKqqcrquC5IkQVEU
    KIoCWVahqgSZjApN4+BwuCFJGjiOZ2yzBaJohcUi0MnJpAP59DDNdO0XNOeEuTQsDKAIANb7
    fL5H169bc+m6NasDtTXVnEXkEQwGcfxEm/TWrj1Dp8/0vkIpfZkQMmYKAU95eb8D3jcBUOLz
    +T526623Ft50003geR6UUui6BosoQBTym5mmqrBaLFi6ZDEWtjRz112zo+A/f/gj/+NPPBmL
    xmKjAKw8zzeWl5c1+33e7aOjYyOhcORxSuk+Qoid3buho6fgA9S94jgOkiR5rFbbH9133723
    fP1rXxUEnkNlRfmSv/uHf/RMToZeI4QEtl5+6ZV/9Rffci1atBCN8xocQ0PDl59sa5MZk5UG
    AJ7nr1m2bNll69evL3j11VfjuVyuB9OJ0nOmFsvzPDRNGx4cHMxms1m3xSKC6nqe7QEgigKG
    R8fxj//+XxB4HjlJxtDIuKWupnrpsiULHXabbXpuMYaI4wjqaqrx2ptv4fipHrgcVjhtVkRi
    CeRyMrwe9xQ6pJSC6joIAWLxBHKSRNjCTJlCLPG5IEU4jvdH4mlSXFpOmpsaN3ed7pYpxWG2
    djPsM3OEEJ1S2nPg4MHxt3ftqqosLcTzL7+GYCgcNXnMM7xnSqkbwB3FRYU7ZFmpW7igqeWy
    jWsQnJioazt1+l5No+OU0mFWeZv9AFm+aE9Pz/hTTz1V7HK50NNzBm1tJ5Xenl5tQVOj5ZYb
    r+W+9Z2/USORqOJ2Ojif18UFvB4ulkjQ/qERIx/MxhwYfi7n3lw4cIbDcN999xWVl5ffkMtJ
    N8diidZUKuPTdcrZ7TZl3rwF442N8492dLS/smvX7g5d1xOs6nRGTuGHZSfNzGBRURG3bdu2
    7SMjI3cdOXJEiMVis36PAyEEHMcxpotAlmWcOdODsbFxS3NzU1VdXU06k8lM1tfXVxQWFrrD
    4QgXicT1hQuXTgqC+H/ffPPNPZRSJ4CYKbXkAynYMWnz4jOf+Uy1KFq+smfP3vWHDx8jLpfL
    vn792o3btm3z7t69e/f69etbgsHopj17DorJZBKaphctW7boypGR0Zd+9atf7aeUpurq6gJX
    XXXVJlVV18bjcXtBQcHIvn37RjBdDDInTnc8Hp90u11Rm81WoKraFCAkxJCY4dDcXAFZVsFx
    PHw+BxEER2soFLfmcjIMUGmW0kylcqipUdDYaIGu89D1PItpsQDptArzVDQ+TxA4qKqsh0Lh
    QkaqSCYbORdrk9hsNj8hmpXSJJJJS9XmzZfe/uKLL7waDodH2DwxCp3U3t6+RHV1TXtjY8OC
    gYEwKS0thtMpZMbHJxJsPsmMJVYopXpZWRl/1113reU4/uOaps3XNCwaGYlYPR6vtbi46KbW
    1pa+4eHhXZRSnjl+mQ/C5pw4cUJduHBh77x588DzPDweDwoKijSv16vY7Q7h+PFuoaamXPf5
    nHJ++gocpRzHcSKxWvVET09vDNOKJ8Jc7NNzGvJm+oPbW1tavvuZz/xh87VXXwW/1wNAB9UU
    aKqCXCZjbWvvmPdv//nf1c+88EqjJMk/R15iZhLTxQJZsEKDD9pYGlWmlFKrIAgfu+OOO7bf
    e++9XB5M6qA6zb9SnYGR/PdAdehUBwGweNFCfOdb3+BWr1oR+O7f/YOzq6s7ywyS/YF77rCG
    QqHSv/qH72mqqrkopT2EkNMsDBA1GUr1Yif1MpZEAHDHVVdte/BLf/zHFp/XC11X8QeffIRY
    LWLdN779lzfUVFXZv/GVP3W1NDVCkXJYunghPveZP/B/4Ut/uj4aiykAUk6nc82VV1652O/3
    i88999xEb2/vPgB9mC5CMjbBubqhyYmJiXQymXQXFgRAYWzUOurr6vD5P/os0ukURFGERRRw
    1113csVFhe7SkiI47Pb8uAGghILqAAcBH7v7Vlxx2UYMDY9iMhxGKplCqSJj0eIlWLm0FSDI
    j3/+k0AAjE5MIpuVeAYoYwDCDMzwvw0bPvuw2+3IZDL1SxYvuv8f/+6vnQsa67B3z+7iz37p
    q1v7BoaSzHOOE0IizFiqANoPHT76w088+oUvetxOsX9gaDCTzbWZgKSRl6XmQTR307atW/76
    kQfv82WSCfzgRz/Go1/6+ngylerSNL2bMZcuxvJxH4SRZPN9sre396df+MIXtmuaxieTSZlJ
    HgUI1Rf39rbajhw7mZAkaRTT1Z2GDmyfaa7NWeL/XIaHAeDRRx+tt9nsXztzpvfmjo7TzlAo
    Bk2jIESE1WqDx+Px1tZWNLW0LFnv8XifePnll3cqihIkhITZRmwUm+kftvN9zz33zE+n05/f
    vXt3YTQanQUopkEkIRwDlWTqe7mcjBMn2hEOR50rVy61pdPp+MDAwEh5eYV/dDToUVVa1tq6
    8La6urqVp0+ffn3Pnj2HKKUWVjGWYvNav5ihbwNQ3nbbbS6Hw/GZo0ePXX306HFCKUUqlcHu
    3QeEdetWLb7mmmvLwuF44ODBo2Iup0IQrOjtHUJxcVHlmjVr1j7zzDP7Kioqii+55JLN0Wh0
    Hs/ztLCwsG///v17xsbGxmfZygsOmQ4NDcUrKiqiTqcTqVQaZuldSZJx4MARuN0u6DqFrgOq
    qpFcTnFlMjnIcp7VohQMgObHc2QkhFgsDbfbAbvdBqvVAkEQMDlJkEhIbA5zM8CozSYgl0vT
    dDpVhHxxYASA/b3YyHezkwCwdetWa2Fh0d0HDx5rPnVqAB5PoW3t2oVrFy5cmHzzzTf3srB3
    hK2b3JNPPpm+7bZb/2vp0gWLFizQmmQ5F25rO9be398fYnbDsJMyAO3222+vl2Xlb48dO74m
    kZDR2FiHRYtqssHg+Kljxw73dXZ2SiZ2mcfZYvYXZf8eGxvTR0ZGXpo3b15jRUVFIJVKKcHg
    hHLiRJ+wfPmKVp/PW+py2dW9e98eDIcjOafTwblcbt1ms8mJRDzY19c3genOYnNSSDrXIe/W
    xsbGb/7FX3yr+eqrtkHXVGiKAl1TQDUVuqaC5zksXdSCb/7ZFyw8x132i6eek3Rdf5pSKrKJ
    o32YGwHzpKmmacuXL1/+4KOPPmr1er15D5UyFozqgG76mk6DTEopoOmw2Sy449abUFNVaf3K
    N77N7d13YHhoaDh05Njx+pt2XMFdsXntsjN9AwvP9A+/Tin9Iftsc+s4gotUAGEqxIGmaSsX
    Llz4hS998YveysoKqIoMqusQRREfv/9eEo1Gi44ePYaSogKoqgwKAo5SXLvjSuzavafyRz/+
    6SIAytatW5c4HA48/vjjx1Op1D4AIQZ0ouxMYDrJ/EKrnwmAZGhyMpZIxEuLiwqhqQpUVYWi
    yNA0DevXrYEkSZAlCYoiQZEVyLKMnt5+qKoCXVWh6/kxI6DgSP55CByHyvIS1FaVQ+A5CAIH
    qyhCFDhoqgZCAI4QUDYuo+OTUPI5bx4ALgYmhXczKO/WRSmbzQKAv6qysqB5fiPsFg4L5s9D
    cWGBs29gqIiFZlwAHMbnMY96vCDgJ0PDo9FUOrNnFpg0Uit0ABa3y3XFvXff6du+bSvkdBxD
    g/14e++BcV2nHWzM7Ox+uA9yCQJQdV1/MhqNHmCMRgnyOb1LykqLOb/XA7vNSiVJSmK6gj/G
    rjnI5l4Y+aKcqc35wwZexuc/8sgjRTab/etHjx676+DBo7wsaxAEC3jeAp7PhwtjsTTa2voR
    jWarFy+u+fi6dessb7311tsmMGWkH3xoeqKUUlxxxRUWm81219GjR5cODAyY7pWbASjNeXiz
    wSXAYWxsHLt27edXrVruJYSkJyeD8erqCncymSPZLG0JBEqa/P7JIPKNJFRKaZYQIn0Q96+q
    Kvx+P9fQ0HDtwMDgx/fvPygqijoVQs7lJBw4cIx3OJylqqpDUXQIggWUatA0De3t3dwll6yZ
    v2TJkkRVVVVBOByeZ7fbJU3TDr788stHJyYmjGrpKJvHSbZmLyikPzY2JtXU1Ew6HI4pQEhI
    vlpbEERMToYQjcYhCCJ4XoQg8OB5AW63c+re8iHy/LhNE1gE2ayKbDZfW0MpB0oJKCWseIdM
    AVEAsFoFpFJx5HKSG4Cb2RVxLuwKpRQlJSWipunlwWCYqKqObFZBMikJDoejiAE9D6XUSQix
    EEI4SikdGRmdLCoqzVEK2t196vDevXsNTWXDTuaYE0sdDsfizs4zS9rbuyGKHhDigM/nyLz+
    +qsd4+MTE+xSnAAEkzrGB2JSHnvssXaXy/VtTdMqcrlcKaW0yG63Vy9fvmpBLqdC1wFJkrPh
    cDgRDocVtg9HmY2cZKdBZL3nzlUXFVAyzUaL1Wq9+56771q8/cptoLoOXdMY4JoGW5Tq0DQV
    RQV+fPqBu4RjJ9rWd3b3DjDGRWVGYs5EsN8P0NI0zWa322+9/fbba5qbm6eAx0wQeQ5Aqc8E
    l6AU61avxN//1bfFL335axVv794bfvm1NzP9A4OiyIGziCJHKbLI53zlmJHM4gNI6rXZbMhm
    swVer/ePPvnJTzSuXrUKmmZ8ZJ5ttVos+NQjD+Fbf/lX+M///gk+/+inYLFaoVEdDpsND37s
    Hv6tXbubM9mcXl1dLfz0pz/tTaVSb7EJauQCxZCXTDLyn9S5MCIA4sPDw4MvvPBi87q1a/D8
    Cy8iHAohl8tBkiUosgxd0/PMpQH0DcaBEIAAhIXIjbHTNObDUDrFeBIWQreKIux2K7xuF8qK
    C7FuxSIMjozhrf1HjfyT2Tqj7wicTd1tpnrDmyqup0iGYydOdP7yyafWrV6+GL959TXa3dtv
    VC4KACyUUsGohKWU8i6Xa8XG9WttL7z8ajgUDhsbUxb5gpwZ7IesKNHBwQGaSSVJNhXHyOg4
    pRQC8gVXCgNl9FxhxfMJJ79bFfJ76KRBWSFQFkCIvYdhD1YOj44JO9/aDVXVDBY4w+bZBJi0
    Ertuw5ExQCd9L0D+Yh91dXWksLDw+s7O0zfv339oCkwam/d0IUR+I5+YiKO7ezLQ2Ni8o7Pz
    VCQYnOQwU3j/A827nn20trZWx2Kxa0+dOsXnozDEFFrlZoS53wlcEsIhHk/i4MGj/Jo1y8sJ
    oVFFyUx6PFbC86o1FJqIHjlyJMUYLtnkKF30sDelFJ/73Ofmp1Kpz+3ffyCQSCTB88LU9Rug
    ct++o9i0aTUaG+tx+nQfKOVBiIZ0WsLp0wO2VatWL0mlkpZ4PE41TWt77rnn9kuSZKxJA1BG
    WUg/e6H31N/fn92wYUNvY2Ojnk5nuYaGeni9XlgseVYxP8emx8oYO8OvM7OM5t+bzq+cmWeZ
    Zzl1qCqFLOuIRCRwHI/iYgvt6grndF2XcLa8Gn2PY2AuJJzxs/b2ttyWLVsPL1gwfwswIHo8
    AVpQ4MwdPnw8ykCryE6BvY/e2trSEI+nalVV0xRFTTO2W2evKUJIhukP01wum3Y4HFmXy2PT
    dSs8HhcURaaZTNZrCt+DRVVn3NP7tZPvxVaxNCcplUrF2H6gAsg1NTUVFxQE7LJM4Pd7hfr6
    OvvY2NikLMspBiQNOxk0zbm4ec69345Hc5lDWVRVVXXpjqu3czzPQ5XVs4DXFIOn69B0FQ21
    1di0bpWvs7t3Ebu5DKZL/DP4EAp02IOqqK+vv/yKK64gHEdA9XyOHUWeYdBUfUb4ezaINN+n
    TnUsal2Av/j6ly2f/eM/Kzh6/MT4RHAyzRgtI9drHqb725oXm4KLIwKObDbLAbhh+1VX7bjj
    ttsIIYCu0+mqU5Yr6vW48alHHsSff+2bePk3r+Haq68ERb4IYkFTIzZvWGdPZCR0dHTI0Wh0
    0PRRaeR1HMMsxDFngJId0Wgs9u/f/ou/sK5ds2b9hvXrxY0b1sPtdsHjdsFmtUJkea4Cz+er
    EzmWXA6AYNoJoLoGTVOhqRpUVYGqyJBlBbIsISflkMlkEY8nEI3FEI3G0d07gD0Hj2LfkZPK
    yFgwwsDM7Ps8b3jf1M2HUEodLDcsae6bzsZ9YmBg8G///Ovf/mxBwLcmOBlS44lkG6Yr8ThT
    iIUC8FeUlS5ZtnQxefHlVymmxdkzswSTVQB8NpuN/uv3f6B3dXUSKZfVX37tLaOFoQHokixv
    L2NiNs0G7beed+dhJE2/NvWLCoC00d3IZrVev2L50rWrly8mdguPm3Zc5ti178iC3sHRNuT1
    OSPMSI6wsUiye5DYszHagdJ3a615sYAlpRRbtlzuz2ZzN7W1tTtyuRwEwWYCXdyUPEseeOW/
    Hh9PoLIyUFVf37A8GJxMUkpzhBCj0O0Db6FpZnrtdvuq3t7exnA4PPUzi8UCnhegaTqbJ+cL
    eZ+dW5lKZXDkSBu/Zs1yX29v74iqqsrKlSv9HAdh9erVa/v6+qzt7e2apmlmB07DBYiAv9ux
    Y8cO2+LFix/cv//A8oGBwXOA5Dwzl0ikcPJkF1asWIRYLInJySjTcNQwOhqCJKnueDwOj8eT
    fPvtt0ckSeLZtaeQzw2NMBtijua8/5ygyUm5ra3tscbGxpLi4qIdo6PjjtHRcciyAkVRoao6
    s/lAHmcZp3k+clM6lEZhB8/nmUyeFyAI+VdRFGCxiLBa86fdbkF9vQOEZLXTpw8ljh07Zjh6
    YUYyGKld7xo5oJSiuLiYv/rq7YFDhw5n29raVFPdhn7s2HHq9/sfa2paUFJRUXa91WotPHOm
    63RHR4c5JWYqN9Xn8xGPx7u0v7/b43I5VEHgjVagqslOGukUJBaLZSory9KXX77Ol0xKus9n
    z7a1He1LJBKGY5My5TdLTOCevhdg9l5s5TlaYRqSb0bxUJYBQ/WKK65oqq2t3zExEfMqCkEk
    onLz5i2vdTjc3Ouvv3o0m83G2H41zDBXHECGidQbTVkEQ4GG2cv3bCuFOQRhBeVlZWWlJSV5
    fb9zAC3oJpZP1yEIHBpqKzmOkFKd0jLkK8BCDITwH7SRNB0rly9f3tDQUA9NVXH69GkcOXIE
    k5OTaKivw2WXXgJRFGYCZl0HBbtndo8GM6vpFEsXteIrf/J5y2e/+OWioeERDoCnvLzMvqBp
    funJtvbW4GToV5gpxDzlec9VqG7W+9TV1dU9/MgjD7sCAT9UVQGzLFMnpRQ61VFXW4177rwN
    P/nZY1jU0oya6iqAA0ZGRhFPplBaXolXXnklxa7XyYxiio2jQalP5TvNwXyjLLx7OBQK//Wr
    r73+hyuWL7/y5ptu5Hme5JlxXYOua1PV3wZg1DU1/30GJGHMTWYDyBQzqU89A1BtajwVWca/
    /NfP6H8//rSck2QDxIwyr2+C3et7As6MXXyAAJfqlP4lW+QwbZAAcCgSjf4bLwi1JaVlRfFE
    0lxYM9WNgsmr1DfOa6ivq6k2/q+wOWV44GaGcl1NZfkdOtXV//7p43tMBj7DvNgRdk8htsHJ
    zLu3YGZrNToLPJvbjZ4PTJoNolFlbg6FGXIdBmVeXlBQsH3xwlaPpukIR5NwOuxkwbwa9/DY
    ZLWsKKeY3mbOFH6ysucosXvKsg1DncXuaabw8bnWyZweXq+vJBaLzw+FwqzqciaInAaT06+a
    piORkIVAIFCLfMeVBKU0yiSqjPZwHzig3LJli11V1csGBwftqqqioKAQzc1NKCoqAs8L6O3t
    w6lTXbPYrbNB5Gy2MpFIor39tLBoUUtpd3dXKJfL8RaLtVxRaMWaNetqNU1LtLe3G46OuUnE
    RSEgFi1atHZ8fPyO48dP8Lqe75ecZ+a4s4qOxsYmMTw8jpaWBuzf3w5ZlqHrHAoLA8hm0yCE
    0GAwGB8eHjbk9FSWmmGEIGeArQscI+3VV1/taWtr+8dt27bF3G7XPfv3H7JLkjKlLzld/cub
    wtzGKxibnN/HdZ2AEBWqSkGIDkI0EKJOzeFpvUoeVVVe8HxC3bVrZ2xgYHCcUmrYyKCxP/w2
    93jXXXctdblcX3O53L9qb29/gYWWdeQVENQ33tgZPHz4yH/ccccdBZom36Bpmqqqqjlv34jX
    0+3br3Rrmr5scjLKOZ0OcBynmvbctInQkm+55eZAIFD0SDarljmd4qnh4d4zx4/3o7e31yC/
    xgCMUkoN228AM8EUAqcmJ1k32UBu9pw1q+WYNC4JAL6iosLe0NDgttlsTkqpVZZlPhaLkcHB
    QS0ajWpOp5NbsGDBlZFIoj6bzcFmc4LjAFUVLY2NLVVnznSd7uo6LQHIOZ1Orays1Ob1ejmO
    4zyyLCuRSDQ7PDycoZQaOeuGvVQopWb29bzSZXMCKC0WC1RVJTorTKHm0LBOzwp5G18TSvMb
    e36T8rDT2Az4DwlM2h0Ox+WbNm10UErxH//xH3jlld+gpKQYDrsdTocdqqpA5LlpQKmfJwyu
    T4Npneq4ZMNafPrh+23f/Ku/d2WzuXFVVcvuveMW10uv+Gofe/LpJgA9LIHYqPrOzfV9sg4z
    As/zt9x44w3L1q5dC03TpsDT9KlPAUtd13Dp5g3YvXcffvGrp/HZTz8EQRDxwkuvwOFyo62t
    DYlEgmI6f9CoqjM80oQJIF/wZm3Ktc0RQvpzudyTTzz55LKWluYyjhCEQiHE43FkMhlks1nk
    ZRVkqKqal//RGEBk4W1DEoPjCASOA8/zEAUBFosAm8UCu80Kl9MOn8eFnCThiWd+o+ckOcoA
    5AgDgiMAJtgm/46AMl9hyUNV1aKaqoq7GmqrV7+99+CArCg/ZB67wgB5ji3mWGFhobJs2VLb
    mTNnPLquT7WwNBkrjhCydFHrAp/X4wbhpopodGbIplJJKKUlhQH/5y/duGbea2/u6QWwi605
    F7v2EeRlrYIs7COwUON6ANUAXmJA02DVEyZQYyRb6bMNpAlIigAqASwTBGGFy+Vq9Hq9pQ6H
    3cNxnKiqqpqIJxLhSGRAluWjAOo8HrfP5XahsaEeNqsVsVgMaZnA7e70hCPRMgBWgefX+7ye
    gNfr8YiCIGiapqUz2VQylQqlM9kBXde7KKWnGaOZNjlw5hQb/WKCSkqpTVEUu67Ts0DkbGbS
    zFhqGiWEcC5CSIBS6mMA3IYPwfE2Kp5bW1urE4nEhuHhEVJeXoGNGzcinU5jfHwCsqxgbGzc
    BBTfGUTODoNPTIRQUBCwVFRUOnfu3Dm+efMlYiKRcZaW6g5KqR+An1KaYqG61Fwweufa2O+/
    /363rusfP3HiZEU0GmOh7rPBpPE9SglOnx7Epk1FqKkpR2/vCETRgoaGaiQSMVRXV+PAgQPQ
    dd3ITTZSg8ImoDUFPN7vHDRCogCk8fHx4Ouvv/7Mjh07Vra2tixLJlNwOp2w2WwQRQtEUWRM
    I29Kt8iP1+zQd/69yRTvwLK8oGmAplGoKgWlHMrK7HT37rfi/f0DQQa6htk5xu4z/V7Tupqb
    5xOr1Xb1+Hj8GofDUrps2bLeI0eOhJiWs6F0QdPpdFLX9XQwOEm8Xm+B1+tV4vG4UeBLmZNN
    S0vLKlKp7IJYLMmGmjM7sBJjGeXGxnlk3rzGOyYnMzcQAqWnp2PfG2+8McxSgnTmAPQDGPF6
    vRm/38epqua47LLLlhYXF28aGRk5/thjj51gqXwKA5wGoOSZ85w9Fw4xuvns2LHD19DQ0Gqx
    WFbpur5QlpVKRVECbP5QnuezGzZsGE8mk+3RaHSS5/kmWVYQj0cxMREEz/Pw+32w28vFqqrq
    MlXVYosWLWoqLi71C4LoA4iNpS3kCEFE05SxZDLRPTg40Hn8+ImBeDweR76V7blaoZ6lGz6X
    Ie/YyPDI2PDQUFVpcWGe8dFNoWF9JqgE1SHJErq6+yhrLC+acx3wAUp9GBseK1Kpra+v37Rx
    wwbS19uDzs5OrFq5AtlcFk6HE3fefitEQYBONUA3hbepPqPi24hamplajiO4/abryf6DR9y/
    fu6lRDwWzwwO9LnGJyYUZkS8AJKs57QhjTFnldHGRgCgoamp6c4777zTYrNa88U2hoVg1ctG
    JbPhAFgtFtxxy4349l/9Ldo7TsHv82H/oWMoq6yGIp1GYUHAEQpHLMbCZBMwZQrLzfXurLHP
    SADo6uo6vf/Bhz95maoqLpfLxd1z9z2ktrYaFosFVosIiyhC4PPCvISQfP4kNcLdGjTV0PGS
    IcsyJEmCJOUg5XJ46fVd2HvoKHiepzzHaZlsLs5A1xAzKH0AhgghEyyE8I6bG6U0D+KBitKS
    kqqH7r+D6x8c3tHTP3gCwBgLaRo5gAoLWSurVqwgL77wYlE0Fsudg+21ez2e5QtbF/CiKEDg
    BQH5gh1jHIwQhsDz3D3bt26+IuDz0vFgqJONEWdyAIYAZDmOu7YgEFiRk3JJRVHcl25Ys57n
    iO2NXfuXZLK5HwEImrpeKCYnlTf+bwaTzEhWAbi1oaHhlpUrV7asWLHC3dzchKLCQjjsNnAc
    gaLIiEWjaG/v2PD8iy9d39beQb/zra+JWy7bDI5woFQD1XXcfvP1KAj4XT/88c83rVm5TLhk
    41pbS3MjAj4vBJ6DqqpIp9OYCIbQ3dNLT7R3Zk50dA0MjYzvppTuBNDDAEnSNE9ntPiba1CZ
    TCYzNpstbrPZyyVJOws4zgx/TwNMm02koVCKUErtmC6Wmktd198mGgUAxOFwXNLb21sfi8VR
    UlKC9vYOFBcXo7FxHtrbOxGNxqdAyrkYSsCQnBFZ/vJMcNnbO0TWrFniqampSSuKovp8fnAc
    Txnp4AHgpZR6TLbSaK03ZwRDRUXFuv7+gatOn+4m0/nN5waTxn3lchK6u4fQ3FyPiYkYvF4n
    bDYCTRNAiIjKyirXiRMnBFVVKQDFBIouqH/3+ZxuAMmJiYnBbDa7e+PGNZWiKPoJ4fjOzk4y
    NjYOTaOs0ns6/D19f9wUk2kOdfM8PxXuFgQBgYAbVVUB6Dql2WxWHxkZygwNDY8xWzIAoBfA
    ICFk/LeJ4gDApk2b7ZTSlr6+MZSUFCxsaVlwdV9f75vRaEwy6cymKaWSKIrZWCwOq9UaqKqq
    EuPx+FmfY7FYWmKxUHEuJ4PmtztjTalsjusA6HXXXbssnVY/PT4ed1RUuPq7ujqNlC4jLD4m
    CML4jh07mqqrq7cRwjssFiGezaprg8FsTVFR+anLL7/8H15//fUTjChKm+ykyOxzDjMF+gkA
    XHnlla7FixdfrmnavZOTk+tHR0eLg8FJPpFIQlFU6DoFx/GwWKzw+XyorKy4sqGhMdXdPeDZ
    t+8Y8jJRBgM9jPHxMLd27cIF1dX1jZFI2tHRMcplMvLU+wiCAKvVArfbAZ/PpbS0LA8uWLDw
    WG/vmdf27NlzLJ1Oh2e1QpXNDoFhK+cEULKK1OjwyMjbTz/z7IrFC1t4DnQGIwkT6ALVwRGg
    83QP3tp7UDHlS5rPD6wc01SMwwPYdukllzTU1taA6jq++Y2v4cCBA/j5zx/HNTu2QxD4s0L6
    s9lITVOx861dmFdfi6rKiqmiHp3q8HndePhjd/GHjhwvHBkbT59s69BryostVoulVZLlHrbY
    jOpdi3nCXaihZPIXvCAI11199faWRQsXQtM1U4g7f/3JVBI8IbBYRBNLqaNpXj1Wr1yOJ59+
    HoUFATg9PoyNjuDh++/Er597yfL08y87ZuUAnjWWc7hBG/2skwBGNE3711QqtR/AqvLyissf
    evABX0vLAuiaMj0++nTIm7IxpCw8Tk1hcCNcboTEM9kcfWvvQVVRVEP8dYIZyX4AvYSQQQYw
    jcp26d3mL3sO4Z6+/pOv7txdanfY60pKiv8kFoufliRpJ6W0k0n1RAHIiiIrjY3z0NDQUHro
    8GE7A34cpvXDikuKixbMn1cPPr9Ji2wepQ0HjX3mitbmxodvumab7d9++LO0qmm9zHnLstDb
    GKV0TBCE62+47trP3HX7zfaRoUH86Kc/Q2V5KTxOG463nbouk831AHidhXTMLBmHfIcK88Zs
    tHVcUVJS8vVbbrlly913321ZuHAhXC4nCBOmz68hbaq4bdOG9WhpbnL+0/f+DcuXLck3OFaV
    qdxXu92GxQtbyFVXXOb69le+hAK/l6Wf5McSRhqKrkO7fANJJVPO02d6W57/zc6mZ1/eecnw
    2MSvKKW7CCHBWfmvUxW2cw0qDx8+HNq8+ZKTVVWVCxKJ01Mg0lyIM/MksNutcLl4/dixoRw+
    xMOcN3X//fcXS5J0c3f3GZuu65iYCCIYnEQsFkckEsHQ0KgJTJ6PjST5MVzcglOnziCTyc0A
    n4qioqdniGtubvBHo+FMUVGFZrHY3Rs2bFg9Ojo6GIvFIoQQpykyMicpC8bf33fffS5FUe88
    daqjKJ1Og+cFE4jMj5nVamUs3cx8yrGxEGpry9HQUAmn04J4fBI+XxGOHeshDQ3F9uLiYuvo
    6KhhK7W5tpWm/tIKgJSiKPxLL7308/r6uq7CwqKVNTU1W0+d6qzo6uoGx4mzwt+CKez9Tq/T
    Qum1tcUQBEk7ceJ4NhgMJkOhUDCdTg8aDjchpJ8xlJOzne53u89YLKZ6PN7jVVWF2wHe7fMV
    33fLLbet7+vr3ffmm28eUBRFABDUdV0lhGQVRUU4HHXW1dUVt7W1nWafwQPgXS6nlVKyJBKJ
    2/NAWidsDjnYuucAkOXLl/kFwfbp06fH5nMcR9PpeO/ExITRUCLK7mNw69atRaWlZV88fLit
    UZKA1tZGWK02pNMaPB5La2Pj/Nv27dsXymQywVnRBCPd56yx/tjHPlZUUlLy2ZGRkYeOHj1a
    3N/fj3Q6A3NfdeO5cxyHiYkQBgZGRYfD7h8ZGYdZgcBwVGOxFLJZxdbWNoDx8Qhz5ma+FyFZ
    BIMpCIIgOp32ipISb0V9feu6srKy11599dVnRkdH+wFECCExts+d1TpzznIoCSGqoijP/vyx
    x7e2LmhacuN110wJqJpD3qAUHCEYGhnHP//gJ7R/cMRoMp+a5aXpH5SRNEKQmqZVVVdX33bT
    jTdYLKIIXVMhCjZs3rQRa1evgsDz08BRp6AwfW0CzrIs48lfP4ebrrsa1ZXlpp/l8ymXL27F
    Dddss//rD36cfeal12KFAZ9D1/UKFgYcYx64WVpBn4v7ZAxlRU1NzU033HCDaLFY8jJBhrg3
    ATq7uvGXf/03WLywBX/wyYfAsTGkVAchBNdcdQU+/ydfRVZSsGrNWghUxpbN6zAZCnHPv/ya
    W1VVg2EWTInQZDrNb+7IEjaZMyyhr5dSmgIQslgsyzmO80lSDvFYDJl0GrKcg0UQUFjgZxej
    zwCOdApM6qCsJeO00zBVaRxhhmSYGcpu9jpuMpTvKU+UFZuMTYbCP3j+lZ3L//jznylesXRR
    04svvjT/+//14/JEMvUTFtLRAOiyLMterxcrVy73HTp82JgnVgA2SqkFQH1NdVVlSXEhspks
    bFarkZtoM42D3+V0fPLOm65psFkt6Og6M8kMgyGSbuSEZoqKCld94pEH7ZvWrEAmEUFvbw/+
    +fs/GuAFPqhpmmFM3IwZILPyI8/FmjVVVVX97Ze//OVN99xzD5xOJwP4M4GkOY2E4wCbzQpR
    FI0JBApTWoZOEU8kUF5ajIDfZxrTma8GAHXYbVi2aAFa5tfz61Yunv/33//xJ461dQUopa8B
    sBoddUzhnDm3QUeOHMkuWrTomdbW5ssnJiKFsVjqPKFvnrW6E1BbW4BIZDQzODiUNOVYS7gI
    RXvvBUwC4MrLy7d1d3evGx4eAc9PA+FwOIJwODoFkt85b5JAFEX4/T4IgjCryMV4vxgyGdku
    SVKmu7s9VFdXb9V1TeR53g/ATSl1EUIM55vHHBT96Sxl60//9E8Xj42Nbe3p6SVGJoe5Irqq
    qgLLly9GZ2cvenuHZly/quro7R3FunWLEYuFIMsCEgkF4+NxVFQUijU1Na7R0VHhHLZyrh1v
    1WivGgwGh4PBoCQIwtjdd9/TRCmt4Die6UnaIAgWEMIhz9wRzGy/ON2Gcfo1fxrM8vj4uHrw
    4MGYKQe7n+X89rE886Apz/A9ywI+8cQvtcWLF/9y27Ztlw4MDG87cOBMqc9XWNrS0tgciUS4
    I0eO7GXjHtM0LcvzPB0bG+dXrlzW4HQ6D6XTaZnZQVtVVZVPUZRF0WiSAASapnM8zxv7bYo5
    vvzq1auvCoVS142MxElra5nU3989omm6yD7HIBXG5s+fv7arq6euvb0bVqsPPO/EmjXzEoSE
    z4yNheSJifEBSZJcLO+ZO4ednCEPeOuttzpLS0s/29HR8bm33nrLPt0kgJsFJPkZ6TH5qJfO
    5Jq4GakmxjpTVQ3JZB6YznZezXmwlBKk0wr6+6MIh7OBefMKbty27cri559/7rHJyckehksM
    m581O0LCXAEytvGdGRwa/t5Xv/HtL46MjDTedP0OUlzgZzIt+XKHXDaLI8dP4t/+66f09bf3
    pmh+koXZJIyYJBO0cxiyi2IkCSFE0zSeEHLj1du3L1+1cgV0TZ0RphcFflZl93nyJZmm4eoV
    S1FaUjTNZs56r5uv3U6ee+k1V//gcGJkfNLQ0RNZ2NtNKXWzLjqGnMFcFeds2rBhw+IlS5ZA
    ZzJBHCEATwDK4Y0338Lg8BgAQJYk2O02cITkkyZ0DZUVpaitqUJW5TA2Moybr74MTocNyxa1
    kOLCAu/o+IQLgJ1SamP9UC1sPOdsAE3slwEqs8hXlINSGuR5XuV5Ds89/zye+MUvoes6dF1H
    fV0N/ujRT8HrcbNxpOZOrSacOt0Rh0xftsrAl9lY9hNCRjBdmflewaTxqlNKyerVK63333M3
    nFYepQUe8pvX32w8eqK9CUCcVd/JsiTnFEXGJZs2iT/9358vSKVShuftYkxkS0N9rcfldEJV
    FNjtNoOhtLHkcB7AZetWLbv6uqsuJy/8ZiedDEUmTCynxMK/EQCZXE7KjA4PQ1rSgngijuBk
    SNd0fUST9S5TrpfwHsfLZrfbH3744Yc3PPDAAxBFkYG9mY4YYbkWmUwWuVwWoBR2uw3ZTAb9
    /QNYtmSRCeBTSJKE9o5OVJSVYnBwCNlcDrquwWaxwO1ywO10TAFRg4HXdQ0Cz+HS9atgt1r9
    f/qX/3Tjmb6hNICDjG3VTAUedC7nLZuf9Pnnn9934403/HTTptUPHzzY5oxEkjBX2BpfW60W
    NDSUwGbLya++undSlmWjCtjMEFwUJvUdQt345Cc/WZnNZh84ebLdLUky2+jeXcTc+J45tC3L
    CtraTiOXk6dkk8y/TykwNDROmpqq3ZlMSvJ43EIwOIni4uKiaDQ6ouu6yxTRycyVrdywYYO4
    efPmHT09PeXJZAocJ0zlElIKCAKP+voaWK08iosL0Ns7gmmJnfx9RqMpZDIZRCIhBAIl6OgI
    glIO0WiGKysr9wuC4FBV1bCVVlOq15ynB7F88wSllLhcLrvVaslpmo7S0hKsXr0y77QRDpKk
    4MiRNsTj6XMU05n1umdKCrE5qHIcl9J1PcwA5CCzkUNsj3/XdKDzHYqiaNlszn78eAcmJuKI
    RjUUFxcXlpWVLWGfkwWgSpIkW602PRgM8bpOq6urqytOnTqlMufXUV5e7s7l5PpkMgNCOKiq
    DkEQzAQOWbCgudJqdT7Q2Tnq43kOVisSvb19KWZLDfIrCiDG83zC6XTpdrsLgmCF2+1EJBKO
    Pffcs12SJMXYc3jPmpvz5s3bODQ09NCbb75pz6cvggHAmc4ZAPC8AIvFyhw6HtmshKKiAgwO
    TsxwygAOfr8HhOTTS0Qxr8Jg5L4qig5KuXPKfaXTKrq6osKCBQUb1q9fn3ruueeeZVjJiEDO
    0AwXLtRAMmYPbBHnKKX7+wcG//Vr3/rOvb/81a+XrF29QqyrroQo8AiFQjh2sgP7Dh6lk+GI
    ZMpFMM4ZbcU+CI/bhCJampuaPnbfvXfb7DYrVE2dWbEN/azqbaO6e3Y1OyEESxa1oqggMMXA
    mHMsNVA0NtRg22WbLP/xPz93WK0W6+KFra7BoSFxIhhKIF/c4WSGMml+HhdoKJ1er3fHlVde
    6XA6HNBUBbqmY2hoAJOTk/C43QhOTqKgIAAAiESjoOF8ZXNBwAeH3YbBoWHEEmlU1TUgERzB
    6mULocgSqspLMK++xj46PuFnzJhh5K1zxbKeB1QaYC/DxjHGc5zGcRxWLF+OirIyOB12CDwP
    q9UCj9s9y1yb+npP6VKeE0coJkA5zlqGGlXQaZxDVuc9sqzxYDCYGRoa9NZVlmJgcAjxeEJl
    z6+IGa54TpJS8Vgca1avRFNTY93hw0cr2YL2APCJotAyr75W5DkCm9UCj9slsPcw8uxKCgv8
    t99963Vej9uFEx2ndVlRomx8DPmVDKuWbkok4nXf+s539YMHD9KJ4IT+8qs7xxmQsZhYPPUc
    a3UGsyeKImRZrl2wYMFVt912G2eASVB9usKePevx8Qk888yzeGPnTprN5YjX6wXVdfQNDGLX
    nn1Ysqh1ypkjAEZGx3Gi7RR6B0dw4lQ3KNUxMjKKcDhES4uLcPXWS8gt110Jl9NhSr+ZdghX
    LlmA26/bVvjdf/ufTYqihjFdFW6upJ9r51ULhULJJ5/81eNXXHGFsHHjipvHxsIl4+MRLpfT
    wHECrFYbAgEvSks9NJEIqr/5ze7o5OSk0UnL0I07qxDlYoFKc37X+vXrRb/ff+eJEyfXDQ0N
    mdjJdxMxJ2eF8vNt+UT4/V4kk2lQqpzjPQhisRQkSRfzuEIVAb56/fqNLgDhU6dOjTN745xt
    K8/Drr6nY+XKlWXJZPKK3t4+zgBNPC/A7w/AarVC0yisVguy2Qx4Ps/y2Ww2xvCpoJSgsNCH
    TCYBnheQShGk0yoEwYJ4XEJ5eaHb43F7I5Goc1aKU2auAOUsG2lIgcHpdDpEUVQopYjHEzh8
    +Dg0bTp/MpeTGBDBDJbLKF6cBpLmZwwQwmnMFobZHjaKab3Ds+brbzEeVJJyEs9zkYKCAGIx
    GQ6HA3a7RQuF4oLJTiaz2WzOZrMqmUyODwZD3oaGhuZTp05FjIiKy+UqS6dzJZKksBagKvF4
    LAagFDiOs65cuWrT5GR6WTCYRHm5H7lcKjo5OWnkPaqEkCylNL127RqPxWJZOX9+PWex2FRJ
    0onf70gfOrR3SJIkIxWJvoudpMazuOaaa2yapl3T0dFRPA0mMWPdAPk1U1NTjYaGOur3+yBJ
    EtF1Co/HDV2n6OjohSyrM3KwS0sL4HQKaG2tYM6qFW63iyYSKWSzOgYGoiSRkM5aoxzHQZZ1
    DA6mhYaGqjUVFeX9g4NDKqVUZkWj5rzz9wcozQuUgUlCKbUCaACwyWIR1/h8XncimaLHTnRg
    cHgUAs8jkUwiFArDYrXC43bzyVTKSSn1ssHkWS6WUZhjMeWX0IvIVBJKqcvhsH/irjvvaF22
    dEn+ns7TCccMMs9mKvM/z2Yz+I8f/hi33ngNNq9fw0J4Mwt0LIKA7Vs3kyefeZGPJZLZSzas
    ccUTCyp+8N8/awDQxRhK9yyvjr7f8WLPrX5+Y+PqtWtW53tRAwgGg/jSn/wZ0pksfF4vMpks
    tl5xBfbs3oVvfufvoOs6gsFJbLlkPT75wL14/uXXECgqQdvJk1i3pBEOmwWaosBlt2J+Q63w
    1p4DPmYgXaw7gW2uQlHvYDB1EwBIchyncxyH2upqVFWUT3XLURUV2VwOPCEQBX7G45ySMp/q
    8c2+Ow14VGYsjZBOyOR1q3h/FewUQNeRo8cf/6M//pN7mxrr/QcPHSF9g8MT7J7cDDDmstlM
    KhKJoKK8HFsvv8xz9OjxBbquxxijXeJwOBrqaqoJaF6IPeD3cezvLezcuHXz+sWb1q5ALJ5A
    d2+/AZqsRqiesaGNxQX+b9pslgVnenp3f6+nd9Lk2GRM9x9mhUMpAJrpvjWYxIcVRQGAuubm
    5sqqqioTWzgdviYA+vr68X/+7Mv0+RdeVCil/Fe/8hV+65bLIGVTeOaZ53Dg4GHcf/dtrG1m
    Xhe2s+s00lkJf/SFT2L9ujXgoeGll17EF770Z3pP32DuyPF2PhqLWf/gwbuIwPMzJbH0fCRh
    0+pl5H+ffKGuf3hsHgt5JU3J5xfcrcRYe6bnoxcVFXE1NTWOSCQy5nA4upcsme9fsEC1ZTI5
    6Dpgs1ngcFgRCk3i9OlRzWazwev1IJlMKbquz6hGZ2vrolWnz7a7GzZsWBMORx46duyEVVU1
    E6t4brFyMzA8uyiHwOFwoKGhBhMTYSiKds73opRicjLKFRY6uPHxcUlRONFqtXm8Xm8R8l1Q
    fMjrOCZNUYL37cB2d3fjhz/84fLTp083hULhqY28uLgIW7Zcglwux8KLOnp6etHY2ISNG5eC
    kPz9nDo1hNHRMGpqihGNTqCurgFnziQZy8lBknQAos3vD/gikaiLgeGL4nzPiuYoAPSCgoIk
    z/MqZdEASRqHIFimOuYYou3TmSwzX88vfg6d2Yi4Kepo6E7K5j3st7WTfX39ycnJyceWLGmZ
    X1FR0eByuS02G5d49dWuGKaLtJzpdCpnt7sUQjhbf/8wt3btsuaCgkBPOE9g+W022/xUKuvM
    F61wyOUUFBa6HKIo2hVFERoa6mtdLt/248fHRF0HPB47DYeHE7IsGyyjQinNVlZWWNav3/SI
    pol3aFpuJJ2O7M9mM97jx/vkvr4+I31vEkCI6YsmzmEnZfM6Ky8vd2cymUXj4+NngUnjWYui
    iOXLl9Dy8hKNEFBVlYUDBw6A4ywoKCjG8uULEQj4MD4enkrRsNmsKCz0ore3G4cOHYamEfj9
    Jdi0aQ26utolQRC05uZF9lOnQlw6rZxD5ivPVMqyy19ZWblgcHAoyPZCs7bx+8+hNB4KK/IQ
    ASyxWCz3NDc3X7l61cqaZUsW25vmN6K8rAQuhwOiyDMNPwmZdBqxWJQMDg2L3T293uMnO9yH
    jh4v6hsYDjDtowTrlkMuFmNpCtMTFgq8/pJLNt9x7z138TzPQVPV84DIWcLls8Ta8+9JEYvF
    0ds/MM1M6vpZf6PpFC1N87BscYv46pt7pLb2Dn31skWc1SKWSrJSynIukoyJMLQDL7R7ztJl
    y5dVlJeXQ2cFJ1arBUVFRcgMDIFwPFatXo2bbroJLpcLb775JqBpyORyEEUBvf0DOHLiFBoa
    mxAKjqPzjIBYLAaPxw2O41FbVc5xhLj0vFC3A9PVqAIz9Bfz0DmO0zVNkwmXD1S/8NLLeOON
    N5BOp9kmoELXdMxrqMXDH78PbsZcTYHIGY+Wzn7SZk1Do+euUcBB398yohohJCHL8o/e2rWn
    561de64pKyvb4vf7CyKRiMEuOgFYJUlOhsNhcBzBtVdv5x7/xS+b+gcG+9mipgV+f2VJcSF0
    qoMjBGWlJRwhxM3yK0vLS4svue3Gq60upwN9/YMYnwgZlfcG2M8BKCrwez+7cfWSNQePn+oH
    8Bu27gLs95KMbRjHTE05ddZzmto0BEGAqqoFBQUFFpvNNt2FyKQqIEkSvv8fP6C/fvqZHKU0
    t2TJYs/dd96GyrJivPzyy3jpN6/qlFISi8eJ02GbAoTjwUnaceoU/Zu/+S6+9Y2vcpdtXIer
    tmzGr9evIc+99Goum8tlf/bkc4WXrFtpW7Z4AfQpXdE83tB1isKAFyVFAWf/8FgpYzoMPUAL
    LlBpwjDMhr285pprvM3NzRsopTekUul18XiicnR00tXZ2cdnszJUVQchHETRArvdDqfTgaqq
    Jmtj48ICjqP2VCpm7e3tzXV2do5LkuQG4GEh+ox5w75Izjd56KGHKiilf3z06LGGUCg0I3fy
    3ft2m/Mop/MQ/f5855azc79mAtJoNInKyiJLOp3K+P0leiaj8fPnNy0cHR0dGRwcTJmiXFlc
    YPeg2267Tdi2bdu64eERt6Io4Pl8OFiWFWSzEgihUFUZ/f396O7uhsViYdqbPATBDV3XUVpa
    AJ5XIQgirFYHAgGKiYn0lAauphHB4/EYzKTBkFkvlvNt2Ejm0EhGNXxpaQlaWxewHEphykGQ
    JAUnT3YjlcqeFeI+G1ROzXejFajZRp4lvv/bzkt2rcoTTzyxZ/Xq1X9eXV11RUGB45ZMJutu
    bGwsDIVCQ0z9wBWNRmWfr0C2WCwIBkNIJrPFLS2tjW+//XYCQIEgWBqj0Yxo5IhKkgpRtDpE
    UbSrqmpfvHjJ0snJbGUsloUgCLDbRW1gYEJiJJcOQLLb7XTHjmtuz+WEBwhRyeBg12vPPvts
    O7MfdvYMjEiWoU8cwsw2oao5FYxSClEUnYqieFiRM6ZF5s15u5WoqCjV3nxzZ3LZsmW2iYmw
    MD4eRUFBCaqqyqjVKlCv18WNj0emwL7VagEhOi0oKKQLFy4mJ06cJvG4hOHhCAKBAv03v3kl
    6nQ65erqWl9XV4jMZkQJ4aDrBJIE3uFwliDfCjdMKZ2crRl+QQylrutOAHe0trZ+4d577m6+
    9podpKa6ClaLOF1Ra5y6Bmqzwut2orS4AE3z6rHlkvUkk07zPb19BU+/8JtLnnz2pYbB4bFn
    KKWvMWbL6G1q6O5pc+GBzwoFrGxsnPelz33m0UB5WRkDkybNTH1mfpdOzy7EmQaKebZl7/6D
    GJ8Iwu1yTlV4myWFjL9xO+3YvG4V98au/cLxtg71ys2rrJetXzHvN28d2KrpeoJt1i42SS+0
    DzYviuKSJYuXWK0WEaqazxH1ej342p//Gdra2jA4NIRnn38RTqcTr736Gyxc0IjS4iLcfv2V
    WL5kIX7445+jqKQMA/19ePje27Bz116c6urGmhVLQAkQ8HkgCIJNVhQDSNpMyfIXIzdoxlga
    QrCEEWRejwctLQsQCUfQ29eH7du2gucI3C4HrFbLFCs5xUwauZR0SjQJZOr7U7+qm76eERt/
    P4YSedmQCKV0NwDZ7Xa3btu2reKpp55anEgkRpghs1FKMxPBoCZLEr+wdQGuufqqwL9+/wcL
    KKUhAC6/3+cP+H2gugaOALVVFUQUBacsKw4A5VdctrFq2cIFoLqOYCiMSCwmmQyBDMBms1ru
    umn7pZckUmllZDy4hzENBkueYUBrnIkwG1XR6Vks3rna4FFN0/JFWQQznDUCYHJyEq++9rrO
    iqq0hrpa4nFY0NXRhm/95XeVjlNd8ZqqSm82mxXzOFQHJQSCwBMA8pEjR+Q//uKX7P/3X/5O
    WNRUjyWtTeS5l15VAUTGJib5w8fbS5ctaiYGkDR3g8pfBwhbZx7GrDsIISLmQLqMFcJxn/nM
    Z1qsVutnx8bGrz99+kzRyMgYUqksNC2vuZyvyhTZ69TXRBQtsFptgsfj8pSWBhauXLmxetGi
    xTUHD+5/uqvrNGG9veMmMDVnMmPmUPcNN9zgLiws/MzJk+1XdXZ2kXNVo79zRffZ+ZRWqwXz
    5tXmZbs03STmfvbfyLKKTEbhOY4jPE8Vu91u9fs9Ddu3b9/61FNPjQSDwQlmJ83FOe/rOSxY
    sMCZy0mLg8FJYs6LjMcTeOONXSgqKoLf70NzcwssFgsaGpowNhaGLEs4fboTyWQWK1c2IRIZ
    Q1FRBdrahlBbWwqn04E8C61DUUCcTofZ6TbbyotmI4G8bjQAjlIKVVURicTB8ylUVJSht3cQ
    iqKBUkCS5HcIec8ElWfnEM2wk3Nx/aqqqpk9e/Z0HDwopu66667qZDK5vaGhYVEulxs7cuTI
    CAB7OBxRGxuFnM1mQzyexpkzA+KSJU0t7e3to9FoNMXzfFU6nSPGmGazMnhetDqdTmcg4K/w
    +YqXtrUFecoiPYJA1UgkqpvmlLJ9+1WbeN73YDSqOr3e3In9+/edYljKkLGLznK4jQYE6ixA
    OcNOZrNZiKJITekxs5wrDlVVlXRwcCAbiURSLpfb1ts7TtzuQixdupBms/H46dMxwWq1umaD
    wlQqre/duzuzdOlycdmyZdaurlGSSKikqakQFouY6Orqylx2WYPTYhEsqgqYxfrNKSqaplnZ
    nuDGtGY4NxcMpY3juIc2bdr01a9+9auBNatXgSMUmqoik81MybFMgUmTPMsMWRYADXU1+MOH
    7+VWLmmt/uf/+PHHDhw9WUop/TWAAWIkc+Qn5px0WjEJfFcXFhZ85bOP/uHCjRs2QGOFODgH
    UOzt7YfVKqKspPjsvElW8c0RgjO9ffjP//kZPG4XSosLoc+oGDblXSK/ua5c0kKKCvzC2EQo
    8/f/96e63WohhJACtrlxlFIB+VZIF7q5OVwuV019fZ2xS0xdS8DvwyWbNwJUR3BiAl1dXXA6
    7Lj/7ttRXVkOUB1dp8/gWFsnahsaIadj2HHl5RibCGL/kRNYsXgBeCrAbrXAYhFFWVGsbKIZ
    4VYBF/EwgJxRgQoCjiME69etxcYNa7Fz55s43d2NzZvWw2Gzsq456hRLq+nnYCXPDw7JOxUs
    vI8wlFGpnqCUnhoYGNh/55133PSx+++v//73v79aVpQJZoyk8fFxRVFk3mIRcc+dt3Gvvb5z
    /qmu0yMAMgG/z+FyOqDreUmumuoK+LweZ3AyXFRWUjTvhqu3CjarBbquIRyOQpJkzZTjQzhC
    rrr68vVXrluxkP/63//XkK7T02zsZHZ9kywfyuick3kHADO7l214YmIil0qlrG6X05TLmEdy
    Ui6HZDJhtBDTOVCq5tLYvWc3PdHeEQEQFUTBzfO8qbMREPD7YLNZaSaTTZzqPC0/9etnAk2f
    fZjYLALhCJF0SqOUUmsoEi1VjaiDqb0oB4poLIFgOKKzZ2HMV3EuNnZKKSwWC3n00UfXyrL8
    3T179qw/caKNZDI5ECJMhReNziv5v5nZH1nTKHI5BbKcRDSaI8PDTm9DQ8nWSy+9ojoQKHh8
    3759ByilQVahzmEOtQyNe6irq+Oam5tvHhoafvjgwUMWRVFNVaIzWUdBENDY2IBoNIZQKDqr
    knSacREEAU1NDaioKMHp0/3Ih8+5d2Qz4/E08fsdwsGD+6Lz5jVyqhqXEon4SC6X01jqEiFk
    StD/fR9+vz+Qy+XKE4nkjA0V4JBMZpBKDWFkJIhAwAe/349UKoO2th5oGgUhPGpqykCIAovF
    imRSw8hIGOXlARQUOCFJ+Q4zuk4gihYLs5NW07zjLqaNNKV5cYQQhMNRRKMJeL0+1NZWY2Ii
    hGQyw+alyKp+cVbI++x+3u+v9/Nv4djohBCJUppSFGVifHx8p9/v39DRccq7ePGS9ZIkjbW3
    t0fi8bjAcSRts9mQTGYxOhpEXV114apVq1p3796VBUhRLidPzTNZ1qBpsBQWFvoqKysL43HF
    n0zmQAgPURRAiK4nk0nj5tXVq1c1lpc33Ds4KBUUFYm5/v7uw8HgpGFLEyy/2WAlY6ZI1rnW
    5AwsMz4+nq2vr4+6XC5EIpFzOGw8LBYLJiczMiEky/OCZrG4UFDgRkGBJ/fqq7vGWlpai3Rd
    dJnBaD7ywSOXk3IHDuxPbNlyVXEwqIg8bwfPi7ooiolUKgVCaKUoCpb8Wpzp1PE8D4sFNJGI
    4xx2klwQoDQcOULIw5qmWX/2s/9N/+TH/8Ppuk4opRylOqFUJ1TXOYsocoIgsI1bRy6bpaZO
    JXn1+nw1bb7HLoGV57htqqaNAwixsO9ZF36hXreu6xZRFO+/647br7jnrjsIR8AAxtnMIyjF
    3n370d3Tgz/94z+CKArQ1em2fQZ7eaqrG3/xN/9EDx89gUc/+XFSEPBB18xgclaVOIDqylI0
    1tcI48FQdmh0YoR5M71s8zZ7MeT93j9bzA6v11vs9/uncz6ZA0lZz3ECIBDw43TvIHhBRDwW
    By0vg6IoeP6lV1FYUoahwQHceNUl8HncWLF0If73sScRi8UQCPggCgQ8z5nzYA0JoQ+ymwfh
    OI4ABKFwCOFQCKFQGILA4623d4NSHaqi5E9VQVlpMRa1NE09WDr1D9Ph2Ytk4E3iw1kj71CS
    pFd3vb3r0r/9m+8WRCLh5T9/7PGEpmm/AaCNT0zI2WzWJvAcWprn49OPPOD88tf/YlkimRwr
    CPhFQeCnNEOrK8pRVVFuD06GazevX+Vf3NIETddAqI54MgUtz5wZ47J09bLWqz9xz422Z3/z
    tj40OtHJrsvCwKwhlTTGjKRkSsHQ32kD0TSNABjt6uoaGR4e9rYsaJ6Ry6iDwuVyoby8nOvr
    H9AA5GKxqJ6Oh/lTpzppLieFAeSKiwqJy+FgnbXywLC4sAAetxuZTDYLIHv8RJs3Fp4U0qkk
    1fNsZxxAkctpB8d61cMUbqeE4tSZPowHw+b8Loo5rO7+zGc+U2ez2b4zNDS0IR5PoLq6eoZG
    nFn+4+zvCbPkQXiWzyRzPp+zecOGDQ8TQrBnz55dlFKJpQrJc8lSAsB1111Xm06nP7l//yFf
    PJ5g139u1pHSfPFKUVERIpHEDCCS/xkHl8uJxsZaWlFRhGQyjaGhcTLdypA7T04mQTKZRUVF
    QAiHQ9murq4gx3Gjmqb1YFqua070i0VR9KVSKa+iKGfdo6EXSikgyypL/SJMuUCF1WpBdXUx
    otExlJRUoKNjEoqiIxxOIRAoRDAosboDnvC8wGNaNkicK0fm3Q6Oaf1QCiZm7YXb7YYsyygt
    LUZhoTbl6PC8AEXREAolkDcZZ+dVngNLXoymJBR5JYokpRRtbW37r7322hPZbG7TyZPtZWvW
    rN0uSVJ6eHg4pOt6xuFwgJA4VFVHR8cZft26pfOXLl0mUQpf3iHiWAcqHamUzDU3N5W43QX2
    7u4Ylw+HE4giD13XqKLIHABUVlYWLl++ZsvoqFalaYAgZEcPHz4yjmmB8hgrRDI0iY2GD9p7
    ycU+fvx4dv78+Z0lJaWXDQ4Ok5nSP3nFg3Q6g4KCAkIpzem6rrhcAXCcCFWVk8lkMuz1eotG
    RpIzwGheb5MQj8ej9vb2xmU56yosLPDqugWEQFJVNebxeGyEcETXzy4CAjjYbAJ4XlbHxsal
    WUz0jON9h7wppaOapn1l165dgV27dvkBFLDTb9ChhJDGR//gUyWXXboZmqogGongr/7mH9DT
    159lm2hm1pljrwk2KDzyIsncHDB05gUFTdPqmpuabr37zjtEquvQmITOFKA0hacBih1XXYFT
    XdU4duIkHHY7ykqKIAg8VEXB+EQQb+3eh/99/Ffa8bYOunRRC3/b9VeDIyTPUJpyLGcX8rid
    DrQ2NXBv7zvMM29mEtMC2WnMXUsxi91mc9lttimGdWpjNYojCODzeqHrOmx2O4KTIXAc0NPb
    h2NtnZjXtAByOoa1K5dCVVU01tVA1XUMDo/C73OD5CWIDO0THjN1KC9KyPucgJJwBJTihRdf
    wvPPv8AMI49f/foZKIqCVCoFv98HjhAsX7IILU2NEHhTKgWdxVBe3J5NRt6RgfD3Hz58+OiZ
    M2e2fvPrX7Ok0+nNv376GVBK+yYnQ3IkEoHH5QQA3HLTdaSvf6D8X//vf/ndLqfAc9zUvPL7
    PFi+uFXs6u4p2rHtMs5ht0JV8jgjX2U9NT418+qqVn3u4Ts9dqsFO/ceSbC+uw42B1MsxG3k
    AWXfaeOezUQwjcrxvt6+w2+//daCBc3zibnFJ2ieIb9q21Zu/4EDTlXVUpOhkBoNB8VYNELZ
    GrAtbGkWvB7X1NzVKUV5SRHKSor58YmgDiAbCoeVWDQshMNhymyI6vO4fS2N9YRM6alOF9pl
    slm8vusQzUmysc5kEyCbq+YKVlVVT5aUlIz4fH67oshOWZZdqqo6NE2z6Tq1FBcXF6bTWXc0
    Gp9iIUpLC+nIyLAqSZLCcRw7eZnnOZkQRUkmJVmWxSyTPDE7btxczVhKKdxuN5xO5+b29lOL
    8oCjFJFIdIqVOjvMDZw504eKijKsWrUUmUwOyWQKmqbDarWioMAHn8+tqqqkSpIsDg1NcpFI
    /LzSQ+bPkGUVus7xxcUl/ORkKKdpmpGjZ+RQ5uYo196uKKpd1/XzhusBwgSkdRCSB2aSpKG0
    tBCEyLBYLEgkNCQSWSYhlEZFRQnsdiuy2Smgys3a4z4QW8lxHKE0H/IuKyvFsmVLwPN5/cGG
    hpr8pLVa2POmSKVyiEbTyLcLnc1UXhDX8dvaSUPNQxsZGenPZrOvzJs3b/Vrr+202mz2hk2b
    Nt+4c+cbr6VSKdnjcVNCxkn+2SfR3t7nWLFi0WKeFwRF0WcAplAohXXrGgOjowkukcjNAFSU
    UqLruuDxuN2XXnr58kTC0hgM5lBf79RHRs70T05OGgLlhuTaBAt1J0xpF/Td7CQhBCMjI3I6
    nd5XW1tzT1tbh0eWlbNYyuHhMbJ27TJ7YWEhJ0k5yePxAxCgqinJ5XJrTqfHFouNm9JH8qA5
    m1VIaWkp193dnc3lclmXq8SraQJUNZmTJDlZX99QJEnEoih0RstN47P9fitisSEpFJrMMudN
    Ppfz+r4ZSpaM+RqlNMCSNEvZa6GRj+T1eMq3Xn4Zdmy/AoosIxGP4RdPPkV7+vpzmG7zFjbl
    ShpAM842Lw0XQW6G5TUtW7p0cf1bb+/C4088gc9/9lEEWKeN/O/NlDWJRKN45vmX6K+feUEV
    BIHWVFVyHo8LqXSGDg4NK4PDI5IsK8L8eXWOL3/hD9BQW53XsjxXAY+JqeQ5ggWNdcRqEQVJ
    VlQTqE4boJL1AdUv0MhwhCOcEWk1C0RjqlCBwO/zQNc0eNw+jIyOQVVVPPfSqygpr8TgQD+u
    v2Ij3C4nNFWB3+tGcWEBzvQPYvGCBuSyOSj5dmLknJncHxBDSQghhCPYfuU2bFi3Zqr1oigI
    6Ontxf/85Gf4wmc+BY/byYT28/3Kp0PdpiIdFpa9mBH7WfN8NBaPv/L0M89uvGrbVttffvsb
    Nqrrlzzz3PNl4xMTdHIyhNrqKlCqw2G347OffpgTBd5ZXlaSZzx1jRXD8Nh66QaSTCb55YsW
    QNfYXCYUbqcDHEcs0FBSWlxQ+UcP3RFYtWQBnnpxJ3r6h8eZIbRgurLdyNE7Jwt0vnCWyQfM
    pDPp3zz966d3XHfNjkBxcdEM1p7jCG675SbyxptvBV5/4005FovTSGgCXP6R2LweT/GWSzZx
    giBMdcoBS9XYuHalcOxku5NSmqa6hkwqiYnJkFFBWbl+9dKiZYuaWb96fUZU4URHN3YfOmHI
    QRnt79KsFd6chI1/8IMfnHE4HF/P5XIFkiQVa5pWput6GaW0GICH53nPddddt6m3d9h9+nQ/
    BMECp9OFTZuW04MHD6RHR0djhCAGkBgbhzghyAAkDSCq6/qYGbzPod8NAGhtbRUVRVmeTqdt
    Gzeux/DwCGKxBCjFWQAQILDb7Whuno+iIr8yOjqUsdnsWmFhOWG965VYLCqNjETEyspqXzic
    4np6hohRIPFOBTwGQ5PLKaS4uFhsb29XkNdWzJgA5Zy0qtV1jc+LUJNZGyuBOaIuSQqcTgJN
    U+Bw2JHLqaiuLkIkMoqSkgqcOjXJ1gGHTEaBrqtwOGyQZQJBEKiqKvQcNpJcfAMJYtRujY8H
    8fbb+0x9uQksFitWr16KU6f6EI+np6rTOU6YYiRn9veeHQq/aPdhtESllNJ4d3f3W4sWLe4J
    BAItR4+eIDwvzN+0abNd09Ss2+2ihPDEGLPh4SBE0WKvrCyBoqgzxjMez5BgMMWPjcWhaWDt
    eTnoOsBxPO/xeFwrVqwq4fnCqoGBLGexCLDb1XR7e3uI5cBmke93nTClA72fwjB67Nixo6tW
    rT5WXV29+cyZ3ikdSaMD0+RkGGNjIdvatWtLKaW62+2hksQTjsvyCxYsKEqlZHsqlWXRjOmU
    hGAwSerra5x+/0mOUqqLog1Wq4hsdjxbXl7unjevuXZgIMOfa11bLDx8Puj7959KybJitFM2
    F6VqcxHyJqYcLENT0mjhZgFQ7Pf7xLKyUsiyAk1RIIoCaqurCLuAJAOTo5jW8zNa+eQwXZI+
    Z6EM8wbI83xlUUGB/cTJk7j6qm3I53ZRjIyMICflUFtdBYBCVRW8+fZu/M0/fE/fe+BQilIa
    BZDqGxg00LkKwMrzXOXm9avdX3z0EX7VskVTguYzqsShz+j/nWdCCaorSuH1uC3BUERn955m
    EhiJufC8WaWtLsuyJMumvt1mCRcWfvT7fJBlCU6XC+FIBN1nenGi4zSaWxYimwhj7apl7N4o
    LKKA6qoKDA6PQVUkTIRCyOYksyj0nCVm/7ZZDQRAYWEhCgJ+VhiWHw+B5xFPxPPhX9Y9SKf6
    OcTN6Ttd9ZwxQCYDTCmlhsbZ3t179vS0tbe3rlqxHH/9nW9ZXU5H63MvvKT3Dwxg1fIlUy0G
    PR4XPv/oJ0zKBPmdQac61q5cikXN8+BxO6fGjICioqwYDrvd7fMKjZ9/+C77ts1rkMtJ2HPo
    hJaV5DDzuCnbsA0jmcWsQoffQmtTA3Bo9549u5555tlrH3zgYwQmh0bXdVRVVuCbX/kzIZPJ
    lHZ1dpKJsTEUeB2EENRsvWyTY93qFZhKlWF/Jwgcbrl+O3lz977CE+2d1OdxCdBlsHVUsaS1
    qexT998uetyuvHNnkoFKpbP4xbOv0VAklmLOrAHYjL7QFwwoCSGIx+OIx+Mqez8jaV9ktiNs
    t9v9druDJJPpKX9GlhXIskY8Ho86PDycYFGLCfZqaOopJrBvrmyeUyF2QeB5Xdd9kiST0dEx
    dHWdmRL2bmych1gsjsnJfL5XYWEBFi9eQHO5TPa1134TnpiYiBBC4haLJcNxnGS3261Lliyp
    q6lpqBkZCVmHhsZIvpcw964FPAZrks3KxO8PiBzHqbquZymlqVm28oL3C1lWNJ7n5bx8jjYj
    9GcGmLmcjEDAA1nON4AoLbVMsZPJpI5EIjuVpqAolAFQB1IpCquVRzKZMMZMn+tUi3e3PSBA
    Xlg+FktMpWFwnABBEGGxiCy3FyYAORtInjfkfbHSg8z2RD58+HD/kiVL3m5oqG+enDzEHT3a
    RlRVq161aqmiaRwRRWFGS8zBwXGMjUWYiDc/Na9kWcexY4PQ9ZlOjCxTcJxV2Lbtyhpd9zr6
    +jKcogBFRSKSyWBkZGRYYwyzzJwaQ25M+W3tJPsd/eDBg8HW1panFy1auHx0dNwlScqMdaHr
    wMmTXWTVqiWBpqZ6LRazQVV5FBZyPrfb5Tt0qJvXdToFio2uW+FwEpWVBY5NmzaXezw+azJp
    hd0uUkLs9ksv3bIsGiWBSCQ3K+KQP4uKbEgkRuWenp4ppzbv2JLk7AjqhQBKirzIZ4bNLgNU
    WpAvJnHW1da6y0pLpnQbBZ7HotYFRBB4XlW1LGMmxwkhAyw/y+gKo5mYurNQ8IUaSaafmc7m
    stqffukLQllpCSyiiHg8hr/9x3/B4oUtaLj3TkyGwvjJz36Bf//Bj5TRsfGoKRwdZveqAKgp
    LipovPPm6ws+dudNXHlZ8dTGd7bUkH5WSF0nQHGBHwV+rxgMRYznaPa6zdpq71uHEoCcSCSj
    ecHUc/VZz3+vuKgQhEm5KFkJTz//Msorq9HX24Nrt26A2+WEavRTJgTlJUU43dmJdDqNU6f7
    qKbp5vw64/wge7MTJgeFg4cO4ciRo0ilkvnuK9ksEskkwpEo/u6f/i3fSpPqaG2ej5uuvQo8
    R5ikzVR2KQimjNkHwBxMyW4MDg0N7Xr1tdcXLF28iKuuqsR3vv11bvGiVq6wIGASBs/PMYHn
    wXOYnndsfEWBh8/rhq5pUyy0rutorKvCXTddZV3YVI8tG1eB5zgMToRworMny9YdbwIsRsMB
    6X2ASdA8baZSSsPJZPKXP/zv/169aeP60vmN80xOTf66li1djO/9/XeFf/rev+Px598ARzVu
    xdLFnj/85IPwuF35tBSzA6RraJpXhz//4z+wfutv/qWc6jrZfeA4QpGE5YpL19d95sG7uMUt
    jVNg2iwR9da+I3h99yEZ02LhEeQ7dUXniuliY0oB6EwwPsHeM8vsiFBXVzeP43hHMpme2uB0
    HUilsiguLkZHR0ca0y0xhxiwNDrCKGZmda5zJ0VRRF9fv7Z27bo+j8et7d9/mJckGYIgoKam
    Gi0tTdi9+wBEUURDQy3q66u13t4ziSNHjkzmcrkggCClNCxJUnrevHmFa9asbdV1vq69vZeP
    xZJTm925tCrPBy5zOQU+n9Nis9m4TCajsPmaNbEmFyqthlQqlbLZ7ClRFJHLyZjO75wZBkwm
    M7DZSpHJpOB2O+DxOBAKjaCsrAqnTk3CzLxSCmSzCux2AVYrIAiqHgqFpXPYSf1i20pGYxNK
    Ab/fh5qaathsNoiiBaKY76TicNiwcGHDlJRVJiOhu3sE+ZDouULeF8/xPg+oVCRJSk1MTOys
    qqq4ua2tszCTkdDWdhqZjCT6fD5wHI/ZOYGaRtncmqmzqCj6rHaE+YK4cFjlXS6va2goA0XJ
    5wd7PLx+5kxfQpYVno2XwU6aAdZvpbdpFB5RSqVdu3a/uW3btj3NzU1XnDzZQfIMJWcCwAoO
    H27ncjmFmz+/CaWlAVitcHR19SIYjDLni5vBbmoaRVfXOLdwYVVRSUkBKBXg83Hg+dLSvr4Y
    PzwcN81Xc8RBhN+v62+9dSSey+UMOxlmr2eRXRcCKHUT0MuZcnh4SilPCNmxcsVyt8/nnWLr
    CIAli1pRWlJiHR4Z1ZkhNPIOhphh10xeyGxgos/RguIopQeff+HlY3U11Su3brkMmXQajz3x
    K/T29eEzn3oIh44cxT//23/oz7/0m0wuJ0Uwrb9nVLrqhJBVK5cuWveHj9zvvnzTOlhEYSq0
    dq4WjTPyMk16lm6nDYUBH4+ZJfhTnhjbKN73oeZ1ADLxeHykv7+Pbt6wnsC0uU6BEF2H1+NG
    XXUl3ti1F36PExabDQsXLUUyMoF1q5dPFS4ZuZcBnxfpbBYjYxM43tGtmRwAc56Fhg/umAKU
    g4NDGBgcRDqdRnAiiI0b1mKevRZrVi7Lt3rRNbz59h4Qkg9zgDGVM2WEcFE9cMPgzKr6Tqqq
    +ubON9+69WP33hUoCPgR8PvwyYc/Dl3TZgCkmXNruojM3Mlp6mvkv+/zuPBHD92VB9B6frl1
    9fRjNN8C1LiOjIn1yeJ9ihIbhUesQvPQ0aPHnvvP//rRx77xtT8XLKIw89o0Hc1Njfirb38N
    r7z6Og4eOoxP77gKSxa2QNO1qY5V5tA1pTo2rV2Jf/rOn3M//+WzONY9gkcfuY9csn4lCcwA
    09PpC6MTk/jxL1/QE6l0AtMCzIazGJsrYGJ6ljJ7DgqzeRam3kCWLFmyMB5PerPZ3BT4AIBI
    JEHq6kosNptNzuVyCQCTLOF/iDEEuimvzFhn0lyCSk3TMDIyQru6Op+tq6tdbbfbLw0GQ1aX
    y0XnzaujZ87ku8isXbuC2u2W7L59eyI9PT0hSumEEXkSBCGxcePG+fPnN18+PBws6+0dJjMr
    un+7lo35VnkW0eGwWzOZjFHQYlCI+lzsEUNDQ9FFixYF3W7PfKNF37mKhFKpDCjlUVpaClnm
    kE5HYbFYkErpLHeSn/F3sqzB4+Hh8fCQpJAcCk1O2UpTQdUHYCunUaDT6YDf7wMhHAoKAhgd
    nUAikUEoFJ0CJcXFAQgCz6rYzy9ufp50C3IxbCVrWyudPHmy7Yortp0sKyu9rLd3EABBf/8I
    eD44FaafDv2em/0+l5C3cY6NZUCINFUUZ7UKEARZHRgYkE1EmjnlQsb7E28HzRdsyGfOnBlf
    sKD5sYULW5aMjk6URCLxWdfIQ5ZVnDhxGuPjYcybV4tUKovu7iGYi9tmz9tsVsHJkyNcMimj
    pMSHiYksGR1N8nkxc/4s7UmO41BWZqMjIx3Zvr6+iMlOTjKx9iRmdYi7UEBpGDTz5KEAKspK
    S9dffvmlPM9x+TaGoNA1DfPqa7Fm5TLb8Miox2QEjEGJzmJCZilNz+mK6hscGvqLb3z7rz7+
    f//zR4tlWeZHRsdsN1x7ddHjv3wKjz3xVLa7pzdkyvGcQL5QaAyAaLNar77u6q1rPvOJj1sa
    62uQL+zRZrZlnC0VdA6xdCDPJBUGvDzyWmRzLiPBJrYiSVL74SNHldtvvcVisHFTnWAYGOF5
    DjdddzVGx8bQcaoL6zZuRk/PGVxz+UZ4XM4ZoVUCCofdClXVsO9IG870jxjjmJnFLqv4gA+e
    43D9ddfgxuuuwc633sITv/wV7rnzNtht1jyI0nXIsoQDh47A7XLCYhEhy/J072dT6Ju7uHlB
    ZxkVQohMKW1vb+/oOH7i5Matl10CJV9IA0IwCzzqM9sKgppYZ30GC2gGoILAMQWC/Lo80dlD
    M9mcZEphMdIuUmaj8T7lP4wKzaiqqk/+4olfrr70ko2Lr9q29azUC13X4XI6cNN1O3DtVVeA
    58kMhpXO6rJjhPgXzKvHn3z2ESiKApfDlheQ1rSzCnFUTcWTz79Oj7R1ZdnaNsLJE6z4KIG5
    K4SDiXUy2EQjdooVK5a7i4qKl3Z27rdomg5ByGMjjuMQDifQ1FRtq6ystJ45c8b425gJ+Co4
    uyp9TlNMWK45efbZ5wZaWlr+qqWl5XhFRUlrNBq1xWLR+uLiwpqCAn9ufHx08s03D07GYjED
    oI8BGHE6ncnt269e5/MVXNPR0esfHw/hXH2J33vLxjzTwvOCYLXabCY7aWW2krwTEHmvR3t7
    e3Lp0qWnSkuLN46NBc8JPPJMkYqenlG0tNTD5RIxMBBERUUNYycxg3k1wLAo8vD7Ke3p6cum
    05m0yU5m5zIS9245lAbjNzo6gfHxMLxeLy65ZB3OnBlAMpmd0kMlhIPDYYfVmrfzPC9gZg9v
    Mntf/UDy5Rn4Urq7uycvu+yyXVVVFRsHB0fFaWYO53VYzge4zgaXZhCaP51OAZnMpBwKhY3K
    7Qxmto2+oI52AFRKaea1114/cNdddz2zZMnCj+/adUCYzbQaDtbkZAyRSJuJ7efPC5I5Ls/E
    9vSEMDQUh64TZm/OblCQ7/9thcWS1A4fPhRTVTXMSDUj9SaK6Sr2Cy/KmQ302ASl+WRmXHXp
    JZtXrFi21CTFkzfqTocdN123nX/9zV2V0Vjc0Fs0RDJFzJHW5HsMw7WlM5l/7untm498UVHj
    iy+/uurp516UNU0LmrwOA1COAnD5fJ6bH7j7tuWP3H8HF/B5oWnaOVsznkvP0igoMG9yAs/B
    53ERk2EUmWDxFKCcCx0vSumRPXv2jfb19dc2zqs/ByjJX3vz/AY8/LG78Hff+094fT5Eg6NY
    v3o5E2mflgPPd7cgiMWTePa1fXomJ5n7sidMi2yu2J735iwY0lCajqyUgygIKCwowMREEALP
    QVXzYvuapsLlciKXkzAyOg5KdXhcTogCB+jTauaz4jrkYhhPw8AbYucAgpOhyb279+xdd8mm
    9Xwe9M8CjzOKqszs5Cz20twhxtR60FAwSGUy6DozQGkeOKqzwqjvWyx7VmtMozNE9+jY2GM/
    +M8f1i9bsthVVBiADmpiHac7S3E8mdkYgJ6jkIz9TNd1WEUBFoHLr0cjLQAzC3GOtZ/GE8+/
    pqmqFmPAbIJFHiZYtCQ116FjU7RhOiJAKf7P//k/60KhyKbh4VFibq9m5OeFQimxtbW1rL+/
    36mqqtFJRTBYz4ttJ42QPaVU6ejoGO7o6HgWwFEAZdXV1U11dXUNg4MD2YGBwbSu65Ip4jTm
    8/lS119//ZWEiDccPXrKGYslZwDJ87VkPB9DOd2GEQA4YrVaDYfbeBXmwlYCQCgUktLp1J7q
    6qo7OzvPuGRZm3Ed0x1MOIyMhGC12lBR4YbVakU6TZFIZGZs7EbYMd9qkkMqFda6ujoTlFKz
    vUzNsSPznveGfIccDslkCjZbPkiWlw0SQAjPHD0Kp9MOjuOgqpiSEJptGskHVPbNUtc0ANmJ
    iYkDJSVFQZfLVZFK5c6ZKnF+dvLsrjDvBDCdTh7hcFDJZrNGrUeKdbPLzIXjzbqn5TKZTPTw
    4UNPrVmzbnVNTeWSvr6RWSLzM1uYTq+b9wacVZUyCSzunADUYuFRWirQEyeOpCYnJ8OmCO1s
    x3uGDbqg1ovmnEQj50aW5daG+rpPfuy+u11ulwuqIk+1WjMAy6b1a3Dt9q3+nzz2q2ZK6Sil
    tNB0keq5FtZF6E+rEELSLM+HGjlU2VyujxlumDZXI3/SW1xYcPejn7h/+X2330RsVkuefT0P
    UKRntWY8B7ikeSFqp8MOnK3fOJcajhRAz6nOztd+/vgvPv4nX/wcx5v7DNNpokNRVPzm9bdQ
    XVuPnjPduPLS9fC43dBUVmxuEqVWZBlnBkcxGY5KmK7WjyKvyWWuePvgjCTLf3z+xZfw/Asv
    IpvNIpfL4mvf+k6+CEfTQDgCUApZltHZ1Y0XXnkdVquIj991C5YubGYION8pJ79oL04Y53zg
    gxCS0TT94MFDh6PhcLiwIOA/q5jKyO+kHDGyPUF1AsJxrI4//z3deNVwNigDRSQax+DohNFa
    ModpKZb0HLImGvJFPklK6Rtvvb1787PPPX/lx++7h2AWWDS0PwmQHydweUzI5dlZyuXvk3Km
    74OC6oT1CzAxmPo0C59MZfC/v3qJDo9Npkzs5Dimu//E3m8O1HtIZ5jxvQcffKCgsLDo4VOn
    ustTqcxUez/z5jYwECSrV88vXbiwtfbYseOjlNIAIcSH6fztswqH5rjdIlhqj4R8n/MI+152
    cHAwMTg42M2iKmDXkgYQ9fl82Ztvvvl6WaY3Hj9+ypFOzwz/vpeK7rO7dEyDMkpBrFaL2Uaa
    Q99zAVT0o0ePHlq3bsPBhoa6yzo7e84JLoxuPxUVhZicHERFRTU6O0NThSAzcy858DwPh0Ol
    x4+3ZYLB4AxbibmVPXoX5jnvJudbLxZjyZKFsFjy+ZMrVy5iLDI/9TyMsSss9IEQYHAwjJGR
    2Ay/2iAXPgg7aXK+dUqp3Nl56swVV1R3FBUVVqTTI2eBJzOgf+fT7Cycm+Gz2UAnJyeMlC5D
    jSWDC+xSZb4n9j7po0ePdTc3N/+ypaVp3sRExClJ6jmZRNO6mCHjlJ+H5xJcmd2D/WyWtrjY
    hkRiRG5vb4/MspOG423kbM8A0MIcALOpxS/LcqXH4/nyQw8+sHT9urXQdXUGA2awd06HHZ9+
    6D6+t39w/q69B2MA4pTSOCvwMTavGaByLlouztq0jaIDjhCiAkhTSuPswdnZ0zZYlQQAb0HA
    f99nP/XA8vvvuIkIjAmZauWm67OA4rmLcmYASvY3hCOwigJh4FGYBSa5OdwYUpIk/eJ/f/bY
    2nVrVrVuuWwzoE0DSYK8RmdXTy/aOs9g8dLliAZHsXndqvzPWULDVEtCAkRicaRSGU3T9Dgz
    joYUVOR8XszFD4fkr3NB03w47DY47DaIAg9B4DExMYE3dr6Nm66/Gg6bDVTX8q3fVCXfg7e4
    gIX6pnMoeZJXnziPgtDF0IwzinO6Oru6z5w501NYtHolKAE4QqDqFJKUQzQWQzQaRTQSRSQa
    RSqVhiTloKoaCKEQeR42mwVelxM+jwselwMetwM2iwDOIGEpMDI+iUgsocCkDcvWYu5CcoJm
    sZTU5KCNJlOpXz3+xK9WbNtyWVF5WSko8iFCqhNIsoJ4PI5wJIJwOIJQKIx4IoFMJgNJkvKh
    bJaKYBEF2G1WeFwOBHwe+Dxu+DxOuJx2CDwHwhHoWv66dx88bi7ECbG1PsaMZNiUA6XPNTij
    zDGhlOLyyy+3lpaWfay7u/e606d7yLRRnz45jkM6nUNf36R12bJVi5LJZLCnpzdCKY0xRkQ1
    san6RbKTZluZYuy5BCDJwKXb5HzLALJ2u12++eabr9c0csvx452OdDpn6vlN3lNrxnMLm8+Q
    7SEcx/Mmezmn+psA9MOHj0zU1zf8dNGiltZgMFpsFBHN3sDLygpBaQ4WiwXpNEUslj5LssW4
    bptNwPDwgHLkyOGoruuRWbYy+UFE52bu3xySyTROn+6FrueBJuFzy44AAC2ISURBVKV5kfbG
    xjoMDU0gk5GnRPc5Li92Lsv6DLxo6uVN8pJEH6zz3dPTG96yRT1UWlp8+eDgOG8GURyXB/12
    ux02mxU2mw1WqwWCILB5me+Xouv5dApVBTsJCwlPA01R5MHzCo1EIjmTrTQA5RRemYPW0Boh
    JEcpTezbt//1q67avrW6uvKSM2cGYG4SwHEcLJZ8AZXNZoPDYYPVaoXFIprubxpcahqgaRSK
    okOWKVQ1f7/5oPI0wLTZRPh8mv7mm4cS2WzWXD9iqPGcNy3oQnQozV8TTdNqnU7n1x74+Mdu
    euiBj/F5zTh5qmoWps7JuqahobYaX/8/n7V+7Tv/sGr3/sNG7oDMEDpnuuipYpw5NJZGCMoA
    OzkACZZoau5PqQKQKKVOp8PxqYfuvX3tPbfdQIz8s7PzJc9mKs9V2X2WNqU+tQpnC4LPpUix
    zvQsTw8MDv7kH//5e39aV1Plra+tRi4nIZ1KI5vNIJfN4olfP4fqmjq0t7ejqbYMwckwJkNh
    WEQeAs9B4DmIPA+7VcTQ6DiykiQxL9swkqHzJe1ezBDI9GLOg9358xvROK+etfvUQQhFWxuH
    4ZFRVJaXo7DAB521YMyPg5ZvyaipU9pDBKxgJ5+9iHO4tbN7iV+w522an8GJYPDY0WMn1qxY
    upgMj4ygo6MTHZ2dGBweQSKRhKIoU0ylwBubcZ7J06kGVckX8eRbMhLYbBaUFPrRUFOBproq
    VJUVYWwihGQ6Y8jaZE0ed26O0xXM+dIHjxw7tve1nW9ee99dt5NYLIaurm4cP9mGnt4+hCMR
    yLKMvLh0/go0No7TED5/3/niomlxfoHn4PW4UVNZigXzatFYWwmA4skXXqfJdCbJ5mnQFOqe
    xNznTk7NS8Nu6bqOdevW2desWXPX6Oj4F/ftO+TK5SQIgmVGrpax9PMh1SgcDof/kksu32y3
    O6S2trY0pTTH7CQxgeCL6XxrpmeTZdEHowe1yNguFYB22223Xi4I1vuOHWt3ZTK5WfIl5Lwd
    dt7p57O/ZwIqvGnHnxNbydagRimVXnnllb233XbbU8uXL3pg794joixrEAQeFosVomiBzWZD
    TU0xxsf7UVs7D0NDCbjdjqnLycvTcGzD5mC1cvTkyZOZeDxu2MoQs5VzqizwLuzrjFS1VCqN
    bFYydWUSYLfb4XLZIUkK8qkKgimnkp/6+twMJTmn9vBc2chzMHqaqqrZWCx2tKgoELPZbAUc
    xyEQ8KOw0A+v1w2HwzbV3SkPHqkJDHNTwNOYq5SZeUWhyGR0pNM6slmwzjlpNZFITtlJQkjW
    5Hjrc3RPU853X1/fcCwWeW7evJqVQ0PjTkoBr9eDoqICBAJeuFxOCAI/o32rmV09OzuLTEXc
    KAU0DcjlNKRSKlIpDZIEFBVZEQyeyfX3D8xmJ4PMGZ9RtGgeU2EOBlaklK4qKCj48oMPfHzb
    Hz36h4LdboMk5fLab1q+nzfV1Wk9QPZ1S9M8/PXXvmj7lx/8eN3zr+wsSKUzPkqpwHIHLchX
    M2ZwcarfDGpZM21y5nAzq8+gPEfIIzu2XXbVA3ffwlktgkkPT58FFM+u7DZXfJ+VV2mEwTkC
    VdNme3Vzmodi5EIRQtKU0lfffHvX/D/7yjfu3LBujX1waAiRSBQ5KYdQKIycouH662/AM888
    jZPHeLz86psoLAjkAQvPQRR4WEURbpcDx9o6qabpRsGAUTEbxnS1rHIxQnHnCiPabDZks1mq
    qiqSySREnoMsS9BUBZqa76meSqXQOK8e48Egksk4K0yZbrlJTGL2HAMnsqKyRPYZgP8s9vhc
    13QBoFIjhKQVRTn29HMv5PoHBuwDg0PQNQ02m5WBXLC5lJ/OiqIgHk8gGo+jpKgQNqsFHJkW
    IrZYBHCEYHwihJ7+YTz36i4UBbwYmwhB13Qj1G0O4Uhz4XXPBspMa3Minc68+qunntmSSqac
    R44dQzabg9vlhCgKsOTTZ1jRUL7dosDzEC0W8DwHXdehqSpkRYEs5x0AjhCIogC30wlR4NE3
    OIIjJzpAQOB02HDo+CnJxEyOAhg1GcnM+Yzkhc5RVtyCO+64o6C2tvaBsbHxL+zcuaskGAxB
    ECxTTZlMwgszwpO9vZOE0pKSdes2X11ZWeHev38/H43GAEBgjlvqIoD/2Tmw1GQrCftMHnmJ
    OAKA3nLLLfWBQOGDR450FucFsfn3JAV0vorv84XBOY5Qplwx5wwYA12UECLH4/HYW2+99YtL
    L72sbv36FZcnEmnB6/XAarWA53m43U7EYpOwWCwoKiqC1+uHomiQJHUKZBmC4ZoGuFygwWAw
    ZYrkGI63oSWqzbWtNAM5A1BKUo7ZgzxzJ4qWqZzJfKcmK6LRBERRgMfjMjGURvtPfkYrUEoN
    MGbyhs7ulHaWjZyje6UAlIGBge7ly1cNrVmzuMBms1Kr1UI0DabuPtMHzxvpB1aoqg5Z1kzM
    n3HmdW79fgFFRRw0jYBSDrlcWM1ms+lz2Mq5rhOY6l3e1ta2d9OmS7sWLWpa7nQ64HY7oao6
    NE1nOebTWXEcl2cjdT0vq2aOBPA8mXGfqkqh63mgXFJiQUVFXnvTatX0V19tj6mqGkS+wM7Q
    Cg/PJolmj59wgZPUAeCOha2tX/rDP/yDpttuvRkvvPAC3nrrbYbGzgZWBtjSdQ1rVy7Btdsu
    xde/+AfCyiUtLT/5xdPlbZ099bquP00I6WHgLoppHTxtDo2k2fvWTGEbblYoc3VLc+O9n/jY
    nTaf1z1VgGMOaRMAuhlc4mzZILxDjqWmUmSzOXqxmDyT56Mir5kVVFXtJ888/6J+5NjxOz/3
    6Ced1169DR6XEz/6yWP4/8r70uC4riu97761V6ABEBsBgju4kyIFiZREiRJlWaZk2VZkj6MZ
    eRLPTOQsPxJXPD9Sk6RmylNxqpL4j5OaZFLjSlwjT3nssi1Lli1Tli3REkmbm8QFJEACIPal
    9+X1W+/Nj3dv9+smwAWbHM+retWNJoju0/fec76zfUeLNuL8+fMolYpoW78OtmXj9z5zDDt7
    N6NYLKJYKqFQKODNt3+F31y47HCFWEl3cyW5YuluoRhlWeYRLH94fblcjgDYMz4+0fAf/uNf
    QFbkCqF5JUrOI0X9V69V6YFQS/AeHEcpE4LJ6Vn4ERkW4+UQYfijCcPc8XHq5VyqRy4axwCc
    P3PuwnFZlo7GY9EYZf7brGluRvfaDqztaENTogGxSBiOY+N//p+/w+DQCD751OM4dvQRGOUy
    CoUiUpkMpmbmMDE1i7lUBo7jQlcVTM2k2KnzVxxW20wlmtFWIgVH4dcvG4yxM2+/c+L9y/1X
    2z73/Ke2GOVydGZ2FpFQCD3rurB5Qw/WdrYj0RhHPBpBSNcAxlAsFhEO6ZAlwHYcGKUS8vki
    kukMRiemMHxzDJMzcyAANvV0Y3xqxvvOa2/ZnkcrHcjw6Xem+J4troSsgjgfgPTlL395h6qq
    /3Zw8PrvnTz5m6gsK9ixo7eSShSPvrFWatKMhuFgbCyDcpk2b9rUe+z559etv3jxwo/Pnj13
    ilKq86llol7ZWSE9Us+84QWBwr59e7UtW7b8oxs3xu+fnk7WgMT5Cv/vnnuyFlz6aTzAskxa
    93mWU1cyxpgDwLh27doIY+x/HDlyxG5vb/3EBx8MKP4kEgn79m3B5OQktm7dCsYAx/FAKXD1
    6hQch0FVfYJwXdewcWMTJiYGzGQyOV9p0LLvv3ogGXBs1A0bNm4Ph0MdDzywH7btVCJzwZo6
    SZLQ0tJUV2dHaqYFBdP6mqbCdTWtra0tNjExEQYQZoxFCCERADqfNX8Lx+ZSo+m8/8G7fPnK
    zM6du15bs6ap2bKcdaJhyHVdGIaFUsmGZbkVENXR0YTt2zthmiYGBpKglPC1UhEOawiHVYTD
    Pgjz/SXAMKa84eGBkm3b8zVTuctpq7loHgCzv79/7NChh97avn29Mjo63ZzPl7rj8SgURUG5
    bCGTyaBYtGBZLhyHViYeKYpSAfySJENRlIqMkYiGWExHNKpBUXwgmUzmWD4/bMuyZ4yPTwin
    ewzABCFkDtVphgt2sitLEJwQQp45cuTI1/7yq19tO/jgA6DUxbbeXqiKzKd1+CkqP+UYfE5B
    qYf1XR1glCIei+Dzn34aB3ZvS/y3v/q/z/zk7fccxtiPRPkaql2iK90BV9+NGQ7p+ouf/dQn
    Nu3Ytrmme7TK/0eRzRcQDmlQZXmeEYuUp/0XoBLiTTDZfAFY5gkz8xRJV1KOhJBxxthriqIc
    fuTgA9v379uDi5evYC6dxf2bt2HkxgB6N29CJBxCxsuhpakRWzf1+KTmlIIwhsGhEQAVwmah
    KDP8eU0KZz6OsqVG8fgjoZTqrutukWX5xUMHH/zDP/6jf9rcu2VLpW5S4sExgiCFTrWbuLIn
    OcejiKS7ngPXdlAoFvHqT9+O/+CNn2/PF0plABnGWIoQkuP1LgZXmEFy4iVHlPnfGnFd92+H
    R0Yf/ON/8vuxw4cewLquTjTGY5ClKiA+c+4D/M0r30Uul8czHzuC9359HgDFFz77LBpim3it
    qAvHtpHLFzAxNYPzl67hb3/wM1i2kw/UcQUB5bKl4OocORHpGvM872uZbPa+nx5/+9+98Oln
    o7//uc9g4/p1iEUjKBQKMMtlJBrjkCUC13XxozeO481fvocDu7fjD154FrFIGI2xCHRNxdr2
    FjzywF64joN0Nodr14fxyg9+il/9+oLteTTJFeQUgPFAdDKHJdJ9LIiceWTyK1/5yh7Hcb7x
    /vsnD3/wwUViWQ527tyJnp5u1JMWB6dbiJ9TqRKKxTzSaQOGwbR16xL39fUd7lQUJXTy5KkT
    PKtDArqDLjPAuoM9B5544okthmF+bmRkUvEpc24FhpqmorExhlyuxFOk997xraoKKPUo77Kl
    dTdbLl3JwY/JGCsMDAwMtLS0vNbbu/Oh8fGZFtN0sGFDN0yzCEVR4DgKymUbjuNBln2+P8Nw
    IUkuJMmBothobQ2z69cHS5Zl5blhzhBCsqgd1bdsUbzg/+ERZOWLX/xiV1tb23P5fOFLp079
    Zuvk5EwFfPgATK7sPbH/RCSy1smp/uxHNmUoiorW1pj+1FPPbL127ZI5MHAtm0qlU4yxLKqT
    ZKxAVnBZCNwF+Mpms8b3v//9733+8/+4r1j0eq5fv4l8vgzLcnlziv+5Q6EQduzoRmtrA6am
    8ohGdWza1IyRkRwyGauyZrKsQFEUhMMq4nENnZ0a5uYm7JMnTyYppfmAniwGAyfLHF1mABzX
    9YpvvvnmdwCceeyxx/8loHYPDIwhkynANF1Ikgxd10Ap4UTuMhKJKO6/33d0PvhgFKWSA8/z
    60IdhyGXc0CICVVVEI1q6OyMormZeW+/fSpfLBZmGWOT3PGe4NHJWxpxlhVQAggnEonn//mX
    vtT2yOHDcG0TYAz79uzG3t07wTye4qZu5TnjP4tUuOc6cF3bT2lRit5NPfjCC89op85+eH86
    m+/ndZUWr1MoL5eBm2/R6w+xryicrdu2bvrEx594VJIAeJUmG1qp1Uum0vhf3/p7vPTCs+jp
    7qghkl6ws5uDTEHbYlkmpufSwRR88KZLVZB10TLRaJQFMGsYhnny12cwNjGBnx5/m61bv5Gc
    PXsWxUIekUgYN4ZvIhoO4Re/OoVL/QPQVQXhsI5YJIz+gWFQyswAmMwSQnLzedzzGabFeqf8
    /8mMsRYAO2VZ+vjOHTuf/dSnntv50h+8qPhTWDhQFETgvI6QUZc7Ni7/2f8dxsTv1TpA/hp6
    2LVtk/TAvp1t33v9rccuXLq2IZsvbGWMnQNwiTE2yjvaS1y5LGn8WwAwe4QQy3XdEUrp1KMP
    H+zou28PHNsGpS48nq4nYJiZm8OFi/04dGAvDvXtxeVrgzhx6iye/8QTiIVD8DwXjHqQCEFT
    QwzNjVE0N8bwd68e9xhjpQCYFIrSWO6arjrv2+XNJSPlshk1DGP2qSePtN5/3x64joOTvz6D
    73z/daQzORx78jBeeO5plMtlHH/nfYRizfjl+2dw7MlH0NG2Bsd/+T6+/9NfIh6N4tNPPYyH
    7t+DlqYGHDqwG8ffPc0KRcNElcNxsi6FU0MTtNylGc8995wmy/Ifnjp16vDJk6cIIENRNAwO
    DmFoaByyrFZuv1ZNPK99XZJUSJIM12W4ebMAQpT2rVu3P3n+/IUp0zTBv08HVd7XZV23211N
    TQkSDkeeGBqa2FIoGPPSAwEEra0JdHSswZUrI5wkuz4qeftUt4iEOY7p8Sk5802ZWWptYUVs
    QogoEZJ1XZsLh0Ned3c7AAmbN69lY2M3yJYtW7mhpmhoCKNcdrBhQytcl1ZS3YCMcFhhnueJ
    0iDR2Z3nZ88OOt71unIJ2Q7y0ksvNba2tm7VNO3xUqn03OXL/X0XL14OT0+LST5VYOg/VwKA
    caHnt75GiIN83iGtrfGGbdsOPrhjx97uQiHdm06nzs3NzX14/fqNIcMwMpxZxcQykPAH9CQF
    YKfT6bTrOv25XPHp8fE5qVr7KVWAsKLIaGmJQZIkJJMlAAQdHXHMzVkVuiex/yglKJcZLMtD
    Q4PHMpmUSSktBvRkfiUyOfVRSkKIOTo6Og2AKYp0Y3Y2/9jY2IwkSTLC4TD6+rZj27ZNuHFj
    FGfPDsG2KdasacCjj+6H53kYHU2hVMojEtGwf/86dHa2oL9/EsPDeXgeUCh4CIddhEI517LM
    AmNsvrKgXH3Gcb79qCzhAOq6rjd1dHYiyK1bifBQrwoqAzWUjEd/KPUNXE3hEKOIRcNQFSUO
    oJujYgFWFKzgVf/lOI4DAIcePLB3fffadt8g1wBFBkkCTp/7EOlMFk18Kkc18lU3FYfOX2NJ
    AGRyBUzOJL3AIRP3oqcDVdPcMih1CfxudjlQ30IBmKlUevwv/tN/aXI9r6G7e13jiy++SF55
    5RUU8nk/ZC75yvLKwHXe1U78WjZVYZ7/ghmotxM8hmCMCaJhEig0pgHFv+gDyPffpqampv/a
    19d38PDhR1rXdXdLa1qaMTIyguuD12sijZ7n8k5un3tScFB6Amx6NDB73V83/wvzu6olya+l
    jMcieOGZo+GnjxzcNjoxteVi/41jH/YPXs0XjW8wxgZ5cX2OHwZzGSLMgm9wLpPJ3rgxNLL/
    wN5dvLyi2pkPxvDYQw/iz/+U4Ft//0N870c/w+b13fijFz+DjtbmiuMiUvuU+RHmqZkk0rmC
    i+qIxTyAAic0X5GoXV0ESNQuj6Qz2atXrw3uvG/3TkKphyv9A7g6OIR4LMY/PxAO6fjE0Ufw
    xlsn8OThB9CcaAQYheu5KJUtJDN5DAzdxMH9uwBGkcsXMTg8Rrl8WXBiXt7VXd+IsyJydnV1
    RS3L2jk6Oko8j0FVq3yKgt+PEHGDRxh4C1ilnuvWjuF83kEioXdGIuGNpmmKSIlIw5lYxevI
    kSNhz6MPJZNZzY9OyrcAQ1mW0d7ejGy2yMffSXed8g6C0khEQ6GQsg3DEPomqC+Xw6hX6gCZ
    6KYBXEqZoWnK8IEDvVYoFErMzs7ENE1DW1uroOEBIQS6riCRCMHzPD6T3Wa27aBUSopmDgGm
    bJ7SBC+bUYAK4TjlTZRLiuS9/PLLiaampr+cm5v75OjoaGc6nVFN04Ku69i4cX1N1FGW6yOP
    cl3tpBQAW/WUPNXGIwBIJm1V10PrdX1dz+bNG57YtYtNNjae/qsTJ06c5iwF2UCmwlsGXcIY
    Yy5jrFQuG9caG2NlVVWj4twEKadM08OFC2PYsqUdPT1NkGUJly7NIZWyUM8FKx4VRYIsuyyd
    TgdHIue5nqxxCFagTE2wLBgAUoVC4UIi0VjWNC3qeQzhsI7u7ja0t7fg5s1JiKabVKqEd989
    D8aATMasyKbrKtraEpiby2JsrAjH8X8/FlMwO5s0bdvOIcDPO0+N+W0jsUsBaeVsNnvpjTfe
    eHrn9m1SLBoORHY4aKwBkl51ekUAaIrbdWxkczn8+K1fsXQ2V4bfaR1FoJMQq3spsixv2bZl
    g6apMlzXrR2dCIZisYx3T57B4YP7EY+GOeisq5Gkt5/lTSSC6yPjmJ5L2wFgZgYA2qIOnD8T
    mCqMudsA7IZPlm6glluwxaN0TtM0CsfRH374YdLffwWff/5ZdHa0YWj4JtKZLNavW4v21hYY
    RhnJVBrTs3MYm5jG6PgkLDunA2gE0MpTRFEAO7h2UXl6+BKv8wrKtlSv7sCxY8c+/vWvfz0U
    j0XxzW9+EydOnAhskmrFAEe1ED2IlUeRJCQMhNMh+aUalJdl+NOPPAFKXX8pPv30Y/jcJ4/K
    U9OzDV/56jd2nL5wZTcAg48c9QLyLUcJhgOgYJTLN6dnZhghfrW/67owyyYMw+/Oty0b3Ws7
    0NKUwE9+/i7+1Rc/D11TMTo5DU2REdJV6KoCWYBkAqSzeZQMM0hkXggoyWWjwVjA+67wrQFI
    GuXylcHrQ59hjMqSJOGzn34Gu7Zvgaaq6F7bjnQ6A8exsX/PDvRuWo+QpiKbzUGWJTzctw89
    a9thmia2blznd30zgnQ2i8npOY/LlkXtiMUMFuBSW85rYmLC6e3tTTU3N2NsbBJ1VKLCJAYe
    a2+RjhSPwuDF4xoMI8VKJSMBoAlADLWjW1ftisfjGqWsybbdAFlyLThsbIwjFNIxNDRTASX3
    0owjfj8WU9nw8ETZ85vIhDNrYokcgM8++2yot7f3PkmSNtu2jWKxWMxk0kYmk7VLpSJTFKXV
    MIxpVdXWMca0yclJdHZ249KlUTDGkEjEoSgy8nkTtu1BVVWEQio0TeIzpRm6utZGZFlulCTS
    HotFnTVrWpvWru3cF483KIRIcjabnTpx4sSV6enpAuf+DDrq9yxba2vrxsnJyedee+21dalU
    Gtu3b0dvby8EP+atYxPn5yqsJWevL0eormVw7dNpB8PDBtE0Nbp5c3hTNBq9j0e8VFRp0ZZ7
    IpVVLpfHGhpiJUVRomI+t6LIfE65TxfEGEE2W8a6dU3I5SyUyy5CIZ99irEqOBbfkSxLYMyk
    pZJRrnPcSqhr5lupMjxOI4RCIT/Q3Nw+p+t61DAs5PNlvPXWWej6JRiGA0mSEA4r8DyGCxdu
    VsB+OKzC84AzZ8Zx+fIsLIvBcaqUj7oOlsmkRPlaGn7DWBK1lFZ3XCtlCUbBM03zu//7r/96
    14Xz5x9ua2sNybIkSYRIvs0OjPRDlYNSUJj44JLC81y4rgPbdtj0bJJevnYj67jecB3woVj9
    izFKC8lUhlHPE0mHCvWRRAiGR8dhWTYOHdgdGHfnczSyQLdwhecx0PRBOLWNaVp45/QHrFAq
    ixFcJVTpCBYNulRVhWVZ7eFw+L8/+EDfwc6Odpim5RQKBTOfzxdLhlHWNS1y8IEDXQ/ev1/7
    7g9fR0NDA2Ymx/H4oaP4xbvv0RPv/9oqlkpOV2eH9uxTR/Snjx4mPV3tkAAYZYMMXB9R3vzF
    e22Xr91oLJvmLkWWaTikyeGQJuuaKruOK394bciamE7+DWPsFQ5W8gEwsejoK4CWnp4epb29
    DYwy/LM/+RM4jg2Iut1KExTv4q4A+8AUFVZb0wtKK5HzYD2lx50jz3XgeS4USQJ4nZhp2QxA
    M/xJSyLill8mgCLqKA3G2MylK1fxox+/ieGbY5ienkE6k0WhUIBl23BdPwVulMtob23B8XdO
    4d2TZyFJPs1TJBxCoiGKNU0JdLW3YH1XG0YnZ+A4riCkDpKZl7E6dE8e/JGMBcbY4PWhYfPC
    h5eiM7OzmJicxuxcEtlcDkbJgFEuw7Z9rlAxKIEQAk2REQ7rvnzxGAZujKCzrQU9Xe0YnZhG
    OpsToFXUrmV5FFmUZbCVApOEELz++uvWyy+//MO+vr5HI5FYT7FoBOrV5AWacaodt0HKFlHX
    FY1qiEQc89Sp81OWZTkIjHNc7YsQgnPnzpnPPNM50tTUwNLpAqlv5AAIwmEdyWQepunUTZy5
    daxh7cSS6vN4PATGyu7Q0FBuHqO+pP26d+/efa7rfWtycmqdbTtUVVWnoaHFaG5uK0qSZHse
    TYyPpzosy5E7OxsRCoXgeTIMo4j29kbPsrJWuUzdWCwRSqepNjGRg2VRyLJMNE1Dc3Nc27Xr
    0e59++gaSSIHZFllhMgyIMuMEUVVJWnTpvasbdv/+dVXX/0FY6wo2B6wyP4BQkhjsViM5HI5
    MEYxMHAdQ0M352kCk2tqJYP7svpvyi3NYtU0ef0+FfyOMo+4M5rJpOMAOgE4jLESLxFaLpqn
    Csm5ZVlz4bBu9PS0Q1EUFo9HSDgc4tyMoi5U8GZ70DQFu3a1otqVT+C6DLYN2DZgmgAhCgjx
    qGma5To9WVqJEpMF7IDNa0VHe3vV2dbWxAbbdhGPRxGJhBEKadB1IaPMo/q1ZDF+CQaD4/i3
    ZTGYJoPnESiKRzOZjBgjKcoyMlxv1mSrbqcrl9KUAwA3M9ns1352/PhBAJsAtHPjGkGVI4zc
    5ksKhpKE15Lj4dYpVDvgVoXLsD6MThk79ZO33p3Yt7O3e+P6bj5vpBrdIgA+c+xxSIQgmUrX
    zMUOprgFwPFr9mpB9ekL/Tj+3jmTp+PSqDa1LKmz3aeGhMEYPZPOZLoSiURbY0M8kkg0xFyn
    vdXzXKJpGhKJRvbueydZNN5Izp07xy5fuui8d/KUmc3l82JD5QvX6dXBG83fffUn7Tu3bY73
    dHWoIV0jEiHQdQ2bN3Tr5bKpMz5k2nM9WjTKVjKdzRtlS3TSRhhjNnyqkyWNSOMp78mf//zn
    +T/7s3/fyKOPBKCEBeZC1zdPoWYUYbUrP1imUPM8UAPLAiMAKf9bqUyODY9NFfnHCgdSV8tJ
    slxpznnzrXcGrvRf28IYkz3Pn7uuKDJURYEi+0okEvLBFaUeXM+DZ9lwXBfTsylYnIZHUxWE
    NJXNpbPUo9QI1APVT8dZESU5T42QDWD0/dNnRg2jvEVRZBVgcB0fzPvTcuCXL9DqkATCI62p
    DINHPUggkGQ/V6xrGlKZHCuVzWDTWC4wvclZDZ3CGMO3v/3tk0ePHv3zbdu2Pu26brfrug2u
    60UppZpfiiIYT+e7XPil5IxRyhilHp2bK9o3blxPjYzcHOXR1twqREoWvC5fvuIdOHDg9fXr
    Oz6mqspW03QCYBE8yiPD8yg6O5sDQFOq6R6ujpCrHy/nR4liMdn74IPTs8lkcpLryyA92ZLS
    joZhJFVVPaPruuY4botpWlFCnLgkyZ2SROA4HgDG1q5dw4rFNOnp6aGMEdsw0uVz597PjYyM
    ZCj1Sj0966U9e/au2b27rc115ahtM5kx31p4HiGKoocBKUwpYQBhhDBPkqjJmFPK5UrD4+Pj
    BrefNpY4/Sebzaba2trmnnjiiYRfwiURwRV5t/OtF5q7XlvbKi1YD+tHB83y5OSUw+XSV0BH
    iovm84Wsrisf7tmzsQUgccG56KtsxkdG+rEe0/QCgwSC9DoSp9/x/5/j2DSXK9m2becDZUG3
    jBVebqe0jm6NAXDT6UxWVaVre/Zs2m1ZTkTUFgcBZD2pfvUcSreMyuSMX8x1826xWCwEdGV+
    MbaALAFwKfAnJbQwxtq599HOU59xVLkc7/Q+rM5wCpJsMWN3mresZwCU2UqEEuaXT2aMtQF4
    qauj7QstzYk2QogGQCXE7z6X/A1I2C0jVGo67GpvShn1qWuY63lsajZt5gqlcQAjAIYBXAdw
    gxAyzEF1Af4c7nsGXQAUxlgTgG0AtgLo4uvTQAjR+b83RKPRg319fYkzZ86MlUqlCVSJjLOB
    +gnKQVMLgAR/rvDfdVAd2VcKRA4KPMUoKAfSANLcqBc4GfJi1oYwxroBPA9gA4A1gc8VDTgz
    t9tvd3qd3eF3hBM0A+AqgJsARgghN3lqJ71Y+erWELxcoV1V1adCIf1fe67XZVlWmAFqc1Oj
    tKalWUo0xEg0EkZI04ii+LxknufCtmyUjDIy+QKbnk2hWDKYoigsEta9XL4047juEIABcRNC
    huB39uVWMhVc16ClMMbamxKNH9u5feufpjO5HclUWiIE6GhvR2d7G9a0NCEei0BTVU5k7sGx
    bZQMA5lsDnOpDKbnUsgXSohHI2hqjLEbNycyuULpJoBr4iaEXAcwysGIW+cgr4ScCmOsEUAr
    IaRbVdUeAGu5rgzqSVK/1WobdRl8UEkrdCJchmm+XhOEEDHJwlhFPQnGmKbreuuTTz75ZGtr
    2zHP89Y6jtPgeW6MUqaJQeV8kooQdR4QzYSxZ5ziqyK353l0YmI8f/HixQnbtie4vhzienKc
    6xgTgc7ze7y0vr6+9q6uru2qqmxmDJ2MsTV8jVTGmBKLRduam1sOZrNZJZVK3Th//nxqdnbW
    cxxHjOfNAigrioLOzo54R0dHSzQabZAkKczRnOensokJoESpV/Q8r1guG6V0OpObmZmdLRQK
    yUB0KMObHA0A3r3IRQjB1q1bmw4dOvQcY2x3qVRqMU1zjW3bTa7rxjh/9EK2mc13LvwvliMQ
    hjocVVm7wBL4BLmFQjE7MDBw1XGcEb5uNwkhE3zNrKVu1UDTEolGo407duzYfeBA37+JxWIP
    K4oclWVJc10qW5YrOQ4lnseIT1RSdWJkmXAH3SegV1UwShlcl9KBgYHC2bOnR6ampgYppUJP
    Cj0yx8H/iutJxhhRFCX+yCMP39fe3vMvWlpaXohGQ6qiyP5wkpIJy3Jg2x5vBpO4bH5mQ9dV
    aJqCUEiGrkscLDOMjY2UL18+Nzk4ODjgOM4At2cD/GzN8P13V3qSLEFICX59YxxAggOXJvj1
    dJE6QFn/fgsZb48vTiXsyhsdBFp2VkNPii5iLksHgO08ArsWQBt/Xa3zsshtvtd6IBIE0VYg
    KjsOYIwQMiZACfeC2GIAJacTiQJoYIyJWqsEgAYeTZMA9CiK8rKiKJZpmif55pH4+wqeNMGM
    L3EPM8KBm8bfTnSXiromEVEWqe1iMIzOAWVxKYCSy7SGOzJr69bmXuefs0X+O+WyJTn4nyKE
    TPG1zAWK7pd6yYSQBsZYJ4CdfC/2AOgiBM0ACRECFfCbrggggQRBCmF+BQej/DMJxy2FKteY
    UPTCOJdWA5TwfSoBiBNCOhljHwdwny8badV1rUHXNF1RFEWWZVmSCKdqZ4Q3GDHPo8xxXGrZ
    NjUtOyjfLJftJnfWhgkhoxyE5RfjqC1Gj/CpMgnOSiCcn2Z+hharJy0ug+B+FUCksBxG+h5t
    gRoAzV2EkG7/PLI2X9eQGtBMasMk82W+aiCMOGs8w5Hn6zrJQfQk36+5xdoHAYrh16E2cB0p
    7FmMr5H80EMP7X/00UdfmpiYGH3ttdfO5fN5cb5LqNblCudb5vYxElhnkboM1pI7gQiQxddP
    AMpsIKJ+z4CS17O3cBvWxXVlB5dLXSUdyQK2RIw6neQ6MgO/OWm59iFhjIUBrOnp6ekNhUJb
    CSHrFUVZpyhKm6IoMVVVVVmWFVmWJUmSZEIkMVaXZw0p8zyXuq5Lbdt2Lct05uaShWQyOc3t
    swgcjHF5sjyFsBrnjDDGQgCaDx482Nfd3X0UQA8hpMPHXyRMiKRKki8c4ePeSCAdRCmljHnU
    81zPdW3PNC1ndHQ019/fP851pdCTI9xRTd8LYF5KU45IURfBp1/wL1eEte+U8p4vQikASGWm
    MGpHwK1qyjsAboc5vcMsNwgCkJE7AMk7yS1GLJVQHQc3i1rOp6Vyq4nOajegzESoXgYw5bru
    91zXFQXHBIFmCfjFuWLGusQYCwLKUGAPBUniaeB9bVRnndqcq3E5OMhcXqic5UDXDuy/5agn
    Y3e5Z8uoMhGIAmYHPu0Ilsn7ZnwsaQ7AIGMsz5VZK2NoAliIMSgAEx380m3OXv05SwayAZmP
    InUqOhm5XO/waGk3Y2ytaVqtpmlFuQEMjtoj8wCv4EQX0RkpwP4k5qEJWgXZhJ4siFnYjLEc
    P+ehhSOUd8zkOABKPPUmbmO19STfnJSfxQJjbI7TuIjvPgYw5S4d79vJzALOaToAomumzCxx
    D1IO3NxA1kU0F8rDw8ODiqJ8u7+/P53P50XTnR3IxKQJIYJaRebGX+hKPeDoBtkugvrS4t+b
    BZ/8f9G6MjhCkoNwle+NAgfJ8irpSAScH6Ej81gBqh2ucz3GmDE6OjrG3yPJgVILl1sEgoJ6
    UrqNfRbOWwbVca1JfASz18WwC8aYefr06aunT5/OcEehS5KkDkmSGgghOiFEqWYFIFUTqAx+
    gsMHltQnyg1mI0W24xaaoLu1Y0ul4nEDxsngf0/c91IjweoOWj2/mIfVrw1iAcAsiY3KjZIw
    cAQLp/XJHQ5fEFCWucHJ8YVdrrnClbXhwESAiHDAQCcBTHHlpwRAYJFPvBFRUsqDgzo3hGH4
    ExBUTkdE6uSit3EQlloTywIGVRjpPFfe2iIAJVvk74g9Uub8aqKpatloJMTkDs4xWOLRSga/
    /CPDMwTBuqR60DXfnqOBfZfjkZBKyg4rP0BgvsvhjkuWRyyF8Zvm63o7QFkPloV8lZGgnGUg
    6Kitph5xAmfd4N+1VifTnQBWveMtHERxO4Fo16peAjTzpjtBmZXnEbIQFi5/Ivdw5mjg+xPl
    NMVlPG8C+JcCVC0lVFlG5Onp6dT09PRN/pokjDGXW9RzCrowmZdIhQO6smLo5zmPQu/W68pF
    OT9c37v8TBHusAngG1pFHVmxO/yziLKoZZ+iFhhZK7iRwSOgIqodWQBQ3k6X1GRNuU3MrLZj
    GgwQcectz2VgAMqU0hSlNBZwXKS7tAPBbIfo7k4tFjCTJQq30N8ji/j7bIHHBdIhK75w4Auj
    8kUSICoUMOB3kpPcAVAisKDz0WDcVWfVHWQIzlUVc8rVwGs6gCiPPEqojsYro9qsIQCg+Fvi
    b6iYZ15rnWEXDoKDWgJiupSmHP6eSp1MyiIi40tRpsFon1sn57LVHwbkFXsxOP4xhNpmIOkO
    gCvoyFgB41WqywbQ1SovCexTlcsV5dGEGGqpw+S6iMKdALMZiN4FO4IrwGulU951lxR4rFf2
    5B51ZP1ErZrJWh+BrgyeRy1wq4Fo2FLtglcHom3UTl5Z9KCEOl2pzKMrRYlXhHPsShy8CBCY
    R21WiQSCK0HdNB+gDGZ33LqbLsYGBMoKlDp9ra6yjsRtdOSynsE6mYNgvgLq69bhToCSLQD0
    BSC2V0OPLCCfPo+eFAEV+S5lC0ZggxybIttR4ei9W/nIMivMlQz3rqbHXW8EpDqvRr5DdPJe
    DiFDbfrDqzMUS1WStwP8UsAACIAcNMgLTTOoJ867GwfhFqWzREC5VAdmOSNQ9eu5rPu2zuBJ
    dYZPXkCBkNs4NcE19hbKBqwyoKw36EEjqNwGTJIFnNGgI+PURfDoaumV1dSRvwW6sn7/3a70
    gtzj2cJtQPSSgPRtdGW9nguCQzJPlMddQFfeST/dMZiySEB5L/p6tfTkLTIuM6BEYO8tpCel
    O+jJ+UDXQlnTj0JXkjoHbj7nZz752G2CC/V6clHODPltU6KrqQx/2wzBcn8nv00y/EOSawUV
    5e8cMPko1/IjBF6/EzL9tp3Hpcj+u6Yr/6HpyN91Pfn/i678fx9hAh9/r+iSAAAAAElFTkSu
    QmCC
  }
  set pieceImageData(Fantasy,60) {
    iVBORw0KGgoAAAANSUhEUgAAAtAAAAA7CAYAAABSS730AAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAALQAAAAOwCswV83AACAAElEQVR42uy9d3wc
    13k1fO7MbK/AovdKECxg752ieqVkS5ZkSbYlW+524rglTrETO+VNHDuJv7x27CS2IyeWZVmy
    itUliqTYwQqi97JYYHufer8/9g4wWIKyZIKUnFfz+y0BAgS4M/fe5557nvOch+C9673rveu9
    673rvet/0UUIeVe8D0rpu+5+r9R7eu967/rffgnvPYL3rveu9673roUBX++Bk3fvsP6Wv//O
    Q/5b/n6lL26e+6OGj+9N0Peu9+LzewD693tQ39to37v+Xwhcv2/z/E3ukeSBD/IeGPm9ut5t
    QPdyXdp7Q/1efP5/AYfkvXdCCPmt8ZkQsqD3/B6AvjwTlbyFAP7/7Cnw3XByfO/63cebUvqm
    f/+9R1q5+yHscwJAIIRwbF1z7PsaAJUBFg3vsXvvurlqjBmGrxknKqcPc16cpv+vrOX34vP/
    vjEVBAGKosyMyf+m2GyYb8SAtTgAPPsaz+6ZGuKzernWNXlv6i1MAGKbKscGkWMbMMkLyqqR
    Ifh9DTr6/XMcB027OOHhdDqRTCbfNOguXboUHR0dv3WRv1ue1aUGo9/HMddP7ezejYGLzBND
    9Lmu/R7fL6GUWgA0AagH4GX3EwYwSghJApAAZAGI7HMVgPYekHhXAmiiz2Gz2czfeeedReFw
    mD777LNJ/XuGg5BsOBD93q7nNztEvNl7z1vrv/V3vJuexf+LsXm+scuLy8Qwx43xmeZu+ffz
    np1OJ3/XXXeVejyeFgClsiybE4lEsqenZ+Lo0aPTiqJIhBCRxWjJuK4X8p7fA9CXsMmyCauf
    gEAptQK4iW24zxFCEuwZawAUttnKeItptndzcNaDLPucw6z2Tn/vNIdDKGXf49gBA3kblvFZ
    EMP3jbq934sAbdx4LoWVfbeNex57xwFwAPABcAMIEkKyhjmusDmu/r4GaDant5SUlPzT6tWr
    m9wul5DNZui5c+fiA4NDjwB4nIHoBIAkgBRb29r/pg35fwGQ4HRGin1O7r777tra2tpvapqm
    nD9//jtPP/30CAvnKttoM/qB6K2u4XfjOM+TqtZjsPFrmmGBcnn/RgdYF8RoQ/x/18ToK5UR
    e5fvyQQA8Xq95k2bNhYUFBT4wuGI+MILL4Q1TaNsjuvxWfl9PvA//PDDnsLCwr+Znp5+XzQa
    tRHCEZfLLblczlO/+c2z3+7p6Rk3xOgUW9dK/sH4Uu//PQnH7z5ZzZTSJhZsp1gwaamoqPjz
    ioqK6vb29kZN034MQGITN4tZXY6Ct5BSmI9FeDc9Bkqpk1K6GIAJwCADEXp6W2YLehWAFQB6
    KaVnCSGK4RlIlFKV/TwBQBn7B8NiV3UwttD6pUthLdj71E/6nH6mMtyH8aT/tlL8F2N/3on7
    pJQaNyQzgGssFsttbqdjRTQeN8my8g0Ap9g9ZthLH7N3/Tq+2P0CKFi+fHnDv/7rv7qLfYWI
    xyL4whe/ZBsYHCoHUE4pjbAxV9jafisg5r3rym6yZT6f755sNjuxf//+Q8eOHUs6nc5N4+Pj
    10ciEVtDQwNXVVX13bGxsSQAhVKaNsRoMssDzMvezQDHd/E4k3vuuaegtLRkcSKRkJ977vnh
    sbExhd2PSimVN27caNq6dcsmjuMbA4HJsz//+aM92WyWEkJUSqkCQHI4HLSxsVGYmJggwWDQ
    +ExUw6FZW+gYfQnOIsbYPBOfDWNrjM/veGy+lBitx2b9vXzqU59qcrlc96gq3cLzQl1xcfKZ
    V1999f+Koiiz/TljIDt+7+Kzfs88z5tkWa4/dOhQUU9PHwTBitbWVvvGjasrUqlUOcMfAgCV
    MdHkcsTo9wD0736V22y279lsNns4HH6EUvocgGWtra1lH/3oRx1//dd//cDp06enAByhlCqE
    kBRjQzgGumFY3JoBdFzAzL5LA7RJEISHmxvrPkM1jevpG/yhRul/M0ChEEKSlNIVNdWV/7p+
    dVvD4aPtw2P+wFcYiKaUUhGAlxByrdvtXi7LciadTh+hlB4nhGQMi13/XMECaZjebGHmyRTm
    +zcEgINSWgDACcDKDgA8GzOJnXjThBCJ/V1PISnscDET4PMY9wsC+eUM2HgLhXDGbAOltLy4
    yPcnD9531wYeKr737z/ri8YShFJaxMboTQHl78vhmBBCNE2bjEajUVkSvXabBb3dfpw+fSYK
    IASgzLCOUyyOcnls3HvI+R269HXidrubU6nUVxKJhLB169YnRVH8kSzLi9LptNnv95OioqLr
    N23aNPrYY4+9QimVACQ8Ho+joqIi63K5NEkSta6ubimbzapsI9bZO9kQq+m7lez46Ecfcvp8
    xX+cTIr3uFyF6euvv/7bP/rRj55nbKQMILl586ZrJAnfjsVSxUVFZac3btzwldde2zfKmGdx
    7969ZU1NTTdomtZMKY0GAoEDTz755JlkMqmnyPNj9GU51M6nYZ9HbkJuvfVWq9vtKgDgVhTF
    JkmSKZvN8plMhiYSCSkUCqfHx8dToihKhvgsswOUfujXCRH9/zQC7isVm99yfNavtrblnMfj
    uWtiIvw1vz/JNzWVKclkVBJF0csOTEm2/4j4PVAfvEkmgRw5ciSzZ8+eUZvNTlUVxGwWUF5e
    pk1NTQ37/X4XgBK2RjPsvvMPUQsSo98D0L87A+2sq6urffjhh+see+yx5gMHDlQAUDiOs770
    0ktYuXJl8djY2P2hUEgCEKSUmgBYGOAys2AzDKAbQJYt5ixmNTtKfoB+l4Fos9ls2vbB999a
    k4jH8fff+/f1mqq+SCnNAhAZg+wtKykuvu36q0ydXb0FY/6AD4CHpQ3dDofjs7t3776uqanJ
    kslkcOzYsTtOnjz5U03TniKExAHEMZuCpQt1as4PunrQZBITgY2TWT/UGArGFACFJkH4iq/I
    t9ZqsTh4jjOBgNNUjaqqqkqyJEYiMb8oST8G0AMgrQNqSqlEKXWw3y8yQG1c1Eoeo6NhAYEY
    Y4504M6xoMIZnsF8hXEghMBkMkEURVd5WYlvz/aN+NFP/keOxZNvsM3Hh5yUQQeUl8SsXKl5
    rhcLEkKgaRoHgGdzgAeQHhsb8w/099XVVVfg2d88R3t6+/wsZlbq85KxdPo46qlR42H4veud
    24BJJBKhL7/8smfNmjX3bNq0yaEoCpFlmW9oaEA8HncWFRW9f/ny5dlMJpNau3ZtbWFhoY8Q
    4gGIhed5af36DcdfffXVF3p7e6cppSI73KcNcfqyFShdysVxHNxut0tV6Y6uLn95aWkhvF7z
    Gp7nj2mapjACQ6GU+hKJbOH4eNRUUGDziaJUyOKutn379vKampovDgwMbEkkErzFYkFFRcVt
    11133b89/vjjr2qalmAyRWOMVhdiXebFZsJkggLP8+a6ujprKBRSo9GohpwMQQWg7Nmzx7pk
    yZLPyrJytSQpTk3TzJSCY+FH5TgichwJd3V1/vdzzz13glKa0WMzgOxdd91ps9vt9o6ODuno
    0WOSgcUGI8Hki8Xmhdqf2X3qIH4+Wc0cks1I+mQyaUHTtPJAIMarKg+OU3oOHTrUxeKzLjFL
    54Pnd6t7h0GWos8FnYDkT5w4Yd68efNoaWmZYjb3mIqLi1FS4ku8/vprYU3TytlcAc/zmiRJ
    MqVUYwc+xRCjL3l/FS73A8h/4P8bDN/ZPWRFUYxu374dGzZs8H7iE5/48KlTp0aam5tNixcv
    xjPPPIP169e3vvjii3coijIGYIPZbC5csmSJraW5SRAETjt67Phkb9/AkwBeppT6GUudYmAk
    kwekr8SJ9y1dDocDyWQym8lkX391/xvXxRNJWVHVowwYmthLAtDbPzB09oc/+fmOwZGx0wAm
    AHhYQHjfunXrbrBareT5558fdzgc9q1bt/ri8fhdvb29Y5TSMwzggU12mYHZS9LVGoEzA8t2
    dlqtQK5grI4QUkEIcWuath/APvZ/6/rIhuqqylu+/qdfqWturAfJJXqhqSoURYZ/MoBv/f0/
    Z0+d7TjDWK0kgBh7CTzPf9HtdCxOZTIjkiT3U0r7AIwjV6CWNLDvomH8FQDaAgRpSiktBHA3
    ABuldALAeeR0zHpaTy+MkzGrAyWiKFoBLA6Fwo5//69H8ezL+/sppR0AigyMRiIPkM/LEl3s
    vV3pjAv7f6yU0jXIFQzylNIUAxBCOByODw0OYmJRI556+llFURQZwFIWNzWW5j4F4JcApgzA
    KoN5NNHv4lj2rtxAL3FcEYvFwna7PZzJZIr27dsnXH/99TfU1dWlTp8+jUQigQ0bNiAUClWt
    WbPmfdlsNlZdXb2Y4zj75GSAn5qagiSpqKgo33LttddubmpqevbQoUOno9FokK3TOIvVWRgK
    lN4tMVpVVRw5cjSyefPmA7W1JWtsNiHY03PmlCzLDrbW0gCyfX19p5qblw5YrbYlmUzi2MmT
    7TEAHq/Xa16yZMlHpqamtgmCINXW1k5ks1nvyMhIZU1Nzd1NTU0TPT09AwbwrBhi9O9874Yi
    OL6pqcm2dOmSosLCwnKHw1lnMpkaOI6rJIQURKPRMz/5yU+ekSRJf/5ZUcwWiqJ084kT59cG
    AmEABIRwIIQHz/OwWi1YvrxRs1gsfXq81WOzw+GQq6trPsHzpu1FRSVjmzdv7g8Gg739/f1j
    XV3doUgkkmQSH5EBMcmQiVgQuRqlFDU1Nbj77rv3cBy3XhTF6clJf/8LL7zYHwwGZUZKGfcG
    vc6IAOA2bNhUzXFcVVlZEaxWIdXTc/5AT08PB6CYxeYsi13kt7DoZJ64TedbY1cgk0Q+/vGP
    17nd7nWaprmz2awYiURiIyMjmcnJSerxeES73WGqr6+lqiolBYEv3bRpY2VxcQkKCgqoIAih
    kydP/ry9vf2sYb1mDOv23Qmg84up5kmNX8zofd7f8W4Kzsx5IjY1NTXa19e38n3vex/+8A//
    sPjLX/5yYWNjI7dq1So88sgjWL58OX/NNVdveP75FxpUVSWSLKcDgUnV5XTamxrqLA/ef0/1
    0WPHP/bq/kPbItHYi5TSNwBMEELMbGHrbN4lD/RC3n8qlQIhRKOUjp7v6huVZJmwYORk75Nn
    k1RJplL9Pf2DPlGSwwBsjNldsWjRols9Ho/w9NNP789kMo8CWOF2u+9bsWJFaW9v70oAYyww
    aIZq2gV5DuyUvwrA1TabbUVhQUFTeUV5ZXVVlbuurs5aW1PNHzp0GD//xWN1ACoopTLbcKYB
    UKfTYV6xfCka6mqhqgoo1QAtF0PLSovhdNh5AHsALGNjGAYwAiBTVOjd+cVPPdjAcWTzuc4e
    tbN3IDk0Oh4IhqN9sqx0UErPAOgihMSM7AgLmNolBiMA8Didzo/t2LJhSSqRSB9pP304kxX/
    DkCA/R9Jw6FF3wwtAB5obmr6ctuyJaUDI+NIZzIqgEJ2WEoAkAkhFsOGms/2c5RSMzs8aQxo
    G2UrykIz7m+FpVNVdWV1dfW/bdq0qYbneYTDESUYnM5MTk5GY7FYQTabwfjYKCLRCN9YX7u4
    oqyUr62u4GurKjhNlfHj/3micSIw3c0OE3EGrIzZhHc1cM4vfDU+mzeLu+/y+gwAIF1dXaE1
    a9aMFRcXLxofH8fZs2dtPM/bQqEQSktLUVBQgKGhIRKNRpubmpqSiUQikclkxmRZVoqLix08
    LxSPjk66QqHEtkWLFrfV19cfO3HixBNHjhw5wzI4MMzfd1WMppTiwIEDytq1a4ddLss4pUps
    fHwii1zhr8r23/T4+ITY2LioRxCgBYPxhChKDkEQ+J07d26TJGkPx3Ga0+l8bv/+/c+vXLly
    t8Ph2JtIJGpramqW9vT0RAz1KjowudTnQD73uc+12Gy26zRNWyVJUnMqlaoOBsOeWCxui8cT
    QmFhAZqbG9puu+02j6ZpWYBmslkxHI/HY9msZAsGowgGo+A4ARyXA9Acx0EQzBBFiZaXl226
    6667Knmez1CqhbPZ7FgmkwmqKt3d3x9ck80qG7xem+Z2l6bWrSubXrNm3WA8HusYGxs7c/z4
    8Y54PB5hBWr5mYhLvqqqKmEymXaFw4kvJZOyVFZWE7juumu/81//9cirBiloQs8W6mzS5z73
    uWUmk/lvx8cnd2gagcXi1RKJhBVAOXuPnE4MzEdiUEoJz/P8ihVtlvb2k6ohW0kNjPcVz7Z8
    4AMfcHg8nq/6/f4PxGIx3mKxqFarVVq8uDVut9uUaDRusVrtKCgogNtdWLx5884iWaZCNqtx
    kQglZWVutampKdje3h6mlEZYfCYGUu6SLuFyBmf9KMNAJ882WwsbTMxarVJdX6a+Exvp2w1M
    bGJl0+l0X3t7u5ZOp7lMJoPvfe97fGtrK44dO4bS0lJs3LABW7ZsNm/c8FjFy6+8LJ0+dTrk
    90/2+/2Twf0HDpRcvWvbivffdoN5enp6xYEj7U2U0noAP2YPRcsrRJszed+JDUw/CLEDBA8g
    HAiGp0kuKLsZG6lhVqJgkhXVnEhlPaqqqgAKAFh5nr/F5XKVnDx5ciKTyTzHgGk4Ho+LTU1N
    VgClDJzpG1OKHSgWagwFQsjHrr322ofe9747uEXNzaiqrICvsBBmswCTYEI0GsXS1kXrFjc3
    rKYUyGQytP10RyQwHXxWVTVOUVSomgpVVQGqgWoaCCgURYEg8Kbbr9uxrqKsCKIoIZlKa/6p
    kNx+rifB87x73aplWNRYi2x2Nx+NxT3Do+Oes529iw4dP33NqwePdWay4j9QSvsJIRHMdXG5
    JEaT2Q5GKNU6t21at3zF4nrXR//gT1vG/FMVTIogGqQIIgCF53mqKEpTY0PD5/7P33yzbtum
    tRgdHsCX//QvFz3/8uvrAJwxBGfTPOlBXTLiAPAgO5D8EsAY+77ufpCfEr8iLB2A5Xv27Gn+
    7ne/y5vNZoiiiHAo5PZPjJVOjI9hyeJmFHrd+ME/f5uzmnmrz+uG1cyBA0UgEMAr+w97JgLT
    jchpo81ML6lnK8i7NY7lb54GUoMDwDFJC2H/3liXcVlS15fh3siZM2fENWvWDFVWVqK+vh7j
    4+N44oknIIoiFEXGa69lMDo6hmQySXp7+1z19fXOmppqryAI4319vd0mk2msuXnxulQqa9U0
    zWy1WldWV1cHjxw54jcyZCxTlsnbu94RNjpvbPlYLDpZXFwSyWYVSdPUQsPBzgQga7VaCwBN
    MJl4dyqV0gD4KioqfLW1tTcnk0mXy+Xq2b9//0tdXV3R0tLSUFlZmZLJZASLxVLO4jmllMoM
    UAqXGp/MZjNcLtcdExMTf3HmzFkhFAojHk8gkxGhaRpUFVixYhkaGhpbLJaCZgCU53laWWlP
    T02NP60osikHmjlwXG6/yn3kwPMEHMfzhBS2mc2eNp7nwPMcLSggit1O44IAeySSxtRUCuPj
    Cc5iMbkcDovL47E1+HxFO1esKBmNxaL/cOJE+xm2x/GGw/KCAOjjx0/Q7du3n8tmpdS5cxOu
    5cvrq3ieb2JxVqGUmgwxUwSAm2660WK32z90+nTHNSdOdBCTyYWlS1ucixYt3nb27FkxkUgE
    WWw3EhyUxeWZdf/pT396g9PpvG/Tpk2vPfbYLw8HAgHZkJnUSSzlSsa0kpKSwlQqtenll192
    +v1+NkcscLk8hQUFPlDKQxQVnD7dS3p7/WZZBlSVB8+bwfNWmEwWzmo1N/A8X6WqqokdfBVD
    fH53Aei8BUwopQ5KaQuAdTzPN/M879E0TVEUZYpS2gVgAEAEs0VjxhTyglf2LhAAIwB4TdPU
    cDhMdT/jbdu2AQAaGhpwy803wWKxwGw24atf+RJ58CMPWA4efKPi0V88VvzKq68FwuHw2Cuv
    vzHRfvqcO55ITlNKQ2xitlJK3QBGGVtrZqygUb/zjhQY6v8PA2IVVqv1I3e+/31rPB4P/8jP
    fnZ3OBx5AcAQAAul1GuxWPbeestNe67ec5XlX773r9eePnNWAtCnqupYe3t7RtO0I8jJOooJ
    Iaubmpoc4XBYP+m6WZBIs2fAL8SE53kemqaphCCyceMG8uBHPgJKNWiqCk1VQakKRZYQj8dx
    1+03cw/ddxenqipisRg+8+W/8ASmg9WqqpoBCrDnof9JAWiaBrNJwN7rdpCNq5aCUhWUanxg
    KsR/4Zv/n3XYH4SqadA0DRzHobDAA5/XjTVti7GitUk4frqjMJMVqwCkGAtgBJiXOn4EQDqV
    Sj/+s188sXNi89qSWCKZRc6Wrow9axjWoqwoigqgor6+rnzdmlUwCRxqKsuwZsUy0wuv7C9h
    wDjOfi5fm8cx7b8NwO0r25b+cUtTg+/V/W9sn5oO/TOALgPrzeEy6L7fwmWy2WzEbreD53lY
    LBY47TYUFXrQ3FALi4lHOpVAWWkRCFVhEjiAatA0BZl0Gul0BsjZVpZgVmOoS1nedZfu3W7U
    F27evNmxdOmSSrPZUq5pqjuTyWjBYDBx5syZ0MjIaFJ3Y5gndf2uOhzkx8LNmzcXcRxXH4vF
    MDAwgFQqBUnK1W+Pjo5hdHRshp0cH5+A3x8gHR2d9ubmpubm5qYqUcxOTkyMToiiKI+M9Iat
    VptosVicN9xww03ZbHakr6/3zMjI6AilNMq0wBnkec6+gwcj8pGPfKTZ6/U+eO7c+aUmk0nb
    smWroGn0pe7u7gkA5tWrV1esWbP2A/39w5uCwaCwfPmSm1evXq329PRMj46O9tvt9rGTJ0/u
    7+joiFVVVVXU19evmpqaMns8HikQCAA5OZ5OcJgwK9/6ne9bkiRKKY0EAlNae/tJaBoFzwvg
    OJ6BYg5WqwUTE2Hu5MkBjhAOgmDCqlX1nKZp1ZRq9hzRo+PC/BcwPS2RyUmZ/U6emM2CubnZ
    XFRYOCdbDknSIMsiYjEF4bDEL1/u8Waz2Srkapp4g6QiuxBsJiEEkiSRU6dOH1q6dPmBlpbK
    6xwOkzIyEtZYfNEPPrIRJxUXl1hlWVk8Pj5JRFEFzxMEg0lSXV1Q4HK5ChKJRNwwLjPxWQfO
    K1eusLS1rWg0mcx/MTQU2ON2O2+7+eZb/vMXv3j017FYLM1khXH8Dq4lC0ByCJRSXpIkaGzf
    VBQVigJksyo4zgyet0BRKLJZGYSYWebBBEEww2wWkMmkzZqmlRgOAsmLMfHvKIDOO3HzABZz
    HPfQokXNN65dvap6UXOT1eN2IZ1KYWBwUGk/eTrS0dl9KiuKz1BKT7G0dZzdYBoL7L6wUMwG
    e62ura29+bbbbuOvv/76GYBJqQaB5+ByOkGpBlVVAEpRUlyMvbfdip07tplefPGlqh/8249K
    D7xxKBQKR4aQK14qd7vd1S6nY30ymYzEE4nDmkafIYT4kSs8NKaMrjhLbxxbTdO8giB8/o47
    br/jW3/1lyaX24XGhvrmv/v7f3BNTPhfAjBmsZi3f/CeuzZ87atf4SsqKuB0OLx/+EdfvsY/
    OakAOKdpmgQgCqDRZrNdtXPnzjVlZWXCI488EgDgZ4FCQJ6m9lIvp9OJSCRCNY2ODA0Ny6Io
    mnmeA9W0HASmsw/10NETUGQFkiQhmUpheGTc5HQ4VtXWVLkcDgfbrGd/hlAKs8kEl9OF/3j0
    Wbx2uB2FHhdcDhsURcFUKAqL2QyTIOjYG1TToFENHCGIJZLIuQ2hmDHu8mUAZDwAjE5MyifP
    96OyorKyt79/r6pqr7NMgK6PTDHQpFFKAwMDg/7DR454tqxfibHhIRw/eUallCYM7PhMNTv7
    Gk8pXUYIeZ/FYm7meX7DXbffXLy4qQ7j4+MbpqZDKQD/QikNsrklsvGWr+SyBjDZ2dmZOnr0
    qAsARoaHcfbcGdrd1a1pqkI+/6mPca+89jp95rkXVIvZjAK3iysscJECjwuBqSAGR8ej7N5t
    jJUS3u3gWT9M7dq1075mzdrNgiDcnk5nNqRSmQpFUc08b9FqahqiDQ2NPVNTgYOvv77/6MTE
    xDSlNJWXulZwiZrXhT7g6weDG264wbF8+fKP9PT0bO7s7EQ6nZ4z7DmgxM0AJo7LhZl4PIGT
    J09jeHjU1ta2rK60tDjZ2dnpb2xsbKirqytIpdJ8OByhFos129jYdPRXv3r8R8FgcJRSamMg
    Wn8+8pVOeRuyg/jkJz9Z7na7//jkydO79+9/g+N5gduwYW3btm3bPABezGazsXXr1l/T0zO4
    /NixdpLNZsFxfNnKlauunZiYeP6JJ544znFcVlXV+MaNG5cvX7786lAotIwQQgRBmOjq6grm
    xehLnvM8z0OWZYiiOO7xeFJms9ksSfLM3mPsTebzudDaWs2kGQIKCx0cz1uXJ5NZRzYrzfy7
    2Z/N8R3ZrISyMhNcLgGKQqCqubdutRIG0OjM/0MIwHG5z00mDqoqacFgqICRDZcjNgMAJ0mS
    QAiVfT4Bspw2VVfXXLN69Wra3t7er2cOMOttrPb09EhbthT31NRUXxsKZYnZbENZWSEkKZuM
    x+O69E9nrGXdbeTuu+8urK2tvZVSbBEEvnliIrSxp8ePurqq0oaGogdXrlwxvW/f6ycppXFW
    QKlLQa6YRenk5GS6pqbGX1NTs5jneTidThQVlaC4uERzu910dDTARaMpsnLlYi3HlnNEUUAU
    hRCOE0hBgUk8enQkwmSIVgAmdvhZEEwhXKYAxgHYXFRU9K2777pz0wfv+QDf3NQAq9kEUA2q
    IkPMZoTRsdHiJ5/+zdX/+V+PLhsaGX2MUvoqIWQKOc2ozjhmkKtyf8eDNKVU1wZWut3uP/z0
    pz+96KqrrpoBUpRSUI3mUvoMTIN91Gju626XC++7/TasXb3K9H9/8MPS//zpfzkjkahEKbVZ
    zGbz9dfu4deuXFbz7X/+fn13b3+CaWL9hJAgA5yJeQL0ldyczADu3bFjx0e+9sd/bC4rL4em
    KvjoRz4Mu81a9hd/+c1r/f7J8K0339j4x1/6Al9WUgRVkXDDddego+N8yf/59ne2iKK4n91L
    Q1VV1c5rrrmmklJK/ud//icwNTX1BnJ63KzhlL1gh6hwOKyPoX/S7xeTyaTZ63HnfjnN8cgc
    R7Bn9y5oqoqsSiBYHCh2FeBzn/kkX1Fe5mtubEBxkQ+gAAEBCJ35WY/bha9+4VPo7O7F4PAo
    pqeDCE7GIMsyNq1fi6b6ahT7CnJzI6+D8PjkFDKiaATQcSYNMC3QIYICKC4qKnr4K1/8w4pb
    b7oOk+Ojjr/8m3/Y+cIrr2sADlFKJXaQjbF5pgIYHhwa/I8vffVPv7xsSUvh+IRfPH2ucwg5
    Xbeclz3Sx2pRcVHRd97/vr3bWpsbudcPHMQjj/5KSyQSocBUyA9gkjG3WUppmhDC4wpbKzEv
    744jR4689sADD6wQRVENh8OZVCqVoZTC6bA33nTtVd5XXtuvHTtxaoLdIw+AY+FIAtDHntMV
    Z2fe7mXsGvqhDz1QXF5e8clIJPpgd3df1diYH6mUCE0jMJkscDqdvvLyksb6+optN998y6ED
    B/Y/1tHR0ct0hMY4JGJhClwXLDvo8Xi4tra2a6enpx84ePCg5c3BMzeT4jd+PRqN4dCho6Sp
    qdHV1rbMJklisqenJ+r1eq2JRMrd2Tng2rhx9cbdu3eFUqn06Llz544ODw8Ps/oVLo8AupJF
    V9i7d6/D6/V+sru757ZDh45wkiSB4zQcOXKCqKpWu3379hsoRXZiIljf3t5BZJmC40w4f74P
    RUVFtZs2bdrw1FNPHVVVNXnVVVctq6ur2zE+Pl5stVppSUnJyMGDBw+Fw+Ew5hbMXnKMttvt
    iMViJBKJTNnttrTNZiuQJHlmzPSP4+OTcLvdoBTQNApVpQgGJ7h0OlsYiyURi6Uxy0LPcl6a
    pqGjYxh+fxhutwM2mxUWiwk8zyMY5JDNykgmJVzIYANWq4BMJkWTyaSPscEJhlMWKjaDUorS
    0lKyZs3qW8bGJq8+cuQ00TQTWbKkuWXlytXC4ODAU5FIlDBbthh7D+LBgweVysrKR5uaateX
    lJRsUFWqCQINHz588HwymUxjbtdUGYB23XXXWWtraz/p9099YWhowu1yeVBTU45165rTkpSe
    GBkZGp2cDFgMWYYFJ7PeyvXqq68m7rnnnt9s3LixOpvNmiml2VQqlZ6cnJRMJlJcUlJUa7HY
    iCQlY0ePHpkymy2c2+3inU4XLBaLOjwcm+7o6JiAQcM9nyXhuwJAG67FxcXF3/ryF/9o60MP
    fhh2mxWqLEFVZVBVgaapEHgO9TXV+ORDD6Cpvrb8L/76H+7tGxgmlNKDLAjpKQejbcs7zW4Q
    TdPMhJC7b7zxxj0f/vCHidlszm1MeYCZ0vmBNCiFRilqqqvwta9+kTQ11jv/9u//URwZHfNP
    TU+Lhw4frf3w3bdbN69b5YxFo/fGk6nb0pns45TSl5nWLl/Uf0WCs6EIdFNzc9MffPlLX3Qv
    WtQMRZFBNRW8wOPuu+5EJpMp/vZ3/rn49ltuRFlJERRZAgiBiRfw4fvvJYePHq1/6eVXwwAm
    ysrKttx9992lp0+flvbt23deFMUTyLkZxBnAjjFGJ7tQh4V7771Xv49IMBhMxONxV0GBN0fL
    5rLz0DQNu3fvxM7tW6FpKjRNzTHUWu6xq4oKWZahqQo0/esMEBNQVJaXoaq8lDEaOWmIpqk5
    /x0+F5w1VQVHOGhEy/0ogIlAEJIkc8hpCz1MWmF5KwzH2zhQlzY21C++9eabSEVpIUq9Dly7
    e7vppdcO1Gqa1gcgTil1EUJsepqLEKJQSg9FY/Hp4yfPFIxP+E8iZ7+oGAD0HOcOAFdfd901
    W/78T77C2UwE2zasxEc+9QVleHTiGHL2fhHGCNgWKp32OzLQ0+l0+pu9vb0NyNk9edmrxu12
    13k9LphMPGWMT5TNQ5FSJNg8DbKNNIacO4PRleFded17773e8vKKLw0NDX/y4MEjtmAwAkIE
    8Hwu7akoufR1LJaB3x91LlvWsGfbtu3FmUz6pwMDg92YlVTphzLxnT40GN0C7r777gpFUT5+
    7NixkqmpqTn/Jh8ov9lHTaPo7e1HKpUW1q5d5UwkEom+vt7gmjUbmsbHg5apqai9qKjy9tpa
    ayadTqvDw8NJpoHXvaLlK71HFRYWci0tLTcGAlMffeONQ9ZUKsUkEBwURUV7+zliMlkqFy9u
    Rn//GSgKhSCYc9I1RcXp0+e5nTs3LW5tbY2IopiqqanZPjg46KmsrEyLonjmhRdeODU+Ph4y
    xGijG8klxWhWk0AikUiirKwsarPZKqPROCiljBnOyd4mJ6cxPR3WJRjgON4g8eBnJB+z389p
    ogEe2ayEiYkIJifjM/+G4/gZEl3/HZRiDoi2WgUkEjEwT2UvZnsAvOXY9VY62S5atMjEcXzb
    wMCQLRiMwGbzYng4iMpKX2lBQUFVJBKNISdvdAAw6919f/nLXw4+/PDHT7lc9vWSlAk8++yz
    B4aGhpKG+Gwko7SWlpbqVCr9gUOHjruDwRQcjkIIghUFBdzIE088sS8ajYaZJa2D7QnkcnV3
    fLNHFg6Htccff/zJ0tLSrlQqVRYMBgtisZhHVdWiq6++eoPVaiaaRpDNZqXBwaEoY9d1WVGc
    xeZ88lFaqAz+ggJonuehqqrNZDJ95AN33rnpox99EDaLBaoi5UAI1WbApUY1UFUBR4Brdm5B
    IBAo/Iu/+c51iWQqzOQfmsG4XtfgvSMshz5xOI6jqqo21dXV3fuxj33M4vP5GHieHyjPygLY
    92eAdu7fWC1m3HfPXSjweix/9o1vlfX1D4yPjI1LP//lr62+Ajd3456t5fsOHZf7hsZ9AAqY
    LVqWbdKi4ZBx2UC0MX2maVqZy+X8w49+9KONO7ZvZwV0dOZeBYHHffd8ANPT03j8yaexemUb
    yspKQSkBpRrKSovx8Yc+bDpz5tziqenpsp07dxYPDAwoL7zwwnEAbxgmd4IBkiirmtUZ90u+
    wZ/97Gf6PcX8k5OxwcHBipLiIpzv6EAwFEQykUQ8HkMsHkcykUQmk0Y2m4UoipAkCbIsQVXU
    HDims+OZA8+558QRAkHgYRIEmM0mWC1m2KwWOOw2eFxOOB02uBx2eNwOFHrdKHA7MR2Koqdv
    SC9IIYag/JZPy2+xhW1iOhgMd3d3lxcXrEEsGkV3X7/e2IZn4Ei3I9S9jimAhuXLlpRbLGZt
    fMKvb6CEjU0aF1q3OQu9Xs5mMUGTs3A5bHA7nYQF5AK22erpT4p5itMuxrC91eKst9h6XWEZ
    D5VSGsWs5ZPbZrWa0qkUBJ7nDKyibGCfguzAF2YHgrDOCmEeW7OL3c+VvFpaWvja2trbxscn
    HnrllddtkUgMgmABz88FIforlRJx5swQWbOmuW3jxk23Tkz4f5nNZoV5OtG9o44j+rP0+Xxc
    YWHhrv7+/o09PT2GufDWgHMObM39+uTkFI4cOSFs2LCm3G63m2RZzKxY0cIBHDSNcolEPD44
    OKghV/gssYzKFYvRxuuDH/xgsyRJf3D06LGSQGAaHCdA7yFBiA6iz8Jms6GlpQHRaBKiKILS
    nFtFJJJAX9+wbeXKVcsURZYCgYC7tLRUDAaDB1555ZX2bDarH5bzY3TmUlnoVCpnOjU9PZ2q
    ra0N+nw+TE+HUFxcDIfDCYvFAqvVBqvVArPZDEEwwWQygecFCALPwDM3o5WelXwQQ9E9GHOd
    Y681DVBVClXVoCgUspxjtGUZkOXc3wWBg8djxuhoXFUUZb7Y/DvH5/yvRSJhBUDA5/NRqzVA
    eF6A1+sCpaqUTKaoHp8ppWZCiM4Kq+vXr3PxvLBkbGySczhMyUAgEDQcbjO40LrNqiiqXRQV
    8LwJhJggSRoyGcmcSCQ8rJFOhu0D8+5D893L24nPb5H40SYmJuITExMDbM4VA/DxPC+VlJSQ
    RIISk0mA2203Wa1WKZPJ6FmRCMveTrE4HWGvaD6eeLPGalcUQLMTZH1dXe2N937wHt7pcECW
    xNn2uBqdYWZngCRLK1531Xb8+tkXa/a9cXQlZpsyzDfw79il5gRTu7Zt29ayfv26GeaR6qvS
    yD5rFwHVeew0AXDT9dcAoNavfu3rlcOjY9F//sF/JniOIxaLyZYVpQB7HqWY1Zpe4BF8uaUb
    mqaZAHxgz56r99z3wQ+C5zloqjIzthS5sbRazfjYRx7AH//Z1/HD//wJ/uhzn4LZbAEIB41S
    bN+6Cdddc5Xn1dcPuqurq7n/+I//mGaMpH4v+skxylLFcwD0Am1AFEBgcnLyxb/4+tcrbrrx
    Rs+Z02eQzWZgtVrhcNhhs+UCtc1qhdfjzmmXzSaYBB4Cz4PnWOqXTUmdaVYUBbIsQ5YkiJKE
    bDaLTCaLVDqNYCiMZCqNVCoNURIhywrKS4uwaU0bnnlpP913uD3NFrneRMboB/5WdWc6IOXy
    rLb0jxNDQ8M/+7Ov/9Xnt2/dWOSf8OO5l16dopQOsXnEMeBslFRwHMetXLdmlWt0bFwvatTt
    7hJMF5s0bKIEQNdvnn8hvmLZYu/iplq88NIrOHO+K85+1sTei2zoBnbZO03O8+/0A0sGsy4v
    EgA3z/NLNE21//hnj2Jqapo47bayZDqTRq64N04ImQbgp5ROMuAcN2hgdTbuTQuqLrYBXc5r
    9+7dJaIo3d/efsoTDochCBbMuhZwDDjOvjiOgyxr6O0NcKtXV69saGjoO3/+vMwstYzM1iV7
    tC+AfAM7duxwqqp6bV9fnyObzc4hB3OsYj5ozgfOc7+ufwyFojh+/LSwceOaIr9/fGp4eNhf
    VVVlrqysdAQCgQlZln0mk6lSlmXVwD7r6/aKZCRuuOEGu9PpfOjUqdNre3p6DAVxZM69iaKM
    kyc7sGPHRrS0NOD8+f4ZgosQDgMDY2hsrC12uaw0FAqB47iJ119/fTCbzequSEkDeNalBPra
    v2Sv+vPnz8eWLVv20saN61u9Xm9JYWEhrFYrVFWDqqqQ5VycVRQN6XRmRsZh2IbzpDq6F7TO
    UOdegpAD3oIgwGo1w2TiIQj6v+OgKBRTUxkUFlqoqobF/v6+sCE257OZb1liZMh+cQarOACg
    58510NWr1/y6ubl+u9PpWi3LEEpLC8Xu7rM9wWAwiVmnHN4gqdCWLl1WncmIzaFQFC5XqcZx
    nC6/yxpitP5+ce7cudDq1at7Vq5cVtvfPwmvtxCVlV753LnjUUVRdBmSitni4TmFsW+RrHnL
    ce9NDpcacg4iSeTqalS320137Nixrri4pNLlUkEpgc1mcu3evbvy8OHDA6FQKEkICQOYZL0O
    jAy07uet/Lb3/1bi84IC6DvuuAO//OUvWxa3LK5sbmpkBXTazIsaPhpfmqai0OPCmhVLTfve
    OFqPnDNHklIaMyzQd8QSSh9Yxq57XC7X7quuuspst9tzDDOlyGYzkCUJZrMJAs/PAcgXZadn
    PtcAClx/9VWIRiLWr33jr13hcGRM0zSbrCgwm0xuQrQdrHW0MTjLBtBx2XSXhBAIggBZlpfW
    1tZ+5JOf+Li9tKQYiqLMFtHNHO1zrKzPV4hPPvwg/vTr38Qrr+7DdddcxdhaDiaBR0lxEdm2
    bRvp6OhQg8HglIH51NPgOpunT/osFrZwgRJCkqqq/nD//gPh/r7+j/zB5z9Xc8/dd8FmtcBk
    EiBwDIcaxml2PDU29ho7KM3+nTK5h2b4nBqkHJIk5kB1Not4PIHHn3kJf/WdH6r+wHSKMaGT
    bMGHGJhOvtUgzTS9TgDbAIwQQoYxt/paBaBqmvaL9lOnU6fOnP2E2+1eFE8kx9j/p6dgOQOA
    pgDcPl/h6jWrVhD/ZMAIOjX2/pKYLWqRAdg4QtoCgYD1j776Z2m3y5WcCgbT6XRmkI2xzuDq
    zK3O2hpb6c4XSN9ywJuneYtupzdj1WaYdxwhRCWEpCilLofDcdveW27auGvbBt7CA2ImSfoH
    BjyP/vrF1r6h8QRy2m8jCx1BzjVFYs/LZGBo57OAm3e9XgmG0u12twYCU8t0B4p8wHwhuMyB
    iURCRCKhOKuqKlvPnz/vZ2vSOPbSO0Vw6OCQNaKoj8fj60ZHR/WTHzweD7xeL0wmE6LRGGKx
    +NuWc+RAdASnT3cKK1cuKZqcnFS8Xq+9tLTUZbFYPJWVVfWhUOj0a6+9+ptoNJrFXHeGyyZB
    NM6ZZcuWbQkGg3efONEuSJICnhfmjGVOopA7LMTjSZw504W1a9sQCsURCIQAaEy6AqTTWRIK
    RUlRUZF87ty56WQyqRfISnkxer71e0n3EwqFxF/84heP7ty5M7lyZdvHurv7Fh84cJhIkjID
    kHVQrGdM9L/P+j4bvzf/3/MzLjlQbYIgCDCZBFRUuFFVZdZ6e8+l29vbp8PhsJ+t96AhNr/l
    vYlSCqvVyn384x9vI4SUPProo8fGx8dVFiv0xkz0pz/9aff27du/vmTJkg97vd5bI5Fo/Ny5
    c8Oapsl5JAdHKaUWiwWFhYXL+/pGi9LpLAghGsuASAAyDHymWIwSAdD6+romnheqW1ub1MrK
    0qgkSdLAwPnJkydPjrJ/GwEQYkA0hlwnXZ3EJPPIVvLnN7lYjJsHlBrt9IzxmWNzzsS+LwKI
    33jjjVc5nZ73nTnT55YkDhaLC06nx1Rd3VZns9np008/5c9ms7rkLsTGK2FYjxxj8Xl9beZZ
    db7l+LygAPrxxx8HgEKfz2c1m80zAHOGgWZFdnP+zj5yHIGv0EsIIW5KaRFyjgBu5DrFvVMa
    yZmNmDVdaKqvq1u9ft06UErR09ONX//6aZw9ewbJVArr1q7Bxz/2UdjtNgPTfKGUYz52mhCK
    O269EYNDw/bv/n8/KJckOU4IsWzdurlg3eqVFY89/mRd/+BQFkCCUpphhuoZA5C+bPINWZZt
    PM/fd8cdd7Ru2bJlxvt4xsqN3YcOpDVNw7Ilrdh7y4145Oe/xPKlragoLwM4oKu7F+fOd2PJ
    8jY8//zzunODkwVjyaBbCrGFm8ElNhG5yKUQQsKU0qcn/H7yP48++tmbb7qhoKTYNwNwFVmG
    qsiMVZagyDIU9ndVVaDpHtDsGRACcCDguFz1Nq97kILNIQJwHOCw2+B1OxEKRfCbVw5q/sB0
    gt1vwPDSpQGpt3L/bGFTQshup93+PVGS9kmy/A0AaQPTmmWuGikAhy0Wy/Xr169veeONN1zJ
    ZFK+yCZIAdTWVFU2LW5pxjO/eV4PjLIBSM9IOGjufe5qaqh9qMDrMR9tP/1qOBLtAODCbIOd
    CeS6L06y+04iVx0ugBWtMGmF0QM9awhqBHnV4PmB2eA/TRjjrfuUlzEJiZ29H5Ud1vXOZNUc
    x60MTE3xv3rquRwzp6rQFAlUo1bkdNIc+70O5LpXNmPWr1sEEGO2lEkAKcay64ykfvidt7bj
    coNojuOq4/G4WxSlizLO8wFrSoFUSubsdkcJx3GlmqZFKaX6BmvGrAXiO3KxAknOZrOtHxkZ
    qYlEonA6XVixYgVqampmnunw8AhOnz7L4vqbs84XFhgS+P0BFBR4zG1tbb6jR48Gcl8XfMPD
    47bVq5dvbmlpGThy5EiYyTiMrN9ldZO6//77CwkhD545c7ZiejqYN7ZknheH8fFplJcHsGRJ
    A+LxNERRAsChvLwIFguPZFKFoiiZrq4ujc31EFuHMfa5LuVakAwx22spIUQNBoPxxx577NUd
    O3Zoixcv/mJvb3/V+PgkTCYLBEFgOmeBscnCzEsHwjpQNo7lLGHLGf7OGSQuhLHcCgSBh8fD
    0aNHD6ROnToVUlV1ykBwTM0j13qL43Rfsc1m/ysAi3fu3PkHjzzySDvLFKqMaZUopdq+fft6
    vV7vMVlWblQUxVFSUiJMT0/PNKoxxrvt27fbALJmejpiUhRV97422p9mGAAWASj3339faWFh
    8ecjkUyLyUTHDh7c99To6JgpGo1aNE1T2L2NAZiglE6z8c4CQFNTk9vhcPCnT5/OsvHSY1gm
    Lz4L+XM+v7uhoVU639raam9oaCgoLCwss1gsxYTApWmaWZYVmkgkxNHR0Uh3d3dUkiSupKRk
    gyyr7oqKYvB8bpwFQYDZTLiKisoil8tlzWazcDgcprq6utK6utoKj8fD8TwPRVHkdDqdnJ6e
    Do+NjcXGxyeSqqqKjPyQDbH6LbVsvxxFhNlkMqEosmwy89wMy2xknpkAaS4TralIJlO6g4ed
    LVijNdSVV7AbJqksyzyArWvWri2vqanGCy88j+9+559QWlaCVatWwmqxYElrK0wCn2MeDUAZ
    88g4MA9LbTGb8eD9d5OzHeedz734aoJSOjo+Pl732Yc/bDly5Kijf3CoCLkChjSzljGm+BfU
    L1v/XU6nE8lkcvWSJUtuv++D9/JWiyVXOEgx53BEDfIUnZm+5YbrcOCNw/jVr5/Bxx+8HwqV
    8etnfoOSsnJ0nj+PcDjMGcaXMoCXYKdfvROfbEwbLdS5CDlT+iwhJE4pfeP8+c5rv/2P391Y
    UVGOqakpJJJJKLIMUZQgyRJUhRUMGg9/mLWxm+N/NwOYcwDaJPAwmUwQBB5mgYfL6UCBx43O
    3gEMDI9mMKulndADF4AAk7C8JXZPd0gRBGHH+2+7oeJ8V891h0+cPgDgDVYAoh9OMozpyFJK
    5WVLl2J4eLiwu7ub6M+bBUXKJDMcgGUti5qLS4uLmN3XHKs21RBwFFBaW1jg/fz1e3ZUvrL/
    UJRSnGVMgP47I4Z7zLDW4mb2vZ2CIFytKMpBSulTzH9Ut0+S8hhqM5v7+RvwTIqU+akvB7DD
    4/Gs9/l8DT5fYUGB12u3WKwmgCKTyShTU1OpiQm/PxgKnQBQuLR1sfv2W2/G8mVL4HTYoSgK
    Jib8eOrZ5zD92K/K4onkYgDLfIUFxWUlxd6iokK71WIRqKbRVCYjRyLRdCQaj0Si0aGsKOnd
    JQdZNi2NuX73Cq6gm46maTyz6MZsq2NuXpbywu9B70rpZgeit11IdTlj9C233OJWVfXqoaFh
    C6XA+vXrUVxcjIGBAWiahmQyjbGxcQODOdts40KwaXw+c7/f3z+CoiKfvaamxnH8+LHQjh27
    3Tab3ayqqjg9PU3YITDFYrTRH5pernuvqqraOTY2dnVnZxdrfsPNkXAYZRw6aKSUort7GNu3
    F6OmpgL9/eMwmcxobKxGOBxEZWUVstksz8ZYP2xmmeY5aiA4lIWM0ayxmkgpTRw6dOhkS8ui
    9rVrV1fV1obgcDhgsZhnZBg5sMxdIFfRbejyHTxmpymZI/nQX7oO2m63IJUKiOfOndPBs37o
    12Nz+O1kB/WroKCgJRJJrYvFMsXFxcUf9Ho9gWg0JrF7NnYZFC0WSzaTSWvJZNpWV1fn6ejo
    6MbcYk3K2OQSUZSXhUJR2GxWIGeLKxgO9TMZ6/Xr1/OVlVX3Tk2lrhNFBZTGO8+cOTvGMtwW
    tgdMAhjjeT68dOlSS3l5WXkikbQvWrSoubi4eC+l1Oxw2H/4xhuHxlkTHaOkVI/PNvZ8aD54
    1oGzy+Uyvf/976/1+XxbAGyVJGlpJpMtzWYzTllWLJRSnuN4zestyJaXV0TWrVvXmUwmhwCy
    bHh4HMPD40inc2UGFosFZWXFWLSo1t7W1taoqqq3sbHJZ7HYimRZcyiKZgFAzGZO8XhKsjU1
    jck1a6hfFNPdk5P+02fOnOkcGhqeYvKODNt7xHky/XNA9EL7QBNK6XBPT29ofHy8alFTwxym
    eb7PwcBWMpXCmfPdOlNm1GHyeId8VY1tyCmlPrfbffWeq3abTSYBoyOjuP76axGcDuLEiXas
    XNGGTRvXg2pqTq6gXah7NgLnHGOi295RUOQOEqXFRfjkQx/iOs53+0bHJ7LJRFIZGR6yTE1P
    6wvVidnOf3FDGklb6HvnOA7JZNJuNpvv2nvbbTVLly6BqqkzY6Z/1OOSfi9gemivx4173n87
    vvv/fR9X79wKRVVx8sx5tC5rw/joMNasajMdbz/twNxuewvaq/5NLpWluRIA/JlM5uUf/PBH
    PuRan9qu3rOHf+ihB+F02GGxmGE25fTPPM+B5zgQkvN91uUbmqZCVRWospzTQMsSJEmCKIoQ
    WRHi6Y7z+N4P/wuSJFMDWznFAvM4kwYME0LGWQYmygDXb2WvmMzGwXFcfdvypaipqvAeO3n2
    FlXTkgCizMPXwtiFLICsoihiRUU5li5pdXV3d7uYdEoxAFECwGw2m9YsW7JYsObkLYQdcO0G
    JlifMzaO4z58/VXbtlSWl2JoZHyE3Z/JkOqfZpuQw2I2f6iysmK1JEliIDAlb9u0btnGNW22
    p59/ZeOZzl4XpfR5pt/TN+x5AXReh1Cw5i1rLRbLh5csWXLt1q1bKzds2MAtXtyC4iIfLBYL
    BC53uJdlCfFY3NPZ2Vnx+BNPrtr3+n7ta1/9onDVzu3MgSV3+G9pbsCKZa0Yn/AXdnR2bbvl
    +muErZvWk6aGWjgdNvAcB01TIUsSUql0gX8yUNnV27/sxOlz15041TE5NDp2VFW1lwCcN9jA
    6Qdgow3YZW0clclkAm63K2m1WguzWWleucZ84JnneTgcZjoxEVM1TdMLTc0wFJu+UzFav5qb
    m5dFo7FNIyOjIATo6+tHb28f1q5dA5+vEIcPH4MsK4Z7vLhcg+d52Gw2iKJkANyEaWMVdHcP
    kNWrlxToPrs2mxOE8Cafz1c8MDDgZHHaxYggPX284HZ/lFI8+OCDPkVR7j13rqMgHo+zwkEj
    cDQW1M31vk6lMujvH0NTUw0CgQg8HidsNg7JpAZN42GxOCxVVVW2vr4+XWYgY6593YI7jRha
    gyckSZqenAzsa2pqamhubqgxm80Ov9/PHT58lGSzIlQ159CRgxIk71BolGjMZat1Fjungxbg
    dtuxdGkVFQQOmUyGRiJR6fTp89OiKOqgeQTAECFkjMU0PTa/rT2K5/nabDbrHBychM/XtHXL
    li3Hjx8/3h8ITCUppQkGfBPIdYFNaxrVJicDXGvrohK73S6n02kdB+jZNbjd7qZIJFmdSKRh
    tdpAKcyEEDuLKWljhnLbtq3rMxnloyMjYVtdXWGyo2OghxUuiCw+hwFMuN3u+N69e3d7PN6b
    VJV67HZrUFG0usHBSKXTaddWr15XNDU1/f2+vr4Jg5xNB9A8A9BpXNj0jQAgDzzwQHl5efmd
    mUzm7qGhoaUjIyP2QGAK8XgCiqJC03JKQp7nYTKZnW63u6i6uqq5rW2JPDQ0Lhw6lGuyMyvf
    ySAYjEPTqGnNmlVtU1MRMjgY4kOhBERRgaaBrXsegmDyWK3mUrfb0VhY6NhaXt4Yr6mp7xwf
    H3354MGDh6ampgJsLBKYdZiZtznSQvtAEwBDvX19R5548teVf/jZTxGiO268iQ6aADhy4jSO
    nDgjsTesGJgZFe+AJVRe4xACYENbW9v6LVs2gyME997zAaiqil/96leIRMLYtnUz03Nrc3TO
    mAc8pzMZPPHUs9i6cT2qKsvn/IxGNaxfswLv33uT5Z/+9UcFwXBYPneuA8taGux9A0MrJVkZ
    YSd/BwMxFgM7sGCbrx5gNU1b1tLSctPe22/nBEGAIstz2GZQing8DlVR4XLac8WEhntev3YV
    Guvr8OivngIAlJRVYGRoCLfesAeiKHKnznQ4FEUxWmLR+VInlwFQ6ObySfb8nkCuM14bgN3L
    li1dffvevYQjAKg6o3HWNBWaoswclGZ1zuqsFlrTQKkKaLM/xxHA63Hh337yqCZJcob9v2H9
    tA9gmOmWx5BrJGPUbf1WjZ2iKACQlSRp3yOPPrGlpqqsqLS0ZBshpDEwNd2tKMqLlNI+BjLD
    uWmtZUyCgI0bNpif/PVTlaqqnmTP3Hhw9Xnc7talSxYTjhBYLBbCgqONvS/eEMw3L25uuP/u
    228yP/bU82oimepj/0ZvdRwmhExQSuNWq/X+D95z9wfuu+f9XCwcxD//6w9BoEGSJRT7vD6e
    425RNa0LwAhjOUievo6fb70yhvT2+vr6P7n//vuX3HnnnWhqakJOUqbmxs0wXqAUvgIvmhrq
    4PV6uEAgwC1uWQRVNdgXspjldNhRV1tNVrUtNX3+kw/NgOY5Y081+Ao8qKksw/rVy/G+m68R
    egeGqp5+/tXKJ3/zyrrRicmnmSVlAAb7O1yZQmkyMDDQ39jY1FdZWbG+v3/kIhpo/gIW2uWy
    wW7n1JGRkew8Ou53TPusx4Zt27bZzGbz3sHBwfJYLAZKCSYm/OB5HkePHofNZsPU1HTeAeHi
    ANrpdGL16jacPHkOqVTmAsAdicQRCIRN9fUN7mg0nK6srDCbzVbHxo0b142NjQ37/f4wWyNW
    dtDgFlKmY1wOpaWlW8bGxnf09fWR+ZhmQgisViso5aCq2hxJAyEcRkcDqK0tR2NjFVwuK4JB
    P4qKStDVNYbiYg/f2Njo6O/v53Tii2Wo5uzLCxyfNZYJS1FK+eeff/7ls2fPjpaVlbU1NTVt
    y2bFrZ2d3byqaqzrnC7ZEBgjLVygdc4dKvg59nY5jXgOgBUUSKiocOH06eOZiQl/KhQKRdPp
    9HyxeYKRALqs4W1Jc0KhcGdBgfNkY2PZRo4jpfX1TZ+vrq4d6+np3rdv377jqqrqcS2tqmrG
    bLZq09NBLFvWWlZZWWnr7e0N5xGMJp4XlkajSU+OUaYAYCaEc7D9bca/edOmjUVms/UTnZ2T
    jaqqgefVyZ6enmnMOgwlCCEBSunEjTfe2GY2Wz9z4MCx4nRaRXNzQ0tdXQUEQaCaxnN2u337
    okXNZ/r6+l6mlGrzxGfditg43wkA8tnPfnaxzWb7s/7+/lva29ttY2NjEEWRHX6EGaCbGxsV
    kqQglcoiFIqhpKTINDUVhKqqbPxmNfC5RkhpxGJp4dixbqRS4hwrQ47LWRPKcq6LYSwmYmIi
    Drvd4i4t9Wyoqqpbduutxetff33f493d3b2U0jDT+s/xvDfirMsh4UhkMplH/+M/f7x2UVNj
    7Q3X7gGjhS4AXqAUHCE419mHf/q3/9KC4Yhe4ZpirzTegf7rRvaZDbrXZrN9YO9ttxRWlJdD
    U3M+1jxHcOf734c79t4GjiOsFfQ8bhuG5ioARTKZxK+efBr1NVWoriwzyFxyP2cSeNy59yby
    wiv7XOfOd4d++otfxzwup1XVtHIA1Qx02VmANi80Q6+vBUVRzDzP33r11VfXtLYuZi4rs+wz
    IcDQ0Ai+8c2/hsVixjf+9KvwetyMhJ616rv9lhvxlT/7SwhmK3bu2o3e0BSu2r4ZwVAE5aUl
    jtHxCT0NLORnHS6jHlTPduiLn0fOyiwEoFIQhNWapkKUJKRSSWQzaciyBIHnUOD15DppGADW
    DIieU0xoAN4k59Sha24xa7EzzgL0AAvWAYO2MIO8yuffhqMppU8caz+11e3ddfsP/+/37FYz
    1/Sj//xpw6OP/9oty8rPmU+tCiCraVo2k8lgw4Z1pLS0pG5iwu9iQcJi2PhrKsrL6murq0Cp
    BrvNRgghdkqpjQVoXUxYbLfbPn7X3huqS4p9OHX2vF4UadbvmTUD8gNwVJSXr/jogx/i2loX
    Qc4mEJicwOe/8vXwS68dHCAcUVRN62c/a7vIQT9fMkCY3OS6pUuXfuuv/uqvam+44QboPu2q
    Is+M0wWFoGyd8hzrSGc4AFLMxipJlpFMJtG6qBEmQWBSJm2mgRLNLyilGixmE5YvbsKi+hqy
    ftWy6n/8/k/uO3m2q4BS+hsAVkOq9Uo0YiGvvPLKVHV19a9XrFi+LBSK25PJ7EWKCGd10CaT
    gPp6HyYnR9Kjo6M62NclKG/HIeaygGcAZN26dW2xWPzWzs4uPtf+mZ8BkqFQGAAxpPsvtK7T
    mUsmUYIgmOByOfIK8ebqoYeHJ8j69csd/f2905FIJFhdXWPmOEI5jvMaJC52g0xtQR2TKKX4
    wAc+4NI07fbu7u7CRCJpABPczDMqLi7Cli3rEA7HcezYGWga5uihJUnB4KAf69YtQSwWQixG
    oaomBAIxABxpaipzulwuSzwe1+OzYJRxXaYYrTCgkhBFkR8cHKSDg4NRt9vldjrdmwkhvCCY
    YLFYmC4610lckmRQSuaRIRGDXGfWdSXXaTA37plMRuvs7ExHo7GwIVM2DGCQMdC6JZqxydTb
    uvnvf//73R/+8Id+XFpauPTo0ZOeQCBWXl1dVb5o0ZLqQGBS7eg4r9d8IJ1Oi3a7U00m04hG
    EwWNjY3Vvb29kywuWgFYfL5Cp6bRFeFwjKM050YCwMLznJ29R55J+IRVq1bfGAqlbhwZCZP6
    +mIajYbGwuGwTpgkAUQopQGLxRKqqKhoPXXqXFFf3zCsVg96e/2orCxRzOZUdyIRSEUiaqyr
    qyuuZ1gM7iIzQFmP0UaXjYcffrjSarV+o729fe8bb7zB6daF+hjMLQo1jiHP3MDoPNKzWcmV
    1WqBJMmQJOVN6hrm/s5sVsXISBThcNbR1OTbvXPn7iJZln42MDDYRSk1sSZfOmaY43kvLGQw
    YxNepZSe6B8Y/MGf/NlffMbvnyi79cZrUeBx5yarRkA5AqoBqXQaBw8fx3e//5/0+KmzSTZp
    jdX5F3gAX8GmIUaR+47Vq1ddfcvNNxNCkMcy59jFfN3zxWzrAAqzScCqFctR4HXP2AfNFhtq
    UDWK+ppK7L3pWqGrp9+eyYrZTFZU2TOBHpwppU6WqrHA0JlwIZ4R+/maysrK62+++SbebDJB
    VeQccM49JBAA+17fj8HhUQg8h2AwiAKvZwYD6IzFksVNKCstQWFpJQYHB7Bj81qU+LywWsxY
    srjZMjo+4dVBG6XUwqQGwqwqaGHH23A4MhaoxQ2ShKQgCMhkMvjnf/4XnDx5EiqzqKuqqsAX
    /+CzKCstZqtojnfS/HQcpTmJS+6DXnARMwTpEQCjhJBJzFa1vy3nEUOQ0pxOh+ND938Qu3du
    g5JNAarEvfb6G4vH/ZNNTD+v+2RmY7EobVnUTJYvXVoyMeGvZwdXFYCDUmoH0FJdVVlQXFQI
    UAqX00E4jrOrqqqn6DhW/Hft+tVtV+298WpyrrMHI2N+XSpiZfeTYvccA6BmxWw2OD0FtbkO
    kphFMBiCqmoRjdJuqDTONqt8O72L3jvP81RV1ari4uLPffGLX6y95ZZbZlpXz4LkWQkZgV4g
    mysIJQCKfIXQNA19/QMoKy2ZY01JCDA1NQ2/P4CykmIEQyFks9lcUyiBh91qgcVsmnk/s9ad
    OUcWnuewc/NaOGwW5x9/659v6O4fSgI4yAqIaF627bK0xyaEIJ1OK6+99tpvrrrqqpVbt66/
    7dixs0I8nr5AyqHvfzabGYsWlYPjkuKRI4dDsixHDPpXXd/7TlqMkltuucVpsVjuP3PmbP30
    9LQBPM9fFDgrY5i/CyEhHLLZLM6e7YYo6izWXKBNCEEmI2JyMiw0Nja5NU3TXC4nNzU1JTsc
    Dg/HcS5N0xzzxOgFlenU1NQsiUQiu/r7+2e0z3N1vxwaG2thsQgoKvLCbLawYsG5HRmnp2MQ
    xSympiZRVFSCkZEoNI1DLJYBz1vtZWWl7ng8bozRZhajpYWM0XnuOXphXZIdjonb7UnkWHQO
    bW3LUV9fy1jlnD3f8eNnEI0mMNvKey6my9dBG7XSAKGEcPpeEMRsTcoIIWTCEJszyCvcf4v3
    TwBQQTCZR0cnrB0dPeA4K0SRR1lZSUllZVVbR8eMw42USCS0srJyCQDGx/2mlpbGxVartTub
    zepuS/ba2jqToqjN0WiSEKJnGDgzzwvG7DTX1tZWa7HYP9TT43crigaPxyp1dXVMa5pmxWwj
    rDiAqKIoMY7jEjabnVosNsLzZthsNqiqIh08+MZwf3+/nz2LtJFpziOnSH7GZc2aNYLP57ur
    q6vrxgMHDnCZTGbmscx1izHq9rmZ4lCAIJFIoaioEDw/PI9NI4+iIi8yGZEVFeY08jm/9lzp
    3VyZz9yP6bSC7u4w19pa1LZ585Z0IDD1WCqVEticVA0xeibeCQsINvXPVUJImlL6Qm9fP/cn
    f/6XH3jyqWdadm3fLCxqrIfdZkU2m8HwyCgOHDpK9x86RkPhaNYQkGdeTFR/QZvQywmijZkI
    9v+Uud3uB++95+6i2prqGZaZGrsPgs44juTbns2xPmPft1ktuOPWXKc+TVUv4soBXHfVdvLo
    r562d/cO8FarxVZdWekY9/tJOp3RXRtsmC22FI1s5QI9oy1r16xpWbliBbvvnMNGLBaFKGZh
    t9oQi8dRWFgIWRKhahpkSUImkwZHCMwmAQTA4NAIVHCoqqxE55l27Nq6AaqqwG61YHXbUu6F
    V14vYAHBwV7GtOeVYLd0EK0D6CzP8zCbTNi0aQPali+Fy+nMASWbFYUF3lk3GaP2hM4CamMx
    JUWu2/dsteEMgA6yVH7AwDzrc159OwHaMG9lVdUy4XAIiixBUxVEo1HIsqQXN/kYAMoCyEai
    UdXjdgu7du6wvPLavlZZlkdz/x11MZ39ksaGOqvVYgEAeD1u8DkAbWf3wAOoKizw3nnvHTc7
    i32FONfZQ2PxREhPMeobIQPvEoDqyclJ7ze++Td4/96bEYtG8Mijv0plRVFnrHVWVg9YykUy
    CDP3rqoqAbBly5Yt626++eZcu3ZNmwOajbaLwyPDePnlV3Ci/SQymQx016DBoWHs238QG9ev
    nbO+AYIz5zrQ0z+Af/vxf8NkMkFRVSTicciyRIuLCrF1w1py87W74CvwGGobDJk3TcPq5a24
    7303uL/53R9ty2TFAACNSVR0P2x5oee8keAAoHR2dk5Rqv1w69Zt3K5dG/f094+7JifDRBRV
    VtFuhtVqQ3GxF1VVhTSVmlZeeml/ZHp6Wic2IgAieU2OtHmY4StCcLS2tu6cng7efu5cBzer
    i/zt7bov/pGA43iYzWYGzui8bBZA4PcHUV7eYolGQxmTyWyzWOy1O3fu8rrd7vTx48enDDHa
    iot063s7jSeM/66mpkYQBOGqkZGRykgkihxRlgP6NpsNPC9A0wCz2YxUKsU8vwl4XoDJZGYN
    rnL3UVjoRjIZh6pqoNSGUGgagmCCLFNkMqqprKy8oKen12GQDlox28Z6QQd7HhA9Y+jtcDhS
    sViuO2EgMIVUKg3d3k7TgGQydZHiybnP+8JW39AdQIwAOsBIDWNszrzd2JwXs1QAGYvFolos
    NqiqwJrC8DSdTluRcwqKAYgnk0lwHJcxm80YG5vA0qUtdTU1NZU9PT2jLLvhLCgocGWzYpUu
    M5JlFZQSk8VidrJ1yvM8b1m9evW1oVC6bXIyBpfLDpNJS42MDKfZOIrsIJShlKY3bFjv4nm+
    trV1ETjOjFRKRnV1qRoM+ifGx8cVzHbHpbh4EfQFX9u0aVNZJpO59fTp09ZZ8IwL1h4hBGaz
    BdXVlaipqYbL5WBNbyi8Xg8sFgtcLgeSycwc0G2zWVBaWgiTicfGjYvBcVyucN9sptmshFRK
    wthYlITDmQtkTPoBWZYphoYSXGtr8YqWlkUD7e0n9eJIyXDQUBYEQOeDTcNkcgKoA+BOJpOR
    A28cyp7tOO8s9vngcNghiiKCoTAi0ShyjY2gW03pbgyqQT7BGVJGyhVkogkrRnr/7l27dt12
    y81kFihfhF2mb948hWo5vXckEsV3vvcDfPSBe9C2dDHTiM/9eapR1FZVYM/2zXxP36BGQMS7
    33+bbWh4uOEn//3LIIB+A+C0Y9axYqFShE6r1XrVrt277W63GxrrGtnR2Ylv/vXfQhRFeL0e
    TExMYOmyNvT39+Fff/Aj2Gw2RKMx2G1WPPyRD6KmqgJPPP08amrr0dHRgdbGGpT4vFAVGQIv
    oLW5njgcdlcymXIZ7iVf231ZJDqGeasZTpUcAInjOZgtZuzeuRMAZe24lTkaaEpJDkgbJKF0
    HvJ5Hn2fXrwYwayvqK6HlY3B523OcQ1AKpPJPPP9H/xoQyIWLbdbzXj08Se16VBkgq0nF3vF
    AaQi4Yiiqaqwe9cO8sN/r27o6x+oZP+/C0CRIAiNjfV1PM9xIKAoLPBCMAk2SZZtbM1aAGzY
    tXXDkh1b1iOTyaCzZ0BVNS3BvichZxmYZa1hl3jczj83m4T6I8dOnD96vD3O5CD6M0iz9xYn
    hCQopUmDlMV4n0reQVcwmUxr1q9fby8oKJht7mPI/OhAet++1/GNb36LHj16TLVYLPxnPvNp
    0tzYAFnMwm634kT7KUQiERR6PWw9Aooi4/DRE1jcuhSf+tSnUOD1QuAoXnrxRXzr7/6BZjIZ
    +aXXDpJznd2mr37uY6TA654tqjW4thBCsWPjavzPE89XnevuX2xgtJIG2dqCgZK86nd9Y5NH
    RkbDhw4den7FihXWjRuXb5ck1ZlIpCHLKgRBgMtlh8XC097ebu3YseNiLBaTmMWWsW27UZJA
    rgTJkf97H3744WpNo586c+ZsaTgcmWEjfxefZ2P3OpfLhaVLFyEajSORSM/r0pFj9LOIxdK8
    KIrw+/3ZaDRtp5R4S0tLK9k+6KKUugghzjy2/pIfztVXX+0TRXH34OAgn9O05s6d1dVVWL9+
    DTRNgyQp8HhcGB0dRmVlNdatawOlgMViQiYjo6NjEIpCUVdXiulpPyoqKhEISFBVDjxvhqap
    SCQk4vMVeQSBdymKatxzdJJDuxzx2ZAl1J+Z5nDYs9FoDJRSjI/72eFIYFkHwcBWvlX2mRjZ
    asocJRLINYkJ5cVmyTjn3+781jOEnZ2d7WvWrD24Zcu63dPTcb68vAQWixbt7u7Wa5vcAJzx
    eJzyPJ+yWKyYnJxGMBjxLFq0aHFvb2+UZRHdTqejIZMRC3IFwRwURYOqarzd7nDqmGrx4pZG
    h8Nz3enTEyZZ1uBy2ZDJpBI5adNMUyuJUpqpr6+3bt689UFKhfdxnBIpKrJ3ud2yc3S0O3Ps
    2PFANptNY7YLZcyQXaT5Gnb9a8w+Dk6ns87v9y+amprKI+W5OTaDDocTGzeupT5fgeZyORGP
    J/jz5zshCBaEwzG0tjahpMSHZHJ8jnzD63XDbOZw/PgRTE5OAuDhdhdi06Z18PuHFFEU1dbW
    xeb+/ig3NZW8wNZRP3ym0yoSCWqvqalddurU6XFN07KsCF+PzzPOK8JCBGaD/rAWwHUul+vG
    hob6tkVNTb7W1hZrQ30dSouL4XY5IPAcZFlCMpFEMBjE+MQE19s/aO3tHzAPDo+6JvyTLkVR
    CaU0wjSTbjYYBLOFNurlCNLGan7GbqxtaKj/+Mcf/qijuLgIqt48hP4WmYbOKGPWjWNG902A
    cCSCzu4eZMWsQepwIeg2CTyu2rGJ/PxXz5iD4Uh2eGjYWltdTjiOuDWNFiDnN5s0+I3KC6Gj
    ZM+0srKycs2mjRsNAQ0Ih8Po7e1Dy+LFsNqd2L5jFz74wQ/i/PnzeOqpp5CVKbKyhiPHDuK2
    m65FKpXCue4+rN+wCc8//xwcwnKkUknYbTaoACrLS+DzeqzJZMoInm2GU+6VuvT0eQaAyhEO
    kiTh9KnT8PsnEI/HkU6lIIoiNE2F02HHVTu3objIl7PEnoOYjZKO2XGf1XxAYuNlzLjoQPFS
    Cif1wPXq+a7ub379W//nBoHn11msVh8hRGDG8VbDBpgJh8NKVsyiZVETdu/c4ekfGFzMWlp7
    AGhul7OmqqKMzVMNhQUeuJxOWzqd0bXJVWWlxTs/cPuNJrfLgfGJSYyOT+ie0mbD/WYBVHrc
    zi9tWLV0XUf3QADAk2wjKDKkuKNMvhEAMMliQDxvk9Y31Zx6LifVsJjN5vKamhqSW8dG2YY2
    k9Xp7u7FH3/tz7TjJ05kAOCq3bvtf/DZz6DA7UBfXw9eeuVldPf2Ub9/kvgKPDPgO5vNoH9w
    mJ44dRYn29vJJz/xMXgcNlQUe3Dg4AHy/EuviZIkZx5/6gXX0pZG2/133kqMMib9vWjsGTbW
    VdnOdfdXMo1ljBWsRA3sO12IeGaYS8Tj8Zhvv/32RqfTuYsQskOS5KXZrFg+MjJuzWREiKIM
    TctteOGwBTabBQ5HAbdz5x6bLItFiUSM+P0T6cHBIX8kEjFTSq2EEL2gVDPooS9f4YLBCmv7
    9u12n8/3ke7unp1dXd3E2ML5Ulp3A4Db7YTZbMrTW+brpXPPd3o6Qmpri8zB4LTo85VQSVJR
    WFhY7vP5ykOhkH4wMnpCL0jTEZ/P1xQKhZb6/YE52m6n0wmPxwW/3w9ZltHZOYKOjg7U1QVQ
    X18PTVNhs7lQVVWC3t5xFBU5YTLluvwVFxcjnY7DZDKzmg0O6bSMggKn3Wy22BUlrcfnC4oj
    L+Olzy1qsVgUvaNkaWkxYyStMJtzOuhcLADGx6eQSKRnxsgoFZirOMgx0QawrndAzo/N6QU6
    KKgHDhwYI4T8fXNz83BlZdFmu93eEolEiNVqFTBb9+GMxaISx5GU3oWxv3+EW79+xaLi4uLB
    qampLACv1WprTiazFkXRm+BQSJLKO51OJwCzIAj2trYVq4LBTGU4nALP83C5LDQanUwxi0KO
    HQqyTqeT3HDDDfdIkvAgpZo5GBx57vHHf3lIVdViRVGc7MHpUtspxs7rjiSqYb+a4wvNCtxB
    CClOp9OsQ6geo+YeTAVBwJo1K6jNZpZff31fdvPmLfaTJ89gcHAUdrsHy5Ytgd1uoz5fAQYH
    /WRW68+xQ7+Z1tTUYGIiQILBKOJxDX19E/B67dprr70WT6dTpqVL13kSCZkTRe0C6VouthAk
    kypxudwlJpOpTBTFKLvHSL5kZSE00IRpJa/3+Xyf3rVz5/qbb77Rtnb1apSXlcBmtcxohLWZ
    yvbcC5pKVFWBKGZJMpHgh0dGHQcPH1v01HMvV54611kvy8pTANpZoY3FcNoRcRlcJ/JSR5UO
    u/2LH3rg/pZtWzdDnWlbPbd9d76n8xxpx7ytuwnOnO1ANBqDzWI2uAFoTBYwKwnRACxuasDy
    JS2mVw8cFk+eOavu3LRSWLW0pbb9XPd65lmYxGxr0YxBo3Op1+JFi5ora2trc2NFKVRNw5rV
    q/CVL30BBw4eRCgcweHREaxatQqH3ngDw4MDsNttsJhN+OzHP4LWRc34p3/9N9Q3NKGnpxs3
    Xb0Tvf2D6O0bxIpli6GpMrxuJ1xOh2BIc+Zr7MiV2JANwEwDQDiOQyqZwuO/+hVGR8cgMVu6
    6spKABROhx2b1q/FTNyd05VxLv08R96R1xUw73VJriOsQFBhc+FVWZYHFEX54KrVq+/KZDKN
    J0+eDCJXgGohhNgopdloPCbH43FUVZTjfbffyj39m+eaJyb8o2xO2bweT0lpSfFM5qWkyAev
    x20JTE3r0qHWq3dsqVjTthRUowiGwghMhUT2lvQMgkQpdVnMpvtv3L15e1aUtMnp0CHkCgrd
    hjkcMwRnvZlOlK15bZ4N1XgJhBCz2Wyeq0PGXO3zb557jp48dUpiEjH7+nWr4bQJGB8dwl/8
    5be0xx7/ddbpcJgi0YjJeCBSFRWyLCMYDEp//Td/Q6an/KavfuGzxO2wYvumdeSlV/dTVVVj
    WVEUn3/lQOUt1+7ivW4HW89zJVw8R+CwWQljKL2Y9VW2YoHrUtg84h588MGq4uLiuzOZ7J0T
    E/7F4+MT9mAwjEQijVz1vl44qFt9mSAIJmKxWOF0OniXy2H3+Ypsq1bVlqxevbZmcHDAc+zY
    MYnZuIEQYkKe5dNCyzmM0g2XyyVs2rTpxmAw9LFjx05YZrXKFysQNBYQvTmwFgQe1dUVoBRQ
    VWoobppf7hGLpcDzlSZNU0Wn06r5/Um+qMjXcMstt+55+eWX0iMjI1E2vsZM4SUDaJ7nV05N
    TRemUuk572twcARmswXl5aVwOu1wOl0IBAKora2F213ArN+AU6d6IYoy6uvLEAiMoaSkHIOD
    0/B63XA4bEilROQK8yhMJovZZrM6mMzACKD5KxSbAYByHD/TGbipqR5lZWUAcjKVeDw1058g
    GIwgmcz3v8YMMz33a2Q+mYWKuU5g9HeUbeTfj0Yplfbv39978ODB/1izZk3n2rVr/yiRSJRt
    37592XPPPRcIhUICAGssFlcURU05HHZwHAe/fwqJRLqora2t5aWXXooB8JnN5tpgMCPohzpN
    A0RR4dxutxOApa6utsbjKVp95oyfV1XAZOJhswk0EAiJLMNOAUgcxyk33XTjbp53fTQcVlxe
    r9p/9OjhY6IoUsOanulZwIgNY62amhef87OG0DTNomkaYU2PYLRU1DMChYWFKCsrpS+//GKS
    4ziF53lHLJaG3V6IpUtb0NBQLo2MDKo2m9U6K9XK/awgCBgeHlIDgYCya9dO87Fj57hwOItQ
    KE3q6ho4t9udOXv2XLS5ebHJ53M6JyYSc2oGjEy0qgKEcBZCiJeRSa68rMulA2gGOp08zz+8
    atWqL3z+c58ru/HG63NODJrKmk8ooEw/a0x159iY3CQ0m0woKPCi0OvG8iWLcOM1Ox0/f/yZ
    DT/75VPVYxOTT1BKX2OnHb24TC+yWjB3DsOCIJRSN0fIZ26+6cbrH/zQA0Tg+RkNMPTKfINu
    WdNy3X9mK/znZ5QJCAJTU/j5L38Nm80Kj9vFLLW0eWQguZ9xuxzYtG4F2ffGUa67b0h64dUD
    /OLGGltX/9DGVDobBDDKBtVs0KRdKugkABqbm5udbpfTkA6nsFqt2HvrzbjxumuQSiXxd//w
    j3jhhRcw0N+H9++9Eds2b4TVYobH5cTJ02dxvncQa9dtwPjIAL70ic/g//77Izh84hSWLW4E
    4TiYeA5Oh41jAdlieF0xf9m8YEgAcIQAHq8HX/qjL4DjCF566WUcOnwEX/vqH8FsMoGqOcI6
    VxWcr8qlFzDS5Hd7L78LwNDBZZoQMkUpPRAMBnc+9NBDFZlMZmVXV5cfQIDJJrRYLJ6ORCIF
    lWWlWLNqJW6+4TrPD370n62sAxXndruchQXeGYvGwgIPykuLTd29/U4ApeVlxUtuv+lq3ma1
    gFIN4UgUsURSMexKCgCeAHv3bF13zZZ1K7i/+d5PxlRVO83WsmpIBU4DmGap0wQu3iaaGgM2
    c05QVFVNRqPRC1hf/e+qqqCzswuqqkoA0jzPWwrcTqKKabz++ut49rkXU5qmBSmlZYqimmbl
    HxrMJgEulxMApGw2m/nJT3/mWbdiieWW63ahrLgAFrNJS2fUBIDU2ESgOBSO2Lwux7zvRZZl
    ROMJvfrdasi46ICELCAA4T//+c+vEwTha729vXtOnz5r8funIMvqHLCcs/6aC7xVVUM2K0EU
    VUQiaUxMRInDYbdUVPiam5qWVVRX19QfOLD/ib6+vm5KqY0x6HED06pepnVKHnroobWiKH3t
    6NFjZTl7Ov4CNksHyzwvwOFwIJsVoWnaRavz9Z+rrq5AbW0lJiamoKpqHui+UMYhyyqSySwx
    mczc6Ohwqq6uyWYyccTpLKyoqakpHRkZ6WTZH4HoQuVLvJYuXWoGsCgUCpk1TYMRTEiSjHPn
    utDdPQCbzYatW9ejsbEBPl8xzp4dRCSSgKpSyLKG2toK8LzC5ooV3d2jWL3aBp/PiUxGYw1X
    OBDC82azxWKYp0YP8MtGcuTHQ0opTyklqqqivf0MOK4DJSXFWL58KQ4fbocoKjPd6WbNE/LZ
    Z5L3NbylIVkg60HdSzujaVr03LlzHcuXL28PhcI3WK2W6quvvnrls88+64/H4xZJkiRRzCYd
    DgflOJ7IsoLu7kF+7dqli6qqqib8fn+W5/mydFqcsS+kFEgms8Tj8TrNZrNz6dJlzZGIWBCN
    5jTSPM/DbObUSCSqW5BqAOQtWzYvKSurfXBkRCoqLBTkycnhYwMDg0kGGvXmTxEDeI4a4nNm
    nvicT+IRSZJSJpNJFQQBqqpecNClNCedkmVJCYVC6erqaprzgrajsNCBpqYa5eTJIwGz2Wyt
    q1tsNWaDAAJZVgCYtBMnTiQEQTCvWLHMfexYH9E0MwgRiNvtzk5NTUWj0bDT5ap0clzyIrUN
    HMxmHqKYJaqqWvOIPWHBGGh2grmvqqrqaw8++KBn6dKlGBocYmBTNdh5Gfxz2ddmG26oM765
    uhUYQLF72waiqUrlj372y/si0biLUvoU00bLBv3dgqULDVIUHsDeTZs3PfSVL/2RpajId1F7
    OlCKbDaLn//il1jU3IgNa9fM6JeBufZ1hAMSiQR+8B//hdffOIL33XI9fIXeC9pB6yy2vnFz
    hGD18lZS6HWbguFo5L+feD5hMZtUWVF0qx29ApozNL+41Gdi4jiuuqqykhMEHqqqzHEkoFrO
    VaDA60FTYwMOHT0Jh9MJh92OyvIyyLIEURLx5LPP59jn7m7s2rIBdTVVWLe6Db954WXEYjF4
    3G4QaLBZLRwLxiZDYH4nGzRweifBggIveKaHTmfSAJ3t+qJpFFRTmLZ9lu3U1dBGRTSd/5By
    uTSDImb9rTsGBgaOZTLpW/7kj7/q/vJXvrJjYsIfYkFRiEQiqXA4DArAarXgw/fdwx1843DD
    ufOdkwCyvkKvxWa1zsxLm9WKluZG7rUDhz0AbDu3bChoW7qYNQbSEI7GICuKZjilUwCbVi1v
    ufljH9xrefqlA9qYf6qDBV1dI50ghEwzdlz3v76gYONiF0sRKqIojvT09KiKIvMcyyjlFw8y
    hlplG5gzlYzTdCJKDh85SuOJ5DSAuNVqKXU67CxGaTOMV1lpsX5AScYTCfryq6+X7NmyhiiS
    CFYcGQNg0jRVzb2nfBCfq4EIR+MYGBnXmS09jcrn7e6XfFVWVpJ77rlnraZp3zl69OiGEydO
    QhSlmQYSOdA82/44B6g5g3fu7Cv3fQ6iqGB4OIhoNOtYtKj0qquvvrbM4XD895kzZ05TSnnD
    HDR6ui+oxO6+++7zmc3mPzx69Piy3t6+31okaLFYsG7dKoyOTmBgYHhe+Yauf6yoKMPy5S2U
    4wgmJ6fJLAN9MdlHbppHIklSXl5geuaZpwPd3V1+t9udyGazI2NjY4OY65m8IGPc1NRkUxSl
    IpFIzDR6yS+I0jSKTEZEPJ6ExWKFKIqQJBmpVBY8L8BisaCurhSBwChKS8sxOhpGKiUiHE7C
    5yvA1FSWrS0eAMcJgmCeJ0Zf6SZnXE4aBeYdzEFRVJhMwhwwrD8DHWQZ2ef8Ichjua/EpbD4
    F89kMn6/3/9aaWnJltdfP+DdsGH98muuuSb87LPPxtLptJZMJtNOp51yHEcI4TAxMYVgsMq7
    cePGZa+99ppEKfFlsxKMzhLxeAalpT57XV1tWXFxeW1nZ4jLSbNylpQcB6RSKb2+TG1sbChd
    sWLNbRMTSo0kabBYpMkTJ070M5KFMpAcYgXvumRDb6TzljsxxmKxaZfLFXY6nS5RlObtgJpr
    4EYoz/Oipmkaz5s0u93LFxb6QIiWGh4eDqxcubJClnOFo7nx5ZhzjgKr1Q2e58Vz584lGxoW
    WX2+IouqmsBxvEZITrtNKS2ea2c5FzxzHAeHg6eTkyFJURRiiNFc/tq91HShA8DKSCQy8d3v
    ftf/j5pmoppmopQKFNREKeWROzFaCgsLLeXlZTn/OgqEwmH4/ZOU6vqF2eYZKgE0EGigULNZ
    UQOwBMABANMGXz5uoUEIs+OpKy8re/gPPvvpgqVLW6HIygU+zdALx6jGNN0yDrxxGOtWrwDP
    51qkanQ2VaSqGoaHx/BvP/4Z/fHPfgGn3Y6br99DLCbTjI2d0TPa+H9pBKivqURddaUQDEcV
    AFOiJAeRsz4bwVwv1oV6Hiar1VpUWFhIZlhUBiaMgEDTKAq8XmQyGRQUFsAfmMoxPAA6znej
    q3cQa9dvwthQP/bs3AJQiuVLFuEXv3oaw6PjWN7qAKgGkyDokzT/xV1uCcdF2HeO4ziSzWZx
    4MABTExMYGBwEPF4Av/4z9+DIstQVBWKLENVFFgsZty592Y0N9blKDdKMbfLN70YiL5cekGZ
    AWRKCIGiKC89/9zzO3784//wfvUrXy77+tf/8sZgKCQDCKRS6czkZIB1x1TRungRPvupj1m/
    +md/uSIUCgcLvF5BEPhclgQUJoHHimWtxGw2+5x2G7n52t2c3WaFosggoEins9BUjRgAYVNt
    Zdmezz14l8tmMeO1QydilNJBxrrqXc1CTM4xyYKzaAAd2m9Ln7K/a5qmnWtvb09MT097y0pL
    ZuVVLHvCcwQrVywnZrNZkCRJ0TQtOz0VQDIWxujomG4TqZaVlgjFRYUzrdtzHtEcaqsrCc9z
    RFW1DABlYHDQF40EhVgsBkVRdGeVQq/HJXhcjjmt36lBC3+uqw+jEwHZsPkYLewWpMCMUoq9
    e/e6bDbbJxOJxCqPxyNv374NhOTO2fpHgCP65pwLgZzRK5XqelMDKKU64OR5Qt1u95KtW7fc
    MTU1Nen3+xOG7NGCAyvd87mqqmrn+PjE1WfOnCN68dybyTIkSUZ//xA4LtddUNM0qKo2A6Q4
    joPT6URdXRXKy4s0VZUwORkkExPTc7TFF5d/ECSTGVgsJbzT6aRDQ0MRzDZI0hn57EISPjab
    zSLLsi+bFfOsvOY2UQEIRFGGzWaBLMuw223IdZQnKC8vAs/rtV5WTE3livJCoSQqKophs1mQ
    ThOd3SdcrjpPwDvXJTjnNceAU0VFBQoKCuB2u0AIwcqVSw3AmZ+xPhsa8iMYjMNoWzereyZz
    QPQVwNK6c0WW4Q167ty5I7feeutZr9e7bf/+g+Y9e3ZvufrqqzPPPvvsG8FgUCotrVRNJhOn
    qoCqajh3ro9s2bKqbvPmzSrP825ZnpslSSQyEASLdePGjTWplObQ2edcNoYDpZQqiswBIF6v
    17l9+86diYRp8eRkBvX1Dm1qaqB3eHhEZcxrEkCMEDKF2QZfqTwC860AaNLd3T25bt263rKy
    8tpQKJpXxJd77/F4Ejxv4svKykg2m83yPK86HIUmr7cA2WwmKUlioqiomA+H02SuHR1BNivB
    bLZyDodDiUYjiUwmnSkoKLAkkxQ8zynZrBgzmUwZt9tjTibVOdKu2cMnB4tFgM2m0bGx0Syl
    9KIyy4UA0EkA34rH48XxeLwAQLHh5cNsK9Ml11yzp+kLn/tMznOVUjz1zLP44z/7S1VWFF3X
    rBda6K8MZgutAuz7HGOJyeU4NbKOgzu2b9u6YuWKNgwODqKstBQCzxnYV6Onc25j/eAH3ofu
    3j78zy9+BYfdhkVNDXDYbVAUBdPTQRw+3o6nf/Oi2n7qrEoB/oEP3cFtXr96ttPZTLFhvsZa
    A9UAr9uJlqY67vjpDsEAjvSXXhm6kA4cPM/zdqvVOsueGQqiZq29AK/XA0VRUFjog98/CVVV
    IEkinnzmedQ1NKG3txs7t6xHeWkxVFVBeUkxiooK0dU7gGUt9TMZB8MJj7vYae9KBmqOcMhk
    sjjwxiGMjo7C5XSgrrYm14yBEIwNDqGuphpOpx02iwU2m3UO02mUKl0WyvnNA7RuG6cXdhw5
    c/bsmTcOvrH9ww88gGQiUfN3f//tWyORyD5RFOWRkVGqaSqhWk5qcttN12PSP1n4d9/5F4/D
    bud5nptxtACAlctaUVleaq+vqcSqtiW5YiN271xuT9Lt68oKve62T9x/R/HmNcvxX48/R4dG
    J8fZe9OtF1Ms9W/UO/+uQKPzzJkznYcPH9502623GFw4ZoH0zh3bsLJtue3o8RNOAPLkZIAm
    oiGSzqT1AiLX8qWtQklR0cx9UdbwqaWpAR63WwhHohqAbCweV5PxmDA9PU1VTUsBkAhB0Zq2
    JZYCr3t2ndBZjXwqncGLrx+hqXRWT4tmDWz7glnYCYKAWCymWCyWJ1RVPVtaWupIp9PuTCbj
    FcWsRxRlt6qqNlVVrXV1dZWUcmW9vQOsYxuHysoyOBy8dvbsWRGAyHGcyHFcluO4LM9zGY7j
    JI7js2NjQkZVFX86nabINW3gyYW58gW77rjjDhul9Jrh4RFPeXkZTCYTBgaGoGl0HnCrF/oR
    BIMRNDXVY8mSRVSSZMRiCSJJMkwmAS6Xi7pcdjWdTojRaIgrKCi2dHUNkGxWwvxM1YUAWpZV
    KAr4kpISYXR0VGHjmTXsbWlmVbhQjXJMmqbZFEV9k/dHGICWwHE2iGIWTqedgQQzamtLMTk5
    grKyCoyOhpEDYgISCRGqqsDlsiGbxUwDEmNsZv7l5B2I0VyOpeRRUVGGkpISKIqKeDw5Q+o4
    nQ6k0yIymQwoBZMr5Us4SB4jjcs1Zd+M5NAA0NHR0ZFkMvniokXN63t7Byyvv37IvnPn1qv2
    7Nlj9vv98erqOtlisZrSaREcxyEeT+HUqR7Tpk1tTXa7jczeX+7gm80qSKUUvr6+2n3kyABU
    lTI/5Jn7I5RSk9lstu3atWuDIPiW9PSkOLNZgMOhJo8ePefXNE33+df9ofX+HInfxjrnExw6
    s9ze3h5dtWrVgcbGxu19fQNmRVGR3xQlkUhhcnKaX7VqlefQoTeSAFVcrgLYbE6o6pTs9RYQ
    j8fr6O3tvWCtZzISVJVwlZUVXDAYFGVZEm02L/N/1uRUKpmsqam2uVyFBWNj8TnMt37oBAgK
    CsxIp0PS6OhYyhCbRbZ+5+CsS9VAa0xGILKCNt3LVLfpsgNwmEymxsUtLairrYGiyOBAsXRJ
    K2x2G5XjiSzbOEOYFaXHDBtLmn1NNy/XLgeZx6xWrA6HY2Pr4hbr33/7O1AUBX/2J19Bka9g
    poEINRQEEQCyLGH/wUP4p//v3+iRYydEnue1Il8h5/G4OFlWtFA4IofCEVFRFJjNJuede28y
    ffLBDxKb1Tyjq/5tVnhmk4CWxjrC87xJzbUDFA0HjCQDIUZN+O/M2jY1NSHXEhYEmFuENVMN
    R2elCjkdtwqXy4X+7hGIkoSO893o7BvEps1bMT48kGOfkevWZrWa0VhXg77BUciSBEWWkMpt
    vto8EeyKBme73Y50Os1oOMDjceMLf/BZQNNy/qmsc1UoGMKXv/bnuOG6PVi5fGmOiVZlJseZ
    H9PSOSnEKwOiWVFhhlI6kUwmX37q6ac33HzTDZZPPPxRQqlW9w/f/q4tFA6nhoaHqSIrhBCw
    7nlmPPShe4nDYRPcTicEjpvjIlFVUYa79t5IqspL4HY6ZrIohAAejwsCzwsiUOiw20o+8oFb
    fLdcsx3JVAb7j5xSZEXRO2lJhsPgBcXBb1d/yNicqWAw+MJjv/jF6p3bt1l0/f5MNkejqK2u
    xmc+9XHhj778J2WBqan49PQ0zSRjMPGEAHC5Xa6y667exVksplznTf2+NQ1LWpqwesUyy0uv
    HXACSHCEQBIzmJyapppGRQCl9TWVdbdct4uYTQJURZ2zlgmA0x09OHDsjGK45/m6ri4IW/vT
    n/40C+B55IoUfQBKAJSyV6FOcHi9Xls8nik7fboTgmACIQJEUUZdXbF26tSphCzLMRajoyw2
    G1lV3ZN+xsrqchEcAFBWVubOZLKtlAIbN65HV5e+kdI5KXtjI5GiIh/a2lopoElnz55MWixW
    ubi4GA6HA6qqKqHQhDg4mCCVlRUFxcXl7q6uIW56OjJHV/zb3DtycgmJFBeXmAghCrP7SxNC
    0gZSaMHGl6WWydziJw5zvY/JDAPN8wISiQQcDh84jkN5eRE4TpepWmbY59xhQEMqJcLlciAc
    VmAyEVCqUVmWaR7SfCcANAFAVFXFqVPnwHG6JJVjBaAm7NixAX5/ECMjk8zWTmAWf/nx12hl
    N29svpz3preEzlJKEz09PQdWrVo9UF5e3joyMoYDBw47Nm9ev6OgoGDEZrNoNpsVmYw0A5Kn
    piI4cuS8UF1dBt3CTgeBlALj4xHCcTzC4fScuaqqAMBxbrfb3tKyeFFJSUN1X19KkCSKkhIT
    kslQcHBwSO9GmwSQYS5fxqZJ2u8Qn5HJZOS+vr79DQ2NA5WVlYuHh0fzQGzuvZ8/30t27tzk
    XbVqtUwIiNvtBc9bIAgm07Jly8rSadkejSYN8g3dw1nD1FSSa2lpdQ8ODgZzjLsFdjsPSYqK
    VqvVvGHDpkWxmGZPpeR5axvMZh5FRQI9c6YrlUwm8+PzBU2jhAWcDDL7j3S/5iT7vNBut5nq
    a2uJpuX8dCkoSot9KCkqIvF4QmKDE2ZpL920XLeN0S2/8k37FxREs0VkdjjsZXa7nUwHg/j4
    Rz+CAk/OzzUUDmFwcAjLlizO6YgIEAyG8JOf/Rzf/9GPlQn/ZJxtMMlkrnxZ12pzAMpKin1V
    93/gDsuDH3w/KfB6DOD5zVt/U6oBHEFtVRlcDrspGk9o+sGCTeykYTO+ZI/Rvr4+AKCqomQz
    6cyFzSAw+3dQCq/bDappMJnNSKbSiMfiePKZ59HY3IKe7m5s27gG5aXFOZDFNN31NZU4f/48
    0ukU4vE0pkNRzXA40i6HfvJtTQXWZ5EjBA67PafhV3POMRp7KYqCdCqNbDanEyQMQM5hHKlR
    xzFvTL5cOmgjqJQZiH7jyNGjw2fOnFm0ZfNGfOJjD8HpsJf+7f/5tjo0PEziiQQ8blYwqqmw
    Wa144J47oakK+92zLatNJgEfe+CunJM+nduJsbK0GB630wZK6z98183W++64gVgtJnR09+F8
    76AejMxsbLN5wXmO3vntNJUghGiEEJFS+sqLL71884svvrT6jttvm5M5AWOEb7rhOoTDYfu3
    /vYfLP7ANJeIRVBc4OQA1F61c5t926YNBjed2blfWODBA3ffznd09ZT6J6csPq+L56FgOhim
    AHylxb6GT334bkdbazNrSKHN/J8ARSKZwqNPvUyD4WjaAER1z+vkfM/gd71y52wQA6rSY2mS
    ZQckABGLxeIqKCgg4+NBoqe+c125RFgsNjgcDikajcYx23o+mEdo6L65Scxq1i/bulUUxcTz
    vC2dTqO7uxfd3T0z7HNNTTU8Hhc6O/tAKYXJZEJ9fQ1aWhq10dHh1LFjx0LxeDzCnneKEJLl
    eZ40NzeXrVmzpkVVec/p071cLJZ4201YAIJMRoLX6zUJgqDKspw1EByJefavS7rS6bRGCMma
    TKY5LNpcbbauDZVgNlsgSSK8XjOsVgtqa0tmtM9jY5EZ9lkvREulRPh8HuTaZfNQ1YSaSqUU
    Q1zWLheZ9dsZ6Jm5AEIoA8lzC0dNJgGCMKvdz2ef85urXNi5cOHB83yuInq28OTJk8MrV648
    0NhY3zI+PsmNjwdw4MAx25Yt65q9Xi/cbicikcQctnZ6OoJwOGloSz0LJoPBJKJREZo2I8EB
    x3GseJRw27ZtL6fUYRoYyAipVE5D7vHwam/vQCibzeqnEqPdqr6+tbcbn433um/fvv66urrf
    tLUtbwwEgqbcGM6du4lEGkePnuE3bVpdUlxcRNNpDrIsoKSkvNDj8RYeP97DKYo2p2GSPtfH
    xkIoLW1wb9++o8Lt9tjDYTNsNhPMZp7ftWv3cpPJV97VFSKUEsO6nl0/RUVWZLPTcldXlx6f
    8/2uxQVjoA3gWWIbgN4kIqbr4CilrRXl5d6G+jpWWEhBqYqS4mI0NtRxfQODYIMUZTqbcaa1
    SRgWqc5qZ9hLWujFy/M8ZFnWMulM0mQS8Lff/AaKfYWMmZbx00f+Gz29/fjbv/pzmAQeZ851
    4Dv/8n361LPPi1lR1C239DR0nL1fL8/zKzesWVn3iQfvNe3YsgFmpnu+sPtgfsfCWf2kplFU
    lBbB43aaovEEMTyPdB6AVi81BWyxWCCKopoVxVAwFKRU08hcMJjvaeuF02FDNBIBBXD0xEn0
    9A9j09ZtGOnvwdW77ptZbHrnvtLiIiTTGaTSaQyNjmMqGNY7/Sh5r3cCQOd4HY4gnUlj377X
    MTwyjFQiiXQ6jXQmB5qjsRh+8t+P4pdPPpXbrAUBt914DVYuX3LBxMylBGacOMjlZDnyGsRQ
    1oJUBDDo9/sPv/jSy83r160hVosFH7rvXlRWlPPnzp0zzMHZsSWg4DgyR8qgyyKsZvOMA40+
    Z1VNQ01lGe648SpTocdluuPG3bBbLdBUDSfP9dBgOBY33LPE5uy8gentBOe8hgvDwWDw8X/7
    4Q8XbVi3xllRUT7LQjN7SJMg4P57P0CKfYXC93/47/j+z36NRCLJrWpb5vr0xz8CjyfXOEg/
    MBoLaHdt3Ygvf/Zj9r//lx9aKdXIkfZzGBoPcMtbm6s//sBd3LW7NjNHnhyRTmdahwOvHWrH
    a4faZQOTG2VgNIoFdhYyPBOjJl73fE0hV1XOVVVVVVssVkc0GpvjXpFKZUGIwBUWFqrRaDQF
    IMyKiCYYkNbjvZoXiy5rpnBy0p9ubl407HQ6V58+fQ6KktN/+nyFWLWqDSMjY6CUwuVyYdmy
    xSgsdMsnThyLnj9/PqSqapBlTCOU0oTb7TZt2bJlRVVVzfKRkYBzeHgCsqzMkX+8FT9pXSaS
    ycgoKnKZ7HY7H4vFjDKOjEFytyASjunpaXnx4sVhm802h23P7+iWOwxlWYdJEwSBQ1VVKThO
    YY4kNgQC/nkaxMgoL88xui6XgEQiKqfTab14X2F2meoVjtHEuF9XVlbA5/PBbLbAbDbBZDLD
    ZDLB6bSjsbEKVVWlM0WeY2NBTExE8mQb8/tCX9bU4Nz4PBO3MplMYnJycl9lZcXtHo/HF48n
    MTUVwmuvHeaWLFkESVLmtLk2MrYXyndy/y53sJzLslJKMD0tktJSl210NIVUSgMhPGw2ATwv
    Sv39/WkDCZohhCQWIjbp7bAzmUyqvb392Q0bNuxqbm5c2dnZcwELnDsARHDgwAlu9eplKC+v
    g6JYYLFotp6eIUxOhi/SPZQgm5Vx/vwE19ZWU1JU5IEo8nC7eTidBQXJJF/Y1RUi2azK9PHc
    nMOVzWZCcTFHjxw5lYzH43PiM8kVDqQXmoE2dgnSg7TeNZBjzVW2LV+21FtZWQ5NZWBRo3A5
    7Vi7eiV58ZV9Fk3TJABJ1jxligXpsGHAtHnAlbKQATpX0QspkUy+/Mh//8/VVRXlvnVrVyOb
    yeDZ51/EY4//Gn/y5T8E1TT8zy8ex7/83x+qZ86dTzLQHGSbyiKxX7sAAEURSURBVBT7ewZA
    fYHXs+nOvTdWPnjfXVxtVcWMG8lv61g45/saBSWAx+VAgcctDI/5jW00NcMGKS1EMJNlGQAU
    TdNGh4aGVVEUc0Vkc9i4WRbaZrdiyeJFeGnfG+Cg4Ze/fgaLWhaju7MT2zauQUVZKTRNmVN8
    6PW4oCgaYrE42s9101gilTVoQPNbGl8xhmM2lUcIRwjS6TQOHT6MYDCIZDKJdCqNVSvbUOwr
    RGNd7UwxytTUNLp7+mCzWoE5XQnnOkAYDPvnA9GX05VDBhBXFHXfvtf33/rgh+/3lJeWgOc4
    XLtnN3Zt3wKe2TAabdfyP4fBfcboeW4EmHabBZ/60J0QuBzDo6kqMpkMTnX0UEVVM5j1hs5g
    br3D79zhyxCLFBaDXj5w8I1tP33kZ9d84fOfJRxHLnifAs/j5huuxdIlLXjs8SfR29ePz/7B
    +7CqbdmsS9AF3UFz7jPvu+U6VJYVcz977Nd46rV23HbTdeT63VtIXXUFoGmG4sPciwMw5g/g
    vx7/jRZPppIsKOtxI0QIiRg2qUt5BvOStobfq7PFZuY2RFauXFmfyYiF8XhyzqYkihLSaZGr
    qqoSBgYGsgASlNIgIcTPSI6oYX2qhnWrv9TLsT5/85vn0h6P95Hly5e0OJ3O1kBgitjtdixa
    1ERVVcPQ0Cipq6tGa+siLZWKp1544fnI5OSkzp7rLgKRpqYmz7Zt27ZRyi8/fbrXFApFoRcU
    vpXmK7Op41kXD0VRwfOCyeFwWGKxmL4P8nnxekEGtqurK7Nt27Yxr9c749Awt4iQzCmuSqWy
    qK6uAs9bUVdXitHRPpSVVWBsLAxJUg2FmLmkRe5rgNVqhtsN2tExlpFlWcyLzzIuU6fYNwfR
    OU1veXkZCgsLwHE8HA47AoEQRDGNeDw1k3yxWMwoLS1ErtEILtKNcM7nl5XgmC8+E0JUSql4
    7ty5juuuu+5sZWXFzkSib6ao7tixs3ke5xcWvs3vZzz3e/r3g8EswuFcclz/vS6XCYnEZGZ6
    OqhjCNEQn3Ut9O/cp4BlQ1UA0rFjxwYaGxsfXbFieePk5LQrFkvMkXHoH2OxJA4ebEdtbQCN
    jbUYHIyhs3OIOXVwc2Qrsw4aPGKxLE6eHOEaG8tQUuJBMimitzfGhUJZ5nd/YZ1ATtZkQzA4
    KHZ1dUUY/tRJUb0T5QUZwoUA0IohgOZPOJ/dbt+we9dOs91mg6rI0DuacRyHbZs3kP8oKXZN
    TAYseUyJLlrP5p3Y8yq0Lsv18vETJ//xM3/wxXvq62qqs6KI7p4+umL5UpemqeTLf/p1+uun
    f5OKxuK6XjvMgrIuP8kAWN/cWHfNpz/6gO+W6/fAYbNCvYgV3psD6dnPrWYTigo9um+sGbN+
    nKYFlOIYzd77Ojo6YqFwyFdaUgyjA4cRXBAAN1x7FY63n8TY2AQUVcOa9aXo6z6Pa3bfO8PC
    wWAnZrOawXEEw2N+HDx2hlJKU4bsgl5YJV3p4JzXuQ2FBQX48he/AJ7j8ORTT2Hfvv341Mc/
    CitrgAMm2zh46AjOnu9EeVnJnAaE+bP2wrThFbkopVRl8oazXV3d3SdPnlpfcf21M6BQ4HkD
    aNQL3+YB05jrUT5TWJh3uDKbhJnDIiFAIBhG98CIYjhs68WDSczVPl9qLFJZLYBfFMWf/+Sn
    jyzbtmVz5aaN66EZ7ezobHFhfW0NPv/ph5HNZKB7WRsPiHPcZxiDzRGCLRvWYMmiBmSyIop9
    XnDssJDfNhygkGQFjz3zCj3Z0SMaYobOhuqeqiksbCGw8fCkZyHmCGXLysqs9fX1S86c6XRk
    syLTk3LMOYgiEIiQmppqh8PhQCqV0gvioob3LeFCy7p8GdaCXpIk4ZFHHjm6Y8eOb9TV1d1a
    VlbcnEqlBFkWPTxvrlu9uo26XPZ0d3fX9KlTp8KZTCZukAf6AYQ2bdpUtWrV6lunpiLNXV2D
    XCYj/lbg/FbY6JzlncBbrVY9RpsvFqONAOR3OSwlEglFFMXukpJi0Wy2WI2uIrPeuDrrSDEw
    MIG1a5fCarUgFguDUg08b0cgMGnQgs6CEUXJ5c1cLhMICatDQ8NvFqOvqEsSpQSKoqC9/QwI
    4VBRUY7Vq9tw9mw3sll5xoGDEB4FBR6UlxchlcqBp9k4TPJafc/5nFzhAK0BkHp7e6d27tx5
    sKamckt//7BJ0zDDoOcX213oYHHx7xsZa52FzkkYZjMtLhdPh4fHRUYk6jZ7qQUkOEAI0VuG
    p1577bVX9u69fVNb29KbDh06TmaBLZkDphVFw8DAOMbGptn4EQN7PL8jTq6gUEFnpx/Dw2HI
    MoWqEjYvyLy+z16vBQ5HVj1w4GhMFEWd4JgGEGRNvXSCQ8Vl0EDPBMu8FMXGFW3LN+/auZ0g
    D0RpmoZlrS3YvnWj7X8ee7IcwCBmWwzrHbl0gH7F5jIhJEUpfWoyEOifDASWIFdsU9XZ1bPl
    s1/4qjAZmJpiD1bf/MMMOPsByISQazauW3Xdlz73ccemtSuZHlG5QK4xx21jDvM3D6gGhcnE
    w+t2EkNANlFKzawDGJ8LLHQhFrM+dj3dPT0Dp06d9l13zZ4LPHWNQKqxrhZ/+42v4a///p/g
    LS5HZ+d5bN2wGpXlpTl/4DzwwrHItf/oGXT2j0gGjZVeEJnGAjfKeUviulxDDhZxcnPYarGC
    4wCXwwmrxQJVzTUIUlXm/0woBJ6H3WqDqurso3bB0pitviFXJlc4N3Dph9PJUDh8ZP+Bg2uv
    3r2Ty9UGzsqGjBKN/OYfoPMdoIxrWpvrFa7/e0IxODKBqVBEP2grhnS/vinL+fHjUuRk7Pef
    6Ovvf+oHP/zRg62LF5ncLueMReScTAoDxHabjbniXHhvFzZDyd2fx+2Cx+XM6eJV1fAstRkA
    zgE42dGFXz77qqYoaswAnqcw2zTm/2/vvePlOq4zwa9u6vxyxgPwgIcMECAJEkwAKeYoibRF
    URJFZWvG0siSxpZ/P1sa72pWs7ujGY+1M5bXI3tGnrW9XEljyTZFKpCiKGYQgUhEejnHzuH2
    TVX7x63qru7XDwCBfg+esfv3u+jw8ELdqjr1nXO+850M6sh/rlp8WI7a9cQTT/TbtnPHyMi4
    IqeCBfCam0ti06be6KZNm1qOHz8uOoOKE110JvNWEWiAEEI9z7NffPHFk6qqJsLh8DrTNDvW
    rFmzad++fTfk81nvzTdfXZyamhJdEUv1NYqixO+7796tmzZt/ejQ0OS6oaEJ3ihFvWh770t5
    5ip7SjAYLAFnLrsq7hth9Ust0OnpqRPd3WumW1tbNs7NLdbkdIq/b2EhjddfP4V9+7ZhamoC
    3d29mJxM8kizukRdxAcqQGsrYRMTI8X5+fmsBKjyUlTyqkSg/W5xlGdNXdi2U5o7oWUuGqn4
    KhXKko6EteoglwlwrGgUmkdnXcZYYXZ29q2urp75xsaGNTLn+ULR5Wo1CXkPXwx0Awp0XYVh
    eGx6eko48EUpwFGoptddSbSdMeYCMGdmZubOnj39/a1bt+8ZH59eNzk5i0pZOnkvqtw5JcuM
    uXLdyo5DoeBy4Lxc7YLfNKWnx2Dnz79VmJycEuB5Hn7TmAUe9MjXyvJrV3JTaqS+S+F1Smlv
    LBb77Ief+GDb2jVreBEZq0j/RiIhPPn4Y8qrb7y1bnJqZhzAPGOsmbdPzEgRDu9Cv7+OQIPB
    53NnAQwxxrLwK9enU+n0PNIIVkV0ctw4LwBwVVV97L67Djz8e1/+XGDrpg1lMCUDYklblpWa
    rbCLRqc1RUE0EqoA0BJdRq3jPRAH7/zCwuIvvv+Dv7n2lpv26dFIqNRJkklFY+CbY2ZmFgvJ
    NLbtvh4DZ9/Bfb/x4dLXZUk3BgbP85BMZ3H8nXO0YFqiwEYQ9jMoV6yvqnHmABrwFSZhWRZ+
    +dJLGBsdw/TMDPKFAv7D//VteJ4Dz6NwXQeUUpgFE9lcDv/hj7/DHZ0Yfu2R+9HV0YoSlhYA
    RSErbpiXoXG4APKMsbcOHTn65Nz8fEt3V+eygJhJ81aKyUg1+IxAUmOpdKgq5A4Zw8DoBPJ5
    U9BzBF+2IBnnegFHQfUyCSFJxtgzP3v+hZt//vwL137g195fiqBDFPfJfzPfh4SvclIRjPKj
    VUyMXSxnSpc4IJXZGYZUNoe//uFP2dTsQl6iblQb59wKRJ8vuC4eeeSRUCQS+cipU2c3LyzE
    K2gJ4uAqFCxMTyf0a67ZvW5kZGQkk8nModzWNo0VqEW5BNskikXznuctZrNZBiA/NjaWm5qa
    mnRdN8jBqidRVpKKoiQeeeSRa9at6/v4mTPDvePjM2AMXN6rko6xfNfB6vbglYCbx0IQCASE
    bV4RG/21r30N3/jGN9gbb7w5+sQTH3xl27bNfYlEWvGjeMtFHYGmphgcxyxFn2dnZysoKDL4
    UBQFwSBBsZimp06dzLium5HstLDRFla9TqUcoOvt7UFbWxvC4RAYY9izZ3spqiqcAk1ToWkq
    du5cD8+j8DyGiYk4eAf66ig0IZWtCVc1yMEYc86ePTu0YcOG052dHWtSqXwNELw0ulwNGqtb
    ZC+NTlf+rGBQg+cVvHg8Ieg5JiqVgew6UcsYD+ZYjLH8a6+9fryvb8PfX3PN9s8uLqYMuf6g
    0lGo3H+V2t3lMctdCWvfs9rv29uDsKwF59ix4ylKqcBzMiU3Ux19FvdBq+ci8IsSCCilUULI
    Zx964P67fv2x9xNC4LftlhoKiI6EN16/G5/66AfD//4/fmd3wTQXJb5JTgLQq+XpighWXoqA
    W/z9Io+OayjzCfP8b6Wqqn7o4fvufPhrX/lCoK+3h1M2yp3MylX9F6dxVMjaccBNCBD0u6ip
    klGWRe3rudkpl9f5yU9+9rN777j9tr1PfuiJSoCFcnTcsmz87Y9/ii1btuHsmTO49cZr0dvT
    VcGplYupCgUTqUyOxVNZC2V5rBTKBVVXyTiLHGE5XXz8+ElMTEwgHA6hva0Nuq4BTMPI2Bj6
    1vYiGg2DMYbN/X1wHQeO4yAYMEqGmZWArG+sldWuWik/PEKIzRg7Ozg4NHzm7LmWnq6OCp42
    UQDCfHRMAdiOg6JpwrIsOI7NZd18YKgqBJqqIKBrUFVurBUCygg8WnawipaN4fFpRn1Zr1JB
    Fc8ymKgDd7+qul3ew4OJRPJv//rp720+cNstkfa21nJTE354EoWAUQWUeLAdB2bBRLFowrZt
    f8yuVyoIVHgjAl1TETA0BHQdKq/+ZwoB9QBa4WAyvFwuHExLzvZcDeNc1+hzLS1W6XNl166d
    9yYSqSdPnjyju67H5evIkkjW6Og8Oju3te3fv3/7888/v+A4jszZtlFZB7OiAQ4pmm7x6BgA
    uFxCNeO67gJ8fXFVAtB5ANmHH354V1/fxk+ePDnQOzExuwxgvlCR4KVEpxUwBqJpWq2mUHWz
    0X/4h38IAHR2djY3ODj0o/7+DfsnJ2f7x8amahaTAQSBgIG+vk7MzIyiu3sNpqaSsG2Xy7tV
    q3cQ6LoKRaHszJnj5sTERIqv3yTKMrP5q2ej/Xloa2tBZ2cbHMeDaRZBqd9+vrGxAaZpwbIs
    ziM2ff1+nmnwHQ2gWsZOer1q9rk6yDE4OBh3HPtIT0/nnUNDE9pScFwJin21EV3qLKqWFDl8
    STsCvxRt6feKKxRSUShk3FwuK+g5Irhhos4NgABQxphLCDEty0oePXr0J7fdduDAunVr9gwN
    TSzRZPbvkWh+4xe16ro/XjGnZX64/z2MKRyqLB17eY79z0MhDW1toG+8cSSXTqeTHOfNAZjj
    AY7UctHnugFosQgURSGU0jCAT91yy82f+53f/lKora0VrmMv0TkWkUxNU/GxJ36NzM3Nd/3F
    03+z37LsImMsTwgxJfoGQZVGrIh2rwCAFnJ8lBAig+QogBAvvqGcrmEyxigh+NBdt9/6vq/+
    9ucDfWt7uP7rZfCdK6LTlaCbwAcsqOwAtWyLySucT8arrEcSieRf/ul3/suGa6/Z1bJr53Z4
    POXN0QfACI6ePoNzQ2O4+977MHDuNH7rk49DU1Vf2JnAT52DgTICQhgWE0lkc3nKwYMAzgkA
    SV7tmkediiLfTXSrBDL4mo5GIvjiFz4HMH/MjIPHxcUF/P4ffAP333sXrt21Ha7r+i3pmdS2
    3vNAPbcU0WQAFEJAFGW5FCFZBQPNaRzJE0eOvn39nbffpqiKAsfzkM1mMT09g4nJSYyNT2B2
    dh7JVArZXA6O41NWfDoOdwQUAl3TEA2H0NgQRUdbM9Z2d6Cnsw3tLY0IBw2oCoFpmhifnGXc
    EMtKOnLh6BXTN2pE24t8fb30xsFD9734y1/d+qHHHysdRMViEXNz8xifmMDI6BimpmawmEgg
    k8mgaFlwbAeu5/LmgawccdBU6JqGgKGjIRpBS3MDejrbsbanEz2dbWhpakBAVwFCML+YxA9+
    /AuWy5u5C0Sfl6z1etq1al4/AOVf/ssv3+A47lePHDnWOz+/yEGUUiMVqsCyHJw5M6led92G
    LXfccUf2lVdeyViWleVOkBASXiLPtkL2WV7HBX4eFAkhwo6E4deGqLxgyQFg3XvvPf39/Zs+
    ffr0cJ8Az5dSKFjNx7z4cymFrKB2R9W6PEzTLKX9n3/++VNPPvnk9667bteXkslsOJczJeDB
    7S8DurvbQIgDz/OgaRHMzAxzsEGWFNb5oMrA9PSUe+jQ4bTneSmUG2kkJBvtrEb2odZa8jyK
    EyfOQlEGIHixhKjQdQMHDtyA2dk4JibmucSdWgEuxWe1+sCQq9DXW1rXLqXUTCSSb7e2Nqci
    kXCbmE9N0xAOhxCLRdHQEEU0GkE4HIBhGJyyotSQ6/MzIo7DYNsUxSKFaVLYNsBlxDmIVJBK
    pWzbdqrtc0UtUp3ss6AU2gDyJ06cGNyzZ89zW7Zs3Do1NR90HK/kIAWDQcRiUTQ2+mMOh0MI
    Bg2oqsrBsyLtU/D1DK5cBi7ZR1EsejBN/7IsBr9fnr/OOzqCiMfH7PPnB4SDuCDZ5wQ/Q5bt
    T6DVyzgDIJTSVgC/sW/fjV/+N//6663X7t7tdx4Uyrrys0iLUqCpsQFf/tynSMDQ1/0/3/vR
    g6l0NshBHOXGR0ellIq3Qka6ooublAIM8svgbcQZj3xQAA9de83OD//ub3023LeutwSeLw00
    L23dvRzoZiiNlVSBZqXeAIxTWVzuILx87PiJZ775h9/6yNf/4Pf1xlgUs7OzmJmdRTKZRDab
    xS9eegXbd+7CyZMnEY8v4icvvARD12DoGoLBAMLBACLhIGKRMFqbGzEwPIZMLi+4zyIyF5ei
    G4XVNM7i4adzUYpOKqrv4TNKeZGd+JoCy7Zg2zbKTbrLes9CQxqKAkYBRfG5sKqqQiUKwdJG
    BGSZSOFKrO2867pHj7597ENDwyPh0bExvHXoCM4PDCKbzYEoBJFwCIauQ9dUhIJBBANl4FyK
    Vpf2H8ViPIHxyWn86rVD8ChFNBzCpr5eXLdrEwK6hsVk2pOM8moUITEpFTmRyWSe//FzP73+
    7vccCMYTCbz+xkEcefs45ufnAQChUBDBgAFFUaBpGgxeHKtRVeLu++NXVRWGrkHXNNiOg6mZ
    eQwMj/EIGEVbaxN2b9uEG/dsx9unzuHtd87ZHNwlpLSgMM7Z5VKDKwSetS9/+Uv7VFX7Pw4d
    Onrj2bMDS1K+lZJo/nMikcc778wEdu7ctjcajSqvvvqqt7i4aPHiVAEWBaBa6SCHWMclug4/
    x9Io09uE6oV3/fXXt+zeveepkZHpXaOj0xdtiuJX8pMLgmSZ1rGUdwxRA3HBfV4Hx5gSQmzb
    tnMvvfTScw8++NCevXt3P3j48EkFIIhGfZAVDAZ518EujIycxbp16xAIBLFxYxcYI6AUJbDh
    eYDr+oCroSGA6ekRK51OLwlyYOU4+xfCFyWnQM54+2BLq7i9Pg9axdKGmKREtRGXCDgzVmJw
    LOUFlPkdK5JhqeJCOyMjI4M33XTzaHt7a1swWEBPTwfa21sQiYRKWs6uSzllqJwJFnCpktqh
    QNP89tQtLT7oZAwoFhmyWQ+5HBAMEpZOpyxKaSnIwQOYFupILasOchBCipTSzOnTp3910003
    v7+rq33H4mIKXV3t6O7uQFNTAzRNBaXl8frzzuBDFAq/0BKlzIJYI5RvQV3XEAwaaG/3Cwg9
    jyGf95BMOnBdgsZGSn/1q5NZy5cilqkborh7CRaR51670sUtDDOAbbquf+nuu+564mtf/b3o
    xg19GB0b85tP8CYU/uXyYp3KzxuiYXzxn32cbFi3pufP//IH7z99fnANY+znAE4QQgISiBYd
    D1equJCiUpqJSJSOUrEeB5k7OjvaPv35z3y0dfeOrVw7tgoIU/buZOuWLTb0U+qoYKJW6PHU
    O2IpUqVxSunTf//sT9pm5+bubmpsDNqOzdtXhzAzM4vFVBoH7rwXzz77YzhWEf/5u3+NDet7
    sb53DWzHhm07sG0blmXBth2MT80w7u1mqiIbKynndVHjnEgkhCNIpqamcfjIURTy+VLHRNdz
    4bkurGIRG/vW4/zgEObn5yu47qWW1gAI8akbqqLA0FWMjk/B9VxSRcGRL0WMu/rvq5P3TzmN
    48wbBw/NfPF3fm+DqihKLBpBKBRALBaFZVnIZnOglCIYCCAYMBAwdMwvxnHk+Cn0rV2DrZv6
    4Lp+4Y5ZLMJxHBi6jlAoiFDAACEE54bG8Naxd5DLFzC7kBDaw3JXp0K9ohvLGGiRQcoyxl5/
    7Y2D41/5/T/YkkylaEtzk9LV0Y5oZD0SySQymSzSmQxCwSAi4RAaolFEwiEEAwYMQ+cdR/0W
    9fmCiXw+7z8XCnBdF7FIGD1d7WhtaoTneXjr2DvsBz9+AfFkGgXTSnPDLFQgZgkhsnFeURAi
    nO4DBw6Eb7nlloccx/n9N988tOftt08Sv8WvviTqXIs/ubCQxcmTM4Ft23pvfP/7H20+ceJ4
    0/Hjx9+0bVsjhAjgmpeCHCuZPaputGRXBRUIADQ0NGh33XXXB5PJ3H0DA+OKvz7encbzUhrH
    8gCbR8Xkjn0r7fi7AIojIyPTBw+++d19+25qam7ev88vXFQ455eiqSmGYjEL1/XQ2dkFQEVT
    UwTptAnTdKBpftGdqir8UmEYjJ08OSsUsUT2JEEISaFMr6QrPL4lHzPGSCQSRmdnB0/l66WI
    sn//NeRyBYTDAfT1dVek96sbzIjlwpiCYFCHpqlE8SUequ2yyCJ4F3BO6xaFPn/+/NxNN910
    at++a64rFm2FUkYcxytRblRVQTBocOeHlvSe29piUFUFi4sFUMpKc6n6mWtQCtg2BWMEuq6g
    q0sHIYwZBkU2my1iBTujXiiTdPLkyeG9e/e+vnfvru2W5VBVVZVcziSZTB6BgIFAwEAwaIAx
    wHU9uC6Fbdvc6WOloJaq+nx3XVdL3HdNU3gUXkSfgUBAw7p1KjMMFfH4lDU+Pi6Kumf4JQIc
    ueogT/Vca3U4rFQAd3V2dv6vH/7wh276wr/4vNLW0oI/+ta3cOjQYd8zQKV+rCzzxHjhzqMP
    3YP33vcefOC992HXtv7oX37/729/9vmXN8STqf/OGHuNq01oElisWVxYjxQDlsrmCWoHkSIr
    IVVVn3jfg/fsuueO27hTQEtjLUWW+Y+pUN2oKLaSQXN1RLpM5/A8DwXTlAF+rb+z3o6EkBsb
    tW37T199/U3nrjsOvPd///pX1fa2FqgKwTf/6NvQQg04deokFhfmsWljH4IBAy1NTfjqb/8m
    DE3zo7WWz0n7u5++iH/37e+6KBcNpjh1IyVx65xVNs6Ec/eDAG6NRCI3vv7GQZw7f74UaVV4
    AaB4rRCCoeHREhVJLq6k1JeHo8xX5fA8f21Ylg1N03T4yi5R+KozUX6FAZi8c+CSqGydonli
    z4wuxhP/LZvNfryzo70/XyhA1zX0retFf9869K7pRkdbCyKhIFSVIJPJ4rv/73/H+cERNDZE
    8Zknfx1d7a1wHJs3lclgZm4BE1MzGBmfxtTsAmzbRigYwODopGsWLVF0JF+C/1zfFJLkLPB1
    VAQwMr+w8MwPfvh3W5/68Ac3h0LBrecGhtDYEMO2zf3YuqUfG9atRUtzIwKGH13WVAWMUkzN
    zIIxiu6ONhAweJ5bcgZzuQJm5xcwPDqBs4PDOHFmAJqqom9tF06cHiiOTc6agk8HX9t+mjcj
    EYWD1kpGnwV4/uxnP9vd2tr6zxcX479x8OChruHhUWzatAnt7W084ip3MlNqgA4Vi4s5xOMm
    TpyYUfv62rbeeOP+zv7+/o2vvPLKs1NTU+ck+6zgAi1/62ija61r2RkjTz75ZB9APjY4OBb2
    ZfrUC0pfXa58XSWABkyz6KFS3m/FbLQoqDxy5MiZdDr9H++//77PxeP52w8fPk0cx4Omadi3
    bydmZobR27sWhhFEMpmFoqjI5Yo4cWKSgzINmuZfmze3wTSnreHhYWGjk9xGi+hzAfVTzrmg
    bWY8rBiJRPSPfvSj1+m6fkdra4uyf/9N0v+ppCxXKspUdotczgHyQamibN++ozmZTMVyuVyE
    MRYBECWERAAUGGM2t810BWxzaR0nk8nCyMjI33V0dHVRyu5WVVUHCPL5IjKZLHK5IkzTgeP4
    YFhVNbS2NqC5OQKAYH4+h7m5HG+eo0HXdQSDOsJhA5GIjkhEg677oDqTSbHR0XcKk5OT6Sr7
    LKus1E08pjrIAcAuFovpqampX6xfv757fn4uSKl+oLW10VAUFdlsAVNTC9zZ80GzoGAABIbh
    g2vbpqBUzKXPj9Z1DYGAgWjUQCwWREODAV1XYZoeTpw45ZrmbDGXyyVM0xQqatNSgOOSsixX
    FIH2u/S5G7u7u7/+ta997eaPf+wphIIB2FYRT3zwA7jvnjtL6W/xTKkcjfZKX+tsb/ELDcGw
    fVMfvvrFzyhb+9et//d/8t9+LZnOpsRG4pFfT6ZyrJKBhlhIfhqEXbNp4/pHP/TYI2ooYJSK
    BqujyKlMBrZto7W5cWn0mZYBci06hywfZtk2UukcQ2VzGVbDUNcLVDIpRZomhIwyxl4DcE9P
    d1e0rbUZbx0+iqHRCdx7/0N44WfP4cF77sT5oWF4HkUkHEIoGEAoYCAcCoBRDwSMa+0yQd+Q
    W2Wmq0DFBdJ4rC7zzA2fwhgLMMY2Afj1rVu3PPXpT31ywyMPPYBQMAhN5QZX6hQpF8GCayn7
    a5uWeNAiwyL40K7rwLFtnDp7Xv2T736v9/Dx08zzPJcxlhOcQp+1RAS9wZFAQb1OKLF30gB+
    eur02QduvP7a/qc+8jh6uzsRCYegKgSU850JKN546yj+6P/+r8jnC3jisYdw5Ngp/C/f/DY+
    /NgDePieA0BDBJ1tzdi6ca3vJNgW0ukMhscm8WdPP4PZhYTND+CsdOVXiaYjuHYZAH8FYO3f
    /O0zv/PQA/ds/cqXPofNGzf4RaCUYnp6BplMBl2d7dA1FWAMrx86iu/85Q/AKMNnnnwMt96w
    BwQEnudhcTGBcCiIPTs249qdm2Fb+5FIpvDCKwfxrT9/msaTaVF4PCdHn/lnQpR/xTV0P/GJ
    TzS1tLR8dWBg4NOvvfZGYGEhDsMIVBSYVae45QRXNRCxbYqhoSSSSaepv7/n/rvvvjv6ox/9
    6OlsNquRsmgtxQp1qVuuOLL60dLSojQ1NT0yOTm/a34+WbMRhXhub2+G51Gk0/lSy+MLA2ay
    JJLt81RVMObRQqEgIvB11cVexkY7hJACYyw9ODh4fs+ePQdVNXxrLlfQKQX6+tpBiAPHcREM
    NmB2NoFwOAgAcF0/eimcfNcFVBVwXcqmp2esQqFQbaMzl2Kj62GfxY/WNM146qmn1nV0dDxS
    LBY/+fbbJ3aePn2e2LYrrVFVcvz8S9aCljnQlZ+XC9FUVUMsFlI2bNjR+b73tV9z5sw77vDw
    SC6dTmcYYzl+rwuSbZbnuJ4OBGOMOc8888zRp5566mA02nTXkSNnkUjkYNse128ujyEYDGDH
    jjVoa4shmTThuhQbN7aguTmMgYEkLIvyIksgnaZQVQeapiEU0tHREUA4DHbq1MlMIpEQ56+Q
    lF2xItGqJjIuY6z485///LBhGNM33XTTfdu27bzt+PFBzMzEYVmuULZBOBwAYz6dQ1H8Jjl7
    9/ajv38NTp8ex5kzsyVwbRj+3Jqmi3zew8KCDV3X0NAQwKZNMShK1j527JjQhxcAekYKcFxS
    bcoVAWjP8wBg3/79+6996qmPIhwKwXUdKIRg44Y+YP06UOqCeS6na3D6hvyeP3uuC891fAK4
    RxEMGHjkngP42Yuv9bx88O3dPFIp2qMKHiVZiQjWhYwWj1QahJCH7jpw87ot/etLTVKWtOD2
    PDz9w+cQCuj42OPvBeVg69JoHOXPCBhM08TcYoJWAStPOqhYHQ1zNQDJ8+LJ1GI8jl+89DLA
    GJ77+S/Y1u07ybFjb2NiYgKRUD9yuQIc10U0EsYf//lfIWDoCAYMREI+B/qds4PgnSdl45yR
    opIXBVVX6vHz79cYYx0AtgO4t6+v74H7779vx8c++qR+w97r/CJIr9LRoxw0M08AZQ6aJUdQ
    BtDlz8S8e+hsb8GmvrXaj577Rd9Pf/l6++DI+NaiZZ8AcAzAGcbYOJdRFOnwuvLQpKhV3HGc
    s21tLbfs3LaF+KDZ4wWRtKRxbds2ZmbnsW5NN7Zv3oiZ2Tn86o3DWFhMSGPzSt+jKgpamhoQ
    DW9CNBxijDGRGhTGOcsVFFZM57tad5Q7JXHGmFowzYGujvb9e6+9RmGUwnEc/M3fPYuf//I1
    pNMZ3H/nrfjYhx6DqhAcOX4KqhEBwHD0+Du45YbdcBwH/+Wvf4jDJ84jEg7i0ftuw5237oWi
    AJ3tzejt7oBlOy6lTMjWCfrGXFXhoIMVKhyUf15PT8+NMzPTT7z44ouBZDINTTPgeQznzg1g
    YGAUqqpzIFF+9lPies2viehdKmVhcDCr7tjRft3GjRvOHD9+wuK8XBflbnUrruG/3H179NH3
    t3kefWBqat7wNWSXRpUBgmAwgP7+HkxOLiKTMS8KnpcrLvS73ulwXdvL5XKlltfSdcU2epmz
    SZZU1UKhYNYwwqSvrwe6bmDjxh42NjZA1q9fj0gkjELBgmH4XQej0SB27FgDxsocaMYUxGIB
    MEaFhnaaB1BK8qJcy7eCximfIVdSx8EYQ1NTk/Loo4+2Njc3bw0EAu8pFosPDQwMXnvy5OnQ
    xMQ0KGVVgFirAZZZ6SKEQVVReq0oKF2uW35vWQUUCp66Zk3zmn377mzZu3fflnQ6cTKRiB+b
    m1s4PTAwMGJZVlpSEBK22bvSM0ncQx7qdQCYqVTyTDAYyyQSmbZCweH7UpEaoPjPwaAOVVWQ
    yRThugzt7THEYgF+T1hJcUbQkCglKBQYTJPBtpNOLpcTtI2sBKALWPlCfkG1sy3LyliWpRYK
    hUHHcZMLC8nufL4IVVXR3NyAW2+9Bhs3rsPAwBjeeus8CgUXwaCBzZt7cfPN16NQKGJwcBGW
    RdHcHMaNN65HW1szTp+ewPnzKXiez/fP5z1YlsUSibjJcUdcCnLM8ffZS80OXhGFg0eFm1ta
    WrRQMMTbVPNGKV4lz1kc0MxzS+8rnrlUVKnjF6MwNBWhUFCFr8XcBb/dd5qn+1WswqP6xum6
    DsdxWttam+84cPMNqq5pvFHKUtm56dk5vHn4GD79kceAKlWNCkUSEX1G7eg0AbAQT2FuMSl4
    2Y50XXEkXjKAohmLQsokMZW/dxlj5tlzA2Nf/t1/1Wjbdqx37drYvlsOkO9///uYnZ3F2fOD
    UFS/SGFkbBw//+Wr5YWmqTB0nXkepZQyiy9QC77utgBSCmPMAKBLckIyJ71ukRzG2M5YLPZv
    d+3aee3+2/a37dy1U+1sb4frunjttTdK61VEkP3mKR48jytSiNeuV/p/zPMq+dClYjtA4coV
    muJHqvbu3k76eruiyXRm1+lzQ9tOD4w8fH54/IxZtL8NYIg7jGnJkakbgObrJssYO39+YMg2
    TTNg6FqlHjRfrzdedw3+zVe/jL94+of4wd//FMlUGl/53Cfwnlv2+t5rhXY0lTIvWUzOLjCU
    NUVl8CycgxWPwMoV34SQBGPs+Nnzg2YikYo0NTXAcWwcPnoc03MLCBgaFM4XZIzhPbftw7mB
    ETBGccetN5ZtG6WwXQ/J2QWcHRjBHTdf59M7XIqBkQlkcwWbH0JCFmmBECIKU5bVFF2J4SuK
    smV+fqExmUzX6OBVzXtWKpowLNVVlXnACopFD5aFUCwW2whgnIOrPHcAcysR4LjUR2tr68Z8
    vrg9lcpWKBRUFgwStLY2ghAFqVR+2W6El0LjAAjC4QByuUU3l8vVstH1ciZKBW7M1+iSud8O
    pSzX2BgZ3bt3aygUCjUkk4mo67ro7e2FqqpobAyVMgvhcCO6u2O8nsGGbTvMdT04TsJLpZKW
    7MBzVRPGf5chU2X4Hqu+LjvQ8bGPfWxtOBz+t/F4fP/Zs2c7Zmfn9HzehOu6WLOmuyKSLF9l
    WTMBNCtlzqqpSrLOsLiNjCnIZh3YNkKBQHhLNNqwqaOj/75du5yxUCj4J4cOHT7BbXOG2zSK
    Omqhiww7Y8xaWFicWL++fzYajbSZZrpmx0HHoTh5cgrr17ehp6cRjAGLiwWMj2dg23TJeMs6
    3776xsJCyrFtuzrAsVoZQnEeuVw2NzM9PTVw/fU3jDc1NXT7tGwFsVgYa9a0o62tCUNDE6Uu
    isWiixMnRpBM5jA0NAfHYaXOjYaho62tEa2tceh6FpT6SzcY1ADYXjweF+pqcW6fq7vCXhK9
    8IoAtKqqzPO8oePHj6dPn36ndeeO7VAUXg2pKBwQlt8TRkFLX1/6TJkv9eWBwXEcHD5+GqfP
    DYlJFDzRACq50Ffj0dnZ3ra2v29dmftcFUUmAF576200RCPYvX0Tdy4uovm8zNeZApwfnsBi
    Mi04nbYAn1Kkh14hoFQB9ADYCCDAfN3egvSzNQCe7ThnDM+9gTKm33HHHeTkyZO44dpd2L71
    UeSyOczMzSMaCaGjrRWu6yKZSiGeSGEhnsDs3CIW4klhhIPwOcAtjLE18PVb+wVgZ4zFAYxz
    b7/WeC9/x/olurc99NBDd3/zm9/UOjs78P89/TT+9DvfkZqDQGoCwyt9S2luKeldes3kzVHZ
    VIRSXuxB4VEP1POl7lqbG/H5j38AH3z4Lm14bLLxK9/4TztPnR/ZyXVtVaGGwsdeT+6ZB5Ai
    wMbm5ueLxWIxEAzEQJmvkuN6Li/4tOG6DrZv3ogbrt2F//Tnf4W9u3fghj07QBlFJpeDpirQ
    VAUqARgh3GVgKBRMxFOZUlcrLC1QqZc4/6VEOVwpYzU4Pjm1sLC4GGlpbkAoGMQXf/NTOHNu
    AE0NMWzZ1AfXdeA6LvrWrsHvf+mzYMxXFjFNE4pC8Ikn3oebrhuCZVnYsWUDVEUBpb6O9NDo
    hIi6Z7kxjvNL6NsXsHpNU5jruulIJOIahqH5ElGlJVoDl6GC0lGtPVtN9wgENGgaY8lkMgig
    lY8zxAGWchXtMxRFabFtJyK60C2NLPvSYF1dLUgksrBt9xLpG8tHpqNRg507N206jiOAZ1Gy
    W1fkLDLGsGPHDv32229fFwgE1luWpaVSKXNubtaMxxNuJpNhjuMYMzPTmUgkOhAMBq8hhOij
    o6Po7V2LqakEslnTV0UKB2HbLopFF6qqIhDQYBh+5oExBtMsskAgoDY3N4dc143qut4SiUR6
    u7o6oj09PVtjsZjGGMjc3Nzcm28enMrlckVesGvJUdnLHWssFtszMTHx4HPPPdeQSCTR39+P
    6667jqskyVq+SylHlVSkSmm+WpSk5fTAKQVGRgqYnraUUMiIbdkS3NzQ0LgbwCLvLgk+TjHe
    ulPP4vH4oqYpc+FwaJeP18t70C+Y87nrrsswO5vFmjXNCAZ1jIykYVkUul5WuZWdBAGmdR0s
    l8vKjVNyHDznscKdgGto91sAEI/H5wB6vqkptm9iYpEQQjA3l8azz74JXdeRTpuwbQpfap3g
    /Pk5DA/HQSlKDnA26+CVV4Zx9OgUikUKyxKOhK/7bJpJN5/PF1CpABaXqIaXXJtyRRQOSqkC
    4MSRI0e+9/l/8YWPHth/WzQaCRNFUYgigAWXWJH5vIzJ/FBaiuB5rgvHcWDbNltYjOP1Q8ft
    iZn5ST4wWSKJXS3D7PhKGJ7jOK5lW1wjtpKWQcCQzuTw+ltv48BN1yEWDfNIPK1oroKKNte1
    lTgIGAoFC68dPcXMol2Q6CtFQogw0G4dqA1rAwHj29u2bt3b2NhACvmCnS8UzEKhkC4UzKzn
    eV4sFl17x/5bN27fuln/8c9eQEdHJw4ffB3/7JMfxtDwKF5/87A3PDbuRiNh8p5bb9Lf9+Bd
    ZOumPqgKQT6fRzKZIifeOac994tX284ODEdMy9pACLEMXUPQMNRg0NDCgYDhuC4ZGJ3KpLO5
    P2OMPcc3tWic4NUhmqMQQnp27Nihrlu3DgDwwccfx0MPPlCiJZRaV5daVFO5BV35M9AyD5pR
    noEo8/1lHrRoAe65DlzXAaMUDZEQNFVBNBKC5wP7Np5tEVrGuRUAIx4hsBnDXCKZNs+eH2hU
    FYLRsQnMzs1jMR5HMplGwTRh2TZcx0U2m0PQMDA1M4c/+Oa3ofsZBYSCATQ1RNHaFENHazO6
    O1rQ1daMRCqDQqEo5OsKVeC5Lg1U3qWBFtrB46lUenZ0fKKvubEBs3NzmJ2bx8JiHOcGBvHz
    F19GNpeDWSyW6CzCUdJUFaFgANFIGI2xCFqaYjg3NIb2lka0NjWAUYrx6TkmOQ0ZACmeMctU
    8QpXNPrMbQEZGxs72tPTc/TWW2+5dWhoBJ7H+GEi6+NqUkpc46ni8rNP5yi/1zQdoZCBrq4w
    SyQmU6OjY3kpW1Xvpk6X9cjlcovBYCAbDoeastkClnK6FYRCQaiqioWFdFXjkdqR+QtxoqPR
    ABTFpsPDw1lpvRckObArjkA/8sgjuxRF+eN4PLHJcTw0NDRazc0tBYCkCUEWAKEUG2Zm4uts
    O6F2djaAMQZdj+Ds2Sl0djYjHNY8Sk03GDQUyyL65GQKuZwNRVFhGAYJBgNobW3S9+59T9f1
    11sNjNFNmqY7um5AVXUNUDXGiK6qCnp71y+GQuFvPfPMM68zxgo8u8SulFZICOmOx+OBxcVF
    AMDIyCimp+cquL8+XUOp+VpQOWRetKJoFdzoSl60VuJBy8+eRwCocF0/65TJpJsBdMNvBlKs
    Opfq7vTPzc2lCcFCU1MU2WwRDQ0RRKNhhMNBBIMGdF0rqWz4Epz+MbFhQzPWrWMlOTfPE1rQ
    gGUBtk3geSo0DSgUClaVfc5L9BRvlbBWqbYrm80lPM8baGiI0FAooPoOXwjhcBChUAAdHU3c
    2fPH7e/BsgNV1n/2L0Vh0DQNjgM4DhAKacjl0naxaOWlIIe4sniXtSn1aKSScxznz1599dXx
    V1999VpOt2gghEQBGKTs/pSDGiXJXFHcyUTQT5YnEkoNovgmcTkDXKGDaX5yevbU8798bXNP
    R5u/cKXUt6Cw3HLDbuzfd22ZK4pK8Lw02lwuJCx18aMeXjt8Eq8eOuWgrMEp0kfCU7yiVAuf
    BFVVtXBLS0vTrp07Ak2NftFjwSygUCiAUoqO9jbs3rUdbx0+irVr1+P48WMYHBpiX/8//8g7
    e26wYNm24DM7A0OjsZ/+4ldt+67fHb1211Z1TXcHAoaOluZGvO++25Xrdm2JJFLpCPU8qAqB
    qijwPA/ZfAHTcwuurqku/LbBUa63bXOHQakTwEqdOHGCPfvss0Qh5ajy0vmQnqkPlktcaMrK
    nGf+f4QCR5kLXfW1UrMVWmq6AsYwNDaJ2YWEGF+MZ1sMlJUN6m2wPADpkbHxhX/1v32zTVUU
    zbJtX0NVUQDCfykhpY7WvWu6QD2KRDIDynzn1/PEuPxmOUFDR2MsyiijyOYLnmyY+YFjYuX0
    n5cD0TIFJplOZyb+5Dt/cVNLcxNyhTwBA6dhMaiKL/PucToOldrW+wosfldVz/UDbKLItKkh
    hqaGKIbHpymqON9VnEJvOXrYSgRiv//970+/973v/Xfbt2/7zJYtm7e6rhvxPGpQyvQyDa6y
    g7HU8ACiL6eUvQDAmOcV3amp4ezhw4dH8/n8vGSbV60l+YXM9KFDh8f27z/wq82b135kbGxW
    8R2HyuilpmmYno5z+kWwouPghYoHq/nUmqaiszPChodPpmZnZxd4NEtQsHJ1pCtplNJIoVBs
    TiSSuut6UBQFum6UOknmciYSiQz6+noQjy+gr68PkUgD27mTuInEXP61195Kzc/P51pbW71r
    rtnduGvXmjbTRDiTcRTL8lUdikUPi4uuGomEY4ahxhyHwHEIT50DmkZYMAjXdWnBtu1Gnkn0
    GGMW75NwRQ6U4zjplpYWd9u2bQG/zqpShq5cEFoGyTI9Q7y+1Oflvke8DgY1qKptz87OMW6b
    RZalrs5itUJFPp8vMsZmN2/uLa5d22nouqqUNauXZo4sy+PRWQW6Tiq03ctaySVAzYJByizL
    MrG8fB1dJfvM/KAOYYwx03Hsse7u1vwttxgxXdeIovgOraJUZsdqZyFQ1bnR/9zzAMuiiEZ1
    vPNOxvI8T+Z8Z6VxVxQ/X8xGX/bkCwUDQkiYMdZCCGnnqfgeAJ08khaVFhmp8btZjYNdPuAd
    PrhqHVVRjFNkq3AKVY2bMMbCAB7o6mj71/tvun5zc1ODxuXNiKJwyTNCQOAXrwgubGX+tLLF
    teDOUkp9YMLT/YVCkb359hl7bHp+GsAIgFH4HNkhQsgIgClusC1cptwMn0sDfkHdAV3XdwYD
    gTWapjUpqhJWFCWg+NIjqut5YcMwum+//Q7thRdeyMbj8QWUG2RkpUPD5YamXde1dkPXo4qi
    6OLXKUShABwG5jDKLI/Soud5ecdxMx6li3yc56XDKCUVsphXWESoMMZuMgzjd8PhcCf8DpNh
    AEHGWAAV/HqGaoYGSs2g5f/Cql6W/0AmNVkp/0dW8ZM9z6OW7UwAOAxgjNNXxuFL6yzAL+Ss
    iwPIHyovory/rbXlN8Oh4A1msaiYZhGKoqC1pQmtzU1oiEURDYcQCBjQVRWEAB6lcBwbpmkh
    m8sjkUpjPp6C67oIBgyEgwEsJFL5TK4wC+AMgEEAA4SQQT62Gb5WVpoDXF0gqzDGGgA8vGvH
    1o9v2bTxzkQypaXSabiuh2g4gmg0glg0jGAg4Ed2iE/D93gGwSwWkcsVkC8UUODtgyPhIBpi
    EQyPTTlDY1NpAOcADPD1O0AIGQYwyYMA7moAaD5uhTEWBdDa3Ny8vqmpaQuAXs/zOimlrXzN
    l+hwpPQvK3X0KtsUxnjTBua6HrUsyzVNs0gpTaMs0zcFYIoQIgIehdW2z9LZErvrrrv2bt++
    /QuMKdd7Ho0wRnXGoAFMFaOV10jZi1gqry/L/spMLf/2UDo9PZF+5ZWXx9Pp9CRf46MARggh
    Y/zsSsPnEl/2oLZt2xa6+eab90Qikf2MsW2Usm7GaAOlLMztt8oYlEAgEFNV0jU7O6s0NjYm
    h4aGEsPDw8Xp6emC4zgiAJPXNM1bt25dpL+/v6O1ta3NMAIRQghfD4QRQjxRFEoILMaY6Xle
    3rLMdCaTXpicnBo+d+7coG3bCeEwVHWSvZzpJ48//vi23t7er9q23ec4Tth13bDrukHPowHG
    mHaxqb+EXoKsNo1pibFnjAGKQtjCwvz8yy+//FaxWByV7PM0fP3gLOok9SbLTwIIP/jgg9ev
    WdP7VCzW+BFN0yK6rkFRCGzbg2W5sG0XruvrIYv21YJSpKp+VNowVAQCSkkTmTGCeHzRHRk5
    Gz927NjQ4uLiOclGj3BbFZepKavUj4Ewxoy777575/r1Gz5lWfhAIGB0BoMGNE2Fz8934boe
    l7NjfLxKSQfal69TOcVFgaoS3ogFsCyTptMz5rFjRyfOnz9/htto+Vya4zjmks8lcoUDJ/A5
    yTEAzYyxFvhcuGYATTyKthyAvtDiZqhsq50SXBWuRZkSm3S17bMEoNsB7ANwgKd1mgE0SGNW
    eLS9RJ2V/AZSC42xSgdCvg9ZvlHn+OIeBzBBCJnkn2fEYr9cAA2/EUKYMRaDr1Ms5lKeRx3A
    tU1NTY9HIhF7amrqEN9oKgcGaSlKLkj4KgfSgsOuoSyiXpTSR3LaSFSWl9LgdQbQKmOsGcBa
    AGv4/HXz7EkLX9MXW7PsUoz0RT6T3wt5OSHoPsUN9Cy/x4V6rXXJ+Y0yxtoB3APgRu78thNC
    mgIBIxwwDF3TVFVTVUVVfWKWCEj7Th+DRz3m2C7LFUxqOw7lkQQHwAJjbJyv1zEAo9whEA5f
    fjX3rlQkGwbQ0tTYcLftOP/cNM0exhAxDMPoaGvVW5qblFgsSsKhIDF0jYgmBIx6cB0XRctC
    vlBgmUyOLSZSLJ5KM4CwUMDwACQKRWsUwIQAUNK45/g+9lbyQKphn0OMsUa+l9v51cbXeaRG
    FI1cwnqWs4RZbpsX4bfAXYTUAvdq4GduoyMAWtva2ta3trZu8zy3x3HcTs/z2rhToRHiJ5/K
    oIsscx8qM6VS4IMBjLmuR9PpVMG2nRTfv8KRmOJ7Oc7tnHe594OPyeC2tFHTtBZd11t0XW/V
    NK1RUZQgIURjjBkHDhzY39/f/3Amk0m99NJLh86fPy9oYDbKvM+clLnUNU0NGYYRVRQl5Nto
    UXzLTMZYwfO8gud5pudRi/pFJKI2I4eynr+w0ZcNoHnxekNnZ2cvgF7btrs9z+umlHZRSlv4
    +KtsM6sGw6zW8mUVQYyl3yP/raUJ57vfdb0spVTYZqHpPiuCG6ijVrKkYhIA0LRnz55rtmzZ
    8oCikD5CSCchSgshSowQxVAUVVMURVEUVSVE4akkUnL6/D+J+j0LmUcdx/Fs2/ZmZmZzBw++
    OW5Z1qRkr0Y5ppjjc+qt1v6VxqwDaOrtXbvx3nvv+62mpsZ9mqY0MIagZdm6Zbmq63qK5zHi
    a1gQIhc2CyqLqipM0wjTdTCAUtf12Pz8YvH55382Fo8vTjDGxrh9Fk7uNN8X1ruxz1cKoAUN
    RBSERQHEuIGK8M+XS3OQiwANuVVrjvOrqkW+6VUw0IR3RmwC0MoY64LPWe0S9BWe3qnVvvVS
    7netJikOH3dJV1bavEmUaS1XDKD5/DVy+kQDd47EPCoAdgN4gHtvgyi3zS1yI7oIv5JXGFCN
    A9KwREsQ3yOq02U1EZFuF0oGJR1SWd7tCudd5RSjFsZYaxWoaOT34t3sDVaH/0clZ1FUBgsw
    kqo3GOHRthBjrImPvZMDaJFBakK5YLdcjVImiS7n9Mr7No7KJiKiy1N6tcGV4ATzrnmNfN/2
    AVjHrx4OMkN8/jVUdrYjNZxbuaW0cPbj3OmZ5CBKHEgJbrdWJSgr2WdD7Gs+brGvG/i+1i5z
    rcu88qwUfZQlKd2rBaDhF0I38j0tHAfhIMekcV+JjUZVsCfL5184E4LOkb2SDKE0Jp07BjFu
    p8RcRvleVQFo99577y1btmy57ciRIycOHjw4xdU6KJ8T0VUwxx1dcBsd5Gs/LJ1fVMoEy/Kp
    Yp8XL2CjncsF0HyMzZLT18ZfN0rBjdW2zaZkmxe5bRbZ0eJK7GtCiM4DWq187XYD6NE0rVtV
    lTZFUcOKQnRCFEVRiOo7hD6K5sECHqVljFLKGGPM8zzq+Q+LMZaRAjbTKOsgJ/gc0tUE0JzS
    oTLGooSQ1ubm5vWGYaxXVXWtrutrdV3r0DQ9pqpqQNNUTVH8qA53fHlgh8LzKPM8j7quSx3H
    obZte8Vi0TFNs5jL5dKMsZKTC2CSB6qEQtK7sln14ECLaKEsFxXgm9CQDt7LMdDiYKqQPcNV
    5NlJjVyK3AjF+c6xuBGJ8gNYqWGQyWVsZjkSL6IHCSkKX7oXV7jYS7QZLimjS0DXRrlA6DiP
    KEL6ujhIU4QQcWDYItrLD7MQN9DBqvvDagAwIddX4PdY8GbrwsuS1C0KPGojHIB0VdaE1Zgz
    dpn39mIPWbEiwyM5suTZSjw8fm8zADS+for8kI1JALpaLossc1+YNH9Fab0uVgGJq713LQ72
    JrnaTIofIM1SlkStMV5SY7zVY84IAMUv0TTFxdV5CGdGNN0Q+tTVPM53myEUe6iIMoXLlN57
    uHoPJq3tLAcTLrcncT7HspNcDxst7FVacibSqJ+aAQNvPMF/V5EDX2GDLQGgX3311TcOHz78
    TiqVYhw8Cxueh99RMI4yzQ6cthHg2ZlQ1b6v3tuy4yg61cq1DS6urNBfjNEEkOY2yeL7KiI5
    Pqtpm8X9FRnRatvMVsBWAVwjmTdyEWem67pu1nUxz+eqOsChXGA/sxoBKuFQLXA7WMAKqm8s
    e4Mrm6s4jLF8IpGYlbJcC9yRkJ3Fyqrf2uuBSs9iDyS4bV6QnIXL6opbj1beTPJOxR9Ya0Lf
    7YKuBlTVXZ2u5kMYj6xY5FL0oRogvlsDzWocVsKg5FHZbrOeUjNigQlgJaS4AtJBW4pU8HSi
    In2fyQ8NGdgLCofBnapAVVST1PgbZP677DjJHaDqccAK0E+XGStWyBhfKOviSGMV10oBThkA
    KdxwiXWc4IepUWMvK8us42pgJQ6crOQMLOnudBX3rtAploFvrGr/KhcxzvKarUhpc+WNFL+f
    q6J5vYx9ppKTLQ5NXdqD6mUCRznbIF8ermKRd43ATha+zqzJAW2o6gCuNfbLsdFOlSMhN9ug
    dRyTXeX8ZqU1qwJQTdMMmaYZqookW1wSNF0FlFBlo4OoLF6uZaNpDRstN33y6mCXTMkm5aSz
    42rYZvBx2dIci/GupKPI4BfPCxkZ4QQm4RfXB2sEOC4GoOXzVQSoZGGCq+nsQ3L0xX4FY0xk
    4BNShkStcv7JRc4kecxpyUanUaOx1SU7OnXwklbqd1x0wV+l9CAkr0eXIu1y1P1CxvldRx0k
    wynrIVvSgXVFxVjSPBJpE6pVl8I3q4gkGyI1yCMGJcBUZVhI1c/RlnGsakX0vBoO1BUXnokC
    q6pxqVh5GS52ka/JY3Wr7gHqTOGQ50aX1rB8iBoXMc7kAsbKrnIEipLD52GVqAwX2LsGKtPW
    taJvlxqBFoe+LYGnQhWIWpF5fJf2ubZo7ru3UwxL6zVqru9/ADZalda4JgPNyxz7hWx0NSXN
    rde817DRtWyX2M+lTB9vdlZto6sdWdnm16JsXcxG1wxwXQGdsNaZcTVtM2qMV76wAvQNeZ9q
    VfZZttHLRZ8vljFzJcdHzhzZV8NWLTPmgGSfxXP1uC803upArOzkypKqRQlAv6tziayQoV7x
    6MrVelQZMlIFOmuluEkdNjXD0k5PrOrwqpeBvtAaUbCUH4qqBXohmgWpl1Gr81ivum7tpYx9
    BQA0llnDykUM1MUKzegFnCCGVVDfuIR5V5Y5pDVUpgVxAYehluMgAye3atxXE1D+Y7TRtZwG
    cplR50ux0Rd0KlbBPqMGECY1gFO9bDSr9xr4J9u87Bpezj6/W8yxKkGqOo252kmsdqQuVMNQ
    q5bMW8ZGV3T6XXUA/Y/pcTUOo9U4sP6hjmslDNX/KGNdRQP9P8WY/mnMq3s//iGN5Z9s9P8c
    4/wn2/yP21b9jzbu/x/jx4iOU3uREAAAAABJRU5ErkJggg==
  }
  set pieceImageData(Fantasy,65) {
    iVBORw0KGgoAAAANSUhEUgAAAwwAAABACAYAAAC3DfDxAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAAMMAAAAQACjgSjqAACAAElEQVR42uy9d5xc
    V3k+/pzbpred7b1ppd3Vqler23KRu7EBgzs2mFATviEEUvimAck3IT8CgZCEJCQUgwFjjAH3
    JluSVVcrrbSr7WVm6/R26/n9Mefu3h2tjIGVbIjux/PR7nhmds6555z3fd73eZ8XuHxdvi5f
    l6/L1+Xr8nX5unxdvi5fl68LXOTyFFy+Ll+Xr8vX5evydRENLSFv1gYvpU2mv+T3/JOUXuo5
    IAXjJAXfjZq/X6rvdvm6fF2+fvklXJ6Cy9fl6/J1+XprnUhK6Rs5lW/sFV52qn6bL1oAFOhF
    +vy345h/JWBz+bp8XbYLb95eXAy7cDnDcAlv/mUn4PJ1+Vq6ffLbvEfM8bKDn1jOYwKAw8Io
    LLU8DMvPl8+K34E1z+4f+SV74fL9vnxdtg3/S2yDaRcsZ8N5D0KIOUjDYhsu6llxOcNwCRb2
    BZwCUmAMrI+5RfO/eeNbx38hlH35+u1eD4tEShZQFdj/XzQC+TuyRwillAAQAbgBeAFIADQA
    CgCdEKJbflfZz4bFQFy+frvAwXn0HEmSOEEQuEwmQyilprmwgsS5+/2/wTYUnguUUnAcN/e7
    YRiXbcP/ItuwCKh+Q9vwOzJ+4nK5uDVr1jh8Pp+X4zhnNBrF6OioOjY2phqGYRBCNGYTzIdu
    ARBL/70uL82LssDJIk4BB6AcgAvAhGWhmwZBszgCv1P8zUUO9Autu8L0fGGU1WpEf+ujDb8E
    YP7ORU7ewIkizEMy94kAgLe8xtwjeuGB+Ns6H+Z5wXEcDMMIEELudzgcV0mSFATAq6qayWWz
    Pbph/JgQMgogByADIMsepnG47Bz9doEFrvBs27Rpk3P37t03AvC//vrrT7344osxzEcQdQYU
    Ffbz70ymwWozLWMhFzgbaYHTaEZZf6dsw/92m1A4ByyYYmZdOUmSRFVVrWCBFtgF47d9Tsz9
    4PF4+IcffnibKIoPaJrWaBiGA4BGKZ0OhULPPfbYY/tlWc4QQqx2QWZ+5EWxj5czDBcBFZqL
    mx1opoEoEQTh86Iotmaz2S9TSp8nhBiWBZ+zOALnRVB/Ww8Lq1PINr7InEFKFg7KoJSaBtEN
    wAkgy6KrhU6jCazm6BuWA9UakXvbRaILI42LASjLvNACQ0EXi8T9LlyUUhuAOgCrALQASAF4
    kjnJOtsXMnOcfmci6yxSWlVcXPz+j33sY60bNqwHqIHenrP4x//vnzYMDY+EKKU5AGkAHDsz
    zGjS5eu3zES8853v9Ho8nqITJ07Ejh07pgDgWlpaOjRN+8tMJlO5evXq1r6+vn8bGxvLsPNL
    ZXsAbP0vONd+W4Mli9hMUllZKeZyOT4SidCCLItBKTUcDgeuvnqvm+M4x5EjRzNjY2NWCoZB
    KdUsjiOx2OC57YaFtL63hW0ozKRcIKh2wYj677BNIF6vl7/uuuuCZWWlbTabrR0gZcPDwy8+
    +uijZ9ki0di+kH/XzsQ9e/bwdrv9hv7+/nuPHz/OqaoBu92Jjo52FBcXe20225AsyxFKaZwt
    cv1iz8FlwLC0Cxwcx1HDMDoA7APQRSk9QQjJAVjd1NS059Zbb6385je/+amJiYkcpfQM2/AK
    cwjMg0IrOExoQRQFhUbj7XxIWIxCG4DbkM+yPEcpPc3GbzpBCoBWm036oCRJjel05kXDML5P
    CEmx8WoAZEqpAKAEgIPNW9wCvgqpG/qlMAy/7NAu4KyD4ziTisJTSnkGouaAJnuPkc+8UoMB
    J2sUhf6qhbJLOf43C2Df5Os4ALcUBQKfammqa0wkkt4z5wYOUkqPUEpj7H5m2D3WcL6qym9N
    xOwCV04QhPS2bduwZ89uUFUGDwpZVuJsnGUAIub6Z4EF8mbX3uXr7WEbAKCpqekdHMd9ZO/e
    vQfq6uoeeeyxx4ZFUVw9PT1ddvjwYcfOnTvfu3379olHHnnkRRZAkZGnqJkOsGbZM6YDaRQE
    SX4rqK3m9yoqKuLuvffe1U6n8xZd14WRkZGnH3nkkXPMITQopSoA9cEHH1zjdnseppRW1NbW
    Pf34448/Njw8nGVnpApA2bp1q9jR0VGq67ptaGgouX///oQsy5TNk1rweEspXosABTOoxhc8
    5rJRLM1isLNQtwTRfiUgeTHPijd7Hr0JKhm99957qj0e719nMsqV6TQt9vvtqs1miwCYZCAx
    yz7nojvLl9ouHD9+3Ojo6IjH43Hj3Lk+DuDhdHrR2rpCn5ycjCSTyaBpCymlKiEku0jwcUnv
    92XAsMSXy+VCMpnc09jY+BeCIMyeO3fuEUrpdwAsKykp8axbtw6hUGj5I4888j5VVf8NQJw5
    wQlmGLLWg9/iMJqUJa3ggHjbGwYWRfXYJOnTV+3edqfTLnFPPv3iFdmc/BkAM2wsWQACIeSe
    vbu3v++KDWvw1W98q318YqqfUnqcfZQCoEEUxbuqqqqudDqdromJidFIJPJTSukLhJA0M7Bp
    FpXLsfctKWh4I4lEs15lLjyWBwXWaBCllPKU0hoADQCKGICSmHGgbJwpSuk0gBA7CMwoisIO
    B43neRiGwQHgLBE54PziWPoGkawlOfvmp+aCc3Me0KWUQhAEaJpmFwTh6ve+85Y1d9x4NT73
    xX9Wu3v7TyKfjQqwtUHZ+H+rI8wLlxEx10wikUiM95w9s2Hnti3IpJJ44qdP0vDExChbE2WW
    8WcppRlCiHUuDOu/v4sc99+Ri3Act3xwcHBtNptd3djYuHb37t1f0jSt3jAMqbKyEqFQqKiu
    ru7u9vb21OnTp0d5nlfLysqUpqYmo7y8nNhsNk5RFPXMmTMzXV1dCUqpRggxgy2FNS5v+6DS
    d77zHRw8eLDI4XD+xfh49AaeF0hVVfXalpaWv+jp6Ymzszvb1NSkezyeh8bGonclEjKWLStb
    1tra2jc8PHyWOdHqDTfcUN7W1nZvLpfboeu6fdWqVUM1NTU/eeyxx16Jx+NZFrhLWwC3stR7
    5gKfU1i3SBb6cJSWl5dzd9xxR5XNJjUahhHUdcOl65pN0zReVVWqKKqazWZTkUhk5vjx46FY
    LJZmgUaZ+Q/mfaeW4BMpCDpeSptQaPsuFOBYNABqfp/PfOYzK5LJ7DVHjgyUFBeXwG7nznV1
    dU0wu6mycarM1v9WUuwvNE+jo6Mkl8sNFxUV5VwujzuX01BZWYFg0B976aUXpymlpdbxU0rT
    hBCRUqpb6FoUvwZV7zJguERXJpMhAMTt27eTu+66q/xzn/vcwy+99JIbgOxyuWw/+clPwHEc
    WbNmzfbDhw9PAHiVUuoCEOA4zi8IgheUSpquK4ZhnKKUvgogZtkUpvOo/DYZBgAcCPFsXrea
    k0QOTz79ogOAhznAOqNuGZTSGVHgFZskSLKiRNi4vWyMbkmSPnLVVVfduGHDBl5RFKiqWvvU
    U0+1nT59OkgpfYoQkmSOJmdxopakCKgwS1BgCDhKqQhAopSaY+MAJCwgzyxQKhJF8QvNTY17
    SoqLBIHneY7jiGEY0DWdqppqxGJxrX9oeECWlX8AMMCcxDQzeDlKqU/X9Vq2HqLMaFDmiFpp
    K9bamCVfKxYKnmSJhlF2PwkW0sgKI2FE13UAEERBcFdXluH4yVM4eLRzEMBJAH72Po6Nk/9l
    EZS3Kqr2yxyIghoNk5bHsXmyp9Ppse7Tpw1VUbie3l48/exzGQamS5HPpNkAuNgaM9j7Zcua
    0pbSMFy+Ls5yoJSmY7GY/vLLL/Pbt2/f0t7eTjVNS4TDYUEURXi9XkSj0ZYNGzbcHo/HX9y6
    devy6urqWkJIIJPJegxDF3w+v1JRUdFVVlb26Msvv3xWUZQMgBw7H7LMTqiLRZ7fbrZBURQY
    hs5RSj1TU0litzvg84mObDbrYc60BoCbmZnJqKo6TSk0SjlB19XZ8fFxw7QN7e3t/mXLlv3B
    yMjI3snJSY7neQiCUNfY2Nh25ZVX+h977LEXKaUpQohpG0wHWv1N9kzhnFqot4Ttb8Hn80nF
    xcWOyspKDyGEP3nyZDIajaosg6QDUNasWeP2er2fTaUyN6bTOdEwKG8GgTjOTh0Op+HzFev1
    9Q2juVzuiwcOHDhDKc0ye5AGkL3jjjs8dXW1dfF4XD516nT05MmTuUwmQ9l3MrOzCgs+6hfK
    TCzFGjEzJg0NDYLb7RbD4RA3O5unmlFKOUtmbMF3sdS0EEIIPvWpP3LJsiYSIqKqypsLhUZf
    6+rqUhlgyLL3pxazC4tl4H9ZVv5S7o9CP8Jmswkcxwm5XI5n90zo6+uLV1ZWzXq9freuJ9Hc
    3EAjkdnJ/v5+EUCVKIp2j8fjkCTJMTs7S1VVBQswFtr/3y7A8GaVDH6bFQ8IITAMgwCYmp2d
    zW3evNn9N3/zN44PfehDt588eXK6sbFRuO222/D5z38era2tjlAotG98fNwFYAWAer/fb9+0
    aRO3oqUZHKF44cWXEydOnnqRUvozSuk5AAlmFFKWCLpi2fhvy4sVdiZlWX7mJz97eq9BKZ/N
    ya+wBe0GoFJKzSzD64ePdk4MDY3URGLx4wwsedim2rdy5cpr/H4/981vfnM0Ho9HOzo6mq6+
    +upAKBS6ORqN9lNKB5kTZtaGqEu5YSzOH8d4914A1cjz7xsJIU2iKFYLglAky3JG1/V/A9DD
    DJ/CNnNVUcC/8k/+6A+Kd2zbAkIpKCioYcAwdOiahmMnTuKTf/pXGBkLLWPrP0UpjbOMVAbA
    3cVF/oclUYzNRuPjsqL0AegFMEIpnWRAJcOialaOp7ZU64VSCkmSqKIoPgCfIIRUUkpDAE4x
    Kp6ZLVPYWjW/g24xDHYAG1VVbfj50y8gNDGZSyTTB9j7SjFfw5C0gMDzCj8LVMh+aXbjUp4r
    lr9XDuAGALXseyYppRE2Rm5kdESLx6PSc889j77+AZmNv4oQIoqCwBOOQFHUU5TSbwAYw8Ii
    6BwuojrG5WvJHOThoqIiWdM056uvvko8Hs/G5cuXy6dPnybd3d3Yt28f0uk05/P5Nu3Zs8fr
    druDkiQFdF23p1JJbmYmQrJZBSUlJZXt7SvbqqqqXjh69OhL3d3dQ4ZhJFnAJGmxD3O0zLdb
    UIkQgvvvvx8AYp/85CefbWwsv4LjiDo+PvzqyMgINQNKAPR4PJ6dnZ05UFZWfk9RkRGMx2eO
    dXV1pQF4JEniN23adGsikditaRra2tqGCCGpWCzWPDQ0VFJfX39zdXX10NjY2Bjm6YxLYhss
    wgUEAKmtrbU1NTV5q6qqKj0eT53dbm/keb6Z2YggIUR1u93/+eSTT3Zinj6bSyaTQVlWOo4e
    7S4ZHg6DEG7uwXE8OI5HUZEPGze28JQaLYQQWRCEtGEYcV3X4wDSpaWltwmC/fe9XjG1c+eu
    8S1btvRHIrM9w8MjIz09PRPT09MJXdfTlFIZgMyylEqhTfhNgaXlvfTOO++8muO4d+u6Fs1k
    soPd3d2vP/fccyFGN1MtdkHBfAaeAODuvPPOKkK4zV6vy9bRUQOOk3tfeOGFXsMwitnrE2yd
    i4XnfkEx/YIMeEG94wJ7cqlBNaUUHR0d4jXXXLNdkqStuq47NU3LZTKZ6PT0dDwWi7kFQcj5
    fH4AAqqrK7SRkWGybt26VaWlpbaioqDgdLo4QeBTZ8+e+fYzzzzzMqU0yey+aRuWLMsgXCpj
    WRiVJIS8EX+BFt7E3ybgAGB4cHAwOTEx4d68eTM+/vGP+//0T//U39jYiKKiIoiShHA4jPvu
    uy/47//+71dNTU2lAEzH43F0dnY6Z6an3M2NDdL9d73be3JV240/e/r59eHJ6VeQ5/33EUJs
    7N6Z86fgEqUZf53LMAxzI86OhSZGQGBjh5SLjUFGXj7SRilVYonkaVVTZ1gExcwurKisrLx9
    xYoVjueee254cnLyawCiBw4cuL+jo2Nrc3Nz1eHDh1sARNiYdZayNTMySwKoeJ6HrusigKs4
    jrvG4XCsLCoK1FZXVZfV1NQ4GxvqhYaGeiIKAv7u7/8hd7anNw1giB3IGUrpFIC4KAhSaUkx
    goEANE3N3ydqgBoGOA4oDhYBIE4At3EcSRgGNTMJYeYsbrpl31UVt16/t+Jc/1Br15neK8+e
    G5CHRsanZ6PxYUVVuwF0Uko7CSExBjKtSgom5e03WiOKogCA3WaTdl995a5tAY/TeP7lAzPj
    E1NfpJQ+ze5Dljn7ZkTJXA8CgJurKiv/ase2rU0Cz2FgZIwACAKoYN8zzqKn0htkGAirA3Gy
    v2cUZJisShqXnMbHcRx0XecA3LF69erPbd++3aHrOo1Go8bU1JQaCoWSoVAIBBCymTRC4RAq
    yst8VRVla6srK7iGuipSW1VO5FwOX/63bwVHxsNHGAUhASD5287h/d/UtG52dnbI4/FE/X6/
    U1VVnDp1SuQ4TgyHw6ipqYHf74eu6zh48KBt3bp17YSQdCwWm0kmkxFd1/Xy8lKP3e4oHxwc
    93V29ta1tjbffd11+3a0trb+/KmnnnoqkUjwBXUNdDEg+XbJNrDvYciyPC1J9mFKDS6RiOss
    kARLFi0TjcayXm9Rl2EYgWQyqTBAwe3cuXO13W6/ZXJyUqqqquo9cuTIvwwPD8s33njj+2dn
    Z9coilJXWVnZNDY2lmRjNpizbJ6Fv7F927p1K799+/adgiDs03W9Q5blumQyVTYzE3HFYnEx
    kUgQVdWwceM6paGhIfve9753HccRg1KayWaz09PTM9OKotqzWRm5nAKe50EIzwBD/hZmswo4
    jrOvWrX6xuXLV2zjOC5HKY2pqhyOxxMjHMdtnZhIVA0MRODx2JcHAs49Hk+pvHZt+czq1WtH
    UqlEdygU6jx06FBnNBqNWAKPC2zCUvrDPM83JRKZOycmYkIwWKSuWrX6uZGRkf/X29sbZX5L
    ir1WR14BCADw8MMP1wQCgc9PTc3eEo0mHcXFQQBwiKJYyYKIafZd4wW+kBUsEABk9erVNofD
    IR48eFBl2Q2rj6m/FZk45v9C13V8/OMfryOEfL63t3d9Op2mkiRRp9Op+3y+bHl5RVoQhADH
    CXC7PRAEia+paWguLa3lslmVm53VSTicRHV1gFZVVY1JkjSgKMoMAw2cJYu2JJdwKQyCJfJn
    PgQWneULooPmDbTSKHS8zSk3BePkAExNTU2FDh8+XPH444+jpKSEfP3rX8eKFSvQ3d0Np8OB
    5cuX44MPP0zWr1vrfeKnP/W89tqB7ODg4GQ4HD4dDofjR44eq5uZnlx57523CZNTU1W/ePbl
    OzRdbwXwFUrpOVYMZhRw1YzFwNZbOV/m32YO3fR0JDbLcVwpiwoE2XeVLYVrJaqmu1IZuYxS
    GMjTUmwA9gUCgequrq7s5OTkCwAGAXh0XU8qikIDgYDIIrJ+9qe1C0UffpOLUWgcPM9/5IYb
    brj+9tvfgZZly1BRXo6iogBskgie5zA8PIyv/svXbbu3b7mpOFhEDcNAOp3G0ROnojOR6I91
    w+A1TYdBDVBKQRlYyIMGAlXT4PO4HPfevm9dwOeBoihIpTN0OhIzTvUMpMYmZviSYBE2rV2J
    tSuX47brr+SisZhjeDRU23XmXO2BI53bX3ztyEA2J/8DpfQMc7jjizjPS2HwE4qinqiqKLvi
    Yw++h5uZmSkan5iqRj5trDMqgJlpkJmDTwGU+Xy+D37i9z+27P673w1DzeHHP3nC9sef/fyW
    2Ug0wcCRwQCPgIXFztYaEQJgGYDfBzBIKX2CGRVYehiYkSwdlzgKX1xcjMnJSVEUxfXvec97
    XH/4h38IAFBVlY/HYuLU1IQzNDYKt9OBgM+LDz54P+58x8283+uCx2WHQChg6JicnMRjTz7t
    HhkP1wMYxXwxrKkgxVmjyW93oGA5H6wNijgsLP4nFkGDwuLet30wyfx+bJ9wIyMj0Y6Ojsn6
    +vqqqqoqnD17Fk888QQURYHf78dLL72EeDyOUCiMUGhCqKys9DU1NXoqKyuCuVxu+OTJk71u
    tzvc1taxOZHI2DkOOqVGMBgMbvf7/WcSiYRqoSZa51LD26jurUA9T5idnZ2oqqqK6Tr1ZLM5
    O7MNhunIchynAigWBM4tCKQklUoZAAJ+v9/V0tJyvWEYZSUlJalQKPTcgQMHxjiOC2QymSTH
    cZBl2eZwOMoA+DCvPJUudDR/Ext35513Sjab7aHBwcH3nD7djdnZCFKpNHI5GbpOARC4XC6s
    XbtaBKRred5zDSEEgiCgrMyZFATxMU3TRI7j2NyQOacy/zPA8wQAbzOMwGrDoAA4CAJHHQ6O
    lpfTlMNBMDMzQ2dn0yQazWF8PElsNtHuctmq/X5HdTAYuKK9PTg6OzvzxcOHj3RiYd3cUtsE
    AEA2mz0N0KnBwZmqRIJIra2BasPQi9m8yWydmpkWM/DJBYPBG0dHx2994YUD9lwOKC4uw5Yt
    KxtXr161cXx8/Di7d1lLIMn0H4nlPOA2bNhgv+qqq+4VBH7dihUrfvDoo492p9Np028qVFi6
    pBlac46cTmfd1NRUy7PPPsvF43EQQiCKouByeWxeb8DvcnkwMxMHIQKee+4wBwicpgGACEGw
    gedtcDodJBgUqyVJrFIUxZQmN0UT+Lc9YCgEC5RSiaXlVgNoBVDMBpKklI4A6AMwxRaBeRNz
    BSmzt204ydKZjwBAJpORp6amwPM82tvbsXHjRgBAQ0M9du/aBUHgYbfbceutt5Brr7ma9Pb2
    uJ782c8bHnvs8YquU6fGVFWdeGn/wYnOU92+RCI1q+n6NPJKKVUMiduZMyxZIgXnpRffKqO6
    yP2vEgThfTfecMPGmppq4Xvf+/47p6an/QDOssg7B6Dd7/Pdd/dd71lZW1tLvvRPX7lpPBTS
    mXM0febMmZOGYXQDOMRARIPX622prKwkx44dMx0lrwUsSJZI85JFiw3DUDmOzG7btg333nsv
    qKHD0HUYhgFq6KCGgUQiAY/bSf78kx9Ba0sT0XUdMzOz+OAn/tQ5E4lWGpTaKDXOS4pSSkEJ
    gaHr8HvduOcd16GpvhKGpsMwNJLOZPkv/ecPfP/1g6egGwZ0w4BhGOA4DsVFfhQH/NiwuhWr
    25rJ0ZPd3mxOrmBZF54536rlkFySW418HcpPfvrzZ64rD/qaxiemzIO4mK1Dm+XwMmkSFEBx
    eVlpw45tW2G3SVB0GZvWdqCqotQ9G4kWs/XOs7015yxaawLYGVYmCsIfbdu66R7D0JVDh49v
    lhXl3xjgMGUpTZraxYikveE1OTkJ5PnMktPpBM/zc9kqe1kpggEv6qsrwcGArinw+9xw2QXw
    hELgAKproIYOWc4hk8mZVI0SSwQ2vZRG4RKemQQAAoGAuHv37qKiokAppQgoiiLE4/Hc4OBA
    6syZs0ld11UA1uJea5Oi3xY1IAKAVFdX11JKg5FIBP39/YjH41BV1cw+YHZ2do6Kkslk0NfX
    h+HhEa6kpMTd1tbatn79+vpIZDY0MjI4TgjB8HDvDICky+WSly9fvrGpqaluenq679y5c+dk
    WY4QQhJsfeSwUB2Ivk1sA+6+++76ioqKBwcGBtfmcjLX3t7+Lk3Tnjl48GCfrusZjuO466+/
    fl1dXf19p06daZFlmbS1tdyyZs0a2t3dPTk6OhouLy/nQqHQyZdffvk4AMfatWuXud3uptHR
    UZSXl2uRSIRYbEMWC2uufuPxrF+/XmtqapodHw/hxIlOAGSOSsRx+a0pSRIoJeTEiQFMTycI
    z/OQJBs2bWpyqKpaqeu63QowzePVpLkTQqBpFGNjMpJJw8xCEEEQSG2t4K2tFcBxVqABKIoB
    VZURj6uIRlV0dHjduVyuAsAkswm0oGh+KTNH/CuvvHJq586dT7W21tynaRyvqrnM5OSUkwX3
    cpaAhxnQoVu3buUBLJ+YmLInkxnY7T5kMhqi0Qzv9fpKJUlyKYqSKggym8qB5sTxoihK27Zt
    u0mWtc+EQtHq4uKSXbfccsu/PProo6+oqqpYiuCxWLD1Ul2GYYi6rvNaHgWAUgpFUaBpSeRy
    OkQxDY6TYLPZkMko4HmA4yRwnACOE8DzIkRRhKLEBVXVSix+YIb5hkvmAwkX80CwNKqpBPCu
    gN9/x/LlLctXLF/mKy0p5imlmJqcMs72nkv19PYNx+LxlwC8QCkdt3D10yhoSPF21NW3HIIS
    gOs7OjpWXnfddWhra7NG2cERAo/HzSLKOgxKIUkiVnV0oK11Bbn15psc333ke83f+e4jVUPD
    IzPTM5FetiHcNpvNZ7fbGuScnM7Jcj+ApymlZ1kENmkxDNbmHfRtYDADhJDf33fddbf97Rc+
    J1ZUlKOtdUXVF/7u/906PDzyEoBOAE1FgcBNH//ohxs+9MEPELvDDkVRSj//hb+7Np3J/AJA
    t2EYp5Gn5UgAVvn9/uvuuOOOuomJCXR3d0+wCLrEgJRgiVIu9aWrqjY8ODhoqKrCcSB5JagF
    xW9AJpPDS/sPovvsOSiyjNloDGOhCRHAap/HE/C43aCg7H10/v2UwmaTkMrk8NX//iFaGmtQ
    UuSD1+WEIHCYmokCAGySaAEcFAbLVHCEIJFMQ5ZVA3mVoXLM9/pIL5WhLMiqJSemZqLffezn
    0HVDFEXxSlVVTbBnY0Yhyw4xleM41TCMxMzM7MTx48frljVUQ9dUnD7Tg8mpWdni2J/X4ZgB
    cy+A3QA2AljWumLZvk9/4sN8ZGbKMTU1ddPZc4NZAN+ilCZYpNXax+GSXkwJylAUJXTmzBmE
    QiHIsoxwOIQz3d04c6abTk5M4MZ916Cmqpx8+av/SqPRGPV6XCQY8JFgwAuf24mRsRAGR8dT
    7D462bzOFVD/toGFhoYG4brrrlvu9/uvNwy6I5eTl6mq5rbbXaSoqDjb3LxsdOvWrSe6uk4d
    OHz48JCu62nWpChTYBOMt5tdKLANhFKK+++/v7qsrOxDR48erR0aGpoDCtbX58GC9V8Cw6CY
    nJzC7GyEVFVVutasWd1UWupNHD16ZLy6uqa0o6NjpabpQjKZhGFAW7myY7q+vv4HP/3pT59l
    gaUkk6ZOW4Jwb4mDVNis7cEHHywrKyv7ZE/PuX3PP/8iL8sq1q1bXb9q1arbOI578bXXXju9
    efPm1vr6hpuOHz9Vc+LEKVCqQ5Kkqo0bN147MTHx1JNPPnnS4XAcz2QyUbfb7bzhhhs21dbW
    Xjs0NFQRCASooiihwcHB1MXYL+Y4jh49Sq+++uqRQMCvCIIgGUa+tqGwtEoQeJSXB+B2OyAI
    AiRJgtfr5D2emg5dh0+W1QuABUDXKTiOor7egWRSh6oS6DoBpRxsNm7uNdbMxPw6JBBFDrqu
    0pmZWd8iNiF1MW732NgYVVV1sqLCo2UyMi8IpHH37t3XPvvss8ey2WzcEsQxa26MY8eO0V27
    do2WlJSoHo9P1HUBLpcTfr/LmJ6eTquqai1YnwuSUkoRCAT4u+66q9Xj8VwN0DpCuL2dnQPV
    4+NxrFwpLauurvnw6tWrU0eOHOllqkIGForIXPIz4sMf/nDU5XJFi4uL3YQQOJ1OFBUFUVZW
    gaKiIJVlFd3dA2TZskZaWVlCKSVE0wBFoUTTAEIElJQ49N7eszFZlkXkRTIkSqlg6QP29gQM
    1ugBIYSjlK6UJOnPdmzfdsO999zl2LZlM4qK/BB5HtTQkMtmuJmZGe+Bg693fOuRHyx/9eDh
    9Yqqfp/RKKLIUwvM1GHu7QgaLCcCAXBFTU3NBz/1qU95W1tbLWAh71Ca9BPzZzBKisaea1nW
    jE//0f8hO7df4fzHL325+rkXXnJrmqYD8NvtdnHLpg3Ys/MKvPbawTU/e+b5Ek3Tv8HkN6Ms
    mpRkmz+Hheo4b5VRsAO4d/Pmzfd99s//zNbU1AhD13HfPXfB43b5P/sXf7Wnf2CwyOV01v7e
    Bx6s/fDDDxKX0w4QDg/cdze6u8/UfOeR720F8BobVwkhZG1bW9uWa665xjM9PU2feOKJCVmW
    DzPAoFge2lIbRJMDCyAUDoW0TDojuV0uFurI219K8zUI7e1tOHT8NBx2O3iBh02ScOP1+4TK
    yvLKlW2tWNbcCGoY531BwzDQ3FCPT/3Bh3Dy1BmMjI2j69wYsrksDN1AJpvD1o1rsHLFsrm1
    VJCmQGhiGrl8fUEQ85zPODOaSxmNpgBcvCA8cOvNN6794EMPQOTBf/s731vzP4/80JmT5Z8B
    GGV62WZBZo4phEzNRiLf/bsvfqnxxIkTJQJP8OwLL8uT0zP9DBhaU8ZzBzql1EkI+Uhb64qP
    7di2pTgyGyFne3rxxa983RgdHU+PjIXNcQaZIZEJIQv0zC/lxWhsBoCDjz322NiZM2eKk8mk
    MjExIUej0XQmk8kBCAZ87uKGulr84Mc/TWuaNmtmUDiO40VRIIZuqKqmnbMY9yWXzbtEZwK5
    7rrr3GvWrLld0/SPDA6OdAwPj0mRSByyrIHjBLhcLgSDgaaamvJtmzdvvaG6uuqJp59+5qVE
    IjFDCIlbzrks5oUf3rYZhquuuspZVVX1YF9f35WHDh0i54OF84FC4XOUAuPjYUSjcW7lyjb/
    li1XONPpVGZyclJzu92cqmr2kyfP2uvqatz19fX71q1bl9J1faavr+9cOp2eZfSNhCWqq70V
    68ZcC1dffbWrvLz8/aOjY+965ZX9QjKZAsfxOHbsJFRVL16zZuVVgUCgtLKyqqm7u7+qq6sX
    lBIYBsGJE6dRVFTUcMUVV2x+4oknDqfT6VRpaWnZtddeu1GSpI19fX0ur9dL/X7/+AsvvHAk
    lUolcb707FLuG5rNZkNut1uVJJuUy+UKgCCBLCsIhycQCPjhcjlgGBS6rmF4eITPZHIVs7Nx
    zMxEF9CQ5p1+Dul0DseP96GkxAe32wm32wZRFMBxPHhewfQ0RSSSWcQfy3+WwyEgm83QZDIZ
    YBnKNFsPsaW0Cdai53e84x1bDIPe8/LLB20zM0lUVFSUrFzZeNO6devEV199tZOdz2lLJkyV
    ZZn29/c/U1tbe92ePVt2xWIZrqjIb/C8MtHZ2TlMKTXv4xydiImmkAceuH8Dz4t/PzQ0vjGb
    VcTy8hJUVZXQyspS1Wbj4vF4PKGqajGACXYmpyyBRfIW7APu9OnTYxs2bDhy9dVXF8uybNhs
    NhlALpFIpDVN5Wtrq6omJuK22toyY3R0cCoajSgul5t3u92C0+kioihhfDwUOXHixMgi9mBJ
    97dwMQ5HS1Sl0W63f+7uu+7a96k/+j9cXU01DF2DoanQ9fx+tUkSKivK8I5brseGNSulf/zq
    v2155Ac/8eRk+VuU0h4LR806EdrbJQ1tLdijlFba7faPPfjgg43XX3+9qZrEnEgKGEZeDccC
    FLAAQBgwDAqB57Fr53Y01tdx/99Xvhr47299J5dKpSfj8bhy+Mix0vfcfrP73e+4ibxy4PVV
    sXji/aA0SoGfADhXcE9zlzqiWqBCwAHYWVtb+7FP/dEnvatXr4ah5yk7PMfjtltvhq7rnj/+
    zJ9dsWplG/f+B+4hDrsETVNBCIcivw8f/dDD3MmurpZTp7ujAPoBtO7YsWPD7t27xaefflo+
    cuRIj6Zpx9gBkGQHoKmeIF8Mo8jGODM9PZ2Nx+OSx+0yj3UABJQaKC0twT/83eehaRoIKCNj
    AzzHQRB4cATQNQ2GobPOK2Yv9zy49HhcuPaqXdi7exvknAxZkSHLMlRFATUMCAIPt9MOw6D5
    DzbpTTQPW8YnpiDLCliGIYA8vceBJa7pYPuvtKqiYsfvfeAhfvuWjVByKbjtAnn14OGa7p5z
    DRbDFAFgZ2tUYXU4Tw+PjF756GNP3ByPxyPq/L1ULFkBa4dnA8DK+vq6hz7/139ZsmPrBqQT
    UXztX/8Df/dP/xJRVe1FAOMAptnfciJPe+TIr1tZuzRrhlBKD09MTHx6YmKiHvmCfzMaVEQI
    2eh2OYt1TTO5vTMm4DcMQ5FlxUwvz2ChCo5VXvltfZlg4ZprrnGtW7fuA7FY/FOvv360pK9v
    CKqqz6XXOU5EMpnD9HQCw8PTYkND1bLW1pr379u3r/KJJ554MpPJTCxSBC+z7NPbqZh3jjq3
    YcOGbfF4/J4DBw7YksnkgjjThQDChZ7L5WScOHESiURSWrt2NRcKjU3u379/eMeOHdV+vy84
    MTGLysqKlh079vw+zyMC4KudnZ1dbK6sRZ5a4fe9FPaSXfz69euuicXiH3z11ddcs7MRcJwA
    QjgYBsXp0z3gON6/bdumK8bHp/ju7n5QStj64JDJyDh+/BS/a9fmtvb29tjp06dHd+3a1WEY
    xtqBgQG+oaEhpyhK97PPPnt8eHh4ymIbzEz8koFMc+7S6fSs3W5POxx2Vy4nz91fyo5oWVbw
    0ksHwPMCs4956hmlBCZrjecFRmGi7DmaP94phaYZGB+fxcREDIIggud58LwAQcjPG0CQjzhz
    5wVvAQK7nUcyGaOyLJs2YfZi2ATL3+S8Xu/Wkye7q7u7z0EQXMhmCYLBoLumprpVEISwpmmm
    mIcTgMSCzPTRRx8dufvuu75XVVWztrISnpGR4dMvvfRSTygUSlmyzgvYFDt37rTb7Y77u7rO
    XHHoUCc4zoGKiirs2NFhTEyMHD1w4OCpSCQylc1m08jTOrOUUp5Ykdkl9h0B4OWXX04SQv4p
    EAjsTyQSgYmJCVsoFJLi8binsbGxYfv24or8OqHGwEB/vK+vP4F8XY8mCHyWEC6j63pU07Rp
    zPegkpnoy5JS+YWLcUgynrcTwAduvOGGaz/753/KlZeVQFMVUF2DQc3iTuYk63kOeHVVBT75
    0YdJLBZv+/GTT9+APDdasKiAWPXc6dvFKLCbzwO4cdu2bXseeOAB2Gy2OZpKYUZhwXOglrnI
    Z9oMBi6qqyrxZ5/+JCktDjr+6Z+/HohEo0ORaDR+4PXD7puu2Y33vmNfYHw8tOP5144mE8n0
    KUqp6WSYi0XFIg1TLoVhYGCpxuFwfOLB9z3QcO211+RpWNQETXnqzG233ERmZ2b4H/34cQwO
    DSFYFIABgBADVDWwsm0FHrz/HvuffvavWtOZjFRdXd1+5ZVXio8++mj69OnTryJPZzIPjjnA
    wKLZ2YsYRYtOz8yk4vG4r6amGko2C1nOQVUVZLM5pNNJZDMZZDIZ5LI55HJZKIoMRVagqAo0
    VYOua9A1Dbqhgxp5LJwX7wY4nsuDC56DKAiQRAE2mwi7JMFul+B02GHoGgSeg8DzEHguz181
    CGRFwfjEtBldsBWk4smbvX+/wvqn5royi7fzmRNK2N92AHBQSu0Fjh4F4CoKBCqu2LoJzzz3
    QkTVtH728WahsvlQka9foACqG+rrS9atWQWeA1wOG7ZsWA2v201mozGzZsLFjJBo+Vtv2ZnB
    ZATTAF6nlA4i34ytmGVBNLvNRirKSonDboPdbuPS6YxsAU0pFgWMMcAQZY8IW+e5typa/Gs4
    itz69etviMcTn3zuuRdLhoZGwXECBEFcwPnmOA4cx0FVdfT3hyHLhmfNmvobt227Qn7mmWdf
    pJSKBfLJpkoUfbuBhjvuuMPD8/x7u7u7a/LKngsjz3nHkTBFnPOfWww8UEowMDAMRdGETZvW
    lum6bsTjcbW+voZGoykSi6Ulp9MpOZ1cNpVKuQD4WGTWKrVMLmTPLtZ8mGvhox/9aLNh0E8c
    PXq0cnh45LxxGgZFd/c54nA4hfr6Wvh8XuQZjgYMgwMhPCYnZzEwMObs6FjVLoqiy+v1tvb1
    9fEtLS2JkZGRl1955ZXuTCZj1k0tZhvUpQTaiUQiUVZWlnA4HKWRSIw58yJ4XoAoipAkGyRJ
    gihKEEURgiBCEATwvMCc/4Xr35wTa/A7Dy7yAMMwrA8KXQc4Lv//80BlIS3J4RARDsepqqoS
    5qm7v1Lm9c3GXPJ+YL7+ON9LwhzXHCgUTLvAAjum8qNZoM95PN6KqamISxR5rb+/fyAUCk1j
    vv9MriDzTMvKSu2apreMj09AUXQ4nSKSSQWxWJZEo1GMjo5yzC5k2FrgCjLlb0mw2TAM44UX
    XhhgDIBS9igGEAgGgxWGAUFRdBgGOI/Ha8rP64ZhZBXFMKXWZ1lALsr+jbFxLukavyiUJNbZ
    t62+vv72D/3eB4WKinJoqmKh5CyMspuOsq7rKC0uwoP3vJM7eqJr1eh4uB/5Al/d2tUQ8wVv
    bwujwA7aUo/Hc/u73/1uV21tLQMLC7MIi2UUTNCwAFQg/1rD0OFxufDhhx+CKAquv//HL9dH
    Y/HEo4/9NP3iy6/yNkkgAs/xsqyMs00TKEjVKVjIV73oc2FZA3YA91555Z7d73vf+yCJIgxd
    N3fK3FgFnsPd7303RkdH8aV//hd84a/+HJUVFcgnGCk4nsctN12Pp599Pvjzp55x7dq1y3Hm
    zBl6+vTpswC6LOPMFGQYzCischHGTgFMjY+Pj33rW9+qes+dd+Kxxx7D2Z4eZDIZaKoKjYFg
    sKxC3gHKrxWOkLnfOcLBegZTVsRsvt9gv+uGDkPP7xEKysAED7tdgtfjRtDvw9qO5ehY0Yxf
    vPAqDh7r0jFf/5NFXprUunfomzzQFoSpCiUwmbrVdCgceuVrX/+3Zbqa4yWe4NuP/IAODI3M
    sMOMsIPefFj7KdQ31Nc2dbS34plnXzA5tWZgIMMcnDlNeRapnx4ZGYmc7Drp3LphNTKZNI6c
    6EIqnVEZSJAx38xHZ+/T30rQwFQ5VHYfkizAYFKuynVDdz/93It5qT1Nd7BxpNj4Y4x2OI28
    MESMzWvSkk1T8TYFDKYajKIo+PCHP1xlGMbDR48eKxscHGKFe9wC3fnzHwQTEzEMDs64m5qW
    7T516vREOBwmTCXOWgit4+1HTaJ1dXWtiURi19mzZ03biPm+hmSO6246y/msQ/65vNN1YfAQ
    Ck3g8OFOYfPmtRWRyGxkcnI87HA4wPOiKAii2NV1tmdoaEhEnoJirQey0nKMS7EGeJ6HpmnY
    u3evy+v1Pnj2bM+Wrq5ToBQLinXN8em6ga6us/B6PVi7tg2vvXYCmUwOPG+wbLyOnp5BVFWV
    l27bts3b19fnLCsr06empk49++yzZ3VdP882MDqbKRKypCA7HA7P1NbWDq9a1dGs6waWLWtG
    aWkZJCkPDDiOZ9kCM7gybw7nz1xrsHteLSk/P9x5czRnU7j8aw0D0HUDqkqhKAbicRnRqILy
    cje8XhgnT85krIGYApvwqzq8Cwo0CsGEYRhGIpE4UF9fOxaLpatnZlIoLy9HWZkne+TIqTCj
    WnPIC1vwlmafdMOG9Q6n07VuaGhALCsL5jiOMyXSiRlFZ3bBpCQa/f39akND43BlZQXC4Tg4
    ToTf74XTKeozMzOEZRXM7tAUeQnut6yHlSULadqBDKU0ZmZMi4qKSHNzs8fv9/JtbQ3w+71c
    Y2OjZ2BgYDoej5sZ54jFLswyu5BgFK+UxZ6+PRu35au1FQ7A1g3r19esW7c27zhZwYEZaZ5z
    mvMRdrBDoK2lGRvWdHhGx8MrkFc6Mdtepy2peP2tNoIL8n3AqmXLlq3ZvXs3zALW/KbO/2oY
    JB91tYCFxahJhaDCoAbsNgkP3nc30um06//78r/QSDQWikRjMkPLNhZ1rLCk57RFjMIly8qw
    jbCxtqbmvo98+MO2yooK6Lo2R8cqBI0ulxMf/MD78Mk//jP8539/G//nYx+CZLPNZSRKi4O4
    4bpr+L6BIVdjYyO+8Y1vJAGEWJQkZ3EQE6wA3HSmljzDwDY4BRDOZDJf+spX/vlDrx8+vHnd
    2rXilXv2wO/3wufxwONxw263QRLFuQwAz7IG+UPe9Jrni57Nxm0GU13SdR26pkLVVKiKClVV
    oSgycjkZqVQasXgc0Vgcs5EoZiJRPP/K6/je40/R/YeOa+lMdpYdJDEWdYhjYW0LfTPAjxkF
    ia0zuUAC2XQ2spqmf+MnP/2Z69Drh68RBL44PDEVVxTlJCEkY8o8WgCDKX3HAVi9sr3VV1Fe
    DoMaBuZpNior1LRqhZtdn1ODQ0PxT//p/63es/MKxKJR+vNnX0zJijJkWfcZS6Fn5kIUhF8l
    YvZmIm6LgSzLw5SWTTGgpQFYU1pSfPPWTRuKmxtqQagGp7TbdujoyeXDY2GJAqeZLG2U3c9J
    dk+tzRtNsDAHxNgaXTR69lZE31kHUvKZz3xmUzg8saGn5xxzhvjzwMG8Q8zNgQmAYHw8hqqq
    ooqmpqZV4XA4gXl1lSwWNrF8O2VVBLvdvnVoaKhqZmbG6l/NZVHylJXzQcNiQGKx58LhSZw4
    0S2sXbvSf/Dga2OpVCq3devWIlXNN3jbu3fvtrNnz54YHh5WLftDsaybSwI2WS0PWbdu3Y54
    PP7ew4ePCHkAYFJqOBRStGRZxYkTZ7B792asWNGIkyf7YBg6KNUZr1/G2Ngk73bbXZlMBmVl
    ZYlXX311gvXKMR3jC9mGJRm3mWU9cODAbDAY/OqyZc0oKyvbPjExZRsZGUUup7CzW4Ou63NZ
    ATNjsPCI4AqatuWzKfNZN57VLPCWzIQAQeAhiiJsNhGSJMBuF2G3iygtdaGy0kMNI6699tqB
    2JkzZyYtGco3bRMWAwt+v5/fs2ePs7OzUxsYGChsjmlQSsnjjz9++JprrvnHdeva7yeEW2Gz
    iXxPz9nezs7OMYujzhUEkozVq9dUqKq2YmpqFmVlQcrznIJ5RaNUgW1QANCRkVHouj7R3t5i
    eDw+LpfTUVLi12ZmQqO9veci7GzIAjB9ybTFriwIqv6mdqEAEFhfSwrAFsee15BXCCWUUm3Z
    smXu3bv3XAvw68fHZzhd5zA1lSOlpQ2Vt9xym+2FF547OTo6OgUgTgiZZc1aZ1kAKU0pzVn8
    ZMJq+OiFGta9Wduw5ICBNXLiOY5rWb68RXQ67MxRNuk4JjhgvkZhtsEw4LTb0NJUx7HUTBmb
    hDgzlDa8zRRBDMOQAOzcuHFjsLa2FpTmi5nC4RBC4+PI5nKorChHbW0NOMLNASYskm05H1jl
    wYPDbsPD77sX4fCE67+//b0S3TBiAIolSZJqqqsDiWRixczMbBel9KlFsgy6BTRcVADFmlQF
    OI5/3x3vfGfjzp075hWEqEURCHTBva+qrMAH338//vrzf4+N69Zg756d7OThkM3IONvTg61b
    t6K7u5uGQqEo23gOtjZklmaOWA7CNC5CDQM7BCghRKaUHsxks6mXXnr5IY/bfd3HPvJhqbqq
    YiEonOutYMz9bhgGKNUtz1FQquffMwcqjQXgGsZCgG2VcAUounv68Om/+RI92tmdo/l9MsUc
    zGlLuvK8iMMvOSBMStG9AFYC+BpzWAu7pXIARjVN+8p4KCzX1NTc6/f76dTUVJr11tAtDq0V
    cLidTufaDWvX8Ha7bc63tDh/aRQo4lBKPRxH7q6trlzWPzCY7uw6fZp9vplZyjIAPU0pnSGE
    RJiDYA0ycOZBbTEMxiL7g1ss+vQGIIFYuoALmFfrMoGSCEAkhJga2S6Hw3HLDfuuq1+3uh2p
    RAxKLoPVbctIaZHH8cjjz9RPR+LDyFMzTcfHKpFJLKl8G4tUGZaxWCWW5/oXLPb9L4Xz7HA4
    eADrJicn3el0hkVcSQFY4BYBEHnHSZY1JBKyWFwcrCOE9FNKM6xBkZlRfFvYBUvWGe985zt9
    uq7vGh0dFRVFASEciouL0dBQj2CwGIIgYHR0DD095+aci1+lnsHMPoyNheD1eqSNGzdWnjhx
    YobneUdRUZEnkUiVlJSUN9fV1dc999yzuf7+/kwByNIudmbGml249957S3ief39X16mqUCiE
    8/sOcJaMSh4gxeNJdHWdw/r1KzE9HUc4PA2Ag2HoEEUOgYAPw8PDKC4uNgYHB6OhUEhgGboE
    G2OCAe6I6VBhvifLUtk+qmma8uMf//hoS0vL3+/cuXOmuDh4y7FjJ+zRaJytdSs4ngcA5s8m
    MDC7OxNCQQgFx4H9zLPfDRCSf+Tfp7L3ygs+w+Oxo709SCcn++TXXnstNj09bdqEGWYTopaz
    8U2DJ/O8uPfee9u9Xu9HampqXvjyl7/8NGvIawaRNEKIPjo6qv7Xf/3Xk9u2bYtv2LDhz1Kp
    VJ2iKLKiKFZHfQ4tmsGlYLBo+dRUrCoeT+WDaxxnynIbZubZXMPMxtB3v/vdWzhOvF3XwZWW
    evsmJyemzpw5rnV2dkbS6XSW+Y9WmxhjUX2ThWDKdFtjPnM1swV2YdFs9SJAg1jBAbMDpm1Y
    YBeQ709GAGjbt2/fyPPCrp6eYRvP2+BweJDNatA0gy8vry5Zv359VSgUGtJ1XaaUpi220pwj
    cw9IJkvHtAtmbyKWobX2taG/rOnvxerDwHEc53A4HOwPG4s7wgYt+H/MSQKF3SaBENgphQ95
    /WQ35gt0rBzotzS7wBBhidfr3b5r505is0kYGR7Gt7/9bbz08stQZBkGpdiyaRM++Yd/ALfL
    dUF60gUpS+xnv9eDj/3eQ6RvYND7yqsHFQDTBCjdd+1e1+b1qx1/87df3Hr2XP8YgFmmSpOx
    HI5qIfK9CNE0GIZBAOzs6Fh5/b333kPsdjsMbT67MBdRn/vZ/NXA5o0bcNWenfif7z6KjrYV
    KC0pBjiC450ncfL0WWzfsQv/8R/fMOXUHGwzmJy+hCVyksIiTbqWaswFTV+GKKXfefa55+t+
    /otfrL7rPe9GKplEKp1CJp1BLpeFLMtQ5HzRsqLkC5dVTYWusWzCnCRrvsCNIwQ8z0FgBdKC
    wENkWQpB4CEKAmyS+ZAg8Bx++NNncfTkGZnOcxjNaLRpJKzG4U3Jx5WUlNCpqammooD/Ix63
    q2VkLBSjlP43O1jM+5A1HVVKaRxAZNmyZVSRZdfU1JQL+QJks5N3YdS7qqy0pLW9bQWGhofN
    U9oEIibozQGQmVEgAK5f3tRwV+vyJunJp1/qBvAUOxvcmO/8OQ4gRAiZYuPOsO/nQL4XjAhg
    jI3BPEhVSwTLvHhcoOt8QSqeMIDgQb5PSh2AeuTlpH2sg7m5TidZ35kxAHW6rtcdO3ECk1NT
    KAr4wXMcYvEYQuMhZGXVhnyzKRvydEMP8jKyPrb2weYoxj43zO55lGVYrGoi1t4Fl7yB1/Dw
    MMrKynhKaWkqlSam/fxlVCTr7wCBouhEFCWvKIpBRVFmmW0wJWbfUrtgXR9mNLG6urolmUyt
    HR0dhSCI6OjoQEfHSmSzOcTjCShK1lr390sBwoWeo5Sgr28IwWDA1tDQ4Hr66adD27dvLzYM
    rri3t5fftm1DdWlpaU1/f/8Ysw0pyzmpXEzbAABMY56rrq6+dmJi4qqurlMwpUcXZhXIouMb
    G5tEVVUZWlsbEIulIcv5PoWVlcXweOyIRMKorq7WX331VZXtc8F0LhkNyeR1LxpI+k3HzIJI
    GqVU7u3tHc1kMt+75ZZbaltalm09duwkJMkGm812Xv3CwhoGwQIiuAVZJNPnnK9PWFjPYP6c
    NyV5Janqai9isXH12WefjaXT6QgDClMWuxCxZFveNE2VUoodO7aLLpfr3ZmM+j6bzbF6165d
    oy+++GKIgV7dQnmCruu5WCw2JcuyHA5PoLi4JOByuTTmxKtgYgXm1d7eLvG8sH52NuaUZZWR
    NTgzQ6tZ7EKOOfvaAw88UOPx+D4+MZFodrul1IkTR1/Yv3//FPLNQyXz7GW2YYLVe5rKWfwV
    V2wNNjU1V3V1dU2cOHEixe6nbrEL1oahUiGNq7BrvaWhKKmsrLRv3ry5pKSkpM5ut9fzPF+D
    fI2Ci1LK6bquyrIcm52dHe3t7R0ZGRlJezyedlGUbA0NlchmZaiqBkHIwuXiYLPZSVFR0G+3
    2x2qqroaGhpqWlpaGsvKynwOh8PJcRxPKdU0TU3lcrmZWCwWDofDk+fO9U3PzMwkmaiGYqH5
    q1ioqHnBHi0XrXGbpmmRiYkJqmka4WAFB5ZCX0uUmdL5rIOmaZieiZiCLxLmi2LM4s23TO2k
    IKpoTuaapqam9k2bNmJsdBSf+ZM/wezMDG69+WZ4PG54vR6sWbUKTodjjmYDY2G9glVFyRqN
    p5bndGqgvrYaH/vgg1z/wKA/FJ6My4oyeerU6fqdW9Zxap4TyDMjmmXRN5clQntRjGlBdiFo
    s9nufte73lna1toKQ184pjk6UiE9yTAgCgLedfutOHT4KJ78xdO4973vgpLL4rGfPImGxib0
    9vYgFotxFuAIFsnIss2fvBBvb4mNILX83RSldCibzZ742r98veOpp5/hYrEo0uk0HA4nRFHM
    KxmBWuoXCDOUmK9fKJgLo+CR9zcxp5iRTmcgCjxcTgecTgeOdnYblNKExSiMs0cYwBSLsP1K
    6iCMQrFiVfuKuuuv3i393Zf+5R0zkdgRAGG2/mVLxM6kE+W8Xg+tq10p7X/11aAlkmmtIaCM
    jtRaV1tTVl9Xg+GREfAcb43Kc5ZIuc5S3Mv9Pu9H33HjNSUHj3aqsqJ0Y75BX9KSXQgDmGGO
    wzr2fIoQstvrcd+sKKqezeW+RCk9bnGqzfS0UWAYzus0b6aULUChGsAuSZKuLi4uXl1TU11Z
    UVHhKCstFbxeLycIAtE0lcbjcTo6Mqqe6+uLjYyOdSuKIl25e6fv4x/5INpbV8Bht4FSClVR
    MDwyim9881v8t777aJOsKHaX0+mvrakqbqircVSUl/Jup5MAFNlsjkZiMWNyakabmJxOTExN
    j6XSmW5K6WFK6RkAM4v0LlAK0/AXGzSMjIwgk8mAUqqKongBKtL5oGExioamaRwT1TAL280m
    lm+bzLPZwdjhcOwaGRmtnJ2NoKysDC0tLRgdHUM8HkcgEEA4PImBgcEL0rIW1jC8sVOtaTq6
    u8+RrVvXe6urqxODg4Opjo41wdLSEmIYenJkZERmoDPDaG5Jy7q4aLaB2QV88IMfrNB1466u
    rlO+WCzG1gAp6Btg7W48f/913cDZs0PYsWM9amsr0N8/DkkiaG6uxdTUBMrLy+HxePmioiLn
    +Pi46deorMA7xc6pFBahM/8m696SFTJtQo5Smh4bGxuNx+PHVq1q31hdXSnYbHZIkghN06Bp
    OgjBnFQuO/4XSe4ufr8LszLma0Uxj5d1nULTKNxugT7/fE8inU6bUXWrTZhmNiH1q1L5/v7v
    /x4rVqzwGQbdcPbsKFdREVzV3Nx0w4svvvgjFlHXGOXHpA2qoihmOY7Tp6amUFQUKKqoqOD6
    +vqsRblzAYw1a1YX6brRMT0dhWFQBhiIxOyCOVNmVkzfvHmzraKi4p7p6fSV8XgWTifGenrO
    jrLXZ9kY48jTmCcDgUBu69atjaWlJa3pdGba5/NW+v3BO3ieX+P1ep+Ynp7+9vj4eALz9S+Z
    AsDgsGakC8CCmVEgV111lae9vX2t0+m8VtO0K1KpVFMkEg2kUikxl8vxmqaT/H0Tqcvl0v3+
    ouzu3XtGdF0bAMimo0e7MDCQp7SZjQBtNhtqairQ3t5YtGfPnjVer3eN3x8sURTDm8nIYjSa
    5CilhOM4KkmiYbe79PJyf66mpnF648ZN/bOzM0dPnz594vTp7hFVVZMs4JfFwr42+oUCShcL
    MBgATp040ZmenJx0V5aXWsCBhYZkGJaswrxTNBuJ4njXGZPvS6wpq7caLBQ0aiPMKbl2+/Zt
    gZqaaoyOjmLH9m0oDgbx2I8fx/DwCO695y6UX12aL1g1FqFhndeXwYCqquB5jt22eafboBTb
    t27CnXfcKv3T175RomlaLJtN08jsNGQ5ZzAnx8Uirh7LQZmxZhkuYnZhx6pVq/bcdtttEHge
    uq4WZBToXF3HfMYhf14YhoGaqkrcdvP1ePRHj2PPziswOTWD7p4+XLF9Jw68+greddtN5JEf
    /NilqKoZ/TWjDgoWykwuOeWigMJipkhThBCRUvrCya5TFSe7TrUCCAYCAfsXPv85bsvmzRAF
    HqIozKsZEYBwBGQOSM1TlvK1C3n1JE1T8w9VhaIoUFiGYnpmFn/6uX9Eb9+gdfxJZghC7DHG
    ouhh5kTHLYDhTU2IYRj5poscL21ctwatLc2Nrxw8chPynbYjDJCa3bSzczxMgG7YsA5er7cs
    kUjAku2xknVFAGvaVrQ4vR43BJ4Hz3MicwA1y943594D4MFr92xfv3xZI/7zuz+aBTBkibKb
    fSbM6FmTx+N+X3NjY5uua3rPuf5U+4pllfffeaut51w//uf7P/lUIpX+J0ppNzNuhKWpC+lY
    2cUyC5bmcTf4fL6H1q1bt3Hv3r2urVu3orGxAcGiAASeZ2udghoG0XWNZDIZ28DAQNkPfvhY
    6aM//BE+9PBDZPeO7dA0bS6YYBMFdLSvwAfedw8OHDpSWhIMlL7rHTeT9Ws7UFFWCklkdjNf
    80J0XefS6YwwMTlp7+kbLD145Pjag0c6b+gfGjmhqtrTDBhFC84CawMv42KDhuXLl2N6eppq
    mjYQDAZVSZJEa5OpN6Iimb+LogC320aHhyO6pmm8BVwKWPpmhL8uXXFurdx3330Vmqbt6+/v
    F1RVQywWx7PPPg+Px40dO7ZDFEUMDY1ivrCVKyh8JguUcgjhIIoCvF4Pksk0DIMWzBVBPJ7C
    wMAo39Gxqri3tycqSZLBcSJHiOBZvnx5fTgcDhuGkWQ2woV5mVH1Ys0Lq13ggsHgVWNjY1t7
    e3vnCp1NQLDQ+UUBTSk/vng8jaGhEBobKzExEUUg4IbTyWN6OouysmqMj89wzc3NrrNnz/IF
    zb2sAiD6xbANFr9HJoSkKKXCwMDA/jVr1tTX1FStlCSp2DAMxwsvvMQNDg4zlSOKhWpGVmDM
    L6At5SWHzX8FSw1DPkths0lYt64ePp+dZrNZGovF1DNnxtPDw8OmTQgzmzDKbMJ0QcbFeLNz
    Mjs7C6/X66MUwUwmi3A4ZluxovTaK6/cM93be25ifHw8xmwDz84cjVKa43lejceTSKcznoaG
    hkBfX9+gxVbPUXeCwWB1Nis3x2JJ5Au5KUcIMRWVzFq4uSzejh3bt8qy/sDAwLS9stJvRCLT
    vdPTM1m2vk0a5yyAidraWv3GG2+8m+OEfbKs+mpqSqKiKLgHBmaCqZRGmptL77/66qul73//
    +z/MZDIJQkiUUjppOf8JC1Cc1wTUBAtFRUX8e9/73naPx/NQIpG4+dSpU1WDg4Pc1NQ0Uqk0
    8nFdYqWmEZ4XBJfL5amqqmzftGltWyg0ha6uHtasj2dZRAPZrIy+vjEEgwGppaW1rb8/hN7e
    XpJIZKCqxtznchxPOI7neZ7n7Xab5PE4vMGgp6mkpHT3rl3lQ+3tK188cODAiwMDA6PsXhX2
    tTmvruOiAAYL2jpx8uTJU0888dMt73/ffW9AtTEj6cwh1nU88+JrONndY0b9NEvK5G2jgmEx
    DMsrKir23njDDUTgedTW1OB9D9yPWDSKiYkJrFjegquu3M0KtYxF6xWslCQCiumZWTzy6I9w
    +y03oKK8lKHs+TmziQLec/steOmV1zxHT3Tles8NyD2955xr21s84cnp9bpuhJjzZG4yk+e8
    5FGkAmUkvyRJ77zpppuCzU3NeblQUxHCQkmKxxPIZjIIFvlNsbgFoOLqPbvw86eeww8e+ymm
    ZmbR0NiMgf5+bN2wGnv37MRL+w84hkfH7AUOwnkNrC4yzWIO0LJo+RHk+wcsB7DO6XDc3LFy
    ZWDVqg7oqjIvN8o6fOfBo77gOWvNw5zKFnseloL5SDQKj8tFLY5ykh2Ik6ZRADDODIO1GMra
    GffNXgeOHus88A9f+bedyXTGXldX+25VUa6emJw6aRjG05TSIbYGomyfZmRZpq0rVqChob64
    s/NkgBknwpQwzOyB3+Gwt7e3LieiIJhKIhIDDGZkx+S18gD2tDTVv/ued98ivH7sJKamZ0cx
    z1vPMeA2zdLNcLvd93/o4Q9sv+vOOyBnUvi3//hmycEjR+mxzlNUVRXidNhaEqn0VgBmGj2z
    SIjvPCfUkmoOAvjoqlWrfu8DH/hA8S233IKKigrwXJ5bTS1UMwoD4AgE8PC43Vi/dg2y2Qx5
    /cgRNDU2sH1iLKAl6roBn8eDqooy8sA9d+KGa66Exvp2UHNfmWpbBPB53fB7XVjR3IBrd19B
    BoZHAk/84oXdP3ry2ZUj4+GnKaW/IISE2H2KWcamXAp6UllZGQDQqamp46WlpRPl5eU14+MT
    i2YVzldNyjtTfr8LdruhDw0NZy3BAgNvv+Z1BABXUVFx5eTk5Lrh4WFwHJcH/YqKdDqNZ555
    DjzPIx5PMIfwl9OQAAKfz4eNG9fg4MFjSKXSixZBj46GUVVV7gwGg9lYLJKqq6vwiqJkW7Nm
    7bqRkZGhvr4+sx+KnQWYFlC5lmoNWEH2Qw89VKLr+jvPnDnrTSSSlvqVhWN0OOzgeRFmx+PC
    DMTw8ARqayvQ3FwFr9eJyckQSktLEQpFMTMTw5o19c7i4mJbOBwuDC4al8A2mAAlA4B77bXX
    Oru6umYqKipWVFRUrO7o6Lh1enqmeGpqiqmDzcuo5h1H63Nm/QLPahjAnjPAcTp7vTZXsyCK
    KjIZhfb0dCmnT59OJxKJZDqdjhiGMcGyCqZNCBXYhNyvahMCgQCee+656fb29p+0t9c1ZLNq
    gFLSvmJF+x+tWNEWOnWq69n9+/cfNQyDY2sgJcs5hRCiaJqOiYkpqba2qspmk47KsmLWtpmg
    X7DZbMtnZhLBVCpr+hWE4zinJbJvFkhzu3btKpMk2wd7e6cb0mkFHo+Uev31/mH2GpO+FGW2
    eeq66667LpdT7j548JArk9FQVVXtW7t2OTweB7XZCAV4d1FRcFd5efmRgYGBQUbfKwxEzGUz
    Czn/wWCQf9/73rcbwF+cPn1609GjR/mJiQkGmonlXlulcvPgMZlMo69vCA0NtSQSSUDXjUXq
    vPJJ1EwmRyYnozhx4twcqDBfO98PKt/5O5NRkM3qmJnJYHg4KpWWelvq6srq9+27fu2hQwd/
    eOTIkVOU0oilsaO1Tm7BmSBcREd6IpFMfu9rX//Xlob6mqIrd+3Iy4Qs4OnPAwjCHMpXDx3D
    v/7394x0JmttsmJNqS+pruyvewAyNGkDcNue3buaN25Yv0ANyufz4v0PPjBHKTIdwF+WXQAh
    iEQiePIXT2P3jq2oKC89r85BNyjqa6tw5x238N1nz3mi8UTs69/8vuFy2CVKaTmAGhZRMAGD
    VEDzWHLHgM3N2mXLlu256aYbIQgcdE1bQK8iBAiHJ/CXf/MFhEIh/NVn/wTtrcuRfxkDFYaB
    YJEft9xwLf72i1+G2+vD9TfciFdfeREfeeCdqKupwaqVrcLw6JjHPGCYI1oo2XmxaRYm1z7H
    xs4z51YGAJ7n93IcF9B1HdmcjGw2DUWWoaoKeI7A5/WAJ2SemmUaMUrzkSerkphhARs0v5by
    74BJCYpaaEijAIbNtDPmC/1+HcUoAqA3mUr99ysHj6z6+Ec+VHTT9de4J8Mh9xe//NXK5196
    VQDwE1Y8pTO50FwmmzWKg0FsWLfO1dl5sh7AADtkTVqhBKCsuKioubmpEZRS2G02iKJoAoas
    Zb1yACodDvsH3nnLvoqm+lp85d+/pWm6PmoBwWb62wRNy+vr6pbde8970VBdCU1O487bb8LT
    z7+k/NcjPz6bpynQFJsvF6No8IsVrF0giuwE8IHt27f//uc//3nP1q1bmdqNkVcDs1IKzxM4
    MKAbFIQ12cufGVYaYv4ecwRIplJQNRUBvzeffbJ29aYLhQPm/r9BIQg8ljc1oOH9lWT96taS
    f/yX/7n9WNcZP6X0ceY0mHVgWMTxvigXz/PQdR3PP/987+233/7U2rWrHkgk0nw2qy5CQyLn
    0ZIcDgmNjUGMjvanx8bGTGqJqZB2URo0/rqBE0op7r///goA7z1z5qzb7GBsdXxjsfgcOFoo
    mXnheoU8fUGA3W4/j+dufY+iaBgaGiMtLbWe/ftfma6qqlIqKytFQRAUlqlzL0Ln4i8G6DLP
    35KSki3T09Nb+/r6zqPcmNvM6/Vi+/bNkCQJ+/cfQSKROS/TkMnIGB6ewOrVLUgmo5iYyKK8
    vBanTp2DLCvQNCLV1dW6w+Ewj7xcp3AJbYNpE2TmH/DJZJJLJpNKKpVSW1vb9lJKiwnhIAgC
    JMnGahnyLGtF0QooWfOqpeaPhfMxL6uaP39CoZDC6DSmTQhbbEII8wXPceZT/UqyooQQ/Nmf
    /RkURcmFQqFH9+3btzWTSV771FMHRVUlpcuWNZV2dKwOhkIhua+vj5rFtrFY3ACgCIKI8fEw
    li9fVl9WVu4bGRlRLHbBRghxEMKtisWSDk3TIQgidN3geJ4316qKvKgEASB2dHTsi8Vy1w4O
    ziAYdINSZXpwcCCJ+TqDNCEkQimdrqysTPv9/pUHDx51jY6G4XD4MD4eQW1tEkVF4tjs7Hgo
    k+GNSCTSPzk5aVKfBUszYlyI5WIGku677751lNLPHTx4cMPhw4dZrY15D/kCquX5UsomVXke
    UMxTEq1ZR7vdBllWWKaOWzQruTAIY54NBsbHE4jFZKmpqXjDli3bijRN/e6JE52dlFKJBfSs
    fYs0q894sTIMlBCiUEqfO3O2p/GP/+Sz9/7+R3/Pt++aq+B1u/KvIQQGITDnJZZM4ekXXsaX
    //W/jXMDw0m2sGOYl4W0du+95I3bFuEvA0BHRUXFHe99z3sEr9cz5yBTvHFX5zfq9AwKOOw2
    tK1YDpfTAWroixaMEwJce+UO/OgnP3MceP1YOpPNZTLZnKkEoDE6UpJS6mJ1DDYLR3tJQIN1
    TnRdtwO4cffu3WUrVizPd3Q254JdHCHY/9oBnOo+C0oN9PUPYGXbivlbackybNm4Fh6PB+0d
    q9DX14cNq9rQVF8DQRCxaf1q7mdPPefVDcPMntgtjqiAi9N7YTFakrVAN50fIqGU0ijHc7oo
    ChgY6Mc/ffkrmAhPQFZkqKqK0pISfPITH0NDXS1gWAvBAYp8lsmw+ITn2aT5ZmlmQVaMgYOw
    yVNl0fYYFhY6v+kIm6VGhwKwr1+31v7Qg/ej2O/G8sZqzExPCq8fOdGeSqdPIl8vkzOLxLKZ
    jEYIsGPHNu57j/5gWSqVOs6+iwuAk9H4mktLS0pqqythGDqcDjskSTIBQ4oZe47d0+vWr165
    4/abrsHE5DTO9Q+l2Zlgt1ANTJ5qAkBClnNKPBoDrSyFoWuIx+NQFCUHYJBSOms5W34l7jub
    t83Nzc0f+OxnP+vZvn37QplgqxPPzlyO7TMDpnIWQUVFGQReQE/vOTTU1y6o6clvT4Jzff3g
    OA7FRUVIptJQFBnUMMBxBJIoQOA51uR7kRohaoDnOezash4uh93+p1/4yp7u3oEUgOdZxgaW
    om8dF1laU9d1puYzlj527Nj31q9f37Z164atR4+eIum0jPM7Hs8zUN1uO1pbK6CqUfn111+P
    aJo2d58taXT57RBIMmsXKisrbxofH7+it/fcr1XEfKHnUqksTp/ugSwrC+hahRr9U1MR1NVV
    SS0tLR6Px8M7nU46NTUVBeDgOM5jGEacUuokhJiR21whL3sJCoEBADfeeKObEHLTwMBA0ARK
    C6lW+UddXQ18Phc4jofP50UikVmUshQOR9DeriIcHkdpaRmmptLIZjUYBhCLZbnKyiqfIAgO
    TdOsAYrzMikX2SbIjOrIUUqpx+OJ8DyvGYYBj8eDLVs2wufzzTVsy+VUHDlyEtFo4gJri5w3
    X+f7rQSsL0mGnYWFNmHKYhOy+DULv1kBOwghRNN0X2fnKYyOhmG3e9HTM4aKiuKKmprqlX19
    fdMmdSeZTFLDMHJ2uw3j4yFkMtnipqamhpGRkWT+T1MXAEdjY2OQUrREo0liaoHoukFEUTQB
    Q84MJG3atLHRZnPe090d9mQyClpaXDQSmZ2MRKJm/VkO803aYqxuJ+VwOCCKNnCcyIrQBdrf
    PzD1i1/84iyzp9OWrAd5g2Daguuuu+7y2+323zt69Oj6Q4cOQVVVy33hFr1/lM732MjX+lDE
    4ykUFQUgSRIKaZsAB5tNQiDgQTgcAcdxcz0+TNZ+XrL3QvVO+f2Uzero7Y2Q5cuDjRs3br4t
    FArLU1NTAvNtrap6unXPCBfJqTIIIRohJE4p/cHp7jP6pz7z2dt+9vOnqq+5aje/YlkT3K68
    M5xIxHG2pw9Pv/AyXn7tdSMWT2axsClRAkCSqTqcl2G4lF09CzR1PQDuueH6fS3btm2dixSe
    p3YEOtdjgpr0BGp5DgWqUaAoLSnGRx9+H8pKi+cd7wI6l2FQlBUHccv1V3NHT3R5FUWlNpvN
    5vN6nLORqKrreoJFl80mUE4sbFZDl2pO2FVXWlp65fX79hGbZIOuqXPOk6oqeQk8QUAykURR
    UREMI4/58g5PnoKT1yrLf97g8Cicbg/q6+vx8gvP4uH3fAg8RwBqYHXbcgSDRZ6p6RmvJYvi
    sETLLqpSSkHEAZivZzD/boIQjnIcD7vdjrbWVqzu6IDH42aFyk6UBINzNTuWI9v8bx5DWahc
    c13F57+KCRiiyBe3mgoYZmYh/euAhUXGphtGvicENQxQXYeeV74SAfgZPWdObSOTycrpdNqz
    edNGNDc1Vp7oPFmHed13s75mRX1tjcvr8YAaBjxuF+w2ySoFZ2YYGgN+7x3vuf1GR2V5GV4/
    2onJ6dkY+ywz4qSahVssohUYGBwSPv93f4973nMH5Gwa//k/j+jhyWlTErZQUce4ANWRLrL3
    nRzH3XLzzTfX7tixYyHQtQIFmu/arRsGhoeGcfDQIZw71wdZkVltj47w5CReemU/rty9HTzH
    WT4HkFUFrx06goGhEfy/f/oaa8aUr2XhOIJgwE9Xr1xBrtqxFWWlwQKpamMBcFjbsQL3v/tm
    519+8V+3pdKZEBZK4iooqPu5GGcqmzsDgPbKK68MUEr/ec2aNfB6PRt6eoakqakYFMWYM5w8
    z8PhsKO0NID6+mKaSk2rL7/8UnRmZsZUQosCiDFJVWufjbc0kAQAH/nIR5arqvrAyZOnXAuz
    CwujgGYUsbAOofA5a2MuAJBllf3tCxVJE6iqjvHxadLUVOFIJOKKx+N1K4rm3rv36kBX10n9
    tddem7HYBgfm6xiWJNNkLQRdtmxZczqd3tnX1w/DoOB5fo5mllcG4kEpYLNJyGQyyCcaFyv6
    zT98PjeSyQQymQzKy+tx7twYCBHAcQYikTSam4s8Ho/HE41GnWaAghBiCqbIF2ONvIFN4ABQ
    j8cT53nOMBuyzs5GMTsbg6pqrEtzXshisczL4uDAFMxY+FqOI9Yg0gwhxKqU9xvZhMIhC4Kg
    EUJo/h6Kc/UVhBCiKKqL2YUEgJSiKDld19MOhwOZTA6h0KStpqZmhc1mG5ZlmTC74AoGgz5V
    1RoSifRcUbiq6kSSJHOdpgBwoig6Vq1afW00mlk5Ph6DzSbC47Fpp0+PRiildgsAlpldytXX
    1zs4jnja2pZD0wgSCRnV1RVwOBDv6ekxKdwKLI3dFrELF6Q+V1VVrYpGo3tPnDhB5sECFnXc
    AQKbzY6KinKUl5ciryhKYBgUwaAfwWARfD4vIpH4AvnkPC3Rg6IiH0RRREmJ39IlnKOqqiGT
    UTExkSCRSBrz4gELgwp58QiKoaEE6egItqxatWrLs88+m7XQ6szaH9NG4KIBBrZpdDPaAmAk
    Eo12PvaTJ71PPft8oLS4GEVFfoACkWgUUzOzyGSyVsNt3iwNeeUB6yFGMF8AfUkUPhZpZMUB
    2N26YsUd9993L+9xueaoCBfqpfDLJVQZNYsA4YkJfP0//gcPP3AXaqor57T2595nzL9297bN
    WNbUIJ4+05t0OhzCw++7x9XZdbr1iZ8/o1BKRy3OtCk9uEB/eimyDGxONq1sb1+2bt06puqT
    /8zDhw/jW9/+DjRNQ1lpKXrPnYPL5YKiKHjplVeRiCcwOTWFXE7G+jUrsWv7ViiKjMee+DmW
    r2jFmbNnsaKpFk31NdBVFYTjUVtVgaqKUmlqesZvHjQsWmZVS7mU0UbDEqE1AKQ5jhiEI6iq
    rMQHHnoQZnG3SS3SNTUvqUop+6J0kfOIWgxS4Q9zgMGMoESwsD18AguLl34dw0CZOlHX8eOd
    p//5q1/feP21V2J6cgL/+e3v6+l0ZpLtb5Pm4ASgpVKpbDKZREd7G3bu2O480XlyBQMyOgPa
    AQDNjQ11kt0mwTAMuN0uuFxOnn2WGQ10ANi584pNy67cvhm6puF0Tx/SmWwc8/rVc4WNjA62
    2uW0/zHPcZU//flT4Rdf3p+j1JDSmewskzOVLZEnU3Y4u0hWilqNqskV13U9WFxcvGnnzp2w
    2WwL63MstCKAIpfL4gc/fAxf/Zev01Ndp6jH6yW33nor8Xrc0FQN7W0rcLyzCxOTU6iqKJ/L
    SIAA09MzOHn6DLbv2I0tWzZDkkSIPIfXXnsN//0/36ayomjff0wiz+3cyn/mDx4mjXXVrNaJ
    ngdgCIBdW9ehfXlj2aFjp9qZMyFbpJetEfolr3EqyFZpAOT9+/f3TE5O/tfGjRsT27at3Z7J
    KO5YLIlcLk9Rcjod8PvdEARKu7u79CNHjmaj0ehc1/ICOtJ53csXkwS8FIGkffv2eXw+3wNn
    z/as7evrv0BDuvOjfnl+8xs/BxAEAn6sW7cSr756BKlUZoGKUuFrZ2ZiaGioFKanp9J2u10e
    G5u2B4NFgcbGxuZDhw516brusmSgz5OivlDzqjc7p+x1nMPhuKK/f6Buamp6gcJPbW012ttb
    TU42SkqKIctp8DyP+voq+HxeOJ0OcByHqakoJiaikCQRDQ3lmJwMoaysAvG4hmw2T10xDAOp
    lAKel+zBYNBnAgYLKLqUtsFc61kAus/nS7GzFOl0FidOdJ3Xi8EET4WAYHFwQBawYxhpgzKJ
    64yFqmq1CwksbNj3a+0Rs1FdT09P/Iorrnixo6N9DaWiQ1UJGhtrQIga6+/vTzOb4AXg0jRN
    1zQtabfbAQDDw6NoaqprqK2trTx37pxu2hCv11umKFpZvn4hvwdkWYXHIzkEQbBrmiYAEFau
    XLnM5fJde/JkSJRlHcXFTnCckR0fD2WZn6Mw31GmlOZaWlqc27dvfwgQb3Q6hUxbW31YVXV7
    Op2S9+9/eWRoaCiKhapI1r4GhRM0F7S2zB0niuL6cDhcFolEChIR1oL2/O8lJcXYuHEdDQYD
    NBDwk6mpaRIOhyEIEtLpDPx+PyorSxGNJs/LDpSVFSGTSeLs2S7kcjIMA3A6PVi7dhUikaiR
    y8X19vY6YXQ0xY2MRM8DC9YmkLmcgZkZRayurmlzu90DqVRKYWyBrGX86pJlGAp1p5nzWARg
    IyHkarfbvam0tKShvq42UFNdLVVXViAYLMpLCBoGMpkMZiMRhEJhhMIT/OjYuHtialpKptI2
    lvqKU0rDzBk0N7/G7sKcBNQlMhCmOkqdy+X88N13v7di3do1+QJWxjUH6AWUj94YSNAFPP9J
    vHrwdbznjpvnHU1Kz8s0UEpRXVGGPds2c6fP9PLRWCwn57LixjUryZNPPefVdd0DwEMp9Vqq
    4GULcv6NARd7r50QsvOKbducxcXBuXkANfDc8y/guRdfwrYrtmEsPAmvvwh33nknYrEYvv/9
    7+HpF1+F3W7HmTNncPjoMWxavwYnu07j3OAI9l59Lb72ta9h746N0FQlX/xFDbhddjTW1QjH
    T3Z7FskwmM6mdrHB5CKqSSbgzZkSqpqmIR6LIZNJI51KIZvLQlXyGReeEFRWlMHn9SzEAQuw
    w4X+xwKDlLJk48zsXAa/eaEfZf0DBuOJxJe+8rV/ffDRHz7Wls3limPxhG5Ge9jcm5xoLZlM
    ZpKJBCSbiOuvuwaP/vBHjeHwxDl28HgAlNkkqa62pgo8z8MwNNglCcGiAM8+x1S/aSwvLdl5
    5203CH6fF4lkEgNDozqlNMVeY0okmmChwe1yfGZNW8vGvqHRaCKV+WEylUogr3ktsrnIMgM6
    DWCSEGLSk9KLUAvmwkSGYUAQBAAoCgQC5Y2NjQUUMStwyL/+hz96DH/8mT/Rp6amZQD8e264
    wfa3n/8b2EUeiXgE//rv/4HDh4+iv38Q1ZXlc+cFRwgmJ6dwrm+AurwBbN68iaxbsxqGmsO2
    TWtxrrcHL77ymiorSvap5192BPwex2f/8MPE5bAvCmAMaqDI58Gq1mXioWOnapBvhpmyrJeE
    dc9cpL1CAHCbN2/21NXVrXS5XFsEQVjHcXy9pmmE5/PUI4dDnOPqc5wBTdNRVlbB7dix0x6P
    Rz2zs7O+cHjCHo1GOcMwOMZRFy20E4pLpPy0SCCJX7Vq1TWxWOy9x44dF2RZLihaJL8RJQkA
    3G4n7HbbBSRpF75HUVREIkmuqCgoRSIRNRgM2lRVJ2633e1yuYoSeQmzNKU0zYCjbAnULYlT
    feONN7oppbtGRkak/HwIc85Kc3MjqqrKMTo6CpdLQiIxi5Mnu+Dz+dDW1gaHwwdVVVFWVoLi
    4gCmp0+gvDwImw1Ip9NYubIDo6NJ8LwIw+BAiA5VpdA0IhYVBTwFGRTTNlzUOoYL2QS/358z
    I+Ycx8HhcECSJEhSvjeDIIgMLOTrNGRZOw8YzK83LAIq5taiBkt3aytLA/P9BIzfdIwAjEwm
    k92/f/8PtmzZ4tuyZdXVgiBUOZ0O5+joSDqZTCpsvp0MMMiKoqScTofBcTw3PR1FJBL3tra2
    Lu/r64tRSjMAfG63uymblV2yrM716MhmFQSDHqckSQ5N00RJklwdHR3bZmczVdPTSXAcB7fb
    AUXJpmdnZyjme3ColFLZ5/Nx11137V26Lr1fVQ1XKjX14s9+9tMXdF0vSSSSzmw2a/qSMUaF
    n2SUXlN2ttAInyfN29bWJlJKa6PRqHB+doFbAPR8Ph927NhKU6mEcuTIYX39+vX2l1/eT8Lh
    adhsLlRWVqOysgKlpUH09AzPdYIH8g0Qi4q81Ol0QFFUcupUNwxDgMPhgyS5SVmZh77yyivp
    uroRcsUVOz2ZjMbPzKQtWYqFAQmAIB5XEQy6i30+b3UqlYoxKnuCZW9T1vSWsFRggTnSdgBb
    BUF4YNmyZVfu2L6tbNsVW/jW5ctRUV4Gr8cNnslJ5h1sHYbBJCRVhaRSSUxPz5Cx8ZD9RNfp
    sv2vvV58tPNUXSKZKqOUPod8swnKjNtiElAXq5jXOkY3gIeu3LNn993vfS8Enp9TQFqQBXgT
    GYXFm9kRnOvvz6dmBX5Bh19qjWCy5wSew64rNpJHfvRT20wkmj16vNN4183XcqXFgeLw5Mxy
    03lkRqGw+v3Xjiia82LkCXNlgUBg1ZbNm1lHT3XO0b366qtw6nQ3hoeHIEkS6usbUFpaCkEQ
    4PX6EI1GEY/H4LTbsO/qPaCU4kdP/BytbSvR03MWy5vq0D80hsHhEaxY1gjDMGCTBFSWlZBF
    wILVKFySBk4FqWjKcZym67pBCAHH8zjReRJ//w9fRCwWg6IoMHQdvMADNF+c+uB9d+P66/YW
    phEWZhisTe4WcpJM465aUofWBl34DcDCXDMi5pAfysny9PDo2AaHw/Hg6tWrG3t6epYlEgmT
    AmVjxsHIyXJ2ZnYWhqZj3drV2L1ju++73//BcnYg+wBIHo+7orKifK7zuySJqCov41hUypQG
    7rhq59byjWs7AEoRjcUxHp40x2v24VBZvVRAFIQP7t2+abth6HRqNnYYwDkGUKLsrDMVM2Yw
    X/wXYU5zmmUyrYZBWQRwuSRJklwu19xzC/vI5EH/2NgYvv6v36BTU9NZAFmXy+W7Zu+VcNlF
    pBJRfOM//ov+45e/pqdSaRKemOCJhdpEASSSKWQyWfrzn/3MmJyY4P/6L/6M7NiyESVFXly3
    dxd57eBhoqhqyjBo+tmXDpTduu8q27aNaxb0s7F+L57jUBoMgOOI2zBoERt7YdOzi3FuEgDk
    5ptv9re0tOwWBOGOVCq9ZXY2UjE7O2mbnY2SRCINWVah65RxenkIgghJssHhcBCPx0V8Po9U
    WVkSbGmx+QCtdGpqInDq1CltcHAgbhg0w9YBLPdZXUo+/psBQx/5yEfaKaWfOH68szwcnlgE
    LHC/QT1D3lkoLy9hyjH0DbIX3Nz8T0/H0NpaIw0NDWTKyz10eHialJeXVF177XVXvPTSi/LU
    1FSKrQPTofyVNPnfKKtEKcXHPvaxqmw2uzIUCi1woCgF+voG4ff7UVQUhKZpmJ2dQTqdhiiK
    yGazsNvtcDicUFWKkZFJCIKAhoYKhMNjKCsrRzyeg89nh9vtQColg1IehmFAVSlxuz3OArtg
    KwQMl4i6TAFoDofDMBuUBoNF2L59K1wuF3ieZ1x2OueXnT07iL6+UWvm4LyItZUHX5AJMiw2
    QbE8lsQmWOmFlFK1s7NzYmBg4D9ra2sObdiw8VZN89zkcDjKr7jiirrnn39+Rtd1kd0HWyqV
    StntdiqKIlRVxblzQ9ymTauaKysrR8bHx7MA/Ha7vT6Tydl03ZhzcrNZBTab3W6z2RyZTEZc
    tmxZo8cTWHfyZJjTtLxCkNstIZmM5jKZrNlNWQeg8Dyv33zzzdfwvOfhmRnVGwjQ8LFjR/eH
    QuGUhaJmsPU/w2hcs5jvhJ1YBDBkrcEVSil27dolUkpdVrBgLVi2ykSvWNECSnX1hRdeSLa1
    tYqpVMYRjSZhs3lQWVmFzZtXGbou606nXeB5geT7dsyfAYLAY2RkRF++fDkJBIq5Y8fOEEDC
    xEQcTU21XGlpiXL69Ol0dXW1UVPTGIhGc4TShf1NrDQpVaUgRJBEUQwwirGP2WE7Cuo4hCVy
    lgil1Afgvprqmo/eeee7G+5897u4lpZm2G0SkwnUYOgaKOtFYJiSkkY+ZS4IPPxeL/xeD5ob
    arF983py523XC6+8drjq248+ftOho52Niqo+xjTF7cz5sDZjUZcaNCyiiiQAuLV1xYoH/+Dj
    H5Mqysug69pCAICFDdiIOU8XyChYAQRBXjLzF8+8AJvNBofDDmOB3OL5mQqDGlixrAErljUK
    +w8dRVd3j37btTu5O2/a6/3GIz/ZlUhldDZXTotzIC5FFMkyx9VVVZW1TU1NrEibznVvXrt6
    Nf7x/30Bo2OjCIXC+Ob/fBvf+ta3MDs7i+mJcVx95S54PG40N9ShpbkRhw4fxcDIOK68qh3P
    PfNz/P4HH8B/fftRHDraiWUNtQBzxH1eNwiBjdI5kGBjVf7iUjs/v4qRYGuGA/IhweqqStz1
    njvhcNjR2dmJ2dlZ3H7bzeAIB1ADJcVFc7UrlDmNBaBgIeq5YCX04uDo1wQK1rFQS41AmFJ6
    JJvNdpSVldW1traWfPe7312lqmoY+SJlGwBZVdX05OQkdF2Hx+XCne+6nXv+xZebJqemxtme
    LXK5nP7y0pI5MGyTRNTWVBFCiIsFHqory0vbbrvxas7ltIMaBuLxBKZnIibY5U3jSCmVANy2
    c8vavTs2reb+8d8fmTQM47AFNJr87JjFKMQs0TczuopFUs+FgEHXNE1XFKUAzFmzhAS9vefQ
    feaMbn62z+f1VZYVQ5fTOPDaAXztX/9DjUZjUZ7nPalU2mlYRBIIALtNAs9zFEDm2LFj3F/8
    5V87v/qPf8s11VVgeVMd3G4nItF4FoA8MxtxnurutV2xYdVCueoCFSVFVYG8Mt4cVYPxu5e0
    6VkBWOA//OEPt3m93o9GItFbenvPFQ8MDJFYLAFV1QHM68ybHGiO46GqBnI5DalUDrOzSfD8
    DBFFCS6XUywp8ZdWV1cH9+2raxoeHnh6//5XX4hGozzb/wkLnUC5WBnoAhtD7rnnngqv1/tH
    p0+f2Xz6dPeidQi/LlAwswvFxUWoqalCPJ5kkovcmyqS1nUiAICiZJSqqlK7LFOpublljSDw
    6ccee2w0l8uZGRrz7OSWAjQAgMPhqJuZma00i52tPSfGxycQj78Cr9cLr9eLjo7l6OjoQDBY
    DJvNjfHxKSiKjnh8GIlEGrW1FZAkA9lsBlVVjTh5chgrV9YiGPQgk8kH803A4HA4bBzH2QzD
    sBY+m+O76MGkwnXGcRxjXxBkMhl0d/fAMCiKiooQCPhx9mw/iyTzczUq52cXyAVpYos8Txcz
    H0u4ByghRKeUKslkMn76dHevzWZ/cfPmzZsGBwer6urq2rds2TL12muvTbLz2ZZIJDJlZZWq
    zWbjdV1m978psHbt2hXhcDhKKc1IklgVjeb4PCDm55SxOE6QvF6vK5FIuNvb25dFIrmiSCQ9
    1wTV4RDp5GRM1nXd9Gk1ANqePbtXlZZWfWB0VC7xegU9Gp04cupU1yTm5btz7LyIIN/MLmLJ
    uppB6ULAIBfSH9va2iilVDNVjvLPc+dF9kVRQkVFOR0Y6JPT6XQmEAh40mmZ8LwLDocHa9a0
    0pmZcHxsbMxYu3ZjQBB4Yi18pix7PTY2po6Pjyt79+51rFrFSz09IWiaAFUFKSkp1fv6+hOD
    g4N6Tc0yj90uibmcXqBCN09x4nkOlBpE03S7hSlQKMcPLIWRYAenG8CH161b93+/9E9favqL
    v/gLbu3aNbDb7aygZ75hGRbo5lijqPmFrOs6dMMACFAcLMKtN+zFF//609L7733XmmDA9wEA
    V1NKqymlJYwL7VkMCS11BIkptmwtLSn51B98/KOlW7dsgqFrc/KXczKYxnydQS6Xw9DICGRZ
    LpDHZKDCMObqEUDzHa4f/+lTeOnVg2isr4HP454DWAv/zvx7DMOA3+vGpnUdHAB+aiaS/tw/
    /Yfy8qHjGvKFXtXsoCSUUp4ppCyZc8BoGjVlZeXu0tLiBffZ/DcYLMKa1atw475rceWenRgd
    GYGqKNi4bg3uu/tO3HbT9ehob4Wqqvjxk0+htW0lzp49g/Wr2rBp3WqsW7MSh4+fQjKVBNU1
    UF2Dy2mHkP/jVmMnWozCW3KZvF3T1y4rK8P1+67F3iv3wOFwYHp6Bi3NzWhvW4G21uUIFgWY
    ln7+QDHBgimzaqomoUBx6hJfc2luFn159dSpU1P7rruO3HzzTQ2EkE0sImEDYFcUJRkOT8wV
    s2/dvBE333CdhxDSBqAUQLXP67X7fd75/iOEoKGuBm6X08n2dMvubZv9a1bmu4WDUkRicWRy
    OR3zdUymU7Nn5Yqmmz5w163SybP9dCw81cUiRBLmm9pNEkLGCSFjhJAJzDezM7OUxgV4yIXh
    vUQqlUpOTEwsyDBYnXOAIpVKIpvNznW/JgA1NBmZZBzPPv8CHQ9PRADMEkJ0SRLmsq5mEMHj
    ccHrcZugJXbsRGf2F08/A03OQeQAURAMZujihkHTM5EoVFUDsEgBNgBVUTEamqBGPhUqWPbK
    kjc9m58TcJ/4xCe2OJ3OL586dfqBxx9/vOSVV/aTsbEQstkca1wFKzidF0oDCjTFCTTNQDKZ
    w9DQNI4cGeL7++O1NTXL3nvzzTffW1tb2w6gklJayqJkZoRsyRu6FTbv3L59u6OmpuahsbHx
    dxw69DpnKhiZD9NBnpdQJXC7XWhrWw63231eF9+FUqv5o9rr9WDVqla43U7MzsaQj8ByFwAN
    3ILuz4lEhng8HuHAgdfi8Xg4YhjJ2VBobDQcDs/ouk6tWZKlmiv2vQjP83WpVMqZzeYWABrT
    aclkspicnEFf3xDGxyfh8/lACEEoNI3u7kEMDIwjGk1AFEXU15cjFBpHeXkFpqdTmJlJIhJJ
    oaTECUmSwPMieF6ArgOiKIkcxxXaBQFvXTdwDqAcAGSzOfT3D2FwcASyrMDr9SAWS2BmJobZ
    2RiyWRlv5NOf3xX7ktu3uWwGIUQxuzr39PSc0TTtOM/ztLPzpKujo2PzunXratjcO2KxqC5J
    YtZms81R5s6c6ecqK6sbly9fXi9JUhnHCSWZjLwAWMqyBlU1hGAw6Kmuri4vKiptGBuLEl3P
    O+WCwMNmE2gymTB7NFAAemvrioqVK1fdOzGh1WWzOpxOffrEieOnFEU1hVEyACKEkBAhZJT1
    qJiyUJFk/BIxDPPq7u7WdF2f9Xg8LGt0Pog3KUWiKCKTyagAshzHazwvwmbzoKysHB6PUz5x
    4sSEqqpZ8yy0gg5KAVnW4PG4aSgUSnV2nojV11cbPl8RBMEBSjnYbDYFQDKVSsY5jubtywW6
    xAMEdrsAVc3p2WyGw8JmmOftF2GJFtA+SZJ+v6GhwT80NIiv/cvXLE5uATWHORFW55cwL4kW
    RtOZkhAhgMvpIFUVZdWz0fh7mBF+0VK0Jyw1DcVarMd+LrPb7b//wP33tr3rnbezzMGFaxMI
    AWZmZvCXf/O3uOn6a3Hz9deyCOJCalHeIQQUVcXPnn6Ofulr34CiKNixdSNxzWUYChSVDDrX
    EIpSCp7nsGblcvh9HikWT0ZDk9PTocnpHFv8ZzCvGEBRWC31G166rnMAKsrKSiWbZFtQa0Gt
    Hb0NA5QDioNBJFNJ1Nc3YCYSgaaq+bQrgGMnOjE4Mo6r9rbjzKmTeOjOj8Amidi4dhWefu4l
    DI+MoX15MwwdkAQeAs/zqqoVLnD+UlKSLrR8OMIRAjLXtVtDvjeHx+vB9MwMeI5AU7V8V2eW
    fXM67PD7PIsA6YVn1SKDuphWg1oAg7luTg4PDx/rOtV1w598+tP8zMzM6pdeejkB4GkAKUpp
    OhQOG5qucRwBXE4nHrz/HnLk2PGa451dbQByRQG/ZLNJlt4TBpY11qM4WGRLptJlJcVFZTde
    u4dzOR1McQtIJJLQVI0WHGLtVeUlN3/0gXe6HXYJLx04FgfQy5xFMEAQY5xUs2FREvPa/ed1
    tCyMwFnOAAIgOTMz03/69OnWXbt2FkTx6Vx2LRAIwOv1kNnZiAYgJ8uylohFbInYLE51n9GZ
    tGvG6XCQYFHR/HsZWvS4XCgpDpKhkTEdQEpVNe7Y8RPO5K1Xk1w2DSWPDsz+GrrNJjH51oVU
    JLDM5WwsjrN9Q8YFMilLeq1cuRKnTp3CH/7hH7bbbLYvnDlzZtuxY8egqhqKiooszYvymQWz
    s23+Z2szK+6CDa4I4TA7mwZAnMuWlV21b98+/zPPPPPtgYGBHgtVb9H7u9SO05YtWzamUun7
    Dh06bI/F4m8gd7qwa3NtbTU4jseZM+cWaVI2DyACAT86Olqo3+9GIpEiodAUc8p/ecaCUiAW
    S6GuLihOT09nf/KTn4RtNlvUMIxxWZaHKKW5ArrKUtkFsKxjZTqd5g3DWFRacl57nkM2KyMY
    dCOTycDhsLN7np/LiopiiKKBbDaLmpom9PYOgVKK6ekkysuL4XLZkUrJcxFUQeDNJpEizu8I
    fukNAiGEJQ9h1nAQwkNRVGSzOTgcDthsdK57s7nedR3n9WcwqUqFNKWCW3cpkITZvDQDgKTT
    aS4UCr1QU1Oz7eTJUwG73VGyadOmvbIsJ0+dOtU3MzNLBYHP5Quf0yCERzg8jbGxKdemTZtW
    ZrNZjRDin2/al18vmqYjlVJIZWWlr7y83JlK6a5IJLOA189xQDabNYPgtLg46Nm+feeuZFJs
    Gx9Po6HBRaPRcO/Zsz0pFj031aTM3j2mDHm2gNb7ZmsAaTab7S8uLk7a7Q5POp0pAAv5dWwY
    FLIsw+/3UwCyoiiy1+ukDoefFBcXI5dLJyORSKS+vt6pKBrJZ1rmPyNf06ESj8cLjuNyw8Mj
    6fXrN3uKi4OOeFyHKIpU1/UUgIQkSTzAzcmsWjMLZsyN4zh4PCLi8ZCSTKZ+6RmwFCpJXN7f
    VX/x5JNPSj/96U/tTGfdAcDB6AUSANEwDG91dVVg966dRODzMoK5XA6vvHqAjofCBuNLmw+V
    WLo8E0JUSqnKHBfzIOBZwyXuYkQOClSR9m7csP7Khx58AG63C6qiLCp3ajr3hkFRWhzEtXv3
    5Dt8qgo4QiDwPDPqACiBqmoYD4Xxw8d/Rv/jW4/Q8dAEtm5aR67auW2O3nTB2gcGIAwdaK6v
    QWVZKR+LJwlzJKYx36glW0jbWsJIGwfA7/f5hXzzmIXZBWukk1Ag4Pchl8shGAyir+c0FEWB
    3SYhnU7jxz99Cita29HT04N1q1rR0twAXddRX1sFn9eL7p5+tDbXg3AceJ7Act/5gsdbFUWy
    ZCMJ4TgOPT29+OZ//w8ikQgi0SiSiQT+z6f+BJqmQtfnVZMMw8DWTevxwQfvZYIAC4MZ1Ioc
    6KWxBpYvYBoGE8lwlNLnfvGLp7bff9+9vs/99V9JH/+DT1xx5MhRBcBBAOrk1JSRSiU5r8cN
    3TDQumIZPvGx3xP/+M/+sm08NJHweb28KAjzkrGGgYqyErS2NAmDw6OVm9atsq1b1W5R3KJQ
    FBUGo7mwe1zpdbu2v+/Om0t2bl6Lb3z3cYyEJs0O0KaMcA7zfSpMHfKcxRi8qe7gFl5yKpvN
    Hn91//7r7r7rvYLb7VpQv2A2aVzW1Ij21lb+5f2vSgBi6UxGnZ2ZRiJWhGg0arA9ypWUBMX6
    2uq5rtDmfvF5PaiuLCeHj3WCfV9uLBTWE/GYEI1Goaqq2bSPczrsrvrqSnAcgabND8k8M0CA
    rjPn0D80bgI/reCxpHKk69atw9DQELHZbNsIIbVlZWUT1157DaPpUZ7S/J5l/xJBEASeF4S8
    vOK8UbPZbFRVFWiaTgkhlgdnmD8DxAA06vP5lrW3t69k3VlFS0dxcpEcQFBKsXv3bofNZnvH
    mTM9daFQGIIgzBUpvhHlKJvN4vXXj0GSbCgtLYam6XNZFzNq6na7UVVVTqury6gsZ4xsNsdN
    TkZJNJoo6IZN3pDSlE7nIAgSX1pawsdiMTmTyaQttK3z6FtLFlLPV/H6c7kcP+/wcudRI8x/
    ZVmBKIrI5bKsRwEHSglsNgn19WUIhUZRXl6BqakUUql8xiKRyEHXNXi9DmQyLBAJHoRwnOWc
    4AoCSZc8mMQaNxHDoPD7fVi9ehXcbhccDgccDjv27NnK5mXh2pmZiePkyX62rxcWOS9OU7qk
    GYc5dUBTAer06dOvNzc3d9XW1u48evQEnE5n7fbt229UFOWnk5OTaV3X5XxWLcJU5wycPt2P
    bdvWlu3cuXONzWb3Koq2gM5jGMDsbJK0tzcEDAM4c2aG0/V8oNTS7Ixqms4D4Ox2u2337j3b
    BcG/rqcnzYkiD7ebpl999fSQoigOWMRCWMZ8igGHFM6XFqZvkppIx8fHz9bU1IyXlZWvGBgY
    xPmN1PLjDYenSF1dne348eNIp9O52loHdTr9xOPxIpeLZAxDz5SXl4uJRJoU1ioBBKlUDnV1
    PuJwOLRsNpvJ5bI5r9fryO8fgWazmSSAdFVVVbmmEVFR9AuKI0gSD6+Xo8ePj2QVRVEXsQsL
    gi3CEjkUT1NKD+VyOT/y2rsljHpg0oZMycX2tWtW+//2c39F7JLI+jAk8NE/+CRGRscUC28s
    WfCvafhTLF00eTGi5Ys5CRzHUcMwvJIk7dt33bXedCqFl19+BWtXr4KN1WdcqBmbIPC46fpr
    ceDQYfz5X34BbrcLq9pbESzyg1JgNhLByVPd9MX9B+jJU92KLCukprpS+shD95LqyrIFvR3e
    qMcDBUWR34vmxlquu7dfYDdbwbyEpGkgsgyUGUs4R3nKtU2ao7xTen4jtnyGhMDv88IwDPh8
    fsTiSSQSSThLi3HkeCcGhsdxzXWr8cxTT+L+//MhSCIPXdPg87jQ3FiH0739uPHq7bBJUn5u
    mOKl5XFR18OvZCuZheA4Dn6/D0UBP7weD9z5fgMghODgoddRWVGO5cuaQKmBkmARJEmcB13W
    M+tNHV9Lu/YthsjAQvUZDsDBM2fOdD/33HNbH3rf+/D5v/4rxyc/9endJzo7HQDikxOTSjQa
    E3wezxzQ2XfNXkQiUddf/+0XXW6XE/mgwXxE3O124orN67n9B484rtm9HT6vO59doBQgFKLA
    g3VtFQEU2SSx5d23XF31zhuvRDyexEsHjyuGYYyyAAXYgZdhvFmroQUAAF35SURBVPYYLtCw
    6Ffk9qoADh44eHDsxIkT9Tt2bF9QJ2BmS0pLS3Dnu+7gjp04EUil0jE5J6uTE2FkGyqgqaop
    G+1ftbJNrK6qWNDJG9SAw27DqvZW8sQvnpM0TTMAKJlMhuYyKUxOTUNVNVNWtKq5oTawduVy
    i/CCZd+BIpXO4qmXDiGZzpiypPIcVYqQRQsif5Ors7MTuVwOk5OTPwfQl0qlvMlk0pdOpwLZ
    bC4oy3JA0zS/rutOXdcdy5cvrysuLm0+ePAoMXncdrsd69evxOBgr9Lbey7L83yW40iWEC7L
    cSRDCJfhOJLjOC7HcVyG54XYzMxMn3kWUEo5Mh9KW/LL7XYjmUxi1apVFblcbms4PEHWrVsD
    nudx4kQXVFX7paBBEES0ti6jRUU+wzAoyeVkksspIITA4bBTu13S0+mkPDo6qJeVVTgSiSx/
    7tzIHNf9lwEF82dF0aCqlCspKRV6e8/plvufxUJp4SXtli1JEqGUOli2oaDhGDmvu3c+sswj
    m82irEyCKIpQFJ1lF+hcduHcuWFQSsDzHGRZRzKZhd/vxsxMXn3OpHFZqItvuW1gNZBz2gaK
    oiAW0zA9PQtV1aFp+SBiRUUpZFlFJJIAIXk6jkk/W+zrk7fW0lELaDAAYGhoaCyRSDzX3Ny8
    +cyZXtuBA0eIIIhNu3fvvvmVV155PZPJyF6vx9KBmEM6ncPRo2f5rVtXV7hcblbbtHDPRCJp
    8LxNVBQNs7PpxbphE44jAiHEvm3bFe3l5Q0r+/oyYjZroK7OgUwmNtnb22tmF8wMs7XZZxpv
    0PzxjWyDCRoOHTo0Vl9ff7ClZdny0dFxkn/P+bVIg4OjaG6ud6xbt84fi0WzkiQZbrefkyQH
    FEXXi4tLpKKioOf48QHki5UXFinH4xnYbOViRUU5PzQ0rOq6pklSXmGO46CnUqmMx+MRm5qW
    VUQiMp8vDOcWZBbmZZpt0LS4NjAwYAriWCWrF/Tn+Y0BQ4FDoTBpQ9NBTZjZB3ZTHABaamuq
    OZfTCZ4DDJ2Dz+tBZUUFYZ9hNm0zteRjmOcZq5abnLYMxmBqLhcHQufzOSWVFRXtXq8Hf/Tp
    P0FtTQ2amxphswXy/RCA82RPCfJa6v/z3e/jP//7O/rw6FiO4zjd6XRwbpeLIwBNpTNqOpOR
    DcPQALjqa6s9f/wHv0d2b98y50C8OWnWvIOxrL6GEAKR0rnuwyZlq1BvfilT9MScJ1NSct6B
    Or/xmMftgsDlI2iapiESjcLjduKxJ36OtpUd6Ok5izVtLVjR3AhDZ5QrjsOKZY147ImzSCWT
    kAJ+JJIpKKpmYAmb0C2xfSAUFM3NTfiDj38MhM5/VQIgm8ng5794Gs1NDbhq9w7oGqMnseZo
    c8dxgawqBUU+sEp+2UG+1KCBIt9HAYSQLKU0lM1mX3zyZz/fcPutt4o7d2zD333+rx2f/tM/
    33r02PHQxNQUZmZmUF9bPVe/Iwg87rzjVhLweyEKPERRmFu/ZsOyK3dsRSQSxRWb1s7V+8xn
    p7yQRJEH4Bd4vu6mq3cWP/SeW4jb6cSho13oHRgxZULNDtAqW/Npi0H4VdLMFzKQvcPDwy9/
    /9FHa9etXcPZbbbzOi0TALfefCOOHjvu/Oa3vlOlaRoJT0wAmgxJ4AmAgMftLr/h2r2c2+mc
    a2Q4J63KEezathnf/cHjrt7+QTcATRR4GKpiAgYZQMDldCy//carpZqqirlaj8LvcbzrLPa/
    fsKs5UhbHhlLdHnJggidnZ0AgH//938Ps/thBpKKWRApyJ5zIt/d1R+Nxsn4+ASjawiQJDuW
    LUshl8tpY2NjKYstMB9Jy3fPWOzGXFSsYP0u6ZVMJkEpxac+9anSeDxR5XK50NraimPHTrB9
    k69DyPcHoGbEndFlBDQ21mHFimXGzMxk7plnDiVEUVBKS8uo2+2mAIxwOClHIlEjEAh4V6xo
    LUmnVfHMmT4iy8qvXERtGEAmI5OSkhKJ53ld1/XFbMOi++M3tJ0kb5sLiy0XU/nhoCh5KU1V
    1cDz+cwCIfqC7ML0dBqpVG5uLikF4vEcqqv9yKvvcJAkAclkzjAM40Jn2SW3B2bWmxCCRCKJ
    I0eOM9DHzf3L8wLq6qogyxpGRyfnaEl5ihKZO/ILgVeBrOpFp6suYhcoyzLIlNJUb2/vaxs2
    bBiorq5q7e0dwIEDR8iWLRsbtm/f4eY4ovl8bsrNh9TAcRxisSQOHjxFqqtLkU5nz6vNyWZV
    9PZOEE2jUFXdMnccdB3QdRCPx+vYtGlj7YoVa6qGh2VbPK5BFAX4/bx+7tzARCqVMmlppvR5
    GvOFzed1uv9VbUMoFEqPj48/09BQf215eXlFODyxqJxpIpFCZ+dZfsOGlWUTE6FoPpvoByE2
    SJLNtmrVqspsVnfNzMQX9C0xx5tK5RCPy/yqVav909MzCY7jCMdJcDgIKNUUVdXI9u3bm202
    f2k4PH0BBbV8dqG0VKR9fb2Z2dnZVEHGMYNFGvwtJXXDsKR6khZu2BiAIQDjoiiKjQ0NEAV+
    ro6B5zg0NtRBFARiAQTRgvcOAhhGvsX5JPv/SXNABY3dljBMPDc9bq/XG+AIgdvtxsMPPYCi
    gB+GYSCbyeCll/djeno6DxSYc3TiZBc++Sf/l37hH74kD4+ORQCEDMMYSKXSXROTU0fCk1OH
    kqnUUcMwxgRBsG/bvMH7d3/xx/zN110FniMLCqnPBwvnP0cIUF9TAWdeyJxiYYMqc2OkFlsE
    SwAaM+l0WjcMfb75lJWaNKeaROF2uWC3S5BlGYIoIR6P4+jxTgyOhFBXX4/hgT7ccO1VFmcy
    /zm1VRXI5hTMzEahqgpCE1PQNF21OAjWKPglBxCkkFRqFkYiHwkj3LwMrab+/+29eZhcV30m
    /J671V7V1Xu3etXS2iXLkm15lxd5Y7EdbAOxA8RAAiSZEGAmHxkyTzIZGDJfhsAkk8wkhOFL
    WG3AgDFgCxu8SJa1L91qtVrqfe+ufb3r+f6451adqq6WZam65e97XDyXbleXuuueOue3vr/3
    1WGaJhSXAkPXkc3lkM3loKqqDVMCSgZWeVrVi0CSlq3LVnafFmzWpCyA/UePHp04fuIECCG4
    9Zab8Ld/82XXnXtu78zlcu6R0bGShIBaFmRZwrvvuxt777iNVWY4xi/LRFf7Kvy73/8wWpoa
    ymA6Fpob69BYH1ZEUWh/995bG/74Yx8Q6mtD0HQNB4/1IpHKOIo5jqZAninE80xq9Ao/X5MQ
    kqKU/vzZZ5+dPHDggK1Wzu11MFhSTU0If/q5T5MPPPa+GkVRgmPjk9TSc2ioDUgAuu68/ZbQ
    nltvKqFmdfa8aZpYv64bT7z/YTng97UCqKsPh0RJsDAfiVGLUr/H7drxgYfvr/utB+6CQBYP
    YBMAyVQKT/3sRboQTeRQ5GN3vjpBd1UDRS4gJKzKz8/BOCKDMwAmJEmarK9vQDqdsfGNLEii
    FMhk8iQcroUoinn2XqPM/k8yfzDGfMQs+1mqDF6z7AJdlmUqlFIxHk/g5MnTuHBhGA6rSTgc
    xu7d1yEYDBactN/vx65d12DjxrXG6dMnEs8///z06Ojo1PnzF4YOHDjQ+8ILLxx94YUXjpw/
    f2G0p6fHu23bNc0zMzH3qVPnSC6nlkCRikPVlQeei0PUNtNMMBiSZVl2fIPTXeA7+bkrOSPl
    D82G7WYVRaHlVdLSZIEUOiE2ra6NGHK5lEJ3IZvNIhSqw8TEwiLIVyajQZJspWhJkuF2S0gm
    E7ppmlaZb7jKxSVarLA63CDccK+tvmuy7pMISZIgSSIXh5R2GHhdhqXIk5azA03Lh+sYsuHY
    sWOjhmG8vnr1aqooCnK5PA4ePEqGhycagsFQcygUtNEZZV2EZDKN/v5R5HL6IjYfSoHh4QVM
    TMQW7SM7ITZw7bU76nbuvKljYkJ3z8/n2XmTIAhqbnBwMAWblINyRYZ0lQpJjl80Xn311ZOS
    JL28fftWqihKhUFj+35HRiZw+vSg0tW1ujEQ8Ip+fwCW5UFzc2vdmjVrO86fnxBsaFaluSSK
    oaE5UlvbEr7ttltb/f6QxzRluN0eSBIh11133dqurvXrz5+PS7mcUVHpmRCCujo3LCth9vb2
    pSilKc4nJFkytcgeVEvp2WKBe579t6M2GEeRoWd1bW04vGF9TylchQAb169DMBgQI9GYyT7A
    OCFkAcA0cwZJLsh14DY5rq1qLIchsCwLHo8HuVxOSyQTmfa2Vvz3v/4iasNhEFYpfuk3L+Pv
    //Gf8X9/6S/R1FCHVCqL5365D1/9+/9t9Z7pz1JKY2UdE8cwywA2NNTXrXvsoXcFPvzB95HO
    9lZb/deyKncUsLTGA7UI2loa4fd55IzNRWxw68MnDDpnRKtVaYjOzy8Yhq5Lsihy4lH8rIXD
    /hJAQ30dpqam4XK5MTU9i1dffwPbtl+Ds/392LppHTb0rIGTfNiYcKA2HLJbk/EE6muDGLgw
    6jg+B2tXjru7Oi6BtZ1ZsISZ2RkcOPA6FhYWkE6nkMlkkctmoekaJiamMD+/gL7+s+wzB1a1
    NOHxxx6CzSC0mM8NpboP5GJdn2WsKDmwhguzs7OH9/3qV9233LQbhAC7dl6Lr/33vxa+9d3v
    IVwTKukS2Pu0qIRcQonDdWAUrqBAOSHE5oY6vOee28VMJiP+ziMPoLEuDGpZiETjOHFm0BkC
    dvQZtLLqaR5VELJjjkEDcHp8fOIX//TPX39y29YtUl1tGBZPagAKy7SwqrUFf/Wf/ox0trWR
    n//i5/jn7z6LdCYrbN+yyfeHn3jSLjww4ceCfgK7Z0kU8cH3vRvJVNL39X99yktAybFT/Ri4
    MIbW5samx9/3LvLEI+8mwYDP7lCUqMbbw86/3n8Urxw87qyN04GJA0gwqNaiDkMVNQsoS7A0
    9hk4n0sagEIpFRsbG5uCwYD/zJnzBUYQp3WfSKTR1dUguN1uLZPJZADECCFzLNmYZ7/HOe9a
    GfxSXW5bQAjBxz72sYVgMDjtdrsb+/r6OWEuN7Zv3wK/3w9dNyAIIlpamrBt2yZqmpr64ou/
    io+OjkYopY4SbxxAWpIkY9u2bZ3XXrtzm6ZZzSdODAiRSALFQdlLoWAlhQ6HE5Dn8zpcLr/s
    9/vFfD5fELViPjvHBQeo4ppR0zQjXq/XFARB4hVvS6EahMFvdGiaAZ/P3s9+vw9dXc2YnBxD
    c3Mz5uczSKVysFGJxfu2Rc4og3MKUBTLWlhY4EkNDAbFNa9iQYk4R9zr9aC9vQ0+nx8ulwJZ
    VhjLk4Rw2Lb7NTWBwjnI53WcOzeBfN5YJNRWQfGZrLC/K/cNWi6XS87MzLzS2tr8YDgcrotG
    49A0HUePniapVJYEgwH2+eMiDF+kTJxQ4GpxwiLhwvn5PPH5alzT0zlEo1oBshcKyUgkplMz
    MzMmK5AbrLu2ZHfhMu0fBaBPT0/Hzp0795OenvW7u7tHu86fHy4ZWHbeM6XA4OAIUVWd7Nix
    BW1tTTAMGYricQ8OjmFycr4CWUJxnRKJLM6cmRa3bVvTGAz6SSpF4PPJ8Pu9nny+pf3s2SiJ
    xdQKAo/273G7ZTQ3S/TkydOZ+fl5B5YV5zq5TpHNXI6EweScjsoMdoGWiQVRO9Z0d9f3rFtX
    qBw6AfGa7k6sXd0tRaIxR3QnQyl1koYZZlB1rkLgHP7yIY2qG4J8Pk8AzE1MTL76jW/+29o/
    /dynxaDfj0xaxcFDh/GVr/0Dbrt5N9at6caFoWF8/Zv/Rr/z/R8aC5Go02WJwh485pUDawRB
    2Llj2+bNn/zoE669e26G2+VigYO1CBNdSBYsWoZTLjJJUYGgtiaIYMAvzs5HFWZBKKXUcrJf
    DsZVze6CBWBiamoqH4/F3Z6WpmIgSLEIIuFSZGzfsgk/+tkLEASC/QcPYXRiGg9s34Wf/vhH
    +PPPfhKKLME0jJJ/G/B5IcsSEokkxiamcW5o3OCqxvxV9SHOy1oa2ErPw8Mj+OGPfgxBIEin
    07AsC+vXrYXH7cbOa7dDFAQQu1KJ0339CNcEGFVt6SBtuf7CSleUKhQIdABxy6KvvPLa/vsn
    pyYD7W2rYBkmVnd34v/67KfZMLdZlhxYiyrqlJY+B4t7PZcgK7KMJz/4XggAZFGEaRkQAAyN
    TWFkYtqBIAnsK99dyKFs4P8KgmKTg1n+dN+vXtz9wx89s+1jT35k8aA/66o01NfhM3/8Sdy4
    exe+99QPEail+OxHP4Rrt2/lVNFLmY0cheaAz4tP/e5vk672VvL0j3+Or/9gH7Zv20refc/t
    ZMeWDXa31pl1ApcsEGBscgbffuZ5mspkM5wzcIoXMa4iry/TeXECeXDd5xI+/G3btjVQioZI
    JLZouC8aTWHDhk6poaGBZDKZHGwF0ighZJYVk+JckGtyNs6Z36oq1KpSE/onP/nJ9KOPPvrd
    66/f2RoKBRtmZubgdruxdm03bWlpxssvHyCEEGzbthmrV7ebIyPDqUOHDsUSiYSTKMwx3xAP
    BoN0z549u9rbO28dG5sJX7gwDoei9Uq0HACbaUYQRNHv97sWFhZ4cggH2LyoiHQliSPDdVv5
    fH48EAhoLpdLcthvStlaiu/XMExEoym0tq6CorjQ1tYIWbaQyWTQ2bkWx46NFjoQPExD1+3i
    gq2cLMIwYsbMzGyugm/QcfWKSYWoPhAIYP36tUygUIIkSYhGEzBNE3NzUa7zIKCuLgTTpAzW
    VtqVqUSrugTsaqX8AiWEGJRStbe3t/f+++/v6+hYdVsslmDQIYpz54YhSXIJNt+5Vx66U959
    4hOIxa8TkEho6OuLwrJIQTBRkkQEgyLt7x/NapomcMgLp7vAF0tote7917/+9an29rZnd+zY
    /vtzcxElnc5WoDW1qaLHxmYQj6exbl0XOjvb0Ns7j76+IZgmZcrfS51rAfPzKRw9Oib09DSj
    sTGIfD6Lvr6EMDOTgapaJXMLxaTFLiS0tnqQTE7rp0+fTnB+IcqKMk7CsIhWtlpDzyYXRJAK
    KXCAELLz1ltu9jfU19lVRxTZURob6nHzjdeJh48e9zMWFAOlQylR5mysctwc55iWJUBkyU7W
    NM3/8+xzvwgOnBvcu2H9umAul6PHT5wyZFlWbr35RuG5X+7D//6Xb1pvHD6WNgzDEf+IMWfg
    0HYlAbR7Pe573nXvXWv/8OMfEjesW13Qn8ASw9PgKvYlA8/cz2FZ8LpdqKsJiYM2hrtcn8Bh
    DaFVriICwOTExMTcheGhmtbW5lJIEsoHn4E9t96MX+57CYPnh7AwN4u7996Lvr4+bN24FhvX
    2wklLQmibGEvl6IgnkxiaGwKcwsxjYNTlA/qXLUOA4/KsCwLO6/dgb/76t9AkiR8+zvfxdmB
    c/izP/0sXIpcotidz+fw53/112hqbEBNKAhd10p2elHMze7KvUmHYSUqSQbDrJ4YHBy8cPjI
    0Ws62lbZ+5TNK1CLMBX00mC4JDCuBF9zKvVYnFi4FLlAzQxKYYGi9+wFxJNpFaWMTjxO/4og
    N0vce55Sej6VSv3gX77xf7pv3H19YMumjbCs0gSvuB4Sbr/5RuzYuhm5fB61NcFSBXfQxWea
    JQ1utwsPP7AXu7ZtRjqTQVd7K9wuGaZpVlhfCwQUmq7jh8+9hJP9gyrXVXAUTCNMuC5VbYji
    En7BSe4LBAXMrsobN25cOzk5U5NMphkUpRgEpNNZZLO61N3d7R4ZGXF+RwpFAb55FEWU+KH8
    8osuxwEHgPn5ef0HP/jBT/bs2WOsX7/2wZ6eNe26rguWZRFdN1rC4Rpp48YeS5KEzMGDry/0
    9/fHDcNIccnCLIBIW9sqee/ee+51uby39PYOeqam5gvdiisVgAMIg0mJotfrdVgL+auEb72a
    HaZoNDpWX18fqamp8c7OzqOySnExsBkfn0FHRzP8fi/cbhP9/afR0tKChYUskskcKkGbTBOw
    LAqXS4LPRzEzM6nG4/EcFg9xVnW4/3L3TSQSxYsvvgpKCXp6VqOzsx2HDp2CYVgFulVC7GTi
    uus2Q9NM5PM64/fHIhiSs5ZOonEVHwX7OzAwMLdnz54D7e1tN509e0HSdatwvovMP6TixTNp
    VdLuqPQ6e4+DC5IFuN0iBEEzJyYmnLi0fKatmoUksG6qqmla8vXXD/7s7rvvvn7z5o03HD58
    ogIkyIEM2uKKp04NYnBwHIZhwbLAqcSXqkWXi64lk3mcPDmJYDAKVTW5REFccu1CIRdCIdN6
    6aXDqXQ6HWdxapQrcCeWSqaqpfSMSvhASikURYGmaZu6Ojtuu+/evZAkqcB84jg6WRJxz523
    4+lnng2PT0zVolTO3RFkM7lq0kXfT5UPOGXy7uOGYfz9mf6zR8/0n90Em/kpWFMTuv6L/+0r
    gcHzF9LRWNzB0mZQxNzOcbCqLXW14UeefOKxVU8+/gipqw3bonYVqFkvNvBciTHJqd7XBP1C
    BYewLKI1XCA1PTs3d+b11w/23LT7+lIICve9jTwx0d7Wii//5Z/hmWd/jn2/OYD2zk4c+cHT
    +MJnP2F3F3j1bGaH7AFCCeNTczh44qxlWhY/vJmBzYaTq3YX5coKSmBDVzX2+5dl5PN5W+Jd
    sDmZncFUQggs02QVRRt+YIGW0asWuzZcAHs1mD8oV2mfisXih159df/2d913T4EumR/aLyYC
    VskAfKXLSSxLOgzgRB8tdrG9kcvlcfbCKDVNy6F9dZyCM7jl7AlaRTthAFAJIWlK6UunTvfe
    8C/f+OYDX/zP/4m4Xa6KCRK1LJjUgs/rgc/rLp77ch0HlK0VLerTtK9qtpMI01gkhllCXwzg
    2Kl+/OgXvzENw0xxFaQIbJVrBwKTQfVJEMo7DCgL0gp79eMf/3iXLMu3DQ+PSYZhQBTlglOz
    KWJNTE9HSFdXV7i2ttYTjUb5arjOQZDMCvtzpQJAa25uLv3000+/VFtbe8Hv969Op1O1Ho+3
    7a677rq1u7vNNzo6unDs2LGFiC0cwc/4zQOY37hxg++OO+58KJczbjhypE+OxZIVEwA74Lo8
    pWh7qxDB7XZXEjNbNtHT/v7+8bvvvvtca2tL+9xcZMkg0Xmf8XgG+/efxrp1Hair8yCbzaC7
    u4d1F0oDQp5NnVLA5xPhcqWtwcGBtGEYWc4G8HbganWfufkFC/m8xjo/liN+WiAKKU0GAEkq
    3mMpJGmxLgODJK2oTygvqLBKe3Z6eupQe3vXbG1teNXsbBQ8O09ph6m0a1ApoSxNIkiFxLMI
    wXMun09CPh835ucXHNSByuIEZz9Uc3aL8v6nr69vtKen56mNG9etm5iYrnWGj8sHoJ2hbxs2
    qBV0aEqhV6VDz+VwPtOkiMfzZYmCsKg7AxDIsoi2NjcdGTmdv3DhQpwrJC2wQpLTeV4E9aeU
    VgeSxDtgjpeWDTJpYULIh+6/797O7du2gjKIAi0bENy2eSPu33un++v/z7e7LIuOM2MaAhBk
    wbZDeWUuZ4JQoa0KDre8QCl9HcAQbNrYlng8kXnj8NEQt8g6VwmLcp2F69tXtf72pz/1ZOMj
    770fbpcC0zDKRNlosfuyhMbDxRIJRRLh93kIFqv1ScuRMHCJYso0zZd/8cvn73nskfd529ta
    C+Jyi5RnmdFrqK/D+MQUrrv+Bpzp68OWDWuwyekulHQnLFY4tWCaJl45dArnx6Z1FKnQ0tyV
    LW+jLfceWdI5MKOvqhqmp6aQTqfhUhSsam3FgdcPwjRNW4vBMAqVYrfbjVQ6g5+/8GsQQlET
    DGDjujVQFKkQerGTBcGZol7skMhKOQZWaU9TSg8fOXbssanp6ZrO9jZYfMJYITlYlAiUB9dL
    /PtKXYlYIoULo5OOoTaWcArVTCCdzekwzcxalvXDnzz7s2337r2r/d69d7J5jDIBNZTCr0q7
    LYvZjVAxsSoqxdMK6+UMOscSSXzrmefp1OxCjut0OhX5BQ6OVNEpVNMfVLIXlFIEAgGpoaHh
    gampmR1jY1PgmS/5IGFqKoLu7tbA1q1bV7388ssTsFXA/QB8rKik8nCaFT7vzoyGSilNRyKR
    mUgkogOoB6Lxn/70pzEAvng8brGik84lDDEAC9u3bw/dfvvtH4xEUtefPn1OrMQQ82bdAz6h
    KB8Wdaq6zpZyu12VfMNyaddYJ0+ejN90002/Wbu2+5bBwWFXLqeWBICLq8Z24BQIeDA2NoKW
    llZEInZ3oZQxplz8jSAcBkZHR9WRkVGelj1V5husq+cb7Pk2URQRCATgKB5Ho3G0tDQUmJJE
    0REoFAoD3u3tjSCEQNctJBJ5OARQZTMNhJAlxdvISp4Ju8tw7sKaNWv7W1qaV83PxxbNKVRm
    zaqkL7I4OVgKm88/5/NJiEYX9Gw2a1QoJOW5BLJaPtHius/pV1555cD73//YS1u3bvytaDQh
    GIZVIWkoLwIs1VERKog/khIChKULB8XnGxrcoDRuHj16NGEYRpwvXFxKIUlajqDC2aCUUgnA
    Q9u3b3v4Ix96QvB6bNVWWl4Vs+y2+4c+8Fvk8NHjzcdP9a0HEKGUxrjWucrBDczy5GQ5kwYu
    O00TQmRKqcAlBnHmvGSu7cVrRqQB3Ly6q+Pxz//JJxvedc8ddvX4YvMKZerOS8KUyp4TRQKP
    W8ESDmG5nIJTUXjt6LHjR779ne/e9tlP/5EtuoPFlVOHVvTQkeMYHpvCe3fuxo+efgqf//Tv
    waUoJd0FPjDSNB3pTBa9g2MWikwvBZpFQohDGam+DSBJ9ngfITh79iz+6otfRjKZKLBhHHzj
    EAhTehZtxeqCNbeohaPHT8I0TXSsasGffOqjaKwPczdk06ragcBV7T/zXYYzw8Ojo6dP99Z0
    trdVTAQIbFw9FQhgEVCBgFoElHDfC8SOnYlNKeD8twVUhCcRADPzEcwtRJ3zqS7RXbjiCnoF
    x2By5/zE1NT0L//1W9/53et2XiOFgsECPKi0e2BnkraPsDGs9n0LNm0KSFFdhmlLEe51gL1e
    VlmSAA62CErx6/1H+EHnpZxCeiUrrs7asXUgH/vYR7dpmv7R3t6zvkwmC1GUFjk4h91neHhW
    XL9+Y9fExPjkhQtDC5TSMKuE8YUkR0BqpYNBh6IxzeyryejFM7FYLMb5BpT5hsT27dtDe/bc
    8cTcXOyGU6fOCfm8WiFZuPzZhfLgw7IAWVZEzicsq29g3Xnt9OnTv7nuuuvfu3Fjz3UnTvRd
    JBC0r9bWBkiShUwmje7uHhw/7nQXSEVIkygKkGUBlCasEyeOpfL5PD/g7/iGqtMHX05cRCkQ
    CoVw2203wufzMXY4ipaWRnaPIiui0pJ7rK8PgRB7eL23dxzZrL6okeAkF0t1Nlawy2ABMAYG
    Bhbuuuuuo62tjXvOnh2SnIC5fKB5ccfkrV2V9pIoCvB6CYaH5zTTNHnGSGfAP1+tYkmFe9cB
    ZCORyNyZM/0/3rp126729lVdw8MTSwT8pU0he76j0lD7UhdXpyRCWTeimJR4PBIaG0V6/PiJ
    zNzcHF9IWmDzwg7T3JJrIy2HU2DJgghgT3NT07//oz/4ZN2WzZsLeFuUs+eAglom1q9bjT/5
    1EflL/yXv9k4NjGVgj3klmLVQgdWkOaDgBVyEE5A4nAp62zzOV0EL6VU5uAKeUJIllKaA3Bj
    Z/uq3/mzz3yq4V333AlC7HtdrKVQimde1F1gqs+LGZOsAkRDIIBstzdFVFY/Xg6VS2cIfTqf
    z3/7X7/1nfU3XL+rac9tN5dUhHm4RDqTwY9/9ktsu2YHzvb3Y1NPF7Zs7ClCWMooKkGAVCaD
    RCpN2WZ2EoU4dzmML1dzsK3EeJsWRXdXFz7/p5+DJApwKQpEUYAoEJiGjm999ymsX7cWt9x0
    PSzTYNh8E5ZpwjQNiIKAmqC/wHhjFfa4DWm6SNubrKBj0AFMR6LRk0eOHt9+7947C39cYImA
    CYpMNot4Io5EPIloLIZoNIpkKo1cLgdd12BZFgRCoMgSPG4Xgn4vwqEggn4vAn4PAj4PJEZP
    axNK2WsxNjGDVCbn4NvLLxVVxKguce85ZgN+8fKrr9368iv7Nzz4nvtZlwUFJhBqEeRVFfF4
    HLF4HJFIFPPzC4jF40hnMmwddFtLAfbfcdbC7/OiJuhHXTiEmqAfwYAPAa8HosTWw0QB2jY1
    t4Cnnv0VzWRzGQIkaJF0wbnKncKyV1zLCjvkySefbPV6vZ/p7T277cKFkSUrj04lbmJiAfX1
    NYFbb719Wzabi05PTycopUkWpPNBYOF+VqKYVJY0Z5yCFmNgSbJOiIcVzgCbjlijlOZ6enp8
    d9xxxwcikeRNp0+fE/J5rQS3fGWJQuUqow3tFMqVj/nB52VZn9dee220o6PjO1u2rF8zM7NQ
    Ozu7UHHYFXBUnZsxMWF3F6LRLBKJLJy3XQm2oigSCLHoiRPHc0NDQ+W+IfH28Q32e06l0njj
    jWOwVYwp7HEeAkJEbNy4GtmsivHxuQI8hb8Ae8jbUYUui7sWtRhWuLvAvw+DUppbWJg/XlfX
    GA0E/I3xeKoQgjiMaIIgwOVywe12w+12we12weVSoCgyRFEsVM4pte2+aVLoOoWuA6YJ2Nwo
    QslWJsSG3oiiYcViMb6QlGOwZV6UbFlmt1iXIfP666/3rlu3bt+GDWt/d3p6QbIFHYtDz46y
    t8ulsPt3w+u110JRFMiyQ60rsn8DthaUrYUFXbegaRYMw16PovxI0Y4KAkFTkxvJ5LTe13dm
    USEJRaa2THkhibejUjWdApcsyABuD4fD//UP/uCTG9/38EMgAmAZtBSOVFIls6+799yCdDrt
    /fJX/9d1YxNTTlVGo5QapNirXkSbt4wOglczLOk2MGPkBeAmhPAJg04p1QBsb6ivffLTn3yy
    5YG9e+ygr8AtX6mjQC8++EwrDD5bfFIBMKCKsMS1LE6BUuq05Q8Oj4z89Gt/9w8fWd3VIXe0
    r4IFCsu0hzcdcbLXDx7GyMQ0HrnxVnznW/+GP/job0PTNGiaCkkU7HsodCMIRIEgFk9gIRKz
    uFaz4xT4LlRVecSrkTUEggHs2L6NwUlMBjmzkM/mMDo2gYb6OjTU18EyDFDKfu5QiloWTNOw
    /x2fiQAQRAFLlBrIit4hpY4g2Imjx088kkwkveGaIOKJFEZGR9F/dgBnBwYxPjmFZCoFjc1o
    iKIARZYhSxKIQGz9EtjD/6qqQWNaFQSAS5FRWxNEZ1sz1q/uQHd7CxpqQ1AkEZOz88jlVY1L
    ECopVdJlsgsFXCyAc5FI9KVnfvLs2jtuu1kKBvxQVRXT0zPoHziH0319GBoeRSQahZpXbePL
    xOuI000grPtA7U4TCvoUFlRVs9eD2Hzz9eEQujpWYcPaTqzuWIX6cBCSQPDS/iM42X9eB5Ck
    pU7BSRYSZU5h2ZKFCjZZ+OAHP9jQ0tLyubGxid86fPi4qKoaJElG5YFGhz3Hwtmzk7jmmu6W
    e++99+bf/OY36sjISJZSmmd0mZTZt5IhRj52qub9VYLmoVRrQkZRrFQhhDh+1qSU6vX19cp9
    9933/kxGvef06XOinSxcagJwOUkDX8Wk5CK+oerFJBY45vft2/fKo48++sudO7c+9vLLh6Rc
    Ti3AaZzAURRFtLU1QpIspFIprF27HqdPT0KSJG5egZR8TymB2y0jnU5YJ0+eSpum6QRDMc43
    XLEOSzXPhK7rsOc5iomAM+QcCPgACLA1N6RFryl+v5gxiXVhlio/r3T32QKgj46OnV+1qn2k
    vr62MRZLQRRF+P1+hMMh1NXVIBTyw+NxQ5Kkgt6EaVqLdCqcPWIHz0X9BdOkyOctpNMmslkL
    +Txgmk7HSbUSiQTvF1Tuquawc8ViEiEkp6pqbGBgYN811+y4q7m5cfX4+DRLjF2oqQmivj6M
    2toQ/H6bBVIUbQ0a07Q4zVa+4yCUnBtRFCCKApxhcsOgyOUMpNP2lc9TmCaBxyMjHAZ99dWT
    6XQ6Xam7EMMl6HRJ1TgAbMEIcxB+AA821Nd//o/+6A83/dGnPgm/zwvTNEDYICeBCDbWxyqn
    QoFzX5ElPPjAXvi9Ht/f/uM3bjrRe7aGUvorFNUSJWaQeSNQ9W7DEtzzjlNQuZauC0WaQGez
    WABWuV2uJ3/n/Q+v/a133wuBEKYaXZYAVHiuVKCNXlThuSSpAGDa6SW9hP5V1daHEGLCHgKN
    U0p/+OvfvLzuS3/9N7d99MO/I8RiMQyeP4+JqSlEIlGk0mlcGBrB3nvvQ39/PwbOncM/fP3f
    8M1vPw1ZliBLEtxuBX6fF0G/DzXBAJob6zA0Mo5oIunodPBOwakiVaQBuxptZ6dFQgQ7FDSp
    BcogaBZLGgCK9vZVCAWDhUSKOvobXMJQit23K9aiQCDZjBnln61Q/jkvcweOss9eo5SeOX9h
    aP5nv3i+Mx6P49CRY4hEonC5FIRrQnaVPBSArukwDAOapiKXzyOXz0PXbfskiiIkSYDX40Zd
    OASF7QcCilxeRd/AEF49eByEEHSsasS1W3owcGEcKOqy5MpazssNuXHsQY7tw9+8/sahdx84
    +EaHJEl4ft+LODswCICiob4ONaEggh3tUFUN2XwOmUwWmUwGqqYtVvRmw+2iKMDtdiFcE4TP
    64bH7YYsidA0DX1nz+M3+w9DEAg2rOnCNZvX4Wf7XqWapqcqQJEu2SlUs4DEfAIBQD7ykQ93
    tbau+tzExNSHX355vycWS0AUJfDDkOXUkU6wkM1qOH16gmze3N5x773333vy5HHX8ePHqaqq
    JiusOvBLp5Jccn8rUFCyuG4DYf5A5ru7bI6BPPLIIw8QIr6/t7fflU6XziwsHuCsJLr01hMJ
    R8DN1jJbROK/bF1IVuhT5+fnIwcPHvz+LbfcuuaGG665vr9/iPh8XoTDIfj9Png8dkU1FPKh
    t/ckWltbEQ7XYOtWuRA8OVVVw6AwDLuyms+bqK31IZ2e15PJRPpNfMOKdRec/SZJEgxnVpHN
    7xSr60WKWAdulUxmoKpG4fOq9LpiUr24q02IcLV9gvO7LQDGqVOn5nbv3t3X0dFyvSAIaG1t
    Qk1NAIIgQtMM6LqBbFYFITpEUSgkBU53wZnjYESQUFUDRT1Pe18rioiWFrt7bxgUyaTdgTGM
    lJlOZ3h/kCvrPC+nX3AkBrJHjx7t37p168E1a9q783mVtLW1oLW1EW63C5pmIJfTkMnkQYhQ
    6CjIsgxZFjnmOCdJKB5fywIMwz4DNvWu3dVXFAmtrW5IkgBVtRCL6UzMcFobGhqKozhb61A6
    88xIF+3CSdVwCg4ulVLaAeD3NmzY8ORnPv3HTQ89+F70nTmDaDTKBUJ8FdX53iwESQQU61Z3
    4O7bb0Rba6PyL996evvPf/VKcyye7KSUvkgIGUKR+YfHJhor1H62uL8lsL9fcigppS4AD912
    8/U3ffiD74NLkZcWY6sIQ6o88IwlnuM53zVNB3cQVpKX3wnaRjVd/+fvPf0j4TevvLa7pblJ
    6e7qQEtzEzZt6MHAufOAIKKnZz3+x9e+hnBNEDOzc3C7Xbj/7tvR3FiPdDqNVDqDRDKF8clp
    /OLFGfSePU9zubwzMMhXkGJl+6CqA5yX4hS44MgJFDwLkYj4gx8+g2DAb8NNNA2GocMwTVhs
    0DkajeFo8gT6BwbY/nc+OpsNiRAKAQREsOchBEGALIkApZicmSNc8syzYfE89+YKBUwGgLHJ
    qem+//aVrzWvW7NaqaurJe1trUil0pibX8D8AuBSFAQDftTWhLCqpRHBgA9zCxE89czPIUsS
    HnvwPoSCfqTSaSSSKURjCSSSKaiqBsM04fO40dbSCJciI5lO45tP/4KygWdnboGnUs0vBwSh
    QiHBYsJkOUrpwPT0zPH/8B//YpXP681fv2uH97abd5NEMonRsXGcHTwPSRARDPjR3NSAdd2d
    qK2tQU3QD7fLBUWWIAg2W5au61BVDal0GtFYHAuRGGbnFzA1PYecqkKRJbS3NmHbxrVQZAn7
    D5+kT/3sVzSdyepcojDDrjnmFBxoxrInC846UUpJd3e38uCDD17n8Xj+dGho+J7XXjuoxGIJ
    KIoCQZAgihIEoXgt9d82fnuKrFnT2Lpr143v6erqaj1y5PCLFy4M9VJKFdbldZIGdTkhKBX2
    AcqKFQaXPBRe+Pjjv91ZU1Pzu729Fxojkfhl0aY6EIOln6s0EG0HYJqm8vM8KzG7UvANx44d
    G3S73f97x44d6OxcdW02q8rJZBrZrIpIJImGhlrk8xmkUils3boNqqrD53PDMEyMj0ehaSZk
    WYKiSFAUGX6/BL/fBY9HoPv3H8uxRDlR1l1Iofr0mZfUVbO7YwaBTSPsdrtdYk/PWrS35yHL
    cmHA2flsCLEF/3w+gp07N1YYgBVKkmkHyuR8DQRccLlcjk/g/cKK+IQyXwgAZi6XyySTyWPd
    3W3vqa2tqclm81Imk4ckSXC5FLhccqEyrmkG8nkduq7CsoBVq8Korw9gejqJSCRb6MgqigS3
    W4Ik2XueUgJdp8jnTQiCgFBIpC6XgZmZpKZpGu8XeJrtqtvAJUhBcplMJjY7O/vK6tXdDzQ3
    17tSqZwYjSYVSgG/3wu/3wtRFGCaFvJ5HalUHqqagqqaJd0Wu5AgssKaCJdLhsslw+OxL1kW
    GdGQhURCg6ZRyDKhgUCehkJeHDx4PpHNZheYP5gBMMsKSXFcouK1VI0FslUMaY/b7f7S3Xff
    /a7PffYzyk037sbc3Cy+873vY/DcIFtIumjguZwNRJYkfOj9D6Lx5uuwfk0X/vwznyA379re
    /M3v/+TBo6fOtJqm9X0AF1hFSahQ5al6xbgsm+dX0Cw3Quy1O1a1NL3/o088pjTV13GzG4sh
    RYSgEOyXzCtwswmV5hUqqT5rmo5MNgeU6lUsa0WdY0pyFBRTAPo0Tftf4xOT3g89/v5d/+6T
    H4MkilDzeXzhP38Zd955FwYGBpBIxOBxubBrxzacPnMWq1qb8Pj73mNj2k0by29ZJp59/tf4
    /Be/5kBfnCpSggmMJLGEKuFyV0+5dSDsDLgA7HS5lCebm5rqjh07DlmRIYv2IXcqKIJgi7op
    igxQmxrUKREVlbttrQHLso2GaRowTAOGbsAwDNTWBInP6wlnsrkG2LS9ftiMYg6LjE4p1VgH
    oKQDd6VOc4nuW8w0zX8GQPKqdu/09AxpamrAzmu2YuP6tVjV0oy6WrvTIAl2W358YhLPPPcC
    QsEAAIpg0I/33HM7QgE/KDVhGgYy2SyisThm5hZwfngcA+dHMDoxDd0wYBgGzatqkoPZXDRh
    qJZzXGIAWgUQNS3rJxeGhmd2X7+r3jCM9/76lf1ybbgG127fgm1bNqGjrRUN9bUFHQ5nPoVa
    JobHJjA6No6eNV1oqq9l+8As7AdD1xFPJjE7t4DB4VEcP30WL752GF6PC5ZlGZlMLmMYZhJF
    7ZfZCk5BezOnUM3A6aGHHvJv3Ljx/aqqfvbIkaPrjxw5TiyL4qabbkA4HGavFSrMMZQGzJQC
    o6NRzM9ncP78AuJxLbB6dctt9933ru7+/t7nXnnl1Vd0XZdZ0sAP8S5r0nARu1CgGadFgnyx
    o6PzgUgksdtmhkKFxKC0K7B0d+HSOw5AEcqRzeZMlOoY0eVIIMp9AyEkRylNHzhw4EQksvCP
    9913/+eGhsa39PUNFSAa9fU1GBoaQXNzM1wuD3TdxPx8EnV1fqRSeYyPxxh0RWTJpIju7jp4
    vSn95MmTDiNS4iK+YdmLR+Vdte7ubuXee+/ZFgrVfDyRSDU1NtazjkmRNZDXT7AF2my4Yikr
    EOHYcIqq304VXhAEmCbI2rU9ofn5+cazZ8/O6rruo5QGYOtgBTifYFTbJ1wMlvT666+/um3b
    9n/0+0Of8HjcDZJkIJHIYnJyAclkDqpqQNMsFhSLcLtdqK0NQJJEpNMqJElAJJJFJmOwxNiG
    bymKDLdbhtcrIxhU4PNJTLhPx/HjR7Vz5/pj+XzeYchyvvJEGCsCywKQO3PmzPFgMPB/Jien
    XIDrgebmpi5RFBGPp3H+/AQSiQyyWQ2GYaFch0JRFITDfhiGhVQqz/ZYkUnLXg8RbrcMn8+F
    mho3QiEXXC4Rc3Mxum/fC1lJMrMLC5HZJQpJFRnzlgWSxDaZRxCE33/44Yff++X/+l/F9rZV
    MA0ddbVh/Nmf/nvkc9nCQCctu0qeM+2YP+jz2MNAlgmvx4V333Mb1nW3K1/467+/7o1jvTOs
    mkcIIU6w4ig9L8s8wyU4hkKFmVLqAfDI3jtu6b7h2m02jewSMwl5VcXo+CTaW5vgVpSLU6ii
    DKYErrtg2UFHTlURT6YotxbLLmBUocuQYdWMC5TSwXg8sVOWJCJLIg4cPInB4TG8/wM34wdP
    fx+PP/oQXnx5P8YmpyBJIhpqwwVHIwrOLIOAbC4PwzAMFGlUHZYknjJvRStIjnAaIUSglLop
    pWsAvLelpfnxRx95X8/vfeyjQmtLs43XJ8QOChmfPkogRyYHQSqfYTC550xY1ILF6Fjj8SSe
    +ukvfd/6wc82TE7PyazdnWM0p3kAIhvwyqNUAbuwD6p0Tvghr96R0fHRttZWfPGvvoDujjZ4
    3C5bO8AyYZmGPdwMijdO9uIv/9v/gGWaeM+9dyCXz+O7P3oOL+8/hD//zO+hu70VEAUE/T4E
    fR50rWrG9ds3Qtc0LMRieOaXr+DvvvlDi9LCfihQ7LKh09wKdB35ve8MQO8HMHDw0JGHI5Eo
    vvgXn8fu666F3+cFAZDJZHDhwjA0XcPqznb4fB4QABeGR/HFr/wvRBNpdLY24D9++vcQDgVg
    WSbmI1GMT0yhJhRAe0sjakNd2LC2A/ft2Y2xiWl86e++iRdfO6yxM+HMKzj6L3Os41Bxxmc5
    zonzO30+n7hx48aHE4n4F1999bXGgYHzsCwKt9uLmZk52AqoYqFytvhr8Xsb160XqqyRSA6Z
    zLzQ0VHTuXnzjseSyZR2+PDhN7hur+MbVmLObcm1LA44UvrRj360hVK8b3R0ypPP6xXoI4vJ
    kt/vQVNTLSYnF2Czy1z+DANgq95SalrZbKbcN5jLUWwre5iUUke3RBocPH/+9ttzF1wuZYvD
    BtTcXA9FARKJBDo71+D8+Sm0tTUgEPDANClU1SgMv9qXDcmQJAHJZMLIZrNL+YZl7y6UzcoQ
    AKStrc199913dzU1Nb3LMIwn+vvPbT527KSYTKY5lWOBC/yKe91OhgRuZkEqmWGoNAwtCHYA
    3dFR47n11rvWrVu3Tj57th9jY+P5dDqdYcxdAkrhmlX3CZWKSWfPnp1tamo61tHRkT96dAAj
    IzPQdbMQ/PMzGrIsYuvWdjQ2hpBOqxgZiaGzM4xdu9owMpLA2FgK9rA4gapa0HUD6TTFwoLd
    fQoEZKxZIyOXS+WnpqZ46nW+mLRs2jNl9w8236mdOXNm+syZM//a2tra8+ijj+0ZG5tHb+8F
    ZLN2E1IQRPh8HoRCbqiqxWyd/fy6dS24+eYtyGbzeO21s5iZSRVmfvx+FwRBQD5vIZczkc+r
    iEZ1KEoOq1b5EA4TK5eL5yKRKA9DmgUwxwpJ/Eyb9Wa+QaqSkWxubGzc85GPfETs6OyAqWuM
    tUdAXV0tYIXsxMA0YVlsiNMsJguWM9jJGGIsU2e0oyxwNi2sW92Oe27frRw+0bfJsmg/AINl
    yzw2bUUEuyolEJRSyLJMdV1f3dxYf88De/cIbrfCFJwXw4cEAhw/dQb/+tRP8YU/+ThaGusX
    0aRWok5dSv0ZxA5G5hbiTvBiljnMqjiFJaTn+S6Do0GhAFBHRsfIs794AYlEgv78+Rex/Zod
    pK+vDwMDAwgHvKCUYn4hCkWW8PN9L+PgkRNQZBlejwKfx4OA34tTZ87BMC2D6y6UJwuLZhcu
    8j4v2yCW4bJlAM0ANgK4u7Gxce/tt9228Xee+G3XHXtuh9fjsQNkagGFxJg5O8LoQ9l7oYDd
    ORKcWR72OhBQsOeIAGLZnQlJIGhuqMXv/84j2LVtg+eHP3tx0/7DJ9omZ+a3WpbVC+A0gLOU
    0nGOUjCHUrXTap6TApUcgNOmaeYa6mp9Ho+7wP5UnMewQIk9yGyaJjxuF1qaGpBOZyAIAtKZ
    rB3tOfu+DL4oCARNdWF43S4Yhqlz1aM05xiWre28lGNkjsGhzJQopecsakXbWluaggE/TMNA
    IpXC//ynb6JvYAjRaBR33XYDPvnk43ArMianZjAXiePW2/bg5Zf2IRqPo7YmgGgsgS999Z+x
    kMhCFgke3HsTHrjzFgC2fa0JBpDPqw57mEPl7DiGSk6BrpSN/MQnPlFnWdYThw4dbuzt7WPw
    IhmGYeL8+WEObiRzMCS5BIYkijIHTZIL9KuCIEDXKcbG0vD5lLq1a9fuPn78+JRhl+hMBhPT
    sDJw1Ut61NfX70qlMtvm5mIVKWR5yElLSz1qa4OYmooWmLYup7tQZB+SYFm6lUymeN9gcN/T
    avsG5z1YlkVZVTsPICWKYkxRFKOmJog1a9rgdrtpd3crRkcHiTO7YBhJABRutz3D0NVVj9bW
    cGHQ1TQB0yQIhdzI5aC9mW8ox+1Xq7LOB9gbNmyQbrzxxsaampoNbrf7Tk3T7hkdHdty+vQZ
    z8jIKDTN5DQWRDabUERY2GiDooAbazqXUJPbr+O7E5R1JSyoqoGhoTjica+7ra2r5447ulpV
    NbUlHo/2RSKRU7Ozc2cHBwfHdF1PlunULNeslxMTaMPDw2M7dlw7YllWez6vsfPN6wcIJfS/
    hBDkcjryeQOaZsLrVbhZH7Koy0KIANMENA1Q1TydmZnJcb4hVVZIWil6XcogeTpjzZQTicS0
    qubP67qxKZ3OF+6ppaUWt966A+3tLRgcHMGBA2eRSuUhCCIaGoLYvHk94vEETp8exexsBgBB
    a2sAu3Z1oaamBn19o+jvj0LXKQRBYOxRBLFYzEinMxnmA5ykwbmc7sKbQpGqkjBwh8Xncrm8
    dXV1JbAjp4pqOU7fMmGZViGAKCYR7Dkniagg1EQA+DxuCILgtyyzCcVhpgQLToWr7RDsoWZc
    t2nDuq7N69faFIl80M+Jkmmajn0vH0B9bciuJFLzigaeCQhm52OYjyVMzknyl3mlWXWF6jpo
    kdOMH0p3RIrSL/76lfFX978u5XJ5f0dnp/9d730YX/nKVzA1PYsfPfvLAjOMZVGcGThfujlF
    kcqyBMui1DRNPgDgqyNgEATFeZtlkIBFHZYrqaJQag/nmqZ5ndfr/au1a9duuXH37robdt8g
    tq1qRSgYRG9fXwFOZbHOmaPOa3Jf7eTY+W8+gWYBsp1JMGVntsBsnkEU7MTBpSh4+IE7hFuu
    31YzMTW74+SZwS2Dw+MPjk7O9Om68T8BDLMz4jhRiurrElBKqc66DCNT0zORqekZX6MDq+Fn
    cBjH+JaN6/Cl//gZfO+Z5/Ds8y9B03TcuGs7HnvPXrS3NhXYxAr/477XNR3D41NgTGS8Y0gz
    qs1FIk0rAFNzZpucmabBhYXIxNmBwaaNPWsBAMlkCoeOHAcVFRBBgEuR7aF408TGnjXYtW09
    3jjwMu7dcyNaGhvscy04nkfAzHwU/eeGce/tuxmtLsXcQhRjU7MmSgXBeEVnZ8YnXwlCuZwP
    v9/fmMvlOicnJzmOeH4Il2+rV7rEEtgFHwDbVTk7uEinDRIIuJvdbndbOp12AiEeokZwFdlx
    mFCdLEnS9ZFIMuR0FxzIER8I2XzpChoaajAzEyvrLlwuWxKB1+tCNpvUk8mkUzQo9xHWlcYC
    ZfAr3jfwVK5UURTDsqxkc3N4orbWL3o87kAqlfIlk0ls3boVLpeEtrYwikPvQCDggmUVZ3sM
    w6A2tCdlRaMRjVXQNaebSh0WEPtvuvj3xZAJJX6BPXdZ5+Khhx7CM888g8997nNbZVn+L/F4
    /NoLFy7Uj49PSqmUPYNVV1dXVk0X2PyCVJhhWNw1ELlOg7NfRI5itpQxqrjMwPR0VnC75aDX
    G9hWWxvetGrVuncD6jlBEP6ht7d3gEuuquYTlgiYTUqpNjs7GzNN40I4HLzVuf9ypic76Kc4
    e3YGqZSK7u76Ak6/r28BkUgePAXvYnIEAW63CF1Pmykbu8PbAKeYlF/uLusSLGoqgHQmk5nL
    ZjO94XDwAZdLkQzDnrsIh4Nobq5DKBQshL12Rw0YGVnAr361H6pqYHY2XVgrSm1Yc21tALW1
    HrhcEnTdLNhXn0/ExMSCqqqqEyc7fmEBRRrV7FtJGK+4w8BYAKLRaHTi9ddf79m+dYtNDUUo
    BIiwCCAQmxGJQrShGcS+CKuekkIoINjZMygEWLCI/XOTmpifj+LAkVPUMEwNNpWpw3GtEEJE
    XF0RK7vvapoKgPWbetZ4HRx2YeaAC/QJIRgem0Tf2fP4xIcfhdtlV1uX1GW4yBA0r/Taf34U
    iWRa56oG/HXFWN6i+AwVAIQppa2wWaKMsio/gS1YdCaXz68xTHOTZVninXfeib7eXtSFg7jt
    8ccgiQJi8YSDyQchBMmUPewaTyQRSyQRjcWRSmcdw+4MdHkBhCilzbDVXjtRZKmKwpY41znH
    6DjHqjAjmKYpArjz/vvv3/PFL35R6OzsxEsvvoiv/O1XoebzBTPB64w4vFVcs7JUcqUc5eYM
    QNPFSsnUsph4l7MXgBt2bMbHf/tBPP6wKZ85NyR//sv/uGlobGoz0wKROAFCZ02q7hjYHpiN
    JxKRhYVIByF2cgNBACUUpmHTxZmGAcsysW3zeqTSaXzhS18FpRR/+OQHsKarDYZhFoTeBCZy
    ZsHWGwAFDMPAzHy0oLaMygPPy9Z2vgTHQADMplLpwYHBCzsdiFlzUwP+4vN/goHBC2hurMc1
    WzayYTcTNcEAPv37H0IqlUYo4IciizBNE+FgAJ//oydx/HQ/DMPAji09kCQRlmmBEIrxqRnE
    kykTRV2YOIAoIwNwxM2uCgd9LpdT3W533uPx8jsdldiQKgsyoeTnxe+LwYYgELjdErLZjKCq
    ag2AMIAopdRdNud21R5PPfUUvvGNbygAmrLZvEApLRt2LoUP1dXVQBQFRCJJjjHp8iBJTkIS
    CLgwNDStqqqqcv6Bpx6+7L1h897L0DQNsiyLjzzySG0oFGrJ5/NKNBoxJiYm1bm5OSOdThNV
    VQkhJHD+/PmTXV1dbW63e70gCOLQ0BBaW1uRy5mYnJwCIQRutwuECAyOhMLAqyzbHyuleSST
    CSuTyVCfzyurqqoQIngkSQx5vb7mlpZmT0tLc7fP5xNN06Tj4xPRU6dOR03T1Jl/4H3jZXcj
    /+mf/gmEEPKFL3zh5qGhobv27dvnikajaGtrx0033QRFUVBkuBHKBoNL2bDLnytVcuYDZJTN
    +oD7zO3vEwkDQ0MZECJKgYASWLdOWV9bG94KIEEplZfZJ/B2UVdVNWMY+qjP57FEURAcliM+
    WXCSJl03MTmZQGNjEI2NAUxPpzA/n2UQpsWFA35dFEVAPp81VVXL84UDrqOyYrOO5WvA5niE
    dDp9rqmpJu52u+rT6RwAgpGRWTz77H4IgoBoNINsVi8kltPTSUSjAwAIDAOFNZidzeCll87B
    7x9FLmchmzUKRQhJEqAooLFYVGVr4JABRNnXBJdAXbKAnVSF4IkAiKdSqae++tWvbp6fm23a
    sGE9FElimxg2xzp1WIKK1JGgRaEqi4MpOcJVhqFD0zTE4gkcOHSC/vrAkSTDX70txLmWsp/O
    HinRSOBhRKaJlw8cRm1NENs2ri10WEoSASbUVmleofBz7vemMjkcOtlPNd3gqcN47uFqrlmn
    JEl/2dLcfJ3b7RKy2ZyqqmpG1dSEpmpx3TCyhBBvbbjmmttuuWlNa3Oj9MqBQ+jq6sILv/wF
    /sMffxI+rwc/ee55OnhhmGq6TrdtWi+89767yDVbN8DrVpDOZBCNxsnM7DyOnToj/eqVg7Xn
    R8YUTdObAGwWBUFnrAmSW1EUt0tx6XYgOa1q+jcopW8wDD8vBV8tiIokSVLnzp07hfXr1wOg
    uPWWm7Fu7WoGQeME6JzEgA23OoPupOTn5Yq9FmDRAgyHPyOWaQ8DG4Z9Pgxdh65r8DLBG1EA
    6mtDILZFbQDQxHcAuIpr1RtsbH2jmq7HJqenEYvFMTM7h5nZWSwsRDC3EEE0GkMqbQuVqZqG
    VCqNbDYHSin+9eln8ZNfvgSXosDrscXbamuCqKsJoK4miPpwCAG/G6quI5HMUJQpO7PPm4cn
    rlg1nXcM7GvUonRwbGLCyuVzgmWZSCZTkCQRLU0NiMUT+PFzLyCRTCGbzULV7EF/R9xQFAW4
    ZNnGtQZ8CAX8aKgNQdU0JJNpeNwKFFnE5Mw80pmcMz9RIARAUbCKp5RcsfUghOCaa66Z37t3
    72s7d+7cnM9rYjqd4TDbS7MilT9XCkuyv0qSDEWRUVfnQ00N9P37z8V1XRdQZIh5WxSRAODR
    Rx/Fxz72MWPHjh1Rn89NRVEkpboTxSBIFEXU14cQjaYL9JpXOvDs87kgSaY1PDycZgUE/pxU
    xTdomg1D/uxnP7vO7Xb/50wms00UZRIK1eTXrFmbsSwrYZpm3DTNHCEkIIryjrm5WFckkhHr
    6gLIZNJoa+vCsWPnIYoiOjubqMslUIBQwyDCxESCxOM5WBZhw64K8XrdqKtrlHfvbqnfsWOn
    S1XVZkkSt7jdHt3lcguyrIiEiC5KBUUQQHt6No03NjZ9/fnnnz/B5r0cW2hdSRDZ0NCA6667
    TqSUds3MzCjz8/Osyj+DF174FYPRCYWZnFKVY6HsObHCa8VFsz2VuhH22XAINkRYlg3TIYRC
    0yyYpiWk0+la2FBaMEi3Yz+XhXKdK6TkDMOYdrkUw+NxK4QI8Pm88Hp5oTKpoEUginaSK0kC
    QiE3tm5tZPoEjmAbhabZQmWaRmAYAixLgCwTaFre0DSN9/3OlUOZ/swKF5MoABKPx4fa27sj
    Pp+nPpvVCkJ1uZwGt1tBY2MNXC65ZC3K9TacNXDYoQwDUBQJpklgWYSRHBhWLBZz1oH3Cw7i
    IPdWiwVSFR3lvgsXLrj+yxe/dL8sS62CIPoEQXATQhRCCLOQTiGVcE3isuqpw1VsWdRmiaGW
    RS3TNK0M7MnuCa7NzlfOrrogC6XUBHDu2Km+zNT0jL99VTMsizis/AUBY4PBtN6191aEQwFQ
    aqvcstWwNwUoKGGSTnZ/yn5OAMCGvlAIL4DDJ8/i8KkBg20MvgVXXnGtxqPZ5XLddu21Ozqv
    2b4Nfp8Xuq4jmUwikUggk81CkWVs3bwBt9x4A77/wx9j46bN6OvrQywWwQ9/8hx9/dBRY2xi
    KkspTQJQz5674PrNa2+Eb77hWu9tN+4SNqztRG1NCD2rO7C6oxU37twq9g2cD03Nzod0TYck
    CXArMhRZAqUWstk8RidnkEpnm1VN3wzgDBtAc/DMVa00Wpalzc/PY3x8nO1oCq/PVxxoLtdT
    KFAHs64TuOcpB9GzrEW0w5ZlLvpdFoP3iKIAgSjQDQPjU7MgoDjRN4hoPKmx813DzoqbBVLC
    MhpFE0BO0/Tod5/+MX798n46vxAhqqZxXRNAFBkMgxmENd0doNRCNJrA/EIUpmnCMAyHMx6i
    SCAKIvw+D5obalEfDmJmPgK2r8udwkroLyy1BhZLzighJEMpHTvV25/7yt//k29mbg7RaJwp
    Otu+w2FCsYoqRXwXD45woQNZIwKBW1Hg93nQ2lSPtV1tON47AAbHKMHrojjwuWJrUWYPyYkT
    J7TGxsZ/27BhQ/N73/uu27LZnNc0LZFSCPayFbRqS+BKlxAIU8ZNT3U9ox47dnq6r69v4mrf
    81IPO3izzHg8frCxsXahsbG2IRZLVYRUCIKA+fkEslntitiRnERDkkQ0N/sxPT2UGR8fj3GJ
    ZIrB93LVLCa5XK420zRvnZ6ea56fj8A0TYiiVFDzlSQJlkUxPx/DzEwEmzevxczMFFataoPH
    40VPzyrqdgvG9PRE+vjx8ylKqdbT0+PesKEznMsRz8JCVkilNOg6RTKpIpMx4Pe7xJqaQI3b
    Ha6xqWMJ8nlSqGBLEoHXKyIYFBsDgcBrAM6zM2ZwvuGKAuaFhQVQSjWv10tDoRCxz6/A7JjT
    UbK4gN/RYqCcEFexm8QPPfNsOMVkodLPSylXBcG+b0JEhEIyAF2fm5sXOZ/gwfLDuSkhxKKU
    6oZhxmprw7kbbtikuFwylWWJFEkBHHtarLnquoVoNAdBIPD7FW52gXAQOLuuqutAPg/4fBIm
    J3XDMIwcX0xCcV7DwMoXm3kIHMlksnOyLMz19LSv7+pqgd/vgSxLhc/QSQp4qB8v3lauUVMs
    3gOaZiGTsZh+RdZMJlM5lJLFlEOzrEookuVOGCz2obxkWdagqmqdLIttAFDLNqaIxUIxpGxR
    y7+nXCCisxuNoMj8kWQZ8ttF3ZcAOHjs5JkDf/P337j7rtt3Cy5FYTSaNjyDsMxpdWcrFFnG
    oeO9xbdegXLWKlCpOjSbjgKwCdOy5z/mI3F879lfm5F4KsI6MI6qa5wNvVbNKQiCAMuyzmSz
    2S+88MK+B944dGhdMBBodLvdHkmSJEEQJFEgRBAE0n92gHz7ez8QE8mUcsedd5Fnn33WmJqa
    ypzq7c9SSjMoMlpkAGhzCxHXM8/ta/jlS682NdSFQ36vVxZF+wSxtaOWnUhalmVZpmmZumHo
    qqrl0rl8KpfLL+iGOQDgKDOEDiSh2irXpmVZR77zne9MvvLKK14CCBQQQanIUjpStHy0qK9Q
    gCkVEmVa6EZxyTOfSBcOMQdtKnaX4MSnJYfd1rBIL7DPvFzJddkE/Nj+ypumua/3zEBDMpW6
    URQEJZvLQdMMSJIIr8cNr8cNWRahyDIkUYQo2AUESiks07KF3XQNuZyKdDaHZDoPURSQzecR
    SyQRiSWNmfmoI5bGV9IrDjuvUHehxEEym3Xh/NDwoaPHa69pbKgLu10KCLFpdB1qRYEQG8Lp
    QBcJZTMrpKAMTymFIAo2c5hAkMurGB6bwuGTZ+jAhTGT6yykUMoOo+ISmC+Wo3ji/LkXXnhh
    7OzZs3/b1dX1sqLIawzDbDAMo9Y0zSCjIS7sTV6UqDgHV9zjtlm0YJoWNU3DyufzRjQay0Wj
    0bhlWc4gXwz2LMuKkWBc6rI8//zzR+6///7vbN265ol0OheyLMrZJX4gCOV89iUwrdKvi+Fb
    7GeUEECSCE2lIuk33nhjXFXVKc4/LKlfczmPBx54AIQQvO997zvW0dHxhfb21nubmhrWGIZR
    b5qWx7KoTCkVKQWhlJJQyE82bOiWFEWUR0aSpLm5WU+nU5nZ2dHcwMDZ9PT0TNIwjBSAzPDw
    sNbR0e7ZtGlTY0tLS2NnZ10QEGRKnUCXACBUEGARQikhMAmhJkANSq2saeqpfD43Pz4+13/o
    0KHT1fYNjz/+OIaHh2kqlTre09Mz1tTUxD5bKlIKkUF4WVfJqZaWinBxUCTKw5Iq7YPicygx
    56WvAS0+D0gSxezs9EI0GnXYksRFeKjlrbBbMzMzF3w+/w9DIe+9AFnl6AnYwa4Fw7CYAJkt
    QlY8C2DJliNiJ0CSCNNhsBMGWSYQRdXKZuf0+fm5BKXU8Q0p1km6rAC5mmvA/pY5MzOTzOfz
    r9fW+tuSyVyHZVHRMEyIYhGm5Xy2TOtxkW0tTxYspseoKCJcLpEKgkYjkWg+l8uV+AW2Fpft
    I0mVHIMCe6agBkAdpbQOQD1LFkKwMeeVWsSkQia2VHbmDPY5097OUF+UBQ5pcDi8lQwSuIEv
    ma3B9aIofNjv824RRdELwEVAFEIgseIbKfGOnNcoXwrKvCbzm8UODPuGUko1TTezeTUKYJLr
    wowDGCOETLDnImz9rMtdH27oWaGUellC2AqgkUsO/ZxBVgDsWLNmza5QKGQdO3bsHIBpdp8O
    7jrOgh3HwQsswXQ0BdzsdzmQlzxKB10zXGs9xz3Hc3I7SUmmSvcvU0pXAbiO3X89d/81bL9f
    iiO6lDdBL+F8lH9vsOBpgq33FIApQsg0ez7htGWrcU44O0AAeCmldZIo3gGC3zNNq4Vxgbtk
    WZLDNSExGPATv9dDPG4XcSkKcTjiARDLYgmDptNsLo9EMo3ZhSjN5vKUEEI9LsWiQDyXV0fZ
    Hh8HMAJglBAyzu43yvbCVbMFsAct64LBwJbuzvZ/H43F90SiMVFVNRIMBtDZtgqtLc2orwsj
    FAzA61Ygy5LdUbQsGLqGXD6PZCqFSDSO2fkIJqbnsBCNQ5Zl1NUEIcuiOjw2PWVa1gRbgxHY
    Q+6j7NzPsXNgXoWEgQBQKKVBdi7qYcPjHFsRZudcugS/sNQ+5/1ClAXC87DZoRwWkCQfEK9w
    8lhuM0MtLS0d27dvv9Xn827RNL1O1/WQaZpBy7Lctn8odQ/FALOyv+SKBZTr1FPLsiillObz
    OWNoaDg2Pz/vFNqmmF2YJIRMsefizH5e1hpxJBIEgCJJkre9va0xHK5tkSSxURDEBlEUawVB
    8AsCkSmFRCl1b9iwfpeiuK5RVdUYGRnpf+ONN+bT6TRhnfo0VwzIAzAFQRCCwaA3FAoFvV5P
    QBBENwCREFiEEJ0QolJKc5ZFM6ZppA3DyGqansvlsloymcqy4c8MigO/jm8o0Wp4K/fP3bu0
    cePG5jVr1lyv6/qqTCbToKpqvaZpdYZh1FiW5YMt4HaJPoErNKHIllR8TTGZ5v6fa1RSbovY
    gWoymYpEo9HxJXxCnPmNqp2RMnpRGUCos7Ore+/evZ8KBPy7BUEICYLgAYhiGKZkGJZgmiCW
    RQmlhBSXijinAoIgUFEkEEUCWSZUkuwiommaNB5Pai+9tG9iYmJiXNO0cQCjAEYIIWMsNiqx
    h1cpRhQJIYH29va2zZuveXj16u5P+/3eGkWRBUop0uksMpkccjkNmmbCMJzkSWCQRRuWaYsX
    SnC5bBE7r1dgQoEWMhmNHj68f+H8+YGphYWFEUrpqOMb2Fo4PvIt28VqJQwiC+z8LMgLMUfh
    iEi5l+gwXEqg5HQZCpPmzoFnhz3FBY3m1XQKzPk5DrINQBfrtDQyhxlksJDySu/lOExallA5
    LCkLLEGYZsbAqShdseMsSxC9AAIsGAyyxNDPnpfY5+0C8IiiKLdYljVgGMZR9hkJLNtPc049
    xxgVHAVjFwsoPGzNnADBGVRzsLc8Tagjx847BWefZKuRMLCH7HzOlNIG9vk6n3GYSxgut315
    pa9zHG5siSAqvUwJA9hZD1FKm2APo7exq5UFigH22UpcpUuocCb4oNAJDPlO4wLb29PMGTjO
    z6ER1a6WLXAcJCEkwPbHDQCudZJrQkitKIo+URRdoiBIRCCCQIhASiYe+aKAjcs0TNPSdYPX
    nUmzNZjhAsEJQohTOIixfW+t5FqUnRMn+a+hlIZZQl3DnquWX8ijiNF1FH7jXAXduErdJn4t
    JGYrw1zy5CRQdexcyBUKDZfiGyrZA159WmNr4SQNc4SQWS5QTLNzdVlrxBcMKKUF38Cuct8g
    AhBlWfY+8cQTH2xubt41NDTU+9xzz51Mp9NOOdV5v87QvlPo4n2DF0WYJTjbwBN+GBV8Q7os
    Ybii+IFLCCUW84SZnWssS459V9knWOw++cR6nvMJqWonDGXrI1JKA4SQ2nA43O5yuToEQVgl
    y/IqWZabZVkOS5LkkWVZliRRFEVJFNijaBed7oNFLcukhmFahmFYmqaZqpo3Mplsfn5+Pm4Y
    xhxnE53EeJbtp8zVKB5wZ0RgWl0169f3rFu3ruc2URS7CCEtgiDUE0KCgiC6RVGUBUEsWQMe
    hmXT6VqWZZnUNE3LMHTTMDRTVVUjm81qAwPnFmZnZ2c4/+j4BidBTF7Oma8mJEnnAjKN4SOd
    gE/BxYfQyJtUTy3O8DkVZv5aKV7dN9sQztBjHjZTB2FrEmeH1Km+v9Vk4c2SB140Lcocg9N2
    doYeqz3owwsjGZyRdroBAvt6RNO0OQBjbF0EFJVxk+z9Jhi0zGSnQqaUuplD8HABZnnHqVxj
    wnEKDo5d51QtrSrfuwFAJYSkKKUSl7TF2ftd6nOlV/h3L+Xnzntxgqg4S5ouSc3xCluvFluX
    JIBJRn2aZIEK34FyEgVhiYThzexAku1zR6gsjlJRHlxle8DbxBPMeTUBaKaUNhmGETYMw8Ml
    1+XrUG4Xy2mCTS45dqCIs5zzfztAckzOLoEx02QppQkW8Llw+cPJtGwdsqzdzl8le2Glk4Xy
    98vw8s7clsjeW56dUR9KZ4wuxy9czDc4FXuHKSXO1kitZgeKoyw1YQu28b5BdAoFhmHQ/v7+
    g1NTUyNHjhwZS6fTGucb8tz5TjJ4mcnWwPENHs43iJfgG8wKvoF/zRXRjbPP12DzRGkU6cXz
    bK3dV9knUK6oGGc+IYEqz7C8yfrolNJsNBqdL0tgplhSyc/ZCUvYxEr3V6KqXJYUlWsNXO1H
    YZ8MDJybGhg49yuWWDYTQpoFQagXBMFPCHEJAhFtBlDiQDYJpYW1cDqK1OkoOlBtbh0SKAq1
    zbPP+4rsYjWHnp1N51R80uzDlzinSC7zMJQrOjuVZaOsinDVNwOKAlYiy66d/44zJ7kcTsGh
    mHSoFZ1W65IqflVIEJ2AWSwLDrLcPcqU0iMA+rjnCkkVe49OhyHPV5EIIS5mQNwM6yyxNSUV
    7p2WBZP8wNNyDEHyQmUOvMo5oB6ui7RcTuDNXmdxnwUfSK1EEGlVOANO0BhjgZFSIUgW3iRh
    oGVJYYZLhpxhTocE4W1jC1ginGIVZnDV0yC3Vy5lzoRPmmhZMB5naxtha7FcRYLL3QvgAsE0
    O9sKLp/NqNwvOLSQVaULXaa1yDEGMwuASilNsTPhJI+VfAO5AvthcTbR0Snhh+KrfV4KvsE5
    /9xedeClAqVUPnjw4BvsOYVLjp014n2DuoRv8LCOhgxAWMI38EkDT8OcLSs20iqed4HBqvKc
    T1Cusk8oYZXjfMJKzYBSAAZL/lKwu1F8cujn9kcl34CL+AaL23dOoSzOOo2OPcxfTXtYXlRj
    8DknsSQsuc6Zphk1TdPHJcJv1T/SsnWIwabZjrK9eEVFQ6mKi0C54F2r4ACFK9xs5YvCX3i7
    VJG4hAmsYu4EDE41RMbFh7/xJodjqSBCRxnNJJZQwK7CGpmcodW5yoXCJYkCM+4eFvRLXNdB
    J4TwQV+6LOt19BZkAAohpHxIrZLBoGXdDh2lHNuFNajC/VPuvk3OCPL3f7kJYDUcCOWS6kqC
    d8udMGhcAGCwM5BEEUagXKSKRJaovvFJobP2vChPdiWqZZdpCwSnwubseUqpj7MHbyVhqLQO
    aQatcPDYy1EkuFy/YJathcg5QvFNgoFLSRisClXlRSKVV9kv8Guh8oUlFhi7ypLoancYHJvI
    02wvB5sY39HgfUOS3V/hHtkcWLlPdHyDU3BMlu1lwv697Pw+Qoi8hG8ohzRaZTaRF68zq3je
    eZ+Qfpv5BKPMJ6jL7RPKbAGvU2NyMUAMpd2ii3UYKvkFWlaszlbwDW8X5rRCsYdpQxDOLqQA
    +Nm5qFRUWwqVQpdYhwwrEPDq51dUNKzadHy51PrVcFJX61F2784HzAe9/PVmbFGX4zT5Slv5
    tagDc7lrVXaf5clgpUvmgkSJQbQckS+eO75cTIX/neJFgkpaofJaDlFalFhW6f75zF/EpbdQ
    l7O6VCmJsioEUVUPpCqsjQM/UHgHj8UMJW/VIPJJocpdThCwomrGl7gOMnOGbu5ycQ5yqTkO
    vMk66Fzw5xQHyrn16dWyjxfxCVdq+96sgPK28Q1l61Bu10TunJQz11RrbXjoqMn5g6rag8vw
    DRJ3Dir5BifwU6/QN5QHUxf1DW91DSrct/Am76/awedbLbpaK+UTLrL/pTKfIKO04yhWqKqT
    N/ELfEJU7hf4buPbwR7y6+BawjfIS/gF8iZ+gZ/3Vcv8QnmR4C2vxdtC2Ob/D48KhgNLfNjL
    ZTiWMpCopjF4C4khKXOMBKWY7PKq4P8nEsurnRi/nZPsNwkaCC7eTahUUaUXcXoXCwCuarC4
    xDqUB4oXSzAvZbC1El67HLf9tgma3/ENF/W7pMqJ1FL75aKMaytsG8vPQqUzvqKQsiolS+/4
    hEvb/0v5BaGCT3gz31Cpm2SV+Ym3m19Yqugo4s277m9WIKBlPsGssCbv+IZ3Hu883nm883jn
    8c7jncc7j3ce7zzeeVT38f8CxY6XkH8FzNsAAAAASUVORK5CYII=
  }
  set pieceImageData(Fantasy,70) {
    iVBORw0KGgoAAAANSUhEUgAAA0gAAABFCAYAAABwkcaYAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAANIAAAARQBKP5TVAACAAElEQVR42uy9d5xd
    V3Uv/t2n337v3Om9STOjUR/1ZlmSi2xccQdsMARMCQGSPJLw0n55aSQvlRcCBAiEGsA2uIK7
    XGT1UR9N7/32ftr+/XH3mTlzNbINSLJNdPw5nhnNnZl79917rfVd67u+C7hyXbmuXFeuK9eV
    68p15bpyXbmuXFeuKxcAgFxZgivXlevKdeW6cl25/gc5fkJ+3bjgl4kd6K/6fUrp27k25E1e
    J7U/97fjuV65rlxXrkt3CVeW4Mp15bpyXbmuBMdvGMFeCf7+x2yPiwR6ftm/+U7dYPQCa3Ll
    QFy5rlxvkz+6XD7pCkC6sjGuXFeuK9f/AFtDKbVsDrEFpsSyKST/TVoQBFIA1G6rrtif37zL
    ek8JIdS2Ty6LX/pV/eBlAkegF3hh7+DnfeW6cr3bYl9CKbX7Jcsf2ZMR9o+XxSddOeHvjE1C
    3uT9KNwgV4KUK9eV65cwwr9K0PebcsYK1oFjYIkAEAFIebtDCCHgKKWEEGIC0AEYto/mFfvz
    Gx+kzJ0Ttj+sB3Bk4YOtN9/EFZrZlevK9ZbPW6Efsp+Xt+KfftN8UoGd4TmOkwAIpmkCAMdu
    Sgix+yLrprYkxiV5nlcqSG+TI7JteELyaTuBvR/mIs7IZLe1KeYO2//ENft1r7caLF9x9u9u
    R2RLOhB2pqwsFSmollDbvQAI/KZctgoRoZQ2ANgAoBaAO29XaJhSnAMwDCBXcKsAtN/Utfmf
    7JPYOSHz/5w/OzzP8x/4wAcqZFkueumll0a6uro025mhNvCsF4Cm3+j1IoSABW9v5tffNDj+
    Vc7wlevdD45soGCxm77B/RvnkwDgtttuczc2NnaIothOKS02TVPI5XLpUCg08vrrr3f19fVF
    KaUqIcTuj1R7PHyprisA6fJuBmIDP5wVtFBKFQDvAdAK4ElKaV+BM9LZhlgQwP0mgiS747YC
    uoLbtqTUXmq1vm9lHQoDX4Pa3oTF3qJCR/9OXd9LAR5/E/eRDQxxlFIXAB+AAIAUpTTK1tHK
    RmnsnP1GOiO2JpRS6uE47vdaWlruW7ZsmUORZZLLZWlvb6926vSZ47qu/x2ldIQQkgGQZD9q
    2ALhK9dv1mXZSws8cwDInj17iisqKv4CwLprr732m+Fw+CfT09Oa7cxoALI2sLSoT7qQrXo3
    2Ru7T7LRgKxqa2Eyk9ooeXM+yarcFiRiTPtjC3zfeT6pANS+43zRrwsAf5N9ke1scABISUmp
    2NKyxFtaWhbgOE7Zv3//9NjYmEYIoQX+SP8NTz7QlpaWzaZp/vPY2FhDKpXiCeHgcDjN8vLy
    2ObNm7/e19f3GIA0gBS7rZjYuNRP7gpAurwBG0cpDSJPa4mzw0AALHG73X/Q0NCwvLu7e2Mu
    l/srAOO24ERlG4Syz88zmr9JBsfCMZRSGUA5ACeAEMseELZuVkXNoJT6AKwF4AJwBsCMzcFY
    wZ3GyrQWiOLZ3yA2o2Td74pMOXvuKHDGC24biFysWvKOB4MX4cy5KaVXA9jtUORVmq6X67rx
    EwDfZO+zZu0r2375Tb5Enucr7r33Xs9nfud3QGBCU7P4yle/Jpz9//4yoOt6MYAspTTG9o1u
    y9hdMDC6cr07k3bbtm2TtmzZsocQUjY0NHT40UcfHctmsyQYDK5IpVJ7zp49W9nW1vbZ3bt3
    p77//e+/SvNGx2DgyEpE5ey2iJ07e49AYV/bOzLYfysmZcOGDXJ7+7Jy0zSdx451hk6cOFHo
    k3QA5p133umrq6tdaxiGs7u7++yTTz41wwCElazTme0x7CAK83RGO2tkgU96J9pqCzgW9JGc
    l9hkr5/a1uJ/hC+yxzWf+MQnWnw+716AbCaEW0opjcVisT8bGxsbsMV4WXb/pvsj8DzvD4VC
    ZT//+c+lSCQGQZARDJbwe/fu8WUymSIAJQASLMlgsvXJXcgfXUy7cgUgXaZLURRkMhk3IeSP
    OI5bYxjGU5TSJwghYQDL6+rqqv7yL/+S+9d//dddzzzzzASl9FsAMgAMls3lbRk/o8Dw2CtL
    ZqHReZc6IwnAXR636yFBEHzxeOJZwzS/DiBmZRAIIVlKqcLz/P+qq626TxJFqW9g6AVN078I
    YNqWachRSj0AVgOoY4erG0A/CwDthy5ny97Qy2WsL8RPXiSrOFdd4zgOpmkKABS2XiLbJzzb
    GybbP4bNeZ/XW2Jr0L6gbb8coPvXzT7anxOlFIFAgEYikY3BYNE/XXv1tvrK0iB+9NOn1OGx
    SR6Aj4HHDAtwjEIQ8G7NVL7B/iIAMpqmjcbjcSiKBEJNZFIJHD3WaeZyuWEADgDF7MesKkHa
    FgCfpzj2m073/U0GSB/84Ae9oih+JpVKba+pqTn2vve976vf+MY3XiOEtKRSKU9fXx8EQahs
    aWn58LJlyxJnzpyZYMF9msUPPLM7lh3h2HkyC5JOlg0C3mUZcVY5wg033CCtWrXqboD7GKXU
    t2dPyS8IId84fvx43PI1lNLs1Vdf7Whqav58NqvfQymRli1b/sLsbOiLBw4cmGbgQAeQ27Vr
    l2f16tVrJEmq1TQtNzo62v3ss88OhEIhK5mnLeaTLqcvWuxsL+KnCABwHCdUVlYqTqdTotQU
    NU0Xcrkcl8vlSDKZNFVVNSilBvJtBHOJy0X2xxtVpi6LL3ozm/qr0vQ/+MEH3H5/4LORSOKB
    qamkWFTkQ1GRdHxqakoGELDOFjs/Gt7lOgFvYY1IJpOZkGU5SQjvNwwKjgOKioogSWL4zJnT
    EQClzNYYlNIcISSN8/v2L4lPugKQLtOVyWQAwOVwODbs3bt389jY2PqDBw92mKb5FQB1LpfL
    cfr0aWzYsEHs7u6+aWhoaBzAIWZ00wC8zFBa77qBfLlRZUGwZWzspVmzECi9iwKZoENRHnrg
    vjs211eX44v/8rWS6dnwMQCHmXGygtlSl9Nxzcc+eF+J3+3EH/7F/90cjsaaKKUZW5DXJIri
    x9ra2q5paGgIJJNJ/dSpU0NTU1M/opT+nBCSZIGyVcK1fvaSgKQLGI0F9Ess7JvhbNk2k1Gl
    BErpRgCbWEDrAiCzgMWqhGQAxCmlPQA62evMsaA3x4yNzvO8aZomB4CnlPI2aiNsWUzTvp8u
    VcbGcrgFamvcIo7iPI52YV9FIpEAgIblbUtL/+B3HsLTzzyHcDQ+AKALgJ+9Lp69z1n8BojW
    2KipwPlqdVbGf6Sr66wWCYXFIr8XR44cwSuvvJYAMAUgyNbEAKBSSrMAsoQQjf28PRHzG033
    /R9yiZqmuQ4ePCgJgrCxo6Oj4tZbb/0X0zSrdF2XVq5cCcMwEIlEVq5fv/6OoaGhx1VVzdXU
    1PBNTU1KZWWlw+PxOHie5w1Dzw4NDfc8++yzvZlMJseSWIV9A/obnNl37PWjH/0IbW2txRzH
    f7y7e3xjJmNi2bLK4qamxs7jx48fsfkkWl5eVgZgT1fXWIlp8li1qmpjUVFRE0s0AIC+c+fO
    po6Ojo/F4/Frx8bG/JIk6RUVFUO33HLLjx555JGfRyKRBEuMpth5zNjs+kXzSYtVgm3VIM7W
    L2PZDrsvopRSc+XKVcJ11127UZLEjayHxG0Ypkwp5U3TpKZpGIZhZDVNi83MzPY8++yznbFY
    LMn2huWL7H2OXEGi7y37oku4jwpV1kgBg6MwUQ37+2Q9zuVyeUzTXNrTMy6GwzrKygLq1NTk
    gdOnTxvI078t2mruNwEcFfRd2X3R3L4aGhoK19fXT5eWllaHw3EoihNLljTSSCQ82tfXTxhA
    IrZkd5btHSsOLvRHF20TXAFIl/niOE7du3cvVq9e7fj85z9/y3PPPUcAZILBoDQxMYHTp09j
    27ZtwZmZmfem02mNOZUGACVOp7PI4XB4CAGfy+WyyWSqk1L6OIBRLCzLZgqyTu/GQCan6foY
    qIlsNotMNhdir83DDHOOZaGSyVT62OnTXcsIoTSeTJ0GELUeB8DBcdxDW7duvXn79u1CKBSC
    JEliR0dH8w9+8IOPDg8PG5TSlwEkCSG8LUO+QBTjYhqMQkPBqBU8AzhudhcxUBwFMGkDvRbl
    yS8Iwu+u71hzY/uyViKJIng+78MM3YCuqcjmVNrd04cjnSdOarr+V5TScZadSrKPaUqpyzCM
    bchXoCYATFFKU2zvwFbStoNvo8AwXdR9xX4Xz0CMjPkKn0kI4Wx/d7HnY/X1EcMwCABF5Hnu
    7LlufOfHj2WSqfSrbB8V2RIOaVsA8Mtkv95S1vDt8E1sTynIV4R4m+OQAYT6+/vjszPTQUXi
    8bPHnqATk5MzbA1qmKMuAlANYAD5aqvA7IpWsBeuCDe8iwOYW2+9VS8vLw9rmoYDBw4AQO3q
    1at/yzCMmYmJCaG3txcbNmzA7OysWFVVdfWmTZvSmqap69ev3yzLsj+VSrkikaisaRqRZcVY
    smTJkN/vf/ill156ZXh4eJYB7DTL+qZtfsl4N+2d+vp6PPnkE9mKCscYITx4noNp6rOjo6ML
    fBIA/cSJE/GamtrOYNDXSggPVc2eZZlwLwBaVVXlWLly5UNTU1M3Dw8PC06nE4ZhiBMTE80t
    LS2/tXXrVuPxxx/fRylNEUIEdi7PS0r8uu+9TXCCFCRR+GAwKDc2Nrq9Xq/b6/UGHA6Hb2Zm
    Jrpv377JXC5n2VwNgFZZWeETRfF3Z2aiN4ZCMWIYJigFCAE4jgfHcRAEgfr9blJXV3ciGAz+
    VSwWG2eJXyspmd6zZ4+zo6Nju2maUiQSnujr65s6d647NT4+rjGbatl8lSVs7LbfvJRgiVUQ
    HcFg0B+JhGlvb5/e09OjMR9jV/60szTmqKYW0Eyn05LL5ZUEQUZjYxEkyeh99tlXTxmG4Wc/
    JzBfJ1zIJy1WvXqzitbb5ZPsr72mpkZsaGhwapomzczMkLGxMTOTyXCvv75fqKmpGS8uLlnL
    80MoKytFcXFR5pVX9oU1TStzuZx+l8vtKyoqKlMUZXZkZORMKBQihJBUQWyiXeyYTXg7Fqzw
    DTVNExzH/cY1dxYAI5immUun0+MjIyN48MEH8ad/+qfi1NTUtadOnYpXV1dzn/zkJ/GHf/iH
    yOVy2Lx5S/Pzzz93M6U0AWAtIcRRUVHBb92yBa1Lm0Gogad/8eyq/QcOrVE17SlK6REAs8wR
    pZBvsM6wYNAu1ftuyTzEdV1/4RfPv7zXochKIpk6CCCCOdUtEOZsE6Zpdr5y4PAtAARd18+y
    g+Jmv+6qhoaGPbW1tcK3vvWt6bGxsROCIJTdeOONy/bs2RP8r//6r2s0TetjwMDikturcZdi
    zSzVQg/L2DcAaAbQyPN8g8PhqFYUOaBruiMWjx+llP4/SmnUyqCw97XYoSgV77/nTu6+u98L
    Sk2AUlBqgpomTNMANU3yw5/8FKe7zvm0pN7EqkxJ1l8SBRAHsMbjcv19fV2VNxyJRSLR2HQm
    m+tlVad+SukYW/ckqyRYWb9CwHTRsprBYJDOzs66AfypKAqrqUmndMPoBfA8EzChtqpPlq2J
    PXAgACRK6QYAu7t6+sS/+aev4My53n4AfchX3DR2TlRCiIR5CusbVWUWk+Mv7O16Wymttkbg
    jQDu5DiuhFJqUkqTAMLsvQxEo1FjcnIChp7DCy+9ZCJPk1pFAIcsy5IsSyIhhCQSyR7DNP8V
    QA8LcDO223y32JQr1+LX6dOnsy0tLUPFxcUwTROHDh1CIBBoXrFiRX13dzfX19eH6upqVFRU
    YHBw0NvS0nJtKpWayeVyPkKITAgRTNPkEokkNzExzU9Ph5Y0NNR+8pZbbtna29v73KFDhzpn
    Z2dnKaUJds4sv6TaA8l3ck8bIQQbNmwAgMTnPve556urg9cbBlVisfChI0eORpgd11mgnDt9
    +kx89+7dx8vKfDdTCm52dqxraGhYB+AihHBbtmzZoev6nng8LrS3t09TSk/wPF82OTm5bHh4
    uLi6unpPMBjsDYVCU/a+pYvpkyzaIABSVFQkrFixwlNaWloUCAQaHA5HsyRJTQAaKKXVpmkG
    eJ53lJaWHu3t7f23gYGBCCHQKc37okgkUqyqWsXZs/3cuXODIISbuzmOByE8OI4jzc3VWLas
    2sdxpMntdrs4jkuaphnPZrMRXdfjRUVFqzlO+Pt0Ouf1+Yqj69eXTK1evaY3Fov2jI6O9vf2
    9o1NTExEMplMkgHvHCFkUV90Md9/tlb0j/7oj3bwPP/56urqTHt7+3gkEj387LPPvtLf35+y
    VX0sf6QiX0G1AAJ3zz33lBUXl9wuikJNU1M5FEXInD7deainpxfI99lkmG+x6KvkAr7ovIrM
    IpWsd4RPsv7ePffcE6ivr38fIWStaZoSpTSraVo0kUiEx8fHU5qmiT6fD4riRH19DSRJ0MvL
    yysbGxvrAoGA4nJ5REmSeEEQ1OHhgR/86Ec/fpRSGmNVVuumF9sfCZfTyCyGLK1eCswPhyp8
    oxdVcnm3gSW2obOmafadO3fOSCaT/Lp16/DJT37S8xd/8Ree5uZmZLNZKIqCgwcP4lOf+iRv
    GPqKl19+OWIYRphSmhsaGuLj8bj74MGAe+mSJvneO2+RNnas7Hj4Z08u7R8c6TQpfYZSegxA
    iFVDrAxEFhcY+vhOWsdC8EwpTYUj0UFJFBW2P1zsoTp7bTlCSI5SGgtHoocB4mAG0sse1+Tz
    +e7cvHmz99ChQ6nh4eEfA3jZMIxlBw4cqHjwwQeLA4FA3fT0dA0zSpRSajCqhBV8X9SL53kY
    huEGcAfP87tcLteyokCgvLa21l9XV6c0NNSThvo6Ultbg+effwF//3//kc+p6oeZAdBZUD8F
    YJQQIiuKDEHgYegUFBSg+cwg4ThwAg+e52GatJjjyH2sZytrmjQOYAbAEIDiuppK/9/+ye85
    Aeru6u6vOdXVvbarZ0AbGBqNhqOx0ZyqdQE4DuAQpTTEMjdpGwC/aOIGlFLMzs4C+ZkI9Vfv
    2LZtdftSvPzaAePA0ZNrTdP8G0pphDkfy37ZqRfWWdvu9/v+fs/VVy9vbqzjOk+cRHffYCCd
    ydSzipzVZJ5i4OA8Gp+Nh89RSiW2/7IF9L8FgiF4G6u1lFIr0eQRRfETu3btum/Tpk3QNI2G
    w2FMTk6aY2Nj2ZGRkZzL6fRyHBAJh6HIMrdqRXt1bVUlV1dTSeprK1FdXopkKom//qevKn2D
    I6sYqI4DSDBvbFdaetdWUN4s+/pWeg3ebUk7u7BLT0+Pls1me4PBYM7r9coulwtnz57lCSH8
    0NAQqqqqUFtbC1mW0dfXj2g0VtTevswDIBWJRKaj0WiIUmoWFfm99fW1VYOD44HDh8+429qa
    tq5atXpFfX39of379/+4s7PzrO2sFt4o9PHvRJBEKUUul0vJsjBAiKnkcjlimqa70CcByMXj
    iajHIxwyTVNJJBI6A1Fkw4YNzeXl5XeNjo56y8rKUuFw+Ec/+clPXlmxYkX7hg0bKrq6uooJ
    IXWlpaU1oVAow/aXyXxSxu6TLsY6feADH3BXVVW9lxCy2zCMtmw2Wx6NxgLxeFyJRuMkFouT
    RCKJ2toadHSs4vfs2f2gYRgZgOiUmql0OjM1Pj42qmmaZBgGOy/2swEQQmGaeUDmcCjFu3df
    c59Fk6LUjOu6Np1IJIfcbndxMpnzHzs25gSI2+dzVAcCzrUeT1Brby+JtrevGEulkl2Tk5Od
    r7322qHZ2dmQVX26FL5okSS3J5PJrenvn/K6XB5UVBTftHnzpi/29/c/zypFGVu1T7fbhrvv
    vtvf2Nj4+XA4+uDoaJ9bFBXU1JQLXq+vTpbliVwul2XJAx35Kv+iAMmKmzmO4/fs2eOenZ2l
    R48e1S6gkngeYLycZ8v+d+rq6jZlMpkvdHV1lWazWUiSRF0uF3W73XpNTW3K4XCKhkGpKEpE
    khQQIrhra5e0ZbMaiUY1Mj4egapS1NUVIxgs3uj3+49Go9FJVkDg2Wu3EtvvLoBU4GCIDQzx
    LPAQYeu5KAh67P00CwzqO50udgHZzoHe3t7k66+/7vvFL36Bjo4OfO1rX8PSpUsxOjoKXdex
    efMmXH311dize7fw6KOPFr/44ktm5/HOeCgUHpuZmTk3MzOT6TrX3ZJOJto+/P67uPHxce/o
    xNT2bDbXAuAbAF5kwb7VpG8HmufxNN8J61jYP8IqLBOhSCzMcVwt2/hB254QkO+R4AEEsqpe
    yeTSKfLULBHAnqKiosa+vj6zt7f3OIDDbH/p2Ww2K8synE6nwioKEzbDlrmQgfp1L8MwACAg
    iuInb7311o7bbrsVS5csQWlJCQIBH2RJAkAh8Dw6O4+jsqKsYtXytrskUYRhGojFEzhy7GQ0
    Eov/mFIq6roB0Lwjyi8dnc8OUkDTdFRXlLp3b1m7XFEkqJqGVCqNUCROewZGcmf7hlWe55TK
    8hKUFhdhRWszbt17NQlHotLI2ETpybM9pfsPda59cf+RqzLZ3D8C6MR8rxOwsAn7Yu6DlGma
    h3mOXHf/3beJfrfCHzp2qsjM07/A3nuuoJmZseuol+f5B+66470r//R//wE8DgljI0P447/4
    24of//TJTQBeYxU0jRCymDMiBdUYEcBt7H6KUrrPClRs1IqsLSv+dlOH/B6PZ8XHPvYx3Hbb
    baAUxDB0JBMJPhyedU1NTLhSyTjaWpaCA8XXv/zPROAI73U7oEgCYOqAqWNqahqVZSVK3+BI
    DfI8cAnzIjHqpUggvE0JO2KbB2QfFWCB5rl+C/Z+nzdu4d0GluxU30gkMuZ2uxNNTU1ya2sr
    jhw5gqeeegq6rkNRFLz44oswTROjo6Po7+8nvb29UnV1jdTc3OQrLS0tjUQiA0eOHOkJBAIz
    q1at2ej3+51Op6zrukZkWW4rLi5eBmDSLpHNEngWFfwdKeCwiBABPzU1Nd7Y2BgRRb4mlUrp
    oigGVVWd80mUUtXr9QoAAg6HWGkYppJMJimAgCRJYnt7+zUcxzUEAgGTENL50ksvHVFVlczO
    zmqU0pxhGDAMQ1EUpYQlciizb2krTrqYe+Azn/lMEc/znzh37tz6rq5zCIcjSKXSyOVUGIaV
    qCQoKSmBYaBCVYW7TJMHIRwkSUBNTVlU17Wf6Lou5tnPC1tMrFxSnm5HYBiCO5v1LjcMAp7n
    wPMEksTRmhrknE6o8XhOSadV5HIGYrEcxsYSRJZFye2WS30+R2lxsXfN0qX+HVNTk/84Ozvb
    SSmV2V666L7Ivg8IIfjt3/5Ut8PhGopG0yuGhzMIBNxuXdeDLN6glFLRxm6wK6Oivr5+YyKR
    vOell/a7JycjcDh8mJ1NiatW1a9tbGwMnz17dpj9uRQhRMT5jAa7SiC5//77a6qqqn7PNA21
    oaH+R4888uiwaZqwiaNYolOXTdzjjbClIAht4+PjxS+++CJyuRwAEJ7nicPhlNxun+T1BpDN
    atB1ihMnetDTM8GZJoFhcOA4CYIggedl+P05uN1imSxLVex1Cra47aIntYXLaGTsMwGKAbQA
    aANQgTxPXqWUzgIYRL6nJok8/WWxJs931cBUe5kVQC4SiWgTExNQFAXLli3DypUrQSlFQ0M9
    1q5ZA0opXC4nOI6gZenvkw998AH+4MGDgYcfecT/7HMv1ExMTAwCmHnp1deDJ890+WLxRDSb
    zU2xDK8DQC2ldIYha5GtZQrz1KjzVGPeYetFKKXVHMc9ePXOnRuXtbVJP3n44VvGJyZ8AI6x
    1yJRSkUAG6urqu778IMfLIsnEviPr3/jtkQiyTHAExkaGjo3MDAwCuA5tn88ANqrq6uDgiAg
    Ho/rzBh52ZpkWSDIXwqApCgKstlshuf58I7t23HvvfeCmgZM0wTN0+JATQrN1BCNRrFl4zr8
    n//9u1BkGdQ0MDo+gY9+5gty5OSZcpNShVIzXzli9q/w/4aho6aiBA/ecxOCfg8MQ4dpGlBV
    lTz+3GvKn/3TfyqGacI0KUzThEkpeJ5DaTCA0qAf61a1YWVbMw6fOOvMZHOlAMoYeL2gBPRF
    2gIGpfQXr+w/dPO3f/CTtZNTU9ANI4n5npoc5gUWLGqDZRfcXo+nZfu2rQgG/MgkYygtLsKm
    davIz556JqiqmpcFHNYssrnkC1O2sygLHLOR2wN+3xc2rlvb3ts/cHXfwOB/U4r/Rl5RUcNC
    yeN3Au1M4HledDgcbDEBQRDgD/jh87pRVV4KU1dhGhrUXBZVFaWghg4OFBwMGGaeqqmqOaQz
    WbDKWQk7DxqlNMOqiHYA8W6sHM0F7Fu2bHHX1tYEOI7zG4bpTCYT+vT0TLanpzsdjcZUW2Jm
    sR4s89322i2f5PF4RK/XW5/L5ZSpqSmMjo4iEolA0/KJ2FQqhe7ubgAEHJenTYXDEUQiUfT0
    9HLV1dW+5cuXrdq+fXvj2NjYWH9/77jD4RBHR/tnUqlU2Ol0qoSQwPbt269KpZJjIyOjvTMz
    M5OUUoUQkmDnUC1MLrxTQKctuUnuuuuumvr6+g9PTExtCIVCUmNj/S3XX3+999lnnz2WTqfT
    ACSv1yvefvvtm7xe332HD3eWyrKE1tbW26anp7mTJ09OjI6Ohpuams5pmjb66quvPjc+Pq4q
    iuJdtWrVck3TigzDAMdxejQa5ZmvsnySjIWiBRdlUdLpdNrhcISHhoZw6tQpABzrG+LZ+03A
    cTwURcbUVBQHDpyDaeZ7i1wuBzZtapZzObXMMHSF44htn1l7zQ6WCDIZA0NDGWgaAc/n6Xc8
    z5PSUkFZskRUCFHnAEl+jwK5nAFVzSESURGPa2hv9zqz2WwZgDIGSi+lL5oL8p966umhW265
    +WctLdXNo6MJhyBAHR+fyLHErdXfaZfp1tjXRBTF5nB4vGh6OgSOk0AIj1AoDVWljqKiohLk
    VXcFW5LA7pNgi0X4bdu2BcvLy38nGk1+OJfT+YaG5i033njjlx977LETlFJLGCWNhf3Ub2fS
    jlBKJcMwOGvIspUsTqUyUFWKRCIHnpcgigo0zQSlKnheBMeJjKIpgudFiKIAXde4XE4tYnGg
    yfx/+lLEbMLlckQs6A0AuFaSxDvKy8o7mpoayirKy2RZkkgymaQjo2OZ/oHB6dlQ+IRpms8D
    OMGCfquMml4kU/tuyNZZE8pLCCE3rV+/3r937168733vm+u9opSCgMDtdgGUBauGCUopSktK
    cNN7bsRVO7aRV1551ffNb3175bPPPR9PJJKhqenZMcuAchznFUXxasPQN+u6MYW84tshQkiI
    Hd6kLWtnn6dE3+6+CathlO2TIIDf2bFj+61/98W/kZY0N2Ht2tXBv/6bv93b3dMbAHAQ+dlI
    u5oaG/Z+4Q8/77/rjtsRTyQRi8ZqvvVf37nWMIxnAPSYptnPqgUxBsa31dfX77zlllsc+/bt
    o+FweIyth8wydEJB9viiXqqqAoCazWaH+wcGYOj6fO/QPC8clO2BcCSKQ0dPAJRC01RMTE5j
    NhQWALRLoljscrlsZo8WfE4giRKGxqbwjR8+hqbaShQHvHC7HFAkEdF4EqZJIQoiOEIAChDK
    /jbNAzWOAxLJFFRVMxmILLM5gYzNEF/sM8MBCCWSyYH//MGja1xOB5ElqTGnqtsBnMW8upOd
    eqIzgKOmM5mprrNdyFy3CzwBEskEenoHoGl61uYsdBSoPrL9x1NKG5FXCKzkCLnxlhuvW/bZ
    TzyIZ59/sfiL//K1B6ZmQiryQ53TLNCzHPQ7ATRk0un0TG9vb2smk0Emk0EkHEZvbw+6u89h
    aGAAdbXVuPnG6/DYE0/htf0HqCTwCPi9pLjIh4DXDbdTQe/AEAZGxqz32IGFcvLcu7lqRCkl
    TqeTu/XWW8urqqq2iaK4Q9eNlYZhBFniRW9tbZvduHHjuZGRkSMHDhw4Mzk5GWXUoIzNjuYK
    A/t3asJukYoIeeCBB9YrivLBgwcPuicmJmAPYKy8Zt4+c7bANf+5qqro7x/AxMQkaWpq8K5Y
    sdwF0Oj+/ftH/X6/snnz5g5RFKVsNsfpum7KspzKZjMnXnzxxR92d3f3sTl3SQa2M7ag8m1X
    X7X/TUopef/73x+sq6v9zMjI+C3PPfe8FA5HMTU1E9ywoWPvddddV/TMM88c1DTNee211+5y
    uTw3vPbaYV9XVzckSYAsyzUbN268dmpq6hfPPvts96FDh/pSqVRUVdVYTU1NxY4dO7Z7vd6d
    Z8+eddTU1JiRSGR8fHw8x86b3SeRi2Vrrfdy9+7danFx8YjfH6Acx5M8EJ6v/Fh/juM4KIqE
    khKfBWrgdCpwOmWhpqa6XZIcJaqqF4CiebAEAKZJ4XDwqKtzIpUyoWmAYXCglIMo5v+madpH
    ThDbc81/FEUOhqHR2dmQ53L5Iutl9Pb2mpFItLuioiLudEoOUSRKc3Pz9uHhIXNoaHiW2UgT
    C4WyTABUVdUZj8eT8vv9vlhMBccJ8HpdEEVOTyTidkW2BYIDlnT+zTff7F2yZMk6URTbCCEt
    mYx279GjvbKqcli7tnnD0qUtfGtrzz92dXWNsnUA+x0q3n41PKpp2ozH48m43W5HMpmEJMnw
    en0oLS1DcXEJZFlBb+8oAA7Lly+liqLAMChRVQpVpTAMAkIElJS46PR0XyyRSAhsva24jb8U
    Pkm4HMaYBR31hJDPNDc13XPrLTcVX3/dNaSxvg5ulxMcAXLZDIlEws6uru76nz3xdN1Tv3hu
    w/Rs6GkmwzzFAtw4C/ItoGQFNe9IzrItA0UopQ4A93Z0dLznc5/7nFBSUjL3/TwdymQfLYpU
    3kdQ29cupxPXXXsN1qxaSX788CO+f//qfzjPdfdYh7JMlmWhtXUpNqxdjUQ8Tp95/qWNM6Hw
    Vymlr7ON5GDBnAU47fSGtxVsFqzT/StXrvzAn/7Jn8grVqwANQ3cc9ed8Hk9zj/+0z/fcvrM
    WRlArLqqctsX/uD3vXfefgs4jqAo4MdnPv1J0tPb2/TyK6/GkJdJTzNHUymK4o6NGze27tq1
    Szpx4gR97rnnRgGctDlmHefzdi9qsMv2qQZgzFJxcSjKAtIjRb6XpKamGo88+lP8w799AwLP
    g+Pz6knLlrUJO6/aUd/e1oL1HWtgUtP286yeRAGTmti8YS2mpm/BuZ4+nH3xEFKpNHRdAyFA
    KpWB1+NG29JGOBQZ89WohYBrYmoGOVWlAHzI0xEtvnQcl4iKyIzeLW2tLbsffOADpKmhFq8f
    OFDzn9/5wZ3TM6GnAJxm62lVM1LIS4CaABK5XO7R//re99cbulq6tKkOR4914uHHf5GklA7Z
    bIe9Om2AiQ4B2OxwOP50eXvbhmAg4DjX3c3NhsLkn/7tP+jJ02e1RDKlA6hlmUNCKdUKKipv
    d6U6lUqlDn/961/ffPToUX52dtYYGRnRp6Yms6FQOK2qqrh2zaqi9tZm/pvf/p55tPNElFXr
    eY7jBEWWeFmWiKpqRjKV7mO2dzFhinctOGpra5VvuOHGbbIsfyIaje8YG+svmp4OkWQyA9ME
    FEWGz+dDeXnxlvr6ppsrKysPHDx48LEjR470MMEBy44WVubf0SDJLgP/0Y9+tNbr9X66s7Oz
    5fjx4+eBIyuIBs4HRwuBkoaurh5MT8/ya9asCl599dWuSCSSzmazkiRJMqWU7+0d5DIZ1bN2
    7Yrg8uXtk+FwmACIhMPhadM0Y2w9Ldrd204LsjfD796921FXV/fB2dnQ+/fte1menp4BITxO
    n+6CququLVvWb9mzZ4+cSqViZWUVWw8dOuHr7h4AwCOb1XH48HGyZ89VTVu2bNnw+OOPH45E
    ImlCiLJ27drKjRs3XpVOp1vOnj0rlZaWUpfLNfrss8+eyOVymUV80kWvAjz//PNqR0fHuNfr
    MQRBEHR94YhF69jEYgk4HDyWL6+e6yeilCIcDgvpdK4+FBrD1NQs5ml151csp6Zi6OkZQiDg
    QVGRAlEUIQj5v8XzJlRVQyyWgWHQud9TWIVSFAHZbMZMJOL+y+iLAIC+733vawgGgx/p7u4r
    6e8fgyy75La2xqt27rza98gjjzwbj8enKKUq8wVWMloHQLu7uw8vWbLk5S1b1t0wNDTNybID
    DQ0VRjg8NdbfPzBrA0cWNU5nvgx33HGHf8mSpZ/OZLIfmp2dKXc4HLwsS6S6ugxut8twuwXV
    MDSf0+msARCz9UPxFxNU/xpJBjI9Pd1VUlIyumfPniW6rptut1t3Op2aYZjpbDajBoNBfzZr
    uAjhUFbmzQ4MDMQFQSBOp5P3eByCJMkczwvIZmfjJ06cGDIMw8Tioz/e+QDJ7iBsdKk/vWrH
    jnv+8A9+X9q8cQMkUYBp6HO3yDvhciqoKi/D+rUryZaNa8v/8Utfu6urpy9AKX0MwESB2tR5
    M1DeiU7J1le1tby8/COf/exnXR0dHey52vtFCoASKKjJlMlYk4nJlMpKioP4yIceQPuyVvFv
    //4fy1986eW0YRiz2WyW9PUNuLdv3uB86DMfJ1PTU6XPvfTaLpZpmWSVl1ksbEq3eKpvZ1Bn
    l3a+ury8/FO/+7nPerdu2TxXWeE4gr3XXQtQU/n8H/3x+pHRMeOhj3xIufU9e0FAYeoaKGei
    ubEen/7kx4W+vv728YmJKPIyxVUej+eq2267rbKqqor87Gc/y508ebLHMIyjyFPxLINmAcfC
    eR0X+zIBTM9MT2vxeFx2OhSY9u1MKUxKcdvNN2HT+nUAKDhCmFwqB1mSiMvpgCJL8/uEA2Ay
    KgMIKFvX+tpqfPqhDyGVSiEWTyCZTCKdySCbycI0TYgCh9LiIgaQbFaGfW5SitHJaeRUjSLf
    /2NV45y4BFRESikURaHZbDbgdrvf85EHP+T/6IP3A6aKjWvaEQmHA1/71vdXAQix9yiOvDKb
    1Utk0RqeHx4Z/e8vfeU/PiYKAp9IJEd1w+hm73ehM1JtAUhQFMVP33PXHbs++9sfh8ep4NCh
    g/jj//NFPP50/zCrao+x28HuFOuHetucUcGlAXi4s7MTnZ2dZZifIQH2fBtFgffrqsonUymT
    rWWMUpqnPaQzWiqdsWgL05ift2YpM72rJJoLwdHatWvla6655nbTNP/kxInTS0+ePEsikRgo
    JeA4gdE5BPD8LHp7x7iyspKSZcvqb9i2bVuTLMs/3L9/fyelNEIIidgyl2lcwgbxi7UGlr1d
    v369UlJScs/Y2NiuAwcOENYXYHv8QiA0X0laHCQRQhCNxvHaawfR3t6qtLe38iMjI7Ovvfba
    UGtrq7+ysrzi0KGTXDSaEOrrG66pqqrZmMvlhp988onvjI6ODmOhFD19O5N2BWwKrqOjY1cm
    k/nk/v2ve0ZHx5g6Wz5Z3dc3AEI4Zdu2DetdLqdx9OhZR1/fCDhOACF5dd5oNInOzlPCli3r
    2pctWxY9fvz4YEtLS9XGjRuvGh8fr0wmk6S5uTmnaVrPCy+8cHRoaMjuk5I2n2RcZJ9EKKVm
    LpebcTpdmiwrgq6nz6sgAgS9vf2YnJxheyH/75QSGAYlum7ANCnbC/yCn7XkvgGCZDKDU6eG
    IIoiZFmCKOYpU4IgzO2xbNaAYZi2v7MQrDscAuLxGZrN5vzMH10yX2TbAwCA6urqTTMzoXWv
    vXaYS6V0KIoHus7xGzYsaampqRk8ffq0hvysqyghxIm8oA1HKTUfffTRmZtvvvmrbW3Lmisr
    y1o1TUsMDPT1vvrqq33JZDJjYyBY/sjyY2hqatqVTCY/8corB4unpmJwuwNYvboFTU3lmf7+
    3mMnT/YPTE/PjE9NTaWQV+9VKaXCO8EfWUmGF198sX/nzp3/HAgE1qdSKXl6etoYGxszpqam
    BMMwAu95z3s6AM4lyzJCoXDmhRdeGFdV1cjvFdmUJEkVBD6Xy6mRWCw2ifMHKBt4twAk+8JQ
    Sp0APrx506a7vvi3fy2tXrkChqHDMPT53gtKmTSxDmoYcDkduO0918OpKM4//su/3zM4PJoC
    8KLtTbcrR5nvRKdkA4iglAZ5nv/wHXfc0XDTTTdZkt9zAMmiWOEC1aQFoIkBJZ4j2LZ5E8q/
    +Ffkr774f50PP/ozQ1W1oXg8Ht3/+sHmD959G79z6wYk4/ENoUh0Q//w+IhpmrOsbEttg2V1
    W/B0WeUgF5kL1CDL8mff/7731d9+261ztDtKKWCaAAGuu/YaxGJx6a+/+PcIBHzgeQLT0AFC
    QJiK166rtuGuO25zfenfvtKuG4ZKCFl6/fXXV1ZXV5Ovf/3rkampqdeRp2lZtI4Eu+MFGeFL
    6aDDs6FQKpFIyBXlZczhEsAkMEGg6yqcTgfq62uhazp0XcufGz1/x+MJRNg5MnQDpqmzPqb8
    PiGg4EieGkEAcASQZQkOJQieAzhCwHN50MVzmDuDlBJQk+SxFgVMw8TkdMh6n5zI96M4fhWH
    9FbmChFCkM1mAUBxOBR/Y2MDBIFHNqXD6VBQV10J5gQ87Lk4KaUOJrZgDbkFpVQDgOXL2oSx
    8YlcJBqzgA1vc0TnUaQAVJSUFK+75+470dzUgFw6gau2rMf2Tetwrqc/xgCWlZ1zsgCGxztr
    qB9liYFvIS+uUMKyrUWs6lVbXlrK+30eeNwugnketzVvKmU7DxEGiiMsGEnYsvyXKoFwqWwy
    AcDt2bPnasMw//zAgYPNnZ2noGkGeF5knHd+rteGEA66bmJiIoRkUuVWrWpqXbu24/2RSISc
    PXv2LGvKtvee2Yc5v6N7Y7du3dqqquq9R48edYZCoQVBrb0HZDEw9Ea0O03TcerUWWSzqrhm
    zYriVCqlDg4OJtav31haUlLMhUJxQqlQ4fd7KlwuQeF5vhhAlJ1Xa4yAWli9v1zrWeiTPv7x
    jzfyPP/Zo0eP1Z07122zY/PAsb9/GLKsSJs2dSBPM+PAcQJL7pkgxMTQ0Dhqaibcq1atXj4x
    MaGtXr16yczMTGUikSCtra2RoaGh11955ZWzsVis0CclCnzSxZ7LRzKZTMjhcKRlWXKkUinY
    l5lSCp4n0DQD8XgCHMeD54W5PiWe5yHLkq1vKX9+8vLe3FwFcr4PKc9eNwwKw9CQzVp6AtZj
    5sGntc/s1SxF4TE2FoGmaYW+6JeuHr0Vf2TfC4LA+1OplJTJZMHzCggRkE6rMAyIDofiY37J
    xfyCRf3iCCEmpZRqmqplszkxnc7CMHITzz777NFEIkFtdvc8RgPyAgcbR0fHiwcHR8HzTqRS
    KgYHQygt9eknTpyc6OrqirDHK+wW3kn+iBCCoaGh7Le+9a1fEEKOUUpLmT8KAgiUlpZysqzw
    mUy+70iSFPA8r1FKVVVVdVVVrSphwuaLrI9xLGRDXVQjIVwGY7OmvLz8/Z/5nU8ra1avhqGr
    c4bDAgGw9V/key8MEAC7r9qMnr7bXF/8569uz+ZyE1bGwxbcvyOHFdpBBnPKW5cuXbr7gQce
    gMfjmQdHpg0UYSFIOh8oLV5NampswF/8yR8RhyJ7vvfDH9fmcmqof3BY+7//76u8Igpk9fKl
    Yt/AMAaGxzMAZEqpF3kqUobNELAywwuoZJfTIbH31QXgwW3btm5/6KGPwel0zvXnWK+bmvlK
    0223vAeTk5P49nd/iCVNjdi0cR2oCVCSXydFkfGB++7BvldeLT167Pjqmpqa4lWrVpH//M//
    TE9NTb0C4JRt76Rthy9hDVG9DABpZmJiInXi5Imi2ppqHO88jt6+PsRiUSQTCcQTccTjCaRS
    SWQzWeRyKnJqDpqmQpsDTPY9MV/2obbPCQDCEQg8D0EQIIkCJEmELElwOmS4XS54PS54XA74
    PB4U+b0IFvlQWVoMp1PGmZ5+nOsbojajXTic7y2dvUWULBc4Ifv3OY6jpmnGYrF437PPPrdy
    zYo2eBwSBgeH8MqBw5QZRZMBE4HddpBCAfjcbtfqzRvXk2dfeMkcGR2zAg8O8xxx6wzkbEkW
    auiGlk4mQQ0DoCY0TUWGgTYGMqxhfpaq1CWZ4v1rnClroGIGQJxVZk0bGOQppVx3bz+4fMQi
    2rJxWQaErJlJs+zrGIA4ISRW4JDeLeAIAPCxj32sShCE3zl8+FjzkSOdMAzKgBFnC/i582a5
    pNMqTp0aJuvXL2lct27d9UNDQ5l0Os1joaSuXXH1HQsSi4qKRLfbfW1/f39rX1/fIuCIe0OQ
    9GbVJNMEensHQCnEtWtXVE5OToQSiVhyxYpmB6UEhPC8IMDs7+8bnJiYEFklQCsIEFW8Tcp2
    FnC49tprXUVFRQ8ODw9vO3LkKDRNB88LCwJ3y2719AzA7XZh2bJGRKMJTEzMAJiPcwxDx+nT
    Pdi1a0vZnj17VrtcruKuri6ybNmy9JkzZ17et2/fadM0rbgmbUtSXNAnXQT/TAHQZDI5W15e
    niwrKwvG4wmUlZWjqKgIiqJAURTIsgxZliEIIgRBgCDwrMLK2QARsVEyic2ekwVYdy7koxSm
    me85Mox8v62uU2iaCU0zoesm8v0nJrJZE4YBeL0y3G6RxmJxjVJqV6wzftXqml2E401iOZrN
    5vqLi4tjlZWVRTMzCUiSiIqKIAgxszMzs9awcaHQJ7G+WDQ0NLYlk+mKcDgGn8+h5eXS5+xH
    miUIMjZbbOZ/3tREUTQFQebmxQokUEo5VVV9mFduS9kqsW+3MEPhPtMBZNk8PpH9uw4gU1ZW
    VuJwOCWn0wGHQ4HH45E8Ho+ZTqetRFyS+aIIYzuEMd9yE2f03IueQLjkFSTTNCUA12zduqV+
    1+5dCxvRbYH/gq8x/+8Cz+OGa3bip08+U3b8VNcKtigapdQaEGbfTO84h8zohS4Ae3fu3Blc
    vny5DRTNB7f2fqPzq0kFX2Nhdck0DVSUl+GPfv9zRNM07w9/9AgJhSOz3/3vRwEAPM8phBDD
    MM1ult0oxkI1Jt12GC/rgSqoHu2oqqp636c+9Smpob4ehqHPr4kNAFDThCiKuP9996Cvvx//
    8m9fRU1VBSorK/JrSzgYlGJJcwPuu/tOoauru3Lnzp2kp6eH9vX1jbEKAIf5ZsoUc0JxzPe4
    ZQoc9EUPYCmlQ6FQ6Bd/9qd/fmdPd493YGAAQ0NDUBQFTqcDHrcbTqcTpSWlcDodcCgyFFmG
    JEuQLGoCz0Pgbc6JradpGjBNA4ZuQNN16JqKnKoil8shm80hk8kglU4jnc5/nJyaQTKVQiab
    B2I8z+OaHRuhyBK++/BT9Gx3f4oZp4TNaf9KlTarOZzjOEop5WzzG+YCIeZQkqqqfvd7P/jv
    ltHR0Zb62iq+8/hJuv/gkQkA/baqJ4d5mpNVPTIB1JaXlTZ1rFmF5154Cez9tJp4s7aByhkb
    cCAAZqZnZk7++1f/o1kSgJKAFy+89DKef/n1DDPMFhCz5mWpBQ7tkp+ft5D5NG2v15JCzzGg
    tM3hUOq7e/vw/776DYTDEeJ0KGXpTFZFfi5Wmp2FMPL0uhmWrbMoqCnb+TDeyvN6uysptr/P
    BYPBq2dmZrceP34cmqZBEKTzANHCwJ+bo1RlMhp6e6e5VauqljU3N/WcOHFSZXPTNFtQf9Gn
    uV/sdbj55puDhmHs7u/vF1Op1KLZeivb/8tXkuZB1MDAMERREFetai86ffrU5MDAwGhZWRnf
    0FDv4XlBHxwcSLrd7qpoNGowcKAvAjQv+1pafROrVq3akU6n33fw4CEpGo0tAEeF/VmmaeL0
    6R4EAn6sXt2GZDKLZDINQqwqUp5q19s7LGzevKby7NkzpKioiCaTydGDBw9OmKZpyee/kU+6
    FEk72tXVNVhSUvLMjh3b7gwGgz6fzw+/3wfDMKDrBjRNh6bpSKcz0HUdum7CNE0YhqV+ah8z
    QRYB0HkQxfP5CpPA5vMJAs8AlwBR5CGKAhwOif07B57P76PJyRRMk6CiQqZTU4PpgYGBsK3C
    bfdFv7TPto1ygM2fENtYFusjOXz48MGOjnUPb9++4e6pqbBHkhSUlvqzZ86cODc2Nha1xTIc
    CkQ1li9fLrvd7lUjIwNOTdMRCLgMBoas5EqqYL7g3GuJRKLHamqqoitXLisaH4/C6/WjpaWC
    hsMz4YmJCQ3zUtca80mFVSj6Nvsji7GUY3sZlFKd4zi1o6Ojcu3atatlWXa2tNSB43g4HJJj
    9+7d9YcOHRrs7e2doJSmCCFR5oummW+KY2FbxFy/18X0R5cMIC1fvhwnTpzwyLK8duOGDZzf
    64Wuq+dVjOZA0lzgb6uimAYqSouxenmbePxUVw2AcrYQSSzk5xK8w/qQbG9OTVFR0ebdu3dD
    URSYhsGaDzmYhg5qmuCYgV20YrQIkMyDyPmvTdNEeVkJPv+5T5NIJOp58ufPapTScQBOwzAt
    6cklrPR7wubI9QKAdNmDPOZcygVB+Midd9xRvYcBaTs4WgCqGUjy+734xMc+jM/9ry/g29/7
    IT77qY9BlKR5cC2IqK4sJ0uXLiVLlizB1772tRyldJqV4yVboJdkWfFogTO6JJlgS1aaEBKm
    lP7bmbNnJ7/4d3//gTvee3vN33/xb0lpaclclUfgOeuH8mtimgVVRmvP5Kl1c1+b5lyVFqZV
    bVz479bnuqYz4JRFOpNBOp3Gud5B/PfPfk4PdZ4yEsl0ihmmGVsWp5BqZf4S7zuhlLpN01yO
    fF/cjK16ad1Wp/CBSDT65z99/Ml7RVG8TpZlSdW0QUJIiFqb5HyHZDm5FUuXNBfX19ValBFr
    TgJlz7tQGdMCSDWCwNe8uv91nDx9JudxO/WJyelsPJEcQH5Ab9ZW6o+ytUjawOKCKtkiRpq+
    xWC+0IlfaJ6c/eYKspiwMtJMmn3L8va22+654zZX25IGCMREOhHF2XPdroeffGHJqXP9JoAZ
    Vl0upDOkbUCQst8v25xz4Xyg86a4v91g6aabbnIRQq4aHh5xRyJRcByPeQGCxatHhcAgEkkj
    nTZc1dU1LSdOnBxHnu+fZoDbHty8o/yR/XkUFxe3JRKJ9uHh4bmtxPM8HA4HHA4nA4NZNq+E
    /IrVpPyW7O8fhtPpklpbW0snJyeN0tJSZ319g0/Xder3B4IbNhgr+vv7Dr/22msvZzIZ1QY0
    rQAvh8vUZ2zfpw888EC5KIq/dfLkyZqBgcE5CvRir9sCk6qqobPzLK6+ehPa2hrR2dmdbyVg
    Ntc0DSSTGUQiES4UCqG1tTW7f//+mXQ6LRf4pMQFfNKlAEjm0aNHw9Fo9Mtbt26d7OhY+4H+
    /sG65557Eel0Zq7Kw9qxbJVWfi5xMH9O+At8bX/shT+3wJMFmgSBh8/nQG2tl5pmyjxy5NXk
    qVOnZpLJ5EXxRdZecjgc3Ic+9KEGQRCCjz32WNfAwIBR4JMMQgh96aWXYuFw+N9XrVo50dxc
    fY8oiktCoVC0s7NzQNd1e2JkLmHHkn107do1QUqxcno6BK/XA44j1GZPdVvCLm17LXTp0iUu
    RVHaZFlxrF+/wkwmU1kAZiQyO/Pyy/t6M5mM9XNWlT/K1iPNkmJGgfst9COFB2oBk+hC/uhC
    ic8CkGj3SaItMZ0DgE2bNtWuX7/hnsnJ0PJTp4aJYYiQZTc8Hi9fUVFbvnt3QNE0LTY4ODiJ
    fJ+vRfkO22J/a905zM/pM22jOygW0St4q/7okgGkEydOAIDicDjKq6uqACwM/EHNC/ba2D8X
    RQHlpcUEeYnhYstpU0pdFxjy+E4BRlZmYl1zU1P9mjVrAFDouob+/j68vv919PX3Q9NUdKxd
    i+uvuxaSJJ4Phi5UTSpYT9OkqK2uwu9/5pNkfGIycOz4SSsz4a2qrJS2bd1c1tfX13Li5Ola
    VdN+wgI6a36MXYXJvBxrZAV8pmkKAG5Yvnz5rgceeIA4nU6YFrUO86Aor842D5pMw0RzYwMe
    fOB9+OcvfRmbN3Rg+5ZNMJFvjI1Fk3j8yZ9j9Zo1OHfuHEZGRpLstbmYATGQp1jZ+ywSi2XH
    L6ZTtmWmNBbo/ywWi2V++rPHPnHVju2V99x95zyYMQuqq5b8u2nANPIf52co2c+TObc/COzr
    iHyVyZYX43kOTocMhyIh4Pcgl83h4SeexasHO42cqsbZusyy0vasrcSdwFukIi4iL3wjx5E/
    M036LID/a6MIWFQJlQ181ZFXGSz2+/1bW1tbSl5++RUX61fQbO+R3ThTAC6O41avXN4uBvw+
    2PiHGhbKsGZt2TaTUhrkOO7jy9uWrglHourQyPg+VnGUbZm+OANK02w9IsxY59jvsAw1sfCg
    zeEYi1S7+QslJQqUQOfGJSA/v8kCQk7kue+WBLfdHho2sOkXRXFHVVWVa2RsHEeOHkM2k4bA
    EXjdCkqDfl4UhXJN0yXk+xStnxWY7XXaqsz2oXw6CyB02+tbTHnLfLvBUlFRkVvX9SWzs7Mw
    TQpBmA/m8kCJK6icFIImAl03kUyqnMfjKZEkqUxVVUtBK2YDjO+kfrQFlO+ysjJJluUtw8PD
    ZeFwGABBUVERli1bhtraWiiKAo7jMDQ0hNdfPwhN09+UdrdQ5W7h90yT4ty5Pvj9PmX9+vUl
    R44cmXG5XHx5eXng7Nluh9Pp8ra3r/DF4/HpQ4cOxW0y6nZAflmonNZ+5HleqKysvHF2dvbq
    Y8eOz1Hr7IpqC0HS/L6JRhM4fboXa9Ysw9RUBOPjeaqdaZqQJAlNTbUYHx9DMBhEOBxO9fT0
    2H2Sfrl9ErOzWn9/f2h4ePixPXv2ZJYuXfqJ4eHRqjNnumxAh19Am8vPRyJzFNR5kGPvPzr/
    Pr+vaKHpztPtDGgaBc+bqK31I5MJGS+++Hx8eHg4SikNFfiiyK8KkKzrvvvuKw8Eiv4PIWjd
    uXPnFwYGBg6zQJvabJpGCCEnT54MDQ4O/uK+++5bZRjmEo4jzkAgwM/OzuYKkqoLFE2Li0ua
    M5lcTSgUg9frsdZhTiyI7XvLpuasHtobbrjhKkGQH4xG0w5RNAdPnDj68tTUtDA2NkaSyaSV
    QJixfBKl1KKfpW0sCythRmyJNvom/si8UNy2CCiygBDv9/vluro6l9vtdnIcp5imKei6zqfT
    aRIOh82pqSld13UKgF+6dOkqQkirqqqkpqYUiuIEz4ugVIBpEvj9Rd7GxoaiwcHBPuaPdEmS
    UFxc7PT5fKKiKB6AUlXVjFQqpc7OzmaTyWTO5r90Bj7t/sjuk94ULF0ygOR0OpFOpzlKKQzT
    sGW8C1TbUAiSCmh3pmnJj1rBgAu/RqP45fRLLHDZvn79end1VSUi4TC+973v4Uc//jEEnkd5
    eTl4jsNQ8RB0TYMkCvmgGG8g1FAIJrEQWK5sb8OnH/ow9/k/+T/+6ZnZUQAjiWSyat2aVY73
    33mL+Jn/9ceNfYNDjQDGWaa4kGp0WcARc0TQdb3G4XDcd++99/iWLWvLB/tzXOV5IIgCoGAN
    R91z9VXY9/Kr+K/v/QjtrUsR8PtBAbx+8BB6B4aw98b34D++9jXKKg5W46SdcmUFN/GC0val
    jN4o8jKeOWbc98/Ozq585NGf3rFm9SpO1zWEQyFEo1HE4wmk0ymk0xlksxnkcjmoqgpd06Dr
    OgzDmDtfFphkkXle9Y7PUxtEwdaDJIqQZRGKLOf7kJwKXE4HfB43xien8dgvXjJz+cAvysDQ
    LAMEdlDwS/N+2XvvkSXp1k3rVi/tGxz2j45PHqaU7mdgwi4SoLFhsSqAnNPpoOs6OnD06LEi
    RguaU3ayGXzreZQG/L7lq1e0QxB4cDxnGXDYAvw5WhQzpjyA21ubG27b1LGK//7DT0wCOMIe
    F2D2J8uc8jhzSBFKaYo9hiAvg96K/ADso8gPvLZoQjoWpwMLhWtYMCJgQUaOyeBXAFjG/k6j
    JEmVkiQFeJ53AOBM09RUVU3m8n2bPchLolcSQspnZ2fRUFeLTRs3wul0IJPJYGhoBN1DkyCE
    OJBvnI2y17KeEBJUZCkgCKJCSP53a7qeUlUtRCmdADAMoJ99Hrf1Ndr7G3UsVIV8W+bbUEoF
    0zRdqqpdgE73RhWk+c8Nw4QkcQrP8z4GHt3v9ISdRRvbvXt3sWEYV42MjPDZbA61tbXYtm0b
    eJ5Hf38/MpksVFVFLBafUxN7M9GGN5MD1zQdp0+fw9at652VlZXKM888M7F3717O5/MXTU3N
    glJTpJR62FpmGaUmUQCSLts6PfTQQ7WU0vedOnXaNzs7a6secYv02RRW2AiGhiZQXV2B1tZ6
    RCIpVokzUFVVCo9Hwfh4BGvXrqHHjnUauVzO7pM0Zvsup0+aoz/pup54/vnnX6+srFixdGnz
    XVNT0xzHCXA6HXO9SKIoMXnufJXHos3lbzI3XNa+Phb1Lh+jkIIeJNh6kQDDoND1/O10yigq
    Esznn389OTQ0ZPmimQv4ogx+yR4UFn/gD/7g82vT6dyuTEYrLioKfrC2tnZ8eHg4Y1sba+4Z
    mF/K8Tyvz8zMwuVyOurq6jw9PT2DVsLIlhSjAOB2uwVZlldOTU0GUqks22/gbIJjhs1Waiwx
    aH70ox9tdDjcvz06Gql1OCQjHJ449corr3aZphlgsaXJ9sgEY2NY/aJZAFQURWnHju2lra1t
    G1RV1R555JEDDMiZb+CPRNvzWZTxw/wQAUAURRG2bt0aaGhoaPR4PMtFUVwCoNYwzBJKTQ8A
    idJ8UpJSM5xOpwdHR0fP9Pb2Trjd7mWCIIgOh4JQKIJMZhIAgcOhoLg4AKCMCwaLi5xOp6+o
    qAjt7csaampq/S6XO8jzgocQIrKkZJZSI6br2nQqlRqdnp4e7OvrHxocHAxlMpk0i7NyNgGY
    QuU7uoitvLQAicmGqul0enJoaAh0DiQtHIp54UpJ/mM2l8PY+KSVgRTYG2jdAt7m2SNvQoWp
    CwQCW67edTUEnsd3v/ddfPe738POq67Czqt2oLevD8taW7FhXQd4gZ+jPV1YuGEB2Tf/vlpV
    BgY0CQGu3X0VjnQeF//9698O6roxHI/HE93d5xwiZ2ImHE6xg+EBkClwRpek0e1C66TrugBg
    79q1azfcdtttEAQepmHMJ/znih82oGQDTNSk8LiceP+9d+J/feHP8NyL+/Dem29EIpHAwz97
    Am3L2tHX24tsJk0EgXfoumFl9k1m6CyqlR0gmm9Gefo16XVzAIkFkykAIUrpieeff+G6s2fP
    +rLZLDKZLNweDxrq66EoCgRRgCjkb4vDzfNcXv6bcCBMic4+ZNgwNOhM+S7fi6QzDnkeXGma
    jvHJKUSiMYiiyGTDgZHxSbXAGY3bQMEs4wMnbRUE+lZeeyAQQDgcLnM4HUvuv/cOvHbgUOnX
    v/OjWwAMMqBC2ZrwbF0sQ502DdNcvnw5Kisq/D29vZINIC0QR2CGe2lZaUnl0qXNIAAEXiAM
    HEs2sDFXgmdZ1FVej/tjd916g693YIhGY/FeBhScbG8YzBFbfTkpSmk9gCa2BlOEkPUlwaIb
    FEWuGp+Y+oVuGF+llI7ZeOEWXcZ+SfbK7SLz06wh204AKwC8x+Px7K6qqmpcunSpv6GhXqis
    qEAgEIDDoeR1+3NZhEJh9Pf341jn8WxX17nReCJB33PD9YEvfP530dhQB1EQ5myKpqoYHBrC
    X33xn5Sf/PTxFQLPN5eXlxYta13qbF3SxFdVlsPjdoFjw0Fj8TimZ0IYHZug/UPD6dHxyVA0
    Fu8yDPMApfQY8iMZ7HSRTIFjwtsBktLptEkISSqKYxEhBu6CoMmqLlnZclEUkMvliKZpMuaV
    o+R3QcKOVFRUrEomkyuHhobB8zwqKysRCoURi0UhiiKamxtx5kwXpqdn5qoGC+llb9SPhAs+
    NhZL4ty5fq69vSU4ODiYisVimsdThJKSEhiGkRwbG8uxxKfbdlt2mbuUVST7HvT5fILf779h
    fHx83dmzXTBNCp7nFgVFCz/OgyhNM9DVNYBt29aipqYM/f3jkCQJzc01mJ6eRDBYDFl2kqqq
    KqfT6ZTS6bTdJ2UxL9JwyXxSoS9Cvi8zqarq7Oxs6OTSpS17b7/9Zp9FeVNVFZFIFLqu25gM
    1DYTKf/2mKa9OmSdGTJ3xvLKd9bnC2+XS4Ysi6AUc7OQwuFJbXR0dFFfRAgprOD/UrOz7r33
    XtTU1PD333//slAo5BscnEVHR9P2FSuWbxweHj7OgIBum3GXYwm5DM/zWVVVkUqlxcrKyqAk
    STmmtlYoGkavumqHhxCyMhyO8ZqmW+GLwBIqug2kzPmkHTt2OEpKiu+fnU1tm5lJYunSktjR
    o309Zn6Srm5LUIaZXw4tWbJE2rRp0w6/31+kaeoUpZC9Xt+tuo71Tqcndu21137pkUceeTGT
    yWSYT7JAuP1y2MDaYpQ6QiklHMfxd911V0VNTc3VgiC8J5vNdkSj0bJwOOKIx+MknU4jl9PY
    iBYekiTD43GjuDhoNjY2RVpaWkcppfWvvnoYfX3D0DSD2Yt8FVKSZDQ2VmPlyuaqW2+9davP
    53eaJueLxzPSxMQUyeV0mCZlo08EKIoEp1OGy+XVmpqC8SVLWkYTifjRnp7ug0ePHjuXSCRi
    jAqdsbFH1IL36zyfdMkAEqv6pDVN69y///W9s7MhPhDw5ee92KogWGwOEPJ0IQJgZGwCx06e
    pXkEyFhDC7mO76hMnQ1hCwB2tre3N6xf1wHTNLB8WTv+5q/+EidPnsRf/tXfIB6P4y/+/E8g
    yRIMXTt/UKwNJNmpVoahI5PNwiHL7AjOy4Cb1IRDUXD/vXfiwKGj7kNHjxcBoIl4nPLEJLIo
    CCw77CxwSJa6SM4erF3s4MX6fUzqvNrpdN753vfe7mqor5+XfKcLwRAh+dk+JubBkvV6DcPE
    imWt2LNzB37y0yewbdM6nDrbjf7BUVx/wyr8/Kkn8fGPPID//M4PleHRMcUWnBVWEi62QtCb
    VZCs8rilPHMoFo9/JxaPr0d+CGlg85Yt8t/+9V/B6/WA5wh4phbEEYu/hYKePoNR8/IfDVOH
    qRv5KpOhw2DASNc16JoGVVXxT//+TXz7B49Qu6NkRneS3ROWUyKEWFmqKOZn47zlJuo8pQde
    wzAcqUwWWzZ24Mc/fWpDLJG8CXlZ6ilKaZzNO+PY3zABZDVd16urKtHW1uro6e0tBXDO5hDt
    nGcBwPLmpkZfWUkxUqkUJFHkmOFXMN+jQ2wGPwDgt/ZctWXF+jUr8OOfPZ1hlRerJJ9i1Jcw
    pdQaWr29OBh8cOWK9loCoPPEyZTH7XL/9m99QKwoLcK/fOXbN7x+9CQP4OuU0hnmZBfLhEuF
    lVs7OGLPrxbAA2VlZfdu27at6frrr+c3btyIyspK+LzufFBKqY1mmb9zuRwmJyeVnzz8aPOX
    v/I13Hf3nVjW2gJdZ5LwbN/wHIelzU245T3Xk9cPHSm9ae81uPXG69DcWA+f1523aQt62ChM
    w0A6nSazobCru6/f9drBYzWvvH5kW3ffYFdOVX9BKT1ACJlhjjyJheIOl63X0X719vamm5qa
    esrKSrfnZ68UBvjcG9z5YFiSRHg8Eu3uDmm6rhN2dq2eAw7vTHAEANi6datLUZQb+vr6S0Kh
    PL3u5MlToBSor6/Dxo0bkM3mEInEbLSpN68QWVnfoqIAQqEwdN08r2eHEILR0UlUVZVLHR0d
    wUgkkvN6PWY0muUIEYvWr1/f9vOf/3w6l8slmG9ysjObvNQAye7r3v/+99cYhnHnmTNnXLFY
    7DzAZ62BvfXv/EoSh1AojpGRKTQ1VWJqKoqiIg9cLhEjIxG0trbj1KkB1NQUyVVVVUpPT0+h
    T1Ivo0+y+yIegHD27NlDXq/3O4FAYL0kibWKogTGxsbkZ555DqqqoZBZNQ8O+Tl6nf3jwluY
    kwe35MLzing8li+vQX19kKbTaRqLxfTp6elsT09POJVKTTIQYPmiMZsvijHb8kvLPPv9fuza
    tUsAEBAEjiSTacTj2WBTU9Mtu3fv8vX19U8NDQ2F2GBonv0t1TAMjeNIllJgYmISTU0NZeXl
    5eLw8LBF1SZs/hEBQMrKyos1TV8WCsVYbEwBQGIVe4tdwNsSfNyGDeu3qKr5gb6+GdnrdUDT
    MiO9vb1Wv64FBuOEkBlK6XRHR4drx44dH81m1R3RaMrt8bjTfr8Ho6Nh9+BghJSXB721tbWf
    vuqqHfzTT//8dQYWtEV6W+U3YDMQSinZtm2bY+PGjTslSXpodnZ2R19fn3dgYAChUAh5CXNz
    0b40juMgyw6utLQ4uHXrhmAup6K3dxCaZpxHy9Q0HUNDk2hqqlUCgdL606cHMD0dRTqt5ie+
    LJLYEgQBkiSJLpcSLC72BEtLPcvXrt1wfXPzklcPHz70zPHjJ/pYfGH5JKtCbaev0stCscP8
    cMKXX3/9wD1PPvV08/vuufM8at3C3iNr5k3+OWZzWfz0yefQOzCcy2ds55TXFi2RvRMu24ar
    kCTpphv2Xu+oKC8HQLF921YAFDXV1TBNE263C5s3bWSNnG8u7U2pCQ5Ab18/fvCjR/DQRx5A
    cbBoHlBhHlw11tfgA/e8lzvb3RtIJlPRV14/pNdVloirli3xPv/KoXVmnhJj8eat216Ro5fC
    CVmOyDRNHsCuFStWrLvxxhvzgMnQbaBw/jXF43GEQiGUlRZDEsWCx1AIPI/bbtqLl155DT9+
    9Amc7upBW/ty9Pb2oqW5Dve892YcP3maHx4dc2HhdOlFX+dlymhbohiWyMgAgO8BeBV5QY0b
    /H7fxqqqSiiyzCqwCwUWTCPfg8RKJ+edPp7jwQkEPM+BUh4wxXnxBvZ7vG43bCDAavYMMYc0
    jvzsoHFCyBTL4oUL6HVvyYHbEgf9iUTyB//85a//dlN9bTAQCJRVVVf91sxsKD4zM3sIwDOU
    0hGb0c4ByBq6bkiShE0bNwqPP/FkNetdy9gAjxWoejmOa29Z2sy73S5kMmnIskRsAAm2PcCx
    RMaupvra2z507+3c6MQkhkbHQwwcCrbgIQpghhAyTSktLSoK3P/7v/vZ5jtvvxmmlsWjP3tM
    +uo3v0t/8cLLZsDrJqZpCBzHrTJNs4FVak1CiLCIQzovqLZPIAfQJgjCH2/evPnmT37yk8o1
    11yDQKBoPnFiGDB1o2B0gjmnAlpfW4sN6zrwyKM/Q31d7flJCFtvm8/rQU11JT5y/31obqqD
    rulziohz1GebCqfDIaO2ugK1VWXYsamDDNxyvfOxn7+w9seP/aJpaHR8GaX0cULIMFs7qz8q
    ZaMTWRW8y3LmOjs7c5s3b95fU1N1R0lJiXdmJrIIre78CondCRcXu8Hzuj4wMJC2ZXzfUTLv
    F9hHpKOjY0U2m73+3LlzRNM08DyPPN2QoL9/EDMz+d6sTCZbAI7eeP4RABQVBbBu3Wq8/PLr
    SCRSiwIqXTfQ0zOE9euXe2ZnZ3OZTDLV2FjpEkVJaGtrWzE8PDzQ2dkZtvkjqdAnXey9UkCr
    4f1+/+6Jicm13d29c778fDEKkp8p53Aik8mxgG3h40yTor9/DFVVpWhqqkIg4MLk5BiKi4sR
    j+fQ2zuGoiKP0NjY4Ort7eUK+jlwmX3SnC8ihJCenp7B3t7e7xYXF78aCASWbNmyZW8ul9sU
    jyfYQFi+oO8ITPDEWlMU5K+J1S3B4j5ryKwBQig4joIQA5pmor+/33j55X2pWCyeSqVSMdM0
    38wXxfErzi08ePAgDh48qH/oQx96vLi4ZHNHR/MWWRZ4So2rW1raN7S2Lhvp7Dz29GuvvXac
    qa1SAIlcLmeaJs0KgoCZmRA0TQ/W19cHh4eHJ23+xfJJgsfjXpLNquXRaAKEcAxAEJnjOCfm
    RyvMJfmuuWZPhSwrH+vrm60Lh1Oorw8aU1OjA9Fo1LAlsZOsh3laUZTY1q1bbw2Ho9e/+uph
    MZ02UFRU4lm/vh1er5M2NeUlwgVBLC8tLe0QRfGcpmmEUhqz0fysN4y30dEXxLSUUrJnzx7X
    +vXr783lcp8/cuRIw7Fjx8js7Czbn6RAqIM/D8ioqobJyRlEozGoqs4qRwvtr5WsMk0KVdUQ
    CsXR0zM6V4G0gyn736KUQFUNaFoasVgOo6MxvqTEU1FfX3Tbzp27lhcVFT2yb9/Lhw3DCLHq
    nciAEikE2Jad4S5DVeVcKBz+yb986f+l973yKqsM0YIZSPmADayfBgTQdR2P/fwFfPcnjxma
    rltKJWksVJ7K2Sgbb3v1iH1OmJzunmXL2ta/hwX/1DTnBuRWV1fi4x/7Lbz/vnvgdjnnMr8o
    CHDmwKSdegdgfGISL+x7BYlkgq2lOUc5s//stbt2YOvGdQoAaXR8MvSlb3w/ffREl0bzwyNr
    bFk6B2yDzS702i5iBYkAKJYk6ba9e/e6Gxsa8oID9l4jUBAQhEIh/H//569x/4O/hWeefR4c
    IfOAwKoimQYa6mux5+od+MZ3fohzfUNoaGxEb/c53Lx3D8pKgti8YS2RJNFjo8Hwi2R/L8lr
    fpPMncocU5wQEmJ0s5MAxgWen5OQzeZyDCiGMT0zg1AoBF3XFlIu6cLfbq+2LQBXc/QIq/I4
    Nw/Kmn8zzbJ1o8j3mIxZZXzMD2b7papHNgOcAfDjvoGhQ+FEGv/0j3/H/fC/vuH+h7/6s8rl
    y1quA7ALQAWltBiAjxCiAFB1w9Cz2Sw2bFiH8vKyKuR7Zfg8x5naA6pij9vd2taylA3D5eBQ
    FI7tdcW2x627XlHk33rvTdcWt7ctwcGjJ5BIpsbYmlj0tzRTlQqxJuHKpUuWVN55x20oLylG
    SVEAN167C1Xlpfj586/0/+DRpw8c7DzzmmmaL7N1cgAQCyTN8QYBkSXGUMXz/P++6aabbv/y
    l7+s3HXXXQgEAqxSaMzTlu2tPQUVecMw5oZLG7rOHruwcm/93GwoDJHRFfI/Z8LqabMLpsz9
    ftOckwLmOA7NDTX45IfvwV/+4W/7OlYu2wvgfZTSFkqp9X76WdX6vF6dS3nmbH1c9NChQwdd
    Lsdra9ashNPpwEIVu4WCDYVO2+NxoKGhiA4M9KbHx8etrLV9ds87RuK7kNGwc+dOt8fjuWt4
    eLh+ZGTkvIwtACQS+V7H8/ux7BQpbtHv5bO3YkFQtPBxHMchEolhejoilJeXu19+eV+kt/ds
    aGZmLDwzM5PWdd1iNLjYeAzH5dgrFvh48MEHSyilt/T0dLtjsdh5AZv1FBwOB7Zu3YD3vGcP
    6uqqbc9p/nEcxyEeT2NkZAqtrdVwuXhEo1GUlJRjcHAG2ayB2dkEKS+vdHs8nsV8En+5zsdi
    vohSGp6ZmRns7u4+qarqBGMEgRACQRAYQHTA6XTOCXssBEYLST555V7MUTHnKZl24AnE43F9
    YmIynUgk4qZpRhgQsnzRCANK02/mi94KoDx48CAA4Jvf/OaZ6emp7weDrkRPzzk89thz0vPP
    HwpMTSVWrl695r01NTUtAEoZ08ClaRqv63pGkiSk0xnMzoYdNTU1jYIgWG0f9qSzLIpSWzye
    9GUyubnh0xzHSRzHWdRcwao4CYIgt7W13RKPZ/f0988w6piQ7Ovrm2SPtah+CUYvnF22bJnh
    cDjaenr6xcnJEDTNRDSaxuhoCLJMEvH46JHZ2d4DZ84cf+Hw4aO9mqbN9e/PJcvePGFHKisr
    hXXr1t2cTqf/94svvtj4wgsvkJmZGRs4Wsw2LLwtG2KdEbt4R2ESIi//LiCX0xbY5sUUJAv7
    JDmOQNMoJiYSOH58kp+ZMVpXrer44NatW3cSQqoppWXIzzT0YF7PgCs8c8JlOHQZAD89fuJk
    /R984Y9v+d3f+ZSye+d2OB0KQBZ6EgvKhsNRPPL4z/Hlb37XnJoJJTA/GMq6Le37X5p3eqkc
    kRX4s69rZFm+9/bbbvM1Nzfl5bzfYoXogtLe1KrAE7hdTjTW10KWpLmBsQt6lVh1Lljkxx23
    3ED2HzzqjicSs6l0JpVCxmBrmLN43qzB2B5A6pfS0bO16qivr994ww17wfM8DD3f+kRtanUc
    ITh46DBeO3AQPC/gaOdxvGfvdex12oI2mueJ79qxFd/+/o+wZm0Hent60FxfhdXL20BNHWtW
    tqMkGHSMTUx6LMNFKZUYncveO3BpuXULKwgLHBPyM2vAqhpZntGAXnjxRTz8yCOIRmPIZjJQ
    NRUBnw+f+fQn0N7awuLWgj4t2OdHLVonsz12TtnNGsg2Y6PXTTKudwxvIMzwVjOc1twJRVEC
    H7z//bhu9y7oagaVpQF09/QoXd29y3XdOIu8kk+ONcnCMIxcLBbFju3b0Nba6hsfn2hiFAti
    UXIopQqABp/PW97cWJ+fmSUIcLmcdoqdaoFi9vib16xoW//em69HPJ7E6bM9GnPAIhYq3iVs
    a5DMZnNGKpGAGfDANHVk0ilkszmDrVkvez9D7GVbA2XfMm2VUioCuHPDhg03/dmf/ZnQ3t5e
    0LN5fhUIlAUic7Am35NYVVkBRZZxpqsL7cta5meL2ZIrmq7h1OmzqK6qgNvlgqqqMBgVL9/H
    xYHjrC1Fz0tm2Ol62zethdMhi3/yxX/bdKqrNwXgaeaMiU1+9Vce7vgr2mlKKaWHDx+eqays
    /E5zc3NjLqct7ew8yxzw4upsVmLX63Wivb0CyeRM7tChQ2Fd1+OYnwtmp2u8rf5oMZ8EAB0d
    HRvS6fQtJ0+e5rNZ1RakvLXhr4tXk+a/F48nceZMD3I59bxKnP2xlALDw+Po6FjmaGpqdFVU
    VMiCIBjT09PhZDLJcRznNk0zzs6r0xa82IfCX3RwBICUlJR0xGKxjb29fTBNc065rrDfqKKi
    HBUVJaBUQ1lZMQYHx3H+fKT8ax0bm0Vrax3Gx0cRDBYjmTQQiaTBcTxCoQTq64udpaWlnng8
    LhX4JJHZKf1S+uEL+SIWsIPnedHhcGRSqTQoBerqatHW1gpFyQ+O5XkeuZyKI0dOIRyOvYFC
    XeGMLRTE4XNrbNlcCwBM23zRBPNFhdLn9Jf1Q4W2WJZl19TUtKOz8xSyWQpFoTh7dgSlpStq
    amqq24aHh8OYFzVIqaqaE0XRME2THxkZI+vXr2ksKysLjo2Nhdg6OgAoJSUlPo7jWmOxlJhX
    hOSg6zoI4SQmqpMBIDBVUm7Tpk3Nsuy8r6tryh2PZ7FkSRlUNRsZGRnOsLjFSsZYbI84z/Mc
    ISQrywpEMT9IludFSJKEWCyW+sUvnulLJBJJ9vgIfsVeyTvuuKOZUvrJw4cP1xw/fhyGYZzH
    Dlqo9LiQipofEg0YholoNI6qqgo4HAqyWe08wAMQuN1OOBwyYrHUgjla1vcXFl0XVritxBfH
    cchmDfT0hAEEK1auXH3z5OREpru757QtYWn3Rwuo38IlttUm4zlOA/j20WPH9d/9/Beuu+6a
    3UV7r9nFtbU0w+1UAFBoqopwOIzjJ07jsZ8/h32vHTSTqXTaFpRYg8Hiizgkc7FN/3bQGSil
    EoBb1qxZvem9t98GURBYdeQNhr/aeojOA0WLgJ/Wpc34wu9/BsVFAVDDuDCgIhRbN6zF+rUr
    pOdees0HQBJFkRNFUUmn002Yn+WiFFSSLkmPQL5hjwNrOL9286ZNxW1tbfn1YSY6H+ARUI6A
    IwTRWAx+nx8+vx/UNMFzBNTMf5+aHCjJzwAiAMbGJ1BSUoolS5bgp488jN/9+P1wOmQYuo7q
    ijK0LGkUxyYm/bbXagXNdnnkS755rP1pc052kAT2fHI8N19BcrvdqKqshNfjgSgKcDkdKCku
    nq8Q2XHQeV9Q29+1gNSCyNTAwvk+s6x/xFIKimKe5/0rOyQbnQOEEF5RZFCYMA0dhFIosggC
    oiCvGlcEIEkpzQDgdF1PRyJRBIuKsGP7NnHfvpeXarp+jtkHp+1uqaqs8JYUB2FSE6IowONx
    2yl2GVv1qMXv9dx69203KnXVFThw+DiGRsctm6Ngvh9PxXxjpwmA6zrXZf7rv/077r/3DlBd
    xQ9+/AiOn+mKscc5YeOVF9CwCpNI9ALBfJXH47n9/vvvdy5fvnzBwxcAJeRVHwkAzdAxOjKC
    U6dOY2JyErqep1HpmoZUOo19r7yKG6+/BrIszf8OUIAAM7MhHDxyDKZp4h++9BVks1nkVA2G
    oUMSRQT8XixprMP6NStRUVqMBbL7i4C2Nctb8eH7bpX//O+/siEaT4zanI999tqChuZLabtZ
    MGhQSrUnn3zy6A033PDvLS1NH/f5fE1nz/ZzoVAMum4P5PKZTodDRnl5ERobS2g8Pq299NKL
    kXA4bM1fsexn3Das8h2RsLOxGXD33XcXK4rywZMnT9WOjIzaMrfnDz3luMWystybzEICNE1H
    KBSds/FvNEA2Hk8hEkly1dU1TkIIiotLXDwveqqrawKnTp3k9u/fHzEMQ7b5JdlGgbloft4O
    JHfs2OEURfGakZHRktnZ0NyMLHtgbwV4iiIjl8vPibJe//xjFgaLbrcTyWQCkUgEy5atxNmz
    MzBNAp4XkUpp0DSIFRUV/t7e3sV8kshsD327fBHP85Isy6plKziOg6qqSCZTUFWNqc4ZyGQy
    Cyh28xVMLBDxWBhAo+BzgBDOKEjWhS7gi847a7/GfjBFUeRzOY2jlLDeKNFS3COmSb2WP2J/
    W89mszlRlHRRFPnx8UnoulHW2NjYMDY2ls4/FeoC4CwuLlZMkzbH48m5vaGqBjiOEyVJdLLf
    yQPgFEVxtre33xyNZtpHRiLgeR5+vxPh8MxMLBa3xkfkwASeKKVZADlJklwcR8xly1qQTuuI
    x3MoLS1BbW1Q7+09PZNKpSQsbKGgBbTgQv9cuKZElmXe4/HsGR4eXnPy5MkF4MgufV9YdVcU
    BcXFQRQVBSBJ8hz91ONxIxj0o6jIj/HxmQWVJ+t3lZYWweNxor6+HNXVpXlFWpbY0XWTZrMa
    CYWSiEQytnN3/iw2juOg6xSDg3F4PMU1q1at3jYwMBjXNA1MhEMr8Elzflm4lAYaC7XWIwBe
    mJya5r/1ne/vfvinj5VUVpSjsrwMiiwhkUxibHwSk1PTyGRz1swSFec30uu2mRsUi/SSXC6Q
    VDjfhTmkdp/P98AH3v8+d3NjA0xTz1PfzlOkW0S6G28k7W2yqkoeCPz8mRdw7523IODzXhB8
    mSZFUcCHG67ZSV49cETMZnOZ4mCR5+733uI/cOjI6tcPHRVY07liy9gpNuNjLpKR/LUAJPtd
    VV6vd8c111xDXE4nDDb3yDQNHD16FC/t2weO41BZUY7TZ87CpCYEQcDE5CQOHz2GUCiESDQG
    t8uJjlXLEfD7EI3F8MhjT2HV6jXo7u5GfVUp1q5oy/9uQuBzu9C2pJF7ft9rXszLxDsppQ5G
    47KMh3m5gpnFlITY5zIA3VKp23X1Tly9c0festn69UxDBzUMtramvSJUUN7Bwn+ni4IWS1Ev
    ivlhbNYwvgQWKr78qvvBeoKRTCZz9L9/9JM1rc0NfFVZMc51deGJnz9vaLpuZbg8rMLpBEA0
    TUtEIhHwPIedV23HN7/17erBwaEam21wIS8T3FRXU+3wetz5CpIoIODzwra3BVbNcAO4Yeum
    joY9O7YAFOjq7Uc0FrdmZMnWa7ZU6Nhsig5JEj9FTbPom9/+TvLpXzxjAOCnpmYSOVXtZQ7P
    KKBfWbe+yJrNOaSCYKW9sbGxdfv27VZSYU4et5BKxwGYDYfwve//EN//wQ9pT08vXC4XWb1m
    DQSeg2HoUGQFncdPYmR0DEuaGmy2Ih/EnO3qxkwogtVr1kKFBMXrhkfgcfrUSTz7/ItU13XT
    6XRgU8dq7nOf+BBZs6JtbjMVSvFTaoIQ4KpNa7FuVVvRsy8fXMGqkjqrCtrnUGmXETiYAHRd
    1zOPPfbY/rVr19JVq1a996qr1q2NRBLO2dkY0mkVlBJIkgSPx4mSEh943qRdXSf0I0eOpOLx
    eNpWVcwgL96RxQXEJy6G7fxV7QqllLhcLr6xsfHGSCS69/jxE5yqqnPqdG+uSrcYTYZbVKSh
    uDiIlStbsX//UaRSmTcEWKZJMT4+Q9rb64XTp08lJEkWenuHxaKiQHD58uUrTp8+dTYSiYYp
    pU7GbHBg4Zwb+ka0s19mra3Htre3V6mqelV/fz/y/VnzIh6VleWoYwOnU6kMiouDAMhcH3FZ
    WQkcDgcURYauG5iZiUFVDciyhIaGCkxM5OceZbMEkUiGzXoxoesUyaTKlZSUeAVBcOu6bq+a
    KXh7JOMtX5QDQCVJkiVJ0qx1GhwcxtDQaMF8JG5OgGGxqtDCr3GBry0AtYDNECOEhDE/7yjK
    fFEOF0/AggLQp6enT1dX14y2tbU0DA5OQpZdaGqqAKVqcmRkJMt8kYf5mUwmk8kWFblyoijJ
    sVgCk5PTSn19fcuhQ4dGs9msNeLF5fV63YZhVsdiqblzoKo6KCWC0+l0s9fFA+BXr17d5nJ5
    rz15clzIZjU4nQpcLlEfGpqIG4Yh2fa+ZvmU5cuXuzZv3vxhQoSri4p8+pYtqxO5nCoSAmNk
    pH/swIEDw6ZpGrafmZu1tIjtpSiYhcnEtOh73/teD6V029DQkJJIJAret0KaXF5Eobq6CqtX
    r6AlJcXw+TwIhUIkkUgyUY68sn11dTkmJ8MonKUmiiLKyooQiYSQSs3AMAyYJoUkKViypAmZ
    TJpGImmjqqqSm5pKk97eGWJPcC0GuHI5ExMTGa66unxpeXn5uZGREasal7PZ9AXzxoSL7IDm
    1C6Qb8iWATQA2AJgoyzL7UWBQLU/4PeUlZbQ4qIi4nblJ3cHVRXBYBCVlRWYnplFLBYX4/GE
    O5vLEcxL7E4SQgQWQNl5nlaToYG3acYGA0d+AB/Zs3vXittuuRmEIzANs2BmEX3zahLOV7Ob
    y9QSgq7uHvzo0cew99qd+eDvAoCLUhMcAbZu6MDSpgbxxOmu9MxsKFddUeaouPEa7kjniRJV
    1bwW55sQYs2Ysjuji1ady88QMQiAjqampsb1G9bb+o0okskk/ulfvoRTp89gyZJmZLM5ZDJp
    3HDDjSgrK8PXvvZV/PFf/G0+K24YGB0Zwacf+jA+9P678errBzEyPoX1W3bgS1/6Eq7dsQGi
    wIEaeYDEczyaG+vgdCiudCbrKnBEVsbu7VSgMm1rngKgE44DCMARDqAEmqZCU3PQtXxm32qe
    F0UBAs/ZrBwFoVZtoYBiRxctXpi2rKF98KVFI0oXZpx+nWwde30/fP6Fl2pGRka2VpWXuQaH
    R/iBoeE0yxTy7L2x3qecpmmJcCgEwzDQ3taKrZs3uQYHh1pZVlFnc1RKANTX1lRxDkXJ02Q4
    gmCwCKIoODRNd2B+aF5raUlw6z23vYcvCvigaip6+4eQyebimBd+sIIFlc1jWupwyF9obazb
    MDIxnZoNR386PDIWQ74fynIu1jDZWVaJC9sce3YRkYbzZiAxAZP6uro6f2VlJeZS0wuSKfPv
    XyQawV//7d/R//j6N810Oq1xHCc++OEP87/3u58DR3VouQyee/4F/MVf/x1OnT6DpU0NC+iY
    1KQ4dbYLsXgCO3bswPvf/z64nQ6YWg5nTp/ERz/x2zh4+JiWTmdyz7+8X8qpOeVv/+T3SENt
    VV6R6QKUP5/HjfWr27mX9h+t0XS9EvPyxUlbNnZBZeBSJrAYlQatra2+YDDY5PV6q3Vdz0mS
    kK6trXBUVJQQTdOYtDMPSRLB8wQzM9PgeR5NTY1COByRYrEYn0gkqGEYpi05xmG+d0TAZRq6
    /QbJKAIAH/7wh9sppQ+dOHGyaGpqugAcvXGVaHFRhsUrSB6PCy6Xi1Ve3pyeF40moKqmoCgK
    Hw6HcoFAQEync6AUEs8LHhaMptgYitQiPuliOnfi8/nWhkKhxvHx8QVVDlmWsWFDB4qL8wp9
    giBCFEV0d59DKpXC+vUbsHXrKkabAnw+P06c6ENX1wgqKoJwOjn094ewadNGTE1pAHjwfD7R
    ZZomEokcSkt9bofD4UwkEk4ADpa0s/cEX/JBuQV2yfIHhsPhSImiqFM7rR15Sns+0BVYxcVq
    lMcc9cnek7RwyCwW0PAKqp4mS0ilASSY6ljcxh5KX8w9wKr1xvPPP3/q1ltv/frate0faW5u
    qBQEUQwEPDh37uz05OSkRW+zWApKIpHIlJdXqJIkwTCSGBgYxo4dm+rq6+ururq6dAtQ+f3+
    Bk3TfalUZi5Q13UDum5wLpfbzd5jweFweNrb2/eEQumKyck4AAKnUwbPIzc+Pp5h8YkVvKuU
    0lwwGBT27Nl9JyHKxzIZ6lXV2PHXX9//gmkawXA4Io+Pj+uqqloVuQjyIkMh5o8stkPhtQA4
    MYAEl8vl1TStYb7n6Hxqnd1u1NfXYtu2TXR6ekrr6+uhDQ2N0jPPPIdwOAZZdsHt9mPr1nUo
    KyuGokjIP835c+dwyAgGfQiHp3Do0CHMzMxCEBxQFC8yGROVlQH6wgvPZ0pKiunWrTuchlEi
    9PWF3rQHKh7XQIjXV15e1jAyMjIDIE0pTRJCkjbaJrmoFaRFyvoS8gMTb/d4PDc0Nzcv6Viz
    2ru8fRlpaVmCmqpKeN1uyLKUD+yoCd3QoeayyKQziMdjZHxikvT1D0g9fQPSydNnvWfP9ZaH
    ItFy9rvTbMCpFVzwWCjacMlkqi8ACDnWM3Dz0iVL7vrEQx/jS0qK5xud57j6b1AhwvnDX7HY
    YFgTGBoegaZpecGCBepV5/9ek1JUlpdg26YO7uSZc5Ku69qRY53K5nWriCJJiqpqFaxakLYN
    ji2cSm1eLCPMwMj6NWtWe6qrquaFAiiFJIlYuqQZ3T09EEUJxSWlWLpkKe6++y4oioJ0Oo1z
    584BAEZGRjA1OYGigA+RaBSPPPY01nSsQ1dXFziYOHW2ByNjE2iorYbJsuQ1lWVwOh1SOpN1
    MoOnFNAZeMs4vA1UTWuys8kqFZRjSi77XnkZ+/btQywWQyqVQjabRb48bAIU2L5tM+6763YI
    PH9e69Hc3rKCbIumtTg4s2eYrIzKeZmmXzNbl89mEdKj6/rfne3qfu1sV/cOv99/VWVllWN0
    dLSSVTUFW2XTAJCcDYehqircbhduuekG8vNnnm2YnQ31M8fhASA4nY7qmupKO+UV5aXFcCiK
    omlJq8fOA2DL7u2bSzZ2rAJAkUgkMTw6blqccFs10cowlQs8/+nt61dv4XkOJ7v6jiM/SFZh
    gb5iA38RBpCs7KfVQ5lZJOut2u0Ve948gCKfz8e5XK45CuX5PUf5JMjjTzyFb/7nt810Op0E
    QCvKy8Xbb70ZpUU+qNk0njuwH//6b1+hwyOj6OsfIAttDoWuaZiensXU1JT553/+55iamuQ+
    8dEPw+tSUF9TgVtuuI50njhFVFVLA0gePHoi8NjPX3B+4kP3ID96wb6v5n8vR4DaqjIoiuTW
    knoxW4sopTTCqrbipUhK2M+upQ62atUqZfPmTa0ej/c6AFfncmprMpkKxmJxx7Fjp5FMppHN
    qshjnnzwJ8syHA4HPB4XSkrqhLq6JW5R5BRNy3nC4ZC7v7/P7OnpDafTaSdLMGWY/yG2iquB
    y6TUZ/NJBAC5/fbbiz0ezyd7e/s6Tp8+MzeT5ML9RouBnzevNPE8h2AwAI4jLIfHvSn4YpUW
    UllZKU9MTORqa5toPD5FRFEObNq0aeVLL72USCQSaRuotjfjX7RZfZRSbNq0SeF5fv3U1JQ3
    Hk/YhCbyQ4HD4SgCAT8Mw0AqlUIoFMKZM2eg6zpEUUQwGAQA+HxeeDxeZLMaFEVCQ0MlxsZG
    UVxcDI6T4PfzcLkcSKWybC8YSKc1yLJbcjgUZyKRsHySXXTm7fBJ1oE2PR6PZlWwCSGoq6tF
    XV0tFEVhCYS8RLe1V8bHp3Hu3ABTrLtQxehCPUhzf9tiBWQLbnWxivOvuSaUEKKHQqHkj370
    o0dXr149WlFRvrmmpnZvNpuuLS0tLW1sbPR3d3dbzAYHAEc4HM4JgqDKsgxCOExOziAcjnmW
    LVvW0tvbG9F1PQ3A63I569PpnFNV9blzYRgUqmrwHo/HZcUey5a1LXW7fWs7O8e4vLIbD7db
    hqZls6FQ2ChM2ImiSG+55ea9guD6eCik+X0+Ej1z5tQLhw8fHmLJOA/7mRzmlWlDNn8Uwbyi
    qP3K2gG5x+NBJBKBKIoe0zSddirlwt6feZ0rj8eDdevW0NHR4dwrr7ya2rFjh3N8fIJMT0fA
    cTJE0YWVK9toSYmfqqpOZFkmqppZYDdkWQZgUofDQffs2UNef/0ImZqKQBAUDA/PoKGhmisv
    LzeOHz+ecDic6rp1W4tmZlJcIqEW0Di5AvlwCsPgBKfTWcRokzEba0a20eIvTgVpEaDgBXCb
    x+P5xJYtW1be8d7bpe1bt6CiohwORQaomZeOZfQgk81tEQQesijA43KitDiApvpabN3YATWX
    Qygc4k+eOuv92VPPrn72pdfqZkLhekrp0wB41tAYw3xfUtZeJrvURoXMd/N2eNzuz3z4wQ8G
    N2/aCEM33rDP6EIVovOrSPPfJwAi0Shee/0QRFGAJAoF4MhcFCSJAo9tGzvwg4cfl8KRWKrz
    5Gnzzhuv5m/cs9X/8JMvbMupmm7L7LqxsAHSxMXtzfEqirJy9erVnCLLNglhClmS8ImHPoq2
    1hacOnUKE5OT+PGPfghd11BaWoqvfuXfUVleBodDQUlRAHf8r89g11Vb8dwLL2N8ahabtu3E
    j3/0Q3zqt+7HT594BoePnUR9dUU+piSA3+uGIsu8jddud0QiLqHE+ZtRPKy96nQ6kUwmWfIm
    v3cnJycxPDICj9uNTCYDjuOwvL0tX20wDFSWl4HMPW1bqYhS278WVJMonf9Y4BQLKoe/Nse7
    sBnYmtwOQsYBPE8pDVNKm6666qrGgwcPNnd3d1tKejwhRGZc62Q4HDbTqRTn9XiwZdMmbN+y
    2fPIzx5vYQYuDsDnUJRgVUX5PA0NQHVlBVxOpxxPJJ3sfW6oLC9ddvtN1xK3ywFQE/FkEmOT
    01Z/jDXZXkd+srkLwN2b163YtWPjau4r3300rBvGa+x8OAsAZcRyQmygrtVDaVVLCq/F5Gkt
    m4I5FdYF/WPzyoXJRBKPP/EETSQS1kwtuaG+jlSVl0DLpnDqxAn8f3/1RfNo54kUACEciTgM
    dubm+xrndocxMzOj/sM//KOsZdPCZz7xESgih45VyxAMBLiJqWkVQErTdLzy+mHl3tv3ckG/
    77yh3/YeSKbEyLNqoMtWFbQrCl5KcMR/8IMfrK2oqLhX1/W7p6amlw4ODsvj45OIx5PI5fJ9
    FPnKinDevBb2OZEkCbKswOVyisXFPn9ZWblvx476mtWr19QdP9757KlTpw1d1yl7rRLm+yQs
    X2Re6oSd7ZyR9evXO1taWt4fDofvOnToMJ9OpxdUjwopLXbAU/i9C80/sh7v9/tQVVWOXE6F
    aZoX6EE6/+fC4RiqqhpFVVWzosgZLpdbmJ5Oya2t7R0ej4d/+umnH49EIlFbpd8uXHDRqib3
    3Xef1zCM1ZOTUzAMY44qZgGkI0eOIxSKorS0GG63CytWVIHneaTT+QpSKpWFYVBksxoOHz6H
    0dFZ1NaWQ1EIwuEwVqxYg9OnR1BfX4aSEg8yGUshkkDTAEJ43ul0OYCZt9UnLbY3RVEEIYTk
    5b0J3G4XfD4vNE2HIAigFJidjbA9yMMSc7CrKJ7fg2TtWaCwH4mQBb6o0B9djH6jRVkNhBAt
    Ho8n9+3bd4LjuJm7777b43A47kokkoHt27evSKfT06Ojo2DvixIKhTQAWZfLCY4jyOU0dHcP
    cBs3rmpoaKgf7enpTQMokmW5OpvNSbpuzK2FrhvIZjXO5/O58kVt2dPWtmxVKJTxh0KpuWDe
    5ZKRSiWymUzGqkxbiUz92muv7QgGyz82NqaWOp08TaXCnYcOHR5ge8dgMZzGfE8I+V4uyx/Z
    4+XFKkhzixuNRvOOUNc5SilZvHrEoVDERJZF8+jRYynTNLPBYNDV1zcCQXDC4fBg1ap2FBd7
    1MOHD+ba21e6JUkiQLagb4ggl8vRl1/el2toaCRXXbVN3r//OAmHM1BVglRKJ1VVVfTEiROp
    np6e9IoVK5Vg0O1OJCLnPa9C1TsGxhXMzwC1i8FcXIBk481bAxc/1dTU9KmPP/RQ8T333I2K
    8rI8KDIM1o9jzdLIP09C8sHrfJI7T/mwAn9B4FFWUozSnVuxYe1Kcu2ubUVf+/YPrz907GSN
    pumPUEqPs2xk4cwEK4N3UUHSBfqO6gWe//xtt96y4gPvvw88zy0I/M+rAi3C2b/QY+dldU1Q
    AC/uew37Dx7Blo0dcChyfi3NhYCq8PcAQEtTHVqaGvj9hztJ3+CI9r2fPMHXV5WRqvKSiv7h
    8R3IS2hOMaMsFqzlxbxKXC5XZWtLy/ygU1uzd1EggPfedjNuuekG5LJZfOnLX8GRI4dRV9+A
    +tpq/OWf/RHcLidkSYIiS4hGo3j0iZ9jzdoOnD17Fk11Vbjp+t3oHxzG/sOduH7XFjgUB0wD
    kCUBLqdiBTCS7bVae4fH23yZ86k3DgTgeQ633XIzbtx7PUSRx7/+vy9jdnYWn/7kxyEK/Jzc
    c36yuQ3T2JW/rTOKQknPy98XUVCxyrBAsj8Wix1NpVJ1H/vYx+S/+7u/WzU5OTkNYIZVjBUA
    mXA4rCWTSdnrdiPg9+G+e+7kXnv9QP3U9MwEA0guh8PhLC0pXqDsVloSRDAY4Cempi0eedPO
    bRu9q1e05ftlAMTjSUSjcYsWxduqXRyAva3NdTd8+O6bxGdfOUhHx6dOIS99bjkjKxiO2sCR
    NcAwZavELRbYGYvYUwognkgkzHQ6zXk9noXgg/1HKEUqnUZf/wBsGVfO7XYRjhrIpuN4/Kmn
    0XniZIJVtEpMw3RQ01xw5gSehz/fp0UBpFOpVPbb3/meb/O6lcL2jWvhczvgdjsJpubkZenU
    TKgkHI5KQb9vgT2zS4dT00Q4EoOm6ZavKQz++IttX+z9N7IsCw899NA6l8v1hxMTE3s6O084
    BgYGkU7nmMKRMAeE5mWpiU2Wev5zXTdhmjnkcgai0TRGRsKkuNjnbWoq23zVVbvqamvrnnrp
    pZdeisViEwBklrhLFFRhzUvZk2S9bqfTKezcufO6XE79nYMHD3snJibPo9YtRoMTRQmyLCGb
    zbHf92azkABRlNDS0gS/34vu7kHoujm3lheS4LV+TyqVhaaZnCRJXG9vd6q1td1FKccRQvjS
    0tKqkpISTyQSAVP44hn4JBd5veD1eos1Ta+anQ2hUI2OEIJsVkV3dx96e4cgiiLWrl2BiopK
    xONRxONZ7N/fyfYHYBiAoihoaKjA2NgISktLkU4bGBqagc/nQnGxGxMTKTYDiLPoaJyiKHKB
    TxLfCT6J4zhCKeWt/sfu7l709Q2BUqCjYxU8Hg+OHDkJSrFgIGwhGCqsIC3sVzmv2vS29F2x
    XsKUaZqR3t7eAxs3btxx+vTp6rq6uordu3evfeKJJ6ZnZ2fDAORUKsUZhpF0uVwLBiEvWdLg
    WbNmbcvw8MisYRiqKEpliYTKGQadAz6mSZFMZuDzeZ2Kojjq6+ur/f5g04kTkyQvAc6D4wgU
    RUQymdRVVbX2vQFAX7VqVW1LS9uHpqf1+mTSQGkpHzl8+PTRRCJBmb9SbQnvMANHEVuiziok
    aG/RH5FYLJb1eDxZRVEW5PAWS3z4fD6k02kjHo9nRVHMiaJETZODLHtQXFyGurpy/dCh16an
    pqbJsmUrXVZCZR5kcWw+EoFhGNq+ffsyN9xwg3vFijbn4cO9hBAHVJXC5XJTURRTiUTCSCRi
    fre71J0XcVj8eQEcBIEDISZNp9PEniRn6pHnxbzcRTQ0TgC/VVdX97k/+IM/KL7vvvsgSRJm
    ZmYwPT2DmdlZzM6GMDs7i9lQCLOhEGZm8/fsbP7rUCiMUDiCUDiMUCjCPo9gNhTBbCiMnKpi
    w5oV+INPf1TYs33zSoHnPwhgM6U0yPp/3AwJWs73ks95YhWzT2zbtu363/vcZ7hgUdHcTB/7
    TCP7EEdCKfr6+/HDHz+MSDSSTw8t8tj5wYz5pueTp8/i3/7j29XT9iwAAHd/SURBVIjFE1jZ
    3gan08GGhtK54aELQZc5NwQyGPBhVXsLB0BQVS32kyefj/zbt38SG52YnmLBXs4CfZbkpG2j
    XExvXu73+71lZWULM8+2eT2mkQ9anU4HGuvrkEgkIDLJa6/Hg4DfB1mWAGritdcPYWI6hMbG
    Rpw+eQK37L0GRQE/Nq1bjYHhcYyOTQA0X7HkCYEiy8TmfOyOSLjU++WXAOAEAMcxfr8kSXC7
    XXDICjRVQzQaBTXN84ZZ8rY5JeeBoAVKd2/4dpLL4IzsynAJQsg0gFcPHjw4u3TJEvzOp3/b
    6/V4tiLfv2hxv9WZ2VAuX13L75PtWzfj1ptucHEc1wagAkCF3+eVPW7XgjPk83rQVF/HM3BU
    VlocrL7pul3E7XTOJWOisThyqkptdtHKWK4tKym65RP33+FyOR3Yd6AzToFTmO810ZkjmiWE
    TLJ7GvNDDO2NnwsyoIW3nRcPYHJsbCw1PT3N3hHzfNU4JuPN54c0WlxzLZfNUC2bRHhmGgcO
    HjZ13Qghr7RmejzufPBqT0yAoiRYxGgN0ADEJian0i/tewW5TAqGlgPNo++05WR13TBzqrrg
    ucCe6EFe3ap3cASZbE7DwiGE9vtSZcK5T37yk+sVRfn7kydPvudnP3vM0dl5HIlEYi57Txfm
    qRYUU+e/pjgfy+QbrScnYzhyZIgbGIhX19cvvffGG2+8q6SkpBn5OXNFAHxs7/5Ksrq/StIO
    APngBz/YLAjCZ0+cOFl77lwPFmtatuaRcByZU4YqLS3Gjh1bUFpafIFZSPOzS4A8oGptbUJD
    QzU0TcfMTIhVkOxzky48Uykv95sixcUl4qFDh+JPP/3EyNGjr/e8+uq+w08//fSLg4ODoYKy
    +BsOU/1l18xaN0VRKnK5nDeVSp0XqNmBcj6TbiAWS0CW5TkKvarqyOV05INggsrKIGSZIBIJ
    o7y8CkNDM8jldExPx+F2i3C7lTlwzrYFJ4qC8AY+ibyN/oitdz6jbRgmVFWDrhvgeR4Oh1KY
    mZ8fy0eBhZLPsO3FtwsLXRggsRgoBSB67ty5s5qmHSstLTNff/0gRwhpueaaazZ6vV4PAEcm
    k5E0TU243a65va2qGs6c6SXBYEnN6tWrm2VZLuN5PpjJ5BaoGwIc4vEM3G6P4vf7va2trY3x
    uOYKh+erR/keSB7pdMo0TdOyk2Z5eZlvy5Yt702l+PbR0QyCQZGmUuGe48dPTLPYlzKfEyWE
    TBNCJgghkyxBFrMxGX4ptc3p6ekUx3GTwWBwkYrzwjPOklSUKVhnKaWGorigKF6Ul5fBMLT0
    wMDgjMPhyOXPlFlQrc7PitJ1Sjwej5nJZNKnTp2KFhV5zUCgCKLoAM+LYL1qSdM0k4Zhqjxv
    j38WF5dRFAGEaGY4HDEW8UXnnTXhIjqkVgDv1zRNefjhh7OPPvIIMU2TMCdsiTYQFrwTAEQU
    BDKnwAXkhw5qOqUw5xWSqE2QeH7YEMKRKOU4rgaGcT2APvbmW8CIvwwnz9odOxsaGu75/d/9
    jNSydClMS3YbF6a8gQDpVBrf/s73QU0D9951O6iZ7xcqrB4RjsDQDRzpPIG/+Ycv0cPHjqO2
    ugpXbd1IeI7M/71FlfDm/74g8Fi9vBUet0tMJFNZSul0Ot+QPg1giGW/LYqReSkcuNPpRDqd
    LioKBJwup3NhszldKGFsVRIDAT/S6TS8Xi9S6TSSyRS8HhdAKaLRGB594mmsXrMWXV1daKyr
    wro1K0BNE8tamiHJEk6e6UZzfTUAE4RQSKJAbMGtgPMHxl5Wit1ilAZr2ThGr4on4ohGItA0
    DR6vG16PB/sPHISu69BUFbqeH0Cs6xoEnsPqlctRVhKcp9HhvMLSfHD89vgnewXJekqd4+Pj
    x59/4flr/tfv/x6Znp4u//K/f+XabDabATAAwIxFY2ooHAbQBEop3C4nPvKh+9F54lT5gUNH
    lgPIBvw+QZakBWfA7XSgvW0JefzpZz2GaUqb1692rl25LE/vZXsvlUpD03VaYDvqXE7l+vvv
    uKH46i1r8eVv/wSjE9PDDPxYUt4ZBj5CmB9gmLCBIgO/2kDdwaGhoYlz5855mpub5mZdLag0
    g8LpcGDpkmYcPXaMY2uajcZiZioR49RcFsOjYxp7vqqiyHx1VeUCgGZtioryUrhdTpLL5XTk
    m+Plru4eTzwWIelUEtls1rBlID1Oh8x5Xa6FSRkLfLN9FYpEceJsL8W8XDpd5L6oV3NzM3p6
    evC5z32u2u12/3EoFNoyMzODyspKVFfXzDWT2yexW055vpLEFYAJYcG/zSt4cXNBYzKpuxsa
    Gq/bu3ev8LOf/ewH0Wg0yzLSVm/sJbctlFKyfv16KRAI3DEyMrru+PET0HXdVj16o34jglgs
    gVQqDZ/Pj+np8IKg1srA5isFHLxeN1pbm1BWVkQzmTSyWY1MTYUu2H+0mHR4nsKTxJIl5YLD
    4aC9vb2R3t7eMGMyjGJ+lMevMpT6rVbrwfN8UTKZdFhzagorZYV9WNmsClEUkcupEEUeoihC
    11UWgOWV68bHh1FSUopUysDMTJwNyU3DMAz4/U6kUgYIMZEvinHgON4u8GH/+LYm7XieBwDO
    Aj2SJMHhcILnBaiqhkwmh8rKsrleJGtWDcfxoJQgFIohk9EWgKM3/vxtA0iGnV4Wj8fHJyYm
    Xqypqd569Ghn0b59r4p79uxaf91116aefPKpl9LpdC6VSqbdbhdbI471IoXQ2zsqr169Zlk2
    mzUJ4bwL56zl91YslgEhgrhy5YrysrJKX09PlFWP8ongvAgGoblcDlYV0el0yldfffU6QfCt
    7+pKcTzPwesl6WPHuroTiYQVOGTZubFo6jMstrNaJhad97NYRdvWE0s6Ozsza9euPVNeXn6d
    JMmcqmqL0nDzry0Bp9PF+f1+MxKJ5DRN1VwuLxQlg0DAj1QqkUgmk7GWlqVeTTPI/LnjbDRE
    E5pmEp/PTwHkZmZmMrquBYqKAk5Ny0KWZaRSWsYwjIQsy9z/3957x8d1nWfCz7lt+gzaoJIE
    2HsvEiWRlEhKsmVJluXeYydO4pI4zm6K42zWaZvdbMmu98vaKRtn7cQlLrJcFFvF6qQoFrEX
    AETvMxhMn1vP+f64986cuRiQlASA3O/z/H4jUAAIYs6cc973ed/nfZ5gMChoGgVjpFyUqdXl
    ikYVFItpI5FI6HPEIzbvAKmlpQWTk5PjAP7L2NhY09jYWBAV2V1XkSaEygBiQ3NzfNmnP/lr
    UrypEdSy6Whnz1/A17/xLZRKqnsp8k9eLtdAhfM/4SQqotv5INUOZQtJG6oXRfE9jzz8YNuB
    fXeBCADo3DNE/P+vX7sKf/C7n4OiKJicnIIkiYhGwhAJASN2wUZVdYyMjuFnzzyHf/r29+ml
    Kz1MliThXQ+/lWxctxrUmmP+CLPV7AgIVi1finhjvZjLF0RnHV3FMleYwZXenXeQBACarUgY
    DoVCiqLIZZlhXrq4SkiAAXWxGEzDQDQahaYbSGcy6GhrBgC8fOw4RsYT2LvvIL7zL9/Cb/3q
    RxAOBWFRC81NDVi5vBPnLvfgrQfvgE9RQBh1qB9VFQNxoSvar+fR0NCAdDpNAAhEEGCZFN/9
    3mP4l+98F4ah2x0PMJw8dRoMzD47gC3QQIBoOIz6WAyt8SZ7K3rnksrrXA5NN7Nix7h9NsIY
    +/lTTz2195c/9kuR3/2df4NcLtf59X/65wc1TfsZgFKxVNJHR0fB2E5HgIRizeqV+N3f/g3x
    977wx6t6+/rVSCQsyrJU5flECMHOrZtQVxcLa5oWeuvhAyQaCTvy8pXijDOy5e6HJlmSdj5y
    /91LPvDI/Uim0njh2GmNUtaPivyuew9lURmATWO2R8fr4s473zOaTCbPPvvzn685dPAeuK+J
    ge/8UASDAdx76B7y4588EcgXCjIALZVK0VwmBbVUQqlUcoUjfA31dcr6tatnzwoBWNbRjvq6
    mDCdmnHpesWpRJIW81lxJpWCqukmB/yiq5Yvk+vrIpUZJm9niwFnL/XgUu+ACaBIAIPZ6+U+
    TSyAZ1A8HodpmpAkqZ5SqoVCoVduv/12kVIqWxaVKbUU+yOVGaMiY0xUFMUXDkfC09MzAj8r
    EAgEEAr5WTKZpJZlUVthS7AEgVBCiEWIYNkfCRUE1cpkTEtR5LpIJBJKp9MSY0wilenlxbhb
    2O7du5eYpvXWy5ev+AzDQiQSQamk4kaEGVRVw/HjryEWi2HDhjUwDBMzFYokRFFCOBxCW1sc
    bW3NVNdL5vR0gjQ1tUhXrgygVNLn8FiaDTpcCp5Nd5TE5uZmcWJiwvLE+hIXmwwsjCogIYSE
    DcOULcuqKWHuNbDUNAOCIELTNEiSAL/f54AAgra2OHw+glRqBlu3bsfFi5NwaVOqaiKXK6Gu
    LoSpKQ32XhNsMqNAasWiRcljrkOxcwCS7YG0fv0arF+/DpIkOsCAoK2tBRXHFff+IjAME2fP
    XkWplEa1ql013a4aHN20eOQCcOoo2+H8+fPHV65ceX716tX7X375KF5++RX/vn137L/33nvx
    05/+9GQqNWM0NrZQGzCYZQn7S5f6EI1G6u68884tsuzz6foEV3ARyiA7n9fFbdu2NSWTJXF6
    Ol+zmOAomoqiKMp33XXntpaWzj29vUUlnzexfHkYmpadPH/+gluwKxvZOvNGriBDHhU/rdcV
    k1zK8szMjJZKpU61trZONzXF42Nj43MWQaamktB1U9qwYUPw5ZdfnlFVVQ+HGxEIqAgEAigW
    kyohRG9v7/DnciXBBpCCR8SFIp/X0NTUJMqybGqapmuaqoVCoaDPJ8Lnk5FMqgXLsgodHR31
    0Wh99OrVYs17zj1GsiyioUFiV68OlvL5vIbqeGRgIYxiCSGYnJyEQyv5LoCIM4vUCFt2t9F5
    1qEyDLWus7Nz6Uc+9EEsaW+FaRogjOGlI0fwgx/+mDkAyZUYzns+lrjLM+tsgCIWebA+GAyy
    YrG4uq2tdc+B/fvws6eeRjqdxkMPvAXhcKg8F1QbwNgysmvXrML3HvsR/vjP/xOLhMPYtGEd
    aW9rgSxJSGcy6Lnaz06+dpZ19/bpJVU1BUHwPXDfPcJH3v8ofIpsV8BqeCV5VfHcZLK5qR4r
    OpcIfYMjIiqGcLwGfJGjBC2IUSwAWZREgQhCJXnnaS78PAOAaDQCURSgKAoEQcT4+Dg2rltt
    K9f98F+xY+cuXLlyBV1LWrFr+5ZyZVxRZGxcuwpP/OxpZLNZNDXWwzJNmKbl7QASZ5bsVmj1
    Ix6Po6+vz45NDr1j184dqK+LIhgIIBQKIhQIwOdTIAoCnnzm5yiVSnjHQw+AgEEgBLFouNyV
    rVAdWDXj7sa6GAsZkPghXEYIERljr/T09F752ZNP7frNz3waf/gHv0dM01jxzW/9y4Oapr1W
    LBa1waHhWaIA9+y/E3/0+X/j++Kf/2dfzDHS9c70rVuzEhvXrhaLpSL27NjCzQLav47fp0AU
    BdeML0YI2Xzwzl3LfvVD7yB1kRCeev4V9A2Npp37xvVlsWAr0+VRLcTwZlW2GIAcpfS5Z555
    5t6PfPhDsc2bN8HyyHy7BsmHD96Dg/cc8P3wx080AUjm8gUrMTUFvyKW6WYAGrZu3iiv6Fpm
    z62Vz5l9T7U0N2HLxnVCb9+Aq8Zn6prOdE3FZCIBTdPduyIai4TbD+27nYRDQdsw0DNHSADk
    8gX867NHkc0VVABFVq1GpaGGyfd8PE6dOgVZlrF3794rzc3Nv5XJZOoymUysUCjUa5rWaBhG
    k2maDZTSOkppiFIaWL9+/bLNm7dsff75I0I+X3ToGyKWLevA5s2r2DPPPF2cmZkpCoJQIoQU
    CEHR/kiKACkRQlRCSFEQhKxlWclMJpPz3C2LdrnEYtH16XRmZamk4tChu2FZFo4cOYZSSS3P
    DdUCSnYCIWPZsqXYsGE1A6jpUH0Ew7CIw0KgkiTSXC6n9vZe0YPBoK+zsyvU0zNMJiaS1+1Q
    VQ9zVzxhdN0S4vG4zMckxwizhIpircYlsfMq8c0YUyil4mxK2OzquP07G+WiCqUWQqEA0umC
    0z1qxejoMOLxOAoFikQiW+X/NDNTRGdnDIoig1LizEQQ2GwbXOt50yl2jAHj41PQNMOVqoZp
    UliW/bXly5dClmVcvTpSxneGQTlKHfFQQm+IakcWKh55xYNQbbVB+vr6RjKZzLMrVizfc/78
    RX9vbx8kSQ7ecceeA4cPHw5nMhmtvX2p4ff7fBWVOhtAv/baZbJnz6ZYR0cYum7OonyZJsPk
    ZJYsWxaXJyenoGnVnV7GAEpBRFGUACg7duxYsW7dlnXDw1pwelqHzyehvl60+vr6R1KplBu3
    XFEY3krhmvnc9Qp2nE8jzp49e/7uu+8+s3r16sNTUwm4ocV7xotFFZcv95IdOzbVj4+PZQuF
    vNnaupQFgzEiST4wxtDc3ByIx1ti588PgFJW7sbz1NdkMosNG1p98XhcnJmZMRmjliz74PcL
    kGWR5vP5kiiKZNu2rR2mKYXSaRUVzbTZMt/19T4IQsnq7u4pUkqrFHsJIXynen47SNzcBN8S
    dluWRVS4+u5Q1Mq2lhYx4PfBMAxQyzYKbWpoQDQawVQiaTlvbA4Vs8o0Kpx+E9VD3nz7cFGA
    ktOeX7lq5YrW4eER/P0/fBV3778L9x2+B2EEKxuvChzZ1EFKLZw5ex5f+vLfsp8+9axZLBZL
    AMwnnnxG8PkUIggCDN0wdcNw3zSfIsv1b7vvHun3fuuTpK0lXp5zmk3h84AkymBTFhkioSC6
    lrYT5zCVpZ0dedoi91Q9Sd58UUMIAGZZFnO7hnYFn+sccRVpAoag349gIIBisYRoLIbRsQkA
    wEtHjmFsahr7D92Pb3/zG/iNX/kgIuEgLGcvAcCalV34dqGEqeQ0mupjKKklZHN56knKbhqd
    rtbj2LFjfJQGEQg2b9qIzZs2lOfRXGBMLQvJxDQmJifR1tIMURTAKAWlptNZYjXqRdUYxRUb
    WgxwVMMDyA1KrknuiKZpL/z0Zz/b+r53v0vuaG/HH33h80SWpKX//I1vhYqlEhsaGoZpGGWB
    D9s8meCB+w+hqaEejFHYFDsKfq6tPhbFJz76XqhqCU2N9eWutWtq2lAfcxUOIwDie3dsjv/m
    r7xPWNIWR65QxMsnzrFiSZt0fmfZuXd07g4qYg4hhtc7kO9UME0Axy9dvnz6+499/8Dq1asg
    S+Js9TnGEG9uwud+89NkcGi48czZc4Ku6cLExDjWLu+AIosCgKZIONzy0FvvE+piUTBXQAaV
    OyISDuKBe+8hz754NDqTzoQAiD6fTGAZmEokoWq6BiAsCGTdwX23RQ/csWu2R5t7bglw8uwl
    vHz8jKumxMs182tlzPf5cygpOHr0qOlSCz0ULffeKzqMBr/P52sslTQhl8ujVNIhihYEQUQ6
    nXMpJmahUChyMchVJnQ9wlzwWHC+pqHaB2nR7hhBEDtyuXy0sbERzc1xHDlyDDYdxpXPte3e
    +E4ZQBCJhLFx4zp0dLRYV65cLpw9ezZHCFGbmpqsWCzGRFG0VFU1UqmU6fP5gps3b26Nx1tD
    V64MCiMjE2Ds+oaytShrjDEUixoaGxtlRVGobqMPlQNGBU+MnzdA3dDQgFQqRSilVBAIrajv
    1ZphqDwti8I0bfPyUqlkF0Qxjba2Jqd7lMLWrTtw6VKle+T+vGxWhSTB8X6xARJAaalUsm7F
    eOQFaIlEEslkyknmeZNYEcFgAKFQCIVCr0N1Ep05NGkWzqkGRsSbSy5asa4GSHLXXwdQ7O7u
    Prpnz56BpUuXrrtw4RJ6evoBkMBtt+3Y09bWlvD7/QgGA7AV6StzOIWCiuPHL2J4eArpdN4D
    AOx9NjWVxblzw5iYyM6imFEKGAZDNBr1b9u2rW337r2tk5M0OD5eAiAgGpUhCFrhypUrKSen
    di0FNOfc8ErEDG9eAZCePHkyuXXrlifXrFm1p7u7Jzo1layaG+J//6tXhxCPNyl7997ZXioV
    DL9fQSAQBWMKAoFAYPv27Us0zQpNTKSqqMuVYoSAZDILXW9TduzYUX/06NGSKEoiIMHvlyEI
    zCoWi9bu3bs7OztXrejpyQmaZnmUOvnij4iWFgVDQxe0sbExb0wqco2X+QdI/CJyVaCiA5Zc
    nr6rMR4GIHd1LhPC4XCV708sFkHnsqWk92o/Q8VNecbhUSa5dqGF6jkG3nTQXIjuhxcQOnSx
    WF0s5iME2LB+HT75a7+Chro6UIvCskwcfeUYwuEwtmzaUFbLyhfy+PG/Pon/+eW/Y+cvXlYZ
    Y27QLVBKS6WSyvtnBAEsa443Nr7v0Yfkj3/oPaStuYmbc7qGr1KNbpJAgOVL2+H3KbKq6QwV
    5asSqk0ci1gA3rczi6aWiiXD0HVUhqF5mg6qqDr+gB/1dTFMTyfR0NCIxPQ0ZmZm8IMf/xQ7
    d+3GlStXsKStCbt3bHUARKX71BJvhM/nw8jYJNat7ERiOoV0JsdXifikid3M4ORe0oqiQNf1
    yqd4dUdHtIO5qpCWhXA4BF9aQUlVITkACU4nqVL8YVXGoLNmkxaTy1BRvfTSGwxn/x05ffrM
    u0+eOrX0Lfffi/a2Vvzh53+P1NfFGv72f38Vg0NDLJPJkLqyehp1DJEJbt+zwwGIVnkvlLsa
    BDi47/by+jBa7RUWb6xHR1uzODYx1bZ352b5dz71YWHtimUAYxgZn8K5y726kyDL3D2jcknc
    fAYjBsAihEzouv6T73zne9vect99sd27d4Gimo7qgpRdO3fgT//9F8Qv/ulfNJy/cJEMDg5j
    88o2xOsjUu8Auu7Zf4fv0N37wM/5eWeaDty5B4+87T7fP337sQ7DNI3mhpigiMB0Kg3DNH2C
    IGzev3dn86c+9j5SH4vYSp2oNoolBEhnsvjuT37OkqmM6un+5wDkaxh/zvs5qt5yzOIqq273
    vABAJoQEmpubtxaLJdEwzPIcku28rsM0KWlsbKSjo6NunPF6WxVQMQjWuIReXeyiHQBCKZUA
    QiYmJlEoFDEyMgZXotnv92P37u2Ynp5Bd/dVR4BAREtLM7Zt28AEAcZzzz2b6e3tTVuWlQWQ
    m56ezruvxe/3K9u2bevasmVrh6qakVOnLpFUKuMRMiA3YERbDUSKRQ3xeFTx+XzgAJI3JpXm
    m5ZZLNoqx6ZpqpIkm3axvrZEMP9507Sg6yaCwSBUtYRAwAe/31dWrmtqakKpxKq6R5X5JROW
    ZSEUklEoMAQCEhjTaCaTNT1xyDuzdxPjEiPV11i1bLf71DQDPp9t2eICpMo6wtNJInNIft+c
    mOQt3BFCTMaYdubMmcFt27YdXbly+dre3n5imhauXh2EaVJl795d7aFQyC7qT814xEwISiUd
    AwMTnLKfMGsv9PYmqsCFCzYYI8jlNKxcuSrS1bUqkkhAGhkpOusqoL5exszMWHpsbNy1pWBc
    3s0Xoebl/iGEWIwx/fjxEy8fPnz41KZNm+5+8cUjoI5YlFeswTBMnDp1gezatSW0fv1KZlk+
    hEIKLEtBW9uShqYmQzh5sluo0HKr55kEQYCmmejpmSSbN69oEgQBoVDEn83K8PtlyLLB1qxZ
    3dHQ0NwwOloKTE0VaqhzViiNjY0+iGLROnfuXM40TT4e5bj4PSvnnU+A5CY6upM0UC4oueBI
    YIx1hELBxg0b1kMSRZimXg6wkXAYG9auIc88+4LIJUwZZwB6EvbQWQbV8ws8TUydz01xvYuD
    MVZIJJPGHbfvkR584H401NWBMQZRIDh3/jL+5D/8JT7w3ndh66YNIAAGh0fw91/9Ovv6t75D
    p6dTea47xmvTq87r6pAkceWubVtaP/HR94uH9u9FwO9zjFWr6XrXMootf6/TLVjW0YKA3yer
    mi5wQV3lKuEFjj40b47V3CWUTs3MlArFYsylN/GzR8wFfS6VMeDHsiUdGBoaQjgcRi6Xw3Mv
    HcV4IoVD978N//T1r+HTH38/YtEwTNOoSv4ioSDCoRAmE9OgloErVweRzRdMVA8rmoQQy1Px
    vWkP0zSr2A2MUVy5fBnnL1xEOp1GPp9HsVBAsVSEYRgYHh5BoVDAF//8L8sdSkEguGvvHrzl
    0P6aNUnm/pfhpg3H1qjcGYQQjTHWk0wmTz319NNL77l7H0RBQDzeiH/7ud8kXZ3LMDIyUpGH
    96rAMa+Soyd5d2Iwoy7IcGYFKUVjXRQPHLxTWLms3ffR9zyI1V1LwZgFAuD85auYSEy7FSYJ
    1Sp8cwajN2Oo6+xJFcBLV7q7j/zvr/7jW9asWU0i4ZAt5uIBSQQM9xy4C7HofxC+9NdfwU9f
    PIlcLgvTNMXNG9cHPv1rH0dTY311EYE7d5RSRCNhfOYTHyaqqkZ/8JOnmEgIzl3qwdnLV1k0
    Eqp/68F95Fc/8m6yqmtJFQCtpsUy/PzlE3jp1TMWF4DcrkuWEJLlAISJhfM2KXcnHbEEgWM1
    5ADIjDGxvr6+vrGxMdLd3e8oc8nlAKvrJkolnTQ3xwkqM5tpQkjSiUVJ52dZXMHFS1c2Frpg
    x4elUqk0XF9flxZFsfny5e5ywiFJElavXonOzmUYGRkHYwR+vw+rVq3A2rUr6cTEWOno0aPp
    RCLhxqQUH5M6Ojrq7rzzro3xePPawcFxX3//CFR17pmjWvNGtY1o7VlbRYmJkUhEyuVyFIDJ
    GNMd2ou7nqX5LmS5lgqqqqb9fn9JUZTobCnz6vkjd4A8n7eFgyzLhM8XQkeHPXuUTCaxfftO
    XL6cgGFYjolqtVy8rpvw+/0QBIpIREGhkDDz+ZzOAW0T1cP0N699ZL9+4v65sbEBzc1xxyjW
    B1lWHMNYEdFoBIoiY+/ereXXa9PypjE8nOT8kapnjq4BjshixiMPpYwCMHK5XHZiYuKllpbW
    h5uaGhsnJhIACIaGxqCqR8iGDWtQPUMjeLzDhJrUUp5KVjkL1YWD6WkNjY0heWZGx/h4CZTa
    4EhRRITDAj13bjCrqqorKmTCpnsXa3Rc5+OOpQCMc+fOja9fv+4Ha9as2jQwMNQ0ODg8h2ea
    AFVVceLEOZLPF8nGjWvR1VUPSmUoSsDf2zuM4eFJjwLi7DM3OZmB369IGzasbg4GgySVAqJR
    GcEgkZualnT096eF0dGcszakhkgMgd8voq3Nx3p6zpVGR0ezTjziYxLPBKAL1UFiHFXG/cgr
    hQmOiMLqlpaWlm1bt8ySpVZkGdu3bkIkHPblbLNMDbaiUoZTiUo5P5txVD6zxnNBQZLzWi6e
    PXf+0je//Z0dv/6Jj0MQBZi6gXNXruAv/+v/QDAQwKG798EwdLx05BV86ct/x1548YiuG0aG
    A0fucLcbjAQAWxrqY7c9+tBb6j/+ofeQFZ3Lyga7XvEFcAnKXKp5ZT8kgaCpoQ6hUFCcyeQU
    l/LGGKMcSHBdrOe1csUlw5OpVCo7Pj7WumHdmtkCDVUdE9ujZcumDfjKV/8JLa1tCIgM33v8
    J7jttttx6dJldLQ0Yu/u7Q7tjOuYMIaA34dwOIjUTAa5XB5nLnQzTTd4oQ/v0PhiJTI3sl5E
    IASmaeLpnz+Hn/zkCQSCARQLBUiihCVL2uFTFHS0t0EUhfIVk8nm0D8whLv27nY+Q8tgiGF2
    54gTaSDXoFgseHziQEeKMfbiy0eOHh4aGgqtXL7crrgGg/jQ+98DVS3BJ8ugvCpkLQVEsFl7
    ywumeEqnIAp439vvA6UWAj7FoeABmqHj9IVuFEtagbvJDad7VJgrGL1Zd3dnP+oAJiml3//h
    D3+0+Z679y9596PvKHdZ+a6r+3HHti34L3/xJ/jxEz/DY4//COH6Fnz+k7+CXdu3Vq0NL4TC
    OJDU0dqMP/jtXyerli8lP3nyWfznv/8uli1dQv7DFx4ld9+xC9FwsKz+5+1CEQIMDI/jmz94
    kuUKxSJX+ElzzyzeoNTsG1xDV5nKZRvkuJhE1q1bF1YUX8vkZHIWrcowLKRSWTQ3t8rBYNCl
    QucYYzOEkARsgaC05740uTvUfS6UwMAsht3Ro0fP3nHHHT++887b319XdykwOZmALCvo6lqG
    9evXoLe3H+PjU2hubsTmzRtYfX3UPHv2dO7MmTMzpVLJBUZJJyalBUEo7dixo3P37t37dZ0t
    f+21y+LUVMql811j5shVALy+H5I9pyJIkUjELaR6VUVRK56/WUDtdOqRSqUmOjs7c5FIuGV6
    eqYmxY4HSYwxJJNprF+/DKVSAYT4sWJFCCMjg4jH41BVYGoqjYo+R+XnWZYtFa4oIhRFRjQq
    oq9vXC0WS/qtGJMq4xP2nEhX1zKsXr0CpmlBlmXH06dY/ugKNAA2nTMaDWF8PMWDrZoGsa7s
    Pg8cbvKDujLVFy5cOL9s2bJLnZ1L75qami7v32QyjaNHX6sCQdVzNLUVHL3Gy96uqyvDXyya
    uHAhBUqrgVYoJAFQjYGBQQ0VA1ltIQt23HqUnnvu+SPvfve7n92xY+s7k8mUUCppc9JpNc3A
    +fO9mJqawfr1K9HW1oKLF8dx9uxVmKblUdgUap674eEUNM0SVq9uRXt7BIZRQG9vjoyOZkku
    p9cAp9U/o60tCE2bNl977XTW6Yy7+Xfa8SvMegDSgnSQqIdi5x0wJLCHm7fs2rE9tqKrszIn
    wCUq27duwsrlncrpcxcCXMLu0hsyzsVd5LiC7DrPhQJIBMBALpf/f778t3//2RMnX9uwaeN6
    uVgssldePU6HhkbIn//xFwTTNPHfvvS/8PVv/Is5ODyScwKqmzS4lUgXIAUBHFyzcvmeT/7y
    h4IPP3AvIuVBaOqplHOgaA4xCO/3UsoQCvrREIuKI2OTPlRMUr3msJinRK9Wsp3M53Oj586d
    X3PPgf2VxNaTcFUq/MCeXTvwncd+iKNHjqCjrRm+QAhvXbsW/+cfv4pPf/z9iEYjsEzT+RkV
    TylZlhAOBpErFNE3OIrXLvRQ2IpaGqtWRdS5rtItgY/KB1SS8MH3vxePPPw2+BQZf/e//xGZ
    dBq//zufgyJLbmhxElSGo68cx//48t9h2+YNEAQCanF3XA1VO6F6BmlRI1MN81jTGc5+7erV
    voFjr57YuGJ5V9nniwAI+HwOGOakpWt0MmrN6syWk+f+LqX2fIZjuOsm/MlUBpd6Byh3p/Gz
    lV4TWDbP96lOCCkxxk4mkskn/uZv/v5jO7dvk5d3ddpdMI84gttJbmyox0c+8B4cvmcfTNNE
    e1uL/XXqNaNmVRQ5l27YVF+HT3z4Pbhzzw6kM1lsWLsC9dFImdpZNXOEijCDpun43k9+jrOX
    ew2uSucm3SnHrHDOYLQAAMnggHeVQphzf0vbtm17Sz5faJqeTs1yhAeAqakZdHW1+lpbW+W+
    vj6XkpzjilsJruvPx0Hq6SwtSqJ77NixTKlU+ts9e/bkd+3afr9pWs2MMUKppYuiWJfPF5WN
    G9dh1aouK5tN55566snU4OBgllKac96nhPOcDgQC2sGD92xdvXrN28bGkq2XL/eRQkG9Yfrc
    7Eo6qdlpsiwKxgQhFAr5a8Qk3lB4XmOSJEkwTRPDw8OpFStWDDc3N68aHBz1mEuSmnNJk5Mp
    rF7diY6OpSgUdBhGEclkAjt27MaVK3b3yJVs5teCUtv3xecTEYkIkKSiNTg4WGKM8fHololJ
    xB6IJIzZIhMXL15Bb689WL9160YEg0EcPXraATcVCXxAQHt7HNu2rUEymXG+jipKHQ+OuJSD
    3CIxuEz9vnTp0tSBA/uPLFnScfuFC92SquplTzCbvgrO72s2AKr1eRc4e7tH1fePLeZQTUET
    EIlIyOUm9WQy6d45htM98prAWm/mzNRgeZgAtKmpqemzZ89+f9eu3VvXr1+75vTp81Wvh7cF
    cFQQMTWVQjqdRzgcQqlk1ABHpOb94P68RCKPXG4IDQ1hqKqJQsEs0w1rAVT3z9GogsZGsJdf
    PlmYnp52wVHKw97iGQ0L2kHCXAdaFEWYprm2Lha7561vuV+MRCKglllRdQbAKMWS9jYcPHCn
    ePbCpSZKaYi7LCWuJO5WWG5KZYXbbCaAl/L5Qv7Z51849OzzL6yGTScUgsHAhudeeDn2zX/5
    nnXq9LmcpmmuHn0RFfU9NxilATSJovjQHXt2bv83n/kV6bad20AIyon/9TpE1UaNc1HugKDf
    h1g0JDi/p4zazt0LpRjDAGQ1TT9z7NVX92ezHxIj4VDVfAxDtXEmoxQtzU34wu98Fs889yJ+
    8KN/xe2378XFixed7tGOKtoQOHoeIYCiyMin03juldcwMpE0ABRYhbLhypq76i+3CkCqQkp1
    dTHUx6J2HY8xpDMZSJIERXE7HZataEYITMtEIjkNQzchCIIza1PjpLqf5FRqbgZIqhGQdAAj
    mWz2xIsvvrTx7Q8+AJ9PBi+4UD1PRWsCItSQjvcaE2MWaKJlg2UXUI6MT2JkPGFx9015iJzb
    Q/Od7DNPRz4N4MfHjh/f+fV/+sbO3//d34YkijWBiqtYCUbR3triGEWb1Z5jnrmh2SCJQhAE
    bFm/uiwl7wqDwKOOWaHvMpw4exE/+NkL1DStPFcESnHPNKqHh+l8dQPmAJhuPKqpI/zOd76z
    JRaL3Xny5NlAoVDiTAYrSf3MTA7FoiGvXbs2NjAwIFFKvcIi7kC0OUc8xGLFKYf2Tc+ePTvV
    09PzzaVLl74WjUZX5nK5ukKhUHf48OG7Nm5c01oqFQsXL56fPH36dCqfz7t0wBwH+JJ1dXXm
    Aw88cCAeb3mou3uo7urVISepmT1sfj2PJT4xtDtK1VQix46IBAJ+Pg7JHgbKvN9Llm2RwC5f
    vpy98847z7S3t93t810kvIBFraq2rdSl4cSJy1i6tBUrVrRjaGgQ8XgzNI1gaipTUyrcTTYp
    ZZAkgoYGATMzE8bIyGgB/5fEJFXVHEqZ/ZYEAn7njmHO+1sBQoJgS6C7SXIl3MwW53P/DpmN
    mhY9t+OLdk7XpDA2Nn6iq2vFZFNTQ8fIyOSsztDszkdteubs7lGtfcJ3U6pnk0RRQCQiYmBg
    XC+VSrx/E0/rVReoO28RQnTGWOnll1++2NXV9dimTet/Y3R0IphIpGqaS1dekwjLoshkCmXx
    jlodVl6Fzrumum5hcjJX9qCbS0TF/R1kWcTSpQGMjfVqly5dynDgKAVg2inYuQCpppm7NB8b
    6jqUKgAgpmkGALxrz55da+/ev6+q8skHa1mW8eBb7sWPfvp0fU9vX7tDY4jCFncIo6L3Tvmg
    tAAB9nptZ35z9jDGMgDaATQBaC4WS+pjP/zxEsbKHiluIuUGI5fKkAKwxOfzve9t992z8bc/
    /SvC6hWddmJLOYCDuTpE1/JAqh5Up2DwyRLCwQDBbMduPhgt5JppAI4dP37yA+fOnW++847b
    Zw2N8zQ55uzXDWtX42pfP/zBENauXYt//Oo/4Nc/+l7EomHbw2ZWwovy/w+NTuHk+R5mmhav
    olVA9UD1Qle0Xz82KgdUahvCGgaWLV0CURCQSqXKAJpSW7CBONLP69esRjqbxdj4BMDszkg4
    GKgGA87ykNoE8EXrKNXoIhmEkCxj7MSp02feMTw8HF29emV5b9eiyzGvh1Ytqp23M8l1j6q7
    TqjqOPX0D7tzayZH2So6yo9u92hekv0aFTu+W9Wn6/r3/+W731916OCB2B233zb7DuW78Q7Q
    YcxRsqwxk+U1pa4GkRQWJwxSDagwa7YrmUrjGz/4GRubTKioUOu8wYin1y0IBZpfe89act9i
    69OuXLnyzny+sLevbxCuz4s36dd1E8PDU2Tt2s54R0dHw/Dw8DhsT7+A81ScO/O6kuWLEKMY
    IcQEoJVKpVx3d7droN5ECIk/88wzpYaGhubR0VFtenpa58QrqmJSPB63HnrooQdCoejbz53r
    CQ8Pj5cNYvnEZ7Yww43LfPNfYwywLAafz18rJnmpdvO+ZtlsVpuamjrW0hJPtLTEm0dGJlHL
    /8ib+KZSOdTXx6DrJSSTSezYsQs9PQnoekWy2fv33c8FAgQ+n8ZOn75YKBQKrshUHraHTQEL
    aLfxxup0DC71yzaFlZDN5pwZD1+Vmh0vSDAzk4OiyAgG7Q4TpYBlVecFbgiym7pzsiluVuHO
    AqD39PT0rlq16nJHR1vH2Fhi1v6oBXhuvMs6G1jU+jpAoCgiZNliU1OTBnd+veBoIYyVGRef
    S5TS7JEjR55+8MG33b558/oDL710HJZFawIb7zzWXCqR11KO5JXuKt06Yc5/DyBoavJDkorW
    yZMnc6qq8myGaa6LlLtWwU5akNPEgSOHykAA3NXS0vyBj374Q76WluayApI3qDNqYeP6NXjP
    Ox6U/+uX/maFqmnjsOcS6hzzq6yzCSwuiShLhi7yw60iFpzqKOHQfIExRLkLnt/MvHx5ZzAY
    +PB73/Hghs/++sccCW9ac4aokuzRWVSZWcINrlqXB1DJsoiA34c5gJG4ENU6LvFjhBDKGDs3
    PDLyyj9945sPb9q4AeFQsEKVqkGTIgCmZ9L4wY9/il279+Dy5ctobarH3j3bZyeI3M+wLApd
    N3D60lVouqmjWlErCyDneNgUvdW6m7CXZh0hZ0AMTz/zLJ544l+RLxSQy+WgaSpePX7C8eGg
    oBz1iVIKXdfxp3/53yEKIghh2L55I37tYx9AKOCf1aQQBaHslj1HYLoZXSQNwIXBoaHhM+fO
    bVy1ajmXuGN2d6jK0wjgIR+BbbxcJm+UrY8qgKt6loeWf75hmOgbGoWmG27HWq9R6V2IYOQN
    SCrs2ZeXrvb13fn1f/7WA5s2rHfOTQ1VOu8ZAmA3b90VIWACAEbAmGC/ZkpAQQBSMcpiNbpv
    qDHHZFkWnnnpOF48ZgszECDLKhS0pCcYzaIyLNRZ84Il7h4iv/qrv7pUkqRf6u3tjycS01UU
    DW/AHR1NorOzNbRr166uRCIxrqpqCkAMFQN0Nx4ZWDjRiRvdM666Yh62eToAGIwxdXh4ODc8
    PBxyWQ6emJQHkG5qarIeeeSRt/t8wUdOn74cGh9PcIms8Lo7RNcCSW43yi4CgSiKLDl+aN54
    tGAxyZHUt44fP37uwQcffGXjxvUPJ5Npxy9vrkRVKAOD5cvbMDw8iKamJhgGweRk+przFDbA
    EBAOM/T1Dend3T1uTKpSekS1qtZNi0l8XCAE6OpailWrVkJRZCiKAlEU0draPIsG5v5ZFEXs
    2rWmXLNMpfK4fHkMthaRF/uwuWaQbhqrwQEh5uXLl5P33HPPydbW5gOBQEBSVeM6QGc23Wv2
    xEntLmWtjqX7PX6/CEo1mkwm3flGzROPtPm8X69VtLt69epIf//AD7q6ujb29w83DQ2N15D4
    985hzabE8U/XLqD687XWR6g5I+h+PhCQ0Noqs8uXT5eGh4d5at20U7BLoZruXTOGzztA8oIj
    h8O6Rpbl3373u965/P57D9s8uVpdA3cwXxLx/nc+jHPnL9b9+KfPbKEVKWzejJGvrLhmk4t5
    kfAUGNExVLMAaIyxgvP7uhRBAbOVr7IAlgX8vl/6wLvevuFzn/pl0tRQZwPHOWaI7MSEzpL4
    5b9vLqNY5gxWi4TAp8jgAo83GAkLfCFZAFKU0u88/qOfbL/rjr1L3/fud1Z1fVgNqtSRV17F
    +NQ07n3rw/g///gP+OUPPoq6WBTMolW0Ib6DpOkasrk8iqpOnaBTpV7iPL1+AbfEwzV3tX1+
    gHA4jHi8CQG/Hz6fAkWWIEki+gcGQS0LG9evrZwhSmFZFkzThG7oaGuO2+apFf268kMQCJ8W
    32xaAy/5PZpOZ869evzkxoceuB+SKFZ77QAgggN+qD1rpRk6SqUSNE21O25Od82V+ZYEAbIk
    wqdIkGUJAiFggv33GSE2NQ2VLkmuUMDQ6CTDbKVMXjFTn89kf46A5Fb5JxhjP3rq6Wd2PPLw
    21rvPXh3xUuM6+YQQgBRAKO255qm6ygVi87aaNCdtbEsy5ntYhAEAlG010eRZfgVGT6fbANo
    EEAAKHNm2jz/3sj4JL7z42dYoVgqwgZH6TmC0YJ2j64Vk9x1ZYyRQ4cOheLx+EcmJxP3nD9/
    GZZFIYrSNdSYDHR3j5IdO1Yu2b179+qXX345RSl1B3xzntdUprbchKIdL0whODHRAFBijOWd
    Ip4bkwgXk1QAxUgkYj7yyCNvDwRCj7722uXQ+PhUjWRkbrGFuel315b5do0xRVEUHIAkeOKR
    sNBdgu7u7lR/f/93ly/v2j46OrH08uW+OShylY+u71EikcDOnbvR25ssd4/4zhH/90VRhKKI
    KBSy9MSJ4/lcLsfHpMwtFpNINaHB7vbpuo5SSYVpWrAsCkrtwlUsFoUgCJiezlYlyS6lShQl
    qKpRpuNVkxdckQaB3GxgxN/BzliAyRgrJpOJMy0t7clYLNqqaSkPAKqkTS4zQxBs9UhJkiFJ
    EkSx0mGrfL8t6e36BFcr/c2eTQoERGha2spksiofk5wiGm/AzRZiPVA9K5w/evToq0uXLn1x
    3brVj0xOpohhWFVdnGrwg7I0vCRJkGUZsmyviyhKjmeWAJ5iZ6evxDHOdbuMQhWQmj2DJKCl
    xY9SKWmePXsmRynNcvEo6Qi+Xbd7tGAdJGdBXXDURQj5w/vvO3zoNz71SRKJhGGZxuy5AY7m
    QSlFa3MT/u1nfpVksrklLxx5dS9jTGOMFZ2NwF8cs/TLFykw8ckLH2wKjmxgGkCQMSZzX3el
    jFUALbIsffTRh9665XOf/HgZHM1Fj+MV/2ZLe9f+Xi8tz+3YiTYnmN+JfJVuIfm/jDFmOjzW
    U9PT09//yt/+/a9t2bzBv2HdWlgmK1eqCCEAIwARkErN4LEf/St27d6D3t4eNDdEsf+OPRAF
    wQHagk1HFAioVW4doFgoIZFMuZ5aLsAuK2tdS3L4VgFJhBAcvOce3HNgPwRnsLViFGsLgCSn
    p/HIg2+FIBAwxwOoPE9DKSxq2bNsVb41LkBySOOVSuHN7iRRxyQ1A+D0mbPn3z6VSAaWtLWC
    OTNVumViejqFqcQUhoZHMDQ8ionJKaRmZpDN5qBVAQBWBoKiIECWJYSCAdRFw2hqqEN7SxOW
    tjUj3liH+lgYPlkCEwhAGTLZAkYmEgzVw9PeQerrUqvmASS5c095AKdGx8Zf+t73H3/n7bt2
    klAowJkLMmSzGUxNJTA8Mor+gSGMjo0jkUwinclCVVXoug7DNEGtag9TWzFScGbbZAT9PtTF
    omhqqENHaxzLOlrR0tSAhroI/D7FKXIBlmXip8+9gvNX+owb7B4t9OzRNQt2Gzdu9O3evfud
    xWLx148ffy2YSs3MGqafPaRvy81evTqlbN26faNpmvlXX321YFlWwZHV5UFyVQV3kUESrwbJ
    g6WCc9cFYZvjylzRzmCMGaIoWo8++ui90WjsA6dPX47anaMbmzGqBXhej4ADlyQTT1wSPJnW
    ggQk1+PlqaeeOvmBD7z/B1u3bvzVZHLGl0pla3QB7D3r9ytYsaK93D3SdYKJiZmqK3N2sksg
    SSIkibAzZ86ovb29fExKw7Y0cQHSLReTGAMGB0dQ3SmoJP27dm1CIBBAf/8oGBO4WROxbCxb
    oeHN9keyaZx23oibT63z5nnGwMBAz9KlywZbWppabTV8lBN1m0YYQDgcRDQaRjgcRCDgh88n
    Q5LEcvLPnxM3naOUwTQZdJ1CVS2oKoWqMhgGYFnV91AwKCCbTRua7Yjttbdx6d5vSpzhRlke
    hJBSIpGY7O/v+9dVq9be3tbW0jY8PF5154qiiEDAj3A4hGg0jGg0hFAoAL/fB0kSHXAkcPOf
    1W+3ze5msCzAsig0zYKqWigWTaiqBV1nXDfSXttQSEJjo8COHTtTnJ5OZeEp2DnxyM39rskA
    kRYiEDnASASwXhCEz99z94F3fvGP/p20fHmnnahxcsy1BofhUIXWrVmBf/+7vyH8x//xleU/
    f+GobJqWzGzZNsv590RPm8zku0kLHJh4V3Z+aFd2Lj0/AB8hROKCkSs/HgXw/nvu2rv3c5/6
    OGlqrOfMX68v1z0X1c4rm16mxtDKbBKpqGLUgvkLXq0jhLjqXDnG2I9PvnZ6y1996a/v/tM/
    +gJpaW6CphrIZLPIZbMoFAsoFUt49eQpjE0m8cDDj+Kv//r/wdb1q3DqzAVIogBFluD3KVBk
    EbIswSfL8CkywgE/ZjIZjE0mGEchyfLB6GZWtW8gMQZPbQAhoMwCs2jlvacMpZKKVCoNwzQg
    S6Izu0ZtoOR2FR2lHRC+hc2cqq0AQmYFo4UGytet6DqFhIu9V69O9fX1dzbEoujrH8Cp02fw
    2plzGBwaRi6fgyiIiEbCCAb8CAUDaI43QpakMnWQgKdcWjBME4auI5vLY3xyCi++cgqarsHv
    U9Da1Ij1q7uwZf0KLF/SilQ6g3Qmb3lAET9AvRhD1Pw9U4I9RP/0Cy+9vP/chYvNe/fsQiKZ
    xPmLl/DKqydw8dJlJKanQS2KSCSESCiEQMCPeGMDCIFT9TVtRTqHVmh34wCRCBAlEQKx/bhU
    TUPP1QEcf+0ciiUViiQh3lSPDWuWY/umtVjdtQTTM2k88czLzDDMWt2jxVaum9U14pIEYefO
    nYFDhw69wzStP3r11VPtV6/2g6dNeSuefDLPGNDfPwVZlkO7du3ZEw6HydGjR418Pq85FVX3
    3BQ8QHDBmQ1zdB15gCSiog6ncJ0Z6sRT9o53vGNrS0vLxy5e7I+7/iTXU+W6cf+jGxFwKK+P
    981YUIDk7BOLEKLn8/ncyy+//KPDh+/dvHPn1gNHjpwipZIGQRDh8/ng8/mgKApkWUZraxN8
    PmBqagq33XY7MhkDjY1RuKITdmJHyk9KbYlvn08CISbr6+srWpaV93SPeMnhWyEmufN6rDou
    MW4epPJeSpLkiDKIVZLds4Fxrfob3z2p8cWbC5YoALO7uyexd+8d51pamnZ3dw8KiqKgqake
    LS1xNDREEQz6IYoiDMOCYViglMEwLGiam/POpmm6lEtJEhEMyojF/JBlewzcMBiKRQvZLEWh
    wGCaBD6fgPHxjG4YxlwFuwWxFJjjjlEBFE+ePHVm1apVx1euXPbQ+HiSEEIQi0XR2tqElpYm
    RCIhKIrsjDyYsCxaNlt212A2LRHlNXP3hSiK8PtlhMN++HyiY8VAUSyaSKcNZLM2YGpu9iOb
    nTSuXOl2RylmPGyGtJMPzuq2ee9oab4uGT4QOeBgv8/n+93777t337//o38nbdm0wQFH1erb
    VeJKHsDEGMOGtavwp5//LWFZe9vS7/34Zw/PpLMtAJ5njImEEF6BrcBtEmuRQZK7YdzL3KWu
    eRV4mNNResf6tase+OwnPyYu7WhzwNHcnkY31CGitcCV93vtpMjmV4N5kuHFuogoVxEftizr
    a4/94EdxatGN27duJt29vRgfn0Aun4eqqiiVVIyMjeMjH/korl69ioH+fkxOjOPp515COBxC
    wOcDg+2XJMuSTQ/yKaiPRVFSVUwmpi1UBmD5Sp1LjykudlV7rvPDzwIxgFmWBUPXoBsGLNOw
    BRksqyzKwKiFlSu6UBeLYGYm7cwsVTpHjDMRBef941r/CRXlIFKjmziL3rJIFFbmgGgTwGA6
    nen/u3/4Wuc3QgH09vZBFEW0NDeha9kSCAKBpuvIpLPI5vKYTs3YtELdgGmZyGRzKKkqGupi
    UGQZogOoZUlyJOADWLtyGfyO79H0TAbPHjmBHz71AtqaGyFLIvLFkoVqxTpeZcrAAooN1JJY
    JYQUGWNnR8fGz33ne48dOnvuHJ557gUkk0m0tbRg6ZIOrFuzCqqmIpGcxvT0DKZTM9B0DYZh
    QiCkfE5E0X5r7YBlQNMNWJYFgdgeJooiIxoOYd3q5Wisi4IxhulUGsdOnccTT7+EjrZmKLKE
    3oERA5WZSt5E1Q1GhRsJRvMNjrjzRB555JHGdevWfUjXjc+9+urJpadPnwOlgCBIc9QFZs8M
    mCZFb+8kTBPhdes239Hc3Bw9duzYs319faCUCgAkJybxFKnF7ibxQdSlqxLubAvw0KgPHDgQ
    X7Vq1YdHR6fW9fUNc4IMXnf76wGeWgPmN/K9dqJtWSa7GXM2zn1jAFAvXLg4FI83f33btu3x
    u+7atSGRSJH6+jqEw8GyIaokSYhEAjh16iQaGxtRX1+PSISio6MJpmnBNGkZVFPKHMDEoOsW
    ZFmCZRWsmZmZIheTXEbDTY9J3rNjGEYZIImi5FDFJAiC6NBS3e6IiHQ6B00z4fMpZeUyl2LH
    f+SlwCv0KMFlNBDifrF6z4o3IRbxMclMp9P5fD53vqWlqbh377ZwLBaB3++DphkoFjWkUjlH
    PU1y7k+xrDxnU5hF+HyyMydMnQ4Jc/aJ/WdNs5DPm+VuYzAoob7eBpylEkM4TFgul9cYY6Ua
    MWmhveWq8l7GmAGgNDk5mZicnHq+vb35wLp1K2KNjfVobm6AZTHkcgUkk2kwBvh8CoJBP/x+
    xaEfinB9xdxz4r63gkCcfeWyI+z1sSwGTTORzeqgFJAkAYGAjM5Oe766ULAQiQjs2LHufC6X
    c4HRlFNYnEbFi6+EG5gfluYruePAUROAD8Xj8U++/33vW/nZ3/wM8fkU/M3f/j1SqRSAatf7
    qk4IpVzVm8LvU3D4wF50LW3H73zm42TrxtV1//itHxw8e7G7yzDMpxhjRzmQJDkXDhYSJHkc
    l/mABK6aLKDaC4oHHTtj0cgHf+kD74xs37yhGhxdU5Guhv+RO2BOqyWPr+WVZJkmNNsgj90k
    KhU/SFxgjJ0plkpf/vZ3v//Rnz759I4HH7hPOnTPPrQ2N6Oxvg5HXjmOx//1aWzduhX/80tf
    gs+noLG+DqqmYc2KLnziI++1E9lCAblcHtlcDjOZDC5c6sWTzx1hxZKqerpHmRrdI3MBW9LX
    DUSe90AEUA/GGn725FOYmJiEpqswdBsgWU6HiDpnxZ6zoXjuxSPO+wxU63ozjsBCyvRF0ZnH
    6RscBqVUgj24rTgf+ad7tqoorAscuF2AlFI17egzzz2/6q7bb2teu2a1UiwVkUxOY3RsHIqi
    oCXehPa2FrQ2N6Gxvg6xaBiRcBCpmTT+6stfRXI6hQcPH8BD998NVVWRy+WRzmaRTKUxOZXE
    ZGIaiekZGKaJcDCIlqYGrOrswOT0DHv2yGuspGr8EGwBtoJdcRGDUTkgOfMkGiFkwjTNo1/7
    xrd2RsLhwnve+UjD4bv3BwaHhnCl5ypSMzMIh4JY2tGOPTu3or2tBfHGBtTXRREJh+CTbb63
    C5wptezummEgny8gnckgOT2DiakkhsfGMTg8jhOnz0ORFXQtbcPhfXsgSyL7+28+bl3qGQCq
    bQsm3YDkdI+8st5ssc4VY4w0NjZK7373u9c2NDR8Kp3OvP/YseN13d1XEQwGIcu+KuUtQZDK
    H+3ET6r6mq1qxzA4OI1SyfStXt28/S1veaDl6tWejhMnTryUSCRkxpjsxCSJ6wQYCw2S5ohL
    3hhVJXnugEdxz549d5dK2n3d3YPEMMw5B+69w+Q3MnPkAio30alFwRNF+6lpOq2xNot1vty5
    iuLzzz//mmVZX966deuHu7o6dg4MjElDQxMolTRomom2tiYsW9aIyclJ3H777bAsBlU1IEkC
    MpkSursnwBiBLMtQFJuy6vNJqKsLorHRx06cOKPm8/nrMRoWjV5Xi7Xg7o22trY6QRAaVqzo
    RDAY5GZphJoglxCCjo7mWd0AL03R2zliDIhG/SgUTCkYDPpUVeVjkX+xY1GNAhUFYIyMjJ7f
    vHnLuZaWxvUzM7lYqaQRv9+HWCwEQRCgaSZKJQ2pVB6qakDXbcBsWcDSpY1YsSKOYtFCd/cU
    NM32anT3h98vIxCQEInIkGV7Ls8wGHI5V3a+wPJ51UylUkXnbuGfN5Twz+eaEELcmftiT0/P
    a8uWLevds2fz8omJhHH16kizKIqkoSGG1tZGx/hWQz5fQj4/g1LJgKaZMAwLlsXgarm5gFkU
    bUqmJNmGyn6//QwGFQSDCpqafM6dYSGXMzE9rcM0MzQS0allKUZfX1+KK9a58cj1G+U7tNfc
    P9J8LZpz8ccA/PaWLZs/+dnf/Gz00UcfQTQSRm9vL7p7ejA1NeUISXlVl7xdD/sZCvhx247N
    YJQiFAzg7W85iM3rV8vfeuyJtd/50VNN06m0jzH2PFdxIFwwWNBg7F1MzyVDa4EPxlgYwHv2
    7d297sH7Dtq0AkrnBD9wlMyo0yGqTa2bbRRbAUzVs0oAg6rpyBdL4A5SLWPdha6Iuzz5osO5
    PkkplQkhS9/x0ANt9x26G9SykM6k8Q9f/yb27duPvr4+lIp5CIKAtauWQ9MNjIxNoL21Ge2t
    ccdTyzEUJQw/feZFPPncEVosqTmug5QFkOVmj3gpVSx2MOIr3A4t1QdgkyAIn1y/ft2du3ft
    RDQagSIrkGUJUpmvSyAQ+0kIPKCoIs3sroebAFNqwTItmJYJwzBg6Doa62MwDNN//nLvEsZY
    1klyI7BpoFFn3RhjTHckhPl9M28BygsWGWPUMUn9HgERB4aGPpXOZpSVy7vwwH2HsGHtKixb
    0o5wOAifokCWBIAxFItFnL1wCecudkNVNWxYuwqXevpw4I5d2Lh2Jepj4TIo0DUdJbWEbC6P
    oZEJXLk6gIvd/bjQ3Y90NgfdMFQn8efl4Xk51QWt8s6lIMQYKwL4uapqasDv9/cPDP7SyVOn
    lzTHm7Bv7x7s2LYZXcuWIhoJIeD32e62li1Y4Up2D42M4oUjr6IuFsFdt+1Afcw2gkW8AYQs
    Ld8dmqohXyxiKjGNC5d7cfzMBXznR09DkkSWSmdd0Ohyu11ft0QNYYZF6R7xIGTZsmXSe9/7
    3gOiKP5hf//AHa+88qo8MjKGxsYm7Nt3ByKRcBUVaLakM6mStDYMiitXJjA9XUIiUUCxOCEs
    XVq/ZPXqTe9oa2vveu65nz/W19d/GbXlqY3FuGeuEZdq3e3kl37plzpkWX5/b+9w3cxMbg6p
    4WpVMreiez363NymsbN/piAQViwWeUPdBTd995wvCzb1u8gYy7744osnVLUk3nHHvs6rV4db
    h4YmIAgS/H4/NmxYjsHBATQ01MPnCwEAMpkiRFFAKORHsWigVDI98zYClixhEIQiPXXqVM40
    TT4m5Zy55XytmLSQnVbPOhA3Hi1fvtx3//33b6qrq/v1bDZ31+joBHRdd5JZ9+n+fVLjzAiz
    1Mwqczj8R9EByCIMg6KuLuY/fPhwxyuvvJIZGRlJUkojjLEogKijzMg8VLJ5j0XXKtq99NJL
    V/P5wn9bv37D7zQ21u0pFFSkUjmkUllks0WoqlmmWLqvLxwOorExgoaGELJZFT6fhEjEj0Ih
    j2zWdoogRCyLFciySyWTEY0qCIcl+HwCRkZS1ksv/TyTSCQyqEh7FzBbUXWxCt3l0ZKenp6R
    pUuX/k0mk2lmTNy1YsWqt0uShOnpDC5dGkQqlYOqGjBNWjZ4rcyl2QIezc0xhMN+pFJF5HI6
    CLHnskolC9msXl5PFzSFwwrq6/2or/ehuVnCiROXzGefPZJTFDmXyWQSHoCUdBgOr6tgJ83j
    pUwAHFi9evXH/+Iv/mP0/vvuBcBgmbZ3yxf/6A9hmTqYRe1ZClp5UvfPjqqS+zkCCkWSHD8P
    +7WsXNaBz/7KB+D3yY3/66vfOagbxijHk3XfMHMxqnbXQ5/8BSQIAhhjG+ONDW97zyMPSI31
    sXIXoFbXhzKKK919UBQZy5e2O+Do+h0i0LlAJwUBUCyVkMnmGUf9qPVc6Ac/s5Vz9mG/ruvT
    iWSyjTng8MgrxzEwPIb7HngY/+cfv4p3PvxWnDl/EZe6r4IxhvVrVsLvV5x1rKwDGEMmm4du
    GC5PtoCKfGoe1f5HCznQeM345OwRgTEWYIytBPC2WCz2nsOHDm76jc98Sty5fZs9I2TX2Zz3
    kRNh4D9S6hjG1vgac2eSLG5GyVW6M3D+co/wt1/7TtOzL726LV8sBWDLA+sOQDEc6hA/e2N6
    9s9CUBpcetBUJpsd8PkU44tf+F3su2MP/Ipiz9A4d4YLAAmA0+cu4ve++J9ACMGBO3Zj26Z1
    +Nq3H8fv/cl/wzseOIjf+OX32Sar1LLldoMBhAM+tDc34LZt61AslXDmYg9+/y++Ak03NM9+
    KXAB6WbMCPBzJf2EkPRMOrP56WdfMH77N34Nv/7LH0E0EgFxhG4SySSOHT8JxijWr1mNhroo
    AIZcLocv/e3X0Ds0Dl21Z9je9463AozBNA30DQxjeHQcLfEGrOxcglg4iGgogFVdHbj/7tvw
    +M+ex5/9939g+WLJ9Txyq3UuQHKDES/rvThcGO4MP/roo2sFQfiT1157be8rr7yKXK4AUZRs
    EH32Inw+HzdAXj1wXqEEVT7HGEE+r5W7J6pqoa8vjUzG8K9Z07jnjjvu1CcmJkvFYtFVNXWT
    CAucZ99iCjfMFZfcmN3S0rIvny/ttr1/cB1wJGH58jaoqo7x8dQNijHUFnBw6XqA4MxcUFYo
    FExuvfh4xLA4tCGX2ZBjjEmTk1P9jNFkIOBvdbsfbW1NCAZlXLw4ge3bd6K/fwrxeBT19WEQ
    AszMFGFZjPNqqYAGRRGhqgVT07SSJya5H4ve87JI+4S4DKCurq7AwYMHVzY3xx+wLPae/v7B
    TSdOvCZNTEw586sVilzlbHj/X6yi3nlFGnjPJP7viKKIurqQsHLlkqa3ve2hbYODfYGenh5h
    dHRMKxaLbiySAZSceOTGIovbMwsCoF0xj0KhUOjt7enbs2d3amIijVdeOQ9dt5wiyuzXKUkS
    1q9vR0dHI/J5FX1906ivD6Kzsx4dHTGcPTuFbNYod2PdrpFpWigUgGSSQlEkLF/uhywzOjMz
    UzAMo8AVffmYZCz0WanBerEIIUYul8s9/vjjLwJo/uAHP7A8GPSxF188Taanc04Ryqam1tVF
    EA4HUCoZyOXUclexpSWGe+7Zgo6OFpw504Njx/qgqhYIEeDzyYhG/SCEoFi054w0zbYzSadN
    TE5q2LgxBlk2jGKxUCgWy/57bkxylevmFGaY65zNC8XOrihRBcAd+/fvjx88eND+nGU5Q8AE
    wUAAjCocIDLL1W1GTXv4nAdLzuctywLjqhaUWggG/Lhv/+343o+faRkem1zjvHDDSep4Gd6b
    PnjvrBFz1ufwjm2bunZt21SZFalh/koAzMyk8aW/+2fsu307li9t4+hzs4FPFQ0PtU1l3Y5U
    NpdHIpVxOdfWHACJzdPrvh5IcgfPRQBZ07Jw5txFSKKERDLJnnjyGezacxu5evUqzp07h4As
    IF8oQNN0MMYwOj6Jv/irr9gtWEVBMOBDKBhENBLEybMXYZim6QVHThWqUKsDcJ3f900HK47+
    QxhjCoA2AOsBHIrFYgf37N69/r3vfY//obe9FfGmxrLPEZhHdIF/0go1k3lpmtQxCy3/jIqX
    EIM9gyRLIrZvXIs/+71Pk6f37oz98Mnnd5671LMilc5uZ4xdBHABQDdjbJjzjHIBwoKouHEJ
    i+EoHl5RNW1SVUv1Ab8PpmFWA0WuQxqLhhGLRlAolhAKBuBz5KoFgSAUDJTfA8ZJovNAMuBT
    QEBQVDXLU6UrcElMzRb9YtAa3IFy53eTAYwYhtFrGObyWDRqU3bBMDg0jP/4V3+NTF7FxMQk
    9u/dgX/z6V9GwOdDoVhC/+Awtu+8DT3dPegbGAKlFKJAcKW3H3/0l/8L4Wg9NLWIhw/txTvf
    dsgZorcLVtlcAaqmG6g2GHVV65JO98ilMtxwMJrPRzAYlMLh8H39/f27X375CPL5AiRJKZu/
    Dg2NOBQ6+2nPVUg39DlRlMpdJYBgZkbH0FBBXLGieVNHR8f5np4e3Xmf3ELdYgl6vC7Q9Oij
    j4YlSTo0NTURy+dLnJJUbfGFSCSIJUuacfXqGCfCIMzhf1QbKLly4NXAQQZjJk2n0zqX7Jqe
    5HfBYhKXBLvMBgGAFA6Hc7Isk3i8AYwRBIN+1tnZhtHRflJfX4+WlmYIQta2DBBcBS0ftm5d
    Wp6XsJ+AaTI0Ngah6wV3prHse8Tdq2WFXm83xPt7z0cscvI23HbbbcqWLVvaIpHIer/ff9Aw
    jEMjI2PrL1687O/t7UeppHLy1BWVWV6qe7YnDTBb7KT63+a80MtzODMzKs6fT5K2tkhs2bJN
    O1euXLeiWMxuT6dnLqZSqQsTE5PdPT09w6Zp5h2q82LEInf9LQD65ORkOp/Pn5Nl6V5KmQjw
    JsrVe54xglLJpsFbFoNpurLorKrTVL1OQpXIiT0rydjExIRWKpW8MclV0lx0hVDuTBouPRVA
    OpVKXaqra8qYJq2nlJUpmStWtGHfvu1obY2jp6cfR45cxvR0HoB9btramrFmzWoMD09AliWo
    KoUoClixogHbt3ciHI7gwoUBXLgwDVWlVUUXSg06NTXljlNk+HiEinLd61aHnJcOkiAIoJTK
    AGL19fXwKQooNSviAE73w+0E2RVsu8rp/tmuCFeq2zZQqla2q2BjCp8iQ5YlCfbMUxMXrLPO
    6yK4RR6dnZ0YGBho9Pt8++7cs0Oui0ZALVqzywPGQASCk2cuYmxiCpvWrZoNhCib8+9ea3aJ
    ABibnMZMJmdxAdsN4iYHmNg8ByL3f7yyra6AhQXA0lQ1+3/++VvjX/vGtwVVVcOdnV3BD390
    K770pS9hJp3Bz37+QtW/0ds/iKPHX6ve0KLIJFkCo4wZhmlw7Xgd1UP1xEkwvaRovmJZRfeY
    j8qvc1YA4IDf7/+DpUuXrt+1a2fjgQMHxFUrVqCpqRHjE5MYHR11zoMrykBBqS3SYFm2Epn7
    Z2qZzkfne6lV/rt24o+aAg02VQ/2PJIoYNO6lehoi4sTk4mmK70DjWcv924bGB4vTiSmz1sW
    /TKAAVSbGWI+u0jXMKVLFovFwSs9V9fZa0c9nleVruuq5Z34089/Fj958jk89/Kr6BsYhqbp
    +MJvfQK379gMSRCqvcY4c2EGe1h0cHQCpQpA4gNS0XG5VxeLMlWr2OKc1ZJzliYZYxcuX+k5
    lE5nhGjUphCOT0zitbMXsHzFKkiSiFDAD8Hxemqoi+Ltbz2E7/7wZ4hFw3jLwYchCnbDQ5JE
    iIIAUZYxPZ7FxZ4+PGzuhyJLAANKqobegRFmWpbrzVR2KHeAEa8SpC4mtY5/HDx40AdgzdTU
    lJTP58tiDJX5Gb5yXZEkvv6T1HSDLxRMUOoPRiLhZQAmXCqks18KqKaA3xIg6TOf+XSraVrb
    E4kZYidhc3ka2c+WlgYYhokKFa9aAfD6cuBkll+JDSoUqGrRzGTS7h1tcADJnC+A5JmXLsck
    x3fJG5NMv99vASyzfHnbxNKlceLz+cKqWgpOTExgz549EEUBzc1RTrzCfi0NDQGYpgnDMB3P
    MYvZd1uJTUxMG6Zp6lw8MlmZ+lCJSfzv6Jz5qgImqTg/v+Ez5cQhcuDAgQOiKH4+m81uGBgY
    aBgcHJJmZtIolVRHtjoyq1vkdny8HVd3doQ/T243abZRqIBqfaDKnzMZHaUSFYNBpSkSqW9s
    bY1v6+oiRV0vnDdN8ys9PT3914pFC9Cl5dVEi/l8vq++Pp4PBPyxXK4EL3204hvF0NeXQKGg
    Y9WqZixbVo9IxIeJiQJGR/MoFEyn+0bmAEoEsixAkiicAkKRB0gcSFy0Tv0c81kuPVUYH5+4
    snbthkR9fbQ+nS4CsGm08XgM8Xg9wuFwuYDg7oVEIo8TJy6hr28EAwMJlEpmeT0YY5BlGXV1
    EdTXB+D3235a7t8NBCTHPHfaXZeME5OmuZiUfSMFu3kBSJZlwdk4AxcuXLDGx8fE9rZWUMZA
    QGHraBEQRkAdc0YKAsFxzmXE/rwgOP/vVKPs5MXO5hgEMFCnBWni1NlLmEqmTOdERWB7PASc
    C2benbffzGNgYAAAWuvqoss3b1jDedlUS3fbXSKgWFTx9IuvYOPalVjZucRJdh2wSefuEDE2
    h6iDA1AtBlzqHUS+UKoFHPjAxObjEHGgKASgAfagJeOCnnsjSgAsBvysUCjuJ4SsBBA5dOgQ
    6e7uhlos4IF770EoFIRhGCiVSgj4/ZAkEcWSikw2i0wmh1yhgHyhiFy+AF033MvMVcBRAAQY
    YxEAcdiDn+7+oc5FW3CqvmaNQD0v1V8nKPkA3H/fffft/+IXv0hWr16Ny5cu4k//7M8qc3oM
    lfe0PGPGFW0YnOKDt5jjuL4Sxo0luR5Y4PYGuOKFTeEjjonwR9/9Nnz2E+8jhUJRPnHmUuyP
    /+p/bxybTG5wlHNkO/6U6az6AlanXCCfNU1rZGxsHJZpViqYcA1eHQTrFAE2b1iDUNCPI8dP
    4fjp83j4/gM4vP82GyBQy0HDBERgtsks4debYTKZgm4YLpXNBUhFTzBaFNGBawBH3XkfJMZY
    98DQcDY5PV0Xi9rBZ/OGdfjj3/8c+gaG0Lm0Hbfv2uYUrizIkoj3vP0tuPuO3VBkCfV1kfJ9
    sXr5Mvzx734SJ0+fB3avx+07NkOR5bIaYjafx9DoOABoBCiyio+Lq2KXRrUB303p4ieTScYY
    K/l8Poii6DEWJNeZjRFq+iLVAgNuXq0oIgCKXC4fBFDvrEnIuWduqYKd+1AUX6Nl0bpiUb2u
    KEMg4EM8XoepqTR03aoCim9GwEEQBEQiPoyMDGilkqpx3TbdU7ibT3Ak3nvvvcF4PN6o67oy
    PT3NRkZGzImJCUtVS8Q0LYExJvf19Vnt7e0/a2tr2+/3B1aKohDp6+tDY2Mj/P4wRkaSME2r
    rGxnWXaBRZJEyLL9FEUfRNGEphWRyaTp6OioIUkSMQxDBKAQQgKCIERCoWC8ubnFH483mYGA
    X9B1kw4ODuaGhoYKTjfSqBGn6Zs8I/jYxz7mUxTlvp6env3PP/+8MD09jaamOA4cOIBQKMSt
    3dwdouqC6GwDWO/nvD+D97/h/0wpMDRUQE9PniiKJNfX+2LLlwc2NDY2bOzpQdERRHE9Jt1Y
    tJBFCPff0fL5/Ehra3ve7/fFcjmV8zaqvi8EQYBhWBgdzSASCWDFiiaYJsXYWA65nOGAS1LT
    HNb9nG3FYbJ8PufaTCy6QMMNrosKADMzM2MA7a+ri6wRhEkQIsCyGC5fHkEmUwJAkEjkkM2q
    5bsgn9dx/Hg/ZFmCrtMynZMxYGAgi3z+MkKhPhQKtigDT2ENBiUUi2mzUCiUuA5SmhCSRkW1
    rlCLVXY9ED2fPkiUMfb8iy+9dOaP//hPdrzrnY+ivi7mVFds/x07oa9QY8DPR3D0OrcSzih1
    quQmLMv2L8kXCjhz4TK+/fiTZr5QSjgbhOHWMRab6yGLgiD6FLnaq8gDkgQCdPf2o7u3H5/5
    +Pvg98l2t2muDtGsblINyp3z/5lcHqcu9DDTsrzmYlqNDsubf8GyDMMwlggC+UwoFN4tiaKi
    aZpqmGbWsqwspTTjbGgLQEddLLpvz66dXeFQQLpwuRfr1q3DY9//Hn7rU7+M23fvwJFjx/H0
    cy+x6ZkM2lt92H/HHrJnxxbEIkFomobUTBqTU0kyMjqOU+cuiUeOn64fHZ8STcsKA2gHsEkQ
    iCkQgYiiICuy7Jdl0Wealpkvlq5Syr7NGOsmhPDKZQtBpfIrirL8zjvvJNu3bwdjFKtXr8Yf
    fP73oZbUqs4IIWUf6nL3hxAnTSNcCCKuWl1Fzru8z5xz5sqDV7pQRrnSaRoGdMM2WO3saIEk
    iKiPRrCkNQ5JFEUHVDY7vz8PHoRFuHjzACamUzOsWCwRSRQwk04jk8lgJp1BMjmN1Ewa2VwO
    hWIRmqqhUCxiJp3FVHIal3v68d++8nX4FBkBvw/hUAB1kRAa6qKIhgMIBwOIhALwySLAGNKZ
    HChl7oWvciCJl1O9mdLw7rrAufj7plMz6cnJRN2qFZ0OCGTYtnk9VnQtRaFQwLmLl5HN5lAo
    FKFqGkzTKIu5EEKgyCL8Ph/CwQCikRB2b9uEUMCHUMgPy7IgOh3HbK6A0YkkBaCy6q69+3RF
    UTQvgFzMNXrllVeM22+//URnZ+fMypWr6oeGRjzy1UINWpA34a89S8ODA0kSEQopWLo0hFRq
    rDA+Pm6goggpwyOpfSs9isViMRar03w+HwShMOfsEGMEDQ1RiKKIZDJ7QzNHteaNagGqYFCB
    ooAODPQXrUpc4p/zlvi5++8zn/lMRzQa/Yxpmnssi8rLly9Xd+7clTNNI6PrelrXtbxpWlRR
    lI5gMLQvlcp1lUoFiTGGRCKBTZs249y5fkxOptHW1oiOjkYmSXZnOpHIkUQiD8Ngji+QgmDQ
    R0IhH+rrO8Q77+yo37Rps5jNZiKEkHa/378pFAqZwWCI+Hw+WRAkP0B8AMwdO3ZePXXq5Lde
    fPHFXsaYSghxY9K8dEtSqRTa2toClmUtHx4eFiYmJpzPz+Cll45AlmXwUtzVBYJaH6u/rzKr
    JHC0seoZpEqnqbor5dKyVNV+XYbBoKoUpkklVVWb3FjkGM9r3D29kMUpuKbC2WxmShCEfCDg
    gyDYczI+nw9+vwK/3w+/3wefT7bFlRxD1EjED79fgq5bWLaszqHbEZimTcM0DMA0bXNYy7K9
    swDBsWPQaLFY5H34+Hi02AINNWeR3GLp+Pj4NKXW1Ugk5NAybZCnqgZGR1NQFBmKIqO9vQGK
    Ulkf/u5wqYUuPbVUMpHPW7B7MdXmuYGAiGw2Y2iaxs+b82rFrjCD+XqLmtI8rhkB0JvP5f7r
    V//xqx///mOPrfX5lIAoipIgCJJAiDu4yiV2tkUlcRbcTuIYGGOMN7m0qMUsi8KyLGaaFisU
    S5puGNMABjk6x02tWF5rI4miCMuypjPZ3NSFSz0rtm5YC0IqVXzi1PKJA/N6+ofQtbQd2zau
    KYMcr6EuuLkLUvVzKt9X/rnOu3PyXA/OXuqzuPXi/V3m/ZA5HgpbA4Hgr9x7+FD9po0bIArE
    FopIZ9jMzAzN5HKmoeusob5OPrDvDvGe/XfhK3/3VciBCHp7e1Eq5pGcTuHf/dl/ZkeOnTDS
    maz7OwsvHj0e2btrW+Cth/aRbZvXob0ljpWdHSC7t+LB+/aTS9194isnztQPjY7XqaoGURTg
    9ykkGPAh4FMgiQIMw8TVwVG8ePzc1lyhOAZgwqHGMK57NN8gQIAthGB7GzGGUCiEXTt3zRJh
    gEdooaJE5lBRWYVKB1e4gHoFHCxuzmb21ymzAEqrihK6YYAAGB6fRDZfMJ2Er8HZJ3knCZz3
    yvg1uiWpnqv9xv/8m39QkslpjE1MIp/PQ9V0WKYJEDgqf6Kj7kfQGm9Ca3MjVFXHsVPnYDkq
    foZpgjnc6IBfQTgYQFNDDMvam9G1pAUj4wm3I17yVOn4c0IX856ZIyBRZy+NFwrF5ItHXuka
    m5hAd+9VjE9MYTqVQqFQgmHonC+LIxlbXuCKMTdQoYCKgoCA34dYNIyWpgZ0drRgzYplGBmf
    RDaXt7iExJ3Ncmf73KKCeZPvYfLCCy8c2bdv3zcPHz70gcnJyZiqaq76mhOHBPDSsnxi51Xb
    4mlCLnVIFEUmyyJ8PkLz+UT+6NGXB/P5fA4L7JE1TwVNHDp0aGrXrl0X2tvja2dmcrAsilo+
    UIQQWBbFyEgCpZLuqXDX9nStBT693ydJApqbw0ilJrWBgcEMt4f4OdF53UuMMXzhC1/Yyhj7
    xODgcF0ikXK6aYprBmv5fIopCBLLZjX58uVhcWRkClu2rEGxOIOmpib4fCH4/QXs2rWayTIz
    RkYGCqnUtBqPx8XOzq5we3t9YHw8S2ZmSiiVDBQKJpLJEsbGCiQWC4jxeFN9PN5aJ4plI1lS
    LBLk8ygrfIVCIurqlK2dnctGX3wRU86iMY7dQN5st6SzsxOmaRLG7Dka970yDAPj4+NVPka1
    xBZ4ut21RBn4r1f/XanGz/B+n/s1IBAQQKluJhJJxenS8rFowVhD3N1bVm7TNC0niiSzfHk7
    WlsbEYkE4fPZarM2oCGVeV/uHZqeLsE2UvVxxRZSNW9kzyoBmsZQLMLxC6JMtS0nyuCIA8wL
    On/1OtaFAWC6rmdN0xyKxUJ6V1ebEgz6EI3ahuU2YBTKHeZq+IBZXUW+80gpg2kyGAZDqURR
    KFjQNCAYFNjISEY3DKNWPCp4sAFeT8FOmsfFcg/vccMwk8lkci2ADtjzQQ2wJYO9G5lcg17D
    /9lrfqc5mySNigFUBjfBQ+AGaVUEwHS+UHzm2489sW1ZR6t/ZdfSSnfAoRO7N9761V1Ysawd
    hADZbA7wyqJzHSd3fqs8qE8riTOllbmu4bEpfO2xJ1kqk8uiYpjFI+x593ZxgOF5Xde/eezY
    q4eHh0fizfGmcCgUkmRZIrIii02NDaJACBRZRk9vH86eu4BjJ17Dvffdj8cff5yNDA9bl670
    aIVCschVqQsAtJl0Rn7i6eebXjh6vHlZR1uka1m7UheNEklyKpfENsGklBFBEGBaFrL5AlLp
    DNM0g+YLRX0mk1NnMrlcSdO6AXQ7e9RwjIjn/dLt6urCwMCAput6z7e//W06MDAglO8AnkIH
    vgvIPNTJapPgMj0MlQ5iRcCh2i+L7zLOFnfwfo1hYHgMuUIx65w714dCXMjKeA1JeBPAUE9v
    38XvlkorGurroq5yISEEoiSCOAtoWRSW5/qorJkNABTZdvW2LBOZXB7ZXAGTyRQudvejqGps
    IjFNueSfl/euWdFeTEUyLiCBS5iyuXy+95++/f0Nq1Z0+QMBvxAI+BEMBiCJIgzHYJgxZncM
    TdvNnDmUXntQXrFncZxqtOg4vKuqhitXB3DmYjdM02LpbI7kiyXTk8gWnFmbUi3wuJidI55K
    derUqezMzMw/bN++fTAajW4Ph4PNpmlFTdMMUkoDjEGyxznKo+KkoglQ9buzipEhY5QyRill
    lFpM13UzlUqpfX39M9PT05Owh4Jv6aKday/wzDPP5Ds6Or7b3t66TRTFFZlMoSYZw+28MQa0
    tjZ4qFA8EEJNSiJPh+G7cMGgAknS9RdfPDGRz+fdtUvBNvJ2DVOrinbzIU7wqU996nwkEvlm
    W1vroUgkEi+V1LBlWRKljDDGRFXVRTchq6uLIB5vQHNzHXp6rmD9+vXM5/Nbra0hbXCwt3jp
    0sXM5ORUzrKsgiiK2pIlS+QtWzY3dXYua16+vD6i61BMk5HKSFG5k0Yq9CkCQmwVXkFguiha
    KiFabmoqe+X8+QvdbjfSE5Pe9N2rKApOnTql3X///T2bNm2yGhoaRHd5vTL3tebuZkvi30hX
    ltScSbv237M/FwpJmJ4eyqXTM5azJt5YRBbp7qX5fKFgmub59vb6DlU1WxiDxN899j4lTtGp
    Nt2w8v2uWAMgigSiSKAoQDBIEItRBugsny8Zuq7ziT8vFjRrHvYmGC67c9qEMWYUCsWh1ta2
    ic2bl7eqqqE4eRgMwwSlFc8sd66z+k5BGVxWXk8FSPl8AoJBGfG4vZaKwlg2m+PjNb9Gb8pi
    gsxXQHI2axBAHYAGxlgjKgIKDQBisPnYYo2KPLkOSOI/57aWdVQGhKdRMYLih4T1m7hh+LUR
    GGNBAOsEQn57SXvrfa0tTWFRFAWBEIEQQuwYQsomnoQQpytUtUiEXxpnEzGbYge+A+eAIwZK
    KbMoxWQiZY1MJNOUsWEAYwBGAYwAGCaEjDifSzhrauBNyoySCvRXGGMNANYCWOOA5rizT8Ke
    i04BsHznzp1t8XicPfnkk5OU0gmuk6NyNJ4iV0kLAmh0fmbI+TkCqt3kdQ9dqsC1Y1OwtfJd
    pZMcKn5J7vfqb3YfcReoD8BuAO9yfu+I8ww7r8X/Oiti7E18D7vG97jrN+yAx3HnOUYImXD2
    S8qlNsznGePODmGM+QG0CIKwPRIO/VtCyE5dN2RV0willISCQbQ0N6KpoR510QjC4SCCfh98
    igybHchALQrdMKCqKvKFItKZHKamZzAxNQ3DNCHLEiLBAABoE4nUBLPPxwCAIadTPUQIGYVt
    hDrjXLw35V7x3ruMsToAO7du3vD2FV3LPjg+MRmdmEygUCwiGAyiJR5HU2MDGupjiEbCdvdU
    tjttcLqYuq45FgA5zKQzSKVtM92SqiEcCqK1qQF+v0KPn76Yz+aLSQB9zroMAhgghAw7d0rC
    OT/Wzbh7PesSce6ERkEQ2iRJanPunianAl11zqr5DbN/b+aiJeayHBijduvNLdplUZGWnQIw
    RQhJOPsl69wh7GbuG/5uZoyFIpFIy+HDhw63tbU/TAhZZlk0QCn1M0YVG0BC4JgfZPaMCReW
    POkM41MSDmg6K8eKxYJ+7ty5xOXLlycopVPO3TIKYJQQMgZb7GLauZONeYhHru+csnPnzsbO
    zs41oVBojSAIHYQQPib5CIHIGESAKY2NTcuz2WxbsVhkoihOnj59enJkZMRIp9MmpbSEilhA
    EYAhSRKJx+PBJUuWNDY01Nf5/f6QIAgKY3b9gRDBIgQGIdAZg8oYK1FKi5ZlFlRVK+ZyuWwi
    kUxNTU1N5XK5WjGpwMW/N7SfOFqe8t73vndPPB5/p2maTaqqRgzDiBiGETZNM2hZlt/pMBFv
    eJjjn2SVwhSfq8CTozLvbnFqWGzWDuKOnTk0NDTS3d3dzRibKxapC3XGuD0k+ny+6KpVq5Zv
    27b9XW1t7b8my1JUFEVRFAXCGKCqRtko1jCoU6h1a9wV82RRFJyZNQGKIsLvFyDLpAyaikWd
    nTlzPDU42DcxODg0YJrmkBuTnDt33HnteRck3QK5rn/Lli3LVq5cdUgUQx+NxaK7QyE//H4F
    AFAqqY7pcsVI114bUiX2IUmSY7TsPgUoil2o0XUKTWOYnBzWEonBzKVLl4YTiUQ/F5MGnbx2
    wtkX2hvZF/MJkAQn4IScblHMCdwx5/8jXAdprn+b3EAix9NuCk7gyTgDWRlUVE1cVY9bCSDV
    AegCsAfACgc41jnrE7xGNYRc4z1jc6yRt/umo+LanXA2zjiAceeCcQFCcT7WjQdIsMUzQk7C
    EuX2RJjbEyKABkLIB1tbW9fk8/lELpd71XmPJefwl1A9dGegWv3HxwEMtztqcuCIfxqoVgbS
    MdtQNscFo3kBSE4lyuecjSYALbD51M1O8uYWE94MfY29we+r9fdMZz1SNRI/t1quLlRActZL
    ARBljMUB3AZgA+yZsmYn+Y2IouAXBEEWiCAKgk3ntesOth4D/wrdFM00KdUNnVLKKAek806x
    YNI5I24xwQ3E01wR4VYASKJztmKKIu80TesTlNKVzn0b8Pv9ciwakcLhkBAM+IlPUQTFpoEQ
    gat4UmrBNE3ousFUVUOhpLJsLs8y2TyjjsqWIkuaaVqjlLExLpkdATDiJLST3B1yU4K1Z10C
    ACKMsXoHEDU4H90793qFCHKdYp03HvGCFa6qX9pT0MHN3Dd8B4kQEmSMxQghjdFodIksSx2W
    RZsppXFKaYOzh+RqkMRVLkgtcFT92pijCMMnwW7Ca5qm6cwNuKyGKWcPTRJCplBhhpRcwP1m
    AZIDDBUnTgRRMcWOEUKioiiGBUHwEUJExphUV1dX/653veuDgUBgTT6fn3ryySeP9/f38zGp
    yDExClxcEdyYRAgJEkL8hJCqmMQYU90nZgu/8DlOjgNIOU938g0DJK5YVyfLchOAFsZYM2Os
    GUDcKXJH30Qsej1Ve1bje9h1YlHCiUXuXpl5o4nw64xH5Xxu/fr1a7q6um4XBGEpIaRNFMUm
    URTrBEEMSZKkSJIkiaIoCIIoCoJABEEQnO4R4X9HSiljjFLLsiilJtV13dJ1zSoWi9q5c+eT
    Y2NjY875GOOKCC44cu/cm1Z8qXG+opFIpOWhhx76YHNz832CIDQQQkKMMZ9hmLJpUsGyqEAp
    iK2t5DbyqzuOTveZCQJhgsCYJIERQpllUWqaJr148VL6hReeHzJN043VI87TXZ8kV5x63Xfv
    fM4gMa7KTwBQx8PEld0LOMnyjVDsrhWM+K5AyVGWcn0Filwgsm6FYOSefM74MskYO+ZUYONc
    UlwORnOsD3kjFxOqJavdID7jVjoJIbyh42IMnvPvn3uZuUpzBcZYenx8vB/AcWfDS9zeKjgJ
    Rxq2qphbkRWcQ+lzkp4At5ZAta+GVzqWT469CkHzblLo3ouOWp4KIMsYc8+hm6CH3uTZZPP4
    vRZXiHCVyniVMroIFy9z/q0CgNNOgHBBZTOltIFSGkJFwdItNPDFBlIj4WWe8+EmJXm3K80F
    IP6M3EpzJQyASQgxdN3oBfAlBzy2A2hXVbVZVdW6yamE37M+tdYF3J7niwcmAFM3TNc4d4ZL
    Zr305luFTlZl/OncZ648etY5Y29mdqEWQCr72zj/pntOdNw8QY9rnSvLidHFTCaT5BJ+t6vO
    d+NrxSDyJu4l5jlvaeffnOZi0oLcMdyd4sppMwAWY8wwTbMqJhmGoRiGkRFFsf/EiROv9vf3
    Tzh7xr2Tis55yMCmmhremMQY8zPG+JjE+HPliUnUEyN5JdV5jUlOss8YY6ZhGC5Q5WNRwdkD
    4i0Ui4qoVs3MesHiIhQXmCvWcOnSpdFLly497+RxzYSQZkEQmgRBiAmC4CeESIJABJsaSQTi
    MIY4gES4jnRVZ9oGS9RwAHTaydmmuCLdvFNQ5+PedQTbzFwuV3j88cd/5Pf7TwmC0KYoSoei
    KK2yLDVKkhyWJEkRRdEBkEJ5bdwfY3fRKLMsi1mWRU3TpIZhUF3XLVVVTU3TjGKxWDRN0+3c
    T7rx2um0vun7YyEAErjDrzomaLVUfcibDEhuZ0DjPmrcpcJutWDk/H5FQojitK150BLyJPXk
    TQSiWmtGnX+/wHWS0qg20FqI5KYcDJzgoTHGZO7SNTmAZAH4ngNwNFTT5NzffcbpFvIUA4EQ
    IgPwO3Qsv2e/eYGi96P780vOnvUazy3EmrgiAHxAcufq/FwQrrUH2Dz8+zfyPa5HlasOwyd/
    RSyeFxCFY0YHIO/sH+K8P26CwhdgvMYa1yrIsBqFF95LIeXsOb4rTXHrPMrnw/kdpxhjFrcu
    46jQm6XXsTasBoA0OIDBG8O6AOmWMUPlzpj7ukx3/zh7xVejYPdG/g1+bXjVQ/ejhpvkmXUD
    D3ffFN37hgORaafDIi8gQHJjUi01xJqeJYsdkzKZjPXEE098lzEWGB8f17i1MG8wJvkccHSj
    MYl6Yp4bk3iz4XkBSVyxrhyLnHzpVoxFbhyYKxYtmsy1t7jg7FfBOT+6ZVk5y1bOdYWMrlew
    u9a94u4zvkDpFqWKuAmCQTcSr521KRUKhVShUIDzGnIOgKnjii+vZ21qFTTLZxAVY1g+p31T
    Rf83DZBqDFQbqNZFF7mndAOb40aTfWuO57wZVy5AkudWGYnTUdK4Ko0ftWVh56Ii3uiF4+0g
    eZU+8rUqEfMciPhExXIu+6InSRGcP6edoOImweVLwhkET6NCuzC5ap8Em87g434ufznNtSZ8
    JZOX0VQXMLmxuGQTXIDKcb+7MA9rPx/fS7m1KXkSQH2RzhrjgJro3Dfu/ZJjjM1w52c+AFKJ
    C8BuwsYXEW7FRNc9U3ySpzp7KlzjfhFuIBjBk7gZ3M/MwvaayHiquLdKsObvPP53zzvr4O2m
    vdnik9c/zeuhdssU7GrczS49xwWROQ5EStc5Q290TpJ/f/jZUK+3y0J03vj9wDwxiT8nIqVU
    GR0dTTtr4Y1JqhOTMteISQohxF8jJpEaQMPbuZ0rJs2n+TJf+HDBknoLxyKjRhFCxeJ2aavM
    Ubk94VIi0w4Fz3eNotTrBUhl+r8Tk2p2p2/aZTJbfZYvSAkOf1LngLd7v3jHSoQbjNU844Mf
    tXGL/zWNyl/vQ5qvxeErEqg2d6uW8Hhjladai4RrfFyMQ/JGg1HJE5iyXHVJus46vZFgxDyX
    S615HG0BkxvLk3yqzuuUUFFFEziA5Ee1hDTlApgLIvJcoHDXReR+nlwjWb5WdbwWBc/wVOsW
    orrNJ7IFVKvEvZkiwnwGMFZjXfjkjy3S+TG5/ze5izHAJSD82glzBCMyR5LrpX7yXhO8gt2t
    2EGiHsDNF1/SnmRXrBGoa90tc3WQNFQr/LlPFbdAp2QOmXi+YCe8zmrujQIkVqMLcNPU/F5H
    cmx6wErRuYcVVJuuL1RMMrm45DWJXajukeUpjKmeeFSOSc78o98p2vExyfTEJC/FtFZMkjzx
    DnOsCcVsCh5vmjufRWDqiUXaLR6LLG4tvB/ZIt4tfOHXjU9uYc3vxCRvDiJ4AAC5gXvX9BQR
    vJ58t2J3mu+AElR3KbOozIkrNQDS6+0g6Vz+xD/nBTwumoGdx8Nj3kDZrf7gh9e4S1PkLkuJ
    u4jE6yDoN3Ph0DmAQM1Ldz7kVOfYa7UqKe7F4fNUFoinkqJyySqfqM71s4VrXEJzJYHez83r
    vquxH+b6nRfrIns9Sfhc67QgZ9Kzh9y1ET0AW66RfHgBEl7HpcuDQJ37yM8B3CpKZN695K6J
    m+Dy3dQ3SmeAZ214OjNvMs1TXG41evNCxsAbepG3UryaY+8I3PkSauyV+cwZarFBqCcO0fm+
    X15HTOLPh4zq7o8XXOqo7uy80ZhUKx4taEy6xWLRjZ6lmxKL5thD7vnwFnxl1LbEuF7Bbq6z
    YXpikeEBzLfEnTvHfpI88Yj/KNcoWOF1nA9eoVjjCpt8kYW9mbz2lnT4/v/SY44AXeuQkEV4
    T2pdtAsCBN4gKK5VZUGNQ7H4Iv/zdPHMd6HglikZLU5AQo0zc61uyI0mdsyT+NxQcnKzk945
    1obUAIi1AlCtNZlLtY151obOkcjeqh38Xzxe3z0034Do9STFbCHvmDcZk+aq9vPn4P+aO/cX
    sWhB8rm5nq/3XLEae8z7Z9xKd+411kaYIyZdDyxeLx6xa8QjulB57S8ev3j84vGLxy8ev3j8
    4vGLxy8ev3j84vGLx/8vH/8vZmOcUdgaUvkAAAAASUVORK5CYII=
  }
  set pieceImageData(Fantasy,75) {
    iVBORw0KGgoAAAANSUhEUgAAA4QAAABKCAYAAADjRJpiAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAAOEAAAASgCrjVbVAACAAElEQVR42uy9d3hc
    13Utvs6t0weDQS9EIyoBEmABm0iREiWK6hJlWbJkq1huspMXJ05x7PzivMRJnJfkpccvcYod
    V9mW1RvV2ST2XtDrzKAMppdbz++PuRe4GICSbIsUZfN833wABoPB3HPPOXuvvddeG7gyrowr
    48q4Mq6MK+PKuDKujCvjyrgyfi0HuTIFV8aVcWVcGVfGlXFlfKDOCHlP7sil8lnou76A0stl
    rsgF5ogudj0f5Oe+Mq6MK+PyHdyVKbgyrowr48q4Mj5AJ/+ycravjMtn+XxAwJAu8v70Mp4j
    8h7n5cqmujKujA+RHbzUtvAKILyygK6MK+PKuDIu9tlF3sVhnZfNsL7HlfPs13bQXxbc/Dxr
    55exuR/wHNELXeuH9JqujCvjV84WUkpB5m/Id7OFdLH3uKif98ot+9Atonc1kFccqCvjyvjA
    gM978+J+Rfdo3tlFLGcXAcACYCwPQnMvogB0AJrxVbc6ulfOtCv7zFxPVr8lzy7SxUDSlbVz
    ZVwZl68t/FXem4SQWf897+xiFnnAYgettpBeyvPsSobwMt5IxmJiLM4Vkwfiad5Dty7EK3P5
    /o0rTsWVkXfIw3LAz3NU835vdVYpfn1oW8QAfDyAQgBFAHyGzZEApAAkAKjGQwEgG1+VPIN4
    Zfyand+L7CECgGlqauKKiorEQ4cOKbIsmwEH6/7SrqydhbZwkfn8hZ6/YguvjAutL4stJBd4
    Hc3/+qvuq5rXxzAMuf76652lpaXFDMP4KaXOTCZNw+GZdH9/f2JoaChLKVUJIYrFBiqGbdQu
    1Vl2JUN4+TiXi6aSKaUsgEoApQCGCSFpI6JgBYGasXjM6MKvzcF9AQcdhjOa72SYVDSaf3hd
    wImfR8d5r4DzwzbvVzJdP/9aywvQMJRSzgA7sKwx6/60Zr9+XTKEHIDrOY77rMPhaOQ4zgGA
    qKoqZzKZgKIozwJ4zTjTsgAyxiNrgEMN70CHuzJ+ZdcNYw0qGBF21NbWinfddddNoihuCofD
    u77//e+/HY/HdeNvzb1lBhbUfFD4q76G8s6nebZwETtJrWYReUGtd7KF7wU0ftjm/IoN/IXn
    i1jWDwOAcTgcPKWUyWQy5msvZAd/pc92K7PhM5/5TInf739U1/Udmqb5dZ3yhEAnhEnIsnRq
    //79Pzpw4MAQISRrsYVZ5IKnZoD0oq+3KxnCy+MQNzeTSYNhLWCliBDyxxzHbVIU5aeU0v8k
    hKSMPzc3mWz8rOA9ZCEuEN35UI48B50YzgRLjGExdJQQolNKzYOIAeA0fq9YQLk5f5rlQY33
    YvKAZn52lloN84fUAJJ3m2urQ3Gh6PKvw36llIpGsKYVwFIALgAvU0qHjXmxZr5gDdb8mpxr
    PICtK1euvPE3fuM32YqKMlBdQygQwL9885sN+/a/nQVwllIaBZA0jj/dMm9Xxq/ZoJSSZcuW
    8Z2dnYWBQEDdvXt3RlVVAoBpa2ur5jjut8Lh8Dq32331jTfe+Gc//OEPj1hox6rhRCHPDtJf
    5YDeBc4n016xJsi2UGxnqWmUUlpaWsq0tDQ7UqkUc/z4CUVRlAW2kFKqWQI0s7bwAqDxQ2EL
    FwHJ72j/FrGDv5Y2MH+/ejwedtOmTd7q6uqldrutmWW52nQ6Pf7EE0+8GggEJGM9qJhjgMi/
    Zmca/b3f+71KXdd3HjlyZFl/fz90HeB5G2pqlqCzs73G5/OdBpCglKYJIQnjT7VLbQevAMLL
    52AqBrCZUhoHcIYQkjQO3fbCwsJr77333urnn3/+0/39/RlK6ZMGiDE3Wjp/DSJPmGERkEjz
    D8YPMyg0gFolgE0AiimlZwCcAaBaIsimo2kDcAPDMDdQSmVK6TOU0kPGfJogWwWgGO9tB8Cb
    G9Qyr+Z7mg/TaH6YDCHJe+Rz203qH83juOtWx+LdQOKvoNNlB/CgzWb7eGmxv45lmMKRsUBc
    1bQggIjxUtmI9JmH+68NI8O4vxRAvKSkRNuxYztb6POBqjIGBwfwL9/8fzKAGeRopGZtoWkA
    s/lz9asUxLoy3nnNXHvttY0ej+dPa2trlba2tidffPHF/X19fbLL5WrNZDI1b775JpqampY1
    NDR8rqOj429PnjwZMs4nhVKaspxjsuVsMgN5iwGXBSIOH9b1ZdrChx9+uKyoqGgTISibmYmc
    f/7550+NjY0pVltIKVVWr14tbNmy5VpRFG/WdR0rV658/qc/ffytcDismiCbUqoCUCorK9HR
    0WGnlPKBQEA7d+6cCR5h8UWstlDHZUoLzK91fjf7Z/nTfLun54PgXwcbaN2vDz74YIvb7f5d
    VdXWS5JexnGCy+kkL/A8fxJzZQGSMY86fs2YiYQQfOELX8g6nc50MBhEf/8AWFYExwnw+4sg
    SfLM8PAwRa6sIk5zE6sa2ULyTmv3/V5HVwDh5eM4rbLZbH9nt9tJLBZ7Qdf1/wEwAqC5pKTE
    s3PnTvj9/oK/+Zu/+XgymZwCcIZSqhmcY9FwqliLEbRG8fILVfMLVq3p7Q+jcw4APpZlvly7
    pPrjvgKP43xvf08imf7fAA5aHHKZEJKilG7yF/q+cd3WqyqikSheeXP/CkVVf59SOpZLMlIz
    61oEYIcgCGtZlnVIkjSm6/oblNKjxmZVLU6/ZIl8XRJQ+C61H+QdfmeNJPPIZba8ABzGWuIx
    l6U25yJLKc0YgHhRnrtFAIks9j8t641eqqjqL5AJpe+2Xx0OB02n0+V2m+3++z96Z/fN123G
    My+8jP/64c9GDTDoM9ZQ2rhW5VfNCL4Hh8dcO0Pj4+PyRCgkFLhdUBUZe/bsxenTZ2IApgD4
    jbVmOg0ZAByllCWEaFik7uRXIYh1ZVx4f/3+7/9+TSaTuToUCvlKSko279ix4zs/+tGPfgqg
    VpIkh9PpRDAYJCUlJetWrly5s6+v76lMJpOllMoA7Ha73e7xeDIul0vlOA6yLOsTExNqOp2m
    lFKdEKJeKJD3YbaHlFJQSvHJTz7sLi8v/51MRv6kJGnuwsKiwWuvvebr3/72d3Yb57FOKZUA
    pNevX9fNsvxfjoyE63meR0lJ0arVq1f9wYsvvtRrvKcGQL722mt9XV1d1/M8v5FS6lqxYkVw
    9erVu1966aVD4+PjaWP+JCOYY1K+lYttC99Fg+G9BJXI5s1X8z5fgZvjOA8hxKlpmqiqKifL
    MitJEslkMlo8HpdjsVgmEolkM5mMTClVjHU0awONubKWoyz2+S5oAy+6gqRlTb/HrOh7+UDE
    7XZvTSTSd506NWpnGDva2yukRCLcFwwGBQBeY18yxnku/yrZwvdoB8ng4GCks7Mz6Pf7wXEi
    GIaDy+VBfX2NHg5Pjw4ODsLwN4klKCpYgqV0kXvyvtvCK4Dw8hkFdXV1rk9+8pPu55577v7X
    X3+9Rtf1fwFQ4XQ6xcceeww8z2PTpk21L7zwwkcppd8FEKWUmgeSGdHijUWVBRBCLgqvGYeX
    jAsLN3yogSGAQlEQN338o3e4uruW4Ytf+Xrd+b6hDgDnLU4Aa1y3z1fg9dz/kVtx/MRpvLb3
    bS9U+Iz5hOEkFLEs+4XOzs7tGzduFAVBIENDQ/qbb7553eTk5PcopS8Y1N2MsY/YvANfv4QH
    0GyE0wLIzADBrMLVIhk+DsDthJDbGUIKCUPsDMPyhBCWUgpd1ymlukopsrquxwHsB/AycoIg
    Js9dMpwLJbdsKGsc/sSgKZlBCfN/q+8WlLgYBtKqeGkcoCwW0p6wSOQ3vwaJyLJMAXhcLqd3
    66Z18Lgd2HfomKSq2tvGvHgtwQHZAq5/pYzgIgEAxhKIYgghHKU0HAwEwgP9/a6WxgZEZmbw
    1DPP0Vg8HjDmttiYH81wqhTMsR9YLFQfvSI08ys8CCH40pe+RDKZjL579274/f7STZs2fWr7
    9u2coiicLMticXExSkpKMD4+LjY0NNywatWq8N69e88tXbrU1dTU5CsrK/O5XK4CnuddDMMw
    lNJ0OBw+8eqrr+4eGhqKGOINMuZqdEy7uOiZ9GGyhc8//zy83gIvQK4eGJj0hkJJdHc3LLHb
    HSsEQTghy7JmBJLZXOaP+FRVKxgcnILH40FxscujaboPgMuwG+rKlSt9XV1dn00kEjeNjIw4
    FUWB1+ul1dXV12/fvv2HP/3pT5+OxWIJQkjGsA0s5lNO9ff77FkM2C3yYC2fxZohng2Qb968
    WdywYf3dHMftoJT6dJ06dF3nLXWsOkBVSmlW07RYIBDY/eSTT76RyWSsdc9mMEIpLi6mHMex
    U1NTjKqqZgkLY9hmqw1coKh8CYOk+aqXpi3MD97m28F8PQVSVFTEASiKxdL8xEQSLS1+sKw2
    fPjwkZOyLLuMv88Y15vFr252MH9Ordlm9tixY2htbR0rLi7RbDY7q6oUFRXl8Pm8qd2735yQ
    JMln+O5muZJi2ELVsLHqIrbwvYL2K4DwwzRYliWapmUASNdee61727Zt3O/8zu9sfvXVVxVK
    aba4uJhvbm7Gd77zHXR3dzNNTU2rz58/Pw3gCHL1SzUcx/mKi4s9LpfLzhCQRCKRDU1MntJ1
    /SfIUSczlgjeYgWrH3ZHayaTzR49ffbcMjmbJsHQ1DiAAeSyXtSgvZgO+ulAcOLkT554dv3Z
    ngFFkuQ9AKLIZcqosTEf7OzsvPGGG27gBwYGaCQSoS0tLcx9991X8d///d/3RSKRGUrpUQB8
    Hugx6aYXxVmyHMam+qxZJ8IZ11poPPyGs80Z9z9o+WxmcMBFCPnIls0bb79x+zbYbTawLAtC
    AKrrUFUVsiwjkUhi16tvYP/Bw8W6TnsopREAaQMQpwghaeMwvAlAO4AJSukggKBRH5YxDJ41
    omqNIF90ZUALCFxCKS3AnNJlymIIzTokJe+zUashVVWVAWCnus4FAgE89+LL9FzvwHkApwEU
    GH9jGsGUea9+gejiO2YDLgPn3XR4BGPNOYxrNeeQBaBNh8NTg4MDNZqq4PCRI9i3/60scpnU
    EuOtygDUG3PWD+AIpZQzHMz8INavk1LrryUmzGQyUVEU4yzL+o8dOwZBEArWrVt3DyFkYnJy
    kj969Cg2bdoEt9uNiYmJko6OjltSqVTBmjVrVpWWljZQSoXJySludHSMZLMS7HYHqqoqbrj5
    5pvXHDt27PlDhw71Z7PZJICMscbSxkP+sNvDHTt24Jvf/GasvX3ZocJCTycgsDxPQj09Q72y
    LNuNl5mlEPKZM6fPrlnTfay+vvwah0NQE4nYW0eOHJkE4AZAnU4n293dfW8ikbitt7dX9Hq9
    tLCwEDMzMzh16lTJsmXLPtrd3R3etWvXAUqpYLGF+SIi7+tYRMafA8DV19c7mpqaCgsKCvyi
    KPptNluRpmnk9OnTp/fv3z9pZPFmbaDNZitkGOae0dHQdaOjIaiqDhNfMgwDhmHAsixEUURN
    TQn8fr9HEISBTCYTMRgzKXP9NDQ0MDt37ryd57kmSZKD0WhkcHw8EDp79kxkeHhEMvQLNKv9
    s9hE7VIFvCiluO+++9x+v79OkiRMTISyR48eiw8PDytG4BQWX0HB4hRgs46e1zTNznE8qago
    QlWVJz042L/3xIkTWcMmyMb9kYx7RN5PW3i5BGsopcRmszFXX3212+/3F0iSJITDYTo8PKwM
    Dw9r4+PjzlAoNFlQUJB1OJxOSVKxdGkdotGZmd7eXuJ0Opd4vV7Z5/NVFBX5ZZvNFhkaGj5y
    /vz5XmNfSZhfg2nek/d1fKCA8L1KI1/ui+GXnQNd1wmAiVAoFAkGg0U7duzAV7/6VTYUCm06
    ffp0prS0lPnYxz6GUCiEN954A1u3brXPzMxsnZqaKkROzKKYYRi+oqKCrF+3Fs2NDWAJ8MJL
    u8peff3N5mQq/SqldDeAUQBJ05HHXKZHskQgPlTCF5boYIJSuv/tQ8fuON874IwnU+cMEOTC
    HO1RNZzXiCRJw28fPr5iJhpTAIwbzquZXe0uLi6+vqOjg//xj3+c7Onp2QsgvGfPnk2f/OQn
    q9etW1f+/PPPb0aO0suaNBzL/1He782aR/VkkKuDLDSc6UYAzQAaeJ6vdrtc5U6ns0AQBdvk
    xKSWSCb/B8ATBiiWLYEBF8eyvk0b1uMzn3wQDDFpRzoo1QGdQtc1yLKMSDSKtw8dderQWgzw
    nKKUxozvIwA8LMs+srytaa3dZpNGA8FUJBqfSqXSQzTn5PdQSkcATAOIAchaFLXkRUDY+zpc
    LhdNJBJ2AL8niuL1HMvOZDKZgE7pHgC7kKt1MFUKMxYjZjXQJsV2NYDPpDPZsm/9z2MYC4Qk
    TdOPG78vMv6ONa5RwCIZwneo4wQWijUgP0p7GWUuHAAeYFn2Rp7nXZqm6YqiJI11MQMgo2ma
    GBgfRzIRx6uvvY7QxAQFUANgqcDzosvlEBx2OweAmZicHlFU9W8JISeM8yljedCLsTaujMtq
    MKOjoxNtbW0Tfr+/bnBwEMeOHUNhYWHZ6tWrSwYGBpiZmRn09fVhw4YNOHfuHFiWrV+/fr1d
    lmUlHA5HRVG02WyizePxiAyT4mKxBE6cOOetq1tyS3f32q6lS5fuO3r06BtnzpwZUBQlQQgR
    DV/IrD+UcQHmwuXscxBCzM+aqa2t3ef3u+52Oh0+Rcn0Hj58eAw5ETWKubKS7JkzZ+MrV64a
    Kix0plmWqIFAcHxmZoYY+5ps3Lhxuc1mu2loaEhsbGxMC4KwPxaLhZqbmzf09vbWBQKBktra
    2s1Op3MwlUoxliychvklBe9bQNT8cfXq1ba6uroij8dT5nA4GkVRbOE4tk7XabWqauWaphbo
    OhUdDrvc0NDwH8ePH3+BUqpqmqbIspwFkA2Hw35FUQtGR0M4caIHhDALHgzDgud5OJ02FBTw
    Hq/X22K322OEkJSmabFUKhVJpVKxwsLCQoZhPxOLyct1nWa93uKUz1c82dbWOhSPJ/oDgfGe
    oaHh0bGxselYLBZDLrMoGXZQtvgopi/2vp/zBq2Y/uEf/mEXy3J/p6qq6PMVTi9d2tg7MDDw
    9FNPPXVGVVWTUWVS+E37o1tYIcxdd93lr62t3cmy7PWFhS6moMAFWU6N7t+/f1DX9ULjGtLG
    36aMPcZcyLdZRO2WXEAHQ8dlxGgz//enPvWpNo/H8xu6ri/VdZ1bunRptru7OyFJ0kw4HA6z
    LFsmijY4nW7Y7RSVlWV6KpWwXX/99V1er9dmtztFQRAFQlhGEDi9oqLyqeHh4f/OZrMzxjxm
    LPdD/5UAhIsAPmI69XnOUf5CWNDL68NyUL8XIwhgIh6PT5w9e7Zx+/bt6OzsxOc+9zn7N77x
    DXtLSwumpqYQDocxMNCPa665Bp/+9KcLvvnNb3aHw+EogFFZlrPHjx8X+vr6PC6X09XW0mS/
    d+dtzIY1nTU/+umT950827tBVdXXAOw2auWs0ZoFstIfBhEH62FpZCvkeDLVr+u6YFybG3Pt
    OCQAOiEkSymlOtWHJ6bC+7KSbDeu22O8bY0oindv2bKleHh4WO/p6dkH4LsASDKZdJw4caJ8
    5cqV3Isvvlij63qJmUGy1HOa2VeCi1BQz3EcFEUpBfBJjuM2ut3uxkKfz19TW+NsqK/n6uvq
    SE3NElRXV4FjWfzu739Z37N33zbDATcBTxrABICzILARhkDXdWhUB0xAqOuA5WdV1QCgluPY
    BwmIqlNd0jQ9ZTj9YwBiAs8V33vnTdhx7SZxdDwknu8bKDxzvr/59Pk+dWB4LBmNxSeyktwD
    4ASAPZTSSSM4YR52xOKsvK9zp2mzZ6e4qmt5zW07ttWeOXNm5RPPvbw2lkjJlNI3jf8pYY6z
    ry7izKyw2Wx/ffWmq7q3bN7IRCMzeOrZF/hz53tbaa4mLmUBL0kDQL5ThpAYmUu3cR6nMUf9
    nTXEyFO9vRyMIc/zVJblBq/X++jdd9/dunz5ckiShEgkgmAwiJGREXVkeFiORCO82+2Eosig
    VEdrS5O9uqK8tnZJFamprkB1RSnKS/yIRqP46l/8Q/X5vsHllNKgEfRIGFkHq1Ldr0Yq7J37
    7v3cgdJfkZpKcuTIkXhLS8tQSUnJOpfLhdLSUvT29hIA7NDQEPx+P1asWAGbzQZVVfHGG28w
    q1evqaisrJBZlk2Ew9PBSCQaBkA9Ho+ntbWpdmxswnfs2Fm2oaG2pr6+qvL660u66+vrn3zu
    uedekSQJRn3TAhpfvjN6uVJI82vDZVmWed42RKkaVFWVUko9xvmiG2eSDiCTyWSURCIxxrLC
    Plmmtlgsppq2sLW1taKlpeWeYDBY5nA4qCAIh5588sn/GRkZUe+44w62uLi4anR0lC8tLa32
    +XxlqVRKNmyhbrGFmYthC7/whS9UejyeRyil61VVXZrJZHyh0IQzGo1x0WiMxGJxJBJJMAyL
    a6+9WisrK9vx8Y9/vNkAq4qu6+lUKjU+PDzcr6qqkNtjJgDMB4U5l5QQAp/Pt/SWW259GCAK
    IZAppUldV8PpdGZMVZWUpumFfX1TmJxM2ZxO0VZQYPcXFNhb3W6v2tLiT7a0tAUymXTv5OTk
    sT179uwNhULTliyj6Tss2ubj/c5oqarqP39+rErTONTUlGysq6urqq2t/UZfX1/U8GfSlrU/
    rxb0mmuusTc1NX0ync787rlzZ32plITy8lKUlhaU1NXVLQsGg/3GWc0ZXwVLhnCxw4wAII2N
    jUJ7e7v39OnT2Z6eHtUoO7H6qFbK7QcaILWeyY2NjZzb7b5tenr6gVOnTvGyLEMURTidTrjd
    burxeGWPx6Ol01mbrgOCYIMkKYTnHSVOJ4t4XCbB4AwyGQ2KQlBTU4SSEndHSUlx3cjIKKGU
    CgA4S/bWDLR8eCmjixSyWvsM8YZMOWuhxFlpePmPDzXXfxEDn9Y0beD06dMbX375ZfLjH/8Y
    t9xyC/793/8dS5cuRTwehyRJWL9+A1avXoVNV12FFcs77Lt27bLt3/+W0tffP5PNZkdjsdip
    WCymj48HOnRVbfzMg/eQdWu6hMHhseZoPLEEOZrW/1iKy3WL8hreCSBebvNrbTlh9DwbjcYS
    0UQy3Wg44z5rNNQQjJEAOCmFLyMp7YqimmquXmM/bPJ6vY1TU1M4fPjwOIA3DDDpASBns1nd
    6/WC53mnwftOWA6prCXLeFGGoepWLoriI3fffXfNrbfegsalS1FU5IfH7YbNJoIY8xIMBkEp
    ZVZ3LW+rr13SRgxgNBON4tDRk/F4IvkYpRA0VZu905TSBUcMpRSqqqKtqc697ao1bo5lICsK
    Uuk0ItEEhscntJPn+tM6pTa7TUSxvxAFXjc6WpdCvl5CJBrjRgLBgpNnegveOnS8+fX9h9dl
    spIM4C3MFU5baUbv+0Fn9ETKAjiYSCR3rlu1wrW6fSlefmO/GEukXMhRPXVKadZ0GjBHzyCG
    oyMCuHPjhvXdf/1Xf87UVVcgm06gs72Z/dJX/nTZeHBiCkDfu2QHrYI75hlYCuCLRrb3MQDH
    DYMMy3yYmV3gEjaqfafzS1VVAKgsLy+vevTRR9HZ2ZkLXeo6Mpk0YtEoFwlPcxOhIKqrKmC3
    ifjCZx7Bgx/7CHHYBDhsAjgG0DUFVFMxMTGJsmK/cL5vsNI4p3jkaEmmgyldCFx/yMEgyVsP
    VoVDJu9Mfkdlww8jKMwDMxgfH9clSRr2+XxaS0sLu2zZMuzduxcvv/wyNE0Dx3F45ZVXYLPZ
    MDk5iVgsgZde2kX8/kKxrq5erK+v9S1ZsiQWDAb7Dxw4MFBcXBxbtWr1WpfL5XA6bZqmqVld
    17xut3s5z/PHJUnSLUFps54qiznl24tSr3OxHH0A7Pj4eLClxT3ldgsdweBMLyHEi7kaJbM1
    h1RcXCwyDONzuextqqrK6XSaAPASQtj29varRFFss9lspKCgIHjy5Mk3RkZG0gDciURC9vl8
    mizLPACnzWYrRI4pYvprGcwXJ6Pv4zXiD/7gD2oopZ88efJkZV9fP2ZmIshkJMiyDF3PgTuA
    wOv1gmEYNpVSl6dSWE5I7nc2mx0VFb6ZZDL5E0VRuBwIBBbJVxhgEGAYgmyW9cZidi+lDFiW
    gOMY8DwDn4+oDgfNALCrqo5sVoGi6IjFJIyNJSCKPOdyiQUFBfaCoiJnW11dw5rx8bF0KBQ6
    SikVjXWX7+/Si7HXCCHk4YcfHiovLz9HKar6+6fgdLpJaSnryGazHjMYaoAxE3xIls9HV6xY
    0aiq2n2HDh3znT7dD46zY2RkBt3dy3wtLS2rT506lYrH41Hk9CvEvMAozbOFhFJKHA4Hd/PN
    N1/tcrk+u3Tp0iOHDh166rXXXgvnZZ3lvGzqZbEva2pqREJI2+DgIP/WW2/NBqAJIRBFG3E4
    XKLbXQCOE5BIpMGyKl555SBhGB45giAHlhXAcQIYRkQyqaCkRPAJglBuCSwTzBcyfN8H9wEY
    QGI2daaUuozMRQuACsMpp5TShJHBGDEOGQk5vrXpFOU3n9U/xAbRWoyanJiY0MfHx1mfz4eW
    lhY0NTXN0vj+6Z/+EYosw+VygWVZ7LzzDty44wbS19cnvPTSS2VPPvV08bHjJ2pSqdQAgMDu
    fW8Xnj3f647GYol0JjtpgJcMgHIAgqWZtmA80sb8apjf2PeymtRFMsMEQDUh5OHu7u71q1au
    FJ559tkbRkZGPAboiCNHzwByfR13ti9ru/4TH7/ffujQYfrTnz1xp6qqzyBHZ0xNT08PvfHG
    G3FK6RvI0UlFw2lvWbp0KR+NRiHLsqnu6sZcwbRVFYpcxHmL8zwfufrqq2vuuusuUE2DbmT1
    qK5DpzoYEKRSSei6jk89cC9u3bENupH16x8cxme++FUhnugrppTadaqDgi76aSkASgFN19DS
    UINPfexWiDwHXdeg6xoUWcHJ8/3sH3zj39zjEzPQ9ZzSHdVza5bjOZQUFaK4qABrVrSho6UB
    h0+ctWWyUhGAUiMIZPafswKwi7BsCKWU7j57rufAf/zPj7auaFtKkqmUZKz5QuNy05bsoHne
    mA6hyPN8W/ea1UztkmpoShYMKLo6WlFfWy2MBydKjPWStWQZ54mh5InbMEbk/pHaJdWfrqup
    dh87eWpdJBr/LwCvUEozBnUni7nWMguAwAe1Bw2qu8BxHCuK4uzvGIaB0+GEw25Dib8ADXXV
    0DUVmiKhwOuC2yEAugZCNVBdyy0wqkNVFGSyWRiZQT/mlEol4/rZX0EgSACQ+vp6sbm52c3z
    vJsQ4spkMiQajcqBQEAJBAKSrus65sTBrFS8BTbww5QtXCxIvHz5chfP81WqqjLDw8MYGxvD
    9PS0GYCALMsYGRmZzdoQwkCWZQSDIUxMTOHMmbNsfX1dYWtrS8G2bdsaBgcHxvv6ese9Xq99
    YmJ0JhyenuB5IctxXHLZsmUrFUUJh8Ph4WAwOCbLctRwYJNYqJZp9qS9rLKyefec3H777RVN
    TU0PRaOxzWNj47ba2pptO3bc4H755Vf2BgKBqOncNzY2Fmzfvv02SsmOt9465Cos9NGOjo47
    pqamnj59+nQwGAymi4uLB+12e+bkyZO79+zZMwBALC8vL66trW2LxWICz/PQNE2Jx+O8xRZK
    hm3k3u+z3LxWVVWjhJB4f/9A5blz54xMHjub3TOXkyAIIITg1KlhjI+Hjd+zKCx0Y82aOkGS
    ssWaptrm7ul8XRrzaZOOG4+rGBrKAGDBMCwIydUYFhRwXFMT5+Y43ULdzT10HchmNchyFjMz
    MpJJFa2tbls2my02A1+WshPFEvi7aEvmiSeemP74x+9/oq6urCubZf1OJ4d0OpGYmJiwGeBD
    tmSUFYvfTQFQm81WGY8nq8bGQtB1gGUFKArF1FQSjY1+n8fj8cXj8TTmSmpoXiArnyLK3XPP
    PZ02m/0rk5ORq3ie375q1apuXdf/84033hgyAqQmEEphPnPmcgjEMJRSQVXVeWcCpRTZrAxV
    TSGT0cBxAjjOBoBBNiuDZQlYVjDWEgtCODAMB57noGkqk81KBYYtBObUR9O4SEJ13CU+sMwF
    4KCUrgFwh8vpvKqioqKmvLzU43Q4iKqqiEQi6ujY+Mx0eGZQVdX9yKkbjhuLwcqllfKA4Yeu
    KbilKX2bKIobV61axd5zzz1gWRaCIBjXkrsep8MJOByG469BB4UgCGhf1obWlibcduvN7JNP
    PVPyvR/80H/q1OmwrCgzgdDEqGHcAIBlWbZe1/VqQxikF8Ahg0JqRnHSlnlVsAhF7XIxhBYw
    WATgN1atWnn7N/7i68LKlV3YuH6t++t/+Y0tZ86cLQSwBzk6XyEBtq3tXrPha3/0Ff7qqzeh
    t6+fTE1PN7762uvXAngNwICu6wEDPE8bzmk1IeSGtWvXtnV2dpJvf/vbqkFpo4bhEyzRUOZi
    Rq1YloWmaelMJjM+ODjQqWtabj0YdX+zwI7ksoG6kRHsGxiGLEtQVRWDw6OIJxIsgCaGYUrs
    NpuBXqnxwXNrzrr2GMLgfP8wfvbC66guL0GBxwmnXYTI89CMQnyWYcCx7Oz7mPcoR0GlAEOR
    zmQgK4qOXI1KifEPTIOTwUVU5DRqL6dVTTv8+DMvbtx74LCYlRQ7gG4TaFuMjalCaLYX0QDo
    qqqGR0dHkUok4LTz0EAxNTWNcDiiY34DetVQM7QW5Juo1AWgzQiGNRf5Cz/1W48+4t56VTf+
    7b/+p+G/vv/459OZrAbggJGxtBrm960m55cZuq6bZ208Ho8nAoGAo7m5GYqiIJVKYWx0BIMD
    /RgaHATVNdy4/VqkUkn89GdPIhaLw+20o6iwAIVeN9wuO+wCh3O9AxgaDZjgz2F8FfP21odW
    pW6xc3/Tpk2uzs7OdqfTeRUhzEpKaVWuxyWluq4nZVkemZ6ePnnq1KnTp0+fDqmqaq29lRax
    gR86xoxZl0QpJfX19cK11157i6IoO44fP06mpqYWm8lZp9t0vBmGmX0+kUjgxIlTGB4eZdra
    WgpbWpo8iURiZv/+/eMMw6ibN29udbvdNl3XOV3XwbJcFqATw8PDu3bt2vVGOp2esYBCqwAb
    cJkxk/Lr6nbu3Olrbm7+7MxM5COvvfaGbXR0HA0NDc6NG9du2r79et9LL+16c3x8fLKsrMy7
    devWrYqibdq796A4MDAIj8dFrrlmc9369eu3TU5Ovrxv377Rnp6eKUVRkpFIZJphGKGzs7Ou
    u7t7u67ry8fGxpiWlhYtEAiEZmZmdIst5CznOLkY1/zAAw8kKyoqggUFBa1mUMBK7TT3mgkQ
    7XYeXq8DLMuBYVh4vU6IIs+Xl1e02Gz2olxJBLH8DzNbSOZlDT0eHuXlNmSzFKpKoOsMKJ17
    jdn6w/oeucfcZ+M4Bpqm0HA47MKcsJbp7F9UG2he3szMDPr7B04sX758rKOj3M+yhGazem1X
    V9eqw4cPD2maljXuo2r5XGZtrSZJUkoUhURBgdcbj8sghAXHCXC57NA0Vclms5rVFmK+OA2l
    lJLCwkLu1ltvrSgq8rfrOi0WRfG2ycno+oMHzxOPx2fv6qq/ccWKFXx/f///GxsbC2OOfmwN
    il0W4k/BYFCjlE57PB7wPA9ZlsGyLBwOJwoLi+H3F8Hj8WBmJoGJiQhqa6tRXl4CQhgoCoUk
    UagqoOsMGIZDUZETyeRk3FLPmzICBxyZi3h8uDKE+dE/49AvBvCg1+N5eOPG9fU337iDW7G8
    AyVFfthsAjRFQSQaYYeHR8rf3LOv7PmXXuk839u/Wdf1JyilxwghMcNZT2JOwU+yLI4PVUTU
    ONArCCGf2759e/tDDz0Eu90+/3Ch1HD0zZouy/OUQjOAQG1tLT7/2U9j81Ub2H/71n8WP/6z
    Jz3RWCyIHN2xiuM4vrq6ijQ3NlCWYejpM2ezw6PjrxstLCaMAz2JOdU18xDQL8dsobGenAAe
    aGxcet9Xv/IVccOG9SAA7rzjdni9HvGP/+RPuw4fOSoAOARgWVdX56o/+9pX+Q3r14HqOhob
    GvC7v/1bzMjIaGtff38MwDGL8fcBaHU6nZu2bdtWsW7dOubpp5+mp06dGjbA9IVozLhYGUIj
    K5PVNC04NjYOSZIg8PyiTpbd7oBoE/Ef3/kRHn/qhdkGSaqmw+P1ctXV1c0N9XXkqg3rLJ+W
    zoezFOBYBtds3oDJqSk89crbkKQsqK6DZQCe5ZCRJExMR+Ar8KLI78uBvzkO6uz7UgoEJ8KQ
    ZIUiJ/bjtwDBpCXDerEGA2Cd2+2+9Ybt24T13asRCgadjz/5zE19A0NFAN5EriZSNda/Wdth
    RkYlSumuXS+/uuOfqitLr9m8HrFoBN/90U/RNzgSATCJ+RRPGXlcf0qpH8BnvV7Px6oqKsqj
    sZjAc6yYSCbw3R/9FMdPnaO6rvsArDDWWMwAlhwWb5T8QTryBEBgYmJi4Jvf/GbpW2+9hVAo
    hMHBQTo6OqIGxgOZRDKpO50OV1lJITc0NIK/+Yd/lRVZSTMsQ1iGZQWeY11OB2OzCUwyldGn
    ZyLjxj14J6GdD/UwaX2PPvpoi8/ne1BV1ZunpsI1k5NhMRZLEFlWwXEcXC4XLSws0IuKSm69
    9tprexobG19+/fXX901PT4eRU/c1hcEymK8Y/aFAglZhJTO4d9ttt3ULgvCZvXv3Fvb29l7A
    fs5lBnNgkFieZ2YBQSKRwOHDRxEIhLhVq7qKt23b5gyFQklRFN0ul8uWTmeY0dFxBAKTWLKk
    qqyhocF39uzZUE9PzwAAG6XURghJGmcTazjGl40Kad78kU2bNtlaWlruTSZTD+3Zs88xNDQC
    AOjrG4AsK/ymTeuXX3fddfybb755oLu7eynD8Gv37DkgDg+PgxAW8XgKBw4cJddeu6lpw4b1
    0eeee/7I5ORkBgCcTqf36quvbmpsbNw8OTlZNTIywtbX11OGYUYPHjx4TlVVJc8OXkx/gZw9
    ezZZWVk56fF4wLIcdH0OhFmPDEVRoGka2tqqIUnKrFkmhECWJYHjbM2joxMkV8qHBdk98310
    nWJsbBqiKKCkxAaW5UAIB0JygJBlCXgeyGQ0SJKW9x5kHji02zlks1kai8Xdhg2UDTvzjjXn
    7+e4+uqrvS0tLffGYvHG/v4RZLMaqa4ub163br1bluUXjh071m/xrU1/MGtm5M6dO9vX0bF8
    z8qVy+8WhH5GkoCyshJUVxeqPT2nxmdmZszelEqeLdSMvc48+OCD6+x2+5dSqUy3qmouQSCi
    oqhcRUUJiosLqSiyhBCxuaSkpGFsbEwy1rp8MYMNv2hS4syZM9r1119/srq6Or1x40YHpRRe
    bwEKCry6zWaTKIVks4n8zEzScfDgKdLR0Uh1XUqlUinV47GxgiAyLMuzDMMxDMMSSuX08eNn
    R9PptHop7f5FzxDW1dVhYGDAdN5LAPzu0qVLH/nCo5/z3HH7rSgt9ud4QZqSo77pKgoLvWio
    XYL1a1aSm67f6vi3//re6qef31WWzmR+TCl9G0DEUp9jzcgouAhCHhfLKTDmxA7gvo6Ojht+
    93d/l62trZ397HOgT7eAQsvP88BiThGSYRh0rliOr//JH5OuzuW2f/inf13S29efApBQVVWY
    mZkRRaFN+P0vPsrs3rvP+aff+Lu1yVS6h1IaAhAAMEoIiWNOEYrBRZS5/SWMoFk3eL3f7//8
    b/7Gb3i3X3+9cXjn5uHaa7aC51juy1/947YjR4+V1dXWFHzl936bX9e9GrqmgRAdYBhsWL8W
    n/30J8U/+bM/X5FIJOPIiaTYAbTX1tZ279y500kIwbe//W3p/PnzfZTSw0bGMZ13WMqXyCGT
    AUxMTkxo8XicLS7y50CYwe+koIBOUVRUiK/90VcwMTEBjuPAsSxYloVNFOByOYnfV0C8Xg9s
    opALNpjRd1BQAhAz2UgItmxaj+5VKzA1PY3JyWnMRKJIJJPIZDLQNA0sw6DIX4CO1sYcNdUE
    g9Y1r1OMhSYgy4qOXM1eoWEEY0YmlrtYh55BF3UyDHPXzTftaPr6n/x/pLjQg0wyQeprKm1f
    +d9/1RWeiYSQU0OVCCEJI2OYBJAxQBkopXsnJif/42//4V8+/90fPObNSll5enompGraWeSU
    Na01F/OcdIMee1d9Xe0Xv/ibXyjoXtWJ6akJ/Md/fxff+Lt/1bLZ7JiqakPG+utFThVQQY7u
    zOb1/LssfHoAU5Ik/eBnP/uZ/NRTTzlUVZV1Xc9Ygyo8z3WosuKemJxENiulAIxpuq4pUElW
    kkg8mTKjyBnjDEpifuPwyzIg9YtmBktKSvj7779/k81m+2owGNpw8uRZYXQ0gEwmVyJJCAeW
    5cBxAhEEkS0sLPA0NFSvqqmpabzllltaXnzxxacDgcA4pTRm2EEzI5OxnNUfJrYMAUAeeuih
    cqfT+ei5c+dajhw5YhWCWhQMzs8Szv/ZfI2uA4FAEIlEknR1rXDW19cIo6OjkRMnTgRLS0tt
    ZWXlxUND48LQ0DiqqspKr756812bNm1KRiKRsy+++OIrqVQqbKnvgsXX0C+X+aOUEkEQ2LVr
    116tKOpvHDp0qLCnpxeUUjBMLr42OjqON998i928eX3LLbfcUqzr8L711nFxbGwSDGMCGx0T
    E9M4ceIsv2pVR0dbW1vs+PHjI6Io2rdt29ZSXV297vz5825FUbBs2TJFVdWB11577WAgEJjM
    s4WZi2wLyYEDB6QtW7ZMut0uyvM8kSQF84/H3BpIpdJ44429cDgcRjlDzq/SNEBRVCJJCpFl
    BZoGXCjxkptjYHR0CqFQFHa7DQ6HCFEUIAg8OC5HVaWUIJtVkUhkF9GImFu7djuLRCKmS5Jk
    2sCUYQNFXECJ8/12Pzs7O5frun7r/v2HHIODAfC8E4FAjGzY0FGxbNmy7nPnziWz2SyllKaR
    E/aKY45KSp977vkYw7Dfam1trdyyZfVVsqyBUjU+MNAzsH///l5d1xXML7uQkWPN6ADw6U9/
    qtpms325r29o+8mT54mmsaiqqsTy5Q0oK/MlJyZCfadPD4xNTU2N9vb2Zo0AskopTZGFiP1y
    yNDTs2fP7mttbf3x8uXLWyRJIslkMhMKhTKBQECdnJzku7q66ktKKhsEQSQ8z+p79x6aPHPm
    TJznedjtdsbpdFK73a5xHKek0+no+Pj4KOazgjSDVnzRapovKiCklGJgYADI1ck5ADzY1Nj4
    qa9//U/dN9+4AyxDoKkKNE0F1XP1JLquzwJDnmfRtaIdX/vyF4nb5aj+nx89/hFJkjUAJ/JS
    p9Y6nQ+FMbQYwjU+n+8Tn/3sZx3r1q1bAPKsmcDcXpp7bnGwmKvfKvB68MD9H0NtTTX7Z3/+
    f1wHDh2OUUpHYrE4eWP33vp77rzJ3rmsBTXVlQWBYOgTWUlGJisdBPBti2M+r4nr5TBn1lpU
    AM2CIHzxox+9e8nHPnavybuemxcAmzddhT/+6pf5L/3eH5a2tjRizcrO3Foz348y4FkWd++8
    Hfvfetv705892WEcYN7y8vK1jzzyiKOvr48+/vjj4Xg8fgjAeYvhS+Rlq62Z6ou5+CiA6anp
    aSmRSDhKiosAYkQhKQGlDKiuQxREXLVxQ27NGPWFdF6tYW6/6ZoGXdOh65pFYdRYY4YPzhAC
    l9MBt7Ma9UuqjHRX7m/NdWkGJXRdz4FJYwWZFFJN1xAMTUPLhXOdyNWcOA0waBUgeN+HAVKd
    NputZuP6daS8tARyNgWeZ9G9cgVKS4ps4ZlIqZEpT1BKnYQQu5m1NI0QISRDKe31ej0Kx/MI
    jYwOI1enqliCUnJeVNRcD26GYW664/bbCj5x/71gqAqtvgo80XHo6HFtZCx1GjlRmpTxtw7D
    uboUTsIvCghVAM9rmnZc07Riw8EpQK4evBCA4HG52IKCXODBIuykWMCzmY0121XEjEcCuX6X
    1szMhz4zeP/993eLou0vzp3rWbN//wFEInEwDGfQ2cgszY1SQFE0TE3FEItlSTic9C5fXnfj
    dddd53j66aefnJmZCWC+KJNVDEz7EM0LCCFsWVnZjkgkct2BAweYVCq1iL1kFgA+a0ZnsefN
    v0sm0zhw4DDS6Qzf1tbsi8Vimbfeeit00003OYuLi4RAYBpjY9OCx+NZ6/M5iNPpLHE6nadT
    qZRiBIMUi3NGLuAUXvIgg5mheOihh+pFUfytU6dONx47dgKaphlAj8yqZQYCIezff4i75ppN
    pel0FsHgtAEYmZzfRXM2oLd3EJWVZe6urq7lQ0ND2ZqaGkd1dfX6kydPulwuF21qaooMDw8f
    2bt379np6WmTVptvC7MXyxYa801lWZ5yOp0yzwtiLvtnjUFSEAKoqoaRkfF5LSSstYZmTSHD
    sPP2Xr7KqAkKFUWDqmaQTEp5qqTs7FrLvRezaECIYQhsNhZTUxEYjdtdxjlvpcZf7IwW7HZb
    RTgcKQwEJkApAcOwyGQUTE8nSEWFs9DhcPiz2WzKsM92y+djCCEapRSHDx8ONDQ0BDUNRBT5
    7JEjxw/s2bNn1BC+UxfLDprnU2GhvzORSK4/fPgYmZpKwGbzQJYnUFjog9/PTz/77LOHpqen
    48aa4o3PkMJlWDZg7v9du3YFDx069Hccx5Vls9miTCbjk2XZC6CA47him81WJ0kq0XVA1ylY
    ltUURckqiqKn02ktHA5nMdcOLm6xhbOBacxpfFyUgBR3sQ8s80dK6eqCgoJPfuELn3fffNNN
    YAmga+o8kGMFP6A5J1bXNRT7C/H5Rz6O4dGxyl2v7bnWmKD8/m8LCuwv10ixhebhBvDRLVu2
    LN25cyc4jjMcV5pHDdVn67lymaA5Cum8OcNc5lCnFAxDsHXzJrhdLuZrf/oXBW/u2Ud1XQ+k
    02l57/6D9o3dnbjr5m1cOpnw79r9No6d7i0E4DYarkrINe41M1/kHQzSJU/RG3P3yNq13es+
    /+ijKPB658Ag6KxQBaUU12y5Gr/3pd/C3//jv+CZ51/AvR/dCYbhQAkBIbn58vsL8fCDHyeH
    jxytGBoeaQZQuG3bNkc4HMYPf/jDyWw2+wqAQUvQwerEJi3tE2SrEbyI8zM9NTWVDYfDjqVL
    G5BJJBCPx5HNZiBJWSSTKcTjMSQTCSSTKaTTaWQyaWSzWWSlLOSsDFmRoSgKdE2DpmvQNG3e
    eiKEgCGYbdDLsQwEngfPcxAFAXabALtNhMNhh8thh9vlhNNhg8hzEAUeoiBAFDgwBlgNRxMI
    Tk7D2K8EC2mB9Bc8X951CIIAWZYlRVFig4NDyGQy4BkCnVKM5zII5n01RZZ4SilvUR00bifl
    ALS3NDd6fQUF6OsfSBvnkSkiY2YGswa9xUpl51mWLagoLwPHMpAzMnRVQaHPA5fTYSrd+jAn
    C5+yOAiXXYbMUMjRDedvxljr1v5jCoDqggIv53La4XW7IAoCl815bxnMiTHFLQYwYjGCceN7
    08FU8SFvSv/II49UOByOL/b09K16443dhuIcb3FE2XlOvLn3dJ1iZGQKDMMKK1bUXLV27drY
    iy+++Lqu66xFlj2/GTi9nNskWD/Xww8/XMowzJ3nz5/3jo+P57323bKBzCKgcOHzqqrh1Kmz
    0DSN7+hYViEIAhOPx5X6+iW0tLQEDMNBllWiaZo2OTk5E4vFTNEwyajlNWsJmfwz61LPs7WH
    6bZt25xFRUUPTE5Objlw4CDJZDKzYDB/TsbHQ9i//zDWrVuJpqZanDrVC13XjM/PGAI9Gk6d
    6iFbt64vW7NmTXNJSYkrFAq5OI5DfX19+ODBg68cPHiwX80V3qmWAKnVFlqzhO+br2C5bprN
    ZqecTqdks9nERCIJnhcgijZwHAee5yEIImw2GwRBgCAI4HkePC/k2DIcB9ZgzDAMC5a1gjuy
    INs4F4OnMB16Xc9lGjWNQtN0qCqFqlJoWu41lBLoek5UxgSrgsBCFDkaj8c1XddNBtYvBXJ+
    gabuVNP0mM1my3o8HrskxQEQ8DwPl8sOSZJUSZLMnrum4GB+wFZfsWJFCc/zzSMj4ygv92vx
    eDymKErasKFmr+Ms5tc5G/sFbllWxJywCg+Oy719NqtB1zm7rms+y7zIWFyl9HIKaFFd15Vw
    OGwCWGoFxaIoMj6fj5mZkQilBJpGSUFBAWPsE1MfIGWxfRHjYbWDCcwv4/rwAMK8CbMDuP2q
    jRvrd955B3iOhaoqs4BnVryCznfkTedU1zVUlpfg/rtvY44cP9UYnokuM5EypVQxeoctEJi5
    nLODRrS4xe/3b/vIRz5CSkpKDBVIk/Knz88SGlmexbOH7/z86q4V+LOvfYX80Z983ff6m3uJ
    rCiZ//7+Y9xjP3ua4TgGXreTD4SmUwbgcVvmUspz7j6w2ok8YSIWwHVlZWV3f/7zn+eam5pm
    5848eSnm6CEsy+D2W2/G8PAI/uPb30N9XQ02rOvOHdS5tBp0SrF29Urccdst/D/88zebKisr
    2fb2dvznf/6nnM1mTyFHZTMjX1krIDTohaYRvNiZDNPxGwmFQuP/+q//WsgwBPv27ceLL76I
    VCqFrCTNo2uyLAue58BzOXoLx+UMIWsYQpZhwLDMLHAjhvHSqQ5d06FpGjRNhaZpUFUVqqZB
    VVQoigJZUaAoxl42Tm9RFOB2OuD1uFBVXoqbt22Cqqn4wc+ex/EzPSoWNhzPLhJF/HkzVbNf
    F+vbZkQtE4qivPiTx59YV+T3+dat6cJEMID/+M4P6HhwYgK52jXzMlgsFDKhAFw8z69YvbKL
    n4lETABkbQeRNrJaGeRR+ADIiqKcff2NN9fdsO1qpqLEj2wmg937DiAQmjSdJ4fxNybA1CzC
    NJdbDa9VjTVNCOFo7rAyBRJsAIomp6e5//tP/4apqTBUTbUbGUTzvqcMOlIYQJhSGjaMn9W5
    TFoA4WVfJ55/bjEMA03TYLPZ2JKSku2RSPS6AwcOMvF4AizLL5KNWPxngGB8PILCQo+jvr5h
    fUVFxfjY2Bg1gLm1Xsdaz/VhsIPE7/d3JRKJrp6eHgtVdCHAuxDom58pvDBg1HWKc+f6wbIc
    v2xZc+nQ0GB4enp61G63w2azEb+/yDU9HYzt27dvRJIkr2Uu1bzHRWsN8HPaQqarq2uTJEn3
    Hzp0WJycnLJkq8gCmi2lwODgKLxeD1pblyIaTWF0NGgERnUQooMQBlNTEQwMjHIrVqxYKkkS
    e/ToUbS2tqq9vb1n3n777TE9x/LQLLYwmWcLFwRH38drpwD0aDQ6tGTJklB392rP0aPHUVdX
    i9raWgP4cbP3Xdf1XJ9djRpZUDoL7DDrJ5iAlSyy9hhjHxNLZpHMBmtY1gzckFngZwJEWdaR
    yagIBFLgeQ61tR4AGXVqatrae/D9sIFmoJxYbNgCO2juuZGRkVO1tbVvd3d3bT9zZpCoKkF1
    dQVKSlyZw4cPjKbTadOmMXkPs284amtr6rNZuXZqagbl5X7KMMQUYrPWHmYMW2jNGDORSHTU
    5XKFamqW1Pb3h8AwHDweF0pKPIhGQ5lEIilYsoJAroe0de9dNrbQVFA1WARZAEnjPmgAJEEQ
    5OXLl9cXFhZ67HYVHo8TBQUupra21rtkSbV9bGw8qut6xtg7USO4Om0BhEnDTpr6HtLFOnsu
    OiAsLy9HIBAodzgc66+/fhtTWloCXVXnARZrNmxOKTGPKkkpujra0NHa7Hx979uNyKmOypYD
    yao8+oFROX4OB5YHsKmlpaV6/fr1cyEkExwbNVw6XaRuEIvUEb7D85RSrGhfhv/vy79LksmU
    9+DhozPJVHo4mUqrAMRAaMptcYYrLMBHyzOG+e0oLqkBtNzTGoZhPrVz553lO264Yfa6KeYU
    vmYFUiiFTnU47DY8/OD9ON/Ti3/+f/+B2iVVKCsrBdUJKKEgRIfdJuKWG7eTnz35tHj11msw
    MDCA3t7eMHJqo4LlwM5YHFmT1mCVKX/fnVYryDH6Rw5IkvR/v/f9H3zqzNmzq7s6O/nNmzah
    qMgPr8cDj8cNt8sJm80GnuPAcwb4MwEgQ8AYGcBZ8rJl/ZnUUV3PqZjOgkJVzYFCNQcIFUWG
    LMvIZrOIx5OIxuOIRKKYnokgPBPF6HgQ33v8ORw+cYaeOtevaJo2g1z9ZSyPDpHOjyK+x71L
    DPVQs4WKtkjG0Zpde254ZKTkz//qb+72F/pqMpksPzUdDuu6fpIQkqJz/3Sx6K0OoMpX4G1c
    vbITb+zeC4tTZGZnUkZDX2ujYbOQXgcQePnV17TflrJM96oVmJiYwPO7XpejsfiQsY5mAw6L
    iIaolusgWET9Nz8an/+79zKn7xBxzhd4YSxnhWQ4hDqlVAFQyrHsLQ31dZ1dK9qZ4kIvKkv9
    KC50cUdPnm0YC045KaWnTQNqrIUZQsi0JStoVXg0z53Z3nyGIcaFosaXy9lv1tTeddddPkLI
    Tf39A+5gMGTJAuYDFyYvGzb3s64Do6MRlJcXlCxdurR9bGwshlxtjZSnPip/WEBzV1eXTRCE
    jcPDwyVzqqLW8gAym6UxHfF3ywZemFaao0eeP98Hm80m1NXV+F599ZWxmZmZ9Pr1632KIqkA
    mJUru9qrq6sLzp8/f3p6etqqkmgNjsoftGP6wAMPlIui8OmTJ08vOXfu/OwcLabEOreGdJw9
    2w+/34cVK5qQSKQQi6VAiD4LCnVdw8DAGFpaGsSJiRBcLhcIIZGjR49O6bq+wBZirs7MtIXv
    e3DUmh0EQF9//fWz11577d81Nzc9XFpa2jkxMckND48gnc5ClmXIsgpFyQUyrVUQc+Lk1geT
    J1TELqCEmlTQXEaRnf1qPjiOhSDwEAQOoshDFHnYbBxcLgF1dRw8Hp5GIkH5rbeOhAcHB8OW
    zE/c4uzLv+icUUqJ3W5n1q1bJ/T19dHR0dF5dsK6Vn/yk58E77jjjn+qrl6ib9ni3wAwXlHk
    9bNnz/QcPnx4mM7WgcyzhbN2ddmyZbzH4+0cH59yxeNJc+7M9QAAssUOpvN8IzYYDEbb29sj
    3d1dtWVlIcgyRWlpIeU4NXLkyJERRVGs/fbSFgZW5kKg+eexg+/FPlhFm/L+xto31rqYzGBo
    xvhbzeFw0JtvvnlLRUXlLZFIqiiRyIJSDqFQBh5PUeHNN9+27NChAz2HDx8e0DQtY+yhiGEH
    ZyilsxRsSql5tpuK5YzFB6DvlD19r7bwogPCQCAAAGX+wsKqluaWnGQF1Q0RC2tGK8f0tNbQ
    zQM7uo4CjxuN9TXM63vfLkJOrjdloPG4pej1YveAe78ONR/DMBvXrVsnVlZUGLSNXKuAdDqF
    ZDIJSincbhdsomhpAaAvAHugC59frO5wZWcH/uB3fpP53T/8mm9gaNis3SkBwDudDo+u04ps
    NjthCPfIizysDukHMb/EbA7e3t6+4eGHHoLb5YSqanlqlpaMM+YATklxER79zCfxu1/+I/zw
    xz/Do596CLzA59YeYaCpwMjIKAoLC9HW2opvfetbqq7rM5jjsKdNiowRzZlX63SpshhGo+4M
    pXS3qqqxgwcPfSKdTl9/15132LduvXquBtAC7OZ9pTqgWwIwVMv7WZ+lbJt7de53VrVbfd7+
    pUYk1nyPbDaL7zz2NL757cf06ZloxjCA00bwwaRERI3nrcbw511bdgB3AKgC8BghZAbza6p0
    AKrR9y8O4EeJRHIslUr/VnVVdZNNTCrpTCZBKVXz/j9jAT1m24qWJdVV5a3NjSYgpJjfE84a
    8TWNoGqIIHXZbeJtfp+Xf+mV1xO7Xn0jSHMjbIBkCXN1dDOYqyFIWAwhzftM+cZuMcGLWUO+
    WJDsAu1kSB7wI4tEi1nkqLWCIW7CWAD5dWtWr1r76U8+wHqdNmQzSWhyBlRrI+s6W4X//NHT
    lX1D42EAAUqpbKxnEwCnjKiyjDmaLgHAG5kwa814/s+zm/4XcQIuCg2H46AoCvx+f7UkSS1j
    Y+PQNB0cxxsO6PxsYA4gMnmAZu41qZSMZFIWioqKqliWLTUk4k3GQirvLLrsAqP55QZdXV2l
    mqatHxsbI5lMZtYh9/kKUFZWBo/HC4YhmJ4OY3R0DJqm/5wZwoXZRVXVceZMD9xut239+vUV
    Bw8eDHs8Hk9FRaU7Go0RSkl1VdWStqVLl1bt2rXr6UAgYG3zIeVlYS9pcNSct/Lycr6iouKW
    SCS69ciRI8hmJQPIkAVqmfO/ZyBJEk6cOI8tW9aipaUOR4/2INdDzTjzQeDxuCFJWQQCQbS0
    tGjnz5+fmZycZPJsYeYCtvCiKb+bpUKhUCj9wx/+8JVVq1aF16xZ82B5edm1p0+fE8bHg3lZ
    dnZebeBczd9iz2FWYMdUBdX1HMjWdSDX2J4amVRtQb2h9b0ZhgHPc1i61I/iYlY/efJg5ujR
    o9FYLLaYDUxgfo0Y/XnnBAAefPDBpT6f75729vaj//Iv/7Jb0zTkU8oJIZqmafjJT35yqr6+
    /m83b94cLCkpuT+VSnGpVCqVzWYzeX4Mk+dX0/b2di/Lsh2Tk2E215KZgGEYa5DEbFWRtthC
    BYC+evVqT2Nj40cURW8WRVGvrS0NJJPJVDgclI8ePRoaGBiIG6+PXcBPkCilmsVOsYuAvne0
    g+81GGrSsi0t8y5kC639vE2mjLxx48bK6urq23p6hioCgRkIghOi6IIgaIhEKFNe7nGvXt1d
    HwgEAmNjY7JRX289y63lEgzmhIdEA7RrlrK5fDuoXyhofCF7cNFFZYzhtNlsDo/HPetU0rzs
    38IaQtMRnfs9yxB43E4QAjul8CJHb8wXprgcxRfyow0EQIO/sLDrqqs2ghcEaKqM3r5+PPPM
    M9i7dy+mpqehazquv+5a/MYXPg+X0zFbX5g/T3PZwcWziVaQuHnjOnz+Mw+zf/qXf1sUjcUm
    jU1WtLKry7Xz9luwd+8+37Mv7ipIpzMRAClL5NmapdAuVRbW+v4sy1JN01ocDsc99933Mdey
    Zcvm5sSaYTXX07w5ASjV0bm8HXfvvB0/eOwnWLdmJdauWZW7LywwNRXG408+jRWdXejv78fA
    wIBp1BzGgQ3MSUTHLRmu1GIG8GLNixENUgghEqW0H8D3T58+4/3Wf/7X5pVdnYzX6zYyerk+
    hKbQS048RoWiqtBUBaqay/hpltdY99wsGiBGZH72qylgMz/wSDD3NyzD4GzPAL7/+LPUAIPm
    gT5jPMIW0JP4eagQVofSWBMrREH4Q5ZlKtKZrEYp/VGeEVQskTWz+fwIy7LJVatX4cCBA+70
    2JhogFUrxTN/2ACsbG5qFEpLSqxGxtprSbY4jbJBadcppVWEkM+v7upoz2YlOhaYOE4p3Ys5
    UQFT1n4GQAi5FhYmfXK29YUBjhzmtRnXaQVCi2WomcWCOIsBJrMfnPGVMQycHTmBmBLkahyd
    xvMmLSZDKY1iTh3USwjpCM/MsP/+n9+GLEs5FT5dh6oqYKEjlc6yyFFHOUuW12GA+nrjZ/Pa
    ZBMoG4BeMiLQal7WJp/Ot0CN7YMARyal+ktf+pJPluWiZDJhAYL5QGbO95oPbOacTl2nyGQU
    2Gw2r81m86dSqRhy7UmihBDbhyEwal2bhYWFy9LpdOvo6GjOWXA60dHRgdbWFrAsC0mSQQiB
    y+VGKDQJXZcWBX0XUhq9UOZQkmScPHkWV13VbW9oaHC88sorwVWrVmWqqqqLz5zpYe12p627
    e8Wypqams4FAYMKoI8xYstfzausvhS20/B9y55131hNC7jt9+ownGAxZmrLPB38XqrucmYnj
    7NkBLF/ejFAogvHxyZwgGdUhiiKamuowOTkBt9sNANKpU6ckQxxwMVsYtdjC7MW0heaZRwhR
    NE2TDhw40BOJRL67fft234oVHWunpsJEUdTZbJ6Z7bM2kjf7Ec495sRgrMEXc97mK/8zlqzi
    Qpq32WaCUsDns6O4mKdvvbU7ffTo0ZiqqlED4IQvYAN/7gyhObc33nijvaCg4AGA+aLL5T5y
    3XXXTb3wwgtjFhtlAjbZON+1gYGBqZUrVwYcDqc+MzPDlpWV+URR1CRJmvVnLD7r7P8qKyur
    kWW1YXo6OtvywxDSUS1ntrX0SKaUana7nbn66s3bGIb/+NRU0uHziTMHDrz9wvHjx2PZbNaj
    KIpg/H3SsIETAKaMvtkmaNYAEKfTafP7/XwwGFSNkhCaZwfzWRJmDaTyTj6FBQRag6FsSUmJ
    rba21lVZWVnscrmKWJb1EELsuq5zqqoinU7LU1NTyf7+/sjExERW13VaUlKylOf5irKyYlRU
    lFn6prJgWQE8z4Ln3U6fz+ccGxtjABCe57mysrLCurq6Ir+/kLfZ7CQnkKSqmUw2E41G42Nj
    o5GJiclMOp2WDEaOagSeF7OFC/QHLqQBctEzhDzPQ1EUVVYUOZ1Oz8/bLwJg5qll5oEZXdOQ
    yWRNX9VmOCo2zG+Ietn2q7IsNA7AVUsbG8u6OrsgS1k888wz+Kd/+mcoioK1a9dg/bp1EEUB
    tbU14Dg2r05Qfw+gMO91xlzyHIedt92IM+fO89/+7o+8qqaNAsicOXt2yWcevl+44+bt5I09
    ewvS6Uyx4axKlNKMQWmYR8m91BFlTdMcAO7sXrOm/c477wTP89ANIZl5WWVrPaqllhCUgmMZ
    3H7Ljdi9dz+++8OformxHl6PB7pG8fqbezEemsSGTVvwrW99C5qmEcwVVM9ywg36grUtQTYv
    e3qxnU4TgEgGLWOEUvrGW2+/vepnTz7pLvT5MDExiXA4jGg01x4inUrlmsJLEmRVgaao0HRD
    XdRoHD9L1c6be8ZSP8GyuebzPMdBFAVDXEaEw2GDy2mHx+mE1+OC1+PCi6/tw/BoULFEj026
    6KTxdZoQEsnLgP08xpAYhflrV3S01jc31Ak/eer5ezJZ6QSlNGB8frPGjzHmSzeMkwRAr6+r
    RSQyYxsbG/Mhp/ApGQervkg2vMhht7d3reiAzSaAZRnTyFhVHrU8gKIbWe17W5bW3bB141rm
    v3/4swSA08b6MUVjNGMepgwjGKaUJoznOeOMKwSwEsB1Bmh83PgbzQKMsIjB4/HuymRW/hRv
    9EtsBbCSYZgunudrHA6H3+VyuWw20cYyLKdTShVFVhKJRDoeT4RlWe5Frt+n1263lzXU12HH
    9dvQuLQeLpcTVKeIxWI4ffYcnnr2eUxHYj5FUSuMa1vOMEyp3WbzeTwul91mExmGYXRd12VZ
    lpOpdDqTycYUVZ3Qdb0XwHlK6SCAKWMPSHl0SXkRg/iBgUJCCL74xS9ylFKWUixSK8hc4Dlm
    UWeeUgLDjjgtDwcunXT9L3WWmz+uWrXKLorilrGxcf/0dBgOhxObN29GdXU1+vsHMDw8BKfT
    hWg0hlgsBkVRLpgBfLc6woWgkCAWS+D06R6ycuUy78jISHLPnj1Td911l6u4uMRl1Exnp6am
    dCPwnKWUWulrEi5Cndx7cf43bNggFhQU3D4xMdF16tRpaJpuAB5cgCqKBfMDAIOD46iqKkNz
    cw1mZpLIZiVQqqOyshRerwMnTvSgs7MTiiIzmqZZbaFuAAvTFpoPycJkoBfbBhJCspTSdG9v
    71BHR/ublZWVy5uaGh2KosDpdMLhsEMUTWEZ3hCUmVMVXTywgNl5sx4T1Npa16hBzAnIUENY
    hkJVdaiqDkXRoShAdbUX09NB+dSpUyYYDBv2751s4M+9niilePTRR4sAsmVwcFL0+90rGxrq
    7xAE4XuyLFtr3DIWX0YFIHEcJ2uaSkOhCbS2NvvKysqY4eHh/ACt9X6yNputJRpNlkWjCfA8
    b7ZR4/KCj7NUayOjpz/00EOtNpvjC0ND4SqGIchm0yOnTp0aTiQSpv9uloFFAASNeYoIgpB1
    u91E0zRB0zR2y5YtVfX19VttNtuywcHBZx577LFjhhqweV3JRQChYHxVFwNH1mCLCQSXLVvm
    WLVq1RK/379cEIQ1hJBWXdfLZFnxqqri1HU6q7zq9xfJdXX18TVr1gQikcjxoaGhnsLCwu5M
    RuKDwUkEg1NIpzPQdYBlObhcDlRVlaG2toKrqakpn5iYmFm6dGl5U1Ojv6CgsJhhWI+uUwel
    4I3PpjIMyTIMSXR3r41IUnYkGo2cHxkZ6T179tzo1NRUzKSV5umq5LMZ6AdGGTWc6plIJDLZ
    399fsmnjekvmRl8A+qwZinmKkaCIJ1MYGBkznS7GOJw4y+Oiyda/j9lBAChmWXbrpk1XCeXl
    Zejv78O3vvUfaGiox+rVq1BdVYWXX34FO++8A2u7V0NTVYO+d+F6QSt1NFeCmBNKyaeUUkrh
    9XjwyY/fg+MnTjsOHT3uAzCdTqWV8dFRYSwQQHgmMmMYPJexOU2HI20BhRfdCOZFbQiAVq/X
    e/tH7/moUFtTk2sfcSEAOPsc5mVJqa6jtKQI999zF/7sG3+LPfvexk3br8XUdBhPPP08Vqzo
    wsDAAKimotBXIM5Eoo5FnH1r3WrWYgAvujNgyXyZdVtpYx/0jo2Nj37xd36vRVVVoqoq4TgO
    a7u7sXTpUhQU+lFus0EwQBzPc+BZFizHgTVrmDA3b2b9oKZp0FQVipoTj1FkBbIsQ5IlSJIE
    KSshmc7gTE8/Dh07CarnPiPPc1TTNKrpetISDZ00gEwQwITBk49aQPV7phjlemwxVNd1B4CW
    1pZm4RN334ajJ0+3nDrbuxXAC8brNCPDzRlG1wRqGU3TZI/Hg1UrV3Kvvfa6H3P1e6pBy83/
    tw0+X0F1W0tTjg7EcSCE8AbgM6kaVqqquffXetyuB+/deYsznc4gGJoMItdrkFg+T9aYi0kD
    OAPAJkLICgAMpXSIYUhDaXHxdXU1VSWDw6Ox4MRUAaX0SQAxwwCYjYNTiwDCeUBxERVok3pT
    AmALy7J3FhcXr21oaCjp6Ojg2lpbSWVlBQoKCuBw2MGxLHRdQzabwUx4Br29fUv2v/VW54FD
    h7cHgyHltltuKvyzr30FZSUlAIGFfpxjKWzZtAG//ft/5Nn39sHVBV4P29y01LWmazmzrLUZ
    ZaVFcDocYAmBqqnIpDOIxRMITkyib2CInuvt3zY0MpacnJrplxXlgEFxH0KujiltoSpJecAw
    /75cUjuQzWZTNps97nA4CxZmrhYHfvMzFYyR4WAgihzS6QyVZZk3QKBoCV5d1oFRY18SAFi5
    cuUSTdO2DA0NMZlMBl5vAbJZCXv37kNZWSmWLVuG4uJi7Nv3FkKhCaN2azGAx8zS/RYDfgsB
    0txcBwITKC8v4VauXFk0PT0tybKsi6JoNDzX5WQyAQNsOwE4jZY0Zs0X+w6MgouVHcSaNWsa
    NU3fefr0GcfMzMxsHar12ueLylizWnO/kyQF584NYcOGFaiuLkV/fwCCwGLp0ipMTATh8Xgg
    ig5wnMDX1tY6jx8/zlgCWCbzwqrfIOfbwou0z6glKJqilPLT0+HzDQ1LA9u3b61nGJYwDCG6
    rmN0dAyRSBTJZMZgzuiG0Ayd7U1ojYnNtY+Yo4FaawYZhgXHMWAYblakjeNY+HwO+P0uSwBb
    p7Is0337RhKZTMbUIpgy7J9pA8PvZAPf69w999xzsNvtS1RVKx8aCiKVksWmppLrmpqaDpw6
    dWrKeC/FAPCmuiUFIDMMk2EYhk5OTqG5uclTXV3tGh4eHsUFSjjWrFltZ1m2MxZLCJmMBEEQ
    QCkYQoh5Dpn2zJq1ww033ODy+QoejETSawOBGFpby+RAYOBcNBrNGueWab9Mn2GaYZj4xo0b
    yzs6OtY6HI4qXdenVFVJ2myOazMZtTOVUsUlS2rat23b9i+vvvrqaV3XTTCkWTLZ5hCNm5y5
    0DYzqaHt7e3iVVdd1VZQUHAbpfSGZDLZHAgEnBMTk0wkEkU6nUauf2XO5LMsC1G0weVylZaV
    lTRWVpZvWL16TVjXde+bb76N/v5Rw3+boxRPTkYwOjqJWCzFdHQ01ldVVZdwnOCKxzPC+fMB
    kkhkIEkqdJ3OZrYFgYfNJsDlssPjcazz+cqU0tKKiY6O5adGRob3HT58+EQgEJwy+kemDXuY
    tQRqrFRgmp8p5C7RwT+RSCROv/7Gm+133HZLjv5I9dlIC8ViWUM6V2to9JTr6R/EydPndQBZ
    41z80EiQm5EHA9h0lZWVdV57zTXgOQ7FxcX48pd/H2UlJfirv/5r/P3f/yPcHjfu+ehdlrqv
    vLpB6AuAtAmIspKEVCoFr8dt5vnngUiqA01LG/CJe+9ievoHfPF4IqNqmh6JzGBJeTE8Lpcn
    EouXGZkKu8UYmkYweymMoGWhEkqpDcDNK1eubL5h+w05KfbZnoPGGso1vQMhAEMYaJTOq6fL
    wZ1c0/aN69Zg5YoO/OTJZ7F2dSd2730bwakwNmy+Bj/9yY/x6Yfvxxu79zPPvfSKwxptX6TN
    iXaJsoL5xlC3UHZYAL26rn8rmUxuANAMoNztdvs+99nPsDfffFOOAkpMSiedXRfzAgvzagw1
    o1eh2RfUFJgxaag6NE3N3QNQvPrmPhw9eYZm5CwFoMmKYtbDTVuoHxMmICSEmMDnF24rYBxm
    HAB2ZHRcB2GYtuZG26mzvTcYznEAwKhRY2xGtpPm3Om6rhBCsLKrCwVeb2k0FrMbkUktjx/E
    UEpZAE1VlRVFNUuqQXUdgsCDYRhR0zS7JWDAWrNtlNISAI9s2djdtOWqbvzJX/2Trqhqv2GU
    bJijYccJIaa6mALg7uqqqnvWda8uFAQOBw4ekcORGe6h++5it21ei1de213wL//9o09EYgkF
    wC5KadKI0KYuEPjT3wUMcgDWEEI+V1tbe8P1119feOONN5Kuri74/X6zh6ARhNGN9je5gB6h
    FKqq4IGPf4w88dTT3j//y/+Dm3Zcj7LS0tk+lfmMhYb6Gqxds5KZnJr2/eZnH8bWzRtQ5C+E
    wHOzQlCwgEiAQtM0SJJE4vEE29M/6N379pGVr+99u/1s78A12az0CoA3KaUhw1m3Kv9a+xjq
    HxAoJGNjY5NtbW0jZWWlS4aGxhaAvvn1hMyi4AUghkMg0OHhadmgSi3GZ7vcs4MAwBUWFm6O
    xeItQ0PDIIRBKpXC/v37jesUUVlZiVBoApOTUxahj3evGTSBj9vtRElJMcbHQzBphPlAUtN0
    9PYOY8OGlfZly5Z5E4mEXFJSRsfGpgkhvH/jxqs6I5FnpuPxeMLYszYLMyl9Kam5lFI0NzcL
    LpfrhlAotKynp8dwNNl5mdA54DefBLAYUJycjCAYnEZDQwVCoQj8fg9cLgEDAxNoa1uGs2eH
    4fd7mPr6esfZs2dZWZbf1RZeAntoDYoyALgzZ86cdblc/1ZaWrrO5XK1OByOUl3XfW+/fZAZ
    HBya7bk4t17YC9QazqeYWqml5mtyP3Oz3xPCoKamGF6vjabTKRqLxbSpqSkpEAgk+vv7TdaH
    GRANAQgRQqbybKCEX7C1TjAYBCFEZBhCPR4botEkgLKGjRs3fLS6uvrs+fPnRwcHB6copQky
    t0jMwH6GZVmaTCYRjUZtS5YsKeN5/qQh7EItJQQEAKmpqfVSimXT0zFoWs6P13WdYRjGZOzJ
    FpbCbLCxvb19i6rSu/v6JgWOYyCKJNrf3zdiyViaqqRRww7OXHfddUs6Ojp+MxpNdE1MjAtu
    t1stLy/SIpGUcPZsiBAioLW1fGVHR8dnR0dH/+n8+fMjRhY/uRhhEfMFYPLr5wkAcvfdd/vr
    6+vvAvDJYDDYfv78eWF4eBgzMxEoyhxAm98qaO4M7+0dgNdbwG/evLZMEASMjYWMADa7gP2h
    aToCgSm0tdWLqZQqnj7di2g0BVXVjf3J5v2v3HM5ESOO2O2iUFTkri4r81Q1Nratr65ecvj4
    8eMvHjhw4LSiKGZNb9KSwJDfiYnFXQIDQACkKaUvvfba69e+/MqrRbffcpNxA/S8NhP6AoBI
    aU5xMxqN4bEnnkdgYkoCkKJ0XjsEZbED6XKkyFBKPQBuXb9uXcmqlSuhaSoKPB5sWL8Omqri
    N7/weazs7ERdXS1aW5qhaeq82sD8usB8VVFCgAOHjuDpZ1/AH/zOb6LA67HQRuccMgLghm1X
    4+XX3xSeem5XgaIomSef2+W8ffsWpq2x1r3v8MmVBu0ubok+W+lIF9UIWiOixmZtcLlct952
    221CdXVVTrzEGoCygN5EIoHx8XGUlhTB43JBhxU45+bM5XLiI3fegq/+yV/i8Sefw+63DmLF
    ik4MDg6gxO/FLTdsA0MI2fXqGw5FVc16KSuo/6CdLi2P7jcF4A0APQCqAXSxLPsRl9vls9vt
    0FR1NvtuXTe6rlucfH2eQIyZgZ3dk2RORj9XT8gBXK5dhSiKILmbYdbYpIzIZ9gwhmZU1DSE
    YczJKv/cTYwJISb7IA3g+/veOtjyh//7r9amMxm2ualxmU0Ul46MjsYi0dgeAK9RSgNG7SW1
    UHOkbCaD9vZlWLJkiSd68mSJkbkznWzOQgl1Aeioq60Riov80HUdNtEGlmEETdNsmGs/wVjA
    oADglrolVTsevPdOEo3F0dM3mAEwbKGKSsiJqEQM+tA0pbSrvLx85x995cv+W3ZcB6rJePvt
    t8U/+Yu/wXO7XtMPHz0OXVMZhiFuAG0GTVM3IsCLUQUvCBIMI8gDuNZut39t+/btqx599FFm
    /fr1cLlcc+vCDA5YWBxzayW3RnwFXtTX1sLr9WJJdZXFKVzYDochBE6nA40Ndbjt5hvgcTln
    W5sgb43S2RY8OgSeQ1FhAYoKu7Cmcxlu33GN8Oyu15t+/NSLlYMj480AnqKU9hNC7Jir8WUs
    xvADE8Tas2fPTENDw776+rq1588P8LFY4l1aJSykkTIMg7IyLwBJHRgYMHtYLSYocLlmBk22
    Bz7xiU9UsCy7s7+/3xmJzMyCNVM05syZcxgYGIau67Ng7uetGSwq8qOzsx0zMzGoavKC8x2P
    JzE0NM7U1tZ5T5w4PuPzFWYaGiptgiAwtbW1jcuXLx/Ys2fPNADRkgnh8vfWxQwymO+7bdu2
    Gl3Xbzt37rw9EonOu+b8DKEgCHC7PchmJciyumj2VFV19PWN4aqritDQUInCQhdCoQA8Hg8o
    5TE0NAFJUkhTU5nd7/cLwWDQPN+Ixb+hlwIM5jGtzPpwEELI9PT0xHPPPfeq3W4/6/F4apqb
    m1d3dnbeJcuyS1VVsKwRLJ5VYZ1ToLSKOVmppAvrMpHXk3DO79B1IBqN0l27XkqHQqF0NptN
    GjRRMygasGQGp4znTbqoNWj1c0/eI488gq1bt55cvnz5d5Ytq/1cMikXa5riBLhbGhqarmto
    aBg8cODA8wcPHjxl3DdqnI2aqioZhmH0HDgJMcuXL6ssKiqyBYPBtMUOztrCoqKiWkVRqyKR
    +Oxc6DplWJYxG9jDeC1j0i9vvPHGKlG0PTI8HK6emIijra0CmUwqMDAwGMNcSYNkfKYwgKmK
    igqpvb39lkAgtGbv3kNMJqPD5SrgVqxo4erry2lLS4VOKUtEkSOCwNQWFxfXnz9/Poy5WvvF
    7CCzGAPNvMEf//jHS5YsWfJbiUTikaNHjxaePHkSsVhstiXJXICAWZTeb9Z4JxJJpFIZyLIK
    WVYWqe2d26OaljvfhodDCIUiRhZ6MeGjHInHPCdVlSKZlJBKKQgGE6SoyFVQV1e4de3a9Y2F
    hb5nX3nl1d3pdHraqCuPGfcknedDzguOXoo+hNQQVTgQCAaf/79//4/3lJeW8KtXdc6nNeZl
    vEzwQgCk0hl8/yfP4JmXXtcopTEL4rVK2kq4zPpU5YNBI8qy2u/3X/eRu3YSr9eTy2CBQldz
    193S3IzWlmZD6l+dF1m3zs2ClhPG84QSjI6O4eDho8hms4DHNac0qluyhJTC7yvAvXfdRt4+
    dNQ9MTk9ebanf2Z0POhiGcKB0mIDWASNLKHdAgzN/mz6pYi0U0p5ANe1tra2bt9+fa6v12zr
    EpPbn1srsXgcf/23f4/nXngR93xkJ37jc58CwzIw6yvpbCsKipUrOrCyazn++Vvfga/Qj83X
    3oCf/vgxPHTvHSgtLkTX8mWoKC8Vh0fHncY1WwGCVWHqg+j3RfM2NGPJZikACliGkTmOmz30
    JEmCIktQFAWapoKAwumwg2UYqymfo9zmAe75mWY69zQsN8EAOJhfMxgyjOG48bNVNewXdtSN
    dUcBnJBk+ccnTp9r+/QjDxc8cN89LM/C8dKulx1/+4/f3DEenJAA7DZooJrFmZASySTKSkuw
    cmWX/cTJk/XI1fbpAARKqWioZwoAbALPt7W1NOWyWKBwOOxgWVaEotgNY85hvgJnkygKD99+
    0zbfyhXL8O0f/gyT0zNTliCLWfw+C54N0ZS69mVthTft2A6P2w4lS7B21Qp0r1qBf/vvH0wh
    1y8UhiNxxkIZ5CxiLMgLGJALPM8AWG2327/2wAMPrPnyl7+M6urq3ATpFoXZBfXJ85kHOYdI
    N8QUKBRZsWSg5zL45jmhKAqmpsJwuRxgGZLLPpuvN9cdvQCgNDKHDCFoqK3CZx+4G8tbG51/
    9+/fvfrw8dMeSvETQ2yJJ4SwmN/QmF5q6qjJDolEInJPT8+u9vb27Z2d7SveeusoNI2+Q//B
    hXL4Pp8T1dVe2tNzIhUKhcysgrVe5LLsxZsviAKAraio2B6LxbrPnj0HVdUXAD5KgUwmu+D5
    d+tHaAV8LJtrQG7NwM5vyzAHpkZHQ6iuLhe9Xq9t166Xptra2hxFRUWszWbTstmszQgKpSml
    DiPgIGKuYfdFyzxbe1kyDMN5vd6rw+Hw8t7eXug6NWqZFyqLCgKPVauWo6VlKXp7h3H48Ckj
    u5HfaoEgEkkgGJxGa2s1Mpkk+vsn0NbWgZGRMDIZBTMzSTAML1ZXV7uCwSBrsYNsnk3ULmH2
    fYENpJQymUyGyWQyakVFhZ9SqpqqqYQwRg0hPysqk3Osc+fWwq465jqZv36ttYZmfab5vSwr
    NBwOS8lkMplnA4OGDQwsYgPT+CXaeZmJl9deey01PT39+I4dN6zlOHX7a6+9RcLhpOD1Fgpd
    Xa3LV61aJYyOjmaCwSA1Sil0Smkmnc5IhDAax3EIBEJYsaKjvKampjQYDMYwp545S0u32cTG
    dDpTkkikckwtnULXdYbjODNDSK1MGbvdbmtqarwtmZS2DAxMgWEY+HwObWysdzSVSpn20qxv
    NJuyTy9fvtzP83xjb+8gEw7HYLd7kM2qGB0No7a2TMtkps9Fo7EEy7J6PJ4InD17Nm6w2SQs
    XkO9IDBqBhgopeTGG290Llmy5NORSORzr732mnthT1RmwR7LB3jzzxRYAgqL1/ZSmgumMwwD
    RdEsr2cWAY8LAz/m61WVYmIihXhcYerrC5c0Nrbdo+vU+dJLL72uKIoIgLMEi602cN6a4y6R
    PdCMm/z9g4cOV3z5q//f1V/67d/kNm1YB5sg5D4Rnf85GZBc066JSXzvx0/iv773Uz2RTMUt
    WQVrDzir6pd+ORlAM/pggMECAHdv3rSpevPmTRYHaS5zl4s4LayhfE/iMQYl0uV0oKy02BCj
    yXu9BUzqlKJ75Qps2biO+9HPnnEBiCZTaVOwxCxyduYystQBwG5EGwQL3eCigkLjfUt5nt+x
    bdu19vr6euiaNgdSZq8HYAjB0WPH8dLLr8Ll9uLtg4fwqYc+DofDPr+W0Mg22O027Ni2BU89
    +yK2r12Hgf5++L0ubFq/BrqmYUlVOVqalnLDo+NeCxg263TMr5dUzS8vQppf12gVN4kThtFZ
    joOUzeJHjz2Gg4cOIR6LI5VOQ5FlOJ0OPPqZR7Cqa8V8E4u8ljb0nexxLnJG59K1ZjuThLFX
    pyxU0UmjJUTcsr4W7Nn3uo4shzkAeJZ3dNg/++lHUFtdASWbxj07b8WRY8dd33vsiXYA/TDE
    ZAx1Sh2AlEwmYBNFbFi/Dj/+yU9rUqlUiWG8qRkIMWoEq+12e0VzY4NBncztM5ZlRcMIZq2A
    0NgvOzvamlfsvOUGqIqK4yfPUkmWJ417Zu4hxTi/TJGiFIC0LEm6lM2AOkVQXYOiyMhmsjCc
    jF7MyXKHLU4p+QXWkR/Ap2+44YZVX/7yl7FkyZK8bLq1DhlzwA50zsm27L/Kygo4HXacOXsO
    3au7FryXqUYbjcbQNzCIDWtXQRAEaIYKrnn2mUJGOUya10rHhIwGOGRZBpvWrYTDLrJ/8tff
    7Dx+picN4CkTIJuNgz/ITFru+Kf01Vdf7ff7/d9raWkskySl9OTJHiiK/g6iKHO1hT6fE8uW
    lSMSCUmHDx+OqKpq0oHMXlVWuf/LJku4SC04HnnkkVqWZe/t6enxTE1NWUAf8641f3POERYo
    seY7Z9FoHD09A7NR+rlau4XZxWxWwuhoiNTULHHG43GtoqLCoaqqHAiMRycnJ2VCiMMoXbBT
    Su0GKLRZbOFFYyiZvSzvu+8+P8MwO4aHh90zMzMXaECfm2+/34/6+hpkMmmUl5eA53lIkoLF
    6gw1jWJkZBJLl1ZhfHwMbrcHhIgIBqNgGA7ptIJkUmYrKio8PM+LhhqktW7VPIPUi7323skG
    EkJYsz7V7XYnAEJ1nUIQBLS3L0NZWSlEUQTP5wRmslkZR46cQjgcnQf8TCC4ECDmx9Pmx9sI
    ASWEyO9gAycMNohVkE75RW3gAsofx7KUksJTp86hv38YPO+AosRw7two2bChrbaysrIpGAxG
    DVsoA6DpdFphGKKIog2Tk1OIxeKOmpqapYcOHRxVVc0Ea3YANrvd7uI4rikSiTvS6exsRkzX
    KcNxvBkkUS1+Ebt+/bpmUbTf29Mz6YpE0igtLQDPk/TQ0HDY2D9mL9+sQfWMA0jyPG8jhMiC
    IIDjeDAMB5blwfM8NE1Vjx49On7y5Klxw4+YMd7nvXQayM/om3WDGyVJemjfvn3uc+fO5d2D
    xc8ha/DKOlRVRzQax9KltXA6HUgmM4vsUfNcz7H4zNfkix1ZP+tiNdFzmUsCSdLQ1xcBpcTX
    1NSyY3JyInXw4KHDFkYSxcKWTZeGMmrdsIQQhVI6pOv6v+9/+6DyW7/zB1fdevMO543XX0ua
    ltbBac8FFqimQpYlTExM4eCRY3j86Rfp/oNHaSabNbMOSYvzlECuGXJ+35vLyhgaThUDYHNZ
    WdmN9913L1tU5J8FNrl6nDmggkUyf1ikLQddhIpFAWxctwZNDfXwul3zxBzywaVOKdwuB267
    8Tq8unufY2p6hgJwsiyrE0J4VVUrMcdxt1kyhaYDrF9sCpahIrmyvLx85fbt28FzXE5kx3Au
    GcIA7JxDOjMzA4fDgabmZkxPBo16QgLKEFCdgU6opdxSx3R4BpVVVWhtbcUPvv89PHD3LSgq
    9EJTFbgcdnQtX0ZeeWOPR1U1p3HY2SilNoM2ZFXGumQ1SYs0WzXlpK0RoCQhhHIMC1XTMDI6
    ikgkCn+hD9VVleAFHg6bDYW+glkHfy7bM5ugWbiTrTRAYK5F+hySlK2RPqNoftp4WNsoLOjZ
    +PPOXR51yOZyOVmX0wFNU6BpCgSeg9ftghHZLzT+d8rS4DUdjyeorChk7ZrVaKivKzpx8lSd
    8TqzDYLDWPONxcV+X2Vl+Wzmy+Nxg+NYa49Ka+a4w+t23fLR228U6muqMDQ8ip7+IdWYB8FC
    kVGNInhrW5f0sRMntG//z3fx0TtvBUt0vPzKa3htz37JiCqbNYu85aB/zyAnr/3N6urq6us/
    9alPMbNgcG5FLNIKiIKAQFUVBANBDA4NIhaPQ9c0cCyLrJQFy7J4c89e7LzjFrgcDuh5oJBh
    CM6e70Vf/yCqK8vxH9/+PlLpNCRJhq5rEAQBBR43KitK0dbUgLJi/yybZJ4ytQlPjee62pvx
    6Y/v5P74//xr1/RMdBRzdc5WFVb1g8igGYU4WjabzT7zzDOv3nTTTUUrVrQ+4PV6S8+eHUA0
    moSuzxl4k8VESK5msKLCj6VLi2k0OqG89tqrkUgkEsWceq/Z3Dr1y1LPLkWmtLOz01ZSUrJz
    enp6zenTZ5HrlcZewOFhLpgJfCfAaAbD0+kMRkYCRgbyQmI0c+8VDE6jpqaCr6qqsttsNt7r
    LXC63R53TU2d7/TpU/xbb70VNyLuNgtrhr9YvkdeYJkpLS1tT6fTa/v6+qGqGhiGm+cY5nrm
    5frl2e0idF3F1NQUCgtL5mU0qFFrb51Dm01EMpnExMQEli1rx/h4FNmsZorrYGYmibKyIrfL
    5XJFIhHbIraQvVTB0UVsoJ5nA4nL5UwRAqrrFAzDwOv1QBRFZLMS4vEkdJ1CUVTIsryIwmg+
    KMS8TKD1NfmBnzwbOAMgbAiomTYwbrGB72d7Dup0uhiGIbZc1wgGLJsDUopCoeuUZRjGZ9hC
    s6xDSafTFEDGZrMZQZEA29bWuLSoqPhkKBQKG2eJA4C9sLCQEMI0x+NJksus5gChqupEEARz
    P0gAWEop63Q6nS0trbfF41LLyMiMwXJwQJYz0UAgIFleb94/swG7nMlkZIYhqba2ZiQSMuJx
    GV6vF42N5UilYtFAIGh+LhMIksWyXlZ3EnMqqPPGzp07PaIo7jx16tSSC4HB+Rk7xpKJF+Hz
    FcDlcoHLCc1B1yl4nofb7UJxsR/J5DjM+tW5rwQ8z6O83A+e51FdXYrq6lzfWpZlQGlOxVaS
    VMTjWcTjGWgaLlBakPucDGMGd+Jwu4uK2ts7NvX09IZjsRgxGFKqxQYqFmB48QDhIqqaZgGw
    jJxE/o9HRsfi//zNb2358U+fKG6oryU1VZXwet1QFAUTE5MYGBrF0OgY0umMSUPLYn6TdAW5
    XmzKIgaQ4ANUk8s7uMzsYBXLsg/ddOOOii2bN+e6neY1j18A7uYpZ77T8/q8KP7UdBhnzp1H
    VUUpWFG4YPN68/tVncvQvXIF++xLr3EApKrKCsemDWuLDxw6UtDbP+CgNNcc1Dj4zYJ6wZj3
    932uzfdhGAZGRHbb6lWrCtvb22cptgCFqqo4d+4s+vr7IQoCSoqKMDE5BdWg2kqyjMmpaTBG
    Daosyygt9qO0pAgEwOTUNJ589iWsXbsOfX19KHQ7sHndKgNEAyzLYnlbE5wOhz0WT3gsgNja
    7oT7oGijeaqjJsCYrZMjhFCGZeB0OPA7X/ytXOaFQV5mOVcfNk8a0yLQM0/FNe8Vc6+cd89N
    6ofZrDiCuSa8UczJQc8rbP4l1g01gk3nz5w9O/Pqq6+V7LhuK6ArOH7iFN4+dMxUGxORk453
    GT/rAFKxWEyVslm+tnYJNmxYJ5w6faZZ1/VBY12bUv4uAEsry8vcxf5CY78ChV4veJ7nLO/J
    GYbQC+Dm9d0rq7dvvQosw6BvaAQTU9NmfaVouVemUp9sSHOv5Tj2/mQy6fibv/8n5dnnXqAc
    xzJ9A4PSdDgybESaYYmK55+J2gUYGvoiTqYIYFNXV1fxmjVr5q0r6xkzjzYMYHBoCI/9+Cf0
    uedfwODgIESbjZSXl+fWkK4jEo1iajqMvv4BdC1vn4WX5vvIsoKDh49C1XUMjk1gbGLGiNgz
    GB0ZxclTp6mu69TtcqJreRv5zAP3kKvWrcoFdvLX5mwNda7ubNPaLlzV3el+4oXXVxiZXt1o
    fJ9PrbxkgMmyTzUASiQSiT/++OPPX3XVVXJjY+PtZWXFjRMTM+LExAzSadlw6jnYbDb4fG6U
    lRVAECjt7T2lHDx4MBmNRk1GjKmmalU8vqwoo4uJNmzZsmW1pun3nzhxyjk9HTaEOt5NJGax
    vozvlkUkKC0tRlNTAw4ePI50OvMOtYe55zOZLCYmZpjCQid35syZ1LJl7ez58wOcz+cr6upa
    uXpsbGxocHAwnBcgXdA/7v30O0pLSxEKhbBixQpRFMWtQ0NDZaFQaN51MgwDv98Pv99v9KmU
    4HK5LNRZBg6HHYIgwm63GdkxBel0rpWizSaivr4cgcAY3G43GMaJUCgn5EMpA13XEI1mUFtb
    aCssLPREIhF7nj00gXH2A7KBs62YzH3tcDgzOYBGIcsy9u1723LvWUttFos5Nh1ZQB9dSBNc
    mCG0gElq+KRmhjBmZATNh1VA5n0FgwD0QCAQ0TTtXH19XXsoFCWyDNhsNlRXF0PX5XQwGFQM
    e+UxgWkqlaIAUjabDboOjI6Oo62tqayhoaE2FApljHiWA4DT4/G4AVKfE63BbHZZUVSIomjn
    OM6uqqoZHGVXrVrV7nC4rjt1KsilUjJ4nkdO9CYci0aj1PAjZUvSSAIgd3V1uTs7Oz9OCLu6
    vLyEbt26Vk6lMqwoClRRspE9e/b3hMNhKQ9bWB/aBQDhYpNMq6qqaiRJWtvX10ey2eyCMyz/
    nAAAnuewZMkStLU109LSYjgcDiSTSWK2xsmp0XKorq7A6GgIlOafZ4DL5UBhoQeynEJRkWAo
    4FJwHEF5eSk0TaOxWJwKQjGZns6Q3t5JZLPqbNx5IU01t6ZlmWJ8PEWamvx1dXV1rceOHTPV
    02Ujgy1Z1uDFyRAu1s/DrG0yxFRaAXQB6GRZdqkgCBxAMDU9g3QmC4Hnoek6spkMUpkMREGg
    OQl9jTfk5RXDoeRNGqZxDXzeg+bTGd+HDfdzz8HsCZFzvHa2trRsfeiBTxCv1zOPHnXBpvJ0
    fs3ffFXR+SIxc2IxFHv3H8APfvIzrF3dBbHYv2g/QliyhF63G9dfs4m8tvstPp3JJCKRqLDl
    qnV8e2sj/7W/+Nsl2azkQ645qIMQYmZNzEPtosi5E0JQVVWFkZGRKrvdvnbz1VeTAq93VtiC
    IQTDwyP4vT/4CgLBEDweN1iGQSKRQNuydqxbtw7/+A8H8Nu//1Xouo5sVkImk0FdTTX+5Ctf
    QmV5GV57Yy+mIjHcsGwZ/uWf/xlb1nXC53VD1xQADCgBKstKUF5awl8AEAofJCBc5LCz9kKU
    CCGUYRiAADYxV7Km69qsaigFASgx7JrRTFe3tiKcBw+RjwSpsd5AFwR+TOATx0J6dyaPuvdL
    XzfNIbSDwWDoqa/92V989I0333TZbSLZ99YBnDh1NmZQdTjMKebaDaORiEaimiRl+ZJiP67f
    di1+9sRTVRMTk9XI1elJBigsBFBbUV7GeT0e6Hqufs3jccHhsLPGa6wtcDqK/YXr7rnjRlLk
    90HTNPQPjSISSyQsdFGTKqsAUIy6zzZR4P+wpqp89eT0TDYaT75x6OjxJHLN4E3VVmuLiqjh
    cJj0I7OBff4evFAPQgfP860dHR2sx+MxvZILKD7nHK8zp0/jj/74T+hLu17WZVnWRFHkv/Sl
    T+GunXdClbNQFQm9PT340z//Kxw7dgIr2tvm1RsSAMlkCoeOHkddXT2+/vWvo7W1FSwBGKri
    yJFD+ML/+h2cOnNOiyeS0ht7D3ATU9Pin/7B/yLr13TOr1s0qKrm59QphdvpwIbVK/DSG29V
    pDPZahMwGcpzVvBELoUtsNaomOvV4XCQwsJCYWpqKiyK4smGhoaC9vallS0tGpEk2ahpIhBF
    HoLAQ1Ekeu7cOTo6OqoLgkDtdjskSdJ1XZ8VUrMIJpmeKvtB2cB3oGmR+++/v8Tlcn2qv3+g
    9dy58wDwDjTOxWi0uAB9lCxaZ+jxuOH3FxiRe+YCjern/5+JiTAqK1uETCaTjsdjstPpsofD
    UZSXFwmEELMFk8tQ9nVZqLqqxfe4oE/w896DUCgEAFi1alUJpfSqkZFRNpVKz7sWt9uFLVuu
    gs/nRTabaxUsCALC4TBGRkZQWVmFLVu6ZzP0HMcinZZx4MBZxONplJf74XTy6OkJYuXKlUil
    NMgyBcPwMGvwMhkVmkb44uIiT39//6wttGQJP2hbqFtBltPplHIso9w5oaoaCKGGKAi9QN3g
    O1FEYfmbxWsICZkN8mUtNnA2i485AZn30waaa0ofGBiIDw0NPVZbW9d87bUb2sPhBON2u1Be
    7tfPnDkxFgwGMwZ4N22hLZVK6QCSdrsdDMMgHI5gcjIsNjQ0NB05cmQ8k8lQA0Q6/f7CYl3X
    /fF4CtZejdmsDJdLsAuCYFNVlQPAOp1OT0tLy3WRSKYsV45IIAgcnE5BGxkJxdWcWJ95/bOg
    rrS0lN+yZctdLGv7rCTRAl1PDw4O9r1Fqe6Lx+N8T09v0rgOM9gbBRAlhJiqmimTtZW31xQs
    UkMIgIiiWJVKpaqmpqYWObYYCwDLnUEcx6Gjox2dne10cnJCjcdjRNNU7pVXXkU6nYUgOCAI
    DrS3t6CkpBAOhwOpVGZBwMrnc8PhEHHw4DEcO3YMkqSAZQWIogvr1q1BVVUJfe21VyWXy6Wv
    XbtObGurYE+dChJFoe94juWEslTIMmOvqqpsOHnyZEDTNLNdWsoSSGStwVHu/TR+eZNsAsFy
    AFsB3FhQ4O1asmRJRUN9nb2tpYWpr68l5aWl8BV4jd5WDDRNRTaTQTweRzg8Q0ITExgaHuF6
    +gbY4dEx23ggVBCJxvwAHJTSjLEQnIaRNylCbF6k9JJbQcMRMFdSt8ftfvgTH7/P1dXVaQgo
    6PNBIBYXiVlUPGbR188Jz4wHg5Alw99crIl9fu9HAnR3daChbgl38sx5Jp5IyCdPneYbaqvB
    5iSNzMxK2mhSn7ZERef1NXk/53BsbIwAaCkrK2tYt27tbO2W+W9UTUE8HkdJSQm6u7vBcRwK
    Cgqwfft2VFVVwW6348SJE+A4DgzLYvebb+J8bx8kScLU9DSeeO5FdHevRV9vDwKBcZztcWNi
    agoVZSWgVIcOigKvGxXlJey53n475suNi5ao6CVRmXsPEUINuQa0lFKqEOScJ1VR0NPTi8nJ
    CcTjcSQSCaTTKcgGRQ+gKC8rxaYN6+B02Bfw4ReWEs7/jsx/pXmwZ/MyGGlLBuP96lNlbYAb
    BvCfg0PDE8Mjo+sJIe1Op7NEEEUhk8nYMde31IzsZwGkorGYlk6noVMda1atxJrVq5zPPPt8
    o0HvUSilLgAsyzLV1VWVhOe52To2p8MBv8/HDo+MOS30zQIAm7ZctbZw3ZouABSZbBbDI2NQ
    VdUEIpz5uQkhZi1HFcMwX1zT2bZOFHgMjgTOA3jdeH2hYbxZw6iZVCTzYSq2mo2h88eCbCzL
    stA0zclxXGF5ebnRS4paMr6zpNFZVeKZmRn8zf/9e/rc8y+ouq5nALAtzc38fffeg6alddAU
    CQMDfXj9zd0YDwZx5tx5gw6IuZ6gBIgnEggEQ/R87wD+6q/+Cl/+g98nK1d0QJVVrFjWijtu
    2UHO9/ZDUZQsAPVcz4D7v3/wuKO5oZYU+jxG0CKvlYxlXdYtqYDP63GkM9li5NT9rEEJ03G9
    aIAw7wyYlV2/+eabiysqKlY7HI7NLMt2aZpeJ0myPxZLiIHAJNLprCFrnhMj4HkeNpsIu92O
    oqIKsmVLpahpCpvJpIVIJMKPj4+rw8PDkUgkwhuKtjajxtuOudYp89S3L9X5tAgIIkuXLhWr
    q6vvisXiNx85cpRJpVLzVBx/vkbz7wwic73BGPh8ngtE+hfvVwgQJJNppFJZUl5eLszMzKil
    pVU0mZQJIYxYX19fMzY2FpJlOQPAY9jDDBaWrLxvk3zNNdfg1VdfJQUFBUszmUzL6OjoLINm
    TlyChSiKSKfTGBkZga7rkGUJ/f0DiEQiUBQFFRUVoJRC01RUVVWjoMAPjuMgiiLq6ioQDI7D
    5XKjoKAQDCPD6bQhmczOzrGqUigKWK/Xa8+zgdZaQuYDtIXUArSIzWabPfNYlkVRURGcTidE
    0QZRFMDzgiEuwxj3PYNAYGpWeGc++JufObwwmAQ1aHnW/oyZRbL472uvRpOBQClVnnnmmSOb
    N2/+xpIlS26sqSnucLlcLaqq8h6P1+F0OvlYLMZZQKE9Ho9D1/WU3W6bFTfp7x/Gxo2rq+vr
    66tOnz5tZhXdHo+nVpIUVyYjw9rfMp2W4fd7bYIg2NLpNA+A7+joaHU6PZ0nTgSIJKkghIHD
    IYJloU5MTGSM9WImFRRKqSwIAm655ZYdguD43MyMWuDxMJn+/qsw7MoAAIAASURBVP43nn76
    6eMACg32jSnIljbAoNUOmmUpi/UalC9AaWYYhimSZVlIp9N5VFFm0Zq9mpol6OxspydPnsic
    O3dOueGGGxw9PX0YGQmCZUWIIsHSpeUoLy+GKApwuZxIpbLI73XpdNrBcSxdtmwZNE0nx4+f
    gqaxoJRFf38Q1dWVpLi4WHv77beTsixnrrlmW0FFRQE3MhJ9V4EbVaXIZHTG7fYUsixbqGla
    DEDcaMMVxyL1ltzFsQWEGI7UVoZhHq6trb3qmq1bfFu2XE2Wty9DWWkJHHY7eJ4FDDEBs8fZ
    XB+0HJjJ9Z7KIJ1Ok6npafb8+T7HnrcOLnnlzX1lQyPjDZqmvU4p3Ws4wqbTl8RcTY5ZN0Iv
    9gG1SPE8QU6p8zevv35b6733fBQ8x80p6lkzgng30JevwrrweQIgEovh+MnT4DgWLMNYfq+/
    Q1N7oLy0GBu6u5iTZ86LANSjx0/Sq9d1kvaWBtfbR06tssxp5gIO/vteTK7rOgdgZWNjo7u+
    rn62LQKlFJpOUVtTgy/9zhfx2GM/xp7db0LTdbhcblRWVoJlWezevRuDAwPQtFydgE3k8flP
    P4TK8jL89MlnEI4kcGNLC777P9/B3bffiMPHTuHoiTMoLymCblyJ0y7C7/OSPJqoSCkVjN52
    VkGPDyTwkGeAqdEfioAQwjAMZiIR/N+/+3scO34cHMtCUVWwLAuHww4CAk1T0dLUiK7l7XA5
    HfPogfmUQTpbXGitIaT5UVqTl259zKtx+2X2oZWOblEOVQCEKKVP6rp+DMBtRUVFH6mvr3ft
    3r27TZKkiGEsOEKIaGTt05lMRp2ZmQHV6+Av9GHn7beSPXv31UajsRHj/PAAcImCWFJdVZG7
    yWY2yu1EaUmRWWtoCg41l5cWt9558/UkV8OrIZVKY2QsqGO+8MxsdtAAkZ9YvbzlmqvXdjHf
    +enzMUVV9xgAr8CSTVSNfRgBMEMImc0SYq4u5UKAkC7irIuEEF4URWvadwETwXz+rbcP4IUX
    X6K6rpvOjburcwWpLCuClEkiND6OP/vz/0Mfe/wJRVFUBEMTgqoq4DluFlwSipzaraJCURTl
    6aefJhOhIPfn//uPyZqVHWCgYdP61fhOWQk7PDpuilvJB46cEI6fPsdfs2ntgkDZvCwmAKfD
    BlHgWCOQZT6cxh4WLiYgXIQiydxyyy3+pqama0RRvDuTyaydng4Xh0KT/PT0DKLRBNLpDFSV
    GpQixiJtbooocEQURbhcTrjdLr6gwMWXlta66uubStasSVUPDg74jh8/rk1NTWWM1iNmQNba
    g1HFJVKFXmS/MgDYW2+99Spd1x89fvxEwejo2Dsoqr47cFuY6VvouLlcTpSWFkNVtdneYe8l
    u6iqOqamoqSyslQ4f/5cur7epjMMz4ZCCbG9fcUqu92O3bt3747meHNOC+sgezGC0K+++ioA
    sKIodoVCoYKcmMz8z51IpPD224exbFkramvrwTAEsiwhlUpDlmUsWbIERUXFRv8yHpQSnDo1
    gHg8hSVLyuF0cjh3LoDly1dgaGgahYUuFBW5kcloBh071zNNljU4nU4bx3E2VVWtYFD4oGxh
    vg00s7Qsy86eCy6XE+vXd6O4uGgWTGuaBlXVYNZeTU1FMTUVMYR3rPTR+ZnDxWoI8wIgNM8G
    qnl28GL1LdYJIUo2m8289NJLx+x2e6ihoaF969atn43FYq1er3fJ1VdfHX/xxRfDkiTNBkgT
    iYQsy3LSbreD4zhoGjA+HsLMTNzV3t7e3NfXF5EkKQPAY7PZqyRJcciyMisEBRCkUlmIYqnN
    brfbo9Eo73Q6va2tLd0zMxnf5GR8do86nQJUVZKi0ahuwR8qcuVf+o033tjt95c8GgzKZYLA
    QJISZw8cOHDK6AOsW3xQU1RtBrlyppjFFiYvQF2WL5C4YCilNl3XGVO8yUJqWHA+iKKI1tZm
    TEyE5IMHDyZLSkqIw+F0hkJT4HkHRNGJhoY6rF7dpk9NheTi4hLB4bAxVoBpfi8IPILBAO3t
    7VVzTB0fe/jwWVDKIZvVMTOTJrW1dczRo0czfX19amNjI1NeXusLhZJEVek7Cs3kWloALMuK
    hBCPJbHjNHzZiwMILc6ZCQaLADxSWFj42dtuu7XqwQceIJ0rlhvZBwMAaho0VZvfA81sjG00
    xTZ/x3EcPG43PC4n6muqsOWqtWTnLdvFx554rvGZF18tn5gK1wB4llI6YFHBNJXXMpZFRy9B
    i4RZMGioin5m+fKOHf/rN75ASktLoKvqIuBvLnM3S8ObbROhXzhDqM+nm4IAbx88gmMnTmPN
    yuWYzWbkZwkXZB51iAKPdatW4IePP8vH4gnp9LlebWRklHtg5w42Go0vPT8wwhlzGV0kQ6Zc
    JAfLBqBl2bI23uVyzmUCDLoiz3G4+cYbsHnjeszMzCA0MYH/+u//wQ9+8APMzMxg9xuv46FP
    fAw11VXwelyoqa5CeVkJJien8NRzu7Bu/XqcP38eBW4HHvnEPYjHv4V9B45g64bVEEUROtXB
    MgQetxOYrywqWB4c3l3V6oMYjMlgKfB68bnPfgbZTBoulxM/e/IpCDyPe+++y1By1MCzDNwu
    56yaHYUlq5N3hNL8BB0lwAdERbM4AAoAmRCSopSGAByYmpq66iMf+Uit3W4vfvbZZ1fouj4B
    YJpSyhugUM5K2ezE5KTXpDVuvXoTNq5b6372hZeaDeCVBGAXBN5TWV6eo9UaARW304HK8jJi
    OIQiAC8hWL7lqrWuVSuW5dowAEil05iYCpuOgQlGNOQ4/TyAG5fWVu146KM382+8dZSOBibO
    IqeKKhjXZjqaaczVosxgrulsMo+2lj+0RSLSBICq67qSyWTm2KILQCFmz5CjR48iHA5bFeHc
    VZXlgKZAkRU8/exzePKZ52VFUcMAnFlJElRFBc+xFpo7YLeJEAQBxvVLbx84aPvGX/+N7e++
    8aekrKgA5cWFKC4qxPDouGae4eFIzHu+b9C7ZcPq+XRW5FFcCSBlJSiqBktG2BTAMNuIMBdx
    Lc7uP0EQ2E9+8pPLCgt9n0+lMredPHmq6Pz5XjI5OQ1JkkEpMZpbs5av1l5nOYq3LAOKQpFO
    K5iaSoDjONhsIvH5PPbq6uL69vbOioaGhqVHjx557siRo8dVVeWN60wYc2ANjl70tgCL9Pdi
    PvvZz7ba7fbfP3v2XPPJk6dytOs8ZdELy7e/k3AMyWsjQWb7NdbVVcPnK8DISACapr+HPoZz
    cvGRSAK1teWcruuIRGbkxsYqeyyWAcA5ly9f3i1J0uSuXbvGDEfVpIuzF2tttbe3CwzDtE5P
    h8VsVvr/2/vvOMmu+kwYf85Nlas6555Ok3PUjKQZZQkhogBbYMAkA7Yxxja297W96/Wu12u/
    r9feXduvWcxiDA4YDCwGIQQSCiNpJI2kkSb1hA7TOVZXrrr5nN8f996qU7erZ0bSdM/4/VGf
    z/1UdXV1qFPnfNPz/T5PFakFIQIYYxgbm8Ts7CLC4TAikQi2bt2AHTt2QBAErF+/ERcujCGT
    KcA0bRQKGopFHcGggv7+dkxPTyKRSEAUQzh/fgRbt3ahpSWG+fkSDMMCIdTVg2RQlKAiy3LA
    bfnjk8E10Sh+HftQcGtEKJVUvPTSCUiSBMui2Lx5AzTNwMWLl8oFGMYA07S5ta0khTzhzNW0
    ll6nm0fgaDDGNFVVM2fPnh3auXPnK4FAYP3Zs4Pyzp07Nh05cmTxySefXLRtWwSg6Lpuq6pa
    DIWCTJYVQqkJTdNx4cIoufnm3d2bNm2aOXXqVIEQ0hgIBNp13RRN06oq1hSLGgRBlOrq6iKz
    s7PKxo0b+uLxhoFTp2aIadKyxEcopEDXVaNYLHpt7R6qax04cKBvw4aNn1xctPqyWQsDA4Hc
    4ODFF2dmZlQ3kfEKop5eoVcYXUmA3X+zVlo4y7J0QRCoKIrc514LgQNCoTAaGurYK6+8ZBiG
    oUWjUZkQQgwDCAQchH3Hjg1senoi+/LLL2sPPPC2JkmSBb+8kPN3bQCgZ86cUbPZLG6//Y6I
    qlJpeHgWghBEsWihszNKIpGInkqlShMTE+K6devjwaAsFQom97tq6dYSiCKBqpqCWwDnQY2a
    ses1Qwi5WYl6AJ/bsGHDZz7/+c8n3v/QzyKRSIBRBwUkrCIMyogzp8WoG1P6H3MzTR7RB5gT
    UOzZuQ0bBnrIzft3x/76b//x7pNnzjfblH6XMXbWRW74KjBb7aSwxryIAuB97W1tH//1X/1s
    aN/eve7c4JVnBvOFAmzbQjwa9SGC1XqEfPsnIQSTU9P42tf/BcmlFDYO9COgyBzLKC6LOjJK
    sXl9L/p7uqRXT58jS+ls6U/+8iuR/nXtRDcMEIJWxtACYNBrgXKSDrIqDlAURViWVS8IQltf
    Xx8CiuK0N/KtYq61T8TjiMeiGOjrwfj4OP7u7/8ZuXwejY0NeMs9d2Lj+n6XmdRZi6eeOYZU
    toBNmzbha1/9O3zofW/DQO863LRvF/7luw9jem4e/es6wZgjtR4NhSAKgmRTys+pSqsdBLyZ
    7QhXRh7EESfetXM7GGOwTBPJ5BIsy0RDQx0UWa4qynhJDLDScINTsGCMXVeXvwL1uAZHe05h
    jA3l8/nXzg0OrvulX/xFIZlM9jz//PMHXJQw7xpGoqpafmF+oRVgoIyiqbERH/q5h8jxl090
    LyaT8+5rG8LhcLCxoa6SIFGn7bKvtxsBRQnrhhEGEG5tbup6+1vuItFIGNRtl8zl8ig6TG48
    y5m3vLc0NdS961MffHc4Ho3g2eMnCwBOuu+Hp+QuuYigR9DDM9XxwT5bIVCoVVVXbdtOz87O
    Qtd1yLJUsTVuVlK2V5Qil8vBJb7Ry60+1IJtqkgvLeGJp4+yQrHote8EZEmCIBCfhipDMBhA
    fV3C+79UANoLL77U9Mwzz8rvfft9ALMRkCUP2Sy63QJaKp1JmKbpMLChWp+QJ5uZW1xCvlCk
    nI+T1vK8MsaIKIriZz7zmUOhUOg/TU5OHXnxxePS+PgEvODIuyrJSHnJfXT3yxk1AQJdtzA/
    n0UqpZKZmURow4bW/bfeeqS9sbHp+0ePHn2uWCwGXAQ4i+X6W2uiFecWh8lDDz3U2NTU+Lnp
    6dnbX3zxJeJpC16u3dN7rrm5GYlEDJOTM+4M2NWRznR2tmHTpgEIAkE6nXVbl68eXSyVNGia
    ISQSCen554/l9uzZYzQ0NAqalrOyWW0pmUzmvLlQrxCOlfU+37Ste//7H4ozho5sNgvLsqpY
    Wfl9YRgmLCuPXK6IcDiMPXs2w2OPnJiYQzKZdXUZHSKV9vZmhEISzp2bw86duzA1lUIup2Jh
    IYctW+oQiQRg23BRQgrbZggGJVEUxVq+8IYojnJFeU8YHZZlY35+AYSIkGUF8XgMgYABVdXc
    ooJ3HkXUoIHwIYa1ZghrfvTkOvhC6qJtKgCZMZYaGRk5fujQoTtKpVLXyZOnQnv27NmvqqXF
    Y8eez7iJAsvn86VEot5UFEVxCgACpqbmsLCQDu/Zs2fTxMREUlVVJstys6rqxLJY1TnVdQu6
    bgvNzc3RkZGR2KZNm/uyWSOSTBarzl0wKEPTCramaQLnC+3u7q76gwdvek+xKGybmiqSjo4Q
    dD0/8uqrr417vhoVQrCcS9TjMeB7c5l84euq/KD3vKZpGUVRjEgkEs5msyt0JVS0LEVRZKWS
    asLlanDQvjACAYKOjjYEg7Jx8uTJBV3XJUEQmEd6xtt0xgg0zYQsJxAMBqzz58+Xurq62MDA
    QGJurkQsywHcRVFikiSpAIqFQkEkhFFFkeB0Jq88RyhJAhSFsIWFHLVtu5YfFP0LIV3jjRkA
    8KFgMPjLt99+e6KrqwsvvPiiI69QRqlohVnTC0Jdkgv+e6TMJEcrr/WQRC6ZCYcCuPeOW6SZ
    uYU9cwvJMIC/Z4yd9LHKSfAxYq5GRZQj0gFjbF80Gv3spz/1C83vftc7ncCI0qrEbhlKCApG
    GR754Y/wzHPP4/O/9hn0dHe5BCArJ5CEEMwvLOCv/uYreOLoc6hLxB2E0G1PvZq5RMoomhvr
    sW3zevLamXMyY0jNLS6l5xaXqFt5mYRDtOEnL1gVJ2hZFgDURaPR+taWlrLgdRUigApLK6NO
    JaGxoQGapjli60RAOp1xW/woCGOYn1/A9374GA4eOoSLFy8iEQ3hzsO3QCAE+3Ztxz9/+/s4
    c+4iejrbHFEhBgQUGYIgiLbTwuo/VDdUVZRHKAQ3TPEchkAAIkuor6tDPp93WmZAYFPLTQZt
    t3BDIbp6OMzPMOrLA9ll7eyaVka9uQ3BjUUFxtgzLx4/fuunPvXJ5v/4+/9B/I3f+Pz2wXPn
    sgB+4iY2kqqqxdm5OS7BpbjjyK1434PvCH7pK1/bbFl2GkC0vi4hR8KhZfIcWzauRywaCegp
    ow5A/Jab9gb37txaTn4YgGy+AF03vEUSuDO0MRwKvvuDD76l4e7DB/CFr30b03ML43DEi8Nc
    m4snXM9Ld3iD87yUArtcO1UNFMswTXNkcHDQzufzYmNDfVnaoTrJYq5eUr0rREwtADpjzMpm
    MzC1EmZnpnH+4rA3y1kEgKbGBkgeOsjNIyqyjLbWZu9zMwDo+UIx9PKJ1+ruv/MQLEODQ5sO
    1bU9IgBbFEVX4aS6TZS3CaZNcWFkHNlcwUuQV81GXWatyec+97ndoVDoj4aHhw8fPXqULC2l
    QYiAQEBxWTXF8v3lHwtVCOLynyVIpQo4fdoi69e3dG3btuMDoVCo/vHHH/9RLpfjadj9Ooxr
    JQ0g9vX1vS2fLzz4wgvHpaWlVNXs25WSu1gsiv37d4MxgrGxiZooIR9oCYKIrq427Ny5icmy
    gHy+QObmkliJpXQlVJJShnS6gJaWFvno0aPa97///WQ0Gi0KgpDSNG2qWCxOcEUYezUgI/6s
    RqOxhGWZjYVC0UWWyWWDVkCArhvuXgFs20IwGKh6j6FQ0GUWnUQiEYcohjA1NQUAWFoqglIb
    DQ1hFItOwZBSCkoJBEESBUHg/d+NWhwVvG1OSEXrmlIKVdXgyVE48gzemRLK56vabBBf2+gN
    hQz6W1U9pm8CgJw/f/7s3r17X+3r6+t84oknSSQSrd+7d/8dum7kX3755UEAQjqd1lpb20xF
    URRABSECdN3E2bND5LbbDrQePHhw0/HjxyGKYp2mqaSSDDv7yTRtFIu60NTUFO3v72ttampt
    O39+kThi685aiqIAWRZRKGjMsixvn9BoNBq88847bxfF2MGhoaLg+BpBPXdueHB+fp65iJZX
    iMy5WsZJrjDqtcXzrfFXnNGsaAgykkqlFjo7O2cbGxvrZmZmliFtfEeC02pskVgsygDolmUx
    QgQaCsVFVZXR3NyEYrGQTyaTqebmpjpBEIiumzW7E3TdhCQpJByO2IuLSXVkZNjeuHFLtLGx
    QUqlTMhyAIwxk1KaB1CUJDHgzOhcvlUUIAgEJAQCjM3Pzxu2bWOFDb16CaHbm7rDNM3Mww8/
    nPnhD38oMEpFBohgzLkHE51REiaAQSSEBFpamgVHPNwVTKcUyeQSCsWi14zGa4swAJSUVckJ
    IwQ0ly8wOCLLfS6KJRHnVHvB+poYKrcqGgfw4bvuvGPLxz768wiHgk5SdxkyGC+AJIRhoL8P
    3/z2d3Hy1BkM9K6DyWwwm5UlF8oafAKBZVm4cHEEX/jy19h3vv9DGIaJu2+/lezb5Uo0+Eln
    lrWEVUTtAwEFe7Zvxre//2NJ1XTbDUI9YdVZ92uvSrMqArSeE3SqmlosEolE4/G4j111OROi
    c+gp6usSME0DSiAAUZKQXFoqB7WUUTxx9Dmksnls2bIFX/27r+Dn3vMAWloaYVOK7s42rOvq
    wIlT53DvkYNQFBmMAZIkgAiEcFUV/hJu0JZRQgQBAhFgGDpeeeUEZmZmoKoqCsUCsrks/uKv
    /hcMw4BhGrAtC5ZlgVKnlXvr5o1477vfhnAwwPFiV+4ZjxZef8fIixN7XxMAr87Pz59++Ac/
    uOtP/usf4ff/w+8F/q/f/b1DY2PjBoBnHeCJqnNz89A0DbIkOoLz0Qg++fGfx/mLQw1PPv3s
    TgBqIh6TAoFAlVwMAPT1dGFdd6eYTKVbG+oS4QfuvYMk4jFQ2yrvS03XYTkGWeDsUZsoCrc9
    cNctnR988H4sLqVx9MVXDcYw5H4/yJ21POcEl7i2GX+A/3rOI3HX6/S5c+cKIyPDiabGA1X2
    qVpQHti1YzsaGuqlZHJJ8BK5xYVFZqgFsrg4j0wma7rJKpUlSV4/0OdIx1Ba9TsDioyerk4v
    aTcBqIyx4ujYeCKfzZJSoYBiSaVcG1BUlqVAW0sjJEmE7XZaVEmiuP/jUjqDl04OMlY9z2Wv
    VTL07W9/G48//ng8Eol8klJ6iyRJ2Lt3H/XIUwipgEnLZ+QE4nf0XuDg0tljpZ91NK0kiKJc
    t3nz5gdUVU3+8Ic/THGzzrwvXPXilbdnPvWpT7ULgvj+CxeG6icmJjmk5epIYmZm5nD69HmY
    pglJkt1kjaJarxGQJBHRaAT9/euwbl07K5XyVJZFYXo6iUwm72unurKOIWME2WwR7e2dYlNT
    o3Dp0piRSqUKrg/02JL5Gc1VkTHxinmf/vSnI7ZtxzVNW2F/LJfh8Oa8KHWIZEKhUFXA2NHR
    hFBIxODgLHbt2o2pqRRKJacdVVVNZLMl1NdHMDurwQkonQYHwSk1Ctye8t+TG8MBEsIYiOPD
    JbS3tyMWi0FRFIiig1bt37/L4VsQ+WTQaTleXMzg0qXZ8h7jC/9+4foa7PLX68YzvDJCCLLZ
    rDA7O/tUT0/PrXV19Q0vvHAcoVCo/dChm+83TdM8efLk+MLCgrVt23YrEgkjnc7Ba8VeWEhh
    cHBU2rFj4wbTNCBJclzXc8v2n20zpFIF9Pa2RkOhoKSqNLi4WKjaq4IgQBQJTNP05oqJLMvy
    bbcd2d/U1Hl4dFRV8nkLvb0RmGZx7uTJU3NuTuHpLKuuf/E0HTOoMGu/UT8IALhw4cLcunXr
    LnZ2dm05d+5ClY3h3TYhAjTNQCaTI+vW9UivvXaSaZqmA8yKROKyqkqIRqMoFmcKqqoWmptb
    6hkThGJR99kYuJ0IOgRBIvF4nAEwstmcoeuaXleXkAqFIhwpkJJhGHoegN7c3KxQSkTDsLn/
    qfZVV6fANIvWzMyshsvwOaxKQnj//ffj0UcfzQP4M9u2vz43NxeFM8SYgEObXuc+jqHSy9qU
    SMS3fvIXPh48sG8vLMsAKIOqlvA//vIL+MGjP7Y5GFjjLt4Ye4yGHsWvpyEmcLIXZI2D021N
    TU13f+ChnxFam5thWiYEt0RVi9TF//yuHdvwp//1D2DbNo6/fAKhUBBdHe0IKLLzByiFpmsY
    G5/E0edewHe+9wg7eXrQsmybdHW0iR95/3vRUF/nbmpagyTC33rqPBYFgoGeLjQ21ElTM/Mi
    KuKVvG6Jyn0mq6bl5SKEQVmWg8FgsIrcooy+8CQn7uN4PObA5aKIaCSK2bl5UNtpg5xfSOJ7
    j/wYBw4cxPDwMOKRIO44fAjEyQwQi0awfcsGPHPsBSyl02jzBtAdZQbBV4IRfAHWjXYTvOAz
    m83hq3//Dzh16jRkWUYo6LCsZTJZiKIA0zBhmiYS8ZhTKRUIDNOsIpZhNUhmKkBZ2WPeCJVR
    CkeKAgCmADz95BNPHhweGYm8653vRC6XC//nP/yj26ampwmA0wDM2bk5VsjnSUN9HaiLOA/0
    9eK3f/2zQjqd6Xzt1BkrEgkLijuTC27Gt7G+Dvt37yAnTp5J7Ny2iRzYs2OZLiipzFJ5VfQE
    AbbevG/nwKc++G7SWBfHw48/g4np+SW3+BLmqryamxRlfcGotZLzu5o2QI+mHMC58fHxoaef
    enr/3j17KvPL8M0nA9izexcO3nRA/MEjj8bcdh1jYXERajGHfDYDy7KpaxciibpEYOe2LRAI
    ge2T0JFlCev7exAKBcVSSfVQUC2dyTBNLZJUOoWSqlIOBY21tzRHt27o5/73Slsrjz6eOjeE
    wQuXKACVAAarVI75CvKq2a14PA7TNKlpmicZY19pbW2VGxsbA6ZpBk3TDJmmGbIsK2TbdoBS
    KlNK5WAwEO7o6GyZmpqV8vlCuR1QFCV0dbVB10t0cnLCJESwCSEWIcQSBMG7NwVBMAkhlmEI
    lqoKhiiKai6XU72iFWNM5Biv1xIpRX19/e5isbhrZGQU9fX1CAQUJJMp2Da9qnlB07QwPHwJ
    nZ3tuPXWA7AsG8lkCsWi6nQyiCIikTCamhrQ0tLIbNs0x8cv2c3NzXKhoGN4eIJDgsgVW1Qr
    QRugqjoAUWxubpEuXRrzglKDi0c8gjU/mcxq7K0ApTTotIsSLhhEVVGAf3+GYZVnUQ1DRzQa
    Ks9RhkIB9PU5s4PxuIMOTk9PgbkjPbZNkU6X0NeXQCikoFRyTKtLu1/NxX/j+kLBi3PC4TB2
    796BxsYGl2nVQTzD4XB5fzj6bWbZ7YmiUJZVWDnhI2vdIXq1xVEeRCGDg4Mv9/f3n924ccOR
    J588iueee4Hcccdt6w4fPvw2XdcfT6VSlm3bRiwWrSqeMAYMDU0gGo2Edu3as0mWZXFoaB7V
    rezOR59M5rFlS1egrq5OOXVq2p09FWsg+UzwUOW9e/es37Rp24GZGSO6uKhBUWQ0NEj25OTk
    xOzsLHXRQc/fFV12TE/GwyuMvuki3+DgYO7QoUMvrlvXfW99fX04mVwqy+H4L9O0MDo6gUOH
    9oYHBgaCyWSywBi1o9EYikUJgYCCXM40RFG0uru7w4WCKpZKGiomuII8apoJTbNIe3u7dPbs
    WcswDNuyTDMYDEFRGIJBBYaR1lRVK0UiEamnp7chnzclXadV2q3+kFSWRTQ2KpidHdWTyaSG
    2n5wmV7tNUsIH330Ubhsf2MAFlyUrB4OZXqje1/HJYQBAGhoaMD+vXuwf+9uWKYBxigMXcfm
    TRvwg0d/7DHzeZXiPCo6I7zulqev4Q2alvVdfP3VWANnKAO4acvmjR1dnZ340z//77BtG7/w
    sY+goaEeoBXyGNSY5fMo2rO5HP72q//Injr6HMLhMDYM9JKuzg6EQkGUiiVMTs/g4vAoHZ+c
    MlRV0wHITU0Nwc9+6qPk1oP7ym22lbZKtqIeoUc6QynQ0daM7o42YWpmXkK1+LWfRll3BVhX
    peLuzSkKAhFEQahOAKvQKsYxGDJEIxEEgwGYloVEXR2mpqdhmgZEQcATTz+LVCaPbdu24mtf
    /Sp+9p33oq2lqYxgiAKweUM//vWRxzA3v4i2pnowyqDrupdhrKROeyMmhERwN35dXQKf++xn
    oGkqgoEAggEFAUWBLEkQRQFPPPk0nnz6KH7z1z6DeCwKRikEgvL8V8XT+ElmWEV2gpQXgVzB
    a17z6rkPv7TgsK0RABJj7IWx8fGRH/34xzt3bNuK9//sz8C2rPB/+eP/+8j09HQcAGZmZq1c
    Pi83NNRVFR0OHdiLP/jd35L+85/8mdTU0OBIM/gQ/kBAwZGb95PvP/oTcvstN6GpwS3ElIsW
    8ISgvbniCICB7ZvX93z24z8rDPR0IZXO4NnjJ5mmG7PumQtwBlt3ZV48giyjVjL4emfBXGkS
    BmBG1/VnfvDIIzvf+94Hld6eHoesii9SuWvS1NSIj/38h8jJk6cSU9MzDQBoOp1hmdQSgW26
    ySQkAI27tm+VN28YKHc1eP+jM38JbN+8EQO96+TTgxeCXvBi2zazTR3JZBKlkubZnbAgkJ7D
    B/com9b3lmVS/MQ3BECuUMQPn3we6VxeB1BivkKiO0awkmjxNbnde++9CAQCxb6+vr+3LOu7
    hUIhrut6nWVZjbZtN1BKGxljDa5/jFBKgxs2bOiKRuP1r756WpqamoMkyRAECbKs4PDh/bCs
    gv3YY48XGWMeu3OREFLkHnsSQCohpAAgY1lW0rc/yFqjGF1dXaIsy7unp2fqI5Ewjhy5FcVi
    Ec888zycxFeoSmhqJWjRaARbt27CunWdNJ1eMhVFptu2bRAIEdzWcAcBLhYL+sWL5zRN08TN
    mzfHRTEgnjx5gRSLao1kULiC6H0lodJ1izQ3N8miKFLbtr3zqK9QmKbX3IiT8gyuyBhEz+bV
    mh/kg3OAwLIcFk1JElEsFhGNRiFJEigF2tubEAyKmJubxc6duzE9nUKhoFd9BtmsCkFgCIcV
    qCqDKFJXroYyjjCodk/lDYIQOiAZQbGo4sUXX4EoSi5iSkFdMnsAGBjoQWtrM1555ZwzR881
    lzktj/7EG6gmnCFXQgtXdV1q+EIKR8qBARAvXrw4nc1mn+7pWbe/oaEhlEym8Oyzz5Pbbru1
    984773zryy+/fEbXNc0rqPN7yrJsnDkzTERRDA4MdMFrf/Sj+YWChqWlAqmri5LFxXxZiN1b
    R8YA22aQJFkihCibNm3qOHDg4IZUConp6RIoJairkyGKZvH8+XMLLmjknSsPISy6l4ZqmZc3
    5Afd1zNN08yRkZEXd+zYMbx+/fqd6XT2sujb5OQM+vt7pJtvvrn1qaeeMg1Dp9FoFJmMBEGQ
    AUBoa2uLtrW11587NwnLsn0z416rLUUymSPd3etC9fX1oq7rBmOMSZKMQCCMYFBGKlXSKKVs
    +/ZtrY2NrS3nzqU51mRSkz05kVAQCJj0woULJVe+yX/xMjnXPiHkNiNfLeL7mfPu93VUmBq7
    mxob5fr6Opim6bQCURuiIKCjvQ2iIMCm1Pv5HDgxZlRIFcqBE6rlEEysLMi8WgbISwg3b960
    KfzYT57AP3/jm/i59/+sm+hRrs2phqg8AfL5PP714R/iC1/6Cjt77oJBKS0CsM6eO0+cRg0B
    lFKbMeYFNyKA+nVdHcFf/fRHhfe9861QZImbO7zMvKKvdZQyoC4eQ3dHK3ne+Yy8z08nhGi+
    qqjKBVer0ioDgNk2pZZlViGEVa2vHkrgBeiKgvpEHKmlJbS1tWF2bgqmaWIxm8X/efhRHLr5
    FoyMjCASlHDnkZtBiABnJMoJ3vvWdUKWJExMz2LH5n7YNsNSOgvTtJbpLbpGeLUqwtfGH8Jp
    l9m4cYOzRjbH6ssc5DSby+H02XOwbRuhYNAlP6KcPEp1+bHWFOH1igJWSArL+xbAuK7rzz32
    48e3fvChh6TOznZ88AMPQRKF8B/9yZ/uHRsfzy0kkySZTKJ3XXc16ROA2249hL/6sz+CZVoI
    h4JVbdveublp7y78x9/+LPbs2OIYPR7NJkBDfQKRcEiAq+W0vre7+dc++QFp7/ZNABiGLk3i
    7NAl3UUHFc5uGS45gBf869ciGeQ2CHW1XJ88ceKV+x9++AdbfunTn+J0A5fPGt91x2345U9/
    UvrTP/+f7elMppTL55FaSiISlCDLogigNRqJtL3rbW8RGhsb3Fb56plnSil61nXi/rtvFy4O
    X2rQDSMDQAoHA0RgNhaTKRQdoSsFQO/WjevbPvDgW0kkEgK1aVURqCK5w/DyyUE8e/w1CqBA
    gCKriPAW18JmeftQ13WcP3/e8nVWlNwAx2M5Nd2iphKPx+sppWKppLqFOcc+m6aFYlFFXV2Y
    CYJg6Lpe4nxfjiuQqlzhTvOhyPYqolZXQktlQtCRzxfklpYWyLKMkyfPolRSywGMoigIBBQX
    jasE14IgoL29FTt3bmPBoGS9+OLzhXPnzuUZY2pdXZ0Zj8epoii2ZVl2LpczbduWN2/e3LJn
    z57WfF4PnDp1DrlcgUsGLy9pUZnJq7yOUqBY1FBf36AoisJUVTUBGJz2YK2zuRrxBnHmlGB7
    81i1Wl39ASylDJpmIBQKo1AooK2tCaIoQlFE9Pd3YGZmErFYHJIUxtTUdHn9PSSjWDRhWTYi
    ERmZjOMTAgEJuq7apmky37660Xygm6U5/xalNpaWUtwMrsCRyDjzlE1N9SAEsCwPfYEvGcQb
    zX/JdfKFjBDixYqlixcvHjt48OD7enrWbU6lMkins3jmmRfILbfc1Hnw4MGoLMuIx4sQRYlD
    RStI1smTQ5iaSiKXK9Xcc7bNcPHiHEKhAIpFw8fqK4BSAl23EY1GlW3btjUdOnRzY6kUjF26
    lIdlOaM59fUy0um5pYmJSd0tjJpcYVT1FUbp6+2MWcFmM8YYfeaZZ8YGBgZ+smXL5k0jI6OB
    VCqL2m2ZTtvoiRNnyG23HYodOXKkMxAISJSGEAjIoFRCJBIN79+/v9swaGR6enFFPUPGgKmp
    FLq6NgT37t3bcPLkyUVJkkTGJAQCEmSZ0EKhYGzatKlt376btiwu6sGlJXXFRJAQBx1sawti
    dnZUHx8fL3B5Ee8HtVVNCGtA1h5y550o0/1nPFaqEACpo71dbGps4hAz58U93V2or68Tkksp
    r23U69/35mg8IWYvePIcsL+abmONhujdzSUTQuJ1iQRCwSBuO3IYH//Ih1GXSDjkOgBefe0k
    pmdmcM+dt0OR5bJdHRubwN985Wv4p298206lM57z9xBRlTKmw6lS2u4atsuS1Lt/z87YZz75
    YXLHrQfdCiBdWeyef55Ws5cCDJIooK+7A4osKYZpMV+LDL+xeFKL1VhfAkA3dF0temKh/gQQ
    1XNEDAyKIqO1tQXT09NYv349Lp47A1038MTTzyKdzWPHjh3427/9Mt73trvR3toCxuxyYEmZ
    I0Tf2FiP8clZ2JaFoqphdn4RXHDlvVfq6v3dUAmhKIqedARxeUbhCTMwDp32ZnUBB1V1Zi9N
    aJo7L+IG2aLIcaB47aHLdAgv55DX3BHClxQWADx36vTpd7308ssdHR1vg6zIeOhn34dYLBr4
    4//nvzWNT0zi0tg49u3ZVUVW4iUwWzZuAGMUtm2V53LL6LSrR/jO++8qEy/wv4Myhsa6OHq7
    O4ThSxPNG/q6xc9/+kPCkZt2O9oTto2XT53DUiqb44plao0i12okM54UxsVCofj4P/3T1wfu
    uvMOZfOmjT4WT5ST32AggI9/5EOwLFP5qy98Scnk8picmsa69kY01sWUucV07x1HblHuu/sO
    EAJQymoWpBRZws+8+wGcPDMYfuyp57oZY6ytqY4oEpDKZKDppgRg84b+nuZf+/SHxO2bN/hm
    ovm5a2ApncW3fvAES6ayOoACq3SRFHyIGp9Ur6YfKAcanC/Mc74w79rwQGtr62bDsCRV1Tgn
    74lll9DeXk9isZjtaoGVdSjde49gyPIVYD3ZJX0VbfRlb86skFOYGh0dx+LiEhYWFt0WPQd5
    2bp1M5qaGvD88y9D1x3d6EBAwfr1/di6dQNLJhf1J544lpmenvZ8YX5ubq44NzenuQUT0tfX
    13LzzTdvrq9vbB0bm5FGR6dgGGZVwHQ54fuVJCgAoFTS0dERk0OhEFFV1cLydlHeF9qrVWzQ
    dd0QBEF15Fpqo4P+lltKGUolDfF4HDMzMwgEZEiS6DKLipidncXOnbswM5NGoaCVkyNvD5sm
    haoaiEQiEEXH9YVCEpLJnKXrOj+LVMXvcCMlhdVAJqkxB0jKBCqa5pDwSBK7LLFM9e/gk8Xr
    f6vhCykhxGKM6adPnx7fvXv3i/39vZvOnx8ium4ik8nh2WePkwMHdie2b9+EeDzGgsEA0TSj
    6ux4gvVzcymOpZYvqjj36XQJmYzmJtzLSU/yeQObN7eG7rrrns5cjkgjI3kYBnMJt0REo8Q+
    dWosXSqVPI4GPzqo+oov1+KsMQBWqVQqnT175if79x+4Z9u2rTuef/4lNzHmu6ErM7upVBbH
    jp0gN9+8N97Y2IR0mrBgMEgsS0ZnZ1ejZVnCiRNDgnO2atkfwdUQVTEysiBs2rS1JRIJS+Fw
    NJBKSVAUCYpC0NbW2tjfP9BVKonx0dEkbBtVGpDVe5ugoSGAUMigzzxzqqiqaoHzgwUABbej
    pCbCuhoJIQVguW0Vnt6W7lVD3Q9ZYIw1iaLYtHXrZoRCQYf6vpzlU/T1rkNLS7OQXEoJXKLn
    MQwtulcWFYpZPinUOVh01SrCK6BaNmMsPzc/j1/69C9AEkXU1yWcNjxBwPz8PP70v/8FEvE4
    bj98CwKKDNMwceyF4/jLL3yJPXn0Ocs0zRyHhFYlhe57TADY1tzUMPCet98f+vkPvAf9vetA
    vCC/hgi9P4gqt3KxarRDEAi6O9ugKIpsmJaIykCv5ksKSzWSpGu5jgRAoVAo5NPpdFVgyger
    fsQwoCjo7+3BU88dx+bNm2GaFsYmJvD9Rx7DrYePYGh4GEFJwN233VoOWPm1CigKWpoaMbuQ
    hG1bmF9cwqXJWepz+LzQrH0jOUFOWLVMX6HpOk6cOIGJiQlkszkUCwUUS0WoJRWmZWJhYREl
    VcWf/8UXHEp/6iBb8VgU73/POzHQt87pPl72EVd0CN3WmTVrF11hz1QJE8Oh39YZYxfS6dSp
    xx7/Scdb7r0bsixCEkW87a1vQWtLM/nGv3wbdXVx55zAz2DrkBGV9W94dBCsNuMl/zsoRTwa
    wVvuOEQCsiS//9334eDubRDcfzObK+C1wSFm2XbOXSu/Eyz5nOCbRgZ9FWQTQJEx9uPXTp48
    8vf/8I+7//3v/DsEgwFQT5cQFVtBXZHnX/yFj5GO9jZ88Ut/iy9/4xHcvHsjCCBu37pF/Myn
    P4aW5kauXZSbu3QTacoo1nW147d+9VPEtu3E0WMvMVEQyPmhS3jt7BAUWYru27Ut+osf/Vly
    64HdZZZh+Gcb3TV/4tmXcOzl0zwRTZ5D0nLuc+oa+oOyJpj7GXpFJc39f2TGmBCJROJtbW3R
    ZDIjeEEY7+Cz2SIUJSjU1dVRdxakACBLCFniiqN5zg/yxVi1Bnq1Zknh+Pg4NQxjtKWlWaP0
    bHBoaNRF4pz32N7ehm3bNmF8fKosKVFfX4cdO7agra3JHhwcLL7yyiuZfD6fdZPfDPdZauFw
    WLzppps2b9++fXexaDScOHGOLC6mykHc1YncrzxHCMClhZfFeDwup1IpL8bwz9Vr3PqvRoGQ
    5PP5UmdnZzYcDl+2ZZR/TClDNltAX1+LWyx0WnC92cFoNApFiWBqaoZDByvvnzEGVTWRSAhl
    ZFFRGFtaShou07Bd47ruBVJSrdsCAJBlCe3tbYjHEwgEHB1URVEgyzJE0SEkCgYV3HTTdvd3
    CG4ibuLixWkUi/oKbm05+6jPC655qlgrKQRgZjKZ3Nzc3LH29o63t7a2NE5MTEMQBBQKJbzw
    wqskmy0gGo04cltYiQlYWFEMvRYjp5/4Kps1kMkwUdeZODFRgGmycnIZi8lgTNdHRy8V3cIo
    4zrU/IXRN40O+vaMzRgzn332uZH+/oEfbNmyuX98fCoyPT27LIHj38/iYgrPPPMS2b17K3p7
    e9HXFwEgQZaDgZGREYyPz1YVICqzftWFp4mJJSiKLG/atKlZUYJElhlCIRmhECWS1NEyNZUj
    o6OLRNfpCnbM+d3BoIiOjgAbHz+nXbp0ye8L84QQL5dYE4SQd0iMq1bytMTEZRjqTMTjHQf2
    7YMoEFiUVc28tbU0Y/uWzcLguQs85WyRMcYnhSn3jVFu43sBu8UZ7zVxhp7YM4AXnnr66IOP
    PX6o6cF3vQOSJMKyLExOzuJvvvx3uHBxCH/yh7+PaCSMxcUkvvmd7+JLX/kHOjJ6SUN1W2yt
    KnC3KIr7d+3Y0vOpj3xAuu/OI4iEQw4ldFm0kVVJc9SaF1yJ7VQgQEtjPcLBgFgolmR3BzPG
    GHVnRG0u8bZWo12EM2bpYqm0NDk56ZLMVLeNLpNDYE7lefuWzfjO9x7B8PAwLNvCIz9+Aul8
    Ebt27cKX/uaLePCBu9DR0VqlC+mthyxJaKhL4OLFOeiahqHRCUzNLNhYTm1srVZCfO0gVsdQ
    pFIpfOnLX8HQ0BBCoRBUVUNAkdHU2AhFkaHIMvp7e6p+dnJyGvX1iRqMJdwsIeNbSMnlZgiv
    S7GUK2YkGcNzL7z44u0jo6OhLZs3gjIbAiE4sG8Ptm3eCFEUqmb/sELC50cHV3qtH8F6+z1H
    cO9tBxELh8otuwIBpmYXMDQ6abl2UuTspraCE7yW+4zCFYcHMGIYxne/8c1v9t925Nb4fffc
    zb0fVCPyjCIUCuFn3vMubN+6GV//5rfxxNNHUdfUhs995lO4ad+eSoJca024hHr7lo34w9/9
    NfJ3X/8OeebYcZz9y39AQ309Pv/LHyMP3H0EXR0tzty1//dwgc/o2BS+8f3HWb5YUjnn5xXT
    si4RQWGlYGKV9p6H1vGjE16HjOxGBUJ/f39vLBZrP3t2qGy/vLkkQgTk80WYpi10dnbKIyMj
    JmPMQxq9bpkF10fo3N6wOVtlcJe1Bu+9nEwYhoGLFy8+uWHDhrccOXLLbWfODIqLi0mIooSO
    jnbs3LkVtk1x7tyQKyS/Dtu3b2aUWsbTTz+du3DhQsayrIz7/pZcf5gDUOzq6orddttttzQ3
    t+4dH5+NDA+PQ1V1d/2EFdHAy2kP1nrOsmwwRoREIhFANbM0Dx15ccdli1VvptA1Pj6e27Jl
    y6IjHi8uQwVqsc8CQDqdw+bNPWhqagQhBN3dLQiHJZw5M41du3Zz6OByxIExR19Olh3ttVhM
    AKVFe3Z2VvX5QD7Osm8Y/1d+QwyRSAR79uxEIpGAZVmQZRmWZUPTdHem0EY2W6hqC43FIhBF
    rWpOcPm84LIZQnIZF0iukx+khBCTMaafO3fu9Lp168739Ky7dWZm3p3zc8h0zpy5CFmWYdts
    hRZHwc98vOzc8K/zJ4WCIEDTbJw7lwZjxJVPEcsMpHV1MtLpKTWZTHp6vR5Jmb8wWhVzXQMt
    Va94Z9q2XXz++ecfv//++2/es2fXbalUhnidCyvNE2azBbzwwknMzi5i06YB1NfX4fz5GZw8
    OQTDsMotyv4WT379bJthZGQBmmYJ69e3obs7DNvWMDtbJFNTWbK0pFYhgyt1PbS1hcBY3n7l
    lVcKhmEs84UcuFSzU2Y1EkLL5wwJljM0BgBs2bxpY9P2bVt9DJgOyhWJhHH45gP4/g9/FFJV
    TeJaqLx2GC9hKvoqvn6JCv/Xa3F7ZnJq+h/+4A//+EPPPvd8447tW0mpVMIzzz3Pnn/hOB56
    34Nk3+5deP7F4/jS3/49e/Sxn5j5QtH74Dx0cMnnBCmA3dFI+K633ntHyy9+/ENk2+YNDipo
    21XoBXyIIM80eiU9QsYY4rEwEomYsLCUDmJl8dma1OXXSujY/T0FAONnz55lpVKJeNIkvB6e
    H62hjGH7ts3o7e7Eww8/jMb6BB5/4ijuuPteXLx4EYpEcO8dh6ssM78ukiggHotAVTVksjkc
    e+U0CiXVQ2v8lWGjVpXlxvGHIAxAU2Mjfue3fxOmaUKRJXzxS1+GIsv41c98GrIseSyqgCsg
    USoW8cd/9hdoqE9gQ3+v26XMfGtfPTvo2KcbilvHr8n0yqWx8ckXXjy+cdPG9VWfeygUcvQX
    lxVUaiB+V4kO8q25oMwl8pErc5nuzOrg0CUsZbJeUMV3VJRqoIN0ley1QQgpMsaempiYvOmL
    X/zS/Tu2bRXaWluqCWZ8JDMAsG3LJvzeb/0afubd74CiyBjoW1c5vzUKT36GUMYY+nq68Vu/
    8gncfvMBpNIZ7N21BV1tLRAEUi7a1EouCQBV1fCtR57EmfMjFipzdR4jK48qrUhEsEp7z+K6
    VnT4qPnd4qFw4MCBg6Zptc7NLZbndniUUNNMJJNZ0t3dHYxEIigUCjocmY4CIYT3EyVfYZTX
    HrRRrUO4JuMTjDHy3e9+d/zee+/971u2bMkeOXLLLZqmJwgBRFEwYrGYcv78cCAUCmHnzm3o
    7Gw1JybGc8ePH8/Mz89785Ep9/15umOFXbt2td166y33AdL2U6cuylNTcxxT5MokMZdDCGuh
    hI7mGAOlRIhGI978p+cPFc4vCtfa/3k3V/cTp0+fLt59991jjY0NNBAICrpu1AwK/fOF6XQe
    mUwRGzduhGkC3d0tmJgYQSwWQyAQxcTEECh10MNav8s0KUSRQFFk1NcTpFILRjK5pPn8oO4r
    vN8oLoB4XHCFQgHPPXe8rIm3f/8ulEoaXnvtPHhCHi9oV5QADh7chkymiHxeLes58kngSlqE
    hNwAOkzL5wltAMb58+fnDx++9fmurvaD0WhUyudLVYyinm7gSklHdfFBuAxKvZK+KIFpei3j
    FUU4WRYRDhN27tyMXiqVvNjSWKEwaq+CvS4XYs+fPz+1YcOGb23YsGHjxo3r28+cOV/jbJGq
    JM80LYyMTGFuLoVYLIpcrlROBv36gCutKaXA1FQGmYyGlpY4VNVCJqPDspgrieJPKqsJfuNx
    GS0tInvllZPqzMxMDtW8K2nXX+S5rhF7LRJC1PrAOHYswhhrlyTp9jvvvF1pb2/l29zKQRYh
    wMEDe7FxoD908sxgg8/4Ej6Yeb1V32tttGusQQ7AP8zOzS3849e/cY8oSZ2MMdGyLIMQ0mmY
    RuLP//ILePiHP7JHL43nKKVJVEhycjWcoAjgjpbmprs/9sH3JT780INo9hM2YAXyGEaryBdQ
    Aznkk0hKgWg4iEQ0Il7GAa6qE/T2CmPMAHDm9Okz6uzsbHj9QF9ZOBvwozUeukzRUF+HX/nF
    j6OttRlDI6NIpnLYsWMHvvjF/4V3veV2dHh7zje3yuAgpMGAAotSXBidwPGT5ykqREV+liYv
    IbRxo94YgyxLGOjvAxiDZZkwDAOFQgGhYNDRWqR2mWTGoWwkWEwuwTAqoAPzsr4qghmPOMXx
    gLVadW4ER0gIMRhjE/l8/rVnnn1u43sffCc8kfkq1Pxy6CBv3ThNT3YZdLCSPLu/37fnDMvE
    +eExlFTd4BJYrz3bz2BoXwsbtsKMifc35wH8n6eOPrPtW9/5Pz2/9KlfKMtQVGRGqm2GTRkU
    RcGObVvcGUqrnESW7Tlfm6uBFDJKEQ4GccetB8ptptS2QW1aheD75TAIGF46OYiHH3+WmpZd
    5NBBLxlME0KynBNc1i66Sr6AoXq2vWZ/2ZEjR+paW1tvGh0dj2cyWS5AqiYGmZ1dQmfnpnB3
    d3f43LlzQDVpUgmVNlkLy0k+/Hoxa1mRYowx+7HHHrswODj41/39/cei0ej6UqkUm52dlQ4e
    PLi3u7uzv729xVZVNfv888cWBgcHM5qmaT5fmASwJAhC4fDhw/379u17XzZb3HDmzHlhaSnr
    kn8IKwai/jZcr211OTqIZc87M7AgoVDI833+Aql4uQLpNSqMglJq5fP5042NDcVEIh5bWFiq
    gerV0iK0cPr0KHTdQl1dDOGwhKmpKezZsxezsxkUCloVC2t1UA8XLQIiERHhsMlOnRpVS6VS
    LV+4puM5V+n6ysfNkdHIghABsuy0ioZCnrwVj1RVfj4cDkLXLR8CWAvwI2V5Cu75G6k6WkbA
    LMsqzs3NnejrG5hvaWnuzOcnViya8Ogej/r50T8/alhLKJ1/Xa3ZwlBIhCiadGZm2uByCN4X
    +pkxr/Ue80bdDMaYevTo0Zc6Ojp+vGPH1p+bmZmX0+lc1Ww3j3HxyaGq6lBVk0MFyWVabGtf
    pZKJ8fF0+Xfytq0aY0MVkUxXVxhLS9Pm6dOnc4yxPJ8Mwtduv9I6StfSaF0WqnArhi4L5/0b
    1g/sefsDb3VIUCyrugrsooS967px/713CucuDrUZhtkAp000Coe6PQyHmEbzV6ZWOeG7ohN0
    /58sY+xJythFaprdcCQ46hljN337u9/faFmWbln2guvwVM658wlhykVT3zbQ13Pn537p46F3
    3n83QsEAR16xcnLHlrVY+gPf2klkKBBAJBwifmSQMSYRQsqtv6tZXUaFIevU2PjE9NFnn93Q
    399biWz8TKM+tHDPzu1orK/D53/3P+H2O+7A8PAwJMJw751HyuLGNYP/siM18Phzr2ByZsEj
    JuFnJ0su49VaIQ6v9zxWKeeWc143eWtva0OhWAClFLYrRu+1MXrFmK6OdiiK7LZM2eUA3B9b
    lud+yRUH669Xu4zXPpcBcPzkqdNvGx29FNm5Y1tVMshW0rhchgTSqsIV+KSwCh2szEPD33IJ
    BhCGdDaH0YkZxu0hDx1Uuf21mnuMcWvkabm+VigUHvmHf/z6x4/cenNg5/ZtleIJqK/9vJJQ
    U97uLGuzrT6bPEPocttEqxFXf7suJ8+zkEzj69/9MZueWzRQmRVM+67MlZzgtQ7eeRvmS8r4
    15EDB/Zvt216+8jImKDrJkRRqhkgLC1lUSzq8tatW1suXbo0rmlaEA5jacAt1ElcEEXfrL++
    lkGWR2gxPT29OD09/QqACQBNABoppdrGjRsX0um0efHixezS0pKOajIl7/NckiSpcN99923f
    smXrz83OJnvPnBkixaL6OkhiVkIJL9866kgTMBIIBCRCiMwY432huNq+kE+sx8fHz+7atWus
    p2fdjmQyjSszjTrPp9M5nD49isOHd2J8fAyxWAzBYBQTE8McKu1viRSqyFMaGwUUCgvWyMjI
    vxlfyCN13myf854Ycrm8q7lYSWwcFKYy15XPq67EQnUSU1tto/L4RiiMrlActQAYQ0PDQwMD
    6y90dbV3jo/PgCfeufxs7eVIjC5/vmr9DJ9wRiISNC1vLy4mLV9htFSjMHpNCnorzFoaANRs
    Nps6ceLEw0eOHNm7ffvmHS+8cMItjqxkM0hN5s/LJX7VgvfLUUdvT/K6hSsVupqaAgiHDXrs
    2MuFXC7Hd8mkAKRcdDCHFYjVvHWUVtmQLQ9SgW3BYODn3/feB2Nbt2wua8DxlWTmij4rsowH
    334/nnj62dhLJ072uRXsNIC4e/lFmqmbSFzPpNCrCmuEkAJjbME1lA3uB6Jpmn4B1QKitZxg
    GkCIEPLgzm2b7/x3n/tF5Y7DhyAKAicHsFxGomarKPW1uy0jmaluHw0oEkJBBahuExW5e5FH
    CFfJiHmf56V8Pv/st7/z3YF77rpT6OrsqCAQy95vJX6n1MYzx15ArlDCjh078OUv/2+89Z7b
    0dXRVmGKXIGp1LIsTM4sYmRillk25el6y6yFWGEw93oWI1Y6g5Zp4sSrr2J8fALFYgHFUgml
    koq//MLfwDQNmIYj+WJZVpmhdn5hEYIg4L/86V9AEgWIkoC9O7fh9ltucghRfHgDPyDNOcMb
    QZuKMcYsl1zm7PjE5OSrr53cvH3r5pVZa5ehWFhZ9sTfSupHB1d4rUCAxaU0JmbmvdYYq0ZF
    VEO17uCqthO5gV2GMfbo2cHBA//49W/sH/jdf+cUoHxzkVWIXVUihyokrxoVxLL5Sj9JzLJk
    sNbcMANsauPHT7+AZ186aQPIEyDPHCdYnjfjnOCKMxOrGYz5zyG37uS9731vfSgU/rnx8cme
    iYlprv2s+nLmbkyMjs6RXbt6mzdt2th+8uSpBTh6vjG3QBp29ww/v38j2CPmVt01AAU3mRI8
    /zg8PFwaHh4ecxNbj43c7wuzsiwX3va2t+1dv37Dh8fHZ7sHB4eJwwj5esXmL58s1m5zc1Ay
    WVYkURQly7JEzgeulS8EAPryyy9Pb9686akNG/o3j4yMy9lsHpcjlakE3ALa2xsRDIqYmprC
    7t17MDubcVshV5qtRHm2KxAARFFnx44NlpaWljz/5/eFqr9b5jrvPcL/eVF0SGUSiTgURXGR
    Qgl7926DKDpSFIIguAyazhqEwwEEAjL27dvgxhQMyWQeCwt5Dhm8Kh3CG0GruEy8ODIykiyV
    Si+3tbUcjkTCSqGgXhb949G9aqIYYZkG5tUnhTyDKUE0KiKTWbKKxaLJdT+oWK6bt1oINONQ
    Qo0xVnr55ZeHNmzY8L316/v7xsdnolNTc6jV8rnSPCXf1unvUlhOwCPgcm22l0ssQyEJ7e1B
    NjJyShsZGc1zOUQKFR6SDKolimoWbqS1co5uq2gzgE/fcvPNu3/u/Q8hoCiV+SSwGnpVNgb6
    1uHjH35IGJuYWreYXEq6yFuOIwrQuE1i8knhdTBKVb3I7v8ncM/prvGcv4wT9OYjQwDeu2/X
    9jv//W99Vjm0f7fbOmLXCML8YvO1nr9aPUJnji6gyLhMMriqbTK+5LoA4AfHnn/hyNf/+Zvr
    f+2zv1yRQmDLE12AghBgbmER33/kMdx8y624NDYG2Cbuu+s2CETwrWG17pxt2ygUS0imc7Ap
    9ZCHPHd5X6/I1HTDJIPuliwWi/jaP/wTTpx41WVXkyHLMmRJgixLsG0bpmUhEYuBCAQETpsp
    pRQzs3OwbQumaaGpvh6UUgiisAwpFAiBUHuogqw1SugPpFwyJBPAZLFYPPfi8Zc3vffd7yAO
    u3GNhM0lgnF0jQUwUFBGnE/ZrTC75KqupIfbAQECCIJ7BG1QwlyeLF+yCUdrbmp2Ebl80WuN
    4Zl8tcu1yFwLm1aj8MLPLl4yTesHD//ghxvfdv998SO33gzwqCernrsFcSiFBEJcpJmAgrjv
    m8MEy+zHzp/ly/dEEFwbRVzmX5ST0Apai/JnMzY5g28/8iQrqZoGJxnknaBXUMteDh1cK9/A
    J4OMMRKNRqUNGzY8oKraO0+dOieWSioEQeJmk5YHFrOzS+jqag7s339gYH5+YX5ubi7DGEsQ
    QhI1CqPlxPcGKI56vlBwST5slxjH83URF+0UUT0GosKVC3nrW9+6a8OGjR++dGl63dmzwzAM
    00eucPUJ3tXPEVYCVsYYJEkSBCcD5f2fyJX5Vz2YT6VSpXPnzj+6a9fuO3bs2LLjhRdOoKww
    5HtfPMIXCgXQ19eBqakJRKNRhMNxnDo15O5JYRnawK+BLIuQZWBqasI8e/ZsnjFW8PnDfxO+
    MBAIYNeurWhuboJtU9i27YrTs/I6CIIAw7DKZ9a2KQghiESC5fktVTVBSKGmS+OYrtfc570O
    X2hqmlZMJpOnurq6lxob69uLRe0KSF4td+5PgElNUiIeZV6OwDv3kiQgGCSYnEwZusPg4jH3
    am4xSce1Z9m+XHHUJISojLHcSy+9dPSBBx44vGXLhtsWF9PENO0VOw+8gngF+6rumqpGBmsn
    lZU1Wz5vuFxiQnCJZIKwrLT96quvekQyGT4hJISkL4cOrklCyDtBQojAGIsD+MT6gYGHPvfZ
    z8h9vT1lMpRl1XUu4CCE4IF778TopfHAX//vv99eLJVKcNhGi25Fmx+OJP4qwnVwiGWE0LUN
    PNtq0Q1SIoyxAOdIPOFNjTFWcpPFD+zavuWe3/vNX1EOHdjjBEu0Wkri6uYFV5ov9OsU0qp2
    LFmSwDk73vEJa1TxYowx2205Olsqlb719//4T5+96cC+yG2HbwFlFQr7arTCOddPHT2GpUwO
    e/bswZe//L9x352H0d3ZvgwN5BEigMEwTCylMqCM2ajWbymLQRPHI6yJptmbW0DnisZi+O3P
    /zo0TUMwoECWxDIXmigQPPXMc3jmuefxq7/0CcSi0cpMoctcy2wblNqQZQmSJHJyAhWQzZ3N
    ITdINdRv6Mtto4yxE2cGB++fnZsLDfT1lslJ4AVFDDANC9l8DrlcHrl8HulUGql0Grm8g64a
    ug7LbXMXRQEBWUYoGEAsGkZdIoq6WBSRcBDRcAiRUBCiQEApASOAC9bDphTjk7MoabqNanIG
    /jLWCIHmGVk9ps7nxsYn7vj2//nX2/bs2kHCoWD57PDJMqU2CoUSstkscrk8kktLSCaXkM5k
    kMvlUSyVoOs6LMsGpRQEDEQQIEsiQsEAwqEg4rEoGhIxNNQnkIhFEI2EEYuEoMiSw0ZHiXPe
    3Q1tWhYeeeIYBocuWTXQwcs5weuG5HvJIADxE5/4xAFBEH51cPBC6/j4JFZivuQr8YZh4/z5
    Kdx006bGO++8Y/uPfvTjbCqVyjPG8q7Goo7qOdNy++h1TAr5feX5RsMlifBsahiAwhgrF0fd
    OR4DgH7fffdt3LRp80cmJ+f6BwdHrjIZFK7IKFobXayNbrhtlYLbIlrLJ5LVLI66towyxswn
    n3xyqKOj/VubNg2sm5lZSIyPT1chetWBqPO4vb0J4bCEU6emsHPnLhcdLPlmorAsmRYEAcGg
    jGw2Q1988cVCJpPhi6I5ODJg+euBwL/es6dpGp577iVIkgzbpqhMVTjvv6+vG83NDTh5cgiU
    MlfAXvDdi65PrZYmASp8aoSU1Z7IFYqj1/M8mqOjo8O9vb2XWlub26emFsrr5Imle3tIFCUE
    Aop7BRAMBhAIOEVlSaogq97PUeog6qbpXJbFYNsETqhPanZByLIISbJZJpMun/sal7la+2uF
    1lodgDo0NDQ1NTX1aHd394729paGyck5n7B85WMWRaFqrUKhIEIhZ71k2Sm+e+tFuHEe22aw
    bQbLojBNBsOwYZoUlgVYFisXfZYXfIBIREJTk8heffWM6hJxeTP0vB/0CqPL0EG/X5BW6wBy
    lRKBMVYH4OMdHR2f+/xv/Frd3XfdWQnhWTWLn1cF5mUSwqEgPvGhn4VaKsW++s/fOZAvFIEK
    46PFJYMilmtNrTVayDtBxlXePdKDEJxWUIVDD3macBHAuwb61r3jtz776eDNB/aWaddrJ32+
    ZA/VkhN+PUIsYxrFsufhtrShenpWqJEUrrYh58XFfzQ0PLL7//3rL76lr6ebdHW2g9q0KpB3
    CB1tzMzO419/8ChuufUwJiYmoJcKuPO2W8oJTpnhyn3vjAmggpNM6rqOqZk55homPhn0aHs9
    hHAZU9ON1i7qWRxRIOjq7HDfKy0TyHjSB7lcDidPnwUARCNhB0GlldfB/TlKKRi1y5uc3/DE
    Q3lWdn7Xu23Um5M7N3ppbOHCxaGeDf19YAKBqmpYSi7h0vg4zl8YwvDoJUzPzCKTzUJVNbfF
    02HaUxQZomfQXVZb27ag6wZ0wwCjTsIUCgaQiMfQ0dqIvu4ObOjrRmdrExKxCBRJgGFZmJxb
    gGlaJpf81UoGV03ncoXqqIcSTlFKH3v8iad2P/S+BxOHbtrvagEC2WwOMzOzOHfhAs5dGMLY
    +ASSSykUi0UHQRYEBBTFDRgEiO558xITSiko9Sr0FLquwzBMMACyJCIei6ClsQE9Xe3YOLAO
    fd0daKyPIxRQQAjByPgUHnniGLMsuwRHhL5Wi0wtdHDNkUHu7xEA4i//8i/vjMVivz88PLrv
    1VdPwTQtiKIEv4xArSA9kynhzJlJYffunt63vOUt1pNPPmnMzc2pjDHd1ZOk3FmrSoKvY8cM
    5daf74SRUZmDlN1kCy6CaAGwb7311o5du3Z+ZGEhvf3s2WHousElg1czC7g88eNJLa5WgsJn
    wwSfT1z1Apj7P3ikIKWnnz76k3e84x179+7d8fZstiBmswVfkF0eF0coFER/v4MORiJhxGJ1
    OHduGF6rXjXiQ8ozhYAzXxcOyxgevmi4s4O8ppnnC/+NFEcZ8vnCsgTPS4pDoQCam+vLTJsO
    URGDIACEsPLX3s9X6w7yyQFu1JZR/jxa4+Pji4ahn2ltbbopEFAkBxklCAQCiEbDqK+Po74+
    gUQigqBLPidJzhH1kFVO8rh8VgSBQBRFiCJxUVYnKTQMClW1USjYKJUoTJPAtp1lURQBgEkz
    mYxRwxcavsLoasdavIaxyhgrvPrqqy92d3e/tn59711zc0uwbQrAeZ+hUAiJRAyNjfWor48h
    EgmVE0CHzMiGbXsxePXZ9JI6zwY5ayc4WtCssm6aZqFQcC5NcxJFr6jV2hpCoZC0zp4dLFBK
    /eig1ymzIqnaqiKEvNMhbmMsY6wDwKe7Ojs//Vu/9fnGn/vAQ047mm1XSmqsRn2NVD4ehzky
    gV/55IcRj0XiX/7Hb90yv5CMMcae5v6u19YocW/eWgu0sEZgRfnDhwrtuASHBMAjAhC511H3
    f7u/qbHh/b/8iQ9H7zxyCEBFX5CxK5DE1JgjZF7iR5cTz/hb2fjEkFLmd4JkhWu1W0Y9cfFZ
    AN947Ikn+/7HX/71pl/7lV9COBTE/Pw8JqemMDs3j1QqhXyhgNHRMSTTOezfvx9//ddfQDK5
    hC9/7RsIuEYtoCgIBRWEwyHEIiHEo1HUxaNoaqhDNpfDxMwcUBlo9ldEczdii4xvRqmyFV2D
    UyWDUJXREYTDYYRDoaoWB8J9up7AhAACRtwLpMzKSgggCgJEp2VieZRxAzhDr22UMXYpnc5c
    euXEaz2bNqzHq6+dxAvHX8bguQvI5nKIRsJobKhHU2MDetd1QXCr717SZ5imU4hABU2XRBGK
    IiOoyGVjbpom0ukspmbmcfLMRRRKKuoTMWwaWIcDOzehq70Z84spViMRrJUQrlkhyw3GvXb3
    42PjE+cff+Lpg9u2bML4xASeefZ5HHvxZYyNjUMQBTQ3NaKlqRE7tm6GKAqwbRuarqNYLKJY
    Uh001bZhlTU/nfZiSZQQCkoIBQOIRkIIh0KQJRECIU7LdiqN546/in/90ZMIBhQM9Hbhpt3b
    sHfbRjx29DhGJ6ZNVLOKlpNBjl57tWjKr+oscoGLEIlEpI9+9KP76urq/uP4+OQ9zz33gpDL
    FVwiGeGyNO/8c/PzWZw+PS1u3961/oEH3ia/8MLz4vnz521Kqe0WP73CqIhqwrXrlRjyvtAr
    lAqcP5RRPYfHALDe3t7IwYMHHywW9cNnz44IpZJeRYe/ckInoJpN9HJi2ldHPONJPzAv4q+O
    UMgank8LgD42Nrbw2muv/cuBAzf1Hziwa/tLL50mpmkhHA4jHo8iGo0gFApClmXE41FEIhJO
    npzE7t27EQqFsGFDp4uSwSPNgWU5CIVlMReloG7AK7L5+Xndtu2VfGHhRvGFhBBs2rQJ58+f
    r9kt40dO+ZthOGMRfsSvQrqDZQG9hxZ6r3eDfh4hrOUDr3cLKQNgJZPJYjabPVVXV1dqbKyP
    U8rQ0dGC1tYmxOMRAA5DraYZyOVK5XjQa6115i6rz4/X7u/cVxJFWRYRCEiorw+gtdVJKlWV
    Ipezkc1ShMMiLEuluVxeuwI6SNfSF3odM6OjozOLi4tPdXS03tTQUBfN50toa2tCV1c7mpvr
    EQgEYBgmNM1Asaghmy2V7YbTVSWVi6MOGFFduHGSa6uMEnoqcs78roRIREFDQwiiKEDXKXI5
    E0tLBiglaGgQ2CuvnC+l02mvSOPN0Ke5LpmCzxesfkLoczSCyya6C8Cvbtm8+d2//du/GXn3
    u94JRhkK+QIYsx2NKUodFsMyKlF5TMuPbRAwxGNR/MKHfgY9Xe2hL371G/tOnb3QYtn2M4yx
    F9wKI08JXeIM1apXrmokhXzbjmdp/DMI/tmDjbIsfeTBt9/X+q4H7nVbzXiqen+bpy8xBE8g
    w7GPXqnVFNWIInVnylZI0df6ZsNppy0xxk5pmva1f/inb3xqaHhknSxJZHZ+HgIhCIdCCAYD
    0HUdr506g49+7OOYnp7G6OgwouEQHnvyGYRCAfSu60JAkcvBvWmaMAwTumGAgMCyLIxPzlBU
    9C49xiZP4JqfmbBuBCfoC0TL1sa0LCSTS5iYnIKqlmCZBkzTgm05JDK2ZTlnixAcvuUghkcv
    YWJykkMGqyn+y57NTQIFQiAKBIosYSmdBgPjg1GpxuUx8l0P1N4LSFOMsXPf/PZ3Dz/9zHOS
    rutobWnG+oFeRMIhlEoqUuk0Ri6NIZvNgzGKQCCAgOLMXIqigJm5BVwan0Rneyt6ujpAKYVl
    WTBME7puwLRMBBQFTQ0J1Cfi6OlqhyyJyOYKGB2fxouvnoUoCJiaWwQqw/Pli5ubWHXWPn8R
    wU2cDThad1OU0he+9/Aje+bn55WTp8+wQCBANm9cjwff9QAYpZidX8DMzCzOXxyCpukIum2g
    oVAQkXAYTQ31CAYDrt6lY88s03KSRlVFvlDE/EISqqajpDrkBs2N9ehqb8GGA7vQWJ9AOpvD
    uYuj+Mo/f5/9HQFJZXLMcmQmvEpoEg4D9dJl2NTYWp5H/iweOnQofOutt94TCAT+3ejo2E3P
    PHNMWFxcgiBI5blMH5dRVbDpBbOOmDMwP5+FZUHcsqWj75577o329PQ0vPTSS8eSySQYYyIh
    hPeDfmKiVV+HGux9vB4wHyQv6zbx9Bnf/va3H5Yk+b2nT48EMpmcL/gUriIxfGN6hBWGTZSr
    9oIgwLIsRrkq6XVCdzxyLPXYsWNnY7HYVzdt2vyL8Xh0wLJsEo1GAACW5aDvgiCgubkOp0+f
    QigUQnOzI7fU2dkAShnyeQ22zcqohKd1xmsSAgZLJpMlrsPJjw7yvvC6kBnxvuTChQvl/WXb
    NhNFEeFwGLbtyOOIogRRrLTuOfciLMvC7GwSDQ1xVzBd4AhBSBXbo1+OwZEnIQgEJG+/CD4f
    yPtEwe8DV3u9+PPojjDZjDFjcTE51NzcOnf48N64NzNZKKiYn0+5aKHT/hiLhR01AOp1dzBX
    u1JAIhGGKArIZlVYFuWSHsHlegAsi8IwGIpFG7mc5bYiS2hpCaCtDZAkAem0bum67s3O8/5w
    VYjVaq0Rt5f8BDPF4eHhlw8fbr+0b9+2HaIoslgsgkJBJfPzKei6iUBAQSQSQl1dFMFgwEX4
    KGybuoUWG7ZtVaGF3hpJkghJEhEMSpAkwU0eCSyLueigiWRSB2NAJCIjkVDQ3BwAIQI0LWON
    jIzwieCi6w9TWN4qekWGVulaL6i72SMA3hUKhX79jjtu3/Wbn/8N8fAtt+DUqVP4whe/iFw2
    V7Zxy/W/OJp2juq9sb4On/jQ+9Df04kH7rkdG/vXif/0nR/0/uCxp1tm5xcHGGMPE0LOY7k2
    EJ/xryoD6QpJIfNVS21/guU6wQiA9+7ZsXXvRz/wXkQjoSokolZrKGo+T5fPFfrWeOVWU2ft
    DdOEpuvAddCu8q0j306rEkJyjLGn84WC9fgTT31o08b123/7Nz5L1vf3Ih6NIhwK4vuP/BiF
    koabbroJf/M3f4NCPo+GRAyJeDPy+QLuuPUg3v/gA9A0HbquQ9U0qKqKUqmEEycH8Vd/+3Wm
    6YaXDHrtovzsRIFDHm4kdlFXCpAIjLFGANvm5xdCv/N7/wEBRYFl22UG0VoZPiHA9x95lNux
    /IwmygQhlX1Uae0WBQGabiCXLwhwZoK8VrAgdwXcizdMq+oUa80GuALsz05NTW/dMNC3o2vT
    xoa5hQWcPnMODAyJeAzrujqxd+d2dHe1o6WpAZFwGKFgAMGAjEwmi//yZ3+N2fkF1MVj+I1f
    +gi62ltQUlWoqopiSXVaKucWMD41i/GpWVyamIaq6QiHAuhoaUJPVyuee/k0XUpnDVQY+njx
    XQ1rOPe2AsGMQQjJM8ZePHvu/L2D5y+wD3/gZ8IHD+zrO3f+In70+JMolVQ0NtRhw0Af7r79
    VvSu60ZDfQLRSBjhUBDBgAJJFEAAWLYF27IguzOolDmSJ5qmoVRyEsNMLofp2XlcHBnH0MgY
    Tg1eBABs6FuHIwd3o6O1mX75n79nllTN5lpivGQwSQhZQkVmoiab2lqdUy+x+djHPtbZ1tb2
    YcuyfuHkydM9L774MjFNCz096yDLipt0iNy9uOL8EiEiKAVyOR3pdAknT06Tvr6mlk2btt/f
    3d3dd+rUqZ+8+uqrr+i6LrtJIe8LvQLDmszWr/S7uX3m7bVlreXvf/9DbYlE4sPj47Md09Pz
    4Nsba89ZoubzbyZZ9P6e0/4mwDAM27IstkKitlbJdbmVzbbt/I9//OPnstmsvX///g8TIu15
    +eWzJJ3OwzQtWBZFd3cbZBmYnJzA9u07EAwGUCho0HUDgYCExcU8hocXIIqii2I4lyxLqK8P
    Y/36Onb69Fl1bm6uyBVHPWQwVwOFX9PRiVqtme7fFu+4446mUCi0XZYV5e67byu3slf0JmuD
    vQMDXagmTOHvBVQrSlQjh05gT8X29vbI9PR00LIszweGfD7Q5n3gGiP35WTn7Nmz4z09PY8B
    sJPJ7AZJkiQnqYnBsihU1UAmU0ShsIRSSYdh2LBtlFHA9vYG7NgRBCHA4mIRly6lIAhi1T4K
    Bh2EKxyWEYlIkGXBlfViKBQoDKPETHPBXliYK2maViVngoocm4G1lzPhY0/t9OnTY/v27Xux
    u7utbnp6dvH06aFNgUAg0traiHg8AkoZCgUVc3MpZLMllEoGTNN2k0JWNbPqdTp4c7qECOUz
    KMsSFEVCICAjGg0gHg+gvj4ASRJhmhSZjIlLlzIslbpotrVFaTabKy4tLS2holuedH2hlxC+
    rvle6VofTBcZfFdLS8t//dhHP9r96U9/Cj3rukFtC6FQED3r1qFQyIPwel98C2NVouI8RylF
    XTyGgCK70DXDhv4e/NYvfwS37t8V/l9f++ahF0+cCTHG/hmASCo8roQ7AGuyoWodaF9Q6g9O
    iMvAujMeiz74gfe9U+nv6XaSQR75WyYqD19S59cj9M8LrvC8T7KCAFBVDcWSBp/R8ovQrVWy
    6LXKaHBY51KMsRcAdAQDgY0H9uwO9PWuA6MU8wsLeOLpZ3Hr4SOYnp5GNp1Ez7ou1MXj2Ltr
    K37y9DFkc3nE3OSxklBTCIQgl89DEAhFRWai7Ai54XkPeV612a4rnTU+SPACT3evBwFsAfCR
    7u6uhx5817siW7dsdip9HMJFCHFbPUkZ8Vsuxs7PG7Ly/GBl9suCbdkwLROmYUDTdJw+d1H4
    10efbF5cSuvuuqXgyMMkuLYFoLqdm65FcsgHVACOM0B67dSZ/7ywmGzYv2cX3vHAvdi8fgBd
    HW2Ix6LeDK3zI5QiVyjg6HMv4sVXXsPI2ATuuPUmnBq8iO/96EncfvM+7Nm+GR0tTeWZS5s6
    HRCGrjvyElOzGBy6hFPnhvH0Cxcxv5iy3YQm53OCvCO8HjM55XYZxpgO4AyAP2aMRV565dX3
    nXjtdF9nRxseuO8u7N+zC/2965CIx5x95bb3U/e9U+qY3YXFJL77yGOYmZ3HXUcO4eDeHRCJ
    ACKKDoFMNIzW5gYQAHu3b4J1pwVV1TA1M4fT54fxzAsn8Jd/+01oumHpuuEVZ5KcE/Sqovy8
    xJqvn39m8LOf/ez6eDz+71Op1HtefvlEeHDwHHTdRE9PDw4d2o9gMMjNbPGJCh9w8kx9jgTF
    2bMzyGYdAeSLF5NIpfRAX1/9zptvPtzR0NDQ+Pjjjz9hGIbsI0HhbekNwwTJGOOLtASAsG5d
    z+2qqt82OjoF07Q5EobaM4OCIEBRZJimjavVRbs6dLE8D8VKpWKVrfLFFGu1x8ojKISQomVZ
    8nPPPfdyIhFv6evbsGVxMRVKpfIQBBHBYBBdXS2Ynp5CIBBEItGIfN5BcRYWsmhvr4eiSDBN
    J8g3TYAQCkFwGgQkSUKxWGJnzpz1AnU+KfSKpaXr0SnjTwQZY+WRlg0bNgTvvvvuHfX19Z8o
    lUrvOXv2QjCVSsO2XR/GUKW7Vy2oLvh05MSq+wohiOBDGL2AXkRDQ1g4cOBQaywWM1599dX8
    0tJSyiVUrANQIITo7nk0/MXRNUQNGQBrbGws+41vfOPv3va2t4e6ulo3pNN5zMwkkUzmkM+r
    0DQTlMKniycikYiivb0eHR110DQLpmmjtTUGTbOQSmkoFq3yXiLEhCAYEAQRsiy6iaGMRMK5
    GJPw+OOn1AsXzqcsy1qpOLqWoxO1Yk+9VCrlX3jhhX+WZenFQqHUcejQLb8SCIQiCwspnDo1
    jFSqgFJJd/MTb88I5T3itbA3N8fR2dkATbMwNZWFrlsuwgwYBoVpWlBVCkIsLC3pEEURiiIh
    FlPQ0BBEc3MI8bjNXnvtovbSSzMFQRDSlmWlOHRw0X2cxfLRpisWRleDVKYjEAh88ud//ue7
    f/d3fxexaAS27TDyrV8/gN/8jc+VWQvL7aHMDTZtVwSbbyX1AgtGQeAEG54kRUBRcOetBxAO
    KcJv/+H/3DI2ObMfjt6Rd7h4spbr5ghXShK5ACIE4B37dm3vu+vIIacfnfqTQV6vEZiZncf5
    oREc2rcTwYByBabRlVtNl8l9EKBQLCFXKPLBA63hANkaHkrvcyyhMncyVSgUtXQmE+hj3aCM
    4qlnjmF2IYn3f/ggvvK3f4vbbzmIUCiIf33kx3ji6AsghGBDfw9My6y04LrrQwhBNleAbhgU
    FeKDYo1rmYG6HoyFXlmSECIyxsIA+gHcHwqF3nfTTQd2fPqTv6C89f77nNlAvv2z3KK9nGAG
    LnEMys/bLrOt+33QSmu3t69ohWzm3ttvxpYNffJXv/m9dYMXRsOWbccYYwoqBFA2AMVtidQ4
    p+hPDlcDweDbQIqWZc0uLaWyH/vw+/Hrv/KLCCgS4BLEVL1HSkEIw7kLw/jD//ZXEAUBh/bu
    xINvvxcNiaP40RPP4tGfPIP/5/d/HQf3bIdt2+WkEMyR7+hsa0ZnaxNu2r0FqUwWv/t/fxGX
    Jmctn/PzJ4RXHABfC0cIZxbhVcZYx7kLQ7mbDx5g//UPfof09/aAgLktMSaGRy/h9JlBiIKA
    Pbu2oaO1xVlPSvGtf30U33vsKFqaW/HaX/8d/vQPfhN96zoBxlAslvDamXOYmVtAd0cbtm3q
    RyjgzPhuHFiHjf3d2L9jE377j/6SHX9t0LMB3tzgEtcew7fI1JyXWMtz+r73vS+WSCR+aWZm
    5qEnnnhCnpycASBAFCUsLi7h6aefhyzLNRDBysUjg95zjBGUSlaFGAsES0sqikUb/f31TRs3
    bn7L9PT00smTJ1+r0WlhX6fA6qqQQ8YYHnzwwYSiKO+cmZlp4FtFV5aWIGhsrEN3dwsuXJgs
    C4nXptBfSedLWFGCQpJEEEJZPl+wOPTcXsEnrgVKWG7pJoRIjDFJVdVpURRKoVAwREgBhAho
    b29CLBbAmTPj2LZtG5LJAlTVwMBAOzo7GwEAuZzKoWakimxHlkVYlklN09R9xVE+GVw2o7va
    Z8wv4eJldVu3bo3ccsstG5uamu4H8N6pqZltJ068Jo2NTbqC4kIVCr+cWEbkEj6RS/qqX3+l
    187NldDdXads3ry7t6enN3zp0kj00qVL0szMrKlpmu6SmylwJBX0tfKB/m4ZQojNGDOWlpYy
    gkBmKGX2Sy+dk9LpfBm5qrz/CrIliiIGBlrQ39+KbFbF2FgKhmFj/fombN/eitnZIs6eTYLS
    5efJsoB8nqJYNJFMUjQ3M3R2UqZpqm5Zlsr5wyKAkqsioHGdDWvd7s7gSnUwxrQTJ06MAcju
    2bNHrK9PGKdPj+DMmVFQivJeCAQUNDbGEYmEUCqZSKUK5XnKSCSIvXv7ceDADqTTGRw9ehoX
    LiyW1ykSCaC+PgTGgHzeIZBhzEFTUykDmYwN0yRoajKpbes6pbRAKeXbRf36u6XXG0dc0xlC
    SikBsLG9vX3ze97zHsQTCdimUZWcCEQAExgIKatTORC0IDg0FZSVJQUqqJUzY0HtarFoL6nZ
    vmkAO7esD4xNzqwHcBEVLSPd14JFcAOxYDHGIAgCGGO9oWDwrvvuOiw3Nza4ifHy9k8+wfve
    j57ASydOY+fWDQgG5AqyWiMZXEmPsFpk23leAEE6m8NSOsezw63kBFez/aNWZdRrsRMBGCVV
    FZ9+9hhOnTmLhYUke/QnT+LmW24l01NTeOWVV1DMpRAOhVzmMBMA8O3v/wg/OXoMsiQjGFQQ
    CTnyAPFYBGfOD8EwLXoZB8i38l3t/37NjDr3dwhjLAigA8BWAHeEQqE7du7Ysfm9730w/O53
    vQt9vT1O4cS2K9y9VcmcO29KudZsLzn0Cgu0um2buc/B91pvD8WiIbz37fdg55b14g8ef6bl
    J88ebxgZm9pcUrVLAM651xBjbNKlyve3hVirnAR5iY4JYMamdGhmdn6nbZmgksAxq3I6e27B
    pLmxHr3rOjE1PQfDtGAYBjTdqeA5s4Kx5efKm8ktJ9QMpmEilckxVM9KlIsPPidor6UtquEI
    PW1E2U24zqbTmbeYphUhBLAtx04df+VV/Le/+CIisTqMjY3h4N7t+L3Pf8ZhrLVtzMzNo6Gh
    Efv278O3/+UbyBcK5XV96thL+PP/9ffo7R9AamkRb7vzED74nvshCQJs6iL3haJHwMMLlnsM
    aiksnx1cRiSzhq2iAICenp71uq6/7aWXXpLHxsYhCFKZec8wTCwuJkGI5Aph83NNknvxj6u/
    9l7LiyEbBsPkZBH19Y1NfX29u0+fPj1PKaWudI9XkDFwHUh2Xs/y9fb2brIse//MzCIsi7lJ
    ykq6XQIkSUBnZzMCAZkjAFmO/r1RlDAQkABQmk6nDV9S7feJWG1/6CYIfLFGACDEYjFNlmWp
    o6MZiUQc4XAQ3d2tmJ2dQiAQQHd3NwoFw2HydWMOAOjubkRra507E+ZJBDCYJlBXF4QgGPws
    PS9EX8IKs121yc2u3Rnkk8F77rknNDAw0BWJRLYGg8E7bdu+Y35+YcP58xdCFy4Mo1AousmN
    eBnSJnCyHUAt3Te+LXT5a6uF6U2TYnw8i0xGFzs64m1bt+5v3LZtx9ZiMX8pnU4PplKpc7Oz
    c8PDw8NTtm0XXQkWDctlhlbTYPHsv+rs7OyZhoamTDgcbE2n81yRYLkAPWMEuZwOw7DKBCmS
    5HzfNClU1eIIfIRlhRf+PCuKgGIxZy8uLi5LBn0I4fWUDGJwGH69uXp5YWFhwjD0kWAw0OmR
    vxAiQFEkbNvWi5tv3oXGxnoMD1/CsWPnMDeXdYssEmKxELq6OiEIAkIhBYJAwJij97llSzO2
    b+9BOBzBmTOjOHMmiULBrrJNoZCIVCpl5fMF71xmPR/oJYSojE3UTKYvt37XmlSGAIgHg0E5
    Ho+DkyTmEhVaaRXlUSvq1yOkFbkF+PX2AF7MXhQIgsEAAEQBNHuGizFWIIQE3fdJcOPe9qzr
    6lh/YM9OCMTRKKuJ8rmshrPzizj6/Ms4uGcHEtFIFZrIrxn861njeT+qSMEwPZ9E1hHM5itX
    Fndd03bJGtTePOkA/zUvXmzPLyym//S//1XJpkzQdS28bl1P6ODBg/jiF7+IXC6H46+cXPa3
    hkbHlv1tSRSZ6M426YZpcYiWWeP9im6Fj59/8bfR0lptINew2icAeKuiKL/e1ta2cefOHQ13
    3XWXtGP7dnS0t4ExiqGhYQd199A+t92T2jZs27u33NY+33NlQqfKvZP8oYwqe62mgtt+KhBH
    01AUBMiyjLfdc4Qc2LVFnpiebT01ONwyeHF07/TcYimVyZ2mjH0RwDjXglTkijXXNFhdQVrB
    kxQZGb00ZuTyBSUUDDhC6M4PufeV5LiroxW//5u/gp8cPYZvf/9HCCgyXj45iHfefwfecvvN
    6O/pctbJL3LPJcwCgIWlNJachNBLtsrIoBsceAmyeZ2LV7wMheDa0MHFZDI9NDIa2bShv1wo
    GBoexcWRS7jppoOO04tG3EIXhSgKePt9d+Iv/uZrePhf/w8euPswers7ygmyx/yrBILI5Io4
    NzQK3TAhBQNlhumpmQUkUxmbCxrKzKLurETa/bqIq2RTW20QX1GUdblcrmFubs61MxVTVmk7
    q8wJ8u1n/Pd5kgbvcS2kTBCIR+0uhELhtkBAaVNVTQOgMcZUd2+VUD1KceNkgu7sVzAY3J3P
    lzoymUKNWcDlCVssFkF9fQxjY3OuXICwgtj86yeXIYQgHA5A11XbTQgtzifw1zX3hay6r9GT
    7/L7QgCwZVmhkiRmN2/u1W3bFgIBJWxZVuj48XGyfft2yLKMWIxAEEJVjJmRiFwhxDJMGIYB
    Sh1qakEwkU6nLcMwDF+Xlc0qjkxAhTGdcOiT3w96DLfXBK33iAv379//HkLIZ4rF4vrh4ZH6
    sbExcXExiWKxBFEU0dBQX4Xo8S181Zew7OsKKihUyVP420orLafV6ly2zTA9XSDptKJEInJ7
    ItHU1t3dum9gAKVSKfeKpmlfHndESPlE27O7q2L7a4nUM8b0dDo9Rwhm6+pirdPTSytqoXo/
    OzXloIK9vU3o6qqDaTqxwdmzSaRSWllc/XJsvoJAEAoJyOdzlqZpmr9bxi2MqlgDQpmrjBk8
    LXFhfn5+VtO0i3V1sSOBQIB4kh2KIqO5uQ4NDQmEwyHYNnXRaWf9ikUDFy7MgNKnUCrpmJ7O
    gtLKvLJtM/esRlBXF0QoJKNYpOU1kyQB4bCI2dmk4bZxF1y/l3EZRT0/mF+py+hK5+6aJYSU
    Uq9yM7e0tJQ/f+5c/batWyC4NPUUTtQoMMLRjZEyZSHhXZQbbTL+efd1AiGg3GsFgWBqdgEX
    hse8Dy7sJoZhOAO8HrPTDXdz6awVAJs3DPTEujpaPXrry5DEAMdPnEYuV8Dtt+yDJImwbXtF
    PcKVyWNqzREClmXh3NA4VE03OTTM7xSuacDFMTwJboth1HU0hEusPCcowJmVG6WUfjtfKO4B
    0AMgfM8995CJiQmMj13C/j07EY9FIUkiDMMEIYAsSTAME4ViEbl8AflCEZquQ9N0aLoBTquO
    ocIIKwMIMsaiABpRGQon7hqocAg4KLc2fCJ5TREvd60iAN5+zz33HP6d3/kdbNmyBblcFn/w
    B3+As2fPljXyeHSPUv88Kar3yTJjwcqrXi1Vwfh/BoyvtJTF7h32rLbmRnz+Fz+Id957hGSy
    OeWZ468p/+1//dO2ZDq7lTGmep+xGyh4FcvVDFb5v6MBmJhfSBqFQkFpbW4o/1lOZNjdfE6i
    u3XjANqaG/HKa2fxwisnEY9F8O7778RATxcsy3R0HQGHaxWk/C7KEquEIZXJIV8oeW3JfpSQ
    Z1Zbi0rxFVFC7zNxW5Onc/nC5NDQaBe7j5b3xX133QbbtjE3v4D7bzuAOw4fRCQcdNttCW7a
    uwN/9of/DsVCEa0tjQgHA64NIjhycC8CioRXTg5iW+9h3HHzPoSDiiO1AwbLtjAyMQXdML2E
    0GM7zKJaG/R1samtdkJoWZYpiiKTJPkyqIQfjUAV6rdcl3C5DAX/nCgKkGUBqVRJsiw7AWd2
    NwuH1ELhkogbDxpkDIcOHZIFQehUVV2pFqAXVkTzWlvrQSnF0lK+Jsq3Erp4JdIZbzYxFgtg
    YWHaKJVKPIJjYPkc9LVMBgkAMjAwIA8MDMRs2w6kUikyMzNDFxcXmduJ5fmnwEsvvTRNKf1m
    Y2PD3mAw1CsIwfClS5cQDAbR0tKGZDLnooOAKDrIMqXMZTj0WA5lBIMyDENHqVREJpO2R0dH
    dV3Xed/raUdGQqFQQ319vVxfX2fLskJ0XadTU1NqPp83XVTaqpFAXxN7xhjDJz7xibgkSe84
    f/78oeeffx7JZBLRaAx33HEHGhoauPWs1SIM7mwBy8lj/HqCy19X++vqx4wBpZKNoaECZmZ0
    EgjIgZaWQKCjI7KjsbFx0/j4uMYYU1yZCsYV3s01LPgZmUwmRQjmIpEQp7FYffHou2VRTE9n
    IQgidu7sRDhMMDmZw+xsoZxAe2iiPynkWVsVBSgU8qZt27wv5AujGq7T6MQKYvWGa9uzpVJx
    KJFoKoTDwZhpFkEIgaZZOH36EpaW8qCUYX4+i1SqWF5Dy6K4cGEeExNpUAroOi3bJ8tiuHgx
    hWz2LCIRBYWChWzWqNq3iiJCkihLp9O6Gz8VuOJohvOFPJHM6zpz0jVeQALg0uLi4pN/8Zd/
    8aFAQBa3bd0GRZbctk+PKMVBHUDd+UB+rolW5gvL6IRtg5bnCV36VtOEqmmYmJrBN7/3Ywxe
    HPVoyIEVKK1vtFs8Hkcmk5EARCLhMAnIMrdGy5M4AiCXL+CJZ17E1k39WN/b5cxUXoFptDJ7
    6E8uq5NDQoBUJodTF0aZTal3GMvioC5sbmAV5jHdYkIbgIdEUdxPCAlalqVzQV+e2+gBADti
    0ehdO7Zt6WKMSnOLS9i1axe+9S//gg/97IP4wPvehYmpaTz7/HGcHjwPWZKwb/d23Lx/N5oa
    6mCaBvKFIhYXk5ienSfjk9M4fW5IPDU4lEhlst7ZiAJog9OWabn7KSCKQkgUBIVSplq2PQjg
    RwDmONSHbwFZjfaseDAY7Lvrrrtw+PBhMEahyDI+/KEPIZlcBOHyaAIHvQMhzmFwiyxCmViG
    e65SdylnMQQcWkaZew5pFapoWZZ7mY6Uh2FCN3QEZBmNdQkEFAXtLU1Y39MFRZFlF8Vvds+m
    zQVZ4hoYd2+OUGeMzZbUkpFKZ7BxoBeGrkN12UJz+TzS6Qyy2RzyhQJKqgpdc4oHxWIRI5cm
    sGl9L3705DEkYhEEFBnhYACRSAiJaBjxaBihgALFZQ0TBedTyRVKUDXdSwh5R1hr31wPdsxl
    a8UVSBZN07w0Pjl1s2makGUJjFI0NzbgZ9/9NqiaBtMwkM3mMDU9i5Jagq7rME3LtVMMlyYm
    IQoCFFlGMCAjEgpifW83tqzvgyKLCCgyBCI4tgoMhm1jfGoWlm2bHKLqET15TIe8OPaqtPC9
    3tv8/Pyljo6OS5s3b25Mp7MwTWsFiYmVJCcqz/OaaB5I5D3mq8ctLWEEg9QaH59QTdNUwAm/
    oyL5csPeLl68SO+6605VFEUmiiIoZZdlD/XExL2EZ6XXXikxXImIJhRSEAwSNj4+rhqGUSWY
    zfnCa6q/553zT3ziE03Nzc0/Qwi5mTEWZoxplNo5XTeymqbmSyW1oOuaRilV6urqtyYSdffk
    8+o6XaeyohiYnJzEpk2bMTa2gHPnJhCNhtHR0YiGhrjj51MFzM/noWkWRFGELMsIhRSEwwES
    jQbR1NQrtrd3xzdt2sjm5+cly7KiihJojUYjm2OxmBWLxYRgMKRIkhQGBIUxqheLhfPHjx9/
    9MSJEzOMMZ2Tz/HIxK4J8sUYQ11dXYNpmj0jIyOYnp4uA5YnT55CKBQuf32lwoq/AFNdjOFb
    J4WqVsoKWUg1su/o84lldlpKnUSAUgGaZqNUsmHbgmJZZguAFhf59dq6vbVaq3NKAZgLCwtZ
    SulCKBRwOzuIy3gpQ5YlR3YpoCAQkMu6eoLgzLzJsqOVFw7L6O2tc3X1CGwbsCzAtgHbdtoi
    HdkcL1EUIElgxWLJpJT6i6N+X3i97TlfHGUApFwuP9rS0p4Nh4OxTKZYbpVOJvPIZtUya2g8
    HoYsy2VpiWp0uSIl5KwTMD9fAmMqnJqPk8Z4dj4QkMCYSTOZjO4rjvJM+CtqZF9NDHHNdQjd
    f+orzz77XGJ4aPhwa2trRFEUUZJEIggCERzZzuo6S5WWtodocOyGTjDKKKWw3LY2y7RhWCZL
    Z3I0mUoXKGUThGCRsSpZgOvdOnRZo5bNZj0jOT81M2cvpdJSR1uz07rGKq22lXZRgrGJaUzO
    zOHTH34vwqFgZd5wBVIZtqJ+IeXkBby/A7xy+iLOj0zYqMwJaDUO6DWH8N3Nelc4HP6Ptx05
    nBjo74dpGigUisjlciyXyxq5fEFXVdUUBEFsb2uNvPOBt8iHbzmI//n/fhE9fesxOTmJbCaF
    gf5efPt7j+B7P3yMnr84bOq6oQFgTxx9PrB317bAO++/U7jlwG70revE5oFeCAJAbRtLqTR5
    6dUz0lPHXmqYmJ6tL6kaJSBMUSQxFAyQcDBAwqEAgoritMqNTeHEmaF7NN2wGGPfd9scCBdI
    rxYyLQmCEIpEIuWzF4vHcM+997qtwU5hpTIHyBHIuCROKLeBLieWqSac4Yll7JVfT6mTLNLq
    v0W9og+cdslCUbXdILWRa8MoolqjaZWPHvNmAtIlVSs+89wLDaOXxjAyOobpmVksLi056LGm
    OyiX16bnhRmEYM/OrQBj+MFjRx175c5VUsogSSJCwQDq4lE01dehs60R3e3NWNfZirmFJGyb
    eq2YnvNTudlBrRbKdR1vnv3U3ILbxOTUND177oKQLxQwPjmFqelZzM3NI5XOIF8owjIth33W
    ssrMxdWosmNrJNFhnpNEEeFQCPWJGFqb69HR2oyuNme9wBhm5pMMgE4AjVUSQv9MU03NwetB
    +MQYI4888sjsQw899A979uzuiscTrQsLC8SybHikwLVb0MQaAadY87F3OcG8hGhURixGrKGh
    wYXz588vYvk80g3nA2vED7RQKF6MxcKZhoZEs6eHVlvb25nJKZV0zM9nXGKH6oTOP3mwHFWt
    9bxQRlubmyPI5VLGpUuXPOZa/3yTvhoxBmOM/d7v/d4hAP9pamqmcWkpDVF0CCsUJcAURTbD
    4ZgWj9eZABE0TY+ePHlRnpiYx44dG5DNLiEcDqOurgEzM5Po729HZ2cjo1Q3FhYmNUEQaGtr
    W6ClJRGcnc0JCwt5FIsG8nkThKgQBBGBgEwaGyNia2tXw5YtvXWCwNYDAgOIwBgRKCWkWESZ
    hTIcFtDUFL53+/bt5MSJE99y14hw6Ip1jfeKTCkNetwAAKDrGoaHR2qQwvjbRmsTy7zx1y9/
    nf95d2wTgYAIy9KtpaVUCECDe055H0hWO+70a+3lcjmVUroUi4VZX187CYWCiMXCCIeDCAYV
    yLK0Ak8CQalkolSyEIs5Egl+ZmSn64zBMJxL0wBNAygVIIqApqkmY6xWMqhd7xi+RnGUEUIo
    Y0wtFApToiikGxriXbpuIR6PIBYLIxIJIRRykmeJY3W/PMIMeEyjnji9YVBomnOpKoWmMYRC
    ImxbtbPZnMbtGe/yfOGbIqRbDZZRBqed73/MzM4+OzM7O+CiAQkAMTitnP7gj1zh9/GP+csL
    rAoA0oyVhyp5trlrWsFbhQ1HGWOvnjl3ceLRnxztf+jdb4UsS0CtWSQC1NfF8NGH3on9u7ZW
    CYg7cQ8FfFqD1eQx1CURoTVJZy5NzeFfHjnKMrliCZVe5FqB12qt6QSldHB+fn5XXSIR6Oxo
    Fzo72okoCMSyrIBpmgHLtiAKIpoa6tHS0oynn3kOpwcv4O3vfBd+8PDDmJycYn/y539FJ6dn
    dNO0Clw7mV4oFoWjx47Xv/La6ab1/T3R7ZvXi+t7u0ldIgZJFB10DMDu7ZtIb1c7KamqUNaC
    BAO1KVRNY5lcgS0spen8YtqglI67axV0W04Nju59tYy7puv63OOPP45IJFKurhCgSj6C1SwS
    VKQkquQl2PLv+x+DuokPl2iCK974k0VaJp9xCjvPv3IKJaf3nbpJocKhF2uG5nvivADyuVx+
    4X9/9Z9aGhvrZQIIkighGAxAkWUEZBk2pe6cpSvKyxHpeKy8hAiOIRXFsvHXDQPTc4u4NDGD
    l046RA6yJLFsvkBshyrZCyr92oN+KvLrXRnlG/l1ANMnTp4u/PZ/+C9xAsIEUSCxqCNcLBAB
    8VgUoiAABLBd9Nh218+bgSbEacEVRAFgDJZlw7ItzC8uYWZuAc+9dAqmaSIaCSERi2J4bIon
    4SlxwXmttqLr2SpaDliTyaT1ne9854e33367sG5d11t7e9d12LYdoZQqzsVcH8hIddN15c7t
    /IYTiLCyUL33LGOMMWYw21ZpsVg0Tp0azQwODs6qqrqIalHi68lYe1U3d3yCvPrqidduvvmW
    ZzZt6nmXJEmiphm+EIG46+K0nU1NJcEYQzQaqkEGQq6oWVgteF8hvojHg4jFBPvZZ0+l0um0
    J2nCtyeXUE3acE33z+c///lZRVHORCLh/ZqmB/P5glAqqcQJGQRFFEXFac0DNM1AqaSho6MZ
    LS11uHBhEBs2bEAsFmVbt66jqpo3zp8/VTh//nw2mUwWCSFaW1sb2blzZ11vb39zT099LJ83
    xULBJM48WGX9MhkDqkoFSRIFQWDlpNkjFJFlgSkKbEmiZqmkTc7NzS25tt1kjPHSJ7iW9n1x
    cbHY2dk5v379+qo2x2oUWfDNlF7917wwfXXrslBFjnK1v8f7XlOTglxuQc1kMozzgbxeqLDG
    9p1SSk3TNBcaGxPZ3bvXRxhjEmMgnlSHaVorMPz6JedI2WY56L7ztSwLUBRvVs7jpGMIBBhz
    i/VqjcuoVeC7QXyhWSqVMgBmt27t3dHf38EIcdbLspz4wHlsltfILcwnVwAAE4BJREFUb2f4
    41Ah4al0eyiKiHhcgCgSlxSRQVFEZDIpS1VV1ZcQ8iRPNUdNrtYXXtOE0NUUoowxixCyyBg7
    DuCSmxA2uhWRhHsQxKtMCC+XHFIOjfFaRnkK8tclyngdKqIeQci5bC7/L3/9la9/+uz5obrW
    liYILkGHwNFBe48B4F8ffcr9PQxV0UQZWYSr4cg4NkjqEoa4wa3t6KXZNoVuGHhtcJi9cna4
    CGAGwAIqlO5ZV4dv1fS9XC3GVzRN+7VXXzt5y6uvndwmSVK3JIoNgiiGRUEIEEGQRYEIhAgC
    Y4xQRiXTMBP79u8PjF26hMHBwaKu65lkKq1yCEyRc+AGAFYsqZGTZ843nTp7oVmWpZgoCAGA
    CKgg19SdCbTBmE0ZMxljJqXUsCktUUrzjCEFYBbAKIAJn1Enq5XcuAhE3rbtH3/ve9/revTR
    R8NcYqUATAKDxC6bYLHlp6nqu6z6hWyF71a9gtU8pOWBRQCWbVPbplPe51BValy7yihv3Ocp
    pf9T1bR3apr+TsaoUiiWUCiWwChDJBJCY0M96hNxxONRRMJBhAIByLIEUXBEij1SBk03oKoq
    8oUSMrk80tk8cvkiFEVCMOCw2ZqWRWcXloqoaObVCjCrnOD1SGr4v1ljuP6Uqmnfam9rOdze
    2rJhcSmFVCqNVDoD22blVqNQMICAIkMUJUiSXGl7dBNsy7ahqypUTYeuG+WkUZElRMIh1LU1
    QRJF9tLJc/ZCMq0ByDJnrXJca8yKKM31Sga5daOzs7Olb37zm0/U1dUNRSKRdYSQVkppE2Os
    3tUmUxhjIuEiBL6C7JvtZdzXzC2yMNu2mWVZtq7rlmGYOsA8/c8kV9Qr4TqQM7yBPUeeffa5
    dCQS/fKWLVtCO3b07bcsGmEMAsDcYJkfuCRu0b62yLiXVPtRDT6G9Y0slx2naerma6+dTJ4+
    fXqaMTbP+cM0qscXVkPOinz72985c9ttRz6fSCRu6exs2wZgHYAGxhBhjAYYg+LEDsxbEzEc
    DieSycVgPB5HJBJRh4cvpsfGxkpjY5f0VCqtUko9dL00PT2tz8/P046O9kh/f39jR0dHS0ND
    LC5JkuLMC3uyRmCATQmxbQAWIbAYg8EYNSi1S7quF7LZUiqTSc9NTk6NDg0Njbt+UILDhk1I
    9ZDsNYkVmpqaMh/84Acf27hxY0t/f3/Atm3FtqlCKZUZoxKlVHL2jTfMTa7yd1c4xVaIOy8b
    mzJmgxDqtz/lLzSNWUNDF6dVVTWu11msRZpy/vz5Jzs6OuREouFDsqxsdOaRJUiS07qt6xZ0
    3XTF1m1YFgOly5MZURTKlzPzJkBRBAgCXFF7J7mZnZ3Qc7lkYWFhYQmV+Te+5fG6jU3U8oX+
    2fqxsbHc1q1bfwBIyOX0m2VZjnnIoNPuTsvcDc7l6H1WbJAnSF998aRPts2g69RtvdVoqZQz
    p6Ym84ZhZDk/WHAZ29VrsWbkGh3QcqHPDU4jcAWpGWN1AOrdRDABZy4rcBWB8+XIJZgvSLFc
    p5dHhXXH22R+barrHixwgT3c9Uq4Bv8BQSC3CYLYRIAQCIIAFALIgNfYXgGDqteqqu+WrGDI
    OIIw8GEGY4wx26YlzvEl4czFzQKYI4TMApiHQ22bQ2Ve502vJxcFBVAhcKmHI+ba4D6Ou+iy
    hyjJANolSXrH3r17m6amptSZmZlT7v/nzaJ56HEZJURlpk+EM2MT5vYkz2LKy5b424Q8shj+
    NR5jJi9i7808sWu5Tq5uZSsc7cEWAE3u1citVfB1JlrsTb6OXcXXphusegKqCwAWCCF88aHg
    fUarcU65QoxMCIkyxuoIIXcyxt4HoN3dc2EAgVAwIIfDYTEUDAgBRSayLBNZEokkiqSij+ZJ
    SzDYtg3TYexjuUKRpTN5ZloWA8AEgVBJFDXTtCaZe6bcszUNYNo9XwvuGuRvFFvF23ZX6qQu
    FAx2t7Q0fiafL747XyiETNMSCSFobGggXR3taG1pQkN9AvFYFOFQEIrsts44GaHLamigpKrI
    5wtIZ7JYWEpjdj6J+cUUGBiCgQDq4lFWUrWlTK4w467XDIAp95ohhHiFq7R7zuj1XDOfHwwC
    iHL+r4EritbB6ZbhC6Nl+rTX6Qe973kzLkXX55WLoy4TawYVWQ7zRvGDvC8khMieiHdfX99A
    d3f3TkJIh2ma9ZZl1tk2TVBKwwAkZyTanT9xvSLxlFk5mdaqQpVbnOLwVQ/tdxNsxxVSSlk2
    m9Pn5uZypmnmXJ+y4Pq/OddeLXL2Sr8WNt5XJFYABERRjMZisXpFUeolSaqTZblBluW4IAgR
    URRlACKlVKqvr289dOjQA6VSqUOSJO306dNnXnzxxQXDMDxfaKAyb8QnsyCEiIqiBIPBYDgY
    DEYEQfDI+DwtZwMOW61GKS1ZllWyLEu1bVu3bcs2DNOyKzO+vExFjgtYS7z/faPrxPvAvr6+
    9sbGxn5d15s1TWsyTbPJtu1G27brKaVxxlhgJR9Y4+9fFlHxJ3h+AjauVrPCi5jbtMMMVVVT
    pml6/m/R5wPTq+0DfbbKk7CKd3evW3f33Xd9OBaL7iNEaBAEIUYICTIGhVImUQrRK8gwRgTn
    3BLCE/Tw94IARgggSWCCwBilNrUsm5mmZZ85cyb93HPPjluWNc8Y4237DCFk3l2XjLuf2PW2
    V76WWYkxFgsEAo379x+8Zffu3f9XPB4dkGVJEgRCTNMmxaKKUkmDppkwDMudIwWHvotuQii6
    RVRHgF5RJAQCIkIhAZLkkD/ZNjA7O6//5CePziwuLsypqjrDrdc0IWTa9Y9J98yZb3TNrnVC
    SFxDEnCTwogbbMfcID/iBlvKm0RSaiWEGipaXl6rY9FXcaA3iiPkjL8CIEoIqWOMNcLRlmv1
    oaoxVLfZkprl0OWfK7sCwuoPLLx1zKMi+uwZrEX3OQ95NXGNEx1334QARNz9wl9hN8jyUDgR
    wBZCyMdlWVZM03yNMTaIyiCIxxia44oCuuvgwBjzksoAqskX+BlAj1nVq67zOnl8yzKvWZgn
    hBS4fadey3VyA6cQV3BpcJPBBl/yrFyD83UtX+/tLz5g9YRUlzjErLia55SzVSKAMGMs4a5f
    G3e1u+evHtUMjXxbj39QqVaA7l0ecY7u7scUlxDPA5h3AwLvfJVWOyB4g7ZdcW15A4DtAHYB
    6HTtVQMhqCNECBFCFEKI5PAWEQEOfxEq6A4HKztxEgVjlDJG3flKmyvKZLgi1QIqybQXmKfc
    NdWud+BQY61Cru2KA4i7yU7cteeRqyyMXq0fLOuKccF4ziXeyXFIoX6j7K0a6ya5tj/uJs2N
    7sUXu2Kuna61buR1xDjsKuIKT56mLPxMCEm6j9Puuqo88vpmE0LXvgsuehziCpZRbt9E/L5w
    69atA29/+9s/IYpi7OLFiyd/+MMfnimVSuD8me7zhdrr9IW12MZr+cLSCr6wSh/0zSaEAELu
    Pkm4fq/JV3DhfSDexNlaDR/I68elavjAVS9u+W0VYyxGCKkLBoOtsiy3E0JaFUVpDwQCrbIs
    N0mSFJNlOShJkiRJoiiKkiAIgigIgiA4xCDEQ+zdrjRm204Hg23b1LIsahiGreu6reu6kc1m
    S6qqZrgYc57zhTwSr91gfhAARMZYGEC8tbWlc/v2HQcCgcB6QRDaBEFoFkWxXhCEmCRJAVEU
    ZUEQRVEURUEQiLdYLhZdLhxQyhillbWyLJOapmnrum7pumal05nShQsXFlVV9YpTc5wf9JLB
    jLu3rDeKEF6TllHfoKoX/JQ4Y6K5huFqWc/IVSISjAu4PKaiWqx9q9HWca02mufIdbeSlmSM
    US7RyLrOwHOCl0sA34zR889kZrwqs4u28tomq7WW5c+UEGK7M3m83pPXHy2493OMsaOuVtKI
    +xrRFxxlvYPiEonYXlWUMebRaHv7UuH2JfMF9LwIsX/flTULXSe7akQOHvmC56AJISXGWJZz
    Nqr7fuU3WfBZDYdocwEDLx1QU0x8lW+e4Ky335fciq7u/m+LblAR4s7eSskguUJw6Ze7yPDB
    AJYLqt+os14UgEkIURljw+7/7jHGNjOGRsZowj1PEle4WYkZhK1gzym3Xt7n47VBJgGk3aKf
    6t83N0iSwycUJfc5j9nWs+khnx98MwkhPzqhwyEp4glQSje6H+Tej+myU5bcQFVwnzdcf8gX
    lK+1LyS+BMfgk2tU6N39/ATXrBDBxVHMjQ0oHO0/2+cHRc4X2slkcnF0dPRZVVVx7NixoVKp
    ZPp8oebzhTrnC70ENMD5Q8UXo/F+0PadUz520Lmk0e8z2TXcJ9Rl5jQAeD4QXOzp+cA3+3eu
    9estrmiT5fZWCdePUZMSQgzGmKaqalZVVYGLnzJu4sF3UV3OD17JF9IaxZaUL6HhidVuDMPk
    a7N1ZVWM+fmF9Pz8T16CMzrUAqBZFMVmQRDqBUGIuMVR0b0c0veqFmpGeHSZMcoodVoWKKXU
    tm3GGPOKMV7ssOTGJzxAs2ye+Y3Yo9UilbF8RtU7AJJ7SCW8uQHaWhuMR3RMXyWL3mABQ631
    0rh1YZxhy6LSKilcoSL6RtsD/U7Qo7PNcxXmtRB95gsKujuQTjhDyutKiowxA8D3OOclcOtp
    uHo2Oc7Q8AZXJIR4VdEAgICL1pbXmdMH4tfInyh6+1vlAq/VZsei3N8lvq+zXEC+mnv2jbyO
    wkcO4gau12vWlw/aRbdCz6O+UXct30hCWCvJ8QIzLxjIEkKyXECg32gJYQ1WOtMt8BXhtM4I
    HAKRR6UVUn4DQYP/jFmobvv2ggfvPKu4DnqNVxk0eHsLvnUruPZKwcpI1xv1gxaXRBtcgM7L
    BV339qur8IUe0kQIIRZjzDuPGV+BZrV9oeeDi+5+5wnWVjvBpiv4Qt5Pi3BaucWFhQXju9/9
    7sOU0gCllPeFtvvzag1f6AXcAgCZEOL5wuCb8IV+xsjVKnLxYuG8D/QKLoFV9oFX6wdZjf+b
    5zfwdPeu15xvuTvMLcQU3D3FF5hzXCFGehMJIeXOuNfenuN8YWal5OYGtFM2V5CXuOKJZdu2
    atu2Z6v4JJqnMsZV+EIefPDOlneG025hNHstY4drrUPILxZv0LQrBFRvxhH6q8q0RlXq34oT
    JG61hm/dDPkq7mvhBDVfdbmEazS0epVVdeYz+HwQVW6TcR1Y0K1uiqhucfESpPwKFTgBZSIW
    53ITROkyVXu/I+Q19HjNRr4YsRqFCB4B9Sql3ntVUN2SvRaV0av9OVpjvfigdS0De36/eQba
    k6PwHFVoheTGr3F6ub0CX0Lo/f5adNE3VFX0Craq6KILfFKdc1tpAr7AgeDKrNK1bDqfHBRc
    VNC7CrhBCcN8+0v3FR4kzsaIuHq27cutF7/HLK5Ayt/fEPqMV/GevOCHcEGXZ9sCvr31RttF
    r8YX8gg1322kYm300by/D18xqej3hW63i2JZVhDVc6leoK9z8YRX3DX9CSHvDwkhSo0z/Hp8
    oWffebSQXeP1MbiYyfOBgRrFlhvVB+o1ijdrUhBcQW/Wi0GZu57englzhRhpBT9IVrBjtew6
    H9uVOB9YqAE83Ki3chGeePTFlTGlAmMsguoukDdSSK4VlxdR4angNXivSeFFWsVN5r0R+00m
    L2/EoP9buvFO0F+ZVGokQrgKR/hG1o53gv4qs1+El63iITN9B4APnvhkTURl5kHm9rLXCmhy
    gavfyBDud4grBGlCDcO2UmWUvyxfQWK11snyBei13sP1ro6uVHTwr5flsxVree5qBV0FLvhU
    3oATvFyCY/iCTA3VrVYUN35lVF9hzUKEkGCNSvLrqYoy3x73ow5eEHFDjgOs4AMp955XCgre
    TODKUBu98X9vNYpTq3EmedvmJdIyZ+fFq1g78gZtGb9mfIumyT3mk2y2yrbpcr6wXGDn0D3F
    RSyACimMyRVWij5b80Z9If9/1rLr/vbSaz1C4d8n2r9RH1ircHM97bq/s8/zgQEsl4i6Wlvm
    t+t88YD3gTeiDu+V9p+G5SMCOZfnIXCFhBC4um4Zfr14H6hyBaprIgVH1noVawtcvnkn/G/t
    5lsHPtAUaxh9ESujVtc6IaQ+A8UbrmUBxjWcm6i1N2srElfWhE8I/bN//haWWpo25DLX1QT5
    uFLwda33qG8YfKUgc63PNnudr73c+q3qma5x7ojvjHkBhXyZwKgWqnM1ht0/D8vfW/7A6QYk
    /eD3mxd8eagCj7avFDhcTUIIX5DpL1CZqEbhKb8HbyRfsBq+7v+rfvIy55L3jbX207W2dbWS
    nct2Hq2Sfa/l42sVVrznZF8AygeuPALkBY//Jn3hCvvkcv/3WidWr8cHXnbNVvu8XiEG5ZNr
    ibuEyySEl/ODtRJicwU/aN+Idn2FvSdyftDvC/kYQniD58pfTPaj8Dzh4ZuOHW4sj/X/Z7er
    MP5XCjhX06AxrMxKupZB++VuAq5+Jqnq/17tYO0GWJt/k7e1MvxXKELgCob7SmRYVyLCWikp
    viED+Kss2FxtIPl6WR/98xSXDTj/LRYHf3p7XbZtrQ3gVW2oG8DeX64t219oWbND8lM/eGMX
    bq4iBl3p8dX4wSvZdVzGH/5/wRcKV1ivq/WFK3V90JXW7Kd+8Ke3n95+evvp7ae3n95+evvp
    7ae3n95+evvp7ae313X7/wE8PpVpOFlfFwAAAABJRU5ErkJggg==
  }
  set pieceImageData(Fantasy,80) {
    iVBORw0KGgoAAAANSUhEUgAAA8AAAABPCAYAAAAk2KwLAAAABmJLR0QA/wD/AP+gvaeTAAAA
    CXBIWXMAAAAcAAAAHAAPAbmPAAAACXZwQWcAAAPAAAAATwBCM+rqAACAAElEQVR42uy9d3xc
    13UtvM5t0xt674Vo7L2LpKje5SrbcontOI5TvrzEz2mOk7y81JeXvOcXJ07sFMcl7pZkdYpi
    k8ReQAIkeh9gep9bz/fH3AvcGQxlyRYpysbh7/6GAC4GM2fOOXuvvddeG1gZK2NlrIyVsTJW
    xspYGStjZayMlbEyfgEGWZmClbEyVsbKWBkrY2WsDJNzRN4y9+infSL60/5BSumtOofkjb7f
    W+U9rIyVsTJ+Pge3MgUrY2WsjJWxMn4eAUsxJ/qN3rcyVsYbBLPkp/idNwJ+yRsExvQdMm+k
    4D2RgtdOTdfKWBkrY8UG3/TDfGWsjJWxMlbGynjHGVxKqfG9N2rbilrdFUC8Mt7CDPBNH7dw
    Bvgd9x5WxspYGW96j7+ZjU/frr2/AoB/AY3yimFZGSvj53ef/6Lsb2OOdNBrTBihlBL9awbL
    M0/mLJNmeswzxCtn5Mr4SWuvYN0V86no6zl7K2tsZayMn08/+xfYBhPDBpvsLqPfS3K3UhBC
    Cu0vfTvOxxUK9Dt8/ZkMrvmRFjG4K9SilbEy3kFG9zoO9uvaXbNz/osGRgAw+nzxlFIbAKt+
    i2Y+GwkhKgAFgKw/qvo92soZuTLeqCOsrzuzo0f0NVhoh68baFkZK2Nl3JK2ZPFb5q9Nga/r
    2t9f0Hkzg17W4/FY6urqbKqq8rOzs1o8Hqf6PBJCiKbbXfOlvh3n4woAfgcaXpMzx1BKed3R
    kwqof+ZMh6pf0B3AX9jo81uQQf9JT/COFy5ZGW+f4TXtYUIIYQodbCOKiuKZzF9UI0x0A1xC
    Kb0NwAYAlfq5qAKIArgK4CwhJKQDX9F0Sfr3yAo4WRnF9qRpH5qzHgQA8/jjj9eWlJR0+/3+
    ie9///uz2WzWuMe8P41gy+Je/UW2w2/WHps+BxT5HAqf56ea1BX7u7LfjaBzkQzmYmazYG9f
    L5P5CzUopWhsbGQfeOCBLofDsZ9hmDaAejWNsqqqZjOZjH98fPzcoUOHrqZSqQwhRCqwvxLe
    hiTdCgC+xY2Cic4HFGR8KaVWAPcC2AHgMKX0OCFEMTlymu4ASvrvqCvGN29ezXNKioBbqoet
    aJF7GSynuxlzrpnDXT8BKNPCA/gXKJiw4oyY5lH//BkT+LUDcAIo0W8NmPawEdQyR1B/UTe0
    IRi02+l0/uX69evrGxoaGIHnIYpZTE1NamfPno8lU6kvUUp/DEAkhKQAsPrvG+ektmJ9VkYR
    pziPSk8pZY3/9/T0OKqrq3+DZdn3NjU1nbv33nv/+jvf+c5V3b4Y9kDR7QUKAy3Fzvzrnas/
    L2dioX+j/3dxjouAWZrDJEumErngP1Nwf54N1u3365nhPBtsBtQrtncF/DqdTqGtrc1RVVXl
    9ng83snJyegrr7yS0O/TCuyv2e/+hbTBd955p8vtdv9qIpH4oN/vt2SzWQIwsFqttLy8XO3s
    7Lw4Njb2F/39/ZMAsgDSpnPRnKRbAcArI++gN+h8or7xjEXTxnHc/9fU1LRxbm7u9lQq9T91
    EGyOSEmm+/NUFt/MIfrzciiajSillAPg1kFGWp9fcwZ90THWD8V6AE0AUgDGCSFikXuNQ1Ez
    1SEWA8tmp/sdnRUorMEsFqx5HeBPfx6CAW/hfgeAEkrpVgA7WJZdSymt0zRtFMD/BOA3OdJG
    FJW+HcbjVgtoAXBVV1d7Pv/5zzObN28CNBWqIuHFFw8xn/n1/8+aTKWcAEoBpHXnGTod2pjL
    lbEyitqLX/3VX22z2+0b4/H45LFjx0b6+/uzANjGxsY2Sum+CxcuVJaVld3e1NQk9vb2/u/+
    /v4QzaFgIwDNmp5WLjzripyPy77383YmUkqJx+PhduzY4XI4HO6JiYnM2bNns4qimLNsGgCV
    Uqp5vV489NCD1R6PpzWTychnz54bO3XqVNqUATbuNQCJEdBiTEELFDx34fWOm2dTEOF6JXEo
    eN8rtvd1QHB1dTX34IMPtrvdrp0sy24FmE6GId7S0tL/OHny5LdVVTUnlgy7oeEXPIBKCOEA
    +MbGxqyHDh2CKCrgOB42m5Ps2rWda26ut6VSKR+AFKU0iVy5iMGOYd+O17wCgG/xs01fGI8A
    uAfAWUrpCwBmdAvR4/P5Gj7/+c/jmWeeaf/GN77xy5qmJSilE1iqdcuaFhfRDUOxNgq0GEjG
    OzA6+pMOOf19cABuI4R8kBBSrWlaP4BvUEqnYapTIIRIlFINwDae53/L63H1ptOZaCqd+Qal
    9Fs6lcMAIDIAWc/M11FKS/T5DgAIE0I0k1Mkmy7F/Lm8HQbop4koG8CXEGJETlnd0TBf5gAA
    1QMDZudDNV1mI0J/2tf0ds7VG5X8v869FIAA4BG3y/UH3avaq1sba8np85dwdXg8BsBOKfXo
    c5TV16kRcPm5FTR8E+tgPp1OpyilXqfDDlkSoRCKa9eGEAgGgwDiAMr0R8PwGucj+Ul/c4Ui
    +Qs7GLfb/Qgh5HM+n2/uzjvv/KHP5/v20aNHg1artU2SpNKRkRGMjo6y+/fv379x48bJkZGR
    JzKZjEgpVfUgFWc6CzkssRaMrKe5XMkMyH7u7LDx2js6Orl7771no81m+wghpK21tWWksbHx
    G9/+9rev6Rl0A9BKAOT3vve9faWlpb8ty9pmq9WR2bVr149SqdS/XrlyJWHMnQ5+pb6+Pn7r
    1q21dru9XBRFOj09HTp69GggFouput0ybK7ZDuPtAsE/yaZc5zwyg1xWX1dmu5t3rul+jGGD
    r2d733Ry5O04H9+MDX6T9po+9NBDVT6f748yGenehYW4hRCOVFW5xGw261BV1ZX7NSoDyOjP
    rZrWzy+q/SUjIyNiWVmZXxAEKssqUVUNhFAIggWlpSVaIBCYmpmZEZBjtDH6fhUJIdli/svN
    sMErAPjWj0hZABxobm5+F8/z946Ojt6uKMo/UUr7ATT7fD47x3E4cOAA6e/vX3fhwoV3A/gm
    clkOhRCSAcDrh2HWBLTMlC5Nd6TzMpjIzyr9XHh/giBAFEUAaOB5/nd3btu0u6utGc+8+PKO
    0YlpFsA/mUCtQVdjALxvw9q+Ax997FEcOfZK1Te+/+P3qqp6mlI6aRyC+qFYBeB9Xq/3gM/n
    K8tms2IwGLwmy/JTlNKj+udhON4Z/dGIIio3c56vJ19foOhnGFizITaMJNWNsIdSWg+gHIAH
    gE0HcoYBNpgIKQBxSmkAQEKfZyP7ZgQPFN0451HNC5yAwvqbt9NBzJuj1zEWy4TpCqiAYFkW
    qqpaAWzcuW1z5R9/7tfJxPgYXj19LgPgjP57Hn2uGOSXN/xCHInX+1qnjPsTiURkcGCgdteO
    bSBUw/T0FH78zLOqLMvT+vyVmgKBkr7/srotLAY6KFbKRn7hATAhpHp2dtYxPj7e3tXV9enN
    mzc3h8Phr1BKayRJstfX14NSivHxcWd7e/sD69evDxw/fnxQX2dZr9frrq+v1yorKzWn08kw
    DENEUZSGhoZi165dS1NK1SKgzFze8HNjgwFAURT8xm/8Rond7vidUCh+fzCYYqqrfbsbGhoc
    DQ0NfzM5OZnS368CgHU4HMTn8z2YSGTvGxiYYX0+D1pby97d1tZ25sqVK+d1O6QCkHft2lW3
    devWRxiGuUcUxWqbzab29PSM1dbWPvPcc8+9ODY2FseSDoBhgwluYh3ideqXiQnYkuuceVS/
    h3o8HuZd73qXz26311OqlVEKt6ZpNlVVBVVVWUWRIcuyJoqSmE6nk4lEIjY0NBTw+/0pSqms
    21/D9ip68EAttL3FKOm3UHDG7Jv8TDbYGE6ns15R1LX9/WPWubkMenoaoSjS+JUrV0YAuPTb
    sqb1mf0Ftb95/s/zzz8vrVu3btLn88lut0eIRuNgWQENDXVwu53xI0fOzEmS5NbXleFfi5TS
    LCGE13FKMR/vhp19KwD41o+UMgCEPXv24D3veY/ti1/84t5nnnnGqyjKPwCocrlcwosvvojZ
    2Vns2bNHmJmZ2R8MBucBnNNBGYMcfdoBwA7Aoi/aBIBBAKNmAIJ8URgZBeps7/QItCQt4gUb
    x7Klu7dtwsE923D6/EV+dGLaC8BFKRX1KKloCgTEOJZVWhpqucNUg6ZpcX0enXqhkYpcdunT
    69atO3jfffcJHo8HoijiypUr1U8//XRnKBQqo5S+oNcfZrCUFQDyM6I3FfAiv6aZMbK5Oojl
    sVzowRw9ZgF8wGIRPuFyuVwcy/Isw3AMyxAARFU1qqoq1TRNVRRFTqbSSVmWfwjgCeRo50Yg
    IGMCu6y+Xg2gourGzQDU5oy7OXCzCIj1jMoNjU4bjoqRBTdlv82Ud7PDoJrmMc/RIoQQTdOg
    z7mzsqKUETgG33vyOYxNzg4D6EeOqq/pe9TIAjP4Bcj+FrZZwFI2zZhvHoCWTCanL1/u7xWz
    GQgci5ePHMOFi5dSABYAePWz0CxqYqY+ZgrWk/rzCD5WxptfhpTStCiK2unTp9nZ2Vnbvn37
    7t2xY4eaSCRisixbEokE+vr6MDc3h2AwWN/b2/vA6OgoZRhG3bx5c3NlZWWFzWYrIYR4CWFs
    DEMIISTb1tZ6sbq66onjx0+MKIoiISdmmdX3uBEclQrPjne6HR4cHATDEB5AeSiUZIaHg/B4
    XKzbTbyaprlMgXkZABVFUZVlKQ5ASqUUm89HoGlKIhgMqvq5CABaa2urfePGjR9Lp9OPDg4O
    2pLJJFiWRUlJSUVbW1vn3r17q8Lh8A9jsVicEJLGUtbULC6o4i0WCy0Mdpprnw0xNdPZxtps
    Novb7ebi8biWyWRoEdurbN261VFRUfEZQsi7JElxaJrGaxplc9nzHKuAEKIRQlSGIQpAw5WV
    ld/47ne/e5hSag7AZ/Wvye23326rra11LCwsoL+/X56cnDTKv8zBbCNY87bZ3oLnZ/SYgGEP
    WCypsaPI3BUTrFoEcqqq2gmBJZ1W4fW6UFXlys7Nzb52/vz5uG5DjJp+2eS3/FzjEHOnBWPd
    FthfFoDg9/sDpaWlUZ+vpCIWS8HhcKC1tZmGw6GFa9euKQAqdBucEzphGKiqqup7nSni0yk3
    Mii1AoBv/YgLBRBOpVLqjh072JqaGiaTyax98cUXPwJAKisr4+677z785V/+Jebm5rBr166S
    H/3oR/fodQpuAJsB+Dwej72xsdHm8bg5lhAsLCxIQyOjo7IsfxfAKwCilFLD+C4ejCYwbDbA
    7+hsCMMw0DRtJpPNHjt89MSqcCjIXrk6MgfglL4nOD0iyugHfQrAoYFrQ/f9079+venYyXMJ
    SumL+jw59HlhATzS3t5++8GDB4VLly7Ra9euKXa7nTl48CD7wAMPVH3ta197WJKkGUrpEACu
    QNHXnHmnN/gwM2f/GR10CbojUQWgWj+oKgH4AAQBHCaEhJEv/CDrv7d925ZNPZ/6xEfh83jA
    sYy+cCk0VYOiyJAkCXP+efzjV76GsxcuTQI4QylN6CA4DSClAxARwG4AH9bX3wSAUZ2a7geQ
    NF6D7hyZs8jm2q8bvl71DDhHKV1NKW3WX1sMwDyW6FEwzZn5NaoFDrbhBDkB2EOhML7zo6fx
    zKHjMU3TTujvsUR/TOmP/PWM71slknKr7HE9KMPoc1Cr7zsjEp/WP2e7pmmB8fFxLZGIMwwo
    nvrx0zQej0f1eWrSf8dYIwqAYQAvAZjTg4NGXbWxpn7h66tXBiBJ0qzb7ZY4jrNdvXoVNpvN
    smfPnjvcbncwFAoJg4ODAIDVq1djfHyctLa29u3evTsTi8WSq1at2ikIgkMURTIzM4P5+QWk
    01lYrTa0tbV0btq0pbeqqvqZV1555eTk5GSAUprRz4+0KUgoIr9dyDs6INPb24uNGzdGDxzY
    /2JlpW+DpnFWr9cSnpgYeXV6eprq+1vVg9GKoijSxMTE8dbWjuHOzto+j8eaDQQCh8+ePRsy
    zgKO47B9+/Y7NE178Nq1azaPx0Pb29sVSikZHx9nr1y54u3t7b1/zZo1U0eOHDlHKeV0pxtY
    Xo5zI5IZ5oAzAcAyDCO43W6hvb3d3dTUVOXxeKoFQajkeb6CYRh3PB6feuGFF45OTEwkTGeW
    DEBSFKUMwM7x8dlVAwOjkGUVlObOfkIYEMKAZVmwLAuPx4menqZaq9XSTwi5SilN6esspa+x
    jN1uV3p6eu53Op0P1tTUxLu7u8cTicTY3NzcdH9///zExERGVVUVOaaWkSCRdDtcLBh9w31F
    Sik++tGPeUpLS7ZQqlkTiWRsamoqdPz48WAsFjMD8kJmBcz7yJR15yVJ8tpsTq6kxIOyMg9V
    1fTQsWPHLmazWZcpAG3YHu56AejrtUx6nTZKt6z9NcTk+vr6hA0bNtRYrdZyWZb5eDyuzM3N
    ZYaGhrLz8/PclStXxL17b4t5vd4KlvWjqqoSpaVe6eTJ1yKqqpbV1tZWlZWVqaWlparP59UE
    wZKYmpp69cSJE2c0TeN1/CEhv75auVHv65YAwG+Fs/ZzTE1TAEyMjo5KkUjEtnr1avz3//7f
    mZmZmQ2Dg4NyWVkZ2b17N2KxGP7iL/4C+/btI9u2bWs6fvz4XfoBVwfAqmma4PF4yMYN69He
    2gybhbe9dPjlnqeeeb46FI5spZQeAjBAKY3oQCSlXwYYNvfMfMdOtin6mgJw6drwWCoWTzjT
    mewsgFnk6LvmTWdE4qPpdCY6ODwaS6ZSom40zNmk1U6n887NmzdbXnrpJeXUqVOnKKUnALTM
    zc0d/OQnP+lob29vuHz58gYdIDEFaoJSweF8o/aYkWF16kC3DkAHgG4ALTzPV3m93lKvx+Nx
    u11COp3mhkdGZ2VZTlNKB7BEZzYCJDwhxN3a3IR9u3fCZrVA0ygo1QCq6Y+55bIQCOAHTz5D
    9cBMn2mNxQFEKKUh/bn3NtXX3rVlwxo6M+dXZv0LYiAYDiZS6TnkGAuDAMYppX4AEd1ZzBZk
    TJbVVr/Vg2VZKIoiAPiwzWr9gMNhSyaT6WhWFAcBfItSetn0+Yr6WhHNEWhTQIIDsAbApywW
    y9Yz5/vJKyfPIhSJzunAvwRLWV/o75lHkfrVIkG0Qkrd9UTJblnHWt+zDQB+32q1bhAEwaIo
    iiqKYkZV1QSAMIAQgIZoNIJ4NIpQKIjTZ85CX+sbWYaxOJ0Owetx8y6HndGoxoyMT82LohQn
    hLyiA+C0HljIID9zvzJ+cQeNx+OjLpcr5fP5bJFIBFeuXEFpaal727ZtrmQySbLZLEKhECil
    KCkpwaVLl4Q1a9Zs9Hq9gYWFhQDHcUFBEKxer9dms9nt2azIhcNR0t9/lautre6tra1reuCB
    B3aOjo4ePnPmzDm/379AKbUU7HGz6GLRMop3RFRfBwanT5+WtmzZfMnp9Earqz2VgBoYGhqa
    MtlVRY99SZRSLhgMJVpbacTptMQ4jlGTyWQ6m83y+r7Fhg0bWsvLyx+cmZlxVVZWahUVFZem
    pqZe5nm+pKur6+4LFy6UhMPhiubm5k3Hjx8fU1XVsMFmRpGEt1jN19w2y263s/v37/eUl5fX
    2Gy2eovF0slxXBfLsk2aplVKklQmSbI7nc7yVquFLSkpuVpTU5OIRCKTLMtqsixLoihmZFnO
    hsPhUkVRHaFQFFNTfgBLwDd3ERDCghCCZFJEa2uNpmm0vKmpaTXDMGlCkFIUNZ5KpcKhUCjs
    drsZlmVvTyalOyKRtOZwWFSHw5dZtcoX6OjomE0kEsPz8/NXZ2amJ8fHJ/zBYDBGKc3oFNbC
    lnI33FfUfTn6uc99rpnn+T+XJKXW4/FFS0pKA62trS8fPnz4B5cuXYpgiW6bNdlPzQzsAODO
    O+909fb2PiQIll/SNK20ra0SVisnX77cf21sbAzIMfwk3UYoevCfQxEWVkELNeY6trjQ/i5j
    ht1Ko6ysjL3rrrvu5nn+1xRFKdM0jSkrK5NbW1tT27Zti6VSqWAikZBYlrE5nS5YLFbU1dXA
    brehs7Ozoaenp9lms1lYVuABlqMUjNXKU5/PV3/t2rXAwsLCrBGM0S9yowPQ3Nt5CF7Hyclr
    +1PQd+t6C+YdaQje4DyplNKx+fn59OjoqK22thZ9fX341Kc+Zfnbv/1bS29vL+bn5/Hqq68i
    HA5jcnICn/nMZxhVVdvOnD6TkHK0oZlEIiGdOHHCdubMaa/VanX0dq+y/9KH3sfs2Lyu5Bvf
    /v6B185cWJvJiq8BeIFSOqgruhWqR7+uUMKtPu9mQ6TXDMrJdHqYDYWJXv/iMa0vo/5XIoSI
    lFJZkuXzUzP+hUxG5PV5cOpzVM2y7Lt27NhRCwAXLlwYp5R+C8A0AP/c3FxvMBhs7+jo4C9f
    vtyogxmKnLq0qkdQjcOZ4Ab2amZZlqqq2gbgM4IgbHC73U0lPp+7uaVZaG9rY1tbWtDQUI+a
    6ipUVVXhxRcP4f/77c+Wy7L8QeSym4YxSetA/hQAGwWgaRoUVQXVNMM4AVQFKAUhgKqqkGUF
    DCG9PM/VA1A1SmVVUdMapXH9+SYBrOpsa8If/NaniKap/PSMn782Ou68cnWkqX/w2paxydls
    JBpfyIriGIDLAA5RSqdNGZNMgaG7IQeo/vlQANnW1mbnJx5/nzcRDdV97ds/6hoYGuMB/J2u
    dqjo+8m4f1G0yrSHmhmG+cKmjRsOPvrQ/azbacfxE6/ih089UxuLJ9YBGMBSHbWKXLa5qPE1
    RZmNLD+vG28jmFOMHnbTo/dvdvsC2FVfX//IJz/5SXdzczNEUUQ0GsXs7CwmJibo2NioOjs7
    i4ryciIIHBRZRlNjA5obG3zNjfVobqwj9TWVqK4oQ1mJB6FQCL/zhb8u7R8c7qWUjmCJCZEs
    yBr8XLa3KLSZryfW9k7LWLzF/gnx+/1+r9frr6ioKAsGg2hoaMDY2BihlJL5+XlYLBZs2LAB
    TqcTqVQKg4NXEY/HrX19fbUlJSWyqqqRYDA4GwqFQgCoz+f1tLU1tfr9odKBgVGSSGRdq1Y1
    b123bl1vfX39iRdffPGbQ0ND46as3/XEsYB3kECWqeUboZQSUZRUm02ZUFVpRlHAUErdekDU
    KH9h9HNPDAQCNJFI9GsaSauqZg2FQlnoNZn19fUlGzZseFc0Gm1LpVLo6OiYv3jx4rdfeOGF
    QZfLVfn+97+/y+v1lgSDQba+vr7e7XaXRSIRIwip6Uwv0eR43whbgV/5lV9Zb7Vaf1lV1TWS
    JDWk0xlnNBoVwuEIE43GEI8nkEqlkUpl0Nvbhe3bN9dv3rz54xs2bEjlziQqaZqWTiSSkyMj
    IwOKIgtL85oDvQxD8rLAQO57LMuwdXX1m++9t6JLD8xKAE1TqsZEUZxLJlOzhJCmUCiFCxdm
    GUHgGIfDyns8VrfHY2v1eNzb29pKMm1t7f6tW7NjwWDw/NGjRw9PT08vUEpTuv01z98NBS/G
    WlJVVWZZlo6Pz5UGApnS2trK1ro6d3t3d3fo0qVLh/USNUnP9pvZdqph3xobG/m1a9fer2n0
    TwYHh2vn5oJwOJxoa2vga2pq1jQ0NAQmJyfDpkB2Fvk6J8Ve32JLpYMHD7o8Ho/ryJEjsfn5
    eVqkn7C5rZJ2q+xps898++23e3ief+/4+Pju/v5+aJoGnufhcDjgdrup1+ulHo9XlWWFzWYl
    MAwHVaVIJLICIbaqeDxLZmfDyGQUiKIGRSFoaipDQ4OztaKionVhYUHUNY8MdqRx9t0wnRPu
    VjAuWFJCNJRkjcgKTK1kzBuq8NLwcyZWYnI6GAD+aDQ639/fXzo9PY3nnnsOjz/+OL785S+j
    paUFiqKAUooNGzZg165dOHjwdnS0tzHPPPOs5+jRo+5z589lFhYCAVVVx9PpzPl0OsMdPf7q
    ep5lm375I+8nmzesYa4Oj1VksoG7ADQC+CKldMSUvdIKIlMqrqMUfSvPu2lOjRZI44lkOpbO
    iN2U0nM6oKUmkMciR/ehAGopRbUoq1tVTZvUDz+Pfs9Wu93eLUkSefHFF5OiKB5FjlJpA8Bq
    miarqgq32010g+0xZQINEQUeN6GeM8dgQrPdbn/vY489VnL33Xeho70dJSUlcDrssFosIIRC
    0zSwTG7bOR12286tGzZ63W5QSqGoCgLBMM6cv5RMJFMCAIuqqDrgXfLJCCgozSWACQCqUWhU
    I9s29Ll2b1nrIgAkWUY6k0E0nsSsP0gvDo6I4VgChBBqtQjEahHg87rR192Gew7sQiweZyan
    5+yXBoaaXj1zsenwidPrM1kxDeAwcnXtXEHk8IbRyjVNI/o6ORMOR8IlPk/FPfu24tirp8nA
    0BhnCpAolFLWUBY3RcmJ7oBxAA62t7Xt+cLnf5/dtW0zVCmL/bu3gmXg+revf2e9RmlMz3JS
    AOmCzJB5L5IC48sC2AbgN/XgwrcBjJkyH+bXZO4XjlsM9LEA2jo7O10f/ehHUV1dvbinJUlE
    MpkgiViMm/fPQeBYlJWUwOO045//3/8mDNVgtfCw8CwIVaEqMqgqQ2AJKspKOOSo/mVYqmWi
    yG8z9XM3ivViLVKLaNR5wbReCgOh2jvJBrzZuTH20czMTGrVqlUzZWVlvatXr0ZHRwcOHz6M
    I0eOQK/dx4svvgi3241EIoFwOIxwOIyxsXGmsrLK0traWtXQUF9WUlIanpgYH37ttZMT1dXV
    6c2bt2yxWm12q1XQGAbZTCYjcxzX7HA46pBjxRifB0sIMeobDYq+Wjj/7yD/hgDgJiYm5rxe
    75zXa98ejUaviqJoRa7O0ghCc0ZbI5/PWysIfKXFYtkiy3I4m82yui0lXV1d691u97pMJsM0
    NTVlw+HwK6+++uoYAKskSbwkSTLDMBBFEYQQh8Vi8WGpdMJg6LwRRs3PNCwWS48oiu86f/68
    c2xsHJFIFNmsBEVRkFPNXcrgMgwDRaHuaFTcqiiaDm4ZOBwWVFVVB8Lh8HdkWeEYhtH3nbF2
    ybK3oH+fJBKMNxrlvYQQsCwDjmPA8wysVqKVlEDkecJSmoaqashmFYhiGpFIFhyXgMXCs06n
    xen1WtsqKuxt9fUNfY2NjZHp6emzpmwoMZ0Lyk3Yq+Shhx6a6+7uvmi1WlYvLMwTwI7qageT
    TqcE5NhmQK7OubDt3WJg84477qhkWfZ9g4ODtcePn4aqsrBYXIjFJLJ1a1djd3d33/T09EVN
    05IAZEKIBfn142Y8Y/4/s2/fPtf69es/yfP83tramieOHDl66OzZswmz0nnBa5JxC7GxjPPc
    6XS6NE3rGBoawoULF/Lu4XmeWCwO4nC4GKfThWQyC0oJLl8exciInwAMKGVBCAeWtYDjBLCs
    gHRaBaXEJQh8lZ4A4ZFf/y8iv4vIOxsAF1AxDdBr1Z2QVuQomSX6RGQppVEdTCzo0SVFj+aY
    qY5yARh+xxpiU4TU7JCI6XQ6PTc3h3g8jpqaGrS1taGxsXEx0/aXf/mXyKTTcDjssFgsWN3X
    i96ebnzog4+RM2fO2H/0xJONhw69VDUxOTmvquoIpXTy6CsnfdeGRxyRaCydTKWDeoZvQTcq
    lXoggkWO5moIIhk0U/WdYoAL6CiGdWgA8JG+vt4d27ZuFV548dDe0dFRF4CjyNW8SoQQXge/
    azmW/eCOHdvX3H/vPewPfvRE97HjJx6klD6tz1k2mUxOHz58WKWUnsFSLbEVQKvT6ayur6/H
    yy+/bABeqw6ODMNrNh43Y4QtFiF22969JQ8+8CCopkKjWi5zq2k5h45qUDQN0VgUTQ11+Pxn
    fwP1tdXQtFxGt3/gKj7923/EJZKpUoDaNCPja8pJUNMXFPp+pBQb+jrxsffeC0IINFWFpqlQ
    FBmz/iD5s//3NetLr5zP0aiNf3pGWeA5lJf6UFbixaY13ejpbMGZi1f4TFb0AajQM53EBDRv
    eH9XfU2dnp3zH/3q1771EIdHmIVgmOpZDLv+WctYan1iZBok095hAazq7lpl7+vpBtVUKIoE
    n8eFbZvW4js//LE7kUyVFGR9C9U4qSFSYQR59OftYhjmsz1dnQcZQrQrV69tkmXlX5BrqaaY
    jEymICNMb7H9ywAQBEEgPM/n/VwQBJT4fPC6nKipLIOmyFAUCTzHoqLMB01VQFUF0Ax2Qi5Y
    oygyslkR+v4rKQh+GbXFBD9HImNFgC8BQKxWK9vX12dzOBwOlmUdqqoKsVhMDYfD8vz8vJrN
    ZlVTtwBjbykFQSbtnZKJfDNTBoCpqakpYximTJIkXL16FSMjIwgEAlCUJR9/YWEBCwsLJvop
    g1QqjdHRMUxNTaOsrJTr6OioaG9vK62trW27du3a7LVrV2fKyspcc3Pz0bm5uTlVVRMWiyXL
    smzlmjVr+GQyOR8IBKYSiURIp0QnsVQXbP4MVNzCbLhiSY+77rqrore390PZrHjg6tUhZ21t
    jW/v3r0OlmVfunLlil9vfyQwDEN37NjRumnT5sdSqdS206fP8/X1td7169c/FA6HnxwYGJhf
    WFhQY7HYtNvtFhYWFi4dOXLk5WQyCUKIpbu7u9HtdjdMTEzA5XJBFEUllUoJBTbYgnxRyhtx
    hlFN04KSJKWHhoac4+PjIIRbBLtL6yYHdC0WC8LhBF577RpkWdXvYVFdXYJ16+qFbDZbpiiy
    ZclnLLZ0c4+E5ALRoZCE6WkRDMOCYdjFOmGHg2Xa2zmbxbL0GsyXpgHZrApRzCIUEpHNamhv
    dwjZbKYUue4PvAnQGefDzRCHIt///veT1dVVT1ZV+fa0tsoNAA9KtezU1LSInIYJsMSuM4PN
    RYElp9NZIstK2/T0HLJZGVarFQzDIRbLIh4XOa/XW26xWByZTEYEwBYEkBcDggUK3kxDQ4N9
    3bq1j1CKX52fj9S63Y5Nu3fv3qAoytcvXrwY1BmARqA1bcTWcQtRoQ3/WdM0nlLKm888Y8iy
    Ck3LQpYpksksOM4CjrNAUSiyWRksK4BlGTAMp689DoRw4DgOlKokk8m6dBtMkGPMGT7JDQ1I
    cW+X4dUzuzyltBPA3YSQ27xeb2dlRXm50+ngGEJIJpPVgqFQMhAMTSuKchnACQBXdOfSmCBD
    OEc0GQQNN5BGehMNL9FB57bS0tKm1atX47777gPDMLBarab3RmG32+Cw2ZaADHKApKKiHHfd
    dSd27dyBU6dOW77+jW82PP3Ms5XzCwsLkiT5p2bmJH0+jQNCALBfd7ZnkKOYjuhAT8BSXbA5
    +KDhFqeiG+BXf6wA8Omurq4H/+SPvyDs3b0bzz3/vPXP/vwvtly4cNFLKT0CYFYHVJ2CwD94
    1x13tP/+736W9PX2oru7i/mt3/7vPVcGBjIAjgGYopT+SJ/DeX0NegFsEAThjnvvvdenaRpO
    nz6d1QMMBjg2lKC5m+VoMwxDNE2LpVLp+bHxsWZtERTotGUzbKUaZDnHLkim0wgEw5BlCZqm
    we8PIJPJMACaQUklz3NY1PQyPndqjovknltVNVwZGsfR1y6gotQDp8MGq4WHwLEQBG4x68yy
    bAHHgJpo1RQaQ5HNZiFJCtWdl3JT9tJMKb+hRlgHZhEALx85fnLfzKzfNz3rB3JMik798zYA
    sOFsGaI2ZsGVRDgcVhPxOOt1WkGQA2iBYAiSLBuBJjMtSClwfmFk7PT13awb/0d7u1fd9mef
    /yxjEzjmL//uH7a88PIJh6pqfwdgCEs1TTA9J2MC57dKBJoCiEQiESkajQplZWWglEKWJQQW
    FjAzM4WpiUlEIiGs6e1BU0Mtjhw7jv7Ll8ExDDxuJ8p8HnhcdtgsPASWwZWrwxifnjXo/Db9
    suhn7nXp5T9P4Le0tJR/4IEHGktKSjbwPL+RYZgWAG5KKadpmihJUiAWiw6NjY1fOXfu3GQk
    EknpwUHRZHelQtv7Tg1CFwI1Sik5ePCgt6ur67FoNNo3MDCAQCBQ1FwbhK1C8AAQaJqGhYUA
    QqEIRkfH2L6+3ore3j7fwsJ88OjRo7PpdDq9c+fO2srKSgfLshwAhmVZhRASSyQSAydPnnz6
    0qVLwzoIFpAvjiUVOM63dECaUop7773X3dfX95FMJvuBY8dOOAcHr6K2to7fuXPb2ttu2+vl
    OO7whQsXxgHwq1evrt+4ceP9kUi079ixk+zMzAwmJ2fIgQN72rZt27Z/YWHh0IULFwJ+v/8p
    nufVhYWFeVEUs3a73bVt27be3t7euwOBQFUymcSqVavk4eHhhUQiweh7XTBlfpkbbIdJMpkM
    2my2sNvtqTCvlcLEoQGCCSEQBA4cx4Jlc2DZahXAcSxfXl7WZrVaS3J6p3idzC8WqdEeD490
    mkJRAFUloDSXOdYWpTooXm/LGrFVlmWgqgoNh8MO3fYWay1HbsJ+JZRS5uTJU5f27dt3vqur
    ul5RKOF52BoaGnpHR0eVeDxu7BWYAtBZcxBaFEXJYrEkXS4XWJbXAxMsBEGAxcLReFxUVFUl
    Jh+j0AYvgt9169ZZt27d2sTzfBXDkBZBsP760NB07ZUr02hsrC3p7q5739atW7np6elvhsPh
    hP5aUqbnlU0f5i2zjxOJhFhRURFxuVyGkKwegLbA7fbB681dmgbMzYVgsdjQ3FwPm80OTQMk
    SYMkATniHAuG4VFSYkM2m0kFg0ENOTG7LAALpZTXGS831Aa/HRRoojtpbgD38zz/ifb2tjX7
    b9vr2LJ5I5oa6uH1uEEIkIjHMTMzY7l46XLpoZeP9py/2L87kUwdAfCcXvOX1IFaCkt9bmFa
    lO8oI1wkSsoA2GyxWD722GOPld1xxx2w2+2LRiT3vpZAwRJAyAFgUApNyz3abTbs2bMLq/t6
    sH/fXsuX//krda++djIpyfKsDtYaCSGWsrIypqa6ChaB1+bnF+S5+flxSZK/QSk9q9M+LCYQ
    bEjAy8inRd/KxtcN4MMNDQ3v/++f/R3rHQcPgmMZ3H/vPfB5Pdwf/8n/WHX8xCtWLQeCFY5l
    999z153NX/jD3yPtbW2gVMOuHdvxa7/6Kf73//ALa4KhUFwPymRN2dxGAFsrKys33nfffe6O
    jg78+7//u7qwsDCCXF1wYS2XViRse6PePwGQlCRpYXp6GrIsg2PZwrv09cjA6XRgdHwSv/8n
    fw2LwENRVWiahlQqDY2Cr6+r7amtqWZ3btsCnudNH76+Nk1Oj9Nux/YtG/H084fwN//8LQgc
    C6tVgFXgYdWf++LgKACgvNQHlmX0dVz42iioBvgXQpBkWdOBS6lh3CilaRNFidyE86yNZdl7
    Ozs7XDu3bwVDwBw78eqmywODJaqqPacHktiCzGIG+YrVx89fuPi+f/7Kvza8/10PQuAIzp+/
    gP/6wY81UZQWkKN3F9KVZeSLfPEAdjIM8wmPx72B5zhnNBZzt7c2W1LJBM5eG0I6lQZDmFYV
    2hYAAUppUo/aZ7G8hdOtNoaGh4ejX/va1yra29sxNzeH0dFRjI6M0LHxMTkUDKmKqvAfe/wD
    3LsfeQB/+b/+nl64dFlkGEbjOY7hOY6x26zE7XYyVotAItEYnQ+EggCChIBQmtceqbAX9s8b
    +GU/9KEPVVdXVz/AcdzD6XSmLxhc8EQicS6bFQlAYLFY4PE4Na/Xnd6wYYO/ra3t5Llz5w6d
    PXt2RFXVVJGa+0Kl4nf0lFFKSV9fn2XdunUPy7L8yCuvvCJMT09fB/wuAZl8AMwsnqUGTXV+
    fgGx2AnMzMzx69atrtq/f79zfHw8VllZ6SsvL7eLokgWFgKYmfFDEKzVra2NbRs3bhQGBgb+
    U1GUiF4nl9Tb6ZnrLm/JuS9kYO3atcva19f3LkVRPv7aayddAwODUBQNk5NTeOklhdm1a1vz
    jh07BE3TXk4kErGtW7ftiUaTXUePnmL8/gUQwiIYDOPkyXPMnj3bVm3fvj327LPPnvf7/cYa
    5Orr62t27969sby8fMvU1FTpzMwM6enp0dLp9MT58+fHkN8OR7tJWTfG7/eHOzo6wgaIMAPL
    pTUDPbgno7TUja1bOxfpzQxDwPMsCIG1vLyqNxZLs/PzIf3n+evRnAHOZmXMz0dQU1OGtjYe
    msaAUgaUEmgaAcsS2O259ZnNKijUazKvb4YhsNlYiGJWi0ZjTt32Gi2B0vgJtbFvdTClqqpK
    2LJly34A62dmZkkkkoDb7fK0trY/xPNc1dNPP/NqOp2O6p+tkWlNm4LQ9OLFiwvbt28/umpV
    W08iIfKRSAZ2uwvt7bWw2Zj06OioX5IkuSCDbGSRVZ1+Rd7znveUtrS0fCDXmkpu4DjOpmnU
    E49nUFlZCq/XBZZlrB6PZ1t9ff2pcDg8rCdmZJO/civaYDIyMpJubm4eaGlp2R6NRsEwDDwe
    L0pLS+B2e1WbzSrzPA9KIRw+fIopKfFi7dp2JZlMSoQQwvMCwzAs0TTC5BqsMEQQIF2+PDoT
    DofFggDUTXn/3M00vianwgPgEz6f9zOPPPxwzUc+/CH0dHXCZrXm6JiqAqqpoJqKnq4O7N21
    HQ/ffxf3gyefrvu3//yvR0bHJxsBfFevU7VhuRpbtgiweEc5KPrXDQB+Zf/+/d2f/vSn4fV6
    F8G8AX6hGSDYCOHRRfBrBsmgGqgGeD0ePPrQg+jr7SZf+qd/dn3zv77TEo3GEsYmFkWR8Xrc
    5JMf+xDnsArCn/zF33afPndxD4AQ1esQCSFRLNExjcCDmY51qwYUeAB3ejyeX/7kJz/hefih
    B8FxHDRVBSEEu3ftxJ/+8efZ3/uDP2o+/sqrAqWU7ti+te5zv/2bpK2lGZoqA4QBy7B48IH7
    cOlSv/2f/uWr62RZTiKn1MsBqGFZdteaNWvaHnzwQTYcDuOLX/yiPDExcRU5anQMS71vDYNd
    GEC40SMNIDDvn0cimUSJ17Ms6ms4LDu3b8OnPvkJyLIEnufBcxwEgYfdbkOJ10uqKiv4qsoK
    lJb6wDLXDzZRLcdS+MSH34cH77kdk1MzmJqZxUIgiFg8jkxGhKIquKOmDg6bFQd2bwXHcbl1
    XGRaNE3DtH/ByI669WxnRgeKVtykmmr97Lm/t6dnz5//jy9wm9avgSplydmzZ/nf/+M/bz9z
    oT+OHK2e6GqZSR2wp3SwbuyXs7F4/F/+4cv/8unnX3ixwm630fHJKdk/H5gBMEgIRErzaN3m
    9mRG9LmH47g/3r9v75Z3PfwA47RZceToMTz17As4cvxVJZFMhrJZ0Y+c2rkfel9hnWrIkbeq
    d9KNG/3BYPDQ3/zN3+zmeZ5kMplsNptNmZwZFkBbNBqtCIfCmJyaUUVRmgeQyOiU8FAEDGYX
    s2UicnXRcUrzWmdppnrXdzyPt4htYT/1qU/1lJSU/FYqlb7/ypWrnqGhMYTDMSiKCkoJGIYD
    xwmwWCyM2+1y1tdXt7W01Dbt2rVrXUlJyQ9eeumlU5IkRQkhCSzV3jOmIPRiv9p3IgvLyObs
    379/HcuyHz979mzplStXiryX5dneHNhFHqXVDEoYhkCSZFy7NoRIJEo2bVrv7OnpsUxNTcbG
    x8ejJSUlfElJiZvnBcvg4AhxuVxcTU3Z6jvvvDPE87w4Ozt7+eTJk1f0bLCZtrtMJfoWA8EE
    ALt58+YdhJBfv3jxUtWFCxehKMoiHdfvX8DLL58gu3Ztr92zZ89+SRJTlDJNr712ngkEImAY
    TmcraZiYmMHAwJClt7dzzapVq+IXL16cBMA1NTWVHTx4cDvLsl0XL14UAGDt2rWKLMvjhw4d
    em1hYSFkssHiTbTB5MqVK/G2traI2+0Gy7JQVYpicTZKgdHRiUUsoGkaNC2nzSHLGkRRYlKp
    LJNOZyGKsgmkLgFhI1tLCCBJCi5cGMPw8BxcLjscDhtsNgGCwIPjckrRkQggyxRzc3ETKF8u
    dsowBFYrh0QiqGWzWbeeQEkX2F72Zm3Vu+++u8Htdn3w8uVrdadOXYSiMHA43KS3t93d3t60
    pbW1NXrp0qVruo+bMiXODCajdvTo0YzX6/lmZ+eqtr17Nx+Ix9M8z3Maz5Nkf/+lwYGBAX9B
    hraQ9YLNmzdbW1pa3i9J8u8ODAyV+v0ROJ1udHQ0Yu3aVqqqWjqTSc3PzU3MLywERmdmZoCl
    dl8ZXaTrlu0pfObMmeyaNWteqKio2LR///4KVVVVRVHSyWQy5ffPZmZmZmSHw+HbuHFzuyBY
    rB6PC9FoJPPss89MZzJZ1WazMU6nk3W5XMThcFCO47RkMhm9evXqiKqqZk2DwoDUDTvLuJtk
    TMxRVRuA95eVlf1/v/Hrv1b+8Y99BD6vB5qq5MRJNA2apuj1iKoOToCGuhp84sOPoa2pwfI/
    /9f/3dI/cI3RQfCkbgRY0+GfN4HvFCNsBAh0WocNwKPt7e37f/3Xf520trbmg1+6BHrNgLcw
    Iwyq5X1P0x8729rwh7/7WbS2NPN//3//wTM1PRMFMB6Px3H8ldeqeld1eH7z0x9DW0sjOXv+
    0lrkaixVCrxIKX1ZF+F5XZXoWyyaz+QAAv9rDz7wQP1HP/Jh2Gw2XRTKmDsNmzZtxBc+//vM
    73zu9+uGh0fwsccfI10dbVAVGSAEhFCoVIPb6cDHPvI4zp6/UPbKq6/1GYchIWTrnj17Wh56
    6CHmmWeeoYcPHw6lUqnzAK7qBiKNpfY/ZuddvtEBG5MImAogEAgE5EQ8zpf6vIsfocG5MfZM
    16pVWNXZkbPIJmaBEWwx5k5TzfWVACijU/GJYTVBQeB2ueBxOdHa1JDb61SFqqpQZDn3HJSC
    YQgYXSCRFmR/F9WtNA2z/gAURaU6kHPrwQUzjfWGGhOGYaiqqg5CyJq9e3Zbtm3dDBYaFCjY
    sK4Pe3dtY89c6K9BjiImAkhQSh2EEKsRJTfV62YppS/LsnxvIBSuCF4LZyVJOqMDNInS1ze+
    +nu9p7ene8Mf/cHvMX3dHVDEDDat7UYmk8K/feN7YU3TDiFHyY5jqUZZLAgeklv0XASlNEIp
    /btkMvmkHvBw6O/Brn/+VRzHtZR4PWDZHNDAEsXbOJ+MTIWRrYjqc2HsxcLe5z836s+GpsSn
    P/3pHq/X+z/8/vk7XnvtFDcxMQNF0Rbrs1iW151uClFUEArFEYtlMDcX5fr6mlf19vZ9mFLN
    /cILL57QNM16nXZcRr37OypQYArS493vfrfP6XQ+PjU11X3mzBnIsvy64NdQ3C0ExEvfWw6I
    g8Ewjh17haxbt0Zobm709Pdfmnv22Wdnt2/fXlFdXV0/MjJNgsEoyspKatrbu95ttQpsaWnp
    4UuXLgXT6XSh2N8N6WH7Vvp/H//4x5tsNtuvjYyMdp0+fQbZrIgc43tpjoLBCE6cOEn27t1R
    WVNThVdeOU/m58NgGA6AtgiANU3DlStDqK6udK9fv75vamoqm81mUzt27FivaVr3+fPn2fr6
    elpfXx+fnp6+eOzYsct+vz+m733D/hbWVN9Q/yUQCMiqqgacTifleZ6oqlQQo19aV3Nz8/D7
    AwWtjZZqhXMUakavqWT0eS48vuniXpYkGbKsIhbLLtYcM0yuBjjXLolZTEDmstNk2Xo3AjhW
    K4OFhQhkWXYX2N6bpmdirCufz9uQzYotV68OI5lMw2p1QZJUTE4G0dRU5aysrKjr7ycLlNKs
    7isYPgKHnGaGRiklTzzx5GRpadnLFRWVO71eB59Op2YPHXr57ODg1agkSYX2t5CBRdetW1fN
    MMx7Bgaulr766nkQYoHFkkAmQ7FjRzedn5++8uKLL56NRCJxWZYNpXM7lthXpGAh3HJb+amn
    njrZ0tLyOQDViUTCHQ6HHZFIxJbNZp2UUs+6det4TdOIqhrrTtMSiWQ2kUhIkUjEOKsMm2wk
    LKKmfWj2h5UbvR+5m2xgCKV0k91u/5WPffQj5Z/65U/C7XJCVWRoea1TDIpvTowHmgZV08Bz
    LA7s3QlJErk/+LP/tX56Zi6EHCAzy2abW3ostkB5J4Bg02skALrsdvv73ve+9zl37dqVn/U1
    1ULmgV+q6YBBWwQphfcZwJlSCq/Hg48+/gGUlZawf/5X/8t39dqwCmBKUZTEteERTywWw8E9
    28ETrTwai5WfOH1JC0ViswAuUEo1HfCZDwNynffztmU9TKCvFMAvb9iwbtOvfeZXUVFevgh+
    FzPnWu61btuyGb/7O/+N/OEX/hRnzl3Avr274HZ7crcaohCUoqOtBR95/APMtWtDDaFwOAyA
    raura7jvvvuYJ554QnvuuefGkasRnjY532mT4U2aWgfkOdw3Yt4KIrmBQDAoJRIJftFRywkd
    5NoZyTJEMQtRzEISRYiiBFkSIckSZFmGYlyqAlVRoaoKVEXR2yCpOvVeW4wkM7rqJEMWWzKA
    ZRhwLAOOY8FzLASOA89zEHgeAs/mAAylS9FtEKg0B4IzWRGBYMTI5HFYquX6qcHcm02AmtsY
    5AJ2FCB0KXiXq5Fh9NdmlA8IKNI/UP+8q3xeb/Xe3TvxzHMvimFJmtYdC05fH4V1l2YDwQFo
    7epaJbQ0N4KqObVjr9uBdX1d+Nb3BC6dyVp0g2sYoiyWi2rRW/Fc1B9V5GrsJUppCXKiGT7k
    GEUSAIfVIvC1tdVwOx3wetxkdm7eyIwZ9PGssfd04GsA4IQ+1zE9Q3BTHOKbDEDw+OOPV3m9
    3t/0+xfuOHToZW5mZk5X5uR0iiVbRAQnR5UMhxM4e3aErFvXVt3d3fvAwkJAPH/+/CXd/jIF
    AWhzNh036ky7AWts8XU2NjZuFkXxjvPnz3O54918P/O6Nb/XA8TLM8UEqVQGp0+fhywrQl/f
    6hpKKY3HY1pzcwtta2sCwMDvj7I8b2FranxaJpOBJEl2AA5KqQggq/dBZwvPvbdz7o33qvt1
    ZN++fY6KiooPRiKRA6+9dpJEIrFFoLUEvnL2IRgM48SJ02Tv3u2oqCiB3W5HOp2ruDIAMCEa
    UqksLl4cJLt3b6np6+trSyQSCZ/P12qA35qamtmTJ08eO3PmzJgoikbNphn8GlR+s7jnDduD
    iUSCyrK84HQ6ZJblhKXy7aU1w7IsOI7Xr5xQEMsaFwuWZRdFrAwAyzCMae+aQbLx+ZsfyeLX
    mkYW49m5WuCln5lBr9mUchwDQWBpNBqjiqLwBbaX/VkCqT8FCYkabJ2lOWDMgJ5oGuVM9tdC
    KRVMQbvFc6u8vFzweNzNoVDMkRNJzESGhoZnJUky3o9UcOX5bBaLxacoSsPs7DwUhcJq5cGy
    POJxEcmkSBRFsYbDYYuiKC4siS2mULybw60YgKbz8/OZ+fn5UeR0T0p1++vT949SWlqiqirl
    RFFBNqvAYnExVqtVSSQShg02wG/SbHNNQfkYgLi+J6UbHYTmbpZh0bNwLgDv3rx586pf+thH
    4XG7cllfI0sJgEIDXQRwWFSCBXLiToQA+/dsx7mL/ZYv/vPXNqqqOoUcjY8WKFTelAjCDQLB
    FgD3dHd3r3rXu94Fm82mO9OF2d18+nPez/Lu1fS5NAFjXV3Xwgt4+P57YbNYmC/82V+UDQxe
    AwDx3MV+5bOf/5+cwLHUYeNhs1iIoihpACE98yIhR+E0mqCbZdwXi/ffDgNcRMiEA3B3WVnZ
    w5/65V/mV69erSsdm+bCHCQAxYH9ezE3N4e//+I/oKWpAR98/3tywkyEgBIGBBQMy+KOA/vw
    4qHDlm99+7udhBDmwIEDwvT0NF566aUQcqJt06YIfbZIBtgsKnazMk4agDm/3589duyYo6W5
    CUPXhnD8xHGEQiFEozGkUikkU0mkUilk0mmIkgRFViArit6yQQe+OtglWKJcgQDEENTAUiaX
    6gEahiFgGTYHglkGHMeB18GvzWqBw26H02GD026Dz+tGqc+L8lIvairL0VhXBVGUcPjEKYxM
    TFNTtHBRlE3v6aj8DPNJXid7vjj0llIpSum5Qy+/fGDfkV3WrZvWQpWyOH3mLA4dOaEh17rI
    2BcscgqSbAHdier/72lsqC/v7V6FZ184ZFYjNtcPZ00CRIVrJjw3O6dFQiHGUV0BhlBkRBFT
    M3OQZQU6+DWEPwzDC9y6zA0UySjKADKEkKSe0TQiylkAdSBEmJqahqaqUFWNQY6SZ249ldKN
    bFS/wrrhTRZc5qDUO3aYAV1paSlXU1NzVyqVfuDVV1/jpqam9awvkwfYjMxSsaxTOi3h8uUp
    bN7cXrN27drbRkdHY/F4nNH3h7mXpXn/veMCCHv37nVYLJYDExMT9aOjo0WzvsXAbW7uitOf
    zVngwt+VJBkXL14GIYRfvXp1zdjYWGh+3h9paKi0MQyhDMMxHMfzqVQ0eObMmSlFUdymeTb7
    O4o5K/V2z70BfgGw69ev36koygfPn79gm5ycWpYVz8+aM/D7Azh16gJ27NiI3t4OnD8/CFmW
    9TXN6H6jhtnZBUxOzvJ9fX2tsixLwWDQxjAM6urqEidOnHj11KlTo6Z1KRbJAJuZWMqNmDMD
    QGQyGUUUxVmv1yvX19cJIyNjqKqqRE1NLWw2K6xWKywWCwRBAM/zJsCbs5WE5ALDuV6/TIFN
    MrfvWvo6vwTC+EwMwKszuLTcpSgUiqJBklSdaq1BFFVksyoyGQ08z6Cy0gGeB43H42YBPPMa
    /KmYCEVKJd/InJJwODJRUVE5vGpVR3kqdQWKQmC3W9HUVAmW1dKzs7MRvUbXaHPHYrnehbZ2
    7VqvxWJbPTo6xziddlgsjGFbDPtjMIeMgFMecy+dTosOhyPq9fpqOC6on60c7HYrLBaOJpNJ
    RlVVN5ZamZmDz7esgJ0pOKiZAvEZSmlcv0UBkLXZbLSyskrgOI6prCyF2+2A0+kUysrKuEAg
    IJl84ITJ/kb0x4QpKZTQ96QxvzfsHONu4kFIADQ7nc69Dz5wP2lqbFzM+ubRdRfDUVrB1wZQ
    0WAVBNx9YA+eeObF0rGJ6VW6E6NQSiV9YWaRT2V7x0Tq9cVWy7Ls/n379lk6Ojry6nvN81M0
    I5wHivEGaNIaGEJw5+37IMsy8/k//fPS0bHxSDAUnn3h8DEDyFpYlrGoqmaoQZeaFr7Z2VFM
    mc63dSMzDGMAFAKgk+O4jz300ENl996rt9/RlbLNGWBz1pzjWDzy8P0YGh7Gv37tm+hob8W2
    LZt0RlFOcUJTKEp9Hjz2nkfJ8ROveFmOx5o1a/CVr3xFlWV5QnesOeSrI6ZNm9wwvjc0+lyw
    tozs30A0Gn32z//8L+6ZnJz0EMLglVdOgOdz9b0OhwMupxOVFRWw26yw2Wyw2aywWSy6geYh
    CLmaYI7VwSxDQBiyFP6lFBrVcq2OVBWKokJWZMiyDEnKZZWzWRGZbAaZdAbJdBrpTAbJZBrx
    eBKzc/NIZ7JIpzNQVBUNtVV49N4DeOXMRTzx3BFtPhBMFR6eWKLWiG8mAFbIGDDV4TOm/6Ng
    XRug7ImBgcH1n/uDP9q/bfMGKwHFK6+d1gaHRmaRU1CXTb9jGF6m4PmcAHp7e7ottbU10FSN
    mtaM0RswTQjJIF/9XjUFl06dPnM2+n+/9E8l73n4PtgFDkdPvILvPfmcKivKAvIVnpUCJV9z
    DQ59OwzsGwjaGCrabO7Xcj0ddTbKKkLIRp7j7E8/fwgcyyKVTBKb1VKeyYoKclTylA6eDQMc
    1G1HzOQEm+d2WVDq9V7rLc4wIvfff38Fy7KPjIyMekdHx0ztV0gReiUpmgkmhEEyKWJsLMj0
    9FS1tba2dp87d07Ug9CyiREkvRMDCAYDq6urq05V1Z3j4+MkmUwWAb9YBmqvD4rfyD0MZFnB
    pUsDEARBaGpq8p08eXLu0KFDM16vF42NjY6Kigr7/Pz8XCqVsjqdzupUKgW9R65aJPBwMwOq
    P3F85CMfqbXZrJ+4dm2o6dKlfmiaBoYpTBbmzxVAMDExi5KSEXR3tyMcjmN8fGaRHWhkgVVV
    w+DgKOrra5wejx2XL18mLS0tmt/vn+rv7w8V2GCj9CFVYINvWBC6wPYq4+PjZ7u6ug7fdtve
    2yoqKuxWqxXV1VVQFBWKokCWFWSzIhKJlP51znaqao6dpap0Ebias7fmtWY85rKizGLGOBd0
    ZvUsc05dmuM48DwLns89CgILu10Axxn3EsRiIubm0qiudsLppNrVq5cT09PTUVMmz2x7zeyk
    n2aujB72eSnoIjaYAmB+9KMfTb/73e/6WltbU5XH42mMx9OMy+VCSYkzOzDQf21sbCxk+kxJ
    Ifg1zu2mpqYmVdUa/f4gWlrqQAin6XjCsIkZ3f6ahf+MM445f/58cP/+/Wd7ejq7JEkjoVAG
    LpcbnZ11IERJDg8PGX29of+OfB0bTG9BG2wuIcoiV75F9fNHbGhosG/btm1NdXVNKyEs6e1t
    ActyEARe2LlzZ6Pb7eYuXrw4nclksrp4X0wHv0H90Rx4zhTx4Yie7X9L7e9NE8HSX1xPZWVl
    9aZNG8GyLGRJyc9mmtWMYcrMFdyjUQ1N9bXo6Wznxyam6wBUYEnUwHBkjEXK3OpG2GRQjVrV
    7oqKip69e/dCEARoOlV30VhSbTECqqla/vxAW5YlXkaTBpYAtP59hmFw9x37EY/H2T/+n3/l
    nV8IZAEEkKuXYNWcQ24HsAY50YOrRaLO5poI5e0EwZqmGfNpB/De7u7ujR//pY/B5/Uu9rKF
    uXft4sNSYMDtcuLjH30cA4NX8aUvfxXNjfWorqyERlU9xEpBGaCxoQ6VFRWkb806TExMYHBw
    MKlnyi2miLOkAxjDYJiNxk2jW+pMCY1SOgfg/0xNT8/83d//n/fs3rWz/r/91m+Rnp4uWAQe
    FkGARchRsIhxuOhOh5lGD5MzAppjFSx+D7n+Ckv3akvOi6aZhNq0xefWNA2SlAPG2UwG6UwG
    qVQa4WgUx187j7/9p/+kA0NjqqIoKX19hrBEnzGorOkCg/Jm9qJhfO2U0gYdICWvYwyMg3lC
    VdW/HRi8Oj4wePVuQRCaAMiapg0TQsLUoLgsGfNiALjC5XL2rF/TB4eu9G7KWAC51jNm42AE
    TRT9fLUCaJZkyfLVf/9PPP/CIdViETA9MyeGI9EZAOOEQKIUGSzRf411aA7CaIXphCJGJ4+m
    X8wovZ4xup4RM7XHK3axWKotM6hURI/sr3c6HO89sG9v/Z0H9pD66goQqiAWDeNi/xXrUy8c
    bbw0OGrRI9aGCmgC+RTowt6qVP+bgp6xLxTloAWO2C3bAs6Y9tLS0q5sNts3PDwCSZLAsgLy
    Kbz5YDg/E5wP6ILBJESROurr69rOnTs3g/zWY+Y1eku18ngj8wSAdblca9LpdOvk5GRebSUh
    BBzHQxAEEEIgy3KOAfM6IljFs8ZMUdp0LhM8AIfDYV2zZk3l/Lxfq6qqsvb29vpYliWlpaXu
    VatWNSYSifnLly+fOn/+/DlFUQrrEs225G3RQTEFFInX6xUqKyvvTySS+8+cOUcSieQi+L1e
    EMH4maZpGBwcQXl5Kfr62hCNJhGNJnRavm53oCKVEpFKZUg4vACO4+DxeDLPP/98MJPJGBRd
    Y29ndPBb7Py7oXX/OmCgP/7xj0fm5+f/ZsOGDVMbN65/aHJyuvLIkeOIRuM6yDUAbm455mji
    TEG/YDPdmc3bp0v3FP7M/HN2GcPDoFMboNgAxlargKoqJzo7vTQWW1AOHz6TuHbtWiCbzRrs
    mVgR2/tTAWDTOmU+9KEPldvtdu+TTz45Mz09rRUBvxohhAaDQfW73/3eszt27JBra2sfqqur
    3AXAubCwEDpz5syIKIqiaT+YVZYXGVgVFRWMz+frjcWSZZFIfDHjqQf0jPdSaH/NNpM0NDRU
    EMLU+HwesnPnBppKZTWOY6miiPHTp08Oj46OhfT7Uyb7kzABv2LzRoqwz26I/dX/1rLAg2mu
    zFpLih4coNXV1ba77rrrgMViu2dwcNwWDCZBqQBBcMDtdpOqqpKSHTv2OCwWi3L8+PGIqqpZ
    neIc14V1o0VYGIZvxemvt5j9zbPDhe/rjZx3N1MFmgFQW1lRYS8vK8uBENC8LFwh0C38v7nN
    j8NuRU1VBQC49IxkHDmRmZhJZOamFOO/xVFCK4CdHR0d3r6+Pj20R6HIMmamZzA+MY5QMASG
    ZdDW2oL2trbcAWkCv8tp0q8HirXFfq0sy+Lh++/G7Nwc97//7z9WpNJp6PPos1gsbEdHe4Wm
    Ku1j45O96XS6HMBLpkyJ2QCba7HflmCCaRNvtNvtj37gscesa1avXgJji+suv7Yapl64mqah
    saEOn/ilD+MP/uhP8b0fPomPf/gDuebdhOifi4rDLx+FomlYvboPX/nKVzRJkmL6enfoBxvV
    AYzheBsGw6g1/KkNxk85PxohRKaULgD4QTabjb585OjHOjs7WrZv2wKX05HXG1hbZGJoJsBq
    ANklYLzsZ9QAwPnfz69ZN4Pp3EzxHAfewcLlsOWCM6A4f3kQR0+epZcGhmTTHEYKLgPMpPCz
    0dk4AA8C+DSA5wD8s/58BPk1jrLJYE8C+B6Aqvq6ukarzcpdvnzFSimVka9JsKyoSg96dZaX
    ldas6etBIBg0lDypKXssmoyuuf+n8Zy7HHbbhzvbmh3XRsZTV4dGTmKpv28mx5CBpM9RSAf2
    EX0ezZRfanIMyHUiwcVarRATOH2jhtf4BqPHHhh9LjjTZaarLX7fRCHnAdzW0dHesGvHNiwE
    Q7jU3w8xk4FFYFHmdWLb+l5mana+LBJL+gBMU0pVff0X1nEZ9dq8+QzTwa9ZW8IsOlS0jcpP
    Y4xv9NYXBL49FIp4I5EI8nuQMlhey8oUAWxLTrQoKkgmJcblcpfZbLaKTCaTRq4cwGC3vKPs
    r3ns3bvXJgjCjunpaW8oFFpkFJWWlqK1tQV1dXWw2x0ghGB2dhanTp1BJpNdNof5AljkDdQN
    576fyWRx4cJl7Nix2bZly9byCxcuhKenp7Otra2e2Vk/6/cvOBsbG0q3bdvuk2U5ceHChbSp
    FMk4HxS8TSJkhWVPH/jABzoZhnnsypUB1+TkZFHgn58Bzv9ZJiPi4sWr2LNnE1atasbZs1eh
    KLLewzaHZWpqKmG3C7h6dQLt7e10amoqMTY2Zu4tSk1BaHPANGUCH3lClDdIh0OllEpnzpyZ
    Hh4e/ua+fftiLS0tjy8sBKrm5s5AVekygPr6rbUYUwslmMBvMQYHlimS538WOUq0LGtQFBWE
    aHrfYR5eL0dHRwfEEydOJILBYMxkb43HWBEA/FNP4Ic//JGq2tqa32cY0nHHHXf81b/8y7+c
    prneObQgG6kSQpj5+Xnxe9/73qtbt26l27fv6IrH406e52xOp5MEAoFCwUhSEIDGtm3b3DzP
    94bDMSGVypjXpWqyEWIR+6tQSunttx/wtre3f0SWtR3ZrAhC5Inp6dH+aDTGjo+Py9PT01k1
    R0lM6bbXsMF5PoveFSIvS13EbhbzFQ1Qr74Z4Guo3RcE5lmz/dX78bKU0rzvG9ns1atX13s8
    nj3Dw5O2RCKJ8nIv7HYHWFaApnFIJlU4HDZLZ2dndX9//1A4HFb1zLFsYgvoWk3gKIVRs60i
    x3BbZn919pdaYIPpmw1I30wATAAIHM+xHMctA2qUasvArrk+Mx8U555XEDgQgKe5Q8647Mhv
    g3Jrh5sJKZyjKp7jt27fto0pLyuDRlVMTUziW//1X3jmmWcRCARAQaGqKu664w787ud+B26X
    KwdSiihAF1Kel9Ok83/HbrPig+97FEPDI8J3fvBkiaZpc8gp1Fas7u2xfuSD72WeeOKp0i//
    239uTaczYwBilFJRjwZlTNnOt8XxMRa63qjbB+A9W7duaXv44YfA8zxUVTGtMeQHXxZbSC31
    WSaUYte2LbjrjtvxvR8+ha2b1mPt6j5AoyAswdT0LH7wxI+xceMmjI2N4erVqwb9zLwGDcGh
    ZEGmMluMLnQTnGXNlF2MA3hFFMXqH/zwiY88/NCDjm1bNkPMZpHVBbAkScoJXik54StJpzDL
    kkkQS6c5q3p23VhbhtFlCAGr1zGxLANep01zZmEs4+cMWRTH4lgWqqLgu0++gAuXr6mmyGlc
    N76FxuRNR6HNDhvDMJRSWiXw/GPVVRVbQ+FIdTKVHqCUnoSuGGnOQJoAkkwpTQOQqmuqUVdb
    x16+fMWn37tIcdcNeCEzQgCwvqW5yV5XV4tAKARGlzA2He6FCtCyDuQ0SmkbIeRXtm5c2+p0
    2HHh8uAUgOP62vPoLA5VX38B5ISkAibqURY5CqsBKEv0szSk9yfUCoB/tsi8stcDwNcBvka0
    2QC2bgBlAKqQa/9Wg5y4hl3/ufF3QwBmKKXzutPlA9A0PjGBf/7Xf4fX40FZaSl4nkMqncaJ
    0/0Ih8NIZ0QeOZq5gJwQilf/uldntNj0v6OZ9kUQgJ9SagQKUnogy1zzJiGfemoODtAimY23
    MbNJSiRJtEqS9DqOMrMsE1xcFAuQJBVWK2cXBMGbyWTcejDagSU1WBbvAABszlYCIO3t7fWa
    pm2Znp5mUqkUBMGCVatWYf369RAEHoFAELOzcwAoksmUTj9lriN2xbyOCNb164IjkTj6+6+S
    TZtWOyoq5lLPP//8bCKRyDY3t1b1919jMhmFbNu2vqysrKwGwKhuo7Om3sxGQPptndqdO3fa
    PB7PI8FgcM3Fixchy0oe9XmpVc/1QFtuDQYCEVy7NoHu7jbMzYUxM7OwmAG2Wi3o6GhCMBiA
    IAjw+XzayZMnRUmSDNFBxrSvU6YAasLkr5jLVG7UXjWylgqlVIzFYpHnnnvu8COPPFLX2dn+
    6NDQqCUUiuQJYBnCVzn6MrdIXV76Hrso/FS4p82x1pyLZ/4/TD/Lx0DGvZQSCAKHxkYPFham
    5cOHDyfi8XjcFHz+SbaXvlnfzdiLlZXlO1WVPiKKamlpaWm4q6trbmBgIKGzhFQTIBVNoFCi
    lKYp1VS/34+6ujp7bW2tc2xsbNacnDHZ4MXPuaamplJVtZ5gMKq3goPRkSUv4F2Y7KGUaqWl
    ZezatevuBph3z81FLOXlLnFycuLsE088eU7TNK9+LrL670R1+7uApdaiKdNe5err661r164t
    j8fj0pEjR6ImfQXNlImWithfFtdhERarrzbZX7aystKyevVqX2lpabnL5aq3Wq11LMtWEELc
    ur0kqqoqoigmYrHY/PT01MzIyGjQ7/dnPB5PgyAIJQ0NNaiqKkcmk4XRMtnhsMDh4OFwAIQ4
    7Ha73R4OhwVBEGyVlZUVzc3NVfX1dW6Xy+0WBEEghDCapimKIqez2WwkFovPLywsBBYWFmJT
    U1OxTC7SKOvlNoUdMcxB6Tw/63r2l7u5BhjJRCIhJZMJG1CQ8dU/nLw2K4XtVrBU/6rICmKx
    hA5XwGNJYdViij7f0ga4iFgTAbC2rLysc8fOHbBYBJw9exZ/8qd/ipGRERzYvx9btmwGQwjK
    ykpRWV4BuyGQZaakFsxhfra3GC16CRRrGkVleTk++dEPYeDqkO3S5QE3gDlJkuiJV16t/+B7
    HmTKSn2gGiW6o+PQI1cZndtvNUXH3rbIs+7Mb/T5fHe+/33vZxsbG6Fqqinbizzgu5QBRt78
    aVSDzWbF+9/9MF49eRrf+s4P0dbSBIfdDkXR8MJLhxGOxnFfTw+++tWvQpZlRp8X3mR4jVoP
    swL0TVG5ex1DrJheU4xSem5+fv6eL//zvzQfOXIUwWAQ4XAYsVgc6Uwa2awIUcxdit7ySFc5
    XnpKM40837ItewGMbqRZltFrj/hF6rXVIsBht8HltMPjcoLnOBw6dpLqANOgzIR1EBfQgUpI
    70+dMM3tmzLCJoPX6HY7uz7x4ffhhUNH6186/tojeoZXNIntZZHf99QQh5AtFgvWr1+HJ598
    ypdMJbmCz1rD8hqfEoZhenq6VjE+ryfnhuR4b4X9FDXTQa/p4NcJ4IMdrU2777l9L/m3b31f
    UVXtmj4PbtM8ZPW5CxhzpWen7frfSep7dweAe3UA+lVK6RETDUw2RcPl6wBg+SeccSBLakt2
    AM0AtgLYYrFY+ux2e53b7baXlpQIbreLs1itDEMIURSFptIpNbAQlAPBYCqRSEyoqnoagOD1
    eusfvP9ePPLAfWhva4XNagEhgKIoiEZjOHn6DP7ta98kr546U6uqahhAMyGk3G63VXjdbkd5
    eangdjlZnuOIpmkQRVGNJ5JqJBqTk6l0Jp3JzMuyMgxgAMAVSum0nkUSTVkBEcvbU2lvxAjf
    rONR0zRKCKMxDMvm11syeXWDyzOZjClLvASOGYZAVTVG0zSrfubZ9DVkMa3dWxoAFwnOsF6v
    d10qlV41MTEJSoGWlhZs27YNgUAAw8MjiEYjaGhoxOTkFAKBAAzl4nz68+vX/b6+WFbuZzMz
    fpSXl7KrV68umZqaSl6+fDnW3t5eVlJSIpSUeCGK2ejIyEhC30dO5IRp0rodNqtC07drPjdt
    2tRDKX3w8uUr1kAgeN1AQGF9dSEDk1JgZGQadXVV6OxsQDicQDYrglINtbWV8HodOHduEJ2d
    nXA4HMTtdpsD0EZJlqgHCMw2WLwOo+VGBp9lQkiWUppOpVKhiYmJ0xs2bLx948Z1FYlEEg6H
    HVarDRaLsEhF5jhWb1dkni9ch6hD8kSw9Mxa3teFIliqmvu/quZEsBQl12+YZVl4PAI9eXIg
    FY/HYyYG0YJ+BQGETbb3Z6qlppSipaWF5zh+/cJCvGRuLka6umr3rFmz5uWBgYFjyGlyaDr4
    SRdkREWO47IMw2jxeAKxWEyor68v4XlelGW5WI2tcbF2u71dFOXaYDBqPqc5vcWqWdBPLchM
    0ve+9z3dFovlk+Pj4apsVgHLauGhoaERLecgGb6COfsb0OcsUVVVxVRXV3sBWEKhUGbDhg2N
    TU1Nd9hstgOSJE2IovgPr7766iylVCkQ95SKYDkBBWVfhVlQU4kXAcDeeeedpa2trb0ul2sH
    wzAbVFXtVBSlJJPJWlOpNC/LMqtpGsn5aiy1WCxKdXWN1NDQEN6wYeMVv99/qaamZkc4HBUu
    Xx7C9LQfmUx2MSjIshzcbidaWurR2lpn7+3tbfZ6vXxXV9eG6urqKp63lMiyahFFhUunNTaX
    vCKUZa2q2+1UysoqpdbW9riqypOpVHJobm6u/8qVgWvj4+NBXZRXNAmSFWpPLLO/hcGtmwKA
    TY7yxOzsbHx0dMzWtdhb1ATOkK9svESrzO9pCwDBcATDY5O6Iw9CaR5VjnmnRJ8LqAgOAHu7
    urpK1q9bB1EUcezYUaiKgk/98iehyDJCwSBePPQSPvfZ30ZHRxsURV5Wz5uXCYa2vM5aV+nV
    zBRoc49XAH09q/CRD7yH/Mlf/K03Eo0lAciKItPgwgJOnz2PTK7+Q8RSHzOH3uPUoBTd1Pov
    s3Opb3IvgEe2bdvacPDg7bmMsKpgUZEYRWjQdDkNGhTQqIbW5ia866H78OWv/jtOnzmHPTu3
    YXpmDk8+/Tw2bd6M8fFxKLKIvp5V7KXLg3Ys70ttRKvM6oHqTYg6v14G2JCk5wBMS5LU/81v
    fbtc0zSbQalpbWnBQw89hMrKipwIltUKi0WARRByAli6YWYZnWKlJwGpXg9MqQZVVXMtkhQF
    imyIYEmQJAmiKCKbzSKTzSKTySAYCuP7Tz2HYCiyOC0Mw1BN02QsiSaETZlMPyFkQTfK0YLM
    +pul4FNN01gAbQ6Hw7NtyybYLALz2tkLW9KZ7FoAw3qmVDW1HWGwREUWAWQ1VaNr165BWXmZ
    M5lKegBM/ITX0+B2OVu6V3VC4Hk9M84yJjChFGNV6EGeXS6n4/3ve+Q+G8MwGJuYjuuv09zq
    SDEZ36Ae8CgBsIfn+PUMywiSJI1TSt3VlRXbN63vK8lksuT4a2c86UzWQSl9DTkBEHONZ7Fg
    qjkgsGyLGoaXUioAWAXgfpvNdm9dXV1nb2+vc926daS3twf1dXVwuZywCBZwHAuSY7wQURSZ
    eDzODw8P248eO15++MiR1YOD19R7777D+oU/+F2UeD2LdH2qi9z5vB40N9ahpqoSv/bffrdk
    cnp6a3NjA79x3Rp+66b16Opsg8/rhkUQwBACSjXIssxks1k+nkhap2fmXINDwxWXB4d6r42M
    3zk7Nz+byYpnKKXHAVwDEDUJoxT2EVZuIRBMJEkK2Gy2jN3u4OPxVNFM5BIguX79L5BjZ1it
    PGKxtJbNZlksUcfN7VCYd5j9JXfeeadbEISDk5NTrmAwCIZhkE5ncPLkKUQiEWzZshlWqwV2
    ux3z84FFVfvX6wNcmAk2AHOudyuuWxusaRqGh8dRWVkubNiwvvTcuXMhVdWoIFj11lUMZ7PZ
    HKbAg9ET24Z8DRTtbZhP3HPPPQ6Hw/HA3Jy/8+rVqybhK1xnzRWjjS9d6XQWg4MT2LKlF3V1
    FRgdnYUgWNHaWoeFBT8EQYDL5UEkkmTa2trs/f39vLlXsiljlEdhvck2OA+MU0q5YDA4pijy
    QG9vt4tlWQvDEIYQwO+fx9Wr15DNSjnbqag6SNUWwWvuNReuMXNdb45pZWSQGYZdstksZ6r3
    ZeF0WtHYWAaez7VEU1WVSpJMFxbmxbm5OcN2hHTb6wcwTwgxdDiMTKZRSvNTrbnPfvaziEQi
    HkJIazyeYCYm5lFV5SspLy8/UFVVNeH3+zNYEtyz6Os7bQKmGYZhFEmSMD+/wHR1dZaWlpYy
    fr8/e73X1dzczFsswpqFhagjmUxD718LALxeTqmZgrx5zJ6HH37Y5/V6PxyNZjZMTYXR0lKO
    ZDIxMTo6GigApFmdVRQCEHY4HNmDBw+ubWpquo3nhSYAcVVVZjiOX5tKST2BQEQoLXWu3rx5
    s5BMJv+jv79/GvlCqoXDSAKmr+dnmDSGyN133+3t7OzcYbPZHpVleXcgEKien5/n/X4/gsEw
    MpnMovCaEdhjWRY8z7N2u8NSVVXpamysa6ivb9hNCBGOHTtDhocnlgVVGUZDNhtBJJIEwzBc
    X9/q9s7OrpZ0WrJMToaYhYUYUqksFEWDppHFGnWWZVme5wSr1WJ3uexen8/e4PU6t3d0dEdb
    WtqG5+ZmT5w/f/708PDwtKqqhqiduT5bxhsQ9uRu1mGoj6FQKDz8/PMvVO7dvRMWi5BX/7cI
    erWlDB1MAjlmIHy+fwBXro0YdIBl/G+8g9ovGCqzlNJWQRD23H77AbastBQgwAP334/7770X
    wWAQ/+13/jvGxsewbs1alPh8OXGs6ypA06L0Z4DqbW7SqCwv07Wc8u+llIJjGdx7x34ce+U1
    /ns/eroUQDiVSqmRcIhd19OOl46cqIgnU/X6hjZnAAzjy5uAwU1x/Ew9BxkA60pKSm5/+OFH
    mJqa2kWHOI9RoM9JrnaGzS0iHSSb5wSUgmEI7jxwG55+7gV8+wdPoa9nFV546QgisSS6u7vx
    9f/8T7z7oXvBshyuDY/ZRFEUCtZhYQ3hDe35+wZBsGQKFs0B+HdN0wYArNYzczVdXV2u3/qt
    30R5WRmopi6r4V2s49UKBLIWa4i1JSEsTcsBFNPXhvAV1anT8wsBvHr6PIKhyCLtSNO0jCl7
    GdQjz/MAFnTwG0C+mu/PIipGAEixWCJ77JXT6GipR2mJrzI9M/c+AJf1TPAElvrzMjroNqKz
    WVmWUFdbi97ubtv4+Hg1gHOGo6Xv9cWUml5X01JaWlLVtaodlGq5wALPGQDYiiWBocJ6nTpC
    yC/t2rax+a79u/DFf/ka4onklL4nDeq9ooPXmE65MtqYfXxVZ+fB2/bssjgdNrx28tT2sxcu
    4eC+neSxR+9FOpmAIonth0+c/hjNZUuuUUp5nT5VzG4sAzzGfjRUtfXX7QRwL8/zv9Ld3b3x
    /vvvF+644w50d3fD43aD6L2f89aPKcAHStHX04V77roDT/34aeF3Pvf7uG3PbpT4vNBUtWi9
    OSFAZ0crWlua2KaGOvvv/Oan0dvVmcsUgy4GAqmZSaPv+7U9nbhr/y4kkgkyPDphO3HyXOsL
    R040XbpybUcmK74M4BCldMqkKJs0GeHCusK31R4Fg4GhqqqaYGVlpXt+PnSdVkfXy0gyearR
    DocVDgdPr12bl2RZpigumnJLjyIZAdLZ2blaUZSdw8MjyGazYBgWs7NzmJ2dg8UiYGxsDC0t
    rRgbG0AgEHidOVvK8harDS4p8aGmpgrDw+OQJNnUDzf/SiYzGBoaI6tXd7r9/vlMMpnI1tVV
    WebmIqCU823fvmN9OBwOzM/PJ7HU69SchTfYFjdcCKtgPklbW1unpmn3XL06aAmHw3nsgUKw
    a94e+WAYefMxNxdAIBBBa2sN/P4oyss9cLttGB7uR0dHB6anQ4hE4ujubrDW1tZahoaGCsGn
    VswG30Q7rJkyeAwAdmhoaMJut3+5vr7+vNfr7XM4HI1ut7t6dtZvP378VSiKaupvy+T93xC6
    yqdBL7/X3B/XuMegTOeCEgx8Pieqqrw0mUzQcDishkIheW5uLjU3NxcNhUJm27ugg18jA2wu
    ERF/BttrdIYhmqamfD6X6nCE2FgszVRUlG+59957xUBgYery5SsjIyMj85RSq4lNlAFAVVUV
    GYYoAIHfP4/e3p7yhoYGl9/vnzedw4z5am9vdzIM0x2NxrlsVgIhDFSVghAiEEJspnWyWJer
    2zOupaVln6rSR0dHA4Isq/B6reK1a6PDqVTKAKFGkCVJCIlQSoMcx8Xuu+++TQ0NDb/u9wdb
    AoEwsdudtLGxGrKsYWhonkSjMmpqYGlvL7lz8+ZN6dHR0W+k0+mYXo/LXQfLCa+T9FsEvx//
    +MdbKyoqPi7L8ntGRkZqBwYGyPT0NKLRmA78SRHBtaW9G40mMDe3gGvXRsnu3VsdJSU+zM0t
    LNur5iCXqmpYWIigvb1RGB6ew8jINNJpyVQ+wpr60DMgRAEhMuJxEYFAEizLwmazMD6fo6S6
    2rO5trapr6qqatfVq1efPXbs2JlkMhnW9SfMqu4M8ksMlwWhbxoA1v9oUFGU555+5pl1dxzc
    bz+4f5/+c62Anqvp13JhLEKA6Rk/vvX9H9NYPJEFkKZ0WR8yGTeon9sNogoRvffvne3tbW23
    HzgAls218amtrQFAUVFRjr/6iz/D4OBV9HR3obq6EppB6dX7qxYVuyrS9ujpZ1/EydNn8IXf
    +x047LZlwBmUQtUoSkt9eP+7HsSpM+cdUzNz6Ug0nv73b32P37ttA2lvrnOe7b+6jlLM6ZEt
    nhAimDIBZit3MylYRKeF3r9+/fr6/fv2gZBctnsJ/5qBMJBMJDA6No4SnwdVlRWmEIqpNlij
    qKosx8P334O//39fxg+eeBpPPfcSNmzciImJCfAswV0HbkNWFPHVqgp+bGLKbjpkf+rG8Dcw
    Cm0c0FkTgLmqG7TzABoBPMDx3FaWZXWQa1pPJoBCdQBLTWB4CRib1qhmFr1SAS2f6WGKWZmb
    xBtN06MF4NdvpmBdJ/v7Zp0bot93JBqL/dv//aevfrypvraEFyz85k0bt4nZzKaR0fFwMpV6
    iVJ6RKdIGU6VIXaWkSSJ2u12bNq0kfz4mWeqNU0zxNDMasZGlswGoK+2utpaW1MNTdPA8zx4
    njd+ZinIsBIdOFsBPNRYV3Pb4+95CIqq4uLla4oOzhV9Dxp1Uka9tBGp393S0rLvT/7oDy37
    9uwAVAnDw0PkC3/213jp6Kt04OqQZrMIzEIgRAghpZTSOgAzeuRdJIYKS/7IU7Y2axuYwK8b
    wOM+n+83HnnkkcZPfOIT6Ovrg9Vq1R0fDVRV9XVWKJqWryTO8xw8Hg9sNhtqq6v0Rb1cINI4
    91iWhSAIaGlvxab1a3LlDYtrc/nfyfv7oHA5HFi/uht9Xe24a/9O9qnnD9d++0fPPTwyPtUK
    4IeU0kE9yCDo69DcY3IZLe1mK/JSSnH+/PmJO+6oOt3R0dYyPj6FVCp7XZGd69X/AgQsy6Km
    xgNFScujo6PpAlBRKAp2yw+DFrhz506n0+l8cHZ2tn5sbGwZPVdRVFy6dBmDg0PQtJxa71Ib
    qTfS7mgJEJeXl6G7uxMzM369R/f1a4JnZxdQX1/Dtra2uq9evRpfu3at0N5eaxUEgXg8VQ1r
    167tePbZZ+f1tWe2wWyhDb5Ja4+sW7fO4nQ6DwYCwVXXrg1BVTWwLFc0wwsQ8DwHr9cHRVGR
    TGaWZYiNuZUkFcPDU9i6tQ9tbbUoLXVhfn4WFosFDocHly4NIpsV0dHRwDY2NjhGRkYYPZv3
    ujb4Rs9JgZKvcRbQnPK3xJ46dar/zJkzfrvdfrqkpKT1nnvufjelWvfS7xXPii+B2Py1Y/QJ
    NtcFL91z/Xp0WZbx6quvSpcuXUrJspyWZTlhYl4t6HbXj6UODAbzKonrKGm/mbn9q7/6KwiC
    EHvPe97zb9XVNfWbN3fspBRcPJ6sALgHm5rapLq6hqETJ44/de7cuQE9oGz8ATmTyUiUQuJ5
    HsFgGOl02tHQ0FB79uzZEUVRqG47OUKIIerE1NRU16gqbY5EElCUHO1bVVUQwlgYhrGa3hNj
    ApLMgw8+2Gi1Wj8yPR2tmZmJor6+BIRo0eHh4Tl9/xEstd6KGgysnp4eS2Nj4wOjoxMtJ06c
    JqJIYbW6STCYxPr1HejsrNEUhYJhOIZhCGexWBtcLldpOp02dD/YInNKCmzOsgA0IYT5lV/5
    lVU+n+/3QqHQA6dOnbIODg4iv83b6wVFl9YTwECSFMiyjEwmC1GUrru2jCtHq5cxOxtAIpEx
    1a8XU8bP//uUEmQyMrLZOILBNEpKnLbmZt/63t7VDR6Pp+n5559/MRwO+02BP64gCFsUBN9M
    ESwQQiRK6YujY+M7/tff/v2BstISZm1f72KbFVqkJRJM9ZmEEARDYXz5P76NY6+dUQtoF4U9
    HG+opP1bCX71D6ON47gH7rn7bltnR7ue3dV0mksuI7t2zRqsW7tmUWwoT+iqqGBYfg9ggEJT
    VUxMTuLyQK6pPIUVRdWjddC3ad0a3HlgL/uVr/2XS1XV8Klz/XRwaNROKeUoRRmAet05Tpsy
    VsUW4M10/DodDsfBu+++i62vr1tsubNIrzeRA9LpNL74pS/jG9/6NrZu2Yg//sPfRanPuzjv
    5rkkAPbu2o4fPPk0/v4f/xUebwke6OnBN77xddy5dwca62uQyYro7V7FjE1MuUxOSLHrplPT
    CoxwnppvQVbS+HmUYzmwOm1No1QXvpKgKsqi4JXNagHLkNfhXJjaTZlo5+a66yL5MaNuNanv
    c4OCNQ9gVjfCIVP02Vz7+7M63zEA318IBLeXlpXt+pMv/CHZtH4NG49G2K9/89vV//LvX78z
    Fk+kAJzS65FUU3YhK+qiYZs2bkB1dVXlzMxslf46uQInVQBgZQjpXtXRzjjtdlBN01tQCWYA
    rCG/dQMDoEcQ+A/de8dt7i0b1uDQ0VcwOT2b0udFMEU/M1hqPB/Wv9e6fu0a+57dO2G1cJCz
    Mlqb6nFg7w4cOfFaenJ6dtAUOZ3Tf8+mn6ucqVVCMSN8PYDBA3jQ5/P9t1/7tV+r/fSnP43y
    8vKcyri21PN9aenlaz8saRZgcc2wTO48kSRpmW4EXRbkSiIej8Ptdua3gzOdB4XnZ+FzaVqu
    Z3pzQy0+/sFH0dPZav0///z1TafOX3ZSSr9DKR3UA4Hms48WsEDeNir0xYuXEn19fU/W1NRt
    7+3tqjt37jJUFQUtjwpBXGE7JAbl5S7U1LjopUtnUgsLC0YdpUEpNddC37IguKA2jgAgGzZs
    WKeq6t0DA4NcIpEwZWWX6nUpzYGE5dTwnyR2lQ9qc71Zl5Ll12uLlANICsbGpsmGDd12VVVT
    zz777MKqVavsJSUljMPhQCqVMgTejDaQtgJHULnRgejCbPq2bdvqKaV3j4yMWEOh8LJAi5kk
    wLIsurs7sW5dH6LROI4cOYVkMn3dFlILC1EEAlGsWlUHUczg6tVpdHZ2IhBIIBpNA6CIRJJM
    bW2t0+l0CvF43FCtXVS0Ndlg9W3Yj9TkmxL9NTGapjHJZJISQnhKkdSzobnoIkN06jIHhuEW
    hcRUVVtWC7w0v9c/no3fMdShzT5CMpmU0+l0psD2GrTnOZPtjRbY3mV7/s3MqbEvJEnCf/zH
    f1x+/PHHv1tbW7v66tXhknPnrhBZBl9RUcmvX9+1etOmTZicnEyHQiHooFCllKZjsZhCKZUF
    QUA6nYHfv8DV19c2ezye06FQSDElagytIMFmszVIklQfjcYX505RVLAsI+g0a9GUsScAiNvt
    sjc3Nz+USkm7RkcXoCgUpaUuJBLxudnZWdFkt2Ustd2KAIh0da3qIIQ0Dg+PkVgsBavVBU0D
    5udjSCazsFq1mamp8SmO44iqaunp6ekroVCI0Z8zcx37S35CABof/ehHa30+3+/4/f6HDh06
    JIyPjxfouFw/0FJc2R56CUgOyOay5ks5n8Lfs9utUNVcfbn5PnOO6Pp/G6ZsMkUgkEIyKZOm
    Jl95Q0PzAwcOHHD8+Mc/fiaZTM7rn7G5BJaiuPbKTRXBMqLCMwD+7ejxE+W/94dfWPPbv/kZ
    ZsumDRB4DsToyLMYvc/9n4BAA8XY+CS+/O/fwrd/+LSaFaU4gCghSFC6rKflDWtq/lbThPTL
    CuD+jo6OvkcefghWq9WUOVty/jRNeQNCVpqJTm6uqV7ywxx2G0q83pwBpsVaJ+XWikYp7HYr
    Hrr3Drxw+JhtbGLKAiCRSKYXaR36fDtymXhqB2DTaSMWUxZKvZGOn8mZMeoL97e1tbXefnuu
    9ldVlOWtjvTaq8tXruAHP3oCvpISnD9/EcFAEGUlvoI5MzCbhvKyEty5fy+Ov3IKd91zHyYm
    JkA0GXfs2wWOZWC3WbB5wxry4+dedKiqZggMCZRSQT94+YJs3k1dn0VAsJFtNRQyjR9aAIgs
    y4LlGPj9c/jud7+PiYkJRGNRpJIpSLIEiyDgYx/+ILZv3Zz3lGZxOxMMvg4RtCgCVglBllKk
    9QymYYQNCtaCSXjD3MbnpzbABfuT53m+5APvfx8euPceEE1BZakHv/T4e3H63HnfkeOv9SBH
    h5YppZIuUJEFIEqSpCZTSa6npwsd7e32mZnZNj0zm0VOPMNGCLHqrI9yjucbOjpaYbHw0HRV
    U4vFYlCgBX3/GG0HGD2T+u6eVe3dj95/JziOxflLA0gkU4aqsxVLYhmiXpdvnI8qdJq2Ioug
    AgHVcjXakiiCUioCGNedm7T+qOKnV9U31lO3xWL55ccee6z2M5/5DEpLS1+35d1SXb6RXiaL
    cjaU5lTFa2tr4HI6cXlgEHt27wCBSd3d/JwEmJicQiweR2d7qy7eVHAPcsJslJB8RX1zAEe/
    X9M0cCyLXVvWw+WwM1/4my91nbs0eK++/ib1dh1mm3c98bObuecppVR98cVDZx944IHvrlnT
    8zFJUp2DgyMG3a9IHWEhcMuB3+7uKszPT4lnz56NKopi2IAUlreUUXELjmKKqA8++KDP5XK9
    b2pqunl4eOQnKDuTnyhyVQz0msWyIpEYhocnIElK0cxcPqAGgsEIwuE429nZ6eJ5nq2urrZl
    s1lxfHw8ODExkSCE2HRWiE0/X2ym88NgxS1mHW8k2CsrK+M8Hs+OZDLZNzw8BFVVFzM9hQ40
    QODxuLFqVRtEMQOfzw2324lkMoPC2uClLKWGiQk/mpurMT4+A4tFgMvlw9Wro7p6MRAMxtHd
    XWsrKytzxuNxviAz/rbY4OsEoI3yFkYPFFKbzeYWBEE2egH7fF50dnbC7XbBYrFAEASwLIts
    VsL581cQDkdhCF/lJ1eIKbNX/P/FHk22rDD4bLCvAibbW9hD+WfKquc6BeWG1Wr1RiJR++nT
    5xEOJ2GxODE3F8a1azPMhg2trbW1NW2hUCiJnPqzDIAmEglomiZaLBaoqobp6Vm0tjbXNDY2
    1oRCoazuI4p6ba8VgF0QLO3ZrOyOxVKmNaaCZTmBZVmr/v44PTnAAGC3bdvWZbVa3zs8HHQE
    gyk4nTa4XIIyMTE+n0gkiL7GjHPQCCYkACR5XkgDROE4Xlf2zgU1WJYHIQTT09Php5768ZCm
    aaL+O+HrJZXe6JG3YcMGS2Vl5cPxePyBI0eOCKOjo8sIXMWytsWu3NkIKIqGaDSBmppquN1O
    hEKxvLPRvH9ZlkVpqRfptIgczbz4ubqkiVBYJrE8QCuKKkZHo6AUjoaGptu2bt2SOnTopeO6
    jkthF41iZbI3FQAbyrMSpbRfVdV/fPnI0Y9MTU2ve9fDD/B33r6PNDXUwWGzLtIlNU2BmM1i
    ds6PV0+dxX/94Cl66twlTZaVlL4xk5QuOnYJ5Hj2ZiGSW5YGbaIlEEppn9VqfeSRhx+y93R3
    L4LfYmAXRfv5akWozvkiWMb3CAHuvesgdmzdDJvVkqMb4vqtk6imoWdVO/bt3sZ85WvTLt3I
    2nVBCVBKS3VgEsdS/ZHVZHzlwuzHDR5VHMfdvnfvXr6luVmPcC29n8U2HloucrqwEADLcli/
    fj1eOXEcqqYaWw4gRosAbTEgo6kaUuk0Gpua0Nvbi6/9x7/j9t3b0VhfA1WRwbAsejrbUVle
    bp31z3uwVJdlMUUdjcyw/HZkg4o0UNdMhthgJNgByAzLgCUMwqEwDr/8MtLpNHw+L9xuF3ie
    g0XICcIsZdIKYK05w5u3AGjBe6aFRlM1gXI92EXCJtpVRP9+uhj16mecTw0Az7KsvaamCgwD
    yJIEVZbhdjrg87gJcu2FSnTDlqaUGloEkiSKUjwet3S0tWLXzh3M8RMnWiRJvqDvE0oIMYRq
    rABavR5PaVNDPQhhoKkq7DYrbDYrY8rkyMgX99vgdjnvePcDd3FtzQ2IxRO4PDgEWVEWsNTL
    1tyuSSw4E0OvnTqt/uCHT3D33rkfLNFw8eIl/OiZF2kqnYnqdsGuz2dhNvP19jG9znqzArh3
    9erVa37pl35pCfwCeWwMsyL74vcJIIsyAgvzCASDBkgHyzJIpVLwej04dvwVPPaeR+H1uvOy
    xsaQZRknz5xDMpnCnH8BP3jyGaRSaYiiBE1TwHM83C4HKspKUF9bjbISH3KEhgJGTCG1GsCa
    7g588oOPMn/011/q9i8EZ/Q5hkkt3NwzWH07gl5G1wsAqt/vj7/wwgvfP3DgQMmmTasf9Hjc
    rsHBsaKAw/jYc71ABdTVlaG1tYyGw3PSyy+/HI7FYkY/c+MxoddgZW51+1uQHWHb29v2SJJ0
    d39/PxuPxxepzT+pXdFPygJf7/ej0TjS6SwURSn4W8UzwYqiYnLSj9Wr2y3V1RnN5/NZLRar
    o7y83N3c3OIdGLgsnDx5KqlrT1gKbLBkWn83fOzfv9/HcdzB6ekZz/x8oEggYCk2SghgtebE
    7qanp9DQ0JgnlJUP2pYeOY5HMpnCzMwM2tvbEQymEY1mwbIcNE1DLJYBpYxQU1PjGR0dXbS/
    et2oORAt38w1+pNsLwBqtVrjHMepxrnjcrnQ2FgPQgiyWVFXvwZEUdIBIymS4V2uBF0YlzT/
    vGCuDdEmw/ZGkOscYNjeMJZUtJeJS/0stlffkxQAtVgsjlQqw2WzEliWXxTxymYVaBrlOY4v
    Ra4N3mL7TUmSGFVV0xZLLkjg9y8glcp4mpqaWi9cuDCvqqphB+0AbG63W+E4blU0GmdFUV4E
    bpKkgGVZgec5Q9NmUWDX4/G42ts7HozHxY6JiRBUlcLttoLnkZmcnIhgqYetiKVuG1lKaRaA
    Eo/H4w0NTLyrqwPxeBaplAK73YampgrYbGxmYmIypWmaXf9wRCz1U6evwxosGlzVzze6devW
    RgCPXL582ZUr71i+FooF74xLECw5YUqLBQzD6SJ9uWCw1SqgsrIc4XCiSL/q3Hw6HHaUl3sh
    SQqqqkpN7bwYnclAIUkKMhkZqZSUF5S9HijPBbIpJicTcDot7s7OVTuGhoYXJiYmqKl1lIL8
    sljtpgHgAqqv8eEZ0aV+TaNfHR4Znf+rv/0/O771ne+VrO7tJm3NTSgr9YEQIB6PY2JyGv0D
    13BteJSmM1mjp2amgG4lIdefUUKRoufr0XRuAdoVdLXi969bu7bnXY88DKtFMNGb6fIs7mKd
    mn52Fq3zvb4CNKVUV+SVi9KkaRGAbbNacNeBvXjq2ZcE/0JABqA1NzXaWpoaay5c6i8NBENO
    A6QUMb7ijTS8ReZ0fXl5ee+BA/ths9mgqsoiE1JVVczNziISjcAiCHC7XJAkCZqmQpblxQhk
    JptFMpFAVsyCZRh43E6whAEBMD07i6effwk7duzAxOQkoMo4eNsOcCwDTQfPTQ01qK+rZmf9
    824s9aW26cbXAME3vUXFG6BkSaavkwAUlmEAQtDR0YF/+tL/A8vkWheRxX1EF6moi2+GFsvs
    5oAwoYVA+XpZ4UUDksYShTeiXwb1KmXa/28V+DXOqbAoijNHjx5rvn3vLnicdqiguDxwFcOj
    40bNrxW5Hn9O/TXKAKRsVhTjsZjLIgjYtXM7/u0/qismJiab9DklpgyNHUBLeVmJt7a6arGu
    2umww2m3ExNIzppoeyUA7tu8fnXNnft3g2NZTM/OYWrWr2BJmAvGvtMDVaIeJacANjOE3DU/
    P8994X/8OX32uedht1vJxUtX1IFrQyE9U62aPgMjSyFjubLxdQ1wgaNXwbLs7r1791q6urry
    nUFTz20zQ4NQClXTcO3aNfzgh0/gyJGjdHxiAoQwxOGwLxrGQCAAlmVxeWAQO7dt0Wv98wN+
    4UgUJ8+cQzyZxle//h0wuqIlwzKIhMPw++dBCKFutxN93Z147NH7yf7dW8Hr/eqLnYvm59+5
    eS327dho+/r3n1mtswIUAIrODChGC775G1yn6lNK5ZGRkUAqlfrmnj17Mh0dTXfX1lZVT035
    Wb8/jHRagqYBDMOB43jYbFaUlXlQU1MCu52hIyMD8muvvZYMh8OG6nXK9FgsA6wVOLe3WvaX
    fPjDH66zWKyPX706VDc8PGpSdmZMVMLrgdvrg+JirY6Me6uqKtDYWIfz568gmxWvU4u99DcA
    gnA4hnRaYlmWJVeuXEl1dHS6BgdH+NLS0sqNGzdtTiSSgfPnz4ewXAxrWYburf4szEJi1dXV
    7ZIkbR0bG1sUEjMzCxwOJxwOBzSNQpYV8Dyvg9rc0cUwBCzLwWKxgOP4xYycwfy0WgU0N1dh
    bm4GHMfB7S7D+fPT0DQChuFBiIZ0WkEmo7Dl5WUujuMciqKY23QZ83IrdAoxfOJFuqbdbk9x
    HKvm9AeA2dk5PPHE04vv31yykL8Gi2V+zS2QflLmNxfz1zPAhu01gs9R3e5GTYHnZbb3LfRH
    5HQ6PeFyuZO1tTXeiYl5AAx4XkBVlQ+qKouBQEDVba9Lf00pRVE0WZaTFouFchxHEokkZmfn
    mKamuvbS0tLBhYWFiH422wDYnE4nyzBMWyyWZHJ08tw85cAwYW02u0O37YZmB7tp08bVNptj
    38CAn00ksnqrKDskSUzNzMxmTDbbUPxexChr1qxxtbW1PcwwbEdjYy2cTieNRBLEarVSr9cu
    DQ9fGx8YGAgjv32XGcAp1/GnXy84zbhcrvXJZLJzaGho0dfNXy+FpS65eRAEAQ0N9ejoaKMV
    FWWwWq1EluVF6jTDMLDZrKivr8bIyJROb14esCop8cDlskGWJaxe3aAzqXIaCm63CwBoKpUG
    pQShUIYMDy8gkRB/Ih2bEAaKQjE9nURvr6+mo6O9b2JiIoYlpXDJhBWXsRS4G21oCmaYUko5
    3YnrANAJQFAUJT05Ne2bnw+QE45TsFmtQE4cAMlUCqIomT80tiCLRg1aT0GWxBCbUQodtFvB
    EOtzwgHY6/P5Hv7gB94vdHS0Q9V0FWzQZRToQnXnYvRnWjRDvPS1oir43hM/xumz5/EPf/vn
    cDntRYWy8n8X6F3Vjs0bVjM/evpFFkBKkiTu8fe/ixkaXmf967//Umc6k72sZ7isBVdR4/tW
    BiJM1CIbgJ2dnZ0l69ev12sLl4Jm58+fxx//6Z9hbs4Pu90Gnxc6TKkAAIAASURBVNeLeCIO
    i8WKrq4uvPzyYfzv//MP8HrcCIbCSKVSsFgsePTBe3D3wf3QVBXPPP8SkuksVq9ejS996UvY
    0NOGxtoqqIoMEAKNAl6XEy2N9cxrp887sdSWwmpySgRTZk17uxZgASVLM2VsNN3AabnsBMBx
    LLwezzJxoEX1Zy0XlFFVtTjANeX78nDvIgO1UMB9sa1Q2kQfipsuQ/BKxVvbx9F4/wFK6Q9/
    8MMn20Bp1bbNGxCLRvCDJ5/G1eHRsE4H4/XP1viMswDEdCaTicXi0KiGvp5ubNqw3jIxMdkO
    YErf9w798gJoKi8rEyoqyhfLFiwWAZ6cUTAywIZoFg9gQ2mJb+O7H7ybVJaVgIJifGoW84GQ
    oYEgmCZTMdHDVABrBJ773crykvXhaFyeXwhc/OFTz4jIUaqzyNHKU5RSo7whacrqmdv8qEX2
    7+s5QVU+n69j/fr1EASh6PmVd64BUFQVzz33PP7yr/+Gnjl7VhNFCU6nk/n0pz+NTRs3QMxm
    oEgigsEA/vHLX8HpM2exddN609meuwgIxicmMTQyjkcffRSPP/44nE4neI4BxwCnTp7E7/3h
    FzA6Nq4GgmH50JFX2OHRCS6VSjEP3LU/V9tuXpt5dci5PuFOuxV7tq7HUy8eK4/Fk836+pRM
    wVpj3kS8Da3h9D1O9ACICkAOhULJ48ePv9bZ2cl1d3fv27Chp0GWZSZHUcvZWkHgYbdbYLHw
    SCbj9Ny5c+qVKwNyMplUCCGKvk4WM9x6xN1MNyOmM+5tt71mn8QQvtqzZ4+9pqbm3bFYfM/5
    8xeQyWTyhINeH9ySnyB4dX1qdEmJD7W1VbhyZRiiKL2hTLMkyfD7g6Shodxy9erVWF1dncxx
    gmV8fAYVFaU2m81qBFztektCRxGK6nUTA29FBq+kpIS32WxbIpFI7fT0dN57oBQoKfFh164d
    KC31QZYViKKkt+QhWFhYQHV1Ndas6UJrayOsVgt4ngOlwOTkPEZGZqBpQE1NGVwuCwYHJ9HV
    1YVsVkMiIYFlOf0MZaBpGuLxLDwen9PpdDqi0ai1APyabbB6C9jeRaac0+nMMAyrLfXq1ZDN
    iiaFXLoIbHN16lgMxuX3B0YRgLs8W7z09eJ9ahHbGzPZ3aQpsfGWK2kb5RoXL144uWvX7sNb
    tqy/p7R0is9kFJSXl6KpqVK9du3KtN/vF7HUgcQOwKooiiRJUsJqtWosy7HZrITx8Sm0tjZV
    dHR0NC0sLBj2ywHAUV5eXk4IqYzHU4uAzMgAaxplHA6Hw4Q1WK/X62tv7zgYi2UrZ2aiyNF7
    GXg8VkQiwVgymWRMGVvNZIOl+vp66/79+x8VBNsnJIl6ATmQzSav2GzEnc1G2VdfPRe7ePFS
    MJPJGEHrQhtsBBuvZ3/lIp8F7ezs5DmO6w6Hw75IJFIk+1ss0wo4HHZs3Lietre3IJGIqxYL
    z2QyKXLy5CkoigpBsIJlraivr0VNTQXcbhcikTjy1d1z9OfKSh+y2QyOHHkZ8/PzenslFhaL
    Axs3rkdDQxVOnDghcRynrV69ml+zpp69eHGGJJMSirdHy6dEJ5MKEgmNr6qqbna5XKOJRELS
    s+4ZU5vCZSCYu5FGxqT+adStrQGwl2XZHXa7rbWivLysqbHRXl9fR+rralFZUQGvxw2rVQAo
    hShmkUgkEAiEMDs3B//8Ajs9O8dMz8zy0VjcJoqiS9OoTY/oRAiBBSA2kzNqvFEZ+c2s3xYl
    ziKnTQvLsp84ePuBuvvvuy/Xq1ZT8Xp9fJe3NtKzwaY+vsXBbw4fKLKMqekZhCPRxbYhha2T
    Ch1USincLif2795OXjh8XEhnspm5Ob88OTlhqa0qB8dxRrbKgVyRfrpITy7N9HnciIknAMoY
    htmyfft2tqy01EQhBwgorg0N4VJ/P3bu3IWWlhYkEgnIsoK9e/dg165dkGUZp06dQiSRQUl5
    FayONJ577jm4XQ7cffs+TE5P46lnX8T2HTsxPj6OwcFBVJe5EY3FUV5WkjNWhILnWLQ1NUDg
    eZsky7YiWXHBdEjeKtkRsyiWwbTQiG5pVVVDNpOBKGaQSqWQTqchZrM5QSxVBUEOuNXVVMNq
    EZaye8twMDWVBufX/lK67LXIJkNsrjNMm1gg9C00wEYG2Xj/z4QjEdu//sfX7/veD37UqChq
    iawonKpqxiFqBOMMI5wBIGUymWw0GgHVKDweD+6+8yBePHS4LhKN1iFXX5ullDr0z7+htqaK
    cRpq7JSCZRhUlJeBYRibpmlWU/S5DMDuPds3e3Zs2QDoqp0Tk9NIZzIZ/YwzKI+aKXIsAWhm
    GPJbfV1tW2wWC/EHwuMAntR/Xqa/ft703o2Me9h0xfX5l4vM3bJsnz6XDIByh8PhbmxsNNmG
    paAHXQYwKU6eOoXPf+GP6YWLl4w1IGzdssXyK5/6JOqqq6AqIhKxCL75X99BIpnExUuXkclk
    YLfbYG7nBgAzs3OYmZ2jR44exaZNm8i7Hn0UdisPRcrAu283zpy5B3/3xX+ELCsiAHlyetb+
    j//2TXt7cwNZ3dOpSyLQ4hRtUIASNNXXoK66whaLJ6v0QEcaQJJSanZe2OvM3Q0BviagxwAg
    TU1NllWrVjVUVFRstNlsa3me72FZtp4QpiQeTxAj0KyqVGfDyFAUCYIggBCCVau62aamZkss
    FiORSAShUEjy++eikUiUVVWV1dVVzfWWgmkdGufK23rWFbTlYtevX78NwEcuX77snpmZuU5t
    b34m941me4uB4hx1j4Hb7TB9Xq8vgmX+XjAYQXNzDev1erlwOCyXl5daUikJlIJxOl1elmVd
    qqpmdDtsOH9mx0/BDShHslgsEEURu3btcrEsu3N2dk6IxeJ5gkyEELjdbtTUVGF+fg7z8wu6
    4j2H2dlZXLuWy061traA4wiSyRAAoKWlBSxbi/HxeQgCi5aWGszPz4HjONTU1CCZVGC1WiDL
    mi5UpoFSglRKREWFx2q322zRaNScETeXIjG3gA02217V5XKJDMPo8arcuuF5ARzHQxAsEAQe
    Rv2osXYURUMqlcFywetiwljEBJaLAmYV+T1nUwWPGRPYuhETpgFQTp485bfbHf/Y09OT6utr
    3cIwXKXVKrhUVdWy2ayk5ZRKOZN/ZVMURctms0mHw6VxHMcSwmB+PohAIGxpa2vruHDhwlwi
    kdAMAFxS4mtQFNWbKwExcmg5xoEsq6zL5TQAMAeAX7NmTa/D4Vp38eIsk8nkKNMWCw+bTaDT
    04Gk3hPdwFUqIUSmlMoOh4O5556777FY7J+ORtVyh4OR5+amjn//+98/BqBUURS3JElGX2Mj
    AB0BENFLv8ydLqQicyabssZ5o7q62koIqUwmk0zOTci3FcXOHJ7nsW7dGnR0tNDTp09lg8GQ
    duDAAfvFi/3o7x8Ax1nBcTaUlZWjtbUZDocNJSUehMPxZZoGHMfB43FSt9uJ9vZ2EgjkWpkJ
    gh2iyOLKlXFUVVXC6XRqR48eTS0sLDC3337Q2dZWIVy+7Iei0NelQud6VgPxuIzSUneJ3W4v
    TyQSBkPQCCJY9bVLblgGuDDiTAhhdKGXdQDeZ7fb/3/q/jtMjuu8EsbPrdQ5TM45IeccmUVK
    DKKYlLNkK3gd5Lj2euWwXu+ud/3Z/j57f5IscRW8qyxKoiSKOQIkiDwABpicU+dUue7vj6rq
    rq7pASibA2D7eQrTM9Ponr59733v+57znnPn+nXrOnbv2SXs3b2b9PV2o7mxEVVVUQic5TFd
    TMR00xrDUjzWNZXk83nEYjGysLjEjk1MsifPnPOcPH2++sroeJuiqK2U4mWAyhbtj1oTgluN
    lnW9Njx3UcBS3A0D+Ni6gYEjv/7pT6G+vtZMft1Ib0V6srEiQV0hggVqoXLO/lcgk81ieHQM
    PMuCYeCyGHFTrg0H1Y9g59aN6Onq4M5fvMzrhqGeOnNOePT+u0ltddSTyeZ6YAokiABClFLJ
    1Y/tVMpdq0PfQCQS6di7dw94njfFr1Dq5du7ZzeOHD6E84MXMDU1hWg0iv7+fnR2dqKqqgr9
    /f2Ym5tDMplEOp3G9PQ0mhrqcfvRQwAonnrmBeQKMrZs2YKvffWfsXv7ZkzNzGPw0hXccnCP
    eSwmZhWxuakOHMdxjgTYDrpumfYbSoOu0JNUrg5toSHnzp3D//P3/4DFhQWIkgRFMRPfYi+n
    YaCmphp/9Hu/hX27d0K3e68p3BZHpQTCwZemqx8KnPQf91fj7UIunE9j9UBpAFKU0p8YhnEp
    lc5sYRjmsfXr1/fncrm6ycnJbisg2YIX9mesaJomx+IJaJoKhhAcPnQQ27dtDT73wov9VlCT
    LeqWn2XZpva2FsKwTHGtEwI0NzVA4HmfJMu2qroAoKuxvrb/ofvuIlWRECg1IIoSJqbnoGm6
    5GCVFCvCFvpbA+DXNq/rveWWfduZ7z75fF5VtVdgCpuEHXQtWPdt6wub8pZ2BN/CKkmc5g4u
    1ppkAIR4nmfNYjpWR3+tzy+TyeDrX/8mzp0f1KwgpvM8773l6BHUVUchiznks2n88+PfwN/+
    wz8Zi0vLdHp2jpEkmfh93lLfuUWjTqUz0DQNFy9c0P/oj/6InZoYJ7/+qY8h5PeCZwnuOHoQ
    3//RT9jxiSnDRswuj4wzv3j+Fd9Abyd4jl2hDO3en0NBPyKhoO1zHEaJGh9ASRl/zSmXFdQ/
    ma1bt/oOHjy4JRwOv4thmFtFUezLZLKRpaU4F48nid2Pqqo6dN0WCDQVZ3meh8fjgdfrJaFQ
    AJFIkAuHG9impg4fz5OoLIsNc3Nz1UNDQxgfH89omhYEILlisOyg7xm4DkJM1xgXBgDz8Y9/
    vDMQCHx+fHyi//z5C5bf6uq2RpVFsConxnaPmlsMhhDA7/ehtrYaum63Ml2d4uf8vlCQkE7n
    mdbWVs/8/LzS3NxmACyTSOS5DRs2blYUpfDGG2+cFM2TruhibqjOQsTbObayLAMA2traOjRN
    65+bm7PUstmyokIslsDo6DhaW5vh9wcgyzISiQSSyRQkSUIikUBVVRUCAT84jkc4HIamUUxN
    LULTDLS1NSIYFHDx4iQGBtYhnZYgCBxqa0MoFLSirZ5h6BBFDRzHs4FAwBmDPa4i9A2Lwe7Y
    a2W8htfrNSzGBggB2tvbsGPHVvj9fnAcV7E/PZPJ4/jxs0ilshWQX+JC/NxIcXkSTMiK2Otu
    gVmr2FscB6tdQ3nhhReGz58//6XW1tYXN2xYf0tLS+u70+m0f/369T2xWCx+9uzZtKMQ7aGU
    qtlsNhuNVhsejweE5CHLKkZGJsnBgztbent720+fPm3H35DfH2hVFC0oirJjHTNWAmwwoVDY
    FjLlotFoVX9//95EQowsLqYd6sYCWJZq8XhCdjH7NJt99c53vnNfVVXt5xYXlSaGYaDr4vjJ
    kyffLBQKdrGh4BjfghVz3TE4bSVzyirxt+L5OhgMCgACZnFTr4DDlSO2ANDQ0ID+/h565swZ
    6cSJN/Pbtm3zEMKQmZl5CEIAPO9HXV0d9u/fTiMRn6GqCgkGA0y5bZndr89BEHhMTk5Sv99v
    3H33PcyxY2+SubkYBMGPQkHD4mKa9PT0sKdOnVImJiaUc+fOaHv2HKqNRv1MLFZYVZDQmRQr
    igGO43me5yKuGOxzgU7F9c69nZug23OKUloL4BGPx/OpXTt3rn/00Uf42269Be1trfB6PWAI
    YOg6DEM30UjD6R1qFP1ECUxUJBwMIhz0o6ujDXt2bMH9d99OJqem2edeeq32R08+feji5ZFO
    VdOec3h02p6MvCMIKA40+HpX/WzRKw+A+6uqoh/65Cc+5t2zZ3fRCoSuQgusZHlUWQHagVTQ
    lYq8p8+ex/DIOHbv2AqWYSomvyuey3rtpoY67Nm+mZy/eFkAIJ+7MEQ//NA95KOPvov7/x7/
    7sbF5QQB8CJKVgx+R9XVmQTTtzsAG4ZBAKxvb2+v6e7qKlmgFKmKFJ0dHfjLP/sizpw5jYnJ
    KczMzuLlV17GzMwMPvvZz+LPvvhFeD082ttaEfD7cPuRfTiwZxd2bNuEiakZ/Pzp53Ho8GGM
    jI6C6ip+83OfwD/98zfxyusnsX/nFvACDxBzDURDQQg8xxbEIuprq0HzFkpio3o3xa1CIkwA
    MAwhIKDw+rzo7GhHZ0c7QqEgRkZGEY1EsGnDOjNiGjo8HgGd7W1m4aoitIpVBLFQbotU/lNa
    ES5+G4PvKkUADYBqFXGmKKWSYRjdhJCu+++/n//e977XOz8/b3vrEpjS+x5KqWEYRmF5eRmy
    LMMj8GhubMB73n0fOfHmyfZsLjdj7UVhAALHsVWtLc3mWrQo+wwhaG9rhuARBEmWfdbcqQaw
    9ZZDewM7t20q9lxLkozZ+UXDUm929pXbNFU/gPu62pru+tBD93AnzlzE9NziMICLjvnn1FIo
    Vp4BpAkhWQcNruBIZq6ZALvONNTuG1qJ/paKIgwhmJqawiuvvkat91SA2cNE+nq6QHUFuiLh
    uedewD9+6av64tJyFgCby+eDkiQBiJSjsxZ93zrUyUtLS+Tv/uEfvB6eYT710Q+AhY6utiZ0
    tjWT8YkpaiUNhqbpwonT573JdIY01FbDgNtGzcFmIGXzlrdpqChvffjXqmj/ykmeNZ8Zr9fL
    PvbYYx1NTU3vBfBYIpHsGxsbF6an50gymYQkmf2+lJKilYUpcMJaNj0aJElFLieDYXJYXEyB
    43hwHEd8Ph+qqkKexsaqhs7O/pru7p6u6enJY2+8ceLFmZkZ6wxLbNsvJwW8yMS6QYgbAUAe
    eOCB6qampl9PJJJ3nDjxJslksmXJ72o9ubYF9rUSY7dwkzOhbW1tQlVVBIuLcdNai2FWeZ2V
    r6/rFLFYGt3djfzY2Jik64re0lLPxON5Eg4Ho3v37jsUCoXoM88886wkSW71Y3aN5yDx+/3d
    hUKheXl5ucIYMMjnC3jttRNobGxAVVUVQqEAmppaUFdXhxdeeAGHDx9BIBBGKpWFrhuIxURc
    uDCLxcUkvF4B3d1NmJ+fhSAIiEZrcP78FDo769HQEMTCQh6qajllgIGqApQSJhAIOmMw77p/
    w7U43OrQLMsSAMQEOE0Rv2QyjXg8BU3TUVMTRSaTRyqVLY6rpukQRck13ito/6vSnsvVoSue
    0Wiln6/F+rV8fW30VInH44l4PD6Uz+eZurr67YlEYj3HceEDBw5sKhQKseHh4bgjCebT6bTY
    2cmofr/fY7YuE8zMzCMeT/k2bdrUNzIyEs9msyKAqNfrbdI03eP2sVVVDZKkknA47Od53qOq
    qrBu3UBvMBjpOX9+jsiyXqRL+3wCDENTUqmUYc0pOAoI+qFDh/q7uro/FY9r3cmkhq4ujzg1
    NXl8ZGQkaRVIdZQszESUhMeSln2SHX/zDvS9EgJcqRWMiKJoFxVc48xUFLxiGBbt7a2QZVm/
    fPmyaBiGVFdXJ+TzIlQVEIQgAoEwtm7dAJ+PVV588cX8jh07gl6vIJRrJ5Cy/GphYVEbHBxU
    jh49yh86tE945ZUzJJUSwTACkkkRra1VTHV1tZrNZvPj4xPy1q3bg9XVAX8iIaKkhVDZMqk0
    fyljq+Hbl0P4boWK9logwHaSVw/gN+vr6z/14Q99qPqTn/wEerq7wTLESng1kwduU1SdeTkh
    xQ4i6rxvJ2SGDliV1A0Dvejt6sCth/ex3/z2jzp++OTTj8aTqRYAT1JKJyzRIWfC4Vz1a54E
    u0U3rOrzPp7nf+fhh97T9N7HHgHHsZbvb7nas7vnt9TnAVO9mV5dBKtEBTT9K2PxOL7/oycR
    jyfQ0dYClmVX0qhXS7BhwOsRsGvbRnz/p7/kU+mMODk9q375W9/37Nm6Hhv7ujyxRGqLrhsZ
    iwJoo+9utPNtv5kVNd0LoKOlpcXT0NBQSuxRQoMAoKW5Ea3N90DXTRPvr3/zX/CNf/kOhoeH
    IcsSfvtzn8Kdtx8FKIXA8+A4Fqqi4KlnXkBBUrBlyxZ86Utfwh2H9+PAnh1449Q5vHrsdcwv
    LqK9tRnUAAxC4PUK8Pm8TCqT5R3j4Ly4tT4Qvw0HRcZeGwN9ffijP/h9EAIosowv/P4fghd4
    PPzQg+BYM3mjhg5d14rK26tiuyviqdMb+Ka5GU4aGCEkTSk9OTIycuS9731v2yc/+UnP3/7t
    327J5XIxKwm2D1bUMIzs0vIyZFmCRzBFXN5xx2348U+f9P/ymef7LTpTHkAVz/P+xoZ6s4hj
    r31C0NzYgGg4xGYy2aCVQNU1NdS13nf37QgHA6bgGgHyhQISybSdsNv7mw7TbokAuKUqEnrg
    Y4/d56+tjuDlN87kKaWnrEDqQ4nurlqBNmUFXqe/o9tf/ZoiWK7DUVZRFC2VSpXRnOG2K7Ie
    n0gkMD8/77Tn4nieg8/DQZdFJJNx/OgnT2JqeiZrodhRatBgmWq+dbEMg2g4DJ7jqabpGgAp
    lUprX//W/w4d2L2NbBrohsARRC0hDuv9agCEWDxZm0ik2Iaaqgp7c2mPJQBy+QKy2bwzCXYe
    tLm1Tj5c7CuG53nu05/+9K5wOPx76XT6rvPnB/2XLg0hlcpavW6s62IcyZ0bxSwdagzDsJBi
    EYWCgsXFDCKRINfZWdfS0dF3X2NjU+eJEyd+eurUqfO6rntc8ZdxzCHteiXBjrEhlFKyZ88e
    3/r1698ny/JH33zzpG96etayJ2LekuWRSQlnLITzrShAM0XUrbo6goGBbvA8j3Q6WyYa81bo
    0wCQSmXBsq0sz/Pk4sWL+e3bdwRrayMsy7KU53lvQ0NDcyAQ8EiSVES7HT7vWMOEj2NZtlMU
    xWAqlXaNRcnHUxQljI9PY3JyDizLoqurHfv2bUNdXR28Xi8uX57AlSsTlqURKTISOjpqEQwK
    GBycxIYNGxCP57GwkILf70V/fytCIS+SSdEaR93qBQbxer1Chdi75mvy3waSgLEPvgsLi1he
    joNSAp/Ph3e841bIcgJDQ6MghLGKVazVI8yu8AUu7++t5BG8StPwDS4EoKSgLFJKPdPT09OJ
    ROJkdXV1z2uvHRN27tzRcPTo0R2FQmF51mzIFQB4stmsBqAQCPiD9h4mihKGhsbIgQPbmzdt
    2tR57NixLMdxmscjNMiyysiyBid6qesGcjkR1dVBbyAQ8CuKEu7vH1iXTsuBpaVsmcqx18tD
    11U9m81SR1HZAKD19PTU7ty5832iyGyemsqjsdELXRcnTp48NaRpGuegPNtODbaYbMqKwU7a
    uQ3grRZ/K/ayz87OaoZh5DweD3ieh6Ioxf3K7dlrt2hEImHkclk9m83KACSPRwgaBiUc5wOl
    BE1NjWhurtWPH391eXx8XN2+fbu/1MqBsj3LMCgURSV+v48mEgnp+eefL9x3332RjRsHPKdO
    jYAQD3TdHMtgMKgCyKfTaZLP5wqBQJWfZVnY3bSrKUMzDIEgsFCUApVlhcXqtmdYswTYrt5Q
    SqsB/EZLS8tnvvCFL4Q/+MEPIhQMQFGUErprIbyUmsgvDAOGA/Wlhl4U2Cl91cu+h1H6WVd7
    C/7dpz+M7s7WwJf+13dumZiejQL4AaX0isMGwunJWFEQYi13NetA2kMI+d3bb71182/9xudR
    XVVVSn5XEbKyE1lNVfHmqdPgOQ5bt2w0fStX2CFhBU2aABBFCd/5wU/wy+dfguARsHnDAHiO
    LaLub0V0iyHAhoEeNDfWMal0hlFULfPkM6/4Xnj1TcY891PbbxQ2hcfe0F2Fh7f1ZtE6ggCa
    mpoaSTAYMOeRhdBQOLyUaWnuCDyHttYWiy4lwu/3Q9N1BANB6LpqJXUGJqdn8POnn8OBg4cw
    MjIK6AruvuMofD4v9uzYgp/98jlcvDKKtuZ6GJSAMQCeZeD1CAzKBYyKSoKO4HuzKEGvmgDb
    gVMQeIBSqLIMURSRSqWgqgoYIhTnkRmQ7bqKAYMQq60BZXR0t0D0Nf6O6x6LUepBFlES1rss
    SdKZl156qfmv//NfscvLS+Gvfe3x/bIsJwGctyqPBoDs4uKiIUkyEw6FYBg6Ghrq8bEPfYCc
    H7zYNL+w2GMlmw3hUFAwk6/S2jMMitrqKrS3tjBTM3NBmJZLnQf37vTs2LyhLFlMZ7IQJcn+
    m51oBgWwxef1PPjY/XdG7zi8G//zGz/A/FJ8CqbSs9d6vE3ByllKn85eI9GFDutvFYF3MYIS
    2Ww2MTo6WnX0yJHyPlqHL7ft2UtKhUNbQdPQdZ2KhRx0VcLk+DjODl7QrcJDxqSy+eCxes/d
    DJpIOIRIOEREqegiII9PTHleO3bcM9DVDF0z45L1WnnrdX2aplFJlkuYb4WWE/u9zy/FML8c
    s0VhGMf6dl7XI9EjTU1N7Hvf+979Pp/vLyYnJw+8+upr7NTUdDHxNYFZJ5ubFlF5Qpzfl9vV
    2B6NTlVZXTeQTOaRyymIxfKe/v76nQcOHKoNh8M/eeWVV16XZVmwGC/u4jO9Hkiwi/oMAOTQ
    oUP7CCGfOXdusHpo6LKl+nxt2yPzoMVi3bp+1NZW4+TJc8jl8qugteVIMQBEIiFs3boe1dVR
    yLKCWCxZRIArIxqV+4IlSUEuJ5GGhgbhl7/8ZXx+fj7T0tJCvV6vLIri8tTU1MVkMim6dley
    1vvpnj17vISQ1ny+wJjK1pUQ8PKeZ8MA8nkRum5YfcQSGMakoAJMUfjJ7/cU0V+Px4NIpBqn
    Tk1AVXXE41lQqqO62o9sVrUo0EVfUcLzPEcI4SwB1ps9AYb5N1HiXIOmaJA5jlZLQtFTujyp
    pWVr1imA5bZJKhUlsJbHs39rDFasOMTKssyOjY29vm/fviPBYLD91VePkdtuu7X7jjvu2P+L
    X/wivbi4WADgy2QyoJTmA4FA2fqZnp7H3FyrZ8uWLX1TU1OxRCLB8DxfI0mKhbaXz9V0uoCW
    lmZvKBTyh0LBpmi0pmloaIkoim4pm5vJosfDQVUVQxRFJ4ynR6NR/9GjR27huMDey5fzLABU
    V3PyxMTIhcnJScmKwXbcES20t8i+QknATnEVn+mvUEjA/Py8qijKYigUUv1+P68oSsW16Z5L
    hJCiIriuGyrHCdTjCRKWZdDY2ABFkQrj4+MxjuM8HMdTVc0X99JSYs2AUkAUFfj9AfA8ry4v
    L4tXrlwhW7fuqo1EYqRQoOA4DwhhqKVZkNd1nRiGrvI8Y+3dq1s0mag1g0CAQyaT1qw+Z6bC
    RdY0AbapVwDuBPDJUCgUnpiYwN/8zX+zBJdKNEc3tRcwPaWKyr201CdYzF+v0hdbVA/VNFRF
    wsLkzNwui3//OKU05/B+Y1HuskzX+EDi7MfyA/jA+vXrbvm93/1t0tvTXUr4V7UyKoldaZqG
    n//iaZw4dQp/85//HOsH+q0CAFYVwWIIQTabw/efeBL/9M9fp5lMFls3bSDbNm9wfGYOivVV
    7JAMSlFfW42NA73MxcujPMweyViuIErWgp0HMI6SQqBTDZSs1eHGuvlYlq1tqK8nDMOUUcDh
    vO8QZzIMimg0At3QoaoaQuEIFpeWHOgxoOkannn+JeREGZs2bcLXvvY13HZkP3q7OmDoBgZ6
    u9BQV4uTZy7g6L7tEAQBhg6wDAFv2jo4E17OlQQT3Nw3UlSZtNYnNXRQAnR1dYJlCBKJpKku
    rmkm+qvr0HUdhqGBAKirrYbP60FJlbeSMBbKaLA3EQKsug6ODIBXTp06tX9sbKz29373C4jH
    YvU/+OGPbtN1PQ9g2EYRl5djRj6fY1BXU2Ru3Hr0EB576N3CP375a32KomQA+CPhMO/3+cpQ
    UUoN1FRH0dfTSV45fsIPoLG2Olpz9+1HSDQSsnqvzRfKFwqQFYWWDk7FkW1lGeaeOw7vaf7w
    w+/EUiyBl984q1BKL9sJnuOAkbeS37gjAbZtLpwetv/agmEynU6Pnjt3rkdRFHAcW7avOBNi
    SoGqqio0NjaSTDZLbGqXoihaKpHgdUXE7Ows5uYXFeugIAEgTY31pKQw7dAbpwai0TCqq6vI
    wtKyrSyuKooSGrw45Mll0tBVGflCASh5X+oADEHgEfB7y+ORc55ae4qq6bg4PI5EKqs56Glu
    +v6aTmxHAknf9773DQSDwT9ZXl4+ePr0aSabzaKmpqaYTJjJnq0oyzpUj9lVHsMU0aXKP2cs
    ZE/F+HiC9PXVd+7cufMxnuf4559/4UVZljWrJ9i2grIFW65bAdqOQQ8//HCV3+//0OTkdN+Z
    M2ehKKoj+b22jy8AZDJZrFvXh6amBgwPj1VEiZ0HSYZhUFNTjc2b+2k0GkShUEAmUyDLywlX
    8nx15Wn79XWdIpXKoa6uhuN5nl65ciV95cqVtLV2FwEsOA7MqisOr1kMDofDAoD6fD7P2Oee
    0jgwKw7a9veqqsEwKASBtwrRXjAM6yi4EDQ31yIQ4HHhwiTWrVuPeLyAWCwHhmGQzcrIZERU
    VfkwN1eApplbFSEmcsSaZqOsI/ZyFYrQN0/2a8ZcxixIlaPnlFLE40nIsgKfz2cpaHNlKLA9
    dpKkVhTFcj7nTXwrs2a0e/cvXLgwuH37tjO9vb2tly5dZl577Th3661Ht9x+++25n/3sZ8+m
    zBuj61o+GPQ7kjEGiqJhcHAYR4/uqT548OCGV155hbAsF5YkhTiZL/ZYp9MFMAzHtre3VzU0
    NPCiqHuXlrJlc5hhCDiOhaLkqW7CmAQAFQSBP3r0yP7q6sYjExOikEqp6OwMwDDEpXPnzk0a
    huGzHmuznGzdDTsGZx3MLM0F2l2zAO28pdNpI5vNjkQikVRVVVVdKpVGZdX6UrEuk8mhq6uV
    CQaDuiRJiizLiiAI1OsNEU1jUFUVRiaTzGYymXRDQ0OVx+Nh8nmp4t5FKVAoKGhoCBKPR9AK
    hYI0OztLt2/fFa2ujvKKIkIQvGAY6Kqq5AHkBUEQBMFDVNUo7gPl+0a5FZjHwyIYZOnk5KJs
    9VW751LF77k1CjQagGPDw8Oe0dFRnlKDp7RMGVIoTwaoQAiJrFu3ju/p7iqidwAQi8Vx5twg
    lSTJIIBBAd3+CvO+DudlCdhQSm0zbwEASylliZtEfn2QX5sWTimlm7xe73s++P73+fbt22NN
    XmMV5BUr7Ig8HgEPP3g/srkcMpmsaSJtSf7bHysFYBYGCTRNx/D4BP73957Av3z3h3RpOU69
    Hg954F13or21uUxxuow+6FKPtpNkg1IE/T5sHOjBDzmO0zTNPjTah+c4Sn2CTvGrNbX6qamp
    QTwe93IcF4pEoiBm5bfMrqSY5DsotxQmOkRg0vKrqqqwsLBULMIQAFPTs3jyqWdx6NARTExM
    QJULeOedt4HnORiGgapoGJs29OPcuUEkU2k01NYUWfwsy5BVKlHXDRV6OxBgQghGx8bwve/9
    AEtLSygUClhYWIAkyTh56gxUVYWmqTAohaHpMKz5FPD78dlPfRRHD+0rzU/qUn525Af05gq+
    tnqtM4EhAE7GYrGLP33yySN33nE7/uSP/z3JZrPtv3z6mbstVco5AGo2l9MXF5e4rs4O870Z
    FIGAHx//yAcwPDoW/tlTz2yilMqhUJDxej1lc5Na9Pttmzcg4PcF8gWR37Z5o7B7+2ZHwc8o
    Hh5NdKBsPlUB2L1zy/qeT3/g3aSuOoIf/Ow5TM8uxgHMokR9tqlXNvU54Qi+dr+mm6v+KxcA
    ASQppedPnjx529T0FNfT3W21cDjaExwFt5bmJuzcsY1cGR72WE+lSpKkLceWoEh5xJaXoCiK
    TdkmLMN41vX3wesRABdNmVKgrqYajfW1uDh0xaabSZRScXZuPprPZ4mhKsjm8kBJ6ZQFEGyq
    r2WrIuEKYl0laJQASKTSOPbmOWoYhuzY83RX4WDNk72f/exn+JM/+RPW5/MdAbDd7/eL+/fv
    t0g4INaWZlMsi1qVAAjHcYxhUGLGktIhmWFMz2Rd16lh2EmN2Vxt9uqZ1Wlrn6CmdyShgiBU
    9fcPHLpyZXhwbGwsYcVf9nrHX/cBsbu7a5csK7dcuHCRyWSyKxK0a9GfCSGYnV3Ayy+/DkVR
    LQ9LAkmSYfds2sJWPM8hHA6ira0ZnZ0tVBTzejweI7W19czIyCQkSXGgeMw1RbCcB8pMpoCW
    lhq2vr6ejcfjTrV3Gy2q5MKwlvOPsCzLAYhIkkzsMViJAK/0N1ZVHYZBwbIcCoUCQqE6q63J
    /HN9Pg+6u5vLen9PnZqArhvFx8XjOXR3h+DzCcjnVSsBZkwM3pkFVUaFbqpM0GbM2dO2oaEO
    fX298Pv9EAQBoVAQtbXV6OxsK+sfdzIJJEnBmTPDiMczqJznltOjK/sI3xQJcFkBMZ1Ok/n5
    hRc6OzsPtbS0VA8Pj8Ln8wkHDuzbe9ddd6k///nPX0un00ahUJADgQB4XihrM4jH0zh37gqz
    a9fGtv379zNery8ky7kiwliqcxMUCgqyWZnZsmVzLccJzNhYmkiS5mKGMGAYAl3XCaUGa52Z
    2F27dq7r7R04srCghBYWJHg8HKqreX1+fmpicnJStsAwp1Wejf7aBeg8VorW/Uox2BF/jZmZ
    meFNmzbNtLW1101NzawQ3ystAxOxnZtbxPr1fWxHR4cQi8WMfD4vezw89fmC0HUWPp8Xi4v5
    gqIohbq6uhqO47l0OlcRUaYUyGZFdHbWMKFQGMlkSsnn87qiSEooFOQTCXONA1QrFMQsALG2
    tsYXDIb8MzOKpVFBVi2gEUIQiQgAZGN6ekay7Pmcl9sq8+0XwbKDpXUI/CWAk7quB3VdD1sU
    virHFYWp0GWLhNR5PJ7tH3j/e/mPffiD0FSLkkaBU6dP4/O//fuYnpm1ldIkx+buVDoUUS7d
    nnVQ5IgVCJ103Ou5ocESvrpl44b1ffe84y4U8nlomo5wMOCYqKsJUpW+37B+AP/hj76AmZlZ
    fP1b30Y4FMSGgT6EwyGwrJkMS6KEyekZvPb6Cfzy2Rfp2fMXNUmWDYYQ/o5bDpIH33UXeI4r
    JsCoKLpVub+YMAQ9nW2oqYpwi8tx1kEVdQZfyXGtefDN5/MAwDMM4/X6vKVKQBnNEg7aX+n9
    hoIhCDwPVVXR0NCAhaV5FAoF+H0eaKqGp555Hnmr9/fxx7+G2w7vQ29Xe5HG7xEEbFrfhxde
    fg1TM/Oor6kyyzOGARdySFa5buYEmDDWISWVTOHCxUuQJAkBvw9trS3w+/3weATwHA9JlrCw
    sIiOtlb4/T4QAvi8XnS0taz45J1ImllwKc0Oa0u71t+1ZgdlVw+S4QpCDIAZAC+99NJLuwYH
    L/j37N6FL/7H/0BUVe1+/vkX7jEofRkAl8/n5ZnZOY+zAGPoBjra2/D7v/15ks1ma1969Tj1
    +3yMwPMr1jtAsWPLRjQ3NnCz8wvcnbccQG111EThaWl+swyxeeo2tS8IYHNfV/u6z37kIWZ9
    bwcWlxN4+Y0zhqbr09b78TpYGrJjv8yi3De0YsD9V9BVZQBvXLp0af7VV15tM4sCqLi/GKAI
    h0N49/334rnnX/QtLi2FAMQVRdWWF5cgF3LI53MwzExNAxCORiPe7Vs2QeB56LpWJoJlWBZu
    /b3d5IVXjgtWfNIAKJlsjqqSSDKZDLK5vOFAvWs8Al+zd8cWEg4FLFE3lwq0Y285c+EKBi+P
    Of9/pUtb60LgPffcg09+8pM0l8u9yrLsnyiK4ieE+CRJCkiSFFAUOagoalBV1YCu6z7D0D26
    bgg+ny+0cePGjrm5Rd/ExHQZqltdHUVfXwe9dGlIW1hYVFiWUQkhKiGMwjBEMb8yMsPYP2NU
    hmFkjuMKkiQtxONxyUbaKKUMKZXvr3f8pXfeeadPEISDc3MLDUtLy+jv74PX68H4+CRMFdi3
    5uNLKUU+X0BfXzc6OloppQaSyQzJZHIWomz2akajYUSjYQPQtcnJMYUQwnZ2dnvGx+fI/Pzy
    Nb2CKx30TFowUCjIoJQwdXV1/KVLlwxHDHbGXxGmIveaCVC6GYWUUp+ua9a4V0Z93e9J100G
    FsexyOfzaGhohSDwkCTTZqapqRaBgIDBwSn09w8gHi8gHs+VFQRSKRGEUITDXoiiuXWXkP2y
    eMu4zoE3XQwuJcBmAc/vD6C2tgYMY4pdpVIZaJoOXTcZkzzPw+fzIp3Ow6byqqqGQkFagfyW
    U6HdFkg3figcMdiep8WWRQtEYoaGhk51dXVd6O/vOzw5OY0LF4bA8x7fnj07Dt555538s88+
    O5hKpZRQKGp4PB5G00QHgg5MTMzB7/fzW7b0tvv9XjI8vFDWt1oSwtIRi2XI9u1d3mSygKWl
    DK7SmsAAhAPADQwMtOzcuXtbKkWqp6dF6DoQiQjgODU/NDQ0q6qqx4FuqzDpzzlHDC444jO9
    VlHvrSTBZ86cme/v73+ju7try/nzF9hUKl0svlVam8vLMSwsLLNbtmwJT05OxjOZjMpxHA0E
    wpBlAo7joWmaxvM87erqCudyEpPJVG4HAQiyWRGUMkxLSws3NTWla5qm67queTxe8LwOn88L
    TZPVfD4vMgxj9PX11TCM12/7Cl9NAZrjWNTVeRCLTStzc3Oig/miAFAs9pFWaTzf1gTYMAx7
    8spWJUOxsnHDVdnRrM25KEsfCASYzvY21FRVQdOUoiJqV2eHVWWFgXJzaKcyWg4ldTTF8bi0
    KwGj1gKj13NBsyxLdV2PMgyze9eunUIsHsN/+Zv/jmgkgj/43d9GY0N9ifqNlWrMNsWZEIJs
    Nounnn4O//y/vkUHL14yPIKHNDXWk5amRgSDAaKqKpaWY3R2foEuLcVkWVHsXTBwcP9u8tuf
    /QSaG+utAsPqKtNOxNQUHrP+FoOgo6UJDXU1zOJynENJ7c4ed7lC8NXWMvhaPcBF1WJncmUn
    WG5KtP3e/H4vQsEAsrkc6urqMHzpAtLpNAL+BkzPzuGnv3gGBw8ewtT0FORCDu+881YIgnXQ
    tiCVns52EMJgfGoWOzcPAISBosg2NZWWlVrfggLFzZQAE8bs4d2yeRP++3/9K9Nnh+PAcSw4
    hgHLMmAIwZunTuOv/+Zv8euf/Ci2bd4ATTPVOA1bFIteRQKalnosb/SIVEiCYTFK7P4UjlJ6
    fHp6ZuIXTz21Yfu2rdixbRv+/It/yvyxrve8+NLLXkrpaDabk6dnZlDsQ3espa2bN+KLf/z7
    zF//979HQ12tSRGvoMbe2tyIQ/t2kdPnL2Dfrm1gCCnaS9l04YDfB0Hg7QTYB6CluaG24zMf
    fg+3f+dmEEJw7uIwhsdnRJj0SMEaYclGgC2la2cRccV6/df0aNp7reUtPJRIJN780RM/arvr
    ztvR0FBfoj67EkoC4OiRw3jowQe4L3/18XpVVRUAxtLyMgq5DDhTaZ2xYkfdpg3rhK2bNxRj
    hruAKPA8Du7die//+Oee5Vjca8UNSmAQQ1cRTySQy+XtPUwA0Nnf0xm57dAe8Bxn7i/OfQQl
    JDibzeHnzx9DIpVRHQVa5yU7Djr6WiYgnPm30r/7u7+7BNNv2rZkqoKpIm5fZQXo9vb2pp6e
    nuaJiWnfuXOXwHE8CDFplQ0NdWhtbcDMzKx6/vz5HEq2Ok5PbludVHYVVJLWz51zglxv6qUl
    koj29vYAQDYvL8c8bW2t2LVrB5aXY5ifX4IkKWViVaupODMMi4aGemzdugHhcEAfHx+XKKV6
    bW0tU1dXxXIcSygF1XVNy+dzytDQBSkWixkdHR3h7u5e39TUIjM8PAFNo2WoxluhQDtRakVR
    Icsqqa2t5QVBoIqiaNZhzx2DryVe93ae/6wEE8W+vZWJl/PQbf5M1w0oigpB8CCZTIHjTIRJ
    klR4vSb6Ozc3A54XUF1dh1OnJizLKtYqCJhonSyrCIUELC8r1sGYASGgqqq627DITR5/i38r
    IcD09CyWl02zAVqsrZfGt7u7DZs2DeDEiYvI58WydgW7V9UtjHU1deirrM/rPVZO5hW195aL
    Fy/O7t+//+W2tpZdtbW1vvn5RQwOXgTLMv6dO7ftv+222yKSJOr19Y2612tSc53rSdcNXL48
    QQDCrl/fiUJBhtsOyC6sLC1lkMmIZGkpg3xeWbE+KQU0zYDPx7Msywqtra01hw8f7tQ0X+34
    eNay5mFRW8sjnV6OjY6O5q0CtOgosEsOIE+qtFb/jRoJdHZ2Nj87O/tyR0fHu7q6ulrPnTuP
    1ZSUCWEgyyoGBy/jttsOBg4ePNg4PDycptSgwWCwKErHMAzb0tISbmxsqh4eniOKoq1SHGCQ
    y0lIp0Wmt7c3MDg4yFp5IViWhyD44PXykKSULMuy3tnZWbVu3fqOWEzkcjmlLFGvZN0UiQjw
    +w3j1KmhQs40dHbHX9nBaCsD4952FWj3AnYkvEVLC+sgUEBJoashHAoKzU1NZv+gbsCgOogl
    YFJfV4sL5nPZDfFZlHtj2Ymw5kjIREeAXuEBfD1v1sEkEgwEunq6uvDNb/0fDA5ewK9/+hPw
    ej3FBIGCguq6tdzL0UtKKcbGJ/Dlr30D3/n+j4xYPCEByOeQV+OJBAYvDtnjbTiSUQCIeASh
    5tYj+7kvfO6TZMO6vpLdFDVwtb7jElJaehylFFXREJob6si5i1d46/MtFh4IIW6EXlpr9MMK
    vJRSqiqyDEoNa/KVwb7lKI71fjiWRW1tDZYXF7F9xw7kRRGpdAYN9XX42S+fhShr2LZtG776
    1a/i6ME96OvutD4vFBGm5oY6VFdVYWJ6zrS+8QjI5PLIZHOVaDzXtPW5uYrR5qGf53nU1tZZ
    AnW212Kpf10sFDA+MYlkMuno5TdsJxrXbu4sSOCmG4ZKSTAhxImyTKiq+vpzzz8/8KEPvI/t
    7GjHjm1b8Z//8s/In/3FX7U889zzYVVVuZmZGciSbLYquIpM2zZvxH//T38KVVURDAaKytnO
    9ej1evDxDz6C5eVb0dHWDN0ot0mjAKqiYYRDQcC0U2ivrY7WfuoDD/LvuGU/WNZUXH39zAWk
    MjlbVMPjWI+qa73KlZDff23wdVTydasg+vQrr7xy8Nlnn6t/72OPltGfqWOfo9REgX/905/A
    7Nyc78mf/aJF03XE4glayGVIyC+AYxkOQFPA72944F13k/q6Wgc67vCfpubM27V9Mw7v3839
    8KdP1VJK8wD4oN8HBgaSySSyuYJmxaa+2upo5wcfvpfp7Wov06RYcYHijTMX8eqJcwaAPAEK
    tNx7VbSKJtJaJyC2QrPDacCtc2En+AULkbU9loVoNBpmWY4vFApg2fI+YFlWoesGwuEwtWxJ
    bLpe1vpqF6OdPpYyyq2PVBcCeUMWu6ZpPgC1+XweNTXVKBREnDhxErYFku1ZGQ6HIIoSFEUt
    64vzej3o6enChg39NJtNK08//XR2bGwsSykVfT6fEgwGdUEQNAC6JEm6JElobW2t2blzZ0sk
    Uh26cmWKmZiYtZK31a2OKvfnrTyc5/MSotEo7/V6YSfAlFLZmnOV2HFrigIrigJKqcqybJmi
    eHkPMFMxiRBFGX6/H7IsgxBz7wNyxd5fE/3tRyJho7/lvdOSpEMUVfj9ptKtqurgeRYsC5rP
    53Tq7n+6GQNPWaJJiX1X13UUCmJZz7351XyISQsPwePhUShIFWit5UdyO7SVkyEJbpYxcYs4
    udBgBYA4MjJ8fM+evdNdXZ39S0sxqKqGc+cuwTDg2bFj8yaWJTmW5Ukg4EMikXHMF8ayjTJw
    5coUFhYSFtOArGAsMAyDTEbCqVOTEEWtrCe9hCgTSJKOaNTHDQwMVG3atCkoCNWRK1eypFAw
    rZL8fg6BANHOnRtfyuVyrCPvsgvQznWqvp3r1BpLg1Kqv/nmm4PNzc2vbty4/qHJySkunc6W
    hYlyUTUGi4sxnDo1yOzataUmHA57WJYhgUAAoggYBodwOBzctWuXV1Wpf3p6cQXO4xwrTTMw
    ORnDjh2dgR07tlePjo6mOI5nDYMBz3vh9XJIp/NKQ0ND9MiRo/267olOTS1D16lDo4JZkWBz
    HIOmJi/i8Tl1ZGQ0X2HfEysU9kvF0TWu3DhV3Gx/ybiFRMwTQmZhCjZwNTU1XGtLi0ktRUkM
    xuf1oK+n2x5Vu1qSh+lRGQewBFN8adaiJ85ZzxmzXi9jPV62DnzXU3ijSPEB4GU5zhcJh9BQ
    X4e77rwdj7znQYRDwaKt08T4BL7+rf+N+fkFm7NtvmlVwQsvvYIv/OF/oF/52je0WDyRAbBs
    vfdZACMABgGcBnAKwAVrnCN1tTV1n/jwY8Jf/PEXyJYNA47euNWSXzupKX11HroNasDDc+hs
    awLHsbyrwCG7JmDBcbBesyTYOqBruq4XMtms2V/oSNipQwjLRs7swzfHsWhtbsLM7Cyi0SgU
    VUMul8f09Ax+9tSzOHT4CKanp5HPJHHPnbeUlGbtOWpQ+H1etDY3YGZ+CZIkQtc0zC0sIV+Q
    dMf7dl72X2DcVFF3ZeWXEBAQq6nZ/Wur6gDd0OH1edFQXwcCAlGSIEmS+VWWLVGScv/XFbGN
    Xv3vuN7V5wpJn3M/ywJ45eLFi4k33jhRTOS3bd2Cv/5Pf04ee+ShkM/n801NzSCVTpdaHMqU
    gylaWprQ2dHm+D1WoKE9Xe3Yt3ubKajmYjBQaiASCqK3q40AiDbW1TR85iMP8w+/6zZ4LcXu
    5UQKpwavUMMwEiipE9NVKs8K3v52BdvTUQFwMh5PHPv6N76pT01NwS6uoJIqtK6jp6sTf/rv
    f588cN+7/B5B8M8tLmNpaQlVIS+qoyEPgPajh/b73nHHLVYC4xTAoiX03TBQUxXFx97/EDat
    7w8DaCUE4ca6auLhCFLpDAqSTAB0VUcj6z7+/vd47rvrFrAsKe2TKB97Qghi8RS+/7PnEEuk
    FEcSmHddzrHVr0fssdQ7qaNoYyekdp9ZzI7BAOYbGho0SimfyeTK1HpNL1UN+byMmpoa8Dxv
    J9B2z3jcirVzVhyatZ5z2Sp4pB3o8HWhga9WgLYSYFBKDUIYXLx4GW+88SaSyXTZwbalpRlH
    jx5EfX1d2SEuGo1iz54d2LJlvTExMVp48sknY1euXFnWNC2m6/piLpebWVhYGJ+amhqdmpoa
    U1U1vn//vqo77rijk+N84VOnLjCjo9PQNL3iQc55qL7agc/+mdmjKCMQCPA+n49xFDgk17rO
    u4pb+lqNs2SqrOc8Ho9LvMl5ASuFxQiy2QICgUCROWTaCJro78KCaZdUU1OPycllSyHa2bdq
    09JleDwMBIEHw3Dw+QQAOk0mU2qFGGzcyGLMteIwpSYF+mpsCXv8FEVFPm/SfFmWA8uaDC2z
    EEEqAN4rSWjOlyHkxrPUnAr7jjhi2KrEg4MXJiRJer2zs50Gg0ErydIxOHgZx46d4jWNRr1e
    DxuJhFfx2TaR4FQqB1nWKiRtJQufpaUscjm5osc3QJDLKfD5gvwtt9zSGAg0RIaHsySb1YqP
    iUR4aJoojo2N5VDyg3fmRoUKTA36ds0l6/m0sbGx5NjY2C8aGurn1q0bKM6PSmr3dmFqZGQS
    p05dYKPR6nAg4GeDQS88ngBUlactLW01nZ2dTSMjM0wmU1jhK+zs/jNp1WlMTyfYLVu2N+7Z
    s6fR6/V7VJUDz/sgCAwNh0OBO++8c0sgUNN05Uqc5HLKNb1/q6s9CAYNeuHCYCGdTudQYgTn
    YDpb5K+Wg6yFCFZZ8mtR1Zy055xFM7PfWQRA7bqBfiZSQXDE4xGwcf0AeJ5nVVWFO5gTQmwf
    zrT1O+frOynRcoVq9HW5WQilLMtybjkWw2/+xmcASuH3+0AtYZFCoYCvPP51vPLqMezeuR3N
    TQ0gFIjF4/juD57Al776dTo6NiG7UG+7Au+US/cA6OY4bt2OrZuqPvGhR5k7jh5EMOCHYegr
    BV3KkmCjsgewK1nmeQ7tzQ3gOY7XNN3ZB+ymXtmHP3fQebs3SwJAVlU1uby8DMPQwRC2HP11
    egI7DrQcy6G3pwtPP/8yJEkCx3FIplI4eeYs8pKC3bt34Utf+hKOHtyLgZ4uq5hR/pwsw6C5
    sR5jY2OQJBkcy+Li8Dhksypvz7dKwjg3K/pb3N0JMX1pEvE4zp49h+XYMjKZDPK5PPKFPCRR
    gqqqSGcyIITgW9/5Hn74kyeLwmyEABsG+vDIu98Fn9fjaI2u9OaLwkJklQdc9yTYdQCxUVOZ
    UnoplUoNPv3Ms7e+6553wOv1gBoG+vt68Of/8Y9Jf283CoWCWVUtW1coL0LZl7tf3X68sbII
    VWQwGAYCPi/uOLyHJFNp7t7bD+GuW/bBYwm0MQQYGpnAzPySnaBxKAHyioX+FiowNf7N6K9z
    GK1ETCOExCilPzl2/Pi273z3ex3/7vOfBcdxZWJ1blr0QH8f/vKLf0I6O9rwwyd+gr/72vex
    ub8dHMOwmzeuZz/3ax9HY0O9g9VSwULOEiHbuXUT/vC3fo38t7//ctXg0BXwLIPhsUm8ee4y
    DIN61/V1NX740QfIA3ffgoDfW2R6uNFp27HgmVdO4LWTg4ad7NJSW04WQJYQknUUX69X8kcd
    c9X20bQVl21bLwEmlZ8RBMHf0tISzOcLbD4votyqxqzcp9M5UldXw3q9Xl0xW2rsAnTCVWh2
    Ht60CvFAuRF7H8uyAEDi8bjY3t4+1tLSdGBo6Aq7tLRcTDYpJYhGw9i6dSM4jkMuVygmFK2t
    TdiyZQMVBFZ77bVXc4ODgylZljOOWGwX2UWWZY2+vr6mffv2bYpEqjonJma50dEZiKK0auJ7
    dRGs8kN3iQ4IiKIKluXZaDTKLy4uwsF+c597pApMpLc9ZqRSKc0wjHggEDAYhmVMHdLV6eTO
    g3Y6nYXX2wmPxwNd1+HxCGhpqUMwKOD8+QkMDPQjmRQRi2VW7QUsFFRwHKzkDwiFeBQKcS2d
    Tq8mTHdTxWGfzwdRFOHuEQgGA6ivr4PP54fH44EgCOB5HjzPg2VNujjDEGzdOgBTqK6Eay0u
    pjA5ueSyMEMF5Jc4itQVG4NvFm9gHYAaj8fTi4uLx1paWt/Z1NRYMzIyDkIYGAbFyMgECgWJ
    rFvXa/X2l9DfSonuatY6V2tLcDIZslkV8bjCcBzHTE5mkMvpxaSb41hEozxSqcX8/PyCasVg
    ugr6u6JI+jbZw1FCiEYpVV588cWzLS0tz27evPGD09Oz/OLisqP/eeV7NgwDV65MQBRlsn37
    BtTUNILnBXAcITzvES5fHsf4+Jxlf1RZVdr+qusUw8MLYFmW7+1dV8uyPOE4A9XVAnw+imCw
    rmppqUAuX14iqZS86p5oJ8ReL4uWFi/m5sbkK1euOJPfrOPKXQ0B5tZo3hoOKp3du+v2QbWp
    WjVer7dl966d8Pl8oFR3IHUUDGGwfqAPzY0N3OT0DOdALgqU0qzLvkN0bfS6IyhoDoqWdj03
    PitBSxYKhRM/+vFPtu7auZ07sHcPOJaFYehIp9P4wRM/wU+e/AXe+8iD6O5sh65pOH/hIr78
    1a/TJ376CyOdyRSsqrsdeJMo+XXah6wIgK011dFN999zp/+j73+YDPR2mR+Irrt8N1cmtqsm
    xS6kmGUJ6uuq4REEVpRk3kGzMxyBxRmIrwfdNw9gYXZ21igU8kw4FF6pOLxCcdZUzO7r7QZA
    ce78eTAMg6Erw3jm+Zdxyy23YmJiAvlMCvfefTs8HsGyoCmnUTMMg9qaKmTzBYiiCEVRcOHy
    GDUMKjsWXqXrpkKAK2SbxN54zpw9h7/8q7+GKIoQBNNMXeAFhENBCAIPjuNQU10NTdWQUbMg
    BJAkCQuLy2iqr3O095ZbyJQUdWmRLk2ubUx4o7yB7bktA1iiFK+dOHly/+XhYe+2LZthWAlr
    Y30dPv+ZT0O2BMOK9GaXJzWcVW5n0otyRfYVyZwrWTy6fyd2bl6HaCgIhkHx9VTNwLlLI0hl
    svYB2EZ/FRdKtJbB1z6Yq5RSGcDpfD7/9De++a0PHTq437N39y5Xco8Ve1BrazN+77d/A3t3
    78T//vb38KPnTqKusRm/9blPY++u7SXUl5bbnVEXxZowBLcc3ItIOEi++s3v4vWzl3DqwhXw
    ggcfevQ+8sDdt2HTul6wDHE4EdAVxQlCgOGxGXzvyedoLl+QHME247psVFi+zuvdcCCCTuZC
    wRGHGQBMc3NzY3V1ddPQ0ChRVW3FIdEwKOLxNDo6Gtja2lqSyWScjK6UFXeXrfuS4z0arhjg
    FEq8rgXogmlxRZ566imxqanpp42NjfsOHdo/MDh4EYlECoQwqKurxaZN69DQUI/XXjuBTCaL
    cDiE/v5e9PV1GsvLS9Jzzx3PTE9Ppw3DcPp1FllmgUCA7Nu3b8OmTZsOFwpKw6lTl5iFhVgx
    RqzmM/xWen5XUhRNxMswKIlEIl7HuYqpIPZ51davt2udj42NKaqqzgaDAcPsvRQriHhVSuyB
    bLYAVTXQ1NQIgCIY9KO6Ooy5uRmwLIva2gacPj0JVdXBMJyL4mtesqyBZQGe58DzBOEwh9nZ
    RTmXy5XFYAtFVG+2BNjWMiFlSlUEbW0t2LVruy0yC5bloGkaVNU8ixgGhSjK4LiSs6Ig8PB4
    BCSTuQp5rBOJd1Oib2p7JKcwpXz58uXB9vb2y52d7Qempmah67RYUJmfX0YikS7rQ3cyEq6m
    /F6Jpu+cr05LL7NISDEykgalDHQdxb5r257H7yf08uWpvCRJtgq+XRx0gkTyGuUlZYBkOp1O
    nT179skDBw7s2rZty+YXX3wVpqGC01oILpo3MDOziHQ6j76+TvT2dsLvD+DKlRmcPTsMWVaL
    LSSVenSdzyfLOoaG5okoaqS7ux4dHV5QqiKTEcncXJrMz+cgy3pZHFq5N5oF2oYGHzhO1E+d
    OpXP5/MrYrCjAC2uxm5bqwRYd6GwlZRwGev1u9taWhp37dgOljEnE3UgdwYours6sGnjenZy
    esZvPb+T1mWLXSWs+7qrCu6km9IbQX+xEuACgO+dPHWm/3f/4I8Pv/u+d/EbN65HPpfDK68e
    w8+eehqtrS145D0PQJQkfO+HP8ZXHv8mPXNuUNZ13f5Qsyi3K0mi5BfWCuDoQG/3wCc/8l7+
    gXfeiWg4VBRwwSoq02UHRpRQJ7cIVhkibADRcAjhUIBNZbJe63N0Fzic/np0LQKua3wlAFOT
    U1PK/PyCNxIOo6RmXY4wOZE4g1L0dHVi47o+/PiJJ0BA8fKrr0OUVezavRv/85/+CYf27cT6
    /l5HIuOkUlMwBIiEglBVHblcHvPLCYxOzhoARAIotFwNVkW5t9vN2wNsob/UoNiyZTP+8s//
    IziWBcsy+MpXH0dzYwM+9uEPgOc5MKYacRG6JZTizVNn8I9f/hruuv0ogkE/dE1dMRtWmB+R
    UgC+GQZmNRTY2lRPTk5OzR07/nr35o3rS/PCMCAIPATeVFqnRol54EY6i8hi2TqDK/HFSvTX
    kSx6BQHeat7yWrctvAiS6QyGRiZhFWLsm5MqudbB1z1mMiEkTSn92dDlyzu+/JWv7ujr7UFV
    NFLequBYWxQU0A34fT7cfedt2LF1M86eG0RNTRU2re+3ArReET0u3Qec1OWdWzai/fcb8exL
    xxCLJ7Fv11YM9HSYDAVrDO0YVFYctHysCwUJP/j587hweUx3VJntwmTKQkczjuqz21LqeiTA
    qnPcXfty8SS3d+/e9QzDNM/PL0LXzcO1+/CXTGag6+Da29s94+PjulXIECmlOUKIXYyNo+Sh
    jAoF0RvGgCnuJ5TSH/zgByfuueeev2tra/1YY2P9plyu4CWEwOfzqqFQAPF4io/HE6Szsx0b
    N/bTUCigXrgwmD558mQ6k8nY6ELSHYPr6ur422+//XBLS+uRubnl6NDQGExV1NWVna/W81uJ
    7lc6nJq/13UKXQcTDAY9KOmpOC/OFYff9tjr3CpVVVULhcKU3+8rVFVVhd3iQ6shwIQwKBQk
    LC4m0NvbB5blYBg8fD4OZ8+OY2BgAMmkhOXlTJlQmZtuqShmjs/zLIJBBoKgGtPTM7KiKO4C
    jOKKwTclA8v+qGZm5pHLFaDrFBzHYceOzZifX8aVK5OupNb06AYIOjub0dPTiqkp09axlJRh
    hRJ0hVrzTaWO7YrBho1mXr58eWH//n2vNzc37gmHQ1wikSn21ptCcXrRt7wSC6FyYWalcvnK
    ZBkuP1oCRaEghBZ7s+3fBYMcAFmfmZlRHOwr1YH+2uyrigjl25gE65aGg/zGG28Md3Z2PtHT
    09UxPT0bHhoacRTjsCJVs8cglyvg3LkrmJ5eRDgcRCyWhigqZfZbq7EznL69qmpgbCyGZLKA
    xsYIJElHLFaAJOmwhQbdbJnyv4VBKMSjsZGnly6dlyYmJtwx2K0NJWGV9o+1okDD8WJXQ3Fa
    ABzZs2eXv7+/ryQu5FCMpYaBqmgEtx7ejxdffjWcyxcCjkDurjQ7+9iu9ffhOgQFpxIqAAxT
    Sv/H4IWLkyMjo4e8Pm+VruukUBAlXdeDe3bvjJw7fwE//+Uz9JnnXtRi8YQzsXf2cNmId8p6
    vxt4nnvXob27Oj7/6Y8w+3ZtB8exJfVSW3wHlSyWjLKDdlnyu8Kb2FaEJggH/QgF/AxMOt1q
    wZe9HsHXUvU2KKWXJyYmk1euDDet6+8rJQ/OQ7Gz19lKkMOhID7x4fdDlmUkEkmMT07hgQcf
    wsT4OLLpJO69+3ZT+VnTXFi2NT6EwOcRAADxVBovnziHRCqrAhBpOf3evhT8X4AAF2MlKGpr
    qlG3f5/Va5VDPl9AIplCVVUUPMeDGlbiYFF2AQpJlnFldNykApMKyg4U5eidlRAyN1kF2mXN
    UKSWUkrH8/n8uZdffqX7sYceRDQattTSqeVzW67svmIOVhBmAyoluit7ZMue092rD/PvXViK
    YWx6zhbFcyZGkiPwSmuM/toqnrpFx5UAjFJKf/TzXzzVdfTIoar3v/fRMmq4+/0696262mrc
    cethGPZcM4yrimlVYrhQaqCmOoqH77sLhm6AZRlQQ1/RIoIKxUIK4PXTg3jy2VepqmkFB+XK
    Tn5TDqaOM/ga1ynmUJRTPEmFQy2hlJKBgYFAZ2fntng8WbO0FFs1KcvnJcTjGdLR0RGMRCJc
    KpVyIsyi4/CRdTC/4Pq62v3ruYZpOp1WfvCDH7y8YcOGeGdn5x6/398jy7L/woVZo62tvau5
    uaVv375dJBQKqslkPPX006/FxsbGMqqqSo4YnHBcmd7e3tCtt956VzAYPnT58oRvdHSqTEDr
    6skvc42EmFk1gTQMCsOgxO/32/GWA8ARQjjH9+x1TGbo8vLyeG9v70J9fX14dnZhlWSikhWS
    KUrk9XoRCvkRDPowNTVmob+NOHduykJ/2YrPZY6HuUwFgYXPB+RyS9rU1JTNwJBc8VfBGvdE
    /1vOz0XGFAFyubxFyWcQDAbh9/vg8QiQJLsvlbUSPYBhzAIJz3OIRoMW/b9SH7BbCdr5mjez
    O2MpmZMkKb+wsHiqt7dvsbGxoSWZzJbNB2cyfHWU9+pMjJVJGINKbQvlqu7m78JhDpnMkhKP
    x+292LlviigXCnzb2o9WERJTCSGypmm511577YX7779/55Ytm96xuBhjTEEssmLPWin6BSQS
    GSSTOUdxgbnqmJU/T2l+JZMSMhkVTnGy1fq1nc/H8wxaW33IZpe106dP5zRNcxegU454VLha
    AZp7OwPM1Ra088NwUHQO1tbU7L//vntJKBiEoTuOyI4DEcswOHpoP9YP9HtPnDrTBFN0wwfT
    TNpnXR7r/aju4LqWCe5brT5blSsVwBil9JuSLB+TZLkDpk2FF8COl189tvnY8TdoMpVOG4ax
    aH2g9iE17wq+Seu97vX7fA/e/847G37j0x9Gb1eHSYuxxnI1ZedrKUCXDujGikOgAYqAz4uA
    38s4kl0OZl+ZMwAz13FTpACG0+n02Asvvth05x23QeC5suTKPQ+cQgvbtmzCf/tPf4qvPP4t
    JDN5bN++HV/+8pdxcO+OMvS3PFmzLwLCMAABhkYm8cqbgxSVlehEq+/yhvSi/yuq0DBJyygK
    i8FqSwj4/eBYtkgJhV2VZ2x7BiAY8CMUDJjKstbmRg0DBlkpiEUrpN3XKKDdyABsJ5JJACcG
    L1x8x/DIqG/Prh2OBM6pnF5Cf8vESJ0opxP9LUN7sQKFXPEaWPkaBjUwNjWHVDqrO4JrsS/T
    oU58PSxSqOP1JSthejkWj+96/OvfvHffnl1MT3dXZXaKS9HZoOVq2tRJUXYgve5iAq2Q0BIC
    MCwpT3zdtGfHfbuo8O0fP425xZgthpZ1VJxTAFIWKnrN4Hud9sRV1/ftt9++nuO4WycnZ/hs
    Nr9qj5ymGZieXsTu3euCPT091SdPnpxbpdBpf8b623R2eLuLpJQQoimKIp45c2bszJkzaY7j
    zlFKq3Vdr1pejiV4nqcAuDffvBi/ePFiKpPJyI6ikbMInQCQ3rx5c/Utt9zyICHcvrNnL/PT
    0wsW4sasOBhWSmzfigfwagmwaYlD4fF4OIZheMMw3DHYzcRa8zl37ty5qZ6enkvt7S39Q0PD
    kGXFJX7FlAkzOQ/J6XQer79+Ab29bVi3rhXj4+NYt24dMhkZS0vpFQJEK1E88+bzMQiHNXr+
    /KiUTCZXxGBH0e9tFRx6m4eyGH+diLdhGJBlxer1rbSWzK+qqkFVtWL8oZRWQH6BlbZUZeGX
    rH4uuOExWAMgj4yMDPf19V1uaWlqGR2dgqYZKwokTuRwtbVXGf1lKqC/zFWe3+1Py8DvZzA3
    F9NyuZxdkHQK1TmLMtoa5izu9i1xYmJi/tKlS09s3bp14/r1/e1vvnm2aK9VTthlUFmV3o3Q
    MlhJ8q2sZ1A+lqhYZLja89TUeBCJUPrii6cLy8vLNu05Ze3LKastNm2dM6SrnbW56z5zLVI+
    pbSDEPKB2249Wndg/76ywwt1UlUtwZHOjjY8eN/d5OLlKw35fKERpvhGCKbXYdBKhguOiXRT
    bWxORU5CSIpSOm59SLY3o5pOZ2LWw/MOJNtJV7TpZknrfd4aCYce/vB731P96Y++H/W1NaBU
    L0tWyvtVjavaHdl0YQqjouWH87E+j2D6l5ZXmitdZA3nUlmBAUCMUvryc8+/uOvixUue7Vu3
    uMaglPQ6ey/t/sBUMoVXjr2Bo7feipmZGSRiS3jg330CXo9gFhQcCQt1+tpaBQdRlPHLl09i
    ZiGmokR/dPpmrqqKfSOLNFdNgS1FKk3TkEwmkMvlkM/nsXHjehiahtdPvAlN06CpKjRdg65r
    MHRz/sQTCWxcP4ChK6NIJJIgDMBzHDrbWtBYX7sqHmQeHlf1aiQ3OBDbbAONECJSSi9Mz8xM
    nzp9un/Xjq3FQ8jKfvPyNeRO8ir2+mI19PcarwEKTdcxPD6NXEHUHUmJ4kJ/3WwEukZr05kA
    y9a4LQH4yalTpzd/+7vf7/qd3/oNCDznSj7dSC6umtDSSuh4WZHFRY12fRbunzkp6aAUqq7j
    qReP47WTgzpKSpPO4JtEuSiS9HZX9v+1SaStNG59Jfv27fNFo5EHUqn0+tHRCau3kCmj9TkP
    H8vLKSSTeX7Tpk3NY2NjM8lkMg7TeivgKESLrvh7s+1tTuuvAqVU0DSNtePy6OiosrCwsKjr
    ulcURaCcsuhMgNMAUjt37qw/fPjwexVF33vu3AVuYSHm2L/eemJb6QBeGSlG2QGSUkDXKXie
    Z3me52RZdsZd5nrGYNvve2RkJJtIJF5sbm68rampMTQxMY1KfZOVKaimgFVra32x97eurhz9
    raRWW34fqKoC8vmYPjQ0lNdMloY7BhcqHYxv9Dy1e3+pXUG2bizLwev1QRAEeL1ezM0tQVU1
    NDc3gGVZsCxX9Py1ExOfz4tEIou6uiiqqowiYyCblSHLWsUeYEqJJZbFrJb03rDkdxUxLG1q
    aiqWy+VONTTUHQiFgt5UKrvKPLv6/Fv9d8xVrckqr2/zq8fDgucNurS0pGqmFYa2SgxeEzCk
    0phRSlVCiEQpLRw/fvxsR0f7U+vW9X5kampOmJ9fdglZVU5Mr46WM6ug5uXdr6uLjq1mAWeO
    qc/HobnZS6emhuXLly/b1Gcn8mvH4Kxjna8oQNtrnbuOC9t+UWIlrO/r6ek++tGPfAi1NdVW
    71V5UuE8jLAMg3vvvh2vHHvD9/Onn+ulFMsAMpTSiNXsnHMF4OLBw/X6N+qmo9zGyZIpLmuI
    X3Qg2cT1fwoOtEEHcGdtTfXDn/rwe6s/9oGHEQmHLBpqZdpyJXTXpj+7EaaVtD9jhYWLIHDw
    CDys98G5Lnev2fVC5SQALwyPjNzz+Ne/ubX7T/8YoVDQOesdqBDKqPaapuGpZ19ATpSxZ/du
    fOUrXzHR34E+a0xX68M0EXFRkpDK5DCzsEQNg9rUSNsKJYuSJHvhOqFub0f+S2xRqktDQ/jL
    v/prLC0tQ9M06EXUbAXZY8X9wYtDRfooyzD4wCMP4L3vuRcMWfkxmjQuslrie7MEYMMhpDIp
    itLlE2+e6n/s4feYtmbORBflyKxbxKp4EAEBZawDiEFAwZiPJaUxNlAuXuek/JYndmaf6tTs
    AnTdcPa+2eivm46/JvTnq/RPS4SQPKX0fEEUn/7Bj5746J233yrs2rkdxioJPXWLgIEW54v5
    CgxAHL3DZerNhmWHREpDRVzPX8GKyblHEACjE9P4wc+ep/mCaKPYTuqzs/p8o9Hfqya/AJgD
    Bw7spRSPDg2N+OLxhAv9XWlfI8saRkfnsHt3f9XOnTs6XnjhxZimaWmrCB1yxF8VLr/Vos3X
    jb853QryFgtNp5QqACRKaSGXy6WsZJ5zJcBOzZHczp07644cOfI+SdL2nTlziV1aSvyKys6V
    6c0ltWcGTkXu1RBhw6BgWZZhTa6r+7qeMdheWer58+dfP3r06MktWzYcjcWSxPamvXYfJoPm
    ZlP5+ezZMfT19Vvob2bVQoLzgM1xDHiegBCZnjx5UVxYWMg6YnDOEZMLuA7e3L/qzev1Qpbl
    FT9vbm7E3r074fV6y1TAnTZQzjDp3HNrayPFeaTrBi5dmsXMTPKqoK5LhfqGJ79XOe+pmUwm
    v7y8fKajozNRX1/TbDrhvJUk1S1sVa6C7fZIrpTMVW5lKD2/18sCUI2lpWX7vFdEf60YrFRi
    Iryde6U7CSaEFO2Xstls4uzZc08fOXJkz/r1/VsTiTQ0zVglCWXKLNjcY0JpOY1+5WMqICvX
    QHvd+yTDMGhs9ALIGZbwlZP6nITpCnRN9Nc5vtxaB17XC9qU2buDweAnP/KhD4QOHthfRudb
    iTqiiAI3NzbgM5/4ICanZ+ouXLqyGSUBEuem5nyzZRPrBgZi6qok23uMZgde630EYNKhOfeB
    ESZ9Nk9Nb4E7q6ORR37to++v/sSHHrWUZnWXgqpRod935c9XCPNQWlRAXVUECxQsQ0xv0vJA
    67yuN/3KIITolNJRVVV/8MMfPdG1f+/u8CMPvXtl8uE67BICzMzM4smnnsWhQ4cxNzeH5cV5
    /M6vfbAojFOGxJV2SvMnOkUqnYGqabbycxYr5djtnxUqUV5uxltpKVJUVVXh1luOQtc1BPx+
    eD0eU22T48DzLBKJJE6dOYv9e3ahprqq1JOq6zAMA7quQdM0GIaOrvY2s8+XVmoLNJNk3HzU
    50rrWbM23jPnBy/cPjs/5w+H+1f09hIAhCEANenhlDDQNQOyLEMsiJBkCYqiQFNVE0E3DLNO
    yhBwDAOeZ+EROHg9AjiGAQhjCTURUIOY/bCOfZMQsxd9bjFOUS7+crVedHqdxszpf5gE8OzQ
    5eEDP3ziJ5s2rh+AIAhlNHB7vTJ2pksZUGLA0AkkWUY+n4coipAkCbIiQ1VVc57pZl80IWZP
    Occy4DgWAs/D6+Hh9QjweQSwDAMKBiCGNZaOooWDRi3JCn769Cu4NDKhA8gTIEdd6K8VfDOu
    ZFC/GdBQR/JLPvGJT3QEAoHPTk7ODFy8eBmaVi5+tVpStriYwtRUjNu4cXN3IpFMnDp1Kk0p
    jRJC7BjsTIDh3ONukiTYjqfE8b1mIcJ5SmkOgJ8Q4qWUOovQZcjNxo2bIkeOHHlUUfQDlZLf
    f1vP77X+L8pQKUthmmFsznXleEzWPlZQEEIMSql+4sSJuc7Ozu+1t3cMrFvX23T27CVQenVE
    CQB8Pg+6upoxNzdjqbw24/z5aSiKVqYyW+ojLC/WCII5h6enJ9WzZ89mdV3PO1gaGZjWZJWE
    6W6SeEsrlqHz+QLGx6dgKn9r0HVq93+DUiAYDKK6OoK5uWUrgWHKeiqd6HAmI5Whv+5k2rJL
    Ijdj/K2UzFFK1cnJibGurq7Rhob65rGx2aLlk1vgy9lXTamZUPE8B44zr5VIOgGljBWPCAzD
    it+UFJO91RXOCfx+BoqSNyzl/Erxt2IB+nrkIRYKzJ89e3ZkYGDgqc7O1p7R0cngzMwiKlG8
    S3PDGVMYS3GdB89zlge16T9t28uVrM5IkWFgMldg9eyTaybE9hUIcKiv5+mFC2el6enpnAP9
    teOv0xnH7b5QcXy56zRpbdqzAOA2j8fzR488/J7OD3/wg/B6PKY6rPPQU6GvzUY0d23bgt/5
    7CeYv/of/9gxOj6pAlAopRJMap0z27dHrqwCcAPRYGcPnj0RFTuxtT40PwDBCr5wTFbFep8a
    gKOhYPB9H/vAI7Ufff9DxeT3aiiuWwTL3Y9o/94cl2uLYNkHQ5YtJiqrJb7keh9uLHrlc0vL
    yzu+/NWv3btp4zp2w7oB6PpKtJKAghJA1zQ89cwLyBVk7NmzB48//jj279qGLRvXoejNg9Jd
    dxKs6xrmF5ehmg0oeRfym3EValaIDt2U9GcHjEspRUtLMz79yY+DFHuJDBiW9yoBxbFjr+Mr
    j38TDz1wL/bs3AZN08w5U/SxNQBLGErXdRiG5mJ8lBatRYG+WhJ8MwRnw1qPIoALc/MLicHB
    i/51fb1FxJAwDAxDRzabQzqdxuLSMqamZzA7N4+lpWUkkimkMxmIkgRFMZNfXTeKgYZlCDiO
    g8Bz8Pt8iIaDqK4Ko76mGi2NdWhuqEEkHETQ7zX73W2EE0AimcZiLEFRLvriDMQVlVDXQKG9
    EhXaiaiN6Lr+7FO/fKbnwfvv9e3cvgW67QdNAIYwkCQJ6Uwa8XgCU9MzmJyawfzCApZjcSSS
    KRREEbIsm2OoaSaKjNJapRRgWQY8x1kJsIBQ0I/qaAS1NVE0N9Shrake9XXVCAcDCPi8Zi+7
    47O4NDKOX7xwjOq6IcJMflegvyhRryr6Dl7vde7W3QBAHn300frGxsbPpVKpd7755mk2k8kW
    bWVWU0C1f6frFFeuzCMU8vv379+/Wdf1/Pnz50XDMApWT7m9r7lRV+MmSIKpCwWGoxgjWIhw
    zipAC1YPrbOvWaWUqp2dnf7bb7/t3YZBbj9//gq/vJxYlfr31vt6f/Wk2HVIJcR8QKXE97rE
    YWuu2Qq98gsvvHD8oYce+vmmTes+uLSUEObmlrDSiqf8oN3cXFtEf3t7e5HNylhcTK2KGrnp
    +n6/gFwuYxw/fryQTCbz1yhCr6Xq7ts1qiCEIJFIoUTttRNbFvZH3tfXgYaGGly8OI5MplD8
    XSmZYx0JMVtm01PpRd9C/L1ZziY6APXSpaHFAwcOXKqvrz7g9/tYW328WFBnWQiCAJ/Pi2DQ
    j1AogEDAB5/PC4/HTt5KxQJnYcAWVjMMCk2jUFUDsqxDkgxIkgFZNqDrZnLs3jN9PhbZbFoT
    RdEdd91q5GuaAK9CH1cIIZKiKJlz5869fMcdd97S19e1Z2kpadlxkTKGAc/z8Hq9CAR8CIeD
    CIX88Pt98Pk8EAQ78XWOIakIptiFG103x1JRNIiiDlHUIUk6FIVaBZ7yPYJhCBoafCgUEvrg
    4Pm8Q/jKGX/tArTznH3V9iNurQOvlfwylFIvgDt4nv/Te9/1zq2/94XfQWNjAwxdK+JxxdTC
    QetzNvGbVGiCd9x+BLquc3/zD1/uuzI6zltJl2YdSA3rNW0akDPhuFFosLsi7kyGeWtT9sKk
    PwvElFaDAzHRqJmZbhUE/iMP3ntXyyc+9BjCIYdwGK2c1FakP5epIq/iOVpBJXrFc5U2ydWC
    7vVMgp3o0hKA7504cbL/H//nl9f/xz/+Q9TV1oBCh6ZRKIoKTVOgqipURcH4xBR+8vNf4vCR
    o1heXsbwlSHcc+tHMbewBIaYvoICz4FjGTCEgCGmfzAhBBxDIOs6JmfmoWm6ilKvUVngdXiS
    3XTUq2tPXadtVEl9155jBICqaUgmU8hmc9B13fJLLk+AqWGU9Zpjxa5vVhhcCPCNmk9vKZBY
    1gIT8URi7Oy5863vvvceUEqRSCRwaegyzg5ewKWhK5icmkYqnQHLsgj4fQgGfAgHg6iKhtHs
    rYfX47F6MFHGepEVBaIoQRRFxJMpTEzPIp8vIF+QwPEs6muq0NXWjHW9HVjf24GWxlr4vQLi
    yTTS2by+CvrrtONa8+rzagHYShTTAF4ZGRu/8+lnn9uwZeN6cBwLUZIwPTOLM+fO4+y5QVwe
    HsHC4hI0TUMw4EcwGEAkFEJ1VRQtTQ3m81NnDHHOLlo2aWRZRj5fwOzCIoZGxpDNFaAoCoJ+
    H1qaGzDQ3Y6NAz0Y6G5DVSQEwzDw1AvHMTm7oAHIVUB/beGrq/b+3ojE1xp/BgBz//331/T1
    9X5OFKVPvP76Kf/U1Awqi+Gs3uMlSQoGB6exdWtX9MiRo3v9fj89efKkYlvNWBQ7Z1woYxm4
    kvIbsWYNx99jI7t2Gw+PkqsBi5K/vU4pNXw+H3v33Xe/w+PxPnju3BXv/PwyTLVZBtcSfPnV
    e4KZa4hgMY7+TVoscGCl1SS5zsFCA6AsLS0lT548+ZNDhw5t3Llz8558/gTJZvOg1GS2sCxj
    IUbm5ff70NPTgvn5WQBAS0srxsfj8HoFeDykiMJZDP6KveqBgIBYbFmbnZ21Y3DOFYNzN8P6
    fKtj6fbnteeandzayTDLsvB6BXCcZUfIVBYqciYmzud2A74VEOCbJgZXApVyuVwhmUydr6mp
    y4TDwap8XgLH8QiHg6itrUZtbRTRaAjBoA8sy0LTDOvSoWkGRFGBphmWCBRWFKU4jgXHseB5
    FoGAgEiEAccxYFnG+v8G8nkd2ayOfJ5C08wivtfLIBbLqrLJay/GYRvQchSgr6cYqjv/EM+f
    Pz+5Y8eOl9ramjZXV0d9S0txMAyBx+NBdXUE9fU1qKmJIhIJweMRYBgGVFWDpplf83kZhkEd
    c3VlMcA5jRiGFMfT5/OhpoaxPKwBRdGRz2vIZFRksxoKBbPAEAjwqKlh6Zkzl6WlpeUVzgsO
    6nPWUeC6JvuKW6vga72YvUNVA3jY6/H+1oMPPtD3J//+j0h/Xy8MXTMXJiGgxKQGUkJBGQID
    BNDNwTIxJpPuRwmBwHO4544jiIYD7D986Rtdx988E1I1rQbAMev13H2pIsq59telIl0h+Dor
    4nbwtVFTp5oy4wi+9tUC4MMH9+4a+MzHP4Ca6mhZ33QJTTNWCrxcw/7oqjRpVH6MSWvVKyFz
    N3Kj1KzKlkgpvaio6re/98MnPhMKBhve88B9SCaTGB4dwcTkNBYWF5FIJJEvFJBMpiB4fThw
    YD8ef/xrWFxcwj9++Rv4569/2wrULDiWgdfjQSDgQygYQDRkJi/1tdXgORZXxibhONQ70d+M
    o0fdpmXcdOrPFawPir4/RdEVUvqrbXskmxQeiYSxYV0/QqFgUal3haiQsxRYIfElFi3J2gzf
    Ci/mRo+ZTY9cAnD55Kkzh5548ufM0NAVHHvjBOLxBHxeL5oa67F+oA+RcMhM7kQJuVwOhYKI
    VDqLmdkFiJJkionpehEFZ605F/T7EAz6UV9bja72Fvh9HjCEQBQlLMcTuDw6gWMnz4EQgq62
    JuzdvgHjU/PQTEaChNWpVzfCBsQdgAsARhRFeeO5F17qv+uOW7nFxSU89fSzOH32PGRZQV1t
    Nbo7O7Bj22YIPAdRlJBKpZFKp7GwuIR0OgNRlqFrWpndQpFAZJjq0SzDQBB4hEMBVEcjaGqo
    x4b+IIIBPxhCkM5kMTE1i+deOYEnnnoBVZEwDuzcjLbmBrxw7JTd25+hJdpV4iro71rZSv1K
    8ccuTH7oQx/qbG9v+5woSh85fvxE5OLFIeg6tajPTJlSb2XhktLhJpuVcP78NNmwobV67979
    h+vr6wLHj7/+2uLiomH5/TFWEZdDORWtzAntehWhV0mCnTZRxHVucBZzbQtD5qGHHtpZVVX1
    wdHRmarJyXmLRrmyr/fqSS2umRhXUo+u5HtrJzWGYVDDMG4YQ8ZlE6cTQhRKqXzixInRurq6
    /7N+/YaGvXu3dV64MAKe51FVFUEkEkQg4IfP5wXPc/B4BBCi48yZMXR3dyMcDqOvz4OuroYi
    cgSYol92AmMiSDpk2SxsR6MejI4mVUmSKsXgDFa2IN0UQm32OshmsxbqaMBJO3VTT8v9fAFR
    lBCPp4vIXTlNHK775QmJ87nNcQaYUqMxWQXUuFkq87A9gRcWFq40NjYudXa2VkWjYbS0NKKm
    JgKGYaEoKnI5CQsLSaiqDkIYCAJXpD/7fEIRBTaLC6QMsbTnnKqaqK+u06JYmJnEsair86Cx
    kUDTgExGR6FA4fEQ5HJZRVVVtw2XU/zqurAQVtPjsM7K2ZGRkeP79u2/r7OzZcAwKFpaGtDe
    3oRQKAhdN5DLiVhaSkBRNKvdQIDHw8Pj4REO+x3jR1zHO1JGuacU0DRz3aqqgUJBg6oa9ryD
    z8chEBBQVeUFIQS5nIZEQkUgwEFRMvrly5czhmHYxecETEvYpAP9LfwqBS7u7V7IrsDLAegD
    8Km6urr3f+D976/7jc9/FtFIGEOXL0O3aJImouTwETUMS83YMFFM1888Ao/62ioc2rsTjfW1
    zDe+/aP6H//iuTsXl+NtAF6glJ6xKEy8o5orOirSuF4IXIXgazM9na/PWAuiUs8OsdDzd3W0
    tRz9tY++j+nuaFuZ/NpoL9wKs4Z1EHxrCtBA5d5huMR3dN2U2b+JEjknrVwihGQppc9nMtnw
    l7/29cee/MUvmwgBEwoF0Vhfh4a6OnS2tyGXy+GHP/k5Hrn7nVheXsaZ06fR19MFRdUgyTJ6
    uzuwa9smgFLk8nnk8gVkszlMzszj7MXLSKWzWIrFkc3ldVfy66Q/2+jvisPxTaI86ZyvNp2O
    pZQyszOzeOHFl6HIMmSrV1VVFbMXSdMt2q4OURTR0dGG146/gTdPnSkhxE47Gti0VhQLXwxD
    wDIMONbsJ9E0Dcl0xrl/ONcx7/iesw8x1xtVcq1lzdp0B988dSY9PTsbqa+tI91dHWTb5o1Q
    FAXLsTiGrgwjXxChqioYhkE4GER1VQRVkTDaWhrh83oxOj6J5185Dr/Ph7tuPYjaqijyhQLy
    +QJSmSzm5heRzuagaToE3hShq62OYl1vJ4J+H3L5Asan5/C17/yMxhIpoqqaipLaZCUbkOsm
    0FQpAFstKzKANKX0+Omz5975W7/77+tTqZTa1dnO3Xn7LaSxvg6ZbBaXr4zg5VePI5vLQdd0
    hEIBNNbXobuzDTVVVYhGwwgF/PAIAjiONXt7Lbq9pmqQFRm5XAHJdBqJZApLsQQuj4wjlcmC
    AAj4vWhuqMPG/m7cfngPZEXB4KUR+t0nn0Mmm0MuL2oo0aziMF0IYgDiDuErt8L7DUN/bbGr
    7u5u4Z577tkRjUZ/K5PJvOv48RO+ixcvQ9cpGIbD1YRdVkcrGeRyMs6fn0VPT32ou3vd/oaG
    xoZz586+dO7c+dOFQsG24uFdSbByo8ZmlWI0dWUVNj16Bfp155131rS3t78vkcisHxmZhKbp
    ruR3NQo08xZUZ/91SLFdJNM0jWqadrPEYWePYf6ZZ545znFcpKen50PNzQ1dABhV1SCKMgoF
    CbFYCgzDoKenBRMTk6CUoqOjE5lMAYDZupDPy1hayoJSWD2HnOX1yyMa9YHnWXi9PDRNNubn
    50XDMJzobwblbUgr+n9vZF++c27aa1bXdZYQMNFoFJ2dbRAEodijavZXskV6s9nHKiCbzaOj
    o6mYmFWaX3YftvmajCPJNj2UzaSQBWdWoCvFXidIo98EjI4iCjw7Ozu/efPmyxs39vaKokRy
    OYlZWkpB03R4PDz8fh8iEdOSkRDGspPSoCgaCgUFqmoKtTU1RRGJ+JBIFLC8nHeglRy8Xg6C
    wIHjGKtQYaK9sqwjkVBBqSl8FYlwqK01KMdpNJfLKxVisNOB4UYUSqkLhBOHhobGtm3bdnrD
    ht6+7u42yjAMXV5OcbOzpkNAMOhHVVUYfr8XHMeCUkCSFBQKCnK5LGRZhaLoFn25JCZmMhLY
    IrAhCJyVOHPw+XiEw2bibBgmJTqX07C4KELXAY+HRSTC0tZWAp9PwIULQ4VYLJayEl47Bset
    RDiDch2KtzSu3Nu9mB1BgwNwmGGYP9y4ceORz33us8KjjzyMoN+HHz3xY/z9P/y/kCSpPA45
    PTNBy2hs9hswDAPdHW34vd/4JDpbm9DT0Yrf/exHsWf7Ru83v/uTrW+cHmwVJbmZUvqMy4+2
    EqpKr0cVehVTaufNCdW7k18CYJvXIzzyyAP3+Pft3m71Xhqr9/uisp9m2WNhWIxBY5Wk2JHA
    WEqqzsfIsgLJVCy82mHmeh+snchSnhASp5Q+WSgUciOjYx9736MPtf3hF/4dqqsi4FgWDCH4
    +r98G03NzTh06DC+9c1vIJ3OANRAX08Xsrkc5uYXccdvfhrrerugKAp0TYNmCTrpmoYTp8/j
    T//r/0dT6azoqDDbwTdXAf3VcIPpz5X64G1vbkIIQymNAjjCcdzAa8eP4/zghbIqclFYg5i+
    v4yVzBKGsXqCHSvNoUxMQUENm0ZtfjUMo8QmoBQGpUhnc4RhGL9hGLZHthclv28/Su0CumPt
    GKsk9Gt9kLZp0CdFSfrf9XV1Rwb6ejdMz8ySc4MX4PEIaG9rwZ6d29Hd2Y62liY0NdQhHAqa
    wkwsA1CKVDqN//L3X4Lf5wU1DHS0NOFDj94HlhBL2EmFqmrI5nJYXI5hZm4RkzPzGJ+axcnz
    Q8jm8oiEguhoaURnawMmZxZkWkrK7MuujkquZORGFA/cglgXCgXx2MnTZ+oO7N1deOjd9x06
    c27Q+/Szz6NQENHV0YZ9u3dgXX8vejrb0dhQb4qCsYzVe2S2JoBSqKqKbDYHjyV2ZRdjDEM3
    16+mQVVVyLKM5XgS41MzuDI6iaHhMfzg588DALZt6MOmdd0YvDwqT5nU5wJKVkd25TluCW+4
    g+/NkPyShx9+ONrT0/MAx3Gfn5mZ3XL8+Bvs5OQ0ampq0NnZWYb+2tTKt3LJsobFxSxkWcPw
    8DLSaUXo7q4ZOHjwSEN3d3fX66+/8dzo6OhlSinviMF5R2wregVfz1akSq9ToTjtvjEAmK1b
    t+wzDPrO0dFpNpstFOmkq6O45Bq2K0zxgLi6ouzq6tHmPmz23SmKbFhsD/f7uBHq40WPVkKI
    KMty+sknn3x+z549hf379388kciue/XVMygU5GLy1dHRjJaWGoyOjqKzsxOBQACapiOVKoBl
    Gfj9HsTj81haylpCRWwxEWRZBuGwD1u2NGB4+KJ85cqVSjHYLZR6Q2NwJbaVnfzeeuuttd3d
    XbexLNve09OJtrYW1xwuzePSdKYu4adK6BupKBzmZMxYQmKkvb090NLSEpqdnXXHXp8j9lJ3
    MesG6OsUE7mhoaHExo0bfxwOR4xMRjzM83xNfX0VOI6DJCnIZApYXEwhlxORzyuQJNUhwmSe
    Z2prw2htrYYkaWBZBvPzGWSzimu+cfB4OPj9PPx+HoEAj0CAQyRidkwoCkU2q9KlpVEjn4+L
    U1NTWUfyW8B1sD/6FZkwGgAlFosll5aWXmlvb+/L5dJLw8MzHYFAaENzcz2i0SBUVUcqlcPC
    wjzS6TxyOQmKopUJWTn3Lp7nLbo5RTl1ny3S9lkL9PD7BQSDHoTDHoTDAurrfZBlA4mEisHB
    ET0ev6y0tDRoY2PjcU3TyuIvgITVfvSvcl7g1ijIEACbBEH44l133XXwD/7g98meXbvAsgS6
    rmHD+nX4yIc+AFVRirYepaSr1CfoFM6xf28YBmos5MQ+QPt9Xtx1ywFs6Osi/+dHP6/9xnef
    vDuVyUqU0pcAsK6eBsNxXb+VusqmcJWqNLH+TwDAQ1s2ru97z313wyvwJkWGVlJpNop2M4ah
    gxSTV3fvrjWeTr/gSsmv21PTeg1CCHIFEaYbSFlvteEqMNwICpbTOzlHCOEppa8DOKCqSltV
    NIJw0LSqmZqewU9/8QwOHDiIRCKOqclxHD6wB8Oj4+jr6cRyLI7R8ani87Msa6pfUxbU4MEQ
    Ap7noKoadW1yTusFt+3CDaU/Oy1RiGneWKQ2UUp9APoBPFZVFX3fve96Z8sD992HqmgEgsBD
    4O0KNLHF70tTdkXhxV7LpX5hahVuDIvxYSa/OgyrZ1hVFSiygvnFZXz7iV94X379VIeiqLo1
    fikAYZSLmLBYKea0woJljYOyHUSmAXzn3PkLm5eWlpm777wNH/nAo1g/0Iu6mmqEggGwDDEF
    64zS2KTSGfz4Z0/jzTODOHHqHA7s3o7FWAI/+eULyOXzOLx3B7Zs6IPAs6BeA+GgD80Ntdi+
    sR+GoSOXKyCRSmNiehZnLgzjlRNncWlkimq6bvuGixUum4ql36B16lyrqmVDswDg/wVQNXjx
    0r6//C//42BXZzvuu+cuHNy3Gx3trYiGQzDdXig01RQbJoDJhjEM6AAymQy+/5Nf4OVjb6K9
    pQkfePg+dLU3F+OMrfzJMgR+nwfRcBB9na24/dBuZDI5zC0u4c0zF/Gz517Fz59/jRZEyVZ1
    T7uTX+t+CpVVJ6/rYdCdSL73ve+N9vT0fEaW5c+dPz/YePLkKcTjSbAsh2g0ip6eTnCcEwGG
    6xADrOzjIhYtTUYiIUHTzEPk0lIOuZxGWlujVa2tHbfeeWekjhDy/ZGRkcuUUpaUvDPgWp83
    XAfhGkkxKKV4+OGHqrxe3yPz88v1Cwuxa6K9dmLr83lBKaAoNqh8tcT2V6dFcxwDhiE0ny/o
    hsmbdcbhGxKLUS50JxJCOE3TuOPHj5/p6+s7w3HefkVRGUVRwTAsvF4vurubsbS0AF3X0dLS
    hvn5BKqrQ8jlJDAMg5qakEP12vxMTBqqAV03FWU1TaOTk5NyoVAQHUwsZwyuyMC6UYmvI+4y
    AMj27dsDe/fu3VRdXf1BWVbec/bsYN3Y2CRUVStTfF5pt8MUxbDs3uDyghVb1jfs/FlJLMtM
    8DiORSDgQUdHXeBd73pX15kzp7XLl6/k0+l0ilIacYyhZhW2nG00hgNUWnPPc5fOkKHruvKd
    73zn5XvvvTe4fv2GI6lUDmNjc1heTiOXE4vopDsRYxgWVVV+tLfXoro6AIYhmJtLo6kpjP7+
    OsRiBSwuFqAo1CqiGtA0szeVEA0sK0MQTHQ4FOIRjfKoq/OQRELXX3vttbQsy7mrxODrrgWz
    SgFfoZRKL7zwwqs1NTXT8Xi8+q673vHx2toGLC7Gcfr0FcRiGYiiAl03igmtLaZmzqfSHhaN
    BtHb2wi/34O5uTRmZ1PQdcAZBkwqNKDrOmRZRiqlgmVFeDwcQiEBdXU+tLX5kUoZ+smTo/mR
    kctZSmnCwcCyC9AJKzbn8RZsj9Y8AbYCsRfAe7Zt27b3i1/8Itm5Ywd0XbXsZIC+3l709Xab
    tGeb/lw8GOrW4wzrsOh8TOmxuq4V7WnsxK+tuQGffP+DmF9cDn/vp8/uBTBhvn9qEEKKao6O
    CzdbAHbRYiildL3X47nr3nfcKnS2NZclv7Y/r9vWSBRFPPPiMXS2t2DTup4yL1+3CFZJAZpe
    oye4hAITAmSyOWRyBQPlwl7GDQ6+zoREQqmnKwFgOZVKG7IsMzQUhK7reOa5lxBLpnH48GF8
    85vfRF93B+689TC++o1v49XXT8IwDGxe349gwG+i3bYnqJ3sESCXL0BRFFtt1xl47cTXWfUr
    bng3gnblTH5Rkg9lKaUBAD0A7uQ47sFNGzdu/dCHPuB57JGH0VBfV5awmYmso13Bts2ielmS
    WxK9Mhw/sy22nC0P1PG85uO2GToGejvIt773ZOCnT784ML8Ui1KKKmtfcYrYeCwLExnlPTU6
    XP2Ga3jYo9aBQKaUpiRZnm5ubjS+8JufYepqqs3+U0M30UdbGMwaE0KAiclp/NNX/wW6YWDX
    1o147N3vxOJyDP/yvZ/i//f172JkfAp/8QefQzjoL3kA20VBgyLg9yLo96CtqQ4Hdm4CxzI4
    e2nUcMzHgusSb5ZiDMoFsfKEkGlKaS6TzfWCEOX9jzzoe/+j7ykTX8vnCzh99jzOX7gEj4fH
    vl070NPZBgLzcPPGyXP4X//nCWzdvgOvvHkePMfidz77UdM+igLzS8t47cRpxBNJdLe3YueW
    dQgF/QClCIcCiIQ60dfZily+gJPnhgxV0+w+ZWffUcKqPDvR35sCWaKUoqamhu3s7LxDluXP
    HDt2rPHkyVOQZQ0sa4b7qalZJBJpMAxXVIl1KsaaP3d+X/rKMBwoJVAUw1KXtcWxdIyPp5HL
    ady6dVWbdu3alZiens7KskzsWOZA/e3e+Zv25ijko729Y6um6QenpxeJLKtlSelq6K/HI2Dd
    ug5kswWMjy9gdWozrqECvfprcBwLlgWy2axT20RfJQ7fCCaWYq0d1jAMj6ZpS14vpwoC7xFF
    BYQwaGqqRSTiw2uvnUR3dzdUlWB0dB4cx6K+PgIAyGYlSJIGjispH5sHbpPWy/MMKDWo1Wvp
    jsHO5HdF0nE9C1QuYIMQQridO3eGtm/fPlBTU3MPIeSBxcWldWfPnuevXBmFLKsOS55yRedK
    PzeTO9aR7LIVHseUPdadFCeTMnI5jXR3V4UOHrxl0/r1G6rGx8cjk5OTwsLCgqYoikbNmxcr
    /Wx1xx54vZTfi2rQAMR8Pj9LCElNTMxXnz8/WjYO5cWBUvGgtbUaAwONSKVETE4msbCQBaUE
    zc0hNDSEoOtLmJ/Pu9Y7U2RvqKrZ15rPq0inKfx+QvP5rKppmuRiXxVcCfCN9oinsNwsCCHy
    0tJScmlpSQ8Gg0og4E8VCgV68uQlYiuL22MZCPjQ0FAFv99rtSdkoKqGtSdx6OtrxJEj21FX
    V4uTJ88hl1OQSIhFplFVlR9VVT4YBkU6rSKX0wCYdlOybECWFYgiRSTCQ5JymqZpEqU0X6EI
    nUS58NWvPKbc270JBgIB5PP5KkLIjkOHDvFbt2yxPGrLe0mdSa/hSoCdya5RIUF29hdSBy2E
    UgORcBAHdm7BT59+uU6SlU6rAqhSShXrsGxfDK6/AMyvFEwopR4AR3q7O7oP798FhmGKVFG4
    kt+SXyZwYWgEX/rG9/CpDz6ETQM9K/t/K/x/vFUFaIvOGk+mkUxnDUfC4Q7A+lqh7BUoRBVp
    MY4k2AuAicUTzBM//Tk0TcPScgzPPP8SDh8+gng8jtdeew211VHE4glkc3lomjnPhscm8af/
    +f+BwPPgeR4+r4CAz1TxjYSCuDI6AVlR7YSjLOgSQgpY2Rf4Vt4D1ipAW/OKsehMrQDWAzgq
    CPzhvr7+dfe+653+hx96D9m8aaPZm2H1mhcZA1S3lJ1p2VWah8YqP4eL7WHSoVc81nru7vYW
    /PanP4hDe7YJP33mpdY3Tl+on5lf3Khp+iSAIesapZROO7wdnb0110tgwpnEZQBciMXi8tzc
    gq+mKlos5oFWUGo3KBrr67B54wDOXRhCLJFELJHE/OIy8qKImuoo1vd1wuvhi8Un93o3EXVz
    ykuSjNnFGHTd0FDed1QMvpZdzQ1TIq+EAjt6gUWLsTFWKIjT0zNzWLW67AAALs5JREFUEcPW
    fwCFrqn40U9/jq9+8ztoa+/EpUuX8MabZ/Bnf/RbqIqGAVBk8znoBsWGDRswMzODZDptFg1Z
    Boqi4P/88Gf43pPPYfPmLfjRU6/gnqO78bH33g+PwBdfR5JkjE7OQNWKPdR5lKs+J1zB1913
    dEP7MW+//fYQz/P3jI6ONp8+fRqSJINl+WLSpmka0ulMMQF2fi0lwJW+2vftxJmUIZsAEI/L
    WFqSubq6uoHa2tqu2dlZFaaKsmZ9zvZhmcHNr4RP9+/f7/F6vYfT6XzD8nKyohCVO6mlFKiq
    CqOmJoLl5TQqWRqtrux8Ndp0eeLs8XBgGGokEgnFUViodK3JfLxGDLNRYFvkM+v3+6nXK7Ad
    Hc0wDAq/34uWlnosLS1A0zT09PSAEB4DAy3wegXrNYBAwIOtW9vK/G81zbZSofD7BXAcsYvQ
    BVcMdseFsjOJqyi8ZrHX8Vzk/vvvD7a2trb7fL6NHo/nVkrpoVgs1n358rDv8uVhJJNpC2Vj
    sZoqu/23l6s6E4cyuBMhdYtgwcH6cD7W/JrJyLh4MUbq6gKepqb6rp07G5u2bdu2JZfLjieT
    yaFkMnl5dnZudHR0dM4wjEpjrDrmwPVkHCjDw1cmd+zYMRmJBLsFgYeuU8d6ZVYUrwAgmSwg
    n5fBcQy8Xrs/lQOlQC6nQBS1VYtQ5cwMBoLAgBCNzs8vyLquS6skvxJuHiE2aol52n37XD6f
    TyQSiQuNjS23eTyeACFSUek+EPBi584+7N69BeFwCCM