# -*- coding: utf-8 -*-
"""
This module contains representation classes for integrals.
"""

# Copyright (C) 2008-2009 Martin Sandve Alnes and Simula Resarch Laboratory
#
# This file is part of SyFi.
#
# SyFi is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# SyFi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SyFi. If not, see <http://www.gnu.org/licenses/>.
#
# First added:  2008-08-13
# Last changed: 2009-03-16

import swiginac
from sfc.common import sfc_assert, sfc_debug
from sfc.representation.integralrepresentation import IntegralRepresentation

class ExteriorFacetIntegralRepresentation(IntegralRepresentation):
    def __init__(self, integrals, formrep):
        IntegralRepresentation.__init__(self, integrals, formrep, True)
    
    def iter_facet_tokens(self, facet):
        fr = self.formrep
        nsd = fr.cell.nsd
        
        # Facet mapping FIXME (needed anywere? quadrature on facets?)
        #for (s,e) in zip(fr.facet_G_sym, fr.facet_G_expr[facet]):
        #    yield (s,e)
        
        s = fr.facet_D_sym
        e = fr.facet_D_expr[facet]
        yield (s, e)
        
        # Facet normal
        for i in range(nsd):
            yield (fr.n_sym[i], fr.n_expr[facet][i])
        
        # Scaling factor
        if self.symbolic_integral is not None:
            yield (fr.D_sym, fr.facet_D_sym)
    
    def compute_A(self, data, iota, facet=None):
        "Compute expression for A[iota]. Overload in subclasses!"
        
        integrand = data.integral.integrand()
        
        if data.integration_method == "quadrature":
            sfc_assert(facet is None, "Not expecting facet number.")
            
            if self.options.safemode:
                data.evaluator.update(iota)
                integrand = data.evaluator.visit(integrand)
            else:
                n = len(data.G.V())
                try:
                    integrand = data.vertex_data_set[iota][n-1]
                except:
                    print data.vertex_data_set
                    raise RuntimeError
            
            D = self.formrep.D_sym
            A = integrand * D
            
            if self.formrep.options.output.enable_debug_prints:
                sfc_debug("In compute_A(", iota, "):")
                sfc_debug("  data.integral.integrand() = ", data.integral.integrand())
                sfc_debug("  integrand = ", integrand)
                sfc_debug("  A = ", A)
        
        elif data.integration_method == "symbolic":
            sfc_assert(isinstance(facet, int), "Expecting facet number.")
            
            data.evaluator.update(iota)
            integrand = data.evaluator.visit(integrand)
            
            D = self.formrep.D_sym
            polygon = self.formrep.cell.facet_polygons[facet]
            if isinstance(polygon, swiginac.matrix):
                # 1D
                repmap = swiginac.exmap()
                repmap[self.formrep.xi_sym[0]] = polygon[0]
                A = integrand.subs(repmap) * D
            else:
                A = polygon.integrate(integrand) * D
            
            if self.formrep.options.output.enable_debug_prints:
                sfc_debug("In compute_A(", iota, "):")
                sfc_debug("  data.integral.integrand() = ", data.integral.integrand())
                sfc_debug("  integrand = ", integrand)
                sfc_debug("  A = ", A)
        
        return A
    
