#!/bin/sh

# Check compiler options for non-configure builds and create Makefile fragment
#
#    Copyright (C) 2022 Genome Research Ltd.
#
#    Author: Rob Davies <rmd@sanger.ac.uk>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.

# Arguments are:
# 1. C compiler command
# 2. Initial CFLAGS
# 3. LDFLAGS

CC=$1
CFLAGS=$2
LDFLAGS=$3

# Try running the compiler.  Uses the same contest.* names as
# configure for temporary files.
run_compiler ()
{
    "$CC" $CFLAGS $1 $LDFLAGS -o conftest conftest.c 2> conftest.err
    retval=$?
    rm -f conftest.err conftest
    return $retval
}

echo "# Compiler probe results, generated by $0"

# Check for sse4.1 etc. support

rm -f conftest conftest.err conftest.c
cat - <<'EOF' > conftest.c
#include "x86intrin.h"
int main(int argc, char **argv) {
    unsigned int i = _mm_popcnt_u32(1);
    __m128i a = _mm_set_epi32(1, 2, 3, i), b = _mm_set_epi32(4, 3, 2, 1);
    __m128i c = _mm_max_epu32(a, b);
    b = _mm_shuffle_epi8(a, c);
    return *((char *) &b);
}
EOF
FLAGS="-mpopcnt -msse4.1 -mssse3"
if run_compiler "$FLAGS" ; then
    echo "HTS_CFLAGS_SSE4 = $FLAGS"
fi

# Check for avx2

rm -f conftest.c
cat - <<'EOF' > conftest.c
#include "x86intrin.h"
int main(int argc, char **argv) {
    __m256i a = _mm256_set_epi32(1, 2, 3, 4, 5, 6, 7, 8);
    __m256i b = _mm256_add_epi32(a, a);
    long long c = _mm256_extract_epi64(b, 0);
    return (int) c;
}
EOF
FLAGS="-mavx2"
if run_compiler "$FLAGS" ; then
    echo "HTS_CFLAGS_AVX2 = $FLAGS"
fi

# Check for avx512

rm -f conftest.c
cat - <<'EOF' > conftest.c
#include "x86intrin.h"
int main(int argc, char **argv) {
    __m512i a = _mm512_set1_epi32(1);
    __m512i b = _mm512_add_epi32(a, a);
    return *((char *) &b);
}
EOF
FLAGS="-mavx512f"
if run_compiler "$FLAGS" ; then
    echo "HTS_CFLAGS_AVX512 = $FLAGS"
fi

# Check for neon

rm -f conftest.c
cat - <<'EOF' > conftest.c
#include "arm_neon.h"
int main(int argc, char **argv) {
    int32x4_t a = vdupq_n_s32(1);
    int32x4_t b = vaddq_s32(a, a);
    return *((char *) &b);
}
EOF
if run_compiler "" ; then
    echo "HTS_HAVE_NEON = yes"
fi

rm -f conftest.c
