import unittest
import yaramod


class BuilderTests(unittest.TestCase):
    def setUp(self):
        self.new_file = yaramod.YaraFileBuilder(yaramod.Features.AllCurrent)
        self.new_rule = yaramod.YaraRuleBuilder()

    def test_empty_file(self):
        yara_file = self.new_file.get()

        self.assertEqual(yara_file.text, '')

    def test_pure_imports(self):
        yara_file = self.new_file \
            .with_module('math') \
            .with_module('pe') \
            .with_module('elf') \
            .get()

        self.assertEqual(yara_file.text_formatted, '''import "elf"
import "math"
import "pe"

''')
        self.assertEqual(yara_file.text, '''import "elf"
import "math"
import "pe"
''')

    def test_empty_rule(self):
        rule = self.new_rule \
            .with_name('empty_rule') \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule empty_rule
{
	condition:
		true
}
''')
        self.assertEqual(yara_file.text, '''rule empty_rule {
	condition:
		true
}''')

    def test_rule_with_metas(self):
        rule = self.new_rule \
            .with_name('rule_with_metas') \
            .with_string_meta('string_meta', 'string value') \
            .with_int_meta('int_meta', 42) \
            .with_hex_int_meta('hex_int_meta', 0x42) \
            .with_bool_meta('bool_meta', False) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_metas
{
	meta:
		string_meta = "string value"
		int_meta = 42
		hex_int_meta = 0x42
		bool_meta = false
	condition:
		true
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_metas {
	meta:
		string_meta = "string value"
		int_meta = 42
		hex_int_meta = 0x42
		bool_meta = false
	condition:
		true
}''')

    def test_rule_with_variables(self):
        rule = self.new_rule \
            .with_name('rule_with_variables') \
            .with_string_variable('string_var', 'string value') \
            .with_int_variable('int_var', 42) \
            .with_double_variable('double_var', 2.6) \
            .with_bool_variable('bool_var', False) \
            .with_struct_variable('struct_var', 'time') \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_variables
{
	variables:
		string_var = "string value"
		int_var = 42
		double_var = 2.6
		bool_var = false
		struct_var = time
	condition:
		true
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_variables {
	variables:
		string_var = "string value"
		int_var = 42
		double_var = 2.6
		bool_var = false
		struct_var = time
	condition:
		true
}''')

    def test_rule_with_tags(self):
        rule = self.new_rule \
            .with_name('rule_with_tags') \
            .with_tag('Tag1') \
            .with_tag('Tag2') \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_tags : Tag1 Tag2
{
	condition:
		true
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_tags : Tag1 Tag2 {
	condition:
		true
}''')

    def test_rule_with_modifiers(self):
        rule = self.new_rule \
            .with_name('private_rule') \
            .with_modifier(yaramod.RuleModifier.Private) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''private rule private_rule
{
	condition:
		true
}
''')
        self.assertEqual(yara_file.text, '''private rule private_rule {
	condition:
		true
}''')

    def test_rule_with_comments(self):
        rule = self.new_rule \
            .with_name('private_rule') \
            .with_modifier(yaramod.RuleModifier.Private) \
            .with_comment("multiline comment") \
            .with_comment("oneline comment", 0) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''/* multiline comment */
// oneline comment
private rule private_rule
{
	condition:
		true
}
''')
        self.assertEqual(yara_file.text, '''private rule private_rule {
	condition:
		true
}''')

    def test_rule_with_plain_string(self):
        rule = self.new_rule \
            .with_name('rule_with_plain_string') \
            .with_plain_string('$1', 'This is plain string.').ascii().wide() \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_plain_string
{
	strings:
		$1 = "This is plain string." ascii wide
	condition:
		true
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_plain_string {
	strings:
		$1 = "This is plain string." ascii wide
	condition:
		true
}''')

    def test_rule_with_hex_string(self):
        rule = self.new_rule \
            .with_name('rule_with_hex_string') \
            .with_hex_string('$1', yaramod.YaraHexStringBuilder([0x10, 0x11]).get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_hex_string
{
	strings:
		$1 = { 10 11 }
	condition:
		true
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_hex_string {
	strings:
		$1 = { 10 11 }
	condition:
		true
}''')

    def test_rule_with_regexp(self):
        rule = self.new_rule \
            .with_name('rule_with_regexp') \
            .with_regexp('$1', '[a-z0-9]{32}', 'i') \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_regexp
{
	strings:
		$1 = /[a-z0-9]{32}/i
	condition:
		true
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_regexp {
	strings:
		$1 = /[a-z0-9]{32}/i
	condition:
		true
}''')

    def test_rule_with_double_values(self):
        cond = yaramod.double_val(3.14159) > yaramod.double_val(2.71828)
        rule = self.new_rule \
            .with_name('rule_with_double_values') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_double_values
{
	condition:
		3.14159 > 2.71828
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_double_values {
	condition:
		3.14159 > 2.71828
}''')

    def test_multiple_rules(self):
        rule1 = self.new_rule \
            .with_name('rule_1') \
            .with_tag('Tag1') \
            .with_plain_string('$1', 'This is plain string 1.') \
            .get()
        rule2 = self.new_rule \
            .with_name('rule_2') \
            .with_tag('Tag2') \
            .with_plain_string('$2', 'This is plain string 2.') \
            .get()
        yara_file = self.new_file \
            .with_rule(rule1) \
            .with_rule(rule2) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_1 : Tag1
{
	strings:
		$1 = "This is plain string 1."
	condition:
		true
}

rule rule_2 : Tag2
{
	strings:
		$2 = "This is plain string 2."
	condition:
		true
}
''')
        self.assertEqual(yara_file.text, '''rule rule_1 : Tag1 {
	strings:
		$1 = "This is plain string 1."
	condition:
		true
}

rule rule_2 : Tag2 {
	strings:
		$2 = "This is plain string 2."
	condition:
		true
}''')

    def test_rule_with_variable_id_condition(self):
        cond = yaramod.id('time_struct').access('now')()
        rule = self.new_rule \
            .with_name('rule_with_variable_id_condition') \
            .with_struct_variable('time_struct', 'time') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_module('time') \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''import "time"

rule rule_with_variable_id_condition
{
	variables:
		time_struct = time
	condition:
		time_struct.now()
}
''')
        self.assertEqual(yara_file.text, '''import "time"

rule rule_with_variable_id_condition {
	variables:
		time_struct = time
	condition:
		time_struct.now()
}''')

    def test_rule_with_string_id_condition(self):
        cond = yaramod.string_ref('$1')
        rule = self.new_rule \
            .with_name('rule_with_string_id_condition') \
            .with_plain_string('$1', 'This is plain string 1.') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_string_id_condition
{
	strings:
		$1 = "This is plain string 1."
	condition:
		$1
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_string_id_condition {
	strings:
		$1 = "This is plain string 1."
	condition:
		$1
}''')

    def test_rule_with_string_at_condition(self):
        cond = yaramod.match_at('$1', yaramod.int_val(100))
        rule = self.new_rule \
            .with_name('rule_with_string_id_condition') \
            .with_plain_string('$1', 'This is plain string 1.') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_string_id_condition
{
	strings:
		$1 = "This is plain string 1."
	condition:
		$1 at 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_string_id_condition {
	strings:
		$1 = "This is plain string 1."
	condition:
		$1 at 100
}''')

    def test_rule_with_match_in_range_condition(self):
        cond = yaramod.match_in_range('$1', yaramod.range(yaramod.int_val(100), yaramod.int_val(200)))
        rule = self.new_rule \
            .with_name('rule_with_match_in_range_condition') \
            .with_condition(cond.get()) \
            .with_plain_string('$1', 'This is plain string 1.') \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_match_in_range_condition
{
	strings:
		$1 = "This is plain string 1."
	condition:
		$1 in (100 .. 200)
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_match_in_range_condition {
	strings:
		$1 = "This is plain string 1."
	condition:
		$1 in (100 .. 200)
}''')

    def test_rule_with_match_count_condition(self):
        cond = yaramod.match_count('$1')
        rule = self.new_rule \
            .with_name('rule_with_match_count_condition') \
            .with_condition(cond.get()) \
            .with_plain_string('$1', 'This is plain string 1.') \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_match_count_condition
{
	strings:
		$1 = "This is plain string 1."
	condition:
		#1
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_match_count_condition {
	strings:
		$1 = "This is plain string 1."
	condition:
		#1
}''')

    def test_rule_with_match_length_condition(self):
        cond = yaramod.match_length('$1')
        rule = self.new_rule \
            .with_name('rule_with_match_length_condition') \
            .with_plain_string('$1', 'This is plain string 1.') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_match_length_condition
{
	strings:
		$1 = "This is plain string 1."
	condition:
		!1
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_match_length_condition {
	strings:
		$1 = "This is plain string 1."
	condition:
		!1
}''')

    def test_rule_with_match_length_with_index_condition(self):
        cond = yaramod.match_length('$1', yaramod.int_val(0))
        rule = self.new_rule \
            .with_plain_string('$1', 'This is plain string 1.') \
            .with_name('rule_with_match_length_with_index_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_match_length_with_index_condition
{
	strings:
		$1 = "This is plain string 1."
	condition:
		!1[0]
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_match_length_with_index_condition {
	strings:
		$1 = "This is plain string 1."
	condition:
		!1[0]
}''')

    def test_rule_with_match_offset_condition(self):
        cond = yaramod.match_offset('$1')
        rule = self.new_rule \
            .with_name('rule_with_match_offset_condition') \
            .with_plain_string('$1', 'This is plain string 1.') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_match_offset_condition
{
	strings:
		$1 = "This is plain string 1."
	condition:
		@1
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_match_offset_condition {
	strings:
		$1 = "This is plain string 1."
	condition:
		@1
}''')

    def test_rule_with_match_offset_with_index_condition(self):
        cond = yaramod.match_offset('$1', yaramod.int_val(0))
        rule = self.new_rule \
            .with_name('rule_with_match_offset_with_index_condition') \
            .with_plain_string('$1', 'This is plain string 1.') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_match_offset_with_index_condition
{
	strings:
		$1 = "This is plain string 1."
	condition:
		@1[0]
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_match_offset_with_index_condition {
	strings:
		$1 = "This is plain string 1."
	condition:
		@1[0]
}''')

    def test_rule_with_of_in_range_condition(self):
        cond = yaramod.of_in_range(yaramod.all(), yaramod.them(), yaramod.range(yaramod.filesize() - yaramod.int_val(1024), yaramod.filesize()))
        rule = self.new_rule \
            .with_name('rule_with_of_in_range_condition') \
            .with_plain_string('$a1', 'This is plain string 1.') \
            .with_plain_string('$a2', 'This is plain string 2.') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_of_in_range_condition
{
	strings:
		$a1 = "This is plain string 1."
		$a2 = "This is plain string 2."
	condition:
		all of them in (filesize - 1024 .. filesize)
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_of_in_range_condition {
	strings:
		$a1 = "This is plain string 1."
		$a2 = "This is plain string 2."
	condition:
		all of them in (filesize - 1024 .. filesize)
}''')
                         
    def test_rule_with_of_at_offset_condition(self):
        cond = yaramod.of_at(yaramod.any(), yaramod.them(), yaramod.int_val(10))
        rule = self.new_rule \
            .with_name('rule_with_of_in_range_condition') \
            .with_plain_string('$a1', 'This is plain string 1.') \
            .with_plain_string('$a2', 'This is plain string 2.') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_of_in_range_condition
{
	strings:
		$a1 = "This is plain string 1."
		$a2 = "This is plain string 2."
	condition:
		any of them at 10
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_of_in_range_condition {
	strings:
		$a1 = "This is plain string 1."
		$a2 = "This is plain string 2."
	condition:
		any of them at 10
}''')

    def test_rule_with_lt_condition(self):
        cond = yaramod.filesize() < yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_gt_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_gt_condition
{
	condition:
		filesize < 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_gt_condition {
	condition:
		filesize < 100
}''')

    def test_rule_with_le_condition(self):
        cond = yaramod.filesize() <= yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_ge_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_ge_condition
{
	condition:
		filesize <= 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_ge_condition {
	condition:
		filesize <= 100
}''')

    def test_rule_with_gt_condition(self):
        cond = yaramod.filesize() > yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_gt_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_gt_condition
{
	condition:
		filesize > 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_gt_condition {
	condition:
		filesize > 100
}''')

    def test_rule_with_ge_condition(self):
        cond = yaramod.filesize() >= yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_ge_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_ge_condition
{
	condition:
		filesize >= 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_ge_condition {
	condition:
		filesize >= 100
}''')

    def test_rule_with_eq_condition(self):
        cond = yaramod.filesize() == yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_eq_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_eq_condition
{
	condition:
		filesize == 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_eq_condition {
	condition:
		filesize == 100
}''')

    def test_rule_with_neq_condition(self):
        cond = yaramod.filesize() != yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_neq_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_neq_condition
{
	condition:
		filesize != 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_neq_condition {
	condition:
		filesize != 100
}''')

    def test_rule_with_plus_condition(self):
        cond = yaramod.filesize() + yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_plus_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_plus_condition
{
	condition:
		filesize + 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_plus_condition {
	condition:
		filesize + 100
}''')

    def test_rule_with_minus_condition(self):
        cond = yaramod.filesize() - yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_minus_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_minus_condition
{
	condition:
		filesize - 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_minus_condition {
	condition:
		filesize - 100
}''')

    def test_rule_with_multiply_condition(self):
        cond = yaramod.filesize() * yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_multiply_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_multiply_condition
{
	condition:
		filesize * 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_multiply_condition {
	condition:
		filesize * 100
}''')

    def test_rule_with_divide_condition(self):
        cond = yaramod.filesize() / yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_divide_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, r'''rule rule_with_divide_condition
{
	condition:
		filesize \ 100
}
''')
        self.assertEqual(yara_file.text, r'''rule rule_with_divide_condition {
	condition:
		filesize \ 100
}''')

    def test_rule_with_modulo_condition(self):
        cond = yaramod.filesize() % yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_modulo_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_modulo_condition
{
	condition:
		filesize % 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_modulo_condition {
	condition:
		filesize % 100
}''')

    def test_rule_with_xor_condition(self):
        cond = yaramod.filesize() ^ yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_xor_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_xor_condition
{
	condition:
		filesize ^ 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_xor_condition {
	condition:
		filesize ^ 100
}''')

    def test_rule_with_bitwise_and_condition(self):
        cond = yaramod.filesize() & yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_bitwise_and_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_bitwise_and_condition
{
	condition:
		filesize & 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_bitwise_and_condition {
	condition:
		filesize & 100
}''')

    def test_rule_with_bitwise_or_condition(self):
        cond = yaramod.filesize() | yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_bitwise_or_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_bitwise_or_condition
{
	condition:
		filesize | 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_bitwise_or_condition {
	condition:
		filesize | 100
}''')

    def test_rule_with_and_condition(self):
        cond = yaramod.conjunction([yaramod.filesize() > yaramod.int_val(100), yaramod.filesize() < yaramod.int_val(200)])
        rule = self.new_rule \
            .with_name('rule_with_and_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_and_condition
{
	condition:
		filesize > 100 and
		filesize < 200
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_and_condition {
	condition:
		filesize > 100 and filesize < 200
}''')

    def test_rule_with_or_condition(self):
        cond = yaramod.disjunction([yaramod.filesize() > yaramod.int_val(100), yaramod.filesize() < yaramod.int_val(200)])
        rule = self.new_rule \
            .with_name('rule_with_or_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_or_condition
{
	condition:
		filesize > 100 or
		filesize < 200
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_or_condition {
	condition:
		filesize > 100 or filesize < 200
}''')

    def test_rule_with_and_condition_with_comments(self):
        cond = yaramod.conjunction([[yaramod.filesize() > yaramod.int_val(100), 'comment1'], [yaramod.filesize() < yaramod.int_val(200), 'comment2']])
        rule = self.new_rule \
            .with_name('rule_with_and_condition_with_comments') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_and_condition_with_comments
{
	condition:
		filesize > 100 and // comment1
		filesize < 200     // comment2
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_and_condition_with_comments {
	condition:
		filesize > 100 and
		filesize < 200
}''')

    def test_rule_with_and_condition_with_comments_behind(self):
        cond = yaramod.conjunction([[yaramod.filesize() > yaramod.int_val(100), 'comment1'], [yaramod.filesize() < yaramod.int_val(200), 'comment2']])
        rule = self.new_rule \
            .with_name('rule_with_and_condition_with_comments') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_and_condition_with_comments
{
	condition:
		filesize > 100 and // comment1
		filesize < 200     // comment2
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_and_condition_with_comments {
	condition:
		filesize > 100 and
		filesize < 200
}''')

    def test_rule_with_or_condition_with_single_comment(self):
        cond = yaramod.disjunction([[yaramod.filesize() > yaramod.int_val(100), 'skip small files'], [yaramod.filesize() < yaramod.int_val(200), '']])
        rule = self.new_rule \
            .with_name('rule_with_or_condition_with_single_comment') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_or_condition_with_single_comment
{
	condition:
		filesize > 100 or // skip small files
		filesize < 200
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_or_condition_with_single_comment {
	condition:
		filesize > 100 or
		filesize < 200
}''')

    def test_rule_with_or_condition_with_comments(self):
        cond = yaramod.disjunction([[yaramod.filesize() > yaramod.int_val(100), 'skip small files'], [yaramod.filesize() < yaramod.int_val(200), 'also too big files']])
        rule = self.new_rule \
            .with_name('rule_with_or_condition_with_comments') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_or_condition_with_comments
{
	condition:
		filesize > 100 or // skip small files
		filesize < 200    // also too big files
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_or_condition_with_comments {
	condition:
		filesize > 100 or
		filesize < 200
}''')

    def test_rule_with_unary_minus_condition(self):
        cond = -yaramod.int_val(10)
        rule = self.new_rule \
            .with_name('rule_with_unary_minus_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_unary_minus_condition
{
	condition:
		-10
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_unary_minus_condition {
	condition:
		-10
}''')

    def test_rule_with_not_condition(self):
        cond = yaramod.not_(yaramod.filesize() < yaramod.int_val(100))
        rule = self.new_rule \
            .with_name('rule_with_not_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_not_condition
{
	condition:
		not filesize < 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_not_condition {
	condition:
		not filesize < 100
}''')

    def test_rule_with_bitwise_not_condition(self):
        cond = ~yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_bitwise_not_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_bitwise_not_condition
{
	condition:
		~100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_bitwise_not_condition {
	condition:
		~100
}''')

    def test_rule_with_shift_left_condition(self):
        cond = yaramod.filesize() << yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_shift_left_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_shift_left_condition
{
	condition:
		filesize << 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_shift_left_condition {
	condition:
		filesize << 100
}''')

    def test_rule_with_shift_right_condition(self):
        cond = yaramod.filesize() >> yaramod.int_val(100)
        rule = self.new_rule \
            .with_name('rule_with_shift_right_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_shift_right_condition
{
	condition:
		filesize >> 100
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_shift_right_condition {
	condition:
		filesize >> 100
}''')

    def test_rule_with_function_call_condition(self):
        cond = yaramod.id('pe').access('is_dll')().comment(message="SOME COMMENT", multiline=True)
        rule = self.new_rule \
            .with_name('rule_with_function_call_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_module('pe') \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''import "pe"

rule rule_with_function_call_condition
{
	condition:
		/* SOME COMMENT */
		pe.is_dll()
}
''')
        self.assertEqual(yara_file.text, '''import "pe"

rule rule_with_function_call_condition {
	condition:
		pe.is_dll()
}''')

    def test_rule_with_function_call_and_oneline_comment(self):
        cond = yaramod.id('pe').access('is_dll')().comment_behind(message="Generated", multiline=False, linebreak=False)
        rule = self.new_rule \
            .with_name('rule_with_function_call_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_module('pe') \
            .with_rule(rule) \
            .get()

        print(yara_file.text_formatted)

        self.assertEqual(yara_file.text_formatted, '''import "pe"

rule rule_with_function_call_condition
{
	condition:
		pe.is_dll() // Generated
}
''')
        self.assertEqual(yara_file.text, '''import "pe"

rule rule_with_function_call_condition {
	condition:
		pe.is_dll()
}''')

    def test_rule_with_structure_access_condition(self):
        cond = yaramod.id('pe').access('linker_version').access('major')
        rule = self.new_rule \
            .with_name('rule_with_structure_access_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_module('pe') \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''import "pe"

rule rule_with_structure_access_condition
{
	condition:
		pe.linker_version.major
}
''')
        self.assertEqual(yara_file.text, '''import "pe"

rule rule_with_structure_access_condition {
	condition:
		pe.linker_version.major
}''')

    def test_rule_with_array_access_condition(self):
        cond = yaramod.id('pe').access('sections')[yaramod.int_val(0)].access('name').comment(message="SOME COMMENT")
        rule = self.new_rule \
            .with_name('rule_with_array_access_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_module('pe') \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''import "pe"

rule rule_with_array_access_condition
{
	condition:
		// SOME COMMENT
		pe.sections[0].name
}
''')
        self.assertEqual(yara_file.text, '''import "pe"

rule rule_with_array_access_condition {
	condition:
		pe.sections[0].name
}''')

    def test_rule_with_dictionary_access_condition(self):
        cond = yaramod.id('pe').access('version_info')[yaramod.string_val('CompanyName')]
        rule = self.new_rule \
            .with_name('rule_with_dictionary_access_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_module('pe') \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''import "pe"

rule rule_with_dictionary_access_condition
{
	condition:
		pe.version_info["CompanyName"]
}
''')
        self.assertEqual(yara_file.text, '''import "pe"

rule rule_with_dictionary_access_condition {
	condition:
		pe.version_info["CompanyName"]
}''')

    def test_rule_with_for_loop_over_dictionary(self):
        cond = yaramod.for_loop(
                yaramod.any(),
                'k',
                'v',
                yaramod.id('pe').access('version_info'),
                yaramod.conjunction([
                    yaramod.id('k') == yaramod.string_val('CompanyName'),
                    yaramod.id('v').contains(yaramod.string_val('Microsoft'))
                ])
            )
        rule = self.new_rule \
            .with_name('rule_with_for_loop_over_dictionary') \
            .with_plain_string('$1', 'This is plain string.') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()


        self.assertEqual(yara_file.text_formatted, '''rule rule_with_for_loop_over_dictionary
{
	strings:
		$1 = "This is plain string."
	condition:
		for any k, v in pe.version_info : (
			k == "CompanyName" and
			v contains "Microsoft"
		)
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_for_loop_over_dictionary {
	strings:
		$1 = "This is plain string."
	condition:
		for any k, v in pe.version_info : ( k == "CompanyName" and v contains "Microsoft" )
}''')

    def test_rule_with_percentage_of_stringset(self):
        cond = yaramod.of(
                yaramod.int_val(50).percent(),
                yaramod.them()
            )
        rule = self.new_rule \
            .with_name('rule_with_percentage') \
            .with_plain_string('$1', 'This is plain string.') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()


        self.assertEqual(yara_file.text_formatted, '''rule rule_with_percentage
{
	strings:
		$1 = "This is plain string."
	condition:
		50% of them
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_percentage {
	strings:
		$1 = "This is plain string."
	condition:
		50% of them
}''')

    def test_rule_with_complex_condition(self):
        cond = yaramod.for_loop(
                yaramod.any(),
                'i',
                yaramod.set([
                    yaramod.int_val(1),
                    yaramod.int_val(2),
                    yaramod.int_val(3)
                ]),
                yaramod.match_at(
                    '$1',
                    yaramod.paren(
                        yaramod.entrypoint() + yaramod.id('i')
                    )
                )
            )
        rule = self.new_rule \
            .with_name('rule_with_complex_condition') \
            .with_plain_string('$1', 'This is plain string.') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()


        self.assertEqual(yara_file.text_formatted, '''rule rule_with_complex_condition
{
	strings:
		$1 = "This is plain string."
	condition:
		for any i in (1, 2, 3) : ( $1 at (entrypoint + i) )
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_complex_condition {
	strings:
		$1 = "This is plain string."
	condition:
		for any i in (1, 2, 3) : ( $1 at (entrypoint + i) )
}''')
                         
    def test_rule_with_string_literal_set(self):
        cond = yaramod.for_loop(
                yaramod.any(),
                's',
                yaramod.set([
                    yaramod.string_val('hash1'),
                    yaramod.string_val('hash2'),
                    yaramod.string_val('hash3')
                ]),
                yaramod.id('s') == yaramod.string_val('abc123')
            )
        rule = self.new_rule \
            .with_name('rule_with_string_literal_set') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()


        self.assertEqual(yara_file.text_formatted, '''rule rule_with_string_literal_set
{
	condition:
		for any s in ("hash1", "hash2", "hash3") : ( s == "abc123" )
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_string_literal_set {
	condition:
		for any s in ("hash1", "hash2", "hash3") : ( s == "abc123" )
}''')

    def test_rule_with_string_modifiers(self):
        rule = self.new_rule \
            .with_name('rule_with_string_modifiers') \
            .with_plain_string('$1', 'Hello').ascii().wide() \
            .with_plain_string('$2', 'Hello').wide().fullword().nocase() \
            .with_plain_string('$3', 'Hello').xor() \
            .with_plain_string('$4', 'Hello').xor(128) \
            .with_plain_string('$5', 'Hello').wide().xor(1, 255) \
            .with_plain_string('$6', 'Hello').base64() \
            .with_plain_string('$7', 'Hello').base64wide() \
            .with_plain_string('$8', 'Hello').base64('!@#$%^&*(){}[].,|ABCDEFGHIJ\x09LMNOPQRSTUVWXYZabcdefghijklmnopqrstu').private() \
            .with_plain_string('$9', 'Hello').base64wide('!@#$%^&*(){}[].,|ABCDEFGHIJ\x09LMNOPQRSTUVWXYZabcdefghijklmnopqrstu').private() \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get()

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_string_modifiers
{
	strings:
		$1 = "Hello" ascii wide
		$2 = "Hello" wide fullword nocase
		$3 = "Hello" xor
		$4 = "Hello" xor(128)
		$5 = "Hello" wide xor(1-255)
		$6 = "Hello" base64
		$7 = "Hello" base64wide
		$8 = "Hello" base64("!@#$%^&*(){}[].,|ABCDEFGHIJ\\tLMNOPQRSTUVWXYZabcdefghijklmnopqrstu") private
		$9 = "Hello" base64wide("!@#$%^&*(){}[].,|ABCDEFGHIJ\\tLMNOPQRSTUVWXYZabcdefghijklmnopqrstu") private
	condition:
		true
}
''')
        self.assertEqual(yara_file.text, '''rule rule_with_string_modifiers {
	strings:
		$1 = "Hello" ascii wide
		$2 = "Hello" wide nocase fullword
		$3 = "Hello" xor
		$4 = "Hello" xor(128)
		$5 = "Hello" wide xor(1-255)
		$6 = "Hello" base64
		$7 = "Hello" base64wide
		$8 = "Hello" private base64("!@#$%^&*(){}[].,|ABCDEFGHIJ\\tLMNOPQRSTUVWXYZabcdefghijklmnopqrstu")
		$9 = "Hello" private base64wide("!@#$%^&*(){}[].,|ABCDEFGHIJ\\tLMNOPQRSTUVWXYZabcdefghijklmnopqrstu")
	condition:
		true
}''')

    def test_rule_with_custom_modules(self):
        cond = yaramod.conjunction([
            yaramod.id("module_test.structure_test.function_test")(yaramod.regexp("abc", "")),
            yaramod.id("module_test.reference_test.function_test")(yaramod.regexp("cba", "")),
            yaramod.id("cuckoo.sync.mutex")(yaramod.regexp("abc", ""))
        ]).get()
        rule = yaramod.YaraRuleBuilder() \
            .with_name('test') \
            .with_condition(cond)\
            .get()
        yara_file = yaramod.YaraFileBuilder(yaramod.Features.AllCurrent, "./tests/python/testing_modules") \
            .with_module("cuckoo") \
            .with_module("module_test") \
            .with_rule(rule) \
            .get(recheck=True)

        self.assertEqual(yara_file.text_formatted, '''import "cuckoo"
import "module_test"

rule test
{
	condition:
		module_test.structure_test.function_test(/abc/) and
		module_test.reference_test.function_test(/cba/) and
		cuckoo.sync.mutex(/abc/)
}
''')
        self.assertEqual(yara_file.text, '''import "cuckoo"
import "module_test"

rule test {
	condition:
		module_test.structure_test.function_test(/abc/) and module_test.reference_test.function_test(/cba/) and cuckoo.sync.mutex(/abc/)
}''')

    def test_rule_with_defined_condition(self):
        cond = yaramod.int_val(200).defined()
        rule = self.new_rule \
            .with_name('rule_with_defined_condition') \
            .with_condition(cond.get()) \
            .get()
        yara_file = self.new_file \
            .with_rule(rule) \
            .get(True)

        self.assertEqual(yara_file.text_formatted, '''rule rule_with_defined_condition
{
	condition:
		defined 200
}
''')

    def test_rule_with_pe_int_constants_condition(self):
        constans = [
            'MACHINE_TARGET_HOST',
            'MACHINE_R3000',
            'MACHINE_R10000',
            'MACHINE_ALPHA',
            'MACHINE_SH3E',
            'MACHINE_AXP64',
            'MACHINE_ALPHA64',
            'MACHINE_TRICORE',
            'MACHINE_CEF',
            'MACHINE_CEE',
            'SUBSYSTEM_EFI_ROM_IMAGE',
            'HIGH_ENTROPY_VA',
            'APPCONTAINER',
            'GUARD_CF',
            'IMAGE_DIRECTORY_ENTRY_COPYRIGHT',
            'IMAGE_NT_OPTIONAL_HDR32_MAGIC',
            'IMAGE_NT_OPTIONAL_HDR64_MAGIC',
            'IMAGE_ROM_OPTIONAL_HDR_MAGIC',
            'SECTION_NO_PAD',
            'SECTION_LNK_OTHER',
            'SECTION_LNK_INFO',
            'SECTION_LNK_REMOVE',
            'SECTION_LNK_COMDAT',
            'SECTION_NO_DEFER_SPEC_EXC',
            'SECTION_MEM_FARDATA',
            'SECTION_MEM_PURGEABLE',
            'SECTION_MEM_PURGEABLE',
            'SECTION_MEM_LOCKED',
            'SECTION_MEM_PRELOAD',
            'SECTION_ALIGN_1BYTES',
            'SECTION_ALIGN_2BYTES',
            'SECTION_ALIGN_4BYTES',
            'SECTION_ALIGN_8BYTES',
            'SECTION_ALIGN_16BYTES',
            'SECTION_ALIGN_32BYTES',
            'SECTION_ALIGN_64BYTES',
            'SECTION_ALIGN_128BYTES',
            'SECTION_ALIGN_256BYTES',
            'SECTION_ALIGN_512BYTES',
            'SECTION_ALIGN_1024BYTES',
            'SECTION_ALIGN_2048BYTES',
            'SECTION_ALIGN_4096BYTES',
            'SECTION_ALIGN_8192BYTES',
            'SECTION_ALIGN_MASK',
            'SECTION_SCALE_INDEX',
            'IMAGE_DEBUG_TYPE_UNKNOWN',
            'IMAGE_DEBUG_TYPE_COFF',
            'IMAGE_DEBUG_TYPE_CODEVIEW',
            'IMAGE_DEBUG_TYPE_FPO',
            'IMAGE_DEBUG_TYPE_MISC',
            'IMAGE_DEBUG_TYPE_EXCEPTION',
            'IMAGE_DEBUG_TYPE_FIXUP',
            'IMAGE_DEBUG_TYPE_OMAP_FROM_SRC',
            'IMAGE_DEBUG_TYPE_OMAP_TO_SRC',
            'IMAGE_DEBUG_TYPE_BORLAND',
            'IMAGE_DEBUG_TYPE_RESERVED10',
            'IMAGE_DEBUG_TYPE_CLSID',
            'IMAGE_DEBUG_TYPE_VC_FEATURE',
            'IMAGE_DEBUG_TYPE_POGO',
            'IMAGE_DEBUG_TYPE_ILTCG',
            'IMAGE_DEBUG_TYPE_MPX',
            'IMAGE_DEBUG_TYPE_REPRO',
        ]
        cond = yaramod.id('pe').access(constans[0])
        for constant in constans[1:]:
            cond = cond | yaramod.id('pe').access(constant)
        rule = (
            self.new_rule
                .with_name('rule_with_constant_condition')
                .with_condition(cond.get())
                .get()
        )
        yara_file = (
            self.new_file
                .with_module("pe")
                .with_rule(rule)
                .get(True)
        )

        self.assertEqual(
            yara_file.text_formatted,
            'import "pe"\n'
            '\n'
            'rule rule_with_constant_condition\n'
            '{\n'
            '\tcondition:\n'
            f'\t\t{" | ".join(f"pe.{constant}"  for constant in constans)}\n'
            '}\n'
        )
