/*****************************************************************************
 * dvdplay.c: DVD Video Manager library.
 *****************************************************************************
 * Copyright (C) 2002 VideoLAN
 * $Id: dvdplay.c,v 1.8 2003/01/29 22:09:46 sam Exp $
 *
 * Authors: Stphane Borel <stef@via.ecp.fr>
 *
 * Adapted from Ogle - A video player
 * Copyright (C) 2000, 2001 Hkan Hjort
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/*****************************************************************************
 * Preamble
 *****************************************************************************/

#include "config.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#   include <unistd.h>
#endif

#include "common.h"

#include <dvdread/ifo_types.h>
#include <dvdread/ifo_read.h>
#include <dvdread/nav_read.h>

#include "dvdplay/dvdplay.h"

#include "command.h"
#include "vmg.h"
#include "msg.h"

/*****************************************************************************
 * Local prototypes
 *****************************************************************************/

/* in case no callback is defined */
static void _dummy_callback( void* p_args, dvdplay_event_t event )
{
    return;
}

/*
 * Exported functions (prototypes in <dvdplay/dvdplay.h>)
 */

/*****************************************************************************
 * dvdplay_open: allocate structures and initialize video manager
 *****************************************************************************/
extern dvdplay_ptr dvdplay_open( char * psz_dvdroot,
                                  void(*pf_callback)(void*, dvdplay_event_t),
                                 void * p_cb_args )
{
    char *psz_verbose = getenv( "DVDPLAY_VERBOSE" );

    dvdplay_ptr dvdplay;

    /* Allocate main struct */
    dvdplay = malloc( sizeof( struct dvdplay_s ) );
    if( dvdplay == NULL )
    {
        return NULL;
    }

    /* Find verbosity from DVDPLAY_VERBOSE environment variable */
    dvdplay->i_verbosity = 0;

    if( psz_verbose != NULL )
    {
        dvdplay->i_verbosity = atoi( psz_verbose );

        if( dvdplay->i_verbosity < 0 )
        {
            dvdplay->i_verbosity = 0;
        }
        else if( dvdplay->i_verbosity > 3 )
        {
            dvdplay->i_verbosity = 3;
        }
    }

    /* Callback for caller application */
    if( pf_callback != NULL )
    {
        dvdplay->pf_callback = pf_callback;
    }
    else
    {
        dvdplay->pf_callback = _dummy_callback;
    }
    dvdplay->p_args = p_cb_args;

    if( _OpenVMGI( dvdplay, psz_dvdroot ) < 0 )
    {
        _dvdplay_err( dvdplay, "cannot open Video Manager ifo" );
        free( dvdplay );
        return NULL;
    }

    dvdplay->p_vtsi = NULL;
    dvdplay->p_file = NULL;
    dvdplay->b_action_highlight = 0;
    dvdplay_reset( dvdplay );

    _dvdplay_dbg( dvdplay, "dvdplay opened" );

    return dvdplay;
}

/*****************************************************************************
 * dvdplay_close: close libdvdread and free allocated memory
 *****************************************************************************/
extern int dvdplay_close( dvdplay_t * dvdplay )
{
    _dvdplay_dbg( dvdplay, "closing dvdplay" );

    if( dvdplay->p_vmgi != NULL )
    {
        _dvdplay_dbg( dvdplay, "closing vmgi..." );
        ifoClose( dvdplay->p_vmgi );
    }

    if( dvdplay->p_vtsi != NULL )
    {
        _dvdplay_dbg( dvdplay, "closing vtsi" );
        ifoClose( dvdplay->p_vtsi );
    }

    if( dvdplay->p_file != NULL )
    {
        _dvdplay_dbg( dvdplay, "closing dvd file(s)" );
        DVDCloseFile( dvdplay->p_file );
    }

    if( dvdplay->p_dvdread != NULL )
    {
        _dvdplay_dbg( dvdplay, "closing dvdread" );
        DVDClose( dvdplay->p_dvdread );
    }

    _dvdplay_dbg( dvdplay, "dvdplay closed" );

    if( dvdplay != NULL )
    {
        free( dvdplay );
    }

    return 0;
}

/*****************************************************************************
 * dvdplay_start: launch video manager
 *****************************************************************************
 * By default (value of 0 for tt), first play pgc is launched ; if a
 * non-null value is given, the related pgc is launched.
 *****************************************************************************/
extern int dvdplay_start( dvdplay_ptr dvdplay, int i_tt )
{
    _dvdplay_dbg( dvdplay, "starting video manager..." );

    dvdplay_reset( dvdplay );

    if( i_tt )
    {
        if( _SetTT( dvdplay, i_tt ) )
        {
            _dvdplay_err( dvdplay, "cannot start title %d", i_tt );
            return -1;
        }

        /* The client requested i_tt,
         * therefore he doesn't want a pre-command to jump back t a menu */
        _dvdplay_dbg( dvdplay, "PGC pre-command deactivated" );
        dvdplay->b_pgc_pre = 0;
    }
    else
    {
        /* Set pgc to First Play ProGram Chain */
        if( _SetFP_PGC( dvdplay ) )
        {
            _dvdplay_err( dvdplay, "cannot start first play program chain" );
            return -1;
        }
        dvdplay->b_pgc_pre = 1;
    }

    _PlayPGC( dvdplay );
    _ProcessLink( dvdplay );

    if( dvdplay->link.command != PlayThis )
    {
        _dvdplay_warn( dvdplay, "link command is not play (%d)",
                                dvdplay->link.command );
    }

    dvdplay->state.i_blockN = dvdplay->link.data1;

    /* Jump */
    dvdplay->b_jump = 1;

    _dvdplay_dbg( dvdplay, "video manager started" );

    return 0;
}

/*****************************************************************************
 * dvdplay_reset: reinit state and resume info for video manager
 *****************************************************************************/
extern int dvdplay_reset( dvdplay_ptr dvdplay )
{
#define R dvdplay->registers
    /* Setup registers */
    memset( R.SPRM, 0, sizeof( R.SPRM ) );
    memset( R.GPRM, 0, sizeof( R.GPRM ) );
    R.SPRM[0] = ('e'<<8)|'n'; // Player Menu Languange code
    dvdplay->AST_REG = 0; // 15 why?
    dvdplay->SPST_REG = 0; // 62 why?
    dvdplay->AGL_REG = 1;
    dvdplay->TTN_REG = 0;
    dvdplay->VTS_TTN_REG = 1;
    dvdplay->TT_PGCN_REG = 0;
    dvdplay->PTTN_REG = 1;
    dvdplay->HL_BTNN_REG = 1 << 10;

    dvdplay->PTL_REG = 15; // Parental Level
    R.SPRM[12] = ('U'<<8)|'S'; // Parental Management Country Code
    R.SPRM[16] = ('e'<<8)|'n'; // Initial Language Code for Audio
    R.SPRM[18] = ('e'<<8)|'n'; // Initial Language Code for Spu
    R.SPRM[20] = 1; // Player Regional Code (bit mask?)
#undef R

#define S dvdplay->state
    /* Setup state */
    S.i_vtsN = -1;
    S.i_pgN = 0;
    S.i_cellN = 0;
    /* XXX blockN ? -- stef */
    S.domain = FP_DOMAIN;
#undef S

#define RSM dvdplay->resume
    /* Setup resume info */
    RSM.i_vtsN = 0;
    RSM.i_cellN = 0;
    RSM.i_blockN = 0;
#undef RSM

    _dvdplay_dbg( dvdplay, "manager resetted" );

    dvdplay->b_jump = 0;

    return 0;
}

