#include <X11/Xmd.h>

#include "lookup_tbl.h"

#define LUT_CODESET_UNKOWN	0xff
#define LUT_CODE_UNDEFINED	0x00

#define CS_ENTRY	CARD8

static int	LookupTable_addtable(LookupTable *lookup, int target_byte);
static int	LookupTableEntry_add(LookupTable *lookup, int tbl_num, unsigned long src, unsigned long dst, int dst_size, int cs);


LookupTable *
LookupTable_create(int src_size, int dst_size, boolean_t cs_on)
{
	LookupTable *lookup;

	lookup = (LookupTable *)malloc(sizeof(LookupTable));
	if (lookup == (LookupTable *)NULL)
		return (LookupTable *)NULL;

	lookup->src_size = src_size;
	lookup->dst_size = dst_size;
	lookup->cs_on = cs_on;

	lookup->length = 0;
	lookup->alloc_len = 0;
	lookup->alloc_unit = 256;
	lookup->tbl_list = (LookupTableEntry *)NULL;

	lookup->tbl_ptr = LookupTable_addtable(lookup, src_size - 1);
	if (lookup->tbl_ptr < 0){
		free(lookup);
		return (LookupTable *)NULL;
	}
	return lookup;
}

	
int
LookupTable_add(LookupTable *lookup, unsigned long src, unsigned long dst, int cs)
{
	return LookupTableEntry_add(lookup, lookup->tbl_ptr, src, dst, lookup->dst_size, cs);
}


int
LookupTable_save(LookupTable *lookup, FILE *fp)
{
	int	i, j, size;
	CARD32	total_size;
	LookupTableEntry	null_tbl[3];
	int	null_tbl_offset[4];
	int	valid_null_tbl;

	/* count table size & set table offset */
	total_size = 0;
	for(i = 0; i < lookup->length; i++){
		lookup->tbl_list[i].offset = total_size;
		if(lookup->tbl_list[i].cs_entry)
			total_size += sizeof(CS_ENTRY);
		total_size += lookup->tbl_list[i].entry_unit;
	}

	/* create null table */
	for(i = 0; i < lookup->src_size - 1; i++){
		null_tbl[i].target_byte = i;
		null_tbl[i].cs_entry = (void *)NULL;
		null_tbl[i].entry_unit = 0;
		null_tbl[i].entry = (void *)NULL;
	}

	null_tbl_offset[0] = total_size;
	if(lookup->cs_on == B_TRUE){
		null_tbl[0].cs_entry = (CS_ENTRY *)malloc(sizeof(CS_ENTRY) * 256);
		if (null_tbl[0].cs_entry == (CS_ENTRY *)NULL)
			goto err_return;
		memset((void *)(null_tbl[0].cs_entry), LUT_CODESET_UNKOWN, sizeof(CS_ENTRY) * 256);
		null_tbl_offset[1] = null_tbl_offset[0] + sizeof(CS_ENTRY);
	}
	else{
		null_tbl[0].entry_unit = lookup->dst_size;
		null_tbl[0].entry = (void *)malloc(null_tbl[0].entry_unit * 256);
		if (null_tbl[0].entry == (void *)NULL)
			goto err_return;
		memset(null_tbl[0].entry, LUT_CODE_UNDEFINED, null_tbl[0].entry_unit * 256);
		null_tbl_offset[1] = null_tbl_offset[0] + null_tbl[0].entry_unit;
	}

	for(i = 1; i < lookup->src_size - 1; i++){
		null_tbl[i].entry_unit = sizeof(CARD32);
		null_tbl[i].entry = (void *)malloc(null_tbl[i].entry_unit * 256);
		if (null_tbl[i].entry == (void *)NULL)
			goto err_return;
		for(j = 0; j < 256; j++)
			((CARD32 *)(null_tbl[i].entry))[j] = null_tbl_offset[i - 1];
		null_tbl_offset[i + 1] = null_tbl_offset[i] + null_tbl[i].entry_unit;
	}

	/* change table entry (table_num -> offset) */
	valid_null_tbl = -1;
	for(i = 0; i < lookup->length; i++){
		if(lookup->tbl_list[i].target_byte > 0){
			for(j = 0; j < 256; j++){
				if(lookup->src_size == 2){
					if(((CARD16 *)(lookup->tbl_list[i].entry))[j] == 0){
						((CARD16 *)(lookup->tbl_list[i].entry))[j] = null_tbl_offset[lookup->tbl_list[i].target_byte - 1];
						if (valid_null_tbl < lookup->tbl_list[i].target_byte - 1)
							valid_null_tbl = lookup->tbl_list[i].target_byte - 1;
					}
					else
						((CARD16 *)(lookup->tbl_list[i].entry))[j] = lookup->tbl_list[((CARD16 *)(lookup->tbl_list[i].entry))[j]].offset;
				}
				else{
					if(((CARD32 *)(lookup->tbl_list[i].entry))[j] == 0){
						((CARD32 *)(lookup->tbl_list[i].entry))[j] = null_tbl_offset[lookup->tbl_list[i].target_byte - 1];
						if (valid_null_tbl < lookup->tbl_list[i].target_byte - 1)
							valid_null_tbl = lookup->tbl_list[i].target_byte - 1;
					}
					else
						((CARD32 *)(lookup->tbl_list[i].entry))[j] = lookup->tbl_list[((CARD32 *)(lookup->tbl_list[i].entry))[j]].offset;
				}
			}
		}
	}

	/* adjust table size */
	if (valid_null_tbl >= 0)
		total_size = null_tbl_offset[valid_null_tbl + 1];

	/* write table to file */
	if (fwrite(&total_size, sizeof(unsigned long), 1, fp) != 1)
		goto err_return;
	for(i = 0; i < lookup->length; i++){
		if(lookup->tbl_list[i].cs_entry){
			size = sizeof(CS_ENTRY) * 256;
			if (fwrite(lookup->tbl_list[i].cs_entry, 1, size, fp) != size)
				goto err_return;
		}
		size = lookup->tbl_list[i].entry_unit * 256;
		if(fwrite(lookup->tbl_list[i].entry, 1, size, fp) != size)
			goto err_return;
	}
	if (valid_null_tbl >= 0){
		if(null_tbl[0].cs_entry){
			size = sizeof(CS_ENTRY) * 256;
			if (fwrite(null_tbl[0].cs_entry, 1, size, fp) != size)
				goto err_return;
		}
		else{
			size = null_tbl[0].entry_unit * 256;
			if(fwrite(null_tbl[0].entry, 1, size, fp) != size)
				goto err_return;
		}
	}
	for(i = 1; i <= valid_null_tbl; i++){
		size = null_tbl[i].entry_unit * 256;
		if(fwrite(null_tbl[i].entry, 1, size, fp) != size)
			goto err_return;
	}

	/* free resource */
	for(i = 0; i < lookup->src_size - 1; i++){
		if(null_tbl[i].cs_entry)
			free(null_tbl[i].cs_entry);
		if(null_tbl[i].entry)
			free(null_tbl[i].entry);
	}
	
	return 0;
					
err_return:
	for(i = 0; i < lookup->src_size - 1; i++){
		if(null_tbl[i].cs_entry)
			free(null_tbl[i].cs_entry);
		if(null_tbl[i].entry)
			free(null_tbl[i].entry);
	}
	return -1;
}

void
LookupTable_destroy(LookupTable *lookup)
{
	int	i;

	if(lookup->tbl_list){
		for(i = 0; i < lookup->length; i++){
			if(lookup->tbl_list[i].entry)
				free(lookup->tbl_list[i].entry);
			if(lookup->tbl_list[i].cs_entry)
				free(lookup->tbl_list[i].cs_entry);
		}
		free(lookup->tbl_list);
	}
	free(lookup);
}


static int
LookupTable_addtable(LookupTable *lookup, int target_byte)
{
	int	new_length;
	LookupTableEntry	*new_list;

	if(lookup->length >= lookup->alloc_len){
		new_length = lookup->alloc_len + lookup->alloc_unit;
		new_list = (LookupTableEntry *)malloc(sizeof(LookupTableEntry) * new_length);
		if(new_list == (LookupTableEntry *)NULL)
			return -1;

		if(lookup->tbl_list){
			memcpy(new_list, lookup->tbl_list, sizeof(LookupTableEntry) * lookup->length);
			free(lookup->tbl_list);
		}
		lookup->tbl_list = new_list;
		lookup->alloc_len = new_length;
	}

	lookup->tbl_list[lookup->length].target_byte = target_byte;
	lookup->tbl_list[lookup->length].cs_entry = (CS_ENTRY *)NULL;
	lookup->tbl_list[lookup->length].entry_unit = 0;
	lookup->tbl_list[lookup->length].entry = (void *)NULL;
	(lookup->length)++;

	return lookup->length - 1;
}


static int
LookupTableEntry_add(LookupTable *lookup, int tbl_num, unsigned long src, unsigned long dst, int dst_size, int cs)
{
	LookupTableEntry	*tbl;
	unsigned char	num;

	tbl = &(lookup->tbl_list[tbl_num]);

	if(tbl->entry == (void *)NULL){
		if(tbl->target_byte > 0){
			switch(lookup->src_size){
			    case 4:
			    case 3:
				tbl->entry_unit = sizeof(CARD32);
				break;
			    case 2:
			    case 1:
				tbl->entry_unit = sizeof(CARD16);
				break;
			    default:
				return -1;
				break;
			}
		}
		else{
			tbl->entry_unit = dst_size;
		}

		tbl->entry = (void *)malloc(tbl->entry_unit * 256);
		if (tbl->entry == (void *)NULL)
			return -1;

		if(tbl->target_byte > 0)
			memset(tbl->entry, 0, tbl->entry_unit * 256);
		else
			memset(tbl->entry, LUT_CODE_UNDEFINED, tbl->entry_unit * 256);
	}

	if(tbl->target_byte > 0){
		num = (src >> (tbl->target_byte * 8)) & 0xff;
		if(tbl->entry_unit == sizeof(CARD16)){
			if (((CARD16 *)(tbl->entry))[num] == 0){
				((CARD16 *)(tbl->entry))[num] = LookupTable_addtable(lookup, tbl->target_byte - 1);
				if (((CARD16 *)(tbl->entry))[num] < 0)
					return -1;
			}
			return LookupTableEntry_add(lookup, ((CARD16 *)(tbl->entry))[num], src, dst, dst_size, cs);
		}else{
			if (((CARD32 *)(tbl->entry))[num] == 0){
				((CARD32 *)(tbl->entry))[num] = LookupTable_addtable(lookup, tbl->target_byte - 1);
				if (((CARD32 *)(tbl->entry))[num] < 0)
					return -1;
			}
			return LookupTableEntry_add(lookup, ((CARD32 *)(tbl->entry))[num], src, dst, dst_size, cs);
		}
	}
	
	num = src & 0xff;
	switch(dst_size){
	    case 4:
		((CARD32 *)(tbl->entry))[num] = dst;
		break;
	    case 2:
		((CARD16 *)(tbl->entry))[num] = dst;
		break;
	    case 1:
		((CARD8 *)(tbl->entry))[num] = dst;
		break;
	    default:
		return -1;
		break;
	}

	if (cs >= 0){
		if(tbl->cs_entry == (CS_ENTRY *)NULL){
			tbl->cs_entry = (CS_ENTRY *)malloc(sizeof(CS_ENTRY) * 256);
			if(tbl->cs_entry == (CS_ENTRY *)NULL)
				return 1;
			memset(tbl->cs_entry, LUT_CODESET_UNKOWN, sizeof(CS_ENTRY) * 256);
		}	
		((CS_ENTRY *)(tbl->cs_entry))[num] = cs;
	}

	return 0;
}
