/*
 Copyright (C) Intel Corp.  2006.  All Rights Reserved.
 Intel funded Tungsten Graphics (http://www.tungstengraphics.com) to
 develop this 3D driver.
 
 Permission is hereby granted, free of charge, to any person obtaining
 a copy of this software and associated documentation files (the
 "Software"), to deal in the Software without restriction, including
 without limitation the rights to use, copy, modify, merge, publish,
 distribute, sublicense, and/or sell copies of the Software, and to
 permit persons to whom the Software is furnished to do so, subject to
 the following conditions:
 
 The above copyright notice and this permission notice (including the
 next paragraph) shall be included in all copies or substantial
 portions of the Software.
 
 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 IN NO EVENT SHALL THE COPYRIGHT OWNER(S) AND/OR ITS SUPPLIERS BE
 LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 
 **********************************************************************/
 /*
  * Authors:
  *   Keith Whitwell <keith@tungstengraphics.com>
  */
           

#include "main/compiler.h"
#include "brw_context.h"
#include "brw_vs.h"
#include "brw_util.h"
#include "brw_state.h"
#include "program/prog_print.h"
#include "program/prog_parameter.h"

#include "glsl/ralloc.h"

static inline void assign_vue_slot(struct brw_vue_map *vue_map,
                                   int vert_result)
{
   /* Make sure this vert_result hasn't been assigned a slot already */
   assert (vue_map->vert_result_to_slot[vert_result] == -1);

   vue_map->vert_result_to_slot[vert_result] = vue_map->num_slots;
   vue_map->slot_to_vert_result[vue_map->num_slots++] = vert_result;
}

/**
 * Compute the VUE map for vertex shader program.
 *
 * Note that consumers of this map using cache keys must include
 * prog_data->userclip and prog_data->outputs_written in their key
 * (generated by CACHE_NEW_VS_PROG).
 */
static void
brw_compute_vue_map(struct brw_context *brw, struct brw_vs_compile *c)
{
   const struct intel_context *intel = &brw->intel;
   struct brw_vue_map *vue_map = &c->prog_data.vue_map;
   GLbitfield64 outputs_written;
   int i;

   if (intel->gen < 6)
      c->prog_data.outputs_written &= ~BITFIELD64_BIT(VERT_RESULT_CLIP_VERTEX);
   outputs_written = c->prog_data.outputs_written;

   vue_map->num_slots = 0;
   for (i = 0; i < BRW_VERT_RESULT_MAX; ++i) {
      vue_map->vert_result_to_slot[i] = -1;
      vue_map->slot_to_vert_result[i] = BRW_VERT_RESULT_MAX;
   }

   /* VUE header: format depends on chip generation and whether clipping is
    * enabled.
    */
   switch (intel->gen) {
   case 4:
      /* There are 8 dwords in VUE header pre-Ironlake:
       * dword 0-3 is indices, point width, clip flags.
       * dword 4-7 is ndc position
       * dword 8-11 is the first vertex data.
       */
      assign_vue_slot(vue_map, VERT_RESULT_PSIZ);
      assign_vue_slot(vue_map, BRW_VERT_RESULT_NDC);
      assign_vue_slot(vue_map, VERT_RESULT_HPOS);
      break;
   case 5:
      /* There are 20 DWs (D0-D19) in VUE header on Ironlake:
       * dword 0-3 of the header is indices, point width, clip flags.
       * dword 4-7 is the ndc position
       * dword 8-11 of the vertex header is the 4D space position
       * dword 12-19 of the vertex header is the user clip distance.
       * dword 20-23 is a pad so that the vertex element data is aligned
       * dword 24-27 is the first vertex data we fill.
       *
       * Note: future pipeline stages expect 4D space position to be
       * contiguous with the other vert_results, so we make dword 24-27 a
       * duplicate copy of the 4D space position.
       */
      assign_vue_slot(vue_map, VERT_RESULT_PSIZ);
      assign_vue_slot(vue_map, BRW_VERT_RESULT_NDC);
      assign_vue_slot(vue_map, BRW_VERT_RESULT_HPOS_DUPLICATE);
      assign_vue_slot(vue_map, VERT_RESULT_CLIP_DIST0);
      assign_vue_slot(vue_map, VERT_RESULT_CLIP_DIST1);
      assign_vue_slot(vue_map, BRW_VERT_RESULT_PAD);
      assign_vue_slot(vue_map, VERT_RESULT_HPOS);
      break;
   case 6:
   case 7:
      /* There are 8 or 16 DWs (D0-D15) in VUE header on Sandybridge:
       * dword 0-3 of the header is indices, point width, clip flags.
       * dword 4-7 is the 4D space position
       * dword 8-15 of the vertex header is the user clip distance if
       * enabled.
       * dword 8-11 or 16-19 is the first vertex element data we fill.
       */
      assign_vue_slot(vue_map, VERT_RESULT_PSIZ);
      assign_vue_slot(vue_map, VERT_RESULT_HPOS);
      if (c->key.userclip_active) {
         assign_vue_slot(vue_map, VERT_RESULT_CLIP_DIST0);
         assign_vue_slot(vue_map, VERT_RESULT_CLIP_DIST1);
      }
      /* front and back colors need to be consecutive so that we can use
       * ATTRIBUTE_SWIZZLE_INPUTATTR_FACING to swizzle them when doing
       * two-sided color.
       */
      if (outputs_written & BITFIELD64_BIT(VERT_RESULT_COL0))
         assign_vue_slot(vue_map, VERT_RESULT_COL0);
      if (outputs_written & BITFIELD64_BIT(VERT_RESULT_BFC0))
         assign_vue_slot(vue_map, VERT_RESULT_BFC0);
      if (outputs_written & BITFIELD64_BIT(VERT_RESULT_COL1))
         assign_vue_slot(vue_map, VERT_RESULT_COL1);
      if (outputs_written & BITFIELD64_BIT(VERT_RESULT_BFC1))
         assign_vue_slot(vue_map, VERT_RESULT_BFC1);
      break;
   default:
      assert (!"VUE map not known for this chip generation");
      break;
   }

   /* The hardware doesn't care about the rest of the vertex outputs, so just
    * assign them contiguously.  Don't reassign outputs that already have a
    * slot.
    *
    * Also, prior to Gen6, don't assign a slot for VERT_RESULT_CLIP_VERTEX,
    * since it is unsupported.  In Gen6 and above, VERT_RESULT_CLIP_VERTEX may
    * be needed for transform feedback; since we don't want to have to
    * recompute the VUE map (and everything that depends on it) when transform
    * feedback is enabled or disabled, just go ahead and assign a slot for it.
    */
   for (int i = 0; i < VERT_RESULT_MAX; ++i) {
      if ((outputs_written & BITFIELD64_BIT(i)) &&
          vue_map->vert_result_to_slot[i] == -1) {
         assign_vue_slot(vue_map, i);
      }
   }
}


/**
 * Decide which set of clip planes should be used when clipping via
 * gl_Position or gl_ClipVertex.
 */
gl_clip_plane *brw_select_clip_planes(struct gl_context *ctx)
{
   if (ctx->Shader.CurrentVertexProgram) {
      /* There is currently a GLSL vertex shader, so clip according to GLSL
       * rules, which means compare gl_ClipVertex (or gl_Position, if
       * gl_ClipVertex wasn't assigned) against the eye-coordinate clip planes
       * that were stored in EyeUserPlane at the time the clip planes were
       * specified.
       */
      return ctx->Transform.EyeUserPlane;
   } else {
      /* Either we are using fixed function or an ARB vertex program.  In
       * either case the clip planes are going to be compared against
       * gl_Position (which is in clip coordinates) so we have to clip using
       * _ClipUserPlane, which was transformed into clip coordinates by Mesa
       * core.
       */
      return ctx->Transform._ClipUserPlane;
   }
}

bool
brw_vs_prog_data_compare(const void *in_a, const void *in_b,
                         int aux_size, const void *in_key)
{
   const struct brw_vs_prog_data *a = in_a;
   const struct brw_vs_prog_data *b = in_b;

   /* Compare all the struct up to the pointers. */
   if (memcmp(a, b, offsetof(struct brw_vs_prog_data, param)))
      return false;

   if (memcmp(a->param, b->param, a->nr_params * sizeof(void *)))
      return false;

   if (memcmp(a->pull_param, b->pull_param, a->nr_pull_params * sizeof(void *)))
      return false;

   return true;
}

static bool
do_vs_prog(struct brw_context *brw,
	   struct gl_shader_program *prog,
	   struct brw_vertex_program *vp,
	   struct brw_vs_prog_key *key)
{
   struct intel_context *intel = &brw->intel;
   GLuint program_size;
   const GLuint *program;
   struct brw_vs_compile c;
   void *mem_ctx;
   int i;
   struct gl_shader *vs = NULL;

   if (prog)
      vs = prog->_LinkedShaders[MESA_SHADER_VERTEX];

   memset(&c, 0, sizeof(c));
   memcpy(&c.key, key, sizeof(*key));

   mem_ctx = ralloc_context(NULL);

   c.vp = vp;

   /* Allocate the references to the uniforms that will end up in the
    * prog_data associated with the compiled program, and which will be freed
    * by the state cache.
    */
   int param_count;
   if (vs) {
      /* We add padding around uniform values below vec4 size, with the worst
       * case being a float value that gets blown up to a vec4, so be
       * conservative here.
       */
      param_count = vs->num_uniform_components * 4;

   } else {
      param_count = vp->program.Base.Parameters->NumParameters * 4;
   }
   /* We also upload clip plane data as uniforms */
   param_count += MAX_CLIP_PLANES * 4;

   c.prog_data.param = rzalloc_array(NULL, const float *, param_count);
   c.prog_data.pull_param = rzalloc_array(NULL, const float *, param_count);

   c.prog_data.outputs_written = vp->program.Base.OutputsWritten;
   c.prog_data.inputs_read = vp->program.Base.InputsRead;

   if (c.key.copy_edgeflag) {
      c.prog_data.outputs_written |= BITFIELD64_BIT(VERT_RESULT_EDGE);
      c.prog_data.inputs_read |= VERT_BIT_EDGEFLAG;
   }

   /* Put dummy slots into the VUE for the SF to put the replaced
    * point sprite coords in.  We shouldn't need these dummy slots,
    * which take up precious URB space, but it would mean that the SF
    * doesn't get nice aligned pairs of input coords into output
    * coords, which would be a pain to handle.
    */
   for (i = 0; i < 8; i++) {
      if (c.key.point_coord_replace & (1 << i))
	 c.prog_data.outputs_written |= BITFIELD64_BIT(VERT_RESULT_TEX0 + i);
   }

   brw_compute_vue_map(brw, &c);

   if (0) {
      _mesa_fprint_program_opt(stdout, &c.vp->program.Base, PROG_PRINT_DEBUG,
			       true);
   }

   /* Emit GEN4 code.
    */
   program = brw_vs_emit(brw, prog, &c, mem_ctx, &program_size);
   if (program == NULL) {
      ralloc_free(mem_ctx);
      return false;
   }

   if (c.prog_data.nr_pull_params)
      c.prog_data.num_surfaces = 1;
   if (c.vp->program.Base.SamplersUsed)
      c.prog_data.num_surfaces = SURF_INDEX_VS_TEXTURE(BRW_MAX_TEX_UNIT);
   if (prog &&
       prog->_LinkedShaders[MESA_SHADER_VERTEX]->NumUniformBlocks) {
      c.prog_data.num_surfaces =
	 SURF_INDEX_VS_UBO(prog->_LinkedShaders[MESA_SHADER_VERTEX]->NumUniformBlocks);
   }

   /* Scratch space is used for register spilling */
   if (c.last_scratch) {
      perf_debug("Vertex shader triggered register spilling.  "
                 "Try reducing the number of live vec4 values to "
                 "improve performance.\n");

      c.prog_data.total_scratch = brw_get_scratch_size(c.last_scratch*REG_SIZE);

      brw_get_scratch_bo(intel, &brw->vs.scratch_bo,
			 c.prog_data.total_scratch * brw->max_vs_threads);
   }

   brw_upload_cache(&brw->cache, BRW_VS_PROG,
		    &c.key, sizeof(c.key),
		    program, program_size,
		    &c.prog_data, sizeof(c.prog_data),
		    &brw->vs.prog_offset, &brw->vs.prog_data);
   ralloc_free(mem_ctx);

   return true;
}

static bool
key_debug(const char *name, int a, int b)
{
   if (a != b) {
      perf_debug("  %s %d->%d\n", name, a, b);
      return true;
   }
   return false;
}

void
brw_vs_debug_recompile(struct brw_context *brw,
                       struct gl_shader_program *prog,
                       const struct brw_vs_prog_key *key)
{
   struct brw_cache_item *c = NULL;
   const struct brw_vs_prog_key *old_key = NULL;
   bool found = false;

   perf_debug("Recompiling vertex shader for program %d\n", prog->Name);

   for (unsigned int i = 0; i < brw->cache.size; i++) {
      for (c = brw->cache.items[i]; c; c = c->next) {
         if (c->cache_id == BRW_VS_PROG) {
            old_key = c->key;

            if (old_key->program_string_id == key->program_string_id)
               break;
         }
      }
      if (c)
         break;
   }

   if (!c) {
      perf_debug("  Didn't find previous compile in the shader cache for "
                 "debug\n");
      return;
   }

   for (unsigned int i = 0; i < VERT_ATTRIB_MAX; i++) {
      found |= key_debug("Vertex attrib w/a flags",
                         old_key->gl_attrib_wa_flags[i],
                         key->gl_attrib_wa_flags[i]);
   }

   found |= key_debug("user clip flags",
                      old_key->userclip_active, key->userclip_active);

   found |= key_debug("user clipping planes as push constants",
                      old_key->nr_userclip_plane_consts,
                      key->nr_userclip_plane_consts);

   found |= key_debug("clip distance enable",
                      old_key->uses_clip_distance, key->uses_clip_distance);
   found |= key_debug("clip plane enable bitfield",
                      old_key->userclip_planes_enabled_gen_4_5,
                      key->userclip_planes_enabled_gen_4_5);
   found |= key_debug("copy edgeflag",
                      old_key->copy_edgeflag, key->copy_edgeflag);
   found |= key_debug("PointCoord replace",
                      old_key->point_coord_replace, key->point_coord_replace);
   found |= key_debug("vertex color clamping",
                      old_key->clamp_vertex_color, key->clamp_vertex_color);

   found |= brw_debug_recompile_sampler_key(&old_key->tex, &key->tex);

   if (!found) {
      perf_debug("  Something else\n");
   }
}

static void brw_upload_vs_prog(struct brw_context *brw)
{
   struct intel_context *intel = &brw->intel;
   struct gl_context *ctx = &intel->ctx;
   struct brw_vs_prog_key key;
   /* BRW_NEW_VERTEX_PROGRAM */
   struct brw_vertex_program *vp = 
      (struct brw_vertex_program *)brw->vertex_program;
   struct gl_program *prog = (struct gl_program *) brw->vertex_program;
   int i;

   memset(&key, 0, sizeof(key));

   /* Just upload the program verbatim for now.  Always send it all
    * the inputs it asks for, whether they are varying or not.
    */
   key.program_string_id = vp->id;
   key.userclip_active = (ctx->Transform.ClipPlanesEnabled != 0);
   key.uses_clip_distance = vp->program.UsesClipDistance;
   if (key.userclip_active && !key.uses_clip_distance) {
      if (intel->gen < 6) {
         key.nr_userclip_plane_consts
            = _mesa_bitcount_64(ctx->Transform.ClipPlanesEnabled);
         key.userclip_planes_enabled_gen_4_5
            = ctx->Transform.ClipPlanesEnabled;
      } else {
         key.nr_userclip_plane_consts
            = _mesa_logbase2(ctx->Transform.ClipPlanesEnabled) + 1;
      }
   }

   /* _NEW_POLYGON */
   if (intel->gen < 6) {
      key.copy_edgeflag = (ctx->Polygon.FrontMode != GL_FILL ||
                           ctx->Polygon.BackMode != GL_FILL);
   }

   /* _NEW_LIGHT | _NEW_BUFFERS */
   key.clamp_vertex_color = ctx->Light._ClampVertexColor;

   /* _NEW_POINT */
   if (ctx->Point.PointSprite) {
      for (i = 0; i < 8; i++) {
	 if (ctx->Point.CoordReplace[i])
	    key.point_coord_replace |= (1 << i);
      }
   }

   /* _NEW_TEXTURE */
   brw_populate_sampler_prog_key_data(ctx, prog, &key.tex);

   /* BRW_NEW_VERTICES */
   if (intel->gen < 8 && !intel->is_haswell) {
      /* Prior to Haswell, the hardware can't natively support GL_FIXED or
       * 2_10_10_10_REV vertex formats.  Set appropriate workaround flags.
       */
      for (i = 0; i < VERT_ATTRIB_MAX; i++) {
         if (!(vp->program.Base.InputsRead & BITFIELD64_BIT(i)))
            continue;

         uint8_t wa_flags = 0;

         switch (brw->vb.inputs[i].glarray->Type) {

         case GL_FIXED:
            wa_flags = brw->vb.inputs[i].glarray->Size;
            break;

         case GL_INT_2_10_10_10_REV:
            wa_flags |= BRW_ATTRIB_WA_SIGN;
            /* fallthough */

         case GL_UNSIGNED_INT_2_10_10_10_REV:
            if (brw->vb.inputs[i].glarray->Format == GL_BGRA)
               wa_flags |= BRW_ATTRIB_WA_BGRA;

            if (brw->vb.inputs[i].glarray->Normalized)
               wa_flags |= BRW_ATTRIB_WA_NORMALIZE;
            else if (!brw->vb.inputs[i].glarray->Integer)
               wa_flags |= BRW_ATTRIB_WA_SCALE;

            break;
         }

         key.gl_attrib_wa_flags[i] = wa_flags;
      }
   }

   if (!brw_search_cache(&brw->cache, BRW_VS_PROG,
			 &key, sizeof(key),
			 &brw->vs.prog_offset, &brw->vs.prog_data)) {
      bool success = do_vs_prog(brw, ctx->Shader.CurrentVertexProgram,
				vp, &key);

      assert(success);
   }
}

/* See brw_vs.c:
 */
const struct brw_tracked_state brw_vs_prog = {
   .dirty = {
      .mesa  = (_NEW_TRANSFORM | _NEW_POLYGON | _NEW_POINT | _NEW_LIGHT |
		_NEW_TEXTURE |
		_NEW_BUFFERS),
      .brw   = (BRW_NEW_VERTEX_PROGRAM |
		BRW_NEW_VERTICES),
      .cache = 0
   },
   .emit = brw_upload_vs_prog
};

bool
brw_vs_precompile(struct gl_context *ctx, struct gl_shader_program *prog)
{
   struct brw_context *brw = brw_context(ctx);
   struct brw_vs_prog_key key;
   uint32_t old_prog_offset = brw->vs.prog_offset;
   struct brw_vs_prog_data *old_prog_data = brw->vs.prog_data;
   bool success;

   if (!prog->_LinkedShaders[MESA_SHADER_VERTEX])
      return true;

   struct gl_vertex_program *vp = (struct gl_vertex_program *)
      prog->_LinkedShaders[MESA_SHADER_VERTEX]->Program;
   struct brw_vertex_program *bvp = brw_vertex_program(vp);

   memset(&key, 0, sizeof(key));

   key.program_string_id = bvp->id;
   key.clamp_vertex_color = true;

   for (int i = 0; i < MAX_SAMPLERS; i++) {
      if (vp->Base.ShadowSamplers & (1 << i)) {
         /* Assume DEPTH_TEXTURE_MODE is the default: X, X, X, 1 */
         key.tex.swizzles[i] =
            MAKE_SWIZZLE4(SWIZZLE_X, SWIZZLE_X, SWIZZLE_X, SWIZZLE_ONE);
      } else {
         /* Color sampler: assume no swizzling. */
         key.tex.swizzles[i] = SWIZZLE_XYZW;
      }
   }

   success = do_vs_prog(brw, prog, bvp, &key);

   brw->vs.prog_offset = old_prog_offset;
   brw->vs.prog_data = old_prog_data;

   return success;
}

void
brw_vs_prog_data_free(const void *in_prog_data)
{
   const struct brw_vs_prog_data *prog_data = in_prog_data;

   ralloc_free((void *)prog_data->param);
   ralloc_free((void *)prog_data->pull_param);
}
